//**************************************************************************
//*                     This file is part of the                           *
//*             AudioCV - a general audio converter program                *
//*                  The source code of AudioCV is                         *
//*          (c) copyright 2001-2004 by PDSoft (Attila Padar)              *
//*                    http://mpxplay.cjb.net                              *
//* email: mpxplay@freemail.hu (please write AudioCV in the subject field) *
//**************************************************************************
//main

#include "audiocv.h"

//------------------------------------------------------------------------
static int  init_input(acv_fileinfo *);
static int  init_output(acv_fileinfo *,acv_fileinfo *);
static int  read_input(acv_fileinfo *);
static int  convert_input(acv_fileinfo *,acv_fileinfo *,int);
static int  write_output(acv_fileinfo *);
static void update_output_header(acv_fileinfo *);
static void close_input(acv_fileinfo *);
static void close_output(acv_fileinfo *);
static void rename_and_delete_at_end(acv_fileinfo *,acv_fileinfo *);

static void display_progress(acv_fileinfo *,acv_fileinfo *,unsigned long,int,int);
static void display_finish(acv_fileinfo *,acv_fileinfo *);
static unsigned long get_timeh(void);
static unsigned int check_esc(void);

static unsigned int get_extension_filetypenum(char *);
static char *get_shortfilename(char *);
static void acv_change_extension_by_newext(char *filename,char *newext);
       void acv_change_extension_by_typenum(char *,unsigned int);
static void acv_rename_extension(char *filename,char *newext);
       int  acv_check_and_del_file(char *filename);
       long acv_filelen(FILE *);
static int  acv_strcmp(char *,char *);
//------------------------------------------------------------------------
extern void get_commandline_options(int argc, char *argv[]);
extern void opt_reset_variables(acv_fileinfo *,acv_fileinfo *);
extern void opt_accept_filenames(acv_fileinfo *,acv_fileinfo *);
extern void opt_accept_variables(acv_fileinfo *,acv_fileinfo *);
extern int  list_open(void);
extern void list_close(void);
extern int  list_getnextentry(acv_fileinfo *,acv_fileinfo *);
//------------------------------------------------------------------------
extern int  conv_main(acv_fileinfo *,acv_fileinfo *,int);
extern int  conv_init(acv_fileinfo *,acv_fileinfo *);
extern void conv_close(void);
//------------------------------------------------------------------------
extern int  wav_readheader(acv_fileinfo *);
extern void wav_readdata(acv_fileinfo *,unsigned int);
extern int  wav_writeheader(acv_fileinfo *);
extern int  wav_writedata(acv_fileinfo *);
//------------------------------------------------------------------------
extern int  ogg_init_encoder(acv_fileinfo *);
extern int  ogg_encode(acv_fileinfo *);
extern int  ogg_update_header(acv_fileinfo *);
extern void ogg_close_encoder(acv_fileinfo *);
extern int  ogg_init_decoder(acv_fileinfo *);
extern void ogg_decode(acv_fileinfo *,int);
extern void ogg_close_decoder(acv_fileinfo *);
//------------------------------------------------------------------------
extern int opt_functions,dispquiet;
extern float cv_normdb;

static struct{
 char *extension;
 unsigned int type;
}filetypes[]={
  {".   ",0}, // FT_UNKNOWN
  {".wav",0}, // FT_WAV
  {".ogg",0}, // FT_OGG
  {".mp3",0}, // FT_MP3
  {".mp2",0}, // FT_MP2
  {NULL,0},
  };

void main(int argc,char **argv)
{
 int eos,eoc,listuse,quit=0,acverr;
 unsigned long begintimeh;
 acv_fileinfo af_in;
 acv_fileinfo af_out;

 get_commandline_options(argc,argv);

 listuse=list_open();
 if(!listuse){
  opt_reset_variables(&af_in,&af_out);
  opt_accept_filenames(&af_in,&af_out);
 }
 do{
  acverr=0;
  if(listuse){
   opt_reset_variables(&af_in,&af_out);
   if(!list_getnextentry(&af_in,&af_out))
    break; // exit
  }
  opt_accept_variables(&af_in,&af_out);
  if(init_input(&af_in)>=0){
   if((acverr=init_output(&af_in,&af_out))>=0 && (acverr=conv_init(&af_in,&af_out))>=0){
    // analiser part ------------------------------------------------------
    if(af_in.filemode&CF_PASS1){
     eos=eoc=0;
     begintimeh=get_timeh();
     while(!eos && !eoc && !quit){
      eos=read_input(&af_in);
      eoc=convert_input(&af_in,&af_out,0);

      display_progress(&af_in,&af_out,begintimeh,0,0);
      quit=check_esc();
     }
     display_progress(&af_in,&af_out,begintimeh,0,1);
    }
    // conversion part ----------------------------------------------------
    if((af_out.filemode&CF_PASS2) && !quit){
     if(af_in.filemode&CF_PASS1){  // re-open input file (rewind)
      close_input(&af_in);
      acverr=init_input(&af_in);
      if(acverr<0)
       break; // exit
     }
     eos=eoc=0;
     begintimeh=get_timeh();
     while(!eos && !eoc && !quit && (acverr==0)){
      eos=read_input(&af_in);
      eoc=convert_input(&af_in,&af_out,1);

      if(af_out.blocksamplenum)
       acverr=write_output(&af_out);

      display_progress(&af_in,&af_out,begintimeh,1,0);
      quit=check_esc();
     }
     acverr=write_output(&af_out);  // flush (ogg encoder)
     display_progress(&af_in,&af_out,begintimeh,1,1);
     update_output_header(&af_out);
    }
    // ---------------------------------------------------------------------
    display_finish(&af_in,&af_out);
    //close_output(&af_out);
   }/*else{
    close_output(&af_out);
    af_out.filetype=0;
   }
   conv_close();

   if(!quit && (acverr==0))
    rename_and_delete_at_end(&af_in,&af_out);*/
  }
  close_input(&af_in);
  close_output(&af_out);
  conv_close();
  if(!quit && (acverr==0))
   rename_and_delete_at_end(&af_in,&af_out);

 }while(listuse && !quit);
 list_close();
}

static int init_input(acv_fileinfo *af_in)
{
 int error;
 if((af_in->fp=fopen(af_in->filename,"rb"))==NULL){
  fprintf(stderr,"Cannot open input file (%s)!\n",af_in->filename);
  return ACV_ERROR_FILE_CANTOPEN;
 }

 af_in->filetype=get_extension_filetypenum(af_in->filename);
 switch(af_in->filetype){
  case FT_WAV:error=wav_readheader(af_in);break;    //wav
  case FT_OGG:error=ogg_init_decoder(af_in);        //ogg
	      if(error<0)
	       ogg_close_decoder(af_in);
	      break;
  default:error=-1;break;
 }
 if(error<0){                                       //autodetect
  if(wav_readheader(af_in)<0){
   if(ogg_init_decoder(af_in)<0){
    fprintf(stderr,"Couldn't handle input file (unknown filetype)!\n");
    return ACV_ERROR_FILETYPE_UNKNOWN;
   }
  }
 }

 if(af_in->channels>2)
  fprintf(stderr,"Warning: %s contains %d channels (more than 2)!",af_in->filename,af_in->channels);

 af_in->buffer=(char *)malloc(PCM_BUFFSIZE*af_in->channels*sizeof(PCM_CV_TYPE_F));
 if(af_in->buffer==NULL){
  fprintf(stderr,"Couldn't alloc input buffer!\n");
  return ACV_ERROR_MEMORY;
 }
 return 0;
}

static int init_output(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 int error=0;

 //set output chan,freq,bits
 if(!af_out->freq)
  af_out->freq=af_in->freq;
 else
  if(af_out->freq==af_in->freq){  // cv_freq is not required
   af_in->filemode &=~CF_FREQ;    // disable it
   af_out->filemode&=~CF_FREQ;
  }

 if(!af_out->channels)
  af_out->channels=af_in->channels;
 else
  if(af_out->channels==af_in->channels){ // cv_channel is not required
   af_in->filemode &=~CF_CHANNELS;       // disable it
   af_out->filemode&=~CF_CHANNELS;
  }

 if(!af_out->databits){    // keep bit resolution at wav to wav conversion
  af_out->databits=af_in->databits;
  af_out->scalebits=af_in->scalebits;
 }

 if(!af_out->bitrate)      // keep bitrate at ogg to ogg conversion
  af_out->bitrate=af_in->bitrate;

 if(!af_out->tag_artist)
  af_out->tag_artist=af_in->tag_artist;
 if(!af_out->tag_title)
  af_out->tag_title =af_in->tag_title;
 if(!af_out->tag_album)
  af_out->tag_album  =af_in->tag_album;
 if(!af_out->tag_date)
  af_out->tag_date   =af_in->tag_date;
 if(!af_out->tag_genre)
  af_out->tag_genre  =af_in->tag_genre;
 if(!af_out->tag_comment)
  af_out->tag_comment=af_in->tag_comment;

 //open output file, write header
 if(af_out->filemode&CF_OUTPUT){ //open output file
  af_out->filetype=get_extension_filetypenum(af_out->filename);
  //if input and output filenames are the same
  //write output into a filename.tmp file
  if(acv_strcmp(af_in->filename,af_out->filename)==0){
   acv_change_extension_by_newext(af_out->filename,".tmp");
   opt_functions|=FUNC_RENAMEFILES;
  }
  if(acv_check_and_del_file(af_out->filename)!=0){
   fprintf(stderr,"Output file already exists (%s)!\n",af_out->filename);
   return ACV_ERROR_FILE_ALREADYEXISTS;
  }
  if((af_out->fp=fopen(af_out->filename,"wb"))==NULL){
   fprintf(stderr,"Cannot open output file (%s)!\n",af_out->filename);
   return ACV_ERROR_FILE_CANTOPEN;
  }

  switch(af_out->filetype){
   case FT_WAV:error=wav_writeheader(af_out);break;
   case FT_OGG:error=ogg_init_encoder(af_out);break;
   default:fprintf(stderr,"Unknown output filetype!\n");
	   return ACV_ERROR_FILETYPE_UNKNOWN;
  }
  if(error==0)
   af_out->headerlen=ftell(af_out->fp);
 }
 if(error==0){ // we always have to allocate the output buffer (for the analyzing functions)
  long mbuffsize=PCM_BUFFSIZE*max(af_in->channels,af_out->channels);
  mbuffsize=(long)((float)mbuffsize*((float)max(af_in->freq,af_out->freq))/(float)af_in->freq);
  mbuffsize+=8192;
  mbuffsize*=sizeof(PCM_CV_TYPE_F);
  af_out->buffer=(char *)malloc(mbuffsize);
  if(af_out->buffer==NULL){
   fprintf(stderr,"Couldn't alloc output buffer!\n");
   error=ACV_ERROR_MEMORY;
  }
 }
 return error;
}

static int read_input(acv_fileinfo *af_in)
{
 switch(af_in->filetype){
  case FT_WAV:wav_readdata(af_in,PCM_INSAMPLES);break;
  case FT_OGG:ogg_decode(af_in,PCM_INSAMPLES);break;
  default    :af_in->blocksamplenum=0;break;
 }
 af_in->currsamplenum+=af_in->blocksamplenum;
 return (!af_in->blocksamplenum);
}

static int convert_input(acv_fileinfo *af_in,acv_fileinfo *af_out,int pass)
{
 int eoc=0;
 af_out->blocksamplenum=af_in->blocksamplenum;
 if(af_in->blocksamplenum)
  eoc=conv_main(af_in,af_out,pass);
 return eoc;
}

static int write_output(acv_fileinfo *af_out)
{
 int acverr;
 af_out->currsamplenum+=af_out->blocksamplenum;
 switch(af_out->filetype){
  case FT_WAV:acverr=wav_writedata(af_out);break;
  case FT_OGG:acverr=ogg_encode(af_out);break;
  default:acverr=ACV_ERROR_FILETYPE_UNKNOWN;break;
 }
 switch(acverr){
  case ACV_ERROR_FILE_CANTWRITE:fprintf(stderr,"Can't write output file!\n");break;
 }
 return acverr;
}

static void update_output_header(acv_fileinfo *af_out)
{
 int error=0;

 if(af_out->filetype){
  af_out->allsamplenum=af_out->currsamplenum;
  fseek(af_out->fp,0,SEEK_SET);
  switch(af_out->filetype){
   case FT_WAV:error=wav_writeheader(af_out);break;
   case FT_OGG:error=ogg_update_header(af_out);break;
  }
  if(error!=0 || ftell(af_out->fp)!=af_out->headerlen)
   fprintf(stderr,"\nOoops, header update failed ... do inform the developer!\n");
 }
}

static void close_input(acv_fileinfo *af_in)
{
 switch(af_in->filetype){
  case FT_OGG:ogg_close_decoder(af_in);
 }
 if(af_in->fp){
  fclose(af_in->fp);
  af_in->fp=NULL;
 }
 if(af_in->buffer){
  free(af_in->buffer);
  af_in->buffer=NULL;
 }
 af_in->currsamplenum=0;
}

static void close_output(acv_fileinfo *af_out)
{
 switch(af_out->filetype){
  case FT_OGG:ogg_close_encoder(af_out);break;
 }
 if(af_out->fp){
  fclose(af_out->fp);
  af_out->fp=NULL;
 }
 if(af_out->buffer){
  free(af_out->buffer);
  af_out->buffer=NULL;
 }
}

static void rename_and_delete_at_end(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 //input and output filenames were same
 //rename input file to .old
 //rename .tmp outfile to the proper output name
 if(af_out->filetype){
  if(opt_functions&FUNC_DELETEINFILE){
   if(remove(af_in->filename)!=0)
    fprintf(stderr,"Couldn't delete input file (%s)!\n",af_in->filename);
   //else
    //fprintf(stderr,"Delete: %s\n",get_shortfilename(af_in->filename));
  }
  if(opt_functions&FUNC_RENAMEFILES){
   if(!(opt_functions&FUNC_DELETEINFILE))
    acv_rename_extension(af_in->filename,".old");
   acv_rename_extension(af_out->filename,filetypes[af_out->filetype].extension);
  }
 }
}

//-------------------------------------------------------------------------
// progress display

static void display_progress(acv_fileinfo *af_in,acv_fileinfo *af_out,unsigned long begintimeh,int pass,int final)
{
 static unsigned long lasttimeh;
 unsigned long curtimeh;
 if(!dispquiet){
  curtimeh=get_timeh();
  if((curtimeh-lasttimeh)>=DISP_REF_INT || final){
   unsigned long inallsamplenum=af_in->allsamplenum;
   unsigned long samplesdone=af_in->currsamplenum;
   unsigned long percent=(unsigned long)((float)samplesdone*100.0f/(float)inallsamplenum);
   unsigned long timedone=samplesdone/(af_in->freq*af_in->channels); // length of encoded data in seconds
   unsigned long songtime=inallsamplenum/(af_in->freq*af_in->channels);

   fprintf(stderr,"\r%s %12.12s:%3d%% (%dm%2.2ds/%dm%2.2ds) ",
	  (pass)? "Create ":"Analyze",
	  get_shortfilename((pass)? af_out->filename:af_in->filename),
	  percent,
	  timedone/60,timedone%60,songtime/60,songtime%60);
   {
    unsigned long elapsedtimeh=curtimeh-begintimeh;               // elapsed time in hundredths of seconds
    float speed=(float)elapsedtimeh/100.0f/(float)samplesdone;    // speed=seconds/samples
    unsigned long estimated=(unsigned long)((float)(inallsamplenum)*speed); // estimated time
    unsigned long elapsed=elapsedtimeh/100;
    fprintf(stderr,"[runtime: %dm%2.2ds/%dm%2.2ds] ",
	  elapsed/60,elapsed%60,estimated/60,estimated%60);
   }

   if(!pass){
    if(af_in->filemode&CF_ANALYZE)
     fprintf(stderr,"Peek:-%2.1fdB ",acv_todB(cv_normdb));
   }else{
    if(af_out->filemode&CF_NORMALIZE)
     fprintf(stderr,"Gain:+%2.1fdB ",acv_todB(cv_normdb));
    else
     if(af_in->filemode&CF_ANALYZE){
      fprintf(stderr,"           ");
      af_in->filemode&=~CF_ANALYZE;
     }
   }

   lasttimeh=curtimeh;
  }
 }
}

static void display_finish(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 if(!dispquiet){
  if(af_out->filetype){
   long outtimesec=af_out->allsamplenum/(af_out->channels*af_out->freq);
   fprintf(stderr,"\nDone ... Length : %dm%2.2ds",outtimesec/60,outtimesec%60);
   if(af_out->filetype==FT_OGG)
    fprintf(stderr," , Bitrate : %d bit/s",af_out->bitrate);
  }
  fprintf(stderr,"\n");
 }
}

static unsigned long get_timeh(void)
{
 return ((unsigned long)clock()*100/CLOCKS_PER_SEC);
}

static unsigned int check_esc(void)
{
 if(kbhit())
  if(getch()==KEY_ESC)
   return 1;
 return 0;
}

//--------------------------------------------------------------------------
static unsigned int get_extension_filetypenum(char *filename)
{
 char *ext;
 unsigned int i;

 if((ext=strrchr(filename,'.'))==NULL)
  return 0;
 i=0;
 while(filetypes[i].extension!=NULL){
  if(acv_strcmp(filetypes[i].extension,ext)==0)
   return i;
  i++;
 }
 return 0;
}

static char *get_shortfilename(char *filename)
{
 char *p=strrchr(filename,'\\');
 if(p){
  p++;
  return p;
 }
 return filename;
}

static void acv_change_extension_by_newext(char *filename,char *newext)
{
 char *ext;
 ext=strrchr(filename,'.');
 if(ext)
  *ext=0;
 strcat(filename,newext);
}

void acv_change_extension_by_typenum(char *filename,unsigned int newfiletypenum)
{
 acv_change_extension_by_newext(filename,filetypes[newfiletypenum].extension);
}

//-------------------------------------------------------------------------
static void acv_rename_extension(char *filename,char *newext)
{
 char oldname[300];
 strcpy(oldname,filename);
 acv_change_extension_by_newext(filename,newext);
 if(acv_check_and_del_file(filename)==0)
  if(rename(oldname,filename)==0){
   //fprintf(stderr,"Rename: %s -> %s\n",get_shortfilename(oldname),get_shortfilename(filename));
   return;
  }
 fprintf(stderr,"Unsuccessfull rename: %s -> %s\n",get_shortfilename(oldname),get_shortfilename(filename));
}

int acv_check_and_del_file(char *filename)
{
 FILE *fp;
 fp=fopen(filename,"rb");
 if(fp==NULL)
  return 0;
 fclose(fp);

 if(opt_functions&FUNC_OVERWRITE) // we delete the file
  if(remove(filename)==0)
   return 0;

 return ACV_ERROR_FILE_ALREADYEXISTS;
}

long acv_filelen(FILE *fp)
{
 long len;
#ifdef __WATCOMC__
 len=filelength(fp->_handle);
#else
 #if defined(__DJGPP__) || defined(_MSC_VER)
  len=filelength(fp->_file);
 #else
  long curpos=ftell(fp);
  fseek(fp,0,SEEK_END);
  len=ftell(fp);
  fseek(fp,curpos,SEEK_SET);
 #endif
#endif
 return len;
}

static int acv_strcmp(char *str1,char *str2)
{
 char c1,c2;
 if(str1==NULL || str1[0]==0)
  if(str2!=NULL && str2[0]!=0)
   return -1;
  else
   return 0;
 if(str2==NULL || str2[0]==0)
  if(str1!=NULL && str1[0]!=0)
   return 1;
  else
   return 0;
 while(str1[0]!=0 && str2[0]!=0){
  c1=str1[0];
  c2=str2[0];
  if(c1>96 && c1<123)
   c1-=32;
  if(c2>96 && c2<123)
   c2-=32;
  if(c1!=c2)
   if(c1<c2)
    return -1;
   else
    return 1;
  str1++;str2++;
 }
 if(str1[0]==0){
  if(str2[0]!=0)
   return -1;
 }else{
  if(str2[0]==0)
   return 1;
 }
 return 0;
}

#if defined(__WATCOMC__)
int matherr(struct _exception *a)
{
 a->retval=1.0;
 return 1;
}
#endif
