//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function:WAV file reading

#include "in_file.h"

#ifdef MPXPLAY_LINK_INFILE_WAV

#define WAVID_RIFF   0x46464952 // RIFF (FIRR)
#define WAVID_WAVE   0x45564157 // WAVE (EVAW)
#define WAVID_fmt    0x20746d66 // fmt  ( tmf)
#define WAVID_data   0x61746164 // data (atad)
#define WAVID_fact   0x74636166 // fact (tcaf)

typedef struct wav_decoder_data{
 unsigned long filedatabegin;
 unsigned long filedatalen;
 unsigned int  bytespersample;
}wav_decoder_data;

static void wav_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);
static unsigned int  wav_chunk_search(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,unsigned long);
static unsigned long read_le32_fromfile(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds);

static void *wav_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 struct wav_decoder_data *wavi=NULL;
 unsigned long indatalen,maxdatalen;

 struct RIFF{
  unsigned long riffID;
  unsigned long rLen;
 }riff;
 struct WAVE {
  unsigned long waveID;
 }wave;
 struct FORMAT {
  unsigned long fLen;
  unsigned short wTag;
  unsigned short wChannel;
  unsigned long nSample;
  unsigned long nByte;
  unsigned short align;
  unsigned short sample;
 }fmt;

 if(!fbfs->fopen_read(fbds,filename,0))
  goto err_out_opn;

 if(fbfs->fread(fbds,(void *)(&riff),sizeof(struct RIFF))!=sizeof(struct RIFF))
  goto err_out_opn;
 if(riff.riffID!=WAVID_RIFF)
  goto err_out_opn;

 if(fbfs->fread(fbds,(void *)(&wave),sizeof(struct WAVE))!=sizeof(struct WAVE))
  goto err_out_opn;
 if(wave.waveID!=WAVID_WAVE)
  goto err_out_opn;

 miis->filesize=fbfs->filelength(fbds);
 if(!wav_chunk_search(fbfs,fbds,WAVID_fmt))  // search for 'fmt' chunk
  goto err_out_opn;
 if(fbfs->fread(fbds,(void *)(&fmt),sizeof(struct FORMAT))!=sizeof(struct FORMAT)) // read 'fmt' chunk
  goto err_out_opn;
 if(!fmt.nSample)   // freq==0?
  goto err_out_opn;
 if(!fmt.wChannel)  // channels==0 ?
  goto err_out_opn;
 if(!fmt.align)
  goto err_out_opn;
 if(!fmt.sample || (fmt.sample>PCM_MAX_BITS)) //bits
  goto err_out_opn;
 if(fmt.fLen<16)    // too short fmt chunk
  goto err_out_opn;
 switch(fmt.wTag){
  case 0x01:adi->bits=fmt.sample;
	    break;
  case 0x03:adi->infobits|=ADI_FLAG_FLOATOUT;
            adi->bits=1;
            adi->bitratetext="32-float";
	    break;
  default:goto err_out_opn;
 }
 adi->wave_id=fmt.wTag;

 if(read_le32_fromfile(fbfs,fbds)!=WAVID_data){ // first (and fast) search for 'data' chunk
  if(fbfs->fseek(fbds,(long)fmt.fLen-20,SEEK_CUR)<0)
   goto err_out_opn;
  if(!wav_chunk_search(fbfs,fbds,WAVID_data))    // (slow and valid) search for 'data' chunk
   goto err_out_opn;
 }

 wavi=calloc(1,sizeof(struct wav_decoder_data));
 if(!wavi)
  goto err_out_opn;

 indatalen=read_le32_fromfile(fbfs,fbds);

 wavi->filedatabegin=fbfs->ftell(fbds);
 maxdatalen=miis->filesize-wavi->filedatabegin; // maxdatalen=filesize-headersize
 if(!indatalen || (indatalen>maxdatalen))  // indatalen have to be less than maxdatalen
  indatalen=maxdatalen;
 wavi->filedatalen=indatalen;
 adi->pcmdatalen=indatalen/(unsigned long)fmt.align;
 if(!adi->pcmdatalen)              // we need 1 sample at least
  goto err_out_opn;
 adi->freq    =fmt.nSample;                       // frequency;
 adi->filechannels=adi->outchannels=fmt.wChannel; // number of channels

 wavi->bytespersample=fmt.align/fmt.wChannel;

 adi->longname="WAV-file";

 return wavi;

err_out_opn:
 wav_infile_close(fbfs,fbds,wavi,miis);
 return NULL;
}

static unsigned int wav_chunk_search(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,unsigned long search_id)
{
 long jump;
 unsigned int retcode=0,counter=10;

 do{
  if(read_le32_fromfile(fbfs,fbds)==search_id){
   retcode=1;
   break;
  }
  if(fbfs->eof(fbds))
   break;
  jump=read_le32_fromfile(fbfs,fbds);
  if(jump!=WAVID_fact)
   if(fbfs->fseek(fbds,jump,SEEK_CUR)<0)
    break;
 }while(counter--);
 return retcode;
}

static unsigned long read_le32_fromfile(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned long x;
 if(fbfs->fread(fbds,(void *)(&x),4)!=4)
  return 0;
 return x;
}

static void wav_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct wav_decoder_data *wavi=infile_data;
 if(wavi)
  free(wavi);
 fbfs->fclose(fbds);
}

static int wav_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct wav_decoder_data *wavi=infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 long filepos=fbfs->ftell(fbds);
 long fileend=wavi->filedatabegin+wavi->filedatalen;
 long framesize=adi->pcmsamplenum_frame*adi->filechannels*wavi->bytespersample;
 long readsize=framesize;

 if((filepos+readsize)>fileend)
  readsize=fileend-filepos;

 if(readsize<1){
  //fbfs->fread(fbds,adi->pcm_outdatabuf,framesize); // !!! to flush out filebuf
  return MPXPLAY_ERROR_INFILE_EOF;
 }else{
  adi->pcmoutsamplenum=fbfs->fread(fbds,adi->pcm_outdatabuf,readsize)/wavi->bytespersample;
  if(!adi->pcmoutsamplenum)
   return MPXPLAY_ERROR_INFILE_NODATA;
 }

 return MPXPLAY_ERROR_INFILE_OK;
}

static long wav_fseek(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,long newframenum)
{
 struct wav_decoder_data *wavi=infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 long newfilepos=newframenum*adi->pcmsamplenum_frame*adi->filechannels*wavi->bytespersample;
 if(newfilepos>=wavi->filedatalen)
  return MPXPLAY_ERROR_INFILE_EOF;
 newfilepos+=wavi->filedatabegin;
 if(fbfs->fseek(fbds,newfilepos,SEEK_SET)<0)
  return MPXPLAY_ERROR_INFILE_EOF;
 return newframenum;
}

struct mpxplay_infile_func_s IN_WAV_funcs={
 0,
 NULL,
 NULL,
 &wav_infile_open,
 &wav_infile_open,
 &wav_infile_open,
 &wav_infile_close,
 &wav_infile_decode,
 &wav_fseek,
 NULL,
 NULL,
 NULL,
 NULL,
 {"WAV",NULL}
};

#endif //MPXPLAY_LINK_INFILE_WAV
