//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: MP4->AAC file handling
//requires the dec_mp4\mp4ff.lib and mp4ff.h files

#include "in_file.h"

#ifdef MPXPLAY_LINK_INFILE_MP4

#define MP4_AUDIO_BSBUF_SIZE 32768 // have to be enough

#include "dec_aac\faad.h"
#include "dec_mp4\mp4ff.h"

#define AAC_SPECTRUM_ANALISER 1

typedef struct mp4_decoder_data{
 mp4ff_t *infile;
 int  track;
 unsigned int audiotype;
 long numSamples;
 long sampleId;

 unsigned char *bsbuffer;
 unsigned long bsbuf_size;

 faacDecHandle hDecoder;
 faacDecFrameInfo *frameInfo;

 unsigned long pcmsamples_per_aacframe;
 MPXPLAY_PCMOUT_FLOAT_T *pcmoutsave_buffer;
 unsigned int pcmoutsave_samplenum;

 mp4ff_callback_t mp4cb;
}mp4_decoder_data;

struct mpxplay_infile_func_s IN_MP4_funcs;

static void mp4_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);
static int mp4_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);
static int mp4_infile_demux_bs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);
static int mp4_get_audiotrack(mp4ff_t *infile);
static faacDecHandle faaddec_init_dsi(unsigned char *bsbuffer,unsigned int bsbytes,faacDecFrameInfo *hInfo);
static faacDecHandle faaddec_init_frame(unsigned char *bsbuffer,unsigned int bsbytes,faacDecFrameInfo *hInfo,int bitstream_alloc_only);
static unsigned int MakeAdtsHeader(struct faacDecFrameInfo *frameInfo,unsigned char *data,unsigned int framesize);

static void *mp4_infile_check(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct mp4_decoder_data *mp4i=NULL;
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;

 if(!fbfs->fopen_read(fbds,filename,8192))
  goto err_out_check;

 miis->filesize=fbfs->filelength(fbds);
 if(miis->filesize<16)
  goto err_out_check;

 mp4i=(struct mp4_decoder_data *)calloc(1,sizeof(struct mp4_decoder_data));
 if(!mp4i)
  goto err_out_check;

 mp4i->mp4cb.read=fbfs->fread;
 //mp4i->mp4cb.write=fbfs->fwrite;
 mp4i->mp4cb.seek=fbfs->fseek;
 mp4i->mp4cb.user_data=fbds;

 mp4i->infile=mp4ff_open_read(&mp4i->mp4cb);
 if(!mp4i->infile)
  goto err_out_check;

 mp4i->track=mp4_get_audiotrack(mp4i->infile);
 if(mp4i->track<0)
  goto err_out_check;

 mp4i->audiotype=mp4ff_get_audio_type(mp4i->infile,mp4i->track);

 mp4i->numSamples=mp4ff_num_samples(mp4i->infile,mp4i->track);
 if(!mp4i->numSamples)
  goto err_out_check;

 adi->freq=mp4ff_get_sample_rate(mp4i->infile,mp4i->track);
 if(!adi->freq)
  goto err_out_check;
 adi->filechannels=adi->outchannels=mp4ff_get_channel_count(mp4i->infile,mp4i->track);
 if(!adi->filechannels)
  goto err_out_check;
 adi->pcmdatalen=mp4ff_get_track_duration(mp4i->infile,mp4i->track);
 if(!adi->pcmdatalen)
  goto err_out_check;
 adi->bitrate=mp4ff_get_avg_bitrate(mp4i->infile,mp4i->track)/1000;

 adi->bits=16;        // !!! scalebits in AAC decoder
 adi->wave_id=0x2500; // ???
 adi->shortname="AAC";
 adi->longname="MP4->AAC";

 if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT){
  miis->allframes=mp4i->numSamples;
  adi->infobits|=ADI_FLAG_BITSTREAMOUT|ADI_FLAG_BITSTREAMNOFRH;
 }else
  adi->infobits|=ADI_FLAG_FLOATOUT;

#ifdef AAC_SPECTRUM_ANALISER
 adi->infobits|=ADI_FLAG_OWN_SPECTANAL;
#endif

 return mp4i;

err_out_check:
 mp4_infile_close(fbfs,fbds,mp4i,miis);
 return NULL;
}

static void *mp4_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 struct mp4_decoder_data *mp4i;
 unsigned int retry;
 int bsbytes;
 //char sout[100];

 mp4i=mp4_infile_check(fbfs,fbds,filename,miis);
 if(!mp4i)
  return NULL;

 mp4i->frameInfo=(faacDecFrameInfo *)calloc(1,sizeof(faacDecFrameInfo));
 if(!mp4i->frameInfo)
  goto err_out_open;

 mp4i->bsbuf_size=mp4ff_read_sample_getsize(mp4i->infile,mp4i->track,0);
 if(mp4i->bsbuf_size<MP4_AUDIO_BSBUF_SIZE)
  mp4i->bsbuf_size=MP4_AUDIO_BSBUF_SIZE;

 mp4i->bsbuffer=(unsigned char *)malloc(mp4i->bsbuf_size);
 if(!mp4i->bsbuffer)
  goto err_out_open;

 mp4i->frameInfo->channels  =adi->filechannels;
 mp4i->frameInfo->samplerate=adi->freq;

 bsbytes=mp4ff_get_decoder_config_v2(mp4i->infile,mp4i->track,mp4i->bsbuffer,mp4i->bsbuf_size);

 if(bsbytes>0){
  mp4i->hDecoder=faaddec_init_dsi(mp4i->bsbuffer,bsbytes,mp4i->frameInfo);
  if(!mp4i->hDecoder)
   goto err_out_open;

  if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT){
   faacDecClose(mp4i->hDecoder);
   mp4i->hDecoder=NULL;
  }
 }else{
  bsbytes=mp4ff_read_sample_v2(mp4i->infile,mp4i->track,0,mp4i->bsbuffer);
  if(bsbytes<1)
   goto err_out_open;
  mp4i->hDecoder=faaddec_init_frame(mp4i->bsbuffer,bsbytes,mp4i->frameInfo,1);
  if(!mp4i->hDecoder)
   goto err_out_open;

  retry=0;
  do{
   if(bsbytes)
    faacDecReadframe(mp4i->hDecoder,mp4i->frameInfo,mp4i->bsbuffer,bsbytes);
   //sprintf(sout,"re:%d bs:%d o:%d s:%d c:%d f:%d bc:%d %2.2X%2.2X%2.2X%2.2X",retry,bsbytes,mp4i->frameInfo->object_type,mp4i->frameInfo->sbr_present_flag,mp4i->frameInfo->channels,mp4i->frameInfo->samplerate,mp4i->frameInfo->bytesconsumed,mp4i->bsbuffer[0],mp4i->bsbuffer[1],mp4i->bsbuffer[2],mp4i->bsbuffer[3]);
   //display_message(1,0,sout);
   //getch();
   if((mp4i->frameInfo->bytesconsumed>0) && (mp4i->frameInfo->error==0) && mp4i->frameInfo->sbr_present_flag && mp4i->frameInfo->samplerate && mp4i->frameInfo->channels)
    break;
   if(mp4ff_read_sample_getsize(mp4i->infile,mp4i->track,retry)<mp4i->bsbuf_size){
    bsbytes=mp4ff_read_sample_v2(mp4i->infile,mp4i->track,retry,mp4i->bsbuffer);
    if(bsbytes<1)
     goto err_out_open;
   }else
    bsbytes=0;

  }while(++retry<5);

  if((mp4i->frameInfo->bytesconsumed<1) || (mp4i->frameInfo->error) || !mp4i->frameInfo->samplerate || !mp4i->frameInfo->channels)
   goto err_out_open;

  //sprintf(sout,"o:%d s:%d c:%d f:%d b:%d",mp4i->frameInfo->object_type,mp4i->frameInfo->sbr_present_flag,mp4i->frameInfo->channels,mp4i->frameInfo->samplerate,mp4i->frameInfo->bytesconsumed);
  //display_message(1,0,sout);
  //getch();

  faacDecClose(mp4i->hDecoder);
  mp4i->hDecoder=NULL;

  if(!(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT)){
   bsbytes=mp4ff_read_sample_v2(mp4i->infile,mp4i->track,0,mp4i->bsbuffer);
   if(bsbytes<1)
    goto err_out_open;
   mp4i->hDecoder=faaddec_init_frame(mp4i->bsbuffer,bsbytes,mp4i->frameInfo,0);
   if(!mp4i->hDecoder)
    goto err_out_open;
  }
 }

 if(adi->freq!=mp4i->frameInfo->samplerate){ // HE-AAC upsampling
  adi->pcmdatalen=mp4i->numSamples*mp4i->frameInfo->frameLength;
  adi->freq=mp4i->frameInfo->samplerate;
 }

 mp4i->pcmsamples_per_aacframe=mp4i->frameInfo->frameLength;

 if(adi->outchannels>2)  // !!! at AAC
  adi->outchannels=2;    //

 if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT){
  IN_MP4_funcs.decode=mp4_infile_demux_bs;
 }else{
  mp4i->pcmoutsave_buffer=malloc(2*mp4i->pcmsamples_per_aacframe*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
  if(!mp4i->pcmoutsave_buffer)
   goto err_out_open;
  IN_MP4_funcs.decode=mp4_infile_decode;
 }

 return mp4i;

err_out_open:
 mp4_infile_close(fbfs,fbds,mp4i,miis);
 return NULL;
}

static void mp4_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct mp4_decoder_data *mp4i=(mp4_decoder_data *)infile_data;
 if(mp4i){
  if(mp4i->infile)
   mp4ff_close(mp4i->infile);
  if(mp4i->bsbuffer)
   free(mp4i->bsbuffer);

  if(mp4i->hDecoder)
   faacDecClose(mp4i->hDecoder);
  if(mp4i->frameInfo)
   free(mp4i->frameInfo);

  if(mp4i->pcmoutsave_buffer)
   free(mp4i->pcmoutsave_buffer);
  free(mp4i);
 }
 fbfs->fclose(fbds);
}

static int mp4_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 struct mp4_decoder_data *mp4i=(mp4_decoder_data *)infile_data;
 unsigned int samplenum_left;
 long bsbytes;
 void *sample_buffer;
 MPXPLAY_PCMOUT_FLOAT_T *pcmout;
 faacDecFrameInfo frameInfo;

 if(!mp4i)
  return MPXPLAY_ERROR_INFILE_EOF;

 mp4i->mp4cb.user_data=fbds; // we allways update it here

 samplenum_left=adi->pcmsamplenum_frame*adi->outchannels;
 pcmout=(MPXPLAY_PCMOUT_FLOAT_T *)adi->pcm_outdatabuf;

 do{
  if(mp4i->pcmoutsave_samplenum){
   unsigned int samplenum_currblock=min(mp4i->pcmoutsave_samplenum,samplenum_left);
   pds_memcpy(pcmout,mp4i->pcmoutsave_buffer,samplenum_currblock*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
   pcmout+=samplenum_currblock;
   mp4i->pcmoutsave_samplenum-=samplenum_currblock;
   samplenum_left-=samplenum_currblock;
   adi->pcmoutsamplenum+=samplenum_currblock;
   if(mp4i->pcmoutsave_samplenum)
    pds_memcpy(mp4i->pcmoutsave_buffer,mp4i->pcmoutsave_buffer+samplenum_currblock,mp4i->pcmoutsave_samplenum*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
  }
  if(!samplenum_left)
   break;

  if(mp4i->sampleId>=mp4i->numSamples)
   return MPXPLAY_ERROR_INFILE_EOF;

  bsbytes=mp4ff_read_sample_getsize(mp4i->infile,mp4i->track,mp4i->sampleId);
  if((bsbytes<1) || (bsbytes>mp4i->bsbuf_size)){ // bad frame
   mp4i->sampleId++;    // seek to next frame
   continue;
  }

  bsbytes=mp4ff_set_sample_position(mp4i->infile,mp4i->track,mp4i->sampleId);
  if(bsbytes<0){        // failed seek
   if(bsbytes==MPXPLAY_ERROR_MPXINBUF_SEEK_EOF){ // seekpoint is out of file
    mp4i->sampleId++;   // seek to next frame
    continue;
   }                                   // else seekpoint is out of mpxbuffer
   return MPXPLAY_ERROR_INFILE_RESYNC; //
  }

  bsbytes = mp4ff_read_sample_v2(mp4i->infile,mp4i->track,mp4i->sampleId,mp4i->bsbuffer);
  if(bsbytes<1)
   return MPXPLAY_ERROR_INFILE_NODATA;

  sample_buffer = faacDecDecode(mp4i->hDecoder,&frameInfo,mp4i->bsbuffer,bsbytes);
  if(sample_buffer && (frameInfo.error==0) && frameInfo.bytesconsumed && (frameInfo.samples>=adi->outchannels)){
   pds_memcpy(mp4i->pcmoutsave_buffer+mp4i->pcmoutsave_samplenum,sample_buffer,frameInfo.samples*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
   mp4i->pcmoutsave_samplenum+=frameInfo.samples;
  }

  mp4i->sampleId++;
 }while(1);

 return MPXPLAY_ERROR_INFILE_OK;
}

static int mp4_infile_demux_bs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 struct mp4_decoder_data *mp4i=(mp4_decoder_data *)infile_data;
 long bsbytes;

 if(!mp4i)
  return MPXPLAY_ERROR_INFILE_EOF;

 mp4i->mp4cb.user_data=fbds; // we allways update it here

 if(mp4i->sampleId>=mp4i->numSamples)
  return MPXPLAY_ERROR_INFILE_EOF;

 bsbytes=mp4ff_read_sample_getsize(mp4i->infile,mp4i->track,mp4i->sampleId);
 if((bsbytes<1) || (bsbytes>mp4i->bsbuf_size)){
  mp4i->sampleId++;
  return MPXPLAY_ERROR_INFILE_RESYNC;
 }

 bsbytes=mp4ff_set_sample_position(mp4i->infile,mp4i->track,mp4i->sampleId);
 if(bsbytes<0){        // failed seek
  if(bsbytes==MPXPLAY_ERROR_MPXINBUF_SEEK_EOF){ // seekpoint is out of file
   mp4i->sampleId++;   // seek to next frame
  }                                   // else seekpoint is out of mpxbuffer
  return MPXPLAY_ERROR_INFILE_RESYNC; //
 }

 bsbytes = mp4ff_read_sample_v2(mp4i->infile,mp4i->track,mp4i->sampleId,mp4i->bsbuffer);
 if(bsbytes<1)
  return MPXPLAY_ERROR_INFILE_NODATA;

 if((adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT) && !(adi->infobits&ADI_CNTRLBIT_BITSTREAMNOFRH))
  adi->pcmoutsamplenum=MakeAdtsHeader(mp4i->frameInfo,adi->pcm_outdatabuf,bsbytes);
 pds_memcpy(adi->pcm_outdatabuf+adi->pcmoutsamplenum,mp4i->bsbuffer,bsbytes);
 adi->pcmoutsamplenum+=bsbytes;

 mp4i->sampleId++;

 return MPXPLAY_ERROR_INFILE_OK;
}

static long mp4_fseek(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,long newmpxframenum)
{
 struct mp4_decoder_data *mp4i=(mp4_decoder_data *)infile_data;

 if(mp4i){
  mp4i->mp4cb.user_data=fbds; // we allways update it here
  pds_ftoi((float)mp4i->numSamples*(float)newmpxframenum/(float)miis->allframes,&mp4i->sampleId);
  if(mp4i->sampleId<mp4i->numSamples){
   if(mp4ff_set_sample_position(mp4i->infile,mp4i->track,mp4i->sampleId)>=0)
    return newmpxframenum;
  }
 }

 return MPXPLAY_ERROR_INFILE_EOF;
}

static void mp4_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,int cleartype)
{
 struct mp4_decoder_data *mp4i=(mp4_decoder_data *)infile_data;

 if(mp4i){
  if(mp4i->hDecoder)
   faacDecPostSeekReset(mp4i->hDecoder,mp4i->sampleId);
  if(cleartype&(MPX_SEEKTYPE_BOF|MPX_SEEKTYPE_PAUSE)){
#ifdef AAC_SPECTRUM_ANALISER
   aac_analiser_clear();
#endif
   mp4i->pcmoutsave_samplenum=0;
  }
 }
}

#define MP4_COMMENT_TYPES 8

static char *mp4_tag_get(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 static unsigned int mp4atoms[MP4_COMMENT_TYPES]={ATOM_TITLE,ATOM_ARTIST,ATOM_ALBUM,ATOM_DATE,ATOM_COMMENT,ATOM_GENRE2,ATOM_GENRE1,ATOM_TRACK};
 static unsigned int id3index[MP4_COMMENT_TYPES]={I3I_TITLE ,I3I_ARTIST ,I3I_ALBUM ,I3I_YEAR ,I3I_COMMENT ,I3I_GENRE  ,I3I_GENRE,I3I_TRACKNUM};
 struct mp4_decoder_data *mp4i=(mp4_decoder_data *)infile_data;

 if(mp4i){
  unsigned int i;
  for(i=0;i<MP4_COMMENT_TYPES;i++){
   if(!id3ip[id3index[i]]){ // genre1 OR genre2
    char *tag=mp4ff_meta_search_by_atom(mp4i->infile,mp4atoms[i]);
    if(tag){
     unsigned int len;
     len=pds_strcpy(id3p,tag);
     if((*(mpxplay_textconv_funcs->control))&ID3TEXTCONV_UTF_AUTO)
      len=mpxplay_textconv_funcs->utf8_to_char(id3p,len);
     len=mpxplay_textconv_funcs->all_to_char(id3p,len,ID3TEXTCONV_UTF8);
     if(len){
      id3ip[id3index[i]]=id3p;
      id3p+=len+1;
     }
    }
   }
  }
 }

 return id3p;
}

static int mp4_get_audiotrack(mp4ff_t *infile)
{
 int i,numTracks = mp4ff_total_tracks(infile),audiotrack=-1;

 for(i=0; i<numTracks; i++){
  if(mp4ff_get_track_type(infile,i)==TRACK_AUDIO){
   unsigned int at=mp4ff_get_audio_type(infile,i);
   audiotrack=i;
   //fprintf(stdout,"n:%d i:%d a:%2.2X\n",numTracks,i,at);
   if(MP4_IS_AAC_AUDIO_TYPE(at))
    return i;
  }
 }

 return audiotrack;
}

//--------------------------------------------------------------------
// init the decoder using a DecoderSpecificInfo
static faacDecHandle faaddec_init_dsi(unsigned char *bsbuffer,unsigned int bsbytes,faacDecFrameInfo *hInfo)
{
 faacDecHandle hDecoder;
 faacDecConfigurationPtr config;

 hDecoder=faacDecOpen();
 if(!hDecoder)
  return hDecoder;

 config = faacDecGetCurrentConfiguration(hDecoder);
 config->outputFormat  = FAAD_FMT_FLOAT;
 config->downMatrix    = 1;

 if(faacDecInit_dsi(hDecoder,bsbuffer,bsbytes,hInfo)!=0)
  goto err_out_dsi;

 if(!faacDecInitFields(hDecoder))
  goto err_out_dsi;

 return hDecoder;

err_out_dsi:
 faacDecClose(hDecoder);
 return NULL;
}

//init the decoder using the first frame(s)
static faacDecHandle faaddec_init_frame(unsigned char *bsbuffer,unsigned int bsbytes,faacDecFrameInfo *hInfo,int bitstream_alloc_only)
{
 faacDecHandle hDecoder;
 faacDecConfigurationPtr config;
 faacDecFrameInfo frameInfo;

 hDecoder=faacDecOpen();
 if(!hDecoder)
  return hDecoder;

 config = faacDecGetCurrentConfiguration(hDecoder);
 config->defObjectType    = hInfo->object_type;
 config->defSBRpresentflag= hInfo->sbr_present_flag;
 config->defChannels      = hInfo->channels;
 if(hInfo->samplerate)
  config->defSampleRate   = hInfo->samplerate;

 config->outputFormat    = FAAD_FMT_FLOAT;
 config->downMatrix      = 1;

 if(faacDecInit_frame(hDecoder,bsbuffer,bsbytes,&frameInfo)!=0)
  goto err_out_initfr;

 memcpy(hInfo,(void *)&frameInfo,sizeof(faacDecFrameInfo));

 if(bitstream_alloc_only){
  if(!faacDecInitField_bs(hDecoder))
   goto err_out_initfr;
 }else{
  if(!faacDecInitFields(hDecoder))
   goto err_out_initfr;
 }

 return hDecoder;

err_out_initfr:
 faacDecClose(hDecoder);
 return NULL;
}

//---------------------------------------------------------------------

static unsigned int MakeAdtsHeader(struct faacDecFrameInfo *frameInfo,unsigned char *data,unsigned int framesize)
{
 int profile;

 profile = frameInfo->object_type;
 //if(profile)
 // profile--;
 profile&=3;

 pds_memset(data,0,7);

 framesize+=7;

 data[0]  = 0xFF; // 8b: syncword
 data[1]  = 0xF0; // 4b: syncword
                  // 1b: mpeg id = 0
                  // 2b: layer = 0
 data[1] |= 0x01; // 1b: protection absent

 data[2]  = ((profile << 6) & 0xC0);             // 2b: profile
 data[2] |= ((frameInfo->sf_index << 2) & 0x3C); // 4b: sampling_frequency_index
                                                 // 1b: private = 0
 data[2] |= ((frameInfo->channels >> 2) & 0x1);  // 1b: channel_configuration

 data[3]  = ((frameInfo->channels << 6) & 0xC0); // 2b: channel_configuration
                                       // 1b: original
                                       // 1b: home
                                       // 1b: copyright_id
                                       // 1b: copyright_id_start
 data[3] |= ((framesize >> 11) & 0x3); // 2b: aac_frame_length

 data[4]  = ((framesize >> 3) & 0xFF); // 8b: aac_frame_length

 data[5]  = ((framesize << 5) & 0xE0); // 3b: aac_frame_length
 data[5] |= ((0x7FF >> 6) & 0x1F);     // 5b: adts_buffer_fullness

 data[6]  = ((0x7FF << 2) & 0x3F);     // 6b: adts_buffer_fullness
                                       // 2b: num_raw_data_blocks
 return 7;
}

struct mpxplay_infile_func_s IN_MP4_funcs={
 0,
 NULL,
 NULL,
 &mp4_infile_check,
 &mp4_infile_check,
 &mp4_infile_open,
 &mp4_infile_close,
 &mp4_infile_decode,
 &mp4_fseek,
 &mp4_clearbuffs,
 &mp4_tag_get,
 NULL,
 NULL,
 {"MP4","M4A","M4B",NULL}
};

#endif
