//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: MP2/MP3 file handling - main
//requires the dec_mp3\mp3dec.lib & mp3dec.h files

#include "in_file.h"

#ifdef MPXPLAY_LINK_INFILE_MPX

#include "newfunc\newfunc.h"
#include "dec_mp3\mp3dec.h"
#include "playlist\playlist.h"
#include <string.h>

#define WAVID_RIFF   0x46464952 // RIFF (FIRR)
#define WAVID_WAVE   0x45564157 // WAVE (EVAW)
#define WAVID_fmt    0x20746d66 // fmt  ( tmf)
#define WAVID_data   0x61746164 // data (atad)
#define WAVID_RMP3   0x33504d52 // RMP3 (3PMR)
#define WAVID_fact   0x74636166 // fact (tcaf)

#define HEADID_MPC   0x002b504d // MP+  ( +PM)
#define HEADID_AC3   0x0000770b // 0x0b77
#define HEADID_OggS  0x5367674f // OggS (SggO)
#define HEADID_MAC   0x2043414d // MAC  ( CAM) (APE)
#define HEADID_ASF   0x75b22630 //
#define HEADID_FLAC  0x43614c66 // fLaC (CaLf)

#define HEADID_ID3V1 0x00474154 // TAG  ( GAT)
#define HEADID_ID3V2 0x00334449 // ID3  ( 3DI)

extern unsigned int channelmode;

static void mpx_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);
static void mpx_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,int cleartype);
static unsigned long mpx_check_header(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds);
static unsigned long mpx_riff_skip_header(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds);
static void mpx_check_xingheader(struct mp3_decoder_data *mp3i,unsigned char *headbufp);
static void mpx_assign_values(struct mp3_decoder_data *mp3i,struct mpxplay_infile_info_s *miis,unsigned long head);
static void mpx_id3v2x_skip_header(char *bufp,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds);

static void mpx_preinit(void)
{
 mpxdec_synth_init(65536);
 mpxdec_layer2_init();
 mpxdec_layer3_init();
}

static void *mpx_infile_check(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct mp3_decoder_data *mp3i;
 unsigned long head;

 if(!fbfs->fopen_read(fbds,filename,0))
  return NULL;

 miis->filesize=fbfs->filelength(fbds);
 if(miis->filesize<32)
  return NULL;

 mp3i=(struct mp3_decoder_data *)calloc(1,sizeof(struct mp3_decoder_data));
 if(!mp3i)
  return NULL;

 if(!mpxdec_bitstream_init(mp3i)) // layer2,layer3 too
  goto err_out_chk;

 head=mpx_check_header(mp3i,fbfs,fbds);
 if(!head)
  goto err_out_chk;

 mpx_assign_values(mp3i,miis,head);

 return mp3i;

err_out_chk:
 mpx_infile_close(fbfs,fbds,mp3i,miis);
 return NULL;
}

static void *mpx_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct mp3_decoder_data *mp3i;

 mp3i=mpx_infile_check(fbfs,fbds,filename,miis);
 if(!mp3i)
  return mp3i;

 if(!mpxdec_layer3_datafields_alloc(mp3i)) // allocates non-used datafield at MP2, but who cares with it...
  goto err_out_opn;

 mp3i->synthdata=mpxdec_layer3_synth_alloc(mp3i->hybridp);
 if(!mp3i->synthdata)
  goto err_out_opn;
 mp3i->synthdata->outchannels=mp3i->outchannels;

 return mp3i;

err_out_opn:
 mpx_infile_close(fbfs,fbds,mp3i,miis);
 return NULL;
}

static void mpx_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct mp3_decoder_data *mp3i=infile_data;
 if(mp3i){
  mpxdec_layer3_synth_close(mp3i->synthdata);
  mpxdec_layer3_datafields_close(mp3i);
  mpxdec_bitstream_close(mp3i);
  free(mp3i);
 }
 fbfs->fclose(fbds);
}

//-------------------------------------------------------------------------
static void mpx_resync_init(struct mp3_decoder_data *mp3i,mpxplay_audio_decoder_info_s *adi)
{
 if(mp3i->lay==3){
  mpxdec_layer3_datafields_reset(mp3i);
  if(adi->bitrate<=96)
   mp3i->resync_counter=2;
  else
   if(adi->bitrate<256)
    mp3i->resync_counter=1;
 }
 funcbit_enable(mp3i->infobits,MP3DI_INFOBIT_RESYNC);
}

static int mpx_resync_sub(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,mpxplay_audio_decoder_info_s *adi)
{
 unsigned int retry;

 retry=MPXDEC_FRAME_RESYNC_RETRY;
 do{
  switch(mpxdec_readframe(mp3i,fbfs,fbds,1)){
   case MPXDEC_ERROR_OK    :retry=1;break;  // exit from while() now
   case MPXDEC_ERROR_NODATA:return MPXPLAY_ERROR_INFILE_NODATA;
  }
 }while(--retry);

 if(mp3i->lay==3){
  retry=MPXDEC_FRAME_RESYNC_RETRY;
  do{
   switch(mpxdec_readframe(mp3i,fbfs,fbds,1)){
    case MPXDEC_ERROR_OK:
     if(mpxdec_layer3_decode_part1(mp3i,channelmode)==MPXDEC_ERROR_OK){ // to avoid clicks (caused by bsbuf and mdct_block)
      if(!mp3i->resync_counter)
       return MPXPLAY_ERROR_INFILE_OK;
      mp3i->resync_counter--;
      break;
     }
    case MPXDEC_ERROR_BITSTREAM:
     mpx_resync_init(mp3i,adi);
     break;
    case MPXDEC_ERROR_NODATA:
     return MPXPLAY_ERROR_INFILE_NODATA;
   }
  }while(--retry);
  return MPXPLAY_ERROR_INFILE_SYNCLOST;
 }
 return MPXPLAY_ERROR_INFILE_OK;
}

static int mpx_resync_main(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,mpxplay_audio_decoder_info_s *adi)
{
 int retcode=MPXPLAY_ERROR_INFILE_OK;
 if(mp3i->infobits&MP3DI_INFOBIT_RESYNC){
  retcode=mpx_resync_sub(mp3i,fbfs,fbds,adi);
  if(retcode>=0)
   funcbit_disable(mp3i->infobits,MP3DI_INFOBIT_RESYNC);
 }
 return retcode;
}

static int mpx_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct mp3_decoder_data *mp3i=infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 int retcode;

 if(fbfs->ftell(fbds)>=(mp3i->filedatabegin+mp3i->filedatalen))
  return MPXPLAY_ERROR_INFILE_EOF;

 retcode=mpx_resync_main(mp3i,fbfs,fbds,adi);
 if(retcode<0)
  return retcode;

 switch(mpxdec_readframe(mp3i,fbfs,fbds,1)){
  case MPXDEC_ERROR_OK:
   if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT){
    memcpy(adi->pcm_outdatabuf,mp3i->bsbuf,mp3i->fsize);
    adi->pcmoutsamplenum=mp3i->fsize;
    return MPXPLAY_ERROR_INFILE_OK;
   }
   switch(mp3i->lay){
    case 2:mpxdec_layer2_decode_part1(mp3i);break;
    case 3:if(mpxdec_layer3_decode_part1(mp3i,channelmode)==MPXDEC_ERROR_OK)
            break;
           mpx_resync_init(mp3i,adi);
           retcode=mpx_resync_main(mp3i,fbfs,fbds,adi);
           if(retcode<0)
            return retcode;
   }
   break;
  case MPXDEC_ERROR_BITSTREAM:
   mpx_resync_init(mp3i,adi);
   retcode=mpx_resync_main(mp3i,fbfs,fbds,adi);
   if(retcode<0)
    return retcode;
   break;
  case MPXDEC_ERROR_NODATA:
   return MPXPLAY_ERROR_INFILE_NODATA;
 }

 mpxdec_decode_part2(mp3i->synthdata,adi->pcm_outdatabuf);

 adi->pcmoutsamplenum=mp3i->pcm_samplenum;

 return MPXPLAY_ERROR_INFILE_OK;
}

static void mpx_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,int cleartype)
{
 struct mp3_decoder_data *mp3i=infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;

 if(cleartype&MPX_SEEKTYPE_BOF){
  if(mp3i->lay==3){
   mpxdec_bitstream_reset(mp3i);
   mpxdec_layer3_datafields_reset(mp3i);
  }
 }else
  if(cleartype&MPX_SEEKTYPE_NORM)
   mpx_resync_init(mp3i,adi);

 if(cleartype&(MPX_SEEKTYPE_BOF|MPX_SEEKTYPE_PAUSE))
  mpxdec_layer3_synth_clear(mp3i->synthdata);
}

static long mpx_fseek(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,long newframenum)
{
 struct mp3_decoder_data *mp3i=infile_data;
 long newfilepos=newframenum*mp3i->framesize;

 if(newfilepos>=mp3i->filedatalen)
  return MPXPLAY_ERROR_INFILE_EOF;

 newfilepos+=mp3i->filedatabegin;

 if(fbfs->fseek(fbds,newfilepos,SEEK_SET)<0)
  return MPXPLAY_ERROR_INFILE_EOF;

 return newframenum;
}

//--------------------------------------------------------------------------
#define MPX_HEADREAD_SIZE 256
#define MPX_XINGHEAD_SIZE 64

#define MPX_SYNC_RETRY (98304/MPX_HEADREAD_SIZE) // 98304 bytes long check

static unsigned long mpx_sync_header(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned long head;
 unsigned char *bufp=mp3i->buf_p;
 unsigned int lay,bitrate,freq,retry=MPX_SYNC_RETRY;

 if(!mp3i->buffer_bytes){
  bufp=mp3i->bsbuf;
  if(fbfs->fread(fbds,bufp,4)!=4)
   return 0;
  mp3i->buffer_bytes=4;
 }
 head=((unsigned long)bufp[0]<<24)|((unsigned long)bufp[1]<<16)
     |((unsigned long)bufp[2]<< 8)|((unsigned long)bufp[3]    );
 bufp+=4;
 mp3i->buffer_bytes-=4;

 do{
  lay  =4-((head>>17)&0x3);
  bitrate=((head>>12)&0xf);
  freq   =((head>>10)&0x3);
#ifdef MPXDEC_ENABLE_FREEFORMAT
  if(((head&0xffe00000)==0xffe00000)&&(bitrate<15)&&(freq<=2)&&(lay>=2)&&(lay<=3))
   break;
#else
  if(((head&0xffe00000)==0xffe00000)&&(bitrate)&&(bitrate<15)&&(freq<=2)&&(lay>=2)&&(lay<=3))
   break;
#endif
  if(!mp3i->buffer_bytes){
   bufp=mp3i->bsbuf;
   mp3i->buffer_bytes=fbfs->fread(fbds,bufp,MPX_HEADREAD_SIZE);
   if(!mp3i->buffer_bytes || !(--retry)){
    head=0;
    break;
   }
  }
  head=(head<<8) | (unsigned long)(bufp[0]);
  bufp++;
  mp3i->buffer_bytes--;
 }while(1);

 mp3i->buf_p=bufp;
 return head;
}

static unsigned long mpx_sync_firstframe_normal(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned int retry=MPXDEC_FRAME_RESYNC_RETRY,okframes=0;
 unsigned long head=mp3i->firsthead;

 if(fbfs->fseek(fbds,mp3i->filedatabegin,SEEK_SET)<0)
  return 0;
 mpxdec_bitstream_reset(mp3i);
 mp3i->firsthead&=MPXDEC_HEADMASK_STANDARD;

 do{
  unsigned int error;
  error=mpxdec_readframe(mp3i,fbfs,fbds,0);
  if(error==MPXDEC_ERROR_OK)
   okframes++;
  else{
   if(error==MPXDEC_ERROR_NODATA){
    if(!okframes)
     head=0;
    break;
   }
   if(fbfs->fseek(fbds,mp3i->filedatabegin+1,SEEK_SET)<0)
    return 0;
   mpxdec_bitstream_reset(mp3i);
   head=mpx_sync_header(mp3i,fbfs,fbds);
   if(!head)
    break;
   mp3i->firsthead=head&MPXDEC_HEADMASK_STANDARD;
   mp3i->filedatabegin=fbfs->ftell(fbds)-mp3i->buffer_bytes-4;
   if(fbfs->fseek(fbds,mp3i->filedatabegin,SEEK_SET)<0)
    return 0;
   mpxdec_bitstream_reset(mp3i);
   okframes=0;
  }
 }while((okframes<MPXDEC_FRAME_HEADSYNC_COUNT) && (--retry));

 if(fbfs->fseek(fbds,mp3i->filedatabegin,SEEK_SET)<0)
  return 0;
 mpxdec_bitstream_reset(mp3i);
 return head;
}

#ifdef MPXDEC_ENABLE_FREEFORMAT
static unsigned long mpx_sync_firstframe_freeformat(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned int retry;

 retry=32;
 while((mpx_sync_header(mp3i,fbfs,fbds)&0xfffffc03)!=(mp3i->firsthead&0xfffffc03) && (--retry)){}
 if(!retry)
  return 0;

 mp3i->fsize=fbfs->ftell(fbds)-mp3i->buffer_bytes-mp3i->filedatabegin; // lenght of 0. frame

 if(fbfs->fseek(fbds,mp3i->filedatabegin,SEEK_SET)<0) // rewind
  return 0;
 mp3i->buf_p=mp3i->bsbuf;
 mp3i->buffer_bytes=0;

 if(!mp3i->fsize || (mp3i->fsize>MPXDEC_FRAMESIZE_MAX))
  return 0;

 mp3i->infobits|=MP3DI_INFOBIT_FREEFORMAT;

 return mp3i->firsthead;
}
#endif

static unsigned long mpx_check_header(struct mp3_decoder_data *mp3i,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned long head;

 mp3i->buf_p=mp3i->bsbuf;
 if(fbfs->fread(fbds,mp3i->buf_p,4)!=4)
  return 0;
 mp3i->buffer_bytes=4;

 head=*((unsigned long *)(mp3i->buf_p));

 if((head&0x00ffffff)==HEADID_MPC)
  return 0;
 if((head&0x0000ffff)==HEADID_AC3)
  return 0;
 if(head==HEADID_OggS)
  return 0;
 if(head==HEADID_MAC)
  return 0;
 if(head==HEADID_ASF)
  return 0;
 if(head==HEADID_FLAC)
  return 0;

 if((head&0x00ffffff)==HEADID_ID3V2){
  mpx_id3v2x_skip_header(mp3i->buf_p,fbfs,fbds);
  mp3i->buffer_bytes=0;
  mp3i->infobits|=MP3DI_INFOBIT_ID3V2_PRESENT;
 }

 if(head==WAVID_RIFF){
  mp3i->filedatalen=mpx_riff_skip_header(fbfs,fbds);
  if(!mp3i->filedatalen)
   return 0;
  //mp3i->infobits|=MP3DI_INFOBIT_RIFFWAVEHEAD;
  mp3i->buffer_bytes=0;
 }

 head=mpx_sync_header(mp3i,fbfs,fbds);
 if(!head)
  return 0;
 mp3i->firsthead=head;
 mp3i->filedatabegin=fbfs->ftell(fbds)-mp3i->buffer_bytes-4;

#ifdef MPXDEC_ENABLE_FREEFORMAT
 mp3i->bitrate_index=((head>>12)&0xf);
 if(!mp3i->bitrate_index)
  head=mpx_sync_firstframe_freeformat(mp3i,fbfs,fbds);
 if(mp3i->bitrate_index || !head)
#endif
 head=mpx_sync_firstframe_normal(mp3i,fbfs,fbds);
 if(!head)
  return 0;

 if(mp3i->buffer_bytes<MPX_XINGHEAD_SIZE) // fill up buffer to xing header checking
  if(fbfs->fread(fbds,mp3i->buf_p+mp3i->buffer_bytes,MPX_XINGHEAD_SIZE-mp3i->buffer_bytes)!=(MPX_XINGHEAD_SIZE-mp3i->buffer_bytes))
   return 0;
 mpx_check_xingheader(mp3i,mp3i->buf_p);

 return head;
}

//------------------------------------------------------------------------
static unsigned long read_le32_fromfile(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned long x;
 if(fbfs->fread(fbds,(void *)(&x),4)!=4)
  return 0;
 return x;
}

static unsigned int wav_chunk_search(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,unsigned long search_id)
{
 long jump;
 unsigned int retcode=0,counter=10;

 do{
  if(read_le32_fromfile(fbfs,fbds)==search_id){
   retcode=1;
   break;
  }
  if(fbfs->eof(fbds))
   break;
  jump=read_le32_fromfile(fbfs,fbds);
  if(jump!=WAVID_fact)
   if(fbfs->fseek(fbds,jump,SEEK_CUR)<0)
    break;
 }while(counter--);
 return retcode;
}

static unsigned long mpx_riff_skip_header(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned long waveID,indatalen,maxdatalen;
 struct FORMAT {
  unsigned long fLen;
  unsigned short wTag;
  unsigned short wChannel;
  unsigned long nSample;
  unsigned long nByte;
  unsigned short align;
  unsigned short sample;
 }fmt;

          read_le32_fromfile(fbfs,fbds);                // RIFF len
 waveID = read_le32_fromfile(fbfs,fbds);                // WAVE id

 switch(waveID){
  // RIFF....RMP3data          header
  case WAVID_RMP3:pds_memset((void *)&fmt,0,sizeof(struct FORMAT));
		  break;
  // RIFF....WAVEfmt....data   header
  case WAVID_WAVE:if(!wav_chunk_search(fbfs,fbds,WAVID_fmt))    // search for 'fmt' chunk
		   return 0;
		  if(fbfs->fread(fbds,(void *)(&fmt),sizeof(struct FORMAT))!=sizeof(struct FORMAT)) // read 'fmt' chunk
                   return 0;
		  if((fmt.wTag!=0x0055) && (fmt.wTag!=0x0050)) // this is not an MP3 nor MP2
		   return 0;
		  break;
  default:return 0;
 }
 if(read_le32_fromfile(fbfs,fbds)!=WAVID_data){ // first (and fast) search for 'data' chunk
  if(fmt.fLen<16)
   return 0;
  if(fbfs->fseek(fbds,(long)fmt.fLen-20,SEEK_CUR)<0)
   return 0;
  if(!wav_chunk_search(fbfs,fbds,WAVID_data))    // (slow and correct) search for 'data' chunk
   return 0;
 }

 indatalen=read_le32_fromfile(fbfs,fbds);
 if(fbfs->eof(fbds))
  return 0;
 maxdatalen=fbfs->filelength(fbds)-fbfs->ftell(fbds);

 if(!indatalen || (indatalen>maxdatalen))
  indatalen=maxdatalen;

 return indatalen;
}

//-------------------------------------------------------------------------
#define HEADID_XING      0x676e6958 // 'gniX' ('Xing')
#define XING_FRAMES_FLAG 0x00000001

static void mpx_check_xingheader(struct mp3_decoder_data *mp3i,unsigned char *headbufp)
{
 int xing_flags,h_id,mode,offs;

 h_id=(headbufp[1]>>3)&1;
 mode=(headbufp[3]>>6)&3;

 if(h_id){       // mpeg1
  if(mode!=3)
   offs=(32+4);
  else
   offs=(17+4);
 }else{          // mpeg2
  if(mode!=3)
   offs=(17+4);
  else
   offs=(9+4);
 }
 headbufp+=offs;
 if(pds_getb_le32(headbufp)!=HEADID_XING)
  return;
 headbufp+=4;
 xing_flags=pds_getb_be32(headbufp);
 headbufp+=4;
 if(xing_flags&XING_FRAMES_FLAG){
  mp3i->allframes=pds_getb_be32(headbufp);
  mp3i->infobits|=MP3DI_INFOBIT_VBR;
 }
}

//--------------------------------------------------------------------------
static void mpx_assign_values(struct mp3_decoder_data *mp3i,struct mpxplay_infile_info_s *miis,unsigned long head)
{
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 float timef;

 mp3i->firsthead=head;
 mp3i->lay          =4-((head>>17)&3);
 mp3i->bitrate_index=((head>>12)&0xf);

 switch(mp3i->lay){
  case 2:adi->longname="Layer II";adi->shortname="MP2";adi->wave_id=0x0050;break; // ???
  case 3:adi->longname="LayerIII";adi->shortname="MP3";adi->wave_id=0x0055;break;
 }

 if(head&(1<<20))
  mp3i->lsf=(head&(1<<19))? 0:3;
 else
  mp3i->lsf=6;

 mp3i->frequency_index=((head>>10)&0x3)+mp3i->lsf;
 adi->freq=mpxdec_freqs[mp3i->frequency_index];

 if(mp3i->lsf)
  mp3i->lsf=1;

 mp3i->padding       =((head>>9)&0x1);
 mp3i->mpg_chmode    =((head>>6)&0x3);

 switch(mp3i->mpg_chmode){
  case MPG_MD_DUALCHAN    :adi->channeltext="DualChan";break;
  case MPG_MD_JOINT_STEREO:mp3i->mpg_chmode_ext=((head>>4)&0x3);
			   if(mp3i->mpg_chmode_ext&0x1)      // intensity stereo
			    adi->channeltext="i-Stereo";
			   else
			    if(!mp3i->mpg_chmode_ext || (mp3i->mpg_chmode_ext&0x2)) // ms-stereo
			     adi->channeltext="msStereo";
			   break;
 }

 adi->filechannels=mp3i->filechannels=(mp3i->mpg_chmode==MPG_MD_MONO)? 1:2;
 if(channelmode==CHM_STEREO)
  mp3i->outchannels=mp3i->filechannels;
 else
  mp3i->outchannels=1;
 adi->outchannels=mp3i->outchannels;
 mp3i->pcm_samplenum=(1152>>mp3i->lsf)*mp3i->outchannels;

#ifdef MPXDEC_ENABLE_FREEFORMAT
 if(mp3i->infobits&MP3DI_INFOBIT_FREEFORMAT){
  adi->bitrate=(mp3i->fsize*adi->freq+(576>>mp3i->lsf)*(1000/8))/((1152>>mp3i->lsf)*(1000/8)); // rounding
  mp3i->firsthead&=MPXDEC_HEADMASK_FREEFORMAT;
 }else{
  adi->bitrate=mpxdec_tabsel_123[mp3i->lsf][mp3i->lay-1][mp3i->bitrate_index];
  mp3i->fsize=(unsigned long)adi->bitrate*144000/(adi->freq<<mp3i->lsf)+mp3i->padding;
  mp3i->firsthead&=MPXDEC_HEADMASK_STANDARD;
 }
#else
 adi->bitrate=mpxdec_tabsel_123[mp3i->lsf][mp3i->lay-1][mp3i->bitrate_index];
 mp3i->fsize=(unsigned long)adi->bitrate*144000/(adi->freq<<mp3i->lsf)+mp3i->padding;
 mp3i->firsthead&=MPXDEC_HEADMASK_STANDARD;
#endif

 if(!mp3i->filedatalen)
  mp3i->filedatalen=miis->filesize-mp3i->filedatabegin;

 if(mp3i->infobits&MP3DI_INFOBIT_VBR){
  miis->allframes=mp3i->allframes;
  mp3i->framesize=mp3i->filedatalen/mp3i->allframes;
  timef=(float)(1152>>mp3i->lsf)*(float)mp3i->allframes/(float)adi->freq;
  pds_ftoi((float)(mp3i->filedatalen)/(timef*(1000.0/8.0)),(long *)&adi->bitrate);// VBR average bitrate (ABR)
  if(!adi->bitrate)
   adi->bitrate=1;
 }else{
  mp3i->framesize=mp3i->fsize;
  timef=(float)(mp3i->filedatalen)/(float)(adi->bitrate)/(1000.0/8.0);
  pds_ftoi(timef*(float)adi->freq/(float)(PCM_OUTSAMPLES>>mp3i->lsf),(long *)&mp3i->allframes);
  miis->allframes=mp3i->allframes;
 }
#ifdef MPXDEC_ENABLE_FREEFORMAT
 if(!(mp3i->infobits&MP3DI_INFOBIT_FREEFORMAT))
#endif
  mp3i->fsize=0;  // to sign 0. frame
 mp3i->bsbuf=mp3i->bitstreamp+MPXDEC_FRAMEBUFFER_SIZE;

 pds_ftoi(timef,&miis->timesec);
 pds_ftoi(timef*(float)adi->freq,(long *)&adi->pcmdatalen); // !!!
 adi->bits=16;
 adi->infobits|=ADI_FLAG_OWN_SPECTANAL;
#ifndef MPXDEC_INTEGER_OUTPUT
 adi->infobits|=ADI_FLAG_FLOATOUT|ADI_FLAG_FPUROUND_CHOP;
#endif
 if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT)
  adi->infobits|=ADI_FLAG_BITSTREAMOUT;
}

#endif // MPXPLAY_LINK_INFILE_MPX

//--------------------------------------------------------------------------
//ID3v1

static char *id3v1genres[]={
  "Blues", "Classic Rock", "Country", "Dance", "Disco", "Funk", "Grunge",
  "Hip-Hop", "Jazz", "Metal", "New Age", "Oldies", "Other", "Pop", "R&B",
  "Rap", "Reggae", "Rock", "Techno", "Industrial", "Alternative", "Ska",
  "Death Metal", "Pranks", "Soundtrack", "Euro-Techno", "Ambient",
  "Trip-Hop", "Vocal", "Jazz+Funk", "Fusion", "Trance", "Classical",
  "Instrumental", "Acid", "House", "Game", "Sound Clip", "Gospel", "Noise",
  "AlternRock", "Bass", "Soul", "Punk", "Space", "Meditative",
  "Instrumental Pop", "Instrumental Rock", "Ethnic", "Gothic", "Darkwave",
  "Techno-Industrial", "Electronic", "Pop-Folk", "Eurodance", "Dream",
  "Southern Rock", "Comedy", "Cult", "Gangsta", "Top 40", "Christian Rap",
  "Pop/Funk", "Jungle", "Native American", "Cabaret", "New Wave",
  "Psychadelic", "Rave", "Showtunes", "Trailer", "Lo-Fi", "Tribal",
  "Acid Punk", "Acid Jazz", "Polka", "Retro", "Musical", "Rock & Roll",
  "Hard Rock", "Folk", "Folk/Rock", "National Folk", "Swing", "Fast-Fusion",
  "Bebob", "Latin", "Revival", "Celtic", "Bluegrass", "Avantgarde",
  "Gothic Rock", "Progressive Rock", "Psychedelic Rock", "Symphonic Rock",
  "Slow Rock", "Big Band", "Chorus", "Easy Listening", "Acoustic", "Humour",
  "Speech", "Chanson", "Opera", "Chamber Music", "Sonata", "Symphony",
  "Booty Bass", "Primus", "Porn Groove", "Satire", "Slow Jam", "Club",
  "Tango", "Samba", "Folklore", "Ballad", "Power Ballad", "Rhythmic Soul",
  "Freestyle", "Duet", "Punk Rock", "Drum Solo", "A capella", "Euro-House",
  "Dance Hall", "Goa", "Drum & Bass", "Club House", "Hardcore", "Terror",
  "Indie", "BritPop", "NegerPunk", "Polsk Punk", "Beat", "Christian Gangsta",
  "Heavy Metal", "Black Metal", "Crossover", "Contemporary C",
  "Christian Rock", "Merengue", "Salsa", "Thrash Metal", "Anime", "JPop",
  "SynthPop"
};

#define MAX_ID3GENRENUM (sizeof(id3v1genres)/sizeof(char *))

static unsigned int id3v10_partlens[5]={30,30,30,4,30};

unsigned int mpx_tag_check_id3v1(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 char tag[4];
 if(fbfs->fseek(fbds,-128,SEEK_END)<0)
  return 0;
 if(fbfs->fread(fbds,tag,3)!=3)
  return 0;
 if(tag[0]=='T' && tag[1]=='A' && tag[2]=='G')
  return 1;
 return 0;
}

//allways call check_id3tag_v1 before this
char *mpx_tag_read_id3v1(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 unsigned int i,partlen,reallen;
 char *readp,readtmp[128];

 readp=&readtmp[0];
 if(fbfs->fread(fbds,readp,128-3)!=(128-3))
  return id3p;

 for(i=0;i<5;i++){
  partlen=id3v10_partlens[i];
  if(!id3ip[i]){ // maybe we've got it from ID3v2
   pds_strncpy(id3p,readp,partlen);
   if(i==I3I_COMMENT){
    if(id3p[partlen-1]<32) // tracknumber check
     id3p[partlen-1]=0;
   }
   id3p[partlen]=0;
   reallen=pds_strlenc(id3p,32);
   if(reallen){
    id3ip[i]=id3p;
    id3p[reallen]=0;
    reallen=mpxplay_textconv_funcs->all_to_char(id3p,reallen,0);
    id3p+=reallen+1;
   }
  }
  readp+=partlen;
 }
 //tracknumber from id3v1.1
 if(!id3ip[I3I_TRACKNUM]){  // maybe we've got it from ID3v2
  i=(unsigned int)readp[-1];
  if(i && !readp[-2]){
   id3ip[I3I_TRACKNUM]=id3p;
   id3p+=sprintf(id3p,"%d",i)+1;
  }
 }
 //genre
 if(!id3ip[I3I_GENRE]){     // maybe we've got it from ID3v2
  i=(unsigned int)readp[0];
  if(i<MAX_ID3GENRENUM)
   id3ip[I3I_GENRE]=id3v1genres[i];
 }
 return id3p;
}

char *mpx_tag_id3v1_index_to_genre(unsigned int i)
{
 if(i<MAX_ID3GENRENUM)
  return id3v1genres[i];
 return NULL;
}

static unsigned int tag_id3v1_get_genrenum(char *genrename)
{
 unsigned int i;

 if(genrename)
  for(i=0;i<MAX_ID3GENRENUM;i++)
   if(stricmp(id3v1genres[i],genrename)==0)
    return i;

 return 255;
}

int mpx_tag_write_id3v1(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char **id3ip)
{
 unsigned int error=MPXPLAY_ERROR_INFILE_WRITETAG_UNKNOWN;
 long fileoffset;
 char strtemp[192];

 if(mpx_tag_check_id3v1(fbfs,fbds))        // is file tagged already?
  fileoffset=fbfs->fseek(fbds,-128,SEEK_END);
 else
  fileoffset=fbfs->fseek(fbds,   0,SEEK_END);
 if(fileoffset>0){                // successfull file positioning
  if(id3ip[I3I_TRACKNUM] && id3ip[I3I_TRACKNUM][0]){       // ID3v1.1
   unsigned int tracknum=pds_atol(id3ip[I3I_TRACKNUM]);
   if(tracknum>255)
    tracknum=255;
   sprintf(strtemp,"TAG%-30.30s%-30.30s%-30.30s%-4.4s%-28.28s%c%c%c",
                   (id3ip[I3I_TITLE])?  id3ip[I3I_TITLE]:"",
                   (id3ip[I3I_ARTIST])? id3ip[I3I_ARTIST]:"",
                   (id3ip[I3I_ALBUM])?  id3ip[I3I_ALBUM]:"",
                   (id3ip[I3I_YEAR])?   id3ip[I3I_YEAR]:"",
                   (id3ip[I3I_COMMENT])?id3ip[I3I_COMMENT]:"",0,
                   tracknum,
                   tag_id3v1_get_genrenum(id3ip[I3I_GENRE]));
  }else{                                                   // ID3v1.0
   sprintf(strtemp,"TAG%-30.30s%-30.30s%-30.30s%-4.4s%-30.30s%c",
                   (id3ip[I3I_TITLE])?  id3ip[I3I_TITLE]:"",
                   (id3ip[I3I_ARTIST])? id3ip[I3I_ARTIST]:"",
                   (id3ip[I3I_ALBUM])?  id3ip[I3I_ALBUM]:"",
                   (id3ip[I3I_YEAR])?   id3ip[I3I_YEAR]:"",
                   (id3ip[I3I_COMMENT])?id3ip[I3I_COMMENT]:"",
                   tag_id3v1_get_genrenum(id3ip[I3I_GENRE]));
  }
  if(fbfs->fwrite(fbds,strtemp,128)==128) // successfull TAG writing
   error=MPXPLAY_ERROR_INFILE_OK;
 }
 return error;
}

//---------------------------------------------------------------------
//ID3v2

#define ID3V2_HEADSIZE 10
#define ID3V2_FOOTERSIZE 10
#define ID3V2_FRAMEHEADSIZE 10
#define ID3V2_MAX_DATALEN 256

static unsigned long mpx_id3v2x_get_framesize_4x7(char *bufp)
{
 unsigned long framesize=0;

 if(!(bufp[1]&0x80))
  framesize+=(bufp[0]&0x7f) << 21;

 if(!(bufp[2]&0x80))
  framesize+=(bufp[1]&0x7f) << 14;

 if(!(bufp[3]&0x80))
  framesize+=(bufp[2]&0x7f) <<  7;

 framesize+=(bufp[3]&0x7f)  <<  0;

 return framesize;
}

static unsigned long mpx_id3v2x_get_framesize_4x8(char *bufp)
{
 return pds_getb_be32(bufp);
}

static void mpx_id3v2x_skip_header(char *bufp,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned long id3v2totalsize;
 unsigned int footer_present;

 if(fbfs->fread(fbds,bufp+4,6)!=6)
  return;

 bufp+=3;   // ID3
 bufp+=2;   // version

 footer_present=bufp[0]&0x10;
 bufp+=1;   // flags

 id3v2totalsize=ID3V2_HEADSIZE+mpx_id3v2x_get_framesize_4x7(bufp);
 if(footer_present)
  id3v2totalsize+=ID3V2_FOOTERSIZE;

 if(id3v2totalsize<fbfs->filelength(fbds))
  fbfs->fseek(fbds,id3v2totalsize,SEEK_SET);
}

//-------------------------------------------------------------------------
static unsigned int mpx_id3v2x_textconv_utf_auto(char *id3p,unsigned int datalen,unsigned int text_enc_type,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 if((*(mpxplay_textconv_funcs->control))&ID3TEXTCONV_UTF_AUTO){
  switch(text_enc_type){
   case 1:datalen=mpxplay_textconv_funcs->utf16LE_to_char(id3p,datalen);mpxplay_textconv_funcs->convdone=ID3TEXTCONV_UTF16;break;
   case 2:datalen=mpxplay_textconv_funcs->utf16BE_to_char(id3p,datalen);mpxplay_textconv_funcs->convdone=ID3TEXTCONV_UTF16;break;
   case 3:datalen=mpxplay_textconv_funcs->utf8_to_char(id3p,datalen);mpxplay_textconv_funcs->convdone=ID3TEXTCONV_UTF8;break;
  }
 }
 return datalen;
}

static unsigned int mpx_id3v2x_tagread_text(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *id3p,unsigned int datalen,char **id3ip,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 int text_enc_type=0;
 if(datalen<2)
  return 0;
 if(fbfs->fread(fbds,(char *)&text_enc_type,1)!=1)
  return 0;
 datalen--;
 if(fbfs->fread(fbds,id3p,datalen)!=datalen)
  return 0;
 datalen=mpx_id3v2x_textconv_utf_auto(id3p,datalen,text_enc_type,mpxplay_textconv_funcs);
 return datalen;
}

static char *mpx_id3v2x_tagend_text(char *id3p,unsigned int datalen,char **id3ip,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 if(datalen){
  id3ip[0]=id3p;
  id3p[datalen]=0;
  datalen=mpxplay_textconv_funcs->all_to_char(id3p,datalen,mpxplay_textconv_funcs->convdone);
  id3p+=datalen+1;
 }
 return id3p;
}

static char *mpx_id3v2x_tagread_txxx(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *id3p,unsigned int datalen,char **id3ip,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 datalen=mpx_id3v2x_tagread_text(fbfs,fbds,id3p,datalen,id3ip,mpxplay_textconv_funcs);
 id3p=mpx_id3v2x_tagend_text(id3p,datalen,id3ip,mpxplay_textconv_funcs);
 return id3p;
}

static char *mpx_id3v2x_tagread_tcon(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *id3p,unsigned int datalen,char **id3ip,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 datalen=mpx_id3v2x_tagread_text(fbfs,fbds,id3p,datalen,id3ip,mpxplay_textconv_funcs);
 if(datalen){
  if(id3p[0]=='('){
   int gennum=pds_atol(&id3p[1]);
   if((gennum || id3p[1]=='0') && gennum<MAX_ID3GENRENUM)
    id3ip[0]=id3v1genres[gennum];
   else
    id3p=mpx_id3v2x_tagend_text(id3p,datalen,id3ip,mpxplay_textconv_funcs);
  }else{
   id3p=mpx_id3v2x_tagend_text(id3p,datalen,id3ip,mpxplay_textconv_funcs);
  }
 }
 return id3p;
}

static char *mpx_id3v2x_tagread_comm(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *id3p,unsigned int datalen,char **id3ip,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 char *s;
 int text_enc_type=0;

 if(datalen<=4)    // real datalen == 0
  return id3p;
 if(fbfs->fread(fbds,(char *)&text_enc_type,1)!=1) // text encoding type (1 byte)
  return id3p;
 if(fbfs->fread(fbds,id3p,3)!=3)                   // language           (3 bytes)
  return id3p;
 datalen-=4;
 if(fbfs->fread(fbds,id3p,datalen)!=datalen)
  return id3p;

 datalen=mpx_id3v2x_textconv_utf_auto(id3p,datalen,text_enc_type,mpxplay_textconv_funcs);

 s=id3p;
 do{
  if((datalen<1) || *s)
   break;
  datalen--;
  s++;
 }while(1);

 if(datalen && (s>id3p))
  pds_memcpy(id3p,s,datalen);

 id3p=mpx_id3v2x_tagend_text(id3p,datalen,id3ip,mpxplay_textconv_funcs);

 return id3p;
}

typedef struct id3v2x_one_frame_info_s{
 char *frameid;
 char *(*frame_reader)(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *id3p,unsigned int datalen,char **id3ip,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs);
 unsigned int i3i_index;
}id3v2x_one_frame_info_s;

typedef struct id3v2x_one_version_info_s{
 unsigned char version_number;
 unsigned long (*frame_getsize)(char *bufp);
 id3v2x_one_frame_info_s *(*frame_selector)(char *framehead,id3v2x_one_frame_info_s *f0);
 id3v2x_one_frame_info_s *supported_frames;
}id3v2x_one_version_info_s;

static id3v2x_one_frame_info_s id3v23_supported_frames[]=
{{"TPE1",&mpx_id3v2x_tagread_txxx,I3I_ARTIST},
 {"TIT2",&mpx_id3v2x_tagread_txxx,I3I_TITLE},
 {"TALB",&mpx_id3v2x_tagread_txxx,I3I_ALBUM},
 {"TYER",&mpx_id3v2x_tagread_txxx,I3I_YEAR},
 {"TCON",&mpx_id3v2x_tagread_tcon,I3I_GENRE},
 {"COMM",&mpx_id3v2x_tagread_comm,I3I_COMMENT},
 {"TRCK",&mpx_id3v2x_tagread_txxx,I3I_TRACKNUM},
 {NULL,NULL,0}
};

static id3v2x_one_frame_info_s id3v24_supported_frames[]=
{{"TPE1",&mpx_id3v2x_tagread_txxx,I3I_ARTIST},
 {"TIT2",&mpx_id3v2x_tagread_txxx,I3I_TITLE},
 {"TALB",&mpx_id3v2x_tagread_txxx,I3I_ALBUM},
 {"TYER",&mpx_id3v2x_tagread_txxx,I3I_YEAR},
 {"TDRC",&mpx_id3v2x_tagread_txxx,I3I_YEAR},
 {"TCON",&mpx_id3v2x_tagread_tcon,I3I_GENRE},
 {"COMM",&mpx_id3v2x_tagread_comm,I3I_COMMENT},
 {"TRCK",&mpx_id3v2x_tagread_txxx,I3I_TRACKNUM},
 {NULL,NULL,0}
};

static id3v2x_one_frame_info_s *mpx_id3v2x_frame_selector(char *framehead,id3v2x_one_frame_info_s *f)
{
 while(f->frameid){
  if(pds_strncmp(framehead,f->frameid,4)==0)
   return f;
  f++;
 }
 return NULL;
}

//2.3 and 2.4 are supported
static id3v2x_one_version_info_s id3v2_all_version_infos[]=
{{0x03,mpx_id3v2x_get_framesize_4x8,mpx_id3v2x_frame_selector,&id3v23_supported_frames[0]},
 {0x04,mpx_id3v2x_get_framesize_4x7,mpx_id3v2x_frame_selector,&id3v24_supported_frames[0]},
 {0x00,NULL,NULL}
};

static id3v2x_one_version_info_s *mpx_id3v2_version_selector(char *verstr)
{
 id3v2x_one_version_info_s *v=&id3v2_all_version_infos[0];
 while(v->frame_getsize){
  if((unsigned char)verstr[0]==v->version_number)
   return v;
  v++;
 }
 return NULL;
}

char *mpx_tag_read_id3v2(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 id3v2x_one_version_info_s *version_handler;
 unsigned long id3v2totalsize,filepos;
 unsigned int extended_header,footer_present;
 //unsigned int unsynchronisation,experimental_indicator;
 char *bufp;
 char readtmp[16];

 if(fbfs->fseek(fbds,0,SEEK_SET)<0)
  return id3p;
 if(fbfs->fread(fbds,&readtmp[0],ID3V2_HEADSIZE)!=ID3V2_HEADSIZE)
  return id3p;
 bufp=&readtmp[0];

 //"ID3"
 if((*((unsigned long *)(bufp))&0x00ffffff)!=HEADID_ID3V2)
  return id3p;
 bufp+=3;

 //version
 version_handler=mpx_id3v2_version_selector(bufp);
 if(!version_handler)
  return id3p;
 bufp+=2;

 //flags
 //unsynchronisation     =bufp[0]&0x80;
 extended_header       =bufp[0]&0x40;
 //experimental_indicator=bufp[0]&0x20;
 footer_present        =bufp[0]&0x10;
 bufp+=1;

 id3v2totalsize=ID3V2_HEADSIZE+mpx_id3v2x_get_framesize_4x7(bufp);
 if(footer_present)
  id3v2totalsize+=ID3V2_FOOTERSIZE;

 //fprintf(stderr,"id3v2size:%d %4.4X \n",id3v2totalsize,id3v2totalsize);

 if(id3v2totalsize>=fbfs->filelength(fbds) || id3v2totalsize<=ID3V2_HEADSIZE)
  return id3p;

 filepos=ID3V2_HEADSIZE;

 if(extended_header){
  unsigned long ehsize;
  if(fbfs->fread(fbds,&readtmp[0],4)!=4)
   return id3p;
  ehsize=version_handler->frame_getsize(&readtmp[0]);
  if(ehsize<4 || (id3v2totalsize+ehsize)>=fbfs->filelength(fbds))
   return id3p;
  filepos+=ehsize;
  if(fbfs->fseek(fbds,ehsize-4,SEEK_CUR)<0)
   return id3p;
 }

 if((filepos+ID3V2_FRAMEHEADSIZE)>=id3v2totalsize){ // no frames (tags) in the ID3V2
  fbfs->fseek(fbds,id3v2totalsize,SEEK_SET);
  return id3p;
 }

 do{
  unsigned int datalen;
  id3v2x_one_frame_info_s *frameselect;
  // maybe there are padding bytes... required ???
  /*do{
   fbfs->fread(fbds,&readtmp[0],1);
   filepos++;
   if(filepos>=id3v2totalsize)
    return id3p;
  }while(!readtmp[0]);
  filepos--;
  fbfs->fread(fbds,&readtmp[1],ID3V2_FRAMEHEADSIZE-1);*/

  if(fbfs->fread(fbds,&readtmp[0],ID3V2_FRAMEHEADSIZE)!=ID3V2_FRAMEHEADSIZE)
   break;
  filepos+=ID3V2_FRAMEHEADSIZE;

  datalen=version_handler->frame_getsize(&readtmp[4]);

  //fprintf(stderr,"\n%c%c%c%c %2d : ",readtmp[0],readtmp[1],readtmp[2],readtmp[3],datalen);

  if(datalen){
   filepos+=datalen;
   if(filepos>id3v2totalsize)
    break;
   if((datalen<ID3V2_MAX_DATALEN) && ((frameselect=version_handler->frame_selector(&readtmp[0],version_handler->supported_frames))!=NULL)){
    unsigned int i3i_select=frameselect->i3i_index;

    if(!id3ip[i3i_select]){
     //fprintf(stderr,"%s : ",frameselect->frameid);
     mpxplay_textconv_funcs->convdone=0;
     id3p=frameselect->frame_reader(fbfs,fbds,id3p,datalen,&id3ip[i3i_select],mpxplay_textconv_funcs);
     //fprintf(stderr," %s ",((id3ip[i3i_select])? id3ip[i3i_select]:"none"));
     if(id3ip[i3i_select])
      continue;
    }
   }
   if(fbfs->fseek(fbds,filepos,SEEK_SET)<0)
    break;
  }
 }while(filepos<(id3v2totalsize-ID3V2_FRAMEHEADSIZE));

 if(filepos!=id3v2totalsize)
  fbfs->fseek(fbds,id3v2totalsize,SEEK_SET);

 return id3p;
}

//---------------------------------------------------------------------------

#ifdef MPXPLAY_LINK_INFILE_MPX

static char *mpx_tag_get(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 mp3_decoder_data *mp3i=infile_data;
 char *id3pnew=NULL;

 if(!mp3i || (mp3i->infobits&MP3DI_INFOBIT_ID3V2_PRESENT))
  id3pnew=mpx_tag_read_id3v2(fbfs,fbds,id3ip,id3p,mpxplay_textconv_funcs);

 if((id3pnew<=id3p) || (!id3ip[I3I_ARTIST] && !id3ip[I3I_TITLE])){ // if we haven't got the required infos from the id3v2
  if(mpx_tag_check_id3v1(fbfs,fbds))
   id3p=mpx_tag_read_id3v1(fbfs,fbds,id3ip,id3p,mpxplay_textconv_funcs);
 }else
  id3p=id3pnew;

 return id3p;
}

struct mpxplay_infile_func_s IN_MP3_funcs={
 0,
 &mpx_preinit,
 NULL,
 &mpx_infile_check,
 &mpx_infile_check,
 &mpx_infile_open,
 &mpx_infile_close,
 &mpx_infile_decode,
 &mpx_fseek,
 &mpx_clearbuffs,
 &mpx_tag_get,
 &mpx_tag_write_id3v1,
 NULL,
 {"MP2","MP3",NULL}
};

#endif
