//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: APE file handling
//requires the dec_ape\apedec.lib file (and include files)

#include "in_file.h"

#ifdef MPXPLAY_LINK_INFILE_APE

#include "newfunc\newfunc.h"
#include "in_funcs.h"
#include "dec_ape\apetypes.h"
#include "dec_ape\all.h"
#include "dec_ape\maclib.h"
#include "dec_ape\apeinfo.h"

#define APEDEC_SEEKBLOCKNUM_INVALID 0xffffffff

typedef struct ape_decoder_data{
 struct IAPEDecompress_data_s *spAPEDecompress;
 unsigned long seek_blocknum;
 unsigned int firstseek;
}ape_decoder_data;

static void ape_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);

static int ape_assign_values(struct ape_decoder_data *apei,struct mpxplay_infile_info_s *miis)
{
 int (*GetInfo)(struct CAPEInfo_data_s *,enum APE_DECOMPRESS_FIELDS Field, int nParam1, int nParam2);
 struct CAPEInfo_data_s *apeinfo_datas=apei->spAPEDecompress->apeinfo_datas;
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 int fileversion;

 GetInfo=apei->spAPEDecompress->apeinfo_datas->apeinfo_funcs->GetInfo;

 miis->filesize=apei->spAPEDecompress->fileio_funcs->filelength(apei->spAPEDecompress->fileio_datas);
 adi->freq=GetInfo(apeinfo_datas,APE_INFO_SAMPLE_RATE,0,0);
 adi->filechannels=adi->outchannels=GetInfo(apeinfo_datas,APE_INFO_CHANNELS,0,0);
 adi->bits=GetInfo(apeinfo_datas,APE_INFO_BITS_PER_SAMPLE,0,0);
 adi->pcmdatalen=GetInfo(apeinfo_datas,APE_INFO_WAV_DATA_BYTES,0,0)/((adi->bits>>3)*adi->filechannels);

 apei->seek_blocknum=APEDEC_SEEKBLOCKNUM_INVALID;
 apei->firstseek=1;

 adi->longname=malloc(MPXPLAY_ADITEXTSIZE_LONGNAME+8);
 adi->bitratetext=malloc(MPXPLAY_ADITEXTSIZE_BITRATE+8);
 if(!adi->longname || !adi->bitratetext)
  return 0;
 fileversion=GetInfo(apeinfo_datas,APE_INFO_FILE_VERSION,0,0);
 sprintf(adi->longname,"APEv%1.1d.%2.2d",fileversion/1000,(fileversion/10)%100);
 sprintf(adi->bitratetext,"%2d/%2.1f%%",adi->bits,
         100.0*(float)GetInfo(apeinfo_datas,APE_INFO_AVERAGE_BITRATE,0,0)
         /(float)GetInfo(apeinfo_datas,APE_INFO_DECOMPRESSED_BITRATE,0,0));

 return 1;
}

static void *ape_infile_check(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct ape_decoder_data *apei;

 apei=(struct ape_decoder_data *)calloc(1,sizeof(struct ape_decoder_data));
 if(!apei)
  goto err_out_check;

 apei->spAPEDecompress=IAPEDecompress_check(fbfs,fbds,filename);
 if(!apei->spAPEDecompress)
  goto err_out_check;

 if(!ape_assign_values(apei,miis))
  goto err_out_check;

 return apei;

err_out_check:
 ape_infile_close(fbfs,fbds,apei,miis);
 return NULL;
}

static void *ape_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct ape_decoder_data *apei;
 long allblocks;

 apei=(struct ape_decoder_data *)calloc(1,sizeof(struct ape_decoder_data));
 if(!apei)
  goto err_out_open;

 apei->spAPEDecompress=IAPEDecompress_open(fbfs,fbds,filename);
 if(!apei->spAPEDecompress)
  goto err_out_open;

 if(!ape_assign_values(apei,miis))
  goto err_out_open;

 allblocks=IAPEDecompress_GetInfo(apei->spAPEDecompress,APE_DECOMPRESS_TOTAL_BLOCKS,0,0);
 if(allblocks<1)
  goto err_out_open;

 return apei;

err_out_open:
 ape_infile_close(fbfs,fbds,apei,miis);
 return NULL;
}

static void ape_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct ape_decoder_data *apei=infile_data;
 if(apei){
  struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
  apei->spAPEDecompress->fileio_datas=fbds; // update allways
  IAPEDecompress_close(apei->spAPEDecompress);
  if(adi->longname)
   free(adi->longname);
  if(adi->bitratetext)
   free(adi->bitratetext);
  free(apei);
 }
 fbfs->fclose(fbds);
}

static int ape_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct ape_decoder_data *apei=infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 int nBlocksDecoded = -1,nRetVal;

 apei->spAPEDecompress->fileio_datas=fbds; // update allways

 nRetVal = IAPEDecompress_GetData(apei->spAPEDecompress,(char *)adi->pcm_outdatabuf,adi->pcmsamplenum_frame,&nBlocksDecoded);

 if(nRetVal==APEDEC_ERROR_IO_EOF)
  return MPXPLAY_ERROR_INFILE_EOF;

 if(nRetVal!=APEDEC_ERROR_SUCCESS || nBlocksDecoded<1)
  return MPXPLAY_ERROR_INFILE_NODATA;

 adi->pcmoutsamplenum = (nBlocksDecoded * IAPEDecompress_GetInfo(apei->spAPEDecompress,APE_INFO_BLOCK_ALIGN,0,0))/(adi->bits>>3);

 return MPXPLAY_ERROR_INFILE_OK;
}

//-------------------------------------------------------------------------

static void ape_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,int cleartype)
{
 struct ape_decoder_data *apei=infile_data;
 if(apei){
  apei->spAPEDecompress->fileio_datas=fbds;
  if(cleartype&(MPX_SEEKTYPE_NORM|MPX_SEEKTYPE_BOF) && (apei->seek_blocknum!=APEDEC_SEEKBLOCKNUM_INVALID))
   IAPEDecompress_Seek(apei->spAPEDecompress,apei->seek_blocknum);
  apei->seek_blocknum=APEDEC_SEEKBLOCKNUM_INVALID;
 }
}

static long ape_fseek(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,long newmpxframenum)
{
 struct ape_decoder_data *apei=infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 struct IAPEDecompress_data_s *iapedec=apei->spAPEDecompress;
 unsigned int blocks_per_frame=IAPEDecompress_GetInfo(iapedec,APE_INFO_BLOCKS_PER_FRAME,0,0);
 unsigned int allapeframes=IAPEDecompress_GetInfo(iapedec,APE_INFO_TOTAL_FRAMES,0,0);
 unsigned int newblock=newmpxframenum*adi->pcmsamplenum_frame;
 unsigned int currblock,newapeframe;
 unsigned long newbytepos,beginbytepos;
 //char sout[100];

 if(apei->seek_blocknum!=APEDEC_SEEKBLOCKNUM_INVALID){
  currblock=apei->seek_blocknum;
  apei->seek_blocknum=APEDEC_SEEKBLOCKNUM_INVALID;
 }else
  currblock=IAPEDecompress_GetInfo(iapedec,APE_DECOMPRESS_CURRENT_BLOCK,0,0);

 if(newblock!=currblock || !newblock){
  if(newblock>currblock){
   if(currblock || (newblock-currblock)<blocks_per_frame) // fast forward seek
    newblock+=blocks_per_frame;
   newapeframe=newblock/blocks_per_frame;
   if(newapeframe>=allapeframes) // eof
    return MPXPLAY_ERROR_INFILE_EOF;
  }else{
   if(newblock>(blocks_per_frame/2)){ // rewind in file
    newblock-=blocks_per_frame/2;
    newapeframe=newblock/blocks_per_frame;
   }else{                 // begin of file (newblock==0 && newblock==currblock)
    newapeframe=0;
   }
  }

  newblock=newapeframe*blocks_per_frame;
  //pds_textdisplay_printf("--------------------------------------------------------------");
  //sprintf(sout,"cb:%d nb:%d nf:%d af:%d fs:%d",currblock,newblock,newapeframe,allapeframes,apei->firstseek);
  //display_message(1,0,sout);
  //pds_textdisplay_printf(sout);
  //getch();
  if(apei->firstseek){ // first seeking in this file, fill bitstream full
   iapedec->fileio_datas=fbds;
   if(IAPEDecompress_Seek(apei->spAPEDecompress,newblock)<0)
    return MPXPLAY_ERROR_INFILE_EOF;
   apei->seek_blocknum=APEDEC_SEEKBLOCKNUM_INVALID;
   apei->firstseek=0;
  }else{
   newbytepos=IAPEDecompress_GetInfo(iapedec,APE_INFO_SEEK_BYTE,newapeframe,0);
   beginbytepos=IAPEDecompress_GetInfo(iapedec,APE_INFO_SEEK_BYTE,0,0);
   newbytepos-=(newbytepos-beginbytepos)%4;
   if(iapedec->fileio_funcs->fseek(fbds,newbytepos,SEEK_SET)<0)
    return MPXPLAY_ERROR_INFILE_EOF;
   apei->seek_blocknum=newblock;
  }
  //sprintf(sout,"asb:%d ret:%d ",apei->seek_blocknum,newblock/apei->blocks_per_decode);
  //pds_textdisplay_printf(sout);
  //display_message(1,0,sout);
  //getch();
 }

 return (newblock/adi->pcmsamplenum_frame);
}

#endif // MPXPLAY_LINK_INFILE_APE

//----------------------------------------------------------------------

#if defined(MPXPLAY_LINK_INFILE_APE) || defined(MPXPLAY_LINK_INFILE_MPC) || defined(MPXPLAY_LINK_INFILE_AAC) || defined(MPXPLAY_LINK_INFILE_AC3)

#include "newfunc\newfunc.h"
#include "in_funcs.h"
#include "dec_ape\apeinfo.h"

typedef struct tagtype_s{
 char *name;
 unsigned int i3i_index;
}tagtype_s;

static tagtype_s tagtypes[]={
 {"Artist",I3I_ARTIST},{"Title"  ,I3I_TITLE}  ,{"Album",I3I_ALBUM},
 {"Year"  ,I3I_YEAR}  ,{"Comment",I3I_COMMENT},{"Genre",I3I_GENRE},
 {"Track" ,I3I_TRACKNUM}
};

#define TAGTYPENUM (sizeof(tagtypes)/sizeof(struct tagtype_s))

static int ape_tag_check_apetag(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct APETag_s *apetag)
{
 int version,taglen;
 long filesize=fbfs->filelength(fbds);
 if(filesize<=((long)sizeof(struct APETag_s)))
  return 0;
 if(fbfs->fseek(fbds,-((long)sizeof(struct APETag_s)),SEEK_END)<0)
  return 0;
 if(fbfs->fread(fbds,(char *)apetag,sizeof(struct APETag_s))!=sizeof(struct APETag_s))
  return 0;
 if(pds_strncmp(apetag->ID,"APETAGEX",sizeof(apetag->ID))!=0)
  return 0;
 version=read_le_long(apetag->Version);
 if(version!=1000 && version!=2000)
  return 0;
 taglen=read_le_long(apetag->Length);
 if(taglen<=sizeof(struct APETag_s) || (taglen>=filesize))
  return 0;
 if(!read_le_long(apetag->TagCount))
  return 0;
 if(fbfs->fseek(fbds,-taglen,SEEK_END)<0)
  return 0;
 return 1;
}

static char *ape_tag_read_apetag(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct APETag_s *apetag,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 long tt,version,taglen,tagcount;
 long namelen,datalen,dataflags;
 char *tagbuff,*tbp,*tname,*tdata;

 taglen=read_le_long(apetag->Length);
 tagbuff=malloc(taglen);
 if(!tagbuff)
  return id3p;
 if(fbfs->fread(fbds,tagbuff,taglen-sizeof(struct APETag_s))!=(taglen-sizeof(struct APETag_s)))
  goto err_out_tagread;
 tbp=tagbuff;
 tagbuff[taglen-sizeof(struct APETag_s)]=0;
 version=read_le_long(apetag->Version);
 tagcount=read_le_long(apetag->TagCount);
 for(;*tbp && tagcount--;){
  datalen  =read_le_long(tbp); tbp+=4;
  dataflags=read_le_long(tbp); tbp+=4;
  namelen=strlen(tbp)+1;
  if(namelen>128)        // impossible data
   break;
  tname=tbp; tbp+=namelen;
  tdata=tbp; tbp+=datalen;
  if(tbp>(&tagbuff[taglen-sizeof(struct APETag_s)])) // bad data -> overflow
   break;
  if((namelen>1) && ((version==1000 && datalen>1) || (version==2000 && datalen>0 && !(dataflags&2)))){
   for(tt=0;tt<TAGTYPENUM;tt++){
    int i3iindex=tagtypes[tt].i3i_index;
    if(!id3ip[i3iindex] && pds_stricmp(tname,tagtypes[tt].name)==0){
     id3ip[i3iindex]=id3p;
     pds_memcpy(id3p,tdata,datalen);
     if(version==1000){
      id3p+=mpxplay_textconv_funcs->all_to_char(id3p,datalen-1,0)+1; // datalen includes the terminating null in v1
     }else{
      id3p[datalen]=0;
      if((*(mpxplay_textconv_funcs->control))&ID3TEXTCONV_UTF_AUTO)
       datalen=mpxplay_textconv_funcs->utf8_to_char(id3p,datalen);  // allways in v2
      id3p+=mpxplay_textconv_funcs->all_to_char(id3p,datalen,ID3TEXTCONV_UTF8)+1;   // datalen does not include the terminating null in v2
     }
     break;
    }
   }
  }
 }
err_out_tagread:
 free(tagbuff);
 return id3p;
}

char *ape_tag_get(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,char **id3ip,char *id3p,struct mpxplay_textconv_func_s *mpxplay_textconv_funcs)
{
 struct APETag_s apetag;

 if(mpx_tag_check_id3v1(fbfs,fbds))
  return mpx_tag_read_id3v1(fbfs,fbds,id3ip,id3p,mpxplay_textconv_funcs);

 if(ape_tag_check_apetag(fbfs,fbds,&apetag))
  return ape_tag_read_apetag(fbfs,fbds,&apetag,id3ip,id3p,mpxplay_textconv_funcs);

 return id3p;
}

int ape_tag_put(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char **id3ip)
{
 int tt,i,extratags=0,cutfile=0;
 long alltagsize=0,tagcount=0,newalltagsize,*tagdlens=NULL,*tagflags=NULL;
 __int32 tagnamelen,tagdatalen,flags=0,version;
 char *tagbuff=NULL,*tbp,**tagnames=NULL,**tagdatas=NULL;
 struct APETag_s apetag;

 if(mpx_tag_check_id3v1(fbfs,fbds))
  return mpx_tag_write_id3v1(fbfs,fbds,id3ip);

 if(ape_tag_check_apetag(fbfs,fbds,&apetag)){ // APE file already has APETAGEX
  // load all tagdatas, update and write back
  version=read_le_long(apetag.Version);
  if(version!=1000)// && version!=2000)
   return MPXPLAY_ERROR_INFILE_WRITETAG_TAGTYPE;
  alltagsize=read_le_long(apetag.Length);
  tbp=tagbuff=malloc(alltagsize);
  tagcount=read_le_long(apetag.TagCount);
  tagnames=(char **)malloc((tagcount+TAGTYPENUM)*sizeof(char *));
  tagdatas=(char **)malloc((tagcount+TAGTYPENUM)*sizeof(char *));
  tagdlens=(long *)malloc((tagcount+TAGTYPENUM)*sizeof(long));
  tagflags=(long *)malloc((tagcount+TAGTYPENUM)*sizeof(long));
  if(!tagbuff || !tagnames || !tagdatas || !tagdlens || !tagflags)
   goto err_out_tagput;
  //load
  if(fbfs->fread(fbds,tagbuff,alltagsize-sizeof(struct APETag_s))!=(alltagsize-sizeof(struct APETag_s)))
   goto err_out_tagput;
  for(i=0;i<tagcount && (tbp<(tagbuff+alltagsize-sizeof(struct APETag_s)));i++){
   tagdlens[i]=read_le_long(tbp);tbp+=4;
   tagflags[i]=read_le_long(tbp);tbp+=4;
   tagnames[i]=tbp;              tbp+=strlen(tbp)+1;
   tagdatas[i]=tbp;              tbp+=tagdlens[i];
  }
  tagcount=i; // correction in the case of a wrong ape-tag
  //update
  for(tt=0;tt<TAGTYPENUM;tt++){
   int i3iindex=tagtypes[tt].i3i_index;
   if(id3ip[i3iindex]){
    for(i=0;i<tagcount;i++){
     if((version==1000 && tagdlens[i]>1) || (version==2000 && tagdlens[i]>0 && !(tagflags[i]&2))){
      if(pds_stricmp(tagnames[i],tagtypes[tt].name)==0){ // found, update
       tagdatas[i]=id3ip[i3iindex];
       tagdlens[i]=strlen(id3ip[i3iindex])+((version==1000)? 1:0);
       break;
      }
     }
    }
    if(i==tagcount){                                 // not found, new
     tagnames[tagcount+extratags]=tagtypes[tt].name;
     tagdatas[tagcount+extratags]=id3ip[i3iindex];
     tagdlens[tagcount+extratags]=strlen(id3ip[i3iindex])+((version==1000)? 1:0);
     tagflags[tagcount+extratags]=0;
     extratags++;
    }
   }
  }
  //write back
  if(fbfs->fseek(fbds,-alltagsize,SEEK_END)<0)
   goto err_out_tagput;
  newalltagsize=0;
  tagcount+=extratags;
  for(i=0;i<tagcount;i++){
   tagnamelen=strlen(tagnames[i])+1;
   fbfs->fwrite(fbds,&tagdlens[i],sizeof(tagdlens[i]));
   fbfs->fwrite(fbds,&tagflags[i],sizeof(tagflags[i]));
   fbfs->fwrite(fbds,tagnames[i],tagnamelen);
   fbfs->fwrite(fbds,tagdatas[i],tagdlens[i]);
   newalltagsize+=sizeof(tagdlens[i])+sizeof(tagflags[i])+tagnamelen+tagdlens[i];
  }
  if(newalltagsize<alltagsize)
   cutfile=1;
  alltagsize=newalltagsize;
  free(tagflags);
  free(tagdlens);
  free(tagdatas);
  free(tagnames);
  free(tagbuff);
 }else{
  // write an absolute new APEv1 tag
  version=1000;
  if(fbfs->fseek(fbds,0,SEEK_END)<0)
   return MPXPLAY_ERROR_INFILE_EOF;
  for(tt=0;tt<TAGTYPENUM;tt++){
   int i3iindex=tagtypes[tt].i3i_index;
   if(id3ip[i3iindex]){
    tagnamelen=strlen(tagtypes[tt].name)+1;
    tagdatalen=strlen(id3ip[i3iindex])+1;
    fbfs->fwrite(fbds,&tagdatalen,sizeof(tagdatalen));
    fbfs->fwrite(fbds,&flags,sizeof(flags));
    fbfs->fwrite(fbds,tagtypes[tt].name,tagnamelen);
    fbfs->fwrite(fbds,id3ip[i3iindex],tagdatalen);
    alltagsize+=sizeof(tagdatalen)+sizeof(flags)+tagnamelen+tagdatalen;
    tagcount++;
   }
  }
 }
 //finish/close the tag
 memcpy(apetag.ID,"APETAGEX",sizeof("APETAGEX"));
 write_le_long(apetag.Version,version);
 write_le_long(apetag.Length,alltagsize+sizeof(struct APETag_s));
 write_le_long(apetag.TagCount,tagcount);
 write_le_long(apetag.Flags,0);
 memset(apetag.Reserved,0,sizeof(apetag.Reserved));
 fbfs->fwrite(fbds,(void *)&apetag,sizeof(struct APETag_s));

 if(cutfile)
  fbfs->chsize(fbds,fbfs->ftell(fbds));

 return MPXPLAY_ERROR_INFILE_OK;

err_out_tagput:
 if(tagflags)
  free(tagflags);
 if(tagdlens)
  free(tagdlens);
 if(tagdatas)
  free(tagdatas);
 if(tagnames)
  free(tagnames);
 if(tagbuff)
  free(tagbuff);
 return MPXPLAY_ERROR_INFILE_MEMORY;
}

#endif // defined(MPXPLAY_LINK_INFILE_APE) || defined(MPXPLAY_LINK_INFILE_MPC) || defined(MPXPLAY_LINK_INFILE_AAC) || defined(MPXPLAY_LINK_INFILE_AC3)

#ifdef MPXPLAY_LINK_INFILE_APE

struct mpxplay_infile_func_s IN_APE_funcs={
 0,
 NULL,
 NULL,
 &ape_infile_check,
 &ape_infile_check,
 &ape_infile_open,
 &ape_infile_close,
 &ape_infile_decode,
 &ape_fseek,
 &ape_clearbuffs,
 &ape_tag_get,
 &ape_tag_put,
 NULL,
 {"APE","MAC",NULL}
};

#endif
