//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2004 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: AC3 main
//
//based on the ac3dec by Aaron Holtzman

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "ac3.h"
#include "ac3decod.h"
#include "bitalloc.h"
#include "bitstrm.h"
#include "coeff.h"
#include "crc.h"
#include "downmix.h"
#include "exponent.h"
#include "imdct.h"
#include "parse.h"
#include "rematrix.h"

static void *ac3_mallc(unsigned int size)
{
 void *ptr=malloc(size);//+256);
 if(ptr)
  memset(ptr,0,size);//+256);
 return ptr;
}

static ac3_decoder_data *AC3_alloc_mems(void)
{
 ac3_decoder_data *amip;
 amip=ac3_mallc(sizeof(struct ac3_decoder_data));
 if(!amip)
  return NULL;

 amip->bs=ac3_mallc(sizeof(struct bitstream_s));
 if(!amip->bs)
  goto err_out_allocmems;
 amip->bs->buffer=ac3_mallc(AC3_FRAMESIZE_MAX);
 if(!amip->bs->buffer)
  goto err_out_allocmems;
 amip->audblk=ac3_mallc(sizeof(struct audblk_s));
 if(!amip->audblk)
  goto err_out_allocmems;
 amip->bsi=ac3_mallc(sizeof(struct bsi_s));
 if(!amip->bsi)
  goto err_out_allocmems;
 amip->syncinfo=ac3_mallc(sizeof(struct syncinfo_s));
 if(!amip->syncinfo)
  goto err_out_allocmems;
 amip->samples=ac3_mallc(sizeof(stream_samples_t));
 if(!amip->samples)
  goto err_out_allocmems;
 amip->mdct_delaybuf=ac3_mallc(sizeof(mdct_delaybuf_t));
 if(!amip->mdct_delaybuf)
  goto err_out_allocmems;
 amip->pcmoutsave_buffer=ac3_mallc(2*sizeof(AC3_OUT_T)*AC3_BLOCKSAMPLES*AC3_OUTCHANNELS);
 if(!amip->pcmoutsave_buffer)
  goto err_out_allocmems;

 return amip;

err_out_allocmems:
 CloseAc3(amip);
 return NULL;
}

static void AC3_init_variables(ac3_decoder_data *amip,void *read_func,void *readfunc_passinfo)
{
 static unsigned int ac3_preinit_done;

 amip->readfunc_callback=read_func;
 amip->readfunc_passinfo=readfunc_passinfo;

 //constant settings
 amip->DRC_type=DRC_HIGH; // dynamic range/compression control (auto volume correction)
 amip->downmix_select=AC3_DOWNMIX_TYPE_DOLBY2;

#ifdef AC3_LINK_DOWNMIX_2
 if(amip->downmix_select==AC3_DOWNMIX_TYPE_DOLBY2){
  amip->surround_delay=ac3_mallc(sizeof(surround_delay_t));
  if(!amip->surround_delay)
   amip->downmix_select=0;
 }
#endif
 if(!ac3_preinit_done){
  ac3dec_exponent_init();
  ac3dec_coeff_init();
  ac3dec_imdct_init();
  ac3dec_downmix_init(amip);
  ac3_preinit_done=1;
 }
}

static unsigned int AC3_sync_header(ac3_decoder_data *amip)
{
 if(!ac3dec_parse_syncinfo(amip)) // search and read frame
  return 0;                // failed
 ac3dec_parse_bsi(amip->bsi,amip->bs);
 return 1;
}

ac3_decoder_data *InitAc3(void *read_func,void *readfunc_passinfo)
{
 ac3_decoder_data *amip;
 amip=AC3_alloc_mems();
 if(!amip)
  return amip;

 AC3_init_variables(amip,read_func,readfunc_passinfo);
 if(!AC3_sync_header(amip))
  goto err_out_init;
 return amip;

err_out_init:
 CloseAc3(amip);
 return NULL;
}

unsigned int DecodeAc3(ac3_decoder_data *amip,AC3_OUT_T *pcmout,unsigned int samplenum_left)
{
 unsigned int pcmoutsamplenum=0;

 if(amip->outtype&AC3_OUTTYPE_BITSTREAM){
  char *pcm_c=(char *)pcmout;
  if(!ac3dec_parse_syncinfo(amip)) // search and read frame
   return 0;                // failed
  if(!(amip->outtype&AC3_OUTTYPE_BITSTRM_NOFRH)){
   pcm_c[0]=0x0b;
   pcm_c[1]=0x77;
   pcm_c+=2;
   pcmoutsamplenum+=2;
  }
  memcpy(pcm_c,amip->bs->buffer,amip->syncinfo->frame_size*2-2);
  pcmoutsamplenum+=amip->syncinfo->frame_size*2-2;
  return pcmoutsamplenum;
 }

 if(amip->pcmoutsave_samplenum){
  unsigned int samplenum_currblock=min(amip->pcmoutsave_samplenum,samplenum_left);
  memcpy(pcmout,amip->pcmoutsave_buffer+(AC3_BLOCKSAMPLES-samplenum_currblock)*AC3_OUTCHANNELS,samplenum_currblock*AC3_OUTCHANNELS*sizeof(AC3_OUT_T));
  pcmout+=samplenum_currblock*AC3_OUTCHANNELS;
  amip->pcmoutsave_samplenum-=samplenum_currblock;
  samplenum_left-=samplenum_currblock;
  pcmoutsamplenum+=samplenum_currblock;
 }

 amip->error_flag=0;

 while(samplenum_left){

  if(!amip->blockcount){
   amip->bs->bitindex=amip->bs->stored_bytes=0;
   if(!AC3_sync_header(amip))
    break;
   amip->blockcount=AC3_BLOCKS_PER_FRAME;
  }
  amip->blockcount--;

  memset(amip->samples,0,sizeof(AC3_DOUBLE_T) * 256 * (amip->bsi->nfchans + amip->bsi->lfeon));

  // Extract most of the audblk info from the bitstream (minus the mantissas)
  if(!ac3dec_parse_audblk(amip->bsi,amip->audblk,amip->bs))
   break;

  // Take the differential exponent data and turn it into absolute exponents
  if(!ac3dec_exponent_unpack(amip->bsi,amip->audblk))
   break;

  // Figure out how many bits per mantissa
  if(!ac3dec_bit_allocate(amip->syncinfo->fscod,amip->bsi,amip->audblk))
   break;

  // Extract the mantissas from the stream and generate floating point frequency coefficients
  ac3dec_coeff_unpack(amip,amip->bsi,amip->audblk,amip->samples);
  if(amip->error_flag)
   break;

  if(amip->bsi->acmod == 0x2)
   rematrix(amip->audblk,amip->samples);

  //ac3dec_downmix_output(amip,amip->bsi,amip->samples,(samplenum_left>=AC3_BLOCKSAMPLES)? pcmout:amip->pcmoutsave_buffer);
  //ac3dec_imdct_decode(amip->bsi,amip->audblk,(samplenum_left>=AC3_BLOCKSAMPLES)? pcmout:amip->pcmoutsave_buffer,amip->mdct_delaybuf);

  // Convert the frequency samples into time samples
  ac3dec_imdct_decode(amip->bsi,amip->audblk,amip->samples,amip->mdct_delaybuf,amip);

  // Downmix into the requested number of channels
  ac3dec_downmix_output(amip,amip->bsi,amip->samples,(samplenum_left>=AC3_BLOCKSAMPLES)? pcmout:amip->pcmoutsave_buffer);

  if(samplenum_left>=AC3_BLOCKSAMPLES){
   pcmout+=AC3_BLOCKSAMPLES*AC3_OUTCHANNELS;
   samplenum_left-=AC3_BLOCKSAMPLES;
   pcmoutsamplenum+=AC3_BLOCKSAMPLES;
  }else{
   memcpy(pcmout,amip->pcmoutsave_buffer,samplenum_left*AC3_OUTCHANNELS*sizeof(AC3_OUT_T));
   amip->pcmoutsave_samplenum=AC3_BLOCKSAMPLES-samplenum_left;
   pcmoutsamplenum+=samplenum_left;
   samplenum_left=0;
  }
 }

 return (pcmoutsamplenum*AC3_OUTCHANNELS);
}

void AC3reset(ac3_decoder_data *amip)
{
 ac3dec_imdct_reset(amip);
 ac3dec_downmix_reset(amip);
 amip->pcmoutsave_samplenum=0;
 amip->blockcount=0;
}

void CloseAc3(ac3_decoder_data *amip)
{
 if(amip){
  if(amip->bs){
   if(amip->bs->buffer)
    free(amip->bs->buffer);
   free(amip->bs);
  }
  if(amip->audblk)
   free(amip->audblk);
  if(amip->bsi)
   free(amip->bsi);
  if(amip->syncinfo)
   free(amip->syncinfo);
  if(amip->samples)
   free(amip->samples);
  if(amip->mdct_delaybuf)
   free(amip->mdct_delaybuf);
  if(amip->pcmoutsave_buffer)
   free(amip->pcmoutsave_buffer);
#ifdef AC3_LINK_DOWNMIX_2
  if(amip->surround_delay)
   free(amip->surround_delay);
#endif
  free(amip);
 }
}
