//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2004 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: SB Live/Audigy (EMU10K1 chip) low level routines
//based on the Creative (http://sourceforge.net/projects/emu10k1)
//         and ALSA (http://www.alsa-project.org) drivers

#include <au_cards\au_cards.h>
#include <newfunc\newfunc.h>
#include <newfunc\dll_load.h>

#include <string.h>
#include <au_cards\dmairq.h>
#include <au_cards\pcibios.h>
#include "ac97_def.h"
#include "emu10k1.h"

#define VOICE_FLAGS_MASTER      0x01
#define VOICE_FLAGS_STEREO	0x02
#define VOICE_FLAGS_16BIT	0x04

#define DEFAULT_PCMVOLUME_AUDIGY  66  // 0-100

static struct mpxplay_resource_s *mrs;

struct emu10k1_card
{
 unsigned long   iobase;
 unsigned short	 model;
 unsigned int    irq;
 unsigned char   chiprev;
 unsigned char   is_audigy;
 struct pci_config_s  *pci_dev;

 dosmem_t *dm;                     // now it's ca. 72k only
 //void *cardbuf_mem;              // but later we should use DPMI memory
 unsigned long *virtualpagetable;
 void *silentpage;
 char *pcmout_buffer;
 long pcmout_bufsize;

 unsigned int voice_initial_pitch;
 unsigned int voice_pitch_target;
};

static void emu10k1_ac97_write(unsigned int iobase,unsigned int reg, unsigned int value);
static unsigned int emu10k1_ac97_read(unsigned int iobase, unsigned int reg);

//-------------------------------------------------------------------------
// low level write & read
#define outb(reg,val) outp(reg,val)
#define outw(reg,val) outpw(reg,val)
#define outl(reg,val) outpd(reg,val)
#define inb(reg) inp(reg)
#define inw(reg) inpw(reg)
#define inl(reg) inpd(reg)

static void emu10k1_writefn0(struct emu10k1_card *card, uint32_t reg, uint32_t data)
{
 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;
  data = (data << offset) & mask;
  reg &= 0x7f;

  data |= inl(card->iobase + reg) & ~mask;
  outl(card->iobase + reg, data);
 }else{
  outl(card->iobase + reg, data);
 }

 return;
}

static uint32_t emu10k1_readfn0(struct emu10k1_card *card, uint32_t reg)
{
 uint32_t val;

 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;
  reg &= 0x7f;

  val = inl(card->iobase + reg);

  return (val & mask) >> offset;
 }else{
  val = inl(card->iobase + reg);
  return val;
 }
}

#define A_PTR_ADDRESS_MASK 0x0fff0000
static void emu10k1_writeptr(struct emu10k1_card *card, uint32_t reg, uint32_t channel, uint32_t data)
{
 uint32_t regptr;

 regptr = ((reg << 16) & A_PTR_ADDRESS_MASK) | (channel & PTR_CHANNELNUM_MASK);

 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;
  data = (data << offset) & mask;

  outl(card->iobase + PTR, regptr);
  data |= inl(card->iobase + DATA) & ~mask;
  outl(card->iobase + DATA, data);
 }else{
  outl(card->iobase + PTR, regptr);
  outl(card->iobase + DATA, data);
 }
}

static uint32_t emu10k1_readptr(struct emu10k1_card *card, uint32_t reg, uint32_t channel)
{
 uint32_t regptr, val;

 regptr = ((reg << 16) & A_PTR_ADDRESS_MASK) | (channel & PTR_CHANNELNUM_MASK);

 if(reg & 0xff000000) {
  uint32_t mask;
  uint8_t size, offset;

  size = (reg >> 24) & 0x3f;
  offset = (reg >> 16) & 0x1f;
  mask = ((1 << size) - 1) << offset;

  outl(card->iobase + PTR, regptr);
  val = inl(card->iobase + DATA);

  return (val & mask) >> offset;
 }else{
  outl(card->iobase + PTR, regptr);
  val = inl(card->iobase + DATA);

  return val;
 }
}

//-----------------------------------------------------------------------
// init & close
static unsigned int snd_emu10k1_buffer_init(struct audio_info *aui,struct emu10k1_card *card)
{
 uint32_t pagecount,pcmbufp;

 card->pcmout_bufsize=MDma_get_max_pcmoutbufsize(aui,EMUPAGESIZE);

 card->dm=MDma_alloc_dosmem( MAXPAGES*sizeof(uint32_t)       // virtualpage
                            +EMUPAGESIZE                     // silentpage
                            +card->pcmout_bufsize            // pcm output
                            +0x1000 );                       // to round
 if(!card->dm)
  return 0;

 card->silentpage=(void *)(((uint32_t)card->dm->linearptr+0x0fff)&0xfffff000); // buffer begins on page boundary
 card->virtualpagetable=(uint32_t *)((uint32_t)card->silentpage+EMUPAGESIZE);
 card->pcmout_buffer=(char *)(card->virtualpagetable+MAXPAGES);

 pcmbufp=(uint32_t)card->pcmout_buffer;
 pcmbufp<<=1;
 for(pagecount = 0; pagecount < (card->pcmout_bufsize/EMUPAGESIZE); pagecount++){
  card->virtualpagetable[pagecount] = pcmbufp | pagecount;
  pcmbufp+=EMUPAGESIZE*2;
 }
 for( ; pagecount<MAXPAGES; pagecount++)
  card->virtualpagetable[pagecount] = ((uint32_t)card->silentpage)<<1;

 return 1;
}

static void snd_emu10k1_hw_init(struct emu10k1_card *card)
{
 int ch;
 uint32_t silent_page;

 // disable audio and lock cache
 emu10k1_writefn0(card,HCFG,HCFG_LOCKSOUNDCACHE | HCFG_LOCKTANKCACHE_MASK | HCFG_MUTEBUTTONENABLE);

 // reset recording buffers
 emu10k1_writeptr(card, MICBS, 0, ADCBS_BUFSIZE_NONE);
 emu10k1_writeptr(card, MICBA, 0, 0);
 emu10k1_writeptr(card, FXBS, 0, ADCBS_BUFSIZE_NONE);
 emu10k1_writeptr(card, FXBA, 0, 0);
 emu10k1_writeptr(card, ADCBS, 0, ADCBS_BUFSIZE_NONE);
 emu10k1_writeptr(card, ADCBA, 0, 0);

 // disable channel interrupt
 emu10k1_writefn0(card, INTE, 0);
 emu10k1_writeptr(card, CLIEL, 0, 0);
 emu10k1_writeptr(card, CLIEH, 0, 0);
 emu10k1_writeptr(card, SOLEL, 0, 0);
 emu10k1_writeptr(card, SOLEH, 0, 0);

 if(card->is_audigy){
  emu10k1_writeptr(card, 0x5e, 0, 0xf00); // ??
  emu10k1_writeptr(card, 0x5f, 0, 0x3);   // ??
 }

 // init envelope engine
 for (ch = 0; ch < NUM_G; ch++){
  emu10k1_writeptr(card, DCYSUSV, ch, 0);
  emu10k1_writeptr(card, IP, ch, 0);
  emu10k1_writeptr(card, VTFT, ch, 0xffff);
  emu10k1_writeptr(card, CVCF, ch, 0xffff);
  emu10k1_writeptr(card, PTRX, ch, 0);
  emu10k1_writeptr(card, CPF, ch, 0);
  emu10k1_writeptr(card, CCR, ch, 0);

  emu10k1_writeptr(card, PSST, ch, 0);
  emu10k1_writeptr(card, DSL, ch, 0x10);
  emu10k1_writeptr(card, CCCA, ch, 0);
  emu10k1_writeptr(card, Z1, ch, 0);
  emu10k1_writeptr(card, Z2, ch, 0);
  emu10k1_writeptr(card, FXRT, ch, 0x32100000);

  emu10k1_writeptr(card, ATKHLDM, ch, 0);
  emu10k1_writeptr(card, DCYSUSM, ch, 0);
  emu10k1_writeptr(card, IFATN, ch, 0xffff);
  emu10k1_writeptr(card, PEFE, ch, 0);
  emu10k1_writeptr(card, FMMOD, ch, 0);
  emu10k1_writeptr(card, TREMFRQ, ch, 24);  // 1 Hz
  emu10k1_writeptr(card, FM2FRQ2, ch, 24);  // 1 Hz
  emu10k1_writeptr(card, TEMPENV, ch, 0);

  // these are last so OFF prevents writing
  emu10k1_writeptr(card, LFOVAL2, ch, 0);
  emu10k1_writeptr(card, LFOVAL1, ch, 0);
  emu10k1_writeptr(card, ATKHLDV, ch, 0);
  emu10k1_writeptr(card, ENVVOL, ch, 0);
  emu10k1_writeptr(card, ENVVAL, ch, 0);

  // Audigy extra stuffs
  if (card->is_audigy) {
   emu10k1_writeptr(card, 0x4c, ch, 0); // ??
   emu10k1_writeptr(card, 0x4d, ch, 0); // ??
   emu10k1_writeptr(card, 0x4e, ch, 0); // ??
   emu10k1_writeptr(card, 0x4f, ch, 0); // ??
   emu10k1_writeptr(card, A_FXRT1, ch, 0x03020100);
   emu10k1_writeptr(card, A_FXRT2, ch, 0x3f3f3f3f);
   emu10k1_writeptr(card, A_SENDAMOUNTS, ch, 0);
  }
 }

 /*
  *  Init to 0x02109204 :
  *  Clock accuracy    = 0     (1000ppm)
  *  Sample Rate       = 2     (48kHz)
  *  Audio Channel     = 1     (Left of 2)
  *  Source Number     = 0     (Unspecified)
  *  Generation Status = 1     (Original for Cat Code 12)
  *  Cat Code          = 12    (Digital Signal Mixer)
  *  Mode              = 0     (Mode 0)
  *  Emphasis          = 0     (None)
  *  CP                = 1     (Copyright unasserted)
  *  AN                = 0     (Audio data)
  *  P                 = 0     (Consumer)
  */
 emu10k1_writeptr(card, SPCS0, 0,
			SPCS_CLKACCY_1000PPM | SPCS_SAMPLERATE_48 |
			SPCS_CHANNELNUM_LEFT | SPCS_SOURCENUM_UNSPEC |
			SPCS_GENERATIONSTATUS | 0x00001200 |
			0x00000000 | SPCS_EMPHASIS_NONE | SPCS_COPYRIGHT);
 emu10k1_writeptr(card, SPCS1, 0,
			SPCS_CLKACCY_1000PPM | SPCS_SAMPLERATE_48 |
			SPCS_CHANNELNUM_LEFT | SPCS_SOURCENUM_UNSPEC |
			SPCS_GENERATIONSTATUS | 0x00001200 |
			0x00000000 | SPCS_EMPHASIS_NONE | SPCS_COPYRIGHT);
 emu10k1_writeptr(card, SPCS2, 0,
			SPCS_CLKACCY_1000PPM | SPCS_SAMPLERATE_48 |
			SPCS_CHANNELNUM_LEFT | SPCS_SOURCENUM_UNSPEC |
			SPCS_GENERATIONSTATUS | 0x00001200 |
			0x00000000 | SPCS_EMPHASIS_NONE | SPCS_COPYRIGHT);

 if(card->is_audigy && card->chiprev == 4) { // audigy2
  // Hacks for Alice3 to work independent of haP16V driver
  uint32_t tmp;

  //Setup SRCMulti_I2S SamplingRate
  tmp = emu10k1_readptr(card, A_SPDIF_SAMPLERATE, 0);
  tmp &= 0xfffff1ff;
  tmp |= (0x2<<9);
  emu10k1_writeptr(card, A_SPDIF_SAMPLERATE, 0, tmp);

  // Setup SRCSel (Enable Spdif,I2S SRCMulti)
  emu10k1_writefn0(card, 0x20, 0x600000);
  emu10k1_writefn0(card, 0x24, 0x14);

  // Setup SRCMulti Input Audio Enable
  emu10k1_writefn0(card, 0x20, 0x6E0000);
  emu10k1_writefn0(card, 0x24, 0xFF00FF00);
 }

 //buffer config
 emu10k1_writeptr(card, PTB, 0, (uint32_t) card->virtualpagetable);
 emu10k1_writeptr(card, TCB, 0, 0);
 emu10k1_writeptr(card, TCBS, 0, 4);

 silent_page = (((uint32_t)card->silentpage) << 1) | MAP_PTI_MASK;

 for (ch = 0; ch < NUM_G; ch++) {
  emu10k1_writeptr(card, MAPA, ch, silent_page);
  emu10k1_writeptr(card, MAPB, ch, silent_page);
 }

 //mixer (routing) config
 fx_init(card);

 // setup HCFG
 if(card->is_audigy) {
  if(card->chiprev == 4) // audigy2
   emu10k1_writefn0(card,HCFG,HCFG_AUDIOENABLE | HCFG_AC3ENABLE_CDSPDIF | HCFG_AC3ENABLE_GPSPDIF | HCFG_AUTOMUTE | HCFG_JOYENABLE);
  else
   emu10k1_writefn0(card,HCFG,HCFG_AUDIOENABLE | HCFG_AUTOMUTE | HCFG_JOYENABLE);
 }else{
  if(card->model == 0x20 || card->model == 0xc400 || (card->model == 0x21 && card->chiprev < 6))
   emu10k1_writefn0(card,HCFG,HCFG_AUDIOENABLE | HCFG_LOCKTANKCACHE_MASK | HCFG_AUTOMUTE);
  else
   emu10k1_writefn0(card,HCFG,HCFG_AUDIOENABLE | HCFG_LOCKTANKCACHE_MASK | HCFG_AUTOMUTE | HCFG_JOYENABLE);
 }

 if(card->is_audigy){
  uint32_t tmp = emu10k1_readfn0(card, A_IOCFG);
  tmp&=~0x44;
  if(card->chiprev==4) // Audigy2 Unmute Analog now.  Set GPO6 to 1 for Apollo.
   tmp |= 0x0040;
  emu10k1_writefn0(card, A_IOCFG, tmp);
 }
}

static void snd_emu10k1_hw_close(struct emu10k1_card *card)
{
 int ch;

 emu10k1_writefn0(card,INTE,0);

 // Shutdown the chip
 for (ch = 0; ch < NUM_G; ch++)
  emu10k1_writeptr(card, DCYSUSV, ch, 0);
 for (ch = 0; ch < NUM_G; ch++) {
  emu10k1_writeptr(card, VTFT, ch, 0);
  emu10k1_writeptr(card, CVCF, ch, 0);
  emu10k1_writeptr(card, PTRX, ch, 0);
  emu10k1_writeptr(card, CPF, ch, 0);
 }

 // reset recording buffers
 emu10k1_writeptr(card, MICBS, 0, 0);
 emu10k1_writeptr(card, MICBA, 0, 0);
 emu10k1_writeptr(card, FXBS, 0, 0);
 emu10k1_writeptr(card, FXBA, 0, 0);
 emu10k1_writeptr(card, FXWC, 0, 0);
 emu10k1_writeptr(card, ADCBS, 0, ADCBS_BUFSIZE_NONE);
 emu10k1_writeptr(card, ADCBA, 0, 0);
 emu10k1_writeptr(card, TCBS, 0, TCBS_BUFFSIZE_16K);
 emu10k1_writeptr(card, TCB, 0, 0);
 if (card->is_audigy)
  emu10k1_writeptr(card, A_DBG, 0, A_DBG_SINGLE_STEP);
 else
  emu10k1_writeptr(card, DBG, 0, 0x8000);

 // disable channel interrupt
 emu10k1_writeptr(card, CLIEL, 0, 0);
 emu10k1_writeptr(card, CLIEH, 0, 0);
 emu10k1_writeptr(card, SOLEL, 0, 0);
 emu10k1_writeptr(card, SOLEH, 0, 0);

 // disable audio and lock cache
 emu10k1_writefn0(card,HCFG,HCFG_LOCKSOUNDCACHE | HCFG_LOCKTANKCACHE_MASK | HCFG_MUTEBUTTONENABLE);
 emu10k1_writeptr(card, PTB, 0, 0);
}

//--------------------------------------------------------------------------
// mixer (FX)
static void snd_emu10k1_set_spdif_freq(struct emu10k1_card *card,unsigned int rateflags)
{
 uint32_t tmp;
 tmp = emu10k1_readptr(card, A_SPDIF_SAMPLERATE, 0);
 tmp&= ~A_SPDIF_48000;
 tmp&= ~A_SPDIF_96000;
 tmp|= rateflags;
 emu10k1_writeptr(card, A_SPDIF_SAMPLERATE, 0, tmp);
}

static void emu10k1_set_control_gpr(struct emu10k1_card *card, unsigned int addr, int32_t val)
{
 if(card->is_audigy)
  emu10k1_writeptr(card, A_FXGPREGBASE + addr, 0, val);
 else
  emu10k1_writeptr(card, FXGPREGBASE + addr, 0, val);
}

static uint32_t emu10k1_read_control_gpr(struct emu10k1_card *card, unsigned int addr)
{
 uint32_t retval;
 if(card->is_audigy)
  retval=emu10k1_readptr(card, A_FXGPREGBASE + addr, 0);
 else
  retval=emu10k1_readptr(card, FXGPREGBASE + addr, 0);
 return retval;
}

static void fx_init(struct emu10k1_card *card)
{
 unsigned int i, pc = 0;

 if(card->is_audigy){

  emu10k1_writeptr(card, A_DBG, 0, A_DBG_SINGLE_STEP); // stop fx

  for (i = 0; i < 512 ; i++)
   emu10k1_writeptr(card, A_FXGPREGBASE+i,0,0);  // clear gpr

  // Front Output + Master Volume
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_AFRONT_L), 0xc0, A_GPR(8), A_FXBUS(FXBUS_PCM_LEFT));
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_AFRONT_R), 0xc0, A_GPR(9), A_FXBUS(FXBUS_PCM_RIGHT));

  // Digital Front + Master Volume
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_FRONT_L),  0xc0, A_GPR(8), A_FXBUS(FXBUS_PCM_LEFT));
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_FRONT_R),  0xc0, A_GPR(9), A_FXBUS(FXBUS_PCM_RIGHT));

  // Audigy Drive, Headphone out + Master Volume
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_HEADPHONE_L),0xc0,A_GPR(8), A_FXBUS(FXBUS_PCM_LEFT));
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_HEADPHONE_R),0xc0,A_GPR(9), A_FXBUS(FXBUS_PCM_RIGHT));

  // Rear output + Master Volume
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_AREAR_L),  0xc0, A_GPR(8), A_FXBUS(FXBUS_PCM_LEFT));
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_AREAR_R),  0xc0, A_GPR(9), A_FXBUS(FXBUS_PCM_RIGHT));

  // Digital Rear + Master Volume
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_REAR_L),   0xc0, A_GPR(8), A_FXBUS(FXBUS_PCM_LEFT));
  A_OP(iMAC0, A_EXTOUT(A_EXTOUT_REAR_R),   0xc0, A_GPR(9), A_FXBUS(FXBUS_PCM_RIGHT));

  for( ; pc<1024; pc++)
   A_OP(0xf, 0xc0, 0xc0, 0xcf, 0xc0);

  emu10k1_writeptr(card, A_DBG, 0, 0); // start fx

  //Master volume
  i=((float)DEFAULT_PCMVOLUME_AUDIGY*(float)0x7fffffff+50.0)/100.0;
  if(i>0x7fffffff)
   i=0x7fffffff;
  emu10k1_set_control_gpr(card,8,i);
  emu10k1_set_control_gpr(card,9,i);

  //for Audigy, we mute ac97 and use the philips 6 channel DAC instead
  emu10k1_ac97_write(card->iobase, AC97_MASTER_VOL_STEREO, AC97_MUTE);

 }else{
  for (i = 0; i < 256; i++)
   emu10k1_writeptr(card,FXGPREGBASE + i, 0, 0);

  // ac97
  L_OP(iACC3, EXTOUT(EXTOUT_AC97_L), 0x40, 0x40, FXBUS(FXBUS_PCM_LEFT));
  L_OP(iACC3, EXTOUT(EXTOUT_AC97_R), 0x40, 0x40, FXBUS(FXBUS_PCM_RIGHT));

  // digital out
  L_OP(iACC3, EXTOUT(EXTOUT_TOSLINK_L), 0x40, 0x40, FXBUS(FXBUS_PCM_LEFT));
  L_OP(iACC3, EXTOUT(EXTOUT_TOSLINK_R), 0x40, 0x40, FXBUS(FXBUS_PCM_RIGHT));

  // Livedrive, headphone out
  L_OP(iACC3, EXTOUT(EXTOUT_HEADPHONE_L), 0x40, 0x40, FXBUS(FXBUS_PCM_LEFT));
  L_OP(iACC3, EXTOUT(EXTOUT_HEADPHONE_R), 0x40, 0x40, FXBUS(FXBUS_PCM_RIGHT));

  for( ; pc < 512 ; pc++)
   L_OP(iACC3, 0x40, 0x40, 0x40, 0x40);

  emu10k1_writeptr(card, DBG, 0, 0);

  //For the Live we use ac97
  emu10k1_ac97_write(card->iobase, AC97_RESET, 0);
  emu10k1_ac97_read(card->iobase, AC97_RESET);

  emu10k1_ac97_write(card->iobase, AC97_MASTER_VOL_STEREO, 0x0202);
  emu10k1_ac97_write(card->iobase, AC97_PCMOUT_VOL,        0x0202);
  emu10k1_ac97_write(card->iobase, AC97_HEADPHONE_VOL,     0x0202);
  emu10k1_ac97_write(card->iobase, AC97_EXTENDED_STATUS,AC97_EA_SPDIF);
 }
}

//--------------------------------------------------------------------------

static void emu10k1_clear_stop_on_loop(struct emu10k1_card *card, uint32_t voicenum)
{
 if (voicenum >= 32)
  emu10k1_writeptr(card, SOLEH | ((0x0100 | (voicenum - 32)) << 16), 0, 0);
 else
  emu10k1_writeptr(card, SOLEL | ((0x0100 | voicenum) << 16), 0, 0);
}

static uint32_t srToPitch(uint32_t sampleRate)
{
 static uint32_t logMagTable[128] = {
  0x00000, 0x02dfc, 0x05b9e, 0x088e6, 0x0b5d6, 0x0e26f, 0x10eb3, 0x13aa2,
  0x1663f, 0x1918a, 0x1bc84, 0x1e72e, 0x2118b, 0x23b9a, 0x2655d, 0x28ed5,
  0x2b803, 0x2e0e8, 0x30985, 0x331db, 0x359eb, 0x381b6, 0x3a93d, 0x3d081,
  0x3f782, 0x41e42, 0x444c1, 0x46b01, 0x49101, 0x4b6c4, 0x4dc49, 0x50191,
  0x5269e, 0x54b6f, 0x57006, 0x59463, 0x5b888, 0x5dc74, 0x60029, 0x623a7,
  0x646ee, 0x66a00, 0x68cdd, 0x6af86, 0x6d1fa, 0x6f43c, 0x7164b, 0x73829,
  0x759d4, 0x77b4f, 0x79c9a, 0x7bdb5, 0x7dea1, 0x7ff5e, 0x81fed, 0x8404e,
  0x86082, 0x88089, 0x8a064, 0x8c014, 0x8df98, 0x8fef1, 0x91e20, 0x93d26,
  0x95c01, 0x97ab4, 0x9993e, 0x9b79f, 0x9d5d9, 0x9f3ec, 0xa11d8, 0xa2f9d,
  0xa4d3c, 0xa6ab5, 0xa8808, 0xaa537, 0xac241, 0xadf26, 0xafbe7, 0xb1885,
  0xb3500, 0xb5157, 0xb6d8c, 0xb899f, 0xba58f, 0xbc15e, 0xbdd0c, 0xbf899,
  0xc1404, 0xc2f50, 0xc4a7b, 0xc6587, 0xc8073, 0xc9b3f, 0xcb5ed, 0xcd07c,
  0xceaec, 0xd053f, 0xd1f73, 0xd398a, 0xd5384, 0xd6d60, 0xd8720, 0xda0c3,
  0xdba4a, 0xdd3b4, 0xded03, 0xe0636, 0xe1f4e, 0xe384a, 0xe512c, 0xe69f3,
  0xe829f, 0xe9b31, 0xeb3a9, 0xecc08, 0xee44c, 0xefc78, 0xf148a, 0xf2c83,
  0xf4463, 0xf5c2a, 0xf73da, 0xf8b71, 0xfa2f0, 0xfba57, 0xfd1a7, 0xfe8df
 };

 static char logSlopeTable[128] = {
  0x5c, 0x5c, 0x5b, 0x5a, 0x5a, 0x59, 0x58, 0x58,
  0x57, 0x56, 0x56, 0x55, 0x55, 0x54, 0x53, 0x53,
  0x52, 0x52, 0x51, 0x51, 0x50, 0x50, 0x4f, 0x4f,
  0x4e, 0x4d, 0x4d, 0x4d, 0x4c, 0x4c, 0x4b, 0x4b,
  0x4a, 0x4a, 0x49, 0x49, 0x48, 0x48, 0x47, 0x47,
  0x47, 0x46, 0x46, 0x45, 0x45, 0x45, 0x44, 0x44,
  0x43, 0x43, 0x43, 0x42, 0x42, 0x42, 0x41, 0x41,
  0x41, 0x40, 0x40, 0x40, 0x3f, 0x3f, 0x3f, 0x3e,
  0x3e, 0x3e, 0x3d, 0x3d, 0x3d, 0x3c, 0x3c, 0x3c,
  0x3b, 0x3b, 0x3b, 0x3b, 0x3a, 0x3a, 0x3a, 0x39,
  0x39, 0x39, 0x39, 0x38, 0x38, 0x38, 0x38, 0x37,
  0x37, 0x37, 0x37, 0x36, 0x36, 0x36, 0x36, 0x35,
  0x35, 0x35, 0x35, 0x34, 0x34, 0x34, 0x34, 0x34,
  0x33, 0x33, 0x33, 0x33, 0x32, 0x32, 0x32, 0x32,
  0x32, 0x31, 0x31, 0x31, 0x31, 0x31, 0x30, 0x30,
  0x30, 0x30, 0x30, 0x2f, 0x2f, 0x2f, 0x2f, 0x2f
 };

 int i;

 if (sampleRate == 0)
  return 0;

 sampleRate *= 11185;	// Scale 48000 to 0x20002380

 for (i = 31; i > 0; i--) {
  if (sampleRate & 0x80000000) {
   return (uint32_t) (((int32_t) (i - 15) << 20)
                      +logMagTable[0x7f & (sampleRate >> 24)]
                      + (0x7f & (sampleRate >> 17)) * (int)logSlopeTable[0x7f & (sampleRate >> 24)]);
  }
  sampleRate = sampleRate << 1;
 }

 return 0;  // Should never reach this point
}

static uint32_t samplerate_to_linearpitch(uint32_t samplingrate)
{
 samplingrate = (samplingrate << 8) / 375;
 return (samplingrate >> 1) + (samplingrate & 1);
}

#define PITCH_48000 0x00004000
#define PITCH_96000 0x00008000
#define PITCH_85000 0x00007155
#define PITCH_80726 0x00006ba2
#define PITCH_67882 0x00005a82
#define PITCH_57081 0x00004c1c

static uint32_t emu10k1_select_interprom(struct emu10k1_card *card, unsigned int pitch_target)
{
 if(pitch_target==PITCH_48000)
  return CCCA_INTERPROM_0;
 if(pitch_target<PITCH_48000)
  return CCCA_INTERPROM_1;
 if(pitch_target>=PITCH_96000)
  return CCCA_INTERPROM_0;
 if(pitch_target>=PITCH_85000)
  return CCCA_INTERPROM_6;
 if(pitch_target>=PITCH_80726)
  return CCCA_INTERPROM_5;
 if(pitch_target>=PITCH_67882)
  return CCCA_INTERPROM_4;
 if(pitch_target>=PITCH_57081)
  return CCCA_INTERPROM_3;
 return CCCA_INTERPROM_2;
}

//-------------------------------------------------------------------------
//start & stop

static void snd_emu10k1_pcm_init_voice(struct emu10k1_card *card,
                                       unsigned int voice, unsigned int flags,
				       unsigned int start_addr,
				       unsigned int end_addr)
{
 unsigned int ccca_start = 0;
 uint32_t silent_page;
 int vol_left, vol_right;

 if(flags&VOICE_FLAGS_STEREO){
  start_addr >>= 1;
  end_addr >>= 1;
 }
 if(flags&VOICE_FLAGS_16BIT){
  start_addr >>= 1;
  end_addr >>= 1;
 }

 vol_left = vol_right = 255;
 if(flags&VOICE_FLAGS_STEREO){
  if(flags&VOICE_FLAGS_MASTER)
   vol_right = 0;
  else
   vol_left = 0;
 }

 emu10k1_writeptr(card,DCYSUSV, voice, 0);
 emu10k1_writeptr(card,VTFT, voice, 0xffff);
 emu10k1_writeptr(card,CVCF, voice, 0xffff);
 // Stop CA
 // assumption that PT is already 0 so no harm overwritting ???
 emu10k1_writeptr(card,PTRX, voice, (vol_left << 8) | vol_right);
 if(flags&VOICE_FLAGS_MASTER) {
  unsigned int ccis;
  if(flags&VOICE_FLAGS_STEREO){
   emu10k1_writeptr(card, CPF, voice    , CPF_STEREO_MASK);
   emu10k1_writeptr(card, CPF, voice + 1, CPF_STEREO_MASK);
   ccis=28;
  }else{
   emu10k1_writeptr(card, CPF, voice, 0);
   ccis=30;
  }
  if(flags&VOICE_FLAGS_16BIT)
   ccis *= 2;
  ccca_start  = start_addr + ccis;
  //ccca_start |= CCCA_INTERPROM_0;
  ccca_start |= emu10k1_select_interprom(card,card->voice_pitch_target);
  ccca_start |= (flags&VOICE_FLAGS_16BIT)? 0 : CCCA_8BITSELECT;
 }
 emu10k1_writeptr(card,DSL,  voice, end_addr);// | (vol_right<<24)); // ???
 emu10k1_writeptr(card,PSST, voice, start_addr);// | (vol_left<<24)); // ???
 emu10k1_writeptr(card,CCCA, voice, ccca_start);
 // Clear filter delay memory
 emu10k1_writeptr(card,Z1, voice, 0);
 emu10k1_writeptr(card,Z2, voice, 0);
 // invalidate maps
 silent_page = (((uint32_t)card->silentpage) << 1) | MAP_PTI_MASK;
 emu10k1_writeptr(card,MAPA, voice, silent_page);
 emu10k1_writeptr(card,MAPB, voice, silent_page);
 // modulation envelope
 emu10k1_writeptr(card, CVCF,    voice, 0xffff);     // ???
 emu10k1_writeptr(card, VTFT,    voice, 0xffff);     // ???
 emu10k1_writeptr(card, ATKHLDM, voice, 0x7f00);     // was 0
 emu10k1_writeptr(card, DCYSUSM, voice, 0);          // was DCYSUSM_DECAYTIME_MASK
 emu10k1_writeptr(card, LFOVAL1, voice, 0x8000);
 emu10k1_writeptr(card, LFOVAL2, voice, 0x8000);
 emu10k1_writeptr(card, FMMOD,   voice, 0x0000);     // (may be 0x7000)
 emu10k1_writeptr(card, TREMFRQ, voice, 0);
 emu10k1_writeptr(card, FM2FRQ2, voice, 0);
 emu10k1_writeptr(card, ENVVAL,  voice, 0xefff);     // was 0x8000
 // volume envelope
 emu10k1_writeptr(card, ATKHLDV, voice, ATKHLDV_HOLDTIME_MASK | ATKHLDV_ATTACKTIME_MASK);
 emu10k1_writeptr(card, ENVVOL,  voice, 0x8000);
 // filter envelope
 emu10k1_writeptr(card, PEFE_FILTERAMOUNT, voice, 0); // was 0x7f
 // pitch envelope
 emu10k1_writeptr(card, PEFE_PITCHAMOUNT, voice, 0);
}

static void snd_emu10k1_playback_start_voice(struct emu10k1_card *card,
                                               int voice, int flags)
{
 emu10k1_writeptr(card, IFATN, voice, IFATN_FILTERCUTOFF_MASK);
 emu10k1_writeptr(card, VTFT, voice,  0xffff); // ???
 emu10k1_writeptr(card, CVCF, voice,  0xffff); // ???
 emu10k1_clear_stop_on_loop(card, voice);      // ???
 emu10k1_writeptr(card, DCYSUSV, voice, (DCYSUSV_CHANNELENABLE_MASK|0x7f00)); // was 0x7f7f
 emu10k1_writeptr(card, PTRX_PITCHTARGET, voice, card->voice_pitch_target);
 if(flags&VOICE_FLAGS_MASTER) // ???
  emu10k1_writeptr(card, CPF_CURRENTPITCH, voice, card->voice_pitch_target);
 emu10k1_writeptr(card, IP, voice, card->voice_initial_pitch);
}

static void snd_emu10k1_playback_stop_voice(struct emu10k1_card *card, int voice)
{
 emu10k1_writeptr(card,IP, voice, 0);
 emu10k1_writeptr(card,CPF_CURRENTPITCH, voice, 0);
}

static void snd_emu10k1_playback_invalidate_cache(struct emu10k1_card *card, int voice, int flags)
{
 unsigned int i, ccis, cra = 64, cs, sample;

 if(flags&VOICE_FLAGS_STEREO){
  ccis = 28;
  cs = 4;
 }else{
  ccis = 30;
  cs = 2;
 }
 if(flags&VOICE_FLAGS_16BIT)
  sample=0x00000000;
 else{
  sample=0x80808080;
  ccis *= 2;
 }
 for(i = 0; i < cs; i++)
  emu10k1_writeptr(card, CD0 + i, voice, sample);
 // reset cache
 emu10k1_writeptr(card, CCR_CACHEINVALIDSIZE, voice, 0);
 if(flags&VOICE_FLAGS_STEREO)
  emu10k1_writeptr(card, CCR_CACHEINVALIDSIZE, voice + 1, 0);

 emu10k1_writeptr(card, CCR_READADDRESS, voice, cra);
 if(flags&VOICE_FLAGS_STEREO)
  emu10k1_writeptr(card, CCR_READADDRESS, voice + 1, cra);
 // fill cache
 emu10k1_writeptr(card, CCR_CACHEINVALIDSIZE, voice, ccis);
}

//-------------------------------------------------------------------------
static pci_device_s creative_devices[]={
 {"LIVE"  ,PCI_VENDOR_ID_CREATIVE,PCI_DEVICE_ID_CREATIVE_EMU10K1},
 {"AUDIGY",PCI_VENDOR_ID_CREATIVE,PCI_DEVICE_ID_CREATIVE_AUDIGY},
 {NULL,0,0}
};

static void SBLIVE_close(struct audio_info *aui);
static void sblive_select_mixer(struct emu10k1_card *card);

static void SBLIVE_card_info(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;
 char sout[100];
 sprintf(sout,"SBL : SB %s soundcard found on port:%4.4X irq:%d chiprev:%2.2X model:%4.4X",
         (card->is_audigy)? (card->chiprev==4)? "AUDIGY 2":"AUDIGY":"LIVE",
         card->iobase,card->irq,card->chiprev,card->model);
 mrs->pds_textdisplay_printf(sout);
}

static int SBLIVE_adetect(struct audio_info *aui)
{
 struct emu10k1_card *card;

 card=(struct emu10k1_card *)calloc(1,sizeof(struct emu10k1_card));
 if(!card)
  return 0;
 aui->card_private_data=card;

 card->pci_dev=(struct pci_config_s *)calloc(1,sizeof(struct pci_config_s));
 if(!card->pci_dev)
  goto err_adetect;

 if(pcibios_search_devices(&creative_devices,card->pci_dev)!=PCI_SUCCESSFUL)
  goto err_adetect;

 if(card->pci_dev->device_id==PCI_DEVICE_ID_CREATIVE_AUDIGY)
  card->is_audigy=1;

 card->iobase = pcibios_ReadConfig_Dword(card->pci_dev, PCIR_NAMBAR)&0xfff0;
 if(!card->iobase)
  goto err_adetect;
 card->irq    = pcibios_ReadConfig_Byte(card->pci_dev, PCIR_INTR_LN);
 card->chiprev= pcibios_ReadConfig_Byte(card->pci_dev, PCIR_RID);
 card->model  = pcibios_ReadConfig_Word(card->pci_dev, PCIR_SSID);

 if(!snd_emu10k1_buffer_init(aui,card))
  goto err_adetect;

 aui->card_DMABUFF=card->pcmout_buffer;

 snd_emu10k1_hw_init(card);

 sblive_select_mixer(card);

 return 1;

err_adetect:
 SBLIVE_close(aui);
 return 0;
}

static void SBLIVE_close(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;
 if(card){
  if(card->iobase)
   snd_emu10k1_hw_close(card);
  MDma_free_dosmem(card->dm);
  if(card->pci_dev)
   free(card->pci_dev);
  free(card);
  aui->card_private_data=NULL;
 }
}

static void SBLIVE_setrate(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;
 unsigned int dmabufsize;

 if(aui->freq_card<4000)
  aui->freq_card=4000;
 else{
  unsigned int limit=(card->is_audigy)? 96000:48000;
  if(aui->freq_card>limit)
   aui->freq_card=limit;
 }

 aui->chan_card=2;
 aui->bits_card=16;
 aui->card_wave_id=0x0001;

 dmabufsize=MDma_init_pcmoutbuf(aui,card->pcmout_bufsize,EMUPAGESIZE);

 card->voice_initial_pitch = srToPitch(aui->freq_card) >> 8;
 card->voice_pitch_target = samplerate_to_linearpitch(aui->freq_card);

 if(aui->freq_card==44100)
  snd_emu10k1_set_spdif_freq(card,A_SPDIF_44100);
 else
  if(aui->freq_card<=48000 || !card->is_audigy)
   snd_emu10k1_set_spdif_freq(card,A_SPDIF_48000);
  else
   snd_emu10k1_set_spdif_freq(card,A_SPDIF_96000);

 snd_emu10k1_pcm_init_voice(card,0,VOICE_FLAGS_MASTER|VOICE_FLAGS_STEREO|VOICE_FLAGS_16BIT,0,dmabufsize);
 snd_emu10k1_pcm_init_voice(card,1,VOICE_FLAGS_STEREO|VOICE_FLAGS_16BIT,0,dmabufsize);
}

static void SBLIVE_start(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;

 snd_emu10k1_playback_start_voice(card,0,VOICE_FLAGS_MASTER);
 snd_emu10k1_playback_start_voice(card,1,0);
}

static void SBLIVE_stop(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;

 snd_emu10k1_playback_stop_voice(card,0);
 snd_emu10k1_playback_stop_voice(card,1);
}

static unsigned int SBLIVE_getbufpos(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;
 unsigned int bufpos;

 bufpos=emu10k1_readptr(card,CCCA_CURRADDR,0);
 bufpos*=aui->chan_card;
 bufpos*=aui->bits_card>>3;

 if(bufpos<(unsigned int)aui->card_dmasize)
  aui->card_dma_lastgoodpos=bufpos;

 return aui->card_dma_lastgoodpos;
}

static void SBLIVE_clearbuf(struct audio_info *aui)
{
 struct emu10k1_card *card=aui->card_private_data;
 MDma_clearbuf(aui);
 snd_emu10k1_playback_invalidate_cache(card,0,VOICE_FLAGS_STEREO|VOICE_FLAGS_16BIT);
}

//--------------------------------------------------------------------------
//mixer

static void emu10k1_ac97_write(unsigned int iobase,unsigned int reg, unsigned int value)
{
 outb(iobase + AC97ADDRESS, reg);
 outw(iobase + AC97DATA, value);
 //outb(iobase + AC97ADDRESS, AC97_EXTENDED_ID); // ???
}

static unsigned int emu10k1_ac97_read(unsigned int iobase, unsigned int reg)
{
 outb(iobase + AC97ADDRESS, reg);
 return inw(iobase + AC97DATA);
}

static void SBLIVE_writeMIXER(struct audio_info *aui,unsigned int reg, unsigned int val)
{
 struct emu10k1_card *card=aui->card_private_data;
 if(card->is_audigy)
  emu10k1_set_control_gpr(card,reg,val);
 else
  emu10k1_ac97_write(card->iobase,reg,val);
}

static unsigned int SBLIVE_readMIXER(struct audio_info *aui,unsigned int reg)
{
 struct emu10k1_card *card=aui->card_private_data;
 unsigned int retval;
 if(card->is_audigy)
  retval=emu10k1_read_control_gpr(card,reg);
 else
  retval=emu10k1_ac97_read(card->iobase,reg);
 return retval;
}

static aucards_onemixerchan_s sbaudigy_master_vol={2,{{8,0x7fffffff,0,0},{9,0x7fffffff,0,0}}};

static aucards_allmixerchan_s sbaudigy_mixerset={
 &sbaudigy_master_vol,     // master volume
 NULL,                     // master mute switch
 NULL,                     // pcm volume
 NULL,                     // pcm mute switch
 NULL,                     // tone treble
 NULL                      // tone bass
};

static aucards_onemixerchan_s sblive_master_vol={2,{{AC97_MASTER_VOL_STEREO,0x3f,8,SUBMIXCH_INFOBIT_REVERSEDVALUE},{AC97_MASTER_VOL_STEREO,0x3f,0,SUBMIXCH_INFOBIT_REVERSEDVALUE}}};
static aucards_onemixerchan_s sblive_pcm_vol   ={2,{{AC97_PCMOUT_VOL,0x3f,8,SUBMIXCH_INFOBIT_REVERSEDVALUE},{AC97_PCMOUT_VOL,0x3f,0,SUBMIXCH_INFOBIT_REVERSEDVALUE}}};

static aucards_allmixerchan_s sblive_mixerset={
 &sblive_master_vol,       // master volume
 NULL,                     // master mute switch
 &sblive_pcm_vol,          // pcm volume
 NULL,                     // pcm mute switch
 NULL,                     // tone treble
 NULL                      // tone bass
};

one_sndcard_info SBLIVE_sndcard_info={
 "SBL",
 SNDCARD_LOWLEVELHAND|SNDCARD_INT08_ALLOWED,

 NULL,                  // no init
 &SBLIVE_adetect,       // only autodetect
 &SBLIVE_card_info,
 &SBLIVE_start,
 &SBLIVE_stop,
 &SBLIVE_close,
 &SBLIVE_setrate,

 NULL,
 &MDma_writedata,
 &SBLIVE_getbufpos,
 &SBLIVE_clearbuf,
 &MDma_interrupt_monitor,
 NULL,

 &SBLIVE_writeMIXER,
 &SBLIVE_readMIXER,
 NULL
};

static void sblive_select_mixer(struct emu10k1_card *card)
{
 if(card->is_audigy)
  SBLIVE_sndcard_info.card_mixerchans=&sbaudigy_mixerset;
 else
  SBLIVE_sndcard_info.card_mixerchans=&sblive_mixerset;
}

static mpxplay_module_entry_s sbl_module_entry={
 MPXPLAY_DLLMODULETYPE_AUCARD,
 0,
 "SBL",
 MPXPLAY_DLLMODULEVER_AUCARD,
 (void *)&SBLIVE_sndcard_info
};

static mpxplay_dll_entry_s mpxplay_dll_entry_structure={
 MPXPLAY_DLLENTRY_STRUCTURE_VERSION,
 {
  &sbl_module_entry,
  NULL
 }
};

extern void dllstrtr_update_crwdata(unsigned long *cwd);

long __export mpxplay_dll_entrypoint(struct mpxplay_resource_s *p_mrs,unsigned long *crwdata_begin)
{
 mrs=p_mrs;
 //dllstrtr_update_crwdata(crwdata_begin); // SBL module doesn't need this
 return (long)(&mpxplay_dll_entry_structure);
}
