;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; FreeDOS Resident Calculator
; Copyright (c) 2006, 2007 Oleg O. Chukaev
;
; This program is free software; you can redistribute it and/or
; modify it under the terms of the GNU General Public License
; as published by the Free Software Foundation; either version 2
; of the License, or (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
; 02111-1307, USA.
;
; FreeDOS is a trademark of Jim Hall
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Includes
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;
%include	"config.inc"
%include	"fdrc.inc"
;
%include	"beep.inc"
%include	"scancode.inc"
%include	"bios_var.inc"
%include	"dos.inc"
%include	"dos_prt.inc"
%include	"cmdline.inc"
%include	"tsr.inc"
%include	"copyrigh.inc"
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Code
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
		cpu	8086
		section	.text
		org	100h
offset_0100:
;-----------------------------------------------------------------------------
		jmp	install
;-----------------------------------------------------------------------------


;=============================================================================
; int09h_handler
;-----------------------------------------------------------------------------
; Handler for INT 09h. Checks for activation keys and calls main procedure
; (calculator) if these keys was pressed.
; In:	---
; Out:	---
; Modf:	---
; Call:	calculator, beep
; Use:	actkey, run_flag, shiftkey, old_int09h
;
int09h_handler:
		push	ds
		push	ax
		cmp	byte [cs:disable_flag],1
		je	@@to_old_handler
		cmp	byte [cs:run_flag],1
		je	@@to_old_handler
;-----------------------------------------------------------------------------
; Check for activation keys
;-----------------------------------------------------------------------------
		sti
		in	al,60h
		cmp	al,[cs:actkey]
		jne	@@to_old_handler
		xor	ax,ax
		mov	ds,ax
		mov	al,[BIOS_KBD_FLAGS]
		and	al,[cs:shiftkey]
		cmp	al,[cs:shiftkey]
		jne	@@to_old_handler
;-----------------------------------------------------------------------------
; Check for usable video mode. !!! REWRITE ???
;-----------------------------------------------------------------------------
		mov	al,[BIOS_VIDEO_MODE]	;Video mode
		and	al,7fh
		cmp	al,03h			;80x25
		jbe	@@usable
		cmp	al,55h			;132x25
		je	@@usable
		sti				;We must enable ints for `beep'
		call	beep
		call	beep
		cli
		jmp	@@to_old_handler
@@usable:
;-----------------------------------------------------------------------------
		key_received
		send_eoi
;-----------------------------------------------------------------------------
		mov	byte [cs:run_flag],1

		call	calculator

		pop	ax
		pop	ds
		mov	byte [cs:run_flag],0
		iret

@@to_old_handler:
		pop	ax
		pop	ds
		jmp	far [cs:old_int09h]
;=============================================================================
; int2fh_handler
;-----------------------------------------------------------------------------
; Handler for INT 2fh.
; In:	AH -- identifier
;	AL -- function:
;		0 -- installation check
;		1 -- get list of entry points (NOT YET IMPLEMENTED)
;		2 -- enable/disable FDRC
;			BL == 0 -- enable FDRC
;			BL == 1 -- disable FDRC
;		3 -- set coordinate of left side
;			BL -- coord.
;		4 -- set coordinate of top side
;			BL -- coord.
;		5 -- set activation key
;			BL -- key code
;		6 -- set shift key(s)
;			BL -- key(s) code
;
; Out:	function 0:
;		if FDRC is installed:
;			AL == 0ffh
;			DI == MAGICK_CODE
;			SI == version number
;			CX == TSR's CS
;		if FDRC is not installed:
;			---
;	function 1: (NOT YET IMPLEMENTED)
;		DX:AX -> list of entry points
;	function 2, 3, 4, 5, 6:
;		---
;
; Modf:	AX, CX, DI, SI
; Call:	---
; Use:	old_int2fh, ident
;
int2fh_handler:
		cmp	ah,[cs:ident]	;Our TSR?
		je	@@yes		;Yes
		jmp	far [cs:old_int2fh]
@@yes:
		or	al,al		;Installation check?
		jne	@@01		;No
		mov	al,0ffh		;Yes, return AL=0ffh
		push	cs
		pop	cx
		mov	di,MAGICK_CODE	;Identification code
		mov	si,VERSION	;Version number
		iret
@@01:
		cmp	al,FUNC_EN_DIS
		jne	@@02
		mov	[cs:disable_flag],bl
		iret
@@02:
		cmp	al,FUNC_SET_L
		jne	@@03
		mov	[cs:left],bl
		iret
@@03:
		cmp	al,FUNC_SET_T
		jne	@@04
		mov	[cs:top],bl
		iret
@@04:
		cmp	al,FUNC_SET_K
		jne	@@05
		mov	[cs:actkey],bl
		iret
@@05:
		cmp	al,FUNC_SET_S
		jne	@@quit
		mov	[cs:shiftkey],bl
@@quit:
		iret
;=============================================================================
; calculator
;-----------------------------------------------------------------------------
; Main procedure. Draws window, reads keystrokes, performs calculations, etc.
; In:	---
; Out:	---
; Modf:	DS, AX
; Call:	update_window, update_results, xchg_window, set_cursor
; Use:	scr_rows, top, scr_cols, left, old_ss, old_sp, stack_bottom,
;	key_table, fucking_proc
;
calculator:
		mov	ax,cs
		mov	ds,ax
		mov	[old_ss],ss	;Save old SS:SP
		mov	[old_sp],sp
		mov	ss,ax		;Set new stack
		mov	sp,stack_bottom
		push	es
		push	bx
		push	cx
		push	dx
		push	di
		push	si
		push	bp
		cld

		push	ax		;<--- AX == CS
		xor	ax,ax
		mov	es,ax

		mov	al,[es:BIOS_VIDEO_PAGE]
		mov	[video_page],al

		mov	ax,[es:BIOS_PAGE_OFFSET]
		mov	[page_offset],ax

		mov	al,[es:BIOS_SCR_HEIGHT]
		inc	ax
		mov	[scr_rows],al

		sub	al,CALC_HEIGHT
		cmp	al,[top]
		jae	@@top_ok
		mov	byte [top],0
@@top_ok:
		mov	ax,[es:BIOS_SCR_WIDTH]
		mov	[scr_cols],ax

		sub	al,CALC_WIDTH
		cmp	al,[left]
		jae	@@left_ok
		mov	byte [left],0
@@left_ok:
		mov	word [video_seg],COLOR_SEG
		pop	es		;<--- ES := CS
;-----------------------------------------------------------------------------
		mov	ah,03h		;Get cursor position and shape
		mov	bh,[video_page]	;Video page
		int	10h
		push	cx
		push	dx
;-----------------------------------------------------------------------------
		cmp	byte [regs_cleared],0
		jne	@@skip_reg_clr
		mov	di,reg_table
		mov	cx,2*10
		xor	ax,ax
	rep	stosw
		mov	byte [regs_cleared],1
@@skip_reg_clr:
;-----------------------------------------------------------------------------
		cmp	byte [autoclear],0
		je	@@skip_clear
		or	byte [autoclear],1
@@skip_clear:
;-----------------------------------------------------------------------------
		call	update_window
@@next_key:
		cmp	byte [quit_flag],0
		jne	@@esc
		int	28h
		mov	ah,1
		int	16h
		jz	@@next_key
		mov	ah,0
		int	16h
;-----------------------------------------------------------------------------
		xchg	ax,bx		;OPTIMIZE: instead of MOV BX,AX
		mov	si,key_table-2
@@next:
		lodsw			;Skip address
		lodsw
		or	ax,ax
		jz	@@found		;Found last proc -- insert_char
		cmp	ax,bx
		jne	@@next		;Any other proc clears flag...
		and	byte [autoclear],~1
@@found:
;-----------------------------------------------------------------------------
		call	[si]
		call	word [fucking_proc]
		call	update_results
		jmp	@@next_key
@@esc:
		mov	byte [quit_flag],0
		call	xchg_window	;Restore window
;-----------------------------------------------------------------------------
		pop	dx
		pop	cx
		call	set_cursor	;Set cursor position
;-----------------------------------------------------------------------------
		pop	bp
		pop	si
		pop	di
		pop	dx
		pop	cx
		pop	bx
		pop	es
		mov	ss,[old_ss]	;Restore old stack
		mov	sp,[old_sp]
		ret
;=============================================================================
; kbd_out_dec
;-----------------------------------------------------------------------------
; Writes decimal form of result to keyboard buffer and sets quit_flag.
; In:	---
; Out:	---
; Modf:	AX, SI
; Call:	get_addr, stuff_key
; Use:	quit_flag
;
kbd_out_dec:
		push	es
		push	ds
		mov	ax,0101h	;See print_table
		call	get_addr
		push	es
		pop	ds
		mov	si,ax

		lodsw
		cmp	al,'-'
		jne	@@loop
		mov	ax,KEY_MINUS
		call	stuff_key

@@loop:
		lodsw
		cmp	al,' '
		je	@@loop
		cmp	ah,COL_LETTER	;Color of letter `u' or `d'
		je	@@quit
		mov	ah,0
		call	stuff_key
;		loop	@@loop		;Why LOOP ???
		jmp	@@loop

@@quit:
		pop	ds
		pop	es
		mov	byte [quit_flag],1
		ret
;=============================================================================
; kbd_out_hex
;-----------------------------------------------------------------------------
; Writes hexadecimal form of result to keyboard buffer and sets quit_flag.
; In:	---
; Out:	---
; Modf:	AX
; Call:	stuff_keys
; Use:	---
;
kbd_out_hex:
		mov	ax,0110h	;See print_table
		jmp	stuff_keys
;=============================================================================
; kbd_out_bin_{1,2,3,4}
;-----------------------------------------------------------------------------
; Writes binary form of result to keyboard buffer and sets quit_flag.
; In:	---
; Out:	---
; Modf:	AX
; Call:	stuff_keys
; Use:	---
;
kbd_out_bin_1:
		mov	ax,0201h	;See print_table
		jmp	stuff_keys
kbd_out_bin_2:
		mov	ax,0209h	;See print_table
		jmp	stuff_keys
kbd_out_bin_3:
		mov	ax,0211h	;See print_table
		jmp	stuff_keys
kbd_out_bin_4:
		mov	ax,0219h	;See print_table
;		jmp	stuff_keys
;=============================================================================
; stuff_keys
;-----------------------------------------------------------------------------
; Writes 8 pairs of char/scancode to keyboard buffer and sets quit_flag.
; All scancodes are set to 0!
; In:	AX -- coordinates of string
; Out:	---
; Modf:	AX, CX, SI
; Call:	stuff_key
; Use:	quit_flag
;
stuff_keys:
		push	es
		push	ds
		call	get_addr
		push	es
		pop	ds
		mov	si,ax
		mov	cx,8
@@loop:
		lodsw
		mov	ah,0
		call	stuff_key
		loop	@@loop

		pop	ds
		pop	es
		mov	byte [quit_flag],1
		ret
;=============================================================================
; stuff_key
;-----------------------------------------------------------------------------
; Writes pair of char/scancode to keyboard buffer.
; In:	AX -- char/scancode
; Out:	---
; Modf:	---
; Call:	---
; Use:	some BIOS variables
;
stuff_key:
		push	ds
		push	bx
		push	di
		mov	bx,40h			;Segment of BIOS variables
		mov	ds,bx
		mov	bx,[BIOS_KBD_TAIL_]	;Tail
		mov	di,bx			;Save it
		inc	bx
		inc	bx
		cmp	bx,[BIOS_KBD_BUF_END_]	;End of buffer reached?
		jne	@@next			;No, skip
		mov	bx,[BIOS_KBD_BUF_START_];Else BX = start of buffer
@@next:
		cmp	bx,[BIOS_KBD_HEAD_]	;BX = Buffer Head?
		je	@@quit
		mov	[di],ax			;Stuff scan code, char in buf
		mov	[BIOS_KBD_TAIL_],bx	;and update buffer tail
@@quit:
		pop	di
		pop	bx
		pop	ds
		ret
;=============================================================================
; set_quit_flag
;-----------------------------------------------------------------------------
; Sets quit_flag.
; In:	---
; Out:	---
; Modf:	---
; Call:	---
; Use:	quit_flag
;
set_quit_flag:
		mov	byte [quit_flag],1
		ret
;=============================================================================
; change_sign
;-----------------------------------------------------------------------------
; Changes sign of result representation.
; In:	---
; Out:	---
; Modf:	---
; Call:	---
; Use:	sign_flag, byte [print_table+4]
;
change_sign:
		xor	byte [print_table+4],'u' ^ 'd'
		xor	byte [sign_flag],1
		ret
;=============================================================================
; enter_handler, enter_handler_2
;-----------------------------------------------------------------------------
; enter_handler called when ENTER pressed. Stores string in history,
; stores '\0' at the end of string in buffer, then calls calculation procedure.
; enter_handler_2 called after each entered digit or letter if switch -a
; specified on command line.
; In:	---
; Out:	---
; Modf:	~AX, BX, ~CX, ~DX, ~DI, ~SI
; Call:	calculate
; Use:	eol_ptr
;
enter_handler:
;		cmp	word [hist_end],history_buffer
;		je	enter_handler_2
		call	put_string
enter_handler_2:
		mov	bx,[eol_ptr]
		mov	byte [bx],0
		call	calculate
return:
		ret
;=============================================================================
; ins_toggle
;-----------------------------------------------------------------------------
; Switches between insert and overstrike modes.
; In:	---
; Out:	---
; Modf:	---
; Call:	---
; Use:	ins_flag
;
ins_toggle:
		xor	byte [ins_flag],1
		ret
;=============================================================================
; move_left
;-----------------------------------------------------------------------------
; Moves calculator's window left.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	left
;
move_left:
		cmp	byte [left],0
		je	return
		call	xchg_window
		dec	byte [left]
		jmp	update_window
;=============================================================================
; move_right
;-----------------------------------------------------------------------------
; Moves calculator's window right.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	left, scr_cols
;
move_right:
		mov	al,[scr_cols]
		sub	al,CALC_WIDTH
		cmp	[left],al
		jae	return
		call	xchg_window
		inc	byte [left]
		jmp	update_window
;=============================================================================
; move_up
;-----------------------------------------------------------------------------
; Moves calculator's window up.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	top
;
move_up:
		cmp	byte [top],0
		je	return
		call	xchg_window
		dec	byte [top]
		jmp	update_window
;=============================================================================
; move_down
;-----------------------------------------------------------------------------
; Moves calculator's window down.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	xchg_window, jumps to `update_window' and `return'
; Use:	top, scr_rows
;
move_down:
		mov	al,[scr_rows]
		sub	al,CALC_HEIGHT
		cmp	[top],al
		jae	return
		call	xchg_window
		inc	byte [top]
		jmp	update_window
;=============================================================================
; update_window
;-----------------------------------------------------------------------------
; Updates calculator's window.
; In:	---
; Out:	---
; Modf:	~AX, ~BX, ~CX, ~DX, ~DI, ~SI
; Call:	xchg_window, draw_window, update_results
; Use:	---
;
update_window:
		call	xchg_window
		call	draw_window
		call	update_results
		ret
;=============================================================================
; update_results
;-----------------------------------------------------------------------------
; Updates results and input line, prints error message (if necessary).
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_addr
; Use:	video_seg, print_table, top_left, error_code
;
update_results:
		push	es
		mov	si,print_table
@@loop01:
		lodsw			;Shift from top_left
		call	get_addr
		lodsw			;Procedure address
		push	si
		call	ax
		pop	si
		lodsw			;Char and it's color
		or	al,al
		jz	@@quit
		stosw
		jmp	@@loop01
@@quit:
		mov	[error_code],al	;OPTIMIZE: AL instead of `ERR_ALL_OK'
		pop	es
		ret
;=============================================================================
; xchg_window
;-----------------------------------------------------------------------------
; Exchanges rectangular area of screen and buffer.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_addr
; Use:	scr_buf, scr_cols
;
xchg_window:
		push	es
		xor	ax,ax
		call	get_addr
		mov	bx,CALC_HEIGHT
		mov	si,scr_buf

@@next_line:
		mov	di,ax
		mov	cx,CALC_WIDTH
@@loop:
		mov	dx,[si]		;|
		xchg	dx,[es:di]	;| rep xchg word [es:di],[ds:si] ;-)
		mov	[si],dx		;|
		cmpsw			;OPTIMIZE: instead of (INC DI; INC SI)*2
		loop	@@loop

		add	ax,[scr_cols]
		add	ax,[scr_cols]
		dec	bx
		jnz	@@next_line

		pop	es
		ret
;=============================================================================
; draw_window
;-----------------------------------------------------------------------------
; Draws calculator's window.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_addr
; Use:	video_seg, top_left, scr_cols, calc_title
;
draw_window:
		push	es
		xor	ax,ax
		call	get_addr
		mov	dx,-CALC_WIDTH
		add	dx,[scr_cols]
		shl	dx,1

		mov	ax,COL_BG*100h+CHAR_TOP_LEFT
		stosw

		mov	si,calc_title
		mov	ah,COL_TITLE
		mov	cx,TITLE_LENGTH
@@loop00:
		lodsb
		stosw
		loop	@@loop00

		mov	cl,CALC_WIDTH - TITLE_LENGTH - 2
		mov	ax,COL_BG*100h+CHAR_TOP
	rep	stosw
		mov	al,CHAR_TOP_RIGHT
		stosw
		add	di,dx

		mov	bx,CALC_HEIGHT-2
@@loop01:
		mov	al,CHAR_LEFT
		stosw
		mov	cl,CALC_WIDTH-2	;OPTIMIZE: CL instead of CX
		mov	al,' '
	rep	stosw
		mov	al,CHAR_RIGHT
		stosw
		add	di,dx
		dec	bx
		jnz	@@loop01

		mov	al,CHAR_BOTT_LEFT
		stosw
		mov	cl,CALC_WIDTH-2	;OPTIMIZE: CL instead of CX
		mov	al,CHAR_BOTTOM
	rep	stosw
		mov	al,CHAR_BOTT_RIGHT
		stosw

		pop	es
		ret
;=============================================================================
; parse
;-----------------------------------------------------------------------------
; Parses input buffer using the following table.
;
;		0    1    2    3    4    5    6    7    8    9    10   11
;		0-1  2-9  a-f  \.   @    op   :    $    \"   ' '  eol  *
; -------------------------------------------------------------------------
; eol,0		exit exit exit exit exit exit exit exit exit exit exit exit
; op,1		exit exit exit exit exit exit exit exit exit exit exit exit
; endhex,2	exit exit exit exit exit exit exit exit exit exit exit exit
; enddec,3	err  err  err  err  err  exit exit err  err  exit exit err
; endbin,4	err  err  err  err  err  exit exit err  err  exit exit err
; endstr,5	err  err  err  err  err  exit exit err  str  exit exit err
; endreg,6	err  err  err  err  err  exit exit err  err  exit exit err
; end:,7	err  err  err  err  err  exit exit err  err  exit exit err
; start,8	bin  dec  hex  err  err  op   :reg reg  str  err  eol  err
; hex,9		hex  hex  hex  err  err  endh endh err  err  endh endh err
; dec,10	dec  dec  hex  endd err  endh endh err  err  endh endh err
; bin,11	bin  dec  hex  endd endb endh endh err  err  endh endh err
; str,12	str  str  str  str  str  str  str  str  ends str  err  str
; reg,13	endr endr err  err  err  err  err  err  err  err  err  err
; :reg,14	end: end: err  err  err  err  err  err  err  err  err  err
; exit,15
;
; In:	SI -> string
; Out:	SI -> next char after token
;	CF clear if no errors
;	CF set if error
;	AX -- type of token (finite automaton state)
; Modf:	AX, BX, CX, SI
; Call:	type_of_char
; Use:	state_table, error_offs, error_code
;
parse:
		mov	cx,P_START*100h+4	;CH -- initial state
@@next_char:					;CL == 4 for shifts
		mov	al,6			;Length of record in state_table
		mul	ch
		add	ax,state_table
		xchg	ax,bx			;BX -> state record

		lodsb
		call	type_of_char
		shr	ax,1
		xlat
		jnc	@@shr
		shl	al,cl
@@shr:
		shr	al,cl			;AL -- new state
		cmp	al,P_EX			;Exit?
		je	@@quit			;Yes
		mov	ch,al			;Save new state
		cmp	al,P_ERR		;Error?
		jne	@@next_char		;No, get next char...

		dec	si
		mov	[error_offs],si
		mov	byte [error_code],ERR_PARSE
		stc
		ret
@@quit:
		mov	al,ch
		cbw
		clc
		ret
;=============================================================================
; type_of_char
;-----------------------------------------------------------------------------
; Determines type of char.
; In:	AL -- char
; Out:	AX -- type of char:
;	0	0-1		1	2-9
;	2	a-f		3	\.
;	4	@		5	operator
;	6	:		7	$
;	8	\"		9	' '
;	10	eol		11	* (any other character)
;
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
type_of_char:
		push	si
		mov	dx,-1
		mov	ah,al
		mov	si,chars
@@underscore:
		inc	dx
@@next:
		lodsb
		cmp	al,'_'
		je	@@underscore
		cmp	al,ah
		je	@@quit
		or	al,al
		jnz	@@next
		inc	dx
@@quit:
		xchg	ax,dx
		pop	si
		ret
;=============================================================================
; calculate
;-----------------------------------------------------------------------------
; Parses input buffer and calculates result.
; In:	---
; Out:	---
; Modf:	AX, BX, ~CX, ~DX, ~DI, SI
; Call:	parse
; Use:	input_buf, proc_table, math_sp
;
calculate:
		mov	si,input_buf-1
@@next:
		inc	si
		cmp	byte [si],' '	;Skip spaces
		je	@@next

		push	si		;parse destroys SI
		call	parse
		pop	si
		jc	@@quit		;Quit if parse error
		shl	ax,1
		xchg	bx,ax		;OPTIMIZE: instead of MOV BX,AX
		call	[proc_table+bx]	;Retrieve token from input buffer
		dec	si		;CF not destroyed by DEC
		jnc	@@next		;Next token if no errors
@@quit:
		mov	word [math_sp],math_stack	;Flush stack
		ret
;=============================================================================
; op_handler
;-----------------------------------------------------------------------------
; Called when parser encountered operator in the input line.
; In:	SI -> operator
;	math stack: 2 values on top
; Out:	SI -> next char
;	math stack: result on top
;	CF clear if no errors
;	CF set if error
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	pop_dw, push_dw
; Use:	error_offs, error_code, op_table
;
op_handler:
		mov	[error_offs],si			;SI -> operator
		mov	byte [error_code],ERR_ALL_OK	;Assume no errors
		lodsb					;Get operator
		cmp	al,';'
		je	@@flush

		mov	di,op_table-2
@@next:
		scasw
		scasb			;Find operator in table
		ja	@@next

		cmp	al,'\'		;Handle unary operators: `sqrt',
		je	@@pop
		cmp	al,'~'		;`not',
		je	@@pop
		cmp	al,'`'		;and `neg'
		je	@@pop

		call	pop_dw
		jc	@@quit
		mov	cx,dx
		xchg	bx,ax		;OPTIMIZE: instead of MOV BX,AX
@@pop:
		call	pop_dw
		jc	@@quit
		call	word [di]
		call	push_dw
		cmp	byte [error_code],ERR_ALL_OK
		je	@@quit
		stc
@@quit:
		ret
@@flush:
		mov	word [math_sp],math_stack	;Flush stack
		ret
;=============================================================================
; colon_handler
;-----------------------------------------------------------------------------
; Called when parser encountered operator in the input line.
; In:	SI -> operator
;	math stack: 1 value on top
; Out:	SI -> next char
;	math stack: result on top
;	CF clear if no errors
;	CF set if error
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	pop_dw, push_dw
; Use:	reg_table
;
colon_handler:
		mov	[error_offs],si
		lodsw
		sub	ah,'0'
		mov	cl,6
		shr	ax,cl
		add	ax,reg_table
		xchg	ax,bx

		call	pop_dw
		jc	@@quit
		mov	[bx],ax
		mov	[bx+2],dx
		call	push_dw
@@quit:
		ret
;=============================================================================
; eol_handler
;-----------------------------------------------------------------------------
; Called when parser encountered end of line.
; In:	---
; Out:	CF is _always_ set
; Modf:	AX, DX
; Call:	pop_dw
; Use:	math_sp, result, error_code
;
eol_handler:
		call	pop_dw
		jnc	@@ok
		xor	ax,ax
		cwd
@@ok:
		mov	[result],ax
		mov	[result+2],dx
		mov	al,ERR_ALL_OK
		cmp	word [math_sp],math_stack
		je	@@quit
		mov	al,ERR_NOT_EMPTY
@@quit:
		mov	[error_code],al
		stc
		ret


;=============================================================================
%include	"math.inc"
;=============================================================================


;=============================================================================
; push_dw
;-----------------------------------------------------------------------------
; Saves double word on the top of math stack.
; In:	DX:AX -- double word to push
; Out:	---
; Modf:	---
; Call:	---
; Use:	math_sp
;
push_dw:
		push	bx
		mov	bx,[math_sp]
		mov	[bx],ax
		mov	[bx+2],dx
		add	word [math_sp],4
		pop	bx
		clc
		ret
;=============================================================================
; pop_dw
;-----------------------------------------------------------------------------
; Stores in DX:AX double word from the top of math stack.
; In:	---
; Out:	DX:AX -- double word
;	CF clear if no errors
;	CF set and error_code == ERR_UNDERFLOW if attempt to pop from
;		empty stack
; Modf:	!AX, !DX
; Call:	---
; Use:	math_sp, error_code
;
pop_dw:
		cmp	word [math_sp],math_stack
		jbe	@@error
		push	bx
		sub	word [math_sp],4
		mov	bx,[math_sp]
		mov	dx,[bx+2]
		mov	ax,[bx]
		pop	bx
		clc
		ret
@@error:
		mov	byte [error_code],ERR_UNDERFLOW
		stc
		ret
;=============================================================================
; show_status
;-----------------------------------------------------------------------------
; Prints status message.
; In:	error code:
;		0 -- no errors
;		1 -- 'Parse error'
;		2 -- 'Stack underflow'
;		3 -- 'Stack not empty'
;		4 -- 'Overflow'
;	ES:DI -> video memory
; Out:	---
; Modf:	AX, CX, SI, DI
; Call:	print_hex
; Use:	---
;
show_status:
		mov	ax,MSG_ERR_LENGTH
		mov	cx,ax
		mul	byte [error_code]
		add	ax,error_messages
		xchg	ax,si
		mov	ah,COL_ERRMSG
@@loop00:
		lodsb
		stosw
		loop	@@loop00

		add	di,(CALC_WIDTH-3-MSG_ERR_LENGTH)*2
		mov	cx,COL_BG*256+04h	;color / diamond
		cmp	word [hist_top],history_buffer
		je	@@stosw
		mov	cl,19h			;down arrow
		mov	ax,[hist_cur]
		cmp	ax,history_buffer
		je	@@stosw
		mov	cl,18h			;up arrow
		cmp	ax,[hist_top]
		je	@@stosw
		mov	cl,12h			;up/down arrow
@@stosw:
		xchg	ax,cx
		stosw
		ret
;=============================================================================
; out_dec
;-----------------------------------------------------------------------------
; Outputs result in decimal form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	~AX, ~BX, CX, ~DX, DI, SI
; Call:	ltoa
; Use:	---
;
out_dec:
		call	ltoa
		mov	cx,11
	rep	movsw
		ret
;=============================================================================
; out_bin
;-----------------------------------------------------------------------------
; Outputs result in binary form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
out_bin:
		mov	ax,[result+2]
		call	print_bin
		mov	ax,[result]
print_bin:
		xchg	ax,dx
		mov	bx,00ffh
@@loop01:
		mov	ax,COL_NUM*100h
		shl	bx,1
		adc	ah,al
		rcl	dx,1
		adc	al,'0'
		stosw
		or	bx,bx
		jnz	@@loop01
		ret
;=============================================================================
; out_hex
;-----------------------------------------------------------------------------
; Outputs result in hexadecimal form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, ~CX, ~DI
; Call:	print_hex
; Use:	result
;
out_hex:
		mov	bp,print_hex
print_color:
		mov	si,result+3
		mov	bl,01010000b
@@loop:
		mov	ah,COL_NUM
		shl	bl,1
		adc	ah,0
		mov	al,[si]
		dec	si
		call	bp
		or	bl,bl
		jnz	@@loop
		ret
;=============================================================================
; out_char
;-----------------------------------------------------------------------------
; Outputs result in string form.
; In:	dword [result] -- number to print
;	ES:DI -> video buffer
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
out_char:
		mov	ax,COL_LETTER*100h+'"'
		stosw
		mov	bp,@@print_char
		jmp	print_color
@@print_char:
		stosw
		ret
;=============================================================================
; print_hex
;-----------------------------------------------------------------------------
; Prints hex value of AL
; In:	AL -- byte
;	AH -- color
;	ES:DI -> place in video RAM
; Out:	---
; Modf:	AX, CX, DI
; Call:	---
; Use:	---
;
print_hex:
		mov	cl,4
@@next:
		push	ax
		shr	al,cl
		and	al,0fh
		cmp	al,10
		sbb	al,69h
		das
		stosw
		pop	ax
		sub	cl,4
		jz	@@next
		ret
;=============================================================================
; get_addr
;-----------------------------------------------------------------------------
; Returns address of char in video buffer
; In:	AH/AL -- additions to top/left row/column of char
; Out:	ES:AX, ES:DI -- address
; Modf:	!AX, !DI, !ES
; Call:	---
; Use:	---
;
get_addr:
		add	ax,[top_left]
		push	bx
		xchg	bx,ax
		mov	al,[scr_cols]
		mul	bh
		mov	bh,0
		add	ax,bx
		pop	bx
		shl	ax,1
		add	ax,[page_offset]
		mov	di,ax
		mov	es,[video_seg]
		ret
;=============================================================================
; update_line
;-----------------------------------------------------------------------------
; Updates input line.
; In:	ES:DI -> videobuffer
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	start_ptr, end_ptr, cur_ptr, autoclear, error_code, error_offs, top_left
;
update_line:
		mov	si,[start_ptr]
@@loop01:
		cmp	si,[end_ptr]
		je	@@end_loop
;-----------------------------------------------------------------------------
; Select color for input line.
;-----------------------------------------------------------------------------
		mov	ah,COL_INPUT
		test	byte [autoclear],1
		jz	@@using_col_input
		mov	ah,COL_INPUT0
@@using_col_input:
;-----------------------------------------------------------------------------
; Check for errors.
;-----------------------------------------------------------------------------
		cmp	byte [error_code],ERR_ALL_OK
		je	@@no_parse_error
		cmp	byte [error_code],ERR_NOT_EMPTY
		je	@@no_parse_error
		cmp	si,[error_offs]
		jne	@@no_parse_error
		mov	ah,COL_ERRCHR
@@no_parse_error:
;-----------------------------------------------------------------------------
		lodsb
		stosw
		jmp	@@loop01
@@end_loop:
		mov	cx,[start_ptr]
		add	cx,LEN+1
		sub	cx,[end_ptr]
		jle	@@skip_fill
		mov	ax,COL_FILL*100h+INPUT_FILL_CHAR
	rep	stosw			;Fill input line
@@skip_fill:
		mov	ax,[cur_ptr]
		sub	ax,[start_ptr]
		add	ax,[top_left]
		add	ax,0301h
		xchg	ax,dx
		mov	cx,0e0fh	;Cursor like `_'
		test	byte [ins_flag],1
		jz	set_cursor
		mov	cx,000dh	;Cursor like `'
;-----------------------------------------------------------------------------
; set_cursor -- sets cursor shape (CX) and position (DX)
;-----------------------------------------------------------------------------
set_cursor:
		mov	ah,02h
		mov	bh,[video_page]
		int	10h		;Set cursor position
		mov	ah,01h
		int	10h		;Set cursor shape
		ret


;=============================================================================
%include	"edit.inc"
;=============================================================================


;=============================================================================
; history_prev
;-----------------------------------------------------------------------------
; Gets previous line from history.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_string, beep
; Use:	current_item, num_items
;
history_prev:
		mov	ax,[hist_cur]
		cmp	ax,history_buffer
		je	@@quit
		mov	si,ax
		sub	ax,[si-2]
		sub	ax,4
		mov	[hist_cur],ax
		jmp	get_string	;call get_string / ret
@@quit:
		jmp	beep		;call beep / ret
;=============================================================================
; history_next
;-----------------------------------------------------------------------------
; Gets next line from history.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	get_string, beep
; Use:	current_item
;
history_next:
		mov	ax,[hist_cur]
		cmp	ax,[hist_top]
		je	@@quit
		mov	si,ax
		add	ax,[si]
		add	ax,4
		mov	[hist_cur],ax
		jmp	get_string	;call get_string / ret
@@quit:
		jmp	beep		;call beep / ret
;=============================================================================
; history_first
;-----------------------------------------------------------------------------
; Gets first line from history.
; In:	---
; Out:	---
; Modf:	---
; Call:	get_string, beep
; Use:	current_item
;
history_first:
		mov	ax,history_buffer
		cmp	[hist_cur],ax
		je	@@quit
		mov	[hist_cur],ax
		jmp	get_string	;call get_string / ret
@@quit:
		jmp	beep		;call beep / ret
;=============================================================================
; history_last
;-----------------------------------------------------------------------------
; Gets last line from history.
; In:	---
; Out:	---
; Modf:	AX
; Call:	get_string, beep
; Use:	current_item
;
history_last:
		mov	ax,[hist_top]
		cmp	[hist_cur],ax
		je	@@quit
		mov	[hist_cur],ax
		jmp	get_string	;call get_string / ret
@@quit:
		jmp	beep		;call beep / ret
;=============================================================================
; put_string
;-----------------------------------------------------------------------------
; Stores current line to history buffer.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	history_buffer, hist_items, num_items, eol_ptr, input_buf
;
put_string:
		mov	ax,[eol_ptr]
		sub	ax,input_buf
		jz	@@quit		;Exit if size of input string is 0

		mov	di,[hist_top]
@@loop:
		mov	bx,[hist_end]
		cmp	bx,history_buffer
		je	@@quit		;Exit if size of history buffer is 0

		sub	bx,di
		sub	bx,4		;For header and trailer
		cmp	bx,ax		;AX -- length of string
		jge	@@enough

		mov	cx,di
		mov	di,history_buffer
		lea	si,[di+4]
		add	si,[di]
		sub	cx,si
		jcxz	@@skip
	rep	movsb
@@skip:
		jmp	@@loop

@@enough:
		mov	si,input_buf
		stosw			;Store length in header
		mov	cx,ax
	rep	movsb
		stosw			;Store length in trailer
		mov	[hist_top],di
		mov	[hist_cur],di
@@quit:
		ret
;=============================================================================
; get_string
;-----------------------------------------------------------------------------
; Copies line from history to buffer.
; In:	---
; Out:	---
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	clear_str
; Use:	result, current_item, history_buffer, input_buf, eol_ptr, end_ptr
;
get_string:
		call	clear_str
		mov	word [result],0
		mov	word [result+2],0

		mov	di,input_buf
		mov	si,[hist_cur]
		cmp	si,[hist_top]
		je	@@quit		;Exit if current string is the last one

		lodsw
		mov	cx,ax		;Empty strings are not allowed,
	rep	movsb			;jcxz instruction is omitted
		add	[eol_ptr],ax
		cmp	ax,LEN
		jb	@@001
		mov	ax,LEN
@@001:
		add	[end_ptr],ax
@@quit:
		ret
;=============================================================================
; ltoa
;-----------------------------------------------------------------------------
; !!! Converts dword [result] to 22-bytes string, 1st byte is `-' or ` '.
; Puts spaces to bytes before digits.
; In:	---
; Out:	SI -> ltoa_buf
; Modf:	AX, BX, DX, CX, !SI
; Call:	---
; Use:	ltoa_buf, result
;
ltoa:
		mov	si,ltoa_buf-2
		push	si
		mov	cx,11

@@fillbuf:
		lodsw			;We can't use stosb because ES:DI
		mov	byte [si],' '	;points to video memory
		loop	@@fillbuf	;Fill w/ spaces

		mov	ax,[result]
		mov	dx,[result+2]

		cmp	[sign_flag],cl	;OPTIMIZE: CL instead of 0
		je	@@unsigned
		or	dx,dx
		jns	@@unsigned
		neg	dx
		neg	ax
		sbb	dx,cx		;OPTIMIZE: CX instead of 0
		mov	byte [si-20],'-'
@@unsigned:
		mov	cl,10
@@next_digit:
		mov	bx,ax
		mov	ax,dx
		xor	dx,dx
		div	cx
		xchg	ax,bx
		div	cx
		xchg	dx,bx
		add	bl,'0'
		mov	[si],bl		;Store digit
		dec	si
		dec	si
		mov	bx,dx
		or	bx,ax
		jnz	@@next_digit

		pop	si
		lodsw
		ret
;=============================================================================
; atol
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word.
; In:	SI -> string
;	BP -> checking procedure that returns CY if BL is not a valid
;		digit, NC and BL == number if BL is a valid digit
; Out:	DX:AX -- number
;	SI -> 2 chars after number
; Modf:	!AX, !DX, SI
; Call:	---
; Use:	BP -> procedure
;
atol:
		xor	ax,ax
		xor	bx,bx
		xor	di,di
@@loop:
		xchg	ax,di		;Mul DI:AX by base (CX)
		mul	cx
		xchg	ax,di
		mul	cx
		add	ax,bx		;Char, converted to digit
		adc	di,dx
		mov	bl,[si]		;Load char
		inc	si
		call	bp		;Check char and get base
		jnc	@@loop

		mov	dx,di
		ret			;Return
;=============================================================================
; atolx
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word. String contains digits in base 16.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI, BP
; Call:	atol, push_ddw
; Use:	---
;
atolx:
		mov	bp,ishexdigit
		call	atol
		call	push_dw
		dec	si
		ret
;=============================================================================
; atold
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word. String contains digits in base 10.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI, BP
; Call:	atol, push_ddw
; Use:	---
;
atold:
		mov	bp,isdecdigit
		call	atol
		call	push_dw
		ret
;=============================================================================
; atolb
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word. String contains digits in base 2.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI, BP
; Call:	atol, push_ddw
; Use:	---
;
atolb:
		mov	bp,isbindigit
		call	atol
		call	push_dw
		ret
;=============================================================================
; atols
;-----------------------------------------------------------------------------
; Converts ASCII-string to double word.
; In:	SI -> string
; Out:	value on top of math_stack
; Modf:	AX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
atols:
		push	bx
		xor	bx,bx
		xor	dx,dx
		lodsb			;Skip '"'
@@next:
		lodsb
		cmp	al,'"'
		jne	@@shift
		lodsb
		cmp	al,'"'
		jne	@@quit
@@shift:
		mov	dh,dl
		mov	dl,bh
		mov	bh,bl
		mov	bl,al
		jmp	@@next
@@quit:
		dec	si
		xchg	ax,bx		;OPTIMIZE: instead of MOV AX,BX
		call	push_dw
		pop	bx
		ret
;=============================================================================
; atolr
;-----------------------------------------------------------------------------
; Converts ASCII-string (reg. name) to double word.
; In:	SI -> string (1st byte -- '$', 2nd byte -- '0' .. '9')
; Out:	value on top of math_stack
; Modf:	AX, BX, CX, DX, DI, SI
; Call:	---
; Use:	---
;
atolr:
		lodsb
		lodsb
		sub	al,'0'
		shl	al,1
		shl	al,1
		cbw
		add	ax,reg_table
		xchg	ax,bx
		mov	ax,[bx]
		mov	dx,[bx+2]
		call	push_dw
		ret
;=============================================================================
; ishexdigit, isbindigit, isdecdigit
;-----------------------------------------------------------------------------
; Checks if char is:
;	ishexdigit -- hexadecimal digit
;	isdecdigit -- decimal digit
;	isbindigit -- binary digit
; In:	BL -- char
; Out:	CF clear if BL is valid
;		BL -- number
;	CF set otherwise
;	CX -- base
; Modf:	!BL, !CX
; Call:	---
; Use:	---
;
ishexdigit:
		mov	cx,16
		cmp	bl,'f'
		ja	..@no
		cmp	bl,'a'
		jae	..@yes_a
		cmp	bl,'F'
		ja	..@no
		cmp	bl,'A'
		jae	..@yes_aa
		jmp	..@check_dec
isdecdigit:				;<---------------------
		mov	cx,10
..@check_dec:
		cmp	bl,'9'
		ja	..@no
		cmp	bl,'0'
		jae	..@yes
		jmp	..@no
isbindigit:				;<---------------------
		mov	cx,2
		cmp	bl,'1'
		ja	..@no
		cmp	bl,'0'
		jae	..@yes
..@no:
		stc
		ret
..@yes_a:
		sub	bl,'a'-'A'
..@yes_aa:
		sub	bl,'A'-0ah-'0'
..@yes:
		sub	bl,'0'
		clc
		ret
;=============================================================================
; beep
;-----------------------------------------------------------------------------
; Generates beep.
; In:	---
; Out:	---
; Modf:	AX
; Call:	sleep
; Use:	opt_q
;
beep:
		cmp	byte [cs:opt_q],0
		jne	beep_quit
		define_beep_core
beep_quit:
		ret
;-----------------------------------------------------------------------------
		define_sleep
;-----------------------------------------------------------------------------


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Initialized resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
;
top_left:				;Col/row of calc's top left corner:
left		db	DEF_LEFT	;column
top		db	DEF_TOP		;row
actkey		db	DEF_ACTKEY	;Default activation key
shiftkey	db	DEF_SHIFTKEY	;Default shift value
opt_q		db	0		;1 if no sound
run_flag	db	0		;1 if calculator is active
disable_flag	db	0		;1 if calculator is disabled
autoclear	db	0		;Input line clearing mode
result		dw	0, 0		;Result of calculations
error_code	db	0		;Error code
sign_flag	db	1		;1 if result have sign
ins_flag	db	0		;0/1 -- insert/overstrike mode
quit_flag	db	0		;1 -- quit
regs_cleared	db	0		;0 if registers cleared
hist_end	dw	history_buffer	;history_buffer+hist_size
hist_top	dw	history_buffer	;Top of history stack
hist_cur	dw	history_buffer	;Currend string in history
math_sp		dw	math_stack	;Math stack pointer
fucking_proc	dw	null_proc	;Call this proc after pressing any key
;-----------------------------------------------------------------------------
; Pointers for editing procedures
;-----------------------------------------------------------------------------
; Input buffer:		This_is_a_string_in_the_input_buffer
; Edit window:		^		   _   ^
; input_buf ------------'                  ^  ^\        ^   |
;					   |  | `cursor |   |
ed_pointers:				   ;  |         |   |
start_ptr	dw	input_buf	;--'  |         |   |
cur_ptr		dw	input_buf	;-----'         |   |
end_ptr		dw	input_buf	;---------------'   |
eol_ptr		dw	input_buf	;-------------------'
;-----------------------------------------------------------------------------
; Table for printing parts of calculator's window
;-----------------------------------------------------------------------------
print_table:
		prtstr	0101h,	out_dec,	'd',	COL_LETTER
;This char used in `change_sign' ----------------^
		prtstr	0110h,	out_hex,	'h',	COL_LETTER
		prtstr	011ch,	out_char,	'"',	COL_LETTER
		prtstr	0201h,	out_bin,	'b',	COL_LETTER
		prtstr	0301h,	update_line, CHAR_RIGHT, COL_BG
		prtstr	0401h,	show_status,	0,	COL_BG
;This char used in `update_results'             ^
;as end-of-table marker ------------------------'
;
;-----------------------------------------------------------------------------
; Macro from messages.??
;
		resident_messages
;
;-----------------------------------------------------------------------------
; Table for `type_of_char'
; Classes of characters delimited by '_'.
;-----------------------------------------------------------------------------
chars		db	'01_23456789_abcdefABCDEF_._@_'
;			 0  1        2            3 4
		db	'+-*/%&|^<>{}\~`;_:_$_"_ _', 0
;			 5                6 7 8 9    10
;-----------------------------------------------------------------------------
; Editing and movement keys
;-----------------------------------------------------------------------------
key_table:
	%ifdef	VI_KEYS
		keystr	KEY_ALT_H,	move_left	;A-h (<--)
		keystr	KEY_ALT_L,	move_right	;A-l (-->)
		keystr	KEY_ALT_K,	move_up		;A-k (^)
		keystr	KEY_ALT_J,	move_down	;A-j (v)
	%else
		keystr	KEY_CTRL_S,	move_left	;C-s (<--)
		keystr	KEY_CTRL_D,	move_right	;C-d (-->)
		keystr	KEY_CTRL_E,	move_up		;C-e (^)
		keystr	KEY_CTRL_X,	move_down	;C-x (v)
	%endif
		keystr	KEY_LEFT,	cur_left	;<--
		keystr	KEY_RIGHT,	cur_right	;-->
		keystr	KEY_UP,		history_prev	;^
		keystr	KEY_DOWN,	history_next	;v
		keystr	KEY_PG_UP,	history_first	;Pg Up
		keystr	KEY_PG_DN,	history_last	;Pg Dn
		keystr	KEY_BACKSPACE,	backspace	;bksp
		keystr	KEY_DELETE,	delete		;del
		keystr	KEY_HOME,	cur_home	;home
		keystr	KEY_END,	cur_end		;end
		keystr	KEY_CTRL_Y,	clear_str	;C-y
		keystr	KEY_CTRL_K,	clear_eol	;C-k
		keystr	KEY_ALT_S,	change_sign	;A-s
		keystr	KEY_ENTER,	enter_handler	;Enter
		keystr	KEY_INSERT,	ins_toggle	;Insert
		keystr	KEY_ESC,	set_quit_flag	;Esc
		keystr	KEY_ALT_D,	kbd_out_dec	;A-d
		keystr	KEY_ALT_H,	kbd_out_hex	;A-h
		keystr	KEY_ALT_1,	kbd_out_bin_1	;A-1
		keystr	KEY_ALT_2,	kbd_out_bin_2	;A-2
		keystr	KEY_ALT_3,	kbd_out_bin_3	;A-3
		keystr	KEY_ALT_4,	kbd_out_bin_4	;A-4
		keystr	0000h,		insert_char	;Any key
;-----------------------------------------------------------------------------
; This table must be ordered by 'operator' field!
;-----------------------------------------------------------------------------
op_table:
		opstr	'%',	mod_proc
		opstr	'&',	and_proc
		opstr	'*',	mul_proc
		opstr	'+',	add_proc
		opstr	'-',	sub_proc
		opstr	'/',	div_proc
		opstr	'<',	shl_proc
		opstr	'>',	shr_proc
		opstr	'\',	sqrt_proc
		opstr	'^',	xor_proc
		opstr	'`',	neg_proc
		opstr	'{',	shl_proc
		opstr	'|',	or_proc
		opstr	'}',	sar_proc
		opstr	'~',	not_proc
		opstr	0ffh,	null_proc	;End of table
;-----------------------------------------------------------------------------
; Table for finite automaton
;-----------------------------------------------------------------------------
state_table:
		state	P_EX,  P_EX,  P_EX,  P_EX,  P_EX,  P_EX,	\
			P_EX,  P_EX,  P_EX,  P_EX,  P_EX,  P_EX		;EOL

		state	P_EX,  P_EX,  P_EX,  P_EX,  P_EX,  P_EX,	\
			P_EX,  P_EX,  P_EX,  P_EX,  P_EX,  P_EX		;OP

		state	P_EX,  P_EX,  P_EX,  P_EX,  P_EX,  P_EX,	\
			P_EX,  P_EX,  P_EX,  P_EX,  P_EX,  P_EX		;ENDHEX

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_EX,	\
			P_EX,  P_ERR, P_ERR, P_EX,  P_EX,  P_ERR	;ENDDEC

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_EX,	\
			P_EX,  P_ERR, P_ERR, P_EX,  P_EX,  P_ERR	;ENDBIN

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_EX,	\
			P_EX,  P_ERR, P_STR, P_EX,  P_EX,  P_ERR	;ENDSTR

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_EX,	\
			P_EX,  P_ERR, P_ERR, P_EX,  P_EX,  P_ERR	;ENDREG

		state	P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_EX,	\
			P_EX,  P_ERR, P_ERR, P_EX,  P_EX,  P_ERR	;ENDC

		state	P_BIN, P_DEC, P_HEX, P_ERR, P_ERR, P_OP,	\
			P_C,   P_REG, P_STR, P_ERR, P_EOL, P_ERR	;START

		state	P_HEX, P_HEX, P_HEX, P_ERR, P_ERR, P_EH,	\
			P_EH,  P_ERR, P_ERR, P_EH,  P_EH,  P_ERR	;HEX

		state	P_DEC, P_DEC, P_HEX, P_ED,  P_ERR, P_EH,	\
			P_EH, P_ERR, P_ERR, P_EH,  P_EH,  P_ERR		;DEC

		state	P_BIN, P_DEC, P_HEX, P_ED,  P_EB,  P_EH,	\
			P_EH,  P_ERR, P_ERR, P_EH,  P_EH,  P_ERR	;BIN

		state	P_STR, P_STR, P_STR, P_STR, P_STR, P_STR,	\
			P_STR, P_STR, P_ES,  P_STR, P_ERR, P_STR	;STR

		state	P_ER,  P_ER,  P_ERR, P_ERR, P_ERR, P_ERR,	\
			P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_ERR	;REG

		state	P_EC,  P_EC,  P_ERR, P_ERR, P_ERR, P_ERR,	\
			P_ERR, P_ERR, P_ERR, P_ERR, P_ERR, P_ERR	;REG
;-----------------------------------------------------------------------------
proc_table:
		dw	eol_handler	;P_EOL
		dw	op_handler	;P_OP
		dw	atolx		;P_EH
		dw	atold		;P_ED
		dw	atolb		;P_EB
		dw	atols		;P_ES
		dw	atolr		;P_ER
		dw	colon_handler	;P_EC
;-----------------------------------------------------------------------------
; Buffer for `ltoa'
ltoa_buf	dw	(COL_NUM+1)<<8, (COL_NUM)<<8
	times 3	dw	(COL_NUM+1)<<8
	times 3	dw	(COL_NUM)<<8
	times 3	dw	(COL_NUM+1)<<8
;-----------------------------------------------------------------------------
;
init_data_end:
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; End of initialized resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;=============================================================================
; Startup code
;-----------------------------------------------------------------------------
; Return codes:
;	0 -- all ok, requested functions performed
;	1 -- unload failed because FDRC was not loaded
;	2 -- unload failed because some INTs intercepted by other programs
;	3 -- installation failed because multiplex numbers exhausted
;	4 -- installation failed because FDRC already installed
;	5 -- invalid command line option used
;
install:
		cld
		mov	si,copyright
		call	dos_print	;Print copyright message
;-----------------------------------------------------------------------------
; Parse command line
;-----------------------------------------------------------------------------
		call	parse_cmdline
		jnc	@@parsed_ok
		mov	[unk_opt_char],al
		mov	si,unexp_param
		cmp	ah,0ffh
		je	@@unexp_param
		mov	si,unk_opt_msg
@@unexp_param:
		call	dos_print
		mov	si,bad_opt_tail
		mov	al,RC_BAD_OPT
		jmp	@@exit		;Unknown option found, exit
@@parsed_ok:
;-----------------------------------------------------------------------------
; -h, -?, -H
;-----------------------------------------------------------------------------
		mov	si,usage
		rcr	byte [opt_h],1
		jc	@@print_and_exit

		mov	si,keyboard_help
		rcr	byte [opt_hh],1
		jnc	@@skip_help
@@print_and_exit:
		mov	al,RC_ALL_OK
		jmp	@@exit		;Print help and exit
@@skip_help:
;-----------------------------------------------------------------------------
; -E, -D
;-----------------------------------------------------------------------------
		cmp	byte [opt_ed],0ffh
		je	@@skip_ed

		call	is_installed
		jnc	@@say_not_inst	;FDRC not installed, exit
@@en_dis:
		mov	ah,bh
		mov	al,FUNC_EN_DIS
		mov	bl,[opt_ed]
		int	2fh

		mov	si,enabled_msg
		or	bl,bl
		jz	@@print
		mov	si,disabled_msg
@@print:
		mov	al,RC_ALL_OK
		jmp	@@exit
@@skip_ed:
;-----------------------------------------------------------------------------
; -u
;-----------------------------------------------------------------------------
		rcr	byte [opt_u],1
		jnc	@@skip_unload

		call	is_installed
		jc	@@unload	;FDRC installed, uninstall it...
@@say_not_inst:
		mov	si,not_inst_msg
		mov	al,RC_NOT_INST
		jmp	@@exit		;FDRC not installed, exit
@@unload:
;-----------------------------------------------------------------------------
		xor	ax,ax
		mov	es,ax
		cmp	cx,[es:09h*4+2]
		jne	@@int09h_hooked	;INT 09h intercepted...
		cmp	cx,[es:2fh*4+2]
		je	@@real_unload
@@int09h_hooked:
		mov	si,cant_unload_msg ;INT 09h and/or INT 2fh intercepted
		mov	al,RC_INTERC	;by other program, exit
		jmp	@@exit
;-----------------------------------------------------------------------------
@@real_unload:
		mov	es,cx
		push	ds
		cli
		lds	dx,[es:old_int09h]
		mov	ax,DOS_SET_INT_VECT*256+09h
		int	21h		;Restore INT 09h
		lds	dx,[es:old_int2fh]
		mov	ax,DOS_SET_INT_VECT*256+2fh
		int	21h		;Restore INT 2fh
		sti
		pop	ds

		mov	ah,DOS_MEM_FREE	;Free used memory
		int	21h

		mov	si,uninstalled_msg
		mov	al,RC_ALL_OK
		jmp	@@exit		;FDRC uninstalled successfully, exit

@@skip_unload:
;-----------------------------------------------------------------------------
; -n
;-----------------------------------------------------------------------------
; REWRITE!!!
		rcr	byte [opt_n],1
		jnc	@@resident
		jmp	calculator	;Non-resident mode
@@exit:
		call	dos_print
		mov	ah,DOS_TERMINATE;Return to DOS
		int	21h

@@resident:
;-----------------------------------------------------------------------------
; -e, -t, -l, -k, -s
;-----------------------------------------------------------------------------
		rcr	byte [opt_e],1
		jc	@@opt_e_is_set

		call	is_installed
		jnc	@@opt_e_is_set

		mov	ah,bh
		mov	al,0

		rcr	byte [opt_l],1
		jnc	@@001
		mov	al,FUNC_SET_L
		mov	bl,[left]
		int	2fh
@@001:
		rcr	byte [opt_t],1
		jnc	@@002
		mov	al,FUNC_SET_T
		mov	bl,[top]
		int	2fh
@@002:
		rcr	byte [opt_s],1
		jnc	@@003
		mov	al,FUNC_SET_S
		mov	bl,[shiftkey]
		int	2fh
@@003:
		rcr	byte [opt_k],1
		jnc	@@004
		mov	al,FUNC_SET_K
		mov	bl,[actkey]
		int	2fh
@@004:
		mov	si,already_ins_msg
		call	dos_print

		or	al,al
		jz	@@not_changed
		mov	si,params_chg_msg
		mov	al,RC_ALL_OK
		jmp	@@exit
@@not_changed:
		mov	si,use_e_msg
		mov	al,RC_ALR_INST
		jmp	@@exit

@@opt_e_is_set:
;-----------------------------------------------------------------------------
; -y
;-----------------------------------------------------------------------------
		mov	ax,[hist_size]
		or	ax,ax
		jz	@@skip_y
		cmp	ax,MIN_HIST_SIZE
		jae	@@ok01
		mov	si,hist_too_small
		call	dos_print
		mov	ax,MIN_HIST_SIZE
@@ok01:
		cmp	ax,MAX_HIST_SIZE
		jbe	@@ok02
		mov	si,hist_too_large
		call	dos_print
		mov	ax,MAX_HIST_SIZE
@@ok02:
		add	word [hist_end],ax	;First byte beyond the buffer
		add	ax,15
		mov	cl,4
		shr	ax,cl
		mov	[extra_memory],ax
@@skip_y:
;-----------------------------------------------------------------------------
; Search for a vacant number
;-----------------------------------------------------------------------------
@@ins10:				;Nums 0..191 are reserved
		mov	bh,0c0h		;Scan nums 192..255
@@ins11:
		mov	ah,bh		;AH=num
		mov	al,0		;AL=0, installation check
		push	bx
		push	ds
		int	2fh
		pop	ds
		pop	bx
		cmp	al,0		;This num used ?
		je	@@ins12		;No, exit loop
		inc	bh		;Next num
		jnz	@@ins11		;Up to 255

		mov	si,cant_inst_msg
		mov	al,RC_NO_MUX_NUMS
		jmp	@@exit
@@ins12:
		mov	[ident],ah	; save num
;-----------------------------------------------------------------------------
; Save interrupt vectors
;-----------------------------------------------------------------------------
		mov	ax,DOS_GET_INT_VECT*256+09h
		int	21h
		mov	[old_int09h],bx
		mov	[old_int09h+2],es
		mov	ax,DOS_GET_INT_VECT*256+2fh
		int	21h
		mov	[old_int2fh],bx
		mov	[old_int2fh+2],es
;-----------------------------------------------------------------------------
; Allocate UMB
;-----------------------------------------------------------------------------
		mov	ax,cs
		cmp	ax,0a000h	;If we loaded via LH, skip
		jb	@@check_opt_w
		mov	word [tail_msg],inst_umb_msg
		jmp	@@set_vectors	;allocating UMB
@@check_opt_w:
		rcr	byte [opt_w],1	;Option `w' -- don't load into UMB
		jc	@@set_vectors
		mov	ax,DOS_GET_UMB_LINK
		int	21h
		cbw
		push	ax			;UMB link state
		mov	ax,DOS_GET_ALLOC
		int	21h
		push	ax			;Memory allocation strategy
		mov	ax,DOS_SET_UMB_LINK	;Set UMB link state:
		mov	bx,1			;link UMB for allocations
		int	21h
		jc	@@restore_umb		;No UMB?
		mov	ax,DOS_SET_ALLOC
		mov	bx,40h			;First fit high only
		int	21h
		jc	@@restore_umb
		mov	ah,DOS_MEM_ALLOC	;Allocate memory block
		mov	bx,((end_of_resident-offset_0100+100h)/16)+1	;Size
		add	bx,[extra_memory]	;of resident part in paragraphs
		int	21h
		jc	@@restore_umb	;Restore UMB if not enough free memory
;-----------------------------------------------------------------------------
; Copy interrupt handlers into UMB and patch MCB
;-----------------------------------------------------------------------------
		dec	ax		;Seg. addr. of MCB
		mov	es,ax
		inc	ax
		mov	[es:1],ax	;Seg. addr. of PSP
		mov	bx,cs
		dec	bx
		mov	ds,bx
		mov	cx,8+(end_of_resident-offset_0100+100h)
		mov	di,8
		mov	si,di
	rep	movsb			;Copy owner's name and code+data
		push	cs
		pop	ds
		mov	word [tail_msg],inst_umb_msg
		mov	word [dos_func],DOS_TERMINATE*256+00h
		mov	ds,ax
@@restore_umb:
;-----------------------------------------------------------------------------
; Restore allocation strategy and UMB link state
;-----------------------------------------------------------------------------
		pop	bx
		mov	ax,DOS_SET_ALLOC
		int	21h		;Restore mem. alloc. strat.
		pop	bx
		mov	ax,DOS_SET_UMB_LINK
		int	21h		;Restore UMB link state
@@set_vectors:
;-----------------------------------------------------------------------------
; Set new handlers for INT 09h and INT 2fh
;-----------------------------------------------------------------------------
		cli
		mov	ax,DOS_SET_INT_VECT*256+09h
		mov	dx,int09h_handler
		int	21h
		mov	ax,DOS_SET_INT_VECT*256+2fh
		mov	dx,int2fh_handler
		int	21h
		sti

		mov	ax,[cs:2ch]
		mov	es,ax
		mov	ah,DOS_MEM_FREE		;Free environment segment
		int	21h
		
		push	cs
		pop	ds

		mov	si,inst_ok_msg
		call	dos_print
		mov	si,[tail_msg]
		call	dos_print

		mov	ax,[dos_func]
		mov	dx,((end_of_resident-offset_0100+100h)/16)+1
		add	dx,[extra_memory]
		int	21h		;Terminate (and stay resident?)
;=============================================================================
; is_installed
;-----------------------------------------------------------------------------
; Checks if FDRC is installed.
; In:	---
; Out:	CY if FDRC is installed
;		BH == MUX number
;		CX == FDRC's segment
;	NC if FDRC is not installed
; Modf:	AX, BX, ?CX, ?DX, ?SI, DI, ?BP
; Call:	---
; Use:	---
;
is_installed:
		mov	bh,0ffh		;Scan nums 255..192
@@loop:
		mov	ah,bh		;AH=num
		mov	al,0		;AL=0, installation check
		push	bx
		push	ds
		push	es
		int	2fh
		pop	es
		pop	ds
		pop	bx
		cmp	al,0ffh		;This num used?
		jne	@@dec		;No

		cmp	di,MAGICK_CODE
		jne	@@dec

		stc			;FDRC found,
		ret			;exit
@@dec:
		dec	bh		;Next num
		cmp	bh,0c0h
		jae	@@loop

		clc
		ret
;=============================================================================
; Procedures from NASMLiB
;-----------------------------------------------------------------------------
		define_dos_print
		define_parse_cmdline
		define_get_param
;=============================================================================
;
opt_a_proc:
		mov	word [fucking_proc],enter_handler_2
		ret
;-----------------------------------------------------------------------------
opt_c_proc:
		dec	byte [autoclear]
		ret
;-----------------------------------------------------------------------------
opt_e_proc:
		dec	byte [opt_e]
		ret
;-----------------------------------------------------------------------------
opt_ee_proc:
		mov	byte [opt_ed],0
		ret
;-----------------------------------------------------------------------------
opt_dd_proc:
		mov	byte [opt_ed],1
		ret
;-----------------------------------------------------------------------------
opt_hh_proc:
		dec	byte [opt_hh]
		ret
;-----------------------------------------------------------------------------
opt_h_proc:
		dec	byte [opt_h]
		ret
;-----------------------------------------------------------------------------
opt_n_proc:
		dec	byte [opt_n]
		ret
;-----------------------------------------------------------------------------
opt_u_proc:
		dec	byte [opt_u]
		ret
;-----------------------------------------------------------------------------
opt_w_proc:
		dec	byte [opt_w]
		ret
;-----------------------------------------------------------------------------
opt_q_proc:
		dec	byte [opt_q]
		ret
;-----------------------------------------------------------------------------
opt_y_proc:
		mov	bp,isdecdigit
		call	atol
		mov	[hist_size],ax
		or	dx,dx
		jz	@@quit
		mov	word [hist_size],0ffffh
@@quit:
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_l_proc:
		mov	bp,isdecdigit
		call	atol
		or	dx,dx
		js	@@quit		;Quit if negative value
		mov	[left],al
		dec	byte [opt_l]
@@quit:
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_t_proc:
		mov	bp,isdecdigit
		call	atol
		or	dx,dx
		js	@@quit		;Quit if negative value
		mov	[top],al
		dec	byte [opt_t]
@@quit:
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_k_proc:
		mov	bp,ishexdigit
		call	atol
		mov	[actkey],al
		dec	byte [opt_k]
		dec	si
		clc
		ret
;-----------------------------------------------------------------------------
opt_s_proc:
		mov	bp,ishexdigit
		call	atol
		mov	[shiftkey],al
		dec	byte [opt_s]
		dec	si
		clc
		ret
;
;-----------------------------------------------------------------------------


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Non-resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

;-----------------------------------------------------------------------------
; This table must be ordered by first field!
;-----------------------------------------------------------------------------
		options_begin
		optstr	'?',	opt_h_proc
		optstr	'D',	opt_dd_proc
		optstr	'E',	opt_ee_proc
		optstr	'H',	opt_hh_proc
		optstr	'a',	opt_a_proc
		optstr	'c',	opt_c_proc
		optstr	'e',	opt_e_proc
		optstr	'h',	opt_h_proc
		optstr	'k',	opt_k_proc
		optstr	'l',	opt_l_proc
		optstr	'n',	opt_n_proc
		optstr	'q',	opt_q_proc
		optstr	's',	opt_s_proc
		optstr	't',	opt_t_proc
		optstr	'u',	opt_u_proc
		optstr	'w',	opt_w_proc
		optstr	'y',	opt_y_proc
		optstr	0ffh,	null_proc
		options_end
;-----------------------------------------------------------------------------
opt_h		db	0		;Help
opt_hh		db	0		;Keyboard help
opt_e		db	0		;Allow loading 2nd, 3rd,... TSR
opt_t		db	0		;
opt_l		db	0		;
opt_s		db	0		;
opt_k		db	0		;
opt_n		db	0		;Non-resident mode
opt_u		db	0		;Unload
opt_w		db	0		;Prevent loading into UMB
opt_ed		db	0ffh		;Enable/Disable FDRC
hist_size	dw	0		;Size of history buffer
;-----------------------------------------------------------------------------
tail_msg	dw	inst_conv_msg
dos_func	dw	DOS_TSR*256+00h
extra_memory	dw	0
;-----------------------------------------------------------------------------
; Macro from messages.??
;
		non_resident_messages
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; End of non-resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; Uninitialized resident data. This variables overlaps installation code.
; Note that some variables initialized by that code!
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
		absolute	install
;-----------------------------------------------------------------------------
old_int09h	resw	2		;Old INT 09h vector
old_int2fh	resw	2		;Old INT 2fh vector
video_seg	resw	1		;Segment of videobuffer
scr_rows	resb	1		;Number of rows on screen
scr_cols	resw	1		;Number of columns on screen
video_page	resb	1		;Number of active video page
page_offset	resw	1		;Offset of active video page
error_offs	resw	1		;Offset of error char in buffer
ident		resb	1		;Multiplex TSR identifier
old_ss		resw	1		;Saved SS
old_sp		resw	1		;Saved SP
scr_buf		resw	SCR_BUF_LENGTH	;Buffer for saving screen
input_buf	resb	BUF_LENGTH	;Input buffer
reg_table	resw	2*10		;Registers
math_stack	resw	2*MSTACK	;Math stack
new_stack	resw	PSTACK		;Programm's stack
stack_bottom:				;Bottom of stack
;
history_buffer:				;History buffer (if present)
					;resides here.
end_of_resident:			;Last byte of program in memory
;
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; End of uninitialized resident data
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; E0F
;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

