;******************************************************************************
;                            DOS-LOGO DEVICE DRIVER
;------------------------------------------------------------------------------
; This driver allows a palette animated bitmap to be shown during the boot
; process of DOS. It has the ability to be deactivated using the Escape
; keystroke. During its session, all text messages will not be erased. The 
; logo is self-sustained and will be hidden once the user requires input. It
; is possible to start a new logo session, or manipulate the current active
; logo session by calling the DOS-LOGO Support interrupts.
;
; DOS-LOGO must run on a 386 or higher CPU. It will only run on VGA graphics
; adapters. It supports the VGA text modes 80x25, 40x25, 80x43 and 80x50. Both
; color and monochrome display adapters are supported.
;
; Requires 3,5 KB conventional memory and a maximum of 20 KB + 128 KB disk 
; space to show a logo.
;	
; Supported logo bitmap formats: Windows BMP and Windows RLE resolution 320x400
; 8-bit colors with the color important word filled with palette animation 
; start and rewind count.
;
; DOS-LOGO is compiled with NASM16: http://nasm.sourceforge.net/.
;
; DOS-LOGO is licensed under the GNU LESSER GENERAL PUBLIC LICENSE. Read the
; textfile lgpl.txt in the same directory as dos-logo.asm, or visit:
; http://www.gnu.org/licenses/lgpl.html.
;
; DOS-LOGO is developed by Ola Salomonsson. Programmed by Ola Salomonsson
; for use with DOS as part of the boot process, but can also be called
; by any loading task within DOS that of its liking.
; Contact: mailto:ola]o[logotypes.se.
;
; DOS-LOGO has been tested and are working with the following versions of DOS:
;
;  MS-DOS version 6.22
;  PC-DOS version 7.X
;  MS-DOS version 7.10
;  DRDOS version 7.X
;
;  MS-DOS is a registered trademark of Microsoft Corporation.
;  PC-DOS is a registered trademark of International Machine Corporation.
;  DRDOS is a trademark of DRDOS, Inc.
;
; Visit http://logosys.logotypes.se/ for technical aspects of DOS-LOGO and for 
; tutorials on how to make propper logo.sys files. And also a nice collection
; of the original logo.sys files for insperation and guidance (note: most of 
; them refer to one or more registered trademarks as stated on the site).
;
; Copyright  2007-2008 Ola Salomonsson, logosys.logotypes.se
; Date 5/15/2008.
;
;
; HOW TO INSTALL:
;  Copy DOS-LOGO.SYS to the bootdisk root (or the OS driver directory). Also
;  store a LOGO.SYS bitmap, or a LOGO.RLE bitmap in the bootdisk root (must be
;  the root on the bootdisk of DOS).
;  Open up CONFIG.SYS, and write this line to the top of the file:
;
;      DEVICE=[d:][path]DOS-LOGO.SYS
;
;  Where [d:] is the drive and [path] is the directory where DOS-LOGO.SYS can
;  be located.
;  It is recommended to add this line to the top of CONFIG.SYS for best visual
;  outcome.
;
;
;
;           This program does not have any warranties, either stated
;           or implied.  By using this program you are assuming full 
;                 responsibility for the program's execution.
;
;******************************************************************************

	org		0
	
;******************************************************************************
;******************************************************************************
	dd		-1					; Driver linked list (-1 = last driver)
	dw		8000h				; Device attributes	(character device)
	dw		strategy_entry		; Pointer to strategy routine
	dw		request_entry		; Pointer to the request routine
	db		'DOS$LOGO'			; Device driver name
;******************************************************************************
;******************************************************************************

_JMPFAR		equ	0EAh
_JMPSHORT	equ	0EBh
_IRET		equ 0CFh

_BEEP		equ	07h
_BACKSPACE	equ	08h
_LF			equ	0Ah
_CR			equ	0Dh

PRH			dd	0

logo_flags	db	0				; Dos-logo flags. Currently only bit 20h is supported
logo_chksum	dw	0				; Logo pixel checksum to see if the video buffer has been untouched

; Interrupt chain
I08			dw	0,0				; Interrupt 08h (System timer 18.2 times per second)
I10			dw	0,0				; Interrupt 10h (Video services)
I15			dw	0,0				; Interrupt 15h (System services)
I16			dw	0,0				; Interrupt 16h (Keyboard services)
I2F			dw	0,0				; Interrupt 2Fh (Multiplex)

;******************************************************************************
strategy_entry:
	mov		word [cs:PRH+2],es	; Store segment address
	mov		word [cs:PRH],bx	; Store offset addr

	retf
;******************************************************************************
	
;******************************************************************************
request_entry:
	pusha
	pushf
		
	push	cs
	pop		ds					; DS = CS
	
	les		di,[PRH]			; ES:DI = Request Header
	
	mov		al,[es:di+2]		; Get function code
	
	or		al,al				; Check for command 0 "Init"
	jz		short .init

	mov		ax,8003h			; Return unknown command error
	jmp		short .exit_status

.vga_fail:
	mov		ax,8001h			; Return unknown unit error
	jmp		short .exit_status

.init:
	; First test if this computer's graphcs card is VGA compatible
	
	mov		ax,1A00h			; Get display combination code
	int		10h

	cmp		al,byte 1Ah			; Return code must be 1Ah
	jnz		.vga_fail
	
	cmp		bl,byte 08h			; Only VGA color (08h) or VGA monochrome (07h) are valid
	ja		.vga_fail
	
	cmp		bl,byte 07h
	jb		.vga_fail
	
	; Do the initial setup of interrupt vectors and display the logo

	call	setup

	call	doslogo_display		; Display the logo and put it in session

	; If the logo failed to be displayed, doslogo_display set the carry flag.
	; But we will not fail the driver. There may come future requests to display
	; a logo
	
	les		di,[PRH]			; ES:DI = Request Header

	mov		word [es:di+16],cs	; Setup resident start address
	mov		word [es:di+14],word end_offs

	xor		ax,ax

.exit_status:
	or		ax,0100h			; Set "Done" bit
	mov		word [es:di+3],ax	; Store status

	popf
	popa
		
	retf
;******************************************************************************

; This string enforces copyright and must not be removed

c_str		db	"DOS-LOGO Device Driver version 1.00",0
			db	"Copyright (C) 2007 logosys.logotypes.se",0
	
;******************************************************************************
setup:
	push	eax
	push	bx
	push	ds
	push	es

	; We must turn off any previous logos before hooking the interrupts. Else we will get the
	; previous logo's logo ints

	mov		ax,4A32h			; DOS-LOGO Support
	mov		bl,byte 3			; DOS-LOGO Restore Previous Video Mode
	int		2Fh

	; Initialize the segment for our logo int dispatchers
	mov		word [cs:logo_int_08h_dispatcher + 2],cs
	mov		word [cs:logo_int_10h_dispatcher + 2],cs
	mov		word [cs:logo_int_15h_dispatcher + 2],cs
	mov		word [cs:logo_int_16h_dispatcher + 2],cs

	push	cs
	pop		es					; ES = CS

	xor		bx,bx
	mov		ds,bx				; DS = Seg0000

	cli							; Disable interrupts

	; Get the original interrupts chained by the logo ints

	mov		bx,08h * 4			; Get interrupt 08h vector
	mov		eax,[bx]
	mov		dword [es:I08],eax

	mov		bx,10h * 4			; Get interrupt 10h vector
	mov		eax,[bx]
	mov		dword [es:I10],eax

	mov		bx,15h * 4			; Get interrupt 15h vector
	mov		eax,[bx]
	mov		dword [es:I15],eax

	mov		bx,16h * 4			; Get interrupt 16h vector
	mov		eax,[bx]
	mov		dword [es:I16],eax

	mov		bx,2Fh * 4			; Get interrupt 2Fh vector
	mov		eax,[bx]
	mov		dword [es:I2F],eax

	; And set the new DOS-LOGO Support Multiplex interrupt

	mov		ax,cs
	rol		eax,10h				; Put our CS segment in the high word of EAX
	mov		ax,int_2fh
	mov		dword [bx],eax		; Set our own interrupt 2Fh

	sti							; Enable interrupts

	pop		es
	pop		ds
	pop		bx
	pop		eax
	
	ret
;******************************************************************************

;******************************************************************************
doslogo_redisplay:
	; Only allow redisplaying of a logo which has a checksum, and is not
	; currently in session

	cmp		word [logo_chksum],0
	jnz		.has_checksum

.fail_exit:
	; Return with CF set to indicate error

	stc
	
	ret

.has_checksum:
	test	word [logo_flags],20h
	jnz		short .fail_exit

	; Save original stack. We will use our own
	
	mov		word [display_logo_sp],sp
	mov		word [display_logo_ss],ss
	
	mov		ss,ax				; Using our own stack
	mov		sp,original_sp	

	; It is time to save this video mode info, and we use
	; our own stack to put it temporarily until the mode change
	; has been completed

	push	ss
	pop		es					; ES = SS
	
	sub		sp,38				; 38 = size of bios video info
	mov		di,sp				; ES:DI will be the destination for the text mode info

	push	ds					; Save DS

	call	doslogo_save_textmode_info

	pop		ds					; Restore DS

	; Time to set the famous 320x400 graphics mode. This must be done in order
	; to validate the logo checksum. If the checksum does not match the contents
	; of the video RAM, the previous loaded logo has been destroyed and redisplay
	; will not be possible
	
	; Indicate we don't want to erase video memory by zeroing CX
	
	xor		cx,cx
	call	doslogo_320x400_A8000h

	; ES now points to SegA000h (Graphics video RAM)

	; Time to validate the checksum of the previous loaded logo against
	; the video RAM

	mov		cx,400				; Check 400 scanlines
	xor		bx,bx				; Clear test checksum
	mov		si,8000h			; ES:SI points to A8000h in video RAM

.checksum_next_scanline:
	call	doslogo_4_plane_scanline_checksum
	loop	.checksum_next_scanline

	; Compare checksum with previous logo's checksum
	cmp		word [logo_chksum],bx
	jz		.logo_validated

	; Okay, previous logo was destroyed. No logo could be re-displayed

	push	ss
	pop		es

	mov		di,sp

	call	doslogo_restore_textmode_info

	; After the call to restore the text mode info, DS points to ES and 
	; ES points to Seg0000
	
	add		sp,38				; Erase video info from our stack

	mov		ax,cs
	mov		ds,ax
	mov		es,ax				; ES = DS = CS

	; We must restore original stack before calling
	; doslogo_restore_previous_video!

	mov		sp,[display_logo_sp]
	mov		ss,[display_logo_ss]
	
	or		word [logo_flags],20h

	call	doslogo_restore_previous_video

	; Return with CF set to indicate error

	stc

	ret

.logo_validated:
	push	es
	pop		ds					; DS = SegA000h (Graphics video RAM)

	mov		si,0FD00h			; Start offset of palette from previous logo
	xor		di,di				; We are not going to remember the palette
	mov		cx,256				; 256 palette entries to read
	
	call	doslogo_restore_palette

	mov		di,sp				; Start of saved video info
	call	doslogo_alphanumeric_addressing

	push	ss
	pop		es

	mov		di,sp

	call	doslogo_restore_textmode_info

	; After the call to restore the text mode info, DS points to ES and 
	; ES points to Seg0000
	
	add		sp,38				; Erase video info from our stack

	mov		ax,cs
	mov		ds,ax
	mov		es,ax				; ES = DS = CS

	call	doslogo_enable_display

	; Put us in a logo session

	or		byte [logo_flags],byte 20h

	call	doslogo_int_open_dispatchers

	; Restore original stack

	mov		sp,[display_logo_sp]
	mov		ss,[display_logo_ss]

	clc							; Clear CF to indicate success

	ret
;******************************************************************************

display_logo_sp	dw	0
display_logo_ss	dw	0

logo_rle	db	'\LOGO.RLE',0	; Filename of logo in RLE format at bootdisk root
logo_sys	db	'\LOGO.SYS',0	; Filename of logo in BMP format at bootdisk root
handle		dw	0				; File handle
sl_left		dw	0				; Scanlines left until logo has been drawn
cmpr		dw	0				; Logo compression type

; Read buffer to cache data from disk. Must be [SIZE % 320 < 160] to not
; overflow! Recommended sizes are:
; 0x8000	Original size (faster on harddisks).
; 0x800		MS-DOS 7 beta (faster on floppy disks).

RBUFF_SIZE	equ	0x8000

; The read buffer will be rounded up to fit a reminding scanline from each new
; package read

RBUFF_EVEN	equ	(((RBUFF_SIZE / 320) + 1) * 320)

;******************************************************************************
doslogo_display:
	; Save original stack. We will use our own
	
	mov		word [display_logo_sp],sp
	mov		word [display_logo_ss],ss
	
	mov		ax,cs
	mov		ds,ax				; DS = CS
	mov		es,ax				; ES = CS

	mov		ss,ax				; Using our own stack
	mov		sp,original_sp	
	
	mov		dx,logo_rle			; Try to open logo.rle from bootdisk root
	mov		ax,3D00h			; DOS open handle
	int		21h
	
	jnc		.found_file

	mov		dx,logo_sys			; Try to open logo.sys from bootdisk root
	mov		ax,3D00h			; DOS open handle
	int		21h
	
	jnc		.found_file

	; Return with CF set to indicate error

	jmp		exit_ss

.failed_opened_file:	
	; Return with CF set to indicate error

	jmp		exit_close
	
.found_file:
	xchg	bx,ax				; Get file handle into BX

	mov		word [handle],bx	; Save file handle
	
	; Time to read the bitmap file header, the bitmap info header, 
	; as well as the palette
	
	mov		ah,3Fh				; DOS read from file
	mov		cx,436h				; Assuming a 256 BGRx-quad palette
	mov		dx,read_buffer
	int		21h

	jc		.failed_opened_file	; Jump if failed to read from file

	mov		si,read_buffer		; SI will be our buffer pointer
	
	; We only read the BM signature from the bitmap file header
	
	cmp		word [si],word 'BM'
	jnz		.failed_opened_file

	; Now skip to the bitmap info header
	
	add		si,0Eh
	
	cmp		word [si],28h		; We only accept the standard 28h byte info header size
	jnz		.failed_opened_file

	cmp		word [si + 0Ch],1	; And only 1 plane bitmap support
	jnz		.failed_opened_file

	cmp		word [si + 0Eh],8	; As well as a 256 palette bitmap
	jnz		.failed_opened_file

	; The logo bitmap must be in the correct size
	
	cmp		word [si + 04h],320	; Width of bitmap must be 320 pixels
	jnz		.failed_opened_file

	cmp		word [si + 08h],400	; Height of bitmap must be 400 pixels
	jnz		.failed_opened_file

	; Doslogo supports RLE8 encoded bitmaps which is a new feature to the original logo.sys
	; format
	
	mov		dx,[si + 10h]		; Get the compression format
	mov		word [cmpr],dx		; Save for later use
	
	cmp		dl,byte 01h			; Check for RLE8 compression or normal RGB format
	ja		.failed_opened_file

	; Now for the embedded animation info stored at the color important word
	; in the info header
	
	mov		ax,[si + 24h]		; Get animation palette entry start offset and rotation info
	mov		byte [anim_start],al
	neg		al
	add		al,255
	mov		byte [anim_size],al
	or		ah,ah
	jz		.use_default_rot

	mov		word [anim_rot_sv],ax
	mov		word [anim_rot],ax	; Animation rotation info holds the palette entry start offset
								; and the rewind count

.use_default_rot:
	; It is time to save this video mode info, and we use
	; our own stack to put it temporarily until the mode change
	; has been completed

	push	ss
	pop		es					; ES = SS
	
	sub		sp,38				; 38 = size of bios video info
	mov		di,sp				; ES:DI will be the destination for the text mode info

	push	ds					; Save DS

	call	doslogo_save_textmode_info

	pop		ds					; Restore DS

	; Time to set the famous 320x400 graphics mode. We also request a clear of
	; the video RAM A000h:8000h to B000h:0000h (8000h bytes). This area is used
	; for the logo bitmap (stored in 4 planes) as well as the palette to be 
	; remembered between logo sessions. This is done by calculating a checksum
	; for logo validation
	
	mov		word [logo_chksum],0

	mov		cx,8000h
	call	doslogo_320x400_A8000h

	; ES now points to SegA000h (Graphics video RAM)

	mov		di,0FD00h			; This is where we put the logo palette so that
								; the logo can be restored once the session is
								; over.
								; Also note that this is at the end of the
								; logo too, so this is where we are going to begin
								; drawing the logo backwards up to 0x8000 (4 planes)
	
	; SI was destroyed by the call to doslogo_save_textmode_info, so we
	; need to point it to the palette start in the read buffer
	
	mov		si,read_buffer + 36h
	mov		cx,256				; 256 palette entries to read
	
	call	doslogo_set_and_remember_palette

	; Now it is time to draw the logo bitmap. Since bitmap data is stored
	; upside down, we will begin drawing from A000h:FD00h in video RAM (pointed
	; to by DI) up to A000h:8000h (4 planes). We will read the logo bitmap as packages
	; the size of the read buffer, draw each package to video RAM in even scanlines.
	; The reminding part of the readbuffer will be shifted to the start of the next
	; package. This will continue until all 400 scanlines have been drawn.
	; However, if the logo bitmap is compressed using RLE8, each package will be half
	; the size, using the upper half of the read buffer as a RLE8 decode buffer, and
	; the lower half of the read buffer as draw package

	mov		word [sl_left],400	; Scanlines left until logo has been drawn
	mov		dx,read_buffer		; Write data to the beginning of the read_buffer

	mov		si,RBUFF_SIZE		; Size of a full package for use to calculate number of 
								; full scanlines to draw

	cmp		word [cmpr],0		; RLE8 compression check
	jz		.read_next_package

	shr		si,1				; RLE8 decode buffer takes up half the read buffer size
	add		dx,si				; Adjust read buffer to the lower half

	xor		ax,ax
	mov		word [fill_cnt],ax	; Reset the fill count, decode pointer, and pitch
	mov		word [fill_attr],ax
	mov		word [decode_ptr],ax
	mov		word [pitch_mod],320
	
.read_next_package:
	call	read_buffer_possibly_decode

	jc		.read_failure		; CF is set if failed to read from file (file might be damaged, or
								; invalid size of the read buffer)

	mov		ax,si				; AX = package size including reminding size from the last package
	xor		dx,dx
	mov		cx,320				; Size of a scanline
	
	div		cx					; Divide current package size with scanline size to get number of
								; scanlines to process in AX

	xchg	cx,ax				; CX = number of scanlines to process, AX = scanline size
	cmp		cx,[sl_left]		; We must not do more scanlines than the ones left to draw a complete logo
	jbe		.no_limit_needed

	mov		cx,[sl_left]		; Limit the number of scanlines to the number left to draw a complete logo

.no_limit_needed:
	push	cx					; We need the information in CX and SI after the scanlines have been drawn
	push	si

	mov		bx,read_buffer		; Read pixels from the beginning of the read_buffer
	
	cmp		word [cmpr],0		; RLE8 compression check
	jz		.draw_scanline
	
	add		bx,RBUFF_SIZE / 2	; Adjust read buffer to the lower half
		
.draw_scanline:
	sub		di,320 / 4			; The logo is drawn upside down, so adjust to next scanline

	push	cx					; Remember the number of scanlines left to draw

	call	doslogo_4_plane_scanline

	pop		cx					; Restore the number of scanlines left to draw
	
	add		bx,320				; Do next scanline from the read buffer
	loop	.draw_scanline

	pop		cx					; CX = package size including reminding size from the last package
	pop		ax					; AX = number of scanlines processed

	sub		word [sl_left],ax	; Decrease the number of needed scanlines to draw the logo
	jbe		.logo_drawn			; Was this the last scanline? If so, jump

	mov		si,bx				; SI = Offset to the reminding data in the read buffer that we
								; want to copy to the beginning of the read buffer for the upcoming
								; scanline drawing

	mov		bx,320				; Find out the size of the scanlines drawn by multiplying number of
								; scanlines processed with 320 (size of one scanline)

	mul		bx

	sub		cx,ax				; CX = size of the reminding data to use in the upcoming scanline
	
	push	di					; Save the video RAM address for this scanline
	push	cx					; Save reminding size soon to occupy the beginning of the read buffer
	push	es					; Save video RAM segment
	
	push	ds
	pop		es					; ES = DS
	
	mov		di,read_buffer		; Start of read_buffer to copy to

	mov		ax,RBUFF_SIZE		; Remember the buffer size to be added with SI at the bottom of the
								; package loop

	cmp		word [cmpr],0		; RLE8 compression check
	jz		.no_adjust_start_size

	shr		ax,1				; RLE8 decode buffer takes up half the read buffer size
	
	add		di,ax				; Adjust read buffer to the lower half
		
.no_adjust_start_size
	cld							; Forward direction

	rep		movsb				; Copy reminding data to the top of the read buffer

	mov		dx,di				; Where to append the next package data in the read buffer

	pop		es					; Restore video RAM segment
	pop		si					; Restore reminding size of package
	pop		di					; Restore the video RAM address for this scanline

	add		si,ax				; SI = size of resized read buffer including reminder of
								; logo bitmap already put at the top of the read buffer

	jmp		.read_next_package

.read_failure:
.logo_drawn:
	mov		di,sp				; Start of saved video info
	call	doslogo_alphanumeric_addressing

	push	ss
	pop		es

	mov		di,sp

	call	doslogo_restore_textmode_info

	; After the call to restore the text mode info, DS points to ES and 
	; ES points to Seg0000
	
	add		sp,38				; Erase video info from our stack

	call	doslogo_enable_display

	; Time to calculate the keyboard buffer timing so that
	; a logo session can survive an interrupt 16h call if
	; the user pressed a keystroke beforehand

	mov		ax,[es:46Ch]		; AX = (0040h:006Ch) Low word of master clock count

sync_timing:
	mov		dx,[es:46Ch]		; DX = (0040h:006Ch) Low word of master clock count
	cmp		dx,ax
	jz		sync_timing

	xor		cx,cx				; CX = reset counter

start_timing:
	mov		ah,byte 01h			; Report keystroke ready
	int		16h

	inc		cx					; Increment counter
	cmp		dx,[es:46Ch]
	jz		start_timing

	xchg	ax,cx				; AX = number of int 16h calls
	xor		dx,dx
	mov		cx,20

	div		cx					; AX /= 20

	or		ax,ax
	jz		.use_default_delay

	mov		word [min_delay],ax	; Remember minimum delay
	mov		word [sav_delay],ax	; Remember minimum delay

.use_default_delay:
	; Put us in a logo session

	or		byte [logo_flags],byte 20h

	call	doslogo_int_open_dispatchers

exit_close:
	; Close the logo bitmap file

	mov		bx,[handle]
	mov		ax,3E00h			; DOS close handle
	int		21h

exit_ss:
	; Restore original stack

	mov		sp,[display_logo_sp]
	mov		ss,[display_logo_ss]

	ret
;******************************************************************************

fill_cnt	dw	0				; Compression count in progress if not zero
fill_attr	dw	0				; Compression count in progress if not zero
pitch_mod	dw	0				; The rewind width of the bitmap
decode_ptr	dw	0				; Pointer into the decode buffer

;******************************************************************************
;	read_buffer_possibly_decode
;
;	Two cases exist. The first is the simple one: The bitmap is uncompressed
;	and by that it is linear. We will fill the read buffer full with the
;	bitmap data, and feed it to the video buffer in packages.
;	The second case is more evil: We will use the upper half of the read buffer
;	as a RLE8 decode buffer, decode it to the linear bitmap data needed to be
;	drawn in video RAM to the lower half of the read buffer. Doing so will both
;	slow down the read process because of the encoding time, but also the
;	drawing process because the packages are half the size of the read buffer
;	instead of the full size. Anything to save memory on both RAM and disk.
;	DX is pointing to the read buffer start + reminding bytes of the last
;	uncomplete scanline read.
;	Returns CF set if failed to read from file.
;******************************************************************************
read_buffer_possibly_decode:
	cmp		word [cmpr],0		; Test for RLE8 compression
	jz		near .read_uncompressed_data

	push	eax
	push	ecx
	push	si
	push	di	
	push	es

	mov		si,[decode_ptr]

	push	ds
	pop		es					; ES = DS

	mov		di,dx				; DI points to the lower half of the read buffer + reminder of last package
	mov		bx,di
	add		bx,RBUFF_SIZE / 2	; The packet will be full once DI reaches BX

.decode_until_packet_full:
	; We may be in a fill count mode. This is a RLE8 color count using either 
	; color 0 (background color) or a color specified in the high byte of fill_cnt.
	; If the fill attribute is signed, we are in absolute mode

	cmp		dword [fill_cnt],0
	jz		near .no_fill_cnt

	js		.abs_mode			; Absolute mode if signed

	mov		ax,[fill_attr]		; Color count if bit 13 is on
	and		ax,2000h
	jz		.do_bg_cnt			; Else we have a background fill count

	; This is a color count
	mov		al,byte [fill_cnt + 1]
	
.clr_cnt:
	call	set_bmp8_byte		; Set the color
	jc		.clr_cnt_ready		; Buffer is ready, but we need to count this pixel first

	dec		byte [fill_cnt]
	jnz		.clr_cnt
	
	jmp		.clear_cnt			; Jump to clear the counters

.clr_cnt_ready:
	dec		byte [fill_cnt]
	jnz		near .buffer_ready	; Buffer is ready. Let us do some drawing!

.clear_cnt_ready:
	xor		ax,ax
	mov		word [fill_cnt],ax	; Turn off cnt
	mov		word [fill_attr],ax

	jmp		near .buffer_ready	; Buffer is ready. Let us do some drawing!

.do_bg_cnt:
	xor		al,al				; This is a background fill count

.bg_cnt:
	call	set_bmp8_byte		; Set the color
	jc		.bg_cnt_ready		; Buffer is ready, but we need to count this pixel first

	dec		dword [fill_cnt]	; Background counts can be higher than a word, so we use dword
	jnz		.bg_cnt

	jmp		.no_fill_cnt

.bg_cnt_ready:
	dec		dword [fill_cnt]
	jmp		near .buffer_ready	; Buffer is ready. Let us do some drawing!

.abs_ready:
	dec		byte [fill_cnt]
	jnz		near .buffer_ready	; Buffer is ready. Let us do some drawing!

	; End of absolute mode. Very important is to check if we are on
	; an odd byte, then we must read one extra byte to stay on a word
	; boundary

	mov		ax,[fill_attr]
	and		ax,4000h
	jz		.clear_cnt_ready

	; Read one more byte to fix word alignment

	call	get_rle8_byte		; Get cmd in AL
	jc		near .failure		; If CF is set, failed to read from file

	jmp		.clear_cnt_ready

.abs_mode:
	call	get_rle8_byte		; Get color in AL
	jc		near .failure		; If CF is set, failed to read from file

	call	set_bmp8_byte		; Set the color
	jc		.abs_ready			; Buffer is ready, but we need to count this pixel first

	dec		byte [fill_cnt]
	jnz		.abs_mode

	; End of absolute mode. Very important is to check if we are on
	; an odd byte, then we must read one extra byte to stay on a word
	; boundary

	mov		ax,[fill_attr]
	and		ax,4000h
	jz		.clear_cnt

	; Read one more byte to fix word alignment

	call	get_rle8_byte		; Get cmd in AL
	jc		near .failure		; If CF is set, failed to read from file

.clear_cnt:
	xor		ax,ax
	mov		word [fill_cnt],ax	; Turn off cnt
	mov		word [fill_attr],ax

.no_fill_cnt:
	; Time to decode the RLE8 data

	call	get_rle8_byte		; Get cmd in AL
	jc		near .failure		; If CF is set, failed to read from file

	or		al,al				; Is this a command, or a color count?
	jnz		near .do_color_count

	; AL indicated a command. Find out which

	call	get_rle8_byte		; Get marker in AL
	jc		near .failure		; If CF is set, failed to read from file

	or		al,al				; Is this the end of line marker?
	jnz		.not_eol			; If not, jump

	; End of line marker. Fill the reminding
	; pixels with the background color

	mov		ax,[pitch_mod]
	mov		word [fill_cnt],ax
	jmp		.decode_until_packet_full
	
.not_eol:
	dec		al
	or		al,al				; Is this the end of bitmap marker?
	jnz		.not_eob			; If not, jump

	; End of bitmap marker. Fill the reminding
	; pixels with the background color. We simply say there
	; are 320x400 pixels to fill, and the logo drawing will
	; end by itself when reaching 400 lines

	mov		eax,320 * 400
	mov		dword [fill_cnt],eax
	jmp		.decode_until_packet_full

.not_eob:
	dec		al
	or		al,al				; Is this the delta marker?
	jnz		.not_delta			; If not, jump

	; Delta marker. Get the relative x byte, and add it with
	; the relative y byte multiplyed by the bitmap width. Fill
	; the relative void with the background color

	call	get_rle8_byte		; Get relative x in AL
	jc		.failure			; If CF is set, failed to read from file

	movzx	cx,al
	mov		word [fill_cnt],cx

	xor		eax,eax

	call	get_rle8_byte		; Get relative x in AL
	jc		.failure			; If CF is set, failed to read from file

	mov		ecx,320

	mul		ecx

	add		dword [fill_cnt],eax
	jmp		.decode_until_packet_full

.not_delta:
	; The marker is absolute mode. Restore AL by adding 2
	
	add		al,byte 2			; Restore marker, now being the counter
	movzx	ax,al				; Word extend it, unsigned of course
	mov		word [fill_cnt],ax	; Put the counter
	xor		cx,cx
	or		cx,8000h			; Turn on the sign bit to indicate absolute mode
	test	al,byte 1			; Test if this is an odd count
	jz		.not_odd

	or		cx,4000h			; Turn on bit 14 so that we can remember this count as odd

.not_odd:
	mov		word [fill_attr],cx
	jmp		.decode_until_packet_full
	
.do_color_count:
	or		ax,2000h			; Turn on bit 13 to indicate a color count
	mov		word [fill_attr],ax
	ror		ax,8				; Save count in AH
	
	call	get_rle8_byte		; Get color in AL
	jc		.failure			; If CF is set, failed to read from file

	ror		ax,8				; AL = count, AH = color
	mov		word [fill_cnt],ax
	jmp		.decode_until_packet_full

.failure:
	stc							; Set CF to indicate failure

	mov		word [decode_ptr],si

	pop		es
	pop		di
	pop		si
	pop		ecx
	pop		eax

	ret

.buffer_ready:
	clc							; Clear CF to indicate success

	mov		word [decode_ptr],si

	pop		es
	pop		di
	pop		si
	pop		ecx
	pop		eax

	ret
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
.read_uncompressed_data:
	mov		bx,[handle]			; Get file handle
	mov		ah,3Fh				; DOS read from file
	mov		cx,RBUFF_SIZE		; Read the size of a full package
	int		21h

	ret
;******************************************************************************

;******************************************************************************
get_rle8_byte:
	; First see if we are at offset 0 in the decode buffer. If SI is zero
	; we must read a new chunk of RLE8 decoded data to the decode buffer

	or		si,si				; Do we need to reload the RLE8 decode buffer?
	jnz		.get_byte

	push	dx					; Save DX
	push	bx					; Save BX
	
	mov		bx,[handle]			; Get file handle
	mov		ah,3Fh				; DOS read from file
	mov		cx,RBUFF_SIZE / 2	; Read the size of half the read buffer
	mov		dx,read_buffer		; We want to put the the decoded bitmap in the upper half
	int		21h

	pop		bx					; Restore BX
	pop		dx					; Restore DX

	jnc		.get_byte

	; Carry flag is set. It indicates file read error
	
	ret							; Return with CF set

.get_byte:
	; Return the byte in AL
	
	mov		al,byte [read_buffer + si]
	inc		si
	
	; Do we need to reset the RLE8 decode pointer?

	cmp		si,RBUFF_SIZE / 2
	jnz		.no_reset_rle8_ptr

	xor		si,si				; Next decode iteration, the decode buffer will be reloaded

.no_reset_rle8_ptr:
	clc							; Clear the CF flag to indicate success

	ret
;******************************************************************************

;******************************************************************************
set_bmp8_byte:
	cld							; Forward direction
	stosb						; Store AL at ES:DI

	; We must know how many bytes are left until next scanline
	
	dec		word [pitch_mod]	; Count backwards from bitmap width to 0
	jns		.not_signed

	mov		word [pitch_mod],320 - 1

.not_signed:
	; Check to see if the packet is full of bitmap data to be drawn

	cmp		di,bx
	jnz		.package_not_full

	; Yes package is full!

	stc							; Set the CF flag to indicate package is full

	ret

.package_not_full:
	clc							; Clear the CF flag to indicate package not full

	ret
;******************************************************************************

;******************************************************************************
;	doslogo_320x400_A8000h
;
;	Turns standard VGA mode 13h to unchained mode to allow using another start
;	address for the video buffer, as well as doubles the scanlines from 200
;	to 400. The only reason for using 400 scanlines is so that the swap from
;	VGA text modes to this mode won't turn off the monitor. Text modes for VGA
;	use 400 scanlines, so the swapping will be smooth, which is required!
;	Using any other modes with 400 scanlines won't work because either they are
;	not supporting display address changing, not widely compatible, or not
;	supporting 256 colors.
;	CX = number of bytes to clear from start of video buffer.
;	Returns: ES will point to segment A000h.
;	Once the mode change is done, the display must be reactivated to show any
;	drawn pixels. This is because it is slower to write pixels in unchained
;	mode, and we want to draw pixels even if the computer is from the 80's.
;******************************************************************************
doslogo_320x400_A8000h
	mov		ax,13h | 0x80		; Set mode 13h (320x200x256) and don't erase the video buffer!
	int		10h

	mov		dx,03DAh			; VGA port 3DAh: Input status #1 register
	in		al,dx				; Reset flip flop between address and data register for port 3C0h

	mov		dl,0C0h				; VGA port 3C0h: Attribute: Palette
	xor		al,al
	out		dx,al				; Turn off display

	mov		dl,0C4h				; VGA port 3C4h: Sequencer address
	mov		ax,0F02h			; Map mask address
	out		dx,ax				; Enable write for all 4 planes

	mov		al,byte 04h			; Sequencer address: Memory mode
	out		dx,al

	inc		dx					; VGA port 3C5h: Sequencer data
	in		al,dx				; Get memory mode

	and		al,byte ~08h		; Turn off bit 3 to select map mode rather than color mode
	or		al,byte 04h			; Enable odd/even addressing mode
	out		dx,al				; Set memory mode

	mov		dl,0CEh				; VGA port 3CEh: Graphics address
	mov		al,byte 05h			; Graphics address: Mode
	out		dx,al

	inc		dx					; VGA port 3CDh: Graphics data
	in		al,dx				; Get mode

	and		al,byte ~10h		; Turn off bit 4 to not use odd/even RAM addressing
	out		dx,al

	dec		dx					; VGA port 3CEh: Graphics address
	mov		al,byte 06h			; Graphics address: Miscellaneous
	out		dx,al

	inc		dx					; VGA port 3CDh: Graphics data
	in		al,dx				; Get miscellaneous
	and		al,byte ~02h		; Turn off bit 1 to not chain odd maps after even maps
	out		dx,al

	mov		ax,0A000h			; SegA000h	Graphics video RAM
	mov		es,ax

	; Time to clear the video ram at A8000h if so
	; stated in CX

	jcxz	.do_not_clear

	mov		di,8000h			; ES:DI = A000h:8000h

	xor		ax,ax				; Clear with zero

	cld							; Forward direction
	rep		stosb				; Clear as many bytes specified in CX

.do_not_clear:
	mov		dl,byte 0D4h		; VGA port 3D4h: CRT controller address
	mov		ax,0Ch | 8000h		; CRT controller address: Start address high
	out		dx,ax				; Set display buffer to start at A000h:8000h

	mov		al,byte 09h			; CRT controller address: Maximum scan line
	out		dx,al

	inc		dx					; VGA port 3D5h: CRT controller data
	in		al,dx				; Get maximum scan line

	and		al,byte ~1Fh		; Clear bit 0 - 4 to use normal scanline timing
	out		dx,al				; This action turns 320x200 mode 13h into 230x400

	dec		dx					; VGA port 3D4h: CRT controller address
	mov		al,byte 14h			; CRT controller address: Underline location
	out		dx,al

	inc		dx					; VGA port 3D5h: CRT controller data
	in		al,dx				; Get underline location

	and		al,byte ~40h		; Clear bit 6 to turn off double word mode addressing
	out		dx,al				; Set underline location

	dec		dx					; VGA port 3D4h: CRT controller address
	mov		al,byte 17h			; CRT controller address: Mode control
	out		dx,al

	inc		dx					; VGA port 3D5h: CRT controller data
	in		al,dx				; Get mode control

	or		al,byte 40h			; Set bit 6 to use word 
	out		dx,al				; Set mode control mode addressing

	ret
;******************************************************************************

;******************************************************************************
;	doslogo_set_and_remember_palette
;
;	Called with CX set to the number of palette entries to read, DS:SI to the
;	source of the palette, and ES:DI to the RAM where the palette should be
;	remembered in case of a future logo session.
;	This routine will set the palette, as well as remember the palette.
;******************************************************************************
doslogo_set_and_remember_palette:
	push	di
	push	es
	push	si
	push	cx

	mov		di,si				; The first duty is to convert the BGRx-quad 8 bit components into
								; into RGBx-quad 6 bit components

	push	ds
	pop		es					; ES = DS

.loop_bgrx_to_rgbx:
	lodsw						; Load BG into AX
	
	mov		dx,ax				; DX = BG
	
	lodsw						; Load RX into AX
	
	shr		al,byte 2			; Convert R component from 8 bit to 6 bit
	shr		dl,byte 2			; Convert B component from 8 bit to 6 bit
	shr		dh,byte 2			; Convert G component from 8 bit to 6 bit
	
	stosb						; Store R 6 bit over B 8 bit
	
	xchg	dl,dh				; Swap B 6 bit with G 6 bit to put them in the RGB order
	mov		ax,dx				; AX = DX
	
	stosw						; Store GB 6 bit over GR 8 bit

	loop	.loop_bgrx_to_rgbx	; And loop through all palette entries
	
	pop		cx
	pop		si
	pop		es
	pop		di

doslogo_restore_palette:
	push	di					; We want to save this address for the logo drawing
	
	xor		ax,ax				; Begin at palette index 0
	mov		dx,03C8h			; VGA set palette index
	out		dx,al
	
	inc		dx					; DX = 03C9h. VGA set RGB components

	mov		bx,cx
	add		cx,cx
	add		cx,bx				; CX = CX * (palette entry components)

.set_next_palette_component:
	push	cx
	
	mov		cx,20
	
.port_delay:
	loop	.port_delay			; We must wait a short delay before setting the next component
	
	pop		cx
	
	lodsb						; Load the component (can be R, G, or B depending on the iteration)
	
	out		dx,al
	
	or		di,di				; DI might be null, which means we don't want to remember the palette
	jz		.dont_remember
	
	stosb						; Store this component at ES:DI for future logo sessions

.dont_remember:
	loop	.set_next_palette_component

	pop		di					; DI is now where the logo should end in video RAM

	ret
;******************************************************************************

;******************************************************************************
;	doslogo_alphanumeric_addressing
;
;	Changes the adressing for the character generator to think it is in text
;	mode (B8000h for color adapters and B0000h for monochrome adapters).
;	DI = start of the video mode info array returned from
;	doslogo_save_textmode_info.
;******************************************************************************
doslogo_alphanumeric_addressing:
	mov		dx,03C4h			; VGA port 3C4h: Sequencer address
	mov		al,02h				; Map mask address
	out		dx,al

	inc		dx					; VGA port 3C5h: Sequencer data
	in		al,dx				; Get map mask
	
	and		al,byte ~0Ch		; Turn off bit 2 and 3 to disable write planes 2 and 3
	or		al,byte 03h			; Turn on bit 0 and 1 to enable write planes 0 and 1
	out		dx,al				; Set map mask

	dec		dx					; VGA port 3C4h: Sequencer address
	mov		al,byte 04h			; Sequencer address: Memory mode
	out		dx,al

	inc		dx					; VGA port 3C5h: Sequencer data
	in		al,dx				; Get memory mode
	
	and		al,byte ~04h		; Turn off bit 2 to disable odd/even addressing mode
	out		dx,al				; Set memory mode

	mov		dl,0CEh				; VGA port 3CEh: Graphics address
	mov		ax,0004h			; Graphics address: Read map select
	out		dx,ax				; Read from plane 0

	mov		al,byte 05h			; Graphics address: Mode
	out		dx,al
	
	inc		dx					; VGA port 3CDh: Graphics data
	in		al,dx				; Get mode
	
	or		al,byte 10h			; Turn on bit 4 to use odd/even RAM addressing
	out		dx,al

	dec		dx					; VGA port 3CEh: Graphics address
	mov		al,byte 06h			; Graphics address: Miscellaneous
	out		dx,al
	
	inc		dx					; VGA port 3CDh: Graphics data
	in		al,dx				; Get miscellaneous
	or		al,byte 0Eh			; Turn on alphanumeric mode, chain odd maps after even maps
								; and use video RAM B8000h (32K CGA)

	; Check to see if we are on a monochrome adapter. Default is color

	cmp		byte [ss:di + 1],byte 07h
	jnz		.use_color_adapter_addressing
	
	; We are on monochrome adapter, so change the cpu memory mapping
	; to B0000h

	and		al,byte ~04h

.use_color_adapter_addressing:
	out		dx,al

	ret
;******************************************************************************

;******************************************************************************
;	doslogo_enable_display
;
;	As soon as we call this routine, the graphics will be shown. Because the
;	display was turned off right after the mode switch, and the new mode uses
;	400 scanlines just like the VGA text modes, the monitor won't waste time
;	reloading itself for a new resolution.
;******************************************************************************
doslogo_enable_display:
	mov		dx,03DAh			; VGA port 3DAh: Input status #1 register
	in		al,dx				; Reset flip flop between address and data register for port 3C0h

	mov		dl,0C0h				; VGA port 3C0h: Attribute: Palette
	mov		al,byte 20h
	out		dx,al				; Turn on display
	ret
;******************************************************************************

;******************************************************************************
;	doslogo_4_plane_scanline
;
;	Drawing pixels in unchained mode requires changes to the write plane. These
;	changes takes time. Instead of changing between the 4 write planes every
;	new pixel, we draw every fourth pixel to one plane at a time * 4. Still
;	not fast, but luckily the display is turned off at this time, so no 
;	flickery screen drawing.
;******************************************************************************
doslogo_4_plane_scanline:
	push	bp
	
	cld							; Forward direction
	
	mov		dx,03C4h			; VGA port 3C4h: Sequencer address
	mov		cx,320 / 4			; Repeat Count
	mov		bp,cx				; Remember repeat count in BP
	mov		ax,0102h			; Map mask address
	out		dx,ax				; Enable write to plane 0
	
	mov		si,bx				; SI = read buffer
	
	push	di					; Remember start address of this scanline for
								; the next planes

.plane_0_loop:
	lodsd						; Load 4 pixels from DS:SI
	
	stosb						; Store first pixel in the 4 pixel group at ES:DI
								; using write plane 0

	mov		ah,byte 0			; Update logo checksum for future logo sessions
	add		word [logo_chksum],ax
	loop	.plane_0_loop		; Do the whole plane 0 scanline

	pop		di					; DI = start address of this scanline

	mov		ax,0202h			; Map mask address
	out		dx,ax				; Enable write to plane 1
	
	lea		si,[bx + 1]			; SI = read buffer + 1 for the second pixel
	mov		cx,bp				; CX = repeat count
	
	push	di					; Remember start address of this scanline for
								; the next planes

.plane_1_loop:
	lodsd						; Load 4 pixels from DS:SI
	
	stosb						; Store second pixel in the 4 pixel group at ES:DI
								; using write plane 1

	mov		ah,byte 0			; Update logo checksum for future logo sessions
	add		word [logo_chksum],ax
	loop	.plane_1_loop		; Do the whole plane 1 scanline

	pop		di					; DI = start address of this scanline
	
	mov		ax,0402h			; Map mask address
	out		dx,ax				; Enable write to plane 2
	
	lea		si,[bx + 2]			; SI = read buffer + 2 for the third pixel
	mov		cx,bp				; CX = repeat count
	
	push	di					; Remember start address of this scanline for
								; the next planes

.plane_2_loop:
	lodsd						; Load 4 pixels from DS:SI
	
	stosb						; Store third pixel in the 4 pixel group at ES:DI
								; using write plane 2

	mov		ah,byte 0			; Update logo checksum for future logo sessions
	add		word [logo_chksum],ax
	loop	.plane_2_loop		; Do the whole plane 2 scanline

	pop		di					; DI = start address of this scanline

	mov		ax,0802h			; Map mask address
	out		dx,ax				; Enable write to plane 3
	
	lea		si,[bx + 3]			; SI = read buffer + 3 for the fourth pixel
	mov		cx,bp				; CX = repeat count
	
	push	di					; Remember start address of this scanline for
								; the next planes

.plane_3_loop:
	lodsd						; Load 4 pixels from DS:SI
	
	stosb						; Store fourth pixel in the 4 pixel group at ES:DI
								; using write plane 3

	mov		ah,byte 0			; Update logo checksum for future logo sessions
	add		word [logo_chksum],ax
	loop	.plane_3_loop		; Do the whole plane 3 scanline

	pop		di					; DI = start address of this scanline
	
	pop		bp

	ret
;******************************************************************************

;******************************************************************************
;	doslogo_4_plane_scanline_checksum
;
;	This routine will checksum the pixels in a scanline of 4 planes pointed to
;	by ES:SI. 
;	Returns the checksum in BX.
;******************************************************************************
doslogo_4_plane_scanline_checksum:
	push	cx					; Save scanline count

	cld							; Forward direction

	mov		cx,320 / 4			; Repeat Count
	mov		dx,03CEh			; VGA port 3CEh: Graphics address
	mov		ax,0004h			; Read map select address

.next_plane:
	out		dx,ax				; Read plane set by AH

	push	ax					; Remember graphics port address and plane number
	push	cx					; Remember scanline count
	push	si					; Remember start offset of scanline

	mov		ah,byte 0

.read_single_plane_pixels:
es	lodsb						; Load byte from ES:SI into al

	add		bx,ax				; Update checksum
	dec		cx
	jnz		.read_single_plane_pixels

	pop		si					; Restore start offset of scanline
	pop		cx					; Restore scanline count
	pop		ax					; Restore graphics port address and plane number

	inc		ah					; Increase plane

	cmp		ah,byte 04h			; Have we done all planes?
	jb		.next_plane			; Jump if not

	add		si,cx				; Next time, do next scanline offset

	pop		cx					; Restore scanline count

	ret
;******************************************************************************

anim_start	db	0				; Start offset of palette entry indexes for animation
anim_size	db	0				; Number of palette entry indexes to rotate
anim_rot_sv	db	0				; Saved animation rotation info between logo sessions
			db	1
anim_rot	db	0				; Animation rotation info 
			db	1

;******************************************************************************
int_08h:
			db	_IRET

logo_int_08h_dispatcher:
			dw	logo_int_animate	; Offset
			dw	0					; Segment
	
;******************************************************************************
;	logo_int_animate
;
;	This is the routine that animates the logo by rotating palette entries.
;	It is called from interrupt 08h and uses the master clock count from
;   the bios to time the animation, as well as checking for vertical blank.
;******************************************************************************
logo_int_animate:
	push	eax					; EAX is the main register for palette rotation storage
	
	push	ds
	
	xor		ax,ax
	mov		ds,ax				; DS = Seg0000
	
	mov		ax,[46Ch]			; AX = (0040h:006Ch) Low word of master clock count
	test	al,1				; Only animate during odd count
	jnz		.exit_anim
	
	sti							; Allow other interrupts to be processed at this time
	
	push	bx
	push	cx
	push	dx
	
	push	cs
	pop		ds					; DS = CS

	mov		dl,[anim_start]		; Get the start entry to rotate palette entries from
	or		dl,dl				; If no start, there won't be any animation
	jz		.end_anim
	
	push	dx
	
	mov		dx,03DAh			; VGA video status bits
	mov		bx,64				; Timeout
	xor		cx,cx				; We won't wait forever for vertical blank

.test_invb:	
	in		al,dx				; Get video status bits
	
	test	al,08h				; Test for vertical sync pulse
	
.opcode_invb:
	jz		.in_vb

	loop	.test_invb			; Loop a while until vb is triggered
	
	dec		bx					; Still no vb... To not hang the system, we have a failsafe timeout
	jnz		.test_invb
	
	; Reached the timeout. We will never wait for the vertical blank again during this lifetime
	
	mov		byte [cs:.opcode_invb],byte _JMPSHORT
	
.in_vb:
	mov		bx,64				; Timeout
	xor		cx,cx				; We won't wait forever for vertical blank to end
	
.test_outvb:	
	in		al,dx				; Get video status bits
	
	test	al,08h				; Test for vertical sync pulse
	
.opcode_outvb:
	jnz		.out_vb
	
	loop	.test_outvb			; Loop a while until vb is over
	
	dec		bx					; Still caught in vb... To not hang the system, we have a failsafe timeout
	jnz		.test_outvb
	
	; Reached the timeout. We will never wait for the vertical blank to end again during this lifetime
	
	mov		byte [cs:.opcode_outvb],byte _JMPSHORT

.out_vb:
	pop		dx					; DL is the start index of palette entries to animate

	call	GetRGBComponents	; Get the first palette entry's components

	push	eax					; Save this entry for last (to make the animation wrap)
	
	mov		cl,[anim_size]		; Get the animation size. This is all the palette entries - 1
	mov		bl,[anim_rot + 1]	; Get the animation stepping (default == 1)
	
.do_size:
	add		dl,bl				; Go to next entry in the palette using stepping
	
	call	GetRGBComponents	; Get the next palette entry's components
	
	sub		dl,bl				; Now make the previous entry the next entry
	
	call	SetRGBComponents	; Set the previous palette entry's components
	
	add		dl,bl				; Next pair of entries to move
	
	dec		cl					; And decrement the number of entries left
	jnz		.do_size
	
	pop		eax					; We still need to wrap the last palette entry with the first
	
	call	SetRGBComponents	; Set the last palette entry's components
	
	; The animation supports reverse mode. This mode
	; allows the palette rotation to swap direction after
	; a custom step, and every full size iteration.
	
	cmp		byte [anim_rot],byte 0
	jz		.end_anim
	
	dec		byte [anim_rot]
	jnz		.end_anim
	
	neg		byte [anim_rot + 1]
	mov		al,[anim_size]
	mov		byte [anim_rot],al
	
	mul		bl
	
	add		byte [anim_start],al

.end_anim:
	pop		dx
	pop		cx
	pop		bx
	
.exit_anim:
	pop		ds
	
	pop		eax
		
	; Chain to the original interrupt 08h vector
	
	jmp		far [cs:I08]
;******************************************************************************

;******************************************************************************
GetRGBComponents:
	mov		al,dl				; Save the entry index. EAX == ??????II
	
	mov		dx,03C7h			; VGA get palette index
	out		dx,al
	
	inc		dx					; DX = 03C8h
	inc		dx					; DX = 03C9h. VGA get RGB components
	
	ror		eax,8				; EAX == II??????
	
	jmp		short $+2			; Wait bus
	jmp		short $+2
	jmp		short $+2

	in		al,dx				; EAX == II????RR
	
	ror		eax,8				; EAX == RRII????
	
	jmp		short $+2			; Wait bus
	jmp		short $+2
	jmp		short $+2

	in		al,dx				; EAX == RRII??GG
	
	ror		eax,8				; EAX == GGRRII??
	
	jmp		short $+2			; Wait bus
	jmp		short $+2
	jmp		short $+2

	in		al,dx				; EAX == GGRRIIBB

	ror		eax,8				; EAX == BBGGRRII

	mov		dl,al				; Restore the entry index	

	ret
;******************************************************************************

;******************************************************************************
SetRGBComponents:
	mov		al,dl				; Save the entry index. EAX == BBGGRR..
	
	mov		dx,03C8h			; VGA set palette index
	out		dx,al
	
	inc		dx					; DX = 03C9h. VGA set RGB components
	
	ror		eax,8				; EAX == ..BBGGRR
	
	jmp		short $+2			; Wait bus
	jmp		short $+2
	jmp		short $+2

	out		dx,al				; EAX == ..BBGG..
	
	ror		eax,8				; EAX == ....BBGG
	
	jmp		short $+2			; Wait bus
	jmp		short $+2
	jmp		short $+2

	out		dx,al				; EAX == ....BB..
	
	ror		eax,8				; EAX == ......BB
	
	jmp		short $+2			; Wait bus
	jmp		short $+2
	jmp		short $+2

	out		dx,al				; EAX == ........

	ror		eax,8				; EAX == BBGGRRII

	mov		dl,al				; Restore the entry index	

	ret
;******************************************************************************

;******************************************************************************
int_10h:
			db	_IRET

logo_int_10h_dispatcher:
			dw	logo_int_video		; Offset
			dw	0					; Segment
	
;******************************************************************************
;	logo_int_video
;
;	The interrupt 10h is different when in a logo session. This is basically
;	true because we need to be able to swap back to previous video mode if a
;	program or driver try to change the video mode.
;	For text mode 02h, 03h, and 07h, we also must handle text scroll down.
;	This is mainly to reduce space that we only process the most common
;	text modes, as well as only display page 0.
;******************************************************************************
logo_int_video:
	sti							; Allow other interrupts to be processed at this time

	or		ah,ah				; Is this service 00h: Set Video Mode?
	jnz		.test_functions		; If no, jump to test other video functions

	cmp		al,7Fh				; Only if the mode was 7Fh, we don't go back to previous mode
	jz		.jump_orig_int10h
	
.terminate_logo:
	call	doslogo_restore_previous_video

.jump_orig_int10h:	
	jmp		.do_original_int10h

.test_functions:
	cmp		ah,11h				; If this service is 11h: Character Generator Interface
	jz		.terminate_logo		; It is counted as a mode change, so go back to previous video mode
	
	cmp		ah,0Eh				; Now we look for service 0Eh: Write Character in Teletype Mode
	jnz		.jump_orig_int10h	; If it is another service, we let original int 10h deal with it

	cmp		al,byte _BEEP		; A beep character will be handled by the original int 10h
	jz		.jump_orig_int10h

	cmp		al,byte _BACKSPACE	; A backspace character will be handled by the original int 10h
	jz		.jump_orig_int10h
	
	pusha						; Save all registers

	push	ds
	push	es

	cmp		al,byte _CR			; Carriage return will be handled by the original int 10h
	jz		.leave_to_original_int10h
	
	xor		dx,dx
	mov		ds,dx				; DS = Seg0000

	mov		dx,[450h]			; DX = (0040h:0050h) Current column, row position for text mode page 0

	cmp		dh,[484h]			; (0040h:0084h) Max rows in this text mode - 1
	jnz		.leave_to_original_int10h
	
	; The cursor is on the last row. One linefeed will force us to scroll the window, but also
	; if a character ends up outside the max columns
	
	mov		ah,[449h]			; AH = (0040h:0049h) Current video mode
	
	cmp		ah,byte 07h			; Text mode 80x25 Mono
	jz		.test_scroll

	cmp		ah,byte 02h			; Text mode 80x25 16 colors (for monochrome monitors)
	jb		.leave_to_original_int10h
	
	cmp		ah,byte 03h			; Text mode 80x25 16 colors (for color monitors)
	ja		.leave_to_original_int10h
	
.test_scroll:
	; We only try to scroll 80 column modes. Not modified ones. Same goes for
	; display page, which must be 0
	
	cmp		word [44Ah],80		; (0040h:004Ah) Max columns in this text mode
	jnz		.leave_to_original_int10h

	cmp		byte [462h],0		; (0040h:0062h) Current display page
	jnz		.leave_to_original_int10h

	; Only line feed of last columns will be scrolled
	
	cmp		al,byte _LF
	jz		.prepare_scroll
	
	cmp		dl,byte 79			; Is this the last column?
	jnz		.leave_to_original_int10h
	
.prepare_scroll:
	mov		si,0B000h			; Text mode video buffer for mono mode
	cmp		ah,byte 07h			; Choose which video buffer to use depending on video mode
	jz		.begin

	mov		si,0B800h			; Text mode video buffer for color mode

.begin:
	mov		ds,si
	mov		es,si
	
	mov		si,80*2				; SI = A full row. A text mode character is 2 bytes
	xor		di,di				; ES:DI = BX00h:0000h
	movzx	cx,dh				; Extend byte (max rows - 1) to unsigned word

	imul	cx,si				; CX = ((max rows - 1) * row size)

	mov		bx,cx				; BX = video buffer size
	shr		cx,2				; CX = count (video buffer size / double word)

	cmp		al,byte _LF			; Scroll down the text buffer if the character is a linefeed
	jz		.scroll_down

	add		dl,dl				; DL = row size
	movzx	dx,dl				; Extend byte (row size) to unsigned word
	add		bx,dx				; BX = Character offset in video buffer
	mov		byte [bx],al		; Store linefeed in the video buffer
	sub		bx,dx				; Restore BX
	
	mov		ax,0E0Dh			; This will call our interrupt 10h again, and it will run the original
	int		10h					; interrupt 10h to do the carriage return. This call is recursive

	; Now we are in the correct column as done by the original interrupt 10h. Now scroll down
	
.scroll_down:
	cld							; Forward direction
	rep		movsd				; Move BX00h:00A0h of size CX to BX00h:0000h (scroll window down one row)

	mov		al,20h				; Character whitespace
	mov		ah,byte [bx + 1]	; Use character attribute from beginning of last row
	mov		cx,80				; CX = count (one row)
	rep		stosw				; Blank the new last row with character, attrib from AX

	pop		es
	pop		ds

	popa						; Restore all registers

	iret						; Exit without calling the original interrupt 10h
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
.leave_to_original_int10h:
	pop		es
	pop		ds
	
	popa						; Restore all registers

.do_original_int10h:
	; Chain to the original interrupt 10h vector
	
	jmp		far [cs:I10]
;******************************************************************************

;******************************************************************************
int_15h:
			db	_IRET

logo_int_15h_dispatcher:
			dw	logo_int_escape		; Offset
			dw	0					; Segment
	
;******************************************************************************
;	logo_int_escape
;
;	At any time during a logo session, the user can press ESC to go back to
;	the previous video mode without any information being destroyed or cleared.
;	This way the user can simply see what he/she missed and can also continue
;	to follow text mode output.
;******************************************************************************
logo_int_escape:
	cmp		ah,4Fh				; Keyboard intercept
	jnz		.do_original_int15h

	stc							; Inform interrupt 09h to process this keystroke
	
	cmp		al,byte 01h			; See if the user pressed the ESC key
	jnz		.do_original_int15h

	call	doslogo_restore_previous_video

	; Now to make this escape keystroke disappear for good so that
	; interrupt 09h will never know about it
	
	push	bp
	mov		bp,sp
	and		byte [bp + 6],byte 0FEh
	pop		bp

	iret
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
.do_original_int15h:	
	; Chain to the original interrupt 15h vector
	
	jmp		far [cs:I15]
;******************************************************************************

; Read keyboard buffer delay values

start_time	dw	0				; Remembers the last time between int 16h service 01h calls
counter		dw	0				; Delay counter
min_delay	dw	0C8h			; Minimum delay before restoration of video mode when being
								; in interrupt 16h service 01h
sav_delay	dw	0C8h			; Original saved delay

;******************************************************************************
int_16h:
			db	_IRET

logo_int_16h_dispatcher:
			dw	logo_int_kbd		; Offset
			dw	0					; Segment
	
;******************************************************************************
;	logo_int_kbd
;
;	The logo seesion will be over once an interrupt 16h service 00h, 01h, or
;	02h is invoked.
;******************************************************************************
logo_int_kbd:
	sti							; Allow interrupts
	pushf						; Save flags for original interrupt 16h

	push	ax
	push	ds
	push	es

	push	cs
	pop		ds					; DS = CS
		
	and		ah,0EFh				; Get the service number
	cmp		ah,byte 01h			; Service 00h restore previous video mode
	jb		.terminate_logo

	jz		.do_delay			; Services 01h and 02h wait a short time delay before
								; restoring the previous video mode. The delay time
								; depends on the keyboard bios settings. This is so that
								; the logo can be kept active if the user answerd the
								; keystroke call beforehand
	cmp		ah,byte 02h
	ja		.do_original_int16h	; All other services are ignored by the logo session

.do_delay:
	; Because these services require repeatable calls
	; to interrupt 16h, we must setup the start time
	; for the delay
	
	xor		ax,ax
	mov		es,ax				; ES = Seg0000
	
	mov		ax,[es:46Ch]		; AX = (0040h:006Ch) Low word of master clock count
	cmp		ax,[start_time]		; See if the timing has begun
	jnz		.begin_timer

	inc		word [counter]
	mov		ax,[min_delay]
	cmp		word [counter],ax
	jb		.do_original_int16h
	
	jmp		.terminate_logo

.begin_timer:
	mov		word [start_time],ax
	mov		word [counter],0	; Start the time delay on the next int 16h call
	jmp		.do_original_int16h
	
.terminate_logo:
	call	doslogo_restore_previous_video

.do_original_int16h:	
	pop		es
	pop		ds
	pop		ax

	popf						; Restore flags
	cli							; Disallow interrupts	

	; Chain to the original interrupt 16h vector

	jmp		far [cs:I16]
;******************************************************************************

;******************************************************************************
;	int_2fh
;
;	Multiplex services 4A32h, 16F1h, and 160Eh: DOS-LOGO Support.
;
;	16F1h is used by Windows 95's win.com as part of the Win386 multiplex 
;			support.
;
;	160Eh is used by Windows 98's win.com as part of the Win386 multiplex 
;			support. Windows Millennium Edition skips win.com and directly
;			executes the win386 extender VMM32.VXD. But MS-DOS 8.00 still has
;			support for service 160Eh like MS-DOS 7.10.
;
;	4A32h is DOS's own access to the logo support and supports all versions of
;			the customer Windows. It is recommended to use this multiplex 
;			service in your programs to access DOS-LOGO Support. All the other
;			multiplex services are there for compatibility with win.com.
;
;	Subservice 06h is new in this version of DOS-LOGO. It allows redisplaying
;			and session start of the logo. And it does all the work for you.
;			All the other subservises are there for compatibility with
;			win.com. More information about each subservice can be found at
;			http://logosys.logotypes.se/
;******************************************************************************
int_2fh:
	cmp		ah,byte 16h
	jnz		.not_win386call
	
	cmp		al,byte 0Eh
	jz		.doslogo_support
	
	cmp		al,byte 0F1h
	jz		.doslogo_support

	jmp		.do_original_int2Fh
	
.not_win386call:
	cmp		ax,4A32h
	jz		.doslogo_support

.do_original_int2Fh:	
	; Chain to the original interrupt 16h vector

	jmp		far [cs:I2F]
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
.doslogo_support:
	push	ds
	
	push	cs
	pop		ds					; DS = CS

	cmp		bl,byte 01h
	jz		.doslogo_01h
	
	ja		.next_not_00h_01h
	
.doslogo_00h:
	; Subservice 00h: DOS-LOGO Is Session Get Checksum.
	; Returns DX == logo pixel checksum. Else DX == 0.
	; Returns AX == 0 if logo is in session. Else AX == -1.

	mov		dx,[logo_chksum]	; Return the pixel logo checksum in DX
	xor		ax,ax				; Return 0 in AX if the logo is in session
	test	byte [logo_flags],20h
	jz		.doslogo_iret		; Jump if logo is in session

	dec		ax					; Else return -1 in AX
	jmp		.doslogo_iret

.next_not_02h_03h:
	cmp		bl,byte 05h
	jb		.doslogo_04h

	ja		.next_not_04h_05h

.doslogo_05h:
	; Subservice 05h: DOS-LOGO Turn Off Session.

	and		byte [logo_flags],byte ~20h
	jmp		.doslogo_02h		; Jump to close the logo int dispatchers

.next_not_04h_05h:
	cmp		bl,byte 06h
	jz		.doslogo_06h

	jmp		.doslogo_iret		; No more subservises, so pop registers and iret

.doslogo_06h:
	; Subservice 06h: DOS-LOGO Redisplay and Enter Session.

	pusha

	call	doslogo_redisplay	; Redisplay the logo and put it in session

	popa
	
	jmp		.doslogo_iret

.doslogo_04h:
	; Subservice 04h: DOS-LOGO Turn On Session.
	; Returns AX == previous animation rotation info.
	
	or		byte [logo_flags],byte 20h
	mov		ax,[sav_delay]		; Restore minimum read keyboard buffer delay
	mov		word [min_delay],ax
	mov		al,byte 0FFh		; Recalculate animation palette start offset
	sub		al,[anim_size]
	mov		byte [anim_start],al
	mov		ax,[anim_rot_sv]	; Get saved animation rotation info from last logo session
	mov		word [anim_rot],ax	; And use it for this session

.doslogo_01h:
	; Subservice 01h: DOS-LOGO Resume Session.

	call	doslogo_int_open_dispatchers
	
	jmp		.doslogo_iret

.next_not_00h_01h:
	cmp		bl,byte 03h
	jz		.doslogo_03h

	ja		.next_not_02h_03h

.doslogo_02h:
	; Subservice 02h: DOS-LOGO Pause Session.

	call	doslogo_int_close_dispatchers
	
	jmp		.doslogo_iret

.doslogo_03h:
	; Subservice 03h: DOS-LOGO Restore Previous Video Mode.
	; Will automatically close doslogo int dispatchers and
	; restore previous video mode.

	call	doslogo_restore_previous_video

.doslogo_iret:
	pop		ds

	iret						; Exit without calling the original interrupt 2Fh
;******************************************************************************

;******************************************************************************
;	doslogo_restore_previous_video
;
;	Returns the carry flag set if video was successfully restored.
;******************************************************************************
doslogo_restore_previous_video:
	push	ds					; Save original DS contents
	
	push	cs
	pop		ds					; DS = CS
	
	btr		word [logo_flags],5	; Copy bit 5 into carry flag and reset it in logo_flags
	
	pop		ds					; Restore original DS contents
	
	jnc		near .return		; Jump if not in logo session

	inc		byte [cs:stack_ref]	; Keep track of stack restoration
	jnz		.no_stack_setup
	
	mov		word [cs:original_ss],ss
	mov		word [cs:original_sp],sp
	
	push	cs
	pop		ss					; SS = CS
	
	mov		sp,original_sp		; SP = offset original_sp

.no_stack_setup:
	call	doslogo_int_close_dispatchers

	pusha
	push	ds
	push	es
	
	push	ss
	pop		es					; ES = SS
	
	sub		sp,38				; 38 = size of bios video info
	mov		di,sp				; ES:DI will be the destination for the text mode info

	call	doslogo_save_textmode_info

	; After the call to save the text mode info, DS points to Seg0000

	cmp		byte [484h],42		; Shall we restore 43 rows text mode?
	jnz		.not_43_rows

	mov		ax,1201h
	mov		bl,byte 30h

	pushf						; Call interrupt 10h
	cli
	call	far [cs:I10]
	
	mov		dx,[488h]			; Save switch settings for 43 rows when setting 50 rows below
								
.not_43_rows:
	mov		ah,byte 0			; Set video mode
	mov		al,[449h]			; Get the "supposed" video mode
	or		al,byte 80h			; And do not erase the video buffer!
	
	pushf						; Call interrupt 10h
	cli
	call	far [cs:I10]
	
	call	doslogo_restore_textmode_info

	; After the call to restore the text mode info, DS points to ES and 
	; ES points to Seg0000
	
	add		sp,38				; 38 = size of bios video info

	cmp		byte [es:484h],42	; Shall we restore 50 rows text mode?
	jb		.not_50_rows

	ja		.set_50_rows
	
	mov		word [es:488h],dx	; 43 rows support

.set_50_rows:
	mov		ax,1112h
	mov		bl,byte 0

	pushf						; Call interrupt 10h
	cli
	call	far [cs:I10]

	jmp		.set_cursor

.not_50_rows:
	; Shall we restore 28 rows text mode?
	
	cmp		byte [es:484h],27
	jb		.set_cursor

	mov		ax,1111h
	mov		bl,byte 0

	pushf						; Call interrupt 10h
	cli
	call	far [cs:I10]

.set_cursor:
	; Time to restore the cursor location for this display page
	
	mov		bl,[es:462h]		; Get the "supposed" display page
	xor		bh,bh
	mov		di,bx				; * 2 for word alignment
	mov	dx,[es:bx + di + 450h]	; Get this display page's cursor position
	mov		bh,bl				; Page number in BH
	mov		ah,byte 02h			; Set cursor position

	pushf						; Call interrupt 10h
	cli
	call	far [cs:I10]
	
	pop		es
	pop		ds
	popa
	
	dec		byte [cs:stack_ref]	; Keep track of stack restoration
	jns		.no_stack_restore
	
	mov		ss,[cs:original_ss]
	mov		sp,[cs:original_sp]

.no_stack_restore:
	stc							; Set carry flag to indicate success

.return:	
	ret
;******************************************************************************

;******************************************************************************
;	doslogo_save_textmode_info
;
;	This routine saves video mode information in an array of 38 bytes located
;	at ES:DI. The information is taken from the bios area in segment 0040h.
;	This information is needed to manipulate a graphics mode into thinking
;	it is still in text mode. The data collected is:
;
;	(0040h:0010h) (W) Bios equipment-list
;	(0040h:0049h) (B) Current video mode
;	(0040h:004Ah) (W) Max columns in this text mode
;	(0040h:004Ch) (W) Video page buffer size in bytes
;	(0040h:004Eh) (W) Offset into start of active page video buffer
;	(0040h:0050h) (W * 8) Current column, row position for text mode pages 0 - 7
;	(0040h:0060h) (W) Cursor size, end scan line, start scan line
;	(0040h:0062h) (B) Current display page
;	(0040h:0063h) (W) Port address to hardware display controller chip
;	(0040h:0065h) (B) Current setting of CGA and MDA register
;	
;	(0040h:0084h) (B) Max rows in this text mode - 1
;	(0040h:0085h) (W) Character height in scanlines
;	(0040h:0087h) (B) Display adapter options (bit3 == 0 if EGA is active)
;	(0040h:0088h) (B) Switch settings from adapter card
;	(0040h:0089h) (B) Reserved
;	(0040h:008Ah) (B) Reserved
;******************************************************************************
doslogo_save_textmode_info:
	push	di					; Save DI
	
	xor		cx,cx
	mov		ds,cx				; DS = Seg0000

	cld							; Forward direction
	
	mov		si,410h				; DS:SI = (0040h:0010h)
	
	movsb
	
	mov		si,449h				; DS:SI = (0040h:0049h)
	mov		cl,byte 30			; Count
	
	rep		movsb
	
	mov		si,484h				; DS:SI = (0040h:0084h)
	mov		cl,byte 7			; Count

	rep		movsb
	
	pop		di					; Restore DI
	
	ret
;******************************************************************************

; Recursive video (text mode) information array and pointers

logo_stack	times 200h	db	0
original_sp				dw	0
original_ss				dw	0
stack_ref				db	0FFh	; Reference to stack restoration

;******************************************************************************
;	doslogo_restore_textmode_info
;
;	This routine restores video mode information from an array of 38 bytes
;	located in ES:DI. The information is put at the bios area in segment
;	0040h. This information is needed to manipulate a graphics mode into
;	thinking it is still in text mode. The data to replace is:
;
;	(0040h:0010h) (W) Bios equipment-list
;	(0040h:0049h) (B) Current video mode
;	(0040h:004Ah) (W) Max columns in this text mode
;	(0040h:004Ch) (W) Video page buffer size in bytes
;	(0040h:004Eh) (W) Offset into start of active page video buffer
;	(0040h:0050h) (W * 8) Current column, row position for text mode pages 0 - 7
;	(0040h:0060h) (W) Cursor size, end scan line, start scan line
;	(0040h:0062h) (B) Current display page
;	(0040h:0063h) (W) Port address to hardware display controller chip
;	(0040h:0065h) (B) Current setting of CGA and MDA register
;	
;	(0040h:0084h) (B) Max rows in this text mode - 1
;	(0040h:0085h) (W) Character height in scanlines
;	(0040h:0087h) (B) Display adapter options (bit3 == 0 if EGA is active)
;	(0040h:0088h) (B) Switch settings from adapter card
;	(0040h:0089h) (B) Reserved
;	(0040h:008Ah) (B) Reserved
;******************************************************************************
doslogo_restore_textmode_info:
	push	es
	pop		ds					; DS = ES
	
	mov		si,di				; DS:SI = array
	
	xor		cx,cx
	mov		es,cx				; ES = Seg0000

	cld							; Forward direction
	
	mov		di,410h				; ES:DI = (0040h:0010h)
	
	movsb
	
	mov		di,449h				; ES:DI = (0040h:0049h)
	mov		cl,byte 30			; Count
	
	rep		movsb
	
	mov		di,484h				; ES:DI = (0040h:0084h)
	mov		cl,byte 7			; Count

	rep		movsb

	ret
;******************************************************************************
	
int_dispatcher_tbl:
		dw	int_08h				; Far jump to interrupt or iret depending on dispatcher
		dw	08h * 4				; Interrupt vector offset
		dw	I08					; Original interrupt

		dw	int_10h				; Far jump to interrupt or iret depending on dispatcher
		dw	10h * 4				; Interrupt vector offset
		dw	I10					; Original interrupt

		dw	int_15h				; Far jump to interrupt or iret depending on dispatcher
		dw	15h * 4				; Interrupt vector offset
		dw	I15					; Original interrupt

		dw	int_16h				; Far jump to interrupt or iret depending on dispatcher
		dw	16h * 4				; Interrupt vector offset
		dw	I16					; Original interrupt
		
		dw	0					; End of table
			
;******************************************************************************
doslogo_int_open_dispatchers:
	cli							; Turn off interrupts since we are going to change interrupt vectors
	cld							; Forward direction
	
	pusha						; Save all registers
	
	push	ds
	push	es
	push	eax
	push	edx
	
	xor		ax,ax
	mov		es,ax				; ES = Seg0000
	
	mov		ax,cs				; AX = CS
	mov		ds,ax				; DS = CS
	
	test	byte [logo_flags],20h
	jz		.is_open			; Skip if logo is in session

	rol		eax,10h				; Put our CS segment in the high word of EAX as a
								; template for a quick far pointer
	mov		si,int_dispatcher_tbl

.next_int_dispatcher:
	lodsw						; Load dispatcher offset from DS:SI into AX
	
	or		ax,ax				; Was this the end of out info table?
	jz		.is_open			; If yes, then exit
	
	xchg	bx,ax				; Exchange AX with BX for use with effective addressing
	cmp	byte [bx],byte _IRET	; Is this dispatcher closed?
	jz		.open_int_dispatcher

	; The dispatcher was already open. This may be because
	; another program took over the interrupt vector
	; once the logo was in session the last time. If this
	; was the case, the dispatcher should point to another segment
	; than our CS. Else we don't need to do anything with this
	; dispatcher since it was left properly open
	
	mov		ax,cs
	
	cmp		word [bx + 3],ax
	jz		.do_next_int_dispatcher

	lea		ax,[bx + 5]			; Load the address of this logo int into AX
	mov		dword [bx + 1],eax	; Open the dispatcher to our logo int

.do_next_int_dispatcher:
	add		si,4
	jmp		.next_int_dispatcher
	
.open_int_dispatcher:
	mov	byte [bx],byte _JMPFAR	; Open this dispatcher up
	
	lodsw						; Load the interrupt vector offset from DS:SI into AX
	
	xchg	bx,ax				; Exchange AX with BX for use with effective addressing.
								; Since this was done before, EAX now hold the far address
								; to the logo int dispatcher

	xchg	eax,[es:bx]			; Swap the logo int dispatcher address with the cpu interrupt vector's 
								; address. The logo int dispatcher will be active once the interrupts
								; are turned on

	mov		bx,[si]				; Get the address to the original interrupt far pointer
	add		si,2
	
	mov		dword [bx],eax		; Save the original interrupt far pointer. Remember that
								; when a logo int session has been terminated, any changes
								; to the interrupt vector will chain to the original interrupt
								; and by that, we must treat this interrupt far pointer as
								; the original
								
	mov		ax,ds				; AX = DS = CS
	rol		eax,10h				; Put our CS segment in the high word of EAX as a
								; template for a quick far pointer

	jmp		.next_int_dispatcher

.is_open:
	pop		edx
	pop		eax
	pop		es
	pop		ds
	
	popa
	
	sti							; Turn on interrupts again
	
	ret
;******************************************************************************

;******************************************************************************
doslogo_int_close_dispatchers:
	cli							; Turn off interrupts since we are going to change interrupt vectors
	cld							; Forward direction
	
	pusha						; Save all registers
	
	push	ds
	push	es
	push	eax
	push	edx
	
	xor		ax,ax
	mov		es,ax				; ES = Seg0000
	
	mov		ax,cs				; AX = CS
	mov		ds,ax				; DS = CS
	
	rol		eax,10h				; Put our CS segment in the high word of EAX as a
								; template for a quick far pointer
	mov		si,int_dispatcher_tbl

.next_int_dispatcher:
	lodsw						; Load dispatcher offset from DS:SI into AX
	
	or		ax,ax				; Was this the end of out info table?
	jz		.is_closed			; If yes, then exit
	
	xchg	bx,ax				; Exchange AX with BX for use with effective addressing
	cmp	byte [bx],byte _IRET	; Is this dispatcher closed?
	jz		.was_already_closed	; If yes, then prepare to close the next logo int

	lodsw						; Load the interrupt vector offset from DS:SI into AX

	xchg	bx,ax				; Exchange AX with BX for use with effective addressing.
								; Since this was done before, EAX now hold the far pointer 
								; address to the logo int dispatcher

	cmp		eax,[es:bx]			; In a normal case, the cpu interrupt vector should
								; be the same as the logo int dispatcher offset.
	jz		.close_int_dispatcher
	
	; Else we have a case where one or more programs/drivers have taken over
	; the cpu interrupt vector and chained the logo int! Now the dispatcher's true
	; purpose will be exposed. We will keep the far call open, but it will address
	; the original interrupt instead of the logo int address. So those programs/drivers
	; will chain this dispatcher which will redirect to the real interrupt
	
	xchg	bx,ax				; Exchange AX with BX for use with effective addressing. BX now contains
								; the address to the logo int dispatcher

	lodsw						; Load the original interrupt vector from DS:SI into AX

	xchg	di,ax				; Exchange DI with BX for use with effective addressing

	mov		edx,[di]			; Get the far pointer into EDX and
	mov		dword [bx + 1],edx	; put it as the new dispatcher address
	jmp		.next_int_dispatcher
	
.close_int_dispatcher:
	xchg	bx,ax				; Get the far pointer address of the logo int back into BX
	mov	byte [bx],byte _IRET	; And close this dispatcher
	xchg	ax,bx				; Put the original vector offset back into BX
	
	lodsw						; Load the original interrupt vector from DS:SI into AX

	xchg	di,ax				; Exchange DI with BX for use with effective addressing
	mov		edx,[di]			; Get the far pointer into EDX and
	mov		dword [es:bx],edx	; put it back to the cpu interrupt vector pointed to by ES:BX
	jmp		.next_int_dispatcher

.was_already_closed:
	add		si,4
	jmp		.next_int_dispatcher

.is_closed:
	pop		edx
	pop		eax
	pop		es
	pop		ds
	
	popa
	
	sti							; Turn on interrupts again
	
	ret
;******************************************************************************

; The end of this device driver's resident portion

end_offs:

; Read buffer to be used during the first time displaying of the logo

read_buffer	times RBUFF_EVEN	db	0
