Program IVTUTIL;
{
  This is a util for processing and converting Interrupt Vector Table files
  in binary format (1024 byte files) and between IVT (Interrupt Vector Table)
  text INI files which support patching and easy analysis of saved vectors.

  PUBLIC DOMAIN - This program/source can be freely modified and distributed.

  Notes: The following code was very quickly written from scratch when I was
  away traveling, so no apologies for any hard coding, compiler specifics etc
  as this code was quickly written to do a job with what I just had to hand.

  ** This released version does NOT currently compile with FreePascal etc **
  Written with TP7.0 - do NOT trust output if compiled with earlier versions
  due to very specific use of things like SaveInt00 in this particular version

  No warranties are provided for this code - use ENTIRELY at your own RISK !

  Regards,
  Richard L. James
  http://www.wimborne.org/richard/
}

Uses DOS, Mathcvt;  { Obtain from http://www.devq.net/pascal/src/mathcvt.pas }

Type
  Vector_Rec=Record                        { Define a vector table pointer }
    Offset:Word;                           { Offset }
    Segment:Word;                          { Segment }
  End; {of Record Vector}
  Vector_Table_Array=Array[0..255] of Vector_Rec;
  Vector_Table_Array_PTR=Array[0..255] of Pointer;
  Filename_Rec=Record
    PathStr:PathStr;  {here for FSPLIT but we don't use this field ! }
    DirStr:DirStr;    {here for FSPLIT but we don't use this field ! }
    NameStr:NameStr;
    ExtStr:ExtStr;
    Size:LongInt;
  End; {of Record Filename}

Const
  VectorTableName:String[12]='vectable.dat';  { Filename of Vector Table}
  InterruptNumberCounter:Word=$0000;          { Default this value to 0}
  BytesRead:Word=$0000;                       { Default this value to 0}
  BytesWritten:Word=$0000;                    { Default this value to 0}
  ConversionType:String[8]='UNKNOWN';
  LineNumber:LongInt=0;
  HexChars:Array[0..$F] of Char='0123456789ABCDEF';  {used by HW+HB routines}
  DisplayLinebyLine:Boolean=FALSE;            { Flag for "more" type routine }

Var
  VectorTable:Vector_Table_Array; { Array[0..255] of vector values }
  VectorTable_PTR:Vector_Table_Array_PTR Absolute VectorTable;
  CurrentVectorTable:Vector_Table_Array Absolute $0000:$0000;
  CurrentVectorTable_PTR:Vector_Table_Array_PTR Absolute CurrentVectorTable;
  Parameter1:String;
  Parameter2:String;
  InputFile:Filename_Rec;
  OutputFile:Filename_Rec;
  IVTFile, CSVFile, TXTFile, REGFile:Text;
  VectorTableFile:File;
  OneLine:String;

{ ************************************************************************** }

Function HW(Word2Convert:Word):String;     { Converts a Word to a HexWord }

Begin
  HW:=HexChars[Hi(Word2Convert) shr 4]+HexChars[Hi(Word2Convert) and $F]
     +HexChars[Lo(Word2Convert) shr 4]+HexChars[Lo(Word2Convert) and $F];
End;

{ ************************************************************************* }

Function HB(Byte2Convert:Byte):String;     { Converts a Byte to a HexByte }
Begin
  HB:=hexChars[Byte2Convert shr 4]+hexChars[Byte2Convert and $F];
End;

{ ************************************************************************** }

Function UpCaseString(InString:String):String;

Var
   StrCounter:Byte;

Begin
  For StrCounter:=1 to Length(InString) do
    Begin
      InString[StrCounter]:=UpCase(InString[StrCounter]);
    End;
  UpCaseString:=InString;
End;

{ ************************************************************************** }

Function Points2IRET(Segment2Check, Offset2Check:Word):ByteBool;

Begin
  { Do we point to opcode $CF which is an IRET (Interrupt Return) ? }
  If Byte(Ptr(Segment2Check,Offset2Check)^)=$CF {IRET opcode} then
    Points2IRET:=ByteBool(1)
  else
    Points2IRET:=ByteBool(0);
End;

{ ************************************************************************** }

Function NullPointer(Offset2Check, Segment2Check:Word):ByteBool;
Begin
  { Do we have a NULL pointer ? }
  If (Segment2Check=$0000) and (Offset2Check=$0000) {Null pointer? } then
    NullPointer:=ByteBool(1)
  else
    NullPointer:=ByteBool(0);
End;

{ ************************************************************************** }

Procedure ProcessINILine(INILine:String);

Begin
  Writeln('INI SECTION: ',UpCaseString(INILine));
End;

{ ************************************************************************** }

Procedure ProcessInterruptLine(InterruptLine:String);

Var
  TmpString:String;
  InterruptNumberString:String;
  InterruptNumber:Byte;
  InterruptOffsetString:String;
  InterruptSegmentString:String;

Begin
  {Check for and remove any spaces if they exist}
  While Pos(' ', InterruptLine) > 0 do
    Delete(InterruptLine, Pos(' ', InterruptLine), 1);
    {123456789012345}
    {Int00=F000:9BD0}

  If Length(InterruptLine)>15 then
    Begin
      Writeln('** ERROR! Line ', LineNumber,' is greater than 15 characters (ignoring line completely)');
    End;

  If UpCaseString(Copy(InterruptLine, 1, 3))='INT' then
    Begin
      TmpString:=UpCaseString(Copy(InterruptLine, 4, Length(InterruptLine)-3));
      If Length(TmpString)>12 then  {another check - paranoid :-)}
        Begin
          Writeln('** ERROR! Line ',LineNumber,' has been ignored due to bad data: ',InterruptLine);
        End
      else
        Begin
          If (Pos('=', TmpString)<>3) or (Pos(':', TmpString)<>8) then
            Begin
              Writeln('** ERROR! Line ',LineNumber,' has been ignored due to bad data: ',InterruptLine);
            End
          else
            Begin
              InterruptNumberString:=Copy(TmpString,1,Pos('=',TmpString)-1);
              InterruptSegmentString:=Copy(TmpString,Pos('=',TmpString)+1,4);
              InterruptOffsetString:=Copy(TmpString,Pos(':',TmpString)+1,4);
{
              Write('Line ',LineNumber,' OK - ',TmpString, ' / ');
              Write('Interrupt: ',InterruptNumberString,' / ');
              Write('Segment: ',InterruptSegmentString,' / ');
              Writeln('Offset: ',InterruptOffsetString);
}
              VectorTable[Hex2Dec(InterruptNumberString)].Offset:=Hex2Dec(InterruptOffsetString);
              VectorTable[Hex2Dec(InterruptNumberString)].Segment:=Hex2Dec(InterruptSegmentString);
{
              Writeln('From Array ',HB(Hex2Dec(InterruptNumberString)), ' - ',
                       HW(VectorTable[Hex2Dec(InterruptNumberString)].Segment),':',
                       HW(VectorTable[Hex2Dec(InterruptNumberString)].Offset));
}
            End;
        End;
    End;
{  Writeln('ERROR: ',UpcaseString(InterruptLine));}
End;


{ ************************************************************************** }
Procedure ErrorHandler({ErrorNumber:Byte; }ErrorMessage:String);

Begin
  Writeln;
  Write('FATAL ERROR: ',ErrorMessage);
  Halt(255);  { ON error halt application and return to operating system }
End;

{ ************************************************************************** }

Procedure ReadIVTFile(IVTFile2Read:String);

Begin
  {$I-}         { Disable Input/Output checking }
  FileMode:=0;  { Open files as Read-Only for safety }
  Assign(IVTFile, IVTFile2Read);
  Reset(IVTFile);
  If (IOResult <> 0) then ErrorHandler(IVTFile2Read+' could NOT be opened');
  while NOT EOF(IVTFile) do
  Begin
    OneLine:=' ';
    Readln(IVTFile, OneLine);
    Inc(LineNumber);
    Case UpCase(OneLine[1]) of
      ';' : {do nothing};
      ':' : Writeln(':COMMENT', OneLine);
      '[' : ProcessINILine(OneLine);
      'I' : ProcessInterruptLine(OneLine);
      #13 : {do nothing};
      #10 : {do nothing};
      #32 : {do nothing};
      '!' : {execute...};
      'S' : Begin
              Writeln('** STOP: Stop command encountered EXITing processing at line ',LineNumber,'!');
              Close(IVTFile);
              If (IOResult <> 0) then ErrorHandler('IVTFile could NOT be closed');
              Exit;
            End;
      'Q' : Begin
              Writeln('** QUIT: Quit command encountered HALTing processing at line ',LineNumber,'!');
              Close(IVTFile);
              If (IOResult <> 0) then ErrorHandler('IVTFile could NOT be closed');
              Halt; {No ERROR, but we now wish to HALT via the QUIT command!}
            End;
      else {do nothing};  { Dump text file }
    End; {of case}
  End;
  Close(IVTFile);
  If (IOResult <> 0) then ErrorHandler('IVTFile could NOT be closed');
  {$I+}           { Enable Input/Output checking again }
End;

{ ************************************************************************** }

Procedure CreateCSVFile(CSVFile2Create:String);

Begin
  {$I-}           { Disable Input/Output checking }
  FileMode:=2;    { Read/Write access }

  Assign(CSVFile, CSVFile2Create); { Attempt to create CSV (text) file }
  ReWrite(CSVFile); { Note that no record size since we are a text (CSV) file}
  If (IOResult<>0) then ErrorHandler('Unable to create file '+CSVFile2Create);

  Writeln(CSVFile,'''Interrupt'',''Ptr'',''Segment'',''Offset'',''NullPtr'',''Ptr2IRET''');
  {Dump Interrupt vector table we read from disk }
  For InterruptNumberCounter:=0 to 255 do
    Begin
      Writeln(CSVFile,HB(InterruptNumberCounter), ',',
                HW(VectorTable[InterruptNumberCounter].Segment),':',
                HW(VectorTable[InterruptNumberCounter].Offset),',',
                HW(VectorTable[InterruptNumberCounter].Segment),',',
                HW(VectorTable[InterruptNumberCounter].Offset),',',
                NullPointer(VectorTable[InterruptNumberCounter].Segment,
                            VectorTable[InterruptNumberCounter].Offset),',',
                Points2IRET(VectorTable[InterruptNumberCounter].Segment,
                            VectorTable[InterruptNumberCounter].Offset));
    End;

  Close(CSVFile); { Attempt to close our vector table file }
  If (IOResult<>0) then ErrorHandler('Error closing file '+CSVFile2Create);
  {$I+}           { Enable Input/Output checking again }

  FileMode:=0;  { Revert to Read/Only access }

End;

{ ************************************************************************** }

Procedure CreateTXTFile(TXTFile2Create:String);

Begin
  {$I-}           { Disable Input/Output checking }
  FileMode:=2;    { Read/Write access }

  Assign(TXTFile, TXTFile2Create); { Attempt to create CSV (text) file }
  ReWrite(TXTFile); { Note that no record size since we are a text (CSV) file}
  If (IOResult<>0) then ErrorHandler('Unable to create file '+TXTFile2Create);

  Writeln(TXTFile, 'INT    VECTOR    POINTS TO');
  Writeln(TXTFile, '---    ------    ---------');
  {Dump Interrupt vector table we read from disk }
  For InterruptNumberCounter:=0 to 255 do
    Begin
      Write(TXTFile,'$',HB(InterruptNumberCounter), '   ',
                HW(VectorTable[InterruptNumberCounter].Segment),':',
                HW(VectorTable[InterruptNumberCounter].Offset));
      If NullPointer(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
                        Write(TXTFile,'  NULL ');

      If Points2IRET(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
                        Write(TXTFile,'  IRET ');
      Writeln(TXTFile);
      If DisplayLinebyLine=TRUE then
        Begin
          Asm
            Xor AX,AX  { Zero the accumulator }
            Int 16h    { Wait for a keypress! }
          End;
        End;
  End;

  Close(TXTFile); { Attempt to close our vector table file }
  If (IOResult<>0) then ErrorHandler('Error closing file '+TXTFile2Create);
  {$I+}           { Enable Input/Output checking again }

  FileMode:=0;  { Revert to Read/Only access }

End;

{ ************************************************************************** }

Procedure CreateREGFile(REGFile2Create:String);

Const
  RegByteCounter:Word=$0000;

Begin
  {$I-}           { Disable Input/Output checking }
  FileMode:=2;    { Read/Write access }

  Assign(REGFile, REGFile2Create); { Attempt to create REG (text) file }
  ReWrite(REGFile); { Note that no record size since we are a text (REG) file}
  If (IOResult<>0) then ErrorHandler('Unable to create file '+REGFile2Create);
  Writeln(REGFile, 'REGEDIT4');
  Writeln(REGFile);
  Writeln(REGFile, '[HKEY_USERS\.Default\Software\Ivtutil]');
  Writeln(REGFile);
  Writeln(REGFile, '[HKEY_USERS\.Default\Software\Ivtutil\InterruptVectorTable]');
  Write(REGFile, '"@"=hex:');
  RegByteCounter:=5;
  For InterruptNumberCounter:=0 to 255 do
    Begin
      Write(REGFile, HB(Lo(VectorTable[InterruptNumberCounter].Offset)),',',
            HB(Hi(VectorTable[InterruptNumberCounter].Offset)),',',
            HB(Lo(VectorTable[InterruptNumberCounter].Segment)),',',
            HB(Hi(VectorTable[InterruptNumberCounter].Segment)));
      RegByteCounter:=RegByteCounter+4;
      If InterruptNumberCounter<>255 then
        Begin
          If RegByteCounter>=24 then
            Begin
              Writeln(REGFile, ',\');
              RegByteCounter:=0;
            End
          else Write(REGFile, ',');
        End
      else
        {do nothing};
    End;
  Writeln(REGFile);

  Writeln(REGFile);

{  Writeln(REGFile, 'INT    VECTOR    POINTS TO');
  Writeln(REGFile, '---    ------    ---------');}
  {Dump Interrupt vector table we read from disk }
  For InterruptNumberCounter:=0 to 255 do
    Begin
      Writeln(REGFile, '[HKEY_USERS\.Default\Software\Flopper\InterruptVectorTable\',
                       'Int',HB(InterruptNumberCounter),']');
      Writeln(REGFile, '@="',
                       HW(VectorTable[InterruptNumberCounter].Segment),':',
                       HW(VectorTable[InterruptNumberCounter].Offset),'"');

      {
      Writeln(REGFile, '"DefaultSegment"="',
                       HW(VectorTable[InterruptNumberCounter].Segment),'"');
      Writeln(REGFile, '"DefaultOffset"="',
                       HW(VectorTable[InterruptNumberCounter].Offset),'"');
      }

      Writeln(REGFile, '"DefaultValueHex"=hex:',
                       HB(Lo(VectorTable[InterruptNumberCounter].Offset)),',',
                       HB(Hi(VectorTable[InterruptNumberCounter].Offset)),',',
                       HB(Lo(VectorTable[InterruptNumberCounter].Segment)),',',
                       HB(Hi(VectorTable[InterruptNumberCounter].Segment)));
      Writeln(REGFile, '"PatchedValueHex"=hex:00,00,00,00');

      If NullPointer(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
              Writeln(REGFile, '"DefaultNullPointer"=dword:00000001')
         else Writeln(REGFile, '"DefaultNullPointer"=dword:00000000');

      If Points2IRET(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
              Writeln(REGFile, '"DefaultPointsToIRET"=dword:00000001')
         else Writeln(REGFile, '"DefaultPointsToIRET"=dword:00000000');

      If (NullPointer(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset)) or
         (Points2IRET(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset)) then
         Begin
           Writeln(REGFile, '"InterruptEnabled"=dword:00000000');
         End
      else
        Begin
           Writeln(REGFile, '"InterruptEnabled"=dword:00000001');
        End;
      Writeln(REGFile, '"UsePatchedValue"=dword:00000000');
      Writeln(REGFile);
    End;
  Writeln(REGFile);

  Close(REGFile); { Attempt to close our vector table file }
  If (IOResult<>0) then ErrorHandler('Error closing file '+REGFile2Create);
  {$I+}           { Enable Input/Output checking again }

  FileMode:=0;  { Revert to Read/Only access }

End;

{ ************************************************************************** }

Procedure CreateIVTFile(IVTFile2Create:String);
Begin
  {$I-}           { Disable Input/Output checking }
  FileMode:=2;    { Read/Write access }

  Assign(IVTFile, IVTFile2Create); { Attempt to create IVT (text) file }
  ReWrite(IVTFile); { Note that no record size since we are a text (IVT) file}
  If (IOResult<>0) then ErrorHandler('Unable to create file '+IVTFile2Create);

  Writeln(IVTFile,'[IVT_PC_Info]');
  Writeln(IVTFile,'; This section is NOT currently used but is here for possible future use!');
  Writeln(IVTFile,'PC_Make=');
  Writeln(IVTFile,'CPU_Type=');
  Writeln(IVTFile,'BIOS_Make=');
  Writeln(IVTFile,'BIOS_VersionNumber=');
  Writeln(IVTFile,'BIOS_SerialNumber=');
  Writeln(IVTFile);
  Writeln(IVTFile,'[IVT_Setup]');
  Writeln(IVTFile,'; This section is NOT currently used but is here for possible future use!');
  Writeln(IVTFile,'; Note: IVT INI files use these special characters in position 1 of each line:');
  Writeln(IVTFile,';       ; = internal comment NOT displayed on console when processing file');
  Writeln(IVTFile,';       : = comment to DISPLAY as console output when processing this file');
  Writeln(IVTFile,';       [ = start of ini header followed by text and closing ] bracket');
  Writeln(IVTFile,';       I or i = paraser expects Interrupt in this format: Intxx=xxxx:xxxx');
  Writeln(IVTFile,';       S or s = stop paraser but still create outputfile up to this point');
  Writeln(IVTFile,';       Q or q = Quit - completely HALT paraser and do NOT create output');
  Writeln(IVTFile,';       X or x = eXecute another command ** NOT CURRENTLY IMPLEMENTED **');
  Writeln(IVTFile);
  Writeln(IVTFile,'[InitialBootupVectorTable]');
  Writeln(IVTFile,'; Following are the values of the Interrupt Vector Table prior to an OS boot');
  Writeln(IVTFile,'; IMPORTANT: If you wish to PATCH these values use patching section down below!');
  Writeln(IVTFile,'; IMPORTANT: All values are in HEX!! - be aware of weird people using decimal!!');
  {Dump Interrupt vector table we read from disk }
  For InterruptNumberCounter:=0 to 255 do
    Begin
      Writeln(IVTFile,'Int',HB(InterruptNumberCounter), '=',
                HW(VectorTable[InterruptNumberCounter].Segment),':',
                HW(VectorTable[InterruptNumberCounter].Offset));
    End;
  Writeln(IVTFile);
  Writeln(IVTFile,'[NullPointers]');
  For InterruptNumberCounter:=0 to 255 do
    Begin
      If NullPointer(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
      Begin
        Writeln(IVTFile,'Int',HB(InterruptNumberCounter), '=',
                HW(VectorTable[InterruptNumberCounter].Segment),':',
                HW(VectorTable[InterruptNumberCounter].Offset));
      End;
    End;
  Writeln(IVTFile);
  Writeln(IVTFile,'[IRETs]');
  For InterruptNumberCounter:=0 to 255 do
    Begin
      If Points2IRET(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
      Begin
        Writeln(IVTFile,'Int',HB(InterruptNumberCounter), '=',
                HW(VectorTable[InterruptNumberCounter].Segment),':',
                HW(VectorTable[InterruptNumberCounter].Offset));
      End;
    End;
  Writeln(IVTFile);
  Writeln(IVTFile,'[Patches2VectorTable]');
  Writeln(IVTFile,'; Add any Interrupt patches below this line, e.g Int05=FFFF:0000 to patch Int05');
  Writeln(IVTFile,'; IMPORTANT: the values in this section *override* any default PC values above!');
  Writeln(IVTFile,'; IMPORTANT: All values are in HEX!! - be aware of weird people using decimal!!');

  Close(IVTFile); { Attempt to close our vector table file }
  If (IOResult<>0) then ErrorHandler('Error closing file '+IVTFile2Create);
  {$I+}           { Enable Input/Output checking again }

  FileMode:=0;  { Revert to Read/Only access }
End;

{ ************************************************************************** }

Procedure ReadDATFile(DATFile2Read:String);

Begin
  {$I-}           { Disable Input/Output checking }
    FileMode:=0;  { Open files as Read-Only for safety }
    Assign(VectorTableFile, DATFile2Read); { Attempt to open vector table file }
    Reset(VectorTableFile, 1);  { Record size = 1 }
    BlockRead(VectorTableFile, VectorTable, SizeOf(VectorTable), BytesRead); { read it }
    Close(VectorTableFile); { Attempt to close our vector table file }
  {$I+}           { Enable Input/Output checking again }

  If (IOResult <> 0) or (BytesRead<>SizeOf(VectorTable)) then { Any Errors? }
    ErrorHandler(DATFile2Read+' NOT found or file is NOT 1024 bytes!');
End;

{ ************************************************************************** }

Procedure CreateDATfile(DATFile2Create:String);

Begin
  {$I-}           { Disable Input/Output checking }
  FileMode:=2;  { Read/Write access }

  Assign(VectorTableFile, DATFile2Create); { Attempt to create DAT file }
  ReWrite(VectorTableFile, 1);  { Record size = 1 }
  If (IOResult<>0) then ErrorHandler('Unable to create file '+DATFile2Create);

  BlockWrite(VectorTableFile, VectorTable, SizeOf(VectorTable), BytesWritten); { read it }
  If (IOResult<>0) or (BytesWritten<>SizeOf(VectorTable)) then { Errors? }
    ErrorHandler('Error during creation of file '+DATFile2Create);

  Close(VectorTableFile); { Attempt to close our vector table file }
  If (IOResult<>0) then ErrorHandler('Error closing file '+DATFile2Create);
  {$I+}           { Enable Input/Output checking again }

  FileMode:=0;  { Revert to Read/Only access }
End;

{ ************************************************************************** }

Procedure IVT2DAT(IVTFile2Read:String; DATFile2Create:String);

Begin
  ReadIVTFile(IVTFile2Read);
  CreateDATFile(DATFile2Create);
End;

{ ************************************************************************** }

Procedure IVT2MEM(IVTFile2Read:String; DATFile2Create:String);

Begin
  Writeln('*** Use this undocumented function with extreme care!!!! ***');
  ReadIVTFile(IVTFile2Read);
  For InterruptNumberCounter:=0 to 255 do
    Begin
      If NullPointer(VectorTable[InterruptNumberCounter].Segment,
                     VectorTable[InterruptNumberCounter].Offset) then
        Begin
          { Ignore any Interrupt values NOT present as we want to EXCLUDE!}
        End
      else
        Begin
          Writeln('PATCHing current Interrupt $',HB(InterruptNumberCounter),
                  ' from ',
                  HW(CurrentVectorTable[InterruptNumberCounter].Segment),':',
                  HW(CurrentVectorTable[InterruptNumberCounter].Offset),
                  ' to ',HW(VectorTable[InterruptNumberCounter].Segment),
                     ':',HW(VectorTable[InterruptNumberCounter].Offset));
          SetIntVec(InterruptNumberCounter,
                    Ptr(VectorTable[InterruptNumberCounter].Segment,
                        VectorTable[InterruptNumberCounter].Offset));
        End;
      {FBh = STI}
      {FAh = CLI}
    End;
  Writeln('*** Use this undocumented function with extreme care!!!! ***');
End;

{ ************************************************************************** }

Procedure MEM2DAT(WeIgnoreThis:String; DATFile2Create:String);

Begin
  Move(CurrentVectorTable, VectorTable, SizeOf(Vector_Table_Array));
  CreateDATFile(DATFile2Create);
End;

{ ************************************************************************** }

Procedure MEM2TXT(WeIgnoreThis:String; TXTFile2Create:String);

Begin
  Move(CurrentVectorTable, VectorTable, SizeOf(Vector_Table_Array));
  CreateTXTFile(TXTFile2Create);
End;

{ ************************************************************************** }

Procedure MEM2REG(WeIgnoreThis:String; REGFile2Create:String);

Begin
  Move(CurrentVectorTable, VectorTable, SizeOf(Vector_Table_Array));
  CreateREGFile(REGFile2Create);
End;

{ ************************************************************************** }

Procedure MEM2CSV(WeIgnoreThis:String; CSVFile2Create:String);

Begin
  Move(CurrentVectorTable, VectorTable, SizeOf(Vector_Table_Array));
  CreateCSVFile(CSVFile2Create);
End;

{ ************************************************************************** }

Procedure MEM2IVT(WeIgnoreThis:String; IVTFile2Create:String);

Begin
  Move(CurrentVectorTable, VectorTable, SizeOf(Vector_Table_Array));
  CreateIVTFile(IVTFile2Create);
End;

{ ************************************************************************** }

Procedure DAT2IVT(DATFile2Read:String; IVTFile2Create:String);

Begin
  ReadDATFile(DATFile2Read);
  CreateIVTFile(IVTFile2Create);
End;

{ ************************************************************************** }

Procedure DAT2CSV(DATFile2Read:String; CSVFile2Create:String);

Begin
  ReadDATFile(DATFile2Read);
  CreateCSVFile(CSVFile2Create);
End;

{ ************************************************************************** }

Procedure DAT2TXT(DATFile2Read:String; TXTFile2Create:String);

Begin
  ReadDATFile(DATFile2Read);
  CreateTXTFile(TXTFile2Create);
End;

{ ************************************************************************** }

Procedure DAT2REG(DATFile2Read:String; REGFile2Create:String);

Begin
  ReadDATFile(DATFile2Read);
  CreateREGFile(REGFile2Create);
End;


{ ************************************************************************** }

Procedure ShowUtilHeader;

Begin
  Writeln;
  Writeln('IVTUTIL v1.01 - utility for processing/converting Interrupt Vector Table files');
  Writeln('                Source and util released as PUBLIC DOMAIN by Richard L. James');
  Writeln;
End;

{ ************************************************************************** }

Procedure Usage;

Begin
  ShowUtilHeader;
  Writeln('Usage:');
  Writeln('       IVTUTIL vectable.dat vectable.ivt   =   Convert .dat to .IVT file');
  Writeln('       IVTUTIL vectable.dat vectable.txt   =   Convert .dat to .TXT file');
  Writeln('       IVTUTIL vectable.dat vectable.csv   =   Convert .dat to .CSV file');
  Writeln('       IVTUTIL vectable.dat vectable.reg   =   Convert .dat to .REG file');
  Writeln('       IVTUTIL vectable.dat con            =   Display .dat on console');
  Writeln('       IVTUTIL vectable.dat con more       =   as above but Line by Line');
  Writeln;
  Writeln('       IVTUTIL input.ivt vectable.dat      =   Create .DAT from .IVT file');
  Writeln;
  Writeln('       IVTUTIL mem con                  =   Output current IVT to CONsole');
  Writeln('       IVTUTIL mem con more             =   as above but Line by Line');
  Writeln('       IVTUTIL mem current.dat          =   Create .DAT from current IVT');
  Writeln('       IVTUTIL mem current.ivt          =   Create .IVT from current IVT');
  Writeln('       IVTUTIL mem current.txt          =   Create .TXT from current IVT');
  Writeln('       IVTUTIL mem current.csv          =   Create .CSV from current IVT');
  Writeln('       IVTUTIL mem current.reg          =   Create .REG from current IVT');
  Writeln;
  Write('Press a key to continue...');  { Wait for a key }
  Asm { Easier to spot errors from batch files or Windows if we wait... }
    Xor AX,AX   { Zero the accumulator }
    Int 16h     { Wait for a keypress! }
  End; { If you don't have TP and hate waits like this then you can ... }
  Writeln; { .. hex edit the binary changing 31 C0 CD 16 to 90 90 90 90 ! }
  Halt(255);
End;


{ ************************************************************************** }
Procedure IVT_Init;

Begin
  FileMode:=0;    {Default to Read-Only files - for safety!! }
  LineNumber:=0;  {Yes, do this again - for safety...! }

  {Reverse all the Interrupt changes made by Turbo Pascal's runtime library}
  SetIntVec($00,SaveInt00);
  SetIntVec($02,SaveInt02);
  SetIntVec($1B,SaveInt1B);
  SetIntVec($21,SaveInt21);
  SetIntVec($23,SaveInt23);
  SetIntVec($24,SaveInt24);
  SetIntVec($34,SaveInt34);
  SetIntVec($35,SaveInt35);
  SetIntVec($36,SaveInt36);
  SetIntVec($37,SaveInt37);
  SetIntVec($38,SaveInt38);
  SetIntVec($39,SaveInt39);
  SetIntVec($3A,SaveInt3A);
  SetIntVec($3B,SaveInt3B);
  SetIntVec($3C,SaveInt3C);
  SetIntVec($3D,SaveInt3D);
  SetIntVec($3E,SaveInt3E);
  SetIntVec($3F,SaveInt3F);
  SetIntVec($75,SaveInt75);
End;



Begin
  IVT_Init;  {Setup + reverse all the Interrupt changes made by Turbo Pascal }

  { Impose some filename rules as we use .DAT or .IVT to determine file type }
  { Shortist filename we support is X.DAT or X.IVT (param = 4 inc. fullstop! }
  { Longest filename we support is 12345678.910 due to DOS limits (12 inc .! }
  { We also expect the files to be in the current directory .. tough rules :)}
  Parameter1:=UpCaseString(ParamStr(1));  {Copy Parameter1 to allow upcasing}
  Parameter2:=UpCaseString(ParamStr(2));  {Copy Parameter2 to allow upcasing}

  If (Parameter1='MEMORY') or (Parameter1='MEM') then Parameter1:='MEMORY.MEM';
  If (Parameter2='MEMORY') or (Parameter2='MEM') then Parameter2:='MEMORY.MEM';
  If (Parameter2='CON') or (Parameter2='CONSOLE') then Parameter2:='CONSOLE.CON';

  If Parameter2<>'CONSOLE.CON' then
    Begin
      If (ParamCount<=1) or (ParamCount>=3) then Usage;
    End
  else
    Begin
      If (ParamCount<=1) or (ParamCount>=4) then Usage;
      If ParamCount=3 then DisplayLinebyLine:=TRUE;
    End;
  If ( Length(Parameter1) <5) or ( Length(Parameter2) >=13) then Usage;
  If ( Length(Parameter2) <5) or ( Length(Parameter2) >=13) then Usage;

  {Split the expected filenames in Parameter1 and Parameter2 into components}
  FSplit(Parameter1, InputFile.DirStr, InputFile.NameStr, InputFile.ExtStr);
  FSplit(Parameter2, OutputFile.DirStr, OutputFile.NameStr, OutputFile.ExtStr);

  {Check to see if our InputFile name is one of the extensions we support }
  If (UpCaseString(InputFile.ExtStr)='.DAT') then {do nothing}
  else if (UpCaseString(InputFile.ExtStr)='.IVT') then {do nothing}
  else if (UpCaseString(InputFile.ExtStr)='.MEM') then {do nothing}
  else Usage;  {If NOT valid extension display our usage screen to end user }

  {Check to see if our OutputFile name is one of the extensions we support }
  If (UpCaseString(OutputFile.ExtStr)='.DAT') then {do nothing}
  else if (UpCaseString(OutputFile.ExtStr)='.IVT') then {do nothing}
  else if (UpCaseString(OutputFile.ExtStr)='.TXT') then {do nothing}
  else if (UpCaseString(OutputFile.ExtStr)='.CSV') then {do nothing}
  else if (UpCaseString(OutputFile.ExtStr)='.REG') then {do nothing}
  else if (UpCaseString(OutputFile.ExtStr)='.MEM') then {do nothing}
  else if (UpCaseString(OutputFile.ExtStr)='.CON') then {do nothing}
  else Usage;  {If NOT valid extension display our usage screen to end user }

  If (InputFile.ExtStr='.DAT') and (OutputFile.ExtStr='.IVT') then
    ConversionType:='DAT2IVT'
  else
  If (InputFile.ExtStr='.DAT') and (OutputFile.ExtStr='.TXT') then
    ConversionType:='DAT2TXT'
  else
  If (InputFile.ExtStr='.DAT') and (OutputFile.ExtStr='.CSV') then
    ConversionType:='DAT2CSV'
  else
  If (InputFile.ExtStr='.DAT') and (OutputFile.ExtStr='.REG') then
    ConversionType:='DAT2REG'
  else
  If (InputFile.ExtStr='.IVT') and (OutputFile.ExtStr='.DAT') then
    ConversionType:='IVT2DAT'
  else
  If (InputFile.ExtStr='.MEM') and (OutputFile.ExtStr='.DAT') then
    ConversionType:='MEM2DAT'
  else
  If (InputFile.ExtStr='.MEM') and (OutputFile.ExtStr='.TXT') then
    ConversionType:='MEM2TXT'
  else
  If (InputFile.ExtStr='.MEM') and (OutputFile.ExtStr='.REG') then
    ConversionType:='MEM2REG'
  else
  If (InputFile.ExtStr='.MEM') and (OutputFile.ExtStr='.IVT') then
    ConversionType:='MEM2IVT'
  else
  If (InputFile.ExtStr='.MEM') and (OutputFile.ExtStr='.CSV') then
    ConversionType:='MEM2CSV'
  else
  If (InputFile.ExtStr='.MEM') and (OutputFile.ExtStr='.CON') then
    ConversionType:='MEM2CON'
  else
  If (InputFile.ExtStr='.DAT') and (OutputFile.ExtStr='.CON') then
    ConversionType:='DAT2CON'
  else
  If (InputFile.ExtStr='.IVT') and (OutputFile.ExtStr='.MEM') then
    ConversionType:='IVT2MEM' {Undocumented on purpose - USE WITH REAL CARE!}
  else Usage;  {Show usage as user is trying to do something we don't support}


  {If we are here then we know we do at least have a valid extension/usage! }
  If (ConversionType='MEM2CON') or (ConversionType='DAT2CON') then
    Begin
      {do NOT show header information when emulating INTS style output!}
    End
  else
    Begin
      ShowUtilHeader;
     {Display header and filenames... note we don't yet validate filename!}
      Writeln('INPUT  filename: ',InputFile.NameStr:8,InputFile.ExtStr);
      Writeln('OUTPUT filename: ',OutputFile.NameStr:8,OutputFile.ExtStr);
      Writeln('Conversion Type: ',ConversionType);
    End;


  If UpCaseString(ConversionType)='DAT2IVT' then
    Begin
      DAT2IVT(InputFile.NameStr+InputFile.ExtStr,
              OutputFile.NameStr+OutputFile.ExtStr);
    End
  else
    If UpCaseString(ConversionType)='DAT2TXT' then
      Begin
        DAT2TXT(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='DAT2CSV' then
      Begin
        DAT2CSV(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='DAT2REG' then
      Begin
        DAT2REG(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='IVT2DAT' then
      Begin
        IVT2DAT(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='MEM2DAT' then
      Begin
        MEM2DAT(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='MEM2REG' then
      Begin
        MEM2REG(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='MEM2TXT' then
      Begin
        MEM2TXT(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='MEM2CSV' then
      Begin
        MEM2CSV(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='MEM2IVT' then
      Begin
        MEM2IVT(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else
    If UpCaseString(ConversionType)='MEM2CON' then
      Begin
        MEM2TXT(InputFile.NameStr+InputFile.ExtStr,
                '' {standard output});
      End
  else
    If UpCaseString(ConversionType)='DAT2CON' then
      Begin
        DAT2TXT(InputFile.NameStr+InputFile.ExtStr,
                '' {standard output});
      End
  else
    If UpCaseString(ConversionType)='IVT2MEM' then
      Begin
        IVT2MEM(InputFile.NameStr+InputFile.ExtStr,
                OutputFile.NameStr+OutputFile.ExtStr);
      End
  else {We should never get here... however just in case... (helps with dev)}
    ErrorHandler('Expected IVT utility function NOT fully implemented');
End.


{ ************************************************************************** }
{ ************************************************************************** }
{ ************************************************************************** }
