/*
cron.c

This program performs the basic functions of
the UNIX cron program. This program is written
for 16-bit DOS/Windows machines and is compiled
with Borland Turbo C.

This version of cron does not support all of the
functions of the UNIX cron program. Some other
features are supported, but implemented differently.
Please see the cron.txt file for more information.

This program starts by opening a cron file
(crontab.txt by default).
It searches all the lines in the file. All
lines starting with a "#" are skipped.

The current time is compared to the time entry on the
crontab line. If the time matches, then the command
at the end of the line is run on the system. With no
match, cron skips to the next line of the crontab.

Cron checks the crontab file once a minute. Since the
file is not kept open during this time, it is possible
to update the crontab file without restarting cron.


Written by Jesse <slicer69@hotmail.com>
http://slicer69.tripod.com/
Date: March, 2003
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#define VERSION 0.1
#define LINE_LENGTH 256
#define COMMAND_LENGTH 128
#define DEFAULT_CRONTAB "crontab.txt"

/* #define DEBUG */



int main(int argc, char *argv[])
{
   FILE *cron_file;
   char input_line[LINE_LENGTH];
   char command_line[COMMAND_LENGTH];
   int minute, hour, day, month, day_of_week;
   int result, found;
   time_t time_value;
   struct tm *current_time;
   int run_cron = 1;

   while (run_cron)
   {
     /* start by opening file */
     if (argc < 2)
       cron_file = fopen(DEFAULT_CRONTAB, "r");
     else
       cron_file = fopen(argv[1], "r");

     if (!cron_file)
     {
        printf("Could not open crontab file.\n");
        return 1;
     }

     /* get system time */
     time( & time_value);
     current_time = localtime( & time_value);
     if (!current_time)
     {
        printf("Could not retrieve current time.\n");
        if (cron_file)
          free(cron_file);
        return 1;
     }

     /* add on to day of week and to month */
     /* this is to fix the Windows/DOS bug of */
     /* starting at month 0 and Monday as day 0 */
     current_time->tm_wday++;
     current_time->tm_mon++;

     #ifdef DEBUG
     printf("Current time: %d %d %d %d %d\n",
            current_time->tm_min, current_time->tm_hour,
            current_time->tm_mday, current_time->tm_mon,
            current_time->tm_wday);
     #endif
     
     /* read in lines from crontab */
     result = fgets(input_line, LINE_LENGTH, cron_file);
     while (result)
     {

       #ifdef DEBUG
       printf("Got line: %s", input_line);
       #endif

       /* check for comment line */
       if (input_line[0] != '#')
       {
          /* get time when events should happen */
          sscanf(input_line, "%d %d %d %d %d",
                 &minute, &hour, &day, &month, &day_of_week);


          #ifdef DEBUG
          printf("Time event should happen: %d %d %d %d %d\n",
                 minute, hour, day, month, day_of_week);
          #endif

          /* check if time matches */
          if (
               ((current_time->tm_min == minute) || (minute == -1)) &&
               ((current_time->tm_hour == hour) || (hour == -1)) &&
               ((current_time->tm_mday == day)  || (day == -1)) &&
               ((current_time->tm_mon == month) || (month == -1)) &&
               ((current_time->tm_wday == day_of_week) || (day_of_week == -1)) 
             )
          {
               #ifdef DEBUG
               printf("Time matches.\n");
               #endif

               /* time matches, so get command part of line */
               /* search for first non-numeric (and non-space) */
               result = 0;
               found = 0;
               while  ( ( (result < strlen(input_line) ) &&
                        (!found) ) )
               {
                  if ( ((input_line[result] < '0') ||
                       (input_line[result] > '9')) &&
                       ((input_line[result] != ' ') &&
                       (input_line[result] != '-'))
                     )
                     found = 1;
                  else
                    result++;
               }

               /* copy command line into a seperate varible */
               strncpy(command_line, &(input_line[result]), COMMAND_LENGTH);

               /* make sure the newline character is removed */
               if ( strstr(command_line, "\n") )
                 command_line[strlen(command_line) - 1] = '\0';

               /* if command is equal to "exit" then quit cron */
               if (! strcmp(command_line, "exit") )
               {
                  printf("Exiting cron.\n");
                  run_cron = 0;
                  if (cron_file) fclose(cron_file);
                  if (current_time) free(current_time);
                  return 0;
               }

               /* run the command */
               #ifdef DEBUG
                  printf("%s\n", command_line);
               #endif

               #ifndef DEBUG
                 system(command_line);
               #endif

          }   /* end of if time matches */
       }      /* end of if line is not commented */

       /* get next line */
       result = fgets(input_line, LINE_LENGTH, cron_file);
     }  /* end of while (result) */


     /* ensure clean exit if cron is killed */
     if (current_time)
     {
        free(current_time);
        current_time = NULL;
     }
     if (cron_file)
     {
        fclose(cron_file);
        cron_file = NULL;
     }

     /* wait for one minute */
     sleep(60);
   }      /* end of while (1) */

   return 0;
}


