//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2008 by PDSoft (Attila Padar)                *
//*                http://mpxplay.sourceforge.net                          *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: ftp drive handling (ftp client)

#include "in_file.h"
#include "newfunc\newfunc.h"
#include "display\display.h"
#include <malloc.h>

#if defined(MPXPLAY_LINK_WATTCP32) || defined(MPXPLAY_WIN32)
 #define MPXPLAY_DRVFTP_LINK_FTPCLIENT 1
#endif

#ifdef MPXPLAY_DRVFTP_LINK_FTPCLIENT

#define DRVFTP_MAX_LOCAL_DRIVES   ('Z'-'A'+1)
#define DRVFTP_MAX_VIRTUAL_DRIVES ('7'-'0'+1)
#define DRVFTP_MAX_DRIVES (DRVFTP_MAX_LOCAL_DRIVES+DRVFTP_MAX_VIRTUAL_DRIVES)
#define DRVFTP_FIRSTDRV_VIRTUAL  DRVFTP_MAX_LOCAL_DRIVES // 0:

// ftp://user:pasw@servername:port/dir
#define DRVFTP_PATHBEGINSTR           "ftp:"
#define DRVFTP_PATHSEPARATOR_PORTNUM  ':'
#define DRVFTP_PATHSEPARATOR_USER     '@'
#define DRVFTP_PATHSEPARATOR_PASSWORD ':'

#define DRVFTP_FTPDRIVEINFO_SYSTYPE_UNIX 1

#define DRVFTP_FILEOPENID_FREE 0
#define DRVFTP_FILEOPENID_BUSY -1000

#define DRVFTP_IP_LEN 4

#define DRVFTP_CACHED_DIRS_INITIAL 1000

#define DRVFTP_DEFAULT_FTPPORT 21

#define DRVFTP_DATATYPE_ASCII  1
#define DRVFTP_DATATYPE_BINARY 2

#define DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE   5000  // in msec
#define DRVFTP_DEFAULT_TIMEOUTMS_SENDBLOCK 30000 // in msec
#define DRVFTP_DEFAULT_TIMEOUTMS_READ       3000  // in msec
#define DRVFTP_DEFAULT_RCVBUFSIZE 65536

#ifndef TRUE
#define TRUE 1
#endif

#define DRVFTP_RESPCODE_OPENING_DATACONN   150
#define DRVFTP_RESPCODE_COMMAND_OK         200
#define DRVFTP_RESPCODE_CMD_SIZE_OK        213
#define DRVFTP_RESPCODE_TRANSFER_COMPLETE  226
#define DRVFTP_RESPCODE_DIRCOMMAND_OK      250
#define DRVFTP_RESPCODE_CMD_PWD_OK         257
#define DRVFTP_RESPCODE_CMD_REST_OK        350
#define DRVFTP_RESPCODE_TRANSFER_ABORTED   426
#define DRVFTP_RESPCODE_NO_SUCH_FILE       550

#define DRVFTP_FTPFILE_FLAG_SEEK     1
#define DRVFTP_FTPFILE_FLAG_WRITE    2
#define DRVFTP_FTPFILE_FLAG_READWAIT 4

typedef mpxp_uint32_t ftpdrive_socket_t;

typedef struct ftpdrive_direntry_info_s{
 char *filename;
 unsigned long attrib;
 mpxp_filesize_t filesize;
 pds_fdate_t fdate;
}ftpdrive_direntry_info_s;

typedef struct ftpdrive_directory_info_s{
 char *dirname;
 unsigned long cached_entries_num;
 unsigned long cached_entries_max;
 struct ftpdrive_direntry_info_s *entrydatas;
}ftpdrive_directory_info_s;

typedef struct ftpdrive_info_s{
 mpxp_uint32_t flags;
 unsigned int drivenum;
 struct ftpdrive_info_s **ftpdrive_info_ptr;
 long connect_id_num;
 unsigned int system_type;
 struct ftpdrive_lowlevel_func_s *lowfunc;
 void *lowlevelfunc_privatedata;
 ftpdrive_socket_t hinternet_connect;
 ftpdrive_socket_t hinternet_filehand;   // one filehand per session
 unsigned int portnum;
 long file_open_id_num;      // id of last opened file
 unsigned long file_bufsize;
 unsigned long socket_bufsize;

 unsigned long cached_dirs_num;
 unsigned long cached_dirs_max;
 struct ftpdrive_directory_info_s *cached_dir_datas;

 mpxp_uint32_t sockethand_connect;
 mpxp_uint32_t sockethand_data;
 unsigned int lastrespcode;
 unsigned int lastdatatype;
 mpxp_uint8_t ip_local[DRVFTP_IP_LEN];
 mpxp_uint8_t ip_remote[DRVFTP_IP_LEN];

 char currremotedir_selected[MAX_PATHNAMELEN];
 char currremotedir_real[MAX_PATHNAMELEN];

 char servername[256];
 char username[128];
 char password[128];
 char initial_path_or_filename[MAX_PATHNAMELEN];
}ftpdrive_info_s;

typedef struct ftpdrive_filefind_s{
 unsigned long entrynum_curr;
 unsigned long entrynum_end;
 struct ftpdrive_direntry_info_s *entry;
}ftpdrive_filefind_s;

typedef struct ftpfile_info_s{
 mpxp_uint32_t flags;
 unsigned int opentype;
 //ftpdrive_socket_t hinternet_connect;     // if no (virtual) drive connected
 struct ftpdrive_info_s **ftpi_ftpdrives_ptr;
 long connect_id_num;
 long open_id_num;
 unsigned long file_bufsize;
 mpxp_filesize_t filepos;
 mpxp_filesize_t filesize;
 mpxp_uint64_t timeout_at_read;
 char servername[MAX_PATHNAMELEN];
 char remotefilename[MAX_PATHNAMELEN];
}ftpfile_info_s;

typedef struct ftpdrive_lowlevel_func_s{
 char *name;
 int   (*global_init)(struct ftpdrive_info_s *ftpi);
 void  (*global_deinit)(struct ftpdrive_info_s *ftpi);
 mpxp_uint32_t (*session_open)(struct ftpdrive_info_s *ftpi);
 int   (*dataconnect_open)(struct ftpdrive_info_s *ftpi,mpxp_uint32_t *socketnum);
 int   (*dataconnect_accept)(struct ftpdrive_info_s *ftpi,mpxp_uint32_t *socketnum);
 void  (*connect_close)(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum);
 long  (*send)(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum,char *data,unsigned long bytes_to_send);
 long  (*bytes_buffered)(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum);
 long  (*receive)(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum,char *data,unsigned long buflen);//,unsigned long requested_bytes,unsigned int timeout_ms);
}ftpdrive_lowlevel_func_s;

static ftpdrive_lowlevel_func_s FTPDRV_lowlevel_funcs;

static ftpdrive_lowlevel_func_s *ALL_lowlevel_funcs[]={
 &FTPDRV_lowlevel_funcs,
 NULL
};

static long drvftp_connectid_num;
static struct ftpdrive_info_s *ftpdrives_info_ptrs[DRVFTP_MAX_VIRTUAL_DRIVES];

static void drvftp_dircache_dir_dealloc(struct ftpdrive_directory_info_s *diri);
static void ftpdrive_drive_unmount(void *drivehand_data);
static unsigned int ftpdrive_checkdir(void *drivehand_data,char *dirname);
static int ftpdrive_chdir(void *drivehand_data,char *path);
static void drvftp_str_localname_to_remote(char *remotename,char *pathname);
static int drvftp_cwd(struct ftpdrive_info_s *ftpi,char *remotename);

//-------------------------------------------------------------------------
//directory cache (filename,attribs,filesize,filedate)
static unsigned int drvftp_dircache_dirs_expand(struct ftpdrive_info_s *ftpi)
{
 struct ftpdrive_directory_info_s *dirdatas;

 dirdatas=(struct ftpdrive_directory_info_s *)malloc((ftpi->cached_dirs_max+DRVFTP_CACHED_DIRS_INITIAL)*sizeof(*dirdatas));
 if(!dirdatas)
  return 0;
 if(ftpi->cached_dir_datas){
  pds_memcpy(dirdatas,ftpi->cached_dir_datas,(ftpi->cached_dirs_max*sizeof(*dirdatas)));
  free(ftpi->cached_dir_datas);
 }
 pds_memset(dirdatas+ftpi->cached_dirs_max,0,(DRVFTP_CACHED_DIRS_INITIAL*sizeof(*dirdatas)));
 ftpi->cached_dir_datas=dirdatas;
 ftpi->cached_dirs_max+=DRVFTP_CACHED_DIRS_INITIAL;
 return 1;
}

static struct ftpdrive_directory_info_s *drvftp_dircache_dir_realloc(struct ftpdrive_info_s *ftpi,struct ftpdrive_directory_info_s *dirdatas,char *dirname)
{
 unsigned int step;
 if(dirdatas){
  drvftp_dircache_dir_dealloc(dirdatas);
  step=0;
 }else{
  if(ftpi->cached_dirs_num>=ftpi->cached_dirs_max)
   if(!drvftp_dircache_dirs_expand(ftpi))
    return NULL;
  dirdatas=ftpi->cached_dir_datas+ftpi->cached_dirs_num;
  step=1;
 }
 dirdatas->dirname=malloc(pds_strlen(dirname)+1);
 if(!dirdatas->dirname)
  return NULL;
 pds_strcpy(dirdatas->dirname,dirname);
 ftpi->cached_dirs_num+=step;
 return dirdatas;
}

static void drvftp_dircache_dir_dealloc(struct ftpdrive_directory_info_s *diri)
{
 struct ftpdrive_direntry_info_s *ed;
 unsigned int i;
 if(diri){
  if(diri->dirname)
   free(diri->dirname);
  ed=diri->entrydatas;
  if(ed){
   i=diri->cached_entries_num;
   if(i)
    do{
     if(ed->filename)
      free(ed->filename);
     ed++;
    }while(--i);
   free(diri->entrydatas);
  }
  pds_memset(diri,0,sizeof(*diri));
 }
}

static void drvftp_dircache_alldirs_dealloc(struct ftpdrive_info_s *ftpi)
{
 struct ftpdrive_directory_info_s *diri=ftpi->cached_dir_datas;
 unsigned int i;
 if(diri){
  i=ftpi->cached_dirs_num;
  if(i)
   do{
    drvftp_dircache_dir_dealloc(diri);
    diri++;
   }while(--i);
  free(ftpi->cached_dir_datas);
  ftpi->cached_dir_datas=NULL;
 }
 ftpi->cached_dirs_num=ftpi->cached_dirs_max=0;
}

static struct ftpdrive_directory_info_s *drvftp_dircache_dir_searchby_name(struct ftpdrive_info_s *ftpi,char *dirname)
{
 struct ftpdrive_directory_info_s *diri=ftpi->cached_dir_datas;
 unsigned int i;
 if(diri){
  i=ftpi->cached_dirs_num;
  if(i)
   do{
    if(pds_stricmp(diri->dirname,dirname)==0)
     return diri;
    diri++;
   }while(--i);
 }
 return NULL;
}

static unsigned int drvftp_dircache_entries_expand(struct ftpdrive_directory_info_s *diri)
{
 struct ftpdrive_direntry_info_s *ed;
 ed=(struct ftpdrive_direntry_info_s *)malloc((diri->cached_entries_max+DRVFTP_CACHED_DIRS_INITIAL)*sizeof(*ed));
 if(!ed)
  return 0;
 if(diri->entrydatas){
  pds_memcpy(ed,diri->entrydatas,(diri->cached_entries_max*sizeof(*ed)));
  free(diri->entrydatas);
 }
 pds_memset(ed+diri->cached_entries_num,0,(DRVFTP_CACHED_DIRS_INITIAL*sizeof(*ed)));
 diri->entrydatas=ed;
 diri->cached_entries_max+=DRVFTP_CACHED_DIRS_INITIAL;
 return 1;
}

static struct ftpdrive_direntry_info_s *drvftp_dircache_entry_alloc(struct ftpdrive_directory_info_s *diri,char *filename)
{
 struct ftpdrive_direntry_info_s *ed;
 if(diri->cached_entries_num>=diri->cached_entries_max)
  if(!drvftp_dircache_entries_expand(diri))
   return NULL;
 ed=diri->entrydatas+diri->cached_entries_num;
 ed->filename=malloc(pds_strlen(filename)+1);
 if(!ed->filename)
  return NULL;
 pds_strcpy(ed->filename,filename);
 diri->cached_entries_num++;
 return ed;
}

static char *drvftp_str_getpath_from_fullname(char *path,char *fullname)
{
 char *filenamep;

 if(!path)
  return NULL;
 if(!fullname){
  *path=0;
  return NULL;
 }

 if(path!=fullname)
  pds_strcpy(path,fullname);
 filenamep=pds_strrchr(path,PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP);
 if(filenamep){
  if((filenamep==path) || (path[1]==':' && filenamep==(path+2))) // "\\filename.xxx" or "c:\\filename.xxx"
   filenamep[1]=0;
  else
   filenamep[0]=0;                        // c:\\subdir\\filename.xxx
  filenamep++;
 }else{
  filenamep=pds_strchr(path,':');
  if(filenamep)
   *(++filenamep)=0;
  else{
   filenamep=path;
   *path=0;
  }
 }
 filenamep=fullname+(filenamep-path);
 return filenamep;
}

static struct ftpdrive_direntry_info_s *drvftp_dircache_entry_searchby_fullname(struct ftpdrive_info_s *ftpi,char *fullname)
{
 struct ftpdrive_directory_info_s *diri;
 struct ftpdrive_direntry_info_s *ed;
 unsigned int en;
 char *filename,dirname[MAX_PATHNAMELEN];

 filename=drvftp_str_getpath_from_fullname(dirname,fullname);

 diri=drvftp_dircache_dir_searchby_name(ftpi,dirname);
 if(!diri)
  goto err_out_remove;
 ed=diri->entrydatas;
 if(!ed)
  goto err_out_remove;
 en=diri->cached_entries_num;
 if(!en)
  goto err_out_remove;
 do{
  if(pds_stricmp(ed->filename,filename)==0)
   return ed;
  ed++;
 }while(--en);
err_out_remove:
 return NULL;
}

static struct ftpdrive_direntry_info_s *drvftp_dircache_entry_removeby_fullname(struct ftpdrive_info_s *ftpi,char *fullname)
{
 struct ftpdrive_direntry_info_s *ed=drvftp_dircache_entry_searchby_fullname(ftpi,fullname);
 if(ed)
  if(ed->filename){
   free(ed->filename);
   ed->filename=NULL;
  }
 return ed;
}

static struct ftpdrive_direntry_info_s *drvftp_dircache_entry_addby_fullname(struct ftpdrive_info_s *ftpi,char *fullname)
{
 struct ftpdrive_directory_info_s *diri;
 struct ftpdrive_direntry_info_s *ed;
 unsigned int en;
 char *filename,dirname[MAX_PATHNAMELEN];

 filename=drvftp_str_getpath_from_fullname(dirname,fullname);

 diri=drvftp_dircache_dir_searchby_name(ftpi,dirname);
 if(!diri)
  return NULL;
 ed=diri->entrydatas;
 if(!ed)
  goto err_out_add;
 en=diri->cached_entries_max;
 if(!en)
  goto err_out_add;
 do{
  if(!ed->filename){ // deleted by entry_removeby_fullname
   pds_memset(ed,0,sizeof(*ed));
   ed->filename=malloc(pds_strlen(filename)+1);
   if(!ed->filename)
    return NULL;
   pds_strcpy(ed->filename,filename);
   diri->cached_entries_num++;
   return ed;
  }
  ed++;
 }while(--en);
err_out_add:
 return drvftp_dircache_entry_alloc(diri,filename);
}

static void drvftp_dircache_entry_copyto_ffblk(struct pds_find_t *ffblk,struct ftpdrive_direntry_info_s *ed)
{
 ffblk->attrib=ed->attrib;
 ffblk->size=ed->filesize;
 pds_memcpy(&ffblk->fdate,&ed->fdate,sizeof(ffblk->fdate)); // ???
 pds_strcpy(ffblk->name,ed->filename);
}

//----------------------------------------------------------------------
static unsigned int drvftp_cmdctrl_sendcommand(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,char *command)
{
 char cmd[MAX_PATHNAMELEN+32];
 snprintf(cmd,sizeof(cmd),"%s\r\n",command);
 //display_message(0,0,command);
 return ftpi->lowfunc->send(ftpi,connect_handler,cmd,pds_strlen(cmd));
}

static unsigned int drvftp_cmdctrl_read_respline(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,char *response,unsigned int respbufsize,unsigned int instant)
{
 long bytes,retcode=0;
 mpxp_uint64_t endtime=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
 char *rp=response;
 unsigned int bufsize=respbufsize,allbytes=0;

 pds_memset(response,0,respbufsize);
 do{
  if(ftpi->lowfunc->bytes_buffered(ftpi,connect_handler)){
   bytes=ftpi->lowfunc->receive(ftpi,connect_handler,rp,1);
   allbytes+=bytes;
  }else
   bytes=0;
  if(instant){ // read only buffered/started responses (don't wait for new ones)
   if(bytes<1)
    break;
   instant=0;
  }
  rp+=bytes;
  bufsize-=bytes;
  if(pds_strchr(response,'\n')){
   retcode=1;
   break;
  }
#ifdef MPXPLAY_WIN32
  Sleep(0);
#endif
 }while(bufsize && (pds_gettimem()<=endtime));
 if(!retcode)
  allbytes=0;

 return allbytes;
}

static unsigned int drvftp_cmdctrl_read_response(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,char *response,unsigned int respbufsize,unsigned int instant)
{
 long retcode=0;
 do{
  if(!drvftp_cmdctrl_read_respline(ftpi,connect_handler,response,respbufsize,instant))
   break;
  retcode=pds_atol(response);
 }while((retcode<100) || (retcode>999));
 if(retcode<100)
  retcode=0;
 else
  ftpi->lastrespcode=retcode;
 return retcode;
}

static unsigned int drvftp_cmdctrl_read_respcode(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,unsigned int instant)
{
 char response[MAX_PATHNAMELEN+32];
 return drvftp_cmdctrl_read_response(ftpi,connect_handler,response,sizeof(response),instant);
}

static void drvftp_cmdctrl_flush_responses(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler)
{
 char tmp[MAX_PATHNAMELEN+32];
 while(drvftp_cmdctrl_read_respline(ftpi,connect_handler,tmp,sizeof(tmp),1)){}
}

static unsigned int drvftp_cmdctrl_send_command_get_respcode(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,char *command)
{
 drvftp_cmdctrl_flush_responses(ftpi,connect_handler);
 if(!drvftp_cmdctrl_sendcommand(ftpi,connect_handler,command))
  return 0;
 return drvftp_cmdctrl_read_respcode(ftpi,connect_handler,0);
}

static unsigned int drvftp_cmdctrl_send_command_passive(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,char *command)
{
 unsigned int respcode;
 drvftp_cmdctrl_flush_responses(ftpi,connect_handler);
 if(!drvftp_cmdctrl_sendcommand(ftpi,connect_handler,command))
  return 0;
 respcode=drvftp_cmdctrl_read_respcode(ftpi,connect_handler,0);
 if((respcode>=100) && (respcode<=399))
  return 1;
 return 0;
}

//splits response to a retcode-number and a string (after the retcode)
static int drvftp_cmdctrl_send_command_get_response(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,char *command,char *respbuf,unsigned int buflen)
{
 int retcode;
 drvftp_cmdctrl_flush_responses(ftpi,connect_handler);
 if(!drvftp_cmdctrl_sendcommand(ftpi,connect_handler,command))
  return 0;
 if(respbuf && buflen){
  char *s;
  *respbuf=0;
  retcode=drvftp_cmdctrl_read_response(ftpi,connect_handler,respbuf,buflen,0);
  s=pds_strchr(respbuf,'\r');
  if(s){
   *s=0;
   pds_strcpy(respbuf,&respbuf[4]);
  }else{
   pds_strncpy(respbuf,&respbuf[4],buflen-4);
   respbuf[buflen-4]=0;
  }
 }else
  retcode=drvftp_cmdctrl_read_respcode(ftpi,connect_handler,0);
 return retcode;
}

//---------------------------------------------------------------------------
static int drvftp_dataconn_send_type(struct ftpdrive_info_s *ftpi,mpxp_uint32_t connect_handler,unsigned int type)
{
 char *command;
 if(ftpi->lastdatatype==type)
  return 1;
 switch(type){
  case DRVFTP_DATATYPE_ASCII:command="TYPE A";break;
  case DRVFTP_DATATYPE_BINARY:command="TYPE I";break;
  default:return 0;
 }
 ftpi->lastdatatype=type;
 if(drvftp_cmdctrl_send_command_get_respcode(ftpi,connect_handler,command)!=DRVFTP_RESPCODE_COMMAND_OK)
  return 0;
 return 1;
}

static int drvftp_dataconn_open(struct ftpdrive_info_s *ftpi,struct ftpfile_info_s *ftfi,char *command,char *openpathfilename,mpxp_uint32_t *data_handler,unsigned int datatype)
{
 unsigned int portnum;
 char *filename,cmd[MAX_PATHNAMELEN+16];
 char newdir[MAX_PATHNAMELEN];

 if(ftfi){ // filename
  filename=drvftp_str_getpath_from_fullname(newdir,openpathfilename);
  if(!filename) // should not happen
   goto err_out_getdataa;
 }else{    // directory
  pds_strcpy(newdir,openpathfilename);
  filename=NULL;
 }
 if(newdir[0])
  if(drvftp_cwd(ftpi,newdir)!=0) // cwd to dir of file (some servers don't like full pathes)
   goto err_out_getdataa;

 if(ftfi && !ftfi->filesize){
  snprintf(cmd,sizeof(cmd),"SIZE %s",filename);
  if(drvftp_cmdctrl_send_command_get_response(ftpi,ftpi->hinternet_connect,cmd,cmd,sizeof(cmd))==DRVFTP_RESPCODE_CMD_SIZE_OK)
   ftfi->filesize=pds_atoi64(cmd);
 }

 *data_handler=0;
 portnum=ftpi->lowfunc->dataconnect_open(ftpi,data_handler);
 if(!portnum)
  return 0;

 drvftp_cmdctrl_flush_responses(ftpi,ftpi->hinternet_connect);

 if(!drvftp_dataconn_send_type(ftpi,ftpi->hinternet_connect,datatype))
  goto err_out_getdataa;

 sprintf(cmd,"PORT %d,%d,%d,%d,%d,%d",(unsigned int)ftpi->ip_local[0],(unsigned int)ftpi->ip_local[1],
  (unsigned int)ftpi->ip_local[2],(unsigned int)ftpi->ip_local[3],(portnum>>8),(portnum&0xff));
 if(drvftp_cmdctrl_send_command_get_respcode(ftpi,ftpi->hinternet_connect,cmd)!=DRVFTP_RESPCODE_COMMAND_OK)
  goto err_out_getdataa;

 if(ftfi && ftfi->filepos){
  sprintf(cmd,"REST %d",ftfi->filepos);
  if(drvftp_cmdctrl_send_command_get_respcode(ftpi,ftpi->hinternet_connect,cmd)!=DRVFTP_RESPCODE_CMD_REST_OK)
   goto err_out_getdataa;
 }

 snprintf(cmd,sizeof(cmd),"%s %s",command,((filename)? filename:""));

 if(drvftp_cmdctrl_send_command_get_respcode(ftpi,ftpi->hinternet_connect,cmd)!=DRVFTP_RESPCODE_OPENING_DATACONN)
  goto err_out_getdataa;

 if(!ftpi->lowfunc->dataconnect_accept(ftpi,data_handler))
  goto err_out_getdataa;

 return 1;

err_out_getdataa:
 if(*data_handler){
  ftpi->lowfunc->connect_close(ftpi,*data_handler);
  *data_handler=0;
 }
 return 0;
}

static long drvftp_dataconn_read(struct ftpdrive_info_s *ftpi,struct ftpfile_info_s *ftfi,mpxp_uint32_t data_handler,char *bufptr,unsigned int buflen,unsigned int requested_bytes,unsigned int wait_at_read)
{
 long total_bytes_read=0,readbytes;
 mpxp_uint64_t endtime_response=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
 if(!requested_bytes)
  requested_bytes=1;
 do{
  readbytes=0;
  drvftp_cmdctrl_read_respcode(ftpi,ftpi->hinternet_connect,1);
  if(ftpi->lastrespcode!=DRVFTP_RESPCODE_TRANSFER_COMPLETE){
   readbytes=ftpi->lowfunc->bytes_buffered(ftpi,data_handler);
   if(readbytes)
    endtime_response=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
  }
  if((ftpi->lastrespcode==DRVFTP_RESPCODE_TRANSFER_COMPLETE) || readbytes){
   readbytes=ftpi->lowfunc->receive(ftpi,data_handler,bufptr,buflen);
   total_bytes_read+=readbytes;
   if((ftpi->lastrespcode==DRVFTP_RESPCODE_TRANSFER_COMPLETE) || (total_bytes_read>=requested_bytes))
    break;
   if(buflen<=readbytes)
    break;
   buflen-=readbytes;
   bufptr+=readbytes;
  }else if(ftpi->lastrespcode==DRVFTP_RESPCODE_TRANSFER_ABORTED)
   break;
  if(!wait_at_read)
   break;
#ifdef MPXPLAY_WIN32
  Sleep(0);
#endif
 }while(pds_gettimem()<=endtime_response);

 //data-reconnect at read-timeout
 if(ftfi){
  if(!total_bytes_read && !wait_at_read){
   if(!ftfi->timeout_at_read)
    ftfi->timeout_at_read=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_READ;
   else if(pds_gettimem()>ftfi->timeout_at_read){
    funcbit_enable(ftfi->flags,DRVFTP_FTPFILE_FLAG_SEEK);
    ftfi->timeout_at_read=0;
    display_timed_message("FTP read timeout, data-reconnect ...");
   }
  }else
   ftfi->timeout_at_read=0;
 }

 return total_bytes_read;
}

static int drvftp_dataconn_write(struct ftpdrive_info_s *ftpi,mpxp_uint32_t data_handler,char *buf,unsigned int len)
{
 return ftpi->lowfunc->send(ftpi,data_handler,buf,len);
}

static void drvftp_dataconn_close(struct ftpdrive_info_s *ftpi,mpxp_uint32_t data_handler)
{
 if(data_handler){
  ftpi->lowfunc->connect_close(ftpi,data_handler);
  if(ftpi->lastrespcode!=DRVFTP_RESPCODE_TRANSFER_COMPLETE)
   drvftp_cmdctrl_read_respcode(ftpi,ftpi->hinternet_connect,0); // wait for close msg
 }
}

//-------------------------------------------------------------------------
static long ftpdrive_drive_config(void *drive_data,unsigned long funcnum,void *argp1,void *argp2)
{
 struct ftpdrive_info_s *ftpi=drive_data;
 struct ftpfile_info_s *ftfi=drive_data;
 unsigned long i;
 char strtmp[1024];

 if(!ftpi)
  return MPXPLAY_DISKDRIV_CFGERROR_INVALID_DRIVE;

 switch(funcnum){
  // ftpi
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_CMD_RESETDRIVE:drvftp_dircache_alldirs_dealloc(ftpi);return 1;
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_ISFILESYSUNX:if(ftpi->system_type==DRVFTP_FTPDRIVEINFO_SYSTYPE_UNIX) return 1; else return 0;
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_ISDIREXISTS:return ftpdrive_checkdir(ftpi,(char *)argp1);
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_ISDIRROOT:
   if(!argp1)
    return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
   drvftp_str_localname_to_remote(strtmp,(char *)argp1);
   i=pds_strlen(strtmp);
   if(!i)
    return 1;
   if((strtmp[0]==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP) && (i==1)) // '/'
    return 1;
   if((strtmp[0]==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP) && (strtmp[1]==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP) && (i==2)) // '//'
    return 1;
   if((strtmp[1]==':') && (i<sizeof(PDS_DIRECTORY_ROOTDIR_STR)))  // "d:/"
    return 1;
   if((strtmp[0]==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP) && (strtmp[2]==':') && (i<=sizeof(PDS_DIRECTORY_ROOTDIR_STR))) // "/d:/"
    return 1;
   return 0;
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_PREREADBUFBYTES:
   return 65536; // !!! ???
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_SET_DRIVEBLOCKSIZE:if(!argp1) return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;funcbit_smp_value_put(ftpi->file_bufsize,(*((unsigned long *)argp1))*2);return 1;
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_DRVLETTERSTR:
   if(!argp1 || !argp2)
    return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
   snprintf((char *)argp1,*((unsigned long *)argp2),"%d:",(ftpi->drivenum-DRVFTP_FIRSTDRV_VIRTUAL));
   return 1;
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_DRVTYPENAME:
   if(!argp1 || !argp2)
    return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
   snprintf((char *)argp1,*((unsigned long *)argp2),"%s%s",DRVFTP_PATHBEGINSTR,ftpi->servername);
   return 1;
  case MPXPLAY_DISKDRIV_CFGFUNCNUM_GET_REALLYFULLPATH:
   if(!argp1 || !argp2)
    return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
   drvftp_str_localname_to_remote(strtmp,(char *)argp2);
   sprintf((char *)argp1,"%s//%s%s%s%s%s:%d%s%s",
    DRVFTP_PATHBEGINSTR,ftpi->username,((ftpi->password[0])? ":":""),ftpi->password,
    ((ftpi->username[0])? "@":""),ftpi->servername,ftpi->portnum,((strtmp[0]!='/')? "/":""),strtmp);
   return 1;

  // ftfi
  case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_FILEBLOCKSIZE:if(!argp1) return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;funcbit_smp_value_put(ftfi->file_bufsize,(*((unsigned long *)argp1))*2);return 1;
  case MPXPLAY_DISKFILE_CFGFUNCNUM_SET_READWAIT:
   if(!argp1)
    return MPXPLAY_DISKDRIV_CFGERROR_ARGUMENTMISSING;
   if(*((unsigned long *)argp1))
    funcbit_enable(ftfi->flags,DRVFTP_FTPFILE_FLAG_READWAIT);
   else
    funcbit_disable(ftfi->flags,DRVFTP_FTPFILE_FLAG_READWAIT);
   return 1;
 }
 return MPXPLAY_DISKDRIV_CFGERROR_UNSUPPFUNC;
}

static unsigned int ftpdrive_drive_check(char *pathname)
{
 char *kukac;
 if(pds_strlicmp(pathname,DRVFTP_PATHBEGINSTR)==0)
  return 1;
 kukac=pds_strchr(pathname,DRVFTP_PATHSEPARATOR_USER); // user@hostname
 if(kukac){
  kukac++;
  if(pds_strlicmp(kukac,DRVFTP_PATHBEGINSTR)==0)
   return 1;
 }
 return 0;
}

static void *ftpdrive_drive_mount(char *pathname)
{
 struct ftpdrive_info_s *ftpi=NULL;
 unsigned int i;
 char *hostname,*portnum,*username,*password,*dirname,strtmp[MAX_PATHNAMELEN];

 for(i=0;i<DRVFTP_MAX_VIRTUAL_DRIVES;i++)
  if(!ftpdrives_info_ptrs[i])
   break;

 if(i>=DRVFTP_MAX_VIRTUAL_DRIVES)
  return ftpi;

 ftpi=(struct ftpdrive_info_s *)calloc(1,sizeof(*ftpi));
 if(!ftpi)
  return ftpi;

 if(drvftp_connectid_num>=0x7fffffff)
  drvftp_connectid_num=0;
 ftpi->connect_id_num=++drvftp_connectid_num;
 ftpi->drivenum=DRVFTP_FIRSTDRV_VIRTUAL+i;
 ftpi->ftpdrive_info_ptr=&ftpdrives_info_ptrs[i];
 ftpi->lowfunc=ALL_lowlevel_funcs[0];

 if(!ftpi->lowfunc->global_init(ftpi))
  goto err_out_mount;

 if(pds_strlicmp(pathname,DRVFTP_PATHBEGINSTR)==0){ // skip ftp://
  pathname+=sizeof(DRVFTP_PATHBEGINSTR)-1;
  while(*pathname=='/')
   pathname++;
 }

 pds_strcpy(strtmp,pathname);
 hostname=pds_strchr(strtmp,DRVFTP_PATHSEPARATOR_USER);
 if(hostname){
  hostname=pds_strchr(strtmp,DRVFTP_PATHSEPARATOR_USER);
  *hostname++=0;
  username=&strtmp[0];
  password=pds_strchr(strtmp,DRVFTP_PATHSEPARATOR_PASSWORD);
  if(password)
   *password++=0;
  pds_strcpy(ftpi->username,username);
  pds_strcpy(ftpi->password,password);
 }else{
  hostname=&strtmp[0];
  username=NULL;
  password=NULL;
 }
 dirname=pds_strchr(hostname,PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP);
 if(dirname){
  pds_strcpy(ftpi->initial_path_or_filename,dirname);
  *dirname=0;
 }
 portnum=pds_strchr(hostname,DRVFTP_PATHSEPARATOR_PORTNUM);
 if(portnum){
  *portnum++=0;
  ftpi->portnum=pds_atol(portnum);
 }
 if(!ftpi->portnum)
  ftpi->portnum=DRVFTP_DEFAULT_FTPPORT;
 pds_strcpy(ftpi->servername,hostname);

 ftpi->hinternet_connect=ftpi->lowfunc->session_open(ftpi);
 if(!ftpi->hinternet_connect){
  display_timed_message("FTP: no connection to the server!");
  goto err_out_mount;
 }
 if(drvftp_cmdctrl_read_respcode(ftpi,ftpi->hinternet_connect,0)!=220){
  display_timed_message("FTP: no server response (possibly down)!");
  goto err_out_mount;
 }
 if(ftpi->username[0]){
  sprintf(strtmp,"USER %s",ftpi->username);
  if(drvftp_cmdctrl_send_command_get_respcode(ftpi,ftpi->hinternet_connect,strtmp)!=331){
   display_timed_message("FTP: incorrect login name!");
   goto err_out_mount;
  }
  if(ftpi->password[0]){
   sprintf(strtmp,"PASS %s",ftpi->password);
   if(drvftp_cmdctrl_send_command_get_respcode(ftpi,ftpi->hinternet_connect,strtmp)!=230){
    display_timed_message("FTP: incorrect password!");
    goto err_out_mount;
   }
  }
 }
 strtmp[0]=0;
 drvftp_cmdctrl_send_command_get_response(ftpi,ftpi->hinternet_connect,"SYST",strtmp,sizeof(strtmp));
 if(pds_strnicmp(strtmp,"UNIX",4)==0)
  ftpi->system_type=DRVFTP_FTPDRIVEINFO_SYSTYPE_UNIX;

 ftpdrives_info_ptrs[i]=ftpi;

 if(ftpi->initial_path_or_filename[0])
  ftpdrive_chdir(ftpi,ftpi->initial_path_or_filename);

 return ftpi;

err_out_mount:
 ftpdrive_drive_unmount(ftpi);
 return NULL;
}

static void ftpdrive_drive_unmount(void *drivehand_data)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 //unsigned int i;
 if(ftpi){
  if(ftpi->hinternet_filehand)
   ftpi->lowfunc->connect_close(ftpi,ftpi->hinternet_filehand);
  if(ftpi->hinternet_connect)
   ftpi->lowfunc->connect_close(ftpi,ftpi->hinternet_connect);
  if(ftpi->ftpdrive_info_ptr)
   *(ftpi->ftpdrive_info_ptr)=NULL;
  drvftp_dircache_alldirs_dealloc(ftpi);
  free(ftpi);
 }
 /*for(i=0;i<DRVFTP_MAX_VIRTUAL_DRIVES;i++)
  if(!ftpdrives_info_ptrs[i])
   break;
 if(i>=DRVFTP_MAX_VIRTUAL_DRIVES){ // no active ftp drive
  // ??? close socket/tcpip handlers (lowfunc->global_deinit())
 }*/
}

static void drvftp_str_localname_to_remote(char *remotename,char *pathname)
{
 int drivenum=pds_getdrivenum_from_path(pathname);
 if(drivenum>=0)
  pathname+=sizeof(PDS_DIRECTORY_DRIVE_STR)-1;
 pds_strcpy(remotename,pathname);
#if (PDS_DIRECTORY_SEPARATOR_CHAR!=PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP)
 while(*remotename){ // convert '\' to '/'
  if(*remotename==PDS_DIRECTORY_SEPARATOR_CHAR)
   *remotename=PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP;
  remotename++;
 }
#endif
}

static void drvftp_str_remotename_to_local(struct ftpdrive_info_s *ftpi,char *localname,char *remotename,unsigned int buflen)
{
 if(*remotename==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP) // skip '/'
  remotename++;

 snprintf(localname,buflen,"%d:%c%s",(ftpi->drivenum-DRVFTP_FIRSTDRV_VIRTUAL),PDS_DIRECTORY_SEPARATOR_CHAR,remotename);
#if (PDS_DIRECTORY_SEPARATOR_CHAR!=PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP)
 while(*localname){ // convert '/' to '\'
  if(*localname==PDS_DIRECTORY_SEPARATOR_CHAR_UNXFTP)
   *localname=PDS_DIRECTORY_SEPARATOR_CHAR;
  localname++;
 }
#endif
}

static void ftpdrive_close_and_lock_filehand(struct ftpdrive_info_s *ftpi)
{
 if(ftpi->hinternet_filehand){
  drvftp_dataconn_close(ftpi,ftpi->hinternet_filehand);
  funcbit_smp_value_put(ftpi->hinternet_filehand,0);
 }
 funcbit_smp_value_put(ftpi->file_open_id_num,DRVFTP_FILEOPENID_BUSY);
}

static void ftpdrive_unlock_filehand(struct ftpdrive_info_s *ftpi)
{
 if(ftpi->file_open_id_num==DRVFTP_FILEOPENID_BUSY)
  funcbit_smp_value_put(ftpi->file_open_id_num,DRVFTP_FILEOPENID_FREE);
}

//-----------------------------------------------------------------------
static unsigned int ftpdrive_findnext(void *drivehand_data,struct pds_find_t *ffblk);

static char *strMon[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
		   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec",NULL };

static unsigned int drvftp_str_listline_to_direntry(char *listline,struct pds_find_t *ffblk)
{
 unsigned int i,linelen=0;
 char *end;

 pds_memset(ffblk,0,sizeof(*ffblk));
 end=pds_strchr(listline,'\n');
 if(end){
  *end=0;
  linelen++;
 }
 end=pds_strchr(listline,'\r');
 if(end){
  *end=0;
  linelen++;
 }
 linelen+=pds_strlen(listline);
 if(linelen<55)
  return linelen;

 if(listline[0]=='d')
  funcbit_enable(ffblk->attrib,_A_SUBDIR);
 else if(listline[0]!='-') // ???
  return linelen;          //
 if(listline[2]!='w')
  funcbit_enable(ffblk->attrib,_A_RDONLY);
 listline+=10; // skip rights
 while(*listline==' ') // skip spaces
  listline++;
 while(*listline!=' ') // skip idontknow (a number)
  listline++;
 while(*listline==' ') // skip spaces
  listline++;
 while(*listline!=' ') // skip user
  listline++;
 while(*listline==' ') // skip spaces
  listline++;
 while(*listline!=' ') // skip group
  listline++;
 while(*listline==' ') // skip spaces
  listline++;
 ffblk->size=pds_atoi64(&listline[0]);
 while(*listline!=' ') // skip size
  listline++;
 while(*listline==' ') // skip spaces
  listline++;

 i=0;
 do{
  if(pds_strnicmp(&listline[0],strMon[i],3)==0){
   ffblk->fdate.month=i+1;
   break;
  }
  i++;
 }while(strMon[i]);
 listline+=4; // skip month
 ffblk->fdate.day=pds_atol(&listline[0]);
 listline+=3; // skip day
 if(listline[2]==':'){ // current year
  ffblk->fdate.hours=pds_atol(&listline[0]);
  ffblk->fdate.minutes=pds_atol(&listline[3]);
  ffblk->fdate.year=(pds_getdate()>>16)-1980; // ???
 }else
  ffblk->fdate.year=pds_atol(&listline[0])-1980; // ???
 listline+=5; // skip time/year
 while(*listline==' ') // skip spaces
  listline++;
 pds_strncpy(ffblk->name,&listline[0],sizeof(ffblk->name)-1);
 ffblk->name[sizeof(ffblk->name)-1]=0;
 return linelen;
}

static unsigned int ftpdrive_findfirst(void *drivehand_data,char *pathname,unsigned int attrib,struct pds_find_t *ffblk)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 struct ftpdrive_directory_info_s *diri;
 struct ftpdrive_filefind_s *ff_data;
 ftpdrive_socket_t hinternet_ff=NULL;
 unsigned int retcode=1,datapos=0;
 char *fn,remotepath[MAX_PATHNAMELEN],searchfilemask[MAX_PATHNAMELEN];
 char remotename[MAX_PATHNAMELEN],command[MAX_PATHNAMELEN+64];

 if(!ftpi->hinternet_connect)
  return retcode;

 ff_data=(struct ftpdrive_filefind_s *)malloc(sizeof(*ff_data));
 if(!ff_data)
  return retcode;

 drvftp_str_localname_to_remote(remotename,pathname);
 fn=drvftp_str_getpath_from_fullname(remotepath,remotename);
 if(fn)
  pds_strcpy(searchfilemask,fn);
 else // should not happen
  pds_strcpy(searchfilemask,PDS_DIRECTORY_ALLFILE_STR);

 diri=drvftp_dircache_dir_searchby_name(ftpi,remotepath);
 if(!diri){
  ftpdrive_close_and_lock_filehand(ftpi);
  if(drvftp_cwd(ftpi,remotepath)!=0)
   goto err_out_ff;
  drvftp_dataconn_open(ftpi,NULL,"LIST",remotepath,&hinternet_ff,DRVFTP_DATATYPE_ASCII);
  if(!hinternet_ff)
   goto err_out_ff;
  diri=drvftp_dircache_dir_realloc(ftpi,diri,remotepath);
  if(!diri)
   goto err_out_ff;

  do{
   struct ftpdrive_direntry_info_s *ed;
   long bytes_read,linelen;
   bytes_read=drvftp_dataconn_read(ftpi,NULL,hinternet_ff,&command[datapos],sizeof(command)-datapos,0,1);
   datapos+=bytes_read;
   if(!datapos)
    break;
   command[datapos]=0;
   linelen=drvftp_str_listline_to_direntry(command,ffblk);
   if(!linelen)
    break;
   pds_memcpy(command,&command[linelen],sizeof(command)-linelen);
   datapos-=linelen;
   if(linelen<55)
    continue;
   if(!pds_filename_wildchar_cmp(ffblk->name,searchfilemask))
    continue;
   ed=drvftp_dircache_entry_alloc(diri,ffblk->name);
   if(!ed)
    break;
   ed->attrib=ffblk->attrib;
   ed->filesize=ffblk->size;
   pds_memcpy(&ed->fdate,&ffblk->fdate,sizeof(ed->fdate));
  }while(1);
  drvftp_dataconn_close(ftpi,hinternet_ff);
 }

 if(!diri->entrydatas || !diri->cached_entries_num)
  goto err_out_ff;
 ff_data->entrynum_curr=0;
 ff_data->entrynum_end=diri->cached_entries_num;
 ff_data->entry=diri->entrydatas;
 ffblk->ff_data=ff_data;

 if(ftpdrive_findnext(ftpi,ffblk))
  goto err_out_ff;

 retcode=0;
 goto end_ff;

err_out_ff:
 if(ff_data)
  free(ff_data);
 ffblk->ff_data=NULL;
end_ff:
 if(hinternet_ff)
  drvftp_dataconn_close(ftpi,hinternet_ff);
 ftpdrive_unlock_filehand(ftpi);
 return retcode;
}

static unsigned int ftpdrive_findnext(void *drivehand_data,struct pds_find_t *ffblk)
{
 struct ftpdrive_filefind_s *ff_data=ffblk->ff_data;
 unsigned int retcode=1;
 if(!ff_data)
  return retcode;
 do{
  if(ff_data->entrynum_curr>=ff_data->entrynum_end)
   break;
  if(ff_data->entry->filename){
   drvftp_dircache_entry_copyto_ffblk(ffblk,ff_data->entry);
   retcode=0;
  }
  ff_data->entrynum_curr++;
  ff_data->entry++;
 }while(retcode);

 return retcode;
}

static void ftpdrive_findclose(void *drivehand_data,struct pds_find_t *ffblk)
{
 if(ffblk->ff_data){
  free(ffblk->ff_data);
  ffblk->ff_data=NULL;
 }
}

static unsigned int ftpdrive_checkdir(void *drivehand_data,char *dirname)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 struct ftpdrive_directory_info_s *diri;
 struct ftpdrive_direntry_info_s *ed;
 char remotename[MAX_PATHNAMELEN];
 drvftp_str_localname_to_remote(remotename,dirname);
 if(pds_stricmp(remotename,ftpi->currremotedir_real)==0)
  return 1;
 diri=drvftp_dircache_dir_searchby_name(ftpi,remotename);
 if(diri)
  return 1;
 ed=drvftp_dircache_entry_searchby_fullname(ftpi,remotename);
 if(ed){
  if(funcbit_test(ed->attrib,_A_SUBDIR))
   return 1;
  return 0;
 }
 if(drvftp_cwd(ftpi,remotename)==0)
  return 1;
 return 0;
}

static char *ftpdrive_getcwd(void *drivehand_data,char *localpathbuf,unsigned int buflen)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 char *remotepath=alloca(buflen+1);
 *localpathbuf=0;
 if(ftpi->hinternet_connect && remotepath){
  if(ftpi->currremotedir_selected[0])
   drvftp_str_remotename_to_local(ftpi,localpathbuf,ftpi->currremotedir_selected,buflen);
  else{
   ftpdrive_close_and_lock_filehand(ftpi);
   *remotepath=0;
   if(drvftp_cmdctrl_send_command_get_response(ftpi,ftpi->hinternet_connect,"PWD",remotepath,buflen)==257){
     char *s=pds_strchr(remotepath,'\"'); // search for "dirname"
    if(s){
     pds_strcpy(remotepath,s+1);
     s=pds_strchr(remotepath,'\"');
     if(s)
      *s=0;
    }
    pds_strcpy(ftpi->currremotedir_real,remotepath);
    pds_strcpy(ftpi->currremotedir_selected,remotepath);
    drvftp_str_remotename_to_local(ftpi,localpathbuf,remotepath,buflen);
   }
   ftpdrive_unlock_filehand(ftpi);
  }
 }
 return localpathbuf;
}

static int drvftp_cwd(struct ftpdrive_info_s *ftpi,char *remotename)
{
 int retcode=-1;
 char command[MAX_PATHNAMELEN+8];
 if(ftpi->hinternet_connect){
  if(pds_stricmp(remotename,ftpi->currremotedir_real)==0)
   retcode=0;
  else{
   ftpdrive_close_and_lock_filehand(ftpi);
   snprintf(command,sizeof(command),"CWD %s",remotename);
   if(drvftp_cmdctrl_send_command_passive(ftpi,ftpi->hinternet_connect,command)==TRUE){
    pds_strcpy(ftpi->currremotedir_real,remotename);
    retcode=0;
   }
   ftpdrive_unlock_filehand(ftpi);
  }
 }
 return retcode;
}

static int ftpdrive_chdir(void *drivehand_data,char *path)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 struct ftpdrive_directory_info_s *diri;
 int retcode=-1;
 char remotename[MAX_PATHNAMELEN],strtmp[MAX_PATHNAMELEN];
 if(ftpi->hinternet_connect){
  pds_strcpy(strtmp,path);
  pds_filename_remove_relatives(strtmp);
  drvftp_str_localname_to_remote(remotename,strtmp);
  diri=drvftp_dircache_dir_searchby_name(ftpi,remotename);
  if(diri)
   retcode=0;
  else
   retcode=drvftp_cwd(ftpi,remotename);
  if(retcode==0)
   pds_strcpy(ftpi->currremotedir_selected,remotename);
 }
 return retcode;
}

static int ftpdrive_mkdir(void *drivehand_data,char *path)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 int retcode=-1;
 char remotename[MAX_PATHNAMELEN],command[MAX_PATHNAMELEN+8];
 if(ftpi->hinternet_connect){
  ftpdrive_close_and_lock_filehand(ftpi);
  drvftp_str_localname_to_remote(remotename,path);
  snprintf(command,sizeof(command),"MKD %s",remotename);
  if(drvftp_cmdctrl_send_command_passive(ftpi,ftpi->hinternet_connect,command)==TRUE){
   struct ftpdrive_direntry_info_s *edn;
   edn=drvftp_dircache_entry_addby_fullname(ftpi,remotename);
   if(edn)
    edn->attrib=_A_SUBDIR;
   retcode=0;
  }
  ftpdrive_unlock_filehand(ftpi);
 }
 return retcode;
}

static int ftpdrive_rmdir(void *drivehand_data,char *path)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 int retcode=-1;
 char remotename[MAX_PATHNAMELEN],command[MAX_PATHNAMELEN+8];
 if(ftpi->hinternet_connect){
  ftpdrive_close_and_lock_filehand(ftpi);
  drvftp_str_localname_to_remote(remotename,path);
  snprintf(command,sizeof(command),"RMD %s",remotename);
  if(drvftp_cmdctrl_send_command_passive(ftpi,ftpi->hinternet_connect,command)==TRUE){
   struct ftpdrive_directory_info_s *diri;
   drvftp_dircache_entry_removeby_fullname(ftpi,remotename);
   diri=drvftp_dircache_dir_searchby_name(ftpi,remotename);
   if(diri)
    drvftp_dircache_dir_dealloc(diri);
   retcode=0;
  }
  ftpdrive_unlock_filehand(ftpi);
 }
 return retcode;
}

static int ftpdrive_rename(void *drivehand_data,char *oldfilename,char *newfilename)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 int retcode=-1;
 char oldremotename[MAX_PATHNAMELEN],newremotename[MAX_PATHNAMELEN],command[MAX_PATHNAMELEN+8];
 if(ftpi->hinternet_connect){
  ftpdrive_close_and_lock_filehand(ftpi);
  drvftp_str_localname_to_remote(oldremotename,oldfilename);
  snprintf(command,sizeof(command),"RNFR %s",oldremotename);
  if(drvftp_cmdctrl_send_command_passive(ftpi,ftpi->hinternet_connect,command)==TRUE){
   drvftp_str_localname_to_remote(newremotename,newfilename);
   snprintf(command,sizeof(command),"RNTO %s",newremotename);
   if(drvftp_cmdctrl_send_command_passive(ftpi,ftpi->hinternet_connect,command)==TRUE){
    struct ftpdrive_direntry_info_s *edo,*edn;
    edo=drvftp_dircache_entry_removeby_fullname(ftpi,oldremotename);
    edn=drvftp_dircache_entry_addby_fullname(ftpi,newremotename);
    if(edo && edn){
     edn->attrib=edo->attrib;
     edn->filesize=edo->filesize;
     pds_memcpy(&edn->fdate,&edo->fdate,sizeof(edo->fdate));
    }
    retcode=0;
   }
  }
  ftpdrive_unlock_filehand(ftpi);
 }
 return retcode;
}

static int ftpdrive_unlink(void *drivehand_data,char *filename)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 int retcode=-1;
 char remotename[MAX_PATHNAMELEN],command[MAX_PATHNAMELEN+8];
 if(ftpi->hinternet_connect){
  ftpdrive_close_and_lock_filehand(ftpi);
  drvftp_str_localname_to_remote(remotename,filename);
  snprintf(command,sizeof(command),"DELE %s",remotename);
  if(drvftp_cmdctrl_send_command_passive(ftpi,ftpi->hinternet_connect,command)==TRUE){
   drvftp_dircache_entry_removeby_fullname(ftpi,remotename);
   retcode=0;
  }
  ftpdrive_unlock_filehand(ftpi);
 }
 return retcode;
}

//---------------------------------------------------------------------

#define DRVFTP_FTPFILE_OPENTYPE_READ   1
#define DRVFTP_FTPFILE_OPENTYPE_WRITE  2
#define DRVFTP_FTPFILE_OPENTYPE_CREATE 4
#define DRVFTP_FTPFILE_OPENTYPE_TEXT   8

static long drvftp_openid_num;

static unsigned int ftpdrive_file_check(void *drivehand_data,char *filename)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 int drivenum;
 if(!ftpi || !ftpi->hinternet_connect)
  return 0;
 drivenum=pds_getdrivenum_from_path(filename);
 if(drivenum==ftpi->drivenum)
  return 1;
 return 0;
}

static unsigned int ftpdrive_file_reconnect(struct ftpfile_info_s *ftfi)
{
 struct ftpdrive_info_s *ftpi=*(ftfi->ftpi_ftpdrives_ptr);
 unsigned int datatype;

 if(!ftpi)
  return 0;
 if(ftpi->file_open_id_num==DRVFTP_FILEOPENID_BUSY) // !!! smp (multithread) (-xr)
  return 0;
 if(!funcbit_test(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_WRITE))
  if(ftfi->filesize && (ftfi->filepos>=ftfi->filesize)) // eof
   return 0;

 if(ftpi->hinternet_filehand){
  drvftp_dataconn_close(ftpi,ftpi->hinternet_filehand);
  funcbit_smp_value_put(ftpi->hinternet_filehand,0);
 }
 funcbit_smp_disable(ftfi->flags,DRVFTP_FTPFILE_FLAG_SEEK);

 if(funcbit_test(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_TEXT))
  datatype=DRVFTP_DATATYPE_ASCII;
 else
  datatype=DRVFTP_DATATYPE_BINARY;

 if(funcbit_test(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_WRITE))
  drvftp_dataconn_open(ftpi,ftfi,"STOR",ftfi->remotefilename,&ftpi->hinternet_filehand,datatype);
 else{
  ftpi->socket_bufsize=ftfi->file_bufsize;
  drvftp_dataconn_open(ftpi,ftfi,"RETR",ftfi->remotefilename,&ftpi->hinternet_filehand,datatype);
 }
 funcbit_smp_value_put(ftpi->hinternet_filehand,ftpi->hinternet_filehand);
 funcbit_smp_value_put(ftpi->file_open_id_num,ftfi->open_id_num);
 if(!ftpi->hinternet_filehand)
  return 0;
 return 1;
}

static void *ftpdrive_file_open(void *drivehand_data,char *filename,unsigned long openmode)
{
 struct ftpdrive_info_s *ftpi=drivehand_data;
 struct ftpfile_info_s *ftfi;

 if(!ftpi || !ftpi->hinternet_connect) // !!! else open internet
  return NULL;

 ftfi=calloc(1,sizeof(*ftfi));
 if(!ftfi)
  return ftfi;

 funcbit_enable(ftfi->flags,DRVFTP_FTPFILE_FLAG_READWAIT);
 ftfi->ftpi_ftpdrives_ptr=ftpi->ftpdrive_info_ptr;

 switch(openmode&(O_RDONLY|O_RDWR|O_WRONLY)){
  case O_RDONLY:funcbit_enable(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_READ);break;
  case O_WRONLY:funcbit_enable(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_WRITE);break;
  default:goto err_out_open; // !!! O_RDWR not supported by FTP commands (?)
 }
 if(openmode&O_CREAT)
  funcbit_enable(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_CREATE);
 //if(openmode&O_TEXT)
 // funcbit_enable(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_TEXT); // doesn't work (ftp server doesn't convert lf to cr/lf?)

 ftfi->connect_id_num=ftpi->connect_id_num;
 pds_strcpy(ftfi->servername,ftpi->servername);
 if(drvftp_openid_num>=0x7fffffff)
  drvftp_openid_num=0;
 ftfi->open_id_num=++drvftp_openid_num; // !!! begins with 1

 drvftp_str_localname_to_remote(ftfi->remotefilename,filename);

 if(ftpi->file_bufsize){
  ftfi->file_bufsize=ftpi->file_bufsize;
  ftpi->file_bufsize=0;
 }

 if(!ftpdrive_file_reconnect(ftfi))
  goto err_out_open;

 return ftfi;

err_out_open:
 ftpdrive_file_close(ftfi);
 return NULL;
}

static void ftpdrive_file_close(void *filehand_data)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 if(ftfi){
  struct ftpdrive_info_s *ftpi=*(ftfi->ftpi_ftpdrives_ptr);
  if(ftpi){
   if(ftpi->file_open_id_num==ftfi->open_id_num){
    if(ftpi->hinternet_filehand){
     drvftp_dataconn_close(ftpi,ftpi->hinternet_filehand);
     funcbit_smp_value_put(ftpi->hinternet_filehand,0);
    }
   }
   if(funcbit_test(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_WRITE) && ftfi->filesize){
    struct ftpdrive_direntry_info_s *ed;
    ed=drvftp_dircache_entry_searchby_fullname(ftpi,ftfi->remotefilename);
    if(!ed)
     ed=drvftp_dircache_entry_addby_fullname(ftpi,ftfi->remotefilename);
    if(ed){
     if(ed->filesize<ftfi->filesize)
      ed->filesize=ftfi->filesize;
    }
   }
  }
  free(ftfi);
 }
}

static long ftpdrive_file_read(void *filehand_data,char *ptr,unsigned int num)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 struct ftpdrive_info_s *ftpi=*(ftfi->ftpi_ftpdrives_ptr);
 long bytes_read=0;
 if(!funcbit_test(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_READ))
  return bytes_read;
 if(!ftpi)
  return bytes_read;
 if(ftfi->connect_id_num!=ftpi->connect_id_num){
  if(pds_stricmp(ftfi->servername,ftpi->servername)!=0)
   return bytes_read;
  ftfi->connect_id_num=ftpi->connect_id_num;
 }
 if((ftfi->flags&DRVFTP_FTPFILE_FLAG_SEEK) || (ftpi->file_open_id_num!=ftfi->open_id_num))
  if(!ftpdrive_file_reconnect(ftfi))
   return bytes_read;
 if(ftfi->filesize && ((ftfi->filepos+num)>=ftfi->filesize))
  num=ftfi->filesize-ftfi->filepos;
 bytes_read=drvftp_dataconn_read(ftpi,ftfi,ftpi->hinternet_filehand,ptr,num,num,funcbit_test(ftfi->flags,DRVFTP_FTPFILE_FLAG_READWAIT));
 if(bytes_read>0)
  funcbit_smp_filesize_put(ftfi->filepos,ftfi->filepos+bytes_read);
 return bytes_read;
}

static long ftpdrive_file_write(void *filehand_data,char *ptr,unsigned int num)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 struct ftpdrive_info_s *ftpi=*(ftfi->ftpi_ftpdrives_ptr);
 long bytes_written=0;
 if(!funcbit_test(ftfi->opentype,DRVFTP_FTPFILE_OPENTYPE_WRITE))
  return bytes_written;
 if(!ftpi)
  return bytes_written;
 if(ftfi->connect_id_num!=ftpi->connect_id_num){
  if(pds_stricmp(ftfi->servername,ftpi->servername)!=0)
   return bytes_written;
  ftfi->connect_id_num=ftpi->connect_id_num;
 }
 if((ftfi->flags&DRVFTP_FTPFILE_FLAG_SEEK) || (ftpi->file_open_id_num!=ftfi->open_id_num))
  if(!ftpdrive_file_reconnect(ftfi))
   return bytes_written;
 bytes_written=drvftp_dataconn_write(ftpi,ftpi->hinternet_filehand,ptr,num);
 if(bytes_written>0){
  funcbit_smp_filesize_put(ftfi->filepos,ftfi->filepos+bytes_written);
  if(ftfi->filesize<ftfi->filepos)
   funcbit_smp_filesize_put(ftfi->filesize,ftfi->filepos);
  funcbit_smp_enable(ftfi->flags,DRVFTP_FTPFILE_FLAG_WRITE);
 }
 return bytes_written;
}

static mpxp_filesize_t ftpdrive_file_tell(void *filehand_data)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 return ftfi->filepos;
}

static mpxp_filesize_t ftpdrive_file_seek(void *filehand_data,mpxp_filesize_t pos,int fromwhere)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 mpxp_filesize_t newfilepos;
 switch(fromwhere){
  case SEEK_CUR:newfilepos=ftfi->filepos+pos;break;
  case SEEK_END:newfilepos=ftfi->filesize+pos;break;
  case SEEK_SET:
        default:newfilepos=pos;break;
 }
 if((newfilepos!=ftfi->filepos)){
  funcbit_smp_enable(ftfi->flags,DRVFTP_FTPFILE_FLAG_SEEK);
  funcbit_smp_filesize_put(ftfi->filepos,newfilepos);
 }
 return ftfi->filepos;
}

static mpxp_filesize_t ftpdrive_file_length(void *filehand_data)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 return ftfi->filesize;
}

static int ftpdrive_file_eof(void *filehand_data)
{
 struct ftpfile_info_s *ftfi=filehand_data;
 if(ftfi->filepos>=ftfi->filesize)
  return 1;
 return 0;
}

//------------------------------------------------------------------------

struct mpxplay_drivehand_func_s FTPDRIVE_drivehand_funcs={
 "FTPDRIVE",
 MPXPLAY_DRIVEHANDFUNC_INFOBIT_SLOWACCESS,    // infobits
 &ftpdrive_drive_config,
 &ftpdrive_drive_check,
 &ftpdrive_drive_mount,
 &ftpdrive_drive_unmount,
 &ftpdrive_findfirst,
 &ftpdrive_findnext,
 &ftpdrive_findclose,
 &ftpdrive_getcwd,
 &ftpdrive_chdir,
 &ftpdrive_mkdir,
 &ftpdrive_rmdir,
 &ftpdrive_rename,
 &ftpdrive_unlink,
 NULL,  // r15
 NULL,  // r16
 NULL,  // r17
 NULL,  // r18
 NULL,  // r19
 NULL,  // r20

 &ftpdrive_file_check,
 &ftpdrive_file_open,
 &ftpdrive_file_close,
 &ftpdrive_file_read,
 &ftpdrive_file_write,
 &ftpdrive_file_seek,
 &ftpdrive_file_tell,
 &ftpdrive_file_length,
 &ftpdrive_file_eof,
 NULL,  // file_chsize
 NULL   // r31
};

#else // MPXPLAY_DRVFTP_LINK_FTPCLIENT

struct mpxplay_drivehand_func_s FTPDRIVE_drivehand_funcs={
 "FTPDRIVE",0,
 NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
 NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
 NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL
};

#endif

//------------------------------------------------------------------------
// lowlevel sockets
#ifdef MPXPLAY_DRVFTP_LINK_FTPCLIENT

#ifdef MPXPLAY_WIN32
 #include <ws2tcpip.h>
 static WSADATA wsadata;
#else // __DOS__ WATTCP
 #include <tcp.h>
 #include "netinet/in.h"
 #include "netdb.h"
 #include "sys/socket.h"
 #include "sys/ioctl.h"
 #ifndef BOOL
 typedef int BOOL;
 #endif
 typedef int SOCKET;
 typedef struct sockaddr SOCKADDR;
 #define SD_RECEIVE      0x00
 #define SD_SEND         0x01
 #define SD_BOTH         0x02
 extern int _watt_do_exit;
 static unsigned int drvftp_wattcp_initialized;
#endif

static int ftpdrv_win32_global_init(struct ftpdrive_info_s *ftpi)
{
#ifdef MPXPLAY_WIN32
 if(!wsadata.wVersion && !wsadata.wHighVersion)
  if(WSAStartup(MAKEWORD(2,2),&wsadata)!=NO_ERROR)
   return 0;
#else
 _watt_do_exit = 0;   // don't exit from the program in sock_init()
 if(!drvftp_wattcp_initialized){
  if(sock_init()!=0)
   return 0;
  drvftp_wattcp_initialized=1;
 }
#endif
 return 1;
}

static void ftpdrv_win32_global_deinit(struct ftpdrive_info_s *ftpi)
{
#ifdef MPXPLAY_WIN32
 if(wsadata.wVersion || wsadata.wHighVersion)
  WSACleanup();
 pds_memset(&wsadata,0,sizeof(wsadata));
#else
 if(drvftp_wattcp_initialized){
  sock_exit(); // ???
  drvftp_wattcp_initialized=0;
 }
#endif
}

static mpxp_uint32_t ftpdrv_win32_session_open(struct ftpdrive_info_s *ftpi)
{
 struct hostent *ht;
 SOCKET portsock;
 //BOOL optval;
 struct sockaddr_in clientservice;
 char hostname_local[MAX_PATHNAMELEN];

 if(gethostname(hostname_local,sizeof(hostname_local))!=0)
  return 0;
 ht=gethostbyname(hostname_local);
 if(!ht || !ht->h_addr_list)
  return 0;
 pds_memcpy(ftpi->ip_local,ht->h_addr_list[0],DRVFTP_IP_LEN); // !!!

 ht=gethostbyname(ftpi->servername);
 if(!ht || !ht->h_addr_list)
  return 0;
 pds_memcpy(ftpi->ip_remote,ht->h_addr_list[0],DRVFTP_IP_LEN); // !!!

 portsock=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
 if(portsock==INVALID_SOCKET)
  return 0;

 pds_memset(&clientservice,0,sizeof(clientservice));
 clientservice.sin_family=AF_INET;
 pds_memcpy(&(clientservice.sin_addr.s_addr),ftpi->ip_remote,DRVFTP_IP_LEN);
 clientservice.sin_port=htons(ftpi->portnum);
 if(connect(portsock,(SOCKADDR *)&clientservice,sizeof(clientservice))==SOCKET_ERROR)
  goto err_out_ccopen;

 //optval=1;
 //setsockopt(portsock,SOL_SOCKET,SO_KEEPALIVE,(char *)&optval,sizeof(optval));

 return (mpxp_uint32_t)portsock;

err_out_ccopen:
 closesocket(portsock);
 return 0;
}

/*static HANDLE netevent = INVALID_HANDLE_VALUE;
static unsigned int ftpdrv_win32_do_select(SOCKET skt)
{
 int events;
 if(netevent==INVALID_HANDLE_VALUE){
  events = (FD_CONNECT | FD_READ | FD_WRITE | FD_OOB | FD_CLOSE | FD_ACCEPT);
  netevent=CreateEvent(NULL, FALSE, FALSE, NULL);
 }else
  events=0;

 if(WSAEventSelect(skt, netevent, events) == SOCKET_ERROR)
  return 0;
 return 1;
}*/

#ifndef SO_CONDITIONAL_ACCEPT
#define SO_CONDITIONAL_ACCEPT SO_ACCEPTCONN
#endif

static int ftpdrv_win32_dataconnect_open(struct ftpdrive_info_s *ftpi,mpxp_uint32_t *socketnum)
{
 SOCKET portsock=INVALID_SOCKET;
 struct sockaddr_in service,add;
 int socksize,portnum=0;
 int rcvbufsize=DRVFTP_DEFAULT_RCVBUFSIZE;
 mpxp_uint64_t endtime=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
 //BOOL optval;

 *socketnum=0;

 do{
  if(portsock==INVALID_SOCKET){
   portsock=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
   if(portsock==INVALID_SOCKET)
    continue;
  }

  if(!portnum){
   pds_memset(&service,0,sizeof(service));
   service.sin_family=AF_INET;
   pds_memcpy(&(service.sin_addr.s_addr),ftpi->ip_local,sizeof(ftpi->ip_local));
   service.sin_port=0;
   if(bind(portsock,(SOCKADDR *)&service,sizeof(service))==SOCKET_ERROR)
    continue;
   pds_memset(&add,0,sizeof(add));
   socksize=sizeof(add);
   if(getsockname(portsock, (SOCKADDR *) &add,&socksize)==SOCKET_ERROR)
    continue;
   portnum=ntohs(add.sin_port);
   if(!portnum)
    continue;
  }
  //optval=1;
  //if(setsockopt(portsock,SOL_SOCKET,SO_CONDITIONAL_ACCEPT,(char *)&optval,sizeof(optval))==SOCKET_ERROR)
  // portnum=0;
  rcvbufsize=(ftpi->socket_bufsize)? ftpi->socket_bufsize:DRVFTP_DEFAULT_RCVBUFSIZE;
  setsockopt(portsock,SOL_SOCKET,SO_RCVBUF,(char *)&rcvbufsize,sizeof(rcvbufsize));
  if(listen(portsock,1)!=SOCKET_ERROR){
   //ftpdrv_win32_do_select(portsock);
   break;
  }
#ifdef MPXPLAY_WIN32
   Sleep(0);
#endif
 }while((pds_gettimem()<=endtime));

 if(portnum)
  *socketnum=(mpxp_uint32_t)portsock;
 else
  if(portsock)
   closesocket(portsock);

 ftpi->socket_bufsize=0;

 return portnum;
}

static int ftpdrv_win32_dataconnect_accept(struct ftpdrive_info_s *ftpi,mpxp_uint32_t *socketnum)
{
 SOCKET portsock=(SOCKET)*socketnum,ps;
 int retcode=1;
 //BOOL optval;
 //unsigned long nonblock=1;
 mpxp_uint64_t endtime=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
 do{
  ps=accept(portsock,NULL,NULL);
  if(ps!=INVALID_SOCKET){
   retcode=1;
   closesocket(portsock);
   portsock=ps;
   //ioctlsocket(portsock,FIONBIO,&nonblock);
   break;
  }
#ifdef MPXPLAY_WIN32
  Sleep(0);
#endif
 }while(pds_gettimem()<=endtime);

 /*if(retcode){
  optval=1;
  setsockopt(portsock,SOL_SOCKET,SO_KEEPALIVE,(char *)&optval,sizeof(optval));
 }*/

 *socketnum=(mpxp_uint32_t)portsock;

 return retcode;
}

static long ftpdrv_win32_bytes_buffered(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum);

static void ftpdrv_win32_connect_close(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum)
{
 //char recvbuf[512];
 if(sockethandnum){
  SOCKET sock=(SOCKET)sockethandnum;
  //sprintf(recvbuf,"wait %d %d",ftpi->file_bufsize,ftpi->socket_bufsize);
  //display_message(1,0,recvbuf);
  shutdown(sock,SD_SEND); // !!! why does this take time? (more without it)
  //if(shutdown(sock,SD_BOTH)!=SOCKET_ERROR)
  // while(recv(sock,recvbuf,sizeof(recvbuf),0)>0){}
  closesocket(sock);
  //clear_message();
 }
}

static long ftpdrv_win32_send(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum,char *data,unsigned long bytes_to_send)
{
 long total_bytes_sent=0,leftbytes=bytes_to_send;
 //mpxp_uint64_t endtime_send=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_SENDBLOCK;
 mpxp_uint64_t endtime_response=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
 if(sockethandnum){
  SOCKET sock=(SOCKET)sockethandnum;
  do{
   long sentbytes=send(sock,data,leftbytes,0);
   if(sentbytes<0)
    break;
   if(sentbytes>0){
    total_bytes_sent+=sentbytes;
    if(total_bytes_sent>=bytes_to_send)
     break;
    leftbytes-=sentbytes;
    data+=sentbytes;
    endtime_response=pds_gettimem()+DRVFTP_DEFAULT_TIMEOUTMS_RESPONSE;
   }
#ifdef MPXPLAY_WIN32
   Sleep(0);
#endif
  }while((pds_gettimem()<=endtime_response));// && (pds_gettimem()<=endtime_send));
 }
 return total_bytes_sent;
}

static long ftpdrv_win32_bytes_buffered(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum)
{
 unsigned long bytes_stored=0;
 if(sockethandnum){
  SOCKET sock=(SOCKET)sockethandnum;
#ifdef MPXPLAY_WIN32
  ioctlsocket(sock,FIONREAD,&bytes_stored);
#else
  ioctlsocket(sock,FIONREAD,(char *)(&bytes_stored));
#endif
 }
 return bytes_stored;
}

static long ftpdrv_win32_receive(struct ftpdrive_info_s *ftpi,mpxp_uint32_t sockethandnum,char *data,unsigned long buflen)
{
 long bytes_received=0;

 if(sockethandnum){
  SOCKET sock=(SOCKET)sockethandnum;
  bytes_received=recv(sock,data,buflen,0);
  if(bytes_received<0)
   bytes_received=0;
 }
 return bytes_received;
}

static ftpdrive_lowlevel_func_s FTPDRV_lowlevel_funcs={
 "FTP",
 &ftpdrv_win32_global_init,
 &ftpdrv_win32_global_deinit,
 &ftpdrv_win32_session_open,
 &ftpdrv_win32_dataconnect_open,
 &ftpdrv_win32_dataconnect_accept,
 &ftpdrv_win32_connect_close,
 &ftpdrv_win32_send,
 &ftpdrv_win32_bytes_buffered,
 &ftpdrv_win32_receive
};

#endif // MPXPLAY_DRVFTP_LINK_FTPCLIENT
