//**************************************************************************
//*                     This file is part of the                           *
//*             AudioCV - a general audio converter program                *
//*                  The source code of AudioCV is                         *
//*          (c) copyright 2001-2004 by PDSoft (Attila Padar)              *
//*                    http://mpxplay.cjb.net                              *
//* email: mpxplay@freemail.hu (please write AudioCV in the subject field) *
//**************************************************************************
//conversion routines

#include "audiocv.h"

       int  conv_main(acv_fileinfo *,acv_fileinfo *,int);
       int  conv_init(acv_fileinfo *,acv_fileinfo *);
static void cv_input_to_standard(acv_fileinfo *,acv_fileinfo *);
//static void cv_scale_PCMF_input(acv_fileinfo *,acv_fileinfo *);
static void cv_unsigned_to_signed_char(char *,unsigned int);
static void cv_wavinput_to_PCMI(acv_fileinfo *);
static void cv_PCMI_to_PCMF(acv_fileinfo *,acv_fileinfo *);

static void cv_standard_to_output(acv_fileinfo *,acv_fileinfo *);
static void cv_limit_PCMF(acv_fileinfo *);
//static void cv_scale_PCMF_output(acv_fileinfo *);
static void cv_PCMF_to_PCMI(char *,char *,unsigned long,unsigned int);
static void cv_scale_PCMI(acv_fileinfo *,unsigned int);
static void cv_PCMI_to_wavout(char *,acv_fileinfo *);
static void cv_signed_to_unsigned_char(char *,unsigned int);
//--------------------------------------------------------------------------
static void analyze_cut(acv_fileinfo *);                   //PCMI only
static void analyze_mute(acv_fileinfo *);                  //PCMI only
static int  analyze_amplitude(acv_fileinfo *,acv_fileinfo *);//PCMI only

static void cv_channels_PCMF(acv_fileinfo*,acv_fileinfo *,unsigned long *);//PCMF only
static void cv_mute_zero(acv_fileinfo *,char *);           //PCMI or PCMF
static int  cv_cut_zero(acv_fileinfo *,acv_fileinfo *,char *);//PCMI or PCMF
static void cv_swapchan(acv_fileinfo *,char *);            //PCMI or PCMF
static void cv_freq_PCMF(acv_fileinfo *,acv_fileinfo *);   //PCMF only
static void cv_amplitude_PCMF(acv_fileinfo *);             //PCMF only

extern int cv_clipslimit;
extern int cv_cutlimit,cv_mutelimit;
extern float cv_normdb,cv_normlimit;

static unsigned long ana_cut_beginend,beginsamplenum_cut,endsamplenum_cut;
static unsigned long ana_mute_beginend,beginsamplenum_mute,endsamplenum_mute;
static PCM_CV_TYPE_F *cv_freq_buffer;

int conv_main(acv_fileinfo *af_in,acv_fileinfo *af_out,int pass)
{
 char *af_destbuff;
 int eoc=0;

 cv_input_to_standard(af_in,af_out);
 switch(pass){
   case 0://analyze input
	  if(af_in->filemode&CF_CHANNELS) // '-oc 1' + '-nr'
	   cv_channels_PCMF(af_in,af_out,&af_in->blocksamplenum);
	  if(af_in->filemode&CF_ANALYZE)
	   eoc=analyze_amplitude(af_in,af_out);
	  if(af_in->filemode&CF_CUTZERO){
	   analyze_cut(af_in);
	   eoc=0;
	  }
	  if(af_in->filemode&CF_MUTEZERO){
	   analyze_mute(af_in);
	   eoc=0;
	  }

	  break;
   case 1://create output
	  af_destbuff=(af_out->filemode&CF_FCONV)? af_out->buffer:af_in->buffer;
	  if(af_out->filemode&CF_MUTEZERO)
	   cv_mute_zero(af_in,af_destbuff);
	  if(af_out->filemode&CF_CUTZERO)
	   eoc=cv_cut_zero(af_in,af_out,af_destbuff);
	  if(af_out->filemode&CF_CHANNELS)
	   cv_channels_PCMF(af_in,af_out,&af_out->blocksamplenum);
	  if(af_out->filemode&CF_SWAPCHAN)
	   cv_swapchan(af_out,af_destbuff);
	  if(af_out->filemode&CF_FREQ)
	   cv_freq_PCMF(af_in,af_out);
	  if(af_out->filemode&CF_NORMALIZE)
	   cv_amplitude_PCMF(af_out);

	  cv_standard_to_output(af_in,af_out);
	  break;
 }
 return eoc;
}

int conv_init(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 long mbuffsize;
 ana_cut_beginend=beginsamplenum_cut=endsamplenum_cut=0;
 ana_mute_beginend=beginsamplenum_mute=endsamplenum_mute=0;

 if(af_out->filemode&CF_FREQ){
  mbuffsize=(PCM_BUFFSIZE+1)*max(af_in->channels,af_out->channels);
  mbuffsize=(long)((float)mbuffsize*((float)max(af_in->freq,af_out->freq))/(float)af_in->freq);
  mbuffsize+=8192;
  mbuffsize*=sizeof(PCM_CV_TYPE_F);
  cv_freq_buffer=(PCM_CV_TYPE_F *)malloc(mbuffsize);
  if(!cv_freq_buffer)
   return ACV_ERROR_MEMORY;
  memset((void *)cv_freq_buffer,0,mbuffsize);
 }

 if(af_in->filemode&CF_ICONV){
  if((af_in->filemode&CF_FLOAT) && (af_in->scalebits<=BITTYPE_UNSCALED))
   af_in->scalebits=16;
 }

 return 0;
}

void conv_close(void)
{
 if(cv_freq_buffer){
  free(cv_freq_buffer);
  cv_freq_buffer=NULL;
 }
}

//***************************************************************************
// input => af_in->buffer
//1. standardization:
//   af_in->buffer : PCM_I (int32) for integer based conversions (analize,cut,mute)
//   af_out->buffer: PCM_F (float) for floating point conversions (normalizer,freq-conv,cv_channels)
//2. wave editor functions (analizer,normalizer,freq,etc.)
//3. scale to output bits (at integer output)
//4. convert to output format (intxx (wav) or float (ogg encoder))
// af_out->buffer => output

static void cv_input_to_standard(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 if(af_in->filemode&CF_FLOAT){
  memcpy(af_out->buffer,af_in->buffer,af_in->blocksamplenum*sizeof(PCM_CV_TYPE_F));
  //cv_scale_PCMF_input(af_in,af_out);
  if(af_in->filemode&CF_ICONV)
   cv_PCMF_to_PCMI(af_out->buffer,af_in->buffer,af_in->blocksamplenum,af_in->scalebits);
 }else{
  if(af_in->scalebits==8)
   cv_unsigned_to_signed_char(af_in->buffer,af_in->blocksamplenum);
  cv_wavinput_to_PCMI(af_in);
  if(af_out->filemode&CF_FCONV)
   cv_PCMI_to_PCMF(af_in,af_out);
 }
}

//scale (div) float input to the range -1...+1 (currently nothing uses this)
//conflict with PCMF_to_PCMI scaling (af_in->scalebits,af_out->scalebits are not good there)
/*static void cv_scale_PCMF_input(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 if(af_in->scalebits>BITTYPE_UNSCALED){
  PCM_CV_TYPE_F *inptr=(PCM_CV_TYPE_F *)af_in->buffer;
  PCM_CV_TYPE_F *outptr=(PCM_CV_TYPE_F *)af_out->buffer;
  const float scaleval=1.0f / (float)((PCM_CV_TYPE_UI)(1 << ((PCM_CV_TYPE_UI)af_in->scalebits-1)));
  unsigned int i;

  for(i=af_in->blocksamplenum;i;i--)
   *outptr++ =*inptr++ * scaleval;
 }else{ // no scale : copy float data into the right (af_out) buffer
  memcpy(af_out->buffer,af_in->buffer,af_in->blocksamplenum*sizeof(PCM_CV_TYPE_F));
 }
}*/

static void cv_unsigned_to_signed_char(char *inptr,unsigned int numsamples)
{
 while(numsamples--)
  *inptr++ -=(char)128;
}

// af_in->buffer (1,2,3,4 bytes) => af_in->buffer (int32)
static void cv_wavinput_to_PCMI(acv_fileinfo *af_in)
{
 char *inptr=(af_in->buffer)+(af_in->blocksamplenum*af_in->bytespersample);
 PCM_CV_TYPE_I *outptr=(PCM_CV_TYPE_I *)(af_in->buffer)+af_in->blocksamplenum;
 unsigned int i,instep=af_in->bytespersample,shift=(PCM_MAX_BITS-af_in->scalebits);
 PCM_CV_TYPE_I insamp;
 inptr-=(shift/8);
 for(i=af_in->blocksamplenum;i;i--){
  inptr-=instep;
  outptr--;
  insamp=*((PCM_CV_TYPE_I *)inptr);
  insamp>>=shift;
  *outptr=insamp;
 }
}

// af_in->buffer (int32) => af_out->buffer (float)
static void cv_PCMI_to_PCMF(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 PCM_CV_TYPE_I *inptr=(PCM_CV_TYPE_I *)af_in->buffer;
 PCM_CV_TYPE_F *outptr=(PCM_CV_TYPE_F *)af_out->buffer;
 const float scaleval=1.0f / (float)((PCM_CV_TYPE_UI)(1 << ((PCM_CV_TYPE_UI)af_in->scalebits-1)));
 unsigned int i;

 for(i=af_in->blocksamplenum;i;i--){
  PCM_CV_TYPE_I insamp=*inptr++;
  *outptr++=(PCM_CV_TYPE_F)insamp*scaleval;
 }
}

//**************************************************************************
//output routines
//**************************************************************************

static void cv_standard_to_output(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 if(af_out->filemode&CF_FLOAT){
  // float output (OGG,WAV)
  if(af_in->filemode&CF_FCONV)
   cv_limit_PCMF(af_out); // clip prevention (OGG decoder and float transformations (norm/freq))
  //cv_scale_PCMF_output(af_out);
 }else{
  // integer output (WAV)
  if(af_in->filemode&CF_FCONV){
   cv_PCMF_to_PCMI(af_out->buffer,af_out->buffer,af_out->blocksamplenum,af_out->scalebits);
   cv_PCMI_to_wavout(af_out->buffer,af_out);
  }else{
   if(af_in->scalebits!=af_out->scalebits)
    cv_scale_PCMI(af_in,af_out->scalebits);
   cv_PCMI_to_wavout(af_in->buffer,af_out);
  }
  if(af_out->scalebits==8)
   cv_signed_to_unsigned_char(af_out->buffer,af_out->blocksamplenum);
 }
}

//limit floating point values to range -1.0...+1.0
static void cv_limit_PCMF(acv_fileinfo *af)
{
 PCM_CV_TYPE_F *ptr=(PCM_CV_TYPE_F *)af->buffer;
 const PCM_CV_TYPE_F posmax=1.0,negmax=-1.0;
 unsigned int i;
 for(i=af->blocksamplenum;i;i--){
  const PCM_CV_TYPE_F insamp=*ptr;
  if(insamp>posmax)
   *ptr=posmax;
  else
   if(insamp<negmax)
    *ptr=negmax;
  ptr++;
 }
}

//scale (mul) to float output (currently nothing uses this)
//conflit with PCMF_to_PCMI scaling
/*static void cv_scale_PCMF_output(acv_fileinfo *af_out)
{
 if(af_out->scalebits>BITTYPE_UNSCALED){
  PCM_CV_TYPE_F *outptr=(PCM_CV_TYPE_F *)af_out->buffer;
  const float scaleval=(float)((PCM_CV_TYPE_UI)(1 << ((PCM_CV_TYPE_UI)af_out->scalebits-1)));
  unsigned int i;

  for(i=af_out->blocksamplenum;i;i--)
   *outptr++ *=scaleval;
 }
}*/

// af_out->buffer (float) => af_in->buffer (int32) & scale to wav-output bits
static void cv_PCMF_to_PCMI(char *srcbuff,char *destbuff,unsigned long blocksamplenum,unsigned int scalebits)
{
 PCM_CV_TYPE_F *inptr=(PCM_CV_TYPE_F *)srcbuff;
 PCM_CV_TYPE_I *outptr=(PCM_CV_TYPE_I *)destbuff;
 const PCM_CV_TYPE_UI scale=((PCM_CV_TYPE_UI)1) << (scalebits-1);
 const PCM_CV_TYPE_UI posmaxi=scale-1;
 const PCM_CV_TYPE_I  negmaxi=-((PCM_CV_TYPE_I)posmaxi)-1;
 const float scaleval=(PCM_CV_TYPE_F)scale;
 unsigned int i;

 for(i=blocksamplenum;i;i--){
  PCM_CV_TYPE_F insamp=*inptr * scaleval;
  inptr++;
  if(insamp>((PCM_CV_TYPE_F)posmaxi))
   *outptr=posmaxi;
  else
   if(insamp<((PCM_CV_TYPE_F)negmaxi))
    *outptr=negmaxi;
   else
#if defined(__WATCOMC__)
    pds_ftoi(&insamp,outptr);
#else
 #if defined(_MSC_VER)
    *outptr=pds_ftoi(insamp);
 #else
    *outptr=(PCM_CV_TYPE_I)insamp;
 #endif
#endif
  outptr++;
 }
}

//scale PCMI n to m bits resolution (scale to wav-output)
static void cv_scale_PCMI(acv_fileinfo *af_in,unsigned int outbits)
{
 PCM_CV_TYPE_I *inptr=(PCM_CV_TYPE_I *)af_in->buffer;
 int bitdiff=(int)outbits-(int)af_in->scalebits;
 unsigned int i;

 if(bitdiff<0){
  bitdiff=-bitdiff;
  for(i=af_in->blocksamplenum;i;i--)
   *inptr++ >>=bitdiff;
 }else{
  for(i=af_in->blocksamplenum;i;i--)
   *inptr++ <<=bitdiff;
 }
}

// af_in->buffer (int32) => af_out->buffer (1,2,3,4 bytes (8,16,24,32 bits))
static void cv_PCMI_to_wavout(char *srcbuff,acv_fileinfo *af_out)
{
 PCM_CV_TYPE_I *inptr=(PCM_CV_TYPE_I *)srcbuff;
	  char *outptr=af_out->buffer;
 unsigned int i,outstep=af_out->bytespersample;
 for(i=af_out->blocksamplenum;i;i--){
  PCM_CV_TYPE_I insamp=*inptr++;
  *((PCM_CV_TYPE_I*)outptr)=insamp;
  outptr+=outstep;
 }
}

// 8-bit wav files are always unsigned ???
static void cv_signed_to_unsigned_char(char *inptr,unsigned int numsamples)
{
 while(numsamples--)
  *inptr++ +=(char)128;
}

//***************************************************************************
// analizer functions
static void analyze_cut(acv_fileinfo *af_in)
{
 unsigned int i,modch=(af_in->channels-1);
 PCM_CV_TYPE_I *ptr=(PCM_CV_TYPE_I *)af_in->buffer;
 if(!ana_cut_beginend){
  for(i=0;i<af_in->blocksamplenum;i++)
   if(abs(*ptr++)>cv_cutlimit){
    beginsamplenum_cut=af_in->currsamplenum-af_in->blocksamplenum
			      +(i&(~modch)); // rounding down to samplenum/channels
    ana_cut_beginend=1;
    break;
   }
 }else{
  int end=-1;
  for(i=0;i<af_in->blocksamplenum;i++)
   if(abs(*ptr++)>cv_cutlimit)
    end=i;
  if(end>=0)
   endsamplenum_cut=af_in->currsamplenum-af_in->blocksamplenum
			   +((end+modch)&(~modch)); // rounding up
 }
}

static void analyze_mute(acv_fileinfo *af_in)
{
 unsigned int i,modch=(af_in->channels-1);
 PCM_CV_TYPE_I *ptr=(PCM_CV_TYPE_I *)af_in->buffer;
 if(!ana_mute_beginend){
  for(i=0;i<af_in->blocksamplenum;i++)
   if(abs(*ptr++)>cv_mutelimit){
    beginsamplenum_mute=af_in->currsamplenum-af_in->blocksamplenum+(i&(~modch));
    ana_mute_beginend=1;
    break;
   }
 }else{
  int end=-1;
  for(i=0;i<af_in->blocksamplenum;i++)
   if(abs(*ptr++)>cv_mutelimit)
    end=i;
  if(end>=0){
   endsamplenum_mute=af_in->currsamplenum-af_in->blocksamplenum+((end+modch)&(~modch));
  }
 }
}

#define ANA_NORM_RES_BITS   11 // norm resolution -> max. +60dB amplitude changing
			       // 1/2048 precision (instead of 1/32768, but it's enough and faster)
#define ANA_NORM_CLIPSTORE  (1<<ANA_NORM_RES_BITS)

static int analyze_amplitude(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 static unsigned long clipcounts[ANA_NORM_CLIPSTORE];
 PCM_CV_TYPE_I signmax,signpeek,clips;
 int i,shift;
 PCM_CV_TYPE_I *ptr=(PCM_CV_TYPE_I *)af_in->buffer;

 if(cv_clipslimit){ //overheat normalization
  memset((void *)&clipcounts[0],0,ANA_NORM_CLIPSTORE*sizeof(unsigned long));
  if(af_in->scalebits<=ANA_NORM_RES_BITS){ // for 8-bit wav files (less than 11 bit)
   shift=0;
   signmax=1<<(af_in->scalebits-1);
  }else{
   shift=af_in->scalebits-1-ANA_NORM_RES_BITS;
   signmax=1<<ANA_NORM_RES_BITS;
  }
  for(i=af_in->blocksamplenum;i;i--){
   PCM_CV_TYPE_I insamp=*ptr++;
   if(insamp<0)
    insamp=-insamp;
   clipcounts[insamp>>shift]++;
  }
  clips=0;
  signpeek=signmax;
  for(i=signmax-1;i>=0;i--){
   clips+=clipcounts[i];
   if(clips>cv_clipslimit)
    break;
   signpeek=i;
  }
 }else{ // single normalization, no limits and faster
  signmax=1<<(af_in->scalebits-1);
  signpeek=0;
  for(i=af_in->blocksamplenum;i;i--){
   PCM_CV_TYPE_I insamp=*ptr++;
   if(insamp<0)
    insamp=-insamp;
   if(insamp>signpeek)
    signpeek=insamp;
  }
 }
 cv_normdb=min(cv_normdb,((float)signmax/(float)signpeek));
 // if we cannot expand the audio, exit from analizing
 if(cv_normdb<=cv_normlimit){
  af_out->filemode&=~CF_NORMALIZE;
  cv_normdb=0.0f;
  return 1;
 }
 return 0;
}

//**************************************************************************
//editor functions
//clear (set to zero) audio
static void cv_mute_zero(acv_fileinfo *af_in,char *destbuff)
{
 PCM_CV_TYPE_I *ptr=(PCM_CV_TYPE_I *)destbuff; // possible int32 and float too (4 bytes)
 unsigned int bob=af_in->currsamplenum-af_in->blocksamplenum;// begin of block
 unsigned int eob=af_in->currsamplenum;                      // end of block
 unsigned int s=af_in->blocksamplenum;
 if(bob<beginsamplenum_mute){  // mute sound at the begin of file
  if(eob>beginsamplenum_mute)
   s=beginsamplenum_mute-bob;
  while(s--)
   *ptr++ =0;
 }else{
  if(eob>endsamplenum_mute){   // mute sound at the end of file
   if(bob<=endsamplenum_mute){
    s=eob-endsamplenum_mute;
    ptr+=(endsamplenum_mute-bob);
   }
   while(s--)
    *ptr++ =0;
  }
 }
}

static int cv_cut_zero(acv_fileinfo *af_in,acv_fileinfo *af_out,char *destbuff)
{
 unsigned int bob=af_in->currsamplenum-af_in->blocksamplenum;// begin of block
 unsigned int eob=af_in->currsamplenum;                      // end of block
 if(bob<beginsamplenum_cut){   // cut sound at the begin of file
  if(eob>beginsamplenum_cut){
   PCM_CV_TYPE_I *ptr=(PCM_CV_TYPE_I *)destbuff;
   af_out->blocksamplenum=eob-beginsamplenum_cut;
   memcpy((void *)ptr,(void *)(ptr+(beginsamplenum_cut-bob)),af_out->blocksamplenum*sizeof(PCM_CV_TYPE_I));
  }else
   af_out->blocksamplenum=0;
 }else{
  if(eob>endsamplenum_cut){    // cut sound at the end of file
   if(bob<=endsamplenum_cut)
    af_out->blocksamplenum=endsamplenum_cut-bob;
   else{
    af_out->blocksamplenum=0;
    return 1;
   }
  }
 }
 return 0;
}

//--------------------------------------------------------------------------
//conversion functions

//stereo to mono, mono to stereo only
static void cv_channels_PCMF(acv_fileinfo *af_in,acv_fileinfo *af_out,unsigned long *blocksamplenum)
{
 unsigned int i;
 PCM_CV_TYPE_F *inptr =(PCM_CV_TYPE_F *)af_out->buffer;
 PCM_CV_TYPE_F *outptr=(PCM_CV_TYPE_F *)af_out->buffer;
 if(af_in->channels==2 && af_out->channels==1){
  *blocksamplenum>>=1;
  for(i=0;i<*blocksamplenum;i++){
   outptr[0]=(inptr[0]+inptr[1])*0.5f;
   outptr++;
   inptr+=2;
  }
 }else{
  if(af_in->channels==1 && af_out->channels==2){
   inptr+=*blocksamplenum;
   outptr+=*blocksamplenum*2;
   for(i=0;i<*blocksamplenum;i++){
    inptr--;
    outptr-=2;
    outptr[0]=outptr[1]=inptr[0];
   }
   *blocksamplenum<<=1;
  }
 }
}

//swap audio channels (only at stereo) in PCMI or in PCMF buffer
static void cv_swapchan(acv_fileinfo *af,char *destbuff)
{
 if(af->channels==2){
  unsigned int i,ch=af->channels;
  PCM_CV_TYPE_I *ptr=(PCM_CV_TYPE_I *)destbuff;
  for(i=af->blocksamplenum/ch;i;i--){
   PCM_CV_TYPE_I lc=ptr[0],rc=ptr[1];
   ptr[0]=rc;
   ptr[1]=lc;
   ptr+=ch;
  }
 }
}

//--------------------------------------------------------------------
//convert frequency
static void cv_freq_PCMF(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 unsigned int i,j,ch;
 PCM_CV_TYPE_F *inptr=(PCM_CV_TYPE_F *)af_out->buffer;
 PCM_CV_TYPE_F *outptr=(PCM_CV_TYPE_F *)af_out->buffer;
 if(!(af_in->freq%af_out->freq)){ // downsampling (2to1,3to1,4to1,...)
  unsigned int scale=af_in->freq/af_out->freq;
  unsigned int instep=(scale-1)*af_out->channels;
  i=af_out->blocksamplenum/(scale*af_out->channels);
  while(i--){
   for(ch=af_out->channels;ch;ch--){
    PCM_CV_TYPE_F insampall=0.0f;
    for(j=0;j<scale;j++)
     insampall+=inptr[j*af_out->channels];
    insampall/=(float)(scale);
    *outptr++=insampall;
    inptr++;
   }
   inptr+=instep;
  }
  af_out->blocksamplenum/=scale;
 }else{
  if(!(af_out->freq%af_in->freq)){ // upsampling (1to2,1to3,1to4,...)
   // sample duplication
   /*int scale=af_out->freq/af_in->freq;
   int outstep=(scale-1)*af_out->channels;
   inptr+=af_out->blocksamplenum;
   outptr+=af_out->blocksamplenum*scale;
   i=af_out->blocksamplenum/af_out->channels;
   while(i--){
    for(ch=af_out->channels;ch;ch--){
     const PCM_CV_TYPE_I insamp=*(--inptr);
     PCM_CV_TYPE_I *putptr=(--outptr);
     for(j=scale;j;j--){
      *putptr=insamp;
      putptr-=af_out->channels;
     }
    }
    outptr-=outstep;
   }
   af_out->blocksamplenum*=scale;*/

   // linear interpolation
   PCM_CV_TYPE_F *save_lastsamples=(PCM_CV_TYPE_F *)alloca(af_out->channels*sizeof(PCM_CV_TYPE_F));
   unsigned int scale=af_out->freq/af_in->freq;
   unsigned int outstep=(scale-1)*af_out->channels;
   inptr +=af_out->blocksamplenum;
   outptr+=af_out->blocksamplenum*scale;
   i=af_out->blocksamplenum/af_out->channels-1;
   memcpy((void *)save_lastsamples,(void *)(inptr-af_out->channels),af_out->channels*sizeof(PCM_CV_TYPE_F));
   while(i--){
    for(ch=af_out->channels;ch;ch--){
     const PCM_CV_TYPE_F insamp1=*(--inptr);
     const PCM_CV_TYPE_F insamp2=*(inptr-af_in->channels);
     PCM_CV_TYPE_F *putptr=--outptr;
     for(j=0;j<scale;j++){
      *putptr=insamp1*(PCM_CV_TYPE_F)(scale-j)/(PCM_CV_TYPE_F)(scale)
	     +insamp2*(PCM_CV_TYPE_F)(      j)/(PCM_CV_TYPE_F)(scale);
      putptr-=af_out->channels;
     }
    }
    outptr-=outstep;
   }
   for(ch=af_out->channels;ch;ch--){
    const PCM_CV_TYPE_F insamp1=*(--inptr);
    const PCM_CV_TYPE_F insamp2=cv_freq_buffer[ch-1];// last sample of previous block
    PCM_CV_TYPE_F *putptr=--outptr;
    for(j=0;j<scale;j++){
     *putptr=insamp1*(PCM_CV_TYPE_F)(scale-j)/(PCM_CV_TYPE_F)(scale)
	    +insamp2*(PCM_CV_TYPE_F)(      j)/(PCM_CV_TYPE_F)(scale);
     putptr-=af_out->channels;
    }
   }
   memcpy((void *)cv_freq_buffer,(void *)save_lastsamples,af_out->channels*sizeof(PCM_CV_TYPE_F));
   af_out->blocksamplenum*=scale;

  }else{ // n to m frequency conversion

  static PCM_CV_TYPE_F inpos;
  const PCM_CV_TYPE_F instep=(PCM_CV_TYPE_F)af_in->freq/(PCM_CV_TYPE_F)af_out->freq;
  const PCM_CV_TYPE_F inend=(PCM_CV_TYPE_F)(af_out->blocksamplenum/af_out->channels);
  PCM_CV_TYPE_F *intmp=cv_freq_buffer;
  unsigned int savesamplenum;

  //if(af_out->freq>af_in->freq){
   savesamplenum=af_out->channels;
   memcpy((intmp+savesamplenum),inptr,af_out->blocksamplenum*sizeof(PCM_CV_TYPE_F));
   while(inpos<inend){
    const float p1=(float)floor(inpos);
    const float m2=inpos-p1,m1=1.0f-m2;
    unsigned int ip1=(int)p1*af_out->channels,ip2=ip1+af_out->channels;
    for(ch=af_out->channels;ch;ch--){
     *outptr++=(intmp[ip1]*m1+intmp[ip2]*m2);
     ip1++;ip2++;
    }
    inpos+=instep;
   }
  /*}else{
   savesamplenum=(int)ceil(instep)*af_out->channels;
   memcpy((intmp+savesamplenum),inptr,af_out->blocksamplenum*sizeof(PCM_CV_TYPE_F));
   while(inpos<inend){
    const float  p1=(float)floor(inpos);
    const float  m12=inpos-p1,m11=1.0f-m12;
    unsigned int ip11=(int)p1*af_out->channels,ip12=ip11+af_out->channels;

    const float  p2=(float)floor(inpos+instep);
    const float  m22=(inpos+instep)-p2,m21=1.0f-m22;
    unsigned int ip21=(int)p2*af_out->channels,ip22=ip21+af_out->channels;

    i=(int)instep;

    //fprintf(stderr,"ssn:%2d p1:%4.3f p2:%4.3f ip11:%4d ip21:%4d m11:%1.4f m21:%1.4f\n",
    //	    savesamplenum,p1,p2,ip11,ip21,m11,m21);
    for(ch=af_out->channels;ch;ch--){
     PCM_CV_TYPE_F outsamp;
     outsamp=intmp[ip11]*m11+intmp[ip12]*m12;

     for(j=0;j<i;j++)
      outsamp+=intmp[ip12+j*af_out->channels];

     if((inpos+instep)<inend){
      outsamp+=intmp[ip21]*m21+intmp[ip22]*m22;
      *outptr++=outsamp/(i+2);
     }else
      *outptr++=outsamp/(i+1);
     ip11++;ip12++;ip21++;ip22++;
    }
    inpos+=instep;
   }
  }*/
  inpos-=inend;
  memcpy(cv_freq_buffer,(cv_freq_buffer+af_out->blocksamplenum),savesamplenum*sizeof(PCM_CV_TYPE_F));
  af_out->blocksamplenum=outptr-(PCM_CV_TYPE_F *)af_out->buffer;
  }
 }
}

//modify sound amplitude (with cv_normdb value) in PCMF buffer
static void cv_amplitude_PCMF(acv_fileinfo *af)
{
 unsigned int i;
 PCM_CV_TYPE_F *ptr=(PCM_CV_TYPE_F *)af->buffer;
 for(i=af->blocksamplenum;i;i--){
  ptr[0] *=cv_normdb;
  ptr++;
 }
}
