//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function:playlist main

#include <malloc.h>
#include "newfunc\newfunc.h"
#include "control\control.h"
#include "display\display.h"
#include "playlist.h"

static void playlist_allocate_memworkarea(void);
static void load_autosaved_list(struct mainvars *);

extern mainvars mvps;
extern char *m3usavename,*mxusavename,*drivescanletters;
extern char *id3tagset[I3I_MAX+1];
extern unsigned int outmode,playcontrol,playrand,playstartsong;
extern unsigned int intsoundcontrol,refdisp,displaymode;
extern unsigned int writeid3tag,playlistload,playlistsave;

static char *memworkarea;

void mpxplay_playlist_init(struct mainvars *mvp)
{
 playlist_allocate_memworkarea();
 playlist_init_pointers(mvp);
 mpxplay_playlist_textconv_init();
 playlist_sortlist_init(mvp);
 playlist_close_infile(mvp->frp0,mvp);
}

static void playlist_allocate_memworkarea(void)
{
 unsigned int allfilenums,pointerssize,filenamessize,id3memsize,memworkareasize;
 allfilenums  =MAX_FILENUM_BROWSER+MAX_FILENUM_PLAYLIST;      //1999+9999=11998
 pointerssize =allfilenums*sizeof(struct playlist_entry_info);//  11998*64
 filenamessize=allfilenums*FILENAMELENGTH;                    // +11998*64
 id3memsize   =allfilenums*ID3LENGTH;                         // +11998*64
 memworkareasize=pointerssize+filenamessize+id3memsize+32768; // =2336384

 memworkarea=malloc(memworkareasize);
 if(memworkarea==NULL)
  mpxplay_close_program(MPXERROR_XMS_MEM);
 pds_memset(memworkarea,0,memworkareasize);
}

void mpxplay_playlist_close(void)
{
 playlist_id3list_close();
 mpxplay_playlist_textconv_close();
 if(memworkarea)
  free(memworkarea);
}

void playlist_init_pointers(struct mainvars *mvp)
{
 struct playlist_side_info *psi0=mvp->psi0,*psi1=psi0+1;
 char *workarea=memworkarea;

 if(playlistload&PLL_DOOMBOX){
  psi0->allfilenum=MAX_FILENUM_PLAYLIST;
  psi1->allfilenum=MAX_FILENUM_BROWSER;
 }else{
  psi0->allfilenum=MAX_FILENUM_BROWSER;
  psi1->allfilenum=MAX_FILENUM_PLAYLIST;
 }

 mvp->pei0=(struct playlist_entry_info *)workarea; // 767872 (11998*64)
  workarea+=(MAX_FILENUM_BROWSER+MAX_FILENUM_PLAYLIST)*sizeof(struct playlist_entry_info);
  workarea+=1024;

 mvp->pei0->filename=workarea; // current song's infos
  workarea+=MAX_PATHNAMELEN+MAX_ID3LEN+(1024-MAX_PATHNAMELEN); // 2048 (1024+1024)

 psi0->filenamesbeginp=workarea;
 psi0->filenameslastp=workarea;            // 127936 (1999*64)
  workarea+=psi0->allfilenum*FILENAMELENGTH;
 psi0->filenamesendp=workarea-300;

 psi1->filenamesbeginp=workarea;
 psi1->filenameslastp=workarea;            // 639936 (9999*64)
  workarea+=psi1->allfilenum*FILENAMELENGTH;
 psi1->filenamesendp=workarea-300;
  workarea+=1024;

 psi0->id3infobeginp=workarea;
 psi0->id3infolastp=workarea;              // 127936 (1999*64)
  workarea+=psi0->allfilenum*ID3LENGTH;
 psi0->id3infoendp=workarea-512;

 psi1->id3infobeginp=workarea;
 psi1->id3infolastp=workarea;              // 639936 (9999*64)
  workarea+=psi1->allfilenum*ID3LENGTH;
 psi1->id3infoendp=workarea-256;           //-----------------
					   //2305664

 psi0->firstentry=psi0->firstsong=psi0->editorhighline=mvp->pei0+1;
 psi0->lastentry=psi0->firstentry-1;
 psi0->endentry=psi0->firstentry+(psi0->allfilenum-2);

 psi1->firstentry=psi1->firstsong=psi1->editorhighline=psi0->firstentry+psi0->allfilenum;
 psi1->lastentry=psi1->firstentry-1;
 psi1->endentry=psi1->firstentry+(psi1->allfilenum-1);

 mvp->aktfilenum=psi1->lastentry;
}

//-------------------------------------------------------------------------
void playlist_peimyself_reset(struct playlist_side_info *psi,struct playlist_entry_info *firstentry,struct playlist_entry_info *lastentry)
{
 struct playlist_entry_info *pei;
 if(!firstentry)
  firstentry=psi->firstentry;
 if(!lastentry)
  lastentry=psi->endentry;
 if(lastentry>=firstentry){
  pei=firstentry;
  do{
   pei->myself=pei;
  }while((++pei)<=lastentry);
 }
}

struct playlist_entry_info *playlist_peimyself_search(struct playlist_side_info *psi,struct playlist_entry_info *pei_src)
{
 struct playlist_entry_info *pei_dest=psi->firstentry,*lastentry=psi->lastentry;
 if(lastentry>=pei_dest){
  do{
   if(pei_dest->myself==pei_src)
    return pei_dest;
  }while((++pei_dest)<=lastentry);
 }
 return NULL;
}

//------------------------------------------------------------------------
void playlist_enable_side(struct playlist_side_info *psi)
{
 if(!(psi->editsidetype&PLT_ENABLED)){
  psi->editsidetype|=PLT_ENABLED;
  playlist_peimyself_reset(psi,NULL,NULL);
  refdisp|=RDT_RESET_EDIT;
 }
}

void playlist_disable_side(struct playlist_side_info *psi)
{
 struct mainvars *mvp=psi->mvp;

 psi->firstsong=psi->editorhighline=psi->firstentry;
 psi->lastentry=psi->firstentry-1;

 psi->filenameslastp=psi->filenamesbeginp;
 psi->id3infolastp=psi->id3infobeginp;

 playlist_fulltime_clearside(psi);

 if(psi==mvp->psip){
  mvp->aktfilenum=psi->firstentry-1;
  mvp->newfilenum=NULL;
  playlist_randlist_clearall(psi);
  playlist_loadsub_sublist_clear();
  if(!(playcontrol&PLAYC_RUNNING))
   mpxplay_stop_and_clear(mvp,0);
 }
 if(psi==mvp->psie){
  playlist_editorhighline_set(psi,psi->firstentry);
  playlist_change_editorside(mvp);
 }
 if(psi==mvp->psil)
  funcbit_disable(playlistload,PLL_LISTS_ALL);
 playlist_chkfile_stop(psi);
 funcbit_disable(psi->editsidetype,PLT_ENABLED|PLT_EXTENDED);
}

void playlist_clear_side(struct playlist_side_info *psi)
{
 struct mainvars *mvp=psi->mvp;

 psi->firstsong=psi->editorhighline=psi->firstentry;
 psi->lastentry=psi->firstentry-1;

 psi->filenameslastp=psi->filenamesbeginp;
 psi->id3infolastp=psi->id3infobeginp;

 pds_memset(psi->firstentry,0,psi->allfilenum*sizeof(struct playlist_entry_info));
 //playlist_peimyself_reset(psi,NULL,NULL); // ???
 playlist_fulltime_clearside(psi);

 if(psi==mvp->psip){
  mvp->aktfilenum=psi->firstentry-1;
  mvp->newfilenum=NULL;
  playlist_randlist_clearall(psi);
 }
 if(psi==mvp->psie)
  playlist_editorhighline_set(psi,psi->firstentry);
 playlist_chkfile_stop(psi);
 funcbit_disable(psi->editsidetype,PLT_ENABLED|PLT_EXTENDED);
}

void playlist_reset_side(struct playlist_side_info *psi)
{
 psi->editsidetype=0;
}

//------------------------------------------------------------------------
static unsigned int startdrive,savedrive;
static char startdir[300],savedir[300];

void save_startdir(struct mainvars *mvp)
{
 struct playlist_side_info *psi;
 unsigned int i;

 pds_getcwd(startdir);
 pds_getdrive(&startdrive);

 psi=mvp->psi0;
 for(i=0;i<PLAYLIST_MAX_SIDES;i++,psi++){
  psi->currdrive=startdrive;
  pds_strcpy(psi->currdir,startdir);
 }
}

void restore_startdir(void)
{
 pds_setdrive(startdrive);
 pds_chdir(startdir);
}

void save_dir(void)
{
 pds_getcwd(savedir);
 pds_getdrive(&savedrive);
}

void restore_dir(void)
{
 pds_setdrive(savedrive);
 pds_chdir(savedir);
}

//------------------------------------------------------------------------
void listline_slice(char **listparts,char *cutchars,char *listline)
{
 char *lastpart=listline,*nextpart=NULL;
 int i,ccn=(int)pds_strlen(cutchars);
 for(i=0;i<ccn;){
  listparts[i]=lastpart;
  do{
   nextpart=pds_strchr(lastpart,cutchars[i]);
   if(nextpart){
    *nextpart++=0;
    lastpart=nextpart;
   }
   i++;
  }while(!nextpart && i<ccn);
 }
}

//-------------------------------------------------------------------------
void start_sideplay(struct mainvars *mvp,struct playlist_side_info *psi)
{
 if(!(playcontrol&PLAYC_RUNNING) && (psi->editsidetype&PLT_ENABLED)){
  mvp->psip=psi;
  if(psi->editsidetype&PLT_DOOMQUEUE){
   playlist_editorhighline_set(psi,psi->firstentry);
   mvp->aktfilenum=psi->firstentry-1;
   mvp->adone=ADONE_RESTART;
  }else{
   if(playrand)
    mvp->newsong=playlist_randlist_getnext(psi);
   else
    mvp->newsong=psi->firstsong;
  }
  playcontrol|=PLAYC_STARTNEXT;
 }
}

//*************************************************************************

void playlist_get_allfilenames(struct mainvars *mvp)
{
 struct playlist_side_info *psi0=mvp->psi0,*psi1=psi0+1;
 unsigned int olddirflag;

 psi0->editsidetype&=PLT_DIRECTORY;
 olddirflag=psi0->editsidetype;
 psi1->editsidetype&=PLT_DIRECTORY;

 if(playlistload&PLL_DOOMBOX){
  mvp->psil=psi0;
  funcbit_disable(psi0->editsidetype,PLT_DIRECTORY);
  funcbit_enable(psi0->editsidetype,PLT_DOOMLIST);
  funcbit_enable(psi1->editsidetype,PLT_DOOMQUEUE);
 }else{
  mvp->psil=psi1;
 }

 if(!(psi1->editsidetype&PLT_DIRECTORY))
  playlist_buildlist_all(mvp->psil);

 if(mvp->psil->editsidetype&PLT_ENABLED){ // playlist has been loaded
  if(playlistload&PLL_DOOMBOX)
   funcbit_disable(psi0->editsidetype,PLT_DIRECTORY);
  funcbit_disable(psi1->editsidetype,PLT_DIRECTORY);
 }else{
  funcbit_enable(psi0->editsidetype,olddirflag);
  if(playlistload&PLL_DOOMBOX)
   funcbit_disable(psi1->editsidetype,PLT_DIRECTORY);
  funcbit_disable(playlistload,PLL_LISTS_CMDL);
 }

 if(psi0->editsidetype&PLT_DIRECTORY){
  mvp->psil=psi1;
  playlist_loaddir_buildbrowser(psi0);
 }

 if(psi1->editsidetype&PLT_DIRECTORY)
  playlist_loaddir_buildbrowser(psi1);
 else
  load_autosaved_list(mvp);

 if(!(psi0->editsidetype&PLT_ENABLED) && !(psi1->editsidetype&PLT_ENABLED))
  mpxplay_close_program(MPXERROR_NOFILE);
}

unsigned int playlist_buildlist_one(struct playlist_side_info *psi,char *listfile,unsigned int loadtype,char *dslp)
{
 unsigned int result=0;
 if(playlist_loadlist_check_extension(listfile) && !pds_strchr(listfile,'*') && !pds_strchr(listfile,'?')){
  funcbit_enable(playlistload,PLL_LOADLIST);
  funcbit_disable(playlistload,(PLL_DRIVESCAN|PLL_DIRSCAN));
  if(playlist_loadlist_mainload(psi,listfile,loadtype))
   result=PLL_LOADLIST;
 }else{
  struct playlist_entry_info *beginentry=psi->lastentry+1;
  funcbit_enable(playlistload,PLL_DIRSCAN);
  if(dslp)
   funcbit_enable(playlistload,PLL_DRIVESCAN);
  else
   funcbit_disable(playlistload,PLL_DRIVESCAN);
  funcbit_disable(playlistload,PLL_LOADLIST);
  playlist_loaddir_scandrives(psi,listfile,dslp);
  playlist_order_filenames_block(psi,beginentry,psi->lastentry);
  result=PLL_DIRSCAN;
 }
 return result;
}

void playlist_buildlist_all(struct playlist_side_info *psi)
{
 // first input file
 if(!playlist_loadsub_buildlist(psi,playlistload,drivescanletters))
  return;

 // multiply input file
 if(outmode!=OUTMODE_FILE){
  unsigned int optcount=OPT_INPUTFILE+1;
  unsigned int allextended=psi->editsidetype&PLT_EXTENDED;
  while((optcount<MAXFREEOPTS) && freeopts[optcount]){
   unsigned int result=playlist_buildlist_one(psi,freeopts[optcount],0,NULL);
   if(result&PLL_LOADLIST)
    allextended&=(psi->editsidetype&PLT_EXTENDED);
   else
    allextended=0;
   freeopts[optcount]=NULL; // load once only
   optcount++;
  }
  funcbit_disable(psi->editsidetype,PLT_EXTENDED);
  funcbit_copy(psi->editsidetype,allextended,PLT_EXTENDED); // playlist (side) is extended if all loaded lists are extended (then we don't check the entries)
 }

 if((playrand&2) && (psi->editsidetype&PLT_ENABLED)){
  playlist_randlist_randomize_side(psi);
  playrand=0;
 }
}

//*************************************************************************
//set starting playlist side & entry
void playlist_init_playside(struct mainvars *mvp)
{
 struct playlist_side_info *psi0=mvp->psi0,*psi1=psi0+1;

 if((psi0->editsidetype&(PLT_ENABLED|PLT_DOOMLIST))==(PLT_ENABLED|PLT_DOOMLIST)){  // ???
  mvp->psie=psi0;
  mvp->psip=psi1;
  if((psi1->editsidetype&PLT_ENABLED) && !(psi1->editsidetype&PLT_DIRECTORY))
   funcbit_enable(playcontrol,PLAYC_STARTNEXT);
 }else{
  if((psi1->editsidetype&PLT_ENABLED) && !(psi1->editsidetype&PLT_DIRECTORY)){
   mvp->psie=psi1;
   mvp->psip=psi1;
   if(playlistload)
    funcbit_enable(playcontrol,PLAYC_STARTNEXT);
  }else{
   mvp->psie=psi0;
   mvp->psip=psi0;
  }
 }

 playlist_editorhighline_seek(psi0,0,SEEK_SET);
 playlist_editorhighline_seek(psi1,0,SEEK_SET);
}

void playlist_init_playsong(struct mainvars *mvp)
{
 struct playlist_side_info *psip=mvp->psip;

 mvp->aktfilenum=psip->firstsong;
 if(psip->lastentry>=psip->firstsong){
  if(playstartsong){
   if((mvp->aktfilenum+playstartsong-1)<=psip->lastentry)
    mvp->aktfilenum+=playstartsong-1;
  }else{
   if(playrand){
    struct playlist_entry_info *newfilenum=playlist_randlist_getnext(psip); // start with a random file
    if(newfilenum)
     mvp->aktfilenum=newfilenum;
   }
  }
 }else{
  funcbit_enable(playcontrol,PLAYC_PAUSENEXT); // ???
 }
 playlist_editorhighline_set(psip,((mvp->aktfilenum<=psip->lastentry)? mvp->aktfilenum:psip->firstentry));
}

//**************************************************************************

unsigned int playlist_open_infile(struct frame *frp,struct mainvars *mvp)
{
 struct playlist_entry_info *pei=mvp->aktfilenum,*pei0=mvp->pei0;
 struct playlist_side_info *psi=mvp->psip;
 struct mpxplay_infile_info_s *miis=frp->infile_infos;
 struct mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 unsigned int i,fnamelen;
 char *sp;

 infile_close(frp);

 if(pei<psi->firstsong || pei>psi->lastentry)
  return 0;
 if(!pei->filename)
  return 0;
 fnamelen=pds_strcpy(pei0->filename,pei->filename);
 if(!fnamelen)
  return 0;

 frp->filetype=GET_HFT(pei->entrytype);
 frp->filesize=pei->filesize;
 frp->infile_funcs=pei->infile_funcs;
 miis->filesize=pei->filesize;
 adi->pcmdatalen=pei->filedatalen;

 get_onefileinfos_open(mvp->psip,pei);
 if(pei->entrytype<DFT_AUDIOFILE)
  return 0;

 frp->filetype=GET_HFT(pei->entrytype);
 frp->filesize=pei->filesize;
 frp->infile_funcs=pei->infile_funcs;
 miis->filesize=pei->filesize;
 adi->pcmdatalen=pei->filedatalen;

 if(!infile_open(frp,pei0->filename))
  return 0;

 funcbit_enable(pei->infobits,PEIF_ENABLED);
 pei->timesec=frp->timesec;
 pei->filesize=frp->filesize;
 //pei->timesec=miis->timesec;
 //pei->filesize=miis->filesize;
 pei->infile_funcs=frp->infile_funcs;
 pei->filedatalen=adi->pcmdatalen;

 //save infos of current song from playlist (entry)
 sp=pei0->filename+fnamelen+1;
 for(i=0;i<=I3I_MAX;i++){
  if(id3tagset[i])
   pei0->id3info[i]=id3tagset[i];
  else
   if(pei->id3info[i] && (sp<(pei0->filename+MAX_PATHNAMELEN+MAX_ID3LEN))){
    pei0->id3info[i]=sp;
    sp+=pds_strcpy(sp,pei->id3info[i])+1;
   }else
    pei0->id3info[i]=NULL;
 }
 /*pei0->infobits=pei->infobits;
 pei0->timesec=pei->timesec;
 pei0->filesize=pei->filesize;
 pei0->filedatalen=pei->filedatalen;*/
 return 1;
}

void playlist_close_infile(struct frame *frp,struct mainvars *mvp)
{
 unsigned int i;
 struct playlist_entry_info *pei0=mvp->pei0;
 infile_close(frp);
 pds_strcpy(pei0->filename,"No file");
 for(i=0;i<=I3I_MAX;i++)
  pei0->id3info[i]=NULL;
 /*pei0->infobits=0;
 pei0->timesec=0;
 pei0->filesize=0;
 pei0->filedatalen=0;*/
}

//*************************************************************************

void playlist_editorhighline_check(struct playlist_side_info *psi)
{
 struct playlist_entry_info *pei=psi->editorhighline;

 if(pei>psi->lastentry)
  pei=psi->lastentry;
 if(pei<psi->firstentry)
  pei=psi->firstentry;
 psi->editorhighline=pei;
}

void playlist_editorhighline_seek(struct playlist_side_info *psi,long offset,int whence)
{
 switch(whence){
  case SEEK_SET:psi->editorhighline=psi->firstentry+offset;break;
  case SEEK_CUR:psi->editorhighline+=offset;break;
  case SEEK_END:psi->editorhighline=psi->lastentry-offset;break;
 }
 playlist_editorhighline_check(psi);
}

void playlist_editorhighline_set(struct playlist_side_info *psi,struct playlist_entry_info *pei)
{
 if(pei>=psi->firstentry && pei<=psi->lastentry)
  psi->editorhighline=pei;
}

//-------------------------------------------------------------------------

void playlist_change_editorside(struct mainvars *mvp)
{
 struct playlist_side_info *psie=mvp->psie;
 if(psie->psio->editsidetype&PLT_ENABLED){
  psie=psie->psio;
  mvp->psie=psie;
  if(psie->editsidetype&PLT_DIRECTORY){
   pds_setdrive(psie->currdrive);
   pds_chdir(psie->currdir);
  }
  refdisp|=RDT_INIT_EDIT;
 }
}

//**************************************************************************
playlist_entry_info *playlist_search_filename(struct playlist_side_info *psi,char *filename)
{
 struct playlist_entry_info *pei;

 if(!filename || !filename[0])
  return NULL;
 while(*filename==' ')
  filename++;
 if(!*filename)
  return NULL;
 for(pei=psi->firstentry;pei<=psi->lastentry;pei++){
  if(pds_stricmp(pei->filename,filename)==0)
   return pei;
 }
 return NULL;
}

//move the cursor to the subdir/list where you came from
void playlist_search_lastdir(struct playlist_side_info *psi,char *lastdir)
{
 struct playlist_entry_info *pei_set;

 pei_set=playlist_search_filename(psi,lastdir);
 if(!pei_set)
  pei_set=psi->firstentry;
 playlist_editorhighline_set(psi,pei_set);
}

void playlist_search_firstsong(struct playlist_side_info *psi)
{
 struct playlist_entry_info *pei;
 psi->firstsong=psi->lastentry+1;
 for(pei=psi->firstentry;pei<=psi->lastentry;pei++)
  if(GET_HFT(pei->entrytype)!=HFT_DFT){
   psi->firstsong=pei;
   break;
  }
}

void playlist_change_sublist_or_directory(struct playlist_side_info *psi,unsigned long head)
{
 if(GET_HFT(head)==HFT_DFT){
  if(head&DFTM_PLAYLIST)
   psi=playlist_loadsub_sublist_change(psi,head);
  else
   psi=playlist_loaddir_changedir(psi,head);
  playlist_chkfile_start_norm(psi,0);
 }
}

//-------------------------------------------------------------------------
void playlist_reload_side(struct mainvars *mvp,struct playlist_side_info *psi)
{
 if(psi->editsidetype&PLT_DIRECTORY)
  playlist_loaddir_buildbrowser(psi);
 else
  if(!(psi->editsidetype&PLT_DOOMQUEUE))
   playlist_buildlist_all(psi);
  else
   return;

 if((psi==mvp->psip) && !(playcontrol&PLAYC_RUNNING)){
  mpxplay_stop_and_clear(mvp,0);
  mvp->adone=ADONE_EOF;
 }
 playlist_chkfile_start_norm(psi,0);
}

//*************************************************************************

void write_id3tags(struct mainvars *mvp)
{
 struct frame *frp;
 struct playlist_side_info *psi;
 struct playlist_entry_info *pei;
 unsigned int error;
 char *filename,*shortfname;
 char sout[128];

 if(!writeid3tag)
  return;

 frp=mvp->frp0+2;
 psi=mvp->psil;

 for(pei=psi->firstsong;pei<=psi->lastentry;pei++){
  filename=pei->filename;
  frp->filetype=GET_HFT(pei->entrytype);
  frp->filesize=pei->filesize;
  frp->infile_funcs=pei->infile_funcs;
  frp->infile_infos->audio_decoder_infos->pcmdatalen=pei->filedatalen;
  error=infile_write_id3tag(frp,filename,&pei->id3info[0]);
  shortfname=pds_strrchr(filename,'\\');
  if(shortfname)
   shortfname++;
  else
   shortfname=filename;
  switch(error){
   case MPXPLAY_ERROR_INFILE_OK               :sprintf(sout,"%.15s  %-30.30s : %-30.30s",shortfname,pei->id3info[I3I_ARTIST],pei->id3info[I3I_TITLE]);break;
   case MPXPLAY_ERROR_INFILE_CANTOPEN         :sprintf(sout,"ID3TAG write error at %.25s (read-only or not exists)!",shortfname);break;
   case MPXPLAY_ERROR_INFILE_WRITETAG_FILETYPE:sprintf(sout,"ID3TAG write is not supported for this filetype (%.25s)",shortfname);break;
   case MPXPLAY_ERROR_INFILE_WRITETAG_TAGTYPE :sprintf(sout,"ID3TAG update is not supported in this tag (ie:APEv2)(%.20s)",shortfname);break;
   default:sprintf(sout,"ID3TAG write error at %.20s (unknown error)!",shortfname);break;
  }
  pds_textdisplay_printf(sout);
 }
 mpxplay_close_program(0);
}

//**************************************************************************
static void save_m3u_playlist(struct playlist_side_info *psi,FILE *fp,char *path)
{
 struct playlist_entry_info *pei;
 int j;
 char cnvtmp[MAX_ID3LEN];

 j=pds_strlen(path);

 if(j>0){ // allways have to be
  if(playlistsave&PLST_EXTM3U)
   fprintf(fp,"#EXTM3U\n");

  for(pei=psi->firstsong;pei<=psi->lastentry;pei++){
   char csave;
   if((playlistsave&PLST_EXTM3U) && (pei->infobits&PEIF_ENABLED)){
    fprintf(fp,"#EXTINF:%d,%s - ",
  	       pei->timesec,
               mpxplay_playlist_textconv_back(cnvtmp,pei->id3info[I3I_ARTIST]));
    fprintf(fp,"%s\n",
               mpxplay_playlist_textconv_back(cnvtmp,pei->id3info[I3I_TITLE]));
   }
   csave=pei->filename[j];
   pei->filename[j]=0;
   if(pds_stricmp(pei->filename,path)==0){  // if path of startdir and save-dir is the same
    fprintf(fp,"%s\n",&pei->filename[j+1]);  // save relative to currdir (subdir\filename.mp3)
    pei->filename[j]=csave;
   }else{
    pei->filename[j]=csave;
    if(pds_getdrivenum_from_path(pei->filename)>=0 && pds_getdrivenum_from_path(pei->filename)==pds_getdrivenum_from_path(path)){ // drives are same
     fprintf(fp,"%s\n",&pei->filename[2]);   // save relative to rootdir (\subdir\filename.mp3)
    }else{
     fprintf(fp,"%s\n",pei->filename);       // save with full path      (d:\subdir\filename.mp3)
    }
   }
  }
 }else{ // ???
  for(pei=psi->firstsong;pei<=psi->lastentry;pei++)
   fprintf(fp,"%s\n",pei->filename);         // save with full path
 }
}

static void save_mxu_playlist(struct playlist_side_info *psi,FILE *fp)
{
 struct playlist_entry_info *pei;
 char cnvtmp[MAX_ID3LEN];

 for(pei=psi->firstsong;pei<=psi->lastentry;pei++){
  fprintf(fp,"%s%s",(pei->filename)? pei->filename:"",
                      mpxplay_playlist_textconv_back(cnvtmp,pei->id3info[I3I_ARTIST]));
  fprintf(fp,"%s%8.8X\n",mpxplay_playlist_textconv_back(cnvtmp,pei->id3info[I3I_TITLE]),
                          (pei->infobits&PEIF_ENABLED)? (MXUFLAG_ENABLED|(pei->timesec&MXUFLAG_TIMEMASK)):0);
 }
}

void save_playlist(struct mainvars *mvp)
{
 struct playlist_side_info *psi;
 FILE *fp;
 char filename[300],path[300];

 if(playlistsave){
  if(playlistsave&PLST_MANUAL){
   psi=mvp->psil; // playlist side ???
   if(playlistsave&PLST_EXTENDED)
    pds_fullpath(filename,mxusavename);
   else
    pds_fullpath(filename,m3usavename);
   pds_getpath_from_fullname(path,filename);
  }else{ // PLST_AUTO
   psi=mvp->psi0+1; // right side
   pds_getpath_from_fullname(path,freeopts[OPT_PROGNAME]); // get directory of mpxplay.exe
   pds_strcpy(filename,path);
   pds_strcat(filename,"\\");
   if(playlistsave&PLST_EXTENDED)
    pds_strcat(filename,mxusavename);
   else
    pds_strcat(filename,m3usavename);
  }
  if((fp=pds_fopen(filename,"wt"))!=NULL){
   if(playlistsave&PLST_EXTENDED)
    save_mxu_playlist(psi,fp);
   else
    save_m3u_playlist(psi,fp,path);
   fclose(fp);
  }
 }
}

static void load_autosaved_list(struct mainvars *mvp)
{
 if(playlistsave&PLST_AUTO){
  struct playlist_side_info *psi0=mvp->psi0;
  struct playlist_side_info *psi1=psi0+1;

  if(!(psi1->editsidetype&PLT_ENABLED)){
   char filename[300];
   pds_getpath_from_fullname(filename,freeopts[OPT_PROGNAME]);
   pds_strcat(filename,"\\");
   if(playlistsave&PLST_EXTENDED)
    pds_strcat(filename,mxusavename);
   else
    pds_strcat(filename,m3usavename);
   playlist_loadlist_mainload(psi1,filename,0);
  }
 }
}
