//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2005 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: DTS file handling
//requires the dec_dts\dtsdec.lib and dts.h files

#include "in_file.h"

#ifdef MPXPLAY_LINK_INFILE_DTS

#include <string.h>
#include "newfunc\newfunc.h"
#include "in_funcs.h"
#include "dec_dts\dts.h"

#define DTS_OUTCHANNELS 2

#define DTS_HEADERSIZE    14
#define DTS_FRAMESIZE_MIN 96
#define DTS_FRAMESIZE_MAX 18726
#define DTS_BLOCKSAMPLES  256

#define DTS_BITSTREAM_BUFSIZE (DTS_FRAMESIZE_MAX*2)

#define DTS_SYNC_RETRY_BYTE   (DTS_FRAMESIZE_MAX*8)
#define DTS_SYNC_RETRY_FRAME  64

typedef struct dts_decoder_data{
 dts_state_t *dtsstate;
 int info_flags;
 int sample_rate;
 int bit_rate;
 int frame_length;

 int config_flags;
 level_t config_level;

 int lastframesize;
 int blockcount;

 unsigned char *bitstream_buffer;
 unsigned int bitstream_storedbytes;

 MPXPLAY_PCMOUT_FLOAT_T *pcmoutsave_buffer;
 unsigned int pcmoutsave_samplenum;

}dts_decoder_data;

static const unsigned int dts_channums[]={
 1,  // DTS_MONO             0
 2,  // DTS_CHANNEL          1 // ???
 2,  // DTS_STEREO           2
 2,  // DTS_STEREO_SUMDIFF   3
 2,  // DTS_STEREO_TOTAL     4
 3,  // DTS_3F               5
 3,  // DTS_2F1R             6
 4,  // DTS_3F1R             7
 4,  // DTS_2F2R             8
 5,  // DTS_3F2R             9
 6,  // DTS_4F2R            10
 6,6,7,8,8
};

static void INDTS_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis);
static void INDTS_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,int cleartype);

static int indts_sync_frame(struct dts_decoder_data *dtsi,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds);

static char *indts_bitstream_init(struct dts_decoder_data *dtsi);
static void indts_bitstream_close(struct dts_decoder_data *dtsi);
static int  indts_bitstream_fill(struct dts_decoder_data *dtsi,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,unsigned int needbytes);
static void indts_bitstream_advance(struct dts_decoder_data *dtsi,unsigned int skipbytes);
static void indts_bitstream_reset(struct dts_decoder_data *dtsi);

static int indts_assign_values(struct dts_decoder_data *dtsi,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,struct mpxplay_infile_info_s *miis)
{
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;

 miis->filesize=fbfs->filelength(fbds);
 if(miis->filesize<DTS_FRAMESIZE_MIN)  // ???
  return 0;

 adi->freq=dtsi->sample_rate;
 adi->filechannels=dts_channums[dtsi->info_flags&DTS_CHANNEL_MASK];
 if(dtsi->info_flags&DTS_LFE)
  adi->filechannels++;
 adi->outchannels=DTS_OUTCHANNELS;
 adi->bitrate=dtsi->bit_rate/1000;

 adi->bits=1;  // output scale : +1.0 ... -1.0
 adi->infobits|=ADI_FLAG_FLOATOUT;
 adi->longname="DTS-file";

 adi->channeltext=malloc(MPXPLAY_ADITEXTSIZE_CHANNEL+8);
 if(!adi->channeltext)
  return 0;
 sprintf(adi->channeltext,"%d.%d chan",dts_channums[dtsi->info_flags&DTS_CHANNEL_MASK],((dtsi->info_flags&DTS_LFE)? 1:0));

 adi->wave_id=0x2001; // ???
 adi->shortname="DTS";
 if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT){
  adi->infobits|=ADI_FLAG_BITSTREAMOUT;
  adi->pcmsamplenum_frame=dtsi->blockcount*DTS_BLOCKSAMPLES;
 }

 dtsi->config_flags=DTS_STEREO;//|DTS_ADJUST_LEVEL;
 dtsi->config_level=(level_t)32768.0;

 return 1;
}

static void *indts_check_header(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct dts_decoder_data *dtsi;

 if(!fbfs->fopen_read(fbds,filename,0))
  return NULL;

 dtsi=calloc(1,sizeof(struct dts_decoder_data));
 if(!dtsi)
  return dtsi;

 dtsi->dtsstate=dts_init(0);
 if(!dtsi->dtsstate)
  goto err_out_chk;

 if(!indts_bitstream_init(dtsi))
  goto err_out_chk;

 if(indts_sync_frame(dtsi,fbfs,fbds)<0)
  goto err_out_chk;

 if(!indts_assign_values(dtsi,fbfs,fbds,miis))
  goto err_out_chk;

 return dtsi;

err_out_chk:
 INDTS_infile_close(fbfs,fbds,dtsi,miis);
 return NULL;
}

static void *INDTS_infile_check(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 return indts_check_header(fbfs,fbds,filename,miis);
}

static void *INDTS_infile_open(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,char *filename,struct mpxplay_infile_info_s *miis)
{
 struct dts_decoder_data *dtsi;
 dtsi=indts_check_header(fbfs,fbds,filename,miis);
 if(!dtsi)
  return dtsi;
 dtsi->pcmoutsave_buffer=malloc(DTS_BLOCKSAMPLES*DTS_OUTCHANNELS*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
 if(!dtsi->pcmoutsave_buffer)
  goto err_out_opn;

 return dtsi;

err_out_opn:
 INDTS_infile_close(fbfs,fbds,dtsi,miis);
 return NULL;
}

static void INDTS_infile_close(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct dts_decoder_data *dtsi=(struct dts_decoder_data *)infile_data;
 if(dtsi){
  mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
  if(dtsi->pcmoutsave_buffer)
   free(dtsi->pcmoutsave_buffer);
  indts_bitstream_close(dtsi);
  dts_free(dtsi->dtsstate);
  if(adi->channeltext)
   free(adi->channeltext);
  free(dtsi);
 }
 fbfs->fclose(fbds);
}

static int INDTS_infile_decode(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis)
{
 struct dts_decoder_data *dtsi=(struct dts_decoder_data *)infile_data;
 mpxplay_audio_decoder_info_s *adi=miis->audio_decoder_infos;
 unsigned int samplenum_left,samplenum_currblock;
 MPXPLAY_PCMOUT_FLOAT_T *pcmout;
 int retcode;

 if(adi->infobits&ADI_CNTRLBIT_BITSTREAMOUT){
  retcode=indts_sync_frame(dtsi,fbfs,fbds);
  if(retcode!=MPXPLAY_ERROR_INFILE_OK)
   return retcode;
  memcpy(adi->pcm_outdatabuf,dtsi->bitstream_buffer,dtsi->lastframesize);
  adi->pcmoutsamplenum=dtsi->lastframesize;
  indts_bitstream_advance(dtsi,dtsi->lastframesize);
  return MPXPLAY_ERROR_INFILE_OK;
 }

 samplenum_left=adi->pcmsamplenum_frame;
 samplenum_currblock=DTS_BLOCKSAMPLES;
 pcmout=(MPXPLAY_PCMOUT_FLOAT_T *)adi->pcm_outdatabuf;

 if(dtsi->pcmoutsave_samplenum){
  samplenum_currblock=min(dtsi->pcmoutsave_samplenum,samplenum_left);
  pds_memcpy(pcmout,dtsi->pcmoutsave_buffer+(DTS_BLOCKSAMPLES-samplenum_currblock)*DTS_OUTCHANNELS,samplenum_currblock*DTS_OUTCHANNELS*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
  pcmout+=samplenum_currblock*DTS_OUTCHANNELS;
  dtsi->pcmoutsave_samplenum-=samplenum_currblock;
  samplenum_left-=samplenum_currblock;
  adi->pcmoutsamplenum+=samplenum_currblock*DTS_OUTCHANNELS;
 }

 while(samplenum_left){
  MPXPLAY_PCMOUT_FLOAT_T *pcmptr;
  sample_t *dtsout;
  unsigned int i,ch;

  if(!dtsi->blockcount){
   if(dtsi->lastframesize){
    indts_bitstream_advance(dtsi,dtsi->lastframesize);
    dtsi->lastframesize=0;
   }
   retcode=indts_sync_frame(dtsi,fbfs,fbds);
   if(retcode!=MPXPLAY_ERROR_INFILE_OK)
    return retcode;
  }
  if(dts_block(dtsi->dtsstate)<0){
   dtsi->blockcount=0;
   continue;
  }
  dtsout=dts_samples(dtsi->dtsstate);
  dtsi->blockcount--;

  pcmptr=(samplenum_left>=DTS_BLOCKSAMPLES)? pcmout:dtsi->pcmoutsave_buffer;

  for(ch=0;ch<DTS_OUTCHANNELS;ch++){
   for(i=0;i<DTS_BLOCKSAMPLES;i++){
    *pcmptr=(MPXPLAY_PCMOUT_FLOAT_T)(*dtsout);
    pcmptr+=DTS_OUTCHANNELS;
    dtsout++;
   }
   pcmptr-=(DTS_BLOCKSAMPLES*DTS_OUTCHANNELS-1);
  }
  if(samplenum_left>=DTS_BLOCKSAMPLES){
   pcmout+=DTS_BLOCKSAMPLES*DTS_OUTCHANNELS;
   samplenum_left-=DTS_BLOCKSAMPLES;
   adi->pcmoutsamplenum+=DTS_BLOCKSAMPLES*DTS_OUTCHANNELS;
  }else{
   pds_memcpy(pcmout,dtsi->pcmoutsave_buffer,samplenum_left*DTS_OUTCHANNELS*sizeof(MPXPLAY_PCMOUT_FLOAT_T));
   dtsi->pcmoutsave_samplenum=DTS_BLOCKSAMPLES-samplenum_left;
   adi->pcmoutsamplenum+=samplenum_left*DTS_OUTCHANNELS;
   samplenum_left=0;
  }
 }

 return MPXPLAY_ERROR_INFILE_OK;
}

static void INDTS_clearbuffs(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,int cleartype)
{
 struct dts_decoder_data *dtsi=(struct dts_decoder_data *)infile_data;
 indts_bitstream_reset(dtsi);
 dtsi->blockcount=0;
 dtsi->lastframesize=0;
 if(cleartype&(MPX_SEEKTYPE_BOF|MPX_SEEKTYPE_PAUSE)){
  dts_reset(dtsi->dtsstate);
  dtsi->pcmoutsave_samplenum=0;
 }
}

static long INDTS_fseek(struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,void *infile_data,struct mpxplay_infile_info_s *miis,long newframenum)
{
 long newfilepos;
 pds_ftoi((float)newframenum*(float)miis->filesize/(float)miis->allframes,&newfilepos);
 if(fbfs->fseek(fbds,newfilepos,SEEK_SET)<0)
  return MPXPLAY_ERROR_INFILE_EOF;
 return newframenum;
}

//------------------------------------------------------------------------
static int indts_sync_frame(struct dts_decoder_data *dtsi,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds)
{
 unsigned int retry_frame=DTS_SYNC_RETRY_FRAME;
 unsigned int retry_byte=DTS_SYNC_RETRY_BYTE;
 do{
  if(!indts_bitstream_fill(dtsi,fbfs,fbds,DTS_HEADERSIZE))
   return MPXPLAY_ERROR_INFILE_NODATA;

  dtsi->lastframesize=dts_syncinfo(dtsi->dtsstate,dtsi->bitstream_buffer,&dtsi->info_flags,&dtsi->sample_rate,&dtsi->bit_rate,&dtsi->frame_length);
  if(dtsi->lastframesize<=0){
   if(!(--retry_byte))
    break;
   indts_bitstream_advance(dtsi,1);
   continue;
  }

  if(!indts_bitstream_fill(dtsi,fbfs,fbds,dtsi->lastframesize))
   return MPXPLAY_ERROR_INFILE_NODATA;

  if(dts_frame(dtsi->dtsstate,dtsi->bitstream_buffer,&dtsi->config_flags,&dtsi->config_level,0)==0){
   dtsi->blockcount=dts_blocks_num(dtsi->dtsstate);
   dts_dynrng(dtsi->dtsstate,NULL,NULL);
   return MPXPLAY_ERROR_INFILE_OK;
  }

  indts_bitstream_advance(dtsi,dtsi->lastframesize); // skips frame on error

  if(!(--retry_frame))
   break;

 }while(1);
 return MPXPLAY_ERROR_INFILE_EOF;
}

//------------------------------------------------------------------------
//bitsteam (input)

static char *indts_bitstream_init(struct dts_decoder_data *dtsi)
{
 dtsi->bitstream_buffer=(unsigned char *)calloc(DTS_BITSTREAM_BUFSIZE,sizeof(unsigned char));
 return dtsi->bitstream_buffer;
}

static void indts_bitstream_close(struct dts_decoder_data *dtsi)
{
 if(dtsi->bitstream_buffer){
  free(dtsi->bitstream_buffer);
  dtsi->bitstream_buffer=NULL;
 }
}

static int indts_bitstream_fill(struct dts_decoder_data *dtsi,struct mpxplay_filehand_buffered_func_s *fbfs,void *fbds,unsigned int needbytes)
{
 if(needbytes>DTS_BITSTREAM_BUFSIZE)
  needbytes=DTS_BITSTREAM_BUFSIZE;

 if(dtsi->bitstream_storedbytes<needbytes)
  dtsi->bitstream_storedbytes+=fbfs->fread(fbds,dtsi->bitstream_buffer+dtsi->bitstream_storedbytes,needbytes-dtsi->bitstream_storedbytes);

 return ((dtsi->bitstream_storedbytes>=needbytes)? 1:0);
}

static void indts_bitstream_advance(struct dts_decoder_data *dtsi,unsigned int skipbytes)
{
 if(dtsi->bitstream_storedbytes>skipbytes){
  pds_memcpy(dtsi->bitstream_buffer,dtsi->bitstream_buffer+skipbytes,dtsi->bitstream_storedbytes-skipbytes);
  dtsi->bitstream_storedbytes-=skipbytes;
 }else
  dtsi->bitstream_storedbytes=0;
}

static void indts_bitstream_reset(struct dts_decoder_data *dtsi)
{
 dtsi->bitstream_storedbytes=0;
}

//--------------------------------------------------------------------------

struct mpxplay_infile_func_s IN_DTS_funcs={
 0,
 NULL,
 NULL,
 NULL, // &INDTS_infile_check, // it's slow yet for an autodetection
 &INDTS_infile_check,
 &INDTS_infile_open,
 &INDTS_infile_close,
 &INDTS_infile_decode,
 &INDTS_fseek,
 &INDTS_clearbuffs,
 &ape_tag_get,
 &ape_tag_put,
 NULL,
 {"DTS",NULL}
};

#endif // MPXPLAY_LINK_INFILE_DTS
