/* 
 *    parse.c
 *
 *	Copyright (C) Aaron Holtzman - May 1999
 *
 *  This file is part of ac3dec, a free Dolby AC-3 stream decoder.
 *	
 *  ac3dec is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  ac3dec is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  -----------------------------------------------------------------------
 *
 *  Modifications by Attila Padar (PDSoft)(Jan 2002,Nov 2004):
 *   - parse_syncinfo
 *   - audblk error check
 *
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include "ac3.h"
#include "bitstrm.h"
#include "crc.h"
#include "parse.h"

static const uint_16 nfchans[8] = {2,1,2,3,3,4,4,5};

struct frmsize_s
{
  uint_16 bit_rate;
  uint_16 frm_size[3];
};

static const struct frmsize_s frmsizecod_tbl[64] =
{
	{ 32  ,{64   ,69   ,96   } },
	{ 32  ,{64   ,70   ,96   } },
	{ 40  ,{80   ,87   ,120  } },
	{ 40  ,{80   ,88   ,120  } },
	{ 48  ,{96   ,104  ,144  } },
	{ 48  ,{96   ,105  ,144  } },
	{ 56  ,{112  ,121  ,168  } },
	{ 56  ,{112  ,122  ,168  } },
	{ 64  ,{128  ,139  ,192  } },
	{ 64  ,{128  ,140  ,192  } },
	{ 80  ,{160  ,174  ,240  } },
	{ 80  ,{160  ,175  ,240  } },
	{ 96  ,{192  ,208  ,288  } },
	{ 96  ,{192  ,209  ,288  } },
	{ 112 ,{224  ,243  ,336  } },
	{ 112 ,{224  ,244  ,336  } },
	{ 128 ,{256  ,278  ,384  } },
	{ 128 ,{256  ,279  ,384  } },
	{ 160 ,{320  ,348  ,480  } },
	{ 160 ,{320  ,349  ,480  } },
	{ 192 ,{384  ,417  ,576  } },
	{ 192 ,{384  ,418  ,576  } },
	{ 224 ,{448  ,487  ,672  } },
	{ 224 ,{448  ,488  ,672  } },
	{ 256 ,{512  ,557  ,768  } },
	{ 256 ,{512  ,558  ,768  } },
	{ 320 ,{640  ,696  ,960  } },
	{ 320 ,{640  ,697  ,960  } },
	{ 384 ,{768  ,835  ,1152 } },
	{ 384 ,{768  ,836  ,1152 } },
	{ 448 ,{896  ,975  ,1344 } },
	{ 448 ,{896  ,976  ,1344 } },
	{ 512 ,{1024 ,1114 ,1536 } },
	{ 512 ,{1024 ,1115 ,1536 } },
	{ 576 ,{1152 ,1253 ,1728 } },
	{ 576 ,{1152 ,1254 ,1728 } },
	{ 640 ,{1280 ,1393 ,1920 } },
	{ 640 ,{1280 ,1394 ,1920 } }
};

unsigned int ac3dec_parse_syncinfo(ac3_decoder_data *amip)
{
 bitstream_t *bs=amip->bs;
 syncinfo_t *syncinfo=amip->syncinfo;
 int syncretry,frameretry=AC3DEC_HEADSEARCH_FRAME_RETRY;
 uint_16 sync_word;
 uint_32 tmp;

 ac3dec_bitstream_reset(bs);

 do{
  if(!bs->file)
   return 0;

  sync_word = ac3dec_bitstream_readbyte(amip)<<8;
  sync_word|= ac3dec_bitstream_readbyte(amip);

  if(!bs->file)
   return 0;

  syncretry=0;
  while(sync_word!=0x0b77){
   sync_word<<=8;
   sync_word|=ac3dec_bitstream_readbyte(amip);
   syncretry++;
   if(!bs->file || syncretry>AC3DEC_HEADSEARCH_SYNC_RETRY)
    return 0;
  }

  if(!ac3dec_bitstream_loadframe(amip,3)) //(pre)load 3 bytes for the freq and framesize infos
   continue;

  bs->bitindex=0;   // else the ac3dec_getbits fails

  tmp =              ac3dec_getbits(bs,8);
  tmp = (tmp << 8) | ac3dec_getbits(bs,8);
  tmp = (tmp << 8) | ac3dec_getbits(bs,8);

  syncinfo->fscod  = (tmp >> 6) & 0x3;

  if(syncinfo->fscod == 3) // illegal freq
   continue;

  switch(syncinfo->fscod){
   case 2:amip->freq = 32000;break;
   case 1:amip->freq = 44100;break;
   case 0:amip->freq = 48000;break;
  }

  syncinfo->frmsizecod = tmp & 0x3f;
  syncinfo->frame_size =frmsizecod_tbl[syncinfo->frmsizecod].frm_size[syncinfo->fscod];
  syncinfo->bit_rate = frmsizecod_tbl[syncinfo->frmsizecod].bit_rate;

  if(!ac3dec_bitstream_loadframe(amip,syncinfo->frame_size*2-2)) // -3 stored bytes (-2 is the syncword)
   continue;

  if(ac3dec_crc_processframe(bs->buffer,syncinfo->frame_size*2-2)==0) // crc of frame is ok
   return 1;

 }while(--frameretry);
 return 0;
}

// This routine fills a bsi struct from the AC3 stream
void ac3dec_parse_bsi(bsi_t *bsi,bitstream_t *bs)
{
 uint_32 i;

 // Check the AC-3 version number
 bsi->bsid = ac3dec_getbits(bs,5);

 // Get the audio service provided by the steram
 bsi->bsmod = ac3dec_getbits(bs,3);

 // Get the audio coding mode (ie how many channels)
 bsi->acmod = ac3dec_getbits(bs,3);
 // Predecode the number of full bandwidth channels as we use this number a lot
 bsi->nfchans = nfchans[bsi->acmod];

 // If it is in use, get the centre channel mix level
 if ((bsi->acmod & 0x1) && (bsi->acmod != 0x1))
  bsi->cmixlev = ac3dec_getbits(bs,2);

 // If it is in use, get the surround channel mix level
 if (bsi->acmod & 0x4)
  bsi->surmixlev = ac3dec_getbits(bs,2);

 // Get the dolby surround mode if in 2/0 mode
 if(bsi->acmod == 0x2)
  bsi->dsurmod= ac3dec_getbits(bs,2);

 // Is the low frequency effects channel on?
 bsi->lfeon = ac3dec_getbits(bs,1);

 // Get the dialogue normalization level
 bsi->dialnorm = ac3dec_getbits(bs,5);

 // Does compression gain exist?
 bsi->compre = ac3dec_getbits(bs,1);
 if (bsi->compre){
  // Get compression gain
  bsi->compr = ac3dec_getbits(bs,8);
 }

 // Does language code exist?
 bsi->langcode = ac3dec_getbits(bs,1);
 if (bsi->langcode){
  // Get langauge code
  bsi->langcod = ac3dec_getbits(bs,8);
 }

 // Does audio production info exist?
 bsi->audprodie = ac3dec_getbits(bs,1);
 if (bsi->audprodie){
  // Get mix level
  bsi->mixlevel = ac3dec_getbits(bs,5);

  // Get room type
  bsi->roomtyp = ac3dec_getbits(bs,2);
 }

 // If we're in dual mono mode then get some extra info
 if (bsi->acmod ==0){
  // Get the dialogue normalization level two
  bsi->dialnorm2 = ac3dec_getbits(bs,5);

  // Does compression gain two exist?
  bsi->compr2e = ac3dec_getbits(bs,1);
  if (bsi->compr2e){
   // Get compression gain two
   bsi->compr2 = ac3dec_getbits(bs,8);
  }

  // Does language code two exist?
  bsi->langcod2e = ac3dec_getbits(bs,1);
  if (bsi->langcod2e){
   // Get langauge code two
   bsi->langcod2 = ac3dec_getbits(bs,8);
  }

  // Does audio production info two exist?
  bsi->audprodi2e = ac3dec_getbits(bs,1);
  if (bsi->audprodi2e){
   // Get mix level two
   bsi->mixlevel2 = ac3dec_getbits(bs,5);

   // Get room type two
   bsi->roomtyp2 = ac3dec_getbits(bs,2);
  }
 }

 // Get the copyright bit
 bsi->copyrightb = ac3dec_getbits(bs,1);

 // Get the original bit
 bsi->origbs = ac3dec_getbits(bs,1);

 // Does timecode one exist?
 bsi->timecod1e = ac3dec_getbits(bs,1);

 if(bsi->timecod1e)
  bsi->timecod1 = ac3dec_getbits(bs,14);

 // Does timecode two exist?
 bsi->timecod2e = ac3dec_getbits(bs,1);

 if(bsi->timecod2e)
  bsi->timecod2 = ac3dec_getbits(bs,14);

 // Does addition info exist?
 bsi->addbsie = ac3dec_getbits(bs,1);

 if(bsi->addbsie){
  // Get how much info is there
  bsi->addbsil = ac3dec_getbits(bs,6);

  // Get the additional info
  for(i=0;i<(bsi->addbsil + 1);i++)
   bsi->addbsi[i] = ac3dec_getbits(bs,8);
 }
}

// More pain inducing parsing
unsigned int ac3dec_parse_audblk(bsi_t *bsi,audblk_t *audblk,bitstream_t *bs)
{
 int i,j;

 for (i=0;i < bsi->nfchans; i++){
  // Is this channel an interleaved 256 + 256 block ?
  audblk->blksw[i] = ac3dec_getbits(bs,1);
 }

 for (i=0;i < bsi->nfchans; i++){
  // Should we dither this channel?
  audblk->dithflag[i] = ac3dec_getbits(bs,1);
 }

 // Does dynamic range control exist?
 audblk->dynrnge = ac3dec_getbits(bs,1);
 if (audblk->dynrnge){
  // Get dynamic range info
  audblk->dynrng = ac3dec_getbits(bs,8);
 }

 // If we're in dual mono mode then get the second channel DR info
 if (bsi->acmod == 0){
  // Does dynamic range control two exist?
  audblk->dynrng2e = ac3dec_getbits(bs,1);
  if (audblk->dynrng2e){
   // Get dynamic range info
   audblk->dynrng2 = ac3dec_getbits(bs,8);
  }
 }

 // Does coupling strategy exist?
 audblk->cplstre = ac3dec_getbits(bs,1);
 if (audblk->cplstre){
  // Is coupling turned on?
  audblk->cplinu = ac3dec_getbits(bs,1);
  if(audblk->cplinu){
   for(i=0;i < bsi->nfchans; i++)
    audblk->chincpl[i] = ac3dec_getbits(bs,1);
   if(bsi->acmod == 0x2)
    audblk->phsflginu = ac3dec_getbits(bs,1);
   audblk->cplbegf = ac3dec_getbits(bs,4);
   audblk->cplendf = ac3dec_getbits(bs,4);
   audblk->ncplsubnd = (audblk->cplendf + 2) - audblk->cplbegf + 1;

   // Calculate the start and end bins of the coupling channel
   audblk->cplstrtmant = (audblk->cplbegf * 12) + 37 ;
   audblk->cplendmant =  ((audblk->cplendf + 3) * 12) + 37;

   // The number of combined subbands is ncplsubnd minus each combined band
   audblk->ncplbnd = audblk->ncplsubnd;

   for(i=1; i< audblk->ncplsubnd; i++){
    audblk->cplbndstrc[i] = ac3dec_getbits(bs,1);
    audblk->ncplbnd -= audblk->cplbndstrc[i];
   }
  }
 }

 if(audblk->cplinu){
  // Loop through all the channels and get their coupling co-ords
  for(i=0;i < bsi->nfchans;i++){
   if(!audblk->chincpl[i])
    continue;

  // Is there new coupling co-ordinate info?
   audblk->cplcoe[i] = ac3dec_getbits(bs,1);

   if(audblk->cplcoe[i]){
    audblk->mstrcplco[i] = ac3dec_getbits(bs,2);
    for(j=0;j < audblk->ncplbnd; j++){
     audblk->cplcoexp[i][j] = ac3dec_getbits(bs,4);
     audblk->cplcomant[i][j] = ac3dec_getbits(bs,4);
    }
   }
  }

 // If we're in dual mono mode, there's going to be some phase info
  if((bsi->acmod==0x2) && audblk->phsflginu && (audblk->cplcoe[0] || audblk->cplcoe[1])){
   for(j=0;j < audblk->ncplbnd; j++)
    audblk->phsflg[j] = ac3dec_getbits(bs,1);

  }
 }

 // If we're in dual mono mode, there may be a rematrix strategy
 if(bsi->acmod == 0x2){
  audblk->rematstr = ac3dec_getbits(bs,1);
  if(audblk->rematstr){
   if (audblk->cplinu == 0){
    for(i = 0; i < 4; i++)
     audblk->rematflg[i] = ac3dec_getbits(bs,1);
   }
   if((audblk->cplbegf > 2) && audblk->cplinu){
    for(i = 0; i < 4; i++)
     audblk->rematflg[i] = ac3dec_getbits(bs,1);
   }
   if((audblk->cplbegf <= 2) && audblk->cplinu){
    for(i = 0; i < 3; i++)
     audblk->rematflg[i] = ac3dec_getbits(bs,1);
   }
   if((audblk->cplbegf == 0) && audblk->cplinu)
    for(i = 0; i < 2; i++)
     audblk->rematflg[i] = ac3dec_getbits(bs,1);
  }
 }

 if (audblk->cplinu){
  // Get the coupling channel exponent strategy
  audblk->cplexpstr = ac3dec_getbits(bs,2);
  audblk->ncplgrps = (audblk->cplendmant - audblk->cplstrtmant) / (3 << (audblk->cplexpstr-1));
 }

 for(i = 0; i < bsi->nfchans; i++)
  audblk->chexpstr[i] = ac3dec_getbits(bs,2);

 // Get the exponent strategy for lfe channel
 if(bsi->lfeon)
  audblk->lfeexpstr = ac3dec_getbits(bs,1);

 // Determine the bandwidths of all the fbw channels
 for(i = 0; i < bsi->nfchans; i++){
  uint_16 grp_size;

  if(audblk->chexpstr[i] != EXP_REUSE){
   if (audblk->cplinu && audblk->chincpl[i]){
    audblk->endmant[i] = audblk->cplstrtmant;
   }else{
    audblk->chbwcod[i] = ac3dec_getbits(bs,6);
    audblk->endmant[i] = ((audblk->chbwcod[i] + 12) * 3) + 37;
   }

   // Calculate the number of exponent groups to fetch
   grp_size =  3 * (1 << (audblk->chexpstr[i] - 1));
   audblk->nchgrps[i] = (audblk->endmant[i] - 1 + (grp_size - 3)) / grp_size;
  }
 }

 // Get the coupling exponents if they exist
 if(audblk->cplinu && (audblk->cplexpstr != EXP_REUSE)){
  audblk->cplabsexp = ac3dec_getbits(bs,4);
  for(i=0;i< audblk->ncplgrps;i++)
   audblk->cplexps[i] = ac3dec_getbits(bs,7);
 }

 // Get the fwb channel exponents
 for(i=0;i < bsi->nfchans; i++){
  if(audblk->chexpstr[i] != EXP_REUSE){
   audblk->exps[i][0] = ac3dec_getbits(bs,4);
   for(j=1;j<=audblk->nchgrps[i];j++)
    audblk->exps[i][j] = ac3dec_getbits(bs,7);
   audblk->gainrng[i] = ac3dec_getbits(bs,2);
  }
 }

 // Get the lfe channel exponents
 if(bsi->lfeon && (audblk->lfeexpstr != EXP_REUSE)){
  audblk->lfeexps[0] = ac3dec_getbits(bs,4);
  audblk->lfeexps[1] = ac3dec_getbits(bs,7);
  audblk->lfeexps[2] = ac3dec_getbits(bs,7);
 }

 // Get the parametric bit allocation parameters
 audblk->baie = ac3dec_getbits(bs,1);

 if(audblk->baie){
  audblk->sdcycod = ac3dec_getbits(bs,2);
  audblk->fdcycod = ac3dec_getbits(bs,2);
  audblk->sgaincod = ac3dec_getbits(bs,2);
  audblk->dbpbcod = ac3dec_getbits(bs,2);
  audblk->floorcod = ac3dec_getbits(bs,3);
 }

 // Get the SNR off set info if it exists
 audblk->snroffste = ac3dec_getbits(bs,1);

 if(audblk->snroffste){
  audblk->csnroffst = ac3dec_getbits(bs,6);

  if(audblk->cplinu){
   audblk->cplfsnroffst = ac3dec_getbits(bs,4);
   audblk->cplfgaincod = ac3dec_getbits(bs,3);
  }

  for(i = 0;i < bsi->nfchans; i++){
   audblk->fsnroffst[i] = ac3dec_getbits(bs,4);
   audblk->fgaincod[i] = ac3dec_getbits(bs,3);
  }
  if(bsi->lfeon){
   audblk->lfefsnroffst = ac3dec_getbits(bs,4);
   audblk->lfefgaincod = ac3dec_getbits(bs,3);
  }
 }

 // Get coupling leakage info if it exists
 if(audblk->cplinu){
  audblk->cplleake = ac3dec_getbits(bs,1);

  if(audblk->cplleake){
   audblk->cplfleak = ac3dec_getbits(bs,3);
   audblk->cplsleak = ac3dec_getbits(bs,3);
  }
 }

 // Get the delta bit alloaction info
 audblk->deltbaie = ac3dec_getbits(bs,1);

 if(audblk->deltbaie){
  if(audblk->cplinu)
   audblk->cpldeltbae = ac3dec_getbits(bs,2);

  for(i = 0;i < bsi->nfchans; i++)
   audblk->deltbae[i] = ac3dec_getbits(bs,2);

  if (audblk->cplinu && (audblk->cpldeltbae == DELTA_BIT_NEW)){
   audblk->cpldeltnseg = ac3dec_getbits(bs,3);
   for(i = 0;i < audblk->cpldeltnseg + 1; i++){
    audblk->cpldeltoffst[i] = ac3dec_getbits(bs,5);
    audblk->cpldeltlen[i] = ac3dec_getbits(bs,4);
    audblk->cpldeltba[i] = ac3dec_getbits(bs,3);
   }
  }

  for(i = 0;i < bsi->nfchans; i++){
   if (audblk->deltbae[i] == DELTA_BIT_NEW){
    audblk->deltnseg[i] = ac3dec_getbits(bs,3);
    for(j = 0; j < audblk->deltnseg[i] + 1; j++){
     audblk->deltoffst[i][j] = ac3dec_getbits(bs,5);
     audblk->deltlen[i][j] = ac3dec_getbits(bs,4);
     audblk->deltba[i][j] = ac3dec_getbits(bs,3);
    }
   }
  }
 }

 // Check to see if there's any dummy info to get
 if((audblk->skiple =  ac3dec_getbits(bs,1))){
  uint_16 skip_data;

  audblk->skipl = ac3dec_getbits(bs,9);

  for(i = 0; i < audblk->skipl ; i++)
   skip_data = ac3dec_getbits(bs,8);
 }

 //check variables

 if((audblk->cplinu == 1) && (audblk->cplbegf > (audblk->cplendf+2)))
  return 0;

 for(i=0; i < bsi->nfchans; i++)
  if((audblk->chincpl[i] == 0) && (audblk->chbwcod[i] > 60))
   return 0;

 return 1;
}
