//**************************************************************************
//*                     This file is part of the                           *
//*                      Mpxplay - audio player.                           *
//*                  The source code of Mpxplay is                         *
//*        (C) copyright 1998-2003 by PDSoft (Attila Padar)                *
//*                    http://mpxplay.cjb.net                              *
//*                  email: mpxplay@freemail.hu                            *
//**************************************************************************
//*  This program is distributed in the hope that it will be useful,       *
//*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
//*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                  *
//*  Please contact with the author (with me) if you want to use           *
//*  or modify this source.                                                *
//**************************************************************************
//function: AC3 imdct
//
//based on the ac3dec by Aaron Holtzman

#include <math.h>
#include <string.h>
#include "ac3.h"

#define AC3_WINDOW_CALCULATE 1
//#define AC3_WINDOW_OLD 1

#if defined(AC3_WINDOW_CALCULATE) && defined(__WATCOMC__)
 #define AC3_WINCALC_ASM 1
#endif

#if defined(AC3_WINDOW_CALCULATE) || !defined(AC3_WINDOW_OLD)
 #define AC3_WINDOW_T double
#else
 #define AC3_WINDOW_T float
#endif


//#define AC3_WINDOW_T double

#define N  512

typedef struct complex_s
{
 AC3_DOUBLE_T real;
 AC3_DOUBLE_T imag;
}complex_t;

static uint_8 bit_reverse_512[128] = {
 0x00, 0x40, 0x20, 0x60, 0x10, 0x50, 0x30, 0x70,
 0x08, 0x48, 0x28, 0x68, 0x18, 0x58, 0x38, 0x78,
 0x04, 0x44, 0x24, 0x64, 0x14, 0x54, 0x34, 0x74,
 0x0c, 0x4c, 0x2c, 0x6c, 0x1c, 0x5c, 0x3c, 0x7c,
 0x02, 0x42, 0x22, 0x62, 0x12, 0x52, 0x32, 0x72,
 0x0a, 0x4a, 0x2a, 0x6a, 0x1a, 0x5a, 0x3a, 0x7a,
 0x06, 0x46, 0x26, 0x66, 0x16, 0x56, 0x36, 0x76,
 0x0e, 0x4e, 0x2e, 0x6e, 0x1e, 0x5e, 0x3e, 0x7e,
 0x01, 0x41, 0x21, 0x61, 0x11, 0x51, 0x31, 0x71,
 0x09, 0x49, 0x29, 0x69, 0x19, 0x59, 0x39, 0x79,
 0x05, 0x45, 0x25, 0x65, 0x15, 0x55, 0x35, 0x75,
 0x0d, 0x4d, 0x2d, 0x6d, 0x1d, 0x5d, 0x3d, 0x7d,
 0x03, 0x43, 0x23, 0x63, 0x13, 0x53, 0x33, 0x73,
 0x0b, 0x4b, 0x2b, 0x6b, 0x1b, 0x5b, 0x3b, 0x7b,
 0x07, 0x47, 0x27, 0x67, 0x17, 0x57, 0x37, 0x77,
 0x0f, 0x4f, 0x2f, 0x6f, 0x1f, 0x5f, 0x3f, 0x7f
};

static uint_8 bit_reverse_256[64] = {
 0x00, 0x20, 0x10, 0x30, 0x08, 0x28, 0x18, 0x38,
 0x04, 0x24, 0x14, 0x34, 0x0c, 0x2c, 0x1c, 0x3c,
 0x02, 0x22, 0x12, 0x32, 0x0a, 0x2a, 0x1a, 0x3a,
 0x06, 0x26, 0x16, 0x36, 0x0e, 0x2e, 0x1e, 0x3e,
 0x01, 0x21, 0x11, 0x31, 0x09, 0x29, 0x19, 0x39,
 0x05, 0x25, 0x15, 0x35, 0x0d, 0x2d, 0x1d, 0x3d,
 0x03, 0x23, 0x13, 0x33, 0x0b, 0x2b, 0x1b, 0x3b,
 0x07, 0x27, 0x17, 0x37, 0x0f, 0x2f, 0x1f, 0x3f
};

static complex_t buf[128];

// Twiddle factor LUT
static complex_t w_all[1+2+4+8+16+32+64];

// Twiddle factors for IMDCT
static AC3_DOUBLE_T xcos1[128];
static AC3_DOUBLE_T xsin1[128];
static AC3_DOUBLE_T xcos2[64];
static AC3_DOUBLE_T xsin2[64];


#ifdef AC3_WINDOW_CALCULATE

static AC3_WINDOW_T window[256];

#else // constant (!AC3_WINDOW_CALCULATE)

#ifndef AC3_WINDOW_OLD

// Windowing function for Modified DCT (multiplied with 2 to avoid an fmul)
static AC3_WINDOW_T window[] = {
0.000271984434,0.000487801502,0.000733004999,0.001015509130,0.001340074814,
0.001710800338,0.002131642308,0.002606576309,0.003139656037,0.003735040314,
0.004397003446,0.005129939411,0.005938363727,0.006826911587,0.007800335530,
0.008863503113,0.010021390393,0.011279082857,0.012641764246,0.014114714228,
0.015703301877,0.017412977293,0.019249273464,0.021217785776,0.023324174806,
0.025574155152,0.027973486111,0.030527964234,0.033243417740,0.036125693470,
0.039180643857,0.042414128780,0.045831996948,0.049440074712,0.053244162351,
0.057250022888,0.061463363469,0.065889827907,0.070534996688,0.075404360890,
0.080503314734,0.085837155581,0.091411054134,0.097230061889,0.103299081326,
0.109622873366,0.116206027567,0.123052969575,0.130167931318,0.137554958463,
0.145217895508,0.153160363436,0.161385729909,0.169897183776,0.178697630763,
0.187789708376,0.197175830603,0.206858098507,0.216838330030,0.227118089795,
0.237698614597,0.248580843210,0.259765386581,0.271252542734,0.283042311668,
0.295134305954,0.307527869940,0.320221960545,0.333215177059,0.346505820751,
0.360091805458,0.373970717192,0.388139754534,0.402595758438,0.417335301638,
0.432354450226,0.447649061680,0.463214516640,0.479045987129,0.495138138533,
0.511485457420,0.528081893921,0.544921219349,0.561996877193,0.579301834106,
0.596828937531,0.614570617676,0.632518947124,0.650665879250,0.669002890587,
0.687521278858,0.706212103367,0.725066184998,0.744073987007,0.763225793839,
0.782511770725,0.801921784878,0.821445524693,0.841072499752,0.860792040825,
0.880593419075,0.900465726852,0.920397818089,0.940378665924,0.960396945477,
0.980441451073,1.000500679016,1.020563483238,1.040618300438,1.060653686523,
1.080658316612,1.100620746613,1.120529770851,1.140374064445,1.160142421722,
1.179823756218,1.199407100677,1.218881607056,1.238236665726,1.257461667061,
1.276546120644,1.295480132103,1.314253449440,1.332856535912,1.351279735565,
1.369513869286,1.387549877167,1.405379176140,1.422993183136,1.440383672714,
1.457543015480,1.474463701248,1.491138339043,1.507560253143,1.523722887039,
1.539620041847,1.555245995522,1.570595383644,1.585663080215,1.600444555283,
1.614935278893,1.629131555557,1.643029808998,1.656626939774,1.669920325279,
1.682907462120,1.695586562157,1.707955956459,1.720014572144,1.731761574745,
1.743196725845,1.754319787025,1.765131115913,1.775631666183,1.785822391510,
1.795704603195,1.805280327797,1.814551472664,1.823520541191,1.832190394402,
1.840563893318,1.848644495010,1.856435775757,1.863941788673,1.871166586876,
1.878114581108,1.884790539742,1.891199111938,1.897345423698,1.903234839439,
1.908872723579,1.914264678955,1.919416427612,1.924333930016,1.929022908211,
1.933489799500,1.937740445137,1.941781401634,1.945618748665,1.949258923531,
1.952708244324,1.955973267555,1.959060192108,1.961975455284,1.964725375175,
1.967316389084,1.969754695892,1.972046613693,1.974197983742,1.976215243340,
1.978104114532,1.979870438576,1.981520175934,1.983058691025,1.984491705894,
1.985824584961,1.987062454224,1.988210558891,1.989273667336,1.990256667137,
1.991164207458,1.992000818253,1.992770671844,1.993478059769,1.994126796722,
1.994720935822,1.995264053345,1.995759606361,1.996210932732,1.996621131897,
1.996993422508,1.997330546379,1.997635126114,1.997909903526,1.998157143593,
1.998379111290,1.998578071594,1.998755812645,1.998914361000,1.999055385590,
1.999180436134,1.999291181564,1.999388813972,1.999474763870,1.999550223351,
1.999616146088,1.999673724174,1.999723672867,1.999766945839,1.999804377556,
1.999836444855,1.999863982201,1.999887466431,1.999907374382,1.999924182892,
1.999938368797,1.999950170517,1.999960064888,1.999968171120,1.999974846840,
1.999980330467,1.999984741211,1.999988317490,1.999991178513,1.999993443489,
1.999995112419,1.999996542931,1.999997496605,1.999998331070,1.999998807907,
1.999999284744,1.999999523163,1.999999761581,1.999999880791,2.000000000000,
2.000000000000 };

#else // AC3_WINDOW_OLD

static AC3_WINDOW_T window[] = {
 0.00014, 0.00024, 0.00037, 0.00051, 0.00067, 0.00086, 0.00107, 0.00130,
 0.00157, 0.00187, 0.00220, 0.00256, 0.00297, 0.00341, 0.00390, 0.00443,
 0.00501, 0.00564, 0.00632, 0.00706, 0.00785, 0.00871, 0.00962, 0.01061,
 0.01166, 0.01279, 0.01399, 0.01526, 0.01662, 0.01806, 0.01959, 0.02121,
 0.02292, 0.02472, 0.02662, 0.02863, 0.03073, 0.03294, 0.03527, 0.03770,
 0.04025, 0.04292, 0.04571, 0.04862, 0.05165, 0.05481, 0.05810, 0.06153,
 0.06508, 0.06878, 0.07261, 0.07658, 0.08069, 0.08495, 0.08935, 0.09389,
 0.09859, 0.10343, 0.10842, 0.11356, 0.11885, 0.12429, 0.12988, 0.13563,
 0.14152, 0.14757, 0.15376, 0.16011, 0.16661, 0.17325, 0.18005, 0.18699,
 0.19407, 0.20130, 0.20867, 0.21618, 0.22382, 0.23161, 0.23952, 0.24757,
 0.25574, 0.26404, 0.27246, 0.28100, 0.28965, 0.29841, 0.30729, 0.31626,
 0.32533, 0.33450, 0.34376, 0.35311, 0.36253, 0.37204, 0.38161, 0.39126,
 0.40096, 0.41072, 0.42054, 0.43040, 0.44030, 0.45023, 0.46020, 0.47019,
 0.48020, 0.49022, 0.50025, 0.51028, 0.52031, 0.53033, 0.54033, 0.55031,
 0.56026, 0.57019, 0.58007, 0.58991, 0.59970, 0.60944, 0.61912, 0.62873,
 0.63827, 0.64774, 0.65713, 0.66643, 0.67564, 0.68476, 0.69377, 0.70269,
 0.71150, 0.72019, 0.72877, 0.73723, 0.74557, 0.75378, 0.76186, 0.76981,
 0.77762, 0.78530, 0.79283, 0.80022, 0.80747, 0.81457, 0.82151, 0.82831,
 0.83496, 0.84145, 0.84779, 0.85398, 0.86001, 0.86588, 0.87160, 0.87716,
 0.88257, 0.88782, 0.89291, 0.89785, 0.90264, 0.90728, 0.91176, 0.91610,
 0.92028, 0.92432, 0.92822, 0.93197, 0.93558, 0.93906, 0.94240, 0.94560,
 0.94867, 0.95162, 0.95444, 0.95713, 0.95971, 0.96217, 0.96451, 0.96674,
 0.96887, 0.97089, 0.97281, 0.97463, 0.97635, 0.97799, 0.97953, 0.98099,
 0.98236, 0.98366, 0.98488, 0.98602, 0.98710, 0.98811, 0.98905, 0.98994,
 0.99076, 0.99153, 0.99225, 0.99291, 0.99353, 0.99411, 0.99464, 0.99513,
 0.99558, 0.99600, 0.99639, 0.99674, 0.99706, 0.99736, 0.99763, 0.99788,
 0.99811, 0.99831, 0.99850, 0.99867, 0.99882, 0.99895, 0.99908, 0.99919,
 0.99929, 0.99938, 0.99946, 0.99953, 0.99959, 0.99965, 0.99969, 0.99974,
 0.99978, 0.99981, 0.99984, 0.99986, 0.99988, 0.99990, 0.99992, 0.99993,
 0.99994, 0.99995, 0.99996, 0.99997, 0.99998, 0.99998, 0.99998, 0.99999,
 0.99999, 0.99999, 0.99999, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000,
 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000, 1.00000 };

#endif // AC3_WINDOW_OLD

#endif // AC3_WINDOW_CALCULATE

void asm_winmake(void);

void ac3dec_imdct_init()
{
 int i, k;
 complex_t *wp;
 AC3_DOUBLE_T *cp,*sp;
#ifdef AC3_WINDOW_CALCULATE
#ifndef AC3_WINCALC_ASM
 AC3_WINDOW_T *winptr;
 double sum;
#else
 int tmp;
#endif

 // compute imdct window - kaiser-bessel derived window, alpha = 5.0
#ifdef AC3_WINCALC_ASM

#pragma aux asm_winmake=\
 "mov dword ptr tmp,2"\
 "fild dword ptr tmp"\
 "mov dword ptr tmp,5"\
 "fldpi"\
 "fimul dword ptr tmp"\
 "mov dword ptr tmp,256"\
 "fidiv dword ptr tmp"\
 "fld st"\
 "fmul"\
 "mov edi,offset window"\
 "fldz"\
 "mov ebx,256"\
 "back1:"\
  "mov eax,256"\
  "sub eax,ebx"\
  "imul eax,ebx"\
  "mov dword ptr tmp,eax"\
  "fild dword ptr tmp"\
  "fmul st(0),st(2)"\
  "fld1"\
  "mov ecx,100"\
  "back2:"\
   "fmul st,st(1)"\
   "mov eax,ecx"\
   "imul eax,ecx"\
   "mov dword ptr tmp,eax"\
   "fidiv dword ptr tmp"\
   "fld1"\
   "fadd"\
   "dec ecx"\
  "jnz back2"\
  "faddp st(2),st"\
  "fstp st(0)"\
  "fst qword ptr [edi]"\
  "add edi,8"\
  "dec ebx"\
 "jnz back1"\
 "fld1"\
 "fadd"\
 "mov edi,offset window"\
 "mov ebx,256"\
 "back3:"\
  "fld qword ptr [edi]"\
  "fdiv st,st(1)"\
  "fsqrt"\
  "fmul st,st(3)"\
  "fstp qword ptr [edi]"\
  "add edi,8"\
  "dec ebx"\
 "jnz back3"\
 "fstp st(0)"\
 "fstp st(0)"\
 modify[eax ebx ecx edi];
 asm_winmake();

#else // !AC3_WINCALC_ASM

 winptr=&window[0];
 sum = 0;
 i=256;
 do{
  double x= i * (256 - i) * (5 * M_PI / 256) * (5 * M_PI / 256);
  double bessel=1.0;
  int j=100;
  do{
   bessel = bessel * x / (j * j) + 1.0;
  }while(--j);
  sum+=bessel;
  *winptr++ = sum;
 }while(--i);
 sum+=1.0;
 winptr=&window[0];
 i=256;
 do{
  *winptr = sqrt (*winptr / sum) * 2.0f; // * 2.0 -> to avoid an fmul
  winptr++;
 }while(--i);
#endif // AC3_WINCALC_ASM
#endif // AC3_WINDOW_CALCULATE

 // Twiddle factors to turn IFFT into IMDCT
 cp=&xcos1[0];
 sp=&xsin1[0];
 for(i=0;i<128;i++){
  *cp++ = cos(2.0 * M_PI * (8*i+1)/(8*N));
  *sp++ = sin(2.0 * M_PI * (8*i+1)/(8*N));
 }

 // More twiddle factors to turn IFFT into IMDCT
 cp=&xcos2[0];
 sp=&xsin2[0];
 for(i=0;i<64;i++){
  *cp++ = cos(2.0 * M_PI * (8*i+1)/(4*N));
  *sp++ = sin(2.0 * M_PI * (8*i+1)/(4*N));
 }

 // Canonical twiddle factors for FFT
 wp=&w_all[0];
 for(i=0;i<7;i++){
  double step_real = cos(-2.0 * M_PI / (1 << (i+1)));
  double step_imag = sin(-2.0 * M_PI / (1 << (i+1)));
  double current_real=1.0;
  double current_imag=0.0;

  for(k=0;k<(1<<i);k++){
   wp->real=(AC3_DOUBLE_T)current_real;
   wp->imag=(AC3_DOUBLE_T)current_imag;
   wp++;
   {
    double tmp=current_real*step_real - current_imag*step_imag;
    current_imag=current_real*step_imag + current_imag*step_real;
    current_real=tmp;
   }
  }
 }
#if defined(AC3_WINDOW_OLD) && !defined(AC3_WINDOW_CALCULATE)
 // to save an fmul
 {
 AC3_WINDOW_T *winptr=window;
 i=256;
 do{
  *winptr++ *= 2.0f;
 }while(--i);
 }
#endif
}

//-------------------------------------------------------------------------

static void imdct_do_512(AC3_DOUBLE_T data[], AC3_DOUBLE_T delay[])
{
 complex_t *bufp,*wp;
 int k,two_m,two_m_plus_one;

 // Pre IFFT complex multiply plus IFFT cmplx conjugate and bit reverse permutation
 {
  complex_t *bufp=&buf[0];
  uint_8 *bitrevp=&bit_reverse_512[0];
  AC3_DOUBLE_T *data_ptr=data;
  AC3_DOUBLE_T *dataptr2=data_ptr+255;
  AC3_DOUBLE_T *cosp=&xcos1[0];
  AC3_DOUBLE_T *sinp=&xsin1[0];
  unsigned int i=128;
  do{
   complex_t *bufpp= bufp + *bitrevp++;
   bufpp->real =   dataptr2[0] * cosp[0] - data_ptr[0] * sinp[0];
   bufpp->imag = - data_ptr[0] * cosp[0] - dataptr2[0] * sinp[0];
   data_ptr+=2;
   dataptr2-=2;
   cosp++;
   sinp++;
  }while(--i);
 }

 // FFT Merge
 {
  wp=&w_all[0];
  two_m=1;
  do{
   two_m_plus_one = two_m<<1;
   bufp=&buf[0];
   k=two_m;
   do{
    int i;
    complex_t *bufpp=bufp++,*bufpq=bufpp+two_m;
    const AC3_DOUBLE_T wmkr=wp->real;
    const AC3_DOUBLE_T wmki=wp->imag;
    wp++;

    i=128;
    do{
     const AC3_DOUBLE_T tmp_b_i = bufpq->imag * wmkr + bufpq->real * wmki;
     const AC3_DOUBLE_T tmp_b_r = bufpq->real * wmkr - bufpq->imag * wmki;

     bufpq->real = bufpp->real - tmp_b_r;
     bufpp->real+= tmp_b_r;
     bufpq->imag = bufpp->imag - tmp_b_i;
     bufpp->imag+= tmp_b_i;

     bufpp+=two_m_plus_one;
     bufpq+=two_m_plus_one;
     i-=two_m_plus_one;
    }while(i);
   }while(--k);
   two_m=two_m_plus_one;
  }while(two_m<=64);
 }

 // Post IFFT complex multiply  plus IFFT complex conjugate
 {
  complex_t *bufp=&buf[0];
  AC3_DOUBLE_T *cosp=&xcos1[0];
  AC3_DOUBLE_T *sinp=&xsin1[0];
  unsigned int i=128;
  do{
   const AC3_DOUBLE_T tmp_a_i_s = bufp->imag * sinp[0];
   const AC3_DOUBLE_T tmp_a_i_c = bufp->imag * cosp[0];

   bufp->imag = bufp->real * *sinp++ - tmp_a_i_c;
   bufp->real = bufp->real * *cosp++ + tmp_a_i_s;
   bufp++;
  }while(--i);
 }

 // Window and convert to real valued signal
 // The trailing edge of the window goes into the delay line
 {
  complex_t *bufp0=(&buf[0])+64,*bufp1=bufp0-1;
  AC3_DOUBLE_T *data_ptr = data;
  AC3_DOUBLE_T *delay_ptr = delay;
  AC3_WINDOW_T *window_ptr = window;
  AC3_WINDOW_T *winptrb=window_ptr+256;
  unsigned int i;
  i=64;
  do{
   winptrb-=2;
   data_ptr[0]  = (-bufp0->imag * window_ptr[0] + delay_ptr[0]);
   delay_ptr[0] =  -bufp0->real * winptrb[1];    // 64 - 127
   bufp0++;

   data_ptr[1]  = ( bufp1->real * window_ptr[1] + delay_ptr[1]);
   data_ptr+=2;
   window_ptr+=2;
   delay_ptr[1] =   bufp1->imag * winptrb[0];    // 63 - 0
   bufp1--;
   delay_ptr+=2;
  }while(--i);

  bufp0=&buf[0];
  bufp1=bufp0+127;
  i=64;
  do{
   winptrb-=2;
   data_ptr[0]  = (-bufp0->real     * window_ptr[0] + delay_ptr[0]);
   delay_ptr[0] =   bufp0->imag     * winptrb[1];    // 0 - 63
   bufp0++;

   data_ptr[1]  = ( bufp1->imag * window_ptr[1] + delay_ptr[1]);
   data_ptr+=2;
   window_ptr+=2;
   delay_ptr[1] =  -bufp1->real * winptrb[0];    // 127 - 64
   bufp1--;
   delay_ptr+=2;
  }while(--i);
 }
}

static void imdct_do_256(AC3_DOUBLE_T data[], AC3_DOUBLE_T delay[])
{
 complex_t *bufp,*wp;
 int k,two_m,two_m_plus_one;

 // Pre IFFT complex multiply plus IFFT cmplx conjugate and bit reverse permutation
 {
  complex_t *bufp=&buf[0];
  uint_8 *bitrevp=&bit_reverse_256[0];
  AC3_DOUBLE_T *data_ptr=data;
  AC3_DOUBLE_T *dataptr2=data_ptr+254;
  AC3_DOUBLE_T *cosp=&xcos2[0];
  AC3_DOUBLE_T *sinp=&xsin2[0];
  unsigned int i=64;
  do{
   complex_t *bufpp= bufp + *bitrevp++;
   bufpp->real =   dataptr2[0] * cosp[0] - data_ptr[0] * sinp[0];
   bufpp->imag = - data_ptr[0] * cosp[0] - dataptr2[0] * sinp[0];
   (bufpp+64)->real =   dataptr2[1] * cosp[0] - data_ptr[1] * sinp[0];
   (bufpp+64)->imag = - data_ptr[1] * cosp[0] - dataptr2[1] * sinp[0];
   data_ptr+=4;
   dataptr2-=4;
   cosp++;
   sinp++;
  }while(--i);
 }

 // FFT Merge
 {
  bufp=&buf[0];
  wp=&w_all[0];

  two_m=1;
  do{
   two_m_plus_one=two_m<<1;
   bufp=&buf[0];
   k=two_m;
   do{
    int i;
    complex_t *bufpp=bufp++,*bufpq=bufpp+two_m;
    const AC3_DOUBLE_T wmkr=wp->real;
    const AC3_DOUBLE_T wmki=wp->imag;
    wp++;

    i=64;
    do{
     {
      const AC3_DOUBLE_T tmp_b_i = bufpq->imag * wmkr + bufpq->real * wmki;
      const AC3_DOUBLE_T tmp_b_r = bufpq->real * wmkr - bufpq->imag * wmki;

      bufpq->real = bufpp->real - tmp_b_r;
      bufpp->real+= tmp_b_r;
      bufpq->imag = bufpp->imag - tmp_b_i;
      bufpp->imag+= tmp_b_i;
     }
     bufpp+=64;
     bufpq+=64;
     {
      const AC3_DOUBLE_T tmp_b_i = bufpq->imag * wmkr + bufpq->real * wmki;
      const AC3_DOUBLE_T tmp_b_r = bufpq->real * wmkr - bufpq->imag * wmki;

      bufpq->real = bufpp->real - tmp_b_r;
      bufpp->real+= tmp_b_r;
      bufpq->imag = bufpp->imag - tmp_b_i;
      bufpp->imag+= tmp_b_i;
     }
     bufpp-=64;
     bufpq-=64;
     bufpp+=two_m_plus_one;
     bufpq+=two_m_plus_one;
     i-=two_m_plus_one;
    }while(i);
   }while(--k);
   two_m=two_m_plus_one;
  }while(two_m<=32);
 }

 // Post IFFT complex multiply
 {
  complex_t *bufp=&buf[0];
  AC3_DOUBLE_T *cosp=&xcos2[0];
  AC3_DOUBLE_T *sinp=&xsin2[0];
  unsigned int i=64;
  do{
   {
    const AC3_DOUBLE_T tmp_a_i_s = bufp->imag * sinp[0];
    const AC3_DOUBLE_T tmp_a_i_c = bufp->imag * cosp[0];

    bufp->imag = bufp->real * sinp[0] - tmp_a_i_c;
    bufp->real = bufp->real * cosp[0] + tmp_a_i_s;
   }
   bufp+=64;
   {
    const AC3_DOUBLE_T tmp_a_i_s = bufp->imag * sinp[0];
    const AC3_DOUBLE_T tmp_a_i_c = bufp->imag * cosp[0];

    bufp->imag = bufp->real * sinp[0] - tmp_a_i_c;
    bufp->real = bufp->real * cosp[0] + tmp_a_i_s;
   }
   sinp++;
   cosp++;
   bufp-=63;
  }while(--i);
 }

 {
  complex_t *bufp0=&buf[0],*bufp1=bufp0+63;
  AC3_DOUBLE_T *data_ptr = data;
  AC3_DOUBLE_T *delay_ptr = delay;
  AC3_WINDOW_T *window_ptr = window;
  AC3_WINDOW_T *winptrb=window_ptr+256;
  unsigned int i;
  i=64;
  do{
   winptrb-=2;
   data_ptr[0]  = (-bufp0->imag * window_ptr[0] + delay_ptr[0]);
   delay_ptr[0] =  -(bufp0+64)->real * winptrb[1];
   bufp0++;

   data_ptr[1]  = ( bufp1->real * window_ptr[1] + delay_ptr[1]);
   data_ptr+=2;
   window_ptr+=2;
   delay_ptr[1] =   (bufp1+64)->imag * winptrb[0];
   bufp1--;
   delay_ptr+=2;
  }while(--i);

  bufp0=&buf[0];
  bufp1=bufp0+63;
  i=64;
  do{
   winptrb-=2;
   data_ptr[0]  = (-bufp0->real     * window_ptr[0] + delay_ptr[0]);
   delay_ptr[0] = (bufp0+64)->imag  * winptrb[1];
   bufp0++;

   data_ptr[1]  = ( bufp1->imag * window_ptr[1] + delay_ptr[1]);
   data_ptr+=2;
   window_ptr+=2;
   delay_ptr[1] = -(bufp1+64)->real * winptrb[0];
   bufp1--;
   delay_ptr+=2;
  }while(--i);
 }
}

#ifdef AC3_SPECTRUM_ANALISER // defined in ac3.h

extern unsigned int displaymode,analtabnum;
extern unsigned long analtab[5][32];

static unsigned long lasts[32],currs[32];
static float scale=4000000.0f;

void ac3_analiser_clear(void)
{
 memset(&lasts[0],0,32*sizeof(unsigned long));
 memset(&currs[0],0,32*sizeof(unsigned long));
}

AC3_DOUBLE_T *asm_band_add(unsigned int,AC3_DOUBLE_T *);

static void ac3_calculate_analiser_bands(AC3_DOUBLE_T *pcm,unsigned int currbs,unsigned int lastch)
{
 unsigned int nbs,bands=32;
 unsigned long anl,*ap,*lp,*cp=&currs[0];

 nbs=currbs>>5;  // /32
 if(!nbs)
  return;

 if(lastch){
  ap=&analtab[analtabnum][0];
  lp=&lasts[0];
 }

 do{
#ifdef OGG_USE_ASM
 #ifdef __WATCOMC__
 #pragma aux asm_band_add=\
  "fldz"\
  "back:"\
   "fld dword ptr [esi]"\
   "add esi,4"\
   "fabs"\
   "dec edx"\
   "fadd"\
  "jnz back"\
  "fmul dword ptr scale"\
  "fistp dword ptr anl"\
  parm[edx][esi] value[esi];
  pcm=asm_band_add(nbs,pcm);
 #endif
#else
  float anf=0.0f;
  unsigned int k=nbs;
  do{
   anf+=fabs(*pcm++);
  }while(--k);
  anl=(unsigned long)(anf*scale);
#endif

  *cp += anl;

  if(lastch){
   unsigned long lnl=*lp;
   anl=*cp;
   *lp=anl;

   anl=(anl+lnl)>>1;

   if(bands<32){
    unsigned long ap1=*(ap-1);  // create fake high bands (for a nicer displaying)
    if(anl<100 && anl<(ap1>>1))
     anl= ap1>>1;
   }
   if(*ap)
    *ap = (*ap + anl)>>1;
   else
    *ap = anl;
   ap++;
   *cp=0;
   lp++;
  }

  cp++;

 }while(--bands);
}

#endif

//static unsigned char d2c[8][2]={{0,0},{0,0},{0,0},{0,0},{2,2},{3,3},{2,3},{3,4}};

void ac3dec_imdct_decode(bsi_t *bsi,audblk_t *audblk, stream_samples_t samples,mdct_delaybuf_t mdct_delaybuf,ac3_decoder_data *amip)
{
 int i;

/*#ifdef AC3_LINK_DOWNMIX_2
 if((amip->downmix_select==AC3_DOWNMIX_TYPE_DOLBY2) && (bsi->acmod>=4)){
  for(i=d2c[bsi->acmod][0];i<=d2c[bsi->acmod][1];i++){
   memset(samples[i],0,64*sizeof(AC3_DOUBLE_T));
#ifdef AC3_SPECTRUM_ANALISER
   ac3_calculate_analiser_bands(samples[i],256,(i==d2c[bsi->acmod][1]));
#endif
  }
 }
#endif*/

 for(i=0;i<bsi->nfchans;i++){
/*#ifdef AC3_SPECTRUM_ANALISER
  ac3_calculate_analiser_bands(samples[i],256,(i==(bsi->nfchans-1)));
#endif*/
  if(audblk->blksw[i])
   imdct_do_256(samples[i],&mdct_delaybuf[i][0]);
  else
   imdct_do_512(samples[i],&mdct_delaybuf[i][0]);
 }

 if(bsi->lfeon)
  imdct_do_512(samples[5],&mdct_delaybuf[5][0]);
}

void ac3dec_imdct_reset(ac3_decoder_data *amip)
{
 memset(amip->mdct_delaybuf,0,sizeof(mdct_delaybuf_t));
}
