/*
 *	Copyright (C) Aaron Holtzman - May 1999
 *
 *  This file is part of ac3dec, a free Dolby AC-3 stream decoder.
 *
 *  ac3dec is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  ac3dec is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

//modifications by : - DVD2AVI
//                   - Mpxplay/PDSoft (Sep 2002,Nov 2004)

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "ac3.h"

#define DOWNMIX_COMPR_INVALID 65536

#define DOLBY_DOWNMIX_1_OLD 1

#define AC3MIXOUT(x) ((AC3_OUT_T)(x))

#define MIXDOWNMUL2 (1.0/(1.0+M_SQRT_2))                   // = 0.5857864
#define MIXDOWNMUL3 (1.0/(1.0+M_SQRT_2+M_SQRT_2))	   // = 0.4142135
#define MIXDOWNMUL4 (1.0/(1.0+M_SQRT_2+M_SQRT_2+M_SQRT_2)) // = 0.3203772

static AC3_DOUBLE_T cmixlev_lut[4]={M_SQRT_2, 0.5956621, 0.500, 0.5956621};
static AC3_DOUBLE_T smixlev_lut[4]={M_SQRT_2, 0.500    , 0.000, 0.500    };

static AC3_DOUBLE_T dynrng_table[256];
static AC3_DOUBLE_T compr_table[256];
static AC3_DOUBLE_T DownMixMuls[3];

typedef struct scaleinfo_s{
 AC3_DOUBLE_T scale_out;
 AC3_DOUBLE_T scale_dr;
 AC3_DOUBLE_T downmixmul1,downmixmul2,downmixmul3;
}scaleinfo_t;

static scaleinfo_t DRC_scale[4]={
 {32768.0, 0.0 , MIXDOWNMUL2,MIXDOWNMUL3,MIXDOWNMUL4}, // no dinamic range control
 {32768.0, 1.0 , MIXDOWNMUL2,MIXDOWNMUL3,MIXDOWNMUL4}, // factory default
 {65536.0, 1.0 , 1.0        ,1.0        ,1.0    },
 {70000.0, 1.1 , 1.0        ,1.0        ,1.0    }      // my tuned values (tuned for/with my AC3s)
};

void ac3dec_downmix_init(ac3_decoder_data *amip)
{
 int i;
 AC3_DOUBLE_T out_scale,dr_scale;

 if(amip->DRC_type>3)
  amip->DRC_type=3;

 out_scale=DRC_scale[amip->DRC_type].scale_out;
 dr_scale=DRC_scale[amip->DRC_type].scale_dr;

 for(i=0; i<128; i++)
  dynrng_table[i]=pow(2.0,(AC3_DOUBLE_T)i/32.0*dr_scale) * out_scale;

 for(i=128; i<256; i++)
  dynrng_table[i]=pow(2.0,(AC3_DOUBLE_T)(i-256)/32.0*dr_scale) * out_scale;

 for(i=0; i<128; i++)
  compr_table[i]= pow(2.0,(AC3_DOUBLE_T)i/16.0*dr_scale) * out_scale;

 for(i=128; i<256; i++)
  compr_table[i]= pow(2.0,(AC3_DOUBLE_T)(i-256)/16.0*dr_scale) * out_scale;

 DownMixMuls[0]=DRC_scale[amip->DRC_type].downmixmul1;
 DownMixMuls[1]=DRC_scale[amip->DRC_type].downmixmul2;
 DownMixMuls[2]=DRC_scale[amip->DRC_type].downmixmul3;
}

void ac3dec_downmix_reset(ac3_decoder_data *amip)
{
 amip->ds_x1=amip->ds_x2=amip->ds_y1=amip->ds_y2=0.0;

#ifdef AC3_LINK_DOWNMIX_2
 if(amip->downmix_select==AC3_DOWNMIX_TYPE_DOLBY2){
  amip->surdelay_counter=0;
  memset(amip->surmixlev_delay,0,sizeof(amip->surmixlev_delay));
  memset(amip->surround_delay,0,sizeof(*(amip->surround_delay)));
 }
#endif
 //amip->prev_compr=DOWNMIX_COMPR_INVALID;
}

//----------------- common ----------------------------------------------

static void downmix_3f_0r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 j;
 const AC3_DOUBLE_T clev = cmixlev_lut[amip->bsi->cmixlev];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[0];

 AC3_DOUBLE_T *centre, *left, *right;
 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];

 j=256;
 do{
  out_samples[0] = AC3MIXOUT((*left++  + clev * *centre  ) * loc_gain);
  out_samples[1] = AC3MIXOUT((*right++ + clev * *centre++) * loc_gain);
  out_samples+=2;
 }while(--j);
}

static void downmix_2f_0r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 j;
 const AC3_DOUBLE_T loc_gain=amip->gain;
 AC3_DOUBLE_T *left, *right;

 left      = samples[0];
 right     = samples[1];

 j=256;
 do{
  out_samples[0] = AC3MIXOUT(*left++  * loc_gain);
  out_samples[1] = AC3MIXOUT(*right++ * loc_gain);
  out_samples+=2;
 }while(--j);
}

static void downmix_1f_0r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 j;
 const AC3_DOUBLE_T loc_gain=amip->gain;
 AC3_DOUBLE_T *centre;

 centre    =samples[0];

 j=256;
 do{
  out_samples[0] =
  out_samples[1] = AC3MIXOUT(M_SQRT_2 * *centre++ * loc_gain);
  out_samples+=2;
 }while(--j);
}

//-------------- Non-Dolby Surround downmixes ----------------------

#ifdef AC3_LINK_DOWNMIX_0

static void downmix_3f_2r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 j;
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T clev = cmixlev_lut[bsi->cmixlev];
 const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[1];

 AC3_DOUBLE_T *centre, *left, *right, *left_sur, *right_sur, *lfe;
 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];
 left_sur  = samples[3];
 right_sur = samples[4];

 if(bsi->lfeon){
  lfe = samples[5];
  j=256;
  do{
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + *centre   + *left_sur++) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + *centre++ + *right_sur++) * loc_gain);
   out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   + slev * *left_sur++) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ + slev * *right_sur++) * loc_gain);
   out_samples+=2;
  }while(--j);
 }else{
  j=256;
  do{
   out_samples[0]=AC3MIXOUT((*left++ + clev * *centre + slev * *left_sur++) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + clev * *centre++ + slev * *right_sur++) * loc_gain);
   out_samples+=2;
  }while(--j);
 }
}

static void downmix_2f_2r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 j;
 const AC3_DOUBLE_T slev = smixlev_lut[amip->bsi->surmixlev];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[0];

 AC3_DOUBLE_T *left, *right, *left_sur, *right_sur;
 left      = samples[0];
 right     = samples[1];
 left_sur  = samples[2];
 right_sur = samples[3];

 j=256;
 do{
  out_samples[0]=AC3MIXOUT((*left++  + slev * *left_sur++ ) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + slev * *right_sur++) * loc_gain);
  out_samples+=2;
 }while(--j);
}

static void downmix_3f_1r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 j;
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T clev = cmixlev_lut[bsi->cmixlev];
 const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[1];
 AC3_DOUBLE_T *centre, *left, *right, *sur;

 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];
 sur       = samples[3];

 j=256;
 do{
  out_samples[0]=AC3MIXOUT((*left++ + clev * *centre + slev * *sur) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ +clev * *centre++ +slev* *sur++) * loc_gain);
  out_samples+=2;
 }while(--j);
}

static void downmix_2f_1r_to_2ch(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 j;
 const AC3_DOUBLE_T slev = smixlev_lut[amip->bsi->surmixlev];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[0];

 AC3_DOUBLE_T *left, *right, *sur;
 left      = samples[0];
 right     = samples[1];
 sur       = samples[2];

 j=256;
 do{
  out_samples[0] = AC3MIXOUT((*left++  + slev * *sur  ) * loc_gain);
  out_samples[1] = AC3MIXOUT((*right++ + slev * *sur++) * loc_gain);
  out_samples+=2;
 }while(--j);
}

#endif // AC3_LINK_DOWNMIX_0

//-------------------- Dolby Surround I downmixes --------------------------

#ifdef AC3_LINK_DOWNMIX_1

#define SURROUND_EFFECT 1.5

static void downmix_3f_2r_to_2ch_dolby(ac3_decoder_data *amip, stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 j;
#ifdef DOLBY_DOWNMIX_1_OLD
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T clev = cmixlev_lut[bsi->cmixlev];
 const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
#endif
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[2];

 AC3_DOUBLE_T *centre, *left, *right, *left_sur, *right_sur, *lfe;
 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];
 left_sur  = samples[3];
 right_sur = samples[4];

 if(amip->bsi->lfeon){
  lfe = samples[5];

  j=256;
  do{
#ifdef DOLBY_DOWNMIX_1_OLD
   //AC3_DOUBLE_T suradd=(*left_sur + *right_sur);
   //AC3_DOUBLE_T surdif=(*left_sur++ - *right_sur++);
   //AC3_DOUBLE_T surval=((suradd+surdif)/2 + (suradd-surdif)/2*SURROUND_EFFECT) * slev;

   const AC3_DOUBLE_T surval = (*left_sur++ + *right_sur++) * slev;
   out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   - surval) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ + surval) * loc_gain);
   //const AC3_DOUBLE_T surval = (*left_sur++ + *right_sur++) * slev;
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   - surval) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ + surval) * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   - *left_sur++ * slev)  * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ + *right_sur++ * slev) * loc_gain);
   //out_samples[0]=AC3MIXOUT(-surval * loc_gain);
   //out_samples[1]=AC3MIXOUT( surval * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left_sur++) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right_sur++) * loc_gain);
#else
   const AC3_DOUBLE_T surval = *left_sur ++ + *right_sur++;
   const AC3_DOUBLE_T ds_sur = (amip->ds_x2 + amip->ds_x1 * 2.0 + surval) * 0.12531781
			      + amip->ds_y1 * 0.77997062 - amip->ds_y2 * 0.28124186;

   amip->ds_x2 = amip->ds_x1;
   amip->ds_x1 = surval;
   amip->ds_y2 = amip->ds_y1;
   amip->ds_y1 = ds_sur;

   out_samples[0]=AC3MIXOUT((*left++  + *lfe   + M_SQRT_2 * (*centre   - ds_sur)) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + M_SQRT_2 * (*centre++ + ds_sur)) * loc_gain);
#endif
   out_samples+=2;
  }while(--j);
 }else{
  j=256;
  do{
#ifdef DOLBY_DOWNMIX_1_OLD
   const AC3_DOUBLE_T surval = (*left_sur++ + *right_sur++) * M_SQRT_2;
   out_samples[0]=AC3MIXOUT((*left++  + *centre   - surval) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + *centre++ + surval) * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left++  + clev * *centre   - slev * *left_sur++)  * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + clev * *centre++ + slev * *right_sur++) * loc_gain);
#else
   const AC3_DOUBLE_T ds_sur = (amip->ds_x2 + amip->ds_x1 * 2.0 + *left_sur + *right_sur) * 0.12531781
			      + amip->ds_y1 * 0.77997062 - amip->ds_y2 * 0.28124186;
   amip->ds_x2 = amip->ds_x1;
   amip->ds_x1 = *left_sur++ + *right_sur++;
   amip->ds_y2 = amip->ds_y1;
   amip->ds_y1 = ds_sur;

   out_samples[0]=AC3MIXOUT((*left++  + M_SQRT_2 * (*centre   - ds_sur)) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + M_SQRT_2 * (*centre++ + ds_sur)) * loc_gain);
#endif
   out_samples+=2;
  }while(--j);
 }
}

static void downmix_2f_2r_to_2ch_dolby(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 j;
//#ifdef DOLBY_DOWNMIX_1_OLD
// bsi_t *bsi=amip->bsi;
// const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
//#endif
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[1];
 AC3_DOUBLE_T *left, *right, *left_sur, *right_sur;

 left      = samples[0];
 right     = samples[1];
 left_sur  = samples[2];
 right_sur = samples[3];

 j=256;
 do{
#ifdef DOLBY_DOWNMIX_1_OLD
  AC3_DOUBLE_T ds_sur= (*left_sur++ + *right_sur++) * M_SQRT_2;
  out_samples[0]=AC3MIXOUT((*left++  - ds_sur) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + ds_sur) * loc_gain);
  //out_samples[0]=AC3MIXOUT((*left++  - *left_sur++  * slev) * loc_gain);
  //out_samples[1]=AC3MIXOUT((*right++ + *right_sur++ * slev) * loc_gain);
#else
  const AC3_DOUBLE_T ds_sur = (amip->ds_x2 + amip->ds_x1 * 2.0 + *left_sur + *right_sur) * 0.12531781
			     + amip->ds_y1 * 0.77997062 - amip->ds_y2 * 0.28124186;
  amip->ds_x2 = amip->ds_x1;
  amip->ds_x1 = *left_sur++ + *right_sur++;
  amip->ds_y2 = amip->ds_y1;
  amip->ds_y1 = ds_sur;

  out_samples[0]=AC3MIXOUT((*left++  - M_SQRT_2 * ds_sur) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + M_SQRT_2 * ds_sur) * loc_gain);
#endif

  out_samples+=2;
 }while(--j);
}

static void downmix_3f_1r_to_2ch_dolby(ac3_decoder_data *amip, stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 j;
//#ifdef DOLBY_DOWNMIX_1_OLD
// bsi_t *bsi=amip->bsi;
// const AC3_DOUBLE_T clev = cmixlev_lut[bsi->cmixlev];
// const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
//#endif
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[1];
 AC3_DOUBLE_T *centre, *left, *right, *sur;

 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];
 sur       = samples[3];

 j=256;
 do{
#ifdef DOLBY_DOWNMIX_1_OLD
  out_samples[0]=AC3MIXOUT((*left++  + *centre   - *sur)   * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + *centre++ + *sur++) * loc_gain);
  //out_samples[0]=AC3MIXOUT((*left++  + clev * *centre   - slev * *sur)   * loc_gain);
  //out_samples[1]=AC3MIXOUT((*right++ + clev * *centre++ + slev * *sur++) * loc_gain);
#else
  const AC3_DOUBLE_T ds_sur = (amip->ds_x2 + amip->ds_x1 * 2.0 + *sur) * 0.12531781
			     + amip->ds_y1 * 0.77997062 - amip->ds_y2 * 0.28124186;
  amip->ds_x2 = amip->ds_x1;
  amip->ds_x1 = *sur++;
  amip->ds_y2 = amip->ds_y1;
  amip->ds_y1 = ds_sur;

  out_samples[0]=AC3MIXOUT((*left++  + M_SQRT_2 * (*centre   - ds_sur)) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + M_SQRT_2 * (*centre++ + ds_sur)) * loc_gain);
#endif

  out_samples+=2;
 }while(--j);
}

static void downmix_2f_1r_to_2ch_dolby(ac3_decoder_data *amip, stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 j;
//#ifdef DOLBY_DOWNMIX_1_OLD
// bsi_t *bsi=amip->bsi;
// const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
//#endif
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[0];
 AC3_DOUBLE_T *left, *right, *sur;

 left      = samples[0];
 right     = samples[1];
 sur	   = samples[2];

 j=256;
 do{
#ifdef DOLBY_DOWNMIX_1_OLD
  out_samples[0]=AC3MIXOUT((*left++  - *sur)   * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + *sur++) * loc_gain);
  //out_samples[0]=AC3MIXOUT((*left++  - slev * *sur)   * loc_gain);
  //out_samples[1]=AC3MIXOUT((*right++ + slev * *sur++) * loc_gain);
#else
  const AC3_DOUBLE_T ds_sur = (amip->ds_x2 + amip->ds_x1 * 2.0 + *sur) * 0.12531781
			     + amip->ds_y1 * 0.77997062 - amip->ds_y2 * 0.28124186;
  amip->ds_x2 = amip->ds_x1;
  amip->ds_x1 = *sur++;
  amip->ds_y2 = amip->ds_y1;
  amip->ds_y1 = ds_sur;

  out_samples[0] = AC3MIXOUT((*left++  - M_SQRT_2 * ds_sur) * loc_gain);
  out_samples[1] = AC3MIXOUT((*right++ + M_SQRT_2 * ds_sur) * loc_gain);
#endif

  out_samples+=2;
 }while(--j);
}

#endif // AC3_LINK_DOWNMIX_1

//------------------ Dolby II downmix -------------------------------------

#ifdef AC3_LINK_DOWNMIX_2

static void downmix_3f_2r_to_2ch_dolby_II(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 i,sdc=amip->surdelay_counter;
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T clev = cmixlev_lut[bsi->cmixlev];
 const AC3_DOUBLE_T slev_d = smixlev_lut[amip->surmixlev_delay[sdc]];
 //const AC3_DOUBLE_T slev = smixlev_lut[bsi->surmixlev];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[1];

 AC3_DOUBLE_T *centre, *left, *right, *lfe, *left_dsur, *right_dsur;
 AC3_DOUBLE_T *left_sur,*right_sur;

 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];
 left_sur  = samples[3];
 right_sur = samples[4];
 //left_dsur = amip->surround_delay[0][sdc][0];
 //right_dsur= amip->surround_delay[0][sdc][1];
 left_dsur = amip->surround_delay[0][0][0]; // Watcom C bug? (surround_delay[][][] addressing is bad)
 left_dsur+= sdc*2*256;                     //
 right_dsur= left_dsur+256;                 //

 if(bsi->lfeon){
  lfe = samples[5];
  i=256;
  do{
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + *centre   + *left_dsur++) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + *centre++ + *right_dsur++) * loc_gain);
   out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   + slev_d * *left_dsur++) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ + slev_d * *right_dsur++) * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + (*centre   + *left_sur++ + *right_dsur++)*M_SQRT_2) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + (*centre++ + *right_sur++ + *left_dsur++)*M_SQRT_2) * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   - slev * *left_sur++  + slev_d * *right_dsur++) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ - slev * *right_sur++ + slev_d * *left_dsur++) * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   + slev * *left_sur++  + slev_d * *right_dsur++) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ + slev * *right_sur++ + slev_d * *left_dsur++) * loc_gain);
   //AC3_DOUBLE_T csurr=(*left_sur ++ + *right_sur++);//*slev;
   //AC3_DOUBLE_T dsurr=(*left_dsur ++ + *right_dsur++)*slev_d;
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + *centre   - csurr) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + *centre++ + csurr) * loc_gain);
   //out_samples[0]=AC3MIXOUT((*left++  + *lfe   + clev * *centre   + csurr  - dsurr) * loc_gain);
   //out_samples[1]=AC3MIXOUT((*right++ + *lfe++ + clev * *centre++ - csurr  + dsurr) * loc_gain);

   //out_samples[0]=AC3MIXOUT((slev_d * *left_dsur++) * loc_gain);
   //out_samples[1]=AC3MIXOUT((slev_d * *right_dsur++) * loc_gain);
   out_samples+=2;
  }while(--i);
 }else{
  i=256;
  do{
   out_samples[0]=AC3MIXOUT((*left++  + clev * *centre   + slev_d * *left_dsur++) * loc_gain);
   out_samples[1]=AC3MIXOUT((*right++ + clev * *centre++ + slev_d * *right_dsur++) * loc_gain);
   out_samples+=2;
  }while(--i);
 }

 amip->surmixlev_delay[sdc]=bsi->surmixlev;
 left_dsur = amip->surround_delay[0][0][0];
 left_dsur+= sdc*2*256;
 right_dsur= left_dsur+256;
 memcpy(left_dsur ,samples[3],256*sizeof(AC3_DOUBLE_T));
 memcpy(right_dsur,samples[4],256*sizeof(AC3_DOUBLE_T));
 sdc++;
 sdc%=AC3_DOLBY2_SURROUND_DELAY;
 amip->surdelay_counter=sdc;
}

static void downmix_2f_2r_to_2ch_dolby_II(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples)
{
 uint_32 i,sdc=amip->surdelay_counter;
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T slev_d = smixlev_lut[amip->surmixlev_delay[sdc]];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[0];

 AC3_DOUBLE_T *left, *right, *left_dsur, *right_dsur;
 left      = samples[0];
 right     = samples[1];
 left_dsur = amip->surround_delay[0][0][0]; // Watcom C bug? (surround_delay[][][] addressing is bad)
 left_dsur+= sdc*2*256;                     //
 right_dsur= left_dsur+256;                 //

 i=256;
 do{
  out_samples[0]=AC3MIXOUT((*left++  + slev_d * *left_dsur++)  * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + slev_d * *right_dsur++) * loc_gain);
  out_samples+=2;
 }while(--i);

 amip->surmixlev_delay[sdc]=bsi->surmixlev;
 left_dsur = amip->surround_delay[0][0][0];
 left_dsur+= sdc*2*256;
 right_dsur= left_dsur+256;
 memcpy(left_dsur ,samples[2],256*sizeof(AC3_DOUBLE_T));
 memcpy(right_dsur,samples[3],256*sizeof(AC3_DOUBLE_T));
 sdc++;
 sdc%=AC3_DOLBY2_SURROUND_DELAY;
 amip->surdelay_counter=sdc;
}

static void downmix_3f_1r_to_2ch_dolby_II(ac3_decoder_data *amip, stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 i,sdc=amip->surdelay_counter;
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T clev = cmixlev_lut[bsi->cmixlev];
 const AC3_DOUBLE_T slev_d = smixlev_lut[amip->surmixlev_delay[sdc]];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[1];

 AC3_DOUBLE_T *centre, *left, *right, *dsur;
 left      = samples[0];
 centre    = samples[1];
 right     = samples[2];
 dsur      = amip->surround_delay[0][0][0]; // Watcom C bug? (surround_delay[][][] addressing is bad)
 dsur     += sdc*2*256;                     //

 i=256;
 do{
  out_samples[0]=AC3MIXOUT((*left++  + clev * *centre   - slev_d * *dsur  ) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + clev * *centre++ + slev_d * *dsur++) * loc_gain);
  out_samples+=2;
 }while(--i);

 amip->surmixlev_delay[sdc]=bsi->surmixlev;
 dsur = amip->surround_delay[0][0][0];
 dsur+= sdc*2*256;
 memcpy(dsur ,samples[3],256*sizeof(AC3_DOUBLE_T));
 sdc++;
 sdc%=AC3_DOLBY2_SURROUND_DELAY;
 amip->surdelay_counter=sdc;
}

static void downmix_2f_1r_to_2ch_dolby_II(ac3_decoder_data *amip, stream_samples_t samples, AC3_OUT_T *out_samples)
{
 uint_32 i,sdc=amip->surdelay_counter;
 bsi_t *bsi=amip->bsi;
 const AC3_DOUBLE_T slev_d = smixlev_lut[amip->surmixlev_delay[sdc]];
 const AC3_DOUBLE_T loc_gain=amip->gain*DownMixMuls[0];

 AC3_DOUBLE_T *left, *right, *dsur;
 left      = samples[0];
 right     = samples[1];
 dsur      = amip->surround_delay[0][0][0]; // Watcom C bug? (surround_delay[][][] addressing is bad)
 dsur     += sdc*2*256;                     //

 i=256;
 do{
  out_samples[0]=AC3MIXOUT((*left++  - slev_d * *dsur  ) * loc_gain);
  out_samples[1]=AC3MIXOUT((*right++ + slev_d * *dsur++) * loc_gain);
  out_samples+=2;
 }while(--i);

 amip->surmixlev_delay[sdc]=bsi->surmixlev;
 dsur = amip->surround_delay[0][0][0];
 dsur+= sdc*2*256;
 memcpy(dsur ,samples[2],256*sizeof(AC3_DOUBLE_T));
 sdc++;
 sdc%=AC3_DOLBY2_SURROUND_DELAY;
 amip->surdelay_counter=sdc;
}

#endif // AC3_LINK_DOWNMIX_2

//------------------------------------------------------------------------

typedef struct downmix_func_s{
 void (*func)(ac3_decoder_data *amip,stream_samples_t samples,AC3_OUT_T *out_samples);
}downmix_func_t;

static downmix_func_t DownMix_Funcs[][8]={
#ifdef AC3_LINK_DOWNMIX_0
 {
  {&downmix_1f_0r_to_2ch},
  {&downmix_1f_0r_to_2ch},
  {&downmix_2f_0r_to_2ch},
  {&downmix_3f_0r_to_2ch},
  {&downmix_2f_1r_to_2ch},
  {&downmix_3f_1r_to_2ch},
  {&downmix_2f_2r_to_2ch},
  {&downmix_3f_2r_to_2ch}
 }
#endif
#ifdef AC3_LINK_DOWNMIX_1
 #ifdef AC3_LINK_DOWNMIX_0
  ,
 #endif
 {
  {&downmix_1f_0r_to_2ch},
  {&downmix_1f_0r_to_2ch},
  {&downmix_2f_0r_to_2ch},
  {&downmix_3f_0r_to_2ch},
  {&downmix_2f_1r_to_2ch_dolby},
  {&downmix_3f_1r_to_2ch_dolby},
  {&downmix_2f_2r_to_2ch_dolby},
  {&downmix_3f_2r_to_2ch_dolby}
 }
#endif
#ifdef AC3_LINK_DOWNMIX_2
 #if defined(AC3_LINK_DOWNMIX_0) || defined(AC3_LINK_DOWNMIX_1)
  ,
 #endif
 {
  {&downmix_1f_0r_to_2ch},
  {&downmix_1f_0r_to_2ch},
  {&downmix_2f_0r_to_2ch},
  {&downmix_3f_0r_to_2ch},
  {&downmix_2f_1r_to_2ch_dolby_II},
  {&downmix_3f_1r_to_2ch_dolby_II},
  {&downmix_2f_2r_to_2ch_dolby_II},
  {&downmix_3f_2r_to_2ch_dolby_II}
 }
#endif
};

void ac3dec_downmix_output(ac3_decoder_data *amip,bsi_t* bsi, stream_samples_t samples,AC3_OUT_T *out_samples)
{
 if(bsi->compre)
  amip->gain = compr_table[bsi->compr];
 else
  amip->gain = dynrng_table[amip->audblk->dynrng];

 DownMix_Funcs[amip->downmix_select][bsi->acmod].func(amip,samples,out_samples);

 //fprintf(stdout,"%3d %3d %6.0f \n",bsi->compr,amip->audblk->dynrng,amip->gain);
}

/*void ac3dec_downmix_output(ac3_decoder_data *amip,bsi_t* bsi, stream_samples_t samples,AC3_OUT_T *out_samples)
{
 unsigned int calccompr;
 int curr=0,prev=0,newc=0;

 if(amip->prev_compr!=DOWNMIX_COMPR_INVALID){
  curr=(bsi->compr>=128)? -(256-bsi->compr):bsi->compr;
  prev=(amip->prev_compr>=128)? -(256-amip->prev_compr):amip->prev_compr;
  newc=(prev+curr)/2;
  calccompr=(newc<0)? (256+newc):newc;
 }else
  calccompr=bsi->compr;

 amip->prev_compr=bsi->compr;

 //if(bsi->compre)
  amip->gain = compr_table[calccompr];
 //else
 // amip->gain = dynrng_table[amip->audblk->dynrng];

 DownMix_Funcs[amip->downmix_select][bsi->acmod].func(amip,samples,out_samples);

 fprintf(stdout,"%3d %3d %6.0f \n",bsi->compr,calccompr,amip->gain);
}*/
