/*
    JPC: A x86 PC Hardware Emulator for a pure Java Virtual Machine
    Release Version 2.0

    A project from the Physics Dept, The University of Oxford

    Copyright (C) 2007 Isis Innovation Limited

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 
    Details (including contact information) can be found at: 

    www.physics.ox.ac.uk/jpc
*/

package org.jpc.j2se;

import org.jpc.emulator.*;
import org.jpc.support.*;

public class VirtualClock extends AbstractHardwareComponent implements Clock
{
    private PriorityVector timers;
    private Timer[] timerArray;

    private boolean ticksEnabled;
    private long ticksOffset;
    private long ticksStatic;
    private long tickRate;

    private static boolean useNanos = false;

    static 
    {
        try
        {
            System.nanoTime();
            useNanos = true;
        }
        catch (Throwable t) {}
    }

    public VirtualClock()
    {
	timers = new PriorityVector(20); // initial capacity to be revised
	tickRate = 0;
	ticksEnabled = false;
	ticksOffset = 0;
	ticksStatic = 0;
    }

    public Timer newTimer(TimerResponsive object)
    {
	Timer tempTimer = new Timer(object, this);
	this.timers.addComparableObject(tempTimer);

	return tempTimer;
    }

    public void process()
    {
	do 
        {
	    Timer tempTimer = (Timer) timers.firstElement();	    
	    if ((tempTimer == null) || !tempTimer.check(getTime())) 
		break;

	    timers.removeFirstElement();
	    tempTimer.setStatus(false);
	    tempTimer.runCallback();
	} 
        while(true);
    }

    public void update(Timer object)
    {
	timers.removeElement(object);
	timers.addComparableObject(object);
    }

    public long getTime()
    {
	if (ticksEnabled)
	    return this.getRealTime() + ticksOffset;
        else
	    return ticksStatic;
    }

    private long getRealTime()
    {
        if (useNanos)
            return System.nanoTime();
        return System.currentTimeMillis()*1000000l;
    }

    public long getTickRate()
    {
	if (tickRate != 0)
            return tickRate;
          
        long msec = System.currentTimeMillis();
        long ticks = this.getRealTime();
	
        do
        {
            try 
            { 
                Thread.sleep(500); 
            } catch (Throwable t) {}
	    
            msec = System.currentTimeMillis() - msec;
        }
        while (msec < 450);

        ticks = this.getRealTime() - ticks;
        tickRate = 1000 * ((ticks + (msec >>> 1)) / msec);
        
        return tickRate;	
    }

    public void pause()
    {
	if (ticksEnabled) 
        {
	    ticksStatic = getTime();
	    ticksEnabled = false;
	}
    }

    public void resume()
    {
	if (!ticksEnabled) 
        {
	    ticksOffset = ticksStatic - getRealTime();
	    ticksEnabled = true;
	}
    }

    public void reset()
    {
	this.pause();
	tickRate = 0;
	ticksOffset = 0;
	ticksStatic = 0;
    }

    public String toString()
    {
	return "Virtual Clock";
    }
}
