/*
    JPC: A x86 PC Hardware Emulator for a pure Java Virtual Machine
    Release Version 2.0

    A project from the Physics Dept, The University of Oxford

    Copyright (C) 2007 Isis Innovation Limited

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 
    Details (including contact information) can be found at: 

    www.physics.ox.ac.uk/jpc
*/

package org.jpc.emulator.memory.codeblock.basic;

import org.jpc.emulator.memory.codeblock.*;

import org.jpc.emulator.processor.*;
import org.jpc.emulator.processor.fpu64.*;
import org.jpc.emulator.memory.*;

public abstract class FirstStageCodeBlock implements CodeBlock, FirstStageCodeSet
{
    protected static final ProcessorException exceptionDE = new ProcessorException(Processor.PROC_EXCEPTION_DE, true);
    protected static final ProcessorException exceptionUD = new ProcessorException(Processor.PROC_EXCEPTION_UD, true);
    protected static final ProcessorException exceptionNM = new ProcessorException(Processor.PROC_EXCEPTION_NM, true);

    protected static final ProcessorException unimplemented = new ProcessorException(-1, true);

    protected static final boolean[] parityMap;
   
    protected Processor cpu;
    protected FpuState fpu;

    static
    {
        parityMap = new boolean[256];
        for (int i=0; i<256; i++)
        {
            boolean val = true;
            for (int j=0; j<8; j++)
                if ((0x1 & (i >> j)) == 1)
                    val = !val;

            parityMap[i] = val;
        }
    } 

    protected int microcodesPosition;
    private int x86Count;

    protected int[] microcodes;
    protected int[] cumulativeX86Length;
    protected int executeCount;

    public FirstStageCodeBlock()
    {
    }

    public FirstStageCodeBlock(int[] microcodes, int[] x86lengths)
    {
        this.microcodes = microcodes;
        cumulativeX86Length = x86lengths;
	if (cumulativeX86Length.length == 0)
	    x86Count = 0;
	else {
	    int count = 1;
	    for (int i = 1; i < cumulativeX86Length.length; i++) {
		if (cumulativeX86Length[i] > cumulativeX86Length[i-1]) count++;
	    }
	    x86Count = count;
	}
    }

    public int getX86Length()
    {
        if (microcodes.length == 0)
            return 0;
	return cumulativeX86Length[microcodes.length-1];
    }

    public int getX86Count()
    {
	return x86Count;
    }

    public String getDisplayString()
    {
        StringBuffer buf = new StringBuffer();
        for (int i=0; i < microcodes.length; i++)
            buf.append(microcodes[i]+"\n");
        return buf.toString();
    }

    public boolean handleMemoryRegionChange(int startAddress, int endAddress)
    {
        return false;
    }

    public int[] getMicrocodes()
    {
        int[] result = new int[microcodes.length];
        System.arraycopy(microcodes, 0, result, 0, result.length);
        return result;
    }

    protected final int getMicrocode()
    {
        int result = microcodes[microcodesPosition];
        microcodesPosition++;
	return result;
    }

    public abstract int operandCount(int microcodeOpcode);
    public abstract int immediateCount(int microcodeOperand);
    protected abstract void loadSegment(int segment, int selector);

    protected final void skipOperand()
    {
	int temp = immediateCount(getMicrocode());
	microcodesPosition += temp;	
    }

    private final void standardFlags(byte result)
    {
        cpu.eflagsOverflow = false;
        cpu.eflagsCarry = false;
        cpu.eflagsZero = (result == 0);
        cpu.eflagsParity = parityMap[result & 0xff];
	cpu.eflagsSign = (result < 0);
    }

    private final void standardFlags(short result)
    {
        cpu.eflagsOverflow = false;
        cpu.eflagsCarry = false;
        cpu.eflagsZero = (result == 0);
        cpu.eflagsParity = parityMap[result & 0xff];
	cpu.eflagsSign = (result < 0);
    }

    private final void standardFlags(int result)
    {
        cpu.eflagsOverflow = false;
        cpu.eflagsCarry = false;
        cpu.eflagsZero = (result == 0);
        cpu.eflagsParity = parityMap[result & 0xff];
	cpu.eflagsSign = (result < 0);
    }

    private double getDoubleOperand()
    {
	switch (getMicrocode()) {
	case FPU_ST0: return fpu.ST(0);
	case FPU_ST1: return fpu.ST(1);
	case FPU_ST2: return fpu.ST(2);
	case FPU_ST3: return fpu.ST(3);
	case FPU_ST4: return fpu.ST(4);
	case FPU_ST5: return fpu.ST(5);
	case FPU_ST6: return fpu.ST(6);
	case FPU_ST7: return fpu.ST(7);
	default:
	    microcodesPosition--;
	    break;
	}

	long w = getLongOperand();
	if (FpuState64.isSNaN(w)) {
	    fpu.setInvalidOperation();
	    fpu.checkExceptions();
	    return Double.NaN;
	}
	return Double.longBitsToDouble(w);
    }

    private void setDoubleOperand(double data)
    {
	switch (getMicrocode()) {
	case FPU_ST0: fpu.setST(0, data); return;
	case FPU_ST1: fpu.setST(1, data); return;
	case FPU_ST2: fpu.setST(2, data); return;
	case FPU_ST3: fpu.setST(3, data); return;
	case FPU_ST4: fpu.setST(4, data); return;
	case FPU_ST5: fpu.setST(5, data); return;
	case FPU_ST6: fpu.setST(6, data); return;
	case FPU_ST7: fpu.setST(7, data); return;
	default:
	    microcodesPosition--;
	    break;
	}
	/*
	storedPosition = microcodesPosition;
	setLastDataOperand();
	microcodesPosition = storedPosition;
	*/
	long w = Double.doubleToRawLongBits(data);
	setLongOperand(w);
    }

    private double getSingleOperand()
    {
	switch (getMicrocode()) {
	case FPU_ST0: return fpu.ST(0);
	case FPU_ST1: return fpu.ST(1);
	case FPU_ST2: return fpu.ST(2);
	case FPU_ST3: return fpu.ST(3);
	case FPU_ST4: return fpu.ST(4);
	case FPU_ST5: return fpu.ST(5);
	case FPU_ST6: return fpu.ST(6);
	case FPU_ST7: return fpu.ST(7);
	default:
	    microcodesPosition--;
	    break;
	}

	int n = getIntOperand();
        int exponent = ((n >> 23) & 0xff);
        int fraction = (n & 0x7fffff);
        if (exponent == 0xff && (fraction & (1 << 22)) == 0 && (fraction != 0))
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
            return Double.NaN;
        }
        float x = Float.intBitsToFloat(n);

        return (double)x;
    }

    private void setSingleOperand(double data)
    {
	switch (getMicrocode()) {
	case FPU_ST0: fpu.setST(0, data); return;
	case FPU_ST1: fpu.setST(1, data); return;
	case FPU_ST2: fpu.setST(2, data); return;
	case FPU_ST3: fpu.setST(3, data); return;
	case FPU_ST4: fpu.setST(4, data); return;
	case FPU_ST5: fpu.setST(5, data); return;
	case FPU_ST6: fpu.setST(6, data); return;
	case FPU_ST7: fpu.setST(7, data); return;
	default:
	    microcodesPosition--;
	    break;
	}
	/*
	storedPosition = microcodesPosition;
	setLastDataOperand();
	microcodesPosition = storedPosition;
	*/
	int w = Float.floatToRawIntBits((float) data);
	setIntOperand(w);
    }

    private double getWordIntegerOperand()
    {
	int w = getShortOperand();
        return (double)w;
    }

    private void setWordIntegerOperand(double x)
    {
        //setLastDataOperand(state);
        short w = (short)x;
        if (x > (double)Short.MAX_VALUE ||
            x < (double)Short.MIN_VALUE)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
            w = Short.MIN_VALUE; // integer indefinite
        }
	setShortOperand(w);
    }

    private double getDoubleIntegerOperand()
    {
	int w = getIntOperand();
        return (double)w;
    }

    private void setDoubleIntegerOperand(double x)
    {
        //setLastDataOperand(state);
        int w = (int)x;
        if (x > (double)Integer.MAX_VALUE ||
            x < (double)Integer.MIN_VALUE)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
            w = Integer.MIN_VALUE; // integer indefinite
        }
	setIntOperand(w);
    }

    private double getQuadIntegerOperand()
    {
	long w = getLongOperand();
        return (double)w;
    }

    private void setQuadIntegerOperand(double x)
    {
        //setLastDataOperand(state);
        long w = (long)x;
        if (x > (double)Long.MAX_VALUE ||
            x < (double)Long.MIN_VALUE)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
            w = Long.MIN_VALUE; // integer indefinite
        }

	setLongOperand(w);
    }

    private void validateOperand(double x) throws ProcessorException
    {
        // 1. check for SNaN.  set IE, throw if not masked.
        //    (actually, this check is already done with the operand
        //    get() method---and SNaN isn't transmitted in the
        //    Java double format.
        // 2. check for denormal operand.  set DE, throw if not masked.
        long n = Double.doubleToRawLongBits(x);
        if (((n >> 52) & 0x7ff) == 0 && ((n & 0xfffffffffffffL) != 0)) {
            fpu.setDenormalizedOperand();
            fpu.checkExceptions();
        }
    }
    
    private void checkResult(double x) throws ProcessorException
    {
        // 1. check for numeric overflow or underflow.
        if (Double.isInfinite(x)) {
	    // overflow
	    // NOTE that this will also flag cases where the inputs
            // were also infinite.  TODO:  find out whether, for
            // instance, multipling inf by finite in x87 will also
            // set the overflow flag.
            fpu.setOverflow();
            fpu.checkExceptions();
        }
	
        // for underflow, FST handles it separately (and before the store)
	
	// if destination is a register, then the result gets biased
	// and stored (is this the Java rule as well?)
	
        // and how can we trap rounding action?  is it possible that
        // something got rounded all the way to zero?
	
        // 2. check for inexact result exceptions.
    }    

    /*
    private void setLastDataOperand(ProcessorState state) throws ProcessorException
    {
        ProcessorState.SegmentSelector seg = mem.getSegment(state);
        fpu.lastData = ((0xffffffffl & getAddressOperand()) | ((long)getSegmentOperand().getSelector() << 32));
    }
    */

    protected final void aaa()
    {
	if (((cpu.eax & 0xf) > 0x9) || cpu.eflagsAuxiliaryCarry) {
	    int alCarry = ((cpu.eax & 0xff) > 0xf9) ? 0x100 : 0x000;
	    cpu.eax = (0xffff0000 & cpu.eax) | (0x0f & (cpu.eax + 6)) | (0xff00 & (cpu.eax + 0x100 + alCarry));
	    cpu.eflagsAuxiliaryCarry = true;
	    cpu.eflagsCarry = true;
	} else {
	    cpu.eflagsAuxiliaryCarry = false;
	    cpu.eflagsCarry = false;
	    cpu.eax = cpu.eax & 0xffffff0f;
	}
    }

    protected final void aad() throws ProcessorException
    {
        int tl = (cpu.eax & 0xff);
        int th = ((cpu.eax >> 8) & 0xff);
        int b = 0xff & getByteOperand();

	int ax1 = th * b;
	int ax2 = ax1 + tl;

        cpu.eax = (cpu.eax & ~0xffff) | (ax2 & 0xff);

	standardFlags((byte)ax2);

	cpu.eflagsAuxiliaryCarry = ((ax1 & 0x08) != (ax2 & 0x08));
	cpu.eflagsCarry = (ax2 > 0xff);
	cpu.eflagsOverflow = ((ax2 & 0x80) != (tl & 0x80));
    }

    protected final void aam() throws ProcessorException
    {
        int tl = 0xff & cpu.eax;
        int b = 0xff & getByteOperand();
        if (b == 0) 
            throw exceptionDE;
        int ah = 0xff & (tl / b);
        int al = 0xff & (tl % b);
        cpu.eax &= ~0xffff;
        cpu.eax |= (al | (ah << 8));

	cpu.eflagsAuxiliaryCarry = false;
	cpu.eflagsCarry = false;
	standardFlags((byte)al);
    }

    protected final void aas()
    {
	if (((cpu.eax & 0xf) > 0x9) || cpu.eflagsAuxiliaryCarry) {
	    int alBorrow = (cpu.eax & 0xff) < 6 ? 0x100 : 0x000;
	    cpu.eax = (0xffff0000 & cpu.eax) | (0x0f & (cpu.eax - 6)) | (0xff00 & (cpu.eax - 0x100 - alBorrow));
	    cpu.eflagsAuxiliaryCarry = true;
	    cpu.eflagsCarry = true;
	} else {
	    cpu.eflagsAuxiliaryCarry = false;
	    cpu.eflagsCarry = false;
	    cpu.eax = cpu.eax & 0xffffff0f;
	}
    }

    protected final void adc_o8() throws ProcessorException
    {
        int o2 = 0xff & getByteOperand();
        int o1 = 0xff & getByteOperand();
        int result = o1;

        if (cpu.eflagsCarry && (o2 == 0xff)) {
	    setByteOperand((byte)result);	    
	    cpu.eflagsOverflow = false;
            cpu.eflagsCarry = true;
	}
        else
        {
	    int sum = cpu.eflagsCarry ? 1 : 0;
            result += o2 + sum;
	    setByteOperand((byte)result);

	    cpu.eflagsCarry = (result & (~0xff)) !=0;
	    if ((byte)(o2 + sum) > 0) // +VE Swing
		cpu.eflagsOverflow = ((byte)result < (byte)o1);
	    else // -VE Swing
		cpu.eflagsOverflow = ((byte)result > (byte)o1);
        }

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
        cpu.eflagsZero = ((byte)result == 0);
        cpu.eflagsParity = parityMap[0xff & result];
        cpu.eflagsSign = ((byte)result < 0);
    }

    protected final void adc_o16() throws ProcessorException
    {
        int o2 = 0xffff & getShortOperand();
        int o1 = 0xffff & getShortOperand();
        int result = o1;

        if (cpu.eflagsCarry && (o2 == 0xffff)) {
	    setShortOperand((short)result);
	    cpu.eflagsOverflow = false;
            cpu.eflagsCarry = true;
        } else {
	    int sum = cpu.eflagsCarry ? 1 : 0;
            result += o2 + sum;
	    setShortOperand((short)result);
	    
	    cpu.eflagsCarry = (result & (~0xffff)) !=0;
	    if ((short)(o2 + sum) > 0) // +VE Swing
		cpu.eflagsOverflow = ((short)result < (short)o1);
	    else // -VE Swing
		cpu.eflagsOverflow = ((short)result > (short)o1);
        }

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
        cpu.eflagsZero = ((short)result == 0);
        cpu.eflagsParity = parityMap[0xff & result];
        cpu.eflagsSign = ((short)result < 0);
    }

    protected final void adc_o32() throws ProcessorException
    {
        long o2 = 0xffffffffl & getIntOperand();
        long o1 = 0xffffffffl & getIntOperand();
        long result = o1;

        if (cpu.eflagsCarry && (o2 == 0xffffffffl)) {
	    setIntOperand((int)result);
	    cpu.eflagsOverflow = false;
            cpu.eflagsCarry = true;
	}
        else
        {
	    int sum = cpu.eflagsCarry ? 1 : 0;
            result += o2 + sum;
	    setIntOperand((int)result);

	    cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	    if ((int)(o2 + sum) > 0) // +VE Swing
		cpu.eflagsOverflow = ((int)result < (int)o1);
	    else // -VE Swing
		cpu.eflagsOverflow = ((int)result > (int)o1);
        }

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
        cpu.eflagsZero = ((int)result == 0);
        cpu.eflagsParity = parityMap[0xff & (int)result];
        cpu.eflagsSign = ((int)result < 0);
    }

    protected final void add_o8() throws ProcessorException
    {
	int o1 = 0xff & getByteOperand();
	int o2 = 0xff & getByteOperand();
        int result = o1 + o2;
	setByteOperand((byte)result);
        standardFlags((byte)result);
        
        cpu.eflagsCarry = (result & (~0xff)) !=0;
        if ((byte)o2 > 0) // +VE Swing
            cpu.eflagsOverflow = ((byte)result < (byte)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((byte)result > (byte)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void add_o16() throws ProcessorException
    {
	int o2 = 0xffff & getShortOperand();
	int o1 = 0xffff & getShortOperand();
        int result = o1 + o2;
	setShortOperand((short)result);
        standardFlags((short)result);
        
        cpu.eflagsCarry = (result & (~0xffff)) !=0;
        if ((short)o2 > 0) // +VE Swing
            cpu.eflagsOverflow = ((short)result < (short)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((short)result > (short)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void add_o32() throws ProcessorException
    {
	long o1 = 0xffffffffl & getIntOperand();
	long o2 = 0xffffffffl & getIntOperand();
        long result = o1 + o2;
	setIntOperand((int)result);
        standardFlags((int)result);
        
        cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
        if ((int)o2 > 0) // +VE Swing
            cpu.eflagsOverflow = ((int)result < (int)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((int)result > (int)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void and_o8() throws ProcessorException
    {
	byte result = (byte)(getByteOperand() & getByteOperand());
	setByteOperand(result);
        standardFlags((byte)result);
    }

    protected final void and_o16() throws ProcessorException
    {
	short result = (short)(getShortOperand() & getShortOperand());
	setShortOperand(result);
        standardFlags((short)result);
    }

    protected final void and_o32() throws ProcessorException
    {
	int result = (getIntOperand() & getIntOperand());
	setIntOperand(result);
        standardFlags(result);
    }

    protected final void bsf_o16() throws ProcessorException
    {
	int source = 0xffff & getShortOperand();
	if (source == 0) {
	    cpu.eflagsZero = true;
	    skipOperand();
	} else {
	    setShortOperand((short)numberOfTrailingZeros(source));
	    cpu.eflagsZero = false;
	}
    }

    protected final void bsf_o32() throws ProcessorException
    {
	int source = getIntOperand();
	if (source == 0) {
	    cpu.eflagsZero = true;
	    skipOperand();
	} else {
	    setIntOperand(numberOfTrailingZeros(source));
	    cpu.eflagsZero = false;
	}
    }

    protected final void bsr_o16() throws ProcessorException
    {
	int source = 0xffff & getShortOperand();
	if (source == 0) {
	    cpu.eflagsZero = true;
	    skipOperand();
	} else {
	    setShortOperand((short)(31 - numberOfLeadingZeros(source)));
	    cpu.eflagsZero = false;
	}
    }

    protected final void bsr_o32() throws ProcessorException
    {
	int source = getIntOperand();
	if (source == 0) {
	    cpu.eflagsZero = true;
	    skipOperand();
	} else {
	    setIntOperand(31 - numberOfLeadingZeros(source));
	    cpu.eflagsZero = false;
	}
    }

    protected final void bswap()
    {
	setIntOperand(reverseBytes(getIntOperand()));
    }

    protected final void bt_o16() throws ProcessorException
    {
	short bitOffset = getShortOperand();
	switch (microcodes[microcodesPosition]) {
	case AX:
	case CX:
	case DX:
	case BX:
	case SP:
	case BP:
	case SI:
	case DI:
	    {
		int bitStringOperand = getShortOperand();
		bitOffset &= 0xf;
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
		skipOperand();
	    }
	break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	}		
    }

    protected final void bt_o32() throws ProcessorException
    {
	int bitOffset = getIntOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
		skipOperand();
	    }
	break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	}		
    }

    protected final void bt_o32_o8() throws ProcessorException
    {
	int bitOffset = 0x1f & getByteOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
		skipOperand();
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void bt_o16_o8() throws ProcessorException
    {
	int bitOffset = 0x1f & getByteOperand();
	switch (microcodes[microcodesPosition]) {
	case AX:
	case CX:
	case DX:
	case BX:
	case SP:
	case BP:
	case SI:
	case DI:
	    {
		int bitStringOperand = getShortOperand();
		bitOffset &= 0xf;
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
		skipOperand();
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void btc_o16() throws ProcessorException
    {
	short bitOffset = getShortOperand();
	switch (microcodes[microcodesPosition]) {
	case AX:
	case CX:
	case DX:
	case BX:
	case SP:
	case BP:
	case SI:
	case DI:
	    {
		int bitStringOperand = getShortOperand();
		bitOffset &= 0xf;
		setShortOperand((short)(bitStringOperand ^ (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand ^ (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	}		
    }

    protected final void btc_o32() throws ProcessorException
    {
	int bitOffset = getIntOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		setIntOperand(bitStringOperand ^ (1 << bitOffset));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand ^ (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void btc_o32_o8() throws ProcessorException
    {
	int bitOffset = 0x1f & getByteOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		setIntOperand(bitStringOperand ^ ~(1 << bitOffset));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand ^ ~(1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void btr_o16() throws ProcessorException
    {
	short bitOffset = getShortOperand();
	switch (microcodes[microcodesPosition]) {
	case AX:
	case CX:
	case DX:
	case BX:
	case SP:
	case BP:
	case SI:
	case DI:
	    {
		int bitStringOperand = getShortOperand();
		bitOffset &= 0xf;
		setShortOperand((short)(bitStringOperand & ~(1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand & ~(1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	}		
    }

    protected final void btr_o32() throws ProcessorException
    {
	int bitOffset = getIntOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		setIntOperand(bitStringOperand & ~(1 << bitOffset));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand & ~(1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void btr_o32_o8() throws ProcessorException
    {
	int bitOffset = 0x1f & getByteOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		setIntOperand(bitStringOperand & ~(1 << bitOffset));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand & ~(1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void bts_o16() throws ProcessorException
    {
	short bitOffset = getShortOperand();
	switch (microcodes[microcodesPosition]) {
	case AX:
	case CX:
	case DX:
	case BX:
	case SP:
	case BP:
	case SI:
	case DI:
	    {
		int bitStringOperand = getShortOperand();
		bitOffset &= 0xf;
		setShortOperand((short)(bitStringOperand | (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand | (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	break;
	}		
    }

    protected final void bts_o32() throws ProcessorException
    {
	int bitOffset = getIntOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		setIntOperand(bitStringOperand | (1 << bitOffset));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand | (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;		
	}

    }

    protected final void bts_o32_o8() throws ProcessorException
    {
	int bitOffset = 0x1f & getByteOperand();
	switch (microcodes[microcodesPosition]) {
	case EAX:
	case ECX:
	case EDX:
	case EBX:
	case ESP:
	case EBP:
	case ESI:
	case EDI:
	    {
		int bitStringOperand = getIntOperand();
		bitOffset &= 0x1f;
		setIntOperand(bitStringOperand | (1 << bitOffset));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	default:
	    {
		int bitStringBaseAddress = getAddressOperand() + ((bitOffset & ~0x7) / 8);
		Segment bitStringSegment = getSegmentOperand();
		bitOffset &= 0x7;
		
		byte bitStringOperand = bitStringSegment.getByte(bitStringBaseAddress);
		bitStringSegment.setByte(bitStringBaseAddress, (byte)(bitStringOperand | (1 << bitOffset)));
		cpu.eflagsCarry = ((bitStringOperand & (1 << bitOffset)) != 0);
	    }
	    break;
	}

    }

    protected final void cmc()
    {
	cpu.eflagsCarry = !cpu.eflagsCarry;
    }

    protected final void cbw()
    {
       if ((cpu.eax & 0x80) == 0)
        {
            cpu.eax &= 0xffff00ff;
        }
        else
        {
            cpu.eax |= 0xff00;
        }

    }
    protected final void cwd()
    {
        if ((cpu.eax & 0x8000) == 0)
        {
            cpu.edx &= 0xffff0000;
        }
        else
        {
            cpu.edx |= 0x0000ffff;
        }

    }

    protected final void cdq()
    {
	if ((cpu.eax & 0x80000000) != 0)
	    cpu.edx = -1;
	else
	    cpu.edx = 0;
    }

    protected final void clc()
    {
	cpu.eflagsCarry = false;
    }

    protected final void cld()
    {
	cpu.eflagsDirection = false;
    }

    protected final void cli()
    {
	cpu.eflagsInterruptEnable = cpu.eflagsInterruptEnableSoon = false;
    }

    protected final void cmovb_o32() throws ProcessorException
    {
	if (cpu.eflagsCarry) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovnb_o32() throws ProcessorException
    {
	if (!cpu.eflagsCarry) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovbe_o32() throws ProcessorException
    {
	if (cpu.eflagsCarry || cpu.eflagsZero) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovnbe_o32() throws ProcessorException
    {
	if (!cpu.eflagsCarry && !cpu.eflagsZero) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovl_o32() throws ProcessorException
    {
	if (cpu.eflagsSign != cpu.eflagsOverflow) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovnl_o32() throws ProcessorException
    {
	if (cpu.eflagsSign == cpu.eflagsOverflow) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovle_o32() throws ProcessorException
    {
	if (cpu.eflagsZero || (cpu.eflagsSign != cpu.eflagsOverflow)) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovnle_o32() throws ProcessorException
    {
	if ((!cpu.eflagsZero) && (cpu.eflagsSign == cpu.eflagsOverflow)) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovo_o32() throws ProcessorException
    {
	if (cpu.eflagsOverflow) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovno_o32() throws ProcessorException
    {
	if (!cpu.eflagsOverflow) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovp_o32() throws ProcessorException
    {
	if (cpu.eflagsParity) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovnp_o32() throws ProcessorException
    {
	if (!cpu.eflagsParity) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovs_o32() throws ProcessorException
    {
	if (cpu.eflagsSign) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovns_o32() throws ProcessorException
    {
	if (!cpu.eflagsSign) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovz_o32() throws ProcessorException
    {
	if (cpu.eflagsZero) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmovnz_o32() throws ProcessorException
    {
	if (!cpu.eflagsZero) {
	    int src = getIntOperand();
	    setIntOperand(src);
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmp_o8() throws ProcessorException
    {
	int o2 = 0xff & getByteOperand();
	int o1 = 0xff & getByteOperand();

	int temp = o1 - o2;

	//flags
        standardFlags((byte)temp);

        cpu.eflagsCarry = (temp & (~0xFF)) !=0;
        if ((byte)o2 < 0) // +VE Swing
            cpu.eflagsOverflow = ((byte)temp < (byte)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((byte)temp > (byte)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ temp) & 0x10) != 0);
    }

    protected final void cmp_o16() throws ProcessorException
    {
	int o2 = 0xffff & getShortOperand();
	int o1 = 0xffff & getShortOperand();

	int temp = o1 - o2;

	//flags
        standardFlags((short)temp);

        cpu.eflagsCarry = (temp & (~0xffff)) !=0;
        if ((short)o2 < 0) // +VE Swing
            cpu.eflagsOverflow = ((short)temp < (short)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((short)temp > (short)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ temp) & 0x10) != 0);
    }

    protected final void cmp_o32() throws ProcessorException
    {
	long o2 = 0xffffffffl & getIntOperand();
	long o1 = 0xffffffffl & getIntOperand();

	long temp = o1 - o2;

	//flags
        standardFlags((int)temp);

        cpu.eflagsCarry = (temp & (~0xffffffffl)) !=0;
        if ((int)o2 < 0) // +VE Swing
            cpu.eflagsOverflow = ((int)temp < (int)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((int)temp > (int)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ temp) & 0x10) != 0);
    }

    protected final void cmps_o8_a16() throws ProcessorException
    {
	int si = (cpu.esi & 0xffff);
	int di = (cpu.edi & 0xffff);

	int b = 0xff & getByteOperand();
	int a = 0xff & getByteOperand();

	if (cpu.eflagsDirection) {
	    si -= 1;
	    di -= 1;
	} else {
	    si += 1;
	    di += 1;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
    
	int result = a - b;
	standardFlags((byte) result);
	
	cpu.eflagsCarry = (result & (~0xFF)) !=0;
	if ((byte)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((byte)result < (byte)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((byte)result > (byte)a);

	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);	
    }

    protected final void cmps_o16_a16() throws ProcessorException
    {
	int si = (cpu.esi & 0xffff);
	int di = (cpu.edi & 0xffff);

	int b = 0xffff & getShortOperand();
	int a = 0xffff & getShortOperand();

	if (cpu.eflagsDirection) {
	    si -= 2;
	    di -= 2;
	} else {
	    si += 2;
	    di += 2;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
    
	int result = a - b;
	standardFlags((short) result);
	
	cpu.eflagsCarry = (result & (~0xFFFF)) !=0;
	if ((short)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((short)result < (short)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((short)result > (short)a);

	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);	
    }

    protected final void cmps_o32_a16()
    {
	int si = (cpu.esi & 0xffff);
	int di = (cpu.edi & 0xffff);

	long b = 0xffffffffl & getIntOperand();
	long a = 0xffffffffl & getIntOperand();

	if (cpu.eflagsDirection) {
	    si -= 4;
	    di -= 4;
	} else {
	    si += 4;
	    di += 4;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);    

	long result = a - b;
	standardFlags((int) result);
	
	cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	if ((int)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((int)result < (int)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((int)result > (int)a);

	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);	
    }

    protected final void cmps_o8_a32()
    {
	int b = 0xff & getByteOperand();
	int a = 0xff & getByteOperand();

	if (cpu.eflagsDirection) {
	    cpu.esi -= 1;
	    cpu.edi -= 1;
	} else {
	    cpu.esi += 1;
	    cpu.edi += 1;
	}
    
	int result = a - b;
	standardFlags((byte) result);
	
	cpu.eflagsCarry = (result & (~0xFF)) !=0;
	if ((byte)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((byte)result < (byte)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((byte)result > (byte)a);

	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);	
    }

    protected final void cmps_o16_a32()
    {
	int b = 0xffff & getShortOperand();
	int a = 0xffff & getShortOperand();

	if (cpu.eflagsDirection) {
	    cpu.esi -= 2;
	    cpu.edi -= 2;
	} else {
	    cpu.esi += 2;
	    cpu.edi += 2;
	}
    
	int result = a - b;
	standardFlags((short) result);
	
	cpu.eflagsCarry = (result & (~0xffff)) !=0;
	if ((short)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((short)result < (short)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((short)result > (short)a);

	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);	
    }

    protected final void cmps_o32_a32()
    {
	long b = 0xffffffffl & getIntOperand();
	long a = 0xffffffffl & getIntOperand();

	if (cpu.eflagsDirection) {
	    cpu.esi -= 4;
	    cpu.edi -= 4;
	} else {
	    cpu.esi += 4;
	    cpu.edi += 4;
	}
    
	long result = a - b;
	standardFlags((int) result);
	
	cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	if ((int)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((int)result < (int)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((int)result > (int)a);

	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);	
    }

    protected final void cmpxchg_o8()
    {
	byte dest = getByteOperand();

	if ((byte)cpu.eax == dest) {
	    setByteOperand(getByteOperand());
	    cpu.eflagsZero = true;
	} else {
	    cpu.eax = (cpu.eax & ~0xff) | (0xff & dest);
	    cpu.eflagsZero = false;
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmpxchg_o16()
    {
	short dest = getShortOperand();

	if ((short)cpu.eax == dest) {
	    setShortOperand(getShortOperand());
	    cpu.eflagsZero = true;
	} else {
	    cpu.eax = (cpu.eax & ~0xffff) | (0xffff & dest);
	    cpu.eflagsZero = false;
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmpxchg_o32()
    {
	int dest = getIntOperand();

	if (cpu.eax == dest) {
	    setIntOperand(getIntOperand());
	    cpu.eflagsZero = true;
	} else {
	    cpu.eax = dest;
	    cpu.eflagsZero = false;
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void cmpxchg8b()
    {
	long data = getLongOperand();
	long reg = ((0xffffffffl & cpu.edx) << 32) | (0xffffffffl & cpu.eax);

	if (reg == data) {
	    setLongOperand(((0xffffffffl & cpu.ecx) << 32) | (0xffffffffl & cpu.ebx));
	    cpu.eflagsZero = true;
	} else {
	    cpu.eax = (int)data;
	    cpu.edx = (int)(data >>> 32);
	    cpu.eflagsZero = false;
	    skipOperand();
	}
    }

    protected final void cwde()
    {
	cpu.eax = ((short)cpu.eax);
    }

    protected final void daa()
    {
	int al = cpu.eax & 0xff;
	boolean newCF;
	if (((cpu.eax & 0xf) > 0x9) || cpu.eflagsAuxiliaryCarry) {
            al += 6;
            cpu.eflagsAuxiliaryCarry = true;
        } else
            cpu.eflagsAuxiliaryCarry = false;
	
        if (((al & 0xff) > 0x9f) || cpu.eflagsCarry) {
	    al += 0x60;
            newCF = true;
	} else
            newCF = false;
	
	cpu.eax = (cpu.eax & ~0xff) | (0xff & al);
	standardFlags((byte)al);
	cpu.eflagsCarry = newCF;
    }

    protected final void das()
    {
	boolean tempCF = false;
	int tempAL = 0xff & cpu.eax;
 	if (((tempAL & 0xf) > 0x9) || cpu.eflagsAuxiliaryCarry) {
	    cpu.eflagsAuxiliaryCarry = true;
	    cpu.eax = (cpu.eax & ~0xff) | ((cpu.eax - 0x06) & 0xff);
	    tempCF = (tempAL < 0x06) || cpu.eflagsCarry;
	}
	
        if ((tempAL > 0x99) || cpu.eflagsCarry) {
            cpu.eax = (cpu.eax & ~0xff) | ((cpu.eax - 0x60) & 0xff);
	    tempCF = true;
	}

	standardFlags((byte)cpu.eax);
	cpu.eflagsCarry = tempCF;
    }

    protected final void dec_o8() throws ProcessorException
    {
        byte result = (byte)(getByteOperand() - 1);
	setByteOperand(result);

	boolean temp = cpu.eflagsCarry;
	standardFlags(result);
	cpu.eflagsCarry = temp;
        cpu.eflagsOverflow = (result == (short)0x7f);
        cpu.eflagsAuxiliaryCarry = ((result & 0xf) == 0xf);
    }

    protected final void dec_o16() throws ProcessorException
    {
        short result = (short)(getShortOperand() - 1);
	setShortOperand(result);

	boolean temp = cpu.eflagsCarry;
	standardFlags(result);
	cpu.eflagsCarry = temp;
        cpu.eflagsOverflow = (result == (short)0x7fff);
        cpu.eflagsAuxiliaryCarry = ((result & 0xf) == 0xf);
    }

    protected final void dec_o32() throws ProcessorException
    {
        int result = (getIntOperand() - 1);
	setIntOperand(result);

	boolean temp = cpu.eflagsCarry;
	standardFlags(result);
	cpu.eflagsCarry = temp;
        cpu.eflagsOverflow = (result == 0x7fffffff);
        cpu.eflagsAuxiliaryCarry = ((result & 0xf) == 0xf);
    }

    protected final void div_o8() throws ProcessorException
    {
	int d = getByteOperand() & 0xff;

	if (d == 0) {
	    throw exceptionDE;
	}

	int x = (cpu.eax & 0xffff);

	int result = x / d;
	if (result > 0xff) {
	    throw exceptionDE;
	}
	int rem = x % d;
	rem <<= 8;
	cpu.eax = (cpu.eax & 0xffff0000) | (0xff & result) | (0xff00 & rem);
    }

    protected final void div_o16() throws ProcessorException
    {
	int d = getShortOperand() & 0xffff;

	if (d == 0) {
	    throw exceptionDE;
	}

	long x = (cpu.edx & 0xffffl);
	x <<= 16;
	x |= (cpu.eax & 0xffffl);

	long result = x / d;
	if (result > 0xffffl) {
	    throw exceptionDE;
	}
	long rem = x % d;
	cpu.eax = (cpu.eax & 0xffff0000) | (int)(result & 0xffff);
	cpu.edx = (cpu.edx & 0xffff0000) | (int)(rem & 0xffff);
    }

    protected final void div_o32() throws ProcessorException
    {
	long d = 0xffffffffl & getIntOperand();

        if (d==0)
            throw exceptionDE;

        long temp = (long)cpu.edx;
        temp <<= 32;
        temp |= (0xffffffffl & cpu.eax);

        // shift division by 1 so we don't involve sign bit
        long r2 = (temp & 1);
        long n2 = (temp >>> 1);  // unsigned right shift!

        // TODO:  check this algorithm!
        long q2 = n2 / d;
        long m2 = n2 % d;

        long q = (q2 << 1);
        long r = (m2 << 1) + r2;

        q += (r / d);
        r %= d;
        if (q > 0xffffffffl)
            throw exceptionDE;

        cpu.eax =  (int)q; //EAX is result
        cpu.edx =  (int)r;    //EDX is remainder
    }

    protected final void f2xm1()
    {
        double x = fpu.ST(0);
        validateOperand(x);
        double y = Math.expm1(x * Math.log(2.0));
        fpu.setST(0, y);
        checkResult(y);
    }

    protected final void fadd()
    {
	double y = getDoubleOperand();
	double x = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x + y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.POSITIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.NEGATIVE_INFINITY))
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
	setDoubleOperand(z);
        checkResult(z);
    }

    protected final void fadd_sr()
    {
	double y = getSingleOperand();
	double x = fpu.ST(0);

	validateOperand(x);
	validateOperand(y);

	double z = x + y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.POSITIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.NEGATIVE_INFINITY))
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
	fpu.setST(0, z);
        checkResult(z);
    }

    protected final void fadd_dr()
    {
	double y = getDoubleOperand();
	double x = fpu.ST(0);

	validateOperand(x);
	validateOperand(y);

	double z = x + y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.POSITIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.NEGATIVE_INFINITY))
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
	fpu.setST(0, z);
        checkResult(z);
    }

    protected final void faddp()
    {
	double y = getDoubleOperand();
	double x = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x + y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.POSITIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.NEGATIVE_INFINITY))
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
	setDoubleOperand(z);
	fpu.pop();
        checkResult(z);
    }

    protected final void fchs()
    {
        fpu.setST(0, - fpu.ST(0));
    }

    protected final void fclex()
    {
        fpu.clearExceptions();
    }

    protected final void fcom()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (unordered)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        } else {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;
    }

    protected final void fcomp()
    {
        fcom();
        fpu.pop();
    }

    protected final void fcompp()
    {
        fcom();
        fpu.pop();
        fpu.pop();
    }
    
    protected final void fcom_sr()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (unordered)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        } else {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;

        fpu.pop(); // ignore popped result
    }

    protected final void fcom_dr()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (unordered)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        } else {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;

        fpu.pop(); // ignore popped result
    }

    protected final void fcomp_sr()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (unordered)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        } else {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;

        fpu.pop(); // ignore popped result
    }

    protected final void fcomp_dr()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (unordered)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        } else {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;

        fpu.pop(); // ignore popped result
    }

    protected final void fdiv()
    {
	double y = getDoubleOperand();
	double x = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	setDoubleOperand(z);
	checkResult(z);
    }

    protected final void fdiv_dr()
    {
	double y = getDoubleOperand();
	double x = fpu.ST(0);

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fdiv_sr()
    {
	double y = getSingleOperand();
	double x = fpu.ST(0);

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fdivp()
    {
	double y = getDoubleOperand();
	double x = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	setDoubleOperand(z);
	fpu.pop();
	checkResult(z);
    }

    protected final void fdivr()
    {
	double x = getDoubleOperand();
	double y = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	setDoubleOperand(z);
	checkResult(z);
    }

    protected final void fdivr_sr()
    {
	double y = fpu.ST(0);
	double x = getSingleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fdivr_dr()
    {
	double y = fpu.ST(0);
	double x = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fdivrp()
    {
	double x = getDoubleOperand();
	double y = getDoubleOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	setDoubleOperand(z);
	fpu.pop();
	checkResult(z);
    }

    protected final void fidivr_wi()
    {
	double y = fpu.ST(0);
	double x = getWordIntegerOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fidivr_di()
    {
	double y = fpu.ST(0);
	double x = getDoubleIntegerOperand();

	validateOperand(x);
	validateOperand(y);

	double z = x / y;

	if ((x == 0.0 && y == 0.0) || (Double.isInfinite(x) && Double.isInfinite(y)))
            fpu.setInvalidOperation();

	if (y == 0.0 && !Double.isNaN(x))
	    fpu.setZeroDivide();

	fpu.checkExceptions();
	fpu.setST(0, z);
	checkResult(z);
    }
    
    protected final void fild_wi()
    {
        double x = getWordIntegerOperand();
        validateOperand(x);
        fpu.push(x);
    }

    protected final void fild_di()
    {
        double x = getDoubleIntegerOperand();
        validateOperand(x);
        fpu.push(x);
    }

    protected final void fild_qi()
    {
        double x = getQuadIntegerOperand();
        validateOperand(x);
        fpu.push(x);
    }

    protected final void fist_wi()
    {
	double x = fpu.ST(0);
	validateOperand(x);
	setWordIntegerOperand(x);
    }

    protected final void fist_di()
    {
	double x = fpu.ST(0);
	validateOperand(x);
	setDoubleIntegerOperand(x);
    }

    protected final void fistp_wi()
    {
	double x = fpu.ST(0);
	validateOperand(x);
	setWordIntegerOperand(x);
	fpu.pop();
    }

    protected final void fistp_di()
    {
	double x = fpu.ST(0);
	validateOperand(x);
	setDoubleIntegerOperand(x);
	fpu.pop();
    }

    protected final void fistp_qi()
    {
	double x = fpu.ST(0);
	validateOperand(x);
	setQuadIntegerOperand(x);
	fpu.pop();
    }

    protected final void fld()
    {
	double x = getDoubleOperand();
	validateOperand(x);
	fpu.push(x);
    }

    protected final void fld_dr()
    {
	double x = getDoubleOperand();
	validateOperand(x);
	fpu.push(x);
    }

    protected final void fld_sr()
    {
	double x = getSingleOperand();
	validateOperand(x);
	fpu.push(x);
    }

    protected final void fldz()
    {
	validateOperand(0.0);
	fpu.push(0.0);
    }

    protected final void fld1()
    {
	validateOperand(1.0);
	fpu.push(1.0);
    }

    protected final void fldcw()
    {
	fpu.setControl(getShortOperand());
    }

    protected final void fmul()
    {
	double x = getDoubleOperand();
	double y = getDoubleOperand();
	validateOperand(x);
	validateOperand(y);

	double z = x * y;
	if ((Double.isInfinite(x) && y == 0.0) || (Double.isInfinite(y) && x == 0.0)) {
	    fpu.setInvalidOperation();
	    fpu.checkExceptions();
        }

	setDoubleOperand(z);
	checkResult(z);
    }

    protected final void fmul_sr()
    {
	double x = getSingleOperand();
	double y = fpu.ST(0);
	validateOperand(x);
	validateOperand(y);

	double z = x * y;
	if ((Double.isInfinite(x) && y == 0.0) || (Double.isInfinite(y) && x == 0.0)) {
	    fpu.setInvalidOperation();
	    fpu.checkExceptions();
        }

	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fmul_dr()
    {
	double x = getDoubleOperand();
	double y = fpu.ST(0);
	validateOperand(x);
	validateOperand(y);

	double z = x * y;
	if ((Double.isInfinite(x) && y == 0.0) || (Double.isInfinite(y) && x == 0.0)) {
	    fpu.setInvalidOperation();
	    fpu.checkExceptions();
        }

	fpu.setST(0, z);
	checkResult(z);
    }

    protected final void fmulp()
    {
	double x = getDoubleOperand();
	double y = getDoubleOperand();
	validateOperand(x);
	validateOperand(y);

	double z = x * y;
	if ((Double.isInfinite(x) && y == 0.0) || (Double.isInfinite(y) && x == 0.0)) {
	    fpu.setInvalidOperation();
	    fpu.checkExceptions();
        }

	setDoubleOperand(z);
	fpu.pop();
	checkResult(z);
    }

    protected final void fninit()
    {
	FpuState64 s = (FpuState64)fpu;
	s.init();
    }

    protected final void fprem()
    {
        double x = fpu.ST(0);
        double y = fpu.ST(1);
        validateOperand(x);
        validateOperand(y);

        // check to see whether should compute partial remainder
        int e1 = (int)((Double.doubleToLongBits(x) >> 52) & 0x7ff);
        int e2 = (int)((Double.doubleToLongBits(y) >> 52) & 0x7ff);
        int d = e1 - e2;
        if (d < 64)
        {
            // full remainder
            fpu.conditionCode &= ~4; // clear C2
            double z = x % y;  // with truncating division	
            checkResult(z);
            // compute least significant bits -> C0 C3 C1
            long i = (long)z;
            fpu.conditionCode &= 4;
            if ((i & 1) != 0) fpu.conditionCode |= 2;
            if ((i & 2) != 0) fpu.conditionCode |= 8;
            if ((i & 4) != 0) fpu.conditionCode |= 1;
            fpu.setST(0, z);
        }
        else
        {
            // partial remainder
            fpu.conditionCode |= 4; // set C2
            int n = 63; // implementation dependent in manual
            double f = Math.pow(2.0, (double)(d - n));
            double qq = Math.signum(x) * Math.floor(Math.abs((x / y) / f));
            double z = x - (y * qq * f);
            fpu.setST(0, z);
        }
    }

    protected final void fprem1()
    {
        double x = fpu.ST(0);
        double y = fpu.ST(1);
        validateOperand(x);
        validateOperand(y);

        // check to see whether should compute partial remainder
        int e1 = (int)((Double.doubleToLongBits(x) >> 52) & 0x7ff);
        int e2 = (int)((Double.doubleToLongBits(y) >> 52) & 0x7ff);
        int d = e1 - e2;
        if (d < 64)
        {
            // full remainder
            fpu.conditionCode &= ~4; // clear C2
	    double z = Math.IEEEremainder(x, y);
            checkResult(z);
            // compute least significant bits -> C0 C3 C1
            long i = (long)z;
            fpu.conditionCode &= 4;
            if ((i & 1) != 0) fpu.conditionCode |= 2;
            if ((i & 2) != 0) fpu.conditionCode |= 8;
            if ((i & 4) != 0) fpu.conditionCode |= 1;
            fpu.setST(0, z);
        }
        else
        {
            // partial remainder
            fpu.conditionCode |= 4; // set C2
            int n = 63; // implementation dependent in manual
            double f = Math.pow(2.0, (double)(d - n));
            double qq = Math.signum(x) * Math.floor(Math.abs((x / y) / f));
            double z = x - (y * qq * f);
            fpu.setST(0, z);
        }
    }

    protected final void frndint()
    {
        double x = fpu.ST(0);
        validateOperand(x);
        double y;
        if (Double.isInfinite(x)) 
            return; // preserve infinities

        switch(fpu.getRoundingControl())
        {
        case FpuState.FPU_ROUNDING_CONTROL_EVEN:
            y = Math.rint(x);
            break;
        case FpuState.FPU_ROUNDING_CONTROL_DOWN:
            y = Math.floor(x);
            break;
        case FpuState.FPU_ROUNDING_CONTROL_UP:
            y = Math.ceil(x);
            break;
        case FpuState.FPU_ROUNDING_CONTROL_TRUNCATE:
            y = Math.signum(x) * Math.floor(Math.abs(x));
            break;
        default:
            throw new IllegalStateException("Invalid rounding control value");
        }
        checkResult(y);
        fpu.setST(0, y);
        // this opcode throws #P exceptions if x isn't an integer,
        // but I can't quite figure out if it actually modifies
        // ST(0) nonetheless (TODO)
    }

    protected final void fscale()
    {
        double x = fpu.ST(0);
        double y = fpu.ST(1);
        validateOperand(x);
        validateOperand(y);
        double z = x * Math.pow(2.0, y);
        checkResult(z);
        fpu.setST(0, z);
    }

    protected final void fsincos()
    {
        double x = fpu.ST(0);
        validateOperand(x);
        // check range
        fpu.conditionCode &= 4;
        if (Math.abs(x) >= Math.pow(2.0,63.0)) {
            fpu.conditionCode |= 4; // set C2
            return;
        }
        double y = Math.sin(x);
        double z = Math.cos(x);
        checkResult(y);
        checkResult(z);
        fpu.setST(0, y);
        fpu.push(z);
    }

    protected final void fsqrt()
    {
        double x = fpu.ST(0);
        validateOperand(x);
        if (x < 0.0)
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
        double y = Math.sqrt(x);
        checkResult(y);
        fpu.setST(0, y);
    }

    protected final void fst_dr()
    {
        double x = fpu.ST(0);
        validateOperand(x); // ?

        // need to check underflow before the store is actually accomplished
	setDoubleOperand(x);
    }

    protected final void fstp()
    {
        double x = fpu.ST(0);
        validateOperand(x); // ?

        // need to check underflow before the store is actually accomplished
	setDoubleOperand(x);

	fpu.pop(); // ignore popped result
    }

    protected final void fstp_sr()
    {
        double x = fpu.ST(0);
        validateOperand(x); // ?

        // need to check underflow before the store is actually accomplished
	setSingleOperand(x);
	
	fpu.pop(); // ignore popped result
    }

    protected final void fstp_dr()
    {
        double x = fpu.ST(0);
        validateOperand(x); // ?

        // need to check underflow before the store is actually accomplished
	setDoubleOperand(x);
	
	fpu.pop(); // ignore popped result
    }

    protected final void fstcw() throws ProcessorException
    {
	setShortOperand((short)fpu.getControl());
    }

    protected final void fstsw() throws ProcessorException
    {
	setShortOperand((short)fpu.getStatus());
    }

    protected final void fsub()
    {
	double x = getDoubleOperand();
        double y = getDoubleOperand();        
        validateOperand(x);
        validateOperand(y);
        double z = y - x;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
	setDoubleOperand(z);
        checkResult(z);
    }

    protected final void fsub_sr()
    {
        double x = fpu.ST(0);
	double y = getSingleOperand();
	validateOperand(x);
        validateOperand(y);
        double z = x - y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }

	fpu.setST(0, z);
        checkResult(z);
    }

    protected final void fsub_dr()
    {
	double x = fpu.ST(0);
        double y = getDoubleOperand();       
        validateOperand(x);
        validateOperand(y);
        double z = x - y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }

	fpu.setST(0, z);
        checkResult(z);
    }

    protected final void fsubp()
    {
	double x = getDoubleOperand();
        double y = getDoubleOperand();       
        validateOperand(x);
        validateOperand(y);
        double z = y - x;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }
	setDoubleOperand(z);
	fpu.pop();
        checkResult(z);
    }

    protected final void fsubr()
    {
        double x = getDoubleOperand();       
	double y = getDoubleOperand();
        validateOperand(x);
        validateOperand(y);
        double z = x - y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }

	setDoubleOperand(z);
        checkResult(z);
    }

    protected final void fsubr_sr()
    {
        double x = getSingleOperand();
	double y = fpu.ST(0);
        validateOperand(x);
        validateOperand(y);
        double z = x - y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }

	fpu.setST(0, z);
        checkResult(z);
    }

    protected final void fsubr_dr()
    {
        double x = getDoubleOperand();
	double y = fpu.ST(0);
        validateOperand(x);
        validateOperand(y);
        double z = x - y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }

	fpu.setST(0, z);
        checkResult(z);
    }

    protected final void fsubrp()
    {
        double x = getDoubleOperand();       
	double y = getDoubleOperand();
        validateOperand(x);
        validateOperand(y);
        double z = x - y;

        if ((x == Double.NEGATIVE_INFINITY && y == Double.NEGATIVE_INFINITY) ||
            (x == Double.POSITIVE_INFINITY && y == Double.POSITIVE_INFINITY)) {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        }

	setDoubleOperand(z);
	fpu.pop(); // ignore popped result
        checkResult(z);
    }

    protected final void ftst()
    {
        double x = fpu.ST(0);
        double y = 0.0;
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);

        if (Double.isNaN(x))
        {
            fpu.setInvalidOperation();
            fpu.checkExceptions();
        } else {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }
        
	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;
    }

    protected final void fucom()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (!unordered)
        {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;
    }

    protected final void fucomp()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (!unordered)
        {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;

	fpu.pop();
    }

    protected final void fucompp()
    {
        // get() will check for SNaNs, but not QNaNs
        double x = fpu.ST(0);
        double y = getDoubleOperand();
        // initial condition:  set flags to "unordered"
        int newcode = 0xd;

        validateOperand(x);
        validateOperand(y);

        boolean unordered = (Double.isNaN(x) || Double.isNaN(y));
        if (!unordered)
        {
            if (x > y) newcode = 0;
            else if (x < y) newcode = 1;
            else newcode = 8;
        }

	fpu.conditionCode &= 2;
	fpu.conditionCode |= newcode;

	fpu.pop();
	fpu.pop();
    }

    protected final void fxch()
    {
	double x = fpu.ST(0);
	double y = getDoubleOperand();
	fpu.setST(0, y);
	setDoubleOperand(x);
    }

    protected final void fyl2x()
    {
        double x = fpu.ST(0);
        double y = fpu.ST(1);

        // check inputs
        validateOperand(x);
        validateOperand(y);
        // check combinations of inputs


        if (x == 0.0 && !Double.isInfinite(y)) fpu.setZeroDivide();

        if ((x == 1.0 && Double.isInfinite(y)) || (x == 0.0 && y == 0.0) || (x < 0.0))
            fpu.setInvalidOperation();
        
        fpu.checkExceptions();
        // do the calculation (masked behavior is same as Java behavior)
        double z = y * Math.log(x) / Math.log(2.0);
        
        checkResult(z);
        fpu.setST(1, z);
        fpu.pop(); // ignore result of pop
    }

    protected final void fyl2xp1()
    {
        double x = fpu.ST(0);
        double y = fpu.ST(1);

        // check inputs
        validateOperand(x);
        validateOperand(y);
        // check combinations of inputs

        if (x == 0.0 && Double.isInfinite(y))
            fpu.setInvalidOperation();

        fpu.checkExceptions();
        // do the calculation (masked behavior is same as Java behavior)
        double z = y * Math.log(x + 1.0) / Math.log(2.0);
        
        checkResult(z);
        fpu.setST(1, z);
        fpu.pop(); // ignore result of pop
    }

    protected final void idiv_o8() throws ProcessorException
    {
	byte src = getByteOperand();
        if (src == 0) {
	    throw exceptionDE;
        }
        short temp = (short)cpu.eax;
        int result = temp / src;
        int rem = temp % src;
        if ((result > Byte.MAX_VALUE) || (result < Byte.MIN_VALUE)) {
	    throw exceptionDE;
        }
	
        cpu.eax = (cpu.eax & ~0xffff) | (0xff & result) | ((0xff & rem) << 8); //AH is remainder
    }

    protected final void idiv_o16() throws ProcessorException
    {
	short src = getShortOperand();
        if (src == 0) {
	    throw exceptionDE;
        }
        int temp = (cpu.edx << 16) | (cpu.eax & 0xffff);
        int result = temp / (int)src;
        int rem = temp % src;

        if ((result > Short.MAX_VALUE) || (result < Short.MIN_VALUE))
	    throw exceptionDE;
		
        cpu.eax = (cpu.eax & ~0xffff) | (0xffff & result); //AX is result
        cpu.edx = (cpu.edx & ~0xffff) | (0xffff & rem);    //DX is remainder
    }

    protected final void idiv_o32() throws ProcessorException
    {
	int src = getIntOperand();
	if (src == 0)
	    throw exceptionDE;

	long temp = (0xffffffffl & cpu.edx) << 32;
	temp |= (0xffffffffl & cpu.eax);
	long result = temp / src;
	if ((result > Integer.MAX_VALUE) || (result < Integer.MIN_VALUE))
	    throw exceptionDE;

	long rem = temp % src;
	
	cpu.eax =  (int)result; //EAX is result
	cpu.edx =  (int)rem;    //EDX is remainder
    }

    protected final void imul_o16() throws ProcessorException
    {
        // explicit casts to sign-extend the words
        int x = (int)getShortOperand();
	int y = (int)getShortOperand();
	int result = x * y;

        setShortOperand((short)result);
        cpu.eflagsOverflow = cpu.eflagsCarry = (result != (short)result);
    }

    protected final void imul_o32() throws ProcessorException
    {
        // explicit casts to sign-extend the words
        long x = (long)getIntOperand();
	long y = (long)getIntOperand();
	long result = x * y;

        setIntOperand((int)result);
        cpu.eflagsOverflow = cpu.eflagsCarry = (result != (int)result);
    }

    protected final void imul_rega_o8() throws ProcessorException
    {
        // explicit casts to sign-extend the bytes
        byte al = (byte)cpu.eax;
        int y = (int)getByteOperand();
	int result = al * y;

	cpu.eax = (cpu.eax & ~0xffff) | (result & 0xffff);

	//answer too wide for 8-bits?
	cpu.eflagsOverflow = cpu.eflagsCarry = (result != (byte)result);
    }

    protected final void imul_rega_o16() throws ProcessorException
    {
        // explicit casts to sign-extend the bytes
        short ax = (short)cpu.eax;
        int y = (int)getShortOperand();
	int result = ax * y;

        cpu.eax = (cpu.eax & ~0xffff) | (result & 0xffff);
	cpu.edx = (cpu.edx & ~0xffff) | (result >>> 16);

	//answer too wide for 16-bits?
	cpu.eflagsOverflow = cpu.eflagsCarry = (result != (short)result);
    }

    protected final void imul_rega_o32() throws ProcessorException
    {
        // explicit casts to sign-extend the bytes
        long eax = (long)cpu.eax;
        long y = (long)getIntOperand();
	long result = eax * y;
	
	cpu.eax = (int)result;
	cpu.edx = (int)(result >>> 32);

	//answer too wide for 32-bits?
	cpu.eflagsOverflow = cpu.eflagsCarry = (result != (int)result);
    }

    protected final void inc_o8() throws ProcessorException
    {
        byte result = (byte)(getByteOperand() + 1);
	setByteOperand(result);

	boolean temp = cpu.eflagsCarry;
	standardFlags(result);
	cpu.eflagsCarry = temp;
        cpu.eflagsOverflow = (result == (byte)0x80);
        cpu.eflagsAuxiliaryCarry = ((result & 0xf) == 0);
    }

    protected final void inc_o16() throws ProcessorException
    {
        short result = (short)(getShortOperand() + 1);
	setShortOperand(result);

	boolean temp = cpu.eflagsCarry;
	standardFlags(result);
	cpu.eflagsCarry = temp;
        cpu.eflagsOverflow = (result == (short)0x8000);
        cpu.eflagsAuxiliaryCarry = ((result & 0xf) == 0);
    }

    protected final void inc_o32() throws ProcessorException
    {
        int result = (getIntOperand() + 1);
	setIntOperand(result);

	boolean temp = cpu.eflagsCarry;
	standardFlags(result);
	cpu.eflagsCarry = temp;
        cpu.eflagsOverflow = (result == 0x80000000);
        cpu.eflagsAuxiliaryCarry = ((result & 0xf) == 0);
    }

    protected final void lea_o16_a16() throws ProcessorException
    {
	int address = getAddressOperand();
	setShortOperand((short)address);
    }

    protected final void lea_o32_a32() throws ProcessorException
    {
	int address = getAddressOperand();
	setIntOperand(address);
    }

    protected final void lods_o8_a16() throws ProcessorException
    {
	byte a = getByteOperand();

	setByteOperand(a);

	int si = (cpu.esi & 0xffff);
	if (cpu.eflagsDirection) {
	    --si;
	} else {
	    ++si;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);       
    }

    protected final void lods_o16_a16() throws ProcessorException
    {
	short a = getShortOperand();

	setShortOperand(a);

	int si = (cpu.esi & 0xffff);
	if (cpu.eflagsDirection) {
	    si -= 2;
	} else {
	    si += 2;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);      
    }

    protected final void lods_o32_a16() throws ProcessorException
    {
	int a = getIntOperand();

	setIntOperand(a);

	int si = (cpu.esi & 0xffff);
	if (cpu.eflagsDirection) {
	    si -= 4;
	} else {
	    si += 4;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);      
    }

    protected final void lods_o8_a32() throws ProcessorException
    {
	byte a = getByteOperand();
	setByteOperand(a);

	if (cpu.eflagsDirection) {
	    cpu.esi -= 1;
	} else {
	    cpu.esi += 1;
	}
    }

    protected final void lods_o16_a32() throws ProcessorException
    {
	short a = getShortOperand();
	setShortOperand(a);

	if (cpu.eflagsDirection) {
	    cpu.esi -= 2;
	} else {
	    cpu.esi += 2;
	}
    }

    protected final void lods_o32_a32() throws ProcessorException
    {
	int a = getIntOperand();
	setIntOperand(a);

	if (cpu.eflagsDirection) {
	    cpu.esi -= 4;
	} else {
	    cpu.esi += 4;
	}
    }

    protected final void mov_o8() throws ProcessorException
    {
	byte src = getByteOperand();
	setByteOperand(src);
    }

    protected final void mov_o16() throws ProcessorException
    {
	short src = getShortOperand();

	if (microcodes[microcodesPosition] == CS)
	    throw exceptionUD;

	setShortOperand(src);
    }

    protected final void mov_o32() throws ProcessorException
    {
	int src = getIntOperand();
	setIntOperand(src);
    }

    protected final void movs_o8_a16() throws ProcessorException
    {
	int si = (cpu.esi & 0xffff);
	int di = (cpu.edi & 0xffff);
	byte a = getByteOperand();
	setByteOperand(a);
	if (cpu.eflagsDirection) {
	    si -= 1;
	    di -= 1;
	} else {
	    si += 1;
	    di += 1;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);	
    }

    protected final void movs_o16_a16() throws ProcessorException
    {
	int si = (cpu.esi & 0xffff);
	int di = (cpu.edi & 0xffff);
	short a = getShortOperand();
	setShortOperand(a);
	if (cpu.eflagsDirection) {
	    si -= 2;
	    di -= 2;
	} else {
	    si += 2;
	    di += 2;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);	
    }

    protected final void movs_o32_a16() throws ProcessorException
    {
	int si = (cpu.esi & 0xffff);
	int di = (cpu.edi & 0xffff);
	int a = getIntOperand();
	setIntOperand(a);
	if (cpu.eflagsDirection) {
	    si -= 4;
	    di -= 4;
	} else {
	    si += 4;
	    di += 4;
	}
	cpu.esi &= ~0xffff;
	cpu.esi |= (si & 0xffff);
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
    }

    protected final void movs_o8_a32() throws ProcessorException
    {
	byte a = getByteOperand();
	setByteOperand(a);
	if (cpu.eflagsDirection) {
	    cpu.esi -= 1;
	    cpu.edi -= 1;
	} else {
	    cpu.esi += 1;
	    cpu.edi += 1;
	}
    }

    protected final void movs_o16_a32() throws ProcessorException
    {
	short a = getShortOperand();
	setShortOperand(a);
	if (cpu.eflagsDirection) {
	    cpu.esi -= 2;
	    cpu.edi -= 2;
	} else {
	    cpu.esi += 2;
	    cpu.edi += 2;
	}
    }

    protected final void movs_o32_a32() throws ProcessorException
    {
	int a = getIntOperand();
	setIntOperand(a);
	if (cpu.eflagsDirection) {
	    cpu.esi -= 4;
	    cpu.edi -= 4;
	} else {
	    cpu.esi += 4;
	    cpu.edi += 4;
	}
    }

    protected final void movsx_o16_o8() throws ProcessorException
    {
	short src = getByteOperand();
	setShortOperand(src);
    }

    protected final void movsx_o32_o8() throws ProcessorException
    {
	int src = getByteOperand();
	setIntOperand(src);
    }

    protected final void movsx_o16_o16() throws ProcessorException
    {
	short src = getShortOperand();
	setShortOperand(src);
    }

    protected final void movsx_o32_o16() throws ProcessorException
    {
	int src = getShortOperand();
	setIntOperand(src);
    }

    protected final void movzx_o16_o8() throws ProcessorException
    {
	short src = (short)(0xff & getByteOperand());
	setShortOperand(src);
    }

    protected final void movzx_o32_o8() throws ProcessorException
    {
	int src = 0xff & getByteOperand();
	setIntOperand(src);
    }

    protected final void movzx_o16_o16() throws ProcessorException
    {
	short src = getShortOperand();
	setShortOperand(src);
    }

    protected final void movzx_o32_o16() throws ProcessorException
    {
	int src = 0xffff & getShortOperand();
	setIntOperand(src);
    }

    protected final void mul_o8()
    {
	int x = cpu.eax & 0xff;
	int y = getByteOperand() & 0xff;

        int result = x * y;
        cpu.eax &= 0xffff0000;
	cpu.eax |= (result & 0xffff);

        cpu.eflagsOverflow = cpu.eflagsCarry = ((0xff00 & result) != 0);
    }

    protected final void mul_o16() throws ProcessorException
    {
	int x = cpu.eax & 0xffff;
	int y = 0xffff & getShortOperand();

        int result = x * y;
        cpu.eax = (cpu.eax & 0xffff0000) | (0xffff & result);
	result = result >> 16;
        cpu.edx = (cpu.edx & 0xffff0000) | (0xffff & result);

        cpu.eflagsOverflow = cpu.eflagsCarry = ((result & 0xffff) != 0);
    }

    protected final void mul_o32() throws ProcessorException
    {
	long x = cpu.eax & 0xffffffffl;
	long y = 0xffffffffl & getIntOperand();

        long result = x * y;
        cpu.eax = (int)result;
	result = result >>> 32;
	cpu.edx = (int)result;

        cpu.eflagsOverflow = cpu.eflagsCarry = (result != 0);
    }

    protected final void neg_o8() throws ProcessorException
    {
	byte source = getByteOperand();
	byte result = (byte)(-source);
	setByteOperand(result);

	cpu.eflagsCarry = (result != 0);
	cpu.eflagsOverflow = (result == (byte)0x80);

        cpu.eflagsAuxiliaryCarry = ((source & 0xf) > 0x0);
        cpu.eflagsZero = (result == 0);
        cpu.eflagsParity = parityMap[0xff & result];
	cpu.eflagsSign = (result < 0);
    }

    protected final void neg_o16() throws ProcessorException
    {
	short source = getShortOperand();
	short result = (short)(-source);
	setShortOperand(result);

	cpu.eflagsCarry = (result != 0);
	cpu.eflagsOverflow = (result == (short)0x8000);

        cpu.eflagsAuxiliaryCarry = ((source & 0xf) > 0x0);
        cpu.eflagsZero = (result == 0);
        cpu.eflagsParity = parityMap[0xff & result];
	cpu.eflagsSign = (result < 0);
    }

    protected final void neg_o32() throws ProcessorException
    {
	int source = getIntOperand();
	int result = -source;
	setIntOperand(result);

	cpu.eflagsCarry = (result != 0);
	cpu.eflagsOverflow = (result == 0x80000000);

        cpu.eflagsAuxiliaryCarry = ((source & 0xf) > 0x0);
        cpu.eflagsZero = (result == 0);
        cpu.eflagsParity = parityMap[0xff & result];
	cpu.eflagsSign = (result < 0);
    }

    protected final void not_o8() throws ProcessorException
    {
	byte result = getByteOperand();
	result = (byte)~result;
	setByteOperand(result);
    }

    protected final void not_o16() throws ProcessorException
    {
	short result = getShortOperand();
	result = (short)~result;
	setShortOperand(result);
    }

    protected final void not_o32() throws ProcessorException
    {
	int result = getIntOperand();
	result = ~result;
	setIntOperand(result);
    }

    protected final void or_o8() throws ProcessorException
    {
	byte result = (byte)(getByteOperand() |  getByteOperand());
	setByteOperand(result);
	standardFlags(result);
    }

    protected final void or_o16() throws ProcessorException
    {
	short result = (short)(getShortOperand() |  getShortOperand());
	setShortOperand(result);
	standardFlags(result);
    }

    protected final void or_o32() throws ProcessorException
    {
	int result = getIntOperand() |  getIntOperand();
	setIntOperand(result);
	standardFlags(result);
    }

    protected final void repe_cmps_o8_a16() throws ProcessorException
    {
        int a = 0;
        int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;

	    b = 0xff & getByteOperand();
	    a = 0xff & getByteOperand();

	    int si = (cpu.esi & 0xffff);
	    int di = (cpu.edi & 0xffff);	    
	    if (cpu.eflagsDirection) {
		si -= 1;
		di -= 1;
	    } else {
		si += 1;
		di += 1;
	    }
	    cpu.esi &= ~0xffff;
	    cpu.esi |= (si & 0xffff);
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a != b) break;
	}
	if (used) {
	    int result = a - b;

	    standardFlags((byte)result);
	    cpu.eflagsCarry = (result & (~0xff)) !=0;
	    if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);
	    
	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repe_cmps_o16_a16() throws ProcessorException
    {
        int a = 0;
        int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;

	    b = 0xffff & getShortOperand();
	    a = 0xffff & getShortOperand();

	    int si = (cpu.esi & 0xffff);
	    int di = (cpu.edi & 0xffff);	    
	    if (cpu.eflagsDirection) {
		si -= 2;
		di -= 2;
	    } else {
		si += 2;
		di += 2;
	    }
	    cpu.esi &= ~0xffff;
	    cpu.esi |= (si & 0xffff);
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a != b) break;
	}
	if (used) {
	    int result = a - b;

	    standardFlags((short)result);
	    cpu.eflagsCarry = (result & (~0xffff)) !=0;
	    if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);
	    

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repe_cmps_o8_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = 0;
        int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;
	    
	    b = 0xff & getByteOperand();
	    a = 0xff & getByteOperand();
	    
	    if (cpu.eflagsDirection) {
		cpu.esi -= 1;
		cpu.edi -= 1;
	    } else {
		cpu.esi += 1;
		cpu.edi += 1;
	    }
	    cpu.ecx--;
	    if (a != b) break;
	}
	if (used) {
	    int result = a - b;

	    standardFlags((byte)result);
	    cpu.eflagsCarry = (result & (~0xff)) !=0;
	    if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repe_cmps_o16_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = 0;
        int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;
	    
	    b = 0xffff & getShortOperand();
	    a = 0xffff & getShortOperand();
	    
	    if (cpu.eflagsDirection) {
		cpu.esi -= 2;
		cpu.edi -= 2;
	    } else {
		cpu.esi += 2;
		cpu.edi += 2;
	    }
	    cpu.ecx--;
	    if (a != b) break;
	}
	if (used) {
	    int result = a - b;

	    standardFlags((short)result);
	    cpu.eflagsCarry = (result & (~0xffff)) !=0;
	    if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repe_cmps_o32_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        long a = 0;
        long b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;
	    
	    b = 0xffffffffl & getIntOperand();
	    a = 0xffffffffl & getIntOperand();
	    
	    if (cpu.eflagsDirection) {
		cpu.esi -= 4;
		cpu.edi -= 4;
	    } else {
		cpu.esi += 4;
		cpu.edi += 4;
	    }
	    cpu.ecx--;
	    if (a != b) break;
	}
	if (used) {
	    long result = a - b;

	    standardFlags((int)result);
	    cpu.eflagsCarry = ((result & ~0xffffffffl) != 0);
	    if ((int)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((int)result < (int)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((int)result > (int)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repne_cmps_o8_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = 0;
        int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;
	    
	    b = 0xff & getByteOperand();
	    a = 0xff & getByteOperand();
	    
	    if (cpu.eflagsDirection) {
		cpu.esi -= 1;
		cpu.edi -= 1;
	    } else {
		cpu.esi += 1;
		cpu.edi += 1;
	    }
	    cpu.ecx--;
	    if (a == b) break;
	}
	if (used) {
	    int result = a - b;

	    standardFlags((byte)result);
	    cpu.eflagsCarry = (result & (~0xff)) !=0;
	    if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repne_cmps_o16_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = 0;
        int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;
	    
	    b = 0xffff & getShortOperand();
	    a = 0xffff & getShortOperand();
	    
	    if (cpu.eflagsDirection) {
		cpu.esi -= 2;
		cpu.edi -= 2;
	    } else {
		cpu.esi += 2;
		cpu.edi += 2;
	    }
	    cpu.ecx--;
	    if (a == b) break;
	}
	if (used) {
	    int result = a - b;

	    standardFlags((short)result);
	    cpu.eflagsCarry = (result & (~0xffff)) !=0;
	    if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void repne_cmps_o32_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        long a = 0;
        long b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    used = true;
	    microcodesPosition = operandAddress;
	    
	    b = 0xffffffffl & getIntOperand();
	    a = 0xffffffffl & getIntOperand();
	    
	    if (cpu.eflagsDirection) {
		cpu.esi -= 4;
		cpu.edi -= 4;
	    } else {
		cpu.esi += 4;
		cpu.edi += 4;
	    }
	    cpu.ecx--;
	    if (a == b) break;
	}
	if (used) {
	    long result = a - b;

	    standardFlags((int)result);
	    cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	    if ((int)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((int)result < (int)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((int)result > (int)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
        }
    }

    protected final void rep_lods_o8_a32()
    {
	executeCount += cpu.ecx - 1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();

	while (cpu.ecx != 0) {
	    microcodesPosition = operandAddress;
	    setByteOperand(getByteOperand());

	    if (cpu.eflagsDirection) {
		cpu.esi -= 1;
	    } else {
		cpu.esi += 1;
	    }
	    cpu.ecx--;
	}
    }

    protected final void rep_lods_o16_a32()
    {
	executeCount += cpu.ecx - 1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();

	while (cpu.ecx != 0) {
	    microcodesPosition = operandAddress;
	    setShortOperand(getShortOperand());

	    if (cpu.eflagsDirection) {
		cpu.esi -= 2;
	    } else {
		cpu.esi += 2;
	    }
	    cpu.ecx--;
	}
    }

    protected final void rep_lods_o32_a32()
    {
	executeCount += cpu.ecx - 1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();

	while (cpu.ecx != 0) {
	    microcodesPosition = operandAddress;
	    setIntOperand(getIntOperand());

	    if (cpu.eflagsDirection) {
		cpu.esi -= 4;
	    } else {
		cpu.esi += 4;
	    }
	    cpu.ecx--;
	}
    }

    protected final void rep_movs_o8_a16() throws ProcessorException
    {
	int operandsAddress = microcodesPosition;
	skipOperand();
	skipOperand();
	while ((cpu.ecx & 0xffff) != 0) {
	    microcodesPosition = operandsAddress;
	    // TODO:  put interrupt handling here, I think
	    int si = (cpu.esi & 0xffff);
	    int di = (cpu.edi & 0xffff);

	    byte a = getByteOperand();
	    setByteOperand(a);

	    if (cpu.eflagsDirection) {
		si -= 1;
		di -= 1;
	    } else {
		si += 1;
		di += 1;
	    }
	    cpu.esi &= ~0xffff;
	    cpu.esi |= (si & 0xffff);
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	}
    }

    protected final void rep_movs_o16_a16() throws ProcessorException
    {
	executeCount += cpu.ecx - 1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
        executeCount += (cpu.ecx & 0xffff) -1;

	while ((cpu.ecx & 0xffff) != 0) {
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int si = (cpu.esi & 0xffff);
	    int di = (cpu.edi & 0xffff);

	    short a = getShortOperand();
	    setShortOperand(a);

	    if (cpu.eflagsDirection) {
		si -= 2;
		di -= 2;
	    } else {
		si += 2;
		di += 2;
	    }
	    cpu.esi &= ~0xffff;
	    cpu.esi |= (si & 0xffff);
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	}
    }

    protected final void rep_movs_o32_a16() throws ProcessorException
    {
	int operandsAddress = microcodesPosition;
	skipOperand();
	skipOperand();
	while ((cpu.ecx & 0xffff) != 0) {
	    microcodesPosition = operandsAddress;
	    // TODO:  put interrupt handling here, I think
	    int si = (cpu.esi & 0xffff);
	    int di = (cpu.edi & 0xffff);

	    int a = getIntOperand();
	    setIntOperand(a);

	    if (cpu.eflagsDirection) {
		si -= 4;
		di -= 4;
	    } else {
		si += 4;
		di += 4;
	    }
	    cpu.esi &= ~0xffff;
	    cpu.esi |= (si & 0xffff);
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	}
    }

    protected final void rep_movs_o8_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
	while (cpu.ecx != 0) {
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    byte a = getByteOperand();
	    setByteOperand(a);

	    if (cpu.eflagsDirection) {
		cpu.esi -= 1;
		cpu.edi -= 1;
	    } else {
		cpu.esi += 1;
		cpu.edi += 1;
	    }
	    cpu.ecx--;
	}
    }

    protected final void rep_movs_o16_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
	while (cpu.ecx != 0) {
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    short a = getShortOperand();
	    setShortOperand(a);

	    if (cpu.eflagsDirection) {
		cpu.esi -= 2;
		cpu.edi -= 2;
	    } else {
		cpu.esi += 2;
		cpu.edi += 2;
	    }
	    cpu.ecx--;
	}
    }

    protected final void rep_movs_o32_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
	int operandAddress = microcodesPosition;
	skipOperand();
	skipOperand();
	while (cpu.ecx != 0) {
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int a = getIntOperand();
	    setIntOperand(a);

	    if (cpu.eflagsDirection) {
		cpu.esi -= 4;
		cpu.edi -= 4;
	    } else {
		cpu.esi += 4;
		cpu.edi += 4;
	    }
	    cpu.ecx--;
	}
    }

    protected final void repe_scas_o8_a16() throws ProcessorException
    {
        int a = getByteOperand() & 0xff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);

	    b = getByteOperand() & 0xff;

	    if (cpu.eflagsDirection) {
		di -= 1;
	    } else {
		di += 1;
	    }
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a != b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((byte)result);
            cpu.eflagsCarry = (result & (~0xFF)) !=0;
            if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repe_scas_o16_a16() throws ProcessorException
    {
        int a = getShortOperand() & 0xffff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);

	    b = getShortOperand() & 0xffff;

	    if (cpu.eflagsDirection) {
		di -= 2;
	    } else {
		di += 2;
	    }
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a != b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((short)result);
            cpu.eflagsCarry = (result & (~0xffff)) !=0;
            if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repe_scas_o32_a16() throws ProcessorException
    {
        long a = getIntOperand() & 0xffffffffl;
	long b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);

	    b = getIntOperand() & 0xffffffffl;

	    if (cpu.eflagsDirection) {
		di -= 4;
	    } else {
		di += 4;
	    }
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a != b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            long result = a - b;
	    standardFlags((int)result);
            cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
            if ((int)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((int)result < (int)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((int)result > (int)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repe_scas_o8_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = getByteOperand() & 0xff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    b = getByteOperand() & 0xff;

	    if (cpu.eflagsDirection) {
		cpu.edi -= 1;
	    } else {
		cpu.edi += 1;
	    }
	    cpu.ecx--;
	    if (a != b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((byte)result);
            cpu.eflagsCarry = (result & (~0xFF)) !=0;
            if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repe_scas_o16_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = getShortOperand() & 0xffff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    b = getShortOperand() & 0xffff;

	    if (cpu.eflagsDirection) {
		cpu.edi -= 2;
	    } else {
		cpu.edi += 2;
	    }
	    cpu.ecx--;
	    if (a != b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((short)result);
            cpu.eflagsCarry = (result & (~0xffff)) !=0;
            if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repe_scas_o32_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        long a = getIntOperand() & 0xffffffffl;
	long b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    b = getIntOperand() & 0xffffffffl;

	    if (cpu.eflagsDirection) {
		cpu.edi -= 4;
	    } else {
		cpu.edi += 4;
	    }
	    cpu.ecx--;
	    if (a != b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            long result = a - b;
	    standardFlags((int)result);
            cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
            if ((int)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((int)result < (int)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((int)result > (int)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repne_scas_o8_a16() throws ProcessorException
    {
        int a = getByteOperand() & 0xff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);

	    b = getByteOperand() & 0xff;

	    if (cpu.eflagsDirection) {
		di -= 1;
	    } else {
		di += 1;
	    }
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a == b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((byte)result);
            cpu.eflagsCarry = (result & (~0xFF)) !=0;
            if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repne_scas_o16_a16() throws ProcessorException
    {
        int a = getShortOperand() & 0xffff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);

	    b = getShortOperand() & 0xffff;

	    if (cpu.eflagsDirection) {
		di -= 2;
	    } else {
		di += 2;
	    }
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a == b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((short)result);
            cpu.eflagsCarry = (result & (~0xffff)) !=0;
            if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repne_scas_o32_a16() throws ProcessorException
    {
        long a = getIntOperand() & 0xffffffffl;
	long b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while ((cpu.ecx & 0xffff) != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);

	    b = getIntOperand() & 0xffffffffl;

	    if (cpu.eflagsDirection) {
		di -= 4;
	    } else {
		di += 4;
	    }
	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
	    if (a == b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            long result = a - b;
	    standardFlags((int)result);
            cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
            if ((int)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((int)result < (int)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((int)result > (int)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repne_scas_o8_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = getByteOperand() & 0xff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    b = getByteOperand() & 0xff;

	    if (cpu.eflagsDirection) {
		cpu.edi -= 1;
	    } else {
		cpu.edi += 1;
	    }
	    cpu.ecx--;
	    if (a == b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((byte)result);
            cpu.eflagsCarry = (result & (~0xFF)) !=0;
            if ((byte)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((byte)result < (byte)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((byte)result > (byte)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repne_scas_o16_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        int a = getShortOperand() & 0xffff;
	int b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    b = getShortOperand() & 0xffff;

	    if (cpu.eflagsDirection) {
		cpu.edi -= 2;
	    } else {
		cpu.edi += 2;
	    }
	    cpu.ecx--;
	    if (a == b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            int result = a - b;
	    standardFlags((short)result);
            cpu.eflagsCarry = (result & (~0xffff)) !=0;
            if ((short)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((short)result < (short)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((short)result > (short)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void repne_scas_o32_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
        long a = getIntOperand() & 0xffffffffl;
	long b = 0;
	int operandAddress = microcodesPosition;
	skipOperand();
        boolean used = false;
	while (cpu.ecx != 0) {
	    used = true;
	    microcodesPosition = operandAddress;
	    // TODO:  put interrupt handling here, I think
	    b = getIntOperand() & 0xffffffffl;

	    if (cpu.eflagsDirection) {
		cpu.edi -= 4;
	    } else {
		cpu.edi += 4;
	    }
	    cpu.ecx--;
	    if (a == b) break;
	}

	if (used) {
	    // now we change the flags to reflect last comparison
            long result = a - b;
	    standardFlags((int)result);
            cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
            if ((int)b < 0) // +VE Swing
                cpu.eflagsOverflow = ((int)result < (int)a);
            else // -VE Swing
                cpu.eflagsOverflow = ((int)result > (int)a);

	    cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
	}
    }

    protected final void rep_stos_o8_a16() throws ProcessorException
    {
	byte data = getByteOperand();
	int operandAddress = microcodesPosition;
	skipOperand();
	while ((cpu.ecx & 0xffff) != 0) {
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);
	    microcodesPosition = operandAddress;
	    setByteOperand(data);

	    if (cpu.eflagsDirection)
                di -= 1;
	    else
                di += 1;

	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
        }
    }

    protected final void rep_stos_o16_a16() throws ProcessorException
    {
	short data = getShortOperand();
	int operandAddress = microcodesPosition;
	skipOperand();
	while ((cpu.ecx & 0xffff) != 0) {
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);
	    microcodesPosition = operandAddress;
	    setShortOperand(data);

	    if (cpu.eflagsDirection)
                di -= 2;
	    else
                di += 2;

	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
        }
    }

    protected final void rep_stos_o32_a16() throws ProcessorException
    {
	int data = getIntOperand();
	int operandAddress = microcodesPosition;
	skipOperand();
	while ((cpu.ecx & 0xffff) != 0) {
	    // TODO:  put interrupt handling here, I think
	    int di = (cpu.edi & 0xffff);
	    microcodesPosition = operandAddress;
	    setIntOperand(data);

	    if (cpu.eflagsDirection)
                di -= 4;
	    else
                di += 4;

	    cpu.edi &= ~0xffff;
	    cpu.edi |= (di & 0xffff);
	    cpu.ecx--;
        }
    }

    protected final void rep_stos_o8_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
	byte data = getByteOperand();
	int operandAddress = microcodesPosition;
	skipOperand();
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    microcodesPosition = operandAddress;
	    setByteOperand(data);

	    if (cpu.eflagsDirection)
                cpu.edi -= 1;
	    else
                cpu.edi += 1;

	    cpu.ecx--;
        }
    }

    protected final void rep_stos_o16_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
	short data = getShortOperand();
	int operandAddress = microcodesPosition;
	skipOperand();
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    microcodesPosition = operandAddress;
	    setShortOperand(data);

	    if (cpu.eflagsDirection)
                cpu.edi -= 2;
	    else
                cpu.edi += 2;

	    cpu.ecx--;
        }
    }

    protected final void rep_stos_o32_a32() throws ProcessorException
    {
        executeCount += cpu.ecx -1;
	int data = getIntOperand();
	int operandAddress = microcodesPosition;
	skipOperand();
	while (cpu.ecx != 0) {
	    // TODO:  put interrupt handling here, I think
	    microcodesPosition = operandAddress;
	    setIntOperand(data);

	    if (cpu.eflagsDirection)
                cpu.edi -= 4;
	    else
                cpu.edi += 4;

	    cpu.ecx--;
        }
    }

    protected final void rcl_o8() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f) % 9;
	if (c>0) {
	    int data = 0xff & getByteOperand();
	    int cfBit = cpu.eflagsCarry ? 1 : 0;
	    data = data | (cfBit << 8); // 9 bit value
	    data = (data << c) | (data >>> (9-c)); // rotate
	    setByteOperand((byte)data);

	    cpu.eflagsCarry = ((data & 0x100)!= 0);
            if (c==1)
            {
	        boolean msb = ((data & 0x80) != 0);
	        cpu.eflagsOverflow = msb ^ cpu.eflagsCarry;
            }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rcl_o16() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f) % 17;
	if (c>0) {
	    int data = 0xffff & getShortOperand();
	    int cfBit = cpu.eflagsCarry ? 1 : 0;
	    data = data | (cfBit << 16); // 17 bit value
	    data = (data << c) | (data >>> (17-c)); // rotate
	    setShortOperand((short)data);

	    cpu.eflagsCarry = ((data & 0x10000)!= 0);
            if (c==1)
            {
	        boolean msb = ((data & 0x8000) != 0);
	        cpu.eflagsOverflow = msb ^ cpu.eflagsCarry;
            }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rcl_o32() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f);
	if (c>0) {
	    long data = 0xffffffffl & getIntOperand();
	    long cfBit = cpu.eflagsCarry ? 1 : 0;
	    data = data | (cfBit << 32); // 33 bit value
	    data = (data << c) | (data >>> (33-c)); // rotate
	    setIntOperand((int)data);

	    cpu.eflagsCarry = ((data & 0x100000000l)!= 0);
            if (c==1)
            {
	        boolean msb = ((data & 0x80000000l) != 0);
	        cpu.eflagsOverflow = msb ^ cpu.eflagsCarry;
            }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }


    protected final void rcr_o8() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f) % 9;
	if (c>0) {
	    int data = 0xff & getByteOperand();
	    int cfBit = cpu.eflagsCarry ? 1 : 0;
	    data = data | (cfBit << 8); // 9 bit value
	    data = (data >>> c) | (data << (9-c)); // rotate
	    setByteOperand((byte)data);

	    cpu.eflagsCarry = ((data & 0x100)!= 0);
            if (c==1)
            {
	        boolean msb = ((data & 0x80) != 0);
		boolean nmsb = ((data & 0x40) != 0);
	        cpu.eflagsOverflow = msb ^ nmsb;
            }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rcr_o16() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f) % 17;
	if (c>0) {
	    int data = 0xffff & getShortOperand();
	    int cfBit = cpu.eflagsCarry ? 1 : 0;
	    data = data | (cfBit << 16); // 17 bit value
	    data = (data >>> c) | (data << (17-c)); // rotate
	    setShortOperand((short)data);

	    cpu.eflagsCarry = ((data & 0x10000)!= 0);
            if (c==1)
            {
	        boolean msb = ((data & 0x8000) != 0);
		boolean nmsb = ((data & 0x4000) != 0);
	        cpu.eflagsOverflow = msb ^ nmsb;
            }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rcr_o32() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f);
	if (c>0) {
	    long data = 0xffffffffl & getIntOperand();
	    long cfBit = cpu.eflagsCarry ? 1 : 0;
	    data = data | (cfBit << 32); // 33 bit value
	    data = (data >>> c) | (data << (33-c)); // rotate
	    setIntOperand((int)data);

	    cpu.eflagsCarry = ((data & 0x100000000l) != 0);
            if (c==1) {
	        boolean msb = ((data & 0x80000000l) != 0);
		boolean nmsb = ((data & 0x40000000l) != 0);
	        cpu.eflagsOverflow = msb ^ nmsb;
            }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rol_o8() throws ProcessorException
    {
        int c = (getByteOperand() & 0x7);
	if (c>0) {
	    int data = 0xff & getByteOperand();
	    data = (data << c) | (data >>> (8-c));
	    setByteOperand((byte)data);

            cpu.eflagsCarry = ((data & 0x1) != 0);       	    
	    if (c==1) {
		boolean msb = ((data & 0x80) != 0);
		cpu.eflagsOverflow = msb ^ cpu.eflagsCarry;
	    }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rol_o16() throws ProcessorException
    {
        int c = (getByteOperand() & 0xf);
	if (c>0) {
	    int data = 0xffff & getShortOperand();
	    data = (data << c) | (data >>> (16-c));

	    setShortOperand((short)data);
            cpu.eflagsCarry = ((data & 0x1) != 0);       
	    
	    if (c==1) {
		boolean msb = ((data & 0x8000) != 0);
		cpu.eflagsOverflow = msb ^ cpu.eflagsCarry;
	    }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void rol_o32() throws ProcessorException
    {
        int c = (getByteOperand() & 0x1f);
	if (c>0) {
	    int data = getIntOperand();
	    //data = Integer.rotateLeft(data, c);
            data = (data << c) | (data >>> -c);
	    setIntOperand(data);

            cpu.eflagsCarry = ((data & 0x1) != 0);       	    
	    if (c==1) {
		boolean msb = ((data & 0x80000000) != 0);
		cpu.eflagsOverflow = msb ^ cpu.eflagsCarry;
	    }
	} else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void ror_o8() throws ProcessorException
    {
       int c = (getByteOperand() & 0x7);
       if (c > 0) {
	   int data = 0xff & getByteOperand();
	   data = (data >>> c) | (data << (8 - c));
	   setByteOperand((byte)data);
	   
	   cpu.eflagsCarry = ((data & 0x80) != 0); // copies MSB
	   if (c==1) {
	       boolean nmsb = ((data & 0x40) != 0);
	       cpu.eflagsOverflow = cpu.eflagsCarry ^ nmsb;
	   }
       } else {
	   skipOperand();
	   skipOperand();
       }      
    }

    protected final void ror_o16() throws ProcessorException
    {
       int c = (getByteOperand() & 0xf);
       if (c > 0) {
	   int data = 0xffff & getShortOperand();
	   data = (data >>> c) | (data << (16 - c));
	   setShortOperand((short)data);

	   cpu.eflagsCarry = ((data & 0x8000) != 0); // copies MSB	   
	   if (c==1) {
	       boolean nmsb = ((data & 0x4000) != 0);
	       cpu.eflagsOverflow = cpu.eflagsCarry ^ nmsb;
	   }
       } else {
	   skipOperand();
	   skipOperand();
       }      
    }

    protected final void ror_o32() throws ProcessorException
    {
       int c = (getByteOperand() & 0x1f);
       if (c > 0) {
	   int data = getIntOperand();
	   //data = Integer.rotateRight(data, c);
           data = (data >>> c) | (data << -c);
	   setIntOperand(data);

	   cpu.eflagsCarry = ((data & 0x80000000) != 0); // copies MSB
	   if (c==1) {
	       boolean nmsb = ((data & 0x40000000) != 0);
	       cpu.eflagsOverflow = cpu.eflagsCarry ^ nmsb;
	   }
       } else {
	   skipOperand();
	   skipOperand();
       }      
    }

    protected final void sar_o8() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;
        if (c > 0) {
	    int result = getByteOperand();
            result = result >> (c-1);
            boolean carry = ((result & 0x1) != 0);
            result = result >> 1;
	    setByteOperand((byte)result);

	    cpu.eflagsCarry = carry;
            if (c==1) cpu.eflagsOverflow = false;
            cpu.eflagsSign = ((byte)result < 0);
            cpu.eflagsZero = ((byte)result == 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}       
    }

    protected final void sar_o16() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;
        if (c > 0) {
	    int result = getShortOperand();
            result = result >> (c-1);
            boolean carry = ((result & 0x1) != 0);
            result = result >> 1;
	    setShortOperand((short)result);

	    cpu.eflagsCarry = carry;
            if (c==1) cpu.eflagsOverflow = false;
            cpu.eflagsSign = ((short)result < 0);
            cpu.eflagsZero = ((short)result == 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}       
    }

    protected final void sar_o32() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;
        if (c > 0) {
	    int result = getIntOperand();
	    result = result >> (c-1);
            boolean carry = ((result & 0x1) != 0);
            result = result >> 1;
	    setIntOperand(result);

	    cpu.eflagsCarry = carry;
            if (c==1) cpu.eflagsOverflow = false;
            cpu.eflagsSign = (result < 0);
            cpu.eflagsZero = (result == 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void sbb_o8() throws ProcessorException
    {
	int o2 = 0xff & getByteOperand();
        int o1 = 0xff & getByteOperand();
	int sum = cpu.eflagsCarry ? 1 : 0;
        int result = o1 - (o2 + sum);
   
	setByteOperand((byte)result);

	standardFlags((byte)result);

	cpu.eflagsCarry = (result & (~0xff)) !=0;
	if ((byte)(o2 + sum) < 0) // +VE Swing
	    cpu.eflagsOverflow = ((byte)result < (byte)o1);
	else // -VE Swing
	    cpu.eflagsOverflow = ((byte)result > (byte)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void sbb_o16() throws ProcessorException
    {
	int o2 = 0xffff & getShortOperand();
        int o1 = 0xffff & getShortOperand();
	int sum = cpu.eflagsCarry ? 1 : 0;
        int result = o1 - (o2 + sum);
   
	setShortOperand((short)result);

	standardFlags((short)result);

	cpu.eflagsCarry = (result & (~0xffff)) !=0;
	if ((short)(o2 + sum) < 0) // +VE Swing
	    cpu.eflagsOverflow = ((short)result < (short)o1);
	else // -VE Swing
	    cpu.eflagsOverflow = ((short)result > (short)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void sbb_o32() throws ProcessorException
    {
	long o2 = 0xffffffffl & getIntOperand();
        long o1 = 0xffffffffl & getIntOperand();
	long sum = cpu.eflagsCarry ? 1 : 0;
        long result = o1 - (o2 + sum);
   
	setIntOperand((int)result);

	standardFlags((int)result);

	cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	if ((int)(o2 + sum) < 0) // +VE Swing
	    cpu.eflagsOverflow = ((int)result < (int)o1);
	else // -VE Swing
	    cpu.eflagsOverflow = ((int)result > (int)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void scas_o8_a16() throws ProcessorException
    {
	int a = 0xff & getByteOperand();
	int b = 0xff & getByteOperand();

	int di = (cpu.edi & 0xffff);
	if (cpu.eflagsDirection) {
	    di -= 1;
	} else {
	    di += 1;
	}
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
	
        int result = a - b;
	
	standardFlags((byte)result);
	cpu.eflagsCarry = (result & (~0xff)) !=0;
	if ((byte)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((byte)result < (byte)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((byte)result > (byte)a);
	
	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
    }

    protected final void scas_o16_a16() throws ProcessorException
    {
	int a = 0xffff & getShortOperand();
	int b = 0xffff & getShortOperand();

	int di = (cpu.edi & 0xffff);
	if (cpu.eflagsDirection) {
	    di -= 2;
	} else {
	    di += 2;
	}
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
	
        int result = a - b;
	
	standardFlags((short)result);
	cpu.eflagsCarry = (result & (~0xffff)) !=0;
	if ((short)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((short)result < (short)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((short)result > (short)a);
	
	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
    }

    protected final void scas_o32_a16() throws ProcessorException
    {
	long a = 0xffffffffl & getIntOperand();
	long b = 0xffffffffl & getIntOperand();

	int di = (cpu.edi & 0xffff);
	if (cpu.eflagsDirection) {
	    di -= 4;
	} else {
	    di += 4;
	}
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
	
        long result = a - b;
	
	standardFlags((int)result);
	cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	if ((int)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((int)result < (int)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((int)result > (int)a);
	
	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
    }

    protected final void scas_o8_a32() throws ProcessorException
    {
	int a = 0xff & getByteOperand();
	int b = 0xff & getByteOperand();

	if (cpu.eflagsDirection) {
	    cpu.edi -= 1;
	} else {
	    cpu.edi += 1;
	}
	
        int result = a - b;
	
	standardFlags((byte)result);
	cpu.eflagsCarry = (result & (~0xff)) !=0;
	if ((byte)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((byte)result < (byte)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((byte)result > (byte)a);	
	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
    }

    protected final void scas_o16_a32() throws ProcessorException
    {
	int a = 0xffff & getShortOperand();
	int b = 0xffff & getShortOperand();

	if (cpu.eflagsDirection) {
	    cpu.edi -= 2;
	} else {
	    cpu.edi += 2;
	}
	
        int result = a - b;
	
	standardFlags((short)result);
	cpu.eflagsCarry = (result & (~0xffff)) !=0;
	if ((short)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((short)result < (short)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((short)result > (short)a);
	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
    }

    protected final void scas_o32_a32() throws ProcessorException
    {
	long a = 0xffffffffl & getIntOperand();
	long b = 0xffffffffl & getIntOperand();

	if (cpu.eflagsDirection) {
	    cpu.edi -= 4;
	} else {
	    cpu.edi += 4;
	}
	
        long result = a - b;
	
	standardFlags((int)result);
	cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
	if ((int)b < 0) // +VE Swing
	    cpu.eflagsOverflow = ((int)result < (int)a);
	else // -VE Swing
	    cpu.eflagsOverflow = ((int)result > (int)a);	
	cpu.eflagsAuxiliaryCarry = ((((a ^ b) ^ result) & 0x10) != 0);
    }

    protected final void setb() throws ProcessorException
    {
        if (cpu.eflagsCarry) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setnb() throws ProcessorException
    {
        if (!cpu.eflagsCarry) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setbe() throws ProcessorException
    {
        if (cpu.eflagsCarry || cpu.eflagsZero) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setnbe() throws ProcessorException
    {
        if (!cpu.eflagsCarry && !cpu.eflagsZero) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setl() throws ProcessorException
    {
        if (cpu.eflagsSign != cpu.eflagsOverflow) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setnl() throws ProcessorException
    {
        if (cpu.eflagsSign == cpu.eflagsOverflow) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setle() throws ProcessorException
    {
        if (cpu.eflagsZero || (cpu.eflagsSign != cpu.eflagsOverflow)) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setnle() throws ProcessorException
    {
        if ((!cpu.eflagsZero) && (cpu.eflagsSign == cpu.eflagsOverflow)) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void seto() throws ProcessorException
    {
        if (cpu.eflagsOverflow) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setno() throws ProcessorException
    {
        if (!cpu.eflagsOverflow) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setp() throws ProcessorException
    {
        if (cpu.eflagsParity) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setnp() throws ProcessorException
    {
        if (!cpu.eflagsParity) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void sets() throws ProcessorException
    {
        if (cpu.eflagsSign) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setns() throws ProcessorException
    {
        if (!cpu.eflagsSign) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setz() throws ProcessorException
    {
        if (cpu.eflagsZero) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void setnz() throws ProcessorException
    {
        if (!cpu.eflagsZero) {
	    setByteOperand((byte)1);
	} else {
	    setByteOperand((byte)0);
	}
    }

    protected final void shl_o8() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;

        if (c > 0) {
	    byte o1 = getByteOperand();
            byte result = (byte)(o1 << (c-1));
            boolean carry = ((result & 0x80) != 0);
            result = (byte)(result << 1);
	    setByteOperand(result);

	    cpu.eflagsCarry = carry;
            if (c == 1)
                cpu.eflagsOverflow = ((result & 0x80) != 0) ^ cpu.eflagsCarry;

            cpu.eflagsZero = (result == 0);
            cpu.eflagsSign = (result < 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void shl_o16() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;

        if (c > 0) {
	    short o1 = getShortOperand();
            short result = (short)(o1 << (c-1));
	    boolean carry = ((result & 0x8000) != 0);
            result = (short)(result << 1);
	    setShortOperand(result);

	    cpu.eflagsCarry = carry;
            if (c == 1)
                cpu.eflagsOverflow = ((result & 0x8000) != 0) ^ cpu.eflagsCarry;

            cpu.eflagsZero = (result == 0);
            cpu.eflagsSign = (result < 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void shl_o32() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;

        if (c > 0) {
	    int o1 = getIntOperand();
            int result = (o1 << (c-1));
            boolean carry = ((result & 0x80000000) != 0);
            result = (result << 1);
	    setIntOperand(result);

	    cpu.eflagsCarry = carry;
            if (c == 1)
                cpu.eflagsOverflow = ((result & 0x80000000) != 0) ^ cpu.eflagsCarry;

            cpu.eflagsZero = (result == 0);
            cpu.eflagsSign = (result < 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void shr_o8() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;

        if (c > 0) {
	    int o1 = 0xff & getByteOperand();	    
            int result = (o1 >>> (c-1));
	    boolean carry = ((result & 0x1) != 0);
            result = result >>> 1;
	    setByteOperand((byte)result);

            cpu.eflagsCarry = carry;
	    if (c == 1)
                cpu.eflagsOverflow = ((o1 & 0x80) != 0);
            cpu.eflagsZero = ((byte)result == 0);
            cpu.eflagsSign = ((byte)result < 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void shr_o16() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;

        if (c > 0) {
	    int o1 = 0xffff & getShortOperand();	    
            int result = (o1 >>> (c-1));
            boolean carry = ((result & 0x1) != 0);
            result = result >>> 1;
	    setShortOperand((short)result);

	    cpu.eflagsCarry = carry;
	    if (c == 1)
                cpu.eflagsOverflow = ((o1 & 0x8000) != 0);
            cpu.eflagsZero = ((short)result == 0);
            cpu.eflagsSign = ((short)result < 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void shr_o32() throws ProcessorException
    {
        int c = getByteOperand() & 0x1f;

        if (c > 0) {
	    int o1 = getIntOperand();
            int result = (o1 >>> (c-1));
            boolean carry = ((result & 0x1) != 0);
            result = result >>> 1;
	    setIntOperand(result);

	    cpu.eflagsCarry = carry;
	    if (c == 1)
                cpu.eflagsOverflow = ((o1 & 0x80000000) != 0);
            cpu.eflagsZero = (result == 0);
            cpu.eflagsSign = (result < 0);
            cpu.eflagsParity = parityMap[result & 0xff];
        } else {
	    skipOperand();
	    skipOperand();
	}
    }

    protected final void stc()
    {
	cpu.eflagsCarry = true;
    }

    protected final void std()
    {
	cpu.eflagsDirection = true;
    }

    protected final void sti()
    {
	cpu.eflagsInterruptEnable = cpu.eflagsInterruptEnableSoon = true;
    }

    protected final void stos_o8_a16() throws ProcessorException
    {
	byte data = getByteOperand();

	// TODO:  put interrupt handling here, I think
	int di = (cpu.edi & 0xffff);

	setByteOperand(data);
	
	if (cpu.eflagsDirection)
	    di -= 1;
	else
	    di += 1;
	
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
    }

    protected final void stos_o16_a16() throws ProcessorException
    {
	short data = getShortOperand();

	// TODO:  put interrupt handling here, I think
	int di = (cpu.edi & 0xffff);

	setShortOperand(data);
	
	if (cpu.eflagsDirection)
	    di -= 2;
	else
	    di += 2;
	
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
    }

    protected final void stos_o32_a16() throws ProcessorException
    {
	int data = getIntOperand();

	// TODO:  put interrupt handling here, I think
	int di = (cpu.edi & 0xffff);

	setIntOperand(data);
	
	if (cpu.eflagsDirection)
	    di -= 4;
	else
	    di += 4;
	
	cpu.edi &= ~0xffff;
	cpu.edi |= (di & 0xffff);
    }

    protected final void stos_o8_a32() throws ProcessorException
    {
	byte data = getByteOperand();
	setByteOperand(data);

	if (cpu.eflagsDirection)
	    cpu.edi -= 1;
	else
	    cpu.edi += 1;
    }

    protected final void stos_o16_a32() throws ProcessorException
    {
	short data = getShortOperand();
	setShortOperand(data);

	if (cpu.eflagsDirection)
	    cpu.edi -= 2;
	else
	    cpu.edi += 2;
    }

    protected final void stos_o32_a32() throws ProcessorException
    {
	int data = getIntOperand();
	setIntOperand(data);

	if (cpu.eflagsDirection)
	    cpu.edi -= 4;
	else
	    cpu.edi += 4;
    }

    protected final void shrd_o16() throws ProcessorException
    {
	int extraBits = 0xffff & getShortOperand();
	int count = getByteOperand() & 0x1f;
	if (count == 0) {
	    skipOperand();
	    skipOperand();
	    return;
	}

	int source = 0xffff & getShortOperand();
	int dest = (source >>> count) | (extraBits << (16 - count));
	setShortOperand((short)dest);

	standardFlags((short)dest);

	cpu.eflagsCarry = ((source >>> (count - 1) )& 0x1) == 0x1;
	if (count == 1)
	    cpu.eflagsOverflow = ((source & 0x8000) != (dest & 0x8000));
    }

    protected final void shrd_o32() throws ProcessorException
    {
	int extraBits = getIntOperand();
	int count = getByteOperand() & 0x1f;
	if (count == 0) {
	    skipOperand();
	    skipOperand();
	    return;
	}

	int source = getIntOperand();
	int dest = (source >>> count) | (extraBits << (32 - count));
	setIntOperand(dest);

	standardFlags(dest);

	cpu.eflagsCarry = ((source >>> (count - 1) )& 0x1) == 0x1;
	if (count == 1)
	    cpu.eflagsOverflow = ((source & 0x80000000) != (dest & 0x80000000));
    }

    protected final void shld_o16() throws ProcessorException
    {
	int extraBits = 0xffff & getShortOperand();
	int count = getByteOperand() & 0x1f;
	if (count == 0) {
	    skipOperand();
	    skipOperand();
	    return;
	}

	int source = 0xffff & getShortOperand();
	int dest = (source << count) | (extraBits >>> (16 - count));
	setShortOperand((short)dest);

	standardFlags((short)dest);

	cpu.eflagsCarry = ((source << (count - 1) )& 0x8000) == 0x8000;
	if (count == 1)
	    cpu.eflagsOverflow = ((source & 0x8000) != (dest & 0x8000));
    }

    protected final void shld_o32() throws ProcessorException
    {
	int extraBits = getIntOperand();
	int count = getByteOperand() & 0x1f;
	if (count == 0) {
	    skipOperand();
	    skipOperand();
	    return;
	}

	int source = getIntOperand();
	int dest = (source << count) | (extraBits >>> (32 - count));
	setIntOperand(dest);

	standardFlags(dest);

	cpu.eflagsCarry = ((source << (count - 1) )& 0x80000000) == 0x80000000;
	if (count == 1)
	    cpu.eflagsOverflow = ((source & 0x80000000) != (dest & 0x80000000));
    }

    protected final void sub_o8() throws ProcessorException
       {
	int o2 = 0xff & getByteOperand();
        int o1 = 0xff & getByteOperand();
        int result = o1 - o2;
	setByteOperand((byte)result);

	standardFlags((byte)result);

	cpu.eflagsCarry = ((result & (~0xFF)) !=0);
	if ((byte)o2 < 0) // +VE Swing
	    cpu.eflagsOverflow = ((byte)result < (byte)o1);
	else // -VE Swing
	    cpu.eflagsOverflow = ((byte)result > (byte)o1);

	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    } 

    protected final void sub_o16() throws ProcessorException
    {
        int o2 = 0xffff & getShortOperand();
        int o1 = 0xffff & getShortOperand();
        int result = o1 - o2;
        setShortOperand((short)result);

	standardFlags((short)result);

        cpu.eflagsCarry = (result & (~0xFFFF)) !=0;
        if ((short)o2 < 0) // +VE Swing
            cpu.eflagsOverflow = ((short)result < (short)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((short)result > (short)o1);
	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void sub_o32() throws ProcessorException
    {
        long o2 = 0xffffffffl & getIntOperand();
        long o1 = 0xffffffffl & getIntOperand();
        long result = o1 - o2;
        setIntOperand((int)result);

	standardFlags((int)result);

        cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
        if ((int)o2 < 0) // +VE Swing
            cpu.eflagsOverflow = ((int)result < (int)o1);
        else // -VE Swing
            cpu.eflagsOverflow = ((int)result > (int)o1);
	cpu.eflagsAuxiliaryCarry = ((((o1 ^ o2) ^ result) & 0x10) != 0);
    }

    protected final void test_o8() throws ProcessorException
    {
	byte result = (byte)(getByteOperand() & getByteOperand());

	//flags
        standardFlags(result);
    }

    protected final void test_o16() throws ProcessorException
    {
	short result = (short)(getShortOperand() & getShortOperand());

	//flags
        standardFlags(result);
    }

    protected final void test_o32() throws ProcessorException
    {
	int result = getIntOperand() & getIntOperand();

	//flags
        standardFlags(result);
    }

    protected final void waitOp() 
    {
	if ((cpu.getCR0() & 0x6) == 0x6)
	    throw exceptionNM;
    }   

    protected final void xadd_o8() throws ProcessorException
    {
	int source = 0xff & getByteOperand();
	int dest = 0xff & getByteOperand();
	setByteOperand((byte)dest);
	
        int result = source + dest;
	setByteOperand((byte)result);
        standardFlags((byte)result);
        
        cpu.eflagsCarry = (result & (~0xff)) !=0;
        if ((byte)dest > 0) // +VE Swing
            cpu.eflagsOverflow = ((byte)result < (byte)source);
        else // -VE Swing
            cpu.eflagsOverflow = ((byte)result > (byte)source);
	cpu.eflagsAuxiliaryCarry = ((((source ^ dest) ^ result) & 0x10) != 0);
    }

    protected final void xadd_o16() throws ProcessorException
    {
	int source = 0xffff & getShortOperand();
	int dest = 0xffff & getShortOperand();
	setShortOperand((short)dest);
	
        int result = source + dest;
	setShortOperand((short)result);
        standardFlags((short)result);
        
        cpu.eflagsCarry = (result & (~0xffff)) !=0;
        if ((short)dest > 0) // +VE Swing
            cpu.eflagsOverflow = ((short)result < (short)source);
        else // -VE Swing
            cpu.eflagsOverflow = ((short)result > (short)source);
	cpu.eflagsAuxiliaryCarry = ((((source ^ dest) ^ result) & 0x10) != 0);
    }

    protected final void xadd_o32() throws ProcessorException
    {
	long source = 0xffffffffl & getIntOperand();
	long dest = 0xffffffffl & getIntOperand();
	setIntOperand((int)dest);

        long result = source + dest;
	setIntOperand((int)result);
        standardFlags((int)result);
        
        cpu.eflagsCarry = (result & (~0xffffffffl)) !=0;
        if ((int)dest > 0) // +VE Swing
            cpu.eflagsOverflow = ((int)result < (int)source);
        else // -VE Swing
            cpu.eflagsOverflow = ((int)result > (int)source);
	cpu.eflagsAuxiliaryCarry = ((((source ^ dest) ^ result) & 0x10) != 0);
    }

    protected final void xchg_o8() throws ProcessorException
    {
        byte tmp1 = getByteOperand();
	byte tmp2 = getByteOperand();
	setByteOperand(tmp2);
	setByteOperand(tmp1);
    }

    protected final void xchg_o16() throws ProcessorException
    {
        short tmp1 = getShortOperand();
	short tmp2 = getShortOperand();
	setShortOperand(tmp2);
	setShortOperand(tmp1);
    }

    protected final void xchg_o32() throws ProcessorException
    {
        int tmp1 = getIntOperand();
	int tmp2 = getIntOperand();
	setIntOperand(tmp2);
	setIntOperand(tmp1);
    }

    protected final void xlat()
    {
	byte b = getByteOperand();
	setByteOperand(b);
    }

    protected final void xor_o8() throws ProcessorException
    {
	byte result = (byte)(getByteOperand() ^ getByteOperand());
	setByteOperand(result);

	//flags
        standardFlags(result);
    }

    protected final void xor_o16() throws ProcessorException
    {
	short result = (short)(getShortOperand() ^ getShortOperand());
	setShortOperand(result);

	//flags
        standardFlags(result);
    }

    protected final void xor_o32() throws ProcessorException
    {
	int result = getIntOperand() ^ getIntOperand();
	setIntOperand(result);

	//flags
        standardFlags(result);
    }

    protected final Segment getSegmentOperand()
    {
	int microcode = getMicrocode();
	microcodesPosition += immediateCount(microcode);

	switch (microcode) {
	case M_DS_EAX: //104
	case M_DS_ECX: //105
	case M_DS_EDX: //106
	case M_DS_EBX: //107
	case M_DS_ID: //108
	case M_DS_ESI: //109
	case M_DS_EDI: //110
	case M_DS_EAX_IB: //111
	case M_DS_ECX_IB: //112
	case M_DS_EDX_IB: //113
	case M_DS_EBX_IB: //114
	case M_DS_EBP_IB: //115 
	case M_DS_ESI_IB: //116
	case M_DS_EDI_IB: //117
	case M_DS_EAX_ID: //118
	case M_DS_ECX_ID: //119
	case M_DS_EDX_ID: //120
	case M_DS_EBX_ID: //121
	case M_DS_EBP_ID: //122
	case M_DS_ESI_ID: //123
	case M_DS_EDI_ID: //124

	case M_DS_2EAX: //338;
	case M_DS_EAX_ECX: //339;
	case M_DS_EAX_EDX: //340;
	case M_DS_EAX_EBX: //341;
	case M_DS_EAX_ESP: //342;
	case M_DS_EAX_EBP: //343;
	case M_DS_EAX_ESI: //344;
	case M_DS_EAX_EDI: //345;
	    
	case M_DS_2ECX: //346;
	case M_DS_ECX_EDX: //347;
	case M_DS_ECX_EBX: //348;
	case M_DS_ECX_ESP: //349;
	case M_DS_ECX_EBP: //350;
	case M_DS_ECX_ESI: //351;
	case M_DS_ECX_EDI: //352;

	case M_DS_2EDX: //353;
	case M_DS_EDX_EBX: //354;
	case M_DS_EDX_ESP: //355;
	case M_DS_EDX_EBP: //356;
	case M_DS_EDX_ESI: //357;
	case M_DS_EDX_EDI: //358;

	case M_DS_2EBX: //359;
	case M_DS_EBX_ESP: //360;
	case M_DS_EBX_EBP: //361;
	case M_DS_EBX_ESI: //362;
	case M_DS_EBX_EDI: //363;

	case M_DS_ESP: //364;

	case M_DS_EBP_ESP: //365;
	case M_DS_EBP_ESI: //366;
	case M_DS_EBP_EDI: //367;

	case M_DS_ESI_ESP: //368;
	case M_DS_2ESI: //369;
	case M_DS_ESI_EDI: //370;

	case M_DS_EDI_ESP: //371;
	case M_DS_2EDI: //372;

	case M_DS_3EAX: //373;
	case M_DS_2EAX_ECX: //374;
	case M_DS_2EAX_EDX: //375;
	case M_DS_2EAX_EBX: //376;
	case M_DS_2EAX_ESP: //377;
	case M_DS_2EAX_ID: //378;
	case M_DS_2EAX_ESI: //379;
	case M_DS_2EAX_EDI: //380;

	case M_DS_2ECX_EAX: //381;
	case M_DS_3ECX: //382;
	case M_DS_2ECX_EDX: //383;
	case M_DS_2ECX_EBX: //384;
	case M_DS_2ECX_ESP: //385;
	case M_DS_2ECX_ID: //386;
	case M_DS_2ECX_ESI: //387;
	case M_DS_2ECX_EDI: //388;

	case M_DS_2EDX_EAX: //389;
	case M_DS_2EDX_ECX: //390;
	case M_DS_3EDX: //391;
	case M_DS_2EDX_EBX: //392;
	case M_DS_2EDX_ESP: //393;
	case M_DS_2EDX_ID: //394;
	case M_DS_2EDX_ESI: //395;
	case M_DS_2EDX_EDI: //396;

	case M_DS_2EBX_EAX: //397;
	case M_DS_2EBX_ECX: //398;
	case M_DS_2EBX_EDX: //399;
	case M_DS_3EBX: //400;
	case M_DS_2EBX_ESP: //401;
	case M_DS_2EBX_ID: //402;
	case M_DS_2EBX_ESI: //403;
	case M_DS_2EBX_EDI: //404;

	case M_DS_2EBP_EAX: //405;
	case M_DS_2EBP_ECX: //406;
	case M_DS_2EBP_EDX: //407;
	case M_DS_2EBP_EBX: //408;
	case M_DS_2EBP_ESP: //409;
	case M_DS_2EBP_ID: //410;
	case M_DS_2EBP_ESI: //411;
	case M_DS_2EBP_EDI: //412;

	case M_DS_2ESI_EAX: //413;
	case M_DS_2ESI_ECX: //414;
	case M_DS_2ESI_EDX: //415;
	case M_DS_2ESI_EBX: //416;
	case M_DS_2ESI_ESP: //417;
	case M_DS_2ESI_ID: //418;
	case M_DS_3ESI: //419;
	case M_DS_2ESI_EDI: //420;

	case M_DS_2EDI_EAX: //421;
	case M_DS_2EDI_ECX: //422;
	case M_DS_2EDI_EDX: //423;
	case M_DS_2EDI_EBX: //424;
	case M_DS_2EDI_ESP: //425;
	case M_DS_2EDI_ID: //426;
	case M_DS_2EDI_ESI: //427;
	case M_DS_3EDI: //428;

	case M_DS_5EAX: //429;
	case M_DS_4EAX_ECX: //430;
	case M_DS_4EAX_EDX: //431;
	case M_DS_4EAX_EBX: //432;
	case M_DS_4EAX_ESP: //433;
	case M_DS_4EAX_ID: //434;
	case M_DS_4EAX_ESI: //435;
	case M_DS_4EAX_EDI: //436;

	case M_DS_4ECX_EAX: //437;
	case M_DS_5ECX: //438;
	case M_DS_4ECX_EDX: //439;
	case M_DS_4ECX_EBX: //440;
	case M_DS_4ECX_ESP: //441;
	case M_DS_4ECX_ID: //442;
	case M_DS_4ECX_ESI: //443;
	case M_DS_4ECX_EDI: //444;

	case M_DS_4EDX_EAX: //445;
	case M_DS_4EDX_ECX: //446;
	case M_DS_5EDX: //447;
	case M_DS_4EDX_EBX: //448;
	case M_DS_4EDX_ESP: //449;
	case M_DS_4EDX_ID: //450;
	case M_DS_4EDX_ESI: //451;
	case M_DS_4EDX_EDI: //452;

	case M_DS_4EBX_EAX: //453;
	case M_DS_4EBX_ECX: //454;
	case M_DS_4EBX_EDX: //455;
	case M_DS_5EBX: //456;
	case M_DS_4EBX_ESP: //457;
	case M_DS_4EBX_ID: //458;
	case M_DS_4EBX_ESI: //459;
	case M_DS_4EBX_EDI: //460;

	case M_DS_4EBP_EAX: //461;
	case M_DS_4EBP_ECX: //462;
	case M_DS_4EBP_EDX: //463;
	case M_DS_4EBP_EBX: //464;
	case M_DS_4EBP_ESP: //465;
	case M_DS_4EBP_ID: //466;
	case M_DS_4EBP_ESI: //467;
	case M_DS_4EBP_EDI: //468;

	case M_DS_4ESI_EAX: //469;
	case M_DS_4ESI_ECX: //470;
	case M_DS_4ESI_EDX: //471;
	case M_DS_4ESI_EBX: //472;
	case M_DS_4ESI_ESP: //473;
	case M_DS_4ESI_ID: //474;
	case M_DS_5ESI: //475;
	case M_DS_4ESI_EDI: //476;

	case M_DS_4EDI_EAX: //477;
	case M_DS_4EDI_ECX: //478;
	case M_DS_4EDI_EDX: //479;
	case M_DS_4EDI_EBX: //480;
	case M_DS_4EDI_ESP: //481;
	case M_DS_4EDI_ID: //482;
	case M_DS_4EDI_ESI: //483;
	case M_DS_5EDI: //484;

	case M_DS_9EAX: //485;
	case M_DS_8EAX_ECX: //486;
	case M_DS_8EAX_EDX: //487;
	case M_DS_8EAX_EBX: //488;
	case M_DS_8EAX_ESP: //489;
	case M_DS_8EAX_ID: //490;
	case M_DS_8EAX_ESI: //491;
	case M_DS_8EAX_EDI: //492;

	case M_DS_8ECX_EAX: //493;
	case M_DS_9ECX: //494;
	case M_DS_8ECX_EDX: //495;
	case M_DS_8ECX_EBX: //496;
	case M_DS_8ECX_ESP: //497;
	case M_DS_8ECX_ID: //498;
	case M_DS_8ECX_ESI: //499;
	case M_DS_8ECX_EDI: //500;

	case M_DS_8EDX_EAX: //501;
	case M_DS_8EDX_ECX: //502;
	case M_DS_9EDX: //503;
	case M_DS_8EDX_EBX: //504;
	case M_DS_8EDX_ESP: //505;
	case M_DS_8EDX_ID: //506;
	case M_DS_8EDX_ESI: //507;
	case M_DS_8EDX_EDI: //508;

	case M_DS_8EBX_EAX: //509;
	case M_DS_8EBX_ECX: //510;
	case M_DS_8EBX_EDX: //511;
	case M_DS_9EBX: //512;
	case M_DS_8EBX_ESP: //513;
	case M_DS_8EBX_ID: //514;
	case M_DS_8EBX_ESI: //515;
	case M_DS_8EBX_EDI: //516;

	case M_DS_8EBP_EAX: //517;
	case M_DS_8EBP_ECX: //518;
	case M_DS_8EBP_EDX: //519;
	case M_DS_8EBP_EBX: //520;
	case M_DS_8EBP_ESP: //521;
	case M_DS_8EBP_ID: //522;
	case M_DS_8EBP_ESI: //523;
	case M_DS_8EBP_EDI: //524;

	case M_DS_8ESI_EAX: //525;
	case M_DS_8ESI_ECX: //526;
	case M_DS_8ESI_EDX: //527;
	case M_DS_8ESI_EBX: //528;
	case M_DS_8ESI_ESP: //529;
	case M_DS_8ESI_ID: //530;
	case M_DS_9ESI: //531;
	case M_DS_8ESI_EDI: //532;
    
	case M_DS_8EDI_EAX: //533;
	case M_DS_8EDI_ECX: //534;
	case M_DS_8EDI_EDX: //535;
	case M_DS_8EDI_EBX: //536;
	case M_DS_8EDI_ESP: //537;
	case M_DS_8EDI_ID: //538;
	case M_DS_8EDI_ESI: //539;
	case M_DS_9EDI: //540;

	    //SIB + IB Operands
	case M_DS_2EAX_IB: //541;
	case M_DS_EAX_ECX_IB: //542;
	case M_DS_EAX_EDX_IB: //543;
	case M_DS_EAX_EBX_IB: //544;
	case M_DS_EAX_ESP_IB: //545;
	case M_DS_EAX_EBP_IB: //546;
	case M_DS_EAX_ESI_IB: //547;
	case M_DS_EAX_EDI_IB: //548;

	case M_DS_2ECX_IB: //549;
	case M_DS_ECX_EDX_IB: //550;
	case M_DS_ECX_EBX_IB: //551;
	case M_DS_ECX_ESP_IB: //552;
	case M_DS_ECX_EBP_IB: //553;
	case M_DS_ECX_ESI_IB: //554;
	case M_DS_ECX_EDI_IB: //555;

	case M_DS_2EDX_IB: //556;
	case M_DS_EDX_EBX_IB: //557;
	case M_DS_EDX_ESP_IB: //558;
	case M_DS_EDX_EBP_IB: //559;
	case M_DS_EDX_ESI_IB: //560;
	case M_DS_EDX_EDI_IB: //561;

	case M_DS_2EBX_IB: //562;
	case M_DS_EBX_ESP_IB: //563;
	case M_DS_EBX_EBP_IB: //564;
	case M_DS_EBX_ESI_IB: //565;
	case M_DS_EBX_EDI_IB: //566;

	case M_DS_ESP_IB: //567;

	case M_DS_EBP_ESP_IB: //568;
	case M_DS_2EBP_IB: //569;
	case M_DS_EBP_ESI_IB: //570;
	case M_DS_EBP_EDI_IB: //571;

	case M_DS_ESI_ESP_IB: //572;
	case M_DS_2ESI_IB: //573;
	case M_DS_ESI_EDI_IB: //574;

	case M_DS_EDI_ESP_IB: //575;
	case M_DS_2EDI_IB: //576;

	case M_DS_3EAX_IB: //577;
	case M_DS_2EAX_ECX_IB: //578;
	case M_DS_2EAX_EDX_IB: //579;
	case M_DS_2EAX_EBX_IB: //580;
	case M_DS_2EAX_ESP_IB: //581;
	case M_DS_2EAX_EBP_IB: //582;
	case M_DS_2EAX_ESI_IB: //583;
	case M_DS_2EAX_EDI_IB: //584;

	case M_DS_2ECX_EAX_IB: //585;
	case M_DS_3ECX_IB: //586;
	case M_DS_2ECX_EDX_IB: //587;
	case M_DS_2ECX_EBX_IB: //588;
	case M_DS_2ECX_ESP_IB: //589;
	case M_DS_2ECX_EBP_IB: //590;
	case M_DS_2ECX_ESI_IB: //591;
	case M_DS_2ECX_EDI_IB: //592;

	case M_DS_2EDX_EAX_IB: //593;
	case M_DS_2EDX_ECX_IB: //594;
	case M_DS_3EDX_IB: //595;
	case M_DS_2EDX_EBX_IB: //596;
	case M_DS_2EDX_ESP_IB: //597;
	case M_DS_2EDX_EBP_IB: //598;
	case M_DS_2EDX_ESI_IB: //599;
	case M_DS_2EDX_EDI_IB: //600;

	case M_DS_2EBX_EAX_IB: //601;
	case M_DS_2EBX_ECX_IB: //602;
	case M_DS_2EBX_EDX_IB: //603;
	case M_DS_3EBX_IB: //604;
	case M_DS_2EBX_ESP_IB: //605;
	case M_DS_2EBX_EBP_IB: //606;
	case M_DS_2EBX_ESI_IB: //607;
	case M_DS_2EBX_EDI_IB: //608;

	case M_DS_2EBP_EAX_IB: //609;
	case M_DS_2EBP_ECX_IB: //610;
	case M_DS_2EBP_EDX_IB: //611;
	case M_DS_2EBP_EBX_IB: //612;
	case M_DS_2EBP_ESP_IB: //613;
	case M_DS_3EBP_IB: //614;
	case M_DS_2EBP_ESI_IB: //615;
	case M_DS_2EBP_EDI_IB: //616;

	case M_DS_2ESI_EAX_IB: //617;
	case M_DS_2ESI_ECX_IB: //618;
	case M_DS_2ESI_EDX_IB: //619;
	case M_DS_2ESI_EBX_IB: //620;
	case M_DS_2ESI_ESP_IB: //621;
	case M_DS_2ESI_EBP_IB: //622;
	case M_DS_3ESI_IB: //623;
	case M_DS_2ESI_EDI_IB: //624;

	case M_DS_2EDI_EAX_IB: //625;
	case M_DS_2EDI_ECX_IB: //626;
	case M_DS_2EDI_EDX_IB: //627;
	case M_DS_2EDI_EBX_IB: //628;
	case M_DS_2EDI_ESP_IB: //629;
	case M_DS_2EDI_EBP_IB: //630;
	case M_DS_2EDI_ESI_IB: //631;
	case M_DS_3EDI_IB: //632;

	case M_DS_5EAX_IB: //633;
	case M_DS_4EAX_ECX_IB: //634;
	case M_DS_4EAX_EDX_IB: //635;
	case M_DS_4EAX_EBX_IB: //636;
	case M_DS_4EAX_ESP_IB: //637;
	case M_DS_4EAX_EBP_IB: //638;
	case M_DS_4EAX_ESI_IB: //639;
	case M_DS_4EAX_EDI_IB: //640;

	case M_DS_4ECX_EAX_IB: //641;
	case M_DS_5ECX_IB: //642;
	case M_DS_4ECX_EDX_IB: //643;
	case M_DS_4ECX_EBX_IB: //644;
	case M_DS_4ECX_ESP_IB: //645;
	case M_DS_4ECX_EBP_IB: //646;
	case M_DS_4ECX_ESI_IB: //647;
	case M_DS_4ECX_EDI_IB: //648;

	case M_DS_4EDX_EAX_IB: //649;
	case M_DS_4EDX_ECX_IB: //650;
	case M_DS_5EDX_IB: //651;
	case M_DS_4EDX_EBX_IB: //652;
	case M_DS_4EDX_ESP_IB: //653;
	case M_DS_4EDX_EBP_IB: //654;
	case M_DS_4EDX_ESI_IB: //655;
	case M_DS_4EDX_EDI_IB: //656;

	case M_DS_4EBX_EAX_IB: //657;
	case M_DS_4EBX_ECX_IB: //658;
	case M_DS_4EBX_EDX_IB: //659;
	case M_DS_5EBX_IB: //660;
	case M_DS_4EBX_ESP_IB: //661;
	case M_DS_4EBX_EBP_IB: //662;
	case M_DS_4EBX_ESI_IB: //663;
	case M_DS_4EBX_EDI_IB: //664;

	case M_DS_4EBP_EAX_IB: //665;
	case M_DS_4EBP_ECX_IB: //666;
	case M_DS_4EBP_EDX_IB: //667;
	case M_DS_4EBP_EBX_IB: //668;
	case M_DS_4EBP_ESP_IB: //669;
	case M_DS_5EBP_IB: //670;
	case M_DS_4EBP_ESI_IB: //671;
	case M_DS_4EBP_EDI_IB: //672;

	case M_DS_4ESI_EAX_IB: //673;
	case M_DS_4ESI_ECX_IB: //674;
	case M_DS_4ESI_EDX_IB: //675;
	case M_DS_4ESI_EBX_IB: //676;
	case M_DS_4ESI_ESP_IB: //677;
	case M_DS_4ESI_EBP_IB: //678;
	case M_DS_5ESI_IB: //679;
	case M_DS_4ESI_EDI_IB: //680;

	case M_DS_4EDI_EAX_IB: //681;
	case M_DS_4EDI_ECX_IB: //682;
	case M_DS_4EDI_EDX_IB: //683;
	case M_DS_4EDI_EBX_IB: //684;
	case M_DS_4EDI_ESP_IB: //685;
	case M_DS_4EDI_EBP_IB: //686;
	case M_DS_4EDI_ESI_IB: //687;
	case M_DS_5EDI_IB: //688;

	case M_DS_9EAX_IB: //689;
	case M_DS_8EAX_ECX_IB: //690;
	case M_DS_8EAX_EDX_IB: //691;
	case M_DS_8EAX_EBX_IB: //692;
	case M_DS_8EAX_ESP_IB: //693;
	case M_DS_8EAX_EBP_IB: //694;
	case M_DS_8EAX_ESI_IB: //695;
	case M_DS_8EAX_EDI_IB: //696;

	case M_DS_8ECX_EAX_IB: //697;
	case M_DS_9ECX_IB: //698;
	case M_DS_8ECX_EDX_IB: //699;
	case M_DS_8ECX_EBX_IB: //700;
	case M_DS_8ECX_ESP_IB: //701;
	case M_DS_8ECX_EBP_IB: //702;
	case M_DS_8ECX_ESI_IB: //703;
	case M_DS_8ECX_EDI_IB: //704;

	case M_DS_8EDX_EAX_IB: //705;
	case M_DS_8EDX_ECX_IB: //706;
	case M_DS_9EDX_IB: //707;
	case M_DS_8EDX_EBX_IB: //708;
	case M_DS_8EDX_ESP_IB: //709;
	case M_DS_8EDX_EBP_IB: //710;
	case M_DS_8EDX_ESI_IB: //711;
	case M_DS_8EDX_EDI_IB: //712;

	case M_DS_8EBX_EAX_IB: //713;
	case M_DS_8EBX_ECX_IB: //714;
	case M_DS_8EBX_EDX_IB: //715;
	case M_DS_9EBX_IB: //716;
	case M_DS_8EBX_ESP_IB: //717;
	case M_DS_8EBX_EBP_IB: //718;
	case M_DS_8EBX_ESI_IB: //719;
	case M_DS_8EBX_EDI_IB: //720;

	case M_DS_8EBP_EAX_IB: //721;
	case M_DS_8EBP_ECX_IB: //722;
	case M_DS_8EBP_EDX_IB: //723;
	case M_DS_8EBP_EBX_IB: //724;
	case M_DS_8EBP_ESP_IB: //725;
	case M_DS_9EBP_IB: //726;
	case M_DS_8EBP_ESI_IB: //727;
	case M_DS_8EBP_EDI_IB: //728;

	case M_DS_8ESI_EAX_IB: //729;
	case M_DS_8ESI_ECX_IB: //730;
	case M_DS_8ESI_EDX_IB: //731;
	case M_DS_8ESI_EBX_IB: //732;
	case M_DS_8ESI_ESP_IB: //733;
	case M_DS_8ESI_EBP_IB: //734;
	case M_DS_9ESI_IB: //735;
	case M_DS_8ESI_EDI_IB: //736;
    
	case M_DS_8EDI_EAX_IB: //737;
	case M_DS_8EDI_ECX_IB: //738;
	case M_DS_8EDI_EDX_IB: //739;
	case M_DS_8EDI_EBX_IB: //740;
	case M_DS_8EDI_ESP_IB: //741;
	case M_DS_8EDI_EBP_IB: //742;
	case M_DS_8EDI_ESI_IB: //743;
	case M_DS_9EDI_IB: //744;

	    //SIB + ID Operands
	case M_DS_EAX_ECX_ID: //745;
	case M_DS_EAX_EDX_ID: //746;
	case M_DS_EAX_EBX_ID: //747;
	case M_DS_EAX_ESP_ID: //748;
	case M_DS_EAX_EBP_ID: //749;
	case M_DS_EAX_ESI_ID: //750;
	case M_DS_EAX_EDI_ID: //751;

	case M_DS_ECX_EDX_ID: //752;
	case M_DS_ECX_EBX_ID: //753;
	case M_DS_ECX_ESP_ID: //754;
	case M_DS_ECX_EBP_ID: //755;
	case M_DS_ECX_ESI_ID: //756;
	case M_DS_ECX_EDI_ID: //757;

	case M_DS_EDX_EBX_ID: //758;
	case M_DS_EDX_ESP_ID: //759;
	case M_DS_EDX_EBP_ID: //760;
	case M_DS_EDX_ESI_ID: //761;
	case M_DS_EDX_EDI_ID: //762;

	case M_DS_EBX_ESP_ID: //763;
	case M_DS_EBX_EBP_ID: //764;
	case M_DS_EBX_ESI_ID: //765;
	case M_DS_EBX_EDI_ID: //766;

	case M_DS_ESP_ID: //767;

	case M_DS_EBP_ESP_ID: //768;
	case M_DS_EBP_ESI_ID: //769;
	case M_DS_EBP_EDI_ID: //770;

	case M_DS_ESI_ESP_ID: //771;
	case M_DS_ESI_EDI_ID: //772;

	case M_DS_EDI_ESP_ID: //773;

	case M_DS_3EAX_ID: //774;
	case M_DS_2EAX_ECX_ID: //775;
	case M_DS_2EAX_EDX_ID: //776;
	case M_DS_2EAX_EBX_ID: //777;
	case M_DS_2EAX_ESP_ID: //778;
	case M_DS_2EAX_EBP_ID: //779;
	case M_DS_2EAX_ESI_ID: //780;
	case M_DS_2EAX_EDI_ID: //781;

	case M_DS_2ECX_EAX_ID: //782;
	case M_DS_3ECX_ID: //783;
	case M_DS_2ECX_EDX_ID: //784;
	case M_DS_2ECX_EBX_ID: //785;
	case M_DS_2ECX_ESP_ID: //786;
	case M_DS_2ECX_EBP_ID: //787;
	case M_DS_2ECX_ESI_ID: //788;
	case M_DS_2ECX_EDI_ID: //789;

	case M_DS_2EDX_EAX_ID: //790;
	case M_DS_2EDX_ECX_ID: //791;
	case M_DS_3EDX_ID: //792;
	case M_DS_2EDX_EBX_ID: //793;
	case M_DS_2EDX_ESP_ID: //794;
	case M_DS_2EDX_EBP_ID: //795;
	case M_DS_2EDX_ESI_ID: //796;
	case M_DS_2EDX_EDI_ID: //797;

	case M_DS_2EBX_EAX_ID: //798;
	case M_DS_2EBX_ECX_ID: //799;
	case M_DS_2EBX_EDX_ID: //800;
	case M_DS_3EBX_ID: //801;
	case M_DS_2EBX_ESP_ID: //802;
	case M_DS_2EBX_EBP_ID: //803;
	case M_DS_2EBX_ESI_ID: //804;
	case M_DS_2EBX_EDI_ID: //805;

	case M_DS_2EBP_EAX_ID: //806;
	case M_DS_2EBP_ECX_ID: //807;
	case M_DS_2EBP_EDX_ID: //808;
	case M_DS_2EBP_EBX_ID: //809;
	case M_DS_2EBP_ESP_ID: //810;
	case M_DS_3EBP_ID: //811;
	case M_DS_2EBP_ESI_ID: //812;
	case M_DS_2EBP_EDI_ID: //813;

	case M_DS_2ESI_EAX_ID: //814;
	case M_DS_2ESI_ECX_ID: //815;
	case M_DS_2ESI_EDX_ID: //816;
	case M_DS_2ESI_EBX_ID: //817;
	case M_DS_2ESI_ESP_ID: //818;
	case M_DS_2ESI_EBP_ID: //819;
	case M_DS_3ESI_ID: //820;
	case M_DS_2ESI_EDI_ID: //821;

	case M_DS_2EDI_EAX_ID: //822;
	case M_DS_2EDI_ECX_ID: //823;
	case M_DS_2EDI_EDX_ID: //824;
	case M_DS_2EDI_EBX_ID: //825;
	case M_DS_2EDI_ESP_ID: //826;
	case M_DS_2EDI_EBP_ID: //827;
	case M_DS_2EDI_ESI_ID: //828;
	case M_DS_3EDI_ID: //829;

	case M_DS_5EAX_ID: //830;
	case M_DS_4EAX_ECX_ID: //831;
	case M_DS_4EAX_EDX_ID: //832;
	case M_DS_4EAX_EBX_ID: //833;
	case M_DS_4EAX_ESP_ID: //834;
	case M_DS_4EAX_EBP_ID: //835;
	case M_DS_4EAX_ESI_ID: //836;
	case M_DS_4EAX_EDI_ID: //837;

	case M_DS_4ECX_EAX_ID: //838;
	case M_DS_5ECX_ID: //839;
	case M_DS_4ECX_EDX_ID: //840;
	case M_DS_4ECX_EBX_ID: //841;
	case M_DS_4ECX_ESP_ID: //842;
	case M_DS_4ECX_EBP_ID: //843;
	case M_DS_4ECX_ESI_ID: //844;
	case M_DS_4ECX_EDI_ID: //845;

	case M_DS_4EDX_EAX_ID: //846;
	case M_DS_4EDX_ECX_ID: //847;
	case M_DS_5EDX_ID: //848;
	case M_DS_4EDX_EBX_ID: //849;
	case M_DS_4EDX_ESP_ID: //850;
	case M_DS_4EDX_EBP_ID: //851;
	case M_DS_4EDX_ESI_ID: //852;
	case M_DS_4EDX_EDI_ID: //853;

	case M_DS_4EBX_EAX_ID: //854;
	case M_DS_4EBX_ECX_ID: //855;
	case M_DS_4EBX_EDX_ID: //856;
	case M_DS_5EBX_ID: //857;
	case M_DS_4EBX_ESP_ID: //858;
	case M_DS_4EBX_EBP_ID: //859;
	case M_DS_4EBX_ESI_ID: //860;
	case M_DS_4EBX_EDI_ID: //861;

	case M_DS_4EBP_EAX_ID: //862;
	case M_DS_4EBP_ECX_ID: //863;
	case M_DS_4EBP_EDX_ID: //864;
	case M_DS_4EBP_EBX_ID: //865;
	case M_DS_4EBP_ESP_ID: //866;
	case M_DS_5EBP_ID: //867;
	case M_DS_4EBP_ESI_ID: //868;
	case M_DS_4EBP_EDI_ID: //869;

	case M_DS_4ESI_EAX_ID: //870;
	case M_DS_4ESI_ECX_ID: //871;
	case M_DS_4ESI_EDX_ID: //872;
	case M_DS_4ESI_EBX_ID: //873;
	case M_DS_4ESI_ESP_ID: //874;
	case M_DS_4ESI_EBP_ID: //875;
	case M_DS_5ESI_ID: //876;
	case M_DS_4ESI_EDI_ID: //877;

	case M_DS_4EDI_EAX_ID: //878;
	case M_DS_4EDI_ECX_ID: //879;
	case M_DS_4EDI_EDX_ID: //880;
	case M_DS_4EDI_EBX_ID: //881;
	case M_DS_4EDI_ESP_ID: //882;
	case M_DS_4EDI_EBP_ID: //883;
	case M_DS_4EDI_ESI_ID: //884;
	case M_DS_5EDI_ID: //885;

	case M_DS_9EAX_ID: //886;
	case M_DS_8EAX_ECX_ID: //887;
	case M_DS_8EAX_EDX_ID: //888;
	case M_DS_8EAX_EBX_ID: //889;
	case M_DS_8EAX_ESP_ID: //890;
	case M_DS_8EAX_EBP_ID: //891;
	case M_DS_8EAX_ESI_ID: //892;
	case M_DS_8EAX_EDI_ID: //893;

	case M_DS_8ECX_EAX_ID: //894;
	case M_DS_9ECX_ID: //895;
	case M_DS_8ECX_EDX_ID: //896;
	case M_DS_8ECX_EBX_ID: //897;
	case M_DS_8ECX_ESP_ID: //898;
	case M_DS_8ECX_EBP_ID: //899;
	case M_DS_8ECX_ESI_ID: //900;
	case M_DS_8ECX_EDI_ID: //901;

	case M_DS_8EDX_EAX_ID: //902;
	case M_DS_8EDX_ECX_ID: //903;
	case M_DS_9EDX_ID: //904;
	case M_DS_8EDX_EBX_ID: //905;
	case M_DS_8EDX_ESP_ID: //906;
	case M_DS_8EDX_EBP_ID: //907;
	case M_DS_8EDX_ESI_ID: //908;
	case M_DS_8EDX_EDI_ID: //909;

	case M_DS_8EBX_EAX_ID: //910;
	case M_DS_8EBX_ECX_ID: //911;
	case M_DS_8EBX_EDX_ID: //912;
	case M_DS_9EBX_ID: //913;
	case M_DS_8EBX_ESP_ID: //914;
	case M_DS_8EBX_EBP_ID: //915;
	case M_DS_8EBX_ESI_ID: //916;
	case M_DS_8EBX_EDI_ID: //917;

	case M_DS_8EBP_EAX_ID: //918;
	case M_DS_8EBP_ECX_ID: //919;
	case M_DS_8EBP_EDX_ID: //920;
	case M_DS_8EBP_EBX_ID: //921;
	case M_DS_8EBP_ESP_ID: //922;
	case M_DS_9EBP_ID: //923;
	case M_DS_8EBP_ESI_ID: //924;
	case M_DS_8EBP_EDI_ID: //925;

	case M_DS_8ESI_EAX_ID: //926;
	case M_DS_8ESI_ECX_ID: //927;
	case M_DS_8ESI_EDX_ID: //928;
	case M_DS_8ESI_EBX_ID: //929;
	case M_DS_8ESI_ESP_ID: //930;
	case M_DS_8ESI_EBP_ID: //931;
	case M_DS_9ESI_ID: //932;
	case M_DS_8ESI_EDI_ID: //933;
    
	case M_DS_8EDI_EAX_ID: //934;
	case M_DS_8EDI_ECX_ID: //935;
	case M_DS_8EDI_EDX_ID: //936;
	case M_DS_8EDI_EBX_ID: //937;
	case M_DS_8EDI_ESP_ID: //938;
	case M_DS_8EDI_EBP_ID: //939;
	case M_DS_8EDI_ESI_ID: //940;
	case M_DS_9EDI_ID: //941;
	    return cpu.ds;

	case M_ES_BX_SI: //182;
	case M_ES_BX_DI: //183;
	case M_ES_BP_SI: //184;
	case M_ES_BP_DI: //185;
	case M_ES_SI: //186;
	case M_ES_DI: //187;
	case M_ES_IW: //188;
	case M_ES_BX: //189;
	case M_ES_BX_SI_IB: //190;
	case M_ES_BX_DI_IB: //191;
	case M_ES_BP_SI_IB: //192;
	case M_ES_BP_DI_IB: //193;
	case M_ES_SI_IB: //194;
	case M_ES_DI_IB: //195;
	case M_ES_BP_IB: //196;
	case M_ES_BX_IB: //197;
	case M_ES_BX_SI_IW: //198;
	case M_ES_BX_DI_IW: //199;
	case M_ES_BP_SI_IW: //200;
	case M_ES_BP_DI_IW: //201;
	case M_ES_SI_IW: //202;
	case M_ES_DI_IW: //203;
	case M_ES_BP_IW: //204;
	case M_ES_BX_IW: //205;
	    return cpu.es;	    
	    
	case M_SS_BX_SI:
	case M_SS_BX_DI:
	case M_SS_BP_SI:
	case M_SS_BP_DI:
	case M_SS_SI:
	case M_SS_DI:
	case M_SS_IW:
	case M_SS_BX:
	case M_SS_BX_SI_IB:
	case M_SS_BX_DI_IB:
	case M_SS_BP_SI_IB:
	case M_SS_BP_DI_IB:
	case M_SS_SI_IB:
	case M_SS_DI_IB:
	case M_SS_BP_IB:
	case M_SS_BX_IB:
	case M_SS_BX_SI_IW:
	case M_SS_BX_DI_IW:
	case M_SS_BP_SI_IW:
	case M_SS_BP_DI_IW:
	case M_SS_SI_IW:
	case M_SS_DI_IW:
	case M_SS_BP_IW:
	case M_SS_BX_IW:

	case M_SS_EBP_ID:
	case M_SS_EBP_IB:
	case M_SS_ESP_IB:
	    return cpu.ss;
	}

	System.err.println("Unimplemented GetSegment Operand: " + microcode);
	throw unimplemented;
    }

    protected final int getAddressOperand()
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case M_ES_BX_SI: case M_CS_BX_SI: case M_SS_BX_SI: case M_DS_BX_SI: case M_FS_BX_SI: case M_GS_BX_SI:
	    return  (cpu.ebx + cpu.esi) & 0xffff;
	case M_ES_BX_DI: case M_CS_BX_DI: case M_SS_BX_DI: case M_DS_BX_DI: case M_FS_BX_DI: case M_GS_BX_DI:
	    return  (cpu.ebx + cpu.edi) & 0xffff;
	case M_ES_BP_SI: case M_CS_BP_SI: case M_SS_BP_SI: case M_DS_BP_SI: case M_FS_BP_SI: case M_GS_BP_SI:
	    return  (cpu.ebp + cpu.esi) & 0xffff;
	case M_ES_BP_DI: case M_CS_BP_DI: case M_SS_BP_DI: case M_DS_BP_DI: case M_FS_BP_DI: case M_GS_BP_DI:
	    return  (cpu.ebp + cpu.edi) & 0xffff;
	case M_ES_SI: case M_CS_SI: case M_SS_SI: case M_DS_SI: case M_FS_SI: case M_GS_SI:
	    return  cpu.esi & 0xffff;
	case M_ES_DI: case M_CS_DI: case M_SS_DI: case M_DS_DI: case M_FS_DI: case M_GS_DI:
	    return  cpu.edi & 0xffff;
	case M_ES_IW: case M_CS_IW: case M_SS_IW: case M_DS_IW: case M_FS_IW: case M_GS_IW:
	    return  getMicrocode() & 0xffff;
	case M_ES_BX: case M_CS_BX: case M_SS_BX: case M_DS_BX: case M_FS_BX: case M_GS_BX:
	    return  cpu.ebx & 0xffff;
	case M_ES_BX_SI_IB: case M_CS_BX_SI_IB: case M_SS_BX_SI_IB: case M_DS_BX_SI_IB: case M_FS_BX_SI_IB: case M_GS_BX_SI_IB:
	case M_ES_BX_SI_IW: case M_CS_BX_SI_IW: case M_SS_BX_SI_IW: case M_DS_BX_SI_IW: case M_FS_BX_SI_IW: case M_GS_BX_SI_IW:
	    return  (cpu.ebx + cpu.esi + getMicrocode()) & 0xffff;
	case M_ES_BX_DI_IB: case M_CS_BX_DI_IB: case M_SS_BX_DI_IB: case M_DS_BX_DI_IB: case M_FS_BX_DI_IB: case M_GS_BX_DI_IB:
	case M_ES_BX_DI_IW: case M_CS_BX_DI_IW: case M_SS_BX_DI_IW: case M_DS_BX_DI_IW: case M_FS_BX_DI_IW: case M_GS_BX_DI_IW:
	    return  (cpu.ebx + cpu.edi + getMicrocode()) & 0xffff;
	case M_ES_BP_SI_IB: case M_CS_BP_SI_IB: case M_SS_BP_SI_IB: case M_DS_BP_SI_IB: case M_FS_BP_SI_IB: case M_GS_BP_SI_IB:
	case M_ES_BP_SI_IW: case M_CS_BP_SI_IW: case M_SS_BP_SI_IW: case M_DS_BP_SI_IW: case M_FS_BP_SI_IW: case M_GS_BP_SI_IW:
	    return  (cpu.ebp + cpu.esi + getMicrocode()) & 0xffff;
	case M_ES_BP_DI_IB: case M_CS_BP_DI_IB: case M_SS_BP_DI_IB: case M_DS_BP_DI_IB: case M_FS_BP_DI_IB: case M_GS_BP_DI_IB:
	case M_ES_BP_DI_IW: case M_CS_BP_DI_IW: case M_SS_BP_DI_IW: case M_DS_BP_DI_IW: case M_FS_BP_DI_IW: case M_GS_BP_DI_IW:
	    return  (cpu.ebp + cpu.edi + getMicrocode()) & 0xffff;
	case M_ES_SI_IB: case M_CS_SI_IB: case M_SS_SI_IB: case M_DS_SI_IB: case M_FS_SI_IB: case M_GS_SI_IB:
	case M_ES_SI_IW: case M_CS_SI_IW: case M_SS_SI_IW: case M_DS_SI_IW: case M_FS_SI_IW: case M_GS_SI_IW:
	    return  (cpu.esi + getMicrocode()) & 0xffff;
	case M_ES_DI_IB: case M_CS_DI_IB: case M_SS_DI_IB: case M_DS_DI_IB: case M_FS_DI_IB: case M_GS_DI_IB:
	case M_ES_DI_IW: case M_CS_DI_IW: case M_SS_DI_IW: case M_DS_DI_IW: case M_FS_DI_IW: case M_GS_DI_IW:
	    return  (cpu.edi + getMicrocode()) & 0xffff;
	case M_ES_BP_IB: case M_CS_BP_IB: case M_SS_BP_IB: case M_DS_BP_IB: case M_FS_BP_IB: case M_GS_BP_IB:
	case M_ES_BP_IW: case M_CS_BP_IW: case M_SS_BP_IW: case M_DS_BP_IW: case M_FS_BP_IW: case M_GS_BP_IW:
	    return  (cpu.ebp + getMicrocode()) & 0xffff;
	case M_ES_BX_IB: case M_CS_BX_IB: case M_SS_BX_IB: case M_DS_BX_IB: case M_FS_BX_IB: case M_GS_BX_IB:
	case M_ES_BX_IW: case M_CS_BX_IW: case M_SS_BX_IW: case M_DS_BX_IW: case M_FS_BX_IW: case M_GS_BX_IW:
	    return  (cpu.ebx + getMicrocode()) & 0xffff;

	case M_DS_EAX: return cpu.eax; //104
	case M_DS_ECX: return cpu.ecx; //105
	case M_DS_EDX: return cpu.edx; //106
	case M_DS_EBX: return cpu.ebx; //107
	case M_DS_ID: return getMicrocode(); //108
	case M_DS_ESI: return cpu.esi; //109
	case M_DS_EDI: return cpu.edi; //110

	case M_SS_EAX_IB: case M_DS_EAX_IB: return cpu.eax + getMicrocode(); //90, 111
	case M_SS_ECX_IB: case M_DS_ECX_IB: return cpu.ecx + getMicrocode(); //91, 112
	case M_SS_EDX_IB: case M_DS_EDX_IB: return cpu.edx + getMicrocode(); //92, 113
	case M_SS_EBX_IB: case M_DS_EBX_IB: return cpu.ebx + getMicrocode(); //93, 114
	case M_SS_EBP_IB: case M_DS_EBP_IB: return cpu.ebp + getMicrocode(); //94, 115 
	case M_SS_ESI_IB: case M_DS_ESI_IB: return cpu.esi + getMicrocode(); //95, 116
	case M_SS_EDI_IB: case M_DS_EDI_IB: return cpu.edi + getMicrocode(); //96, 117

	case M_SS_EAX_ID: case M_DS_EAX_ID: return cpu.eax + getMicrocode(); //97, 118
	case M_SS_ECX_ID: case M_DS_ECX_ID: return cpu.ecx + getMicrocode(); //98, 119
	case M_SS_EDX_ID: case M_DS_EDX_ID: return cpu.edx + getMicrocode(); //99, 120
	case M_SS_EBX_ID: case M_DS_EBX_ID: return cpu.ebx + getMicrocode(); //100, 121
	case M_SS_EBP_ID: case M_DS_EBP_ID: return cpu.ebp + getMicrocode(); //101, 122
	case M_SS_ESI_ID: case M_DS_ESI_ID: return cpu.esi + getMicrocode(); //102, 123
	case M_SS_EDI_ID: case M_DS_EDI_ID: return cpu.edi + getMicrocode(); //103, 124

	case M_DS_2EAX: return cpu.eax + cpu.eax; //2156
	case M_DS_EAX_ECX: return cpu.eax + cpu.ecx; //2157
	case M_DS_EAX_EDX: return cpu.eax + cpu.edx; //2158
	case M_DS_EAX_EBX: return cpu.eax + cpu.ebx; //2159
	case M_DS_EAX_ESP: return cpu.eax + cpu.esp; //2160
	case M_DS_EAX_EBP: return cpu.eax + cpu.ebp; //2161
	case M_DS_EAX_ESI: return cpu.eax + cpu.esi; //2162
	case M_DS_EAX_EDI: return cpu.eax + cpu.edi; //2163

	case M_DS_2ECX: return cpu.ecx + cpu.ecx; //2164
	case M_DS_ECX_EDX: return cpu.ecx + cpu.edx; //2165
	case M_DS_ECX_EBX: return cpu.ecx + cpu.ebx; //2166
	case M_DS_ECX_ESP: case M_SS_ECX_ESP: return cpu.ecx + cpu.esp; //2167
	case M_DS_ECX_EBP: case M_SS_ECX_EBP: return cpu.ecx + cpu.ebp; //2168
	case M_DS_ECX_ESI: return cpu.ecx + cpu.esi; //2169
	case M_DS_ECX_EDI: return cpu.ecx + cpu.edi; //2170

	case M_DS_2EDX: return cpu.edx + cpu.edx; //2171
	case M_DS_EDX_EBX: return cpu.edx + cpu.ebx; //2172
	case M_DS_EDX_ESP: return cpu.edx + cpu.esp; //2173
	case M_DS_EDX_EBP: return cpu.edx + cpu.ebp; //2174
	case M_DS_EDX_ESI: return cpu.edx + cpu.esi; //2175
	case M_DS_EDX_EDI: return cpu.edx + cpu.edi; //2176

	case M_DS_2EBX: return cpu.ebx + cpu.ebx; //2177;
	case M_DS_EBX_ESP: return cpu.ebx + cpu.esp; //2178
	case M_DS_EBX_EBP: return cpu.ebx + cpu.ebp; //2179
	case M_DS_EBX_ESI: return cpu.ebx + cpu.esi; //2180
	case M_DS_EBX_EDI: return cpu.ebx + cpu.edi; //2181

	case M_DS_ESP: case M_SS_ESP: return cpu.esp; //2182

	case M_DS_EBP_ESP: return cpu.ebp + cpu.esp; //2183;
	case M_DS_EBP_ESI: return cpu.ebp + cpu.esi; //2184;
	case M_DS_EBP_EDI: return cpu.ebp + cpu.edi; //2185;

	case M_DS_ESI_ESP: return cpu.esi + cpu.esp; //2186;
	case M_DS_2ESI: return cpu.esi + cpu.esi; //2187;
	case M_DS_ESI_EDI: return cpu.esi + cpu.edi; //2188;

	case M_DS_EDI_ESP: case M_SS_EDI_ESP: return cpu.edi + cpu.esp; //2189;
	case M_DS_2EDI: return cpu.edi + cpu.edi; //2190;

	case M_DS_3EAX: return (cpu.eax << 1) + cpu.eax; //2191;
	case M_DS_2EAX_ECX: return (cpu.eax << 1) + cpu.ecx; //2192;
	case M_DS_2EAX_EDX: return (cpu.eax << 1) + cpu.edx; //2193;
	case M_DS_2EAX_EBX: return (cpu.eax << 1) + cpu.ebx; //2194;
	case M_DS_2EAX_ESP: return (cpu.eax << 1) + cpu.esp; //2195;
	case M_DS_2EAX_ID: return (cpu.eax << 1) + getMicrocode(); //2196;
	case M_DS_2EAX_ESI: return (cpu.eax << 1) + cpu.esi; //2197;
	case M_DS_2EAX_EDI: return (cpu.eax << 1) + cpu.edi; //2198;

	case M_DS_2ECX_EAX: return (cpu.ecx << 1) + cpu.eax; //2199;
	case M_DS_3ECX: return (cpu.ecx << 1) + cpu.ecx; //2200;
	case M_DS_2ECX_EDX: return (cpu.ecx << 1) + cpu.edx; //2201;
	case M_DS_2ECX_EBX: return (cpu.ecx << 1) + cpu.ebx; //2202;
	case M_DS_2ECX_ESP: return (cpu.ecx << 1) + cpu.esp; //2203;
	case M_DS_2ECX_ID: return (cpu.ecx << 1) + getMicrocode(); //2204;
	case M_DS_2ECX_ESI: return (cpu.ecx << 1) + cpu.esi; //2205;
	case M_DS_2ECX_EDI: return (cpu.ecx << 1) + cpu.edi; //2206;

	case M_DS_2EDX_EAX: return (cpu.edx << 1) + cpu.eax; //2207
	case M_DS_2EDX_ECX: return (cpu.edx << 1) + cpu.ecx; //2208
	case M_DS_3EDX: return (cpu.edx << 1) + cpu.edx; //2209
	case M_DS_2EDX_EBX: return (cpu.edx << 1) + cpu.ebx; //2210
	case M_DS_2EDX_ESP: return (cpu.edx << 1) + cpu.esp; //2211
	case M_DS_2EDX_ID: return (cpu.edx << 1) + getMicrocode(); //2212
	case M_DS_2EDX_ESI: return (cpu.edx << 1) + cpu.esi; //2213
	case M_DS_2EDX_EDI: return (cpu.edx << 1) + cpu.edi; //2214

	case M_DS_2EBX_EAX: return (cpu.ebx << 1) + cpu.eax; //2215;
	case M_DS_2EBX_ECX: return (cpu.ebx << 1) + cpu.ecx; //2216;
	case M_DS_2EBX_EDX: return (cpu.ebx << 1) + cpu.edx; //2217;
	case M_DS_3EBX: return (cpu.ebx << 1) + cpu.ebx; //2218;
	case M_DS_2EBX_ESP: return (cpu.ebx << 1) + cpu.esp; //2219;
	case M_DS_2EBX_ID: return (cpu.ebx << 1) + getMicrocode(); //2220;
	case M_DS_2EBX_ESI: return (cpu.ebx << 1) + cpu.esi; //2221;
	case M_DS_2EBX_EDI: return (cpu.ebx << 1) + cpu.edi; //2222;

	case M_DS_2EBP_EAX: return (cpu.ebp << 1) + cpu.eax; //2223;
	case M_DS_2EBP_ECX: return (cpu.ebp << 1) + cpu.ecx; //2224;
	case M_DS_2EBP_EDX: return (cpu.ebp << 1) + cpu.edx; //2225;
	case M_DS_2EBP_EBX: return (cpu.ebp << 1) + cpu.ebx; //2226;
	case M_DS_2EBP_ESP: return (cpu.ebp << 1) + cpu.esp; //2227;
	case M_DS_2EBP_ID: return (cpu.ebp << 1) + getMicrocode(); //2228;
	case M_DS_2EBP_ESI: return (cpu.ebp << 1) + cpu.esi; //2229;
	case M_DS_2EBP_EDI: return (cpu.ebp << 1) + cpu.edi; //2230;

	case M_DS_2ESI_EAX: return (cpu.esi << 1) + cpu.eax; //2231;
	case M_DS_2ESI_ECX: return (cpu.esi << 1) + cpu.ecx; //2232;
	case M_DS_2ESI_EDX: return (cpu.esi << 1) + cpu.edx; //2233;
	case M_DS_2ESI_EBX: return (cpu.esi << 1) + cpu.ebx; //2234;
	case M_DS_2ESI_ESP: return (cpu.esi << 1) + cpu.esp; //2235;
	case M_DS_2ESI_ID: return (cpu.esi << 1) + getMicrocode(); //2236;
	case M_DS_3ESI: return (cpu.esi << 1) + cpu.esi; //2237;
	case M_DS_2ESI_EDI: return (cpu.esi << 1) + cpu.edi; //2238;

	case M_DS_2EDI_EAX: return (cpu.edi << 1) + cpu.eax; //2239;
	case M_DS_2EDI_ECX: return (cpu.edi << 1) + cpu.ecx; //2240;
	case M_DS_2EDI_EDX: return (cpu.edi << 1) + cpu.edx; //2241;
	case M_DS_2EDI_EBX: return (cpu.edi << 1) + cpu.ebx; //2242;
	case M_DS_2EDI_ESP: return (cpu.edi << 1) + cpu.esp; //2243;
	case M_DS_2EDI_ID: return (cpu.edi << 1) + getMicrocode(); //2244;
	case M_DS_2EDI_ESI: return (cpu.edi << 1) + cpu.esi; //2245;
	case M_DS_3EDI: return (cpu.edi << 1) + cpu.edi; //2246;

	case M_DS_5EAX: return (cpu.eax << 2) + cpu.eax; //2247
	case M_DS_4EAX_ECX: return (cpu.eax << 2) + cpu.ecx; //2248
	case M_DS_4EAX_EDX: return (cpu.eax << 2) + cpu.edx; //2249
	case M_DS_4EAX_EBX: return (cpu.eax << 2) + cpu.ebx; //2250
	case M_DS_4EAX_ESP: case M_SS_4EAX_ESP: return (cpu.eax << 2) + cpu.esp; //2251
	case M_DS_4EAX_ID: return (cpu.eax << 2) + getMicrocode(); //2252
	case M_DS_4EAX_ESI: return (cpu.eax << 2) + cpu.esi; //2253
	case M_DS_4EAX_EDI: return (cpu.eax << 2) + cpu.edi; //2254

	case M_DS_4ECX_EAX: return (cpu.ecx << 2) + cpu.eax; //2255
	case M_DS_5ECX: return (cpu.ecx << 2) + cpu.ecx; //2256
	case M_DS_4ECX_EDX: return (cpu.ecx << 2) + cpu.edx; //2257
	case M_DS_4ECX_EBX: return (cpu.ecx << 2) + cpu.ebx; //2258
	case M_DS_4ECX_ESP: return (cpu.ecx << 2) + cpu.esp; //2259
	case M_DS_4ECX_ID: return (cpu.ecx << 2) + getMicrocode(); //2260
	case M_DS_4ECX_ESI: return (cpu.ecx << 2) + cpu.esi; //2261
	case M_DS_4ECX_EDI: return (cpu.ecx << 2) + cpu.edi; //2262

	case M_DS_4EDX_EAX: return (cpu.edx << 2) + cpu.eax; //2263
	case M_DS_4EDX_ECX: return (cpu.edx << 2) + cpu.ecx; //2264
	case M_DS_5EDX: return (cpu.edx << 2) + cpu.edx; //2265
	case M_DS_4EDX_EBX: return (cpu.edx << 2) + cpu.ebx; //2266
	case M_DS_4EDX_ESP: return (cpu.edx << 2) + cpu.esp; //2267
	case M_DS_4EDX_ID: return (cpu.edx << 2) + getMicrocode(); //2268
	case M_DS_4EDX_ESI: return (cpu.edx << 2) + cpu.esi; //2269
	case M_DS_4EDX_EDI: return (cpu.edx << 2) + cpu.edi; //2270

	case M_DS_4EBX_EAX: return (cpu.ebx << 2) + cpu.eax; //2271
	case M_DS_4EBX_ECX: return (cpu.ebx << 2) + cpu.ecx; //2272
	case M_DS_4EBX_EDX: return (cpu.ebx << 2) + cpu.edx; //2273
	case M_DS_5EBX: return (cpu.ebx << 2) + cpu.ebx; //2274
	case M_DS_4EBX_ESP: return (cpu.ebx << 2) + cpu.esp; //2275
	case M_DS_4EBX_ID: return (cpu.ebx << 2) + getMicrocode(); //2276
	case M_DS_4EBX_ESI: return (cpu.ebx << 2) + cpu.esi; //2277
	case M_DS_4EBX_EDI: return (cpu.ebx << 2) + cpu.edi; //2278

	case M_DS_4EBP_EAX: return (cpu.ebp << 2) + cpu.eax; //2279
	case M_DS_4EBP_ECX: return (cpu.ebp << 2) + cpu.ecx; //2280
	case M_DS_4EBP_EDX: return (cpu.ebp << 2) + cpu.edx; //2281
	case M_DS_4EBP_EBX: return (cpu.ebp << 2) + cpu.ebx; //2282
	case M_DS_4EBP_ESP: return (cpu.ebp << 2) + cpu.esp; //2283
	case M_DS_4EBP_ID: return (cpu.ebp << 2) + getMicrocode(); //2284
	case M_DS_4EBP_ESI: return (cpu.ebp << 2) + cpu.esi; //2285
	case M_DS_4EBP_EDI: return (cpu.ebp << 2) + cpu.edi; //2286

	case M_DS_4ESI_EAX: return (cpu.esi << 2) + cpu.eax; //2287
	case M_DS_4ESI_ECX: return (cpu.esi << 2) + cpu.ecx; //2288
	case M_DS_4ESI_EDX: return (cpu.esi << 2) + cpu.edx; //2289
	case M_DS_4ESI_EBX: return (cpu.esi << 2) + cpu.ebx; //2290
	case M_DS_4ESI_ESP: return (cpu.esi << 2) + cpu.esp; //2291
	case M_DS_4ESI_ID: return (cpu.esi << 2) + getMicrocode(); //2292
	case M_DS_5ESI: return (cpu.esi << 2) + cpu.esi; //2293
	case M_DS_4ESI_EDI: return (cpu.esi << 2) + cpu.edi; //2294

	case M_DS_4EDI_EAX: return (cpu.edi << 2) + cpu.eax; //2295;
	case M_DS_4EDI_ECX: return (cpu.edi << 2) + cpu.ecx; //2296;
	case M_DS_4EDI_EDX: return (cpu.edi << 2) + cpu.edx; //2297;
	case M_DS_4EDI_EBX: return (cpu.edi << 2) + cpu.ebx; //2298;
	case M_DS_4EDI_ESP: return (cpu.edi << 2) + cpu.esp; //2299;
	case M_DS_4EDI_ID: return (cpu.edi << 2) + getMicrocode(); //2300;
	case M_DS_4EDI_ESI: return (cpu.edi << 2) + cpu.esi; //2301;
	case M_DS_5EDI: return (cpu.edi << 2) + cpu.edi; //2302;

	case M_DS_9EAX: return (cpu.eax << 3) + cpu.eax; //2303;
	case M_DS_8EAX_ECX: return (cpu.eax << 3) + cpu.ecx; //2304;
	case M_DS_8EAX_EDX: return (cpu.eax << 3) + cpu.edx; //2305;
	case M_DS_8EAX_EBX: return (cpu.eax << 3) + cpu.ebx; //2306;
	case M_DS_8EAX_ESP: return (cpu.eax << 3) + cpu.esp; //2307;
	case M_DS_8EAX_ID: return (cpu.eax << 3) + getMicrocode(); //2308;
	case M_DS_8EAX_ESI: return (cpu.eax << 3) + cpu.esi; //2309;
	case M_DS_8EAX_EDI: return (cpu.eax << 3) + cpu.edi; //2310;
	    
	case M_DS_8ECX_EAX: return (cpu.ecx << 3) + cpu.eax; //2311;
	case M_DS_9ECX: return (cpu.ecx << 3) + cpu.ecx; //2312;
	case M_DS_8ECX_EDX: return (cpu.ecx << 3) + cpu.edx; //2313;
	case M_DS_8ECX_EBX: return (cpu.ecx << 3) + cpu.ebx; //2314;
	case M_DS_8ECX_ESP: return (cpu.ecx << 3) + cpu.esp; //2315;
	case M_DS_8ECX_ID: return (cpu.ecx << 3) + getMicrocode(); //2316;
	case M_DS_8ECX_ESI: return (cpu.ecx << 3) + cpu.esi; //2317;
	case M_DS_8ECX_EDI: return (cpu.ecx << 3) + cpu.edi; //2318;
	    
	case M_DS_8EDX_EAX: return (cpu.edx << 3) + cpu.eax; //2319;
	case M_DS_8EDX_ECX: return (cpu.edx << 3) + cpu.ecx; //2320;
	case M_DS_9EDX: return (cpu.edx << 3) + cpu.edx; //2321;
	case M_DS_8EDX_EBX: return (cpu.edx << 3) + cpu.ebx; //2322;
	case M_DS_8EDX_ESP: return (cpu.edx << 3) + cpu.esp; //2323;
	case M_DS_8EDX_ID: return (cpu.edx << 3) + getMicrocode(); //2324;
	case M_DS_8EDX_ESI: return (cpu.edx << 3) + cpu.esi; //2325;
	case M_DS_8EDX_EDI: return (cpu.edx << 3) + cpu.edi; //2326;
	    
	case M_DS_8EBX_EAX: return (cpu.ebx << 3) + cpu.eax; //2327
	case M_DS_8EBX_ECX: return (cpu.ebx << 3) + cpu.ecx; //2328
	case M_DS_8EBX_EDX: return (cpu.ebx << 3) + cpu.edx; //2329
	case M_DS_9EBX: return (cpu.ebx << 3) + cpu.ebx; //2330
	case M_DS_8EBX_ESP: return (cpu.ebx << 3) + cpu.esp; //2331
	case M_DS_8EBX_ID: return (cpu.ebx << 3) + getMicrocode(); //2332
	case M_DS_8EBX_ESI: return (cpu.ebx << 3) + cpu.esi; //2333
	case M_DS_8EBX_EDI: return (cpu.ebx << 3) + cpu.edi; //2334
	    
	case M_DS_8EBP_EAX: return (cpu.ebp << 3) + cpu.eax; //2335;
	case M_DS_8EBP_ECX: return (cpu.ebp << 3) + cpu.ecx; //2336;
	case M_DS_8EBP_EDX: return (cpu.ebp << 3) + cpu.edx; //2337;
	case M_DS_8EBP_EBX: return (cpu.ebp << 3) + cpu.ebx; //2338;
	case M_DS_8EBP_ESP: return (cpu.ebp << 3) + cpu.esp; //2339;
	case M_DS_8EBP_ID: return (cpu.ebp << 3) + getMicrocode(); //2340;
	case M_DS_8EBP_ESI: return (cpu.ebp << 3) + cpu.esi; //2341;
	case M_DS_8EBP_EDI: return (cpu.ebp << 3) + cpu.edi; //2342;
	    
	case M_DS_8ESI_EAX: return (cpu.esi << 3) + cpu.eax; //2343;
	case M_DS_8ESI_ECX: return (cpu.esi << 3) + cpu.ecx; //2344;
	case M_DS_8ESI_EDX: return (cpu.esi << 3) + cpu.edx; //2345;
	case M_DS_8ESI_EBX: return (cpu.esi << 3) + cpu.ebx; //2346;
	case M_DS_8ESI_ESP: return (cpu.esi << 3) + cpu.esp; //2347;
	case M_DS_8ESI_ID: return (cpu.esi << 3) + getMicrocode(); //2348;
	case M_DS_9ESI: return (cpu.esi << 3) + cpu.esi; //2349;
	case M_DS_8ESI_EDI: return (cpu.esi << 3) + cpu.edi; //2350;
	    
	case M_DS_8EDI_EAX: return (cpu.edi << 3) + cpu.eax; //2351;
	case M_DS_8EDI_ECX: return (cpu.edi << 3) + cpu.ecx; //2352;
	case M_DS_8EDI_EDX: return (cpu.edi << 3) + cpu.edx; //2353;
	case M_DS_8EDI_EBX: return (cpu.edi << 3) + cpu.ebx; //2354;
	case M_DS_8EDI_ESP: return (cpu.edi << 3) + cpu.esp; //2355;
	case M_DS_8EDI_ID: return (cpu.edi << 3) + getMicrocode(); //2356;
	case M_DS_8EDI_ESI: return (cpu.edi << 3) + cpu.esi; //2357;
	case M_DS_9EDI: return (cpu.edi << 3) + cpu.edi; //2358;
	    
	case M_DS_2EAX_IB: return cpu.eax + cpu.eax + getMicrocode(); //2359

	case M_DS_EAX_ECX_IB: return cpu.eax + cpu.ecx + getMicrocode(); //2360
	case M_DS_EAX_EDX_IB: return cpu.eax + cpu.edx + getMicrocode(); //2361
	case M_DS_EAX_EBX_IB: return cpu.eax + cpu.ebx + getMicrocode(); //2362
	case M_DS_EAX_ESP_IB: case M_SS_EAX_ESP_IB: return cpu.eax + cpu.esp + getMicrocode(); //2363
	case M_DS_EAX_EBP_IB: case M_SS_EAX_EBP_IB: return cpu.eax + cpu.ebp + getMicrocode(); //2364
	case M_DS_EAX_ESI_IB: return cpu.eax + cpu.esi + getMicrocode(); //2365
	case M_DS_EAX_EDI_IB: return cpu.eax + cpu.edi + getMicrocode(); //2366

	case M_DS_2ECX_IB: return cpu.ecx + cpu.ecx + getMicrocode(); //2367
	case M_DS_ECX_EDX_IB: return cpu.ecx + cpu.edx + getMicrocode(); //2368
	case M_DS_ECX_EBX_IB: return cpu.ecx + cpu.ebx + getMicrocode(); //2369
	case M_DS_ECX_ESP_IB: case M_SS_ECX_ESP_IB: return cpu.ecx + cpu.esp + getMicrocode(); //2370
	case M_DS_ECX_EBP_IB: case M_SS_ECX_EBP_IB: return cpu.ecx + cpu.ebp + getMicrocode(); //2371
	case M_DS_ECX_ESI_IB: return cpu.ecx + cpu.esi + getMicrocode(); //2372
	case M_DS_ECX_EDI_IB: return cpu.ecx + cpu.edi + getMicrocode(); //2373

	case M_DS_2EDX_IB: return cpu.edx + cpu.edx + getMicrocode(); //2374
	case M_DS_EDX_EBX_IB: return cpu.edx + cpu.ebx + getMicrocode(); //2375
	case M_DS_EDX_ESP_IB: case M_SS_EDX_ESP_IB: return cpu.edx + cpu.esp + getMicrocode(); //2376
	case M_DS_EDX_EBP_IB: case M_SS_EDX_EBP_IB: return cpu.edx + cpu.ebp + getMicrocode(); //2377
	case M_DS_EDX_ESI_IB: return cpu.edx + cpu.esi + getMicrocode(); //2378
	case M_DS_EDX_EDI_IB: return cpu.edx + cpu.edi + getMicrocode(); //2379

	case M_DS_2EBX_IB: return cpu.ebx + cpu.ebx + getMicrocode(); //2380
	case M_DS_EBX_ESP_IB: case M_SS_EBX_ESP_IB: return cpu.ebx + cpu.esp + getMicrocode(); //2381
	case M_DS_EBX_EBP_IB: case M_SS_EBX_EBP_IB: return cpu.ebx + cpu.ebp + getMicrocode(); //2382
	case M_DS_EBX_ESI_IB: return cpu.ebx + cpu.esi + getMicrocode(); //2383
	case M_DS_EBX_EDI_IB: return cpu.ebx + cpu.edi + getMicrocode(); //2384

	case M_DS_ESP_IB: case M_SS_ESP_IB: return cpu.esp + getMicrocode(); //2385

	case M_DS_EBP_ESP_IB: return cpu.ebp + cpu.esp + getMicrocode(); //2386;
	case M_DS_2EBP_IB: case M_SS_2EBP_IB: return cpu.ebp + cpu.ebp + getMicrocode(); //2387;
	case M_DS_EBP_ESI_IB: case M_SS_EBP_ESI_IB: return cpu.ebp + cpu.esi + getMicrocode(); //2388;
	case M_DS_EBP_EDI_IB: case M_SS_EBP_EDI_IB: return cpu.ebp + cpu.edi + getMicrocode(); //2389;

	case M_DS_ESI_ESP_IB: case M_SS_ESI_ESP_IB: return cpu.esi + cpu.esp + getMicrocode(); //2390
	case M_DS_2ESI_IB: return cpu.esi + cpu.esi + getMicrocode(); //2391
	case M_DS_ESI_EDI_IB: return cpu.esi + cpu.edi + getMicrocode(); //2392

	case M_DS_EDI_ESP_IB: return cpu.edi + cpu.esp + getMicrocode(); //2393;
	case M_DS_2EDI_IB: return cpu.edi + cpu.edi + getMicrocode(); //2394;

	case M_DS_3EAX_IB: return (cpu.eax << 1) + cpu.eax + getMicrocode(); //2395;
	case M_DS_2EAX_ECX_IB: return (cpu.eax << 1) + cpu.ecx + getMicrocode(); //2396;
	case M_DS_2EAX_EDX_IB: return (cpu.eax << 1) + cpu.edx + getMicrocode(); //2397;
	case M_DS_2EAX_EBX_IB: return (cpu.eax << 1) + cpu.ebx + getMicrocode(); //2398;
	case M_DS_2EAX_ESP_IB: return (cpu.eax << 1) + cpu.esp + getMicrocode(); //2399;
	case M_DS_2EAX_EBP_IB: case M_SS_2EAX_EBP_IB: return (cpu.eax << 1) + cpu.ebp + getMicrocode(); //2400;
	case M_DS_2EAX_ESI_IB: return (cpu.eax << 1) + cpu.esi + getMicrocode(); //2401;
	case M_DS_2EAX_EDI_IB: return (cpu.eax << 1) + cpu.edi + getMicrocode(); //2402;
	    
	case M_DS_2ECX_EAX_IB: return (cpu.ecx << 1) + cpu.eax + getMicrocode(); //2403;
	case M_DS_3ECX_IB: return (cpu.ecx << 1) + cpu.ecx + getMicrocode(); //2404;
	case M_DS_2ECX_EDX_IB: return (cpu.ecx << 1) + cpu.edx + getMicrocode(); //2405;
	case M_DS_2ECX_EBX_IB: return (cpu.ecx << 1) + cpu.ebx + getMicrocode(); //2406;
	case M_DS_2ECX_ESP_IB: return (cpu.ecx << 1) + cpu.esp + getMicrocode(); //2407;
	case M_DS_2ECX_EBP_IB: return (cpu.ecx << 1) + cpu.ebp + getMicrocode(); //2408;
	case M_DS_2ECX_ESI_IB: return (cpu.ecx << 1) + cpu.esi + getMicrocode(); //2409;
	case M_DS_2ECX_EDI_IB: return (cpu.ecx << 1) + cpu.edi + getMicrocode(); //2410;

	case M_DS_2EDX_EAX_IB: return (cpu.edx << 1) + cpu.eax + getMicrocode(); //2411;
	case M_DS_2EDX_ECX_IB: return (cpu.edx << 1) + cpu.ecx + getMicrocode(); //2412;
	case M_DS_3EDX_IB: return (cpu.edx << 1) + cpu.edx + getMicrocode(); //2413;
	case M_DS_2EDX_EBX_IB: return (cpu.edx << 1) + cpu.ebx + getMicrocode(); //2414;
	case M_DS_2EDX_ESP_IB: case M_SS_2EDX_ESP_IB: return (cpu.edx << 1) + cpu.esp + getMicrocode(); //2415;
	case M_DS_2EDX_EBP_IB: case M_SS_2EDX_EBP_IB: return (cpu.edx << 1) + cpu.ebp + getMicrocode(); //2416;
	case M_DS_2EDX_ESI_IB: return (cpu.edx << 1) + cpu.esi + getMicrocode(); //2417;
	case M_DS_2EDX_EDI_IB: return (cpu.edx << 1) + cpu.edi + getMicrocode(); //2418;
	    
	case M_DS_2EBX_EAX_IB: return (cpu.ebx << 1) + cpu.eax + getMicrocode(); //2419;
	case M_DS_2EBX_ECX_IB: return (cpu.ebx << 1) + cpu.ecx + getMicrocode(); //2420;
	case M_DS_2EBX_EDX_IB: return (cpu.ebx << 1) + cpu.edx + getMicrocode(); //2421;
	case M_DS_3EBX_IB: return (cpu.ebx << 1) + cpu.ebx + getMicrocode(); //2422;
	case M_DS_2EBX_ESP_IB: return (cpu.ebx << 1) + cpu.esp + getMicrocode(); //2423;
	case M_DS_2EBX_EBP_IB: return (cpu.ebx << 1) + cpu.ebp + getMicrocode(); //2424;
	case M_DS_2EBX_ESI_IB: return (cpu.ebx << 1) + cpu.esi + getMicrocode(); //2425;
	case M_DS_2EBX_EDI_IB: return (cpu.ebx << 1) + cpu.edi + getMicrocode(); //2426;
	    
	case M_DS_2EBP_EAX_IB: return (cpu.ebp << 1) + cpu.eax + getMicrocode(); //2427;
	case M_DS_2EBP_ECX_IB: return (cpu.ebp << 1) + cpu.ecx + getMicrocode(); //2428;
	case M_DS_2EBP_EDX_IB: return (cpu.ebp << 1) + cpu.edx + getMicrocode(); //2429;
	case M_DS_2EBP_EBX_IB: return (cpu.ebp << 1) + cpu.ebx + getMicrocode(); //2430;
	case M_DS_2EBP_ESP_IB: return (cpu.ebp << 1) + cpu.esp + getMicrocode(); //2431;
	case M_DS_3EBP_IB: case M_SS_3EBP_IB: return (cpu.ebp << 1) + cpu.ebp + getMicrocode(); //2432;
	case M_DS_2EBP_ESI_IB: return (cpu.ebp << 1) + cpu.esi + getMicrocode(); //2433;
	case M_DS_2EBP_EDI_IB: return (cpu.ebp << 1) + cpu.edi + getMicrocode(); //2434;
	    
	case M_DS_2ESI_EAX_IB: return (cpu.esi << 1) + cpu.eax + getMicrocode(); //2435;
	case M_DS_2ESI_ECX_IB: return (cpu.esi << 1) + cpu.ecx + getMicrocode(); //2436;
	case M_DS_2ESI_EDX_IB: return (cpu.esi << 1) + cpu.edx + getMicrocode(); //2437;
	case M_DS_2ESI_EBX_IB: return (cpu.esi << 1) + cpu.ebx + getMicrocode(); //2438;
	case M_DS_2ESI_ESP_IB: return (cpu.esi << 1) + cpu.esp + getMicrocode(); //2439;
	case M_DS_2ESI_EBP_IB: return (cpu.esi << 1) + cpu.ebp + getMicrocode(); //2440;
	case M_DS_3ESI_IB: return (cpu.esi << 1) + cpu.esi + getMicrocode(); //2441;
	case M_DS_2ESI_EDI_IB: return (cpu.esi << 1) + cpu.edi + getMicrocode(); //2442;
	    
	case M_DS_2EDI_EAX_IB: return (cpu.edi << 1) + cpu.eax + getMicrocode(); //2443;
	case M_DS_2EDI_ECX_IB: return (cpu.edi << 1) + cpu.ecx + getMicrocode(); //2444;
	case M_DS_2EDI_EDX_IB: return (cpu.edi << 1) + cpu.edx + getMicrocode(); //2445;
	case M_DS_2EDI_EBX_IB: return (cpu.edi << 1) + cpu.ebx + getMicrocode(); //2446;
	case M_DS_2EDI_ESP_IB: case M_SS_2EDI_ESP_IB: return (cpu.edi << 1) + cpu.esp + getMicrocode(); //2447;
	case M_DS_2EDI_EBP_IB: case M_SS_2EDI_EBP_IB: return (cpu.edi << 1) + cpu.ebp + getMicrocode(); //2448;
	case M_DS_2EDI_ESI_IB: return (cpu.edi << 1) + cpu.esi + getMicrocode(); //2449;
	case M_DS_3EDI_IB: return (cpu.edi << 1) + cpu.edi + getMicrocode(); //2450;

	case M_DS_5EAX_IB: return (cpu.eax << 2) + cpu.eax + getMicrocode(); //2451;
	case M_DS_4EAX_ECX_IB: return (cpu.eax << 2) + cpu.ecx + getMicrocode(); //2452;
	case M_DS_4EAX_EDX_IB: return (cpu.eax << 2) + cpu.edx + getMicrocode(); //2453;
	case M_DS_4EAX_EBX_IB: return (cpu.eax << 2) + cpu.ebx + getMicrocode(); //2454;
	case M_DS_4EAX_ESP_IB: case M_SS_4EAX_ESP_IB: return (cpu.eax << 2) + cpu.esp + getMicrocode(); //2455;
	case M_DS_4EAX_EBP_IB: case M_SS_4EAX_EBP_IB: return (cpu.eax << 2) + cpu.ebp + getMicrocode(); //2456;
	case M_DS_4EAX_ESI_IB: return (cpu.eax << 2) + cpu.esi + getMicrocode(); //2457;
	case M_DS_4EAX_EDI_IB: return (cpu.eax << 2) + cpu.edi + getMicrocode(); //2458;

	case M_DS_4ECX_EAX_IB: return (cpu.ecx << 2) + cpu.eax + getMicrocode(); //2459;
	case M_DS_5ECX_IB: return (cpu.ecx << 2) + cpu.ecx + getMicrocode(); //2460;
	case M_DS_4ECX_EDX_IB: return (cpu.ecx << 2) + cpu.edx + getMicrocode(); //2461;
	case M_DS_4ECX_EBX_IB: return (cpu.ecx << 2) + cpu.ebx + getMicrocode(); //2462;
	case M_DS_4ECX_ESP_IB: case M_SS_4ECX_ESP_IB: return (cpu.ecx << 2) + cpu.esp + getMicrocode(); //2463;
	case M_DS_4ECX_EBP_IB: case M_SS_4ECX_EBP_IB: return (cpu.ecx << 2) + cpu.ebp + getMicrocode(); //2464;
	case M_DS_4ECX_ESI_IB: return (cpu.ecx << 2) + cpu.esi + getMicrocode(); //2465;
	case M_DS_4ECX_EDI_IB: return (cpu.ecx << 2) + cpu.edi + getMicrocode(); //2466;

	case M_DS_4EDX_EAX_IB: return (cpu.edx << 2) + cpu.eax + getMicrocode(); //2467
	case M_DS_4EDX_ECX_IB: return (cpu.edx << 2) + cpu.ecx + getMicrocode(); //2468
	case M_DS_5EDX_IB: return (cpu.edx << 2) + cpu.edx + getMicrocode(); //2469
	case M_DS_4EDX_EBX_IB: return (cpu.edx << 2) + cpu.ebx + getMicrocode(); //2470
	case M_DS_4EDX_ESP_IB: case M_SS_4EDX_ESP_IB: return (cpu.edx << 2) + cpu.esp + getMicrocode(); //2471
	case M_DS_4EDX_EBP_IB: case M_SS_4EDX_EBP_IB: return (cpu.edx << 2) + cpu.ebp + getMicrocode(); //2472
	case M_DS_4EDX_ESI_IB: return (cpu.edx << 2) + cpu.esi + getMicrocode(); //2473
	case M_DS_4EDX_EDI_IB: return (cpu.edx << 2) + cpu.edi + getMicrocode(); //2474

	case M_DS_4EBX_EAX_IB: return (cpu.ebx << 2) + cpu.eax + getMicrocode(); //2475;
	case M_DS_4EBX_ECX_IB: return (cpu.ebx << 2) + cpu.ecx + getMicrocode(); //2476;
	case M_DS_4EBX_EDX_IB: return (cpu.ebx << 2) + cpu.edx + getMicrocode(); //2477;
	case M_DS_5EBX_IB: return (cpu.ebx << 2) + cpu.ebx + getMicrocode(); //2478;
	case M_DS_4EBX_ESP_IB: case M_SS_4EBX_ESP_IB: return (cpu.ebx << 2) + cpu.esp + getMicrocode(); //2479;
	case M_DS_4EBX_EBP_IB: case M_SS_4EBX_EBP_IB: return (cpu.ebx << 2) + cpu.ebp + getMicrocode(); //2480;
	case M_DS_4EBX_ESI_IB: return (cpu.ebx << 2) + cpu.esi + getMicrocode(); //2481;
	case M_DS_4EBX_EDI_IB: return (cpu.ebx << 2) + cpu.edi + getMicrocode(); //2482;

	case M_DS_4EBP_EAX_IB: return (cpu.ebp << 2) + cpu.eax + getMicrocode(); //2483;
	case M_DS_4EBP_ECX_IB: return (cpu.ebp << 2) + cpu.ecx + getMicrocode(); //2484;
	case M_DS_4EBP_EDX_IB: return (cpu.ebp << 2) + cpu.edx + getMicrocode(); //2485;
	case M_DS_4EBP_EBX_IB: return (cpu.ebp << 2) + cpu.ebx + getMicrocode(); //2486;
	case M_DS_4EBP_ESP_IB: return (cpu.ebp << 2) + cpu.esp + getMicrocode(); //2487;
	case M_DS_5EBP_IB: case M_SS_5EBP_IB: return (cpu.ebp << 2) + cpu.ebp + getMicrocode(); //2488;
	case M_DS_4EBP_ESI_IB: return (cpu.ebp << 2) + cpu.esi + getMicrocode(); //2489;
	case M_DS_4EBP_EDI_IB: return (cpu.ebp << 2) + cpu.edi + getMicrocode(); //2490;
	    
	case M_DS_4ESI_EAX_IB: return (cpu.esi << 2) + cpu.eax + getMicrocode(); //2491;
	case M_DS_4ESI_ECX_IB: return (cpu.esi << 2) + cpu.ecx + getMicrocode(); //2492;
	case M_DS_4ESI_EDX_IB: return (cpu.esi << 2) + cpu.edx + getMicrocode(); //2493;
	case M_DS_4ESI_EBX_IB: return (cpu.esi << 2) + cpu.ebx + getMicrocode(); //2494;
	case M_DS_4ESI_ESP_IB: case M_SS_4ESI_ESP_IB: return (cpu.esi << 2) + cpu.esp + getMicrocode(); //2495;
	case M_DS_4ESI_EBP_IB: case M_SS_4ESI_EBP_IB: return (cpu.esi << 2) + cpu.ebp + getMicrocode(); //2496
	case M_DS_5ESI_IB: return (cpu.esi << 2) + cpu.esi + getMicrocode(); //2497;
	case M_DS_4ESI_EDI_IB: return (cpu.esi << 2) + cpu.edi + getMicrocode(); //2498;
	    
	case M_DS_4EDI_EAX_IB: return (cpu.edi << 2) + cpu.eax + getMicrocode(); //2499;
	case M_DS_4EDI_ECX_IB: return (cpu.edi << 2) + cpu.ecx + getMicrocode(); //2500;
	case M_DS_4EDI_EDX_IB: return (cpu.edi << 2) + cpu.edx + getMicrocode(); //2501;
	case M_DS_4EDI_EBX_IB: return (cpu.edi << 2) + cpu.ebx + getMicrocode(); //2502;
	case M_DS_4EDI_ESP_IB: case M_SS_4EDI_ESP_IB: return (cpu.edi << 2) + cpu.esp + getMicrocode(); //2503;
	case M_DS_4EDI_EBP_IB: case M_SS_4EDI_EBP_IB: return (cpu.edi << 2) + cpu.ebp + getMicrocode(); //2504;
	case M_DS_4EDI_ESI_IB: return (cpu.edi << 2) + cpu.esi + getMicrocode(); //2505;
	case M_DS_5EDI_IB: return (cpu.edi << 2) + cpu.edi + getMicrocode(); //2506;

	case M_DS_9EAX_IB: return (cpu.eax << 3) + cpu.eax + getMicrocode(); //2507;
	case M_DS_8EAX_ECX_IB: return (cpu.eax << 3) + cpu.ecx + getMicrocode(); //2508;
	case M_DS_8EAX_EDX_IB: return (cpu.eax << 3) + cpu.edx + getMicrocode(); //2509;
	case M_DS_8EAX_EBX_IB: return (cpu.eax << 3) + cpu.ebx + getMicrocode(); //2510;
	case M_DS_8EAX_ESP_IB: case M_SS_8EAX_ESP_IB: return (cpu.eax << 3) + cpu.esp + getMicrocode(); //2511;
	case M_DS_8EAX_EBP_IB: case M_SS_8EAX_EBP_IB: return (cpu.eax << 3) + cpu.ebp + getMicrocode(); //2512;
	case M_DS_8EAX_ESI_IB: return (cpu.eax << 3) + cpu.esi + getMicrocode(); //2513;
	case M_DS_8EAX_EDI_IB: return (cpu.eax << 3) + cpu.edi + getMicrocode(); //2514;

	case M_DS_8ECX_EAX_IB: return (cpu.ecx << 3) + cpu.eax + getMicrocode(); //2515;
	case M_DS_9ECX_IB: return (cpu.ecx << 3) + cpu.ecx + getMicrocode(); //2516;
	case M_DS_8ECX_EDX_IB: return (cpu.ecx << 3) + cpu.edx + getMicrocode(); //2517;
	case M_DS_8ECX_EBX_IB: return (cpu.ecx << 3) + cpu.ebx + getMicrocode(); //2518;
	case M_DS_8ECX_ESP_IB: return (cpu.ecx << 3) + cpu.esp + getMicrocode(); //2519;
	case M_DS_8ECX_EBP_IB: return (cpu.ecx << 3) + cpu.ebp + getMicrocode(); //2520;
	case M_DS_8ECX_ESI_IB: return (cpu.ecx << 3) + cpu.esi + getMicrocode(); //2521;
	case M_DS_8ECX_EDI_IB: return (cpu.ecx << 3) + cpu.edi + getMicrocode(); //2522;

	case M_DS_8EDX_EAX_IB: return (cpu.edx << 3) + cpu.eax + getMicrocode(); //2523;
	case M_DS_8EDX_ECX_IB: return (cpu.edx << 3) + cpu.ecx + getMicrocode(); //2524;
	case M_DS_9EDX_IB: return (cpu.edx << 3) + cpu.edx + getMicrocode(); //2525;
	case M_DS_8EDX_EBX_IB: return (cpu.edx << 3) + cpu.ebx + getMicrocode(); //2526;
	case M_DS_8EDX_ESP_IB: return (cpu.edx << 3) + cpu.esp + getMicrocode(); //2527;
	case M_DS_8EDX_EBP_IB: return (cpu.edx << 3) + cpu.ebp + getMicrocode(); //2528;
	case M_DS_8EDX_ESI_IB: return (cpu.edx << 3) + cpu.esi + getMicrocode(); //2529;
	case M_DS_8EDX_EDI_IB: return (cpu.edx << 3) + cpu.edi + getMicrocode(); //2530;

	case M_DS_8EBX_EAX_IB: return (cpu.ebx << 3) + cpu.eax + getMicrocode(); //2531
	case M_DS_8EBX_ECX_IB: return (cpu.ebx << 3) + cpu.ecx + getMicrocode(); //2532
	case M_DS_8EBX_EDX_IB: return (cpu.ebx << 3) + cpu.edx + getMicrocode(); //2533
	case M_DS_9EBX_IB: return (cpu.ebx << 3) + cpu.ebx + getMicrocode(); //2534
	case M_DS_8EBX_ESP_IB: case M_SS_8EBX_ESP_IB: return (cpu.ebx << 3) + cpu.esp + getMicrocode(); //2535
	case M_DS_8EBX_EBP_IB: case M_SS_8EBX_EBP_IB: return (cpu.ebx << 3) + cpu.ebp + getMicrocode(); //2536
	case M_DS_8EBX_ESI_IB: return (cpu.ebx << 3) + cpu.esi + getMicrocode(); //2537
	case M_DS_8EBX_EDI_IB: return (cpu.ebx << 3) + cpu.edi + getMicrocode(); //2538

	case M_DS_8EBP_EAX_IB: return (cpu.ebp << 3) + cpu.eax + getMicrocode(); //2539;
	case M_DS_8EBP_ECX_IB: return (cpu.ebp << 3) + cpu.ecx + getMicrocode(); //2540;
	case M_DS_8EBP_EDX_IB: return (cpu.ebp << 3) + cpu.edx + getMicrocode(); //2541;
	case M_DS_8EBP_EBX_IB: return (cpu.ebp << 3) + cpu.ebx + getMicrocode(); //2542;
	case M_DS_8EBP_ESP_IB: case M_SS_8EBP_ESP_IB: return (cpu.ebp << 3) + cpu.esp + getMicrocode(); //2543;
	case M_DS_9EBP_IB: case M_SS_9EBP_IB: return (cpu.ebp << 3) + cpu.ebp + getMicrocode(); //2544;
	case M_DS_8EBP_ESI_IB: return (cpu.ebp << 3) + cpu.esi + getMicrocode(); //2545;
	case M_DS_8EBP_EDI_IB: return (cpu.ebp << 3) + cpu.edi + getMicrocode(); //2546;

	case M_DS_8ESI_EAX_IB: return (cpu.esi << 3) + cpu.eax + getMicrocode(); //2547;
	case M_DS_8ESI_ECX_IB: return (cpu.esi << 3) + cpu.ecx + getMicrocode(); //2548;
	case M_DS_8ESI_EDX_IB: return (cpu.esi << 3) + cpu.edx + getMicrocode(); //2549;
	case M_DS_8ESI_EBX_IB: return (cpu.esi << 3) + cpu.ebx + getMicrocode(); //2550;
	case M_DS_8ESI_ESP_IB: case M_SS_8ESI_ESP_IB: return (cpu.esi << 3) + cpu.esp + getMicrocode(); //2551;
	case M_DS_8ESI_EBP_IB: case M_SS_8ESI_EBP_IB: return (cpu.esi << 3) + cpu.ebp + getMicrocode(); //2552;
	case M_DS_9ESI_IB: return (cpu.esi << 3) + cpu.esi + getMicrocode(); //2553;
	case M_DS_8ESI_EDI_IB: return (cpu.esi << 3) + cpu.edi + getMicrocode(); //2554;
	    	    
	case M_DS_8EDI_EAX_IB: return (cpu.edi << 3) + cpu.eax + getMicrocode();//2555;
	case M_DS_8EDI_ECX_IB: return (cpu.edi << 3) + cpu.ecx + getMicrocode(); //2556;
	case M_DS_8EDI_EDX_IB: return (cpu.edi << 3) + cpu.edx + getMicrocode(); //2557;
	case M_DS_8EDI_EBX_IB: return (cpu.edi << 3) + cpu.ebx + getMicrocode(); //2558;
	case M_DS_8EDI_ESP_IB: case M_SS_8EDI_ESP_IB: return (cpu.edi << 3) + cpu.esp + getMicrocode(); //2559;
	case M_DS_8EDI_EBP_IB: case M_SS_8EDI_EBP_IB: return (cpu.edi << 3) + cpu.ebp + getMicrocode(); //2560;
	case M_DS_8EDI_ESI_IB: return (cpu.edi << 3) + cpu.esi + getMicrocode(); //2561;
	case M_DS_9EDI_IB: return (cpu.edi << 3) + cpu.edi + getMicrocode(); //2562;

	case M_DS_EAX_ECX_ID: return cpu.eax + cpu.ecx + getMicrocode(); //2563;
	case M_DS_EAX_EDX_ID: return cpu.eax + cpu.edx + getMicrocode(); //2564;
	case M_DS_EAX_EBX_ID: return cpu.eax + cpu.ebx + getMicrocode(); //2565;
	case M_DS_EAX_ESP_ID: case M_SS_EAX_ESP_ID: return cpu.eax + cpu.esp + getMicrocode(); //2566;
	case M_DS_EAX_EBP_ID: case M_SS_EAX_EBP_ID: return cpu.eax + cpu.ebp + getMicrocode(); //2567;
	case M_DS_EAX_ESI_ID: return cpu.eax + cpu.esi + getMicrocode(); //2568;
	case M_DS_EAX_EDI_ID: return cpu.eax + cpu.edi + getMicrocode(); //2569;

	case M_DS_ECX_EDX_ID: return cpu.ecx + cpu.edx + getMicrocode(); //2570;
	case M_DS_ECX_EBX_ID: return cpu.ecx + cpu.ebx + getMicrocode(); //2571;
	case M_DS_ECX_ESP_ID: case M_SS_ECX_ESP_ID: return cpu.ecx + cpu.esp + getMicrocode(); //2572;
	case M_DS_ECX_EBP_ID: case M_SS_ECX_EBP_ID: return cpu.ecx + cpu.ebp + getMicrocode(); //2573;
	case M_DS_ECX_ESI_ID: return cpu.ecx + cpu.esi + getMicrocode(); //2574;
	case M_DS_ECX_EDI_ID: return cpu.ecx + cpu.edi + getMicrocode(); //2575;

	case M_DS_EDX_EBX_ID: return cpu.edx + cpu.ebx + getMicrocode(); //2576
	case M_DS_EDX_ESP_ID: return cpu.edx + cpu.esp + getMicrocode(); //2577
	case M_DS_EDX_EBP_ID: return cpu.edx + cpu.ebp + getMicrocode(); //2578
	case M_DS_EDX_ESI_ID: return cpu.edx + cpu.esi + getMicrocode(); //2579
	case M_DS_EDX_EDI_ID: return cpu.edx + cpu.edi + getMicrocode(); //2580

	case M_DS_EBX_ESP_ID: return cpu.ebx + cpu.esp + getMicrocode(); //2581
	case M_DS_EBX_EBP_ID: return cpu.ebx + cpu.ebp + getMicrocode(); //2582
	case M_DS_EBX_ESI_ID: return cpu.ebx + cpu.esi + getMicrocode(); //2583
	case M_DS_EBX_EDI_ID: return cpu.ebx + cpu.edi + getMicrocode(); //2584

	case M_DS_ESP_ID: case M_SS_ESP_ID: return cpu.esp + getMicrocode(); //2585

	case M_DS_EBP_ESP_ID: return cpu.ebp + cpu.esp + getMicrocode(); //2586;
	case M_DS_EBP_ESI_ID: return cpu.ebp + cpu.esi + getMicrocode(); //2587;
	case M_DS_EBP_EDI_ID: case M_SS_EBP_EDI_ID: return cpu.ebp + cpu.edi + getMicrocode(); //2588;

	case M_DS_ESI_ESP_ID: case M_SS_ESI_ESP_ID: return cpu.esi + cpu.esp + getMicrocode(); //2589
	case M_DS_ESI_EDI_ID: return cpu.esi + cpu.edi + getMicrocode(); //2590

	case M_DS_EDI_ESP_ID: return cpu.edi + cpu.esp + getMicrocode(); //2591

	case M_DS_3EAX_ID: return (cpu.eax << 1) + cpu.eax + getMicrocode(); //2592;
	case M_DS_2EAX_ECX_ID: return (cpu.eax << 1) + cpu.ecx + getMicrocode(); //2593;
	case M_DS_2EAX_EDX_ID: return (cpu.eax << 1) + cpu.edx + getMicrocode(); //2594;
	case M_DS_2EAX_EBX_ID: return (cpu.eax << 1) + cpu.ebx + getMicrocode(); //2595;
	case M_DS_2EAX_ESP_ID: return (cpu.eax << 1) + cpu.esp + getMicrocode(); //2596;
	case M_DS_2EAX_EBP_ID: return (cpu.eax << 1) + cpu.ebp + getMicrocode(); //2597;
	case M_DS_2EAX_ESI_ID: return (cpu.eax << 1) + cpu.esi + getMicrocode(); //2598;
	case M_DS_2EAX_EDI_ID: return (cpu.eax << 1) + cpu.edi + getMicrocode(); //2599;
	    
	case M_DS_2ECX_EAX_ID: return (cpu.ecx << 1) + cpu.eax + getMicrocode(); //2600;
	case M_DS_3ECX_ID: return (cpu.ecx << 1) + cpu.ecx + getMicrocode(); //2601;
	case M_DS_2ECX_EDX_ID: return (cpu.ecx << 1) + cpu.edx + getMicrocode(); //2602;
	case M_DS_2ECX_EBX_ID: return (cpu.ecx << 1) + cpu.ebx + getMicrocode(); //2603;
	case M_DS_2ECX_ESP_ID: return (cpu.ecx << 1) + cpu.esp + getMicrocode(); //2604;
	case M_DS_2ECX_EBP_ID: return (cpu.ecx << 1) + cpu.ebp + getMicrocode(); //2605;
	case M_DS_2ECX_ESI_ID: return (cpu.ecx << 1) + cpu.esi + getMicrocode(); //2606;
	case M_DS_2ECX_EDI_ID: return (cpu.ecx << 1) + cpu.edi + getMicrocode(); //2607;
	    
	case M_DS_2EDX_EAX_ID: return (cpu.edx << 1) + cpu.eax + getMicrocode(); //2608;
	case M_DS_2EDX_ECX_ID: return (cpu.edx << 1) + cpu.ecx + getMicrocode(); //2609;
	case M_DS_3EDX_ID: return (cpu.edx << 1) + cpu.edx + getMicrocode(); //2610;
	case M_DS_2EDX_EBX_ID: return (cpu.edx << 1) + cpu.ebx + getMicrocode(); //2611;
	case M_DS_2EDX_ESP_ID: return (cpu.edx << 1) + cpu.esp + getMicrocode(); //2612;
	case M_DS_2EDX_EBP_ID: return (cpu.edx << 1) + cpu.ebp + getMicrocode(); //2613;
	case M_DS_2EDX_ESI_ID: return (cpu.edx << 1) + cpu.esi + getMicrocode(); //2614;
	case M_DS_2EDX_EDI_ID: return (cpu.edx << 1) + cpu.edi + getMicrocode(); //2615;
	    
	case M_DS_2EBX_EAX_ID: return (cpu.ebx << 1) + cpu.eax + getMicrocode(); //2616;
	case M_DS_2EBX_ECX_ID: return (cpu.ebx << 1) + cpu.ecx + getMicrocode(); //2617;
	case M_DS_2EBX_EDX_ID: return (cpu.ebx << 1) + cpu.edx + getMicrocode(); //2618;
	case M_DS_3EBX_ID: return (cpu.ebx << 1) + cpu.ebx + getMicrocode(); //2619;
	case M_DS_2EBX_ESP_ID: return (cpu.ebx << 1) + cpu.esp + getMicrocode(); //2620;
	case M_DS_2EBX_EBP_ID: return (cpu.ebx << 1) + cpu.ebp + getMicrocode(); //2621;
	case M_DS_2EBX_ESI_ID: return (cpu.ebx << 1) + cpu.esi + getMicrocode(); //2622;
	case M_DS_2EBX_EDI_ID: return (cpu.ebx << 1) + cpu.edi + getMicrocode(); //2623;
	    
	case M_DS_2EBP_EAX_ID: return (cpu.ebp << 1) + cpu.eax + getMicrocode(); //2624;
	case M_DS_2EBP_ECX_ID: return (cpu.ebp << 1) + cpu.ecx + getMicrocode(); //2625;
	case M_DS_2EBP_EDX_ID: return (cpu.ebp << 1) + cpu.edx + getMicrocode(); //2626;
	case M_DS_2EBP_EBX_ID: return (cpu.ebp << 1) + cpu.ebx + getMicrocode(); //2627;
	case M_DS_2EBP_ESP_ID: return (cpu.ebp << 1) + cpu.esp + getMicrocode(); //2628;
	case M_DS_3EBP_ID: return (cpu.ebp << 1) + cpu.ebp + getMicrocode(); //2629;
	case M_DS_2EBP_ESI_ID: return (cpu.ebp << 1) + cpu.esi + getMicrocode(); //2630;
	case M_DS_2EBP_EDI_ID: return (cpu.ebp << 1) + cpu.edi + getMicrocode(); //2631;
	    
	case M_DS_2ESI_EAX_ID: return (cpu.esi << 1) + cpu.eax + getMicrocode(); //2632;
	case M_DS_2ESI_ECX_ID: return (cpu.esi << 1) + cpu.ecx + getMicrocode(); //2633;
	case M_DS_2ESI_EDX_ID: return (cpu.esi << 1) + cpu.edx + getMicrocode(); //2634;
	case M_DS_2ESI_EBX_ID: return (cpu.esi << 1) + cpu.ebx + getMicrocode(); //2635;
	case M_DS_2ESI_ESP_ID: return (cpu.esi << 1) + cpu.esp + getMicrocode(); //2636;
	case M_DS_2ESI_EBP_ID: return (cpu.esi << 1) + cpu.ebp + getMicrocode(); //2637;
	case M_DS_3ESI_ID: return (cpu.esi << 1) + cpu.esi + getMicrocode(); //2638;
	case M_DS_2ESI_EDI_ID: return (cpu.esi << 1) + cpu.edi + getMicrocode(); //2639;
	    
	case M_DS_2EDI_EAX_ID: return (cpu.edi << 1) + cpu.eax + getMicrocode(); //2640;
	case M_DS_2EDI_ECX_ID: return (cpu.edi << 1) + cpu.ecx + getMicrocode(); //2641;
	case M_DS_2EDI_EDX_ID: return (cpu.edi << 1) + cpu.edx + getMicrocode(); //2642;
	case M_DS_2EDI_EBX_ID: return (cpu.edi << 1) + cpu.ebx + getMicrocode(); //2643;
	case M_DS_2EDI_ESP_ID: return (cpu.edi << 1) + cpu.esp + getMicrocode(); //2644;
	case M_DS_2EDI_EBP_ID: return (cpu.edi << 1) + cpu.ebp + getMicrocode(); //2645;
	case M_DS_2EDI_ESI_ID: return (cpu.edi << 1) + cpu.esi + getMicrocode(); //2646;
	case M_DS_3EDI_ID: return (cpu.edi << 1) + cpu.edi + getMicrocode(); //2647;

	case M_DS_5EAX_ID: return (cpu.eax << 2) + cpu.eax + getMicrocode(); //2648
	case M_DS_4EAX_ECX_ID: return (cpu.eax << 2) + cpu.ecx + getMicrocode(); //2649
	case M_DS_4EAX_EDX_ID: return (cpu.eax << 2) + cpu.edx + getMicrocode(); //2650
	case M_DS_4EAX_EBX_ID: return (cpu.eax << 2) + cpu.ebx + getMicrocode(); //2651
	case M_DS_4EAX_ESP_ID: return (cpu.eax << 2) + cpu.esp + getMicrocode(); //2652
	case M_DS_4EAX_EBP_ID: case M_SS_4EAX_EBP_ID: return (cpu.eax << 2) + cpu.ebp + getMicrocode(); //2653
	case M_DS_4EAX_ESI_ID: return (cpu.eax << 2) + cpu.esi + getMicrocode(); //2654
	case M_DS_4EAX_EDI_ID: return (cpu.eax << 2) + cpu.edi + getMicrocode(); //2655

	case M_DS_4ECX_EAX_ID: return (cpu.ecx << 2) + cpu.eax + getMicrocode(); //2656;
	case M_DS_5ECX_ID: return (cpu.ecx << 2) + cpu.ecx + getMicrocode(); //2657;
	case M_DS_4ECX_EDX_ID: return (cpu.ecx << 2) + cpu.edx + getMicrocode(); //2658;
	case M_DS_4ECX_EBX_ID: return (cpu.ecx << 2) + cpu.ebx + getMicrocode(); //2659;
	case M_DS_4ECX_ESP_ID: case M_SS_4ECX_ESP_ID: return (cpu.ecx << 2) + cpu.esp + getMicrocode(); //2660;
	case M_DS_4ECX_EBP_ID: case M_SS_4ECX_EBP_ID: return (cpu.ecx << 2) + cpu.ebp + getMicrocode(); //2661;
	case M_DS_4ECX_ESI_ID: return (cpu.ecx << 2) + cpu.esi + getMicrocode(); //2662;
	case M_DS_4ECX_EDI_ID: return (cpu.ecx << 2) + cpu.edi + getMicrocode(); //2663;
	    
	case M_DS_4EDX_EAX_ID: return (cpu.edx << 2) + cpu.eax + getMicrocode(); //2664;
	case M_DS_4EDX_ECX_ID: return (cpu.edx << 2) + cpu.ecx + getMicrocode(); //2665;
	case M_DS_5EDX_ID: return (cpu.edx << 2) + cpu.edx + getMicrocode(); //2666;
	case M_DS_4EDX_EBX_ID: return (cpu.edx << 2) + cpu.ebx + getMicrocode(); //2667;
	case M_DS_4EDX_ESP_ID: case M_SS_4EDX_ESP_ID: return (cpu.edx << 2) + cpu.esp + getMicrocode(); //2668;
	case M_DS_4EDX_EBP_ID: case M_SS_4EDX_EBP_ID: return (cpu.edx << 2) + cpu.ebp + getMicrocode(); //2669;
	case M_DS_4EDX_ESI_ID: return (cpu.edx << 2) + cpu.esi + getMicrocode(); //2670;
	case M_DS_4EDX_EDI_ID: return (cpu.edx << 2) + cpu.edi + getMicrocode(); //2671;
	    
	case M_DS_4EBX_EAX_ID: return (cpu.ebx << 2) + cpu.eax + getMicrocode(); //2672;
	case M_DS_4EBX_ECX_ID: return (cpu.ebx << 2) + cpu.ecx + getMicrocode(); //2673;
	case M_DS_4EBX_EDX_ID: return (cpu.ebx << 2) + cpu.edx + getMicrocode(); //2674;
	case M_DS_5EBX_ID: return (cpu.ebx << 2) + cpu.ebx + getMicrocode(); //2675;
	case M_DS_4EBX_ESP_ID: case M_SS_4EBX_ESP_ID: return (cpu.ebx << 2) + cpu.esp + getMicrocode(); //2676;
	case M_DS_4EBX_EBP_ID: case M_SS_4EBX_EBP_ID: return (cpu.ebx << 2) + cpu.ebp + getMicrocode(); //2677;
	case M_DS_4EBX_ESI_ID: return (cpu.ebx << 2) + cpu.esi + getMicrocode(); //2678;
	case M_DS_4EBX_EDI_ID: return (cpu.ebx << 2) + cpu.edi + getMicrocode(); //2679;
	    
	case M_DS_4EBP_EAX_ID: return (cpu.ebp << 2) + cpu.eax + getMicrocode(); //2680;
	case M_DS_4EBP_ECX_ID: return (cpu.ebp << 2) + cpu.ecx + getMicrocode(); //2681;
	case M_DS_4EBP_EDX_ID: return (cpu.ebp << 2) + cpu.edx + getMicrocode(); //2682;
	case M_DS_4EBP_EBX_ID: return (cpu.ebp << 2) + cpu.ebx + getMicrocode(); //2683;
	case M_DS_4EBP_ESP_ID: case M_SS_4EBP_ESP_ID: return (cpu.ebp << 2) + cpu.esp + getMicrocode(); //2684;
	case M_DS_5EBP_ID: case M_SS_5EBP_ID: return (cpu.ebp << 2) + cpu.ebp + getMicrocode(); //2685;
	case M_DS_4EBP_ESI_ID: return (cpu.ebp << 2) + cpu.esi + getMicrocode(); //2686;
	case M_DS_4EBP_EDI_ID: return (cpu.ebp << 2) + cpu.edi + getMicrocode(); //2687;
	    
	case M_DS_4ESI_EAX_ID: return (cpu.esi << 2) + cpu.eax + getMicrocode(); //2688;
	case M_DS_4ESI_ECX_ID: return (cpu.esi << 2) + cpu.ecx + getMicrocode(); //2689;
	case M_DS_4ESI_EDX_ID: return (cpu.esi << 2) + cpu.edx + getMicrocode(); //2690;
	case M_DS_4ESI_EBX_ID: return (cpu.esi << 2) + cpu.ebx + getMicrocode(); //2691;
	case M_DS_4ESI_ESP_ID: case M_SS_4ESI_ESP_ID: return (cpu.esi << 2) + cpu.esp + getMicrocode(); //2692;
	case M_DS_4ESI_EBP_ID: case M_SS_4ESI_EBP_ID: return (cpu.esi << 2) + cpu.ebp + getMicrocode(); //2693;
	case M_DS_5ESI_ID: return (cpu.esi << 2) + cpu.esi + getMicrocode(); //2694;
	case M_DS_4ESI_EDI_ID: return (cpu.esi << 2) + cpu.edi + getMicrocode(); //2695;

	case M_DS_4EDI_EAX_ID: return (cpu.edi << 2) + cpu.eax + getMicrocode(); //2696;
	case M_DS_4EDI_ECX_ID: return (cpu.edi << 2) + cpu.ecx + getMicrocode(); //2697;
	case M_DS_4EDI_EDX_ID: return (cpu.edi << 2) + cpu.edx + getMicrocode(); //2698;
	case M_DS_4EDI_EBX_ID: return (cpu.edi << 2) + cpu.ebx + getMicrocode(); //2699;
	case M_DS_4EDI_ESP_ID: case M_SS_4EDI_ESP_ID: return (cpu.edi << 2) + cpu.esp + getMicrocode(); //2700;
	case M_DS_4EDI_EBP_ID: case M_SS_4EDI_EBP_ID: return (cpu.edi << 2) + cpu.ebp + getMicrocode(); //2701;
	case M_DS_4EDI_ESI_ID: return (cpu.edi << 2) + cpu.esi + getMicrocode(); //2702;
	case M_DS_5EDI_ID: return (cpu.edi << 2) + cpu.edi + getMicrocode(); //2703;

	case M_DS_9EAX_ID: return (cpu.eax << 3) + cpu.eax + getMicrocode(); //2704;
	case M_DS_8EAX_ECX_ID: return (cpu.eax << 3) + cpu.ecx + getMicrocode(); //2705;
	case M_DS_8EAX_EDX_ID: return (cpu.eax << 3) + cpu.edx + getMicrocode(); //2706;
	case M_DS_8EAX_EBX_ID: return (cpu.eax << 3) + cpu.ebx + getMicrocode(); //2707;
	case M_DS_8EAX_ESP_ID: return (cpu.eax << 3) + cpu.esp + getMicrocode(); //2708;
	case M_DS_8EAX_EBP_ID: return (cpu.eax << 3) + cpu.ebp + getMicrocode(); //2709;
	case M_DS_8EAX_ESI_ID: return (cpu.eax << 3) + cpu.esi + getMicrocode(); //2710;
	case M_DS_8EAX_EDI_ID: return (cpu.eax << 3) + cpu.edi + getMicrocode(); //2711;

	case M_DS_8ECX_EAX_ID: return (cpu.ecx << 3) + cpu.eax + getMicrocode(); //2712;
	case M_DS_9ECX_ID: return (cpu.ecx << 3) + cpu.ecx + getMicrocode(); //2713;
	case M_DS_8ECX_EDX_ID: return (cpu.ecx << 3) + cpu.edx + getMicrocode(); //2714;
	case M_DS_8ECX_EBX_ID: return (cpu.ecx << 3) + cpu.ebx + getMicrocode(); //2715;
	case M_DS_8ECX_ESP_ID: return (cpu.ecx << 3) + cpu.esp + getMicrocode(); //2716;
	case M_DS_8ECX_EBP_ID: return (cpu.ecx << 3) + cpu.ebp + getMicrocode(); //2717;
	case M_DS_8ECX_ESI_ID: return (cpu.ecx << 3) + cpu.esi + getMicrocode(); //2718;
	case M_DS_8ECX_EDI_ID: return (cpu.ecx << 3) + cpu.edi + getMicrocode(); //2719;

	case M_DS_8EDX_EAX_ID: return (cpu.edx << 3) + cpu.eax + getMicrocode(); //2720;
	case M_DS_8EDX_ECX_ID: return (cpu.edx << 3) + cpu.ecx + getMicrocode(); //2721;
	case M_DS_9EDX_ID: return (cpu.edx << 3) + cpu.edx + getMicrocode(); //2722;
	case M_DS_8EDX_EBX_ID: return (cpu.edx << 3) + cpu.ebx + getMicrocode(); //2723;
	case M_DS_8EDX_ESP_ID: return (cpu.edx << 3) + cpu.esp + getMicrocode(); //2724;
	case M_DS_8EDX_EBP_ID: return (cpu.edx << 3) + cpu.ebp + getMicrocode(); //2725;
	case M_DS_8EDX_ESI_ID: return (cpu.edx << 3) + cpu.esi + getMicrocode(); //2726;
	case M_DS_8EDX_EDI_ID: return (cpu.edx << 3) + cpu.edi + getMicrocode(); //2727;

	case M_DS_8EBX_EAX_ID: return (cpu.ebx << 3) + cpu.eax + getMicrocode(); //2728;
	case M_DS_8EBX_ECX_ID: return (cpu.ebx << 3) + cpu.ecx + getMicrocode(); //2729;
	case M_DS_8EBX_EDX_ID: return (cpu.ebx << 3) + cpu.edx + getMicrocode(); //2730;
	case M_DS_9EBX_ID: return (cpu.ebx << 3) + cpu.ebx + getMicrocode(); //2731;
	case M_DS_8EBX_ESP_ID: return (cpu.ebx << 3) + cpu.esp + getMicrocode(); //2732;
	case M_DS_8EBX_EBP_ID: return (cpu.ebx << 3) + cpu.ebp + getMicrocode(); //2733;
	case M_DS_8EBX_ESI_ID: return (cpu.ebx << 3) + cpu.esi + getMicrocode(); //2734;
	case M_DS_8EBX_EDI_ID: return (cpu.ebx << 3) + cpu.edi + getMicrocode(); //2735;
	    
	case M_DS_8EBP_EAX_ID: return (cpu.ebp << 3) + cpu.eax + getMicrocode(); //2736;
	case M_DS_8EBP_ECX_ID: return (cpu.ebp << 3) + cpu.ecx + getMicrocode(); //2737;
	case M_DS_8EBP_EDX_ID: return (cpu.ebp << 3) + cpu.edx + getMicrocode(); //2738;
	case M_DS_8EBP_EBX_ID: return (cpu.ebp << 3) + cpu.ebx + getMicrocode(); //2739;
	case M_DS_8EBP_ESP_ID: return (cpu.ebp << 3) + cpu.esp + getMicrocode(); //2740;
	case M_DS_9EBP_ID: return (cpu.ebp << 3) + cpu.ebp + getMicrocode(); //2741;
	case M_DS_8EBP_ESI_ID: return (cpu.ebp << 3) + cpu.esi + getMicrocode(); //2742;
	case M_DS_8EBP_EDI_ID: return (cpu.ebp << 3) + cpu.edi + getMicrocode(); //2743;
	    
	case M_DS_8ESI_EAX_ID: return (cpu.esi << 3) + cpu.eax + getMicrocode(); //2744;
	case M_DS_8ESI_ECX_ID: return (cpu.esi << 3) + cpu.ecx + getMicrocode(); //2745;
	case M_DS_8ESI_EDX_ID: return (cpu.esi << 3) + cpu.edx + getMicrocode(); //2746;
	case M_DS_8ESI_EBX_ID: return (cpu.esi << 3) + cpu.ebx + getMicrocode(); //2747;
	case M_DS_8ESI_ESP_ID: return (cpu.esi << 3) + cpu.esp + getMicrocode(); //2748;
	case M_DS_8ESI_EBP_ID: return (cpu.esi << 3) + cpu.ebp + getMicrocode(); //2749;
	case M_DS_9ESI_ID: return (cpu.esi << 3) + cpu.esi + getMicrocode(); //2750;
	case M_DS_8ESI_EDI_ID: return (cpu.esi << 3) + cpu.edi + getMicrocode(); //2751;
    
	case M_DS_8EDI_EAX_ID: return (cpu.edi << 3) + cpu.eax + getMicrocode(); //2752;
	case M_DS_8EDI_ECX_ID: return (cpu.edi << 3) + cpu.ecx + getMicrocode(); //2753;
	case M_DS_8EDI_EDX_ID: return (cpu.edi << 3) + cpu.edx + getMicrocode(); //2754;
	case M_DS_8EDI_EBX_ID: return (cpu.edi << 3) + cpu.ebx + getMicrocode(); //2755;
	case M_DS_8EDI_ESP_ID: return (cpu.edi << 3) + cpu.esp + getMicrocode(); //2756;
	case M_DS_8EDI_EBP_ID: return (cpu.edi << 3) + cpu.ebp + getMicrocode(); //2757;
	case M_DS_8EDI_ESI_ID: return (cpu.edi << 3) + cpu.esi + getMicrocode(); //2758;
	case M_DS_9EDI_ID: return (cpu.edi << 3) + cpu.edi + getMicrocode(); //2759;
	}

	System.err.println("Unimplemented GetAddress Operand: " + microcode);
	throw unimplemented;
    }

    protected byte getByteOperand()
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case CONSTANT_1: return 1;

	case AL:  return (byte)cpu.eax;
	case AH:  return (byte)(cpu.eax >> 8);
	case CL:  return (byte)cpu.ecx;
	case CH:  return (byte)(cpu.ecx >> 8);
	case DL:  return (byte)cpu.edx;
	case DH:  return (byte)(cpu.edx >> 8);
	case BL:  return (byte)cpu.ebx;
	case BH:  return (byte)(cpu.ebx >> 8);

	case IB:  return (byte)getMicrocode();
	}

	return getByteMemoryOperand(microcode);
    }

    private final byte getByteMemoryOperand(int microcode)
    {
	switch (microcode) {
	    //XLAT Operands
	case M_ES_BX_uAL: return cpu.es.getByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff);
	case M_CS_BX_uAL: return cpu.cs.getByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff);
	case M_SS_BX_uAL: return cpu.ss.getByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff);
	case M_DS_BX_uAL: return cpu.ds.getByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff);
	case M_FS_BX_uAL: return cpu.fs.getByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff);
	case M_GS_BX_uAL: return cpu.gs.getByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff);

	    //Memory Byte Reads
	case M_ES_BX_SI: return cpu.es.getByte((cpu.ebx + cpu.esi) & 0xffff);
	case M_ES_BX_DI: return cpu.es.getByte((cpu.ebx + cpu.edi) & 0xffff);
	case M_ES_BP_SI: return cpu.es.getByte((cpu.ebp + cpu.esi) & 0xffff);
	case M_ES_BP_DI: return cpu.es.getByte((cpu.ebp + cpu.edi) & 0xffff);
	case M_ES_SI: return cpu.es.getByte(cpu.esi & 0xffff);
	case M_ES_DI: return cpu.es.getByte(cpu.edi & 0xffff);
	case M_ES_IW: return cpu.es.getByte(getMicrocode() & 0xffff);
	case M_ES_BX: return cpu.es.getByte(cpu.ebx & 0xffff);
	case M_ES_BX_SI_IB: return cpu.es.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IB: return cpu.es.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IB: return cpu.es.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IB: return cpu.es.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IB: return cpu.es.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IB: return cpu.es.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IB: return cpu.es.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IB: return cpu.es.getByte((cpu.ebx + getMicrocode()) & 0xffff);
	case M_ES_BX_SI_IW: return cpu.es.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IW: return cpu.es.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IW: return cpu.es.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IW: return cpu.es.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IW: return cpu.es.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IW: return cpu.es.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IW: return cpu.es.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IW: return cpu.es.getByte((cpu.ebx + getMicrocode()) & 0xffff);

	case M_CS_BX_SI: return cpu.cs.getByte((cpu.ebx + cpu.esi) & 0xffff);
	case M_CS_BX_DI: return cpu.cs.getByte((cpu.ebx + cpu.edi) & 0xffff);
	case M_CS_BP_SI: return cpu.cs.getByte((cpu.ebp + cpu.esi) & 0xffff);
	case M_CS_BP_DI: return cpu.cs.getByte((cpu.ebp + cpu.edi) & 0xffff);
	case M_CS_SI: return cpu.cs.getByte(cpu.esi & 0xffff);
	case M_CS_DI: return cpu.cs.getByte(cpu.edi & 0xffff);
	case M_CS_IW: return cpu.cs.getByte(getMicrocode() & 0xffff);
	case M_CS_BX: return cpu.cs.getByte(cpu.ebx & 0xffff);
	case M_CS_BX_SI_IB: return cpu.cs.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IB: return cpu.cs.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IB: return cpu.cs.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IB: return cpu.cs.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IB: return cpu.cs.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IB: return cpu.cs.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IB: return cpu.cs.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IB: return cpu.cs.getByte((cpu.ebx + getMicrocode()) & 0xffff);
	case M_CS_BX_SI_IW: return cpu.cs.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IW: return cpu.cs.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IW: return cpu.cs.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IW: return cpu.cs.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IW: return cpu.cs.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IW: return cpu.cs.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IW: return cpu.cs.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IW: return cpu.cs.getByte((cpu.ebx + getMicrocode()) & 0xffff);

	case M_SS_BX_SI: return cpu.ss.getByte((cpu.ebx + cpu.esi) & 0xffff);
	case M_SS_BX_DI: return cpu.ss.getByte((cpu.ebx + cpu.edi) & 0xffff);
	case M_SS_BP_SI: return cpu.ss.getByte((cpu.ebp + cpu.esi) & 0xffff);
	case M_SS_BP_DI: return cpu.ss.getByte((cpu.ebp + cpu.edi) & 0xffff);
	case M_SS_SI: return cpu.ss.getByte(cpu.esi & 0xffff);
	case M_SS_DI: return cpu.ss.getByte(cpu.edi & 0xffff);
	case M_SS_IW: return cpu.ss.getByte(getMicrocode() & 0xffff);
	case M_SS_BX: return cpu.ss.getByte(cpu.ebx & 0xffff);
	case M_SS_BX_SI_IB: return cpu.ss.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IB: return cpu.ss.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IB: return cpu.ss.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IB: return cpu.ss.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IB: return cpu.ss.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IB: return cpu.ss.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IB: return cpu.ss.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IB: return cpu.ss.getByte((cpu.ebx + getMicrocode()) & 0xffff);
	case M_SS_BX_SI_IW: return cpu.ss.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IW: return cpu.ss.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IW: return cpu.ss.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IW: return cpu.ss.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IW: return cpu.ss.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IW: return cpu.ss.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IW: return cpu.ss.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IW: return cpu.ss.getByte((cpu.ebx + getMicrocode()) & 0xffff);

	case M_DS_BX_SI: return cpu.ds.getByte((cpu.ebx + cpu.esi) & 0xffff);
	case M_DS_BX_DI: return cpu.ds.getByte((cpu.ebx + cpu.edi) & 0xffff);
	case M_DS_BP_SI: return cpu.ds.getByte((cpu.ebp + cpu.esi) & 0xffff);
	case M_DS_BP_DI: return cpu.ds.getByte((cpu.ebp + cpu.edi) & 0xffff);
	case M_DS_SI: return cpu.ds.getByte(cpu.esi & 0xffff);
	case M_DS_DI: return cpu.ds.getByte(cpu.edi & 0xffff);
	case M_DS_IW: return cpu.ds.getByte(getMicrocode() & 0xffff);
	case M_DS_BX: return cpu.ds.getByte(cpu.ebx & 0xffff);
	case M_DS_BX_SI_IB: return cpu.ds.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IB: return cpu.ds.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IB: return cpu.ds.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IB: return cpu.ds.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IB: return cpu.ds.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IB: return cpu.ds.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IB: return cpu.ds.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IB: return cpu.ds.getByte((cpu.ebx + getMicrocode()) & 0xffff);
	case M_DS_BX_SI_IW: return cpu.ds.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IW: return cpu.ds.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IW: return cpu.ds.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IW: return cpu.ds.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IW: return cpu.ds.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IW: return cpu.ds.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IW: return cpu.ds.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IW: return cpu.ds.getByte((cpu.ebx + getMicrocode()) & 0xffff);

	case M_FS_BX_SI: return cpu.fs.getByte((cpu.ebx + cpu.esi) & 0xffff);
	case M_FS_BX_DI: return cpu.fs.getByte((cpu.ebx + cpu.edi) & 0xffff);
	case M_FS_BP_SI: return cpu.fs.getByte((cpu.ebp + cpu.esi) & 0xffff);
	case M_FS_BP_DI: return cpu.fs.getByte((cpu.ebp + cpu.edi) & 0xffff);
	case M_FS_SI: return cpu.fs.getByte(cpu.esi & 0xffff);
	case M_FS_DI: return cpu.fs.getByte(cpu.edi & 0xffff);
	case M_FS_IW: return cpu.fs.getByte(getMicrocode() & 0xffff);
	case M_FS_BX: return cpu.fs.getByte(cpu.ebx & 0xffff);
	case M_FS_BX_SI_IB: return cpu.fs.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IB: return cpu.fs.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IB: return cpu.fs.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IB: return cpu.fs.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IB: return cpu.fs.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IB: return cpu.fs.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IB: return cpu.fs.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IB: return cpu.fs.getByte((cpu.ebx + getMicrocode()) & 0xffff);
	case M_FS_BX_SI_IW: return cpu.fs.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IW: return cpu.fs.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IW: return cpu.fs.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IW: return cpu.fs.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IW: return cpu.fs.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IW: return cpu.fs.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IW: return cpu.fs.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IW: return cpu.fs.getByte((cpu.ebx + getMicrocode()) & 0xffff);

	case M_GS_BX_SI: return cpu.gs.getByte((cpu.ebx + cpu.esi) & 0xffff);
	case M_GS_BX_DI: return cpu.gs.getByte((cpu.ebx + cpu.edi) & 0xffff);
	case M_GS_BP_SI: return cpu.gs.getByte((cpu.ebp + cpu.esi) & 0xffff);
	case M_GS_BP_DI: return cpu.gs.getByte((cpu.ebp + cpu.edi) & 0xffff);
	case M_GS_SI: return cpu.gs.getByte(cpu.esi & 0xffff);
	case M_GS_DI: return cpu.gs.getByte(cpu.edi & 0xffff);
	case M_GS_IW: return cpu.gs.getByte(getMicrocode() & 0xffff);
	case M_GS_BX: return cpu.gs.getByte(cpu.ebx & 0xffff);
	case M_GS_BX_SI_IB: return cpu.gs.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IB: return cpu.gs.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IB: return cpu.gs.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IB: return cpu.gs.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IB: return cpu.gs.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IB: return cpu.gs.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IB: return cpu.gs.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IB: return cpu.gs.getByte((cpu.ebx + getMicrocode()) & 0xffff);
	case M_GS_BX_SI_IW: return cpu.gs.getByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IW: return cpu.gs.getByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IW: return cpu.gs.getByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IW: return cpu.gs.getByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IW: return cpu.gs.getByte((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IW: return cpu.gs.getByte((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IW: return cpu.gs.getByte((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IW: return cpu.gs.getByte((cpu.ebx + getMicrocode()) & 0xffff);	    

	case M_ES_EBX_uAL: return cpu.es.getByte(cpu.ebx + (0xff & cpu.eax)); //332;
	case M_CS_EBX_uAL: return cpu.cs.getByte(cpu.ebx + (0xff & cpu.eax)); //333;
	case M_SS_EBX_uAL: return cpu.ss.getByte(cpu.ebx + (0xff & cpu.eax)); //334;
	case M_DS_EBX_uAL: return cpu.ds.getByte(cpu.ebx + (0xff & cpu.eax)); //335;
	case M_FS_EBX_uAL: return cpu.fs.getByte(cpu.ebx + (0xff & cpu.eax)); //336;
	case M_GS_EBX_uAL: return cpu.gs.getByte(cpu.ebx + (0xff & cpu.eax)); //337;

	case M_ES_EAX: return cpu.es.getByte(cpu.eax); //56
	case M_ES_ECX: return cpu.es.getByte(cpu.ecx); //57
	case M_ES_EDX: return cpu.es.getByte(cpu.edx); //58
	case M_ES_EBX: return cpu.es.getByte(cpu.ebx); //59
	case M_ES_ID: return cpu.es.getByte(getMicrocode()); //60
	case M_ES_ESI: return cpu.es.getByte(cpu.esi); //61
	case M_ES_EDI: return cpu.es.getByte(cpu.edi); //62
	case M_ES_EAX_IB: return cpu.es.getByte(cpu.eax + getMicrocode()); //63
	case M_ES_ECX_IB: return cpu.es.getByte(cpu.ecx + getMicrocode()); //64
	case M_ES_EDX_IB: return cpu.es.getByte(cpu.edx + getMicrocode()); //65
	case M_ES_EBX_IB: return cpu.es.getByte(cpu.ebx + getMicrocode()); //66
	case M_ES_EBP_IB: return cpu.es.getByte(cpu.ebp + getMicrocode()); //67
	case M_ES_ESI_IB: return cpu.es.getByte(cpu.esi + getMicrocode()); //68
	case M_ES_EDI_IB: return cpu.es.getByte(cpu.edi + getMicrocode()); //69
	case M_ES_EAX_ID: return cpu.es.getByte(cpu.eax + getMicrocode()); //70
	case M_ES_ECX_ID: return cpu.es.getByte(cpu.ecx + getMicrocode()); //71
	case M_ES_EDX_ID: return cpu.es.getByte(cpu.edx + getMicrocode()); //72
	case M_ES_EBX_ID: return cpu.es.getByte(cpu.ebx + getMicrocode()); //73
	case M_ES_EBP_ID: return cpu.es.getByte(cpu.ebp + getMicrocode()); //74
	case M_ES_ESI_ID: return cpu.es.getByte(cpu.esi + getMicrocode()); //75
	case M_ES_EDI_ID: return cpu.es.getByte(cpu.edi + getMicrocode()); //76

	case M_CS_EAX_IB: return cpu.cs.getByte(cpu.eax + getMicrocode()); //69
	case M_CS_ECX_IB: return cpu.cs.getByte(cpu.ecx + getMicrocode()); //70
	case M_CS_EDX_IB: return cpu.cs.getByte(cpu.edx + getMicrocode()); //71
	case M_CS_EBX_IB: return cpu.cs.getByte(cpu.ebx + getMicrocode()); //72
	case M_CS_EBP_IB: return cpu.cs.getByte(cpu.ebp + getMicrocode()); //73
	case M_CS_ESI_IB: return cpu.cs.getByte(cpu.esi + getMicrocode()); //74
	case M_CS_EDI_IB: return cpu.cs.getByte(cpu.edi + getMicrocode()); //75

	case M_CS_ESI: return cpu.cs.getByte(cpu.esi);

	case M_SS_EAX_IB: return cpu.ss.getByte(cpu.eax + getMicrocode()); //90
	case M_SS_ECX_IB: return cpu.ss.getByte(cpu.ecx + getMicrocode()); //91
	case M_SS_EDX_IB: return cpu.ss.getByte(cpu.edx + getMicrocode()); //92
	case M_SS_EBX_IB: return cpu.ss.getByte(cpu.ebx + getMicrocode()); //93
	case M_SS_EBP_IB: return cpu.ss.getByte(cpu.ebp + getMicrocode()); //94
	case M_SS_ESI_IB: return cpu.ss.getByte(cpu.esi + getMicrocode()); //95
	case M_SS_EDI_IB: return cpu.ss.getByte(cpu.edi + getMicrocode()); //96

	case M_SS_EAX_ID: return cpu.ss.getByte(cpu.eax + getMicrocode()); //97
	case M_SS_ECX_ID: return cpu.ss.getByte(cpu.ecx + getMicrocode()); //98
	case M_SS_EDX_ID: return cpu.ss.getByte(cpu.edx + getMicrocode()); //99
	case M_SS_EBX_ID: return cpu.ss.getByte(cpu.ebx + getMicrocode()); //100
	case M_SS_EBP_ID: return cpu.ss.getByte(cpu.ebp + getMicrocode()); //101
	case M_SS_ESI_ID: return cpu.ss.getByte(cpu.esi + getMicrocode()); //102
	case M_SS_EDI_ID: return cpu.ss.getByte(cpu.edi + getMicrocode()); //103

	case M_CS_EAX_ID: return cpu.cs.getByte(cpu.eax + getMicrocode());
	case M_CS_ECX_ID: return cpu.cs.getByte(cpu.ecx + getMicrocode());
	case M_CS_EDX_ID: return cpu.cs.getByte(cpu.edx + getMicrocode());
	case M_CS_EBX_ID: return cpu.cs.getByte(cpu.ebx + getMicrocode());
	case M_CS_EBP_ID: return cpu.cs.getByte(cpu.ebp + getMicrocode());
	case M_CS_ESI_ID: return cpu.cs.getByte(cpu.esi + getMicrocode());
	case M_CS_EDI_ID: return cpu.cs.getByte(cpu.edi + getMicrocode());

	case M_DS_EAX: return cpu.ds.getByte(cpu.eax); //104
	case M_DS_ECX: return cpu.ds.getByte(cpu.ecx); //105
	case M_DS_EDX: return cpu.ds.getByte(cpu.edx); //106
	case M_DS_EBX: return cpu.ds.getByte(cpu.ebx); //107
	case M_DS_ID: return cpu.ds.getByte(getMicrocode()); //108
	case M_DS_ESI: return cpu.ds.getByte(cpu.esi); //109
	case M_DS_EDI: return cpu.ds.getByte(cpu.edi); //110

	case M_DS_EAX_IB: return cpu.ds.getByte(cpu.eax + getMicrocode()); //111
	case M_DS_ECX_IB: return cpu.ds.getByte(cpu.ecx + getMicrocode()); //112
	case M_DS_EDX_IB: return cpu.ds.getByte(cpu.edx + getMicrocode()); //113
	case M_DS_EBX_IB: return cpu.ds.getByte(cpu.ebx + getMicrocode()); //114
	case M_DS_EBP_IB: return cpu.ds.getByte(cpu.ebp + getMicrocode()); //115 
	case M_DS_ESI_IB: return cpu.ds.getByte(cpu.esi + getMicrocode()); //116
	case M_DS_EDI_IB: return cpu.ds.getByte(cpu.edi + getMicrocode()); //117

	case M_DS_EAX_ID: return cpu.ds.getByte(cpu.eax + getMicrocode()); //118
	case M_DS_ECX_ID: return cpu.ds.getByte(cpu.ecx + getMicrocode()); //119
	case M_DS_EDX_ID: return cpu.ds.getByte(cpu.edx + getMicrocode()); //120
	case M_DS_EBX_ID: return cpu.ds.getByte(cpu.ebx + getMicrocode()); //121
	case M_DS_EBP_ID: return cpu.ds.getByte(cpu.ebp + getMicrocode()); //122
	case M_DS_ESI_ID: return cpu.ds.getByte(cpu.esi + getMicrocode()); //123
	case M_DS_EDI_ID: return cpu.ds.getByte(cpu.edi + getMicrocode()); //124

	case M_DS_2EAX: return cpu.ds.getByte(cpu.eax + cpu.eax); //2156
	case M_DS_EAX_ECX: return cpu.ds.getByte(cpu.eax + cpu.ecx); //2157
	case M_DS_EAX_EDX: return cpu.ds.getByte(cpu.eax + cpu.edx); //2158
	case M_DS_EAX_EBX: return cpu.ds.getByte(cpu.eax + cpu.ebx); //2159
	case M_DS_EAX_ESP: return cpu.ds.getByte(cpu.eax + cpu.esp); //2160
	case M_DS_EAX_EBP: return cpu.ds.getByte(cpu.eax + cpu.ebp); //2161
	case M_DS_EAX_ESI: return cpu.ds.getByte(cpu.eax + cpu.esi); //2162
	case M_DS_EAX_EDI: return cpu.ds.getByte(cpu.eax + cpu.edi); //2163

	case M_DS_2ECX: return cpu.ds.getByte(cpu.ecx + cpu.ecx); //2164
	case M_DS_ECX_EDX: return cpu.ds.getByte(cpu.ecx + cpu.edx); //2165
	case M_DS_ECX_EBX: return cpu.ds.getByte(cpu.ecx + cpu.ebx); //2166
	case M_DS_ECX_ESP: return cpu.ds.getByte(cpu.ecx + cpu.esp); //2167
	case M_DS_ECX_EBP: return cpu.ds.getByte(cpu.ecx + cpu.ebp); //2168
	case M_DS_ECX_ESI: return cpu.ds.getByte(cpu.ecx + cpu.esi); //2169
	case M_DS_ECX_EDI: return cpu.ds.getByte(cpu.ecx + cpu.edi); //2170

	case M_DS_2EDX: return cpu.ds.getByte(cpu.edx + cpu.edx); //2171
	case M_DS_EDX_EBX: return cpu.ds.getByte(cpu.edx + cpu.ebx); //2172
	case M_DS_EDX_ESP: return cpu.ds.getByte(cpu.edx + cpu.esp); //2173
	case M_DS_EDX_EBP: return cpu.ds.getByte(cpu.edx + cpu.ebp); //2174
	case M_DS_EDX_ESI: return cpu.ds.getByte(cpu.edx + cpu.esi); //2175
	case M_DS_EDX_EDI: return cpu.ds.getByte(cpu.edx + cpu.edi); //2176

	case M_DS_2EBX: return cpu.ds.getByte(cpu.ebx + cpu.ebx); //2177;
	case M_DS_EBX_ESP: return cpu.ds.getByte(cpu.ebx + cpu.esp); //2178
	case M_DS_EBX_EBP: return cpu.ds.getByte(cpu.ebx + cpu.ebp); //2179
	case M_DS_EBX_ESI: return cpu.ds.getByte(cpu.ebx + cpu.esi); //2180
	case M_DS_EBX_EDI: return cpu.ds.getByte(cpu.ebx + cpu.edi); //2181

	case M_DS_ESP: return cpu.ds.getByte(cpu.esp); //2182

	case M_DS_EBP_ESP: return cpu.ds.getByte(cpu.ebp + cpu.esp); //2183;
	case M_DS_EBP_ESI: return cpu.ds.getByte(cpu.ebp + cpu.esi); //2184;
	case M_DS_EBP_EDI: return cpu.ds.getByte(cpu.ebp + cpu.edi); //2185;

	case M_DS_ESI_ESP: return cpu.ds.getByte(cpu.esi + cpu.esp); //2186;
	case M_DS_2ESI: return cpu.ds.getByte(cpu.esi + cpu.esi); //2187;
	case M_DS_ESI_EDI: return cpu.ds.getByte(cpu.esi + cpu.edi); //2188;

	case M_DS_EDI_ESP: return cpu.ds.getByte(cpu.edi + cpu.esp); //2189;
	case M_DS_2EDI: return cpu.ds.getByte(cpu.edi + cpu.edi); //2190;

	case M_DS_3EAX: return cpu.ds.getByte((cpu.eax << 1) + cpu.eax); //2191;
	case M_DS_2EAX_ECX: return cpu.ds.getByte((cpu.eax << 1) + cpu.ecx); //2192;
	case M_DS_2EAX_EDX: return cpu.ds.getByte((cpu.eax << 1) + cpu.edx); //2193;
	case M_DS_2EAX_EBX: return cpu.ds.getByte((cpu.eax << 1) + cpu.ebx); //2194;
	case M_DS_2EAX_ESP: return cpu.ds.getByte((cpu.eax << 1) + cpu.esp); //2195;
	case M_DS_2EAX_ID: return cpu.ds.getByte((cpu.eax << 1) + getMicrocode()); //2196;
	case M_DS_2EAX_ESI: return cpu.ds.getByte((cpu.eax << 1) + cpu.esi); //2197;
	case M_DS_2EAX_EDI: return cpu.ds.getByte((cpu.eax << 1) + cpu.edi); //2198;

	case M_DS_2ECX_EAX: return cpu.ds.getByte((cpu.ecx << 1) + cpu.eax); //2199;
	case M_DS_3ECX: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_DS_2ECX_EDX: return cpu.ds.getByte((cpu.ecx << 1) + cpu.edx); //2201;
	case M_DS_2ECX_EBX: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_DS_2ECX_ESP: return cpu.ds.getByte((cpu.ecx << 1) + cpu.esp); //2203;
	case M_DS_2ECX_ID: return cpu.ds.getByte((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_DS_2ECX_ESI: return cpu.ds.getByte((cpu.ecx << 1) + cpu.esi); //2205;
	case M_DS_2ECX_EDI: return cpu.ds.getByte((cpu.ecx << 1) + cpu.edi); //2206;

	case M_DS_2EDX_EAX: return cpu.ds.getByte((cpu.edx << 1) + cpu.eax); //2207
	case M_DS_2EDX_ECX: return cpu.ds.getByte((cpu.edx << 1) + cpu.ecx); //2208
	case M_DS_3EDX: return cpu.ds.getByte((cpu.edx << 1) + cpu.edx); //2209
	case M_DS_2EDX_EBX: return cpu.ds.getByte((cpu.edx << 1) + cpu.ebx); //2210
	case M_DS_2EDX_ESP: return cpu.ds.getByte((cpu.edx << 1) + cpu.esp); //2211
	case M_DS_2EDX_ID: return cpu.ds.getByte((cpu.edx << 1) + getMicrocode()); //2212
	case M_DS_2EDX_ESI: return cpu.ds.getByte((cpu.edx << 1) + cpu.esi); //2213
	case M_DS_2EDX_EDI: return cpu.ds.getByte((cpu.edx << 1) + cpu.edi); //2214

	case M_DS_2EBX_EAX: return cpu.ds.getByte((cpu.ebx << 1) + cpu.eax); //2215;
	case M_DS_2EBX_ECX: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_DS_2EBX_EDX: return cpu.ds.getByte((cpu.ebx << 1) + cpu.edx); //2217;
	case M_DS_3EBX: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_DS_2EBX_ESP: return cpu.ds.getByte((cpu.ebx << 1) + cpu.esp); //2219;
	case M_DS_2EBX_ID: return cpu.ds.getByte((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_DS_2EBX_ESI: return cpu.ds.getByte((cpu.ebx << 1) + cpu.esi); //2221;
	case M_DS_2EBX_EDI: return cpu.ds.getByte((cpu.ebx << 1) + cpu.edi); //2222;
	    
	case M_DS_2EBP_EAX: return cpu.ds.getByte((cpu.ebp << 1) + cpu.eax); //2223;
	case M_DS_2EBP_ECX: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_DS_2EBP_EDX: return cpu.ds.getByte((cpu.ebp << 1) + cpu.edx); //2225;
	case M_DS_2EBP_EBX: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_DS_2EBP_ESP: return cpu.ds.getByte((cpu.ebp << 1) + cpu.esp); //2227;
	case M_DS_2EBP_ID: return cpu.ds.getByte((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_DS_2EBP_ESI: return cpu.ds.getByte((cpu.ebp << 1) + cpu.esi); //2229;
	case M_DS_2EBP_EDI: return cpu.ds.getByte((cpu.ebp << 1) + cpu.edi); //2230;

	case M_DS_2ESI_EAX: return cpu.ds.getByte((cpu.esi << 1) + cpu.eax); //2231;
	case M_DS_2ESI_ECX: return cpu.ds.getByte((cpu.esi << 1) + cpu.ecx); //2232;
	case M_DS_2ESI_EDX: return cpu.ds.getByte((cpu.esi << 1) + cpu.edx); //2233;
	case M_DS_2ESI_EBX: return cpu.ds.getByte((cpu.esi << 1) + cpu.ebx); //2234;
	case M_DS_2ESI_ESP: return cpu.ds.getByte((cpu.esi << 1) + cpu.esp); //2235;
	case M_DS_2ESI_ID: return cpu.ds.getByte((cpu.esi << 1) + getMicrocode()); //2236;
	case M_DS_3ESI: return cpu.ds.getByte((cpu.esi << 1) + cpu.esi); //2237;
	case M_DS_2ESI_EDI: return cpu.ds.getByte((cpu.esi << 1) + cpu.edi); //2238;

	case M_DS_2EDI_EAX: return cpu.ds.getByte((cpu.edi << 1) + cpu.eax); //2239;
	case M_DS_2EDI_ECX: return cpu.ds.getByte((cpu.edi << 1) + cpu.ecx); //2240;
	case M_DS_2EDI_EDX: return cpu.ds.getByte((cpu.edi << 1) + cpu.edx); //2241;
	case M_DS_2EDI_EBX: return cpu.ds.getByte((cpu.edi << 1) + cpu.ebx); //2242;
	case M_DS_2EDI_ESP: return cpu.ds.getByte((cpu.edi << 1) + cpu.esp); //2243;
	case M_DS_2EDI_ID: return cpu.ds.getByte((cpu.edi << 1) + getMicrocode()); //2244;
	case M_DS_2EDI_ESI: return cpu.ds.getByte((cpu.edi << 1) + cpu.esi); //2245;
	case M_DS_3EDI: return cpu.ds.getByte((cpu.edi << 1) + cpu.edi); //2246;

	case M_DS_5EAX: return cpu.ds.getByte((cpu.eax << 2) + cpu.eax); //2247
	case M_DS_4EAX_ECX: return cpu.ds.getByte((cpu.eax << 2) + cpu.ecx); //2248
	case M_DS_4EAX_EDX: return cpu.ds.getByte((cpu.eax << 2) + cpu.edx); //2249
	case M_DS_4EAX_EBX: return cpu.ds.getByte((cpu.eax << 2) + cpu.ebx); //2250
	case M_DS_4EAX_ESP: return cpu.ds.getByte((cpu.eax << 2) + cpu.esp); //2251
	case M_DS_4EAX_ID: return cpu.ds.getByte((cpu.eax << 2) + getMicrocode()); //2252
	case M_DS_4EAX_ESI: return cpu.ds.getByte((cpu.eax << 2) + cpu.esi); //2253
	case M_DS_4EAX_EDI: return cpu.ds.getByte((cpu.eax << 2) + cpu.edi); //2254

	case M_DS_4ECX_EAX: return cpu.ds.getByte((cpu.ecx << 2) + cpu.eax); //2255
	case M_DS_5ECX: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ecx); //2256
	case M_DS_4ECX_EDX: return cpu.ds.getByte((cpu.ecx << 2) + cpu.edx); //2257
	case M_DS_4ECX_EBX: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ebx); //2258
	case M_DS_4ECX_ESP: return cpu.ds.getByte((cpu.ecx << 2) + cpu.esp); //2259
	case M_DS_4ECX_ID: return cpu.ds.getByte((cpu.ecx << 2) + getMicrocode()); //2260
	case M_DS_4ECX_ESI: return cpu.ds.getByte((cpu.ecx << 2) + cpu.esi); //2261
	case M_DS_4ECX_EDI: return cpu.ds.getByte((cpu.ecx << 2) + cpu.edi); //2262

	case M_DS_4EDX_EAX: return cpu.ds.getByte((cpu.edx << 2) + cpu.eax); //2263
	case M_DS_4EDX_ECX: return cpu.ds.getByte((cpu.edx << 2) + cpu.ecx); //2264
	case M_DS_5EDX: return cpu.ds.getByte((cpu.edx << 2) + cpu.edx); //2265
	case M_DS_4EDX_EBX: return cpu.ds.getByte((cpu.edx << 2) + cpu.ebx); //2266
	case M_DS_4EDX_ESP: return cpu.ds.getByte((cpu.edx << 2) + cpu.esp); //2267
	case M_DS_4EDX_ID: return cpu.ds.getByte((cpu.edx << 2) + getMicrocode()); //2268
	case M_DS_4EDX_ESI: return cpu.ds.getByte((cpu.edx << 2) + cpu.esi); //2269
	case M_DS_4EDX_EDI: return cpu.ds.getByte((cpu.edx << 2) + cpu.edi); //2270

	case M_DS_4EBX_EAX: return cpu.ds.getByte((cpu.ebx << 2) + cpu.eax); //2271
	case M_DS_4EBX_ECX: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ecx); //2272
	case M_DS_4EBX_EDX: return cpu.ds.getByte((cpu.ebx << 2) + cpu.edx); //2273
	case M_DS_5EBX: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ebx); //2274
	case M_DS_4EBX_ESP: return cpu.ds.getByte((cpu.ebx << 2) + cpu.esp); //2275
	case M_DS_4EBX_ID: return cpu.ds.getByte((cpu.ebx << 2) + getMicrocode()); //2276
	case M_DS_4EBX_ESI: return cpu.ds.getByte((cpu.ebx << 2) + cpu.esi); //2277
	case M_DS_4EBX_EDI: return cpu.ds.getByte((cpu.ebx << 2) + cpu.edi); //2278

	case M_DS_4EBP_EAX: return cpu.ds.getByte((cpu.ebp << 2) + cpu.eax); //2279
	case M_DS_4EBP_ECX: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ecx); //2280
	case M_DS_4EBP_EDX: return cpu.ds.getByte((cpu.ebp << 2) + cpu.edx); //2281
	case M_DS_4EBP_EBX: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ebx); //2282
	case M_DS_4EBP_ESP: return cpu.ds.getByte((cpu.ebp << 2) + cpu.esp); //2283
	case M_DS_4EBP_ID: return cpu.ds.getByte((cpu.ebp << 2) + getMicrocode()); //2284
	case M_DS_4EBP_ESI: return cpu.ds.getByte((cpu.ebp << 2) + cpu.esi); //2285
	case M_DS_4EBP_EDI: return cpu.ds.getByte((cpu.ebp << 2) + cpu.edi); //2286

	case M_DS_4ESI_EAX: return cpu.ds.getByte((cpu.esi << 2) + cpu.eax); //2287
	case M_DS_4ESI_ECX: return cpu.ds.getByte((cpu.esi << 2) + cpu.ecx); //2288
	case M_DS_4ESI_EDX: return cpu.ds.getByte((cpu.esi << 2) + cpu.edx); //2289
	case M_DS_4ESI_EBX: return cpu.ds.getByte((cpu.esi << 2) + cpu.ebx); //2290
	case M_DS_4ESI_ESP: return cpu.ds.getByte((cpu.esi << 2) + cpu.esp); //2291
	case M_DS_4ESI_ID: return cpu.ds.getByte((cpu.esi << 2) + getMicrocode()); //2292
	case M_DS_5ESI: return cpu.ds.getByte((cpu.esi << 2) + cpu.esi); //2293
	case M_DS_4ESI_EDI: return cpu.ds.getByte((cpu.esi << 2) + cpu.edi); //2294

	case M_DS_4EDI_EAX: return cpu.ds.getByte((cpu.edi << 2) + cpu.eax); //2295;
	case M_DS_4EDI_ECX: return cpu.ds.getByte((cpu.edi << 2) + cpu.ecx); //2296;
	case M_DS_4EDI_EDX: return cpu.ds.getByte((cpu.edi << 2) + cpu.edx); //2297;
	case M_DS_4EDI_EBX: return cpu.ds.getByte((cpu.edi << 2) + cpu.ebx); //2298;
	case M_DS_4EDI_ESP: return cpu.ds.getByte((cpu.edi << 2) + cpu.esp); //2299;
	case M_DS_4EDI_ID: return cpu.ds.getByte((cpu.edi << 2) + getMicrocode()); //2300;
	case M_DS_4EDI_ESI: return cpu.ds.getByte((cpu.edi << 2) + cpu.esi); //2301;
	case M_DS_5EDI: return cpu.ds.getByte((cpu.edi << 2) + cpu.edi); //2302;

	case M_DS_9EAX: return cpu.ds.getByte((cpu.eax << 3) + cpu.eax); //2303;
	case M_DS_8EAX_ECX: return cpu.ds.getByte((cpu.eax << 3) + cpu.ecx); //2304;
	case M_DS_8EAX_EDX: return cpu.ds.getByte((cpu.eax << 3) + cpu.edx); //2305;
	case M_DS_8EAX_EBX: return cpu.ds.getByte((cpu.eax << 3) + cpu.ebx); //2306;
	case M_DS_8EAX_ESP: return cpu.ds.getByte((cpu.eax << 3) + cpu.esp); //2307;
	case M_DS_8EAX_ID: return cpu.ds.getByte((cpu.eax << 3) + getMicrocode()); //2308;
	case M_DS_8EAX_ESI: return cpu.ds.getByte((cpu.eax << 3) + cpu.esi); //2309;
	case M_DS_8EAX_EDI: return cpu.ds.getByte((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_DS_8ECX_EAX: return cpu.ds.getByte((cpu.ecx << 3) + cpu.eax); //2311;
	case M_DS_9ECX: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_DS_8ECX_EDX: return cpu.ds.getByte((cpu.ecx << 3) + cpu.edx); //2313;
	case M_DS_8ECX_EBX: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_DS_8ECX_ESP: return cpu.ds.getByte((cpu.ecx << 3) + cpu.esp); //2315;
	case M_DS_8ECX_ID: return cpu.ds.getByte((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_DS_8ECX_ESI: return cpu.ds.getByte((cpu.ecx << 3) + cpu.esi); //2317;
	case M_DS_8ECX_EDI: return cpu.ds.getByte((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_DS_8EDX_EAX: return cpu.ds.getByte((cpu.edx << 3) + cpu.eax); //2319;
	case M_DS_8EDX_ECX: return cpu.ds.getByte((cpu.edx << 3) + cpu.ecx); //2320;
	case M_DS_9EDX: return cpu.ds.getByte((cpu.edx << 3) + cpu.edx); //2321;
	case M_DS_8EDX_EBX: return cpu.ds.getByte((cpu.edx << 3) + cpu.ebx); //2322;
	case M_DS_8EDX_ESP: return cpu.ds.getByte((cpu.edx << 3) + cpu.esp); //2323;
	case M_DS_8EDX_ID: return cpu.ds.getByte((cpu.edx << 3) + getMicrocode()); //2324;
	case M_DS_8EDX_ESI: return cpu.ds.getByte((cpu.edx << 3) + cpu.esi); //2325;
	case M_DS_8EDX_EDI: return cpu.ds.getByte((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_DS_8EBX_EAX: return cpu.ds.getByte((cpu.ebx << 3) + cpu.eax); //2327
	case M_DS_8EBX_ECX: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ecx); //2328
	case M_DS_8EBX_EDX: return cpu.ds.getByte((cpu.ebx << 3) + cpu.edx); //2329
	case M_DS_9EBX: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ebx); //2330
	case M_DS_8EBX_ESP: return cpu.ds.getByte((cpu.ebx << 3) + cpu.esp); //2331
	case M_DS_8EBX_ID: return cpu.ds.getByte((cpu.ebx << 3) + getMicrocode()); //2332
	case M_DS_8EBX_ESI: return cpu.ds.getByte((cpu.ebx << 3) + cpu.esi); //2333
	case M_DS_8EBX_EDI: return cpu.ds.getByte((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_DS_8EBP_EAX: return cpu.ds.getByte((cpu.ebp << 3) + cpu.eax); //2335;
	case M_DS_8EBP_ECX: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_DS_8EBP_EDX: return cpu.ds.getByte((cpu.ebp << 3) + cpu.edx); //2337;
	case M_DS_8EBP_EBX: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_DS_8EBP_ESP: return cpu.ds.getByte((cpu.ebp << 3) + cpu.esp); //2339;
	case M_DS_8EBP_ID: return cpu.ds.getByte((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_DS_8EBP_ESI: return cpu.ds.getByte((cpu.ebp << 3) + cpu.esi); //2341;
	case M_DS_8EBP_EDI: return cpu.ds.getByte((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_DS_8ESI_EAX: return cpu.ds.getByte((cpu.esi << 3) + cpu.eax); //2343;
	case M_DS_8ESI_ECX: return cpu.ds.getByte((cpu.esi << 3) + cpu.ecx); //2344;
	case M_DS_8ESI_EDX: return cpu.ds.getByte((cpu.esi << 3) + cpu.edx); //2345;
	case M_DS_8ESI_EBX: return cpu.ds.getByte((cpu.esi << 3) + cpu.ebx); //2346;
	case M_DS_8ESI_ESP: return cpu.ds.getByte((cpu.esi << 3) + cpu.esp); //2347;
	case M_DS_8ESI_ID: return cpu.ds.getByte((cpu.esi << 3) + getMicrocode()); //2348;
	case M_DS_9ESI: return cpu.ds.getByte((cpu.esi << 3) + cpu.esi); //2349;
	case M_DS_8ESI_EDI: return cpu.ds.getByte((cpu.esi << 3) + cpu.edi); //2350;

	case M_DS_8EDI_EAX: return cpu.ds.getByte((cpu.edi << 3) + cpu.eax); //2351;
	case M_DS_8EDI_ECX: return cpu.ds.getByte((cpu.edi << 3) + cpu.ecx); //2352;
	case M_DS_8EDI_EDX: return cpu.ds.getByte((cpu.edi << 3) + cpu.edx); //2353;
	case M_DS_8EDI_EBX: return cpu.ds.getByte((cpu.edi << 3) + cpu.ebx); //2354;
	case M_DS_8EDI_ESP: return cpu.ds.getByte((cpu.edi << 3) + cpu.esp); //2355;
	case M_DS_8EDI_ID: return cpu.ds.getByte((cpu.edi << 3) + getMicrocode()); //2356;
	case M_DS_8EDI_ESI: return cpu.ds.getByte((cpu.edi << 3) + cpu.esi); //2357;
	case M_DS_9EDI: return cpu.ds.getByte((cpu.edi << 3) + cpu.edi); //2358;

	case M_DS_2EAX_IB: return cpu.ds.getByte(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_DS_EAX_ECX_IB: return cpu.ds.getByte(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_DS_EAX_EDX_IB: return cpu.ds.getByte(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_DS_EAX_EBX_IB: return cpu.ds.getByte(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_DS_EAX_ESP_IB: return cpu.ds.getByte(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_DS_EAX_EBP_IB: return cpu.ds.getByte(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_DS_EAX_ESI_IB: return cpu.ds.getByte(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_DS_EAX_EDI_IB: return cpu.ds.getByte(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_DS_2ECX_IB: return cpu.ds.getByte(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_DS_ECX_EDX_IB: return cpu.ds.getByte(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_DS_ECX_EBX_IB: return cpu.ds.getByte(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_DS_ECX_ESP_IB: return cpu.ds.getByte(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_DS_ECX_EBP_IB: return cpu.ds.getByte(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_DS_ECX_ESI_IB: return cpu.ds.getByte(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_DS_ECX_EDI_IB: return cpu.ds.getByte(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_DS_2EDX_IB: return cpu.ds.getByte(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_DS_EDX_EBX_IB: return cpu.ds.getByte(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_DS_EDX_ESP_IB: return cpu.ds.getByte(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_DS_EDX_EBP_IB: return cpu.ds.getByte(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_DS_EDX_ESI_IB: return cpu.ds.getByte(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_DS_EDX_EDI_IB: return cpu.ds.getByte(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_DS_2EBX_IB: return cpu.ds.getByte(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_DS_EBX_ESP_IB: return cpu.ds.getByte(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_DS_EBX_EBP_IB: return cpu.ds.getByte(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_DS_EBX_ESI_IB: return cpu.ds.getByte(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_DS_EBX_EDI_IB: return cpu.ds.getByte(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_DS_ESP_IB: return cpu.ds.getByte(cpu.esp + getMicrocode()); //2385

	case M_DS_EBP_ESP_IB: return cpu.ds.getByte(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_DS_2EBP_IB: return cpu.ds.getByte(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_DS_EBP_ESI_IB: return cpu.ds.getByte(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_DS_EBP_EDI_IB: return cpu.ds.getByte(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_DS_ESI_ESP_IB: return cpu.ds.getByte(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_DS_2ESI_IB: return cpu.ds.getByte(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_DS_ESI_EDI_IB: return cpu.ds.getByte(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_DS_EDI_ESP_IB: return cpu.ds.getByte(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_DS_2EDI_IB: return cpu.ds.getByte(cpu.edi + cpu.edi + getMicrocode()); //2394;

	case M_DS_3EAX_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_DS_2EAX_ECX_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_DS_2EAX_EDX_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_DS_2EAX_EBX_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_DS_2EAX_ESP_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_DS_2EAX_EBP_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_DS_2EAX_ESI_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_DS_2EAX_EDI_IB: return cpu.ds.getByte((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_DS_2ECX_EAX_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_DS_3ECX_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_DS_2ECX_EDX_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_DS_2ECX_EBX_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_DS_2ECX_ESP_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_DS_2ECX_EBP_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_DS_2ECX_ESI_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_DS_2ECX_EDI_IB: return cpu.ds.getByte((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_DS_2EDX_EAX_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_DS_2EDX_ECX_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_DS_3EDX_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_DS_2EDX_EBX_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_DS_2EDX_ESP_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_DS_2EDX_EBP_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_DS_2EDX_ESI_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_DS_2EDX_EDI_IB: return cpu.ds.getByte((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;
	    
	case M_DS_2EBX_EAX_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_DS_2EBX_ECX_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_DS_2EBX_EDX_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_DS_3EBX_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_DS_2EBX_ESP_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_DS_2EBX_EBP_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_DS_2EBX_ESI_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_DS_2EBX_EDI_IB: return cpu.ds.getByte((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_DS_2EBP_EAX_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_DS_2EBP_ECX_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_DS_2EBP_EDX_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_DS_2EBP_EBX_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_DS_2EBP_ESP_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_DS_3EBP_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_DS_2EBP_ESI_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_DS_2EBP_EDI_IB: return cpu.ds.getByte((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_DS_2ESI_EAX_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_DS_2ESI_ECX_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_DS_2ESI_EDX_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_DS_2ESI_EBX_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_DS_2ESI_ESP_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_DS_2ESI_EBP_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_DS_3ESI_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_DS_2ESI_EDI_IB: return cpu.ds.getByte((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_DS_2EDI_EAX_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_DS_2EDI_ECX_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_DS_2EDI_EDX_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_DS_2EDI_EBX_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_DS_2EDI_ESP_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_DS_2EDI_EBP_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_DS_2EDI_ESI_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_DS_3EDI_IB: return cpu.ds.getByte((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_DS_5EAX_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_DS_4EAX_ECX_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_DS_4EAX_EDX_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_DS_4EAX_EBX_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_DS_4EAX_ESP_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_DS_4EAX_EBP_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_DS_4EAX_ESI_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_DS_4EAX_EDI_IB: return cpu.ds.getByte((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_DS_4ECX_EAX_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_DS_5ECX_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_DS_4ECX_EDX_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_DS_4ECX_EBX_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_DS_4ECX_ESP_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_DS_4ECX_EBP_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_DS_4ECX_ESI_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_DS_4ECX_EDI_IB: return cpu.ds.getByte((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_DS_4EDX_EAX_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_DS_4EDX_ECX_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_DS_5EDX_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_DS_4EDX_EBX_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_DS_4EDX_ESP_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_DS_4EDX_EBP_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_DS_4EDX_ESI_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_DS_4EDX_EDI_IB: return cpu.ds.getByte((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_DS_4EBX_EAX_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_DS_4EBX_ECX_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_DS_4EBX_EDX_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_DS_5EBX_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_DS_4EBX_ESP_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_DS_4EBX_EBP_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_DS_4EBX_ESI_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_DS_4EBX_EDI_IB: return cpu.ds.getByte((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_DS_4EBP_EAX_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_DS_4EBP_ECX_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_DS_4EBP_EDX_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_DS_4EBP_EBX_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_DS_4EBP_ESP_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_DS_5EBP_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_DS_4EBP_ESI_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_DS_4EBP_EDI_IB: return cpu.ds.getByte((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_DS_4ESI_EAX_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_DS_4ESI_ECX_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_DS_4ESI_EDX_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_DS_4ESI_EBX_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_DS_4ESI_ESP_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_DS_4ESI_EBP_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_DS_5ESI_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_DS_4ESI_EDI_IB: return cpu.ds.getByte((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_DS_4EDI_EAX_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_DS_4EDI_ECX_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_DS_4EDI_EDX_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_DS_4EDI_EBX_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_DS_4EDI_ESP_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_DS_4EDI_EBP_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_DS_4EDI_ESI_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_DS_5EDI_IB: return cpu.ds.getByte((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_DS_9EAX_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_DS_8EAX_ECX_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_DS_8EAX_EDX_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_DS_8EAX_EBX_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_DS_8EAX_ESP_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_DS_8EAX_EBP_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_DS_8EAX_ESI_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_DS_8EAX_EDI_IB: return cpu.ds.getByte((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_DS_8ECX_EAX_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_DS_9ECX_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_DS_8ECX_EDX_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_DS_8ECX_EBX_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_DS_8ECX_ESP_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_DS_8ECX_EBP_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_DS_8ECX_ESI_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_DS_8ECX_EDI_IB: return cpu.ds.getByte((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_DS_8EDX_EAX_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_DS_8EDX_ECX_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_DS_9EDX_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_DS_8EDX_EBX_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_DS_8EDX_ESP_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_DS_8EDX_EBP_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_DS_8EDX_ESI_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_DS_8EDX_EDI_IB: return cpu.ds.getByte((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_DS_8EBX_EAX_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_DS_8EBX_ECX_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_DS_8EBX_EDX_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_DS_9EBX_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_DS_8EBX_ESP_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_DS_8EBX_EBP_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_DS_8EBX_ESI_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_DS_8EBX_EDI_IB: return cpu.ds.getByte((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_DS_8EBP_EAX_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_DS_8EBP_ECX_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_DS_8EBP_EDX_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_DS_8EBP_EBX_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_DS_8EBP_ESP_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_DS_9EBP_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_DS_8EBP_ESI_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_DS_8EBP_EDI_IB: return cpu.ds.getByte((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_DS_8ESI_EAX_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_DS_8ESI_ECX_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_DS_8ESI_EDX_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_DS_8ESI_EBX_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_DS_8ESI_ESP_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_DS_8ESI_EBP_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_DS_9ESI_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_DS_8ESI_EDI_IB: return cpu.ds.getByte((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_DS_8EDI_EAX_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_DS_8EDI_ECX_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_DS_8EDI_EDX_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_DS_8EDI_EBX_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_DS_8EDI_ESP_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_DS_8EDI_EBP_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_DS_8EDI_ESI_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_DS_9EDI_IB: return cpu.ds.getByte((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_DS_EAX_ECX_ID: return cpu.ds.getByte(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_DS_EAX_EDX_ID: return cpu.ds.getByte(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_DS_EAX_EBX_ID: return cpu.ds.getByte(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_DS_EAX_ESP_ID: return cpu.ds.getByte(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_DS_EAX_EBP_ID: return cpu.ds.getByte(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_DS_EAX_ESI_ID: return cpu.ds.getByte(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_DS_EAX_EDI_ID: return cpu.ds.getByte(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_DS_ECX_EDX_ID: return cpu.ds.getByte(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_DS_ECX_EBX_ID: return cpu.ds.getByte(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_DS_ECX_ESP_ID: return cpu.ds.getByte(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_DS_ECX_EBP_ID: return cpu.ds.getByte(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_DS_ECX_ESI_ID: return cpu.ds.getByte(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_DS_ECX_EDI_ID: return cpu.ds.getByte(cpu.ecx + cpu.edi + getMicrocode()); //2575;

	case M_DS_EDX_EBX_ID: return cpu.ds.getByte(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_DS_EDX_ESP_ID: return cpu.ds.getByte(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_DS_EDX_EBP_ID: return cpu.ds.getByte(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_DS_EDX_ESI_ID: return cpu.ds.getByte(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_DS_EDX_EDI_ID: return cpu.ds.getByte(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_DS_EBX_ESP_ID: return cpu.ds.getByte(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_DS_EBX_EBP_ID: return cpu.ds.getByte(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_DS_EBX_ESI_ID: return cpu.ds.getByte(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_DS_EBX_EDI_ID: return cpu.ds.getByte(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_DS_ESP_ID: return cpu.ds.getByte(cpu.esp + getMicrocode()); //2585

	case M_DS_EBP_ESP_ID: return cpu.ds.getByte(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_DS_EBP_ESI_ID: return cpu.ds.getByte(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_DS_EBP_EDI_ID: return cpu.ds.getByte(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_DS_ESI_ESP_ID: return cpu.ds.getByte(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_DS_ESI_EDI_ID: return cpu.ds.getByte(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_DS_EDI_ESP_ID: return cpu.ds.getByte(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_DS_3EAX_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_DS_2EAX_ECX_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_DS_2EAX_EDX_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_DS_2EAX_EBX_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_DS_2EAX_ESP_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_DS_2EAX_EBP_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_DS_2EAX_ESI_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_DS_2EAX_EDI_ID: return cpu.ds.getByte((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_DS_2ECX_EAX_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_DS_3ECX_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_DS_2ECX_EDX_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_DS_2ECX_EBX_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_DS_2ECX_ESP_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_DS_2ECX_EBP_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_DS_2ECX_ESI_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_DS_2ECX_EDI_ID: return cpu.ds.getByte((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_DS_2EDX_EAX_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_DS_2EDX_ECX_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_DS_3EDX_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_DS_2EDX_EBX_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_DS_2EDX_ESP_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_DS_2EDX_EBP_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_DS_2EDX_ESI_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_DS_2EDX_EDI_ID: return cpu.ds.getByte((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_DS_2EBX_EAX_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_DS_2EBX_ECX_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_DS_2EBX_EDX_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_DS_3EBX_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_DS_2EBX_ESP_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_DS_2EBX_EBP_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_DS_2EBX_ESI_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_DS_2EBX_EDI_ID: return cpu.ds.getByte((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_DS_2EBP_EAX_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_DS_2EBP_ECX_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_DS_2EBP_EDX_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_DS_2EBP_EBX_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_DS_2EBP_ESP_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_DS_3EBP_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_DS_2EBP_ESI_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_DS_2EBP_EDI_ID: return cpu.ds.getByte((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_DS_2ESI_EAX_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_DS_2ESI_ECX_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_DS_2ESI_EDX_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_DS_2ESI_EBX_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_DS_2ESI_ESP_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_DS_2ESI_EBP_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_DS_3ESI_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_DS_2ESI_EDI_ID: return cpu.ds.getByte((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_DS_2EDI_EAX_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_DS_2EDI_ECX_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_DS_2EDI_EDX_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_DS_2EDI_EBX_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_DS_2EDI_ESP_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_DS_2EDI_EBP_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_DS_2EDI_ESI_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_DS_3EDI_ID: return cpu.ds.getByte((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_DS_5EAX_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_DS_4EAX_ECX_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_DS_4EAX_EDX_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_DS_4EAX_EBX_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_DS_4EAX_ESP_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_DS_4EAX_EBP_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_DS_4EAX_ESI_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_DS_4EAX_EDI_ID: return cpu.ds.getByte((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_DS_4ECX_EAX_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_DS_5ECX_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_DS_4ECX_EDX_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_DS_4ECX_EBX_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_DS_4ECX_ESP_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_DS_4ECX_EBP_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_DS_4ECX_ESI_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_DS_4ECX_EDI_ID: return cpu.ds.getByte((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_DS_4EDX_EAX_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_DS_4EDX_ECX_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_DS_5EDX_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_DS_4EDX_EBX_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_DS_4EDX_ESP_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_DS_4EDX_EBP_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_DS_4EDX_ESI_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_DS_4EDX_EDI_ID: return cpu.ds.getByte((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_DS_4EBX_EAX_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_DS_4EBX_ECX_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_DS_4EBX_EDX_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_DS_5EBX_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_DS_4EBX_ESP_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_DS_4EBX_EBP_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_DS_4EBX_ESI_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_DS_4EBX_EDI_ID: return cpu.ds.getByte((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_DS_4EBP_EAX_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_DS_4EBP_ECX_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_DS_4EBP_EDX_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_DS_4EBP_EBX_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_DS_4EBP_ESP_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_DS_5EBP_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_DS_4EBP_ESI_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_DS_4EBP_EDI_ID: return cpu.ds.getByte((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_DS_4ESI_EAX_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_DS_4ESI_ECX_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_DS_4ESI_EDX_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_DS_4ESI_EBX_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_DS_4ESI_ESP_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_DS_4ESI_EBP_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_DS_5ESI_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_DS_4ESI_EDI_ID: return cpu.ds.getByte((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_DS_4EDI_EAX_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_DS_4EDI_ECX_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_DS_4EDI_EDX_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_DS_4EDI_EBX_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_DS_4EDI_ESP_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_DS_4EDI_EBP_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_DS_4EDI_ESI_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_DS_5EDI_ID: return cpu.ds.getByte((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_DS_9EAX_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_DS_8EAX_ECX_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_DS_8EAX_EDX_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_DS_8EAX_EBX_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_DS_8EAX_ESP_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_DS_8EAX_EBP_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_DS_8EAX_ESI_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_DS_8EAX_EDI_ID: return cpu.ds.getByte((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_DS_8ECX_EAX_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_DS_9ECX_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_DS_8ECX_EDX_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_DS_8ECX_EBX_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_DS_8ECX_ESP_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_DS_8ECX_EBP_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_DS_8ECX_ESI_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_DS_8ECX_EDI_ID: return cpu.ds.getByte((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_DS_8EDX_EAX_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_DS_8EDX_ECX_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_DS_9EDX_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_DS_8EDX_EBX_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_DS_8EDX_ESP_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_DS_8EDX_EBP_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_DS_8EDX_ESI_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_DS_8EDX_EDI_ID: return cpu.ds.getByte((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_DS_8EBX_EAX_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_DS_8EBX_ECX_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_DS_8EBX_EDX_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_DS_9EBX_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_DS_8EBX_ESP_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_DS_8EBX_EBP_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_DS_8EBX_ESI_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_DS_8EBX_EDI_ID: return cpu.ds.getByte((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_DS_8EBP_EAX_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_DS_8EBP_ECX_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_DS_8EBP_EDX_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_DS_8EBP_EBX_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_DS_8EBP_ESP_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_DS_9EBP_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_DS_8EBP_ESI_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_DS_8EBP_EDI_ID: return cpu.ds.getByte((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_DS_8ESI_EAX_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_DS_8ESI_ECX_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_DS_8ESI_EDX_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_DS_8ESI_EBX_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_DS_8ESI_ESP_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_DS_8ESI_EBP_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_DS_9ESI_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_DS_8ESI_EDI_ID: return cpu.ds.getByte((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_DS_8EDI_EAX_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_DS_8EDI_ECX_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_DS_8EDI_EDX_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_DS_8EDI_EBX_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_DS_8EDI_ESP_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_DS_8EDI_EBP_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_DS_8EDI_ESI_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_DS_9EDI_ID: return cpu.ds.getByte((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;

	case M_SS_2EAX: return cpu.ss.getByte(cpu.eax + cpu.eax); //2156
	case M_SS_EAX_ECX: return cpu.ss.getByte(cpu.eax + cpu.ecx); //2157
	case M_SS_EAX_EDX: return cpu.ss.getByte(cpu.eax + cpu.edx); //2158
	case M_SS_EAX_EBX: return cpu.ss.getByte(cpu.eax + cpu.ebx); //2159
	case M_SS_EAX_ESP: return cpu.ss.getByte(cpu.eax + cpu.esp); //2160
	case M_SS_EAX_EBP: return cpu.ss.getByte(cpu.eax + cpu.ebp); //2161
	case M_SS_EAX_ESI: return cpu.ss.getByte(cpu.eax + cpu.esi); //2162
	case M_SS_EAX_EDI: return cpu.ss.getByte(cpu.eax + cpu.edi); //2163

	case M_SS_2ECX: return cpu.ss.getByte(cpu.ecx + cpu.ecx); //2164
	case M_SS_ECX_EDX: return cpu.ss.getByte(cpu.ecx + cpu.edx); //2165
	case M_SS_ECX_EBX: return cpu.ss.getByte(cpu.ecx + cpu.ebx); //2166
	case M_SS_ECX_ESP: return cpu.ss.getByte(cpu.ecx + cpu.esp); //2167
	case M_SS_ECX_EBP: return cpu.ss.getByte(cpu.ecx + cpu.ebp); //2168
	case M_SS_ECX_ESI: return cpu.ss.getByte(cpu.ecx + cpu.esi); //2169
	case M_SS_ECX_EDI: return cpu.ss.getByte(cpu.ecx + cpu.edi); //2170

	case M_SS_2EDX: return cpu.ss.getByte(cpu.edx + cpu.edx); //2171
	case M_SS_EDX_EBX: return cpu.ss.getByte(cpu.edx + cpu.ebx); //2172
	case M_SS_EDX_ESP: return cpu.ss.getByte(cpu.edx + cpu.esp); //2173
	case M_SS_EDX_EBP: return cpu.ss.getByte(cpu.edx + cpu.ebp); //2174
	case M_SS_EDX_ESI: return cpu.ss.getByte(cpu.edx + cpu.esi); //2175
	case M_SS_EDX_EDI: return cpu.ss.getByte(cpu.edx + cpu.edi); //2176

	case M_SS_2EBX: return cpu.ss.getByte(cpu.ebx + cpu.ebx); //2177;
	case M_SS_EBX_ESP: return cpu.ss.getByte(cpu.ebx + cpu.esp); //2178
	case M_SS_EBX_EBP: return cpu.ss.getByte(cpu.ebx + cpu.ebp); //2179
	case M_SS_EBX_ESI: return cpu.ss.getByte(cpu.ebx + cpu.esi); //2180
	case M_SS_EBX_EDI: return cpu.ss.getByte(cpu.ebx + cpu.edi); //2181

	case M_SS_ESP: return cpu.ss.getByte(cpu.esp); //2182

	case M_SS_EBP_ESP: return cpu.ss.getByte(cpu.ebp + cpu.esp); //2183;
	case M_SS_EBP_ESI: return cpu.ss.getByte(cpu.ebp + cpu.esi); //2184;
	case M_SS_EBP_EDI: return cpu.ss.getByte(cpu.ebp + cpu.edi); //2185;

	case M_SS_ESI_ESP: return cpu.ss.getByte(cpu.esi + cpu.esp); //2186;
	case M_SS_2ESI: return cpu.ss.getByte(cpu.esi + cpu.esi); //2187;
	case M_SS_ESI_EDI: return cpu.ss.getByte(cpu.esi + cpu.edi); //2188;

	case M_SS_EDI_ESP: return cpu.ss.getByte(cpu.edi + cpu.esp); //2189;
	case M_SS_2EDI: return cpu.ss.getByte(cpu.edi + cpu.edi); //2190;

	case M_SS_3EAX: return cpu.ss.getByte((cpu.eax << 1) + cpu.eax); //2191;
	case M_SS_2EAX_ECX: return cpu.ss.getByte((cpu.eax << 1) + cpu.ecx); //2192;
	case M_SS_2EAX_EDX: return cpu.ss.getByte((cpu.eax << 1) + cpu.edx); //2193;
	case M_SS_2EAX_EBX: return cpu.ss.getByte((cpu.eax << 1) + cpu.ebx); //2194;
	case M_SS_2EAX_ESP: return cpu.ss.getByte((cpu.eax << 1) + cpu.esp); //2195;
	case M_SS_2EAX_ID: return cpu.ss.getByte((cpu.eax << 1) + getMicrocode()); //2196;
	case M_SS_2EAX_ESI: return cpu.ss.getByte((cpu.eax << 1) + cpu.esi); //2197;
	case M_SS_2EAX_EDI: return cpu.ss.getByte((cpu.eax << 1) + cpu.edi); //2198;

	case M_SS_2ECX_EAX: return cpu.ss.getByte((cpu.ecx << 1) + cpu.eax); //2199;
	case M_SS_3ECX: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_SS_2ECX_EDX: return cpu.ss.getByte((cpu.ecx << 1) + cpu.edx); //2201;
	case M_SS_2ECX_EBX: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_SS_2ECX_ESP: return cpu.ss.getByte((cpu.ecx << 1) + cpu.esp); //2203;
	case M_SS_2ECX_ID: return cpu.ss.getByte((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_SS_2ECX_ESI: return cpu.ss.getByte((cpu.ecx << 1) + cpu.esi); //2205;
	case M_SS_2ECX_EDI: return cpu.ss.getByte((cpu.ecx << 1) + cpu.edi); //2206;

	case M_SS_2EDX_EAX: return cpu.ss.getByte((cpu.edx << 1) + cpu.eax); //2207
	case M_SS_2EDX_ECX: return cpu.ss.getByte((cpu.edx << 1) + cpu.ecx); //2208
	case M_SS_3EDX: return cpu.ss.getByte((cpu.edx << 1) + cpu.edx); //2209
	case M_SS_2EDX_EBX: return cpu.ss.getByte((cpu.edx << 1) + cpu.ebx); //2210
	case M_SS_2EDX_ESP: return cpu.ss.getByte((cpu.edx << 1) + cpu.esp); //2211
	case M_SS_2EDX_ID: return cpu.ss.getByte((cpu.edx << 1) + getMicrocode()); //2212
	case M_SS_2EDX_ESI: return cpu.ss.getByte((cpu.edx << 1) + cpu.esi); //2213
	case M_SS_2EDX_EDI: return cpu.ss.getByte((cpu.edx << 1) + cpu.edi); //2214

	case M_SS_2EBX_EAX: return cpu.ss.getByte((cpu.ebx << 1) + cpu.eax); //2215;
	case M_SS_2EBX_ECX: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_SS_2EBX_EDX: return cpu.ss.getByte((cpu.ebx << 1) + cpu.edx); //2217;
	case M_SS_3EBX: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_SS_2EBX_ESP: return cpu.ss.getByte((cpu.ebx << 1) + cpu.esp); //2219;
	case M_SS_2EBX_ID: return cpu.ss.getByte((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_SS_2EBX_ESI: return cpu.ss.getByte((cpu.ebx << 1) + cpu.esi); //2221;
	case M_SS_2EBX_EDI: return cpu.ss.getByte((cpu.ebx << 1) + cpu.edi); //2222;
	    
	case M_SS_2EBP_EAX: return cpu.ss.getByte((cpu.ebp << 1) + cpu.eax); //2223;
	case M_SS_2EBP_ECX: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_SS_2EBP_EDX: return cpu.ss.getByte((cpu.ebp << 1) + cpu.edx); //2225;
	case M_SS_2EBP_EBX: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_SS_2EBP_ESP: return cpu.ss.getByte((cpu.ebp << 1) + cpu.esp); //2227;
	case M_SS_2EBP_ID: return cpu.ss.getByte((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_SS_2EBP_ESI: return cpu.ss.getByte((cpu.ebp << 1) + cpu.esi); //2229;
	case M_SS_2EBP_EDI: return cpu.ss.getByte((cpu.ebp << 1) + cpu.edi); //2230;

	case M_SS_2ESI_EAX: return cpu.ss.getByte((cpu.esi << 1) + cpu.eax); //2231;
	case M_SS_2ESI_ECX: return cpu.ss.getByte((cpu.esi << 1) + cpu.ecx); //2232;
	case M_SS_2ESI_EDX: return cpu.ss.getByte((cpu.esi << 1) + cpu.edx); //2233;
	case M_SS_2ESI_EBX: return cpu.ss.getByte((cpu.esi << 1) + cpu.ebx); //2234;
	case M_SS_2ESI_ESP: return cpu.ss.getByte((cpu.esi << 1) + cpu.esp); //2235;
	case M_SS_2ESI_ID: return cpu.ss.getByte((cpu.esi << 1) + getMicrocode()); //2236;
	case M_SS_3ESI: return cpu.ss.getByte((cpu.esi << 1) + cpu.esi); //2237;
	case M_SS_2ESI_EDI: return cpu.ss.getByte((cpu.esi << 1) + cpu.edi); //2238;

	case M_SS_2EDI_EAX: return cpu.ss.getByte((cpu.edi << 1) + cpu.eax); //2239;
	case M_SS_2EDI_ECX: return cpu.ss.getByte((cpu.edi << 1) + cpu.ecx); //2240;
	case M_SS_2EDI_EDX: return cpu.ss.getByte((cpu.edi << 1) + cpu.edx); //2241;
	case M_SS_2EDI_EBX: return cpu.ss.getByte((cpu.edi << 1) + cpu.ebx); //2242;
	case M_SS_2EDI_ESP: return cpu.ss.getByte((cpu.edi << 1) + cpu.esp); //2243;
	case M_SS_2EDI_ID: return cpu.ss.getByte((cpu.edi << 1) + getMicrocode()); //2244;
	case M_SS_2EDI_ESI: return cpu.ss.getByte((cpu.edi << 1) + cpu.esi); //2245;
	case M_SS_3EDI: return cpu.ss.getByte((cpu.edi << 1) + cpu.edi); //2246;

	case M_SS_5EAX: return cpu.ss.getByte((cpu.eax << 2) + cpu.eax); //2247
	case M_SS_4EAX_ECX: return cpu.ss.getByte((cpu.eax << 2) + cpu.ecx); //2248
	case M_SS_4EAX_EDX: return cpu.ss.getByte((cpu.eax << 2) + cpu.edx); //2249
	case M_SS_4EAX_EBX: return cpu.ss.getByte((cpu.eax << 2) + cpu.ebx); //2250
	case M_SS_4EAX_ESP: return cpu.ss.getByte((cpu.eax << 2) + cpu.esp); //2251
	case M_SS_4EAX_ID: return cpu.ss.getByte((cpu.eax << 2) + getMicrocode()); //2252
	case M_SS_4EAX_ESI: return cpu.ss.getByte((cpu.eax << 2) + cpu.esi); //2253
	case M_SS_4EAX_EDI: return cpu.ss.getByte((cpu.eax << 2) + cpu.edi); //2254

	case M_SS_4ECX_EAX: return cpu.ss.getByte((cpu.ecx << 2) + cpu.eax); //2255
	case M_SS_5ECX: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ecx); //2256
	case M_SS_4ECX_EDX: return cpu.ss.getByte((cpu.ecx << 2) + cpu.edx); //2257
	case M_SS_4ECX_EBX: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ebx); //2258
	case M_SS_4ECX_ESP: return cpu.ss.getByte((cpu.ecx << 2) + cpu.esp); //2259
	case M_SS_4ECX_ID: return cpu.ss.getByte((cpu.ecx << 2) + getMicrocode()); //2260
	case M_SS_4ECX_ESI: return cpu.ss.getByte((cpu.ecx << 2) + cpu.esi); //2261
	case M_SS_4ECX_EDI: return cpu.ss.getByte((cpu.ecx << 2) + cpu.edi); //2262

	case M_SS_4EDX_EAX: return cpu.ss.getByte((cpu.edx << 2) + cpu.eax); //2263
	case M_SS_4EDX_ECX: return cpu.ss.getByte((cpu.edx << 2) + cpu.ecx); //2264
	case M_SS_5EDX: return cpu.ss.getByte((cpu.edx << 2) + cpu.edx); //2265
	case M_SS_4EDX_EBX: return cpu.ss.getByte((cpu.edx << 2) + cpu.ebx); //2266
	case M_SS_4EDX_ESP: return cpu.ss.getByte((cpu.edx << 2) + cpu.esp); //2267
	case M_SS_4EDX_ID: return cpu.ss.getByte((cpu.edx << 2) + getMicrocode()); //2268
	case M_SS_4EDX_ESI: return cpu.ss.getByte((cpu.edx << 2) + cpu.esi); //2269
	case M_SS_4EDX_EDI: return cpu.ss.getByte((cpu.edx << 2) + cpu.edi); //2270

	case M_SS_4EBX_EAX: return cpu.ss.getByte((cpu.ebx << 2) + cpu.eax); //2271
	case M_SS_4EBX_ECX: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ecx); //2272
	case M_SS_4EBX_EDX: return cpu.ss.getByte((cpu.ebx << 2) + cpu.edx); //2273
	case M_SS_5EBX: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ebx); //2274
	case M_SS_4EBX_ESP: return cpu.ss.getByte((cpu.ebx << 2) + cpu.esp); //2275
	case M_SS_4EBX_ID: return cpu.ss.getByte((cpu.ebx << 2) + getMicrocode()); //2276
	case M_SS_4EBX_ESI: return cpu.ss.getByte((cpu.ebx << 2) + cpu.esi); //2277
	case M_SS_4EBX_EDI: return cpu.ss.getByte((cpu.ebx << 2) + cpu.edi); //2278

	case M_SS_4EBP_EAX: return cpu.ss.getByte((cpu.ebp << 2) + cpu.eax); //2279
	case M_SS_4EBP_ECX: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ecx); //2280
	case M_SS_4EBP_EDX: return cpu.ss.getByte((cpu.ebp << 2) + cpu.edx); //2281
	case M_SS_4EBP_EBX: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ebx); //2282
	case M_SS_4EBP_ESP: return cpu.ss.getByte((cpu.ebp << 2) + cpu.esp); //2283
	case M_SS_4EBP_ID: return cpu.ss.getByte((cpu.ebp << 2) + getMicrocode()); //2284
	case M_SS_4EBP_ESI: return cpu.ss.getByte((cpu.ebp << 2) + cpu.esi); //2285
	case M_SS_4EBP_EDI: return cpu.ss.getByte((cpu.ebp << 2) + cpu.edi); //2286

	case M_SS_4ESI_EAX: return cpu.ss.getByte((cpu.esi << 2) + cpu.eax); //2287
	case M_SS_4ESI_ECX: return cpu.ss.getByte((cpu.esi << 2) + cpu.ecx); //2288
	case M_SS_4ESI_EDX: return cpu.ss.getByte((cpu.esi << 2) + cpu.edx); //2289
	case M_SS_4ESI_EBX: return cpu.ss.getByte((cpu.esi << 2) + cpu.ebx); //2290
	case M_SS_4ESI_ESP: return cpu.ss.getByte((cpu.esi << 2) + cpu.esp); //2291
	case M_SS_4ESI_ID: return cpu.ss.getByte((cpu.esi << 2) + getMicrocode()); //2292
	case M_SS_5ESI: return cpu.ss.getByte((cpu.esi << 2) + cpu.esi); //2293
	case M_SS_4ESI_EDI: return cpu.ss.getByte((cpu.esi << 2) + cpu.edi); //2294

	case M_SS_4EDI_EAX: return cpu.ss.getByte((cpu.edi << 2) + cpu.eax); //2295;
	case M_SS_4EDI_ECX: return cpu.ss.getByte((cpu.edi << 2) + cpu.ecx); //2296;
	case M_SS_4EDI_EDX: return cpu.ss.getByte((cpu.edi << 2) + cpu.edx); //2297;
	case M_SS_4EDI_EBX: return cpu.ss.getByte((cpu.edi << 2) + cpu.ebx); //2298;
	case M_SS_4EDI_ESP: return cpu.ss.getByte((cpu.edi << 2) + cpu.esp); //2299;
	case M_SS_4EDI_ID: return cpu.ss.getByte((cpu.edi << 2) + getMicrocode()); //2300;
	case M_SS_4EDI_ESI: return cpu.ss.getByte((cpu.edi << 2) + cpu.esi); //2301;
	case M_SS_5EDI: return cpu.ss.getByte((cpu.edi << 2) + cpu.edi); //2302;

	case M_SS_9EAX: return cpu.ss.getByte((cpu.eax << 3) + cpu.eax); //2303;
	case M_SS_8EAX_ECX: return cpu.ss.getByte((cpu.eax << 3) + cpu.ecx); //2304;
	case M_SS_8EAX_EDX: return cpu.ss.getByte((cpu.eax << 3) + cpu.edx); //2305;
	case M_SS_8EAX_EBX: return cpu.ss.getByte((cpu.eax << 3) + cpu.ebx); //2306;
	case M_SS_8EAX_ESP: return cpu.ss.getByte((cpu.eax << 3) + cpu.esp); //2307;
	case M_SS_8EAX_ID: return cpu.ss.getByte((cpu.eax << 3) + getMicrocode()); //2308;
	case M_SS_8EAX_ESI: return cpu.ss.getByte((cpu.eax << 3) + cpu.esi); //2309;
	case M_SS_8EAX_EDI: return cpu.ss.getByte((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_SS_8ECX_EAX: return cpu.ss.getByte((cpu.ecx << 3) + cpu.eax); //2311;
	case M_SS_9ECX: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_SS_8ECX_EDX: return cpu.ss.getByte((cpu.ecx << 3) + cpu.edx); //2313;
	case M_SS_8ECX_EBX: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_SS_8ECX_ESP: return cpu.ss.getByte((cpu.ecx << 3) + cpu.esp); //2315;
	case M_SS_8ECX_ID: return cpu.ss.getByte((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_SS_8ECX_ESI: return cpu.ss.getByte((cpu.ecx << 3) + cpu.esi); //2317;
	case M_SS_8ECX_EDI: return cpu.ss.getByte((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_SS_8EDX_EAX: return cpu.ss.getByte((cpu.edx << 3) + cpu.eax); //2319;
	case M_SS_8EDX_ECX: return cpu.ss.getByte((cpu.edx << 3) + cpu.ecx); //2320;
	case M_SS_9EDX: return cpu.ss.getByte((cpu.edx << 3) + cpu.edx); //2321;
	case M_SS_8EDX_EBX: return cpu.ss.getByte((cpu.edx << 3) + cpu.ebx); //2322;
	case M_SS_8EDX_ESP: return cpu.ss.getByte((cpu.edx << 3) + cpu.esp); //2323;
	case M_SS_8EDX_ID: return cpu.ss.getByte((cpu.edx << 3) + getMicrocode()); //2324;
	case M_SS_8EDX_ESI: return cpu.ss.getByte((cpu.edx << 3) + cpu.esi); //2325;
	case M_SS_8EDX_EDI: return cpu.ss.getByte((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_SS_8EBX_EAX: return cpu.ss.getByte((cpu.ebx << 3) + cpu.eax); //2327
	case M_SS_8EBX_ECX: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ecx); //2328
	case M_SS_8EBX_EDX: return cpu.ss.getByte((cpu.ebx << 3) + cpu.edx); //2329
	case M_SS_9EBX: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ebx); //2330
	case M_SS_8EBX_ESP: return cpu.ss.getByte((cpu.ebx << 3) + cpu.esp); //2331
	case M_SS_8EBX_ID: return cpu.ss.getByte((cpu.ebx << 3) + getMicrocode()); //2332
	case M_SS_8EBX_ESI: return cpu.ss.getByte((cpu.ebx << 3) + cpu.esi); //2333
	case M_SS_8EBX_EDI: return cpu.ss.getByte((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_SS_8EBP_EAX: return cpu.ss.getByte((cpu.ebp << 3) + cpu.eax); //2335;
	case M_SS_8EBP_ECX: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_SS_8EBP_EDX: return cpu.ss.getByte((cpu.ebp << 3) + cpu.edx); //2337;
	case M_SS_8EBP_EBX: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_SS_8EBP_ESP: return cpu.ss.getByte((cpu.ebp << 3) + cpu.esp); //2339;
	case M_SS_8EBP_ID: return cpu.ss.getByte((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_SS_8EBP_ESI: return cpu.ss.getByte((cpu.ebp << 3) + cpu.esi); //2341;
	case M_SS_8EBP_EDI: return cpu.ss.getByte((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_SS_8ESI_EAX: return cpu.ss.getByte((cpu.esi << 3) + cpu.eax); //2343;
	case M_SS_8ESI_ECX: return cpu.ss.getByte((cpu.esi << 3) + cpu.ecx); //2344;
	case M_SS_8ESI_EDX: return cpu.ss.getByte((cpu.esi << 3) + cpu.edx); //2345;
	case M_SS_8ESI_EBX: return cpu.ss.getByte((cpu.esi << 3) + cpu.ebx); //2346;
	case M_SS_8ESI_ESP: return cpu.ss.getByte((cpu.esi << 3) + cpu.esp); //2347;
	case M_SS_8ESI_ID: return cpu.ss.getByte((cpu.esi << 3) + getMicrocode()); //2348;
	case M_SS_9ESI: return cpu.ss.getByte((cpu.esi << 3) + cpu.esi); //2349;
	case M_SS_8ESI_EDI: return cpu.ss.getByte((cpu.esi << 3) + cpu.edi); //2350;

	case M_SS_8EDI_EAX: return cpu.ss.getByte((cpu.edi << 3) + cpu.eax); //2351;
	case M_SS_8EDI_ECX: return cpu.ss.getByte((cpu.edi << 3) + cpu.ecx); //2352;
	case M_SS_8EDI_EDX: return cpu.ss.getByte((cpu.edi << 3) + cpu.edx); //2353;
	case M_SS_8EDI_EBX: return cpu.ss.getByte((cpu.edi << 3) + cpu.ebx); //2354;
	case M_SS_8EDI_ESP: return cpu.ss.getByte((cpu.edi << 3) + cpu.esp); //2355;
	case M_SS_8EDI_ID: return cpu.ss.getByte((cpu.edi << 3) + getMicrocode()); //2356;
	case M_SS_8EDI_ESI: return cpu.ss.getByte((cpu.edi << 3) + cpu.esi); //2357;
	case M_SS_9EDI: return cpu.ss.getByte((cpu.edi << 3) + cpu.edi); //2358;

	case M_SS_2EAX_IB: return cpu.ss.getByte(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_SS_EAX_ECX_IB: return cpu.ss.getByte(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_SS_EAX_EDX_IB: return cpu.ss.getByte(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_SS_EAX_EBX_IB: return cpu.ss.getByte(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_SS_EAX_ESP_IB: return cpu.ss.getByte(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_SS_EAX_EBP_IB: return cpu.ss.getByte(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_SS_EAX_ESI_IB: return cpu.ss.getByte(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_SS_EAX_EDI_IB: return cpu.ss.getByte(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_SS_2ECX_IB: return cpu.ss.getByte(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_SS_ECX_EDX_IB: return cpu.ss.getByte(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_SS_ECX_EBX_IB: return cpu.ss.getByte(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_SS_ECX_ESP_IB: return cpu.ss.getByte(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_SS_ECX_EBP_IB: return cpu.ss.getByte(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_SS_ECX_ESI_IB: return cpu.ss.getByte(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_SS_ECX_EDI_IB: return cpu.ss.getByte(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_SS_2EDX_IB: return cpu.ss.getByte(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_SS_EDX_EBX_IB: return cpu.ss.getByte(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_SS_EDX_ESP_IB: return cpu.ss.getByte(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_SS_EDX_EBP_IB: return cpu.ss.getByte(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_SS_EDX_ESI_IB: return cpu.ss.getByte(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_SS_EDX_EDI_IB: return cpu.ss.getByte(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_SS_2EBX_IB: return cpu.ss.getByte(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_SS_EBX_ESP_IB: return cpu.ss.getByte(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_SS_EBX_EBP_IB: return cpu.ss.getByte(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_SS_EBX_ESI_IB: return cpu.ss.getByte(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_SS_EBX_EDI_IB: return cpu.ss.getByte(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_SS_ESP_IB: return cpu.ss.getByte(cpu.esp + getMicrocode()); //2385

	case M_SS_EBP_ESP_IB: return cpu.ss.getByte(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_SS_2EBP_IB: return cpu.ss.getByte(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_SS_EBP_ESI_IB: return cpu.ss.getByte(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_SS_EBP_EDI_IB: return cpu.ss.getByte(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_SS_ESI_ESP_IB: return cpu.ss.getByte(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_SS_2ESI_IB: return cpu.ss.getByte(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_SS_ESI_EDI_IB: return cpu.ss.getByte(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_SS_EDI_ESP_IB: return cpu.ss.getByte(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_SS_2EDI_IB: return cpu.ss.getByte(cpu.edi + cpu.edi + getMicrocode()); //2394;

	case M_SS_3EAX_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_SS_2EAX_ECX_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_SS_2EAX_EDX_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_SS_2EAX_EBX_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_SS_2EAX_ESP_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_SS_2EAX_EBP_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_SS_2EAX_ESI_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_SS_2EAX_EDI_IB: return cpu.ss.getByte((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_SS_2ECX_EAX_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_SS_3ECX_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_SS_2ECX_EDX_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_SS_2ECX_EBX_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_SS_2ECX_ESP_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_SS_2ECX_EBP_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_SS_2ECX_ESI_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_SS_2ECX_EDI_IB: return cpu.ss.getByte((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_SS_2EDX_EAX_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_SS_2EDX_ECX_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_SS_3EDX_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_SS_2EDX_EBX_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_SS_2EDX_ESP_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_SS_2EDX_EBP_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_SS_2EDX_ESI_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_SS_2EDX_EDI_IB: return cpu.ss.getByte((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;
	    
	case M_SS_2EBX_EAX_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_SS_2EBX_ECX_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_SS_2EBX_EDX_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_SS_3EBX_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_SS_2EBX_ESP_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_SS_2EBX_EBP_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_SS_2EBX_ESI_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_SS_2EBX_EDI_IB: return cpu.ss.getByte((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_SS_2EBP_EAX_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_SS_2EBP_ECX_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_SS_2EBP_EDX_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_SS_2EBP_EBX_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_SS_2EBP_ESP_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_SS_3EBP_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_SS_2EBP_ESI_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_SS_2EBP_EDI_IB: return cpu.ss.getByte((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_SS_2ESI_EAX_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_SS_2ESI_ECX_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_SS_2ESI_EDX_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_SS_2ESI_EBX_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_SS_2ESI_ESP_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_SS_2ESI_EBP_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_SS_3ESI_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_SS_2ESI_EDI_IB: return cpu.ss.getByte((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_SS_2EDI_EAX_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_SS_2EDI_ECX_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_SS_2EDI_EDX_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_SS_2EDI_EBX_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_SS_2EDI_ESP_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_SS_2EDI_EBP_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_SS_2EDI_ESI_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_SS_3EDI_IB: return cpu.ss.getByte((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_SS_5EAX_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_SS_4EAX_ECX_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_SS_4EAX_EDX_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_SS_4EAX_EBX_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_SS_4EAX_ESP_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_SS_4EAX_EBP_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_SS_4EAX_ESI_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_SS_4EAX_EDI_IB: return cpu.ss.getByte((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_SS_4ECX_EAX_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_SS_5ECX_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_SS_4ECX_EDX_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_SS_4ECX_EBX_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_SS_4ECX_ESP_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_SS_4ECX_EBP_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_SS_4ECX_ESI_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_SS_4ECX_EDI_IB: return cpu.ss.getByte((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_SS_4EDX_EAX_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_SS_4EDX_ECX_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_SS_5EDX_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_SS_4EDX_EBX_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_SS_4EDX_ESP_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_SS_4EDX_EBP_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_SS_4EDX_ESI_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_SS_4EDX_EDI_IB: return cpu.ss.getByte((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_SS_4EBX_EAX_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_SS_4EBX_ECX_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_SS_4EBX_EDX_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_SS_5EBX_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_SS_4EBX_ESP_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_SS_4EBX_EBP_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_SS_4EBX_ESI_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_SS_4EBX_EDI_IB: return cpu.ss.getByte((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_SS_4EBP_EAX_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_SS_4EBP_ECX_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_SS_4EBP_EDX_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_SS_4EBP_EBX_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_SS_4EBP_ESP_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_SS_5EBP_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_SS_4EBP_ESI_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_SS_4EBP_EDI_IB: return cpu.ss.getByte((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_SS_4ESI_EAX_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_SS_4ESI_ECX_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_SS_4ESI_EDX_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_SS_4ESI_EBX_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_SS_4ESI_ESP_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_SS_4ESI_EBP_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_SS_5ESI_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_SS_4ESI_EDI_IB: return cpu.ss.getByte((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_SS_4EDI_EAX_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_SS_4EDI_ECX_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_SS_4EDI_EDX_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_SS_4EDI_EBX_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_SS_4EDI_ESP_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_SS_4EDI_EBP_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_SS_4EDI_ESI_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_SS_5EDI_IB: return cpu.ss.getByte((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_SS_9EAX_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_SS_8EAX_ECX_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_SS_8EAX_EDX_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_SS_8EAX_EBX_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_SS_8EAX_ESP_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_SS_8EAX_EBP_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_SS_8EAX_ESI_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_SS_8EAX_EDI_IB: return cpu.ss.getByte((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_SS_8ECX_EAX_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_SS_9ECX_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_SS_8ECX_EDX_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_SS_8ECX_EBX_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_SS_8ECX_ESP_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_SS_8ECX_EBP_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_SS_8ECX_ESI_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_SS_8ECX_EDI_IB: return cpu.ss.getByte((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_SS_8EDX_EAX_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_SS_8EDX_ECX_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_SS_9EDX_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_SS_8EDX_EBX_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_SS_8EDX_ESP_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_SS_8EDX_EBP_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_SS_8EDX_ESI_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_SS_8EDX_EDI_IB: return cpu.ss.getByte((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_SS_8EBX_EAX_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_SS_8EBX_ECX_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_SS_8EBX_EDX_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_SS_9EBX_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_SS_8EBX_ESP_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_SS_8EBX_EBP_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_SS_8EBX_ESI_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_SS_8EBX_EDI_IB: return cpu.ss.getByte((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_SS_8EBP_EAX_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_SS_8EBP_ECX_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_SS_8EBP_EDX_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_SS_8EBP_EBX_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_SS_8EBP_ESP_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_SS_9EBP_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_SS_8EBP_ESI_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_SS_8EBP_EDI_IB: return cpu.ss.getByte((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_SS_8ESI_EAX_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_SS_8ESI_ECX_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_SS_8ESI_EDX_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_SS_8ESI_EBX_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_SS_8ESI_ESP_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_SS_8ESI_EBP_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_SS_9ESI_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_SS_8ESI_EDI_IB: return cpu.ss.getByte((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_SS_8EDI_EAX_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_SS_8EDI_ECX_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_SS_8EDI_EDX_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_SS_8EDI_EBX_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_SS_8EDI_ESP_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_SS_8EDI_EBP_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_SS_8EDI_ESI_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_SS_9EDI_IB: return cpu.ss.getByte((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_SS_EAX_ECX_ID: return cpu.ss.getByte(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_SS_EAX_EDX_ID: return cpu.ss.getByte(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_SS_EAX_EBX_ID: return cpu.ss.getByte(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_SS_EAX_ESP_ID: return cpu.ss.getByte(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_SS_EAX_EBP_ID: return cpu.ss.getByte(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_SS_EAX_ESI_ID: return cpu.ss.getByte(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_SS_EAX_EDI_ID: return cpu.ss.getByte(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_SS_ECX_EDX_ID: return cpu.ss.getByte(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_SS_ECX_EBX_ID: return cpu.ss.getByte(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_SS_ECX_ESP_ID: return cpu.ss.getByte(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_SS_ECX_EBP_ID: return cpu.ss.getByte(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_SS_ECX_ESI_ID: return cpu.ss.getByte(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_SS_ECX_EDI_ID: return cpu.ss.getByte(cpu.ecx + cpu.edi + getMicrocode()); //2575;

	case M_SS_EDX_EBX_ID: return cpu.ss.getByte(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_SS_EDX_ESP_ID: return cpu.ss.getByte(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_SS_EDX_EBP_ID: return cpu.ss.getByte(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_SS_EDX_ESI_ID: return cpu.ss.getByte(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_SS_EDX_EDI_ID: return cpu.ss.getByte(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_SS_EBX_ESP_ID: return cpu.ss.getByte(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_SS_EBX_EBP_ID: return cpu.ss.getByte(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_SS_EBX_ESI_ID: return cpu.ss.getByte(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_SS_EBX_EDI_ID: return cpu.ss.getByte(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_SS_ESP_ID: return cpu.ss.getByte(cpu.esp + getMicrocode()); //2585

	case M_SS_EBP_ESP_ID: return cpu.ss.getByte(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_SS_EBP_ESI_ID: return cpu.ss.getByte(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_SS_EBP_EDI_ID: return cpu.ss.getByte(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_SS_ESI_ESP_ID: return cpu.ss.getByte(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_SS_ESI_EDI_ID: return cpu.ss.getByte(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_SS_EDI_ESP_ID: return cpu.ss.getByte(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_SS_3EAX_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_SS_2EAX_ECX_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_SS_2EAX_EDX_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_SS_2EAX_EBX_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_SS_2EAX_ESP_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_SS_2EAX_EBP_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_SS_2EAX_ESI_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_SS_2EAX_EDI_ID: return cpu.ss.getByte((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_SS_2ECX_EAX_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_SS_3ECX_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_SS_2ECX_EDX_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_SS_2ECX_EBX_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_SS_2ECX_ESP_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_SS_2ECX_EBP_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_SS_2ECX_ESI_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_SS_2ECX_EDI_ID: return cpu.ss.getByte((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_SS_2EDX_EAX_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_SS_2EDX_ECX_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_SS_3EDX_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_SS_2EDX_EBX_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_SS_2EDX_ESP_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_SS_2EDX_EBP_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_SS_2EDX_ESI_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_SS_2EDX_EDI_ID: return cpu.ss.getByte((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_SS_2EBX_EAX_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_SS_2EBX_ECX_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_SS_2EBX_EDX_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_SS_3EBX_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_SS_2EBX_ESP_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_SS_2EBX_EBP_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_SS_2EBX_ESI_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_SS_2EBX_EDI_ID: return cpu.ss.getByte((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_SS_2EBP_EAX_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_SS_2EBP_ECX_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_SS_2EBP_EDX_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_SS_2EBP_EBX_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_SS_2EBP_ESP_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_SS_3EBP_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_SS_2EBP_ESI_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_SS_2EBP_EDI_ID: return cpu.ss.getByte((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_SS_2ESI_EAX_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_SS_2ESI_ECX_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_SS_2ESI_EDX_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_SS_2ESI_EBX_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_SS_2ESI_ESP_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_SS_2ESI_EBP_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_SS_3ESI_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_SS_2ESI_EDI_ID: return cpu.ss.getByte((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_SS_2EDI_EAX_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_SS_2EDI_ECX_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_SS_2EDI_EDX_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_SS_2EDI_EBX_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_SS_2EDI_ESP_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_SS_2EDI_EBP_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_SS_2EDI_ESI_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_SS_3EDI_ID: return cpu.ss.getByte((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_SS_5EAX_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_SS_4EAX_ECX_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_SS_4EAX_EDX_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_SS_4EAX_EBX_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_SS_4EAX_ESP_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_SS_4EAX_EBP_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_SS_4EAX_ESI_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_SS_4EAX_EDI_ID: return cpu.ss.getByte((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_SS_4ECX_EAX_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_SS_5ECX_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_SS_4ECX_EDX_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_SS_4ECX_EBX_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_SS_4ECX_ESP_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_SS_4ECX_EBP_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_SS_4ECX_ESI_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_SS_4ECX_EDI_ID: return cpu.ss.getByte((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_SS_4EDX_EAX_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_SS_4EDX_ECX_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_SS_5EDX_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_SS_4EDX_EBX_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_SS_4EDX_ESP_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_SS_4EDX_EBP_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_SS_4EDX_ESI_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_SS_4EDX_EDI_ID: return cpu.ss.getByte((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_SS_4EBX_EAX_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_SS_4EBX_ECX_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_SS_4EBX_EDX_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_SS_5EBX_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_SS_4EBX_ESP_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_SS_4EBX_EBP_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_SS_4EBX_ESI_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_SS_4EBX_EDI_ID: return cpu.ss.getByte((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_SS_4EBP_EAX_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_SS_4EBP_ECX_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_SS_4EBP_EDX_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_SS_4EBP_EBX_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_SS_4EBP_ESP_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_SS_5EBP_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_SS_4EBP_ESI_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_SS_4EBP_EDI_ID: return cpu.ss.getByte((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_SS_4ESI_EAX_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_SS_4ESI_ECX_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_SS_4ESI_EDX_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_SS_4ESI_EBX_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_SS_4ESI_ESP_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_SS_4ESI_EBP_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_SS_5ESI_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_SS_4ESI_EDI_ID: return cpu.ss.getByte((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_SS_4EDI_EAX_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_SS_4EDI_ECX_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_SS_4EDI_EDX_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_SS_4EDI_EBX_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_SS_4EDI_ESP_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_SS_4EDI_EBP_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_SS_4EDI_ESI_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_SS_5EDI_ID: return cpu.ss.getByte((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_SS_9EAX_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_SS_8EAX_ECX_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_SS_8EAX_EDX_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_SS_8EAX_EBX_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_SS_8EAX_ESP_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_SS_8EAX_EBP_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_SS_8EAX_ESI_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_SS_8EAX_EDI_ID: return cpu.ss.getByte((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_SS_8ECX_EAX_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_SS_9ECX_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_SS_8ECX_EDX_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_SS_8ECX_EBX_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_SS_8ECX_ESP_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_SS_8ECX_EBP_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_SS_8ECX_ESI_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_SS_8ECX_EDI_ID: return cpu.ss.getByte((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_SS_8EDX_EAX_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_SS_8EDX_ECX_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_SS_9EDX_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_SS_8EDX_EBX_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_SS_8EDX_ESP_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_SS_8EDX_EBP_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_SS_8EDX_ESI_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_SS_8EDX_EDI_ID: return cpu.ss.getByte((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_SS_8EBX_EAX_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_SS_8EBX_ECX_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_SS_8EBX_EDX_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_SS_9EBX_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_SS_8EBX_ESP_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_SS_8EBX_EBP_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_SS_8EBX_ESI_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_SS_8EBX_EDI_ID: return cpu.ss.getByte((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_SS_8EBP_EAX_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_SS_8EBP_ECX_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_SS_8EBP_EDX_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_SS_8EBP_EBX_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_SS_8EBP_ESP_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_SS_9EBP_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_SS_8EBP_ESI_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_SS_8EBP_EDI_ID: return cpu.ss.getByte((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_SS_8ESI_EAX_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_SS_8ESI_ECX_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_SS_8ESI_EDX_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_SS_8ESI_EBX_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_SS_8ESI_ESP_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_SS_8ESI_EBP_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_SS_9ESI_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_SS_8ESI_EDI_ID: return cpu.ss.getByte((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_SS_8EDI_EAX_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_SS_8EDI_ECX_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_SS_8EDI_EDX_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_SS_8EDI_EBX_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_SS_8EDI_ESP_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_SS_8EDI_EBP_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_SS_8EDI_ESI_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_SS_9EDI_ID: return cpu.ss.getByte((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;
	}

	System.err.println("Unimplemented GetByte Operand: " + microcode);
	throw unimplemented;
    }

    protected final short getShortOperand()
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case AX:  return (short)cpu.eax;
	case CX:  return (short)cpu.ecx;
	case DX:  return (short)cpu.edx;
	case BX:  return (short)cpu.ebx;
	case SP:  return (short)cpu.esp;
	case BP:  return (short)cpu.ebp;
	case SI:  return (short)cpu.esi;
	case DI:  return (short)cpu.edi;

	case IB:  return (short)getMicrocode();
	case IW:  return (short)getMicrocode();

	case DS:  return (short)cpu.ds.getSelector();
	case SS:  return (short)cpu.ss.getSelector();
	case ES:  return (short)cpu.es.getSelector();
	case CS:  return (short)cpu.cs.getSelector();
	case FS:  return (short)cpu.fs.getSelector();
	case GS:  return (short)cpu.gs.getSelector();
	}

	return getShortMemoryOperand(microcode);
    }

    private final short getShortMemoryOperand(int microcode)
    {
	switch (microcode) {
	case M_ES_BX_SI: return cpu.es.getWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_ES_BX_DI: return cpu.es.getWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_ES_BP_SI: return cpu.es.getWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_ES_BP_DI: return cpu.es.getWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_ES_SI: return cpu.es.getWord(cpu.esi & 0xffff);
	case M_ES_DI: return cpu.es.getWord(cpu.edi & 0xffff);
	case M_ES_IW: return cpu.es.getWord(getMicrocode() & 0xffff);
	case M_ES_BX: return cpu.es.getWord(cpu.ebx & 0xffff);
	case M_ES_BX_SI_IB: return cpu.es.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IB: return cpu.es.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IB: return cpu.es.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IB: return cpu.es.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IB: return cpu.es.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IB: return cpu.es.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IB: return cpu.es.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IB: return cpu.es.getWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_ES_BX_SI_IW: return cpu.es.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IW: return cpu.es.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IW: return cpu.es.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IW: return cpu.es.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IW: return cpu.es.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IW: return cpu.es.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IW: return cpu.es.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IW: return cpu.es.getWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_CS_BX_SI: return cpu.cs.getWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_CS_BX_DI: return cpu.cs.getWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_CS_BP_SI: return cpu.cs.getWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_CS_BP_DI: return cpu.cs.getWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_CS_SI: return cpu.cs.getWord(cpu.esi & 0xffff);
	case M_CS_DI: return cpu.cs.getWord(cpu.edi & 0xffff);
	case M_CS_IW: return cpu.cs.getWord(getMicrocode() & 0xffff);
	case M_CS_BX: return cpu.cs.getWord(cpu.ebx & 0xffff);
	case M_CS_BX_SI_IB: return cpu.cs.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IB: return cpu.cs.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IB: return cpu.cs.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IB: return cpu.cs.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IB: return cpu.cs.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IB: return cpu.cs.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IB: return cpu.cs.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IB: return cpu.cs.getWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_CS_BX_SI_IW: return cpu.cs.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IW: return cpu.cs.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IW: return cpu.cs.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IW: return cpu.cs.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IW: return cpu.cs.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IW: return cpu.cs.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IW: return cpu.cs.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IW: return cpu.cs.getWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_SS_BX_SI: return cpu.ss.getWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_SS_BX_DI: return cpu.ss.getWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_SS_BP_SI: return cpu.ss.getWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_SS_BP_DI: return cpu.ss.getWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_SS_SI: return cpu.ss.getWord(cpu.esi & 0xffff);
	case M_SS_DI: return cpu.ss.getWord(cpu.edi & 0xffff);
	case M_SS_IW: return cpu.ss.getWord(getMicrocode() & 0xffff);
	case M_SS_BX: return cpu.ss.getWord(cpu.ebx & 0xffff);
	case M_SS_BX_SI_IB: return cpu.ss.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IB: return cpu.ss.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IB: return cpu.ss.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IB: return cpu.ss.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IB: return cpu.ss.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IB: return cpu.ss.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IB: return cpu.ss.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IB: return cpu.ss.getWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_SS_BX_SI_IW: return cpu.ss.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IW: return cpu.ss.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IW: return cpu.ss.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IW: return cpu.ss.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IW: return cpu.ss.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IW: return cpu.ss.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IW: return cpu.ss.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IW: return cpu.ss.getWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_DS_BX_SI: return cpu.ds.getWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_DS_BX_DI: return cpu.ds.getWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_DS_BP_SI: return cpu.ds.getWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_DS_BP_DI: return cpu.ds.getWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_DS_SI: return cpu.ds.getWord(cpu.esi & 0xffff);
	case M_DS_DI: return cpu.ds.getWord(cpu.edi & 0xffff);
	case M_DS_IW: return cpu.ds.getWord(getMicrocode() & 0xffff);
	case M_DS_BX: return cpu.ds.getWord(cpu.ebx & 0xffff);
	case M_DS_BX_SI_IB: return cpu.ds.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IB: return cpu.ds.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IB: return cpu.ds.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IB: return cpu.ds.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IB: return cpu.ds.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IB: return cpu.ds.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IB: return cpu.ds.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IB: return cpu.ds.getWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_DS_BX_SI_IW: return cpu.ds.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IW: return cpu.ds.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IW: return cpu.ds.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IW: return cpu.ds.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IW: return cpu.ds.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IW: return cpu.ds.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IW: return cpu.ds.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IW: return cpu.ds.getWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_FS_BX_SI: return cpu.fs.getWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_FS_BX_DI: return cpu.fs.getWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_FS_BP_SI: return cpu.fs.getWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_FS_BP_DI: return cpu.fs.getWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_FS_SI: return cpu.fs.getWord(cpu.esi & 0xffff);
	case M_FS_DI: return cpu.fs.getWord(cpu.edi & 0xffff);
	case M_FS_IW: return cpu.fs.getWord(getMicrocode() & 0xffff);
	case M_FS_BX: return cpu.fs.getWord(cpu.ebx & 0xffff);
	case M_FS_BX_SI_IB: return cpu.fs.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IB: return cpu.fs.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IB: return cpu.fs.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IB: return cpu.fs.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IB: return cpu.fs.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IB: return cpu.fs.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IB: return cpu.fs.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IB: return cpu.fs.getWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_FS_BX_SI_IW: return cpu.fs.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IW: return cpu.fs.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IW: return cpu.fs.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IW: return cpu.fs.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IW: return cpu.fs.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IW: return cpu.fs.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IW: return cpu.fs.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IW: return cpu.fs.getWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_GS_BX_SI: return cpu.gs.getWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_GS_BX_DI: return cpu.gs.getWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_GS_BP_SI: return cpu.gs.getWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_GS_BP_DI: return cpu.gs.getWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_GS_SI: return cpu.gs.getWord(cpu.esi & 0xffff);
	case M_GS_DI: return cpu.gs.getWord(cpu.edi & 0xffff);
	case M_GS_IW: return cpu.gs.getWord(getMicrocode() & 0xffff);
	case M_GS_BX: return cpu.gs.getWord(cpu.ebx & 0xffff);
	case M_GS_BX_SI_IB: return cpu.gs.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IB: return cpu.gs.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IB: return cpu.gs.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IB: return cpu.gs.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IB: return cpu.gs.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IB: return cpu.gs.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IB: return cpu.gs.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IB: return cpu.gs.getWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_GS_BX_SI_IW: return cpu.gs.getWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IW: return cpu.gs.getWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IW: return cpu.gs.getWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IW: return cpu.gs.getWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IW: return cpu.gs.getWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IW: return cpu.gs.getWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IW: return cpu.gs.getWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IW: return cpu.gs.getWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_ES_EAX: return cpu.es.getWord(cpu.eax); //56
	case M_ES_ECX: return cpu.es.getWord(cpu.ecx); //57
	case M_ES_EDX: return cpu.es.getWord(cpu.edx); //58
	case M_ES_EBX: return cpu.es.getWord(cpu.ebx); //59
	case M_ES_ID: return cpu.es.getWord(getMicrocode()); //60
	case M_ES_ESI: return cpu.es.getWord(cpu.esi); //61
	case M_ES_EDI: return cpu.es.getWord(cpu.edi); //62

	case M_ES_EAX_IB: return cpu.es.getWord(cpu.eax + getMicrocode()); //63
	case M_ES_ECX_IB: return cpu.es.getWord(cpu.ecx + getMicrocode()); //64
	case M_ES_EDX_IB: return cpu.es.getWord(cpu.edx + getMicrocode()); //65
	case M_ES_EBX_IB: return cpu.es.getWord(cpu.ebx + getMicrocode()); //66
	case M_ES_EBP_IB: return cpu.es.getWord(cpu.ebp + getMicrocode()); //67
	case M_ES_ESI_IB: return cpu.es.getWord(cpu.esi + getMicrocode()); //68
	case M_ES_EDI_IB: return cpu.es.getWord(cpu.edi + getMicrocode()); //69
	case M_ES_EAX_ID: return cpu.es.getWord(cpu.eax + getMicrocode()); //70
	case M_ES_ECX_ID: return cpu.es.getWord(cpu.ecx + getMicrocode()); //71
	case M_ES_EDX_ID: return cpu.es.getWord(cpu.edx + getMicrocode()); //72
	case M_ES_EBX_ID: return cpu.es.getWord(cpu.ebx + getMicrocode()); //73
	case M_ES_EBP_ID: return cpu.es.getWord(cpu.ebp + getMicrocode()); //74
	case M_ES_ESI_ID: return cpu.es.getWord(cpu.esi + getMicrocode()); //75
	case M_ES_EDI_ID: return cpu.es.getWord(cpu.edi + getMicrocode()); //76

	case M_SS_ID: return cpu.ss.getWord(getMicrocode());

	case M_SS_EAX_IB: return cpu.ss.getWord(cpu.eax + getMicrocode()); //90
	case M_SS_ECX_IB: return cpu.ss.getWord(cpu.ecx + getMicrocode()); //91
	case M_SS_EDX_IB: return cpu.ss.getWord(cpu.edx + getMicrocode()); //92
	case M_SS_EBX_IB: return cpu.ss.getWord(cpu.ebx + getMicrocode()); //93
	case M_SS_EBP_IB: return cpu.ss.getWord(cpu.ebp + getMicrocode()); //94
	case M_SS_ESI_IB: return cpu.ss.getWord(cpu.esi + getMicrocode()); //95
	case M_SS_EDI_IB: return cpu.ss.getWord(cpu.edi + getMicrocode()); //96

	case M_SS_EAX_ID: return cpu.ss.getWord(cpu.eax + getMicrocode()); //97
	case M_SS_ECX_ID: return cpu.ss.getWord(cpu.ecx + getMicrocode()); //98
	case M_SS_EDX_ID: return cpu.ss.getWord(cpu.edx + getMicrocode()); //99
	case M_SS_EBX_ID: return cpu.ss.getWord(cpu.ebx + getMicrocode()); //100
	case M_SS_EBP_ID: return cpu.ss.getWord(cpu.ebp + getMicrocode()); //101
	case M_SS_ESI_ID: return cpu.ss.getWord(cpu.esi + getMicrocode()); //102
	case M_SS_EDI_ID: return cpu.ss.getWord(cpu.edi + getMicrocode()); //103

	case M_DS_EAX: return cpu.ds.getWord(cpu.eax); //104
	case M_DS_ECX: return cpu.ds.getWord(cpu.ecx); //105
	case M_DS_EDX: return cpu.ds.getWord(cpu.edx); //106
	case M_DS_EBX: return cpu.ds.getWord(cpu.ebx); //107
	case M_DS_ID: return cpu.ds.getWord(getMicrocode()); //108
	case M_DS_ESI: return cpu.ds.getWord(cpu.esi); //109
	case M_DS_EDI: return cpu.ds.getWord(cpu.edi); //110

	case M_DS_EAX_IB: return cpu.ds.getWord(cpu.eax + getMicrocode()); //111
	case M_DS_ECX_IB: return cpu.ds.getWord(cpu.ecx + getMicrocode()); //112
	case M_DS_EDX_IB: return cpu.ds.getWord(cpu.edx + getMicrocode()); //113
	case M_DS_EBX_IB: return cpu.ds.getWord(cpu.ebx + getMicrocode()); //114
	case M_DS_EBP_IB: return cpu.ds.getWord(cpu.ebp + getMicrocode()); //115 
	case M_DS_ESI_IB: return cpu.ds.getWord(cpu.esi + getMicrocode()); //116
	case M_DS_EDI_IB: return cpu.ds.getWord(cpu.edi + getMicrocode()); //117

	case M_DS_EAX_ID: return cpu.ds.getWord(cpu.eax + getMicrocode()); //118
	case M_DS_ECX_ID: return cpu.ds.getWord(cpu.ecx + getMicrocode()); //119
	case M_DS_EDX_ID: return cpu.ds.getWord(cpu.edx + getMicrocode()); //120
	case M_DS_EBX_ID: return cpu.ds.getWord(cpu.ebx + getMicrocode()); //121
	case M_DS_EBP_ID: return cpu.ds.getWord(cpu.ebp + getMicrocode()); //122
	case M_DS_ESI_ID: return cpu.ds.getWord(cpu.esi + getMicrocode()); //123
	case M_DS_EDI_ID: return cpu.ds.getWord(cpu.edi + getMicrocode()); //124

	case M_DS_2EAX: return cpu.ds.getWord(cpu.eax + cpu.eax); //2156
	case M_DS_EAX_ECX: return cpu.ds.getWord(cpu.eax + cpu.ecx); //2157
	case M_DS_EAX_EDX: return cpu.ds.getWord(cpu.eax + cpu.edx); //2158
	case M_DS_EAX_EBX: return cpu.ds.getWord(cpu.eax + cpu.ebx); //2159
	case M_DS_EAX_ESP: return cpu.ds.getWord(cpu.eax + cpu.esp); //2160
	case M_DS_EAX_EBP: return cpu.ds.getWord(cpu.eax + cpu.ebp); //2161
	case M_DS_EAX_ESI: return cpu.ds.getWord(cpu.eax + cpu.esi); //2162
	case M_DS_EAX_EDI: return cpu.ds.getWord(cpu.eax + cpu.edi); //2163

	case M_DS_2ECX: return cpu.ds.getWord(cpu.ecx + cpu.ecx); //2164
	case M_DS_ECX_EDX: return cpu.ds.getWord(cpu.ecx + cpu.edx); //2165
	case M_DS_ECX_EBX: return cpu.ds.getWord(cpu.ecx + cpu.ebx); //2166
	case M_DS_ECX_ESP: return cpu.ds.getWord(cpu.ecx + cpu.esp); //2167
	case M_DS_ECX_EBP: return cpu.ds.getWord(cpu.ecx + cpu.ebp); //2168
	case M_DS_ECX_ESI: return cpu.ds.getWord(cpu.ecx + cpu.esi); //2169
	case M_DS_ECX_EDI: return cpu.ds.getWord(cpu.ecx + cpu.edi); //2170

	case M_DS_2EDX: return cpu.ds.getWord(cpu.edx + cpu.edx); //2171
	case M_DS_EDX_EBX: return cpu.ds.getWord(cpu.edx + cpu.ebx); //2172
	case M_DS_EDX_ESP: return cpu.ds.getWord(cpu.edx + cpu.esp); //2173
	case M_DS_EDX_EBP: return cpu.ds.getWord(cpu.edx + cpu.ebp); //2174
	case M_DS_EDX_ESI: return cpu.ds.getWord(cpu.edx + cpu.esi); //2175	
	case M_DS_EDX_EDI: return cpu.ds.getWord(cpu.edx + cpu.edi); //2176

	case M_DS_2EBX: return cpu.ds.getWord(cpu.ebx + cpu.ebx); //2177;
	case M_DS_EBX_ESP: return cpu.ds.getWord(cpu.ebx + cpu.esp); //2178
	case M_DS_EBX_EBP: return cpu.ds.getWord(cpu.ebx + cpu.ebp); //2179
	case M_DS_EBX_ESI: return cpu.ds.getWord(cpu.ebx + cpu.esi); //2180
	case M_DS_EBX_EDI: return cpu.ds.getWord(cpu.ebx + cpu.edi); //2181

	case M_DS_ESP: return cpu.ds.getWord(cpu.esp); //2182	    

	case M_DS_EBP_ESP: return cpu.ds.getWord(cpu.ebp + cpu.esp); //2183;
	case M_DS_EBP_ESI: return cpu.ds.getWord(cpu.ebp + cpu.esi); //2184;
	case M_DS_EBP_EDI: return cpu.ds.getWord(cpu.ebp + cpu.edi); //2185;

	case M_DS_ESI_ESP: return cpu.ds.getWord(cpu.esi + cpu.esp); //2186;
	case M_DS_2ESI: return cpu.ds.getWord(cpu.esi + cpu.esi); //2187;
	case M_DS_ESI_EDI: return cpu.ds.getWord(cpu.esi + cpu.edi); //2188;

	case M_DS_EDI_ESP: return cpu.ds.getWord(cpu.edi + cpu.esp); //2189;
	case M_DS_2EDI: return cpu.ds.getWord(cpu.edi + cpu.edi); //2190;

	case M_DS_3EAX: return cpu.ds.getWord((cpu.eax << 1) + cpu.eax); //2191;
	case M_DS_2EAX_ECX: return cpu.ds.getWord((cpu.eax << 1) + cpu.ecx); //2192;
	case M_DS_2EAX_EDX: return cpu.ds.getWord((cpu.eax << 1) + cpu.edx); //2193;
	case M_DS_2EAX_EBX: return cpu.ds.getWord((cpu.eax << 1) + cpu.ebx); //2194;
	case M_DS_2EAX_ESP: return cpu.ds.getWord((cpu.eax << 1) + cpu.esp); //2195;
	case M_DS_2EAX_ID: return cpu.ds.getWord((cpu.eax << 1) + getMicrocode()); //2196;
	case M_DS_2EAX_ESI: return cpu.ds.getWord((cpu.eax << 1) + cpu.esi); //2197;
	case M_DS_2EAX_EDI: return cpu.ds.getWord((cpu.eax << 1) + cpu.edi); //2198;

	case M_DS_2ECX_EAX: return cpu.ds.getWord((cpu.ecx << 1) + cpu.eax); //2199;
	case M_DS_3ECX: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_DS_2ECX_EDX: return cpu.ds.getWord((cpu.ecx << 1) + cpu.edx); //2201;
	case M_DS_2ECX_EBX: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_DS_2ECX_ESP: return cpu.ds.getWord((cpu.ecx << 1) + cpu.esp); //2203;
	case M_DS_2ECX_ID: return cpu.ds.getWord((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_DS_2ECX_ESI: return cpu.ds.getWord((cpu.ecx << 1) + cpu.esi); //2205;
	case M_DS_2ECX_EDI: return cpu.ds.getWord((cpu.ecx << 1) + cpu.edi); //2206;

	case M_DS_2EDX_EAX: return cpu.ds.getWord((cpu.edx << 1) + cpu.eax); //2207;
	case M_DS_2EDX_ECX: return cpu.ds.getWord((cpu.edx << 1) + cpu.ecx); //2208;
	case M_DS_3EDX: return cpu.ds.getWord((cpu.edx << 1) + cpu.edx); //2209;
	case M_DS_2EDX_EBX: return cpu.ds.getWord((cpu.edx << 1) + cpu.ebx); //2210;
	case M_DS_2EDX_ESP: return cpu.ds.getWord((cpu.edx << 1) + cpu.esp); //2211;
	case M_DS_2EDX_ID: return cpu.ds.getWord((cpu.edx << 1) + getMicrocode()); //2212;
	case M_DS_2EDX_ESI: return cpu.ds.getWord((cpu.edx << 1) + cpu.esi); //2213;
	case M_DS_2EDX_EDI: return cpu.ds.getWord((cpu.edx << 1) + cpu.edi); //2214;

	case M_DS_2EBX_EAX: return cpu.ds.getWord((cpu.ebx << 1) + cpu.eax); //2215;
	case M_DS_2EBX_ECX: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_DS_2EBX_EDX: return cpu.ds.getWord((cpu.ebx << 1) + cpu.edx); //2217;
	case M_DS_3EBX: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_DS_2EBX_ESP: return cpu.ds.getWord((cpu.ebx << 1) + cpu.esp); //2219;
	case M_DS_2EBX_ID: return cpu.ds.getWord((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_DS_2EBX_ESI: return cpu.ds.getWord((cpu.ebx << 1) + cpu.esi); //2221;
	case M_DS_2EBX_EDI: return cpu.ds.getWord((cpu.ebx << 1) + cpu.edi); //2222;

	case M_DS_2EBP_EAX: return cpu.ds.getWord((cpu.ebp << 1) + cpu.eax); //2223;
	case M_DS_2EBP_ECX: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_DS_2EBP_EDX: return cpu.ds.getWord((cpu.ebp << 1) + cpu.edx); //2225;
	case M_DS_2EBP_EBX: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_DS_2EBP_ESP: return cpu.ds.getWord((cpu.ebp << 1) + cpu.esp); //2227;
	case M_DS_2EBP_ID: return cpu.ds.getWord((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_DS_2EBP_ESI: return cpu.ds.getWord((cpu.ebp << 1) + cpu.esi); //2229;
	case M_DS_2EBP_EDI: return cpu.ds.getWord((cpu.ebp << 1) + cpu.edi); //2230;

	case M_DS_2ESI_EAX: return cpu.ds.getWord((cpu.esi << 1) + cpu.eax); //2231;
	case M_DS_2ESI_ECX: return cpu.ds.getWord((cpu.esi << 1) + cpu.ecx); //2232;
	case M_DS_2ESI_EDX: return cpu.ds.getWord((cpu.esi << 1) + cpu.edx); //2233;
	case M_DS_2ESI_EBX: return cpu.ds.getWord((cpu.esi << 1) + cpu.ebx); //2234;
	case M_DS_2ESI_ESP: return cpu.ds.getWord((cpu.esi << 1) + cpu.esp); //2235;
	case M_DS_2ESI_ID: return cpu.ds.getWord((cpu.esi << 1) + getMicrocode()); //2236;
	case M_DS_3ESI: return cpu.ds.getWord((cpu.esi << 1) + cpu.esi); //2237;
	case M_DS_2ESI_EDI: return cpu.ds.getWord((cpu.esi << 1) + cpu.edi); //2238;

	case M_DS_2EDI_EAX: return cpu.ds.getWord((cpu.edi << 1) + cpu.eax); //2239;
	case M_DS_2EDI_ECX: return cpu.ds.getWord((cpu.edi << 1) + cpu.ecx); //2240;
	case M_DS_2EDI_EDX: return cpu.ds.getWord((cpu.edi << 1) + cpu.edx); //2241;
	case M_DS_2EDI_EBX: return cpu.ds.getWord((cpu.edi << 1) + cpu.ebx); //2242;
	case M_DS_2EDI_ESP: return cpu.ds.getWord((cpu.edi << 1) + cpu.esp); //2243;
	case M_DS_2EDI_ID: return cpu.ds.getWord((cpu.edi << 1) + getMicrocode()); //2244;
	case M_DS_2EDI_ESI: return cpu.ds.getWord((cpu.edi << 1) + cpu.esi); //2245;
	case M_DS_3EDI: return cpu.ds.getWord((cpu.edi << 1) + cpu.edi); //2246;

	case M_DS_5EAX: return cpu.ds.getWord((cpu.eax << 2) + cpu.eax); //2247
	case M_DS_4EAX_ECX: return cpu.ds.getWord((cpu.eax << 2) + cpu.ecx); //2248
	case M_DS_4EAX_EDX: return cpu.ds.getWord((cpu.eax << 2) + cpu.edx); //2249
	case M_DS_4EAX_EBX: return cpu.ds.getWord((cpu.eax << 2) + cpu.ebx); //2250
	case M_DS_4EAX_ESP: return cpu.ds.getWord((cpu.eax << 2) + cpu.esp); //2251
	case M_DS_4EAX_ID: return cpu.ds.getWord((cpu.eax << 2) + getMicrocode()); //2252
	case M_DS_4EAX_ESI: return cpu.ds.getWord((cpu.eax << 2) + cpu.esi); //2253
	case M_DS_4EAX_EDI: return cpu.ds.getWord((cpu.eax << 2) + cpu.edi); //2254

	case M_DS_4ECX_EAX: return cpu.ds.getWord((cpu.ecx << 2) + cpu.eax); //2255
	case M_DS_5ECX: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ecx); //2256
	case M_DS_4ECX_EDX: return cpu.ds.getWord((cpu.ecx << 2) + cpu.edx); //2257
	case M_DS_4ECX_EBX: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ebx); //2258
	case M_DS_4ECX_ESP: return cpu.ds.getWord((cpu.ecx << 2) + cpu.esp); //2259
	case M_DS_4ECX_ID: return cpu.ds.getWord((cpu.ecx << 2) + getMicrocode()); //2260
	case M_DS_4ECX_ESI: return cpu.ds.getWord((cpu.ecx << 2) + cpu.esi); //2261
	case M_DS_4ECX_EDI: return cpu.ds.getWord((cpu.ecx << 2) + cpu.edi); //2262

	case M_DS_4EDX_EAX: return cpu.ds.getWord((cpu.edx << 2) + cpu.eax); //2263
	case M_DS_4EDX_ECX: return cpu.ds.getWord((cpu.edx << 2) + cpu.ecx); //2264
	case M_DS_5EDX: return cpu.ds.getWord((cpu.edx << 2) + cpu.edx); //2265
	case M_DS_4EDX_EBX: return cpu.ds.getWord((cpu.edx << 2) + cpu.ebx); //2266
	case M_DS_4EDX_ESP: return cpu.ds.getWord((cpu.edx << 2) + cpu.esp); //2267
	case M_DS_4EDX_ID: return cpu.ds.getWord((cpu.edx << 2) + getMicrocode()); //2268
	case M_DS_4EDX_ESI: return cpu.ds.getWord((cpu.edx << 2) + cpu.esi); //2269
	case M_DS_4EDX_EDI: return cpu.ds.getWord((cpu.edx << 2) + cpu.edi); //2270

	case M_DS_4EBX_EAX: return cpu.ds.getWord((cpu.ebx << 2) + cpu.eax); //2271
	case M_DS_4EBX_ECX: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ecx); //2272
	case M_DS_4EBX_EDX: return cpu.ds.getWord((cpu.ebx << 2) + cpu.edx); //2273
	case M_DS_5EBX: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ebx); //2274
	case M_DS_4EBX_ESP: return cpu.ds.getWord((cpu.ebx << 2) + cpu.esp); //2275
	case M_DS_4EBX_ID: return cpu.ds.getWord((cpu.ebx << 2) + getMicrocode()); //2276
	case M_DS_4EBX_ESI: return cpu.ds.getWord((cpu.ebx << 2) + cpu.esi); //2277
	case M_DS_4EBX_EDI: return cpu.ds.getWord((cpu.ebx << 2) + cpu.edi); //2278

	case M_DS_4EBP_EAX: return cpu.ds.getWord((cpu.ebp << 2) + cpu.eax); //2279
	case M_DS_4EBP_ECX: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ecx); //2280
	case M_DS_4EBP_EDX: return cpu.ds.getWord((cpu.ebp << 2) + cpu.edx); //2281
	case M_DS_4EBP_EBX: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ebx); //2282
	case M_DS_4EBP_ESP: return cpu.ds.getWord((cpu.ebp << 2) + cpu.esp); //2283
	case M_DS_4EBP_ID: return cpu.ds.getWord((cpu.ebp << 2) + getMicrocode()); //2284
	case M_DS_4EBP_ESI: return cpu.ds.getWord((cpu.ebp << 2) + cpu.esi); //2285
	case M_DS_4EBP_EDI: return cpu.ds.getWord((cpu.ebp << 2) + cpu.edi); //2286

	case M_DS_4ESI_EAX: return cpu.ds.getWord((cpu.esi << 2) + cpu.eax); //2287
	case M_DS_4ESI_ECX: return cpu.ds.getWord((cpu.esi << 2) + cpu.ecx); //2288
	case M_DS_4ESI_EDX: return cpu.ds.getWord((cpu.esi << 2) + cpu.edx); //2289
	case M_DS_4ESI_EBX: return cpu.ds.getWord((cpu.esi << 2) + cpu.ebx); //2290
	case M_DS_4ESI_ESP: return cpu.ds.getWord((cpu.esi << 2) + cpu.esp); //2291
	case M_DS_4ESI_ID: return cpu.ds.getWord((cpu.esi << 2) + getMicrocode()); //2292
	case M_DS_5ESI: return cpu.ds.getWord((cpu.esi << 2) + cpu.esi); //2293
	case M_DS_4ESI_EDI: return cpu.ds.getWord((cpu.esi << 2) + cpu.edi); //2294

	case M_DS_4EDI_EAX: return cpu.ds.getWord((cpu.edi << 2) + cpu.eax); //2295;
	case M_DS_4EDI_ECX: return cpu.ds.getWord((cpu.edi << 2) + cpu.ecx); //2296;
	case M_DS_4EDI_EDX: return cpu.ds.getWord((cpu.edi << 2) + cpu.edx); //2297;
	case M_DS_4EDI_EBX: return cpu.ds.getWord((cpu.edi << 2) + cpu.ebx); //2298;
	case M_DS_4EDI_ESP: return cpu.ds.getWord((cpu.edi << 2) + cpu.esp); //2299;
	case M_DS_4EDI_ID: return cpu.ds.getWord((cpu.edi << 2) + getMicrocode()); //2300;
	case M_DS_4EDI_ESI: return cpu.ds.getWord((cpu.edi << 2) + cpu.esi); //2301;
	case M_DS_5EDI: return cpu.ds.getWord((cpu.edi << 2) + cpu.edi); //2302;

	case M_DS_9EAX: return cpu.ds.getWord((cpu.eax << 3) + cpu.eax); //2303;
	case M_DS_8EAX_ECX: return cpu.ds.getWord((cpu.eax << 3) + cpu.ecx); //2304;
	case M_DS_8EAX_EDX: return cpu.ds.getWord((cpu.eax << 3) + cpu.edx); //2305;
	case M_DS_8EAX_EBX: return cpu.ds.getWord((cpu.eax << 3) + cpu.ebx); //2306;
	case M_DS_8EAX_ESP: return cpu.ds.getWord((cpu.eax << 3) + cpu.esp); //2307;
	case M_DS_8EAX_ID: return cpu.ds.getWord((cpu.eax << 3) + getMicrocode()); //2308;
	case M_DS_8EAX_ESI: return cpu.ds.getWord((cpu.eax << 3) + cpu.esi); //2309;
	case M_DS_8EAX_EDI: return cpu.ds.getWord((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_DS_8ECX_EAX: return cpu.ds.getWord((cpu.ecx << 3) + cpu.eax); //2311;
	case M_DS_9ECX: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_DS_8ECX_EDX: return cpu.ds.getWord((cpu.ecx << 3) + cpu.edx); //2313;
	case M_DS_8ECX_EBX: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_DS_8ECX_ESP: return cpu.ds.getWord((cpu.ecx << 3) + cpu.esp); //2315;
	case M_DS_8ECX_ID: return cpu.ds.getWord((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_DS_8ECX_ESI: return cpu.ds.getWord((cpu.ecx << 3) + cpu.esi); //2317;
	case M_DS_8ECX_EDI: return cpu.ds.getWord((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_DS_8EDX_EAX: return cpu.ds.getWord((cpu.edx << 3) + cpu.eax); //2319;
	case M_DS_8EDX_ECX: return cpu.ds.getWord((cpu.edx << 3) + cpu.ecx); //2320;
	case M_DS_9EDX: return cpu.ds.getWord((cpu.edx << 3) + cpu.edx); //2321;
	case M_DS_8EDX_EBX: return cpu.ds.getWord((cpu.edx << 3) + cpu.ebx); //2322;
	case M_DS_8EDX_ESP: return cpu.ds.getWord((cpu.edx << 3) + cpu.esp); //2323;
	case M_DS_8EDX_ID: return cpu.ds.getWord((cpu.edx << 3) + getMicrocode()); //2324;
	case M_DS_8EDX_ESI: return cpu.ds.getWord((cpu.edx << 3) + cpu.esi); //2325;
	case M_DS_8EDX_EDI: return cpu.ds.getWord((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_DS_8EBX_EAX: return cpu.ds.getWord((cpu.ebx << 3) + cpu.eax); //2327
	case M_DS_8EBX_ECX: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ecx); //2328
	case M_DS_8EBX_EDX: return cpu.ds.getWord((cpu.ebx << 3) + cpu.edx); //2329
	case M_DS_9EBX: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ebx); //2330
	case M_DS_8EBX_ESP: return cpu.ds.getWord((cpu.ebx << 3) + cpu.esp); //2331
	case M_DS_8EBX_ID: return cpu.ds.getWord((cpu.ebx << 3) + getMicrocode()); //2332
	case M_DS_8EBX_ESI: return cpu.ds.getWord((cpu.ebx << 3) + cpu.esi); //2333
	case M_DS_8EBX_EDI: return cpu.ds.getWord((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_DS_8EBP_EAX: return cpu.ds.getWord((cpu.ebp << 3) + cpu.eax); //2335;
	case M_DS_8EBP_ECX: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_DS_8EBP_EDX: return cpu.ds.getWord((cpu.ebp << 3) + cpu.edx); //2337;
	case M_DS_8EBP_EBX: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_DS_8EBP_ESP: return cpu.ds.getWord((cpu.ebp << 3) + cpu.esp); //2339;
	case M_DS_8EBP_ID: return cpu.ds.getWord((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_DS_8EBP_ESI: return cpu.ds.getWord((cpu.ebp << 3) + cpu.esi); //2341;
	case M_DS_8EBP_EDI: return cpu.ds.getWord((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_DS_8ESI_EAX: return cpu.ds.getWord((cpu.esi << 3) + cpu.eax); //2343;
	case M_DS_8ESI_ECX: return cpu.ds.getWord((cpu.esi << 3) + cpu.ecx); //2344;
	case M_DS_8ESI_EDX: return cpu.ds.getWord((cpu.esi << 3) + cpu.edx); //2345;
	case M_DS_8ESI_EBX: return cpu.ds.getWord((cpu.esi << 3) + cpu.ebx); //2346;
	case M_DS_8ESI_ESP: return cpu.ds.getWord((cpu.esi << 3) + cpu.esp); //2347;
	case M_DS_8ESI_ID: return cpu.ds.getWord((cpu.esi << 3) + getMicrocode()); //2348;
	case M_DS_9ESI: return cpu.ds.getWord((cpu.esi << 3) + cpu.esi); //2349;
	case M_DS_8ESI_EDI: return cpu.ds.getWord((cpu.esi << 3) + cpu.edi); //2350;
	    
	case M_DS_8EDI_EAX: return cpu.ds.getWord((cpu.edi << 3) + cpu.eax); //2351;
	case M_DS_8EDI_ECX: return cpu.ds.getWord((cpu.edi << 3) + cpu.ecx); //2352;
	case M_DS_8EDI_EDX: return cpu.ds.getWord((cpu.edi << 3) + cpu.edx); //2353;
	case M_DS_8EDI_EBX: return cpu.ds.getWord((cpu.edi << 3) + cpu.ebx); //2354;
	case M_DS_8EDI_ESP: return cpu.ds.getWord((cpu.edi << 3) + cpu.esp); //2355;
	case M_DS_8EDI_ID: return cpu.ds.getWord((cpu.edi << 3) + getMicrocode()); //2356;
	case M_DS_8EDI_ESI: return cpu.ds.getWord((cpu.edi << 3) + cpu.esi); //2357;
	case M_DS_9EDI: return cpu.ds.getWord((cpu.edi << 3) + cpu.edi); //2358;

	case M_DS_2EAX_IB: return cpu.ds.getWord(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_DS_EAX_ECX_IB: return cpu.ds.getWord(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_DS_EAX_EDX_IB: return cpu.ds.getWord(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_DS_EAX_EBX_IB: return cpu.ds.getWord(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_DS_EAX_ESP_IB: return cpu.ds.getWord(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_DS_EAX_EBP_IB: return cpu.ds.getWord(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_DS_EAX_ESI_IB: return cpu.ds.getWord(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_DS_EAX_EDI_IB: return cpu.ds.getWord(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_DS_2ECX_IB: return cpu.ds.getWord(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_DS_ECX_EDX_IB: return cpu.ds.getWord(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_DS_ECX_EBX_IB: return cpu.ds.getWord(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_DS_ECX_ESP_IB: return cpu.ds.getWord(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_DS_ECX_EBP_IB: return cpu.ds.getWord(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_DS_ECX_ESI_IB: return cpu.ds.getWord(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_DS_ECX_EDI_IB: return cpu.ds.getWord(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_DS_2EDX_IB: return cpu.ds.getWord(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_DS_EDX_EBX_IB: return cpu.ds.getWord(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_DS_EDX_ESP_IB: return cpu.ds.getWord(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_DS_EDX_EBP_IB: return cpu.ds.getWord(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_DS_EDX_ESI_IB: return cpu.ds.getWord(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_DS_EDX_EDI_IB: return cpu.ds.getWord(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_DS_2EBX_IB: return cpu.ds.getWord(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_DS_EBX_ESP_IB: return cpu.ds.getWord(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_DS_EBX_EBP_IB: return cpu.ds.getWord(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_DS_EBX_ESI_IB: return cpu.ds.getWord(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_DS_EBX_EDI_IB: return cpu.ds.getWord(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_DS_ESP_IB: return cpu.ds.getWord(cpu.esp + getMicrocode()); //2385

	case M_DS_EBP_ESP_IB: return cpu.ds.getWord(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_DS_2EBP_IB: return cpu.ds.getWord(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_DS_EBP_ESI_IB: return cpu.ds.getWord(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_DS_EBP_EDI_IB: return cpu.ds.getWord(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_DS_ESI_ESP_IB: return cpu.ds.getWord(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_DS_2ESI_IB: return cpu.ds.getWord(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_DS_ESI_EDI_IB: return cpu.ds.getWord(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_DS_EDI_ESP_IB: return cpu.ds.getWord(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_DS_2EDI_IB: return cpu.ds.getWord(cpu.edi + cpu.edi + getMicrocode()); //2394;
	    
	case M_DS_3EAX_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_DS_2EAX_ECX_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_DS_2EAX_EDX_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_DS_2EAX_EBX_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_DS_2EAX_ESP_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_DS_2EAX_EBP_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_DS_2EAX_ESI_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_DS_2EAX_EDI_IB: return cpu.ds.getWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_DS_2ECX_EAX_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_DS_3ECX_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_DS_2ECX_EDX_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_DS_2ECX_EBX_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_DS_2ECX_ESP_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_DS_2ECX_EBP_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_DS_2ECX_ESI_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_DS_2ECX_EDI_IB: return cpu.ds.getWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_DS_2EDX_EAX_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_DS_2EDX_ECX_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_DS_3EDX_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_DS_2EDX_EBX_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_DS_2EDX_ESP_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_DS_2EDX_EBP_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_DS_2EDX_ESI_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_DS_2EDX_EDI_IB: return cpu.ds.getWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;

	case M_DS_2EBX_EAX_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_DS_2EBX_ECX_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_DS_2EBX_EDX_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_DS_3EBX_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_DS_2EBX_ESP_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_DS_2EBX_EBP_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_DS_2EBX_ESI_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_DS_2EBX_EDI_IB: return cpu.ds.getWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_DS_2EBP_EAX_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_DS_2EBP_ECX_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_DS_2EBP_EDX_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_DS_2EBP_EBX_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_DS_2EBP_ESP_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_DS_3EBP_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_DS_2EBP_ESI_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_DS_2EBP_EDI_IB: return cpu.ds.getWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_DS_2ESI_EAX_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_DS_2ESI_ECX_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_DS_2ESI_EDX_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_DS_2ESI_EBX_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_DS_2ESI_ESP_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_DS_2ESI_EBP_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_DS_3ESI_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_DS_2ESI_EDI_IB: return cpu.ds.getWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_DS_2EDI_EAX_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_DS_2EDI_ECX_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_DS_2EDI_EDX_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_DS_2EDI_EBX_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_DS_2EDI_ESP_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_DS_2EDI_EBP_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_DS_2EDI_ESI_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_DS_3EDI_IB: return cpu.ds.getWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_DS_5EAX_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_DS_4EAX_ECX_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_DS_4EAX_EDX_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_DS_4EAX_EBX_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_DS_4EAX_ESP_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_DS_4EAX_EBP_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_DS_4EAX_ESI_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_DS_4EAX_EDI_IB: return cpu.ds.getWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_DS_4ECX_EAX_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_DS_5ECX_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_DS_4ECX_EDX_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_DS_4ECX_EBX_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_DS_4ECX_ESP_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_DS_4ECX_EBP_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_DS_4ECX_ESI_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_DS_4ECX_EDI_IB: return cpu.ds.getWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_DS_4EDX_EAX_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_DS_4EDX_ECX_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_DS_5EDX_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_DS_4EDX_EBX_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_DS_4EDX_ESP_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_DS_4EDX_EBP_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_DS_4EDX_ESI_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_DS_4EDX_EDI_IB: return cpu.ds.getWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_DS_4EBX_EAX_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_DS_4EBX_ECX_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_DS_4EBX_EDX_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_DS_5EBX_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_DS_4EBX_ESP_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_DS_4EBX_EBP_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_DS_4EBX_ESI_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_DS_4EBX_EDI_IB: return cpu.ds.getWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_DS_4EBP_EAX_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_DS_4EBP_ECX_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_DS_4EBP_EDX_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_DS_4EBP_EBX_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_DS_4EBP_ESP_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_DS_5EBP_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_DS_4EBP_ESI_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_DS_4EBP_EDI_IB: return cpu.ds.getWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_DS_4ESI_EAX_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_DS_4ESI_ECX_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_DS_4ESI_EDX_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_DS_4ESI_EBX_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_DS_4ESI_ESP_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_DS_4ESI_EBP_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_DS_5ESI_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_DS_4ESI_EDI_IB: return cpu.ds.getWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_DS_4EDI_EAX_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_DS_4EDI_ECX_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_DS_4EDI_EDX_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_DS_4EDI_EBX_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_DS_4EDI_ESP_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_DS_4EDI_EBP_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_DS_4EDI_ESI_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_DS_5EDI_IB: return cpu.ds.getWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_DS_9EAX_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_DS_8EAX_ECX_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_DS_8EAX_EDX_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_DS_8EAX_EBX_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_DS_8EAX_ESP_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_DS_8EAX_EBP_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_DS_8EAX_ESI_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_DS_8EAX_EDI_IB: return cpu.ds.getWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_DS_8ECX_EAX_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_DS_9ECX_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_DS_8ECX_EDX_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_DS_8ECX_EBX_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_DS_8ECX_ESP_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_DS_8ECX_EBP_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_DS_8ECX_ESI_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_DS_8ECX_EDI_IB: return cpu.ds.getWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_DS_8EDX_EAX_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_DS_8EDX_ECX_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_DS_9EDX_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_DS_8EDX_EBX_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_DS_8EDX_ESP_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_DS_8EDX_EBP_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_DS_8EDX_ESI_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_DS_8EDX_EDI_IB: return cpu.ds.getWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_DS_8EBX_EAX_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_DS_8EBX_ECX_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_DS_8EBX_EDX_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_DS_9EBX_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_DS_8EBX_ESP_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_DS_8EBX_EBP_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_DS_8EBX_ESI_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_DS_8EBX_EDI_IB: return cpu.ds.getWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_DS_8EBP_EAX_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_DS_8EBP_ECX_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_DS_8EBP_EDX_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_DS_8EBP_EBX_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_DS_8EBP_ESP_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_DS_9EBP_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_DS_8EBP_ESI_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_DS_8EBP_EDI_IB: return cpu.ds.getWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_DS_8ESI_EAX_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_DS_8ESI_ECX_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_DS_8ESI_EDX_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_DS_8ESI_EBX_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_DS_8ESI_ESP_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_DS_8ESI_EBP_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_DS_9ESI_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_DS_8ESI_EDI_IB: return cpu.ds.getWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_DS_8EDI_EAX_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_DS_8EDI_ECX_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_DS_8EDI_EDX_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_DS_8EDI_EBX_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_DS_8EDI_ESP_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_DS_8EDI_EBP_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_DS_8EDI_ESI_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_DS_9EDI_IB: return cpu.ds.getWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_DS_EAX_ECX_ID: return cpu.ds.getWord(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_DS_EAX_EDX_ID: return cpu.ds.getWord(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_DS_EAX_EBX_ID: return cpu.ds.getWord(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_DS_EAX_ESP_ID: return cpu.ds.getWord(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_DS_EAX_EBP_ID: return cpu.ds.getWord(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_DS_EAX_ESI_ID: return cpu.ds.getWord(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_DS_EAX_EDI_ID: return cpu.ds.getWord(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_DS_ECX_EDX_ID: return cpu.ds.getWord(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_DS_ECX_EBX_ID: return cpu.ds.getWord(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_DS_ECX_ESP_ID: return cpu.ds.getWord(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_DS_ECX_EBP_ID: return cpu.ds.getWord(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_DS_ECX_ESI_ID: return cpu.ds.getWord(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_DS_ECX_EDI_ID: return cpu.ds.getWord(cpu.ecx + cpu.edi + getMicrocode()); //2575;
	    
	case M_DS_EDX_EBX_ID: return cpu.ds.getWord(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_DS_EDX_ESP_ID: return cpu.ds.getWord(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_DS_EDX_EBP_ID: return cpu.ds.getWord(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_DS_EDX_ESI_ID: return cpu.ds.getWord(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_DS_EDX_EDI_ID: return cpu.ds.getWord(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_DS_EBX_ESP_ID: return cpu.ds.getWord(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_DS_EBX_EBP_ID: return cpu.ds.getWord(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_DS_EBX_ESI_ID: return cpu.ds.getWord(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_DS_EBX_EDI_ID: return cpu.ds.getWord(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_DS_ESP_ID: return cpu.ds.getWord(cpu.esp + getMicrocode()); //2585

	case M_DS_EBP_ESP_ID: return cpu.ds.getWord(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_DS_EBP_ESI_ID: return cpu.ds.getWord(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_DS_EBP_EDI_ID: return cpu.ds.getWord(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_DS_ESI_ESP_ID: return cpu.ds.getWord(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_DS_ESI_EDI_ID: return cpu.ds.getWord(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_DS_EDI_ESP_ID: return cpu.ds.getWord(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_DS_3EAX_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_DS_2EAX_ECX_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_DS_2EAX_EDX_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_DS_2EAX_EBX_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_DS_2EAX_ESP_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_DS_2EAX_EBP_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_DS_2EAX_ESI_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_DS_2EAX_EDI_ID: return cpu.ds.getWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_DS_2ECX_EAX_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_DS_3ECX_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_DS_2ECX_EDX_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_DS_2ECX_EBX_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_DS_2ECX_ESP_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_DS_2ECX_EBP_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_DS_2ECX_ESI_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_DS_2ECX_EDI_ID: return cpu.ds.getWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_DS_2EDX_EAX_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_DS_2EDX_ECX_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_DS_3EDX_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_DS_2EDX_EBX_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_DS_2EDX_ESP_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_DS_2EDX_EBP_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_DS_2EDX_ESI_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_DS_2EDX_EDI_ID: return cpu.ds.getWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_DS_2EBX_EAX_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_DS_2EBX_ECX_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_DS_2EBX_EDX_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_DS_3EBX_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_DS_2EBX_ESP_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_DS_2EBX_EBP_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_DS_2EBX_ESI_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_DS_2EBX_EDI_ID: return cpu.ds.getWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_DS_2EBP_EAX_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_DS_2EBP_ECX_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_DS_2EBP_EDX_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_DS_2EBP_EBX_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_DS_2EBP_ESP_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_DS_3EBP_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_DS_2EBP_ESI_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_DS_2EBP_EDI_ID: return cpu.ds.getWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_DS_2ESI_EAX_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_DS_2ESI_ECX_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_DS_2ESI_EDX_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_DS_2ESI_EBX_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_DS_2ESI_ESP_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_DS_2ESI_EBP_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_DS_3ESI_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_DS_2ESI_EDI_ID: return cpu.ds.getWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_DS_2EDI_EAX_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_DS_2EDI_ECX_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_DS_2EDI_EDX_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_DS_2EDI_EBX_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_DS_2EDI_ESP_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_DS_2EDI_EBP_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_DS_2EDI_ESI_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_DS_3EDI_ID: return cpu.ds.getWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_DS_5EAX_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_DS_4EAX_ECX_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_DS_4EAX_EDX_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_DS_4EAX_EBX_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_DS_4EAX_ESP_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_DS_4EAX_EBP_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_DS_4EAX_ESI_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_DS_4EAX_EDI_ID: return cpu.ds.getWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_DS_4ECX_EAX_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_DS_5ECX_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_DS_4ECX_EDX_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_DS_4ECX_EBX_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_DS_4ECX_ESP_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_DS_4ECX_EBP_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_DS_4ECX_ESI_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_DS_4ECX_EDI_ID: return cpu.ds.getWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_DS_4EDX_EAX_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_DS_4EDX_ECX_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_DS_5EDX_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_DS_4EDX_EBX_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_DS_4EDX_ESP_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_DS_4EDX_EBP_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_DS_4EDX_ESI_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_DS_4EDX_EDI_ID: return cpu.ds.getWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_DS_4EBX_EAX_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_DS_4EBX_ECX_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_DS_4EBX_EDX_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_DS_5EBX_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_DS_4EBX_ESP_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_DS_4EBX_EBP_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_DS_4EBX_ESI_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_DS_4EBX_EDI_ID: return cpu.ds.getWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_DS_4EBP_EAX_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_DS_4EBP_ECX_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_DS_4EBP_EDX_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_DS_4EBP_EBX_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_DS_4EBP_ESP_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_DS_5EBP_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_DS_4EBP_ESI_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_DS_4EBP_EDI_ID: return cpu.ds.getWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_DS_4ESI_EAX_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_DS_4ESI_ECX_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_DS_4ESI_EDX_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_DS_4ESI_EBX_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_DS_4ESI_ESP_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_DS_4ESI_EBP_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_DS_5ESI_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_DS_4ESI_EDI_ID: return cpu.ds.getWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_DS_4EDI_EAX_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_DS_4EDI_ECX_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_DS_4EDI_EDX_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_DS_4EDI_EBX_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_DS_4EDI_ESP_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_DS_4EDI_EBP_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_DS_4EDI_ESI_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_DS_5EDI_ID: return cpu.ds.getWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_DS_9EAX_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_DS_8EAX_ECX_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_DS_8EAX_EDX_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_DS_8EAX_EBX_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_DS_8EAX_ESP_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_DS_8EAX_EBP_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_DS_8EAX_ESI_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_DS_8EAX_EDI_ID: return cpu.ds.getWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_DS_8ECX_EAX_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_DS_9ECX_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_DS_8ECX_EDX_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_DS_8ECX_EBX_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_DS_8ECX_ESP_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_DS_8ECX_EBP_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_DS_8ECX_ESI_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_DS_8ECX_EDI_ID: return cpu.ds.getWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_DS_8EDX_EAX_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_DS_8EDX_ECX_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_DS_9EDX_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_DS_8EDX_EBX_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_DS_8EDX_ESP_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_DS_8EDX_EBP_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_DS_8EDX_ESI_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_DS_8EDX_EDI_ID: return cpu.ds.getWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_DS_8EBX_EAX_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_DS_8EBX_ECX_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_DS_8EBX_EDX_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_DS_9EBX_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_DS_8EBX_ESP_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_DS_8EBX_EBP_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_DS_8EBX_ESI_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_DS_8EBX_EDI_ID: return cpu.ds.getWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_DS_8EBP_EAX_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_DS_8EBP_ECX_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_DS_8EBP_EDX_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_DS_8EBP_EBX_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_DS_8EBP_ESP_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_DS_9EBP_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_DS_8EBP_ESI_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_DS_8EBP_EDI_ID: return cpu.ds.getWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_DS_8ESI_EAX_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_DS_8ESI_ECX_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_DS_8ESI_EDX_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_DS_8ESI_EBX_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_DS_8ESI_ESP_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_DS_8ESI_EBP_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_DS_9ESI_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_DS_8ESI_EDI_ID: return cpu.ds.getWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_DS_8EDI_EAX_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_DS_8EDI_ECX_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_DS_8EDI_EDX_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_DS_8EDI_EBX_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_DS_8EDI_ESP_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_DS_8EDI_EBP_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_DS_8EDI_ESI_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_DS_9EDI_ID: return cpu.ds.getWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;

	case M_SS_2EAX: return cpu.ss.getWord(cpu.eax + cpu.eax); //2156
	case M_SS_EAX_ECX: return cpu.ss.getWord(cpu.eax + cpu.ecx); //2157
	case M_SS_EAX_EDX: return cpu.ss.getWord(cpu.eax + cpu.edx); //2158
	case M_SS_EAX_EBX: return cpu.ss.getWord(cpu.eax + cpu.ebx); //2159
	case M_SS_EAX_ESP: return cpu.ss.getWord(cpu.eax + cpu.esp); //2160
	case M_SS_EAX_EBP: return cpu.ss.getWord(cpu.eax + cpu.ebp); //2161
	case M_SS_EAX_ESI: return cpu.ss.getWord(cpu.eax + cpu.esi); //2162
	case M_SS_EAX_EDI: return cpu.ss.getWord(cpu.eax + cpu.edi); //2163

	case M_SS_2ECX: return cpu.ss.getWord(cpu.ecx + cpu.ecx); //2164
	case M_SS_ECX_EDX: return cpu.ss.getWord(cpu.ecx + cpu.edx); //2165
	case M_SS_ECX_EBX: return cpu.ss.getWord(cpu.ecx + cpu.ebx); //2166
	case M_SS_ECX_ESP: return cpu.ss.getWord(cpu.ecx + cpu.esp); //2167
	case M_SS_ECX_EBP: return cpu.ss.getWord(cpu.ecx + cpu.ebp); //2168
	case M_SS_ECX_ESI: return cpu.ss.getWord(cpu.ecx + cpu.esi); //2169
	case M_SS_ECX_EDI: return cpu.ss.getWord(cpu.ecx + cpu.edi); //2170

	case M_SS_2EDX: return cpu.ss.getWord(cpu.edx + cpu.edx); //2171
	case M_SS_EDX_EBX: return cpu.ss.getWord(cpu.edx + cpu.ebx); //2172
	case M_SS_EDX_ESP: return cpu.ss.getWord(cpu.edx + cpu.esp); //2173
	case M_SS_EDX_EBP: return cpu.ss.getWord(cpu.edx + cpu.ebp); //2174
	case M_SS_EDX_ESI: return cpu.ss.getWord(cpu.edx + cpu.esi); //2175	
	case M_SS_EDX_EDI: return cpu.ss.getWord(cpu.edx + cpu.edi); //2176

	case M_SS_2EBX: return cpu.ss.getWord(cpu.ebx + cpu.ebx); //2177;
	case M_SS_EBX_ESP: return cpu.ss.getWord(cpu.ebx + cpu.esp); //2178
	case M_SS_EBX_EBP: return cpu.ss.getWord(cpu.ebx + cpu.ebp); //2179
	case M_SS_EBX_ESI: return cpu.ss.getWord(cpu.ebx + cpu.esi); //2180
	case M_SS_EBX_EDI: return cpu.ss.getWord(cpu.ebx + cpu.edi); //2181

	case M_SS_ESP: return cpu.ss.getWord(cpu.esp); //2182	    

	case M_SS_EBP_ESP: return cpu.ss.getWord(cpu.ebp + cpu.esp); //2183;
	case M_SS_EBP_ESI: return cpu.ss.getWord(cpu.ebp + cpu.esi); //2184;
	case M_SS_EBP_EDI: return cpu.ss.getWord(cpu.ebp + cpu.edi); //2185;

	case M_SS_ESI_ESP: return cpu.ss.getWord(cpu.esi + cpu.esp); //2186;
	case M_SS_2ESI: return cpu.ss.getWord(cpu.esi + cpu.esi); //2187;
	case M_SS_ESI_EDI: return cpu.ss.getWord(cpu.esi + cpu.edi); //2188;

	case M_SS_EDI_ESP: return cpu.ss.getWord(cpu.edi + cpu.esp); //2189;
	case M_SS_2EDI: return cpu.ss.getWord(cpu.edi + cpu.edi); //2190;

	case M_SS_3EAX: return cpu.ss.getWord((cpu.eax << 1) + cpu.eax); //2191;
	case M_SS_2EAX_ECX: return cpu.ss.getWord((cpu.eax << 1) + cpu.ecx); //2192;
	case M_SS_2EAX_EDX: return cpu.ss.getWord((cpu.eax << 1) + cpu.edx); //2193;
	case M_SS_2EAX_EBX: return cpu.ss.getWord((cpu.eax << 1) + cpu.ebx); //2194;
	case M_SS_2EAX_ESP: return cpu.ss.getWord((cpu.eax << 1) + cpu.esp); //2195;
	case M_SS_2EAX_ID: return cpu.ss.getWord((cpu.eax << 1) + getMicrocode()); //2196;
	case M_SS_2EAX_ESI: return cpu.ss.getWord((cpu.eax << 1) + cpu.esi); //2197;
	case M_SS_2EAX_EDI: return cpu.ss.getWord((cpu.eax << 1) + cpu.edi); //2198;

	case M_SS_2ECX_EAX: return cpu.ss.getWord((cpu.ecx << 1) + cpu.eax); //2199;
	case M_SS_3ECX: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_SS_2ECX_EDX: return cpu.ss.getWord((cpu.ecx << 1) + cpu.edx); //2201;
	case M_SS_2ECX_EBX: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_SS_2ECX_ESP: return cpu.ss.getWord((cpu.ecx << 1) + cpu.esp); //2203;
	case M_SS_2ECX_ID: return cpu.ss.getWord((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_SS_2ECX_ESI: return cpu.ss.getWord((cpu.ecx << 1) + cpu.esi); //2205;
	case M_SS_2ECX_EDI: return cpu.ss.getWord((cpu.ecx << 1) + cpu.edi); //2206;

	case M_SS_2EDX_EAX: return cpu.ss.getWord((cpu.edx << 1) + cpu.eax); //2207;
	case M_SS_2EDX_ECX: return cpu.ss.getWord((cpu.edx << 1) + cpu.ecx); //2208;
	case M_SS_3EDX: return cpu.ss.getWord((cpu.edx << 1) + cpu.edx); //2209;
	case M_SS_2EDX_EBX: return cpu.ss.getWord((cpu.edx << 1) + cpu.ebx); //2210;
	case M_SS_2EDX_ESP: return cpu.ss.getWord((cpu.edx << 1) + cpu.esp); //2211;
	case M_SS_2EDX_ID: return cpu.ss.getWord((cpu.edx << 1) + getMicrocode()); //2212;
	case M_SS_2EDX_ESI: return cpu.ss.getWord((cpu.edx << 1) + cpu.esi); //2213;
	case M_SS_2EDX_EDI: return cpu.ss.getWord((cpu.edx << 1) + cpu.edi); //2214;

	case M_SS_2EBX_EAX: return cpu.ss.getWord((cpu.ebx << 1) + cpu.eax); //2215;
	case M_SS_2EBX_ECX: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_SS_2EBX_EDX: return cpu.ss.getWord((cpu.ebx << 1) + cpu.edx); //2217;
	case M_SS_3EBX: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_SS_2EBX_ESP: return cpu.ss.getWord((cpu.ebx << 1) + cpu.esp); //2219;
	case M_SS_2EBX_ID: return cpu.ss.getWord((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_SS_2EBX_ESI: return cpu.ss.getWord((cpu.ebx << 1) + cpu.esi); //2221;
	case M_SS_2EBX_EDI: return cpu.ss.getWord((cpu.ebx << 1) + cpu.edi); //2222;

	case M_SS_2EBP_EAX: return cpu.ss.getWord((cpu.ebp << 1) + cpu.eax); //2223;
	case M_SS_2EBP_ECX: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_SS_2EBP_EDX: return cpu.ss.getWord((cpu.ebp << 1) + cpu.edx); //2225;
	case M_SS_2EBP_EBX: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_SS_2EBP_ESP: return cpu.ss.getWord((cpu.ebp << 1) + cpu.esp); //2227;
	case M_SS_2EBP_ID: return cpu.ss.getWord((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_SS_2EBP_ESI: return cpu.ss.getWord((cpu.ebp << 1) + cpu.esi); //2229;
	case M_SS_2EBP_EDI: return cpu.ss.getWord((cpu.ebp << 1) + cpu.edi); //2230;

	case M_SS_2ESI_EAX: return cpu.ss.getWord((cpu.esi << 1) + cpu.eax); //2231;
	case M_SS_2ESI_ECX: return cpu.ss.getWord((cpu.esi << 1) + cpu.ecx); //2232;
	case M_SS_2ESI_EDX: return cpu.ss.getWord((cpu.esi << 1) + cpu.edx); //2233;
	case M_SS_2ESI_EBX: return cpu.ss.getWord((cpu.esi << 1) + cpu.ebx); //2234;
	case M_SS_2ESI_ESP: return cpu.ss.getWord((cpu.esi << 1) + cpu.esp); //2235;
	case M_SS_2ESI_ID: return cpu.ss.getWord((cpu.esi << 1) + getMicrocode()); //2236;
	case M_SS_3ESI: return cpu.ss.getWord((cpu.esi << 1) + cpu.esi); //2237;
	case M_SS_2ESI_EDI: return cpu.ss.getWord((cpu.esi << 1) + cpu.edi); //2238;

	case M_SS_2EDI_EAX: return cpu.ss.getWord((cpu.edi << 1) + cpu.eax); //2239;
	case M_SS_2EDI_ECX: return cpu.ss.getWord((cpu.edi << 1) + cpu.ecx); //2240;
	case M_SS_2EDI_EDX: return cpu.ss.getWord((cpu.edi << 1) + cpu.edx); //2241;
	case M_SS_2EDI_EBX: return cpu.ss.getWord((cpu.edi << 1) + cpu.ebx); //2242;
	case M_SS_2EDI_ESP: return cpu.ss.getWord((cpu.edi << 1) + cpu.esp); //2243;
	case M_SS_2EDI_ID: return cpu.ss.getWord((cpu.edi << 1) + getMicrocode()); //2244;
	case M_SS_2EDI_ESI: return cpu.ss.getWord((cpu.edi << 1) + cpu.esi); //2245;
	case M_SS_3EDI: return cpu.ss.getWord((cpu.edi << 1) + cpu.edi); //2246;

	case M_SS_5EAX: return cpu.ss.getWord((cpu.eax << 2) + cpu.eax); //2247
	case M_SS_4EAX_ECX: return cpu.ss.getWord((cpu.eax << 2) + cpu.ecx); //2248
	case M_SS_4EAX_EDX: return cpu.ss.getWord((cpu.eax << 2) + cpu.edx); //2249
	case M_SS_4EAX_EBX: return cpu.ss.getWord((cpu.eax << 2) + cpu.ebx); //2250
	case M_SS_4EAX_ESP: return cpu.ss.getWord((cpu.eax << 2) + cpu.esp); //2251
	case M_SS_4EAX_ID: return cpu.ss.getWord((cpu.eax << 2) + getMicrocode()); //2252
	case M_SS_4EAX_ESI: return cpu.ss.getWord((cpu.eax << 2) + cpu.esi); //2253
	case M_SS_4EAX_EDI: return cpu.ss.getWord((cpu.eax << 2) + cpu.edi); //2254

	case M_SS_4ECX_EAX: return cpu.ss.getWord((cpu.ecx << 2) + cpu.eax); //2255
	case M_SS_5ECX: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ecx); //2256
	case M_SS_4ECX_EDX: return cpu.ss.getWord((cpu.ecx << 2) + cpu.edx); //2257
	case M_SS_4ECX_EBX: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ebx); //2258
	case M_SS_4ECX_ESP: return cpu.ss.getWord((cpu.ecx << 2) + cpu.esp); //2259
	case M_SS_4ECX_ID: return cpu.ss.getWord((cpu.ecx << 2) + getMicrocode()); //2260
	case M_SS_4ECX_ESI: return cpu.ss.getWord((cpu.ecx << 2) + cpu.esi); //2261
	case M_SS_4ECX_EDI: return cpu.ss.getWord((cpu.ecx << 2) + cpu.edi); //2262

	case M_SS_4EDX_EAX: return cpu.ss.getWord((cpu.edx << 2) + cpu.eax); //2263
	case M_SS_4EDX_ECX: return cpu.ss.getWord((cpu.edx << 2) + cpu.ecx); //2264
	case M_SS_5EDX: return cpu.ss.getWord((cpu.edx << 2) + cpu.edx); //2265
	case M_SS_4EDX_EBX: return cpu.ss.getWord((cpu.edx << 2) + cpu.ebx); //2266
	case M_SS_4EDX_ESP: return cpu.ss.getWord((cpu.edx << 2) + cpu.esp); //2267
	case M_SS_4EDX_ID: return cpu.ss.getWord((cpu.edx << 2) + getMicrocode()); //2268
	case M_SS_4EDX_ESI: return cpu.ss.getWord((cpu.edx << 2) + cpu.esi); //2269
	case M_SS_4EDX_EDI: return cpu.ss.getWord((cpu.edx << 2) + cpu.edi); //2270

	case M_SS_4EBX_EAX: return cpu.ss.getWord((cpu.ebx << 2) + cpu.eax); //2271
	case M_SS_4EBX_ECX: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ecx); //2272
	case M_SS_4EBX_EDX: return cpu.ss.getWord((cpu.ebx << 2) + cpu.edx); //2273
	case M_SS_5EBX: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ebx); //2274
	case M_SS_4EBX_ESP: return cpu.ss.getWord((cpu.ebx << 2) + cpu.esp); //2275
	case M_SS_4EBX_ID: return cpu.ss.getWord((cpu.ebx << 2) + getMicrocode()); //2276
	case M_SS_4EBX_ESI: return cpu.ss.getWord((cpu.ebx << 2) + cpu.esi); //2277
	case M_SS_4EBX_EDI: return cpu.ss.getWord((cpu.ebx << 2) + cpu.edi); //2278

	case M_SS_4EBP_EAX: return cpu.ss.getWord((cpu.ebp << 2) + cpu.eax); //2279
	case M_SS_4EBP_ECX: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ecx); //2280
	case M_SS_4EBP_EDX: return cpu.ss.getWord((cpu.ebp << 2) + cpu.edx); //2281
	case M_SS_4EBP_EBX: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ebx); //2282
	case M_SS_4EBP_ESP: return cpu.ss.getWord((cpu.ebp << 2) + cpu.esp); //2283
	case M_SS_4EBP_ID: return cpu.ss.getWord((cpu.ebp << 2) + getMicrocode()); //2284
	case M_SS_4EBP_ESI: return cpu.ss.getWord((cpu.ebp << 2) + cpu.esi); //2285
	case M_SS_4EBP_EDI: return cpu.ss.getWord((cpu.ebp << 2) + cpu.edi); //2286

	case M_SS_4ESI_EAX: return cpu.ss.getWord((cpu.esi << 2) + cpu.eax); //2287
	case M_SS_4ESI_ECX: return cpu.ss.getWord((cpu.esi << 2) + cpu.ecx); //2288
	case M_SS_4ESI_EDX: return cpu.ss.getWord((cpu.esi << 2) + cpu.edx); //2289
	case M_SS_4ESI_EBX: return cpu.ss.getWord((cpu.esi << 2) + cpu.ebx); //2290
	case M_SS_4ESI_ESP: return cpu.ss.getWord((cpu.esi << 2) + cpu.esp); //2291
	case M_SS_4ESI_ID: return cpu.ss.getWord((cpu.esi << 2) + getMicrocode()); //2292
	case M_SS_5ESI: return cpu.ss.getWord((cpu.esi << 2) + cpu.esi); //2293
	case M_SS_4ESI_EDI: return cpu.ss.getWord((cpu.esi << 2) + cpu.edi); //2294

	case M_SS_4EDI_EAX: return cpu.ss.getWord((cpu.edi << 2) + cpu.eax); //2295;
	case M_SS_4EDI_ECX: return cpu.ss.getWord((cpu.edi << 2) + cpu.ecx); //2296;
	case M_SS_4EDI_EDX: return cpu.ss.getWord((cpu.edi << 2) + cpu.edx); //2297;
	case M_SS_4EDI_EBX: return cpu.ss.getWord((cpu.edi << 2) + cpu.ebx); //2298;
	case M_SS_4EDI_ESP: return cpu.ss.getWord((cpu.edi << 2) + cpu.esp); //2299;
	case M_SS_4EDI_ID: return cpu.ss.getWord((cpu.edi << 2) + getMicrocode()); //2300;
	case M_SS_4EDI_ESI: return cpu.ss.getWord((cpu.edi << 2) + cpu.esi); //2301;
	case M_SS_5EDI: return cpu.ss.getWord((cpu.edi << 2) + cpu.edi); //2302;

	case M_SS_9EAX: return cpu.ss.getWord((cpu.eax << 3) + cpu.eax); //2303;
	case M_SS_8EAX_ECX: return cpu.ss.getWord((cpu.eax << 3) + cpu.ecx); //2304;
	case M_SS_8EAX_EDX: return cpu.ss.getWord((cpu.eax << 3) + cpu.edx); //2305;
	case M_SS_8EAX_EBX: return cpu.ss.getWord((cpu.eax << 3) + cpu.ebx); //2306;
	case M_SS_8EAX_ESP: return cpu.ss.getWord((cpu.eax << 3) + cpu.esp); //2307;
	case M_SS_8EAX_ID: return cpu.ss.getWord((cpu.eax << 3) + getMicrocode()); //2308;
	case M_SS_8EAX_ESI: return cpu.ss.getWord((cpu.eax << 3) + cpu.esi); //2309;
	case M_SS_8EAX_EDI: return cpu.ss.getWord((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_SS_8ECX_EAX: return cpu.ss.getWord((cpu.ecx << 3) + cpu.eax); //2311;
	case M_SS_9ECX: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_SS_8ECX_EDX: return cpu.ss.getWord((cpu.ecx << 3) + cpu.edx); //2313;
	case M_SS_8ECX_EBX: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_SS_8ECX_ESP: return cpu.ss.getWord((cpu.ecx << 3) + cpu.esp); //2315;
	case M_SS_8ECX_ID: return cpu.ss.getWord((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_SS_8ECX_ESI: return cpu.ss.getWord((cpu.ecx << 3) + cpu.esi); //2317;
	case M_SS_8ECX_EDI: return cpu.ss.getWord((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_SS_8EDX_EAX: return cpu.ss.getWord((cpu.edx << 3) + cpu.eax); //2319;
	case M_SS_8EDX_ECX: return cpu.ss.getWord((cpu.edx << 3) + cpu.ecx); //2320;
	case M_SS_9EDX: return cpu.ss.getWord((cpu.edx << 3) + cpu.edx); //2321;
	case M_SS_8EDX_EBX: return cpu.ss.getWord((cpu.edx << 3) + cpu.ebx); //2322;
	case M_SS_8EDX_ESP: return cpu.ss.getWord((cpu.edx << 3) + cpu.esp); //2323;
	case M_SS_8EDX_ID: return cpu.ss.getWord((cpu.edx << 3) + getMicrocode()); //2324;
	case M_SS_8EDX_ESI: return cpu.ss.getWord((cpu.edx << 3) + cpu.esi); //2325;
	case M_SS_8EDX_EDI: return cpu.ss.getWord((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_SS_8EBX_EAX: return cpu.ss.getWord((cpu.ebx << 3) + cpu.eax); //2327
	case M_SS_8EBX_ECX: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ecx); //2328
	case M_SS_8EBX_EDX: return cpu.ss.getWord((cpu.ebx << 3) + cpu.edx); //2329
	case M_SS_9EBX: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ebx); //2330
	case M_SS_8EBX_ESP: return cpu.ss.getWord((cpu.ebx << 3) + cpu.esp); //2331
	case M_SS_8EBX_ID: return cpu.ss.getWord((cpu.ebx << 3) + getMicrocode()); //2332
	case M_SS_8EBX_ESI: return cpu.ss.getWord((cpu.ebx << 3) + cpu.esi); //2333
	case M_SS_8EBX_EDI: return cpu.ss.getWord((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_SS_8EBP_EAX: return cpu.ss.getWord((cpu.ebp << 3) + cpu.eax); //2335;
	case M_SS_8EBP_ECX: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_SS_8EBP_EDX: return cpu.ss.getWord((cpu.ebp << 3) + cpu.edx); //2337;
	case M_SS_8EBP_EBX: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_SS_8EBP_ESP: return cpu.ss.getWord((cpu.ebp << 3) + cpu.esp); //2339;
	case M_SS_8EBP_ID: return cpu.ss.getWord((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_SS_8EBP_ESI: return cpu.ss.getWord((cpu.ebp << 3) + cpu.esi); //2341;
	case M_SS_8EBP_EDI: return cpu.ss.getWord((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_SS_8ESI_EAX: return cpu.ss.getWord((cpu.esi << 3) + cpu.eax); //2343;
	case M_SS_8ESI_ECX: return cpu.ss.getWord((cpu.esi << 3) + cpu.ecx); //2344;
	case M_SS_8ESI_EDX: return cpu.ss.getWord((cpu.esi << 3) + cpu.edx); //2345;
	case M_SS_8ESI_EBX: return cpu.ss.getWord((cpu.esi << 3) + cpu.ebx); //2346;
	case M_SS_8ESI_ESP: return cpu.ss.getWord((cpu.esi << 3) + cpu.esp); //2347;
	case M_SS_8ESI_ID: return cpu.ss.getWord((cpu.esi << 3) + getMicrocode()); //2348;
	case M_SS_9ESI: return cpu.ss.getWord((cpu.esi << 3) + cpu.esi); //2349;
	case M_SS_8ESI_EDI: return cpu.ss.getWord((cpu.esi << 3) + cpu.edi); //2350;
	    
	case M_SS_8EDI_EAX: return cpu.ss.getWord((cpu.edi << 3) + cpu.eax); //2351;
	case M_SS_8EDI_ECX: return cpu.ss.getWord((cpu.edi << 3) + cpu.ecx); //2352;
	case M_SS_8EDI_EDX: return cpu.ss.getWord((cpu.edi << 3) + cpu.edx); //2353;
	case M_SS_8EDI_EBX: return cpu.ss.getWord((cpu.edi << 3) + cpu.ebx); //2354;
	case M_SS_8EDI_ESP: return cpu.ss.getWord((cpu.edi << 3) + cpu.esp); //2355;
	case M_SS_8EDI_ID: return cpu.ss.getWord((cpu.edi << 3) + getMicrocode()); //2356;
	case M_SS_8EDI_ESI: return cpu.ss.getWord((cpu.edi << 3) + cpu.esi); //2357;
	case M_SS_9EDI: return cpu.ss.getWord((cpu.edi << 3) + cpu.edi); //2358;

	case M_SS_2EAX_IB: return cpu.ss.getWord(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_SS_EAX_ECX_IB: return cpu.ss.getWord(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_SS_EAX_EDX_IB: return cpu.ss.getWord(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_SS_EAX_EBX_IB: return cpu.ss.getWord(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_SS_EAX_ESP_IB: return cpu.ss.getWord(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_SS_EAX_EBP_IB: return cpu.ss.getWord(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_SS_EAX_ESI_IB: return cpu.ss.getWord(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_SS_EAX_EDI_IB: return cpu.ss.getWord(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_SS_2ECX_IB: return cpu.ss.getWord(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_SS_ECX_EDX_IB: return cpu.ss.getWord(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_SS_ECX_EBX_IB: return cpu.ss.getWord(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_SS_ECX_ESP_IB: return cpu.ss.getWord(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_SS_ECX_EBP_IB: return cpu.ss.getWord(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_SS_ECX_ESI_IB: return cpu.ss.getWord(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_SS_ECX_EDI_IB: return cpu.ss.getWord(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_SS_2EDX_IB: return cpu.ss.getWord(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_SS_EDX_EBX_IB: return cpu.ss.getWord(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_SS_EDX_ESP_IB: return cpu.ss.getWord(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_SS_EDX_EBP_IB: return cpu.ss.getWord(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_SS_EDX_ESI_IB: return cpu.ss.getWord(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_SS_EDX_EDI_IB: return cpu.ss.getWord(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_SS_2EBX_IB: return cpu.ss.getWord(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_SS_EBX_ESP_IB: return cpu.ss.getWord(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_SS_EBX_EBP_IB: return cpu.ss.getWord(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_SS_EBX_ESI_IB: return cpu.ss.getWord(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_SS_EBX_EDI_IB: return cpu.ss.getWord(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_SS_ESP_IB: return cpu.ss.getWord(cpu.esp + getMicrocode()); //2385

	case M_SS_EBP_ESP_IB: return cpu.ss.getWord(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_SS_2EBP_IB: return cpu.ss.getWord(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_SS_EBP_ESI_IB: return cpu.ss.getWord(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_SS_EBP_EDI_IB: return cpu.ss.getWord(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_SS_ESI_ESP_IB: return cpu.ss.getWord(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_SS_2ESI_IB: return cpu.ss.getWord(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_SS_ESI_EDI_IB: return cpu.ss.getWord(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_SS_EDI_ESP_IB: return cpu.ss.getWord(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_SS_2EDI_IB: return cpu.ss.getWord(cpu.edi + cpu.edi + getMicrocode()); //2394;
	    
	case M_SS_3EAX_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_SS_2EAX_ECX_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_SS_2EAX_EDX_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_SS_2EAX_EBX_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_SS_2EAX_ESP_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_SS_2EAX_EBP_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_SS_2EAX_ESI_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_SS_2EAX_EDI_IB: return cpu.ss.getWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_SS_2ECX_EAX_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_SS_3ECX_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_SS_2ECX_EDX_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_SS_2ECX_EBX_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_SS_2ECX_ESP_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_SS_2ECX_EBP_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_SS_2ECX_ESI_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_SS_2ECX_EDI_IB: return cpu.ss.getWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_SS_2EDX_EAX_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_SS_2EDX_ECX_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_SS_3EDX_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_SS_2EDX_EBX_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_SS_2EDX_ESP_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_SS_2EDX_EBP_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_SS_2EDX_ESI_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_SS_2EDX_EDI_IB: return cpu.ss.getWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;

	case M_SS_2EBX_EAX_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_SS_2EBX_ECX_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_SS_2EBX_EDX_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_SS_3EBX_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_SS_2EBX_ESP_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_SS_2EBX_EBP_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_SS_2EBX_ESI_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_SS_2EBX_EDI_IB: return cpu.ss.getWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_SS_2EBP_EAX_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_SS_2EBP_ECX_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_SS_2EBP_EDX_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_SS_2EBP_EBX_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_SS_2EBP_ESP_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_SS_3EBP_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_SS_2EBP_ESI_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_SS_2EBP_EDI_IB: return cpu.ss.getWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_SS_2ESI_EAX_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_SS_2ESI_ECX_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_SS_2ESI_EDX_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_SS_2ESI_EBX_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_SS_2ESI_ESP_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_SS_2ESI_EBP_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_SS_3ESI_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_SS_2ESI_EDI_IB: return cpu.ss.getWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_SS_2EDI_EAX_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_SS_2EDI_ECX_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_SS_2EDI_EDX_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_SS_2EDI_EBX_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_SS_2EDI_ESP_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_SS_2EDI_EBP_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_SS_2EDI_ESI_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_SS_3EDI_IB: return cpu.ss.getWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_SS_5EAX_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_SS_4EAX_ECX_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_SS_4EAX_EDX_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_SS_4EAX_EBX_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_SS_4EAX_ESP_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_SS_4EAX_EBP_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_SS_4EAX_ESI_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_SS_4EAX_EDI_IB: return cpu.ss.getWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_SS_4ECX_EAX_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_SS_5ECX_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_SS_4ECX_EDX_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_SS_4ECX_EBX_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_SS_4ECX_ESP_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_SS_4ECX_EBP_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_SS_4ECX_ESI_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_SS_4ECX_EDI_IB: return cpu.ss.getWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_SS_4EDX_EAX_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_SS_4EDX_ECX_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_SS_5EDX_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_SS_4EDX_EBX_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_SS_4EDX_ESP_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_SS_4EDX_EBP_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_SS_4EDX_ESI_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_SS_4EDX_EDI_IB: return cpu.ss.getWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_SS_4EBX_EAX_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_SS_4EBX_ECX_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_SS_4EBX_EDX_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_SS_5EBX_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_SS_4EBX_ESP_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_SS_4EBX_EBP_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_SS_4EBX_ESI_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_SS_4EBX_EDI_IB: return cpu.ss.getWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_SS_4EBP_EAX_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_SS_4EBP_ECX_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_SS_4EBP_EDX_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_SS_4EBP_EBX_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_SS_4EBP_ESP_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_SS_5EBP_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_SS_4EBP_ESI_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_SS_4EBP_EDI_IB: return cpu.ss.getWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_SS_4ESI_EAX_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_SS_4ESI_ECX_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_SS_4ESI_EDX_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_SS_4ESI_EBX_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_SS_4ESI_ESP_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_SS_4ESI_EBP_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_SS_5ESI_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_SS_4ESI_EDI_IB: return cpu.ss.getWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_SS_4EDI_EAX_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_SS_4EDI_ECX_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_SS_4EDI_EDX_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_SS_4EDI_EBX_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_SS_4EDI_ESP_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_SS_4EDI_EBP_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_SS_4EDI_ESI_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_SS_5EDI_IB: return cpu.ss.getWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_SS_9EAX_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_SS_8EAX_ECX_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_SS_8EAX_EDX_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_SS_8EAX_EBX_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_SS_8EAX_ESP_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_SS_8EAX_EBP_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_SS_8EAX_ESI_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_SS_8EAX_EDI_IB: return cpu.ss.getWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_SS_8ECX_EAX_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_SS_9ECX_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_SS_8ECX_EDX_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_SS_8ECX_EBX_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_SS_8ECX_ESP_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_SS_8ECX_EBP_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_SS_8ECX_ESI_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_SS_8ECX_EDI_IB: return cpu.ss.getWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_SS_8EDX_EAX_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_SS_8EDX_ECX_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_SS_9EDX_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_SS_8EDX_EBX_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_SS_8EDX_ESP_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_SS_8EDX_EBP_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_SS_8EDX_ESI_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_SS_8EDX_EDI_IB: return cpu.ss.getWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_SS_8EBX_EAX_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_SS_8EBX_ECX_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_SS_8EBX_EDX_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_SS_9EBX_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_SS_8EBX_ESP_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_SS_8EBX_EBP_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_SS_8EBX_ESI_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_SS_8EBX_EDI_IB: return cpu.ss.getWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_SS_8EBP_EAX_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_SS_8EBP_ECX_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_SS_8EBP_EDX_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_SS_8EBP_EBX_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_SS_8EBP_ESP_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_SS_9EBP_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_SS_8EBP_ESI_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_SS_8EBP_EDI_IB: return cpu.ss.getWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_SS_8ESI_EAX_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_SS_8ESI_ECX_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_SS_8ESI_EDX_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_SS_8ESI_EBX_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_SS_8ESI_ESP_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_SS_8ESI_EBP_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_SS_9ESI_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_SS_8ESI_EDI_IB: return cpu.ss.getWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_SS_8EDI_EAX_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_SS_8EDI_ECX_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_SS_8EDI_EDX_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_SS_8EDI_EBX_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_SS_8EDI_ESP_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_SS_8EDI_EBP_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_SS_8EDI_ESI_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_SS_9EDI_IB: return cpu.ss.getWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_SS_EAX_ECX_ID: return cpu.ss.getWord(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_SS_EAX_EDX_ID: return cpu.ss.getWord(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_SS_EAX_EBX_ID: return cpu.ss.getWord(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_SS_EAX_ESP_ID: return cpu.ss.getWord(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_SS_EAX_EBP_ID: return cpu.ss.getWord(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_SS_EAX_ESI_ID: return cpu.ss.getWord(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_SS_EAX_EDI_ID: return cpu.ss.getWord(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_SS_ECX_EDX_ID: return cpu.ss.getWord(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_SS_ECX_EBX_ID: return cpu.ss.getWord(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_SS_ECX_ESP_ID: return cpu.ss.getWord(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_SS_ECX_EBP_ID: return cpu.ss.getWord(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_SS_ECX_ESI_ID: return cpu.ss.getWord(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_SS_ECX_EDI_ID: return cpu.ss.getWord(cpu.ecx + cpu.edi + getMicrocode()); //2575;
	    
	case M_SS_EDX_EBX_ID: return cpu.ss.getWord(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_SS_EDX_ESP_ID: return cpu.ss.getWord(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_SS_EDX_EBP_ID: return cpu.ss.getWord(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_SS_EDX_ESI_ID: return cpu.ss.getWord(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_SS_EDX_EDI_ID: return cpu.ss.getWord(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_SS_EBX_ESP_ID: return cpu.ss.getWord(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_SS_EBX_EBP_ID: return cpu.ss.getWord(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_SS_EBX_ESI_ID: return cpu.ss.getWord(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_SS_EBX_EDI_ID: return cpu.ss.getWord(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_SS_ESP_ID: return cpu.ss.getWord(cpu.esp + getMicrocode()); //2585

	case M_SS_EBP_ESP_ID: return cpu.ss.getWord(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_SS_EBP_ESI_ID: return cpu.ss.getWord(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_SS_EBP_EDI_ID: return cpu.ss.getWord(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_SS_ESI_ESP_ID: return cpu.ss.getWord(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_SS_ESI_EDI_ID: return cpu.ss.getWord(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_SS_EDI_ESP_ID: return cpu.ss.getWord(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_SS_3EAX_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_SS_2EAX_ECX_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_SS_2EAX_EDX_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_SS_2EAX_EBX_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_SS_2EAX_ESP_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_SS_2EAX_EBP_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_SS_2EAX_ESI_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_SS_2EAX_EDI_ID: return cpu.ss.getWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_SS_2ECX_EAX_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_SS_3ECX_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_SS_2ECX_EDX_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_SS_2ECX_EBX_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_SS_2ECX_ESP_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_SS_2ECX_EBP_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_SS_2ECX_ESI_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_SS_2ECX_EDI_ID: return cpu.ss.getWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_SS_2EDX_EAX_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_SS_2EDX_ECX_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_SS_3EDX_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_SS_2EDX_EBX_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_SS_2EDX_ESP_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_SS_2EDX_EBP_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_SS_2EDX_ESI_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_SS_2EDX_EDI_ID: return cpu.ss.getWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_SS_2EBX_EAX_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_SS_2EBX_ECX_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_SS_2EBX_EDX_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_SS_3EBX_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_SS_2EBX_ESP_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_SS_2EBX_EBP_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_SS_2EBX_ESI_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_SS_2EBX_EDI_ID: return cpu.ss.getWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_SS_2EBP_EAX_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_SS_2EBP_ECX_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_SS_2EBP_EDX_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_SS_2EBP_EBX_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_SS_2EBP_ESP_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_SS_3EBP_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_SS_2EBP_ESI_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_SS_2EBP_EDI_ID: return cpu.ss.getWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_SS_2ESI_EAX_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_SS_2ESI_ECX_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_SS_2ESI_EDX_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_SS_2ESI_EBX_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_SS_2ESI_ESP_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_SS_2ESI_EBP_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_SS_3ESI_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_SS_2ESI_EDI_ID: return cpu.ss.getWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_SS_2EDI_EAX_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_SS_2EDI_ECX_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_SS_2EDI_EDX_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_SS_2EDI_EBX_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_SS_2EDI_ESP_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_SS_2EDI_EBP_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_SS_2EDI_ESI_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_SS_3EDI_ID: return cpu.ss.getWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_SS_5EAX_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_SS_4EAX_ECX_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_SS_4EAX_EDX_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_SS_4EAX_EBX_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_SS_4EAX_ESP_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_SS_4EAX_EBP_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_SS_4EAX_ESI_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_SS_4EAX_EDI_ID: return cpu.ss.getWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_SS_4ECX_EAX_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_SS_5ECX_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_SS_4ECX_EDX_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_SS_4ECX_EBX_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_SS_4ECX_ESP_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_SS_4ECX_EBP_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_SS_4ECX_ESI_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_SS_4ECX_EDI_ID: return cpu.ss.getWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_SS_4EDX_EAX_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_SS_4EDX_ECX_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_SS_5EDX_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_SS_4EDX_EBX_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_SS_4EDX_ESP_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_SS_4EDX_EBP_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_SS_4EDX_ESI_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_SS_4EDX_EDI_ID: return cpu.ss.getWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_SS_4EBX_EAX_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_SS_4EBX_ECX_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_SS_4EBX_EDX_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_SS_5EBX_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_SS_4EBX_ESP_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_SS_4EBX_EBP_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_SS_4EBX_ESI_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_SS_4EBX_EDI_ID: return cpu.ss.getWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_SS_4EBP_EAX_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_SS_4EBP_ECX_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_SS_4EBP_EDX_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_SS_4EBP_EBX_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_SS_4EBP_ESP_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_SS_5EBP_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_SS_4EBP_ESI_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_SS_4EBP_EDI_ID: return cpu.ss.getWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_SS_4ESI_EAX_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_SS_4ESI_ECX_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_SS_4ESI_EDX_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_SS_4ESI_EBX_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_SS_4ESI_ESP_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_SS_4ESI_EBP_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_SS_5ESI_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_SS_4ESI_EDI_ID: return cpu.ss.getWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_SS_4EDI_EAX_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_SS_4EDI_ECX_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_SS_4EDI_EDX_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_SS_4EDI_EBX_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_SS_4EDI_ESP_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_SS_4EDI_EBP_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_SS_4EDI_ESI_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_SS_5EDI_ID: return cpu.ss.getWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_SS_9EAX_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_SS_8EAX_ECX_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_SS_8EAX_EDX_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_SS_8EAX_EBX_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_SS_8EAX_ESP_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_SS_8EAX_EBP_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_SS_8EAX_ESI_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_SS_8EAX_EDI_ID: return cpu.ss.getWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_SS_8ECX_EAX_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_SS_9ECX_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_SS_8ECX_EDX_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_SS_8ECX_EBX_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_SS_8ECX_ESP_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_SS_8ECX_EBP_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_SS_8ECX_ESI_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_SS_8ECX_EDI_ID: return cpu.ss.getWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_SS_8EDX_EAX_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_SS_8EDX_ECX_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_SS_9EDX_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_SS_8EDX_EBX_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_SS_8EDX_ESP_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_SS_8EDX_EBP_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_SS_8EDX_ESI_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_SS_8EDX_EDI_ID: return cpu.ss.getWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_SS_8EBX_EAX_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_SS_8EBX_ECX_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_SS_8EBX_EDX_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_SS_9EBX_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_SS_8EBX_ESP_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_SS_8EBX_EBP_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_SS_8EBX_ESI_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_SS_8EBX_EDI_ID: return cpu.ss.getWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_SS_8EBP_EAX_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_SS_8EBP_ECX_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_SS_8EBP_EDX_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_SS_8EBP_EBX_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_SS_8EBP_ESP_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_SS_9EBP_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_SS_8EBP_ESI_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_SS_8EBP_EDI_ID: return cpu.ss.getWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_SS_8ESI_EAX_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_SS_8ESI_ECX_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_SS_8ESI_EDX_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_SS_8ESI_EBX_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_SS_8ESI_ESP_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_SS_8ESI_EBP_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_SS_9ESI_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_SS_8ESI_EDI_ID: return cpu.ss.getWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_SS_8EDI_EAX_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_SS_8EDI_ECX_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_SS_8EDI_EDX_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_SS_8EDI_EBX_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_SS_8EDI_ESP_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_SS_8EDI_EBP_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_SS_8EDI_ESI_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_SS_9EDI_ID: return cpu.ss.getWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;
	}

	System.err.println("Unimplemented GetShort Operand: " + microcode);
	throw unimplemented;
    }


    protected final int getIntOperand()
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case EAX: return cpu.eax;
	case ECX: return cpu.ecx;
	case EDX: return cpu.edx;
	case EBX: return cpu.ebx;
	case ESP: return cpu.esp;
	case EBP: return cpu.ebp;
	case ESI: return cpu.esi;
	case EDI: return cpu.edi;
	    
	case CR0: return cpu.getCR0();
	case CR2: return cpu.getCR2();
	case CR3: return cpu.getCR3();
	case CR4: return cpu.getCR4();

	case DR0: return cpu.getDR0();
	case DR1: return cpu.getDR1();
	case DR2: return cpu.getDR2();
	case DR3: return cpu.getDR3();
	case DR6: return cpu.getDR6();
	case DR7: return cpu.getDR7();
	    
	case IWIB: return getMicrocode();
	case IB:  return getMicrocode();
	case IW:  return getMicrocode();
	case ID:  return getMicrocode();
	    
	case IA32: return getMicrocode();
	    
	case ES: return 0xffff & cpu.es.getSelector();
	case CS: return 0xffff & cpu.cs.getSelector();
	case SS: return 0xffff & cpu.ss.getSelector();
	case DS: return 0xffff & cpu.ds.getSelector();
	case FS: return 0xffff & cpu.fs.getSelector();
	case GS: return 0xffff & cpu.gs.getSelector();
	}

	return getIntMemoryOperand(microcode);
    }

    private final int getIntMemoryOperand(int microcode)
    {
	switch (microcode) {
	case M_ES_BX_SI: return cpu.es.getDoubleWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_ES_BX_DI: return cpu.es.getDoubleWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_ES_BP_SI: return cpu.es.getDoubleWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_ES_BP_DI: return cpu.es.getDoubleWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_ES_SI: return cpu.es.getDoubleWord(cpu.esi & 0xffff);
	case M_ES_DI: return cpu.es.getDoubleWord(cpu.edi & 0xffff);
	case M_ES_IW: return cpu.es.getDoubleWord(getMicrocode() & 0xffff);
	case M_ES_BX: return cpu.es.getDoubleWord(cpu.ebx & 0xffff);
	case M_ES_BX_SI_IB: return cpu.es.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IB: return cpu.es.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IB: return cpu.es.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IB: return cpu.es.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IB: return cpu.es.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IB: return cpu.es.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IB: return cpu.es.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IB: return cpu.es.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_ES_BX_SI_IW: return cpu.es.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IW: return cpu.es.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IW: return cpu.es.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IW: return cpu.es.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IW: return cpu.es.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IW: return cpu.es.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IW: return cpu.es.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IW: return cpu.es.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_CS_BX_SI: return cpu.cs.getDoubleWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_CS_BX_DI: return cpu.cs.getDoubleWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_CS_BP_SI: return cpu.cs.getDoubleWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_CS_BP_DI: return cpu.cs.getDoubleWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_CS_SI: return cpu.cs.getDoubleWord(cpu.esi & 0xffff);
	case M_CS_DI: return cpu.cs.getDoubleWord(cpu.edi & 0xffff);
	case M_CS_IW: return cpu.cs.getDoubleWord(getMicrocode() & 0xffff);
	case M_CS_BX: return cpu.cs.getDoubleWord(cpu.ebx & 0xffff);
	case M_CS_BX_SI_IB: return cpu.cs.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IB: return cpu.cs.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IB: return cpu.cs.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IB: return cpu.cs.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IB: return cpu.cs.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IB: return cpu.cs.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IB: return cpu.cs.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IB: return cpu.cs.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_CS_BX_SI_IW: return cpu.cs.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IW: return cpu.cs.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IW: return cpu.cs.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IW: return cpu.cs.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IW: return cpu.cs.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IW: return cpu.cs.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IW: return cpu.cs.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IW: return cpu.cs.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_SS_BX_SI: return cpu.ss.getDoubleWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_SS_BX_DI: return cpu.ss.getDoubleWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_SS_BP_SI: return cpu.ss.getDoubleWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_SS_BP_DI: return cpu.ss.getDoubleWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_SS_SI: return cpu.ss.getDoubleWord(cpu.esi & 0xffff);
	case M_SS_DI: return cpu.ss.getDoubleWord(cpu.edi & 0xffff);
	case M_SS_IW: return cpu.ss.getDoubleWord(getMicrocode() & 0xffff);
	case M_SS_BX: return cpu.ss.getDoubleWord(cpu.ebx & 0xffff);
	case M_SS_BX_SI_IB: return cpu.ss.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IB: return cpu.ss.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IB: return cpu.ss.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IB: return cpu.ss.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IB: return cpu.ss.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IB: return cpu.ss.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IB: return cpu.ss.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IB: return cpu.ss.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_SS_BX_SI_IW: return cpu.ss.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IW: return cpu.ss.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IW: return cpu.ss.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IW: return cpu.ss.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IW: return cpu.ss.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IW: return cpu.ss.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IW: return cpu.ss.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IW: return cpu.ss.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_DS_BX_SI: return cpu.ds.getDoubleWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_DS_BX_DI: return cpu.ds.getDoubleWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_DS_BP_SI: return cpu.ds.getDoubleWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_DS_BP_DI: return cpu.ds.getDoubleWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_DS_SI: return cpu.ds.getDoubleWord(cpu.esi & 0xffff);
	case M_DS_DI: return cpu.ds.getDoubleWord(cpu.edi & 0xffff);
	case M_DS_IW: return cpu.ds.getDoubleWord(getMicrocode() & 0xffff);
	case M_DS_BX: return cpu.ds.getDoubleWord(cpu.ebx & 0xffff);
	case M_DS_BX_SI_IB: return cpu.ds.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IB: return cpu.ds.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IB: return cpu.ds.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IB: return cpu.ds.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IB: return cpu.ds.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IB: return cpu.ds.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IB: return cpu.ds.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IB: return cpu.ds.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_DS_BX_SI_IW: return cpu.ds.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IW: return cpu.ds.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IW: return cpu.ds.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IW: return cpu.ds.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IW: return cpu.ds.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IW: return cpu.ds.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IW: return cpu.ds.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IW: return cpu.ds.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_FS_BX_SI: return cpu.fs.getDoubleWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_FS_BX_DI: return cpu.fs.getDoubleWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_FS_BP_SI: return cpu.fs.getDoubleWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_FS_BP_DI: return cpu.fs.getDoubleWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_FS_SI: return cpu.fs.getDoubleWord(cpu.esi & 0xffff);
	case M_FS_DI: return cpu.fs.getDoubleWord(cpu.edi & 0xffff);
	case M_FS_IW: return cpu.fs.getDoubleWord(getMicrocode() & 0xffff);
	case M_FS_BX: return cpu.fs.getDoubleWord(cpu.ebx & 0xffff);
	case M_FS_BX_SI_IB: return cpu.fs.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IB: return cpu.fs.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IB: return cpu.fs.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IB: return cpu.fs.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IB: return cpu.fs.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IB: return cpu.fs.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IB: return cpu.fs.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IB: return cpu.fs.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_FS_BX_SI_IW: return cpu.fs.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IW: return cpu.fs.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IW: return cpu.fs.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IW: return cpu.fs.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IW: return cpu.fs.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IW: return cpu.fs.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IW: return cpu.fs.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IW: return cpu.fs.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_GS_BX_SI: return cpu.gs.getDoubleWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_GS_BX_DI: return cpu.gs.getDoubleWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_GS_BP_SI: return cpu.gs.getDoubleWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_GS_BP_DI: return cpu.gs.getDoubleWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_GS_SI: return cpu.gs.getDoubleWord(cpu.esi & 0xffff);
	case M_GS_DI: return cpu.gs.getDoubleWord(cpu.edi & 0xffff);
	case M_GS_IW: return cpu.gs.getDoubleWord(getMicrocode() & 0xffff);
	case M_GS_BX: return cpu.gs.getDoubleWord(cpu.ebx & 0xffff);
	case M_GS_BX_SI_IB: return cpu.gs.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IB: return cpu.gs.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IB: return cpu.gs.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IB: return cpu.gs.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IB: return cpu.gs.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IB: return cpu.gs.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IB: return cpu.gs.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IB: return cpu.gs.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_GS_BX_SI_IW: return cpu.gs.getDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IW: return cpu.gs.getDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IW: return cpu.gs.getDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IW: return cpu.gs.getDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IW: return cpu.gs.getDoubleWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IW: return cpu.gs.getDoubleWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IW: return cpu.gs.getDoubleWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IW: return cpu.gs.getDoubleWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_ES_EAX: return cpu.es.getDoubleWord(cpu.eax); //56
	case M_ES_ECX: return cpu.es.getDoubleWord(cpu.ecx); //57
	case M_ES_EDX: return cpu.es.getDoubleWord(cpu.edx); //58
	case M_ES_EBX: return cpu.es.getDoubleWord(cpu.ebx); //59
	case M_ES_ID: return cpu.es.getDoubleWord(getMicrocode()); //60
	case M_ES_ESI: return cpu.es.getDoubleWord(cpu.esi); //61
	case M_ES_EDI: return cpu.es.getDoubleWord(cpu.edi); //62
	case M_ES_EAX_IB: return cpu.es.getDoubleWord(cpu.eax + getMicrocode()); //63
	case M_ES_ECX_IB: return cpu.es.getDoubleWord(cpu.ecx + getMicrocode()); //64
	case M_ES_EDX_IB: return cpu.es.getDoubleWord(cpu.edx + getMicrocode()); //65
	case M_ES_EBX_IB: return cpu.es.getDoubleWord(cpu.ebx + getMicrocode()); //66
	case M_ES_EBP_IB: return cpu.es.getDoubleWord(cpu.ebp + getMicrocode()); //67
	case M_ES_ESI_IB: return cpu.es.getDoubleWord(cpu.esi + getMicrocode()); //68
	case M_ES_EDI_IB: return cpu.es.getDoubleWord(cpu.edi + getMicrocode()); //69
	case M_ES_EAX_ID: return cpu.es.getDoubleWord(cpu.eax + getMicrocode()); //70
	case M_ES_ECX_ID: return cpu.es.getDoubleWord(cpu.ecx + getMicrocode()); //71
	case M_ES_EDX_ID: return cpu.es.getDoubleWord(cpu.edx + getMicrocode()); //72
	case M_ES_EBX_ID: return cpu.es.getDoubleWord(cpu.ebx + getMicrocode()); //73
	case M_ES_EBP_ID: return cpu.es.getDoubleWord(cpu.ebp + getMicrocode()); //74
	case M_ES_ESI_ID: return cpu.es.getDoubleWord(cpu.esi + getMicrocode()); //75
	case M_ES_EDI_ID: return cpu.es.getDoubleWord(cpu.edi + getMicrocode()); //76

	case M_GS_EAX: return cpu.gs.getDoubleWord(cpu.eax); //161
	case M_GS_ECX: return cpu.gs.getDoubleWord(cpu.ecx); //162
	case M_GS_EDX: return cpu.gs.getDoubleWord(cpu.edx); //163
	case M_GS_EBX: return cpu.gs.getDoubleWord(cpu.ebx); //164
	case M_GS_ID: return cpu.gs.getDoubleWord(getMicrocode()); //165
	case M_GS_ESI: return cpu.gs.getDoubleWord(cpu.esi); //166
	case M_GS_EDI: return cpu.gs.getDoubleWord(cpu.edi); //167
	case M_GS_EAX_IB: return cpu.gs.getDoubleWord(cpu.eax + getMicrocode()); //168
	case M_GS_ECX_IB: return cpu.gs.getDoubleWord(cpu.ecx + getMicrocode()); //169
	case M_GS_EDX_IB: return cpu.gs.getDoubleWord(cpu.edx + getMicrocode()); //170
	case M_GS_EBX_IB: return cpu.gs.getDoubleWord(cpu.ebx + getMicrocode()); //171
	case M_GS_EBP_IB: return cpu.gs.getDoubleWord(cpu.ebp + getMicrocode()); //172
	case M_GS_ESI_IB: return cpu.gs.getDoubleWord(cpu.esi + getMicrocode()); //173
	case M_GS_EDI_IB: return cpu.gs.getDoubleWord(cpu.edi + getMicrocode()); //174
	case M_GS_EAX_ID: return cpu.gs.getDoubleWord(cpu.eax + getMicrocode()); //175
	case M_GS_ECX_ID: return cpu.gs.getDoubleWord(cpu.ecx + getMicrocode()); //176
	case M_GS_EDX_ID: return cpu.gs.getDoubleWord(cpu.edx + getMicrocode()); //177
	case M_GS_EBX_ID: return cpu.gs.getDoubleWord(cpu.ebx + getMicrocode()); //178
	case M_GS_EBP_ID: return cpu.gs.getDoubleWord(cpu.ebp + getMicrocode()); //179
	case M_GS_ESI_ID: return cpu.gs.getDoubleWord(cpu.esi + getMicrocode()); //180
	case M_GS_EDI_ID: return cpu.gs.getDoubleWord(cpu.edi + getMicrocode()); //181

	case M_CS_EAX_ID: return cpu.cs.getDoubleWord(cpu.eax + getMicrocode());
	case M_CS_ECX_ID: return cpu.cs.getDoubleWord(cpu.ecx + getMicrocode());
	case M_CS_EDX_ID: return cpu.cs.getDoubleWord(cpu.edx + getMicrocode());
	case M_CS_EBX_ID: return cpu.cs.getDoubleWord(cpu.ebx + getMicrocode());
	case M_CS_EBP_ID: return cpu.cs.getDoubleWord(cpu.ebp + getMicrocode());
	case M_CS_ESI_ID: return cpu.cs.getDoubleWord(cpu.esi + getMicrocode());
	case M_CS_EDI_ID: return cpu.cs.getDoubleWord(cpu.edi + getMicrocode());

	case M_SS_ID: return cpu.ss.getDoubleWord(getMicrocode());

	case M_SS_EAX_IB: return cpu.ss.getDoubleWord(cpu.eax + getMicrocode()); //90
	case M_SS_ECX_IB: return cpu.ss.getDoubleWord(cpu.ecx + getMicrocode()); //91
	case M_SS_EDX_IB: return cpu.ss.getDoubleWord(cpu.edx + getMicrocode()); //92
	case M_SS_EBX_IB: return cpu.ss.getDoubleWord(cpu.ebx + getMicrocode()); //93
	case M_SS_EBP_IB: return cpu.ss.getDoubleWord(cpu.ebp + getMicrocode()); //94
	case M_SS_ESI_IB: return cpu.ss.getDoubleWord(cpu.esi + getMicrocode()); //95
	case M_SS_EDI_IB: return cpu.ss.getDoubleWord(cpu.edi + getMicrocode()); //96

	case M_SS_EAX_ID: return cpu.ss.getDoubleWord(cpu.eax + getMicrocode()); //97
	case M_SS_ECX_ID: return cpu.ss.getDoubleWord(cpu.ecx + getMicrocode()); //98
	case M_SS_EDX_ID: return cpu.ss.getDoubleWord(cpu.edx + getMicrocode()); //99
	case M_SS_EBX_ID: return cpu.ss.getDoubleWord(cpu.ebx + getMicrocode()); //100
	case M_SS_EBP_ID: return cpu.ss.getDoubleWord(cpu.ebp + getMicrocode()); //101
	case M_SS_ESI_ID: return cpu.ss.getDoubleWord(cpu.esi + getMicrocode()); //102
	case M_SS_EDI_ID: return cpu.ss.getDoubleWord(cpu.edi + getMicrocode()); //103

	case M_DS_EAX: return cpu.ds.getDoubleWord(cpu.eax); //104
	case M_DS_ECX: return cpu.ds.getDoubleWord(cpu.ecx); //105
	case M_DS_EDX: return cpu.ds.getDoubleWord(cpu.edx); //106
	case M_DS_EBX: return cpu.ds.getDoubleWord(cpu.ebx); //107
	case M_DS_ID: return cpu.ds.getDoubleWord(getMicrocode()); //108
	case M_DS_ESI: return cpu.ds.getDoubleWord(cpu.esi); //109
	case M_DS_EDI: return cpu.ds.getDoubleWord(cpu.edi); //110

	case M_DS_EAX_IB: return cpu.ds.getDoubleWord(cpu.eax + getMicrocode()); //111
	case M_DS_ECX_IB: return cpu.ds.getDoubleWord(cpu.ecx + getMicrocode()); //112
	case M_DS_EDX_IB: return cpu.ds.getDoubleWord(cpu.edx + getMicrocode()); //113
	case M_DS_EBX_IB: return cpu.ds.getDoubleWord(cpu.ebx + getMicrocode()); //114
	case M_DS_EBP_IB: return cpu.ds.getDoubleWord(cpu.ebp + getMicrocode()); //115 
	case M_DS_ESI_IB: return cpu.ds.getDoubleWord(cpu.esi + getMicrocode()); //116
	case M_DS_EDI_IB: return cpu.ds.getDoubleWord(cpu.edi + getMicrocode()); //117

	case M_DS_EAX_ID: return cpu.ds.getDoubleWord(cpu.eax + getMicrocode()); //118
	case M_DS_ECX_ID: return cpu.ds.getDoubleWord(cpu.ecx + getMicrocode()); //119
	case M_DS_EDX_ID: return cpu.ds.getDoubleWord(cpu.edx + getMicrocode()); //120
	case M_DS_EBX_ID: return cpu.ds.getDoubleWord(cpu.ebx + getMicrocode()); //121
	case M_DS_EBP_ID: return cpu.ds.getDoubleWord(cpu.ebp + getMicrocode()); //122
	case M_DS_ESI_ID: return cpu.ds.getDoubleWord(cpu.esi + getMicrocode()); //123
	case M_DS_EDI_ID: return cpu.ds.getDoubleWord(cpu.edi + getMicrocode()); //124

	case M_FS_EAX: return cpu.fs.getDoubleWord(cpu.eax); //104
	case M_FS_ECX: return cpu.fs.getDoubleWord(cpu.ecx); //105
	case M_FS_EDX: return cpu.fs.getDoubleWord(cpu.edx); //106
	case M_FS_EBX: return cpu.fs.getDoubleWord(cpu.ebx); //107
	case M_FS_ID: return cpu.fs.getDoubleWord(getMicrocode()); //108
	case M_FS_ESI: return cpu.fs.getDoubleWord(cpu.esi); //109
	case M_FS_EDI: return cpu.fs.getDoubleWord(cpu.edi); //110

	case M_FS_EAX_IB: return cpu.fs.getDoubleWord(cpu.eax + getMicrocode());
	case M_FS_ECX_IB: return cpu.fs.getDoubleWord(cpu.ecx + getMicrocode());
	case M_FS_EDX_IB: return cpu.fs.getDoubleWord(cpu.edx + getMicrocode());
	case M_FS_EBX_IB: return cpu.fs.getDoubleWord(cpu.ebx + getMicrocode()); 
	case M_FS_EBP_IB: return cpu.fs.getDoubleWord(cpu.ebp + getMicrocode());
	case M_FS_ESI_IB: return cpu.fs.getDoubleWord(cpu.esi + getMicrocode());
	case M_FS_EDI_IB: return cpu.fs.getDoubleWord(cpu.edi + getMicrocode());

	case M_FS_EAX_ID: return cpu.fs.getDoubleWord(cpu.eax + getMicrocode());
	case M_FS_ECX_ID: return cpu.fs.getDoubleWord(cpu.ecx + getMicrocode());
	case M_FS_EDX_ID: return cpu.fs.getDoubleWord(cpu.edx + getMicrocode());
	case M_FS_EBX_ID: return cpu.fs.getDoubleWord(cpu.ebx + getMicrocode());
	case M_FS_EBP_ID: return cpu.fs.getDoubleWord(cpu.ebp + getMicrocode());
	case M_FS_ESI_ID: return cpu.fs.getDoubleWord(cpu.esi + getMicrocode());
	case M_FS_EDI_ID: return cpu.fs.getDoubleWord(cpu.edi + getMicrocode());

	case M_DS_2EAX: return cpu.ds.getDoubleWord(cpu.eax + cpu.eax); //2156
	case M_DS_EAX_ECX: return cpu.ds.getDoubleWord(cpu.eax + cpu.ecx); //2157
	case M_DS_EAX_EDX: return cpu.ds.getDoubleWord(cpu.eax + cpu.edx); //2158
	case M_DS_EAX_EBX: return cpu.ds.getDoubleWord(cpu.eax + cpu.ebx); //2159
	case M_DS_EAX_ESP: return cpu.ds.getDoubleWord(cpu.eax + cpu.esp); //2160
	case M_DS_EAX_EBP: return cpu.ds.getDoubleWord(cpu.eax + cpu.ebp); //2161
	case M_DS_EAX_ESI: return cpu.ds.getDoubleWord(cpu.eax + cpu.esi); //2162
	case M_DS_EAX_EDI: return cpu.ds.getDoubleWord(cpu.eax + cpu.edi); //2163

	case M_DS_2ECX: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ecx); //2164
	case M_DS_ECX_EDX: return cpu.ds.getDoubleWord(cpu.ecx + cpu.edx); //2165
	case M_DS_ECX_EBX: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ebx); //2166
	case M_DS_ECX_ESP: return cpu.ds.getDoubleWord(cpu.ecx + cpu.esp); //2167
	case M_DS_ECX_EBP: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ebp); //2168
	case M_DS_ECX_ESI: return cpu.ds.getDoubleWord(cpu.ecx + cpu.esi); //2169
	case M_DS_ECX_EDI: return cpu.ds.getDoubleWord(cpu.ecx + cpu.edi); //2170

	case M_DS_2EDX: return cpu.ds.getDoubleWord(cpu.edx + cpu.edx); //2171
	case M_DS_EDX_EBX: return cpu.ds.getDoubleWord(cpu.edx + cpu.ebx); //2172
	case M_DS_EDX_ESP: return cpu.ds.getDoubleWord(cpu.edx + cpu.esp); //2173
	case M_DS_EDX_EBP: return cpu.ds.getDoubleWord(cpu.edx + cpu.ebp); //2174
	case M_DS_EDX_ESI: return cpu.ds.getDoubleWord(cpu.edx + cpu.esi); //2175	
	case M_DS_EDX_EDI: return cpu.ds.getDoubleWord(cpu.edx + cpu.edi); //2176

	case M_DS_2EBX: return cpu.ds.getDoubleWord(cpu.ebx + cpu.ebx); //2177;
	case M_DS_EBX_ESP: return cpu.ds.getDoubleWord(cpu.ebx + cpu.esp); //2178
	case M_DS_EBX_EBP: return cpu.ds.getDoubleWord(cpu.ebx + cpu.ebp); //2179
	case M_DS_EBX_ESI: return cpu.ds.getDoubleWord(cpu.ebx + cpu.esi); //2180
	case M_DS_EBX_EDI: return cpu.ds.getDoubleWord(cpu.ebx + cpu.edi); //2181
  
	case M_DS_ESP: return cpu.ds.getDoubleWord(cpu.esp); //2182

	case M_DS_EBP_ESP: return cpu.ds.getDoubleWord(cpu.ebp + cpu.esp); //2183;
	case M_DS_EBP_ESI: return cpu.ds.getDoubleWord(cpu.ebp + cpu.esi); //2184;
	case M_DS_EBP_EDI: return cpu.ds.getDoubleWord(cpu.ebp + cpu.edi); //2185;

	case M_DS_ESI_ESP: return cpu.ds.getDoubleWord(cpu.esi + cpu.esp); //2186;
	case M_DS_2ESI: return cpu.ds.getDoubleWord(cpu.esi + cpu.esi); //2187;
	case M_DS_ESI_EDI: return cpu.ds.getDoubleWord(cpu.esi + cpu.edi); //2188;

	case M_DS_EDI_ESP: return cpu.ds.getDoubleWord(cpu.edi + cpu.esp); //2189;
	case M_DS_2EDI: return cpu.ds.getDoubleWord(cpu.edi + cpu.edi); //2190;

	case M_DS_3EAX: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.eax); //2191;
	case M_DS_2EAX_ECX: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ecx); //2192;
	case M_DS_2EAX_EDX: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.edx); //2193;
	case M_DS_2EAX_EBX: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ebx); //2194;
	case M_DS_2EAX_ESP: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.esp); //2195;
	case M_DS_2EAX_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + getMicrocode()); //2196;
	case M_DS_2EAX_ESI: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.esi); //2197;
	case M_DS_2EAX_EDI: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.edi); //2198;

	case M_DS_2ECX_EAX: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.eax); //2199;
	case M_DS_3ECX: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_DS_2ECX_EDX: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.edx); //2201;
	case M_DS_2ECX_EBX: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_DS_2ECX_ESP: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.esp); //2203;
	case M_DS_2ECX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_DS_2ECX_ESI: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.esi); //2205;
	case M_DS_2ECX_EDI: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.edi); //2206;

	case M_DS_2EDX_EAX: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.eax); //2207;
	case M_DS_2EDX_ECX: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ecx); //2208;
	case M_DS_3EDX: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.edx); //2209;
	case M_DS_2EDX_EBX: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ebx); //2210;
	case M_DS_2EDX_ESP: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.esp); //2211;
	case M_DS_2EDX_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + getMicrocode()); //2212;
	case M_DS_2EDX_ESI: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.esi); //2213;
	case M_DS_2EDX_EDI: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.edi); //2214;

	case M_DS_2EBX_EAX: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.eax); //2215;
	case M_DS_2EBX_ECX: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_DS_2EBX_EDX: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.edx); //2217;
	case M_DS_3EBX: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_DS_2EBX_ESP: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.esp); //2219;
	case M_DS_2EBX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_DS_2EBX_ESI: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.esi); //2221;
	case M_DS_2EBX_EDI: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.edi); //2222;
	    
	case M_DS_2EBP_EAX: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.eax); //2223;
	case M_DS_2EBP_ECX: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_DS_2EBP_EDX: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.edx); //2225;
	case M_DS_2EBP_EBX: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_DS_2EBP_ESP: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.esp); //2227;
	case M_DS_2EBP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_DS_2EBP_ESI: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.esi); //2229;
	case M_DS_2EBP_EDI: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.edi); //2230;

	case M_DS_2ESI_EAX: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.eax); //2231;
	case M_DS_2ESI_ECX: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ecx); //2232;
	case M_DS_2ESI_EDX: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.edx); //2233;
	case M_DS_2ESI_EBX: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ebx); //2234;
	case M_DS_2ESI_ESP: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.esp); //2235;
	case M_DS_2ESI_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + getMicrocode()); //2236;
	case M_DS_3ESI: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.esi); //2237;
	case M_DS_2ESI_EDI: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.edi); //2238;

	case M_DS_2EDI_EAX: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.eax); //2239;
	case M_DS_2EDI_ECX: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ecx); //2240;
	case M_DS_2EDI_EDX: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.edx); //2241;
	case M_DS_2EDI_EBX: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ebx); //2242;
	case M_DS_2EDI_ESP: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.esp); //2243;
	case M_DS_2EDI_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + getMicrocode()); //2244;
	case M_DS_2EDI_ESI: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.esi); //2245;
	case M_DS_3EDI: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.edi); //2246;

	case M_DS_5EAX: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.eax); //2247
	case M_DS_4EAX_ECX: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ecx); //2248
	case M_DS_4EAX_EDX: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.edx); //2249
	case M_DS_4EAX_EBX: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ebx); //2250
	case M_DS_4EAX_ESP: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.esp); //2251
	case M_DS_4EAX_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + getMicrocode()); //2252
	case M_DS_4EAX_ESI: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.esi); //2253
	case M_DS_4EAX_EDI: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.edi); //2254

	case M_DS_4ECX_EAX: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.eax); //2255
	case M_DS_5ECX: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ecx); //2256
	case M_DS_4ECX_EDX: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.edx); //2257
	case M_DS_4ECX_EBX: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ebx); //2258
	case M_DS_4ECX_ESP: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.esp); //2259
	case M_DS_4ECX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + getMicrocode()); //2260
	case M_DS_4ECX_ESI: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.esi); //2261
	case M_DS_4ECX_EDI: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.edi); //2262

	case M_DS_4EDX_EAX: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.eax); //2263
	case M_DS_4EDX_ECX: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ecx); //2264
	case M_DS_5EDX: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.edx); //2265
	case M_DS_4EDX_EBX: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ebx); //2266
	case M_DS_4EDX_ESP: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.esp); //2267
	case M_DS_4EDX_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + getMicrocode()); //2268
	case M_DS_4EDX_ESI: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.esi); //2269
	case M_DS_4EDX_EDI: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.edi); //2270

	case M_DS_4EBX_EAX: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.eax); //2271
	case M_DS_4EBX_ECX: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ecx); //2272
	case M_DS_4EBX_EDX: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.edx); //2273
	case M_DS_5EBX: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ebx); //2274
	case M_DS_4EBX_ESP: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.esp); //2275
	case M_DS_4EBX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + getMicrocode()); //2276
	case M_DS_4EBX_ESI: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.esi); //2277
	case M_DS_4EBX_EDI: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.edi); //2278

	case M_DS_4EBP_EAX: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.eax); //2279
	case M_DS_4EBP_ECX: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ecx); //2280
	case M_DS_4EBP_EDX: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.edx); //2281
	case M_DS_4EBP_EBX: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ebx); //2282
	case M_DS_4EBP_ESP: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.esp); //2283
	case M_DS_4EBP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + getMicrocode()); //2284
	case M_DS_4EBP_ESI: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.esi); //2285
	case M_DS_4EBP_EDI: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.edi); //2286

	case M_DS_4ESI_EAX: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.eax); //2287
	case M_DS_4ESI_ECX: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ecx); //2288
	case M_DS_4ESI_EDX: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.edx); //2289
	case M_DS_4ESI_EBX: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ebx); //2290
	case M_DS_4ESI_ESP: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.esp); //2291
	case M_DS_4ESI_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + getMicrocode()); //2292
	case M_DS_5ESI: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.esi); //2293
	case M_DS_4ESI_EDI: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.edi); //2294

	case M_DS_4EDI_EAX: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.eax); //2295;
	case M_DS_4EDI_ECX: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ecx); //2296;
	case M_DS_4EDI_EDX: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.edx); //2297;
	case M_DS_4EDI_EBX: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ebx); //2298;
	case M_DS_4EDI_ESP: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.esp); //2299;
	case M_DS_4EDI_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + getMicrocode()); //2300;
	case M_DS_4EDI_ESI: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.esi); //2301;
	case M_DS_5EDI: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.edi); //2302;

	case M_DS_9EAX: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.eax); //2303;
	case M_DS_8EAX_ECX: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ecx); //2304;
	case M_DS_8EAX_EDX: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.edx); //2305;
	case M_DS_8EAX_EBX: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ebx); //2306;
	case M_DS_8EAX_ESP: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.esp); //2307;
	case M_DS_8EAX_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + getMicrocode()); //2308;
	case M_DS_8EAX_ESI: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.esi); //2309;
	case M_DS_8EAX_EDI: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_DS_8ECX_EAX: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.eax); //2311;
	case M_DS_9ECX: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_DS_8ECX_EDX: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.edx); //2313;
	case M_DS_8ECX_EBX: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_DS_8ECX_ESP: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.esp); //2315;
	case M_DS_8ECX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_DS_8ECX_ESI: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.esi); //2317;
	case M_DS_8ECX_EDI: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_DS_8EDX_EAX: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.eax); //2319;
	case M_DS_8EDX_ECX: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ecx); //2320;
	case M_DS_9EDX: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.edx); //2321;
	case M_DS_8EDX_EBX: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ebx); //2322;
	case M_DS_8EDX_ESP: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.esp); //2323;
	case M_DS_8EDX_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + getMicrocode()); //2324;
	case M_DS_8EDX_ESI: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.esi); //2325;
	case M_DS_8EDX_EDI: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_DS_8EBX_EAX: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.eax); //2327
	case M_DS_8EBX_ECX: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ecx); //2328
	case M_DS_8EBX_EDX: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.edx); //2329
	case M_DS_9EBX: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ebx); //2330
	case M_DS_8EBX_ESP: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.esp); //2331
	case M_DS_8EBX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + getMicrocode()); //2332
	case M_DS_8EBX_ESI: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.esi); //2333
	case M_DS_8EBX_EDI: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_DS_8EBP_EAX: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.eax); //2335;
	case M_DS_8EBP_ECX: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_DS_8EBP_EDX: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.edx); //2337;
	case M_DS_8EBP_EBX: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_DS_8EBP_ESP: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.esp); //2339;
	case M_DS_8EBP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_DS_8EBP_ESI: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.esi); //2341;
	case M_DS_8EBP_EDI: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_DS_8ESI_EAX: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.eax); //2343;
	case M_DS_8ESI_ECX: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ecx); //2344;
	case M_DS_8ESI_EDX: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.edx); //2345;
	case M_DS_8ESI_EBX: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ebx); //2346;
	case M_DS_8ESI_ESP: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.esp); //2347;
	case M_DS_8ESI_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + getMicrocode()); //2348;
	case M_DS_9ESI: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.esi); //2349;
	case M_DS_8ESI_EDI: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.edi); //2350;
	    
	case M_DS_8EDI_EAX: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.eax); //2351;
	case M_DS_8EDI_ECX: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ecx); //2352;
	case M_DS_8EDI_EDX: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.edx); //2353;
	case M_DS_8EDI_EBX: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ebx); //2354;
	case M_DS_8EDI_ESP: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.esp); //2355;
	case M_DS_8EDI_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + getMicrocode()); //2356;
	case M_DS_8EDI_ESI: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.esi); //2357;
	case M_DS_9EDI: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.edi); //2358;
	    
	case M_DS_2EAX_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_DS_EAX_ECX_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_DS_EAX_EDX_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_DS_EAX_EBX_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_DS_EAX_ESP_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_DS_EAX_EBP_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_DS_EAX_ESI_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_DS_EAX_EDI_IB: return cpu.ds.getDoubleWord(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_DS_2ECX_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_DS_ECX_EDX_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_DS_ECX_EBX_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_DS_ECX_ESP_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_DS_ECX_EBP_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_DS_ECX_ESI_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_DS_ECX_EDI_IB: return cpu.ds.getDoubleWord(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_DS_2EDX_IB: return cpu.ds.getDoubleWord(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_DS_EDX_EBX_IB: return cpu.ds.getDoubleWord(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_DS_EDX_ESP_IB: return cpu.ds.getDoubleWord(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_DS_EDX_EBP_IB: return cpu.ds.getDoubleWord(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_DS_EDX_ESI_IB: return cpu.ds.getDoubleWord(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_DS_EDX_EDI_IB: return cpu.ds.getDoubleWord(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_DS_2EBX_IB: return cpu.ds.getDoubleWord(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_DS_EBX_ESP_IB: return cpu.ds.getDoubleWord(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_DS_EBX_EBP_IB: return cpu.ds.getDoubleWord(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_DS_EBX_ESI_IB: return cpu.ds.getDoubleWord(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_DS_EBX_EDI_IB: return cpu.ds.getDoubleWord(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_DS_ESP_IB: return cpu.ds.getDoubleWord(cpu.esp + getMicrocode()); //2385

	case M_DS_EBP_ESP_IB: return cpu.ds.getDoubleWord(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_DS_2EBP_IB: return cpu.ds.getDoubleWord(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_DS_EBP_ESI_IB: return cpu.ds.getDoubleWord(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_DS_EBP_EDI_IB: return cpu.ds.getDoubleWord(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_DS_ESI_ESP_IB: return cpu.ds.getDoubleWord(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_DS_2ESI_IB: return cpu.ds.getDoubleWord(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_DS_ESI_EDI_IB: return cpu.ds.getDoubleWord(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_DS_EDI_ESP_IB: return cpu.ds.getDoubleWord(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_DS_2EDI_IB: return cpu.ds.getDoubleWord(cpu.edi + cpu.edi + getMicrocode()); //2394;
	    
	case M_DS_3EAX_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_DS_2EAX_ECX_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_DS_2EAX_EDX_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_DS_2EAX_EBX_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_DS_2EAX_ESP_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_DS_2EAX_EBP_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_DS_2EAX_ESI_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_DS_2EAX_EDI_IB: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_DS_2ECX_EAX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_DS_3ECX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_DS_2ECX_EDX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_DS_2ECX_EBX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_DS_2ECX_ESP_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_DS_2ECX_EBP_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_DS_2ECX_ESI_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_DS_2ECX_EDI_IB: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_DS_2EDX_EAX_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_DS_2EDX_ECX_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_DS_3EDX_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_DS_2EDX_EBX_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_DS_2EDX_ESP_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_DS_2EDX_EBP_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_DS_2EDX_ESI_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_DS_2EDX_EDI_IB: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;
	    
	case M_DS_2EBX_EAX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_DS_2EBX_ECX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_DS_2EBX_EDX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_DS_3EBX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_DS_2EBX_ESP_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_DS_2EBX_EBP_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_DS_2EBX_ESI_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_DS_2EBX_EDI_IB: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_DS_2EBP_EAX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_DS_2EBP_ECX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_DS_2EBP_EDX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_DS_2EBP_EBX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_DS_2EBP_ESP_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_DS_3EBP_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_DS_2EBP_ESI_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_DS_2EBP_EDI_IB: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_DS_2ESI_EAX_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_DS_2ESI_ECX_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_DS_2ESI_EDX_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_DS_2ESI_EBX_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_DS_2ESI_ESP_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_DS_2ESI_EBP_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_DS_3ESI_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_DS_2ESI_EDI_IB: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_DS_2EDI_EAX_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_DS_2EDI_ECX_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_DS_2EDI_EDX_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_DS_2EDI_EBX_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_DS_2EDI_ESP_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_DS_2EDI_EBP_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_DS_2EDI_ESI_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_DS_3EDI_IB: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_DS_5EAX_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_DS_4EAX_ECX_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_DS_4EAX_EDX_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_DS_4EAX_EBX_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_DS_4EAX_ESP_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_DS_4EAX_EBP_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_DS_4EAX_ESI_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_DS_4EAX_EDI_IB: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_DS_4ECX_EAX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_DS_5ECX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_DS_4ECX_EDX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_DS_4ECX_EBX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_DS_4ECX_ESP_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_DS_4ECX_EBP_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_DS_4ECX_ESI_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_DS_4ECX_EDI_IB: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_DS_4EDX_EAX_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_DS_4EDX_ECX_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_DS_5EDX_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_DS_4EDX_EBX_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_DS_4EDX_ESP_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_DS_4EDX_EBP_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_DS_4EDX_ESI_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_DS_4EDX_EDI_IB: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_DS_4EBX_EAX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_DS_4EBX_ECX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_DS_4EBX_EDX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_DS_5EBX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_DS_4EBX_ESP_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_DS_4EBX_EBP_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_DS_4EBX_ESI_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_DS_4EBX_EDI_IB: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_DS_4EBP_EAX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_DS_4EBP_ECX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_DS_4EBP_EDX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_DS_4EBP_EBX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_DS_4EBP_ESP_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_DS_5EBP_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_DS_4EBP_ESI_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_DS_4EBP_EDI_IB: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_DS_4ESI_EAX_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_DS_4ESI_ECX_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_DS_4ESI_EDX_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_DS_4ESI_EBX_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_DS_4ESI_ESP_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_DS_4ESI_EBP_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_DS_5ESI_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_DS_4ESI_EDI_IB: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_DS_4EDI_EAX_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_DS_4EDI_ECX_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_DS_4EDI_EDX_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_DS_4EDI_EBX_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_DS_4EDI_ESP_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_DS_4EDI_EBP_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_DS_4EDI_ESI_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_DS_5EDI_IB: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_DS_9EAX_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_DS_8EAX_ECX_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_DS_8EAX_EDX_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_DS_8EAX_EBX_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_DS_8EAX_ESP_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_DS_8EAX_EBP_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_DS_8EAX_ESI_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_DS_8EAX_EDI_IB: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_DS_8ECX_EAX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_DS_9ECX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_DS_8ECX_EDX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_DS_8ECX_EBX_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_DS_8ECX_ESP_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_DS_8ECX_EBP_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_DS_8ECX_ESI_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_DS_8ECX_EDI_IB: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_DS_8EDX_EAX_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_DS_8EDX_ECX_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_DS_9EDX_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_DS_8EDX_EBX_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_DS_8EDX_ESP_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_DS_8EDX_EBP_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_DS_8EDX_ESI_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_DS_8EDX_EDI_IB: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_DS_8EBX_EAX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_DS_8EBX_ECX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_DS_8EBX_EDX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_DS_9EBX_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_DS_8EBX_ESP_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_DS_8EBX_EBP_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_DS_8EBX_ESI_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_DS_8EBX_EDI_IB: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_DS_8EBP_EAX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_DS_8EBP_ECX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_DS_8EBP_EDX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_DS_8EBP_EBX_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_DS_8EBP_ESP_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_DS_9EBP_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_DS_8EBP_ESI_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_DS_8EBP_EDI_IB: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_DS_8ESI_EAX_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_DS_8ESI_ECX_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_DS_8ESI_EDX_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_DS_8ESI_EBX_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_DS_8ESI_ESP_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_DS_8ESI_EBP_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_DS_9ESI_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_DS_8ESI_EDI_IB: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_DS_8EDI_EAX_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_DS_8EDI_ECX_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_DS_8EDI_EDX_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_DS_8EDI_EBX_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_DS_8EDI_ESP_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_DS_8EDI_EBP_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_DS_8EDI_ESI_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_DS_9EDI_IB: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_DS_EAX_ECX_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_DS_EAX_EDX_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_DS_EAX_EBX_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_DS_EAX_ESP_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_DS_EAX_EBP_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_DS_EAX_ESI_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_DS_EAX_EDI_ID: return cpu.ds.getDoubleWord(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_DS_ECX_EDX_ID: return cpu.ds.getDoubleWord(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_DS_ECX_EBX_ID: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_DS_ECX_ESP_ID: return cpu.ds.getDoubleWord(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_DS_ECX_EBP_ID: return cpu.ds.getDoubleWord(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_DS_ECX_ESI_ID: return cpu.ds.getDoubleWord(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_DS_ECX_EDI_ID: return cpu.ds.getDoubleWord(cpu.ecx + cpu.edi + getMicrocode()); //2575;
	    
	case M_DS_EDX_EBX_ID: return cpu.ds.getDoubleWord(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_DS_EDX_ESP_ID: return cpu.ds.getDoubleWord(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_DS_EDX_EBP_ID: return cpu.ds.getDoubleWord(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_DS_EDX_ESI_ID: return cpu.ds.getDoubleWord(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_DS_EDX_EDI_ID: return cpu.ds.getDoubleWord(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_DS_EBX_ESP_ID: return cpu.ds.getDoubleWord(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_DS_EBX_EBP_ID: return cpu.ds.getDoubleWord(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_DS_EBX_ESI_ID: return cpu.ds.getDoubleWord(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_DS_EBX_EDI_ID: return cpu.ds.getDoubleWord(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_DS_ESP_ID: return cpu.ds.getDoubleWord(cpu.esp + getMicrocode()); //2585

	case M_DS_EBP_ESP_ID: return cpu.ds.getDoubleWord(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_DS_EBP_ESI_ID: return cpu.ds.getDoubleWord(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_DS_EBP_EDI_ID: return cpu.ds.getDoubleWord(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_DS_ESI_ESP_ID: return cpu.ds.getDoubleWord(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_DS_ESI_EDI_ID: return cpu.ds.getDoubleWord(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_DS_EDI_ESP_ID: return cpu.ds.getDoubleWord(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_DS_3EAX_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_DS_2EAX_ECX_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_DS_2EAX_EDX_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_DS_2EAX_EBX_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_DS_2EAX_ESP_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_DS_2EAX_EBP_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_DS_2EAX_ESI_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_DS_2EAX_EDI_ID: return cpu.ds.getDoubleWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_DS_2ECX_EAX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_DS_3ECX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_DS_2ECX_EDX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_DS_2ECX_EBX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_DS_2ECX_ESP_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_DS_2ECX_EBP_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_DS_2ECX_ESI_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_DS_2ECX_EDI_ID: return cpu.ds.getDoubleWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_DS_2EDX_EAX_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_DS_2EDX_ECX_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_DS_3EDX_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_DS_2EDX_EBX_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_DS_2EDX_ESP_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_DS_2EDX_EBP_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_DS_2EDX_ESI_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_DS_2EDX_EDI_ID: return cpu.ds.getDoubleWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_DS_2EBX_EAX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_DS_2EBX_ECX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_DS_2EBX_EDX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_DS_3EBX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_DS_2EBX_ESP_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_DS_2EBX_EBP_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_DS_2EBX_ESI_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_DS_2EBX_EDI_ID: return cpu.ds.getDoubleWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_DS_2EBP_EAX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_DS_2EBP_ECX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_DS_2EBP_EDX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_DS_2EBP_EBX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_DS_2EBP_ESP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_DS_3EBP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_DS_2EBP_ESI_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_DS_2EBP_EDI_ID: return cpu.ds.getDoubleWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_DS_2ESI_EAX_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_DS_2ESI_ECX_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_DS_2ESI_EDX_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_DS_2ESI_EBX_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_DS_2ESI_ESP_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_DS_2ESI_EBP_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_DS_3ESI_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_DS_2ESI_EDI_ID: return cpu.ds.getDoubleWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_DS_2EDI_EAX_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_DS_2EDI_ECX_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_DS_2EDI_EDX_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_DS_2EDI_EBX_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_DS_2EDI_ESP_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_DS_2EDI_EBP_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_DS_2EDI_ESI_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_DS_3EDI_ID: return cpu.ds.getDoubleWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_DS_5EAX_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_DS_4EAX_ECX_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_DS_4EAX_EDX_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_DS_4EAX_EBX_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_DS_4EAX_ESP_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_DS_4EAX_EBP_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_DS_4EAX_ESI_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_DS_4EAX_EDI_ID: return cpu.ds.getDoubleWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_DS_4ECX_EAX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_DS_5ECX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_DS_4ECX_EDX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_DS_4ECX_EBX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_DS_4ECX_ESP_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_DS_4ECX_EBP_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_DS_4ECX_ESI_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_DS_4ECX_EDI_ID: return cpu.ds.getDoubleWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_DS_4EDX_EAX_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_DS_4EDX_ECX_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_DS_5EDX_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_DS_4EDX_EBX_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_DS_4EDX_ESP_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_DS_4EDX_EBP_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_DS_4EDX_ESI_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_DS_4EDX_EDI_ID: return cpu.ds.getDoubleWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_DS_4EBX_EAX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_DS_4EBX_ECX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_DS_4EBX_EDX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_DS_5EBX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_DS_4EBX_ESP_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_DS_4EBX_EBP_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_DS_4EBX_ESI_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_DS_4EBX_EDI_ID: return cpu.ds.getDoubleWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_DS_4EBP_EAX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_DS_4EBP_ECX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_DS_4EBP_EDX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_DS_4EBP_EBX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_DS_4EBP_ESP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_DS_5EBP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_DS_4EBP_ESI_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_DS_4EBP_EDI_ID: return cpu.ds.getDoubleWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_DS_4ESI_EAX_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_DS_4ESI_ECX_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_DS_4ESI_EDX_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_DS_4ESI_EBX_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_DS_4ESI_ESP_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_DS_4ESI_EBP_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_DS_5ESI_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_DS_4ESI_EDI_ID: return cpu.ds.getDoubleWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_DS_4EDI_EAX_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_DS_4EDI_ECX_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_DS_4EDI_EDX_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_DS_4EDI_EBX_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_DS_4EDI_ESP_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_DS_4EDI_EBP_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_DS_4EDI_ESI_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_DS_5EDI_ID: return cpu.ds.getDoubleWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_DS_9EAX_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_DS_8EAX_ECX_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_DS_8EAX_EDX_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_DS_8EAX_EBX_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_DS_8EAX_ESP_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_DS_8EAX_EBP_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_DS_8EAX_ESI_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_DS_8EAX_EDI_ID: return cpu.ds.getDoubleWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_DS_8ECX_EAX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_DS_9ECX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_DS_8ECX_EDX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_DS_8ECX_EBX_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_DS_8ECX_ESP_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_DS_8ECX_EBP_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_DS_8ECX_ESI_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_DS_8ECX_EDI_ID: return cpu.ds.getDoubleWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_DS_8EDX_EAX_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_DS_8EDX_ECX_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_DS_9EDX_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_DS_8EDX_EBX_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_DS_8EDX_ESP_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_DS_8EDX_EBP_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_DS_8EDX_ESI_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_DS_8EDX_EDI_ID: return cpu.ds.getDoubleWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_DS_8EBX_EAX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_DS_8EBX_ECX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_DS_8EBX_EDX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_DS_9EBX_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_DS_8EBX_ESP_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_DS_8EBX_EBP_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_DS_8EBX_ESI_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_DS_8EBX_EDI_ID: return cpu.ds.getDoubleWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_DS_8EBP_EAX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_DS_8EBP_ECX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_DS_8EBP_EDX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_DS_8EBP_EBX_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_DS_8EBP_ESP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_DS_9EBP_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_DS_8EBP_ESI_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_DS_8EBP_EDI_ID: return cpu.ds.getDoubleWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_DS_8ESI_EAX_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_DS_8ESI_ECX_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_DS_8ESI_EDX_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_DS_8ESI_EBX_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_DS_8ESI_ESP_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_DS_8ESI_EBP_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_DS_9ESI_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_DS_8ESI_EDI_ID: return cpu.ds.getDoubleWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_DS_8EDI_EAX_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_DS_8EDI_ECX_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_DS_8EDI_EDX_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_DS_8EDI_EBX_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_DS_8EDI_ESP_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_DS_8EDI_EBP_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_DS_8EDI_ESI_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_DS_9EDI_ID: return cpu.ds.getDoubleWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;

	case M_SS_2EAX: return cpu.ss.getDoubleWord(cpu.eax + cpu.eax); //2156
	case M_SS_EAX_ECX: return cpu.ss.getDoubleWord(cpu.eax + cpu.ecx); //2157
	case M_SS_EAX_EDX: return cpu.ss.getDoubleWord(cpu.eax + cpu.edx); //2158
	case M_SS_EAX_EBX: return cpu.ss.getDoubleWord(cpu.eax + cpu.ebx); //2159
	case M_SS_EAX_ESP: return cpu.ss.getDoubleWord(cpu.eax + cpu.esp); //2160
	case M_SS_EAX_EBP: return cpu.ss.getDoubleWord(cpu.eax + cpu.ebp); //2161
	case M_SS_EAX_ESI: return cpu.ss.getDoubleWord(cpu.eax + cpu.esi); //2162
	case M_SS_EAX_EDI: return cpu.ss.getDoubleWord(cpu.eax + cpu.edi); //2163

	case M_SS_2ECX: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ecx); //2164
	case M_SS_ECX_EDX: return cpu.ss.getDoubleWord(cpu.ecx + cpu.edx); //2165
	case M_SS_ECX_EBX: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ebx); //2166
	case M_SS_ECX_ESP: return cpu.ss.getDoubleWord(cpu.ecx + cpu.esp); //2167
	case M_SS_ECX_EBP: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ebp); //2168
	case M_SS_ECX_ESI: return cpu.ss.getDoubleWord(cpu.ecx + cpu.esi); //2169
	case M_SS_ECX_EDI: return cpu.ss.getDoubleWord(cpu.ecx + cpu.edi); //2170

	case M_SS_2EDX: return cpu.ss.getDoubleWord(cpu.edx + cpu.edx); //2171
	case M_SS_EDX_EBX: return cpu.ss.getDoubleWord(cpu.edx + cpu.ebx); //2172
	case M_SS_EDX_ESP: return cpu.ss.getDoubleWord(cpu.edx + cpu.esp); //2173
	case M_SS_EDX_EBP: return cpu.ss.getDoubleWord(cpu.edx + cpu.ebp); //2174
	case M_SS_EDX_ESI: return cpu.ss.getDoubleWord(cpu.edx + cpu.esi); //2175	
	case M_SS_EDX_EDI: return cpu.ss.getDoubleWord(cpu.edx + cpu.edi); //2176

	case M_SS_2EBX: return cpu.ss.getDoubleWord(cpu.ebx + cpu.ebx); //2177;
	case M_SS_EBX_ESP: return cpu.ss.getDoubleWord(cpu.ebx + cpu.esp); //2178
	case M_SS_EBX_EBP: return cpu.ss.getDoubleWord(cpu.ebx + cpu.ebp); //2179
	case M_SS_EBX_ESI: return cpu.ss.getDoubleWord(cpu.ebx + cpu.esi); //2180
	case M_SS_EBX_EDI: return cpu.ss.getDoubleWord(cpu.ebx + cpu.edi); //2181
  
	case M_SS_ESP: return cpu.ss.getDoubleWord(cpu.esp); //2182

	case M_SS_EBP_ESP: return cpu.ss.getDoubleWord(cpu.ebp + cpu.esp); //2183;
	case M_SS_EBP_ESI: return cpu.ss.getDoubleWord(cpu.ebp + cpu.esi); //2184;
	case M_SS_EBP_EDI: return cpu.ss.getDoubleWord(cpu.ebp + cpu.edi); //2185;

	case M_SS_ESI_ESP: return cpu.ss.getDoubleWord(cpu.esi + cpu.esp); //2186;
	case M_SS_2ESI: return cpu.ss.getDoubleWord(cpu.esi + cpu.esi); //2187;
	case M_SS_ESI_EDI: return cpu.ss.getDoubleWord(cpu.esi + cpu.edi); //2188;

	case M_SS_EDI_ESP: return cpu.ss.getDoubleWord(cpu.edi + cpu.esp); //2189;
	case M_SS_2EDI: return cpu.ss.getDoubleWord(cpu.edi + cpu.edi); //2190;

	case M_SS_3EAX: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.eax); //2191;
	case M_SS_2EAX_ECX: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ecx); //2192;
	case M_SS_2EAX_EDX: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.edx); //2193;
	case M_SS_2EAX_EBX: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ebx); //2194;
	case M_SS_2EAX_ESP: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.esp); //2195;
	case M_SS_2EAX_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + getMicrocode()); //2196;
	case M_SS_2EAX_ESI: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.esi); //2197;
	case M_SS_2EAX_EDI: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.edi); //2198;

	case M_SS_2ECX_EAX: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.eax); //2199;
	case M_SS_3ECX: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_SS_2ECX_EDX: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.edx); //2201;
	case M_SS_2ECX_EBX: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_SS_2ECX_ESP: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.esp); //2203;
	case M_SS_2ECX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_SS_2ECX_ESI: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.esi); //2205;
	case M_SS_2ECX_EDI: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.edi); //2206;

	case M_SS_2EDX_EAX: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.eax); //2207;
	case M_SS_2EDX_ECX: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ecx); //2208;
	case M_SS_3EDX: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.edx); //2209;
	case M_SS_2EDX_EBX: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ebx); //2210;
	case M_SS_2EDX_ESP: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.esp); //2211;
	case M_SS_2EDX_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + getMicrocode()); //2212;
	case M_SS_2EDX_ESI: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.esi); //2213;
	case M_SS_2EDX_EDI: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.edi); //2214;

	case M_SS_2EBX_EAX: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.eax); //2215;
	case M_SS_2EBX_ECX: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_SS_2EBX_EDX: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.edx); //2217;
	case M_SS_3EBX: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_SS_2EBX_ESP: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.esp); //2219;
	case M_SS_2EBX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_SS_2EBX_ESI: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.esi); //2221;
	case M_SS_2EBX_EDI: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.edi); //2222;
	    
	case M_SS_2EBP_EAX: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.eax); //2223;
	case M_SS_2EBP_ECX: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_SS_2EBP_EDX: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.edx); //2225;
	case M_SS_2EBP_EBX: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_SS_2EBP_ESP: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.esp); //2227;
	case M_SS_2EBP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_SS_2EBP_ESI: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.esi); //2229;
	case M_SS_2EBP_EDI: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.edi); //2230;

	case M_SS_2ESI_EAX: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.eax); //2231;
	case M_SS_2ESI_ECX: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ecx); //2232;
	case M_SS_2ESI_EDX: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.edx); //2233;
	case M_SS_2ESI_EBX: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ebx); //2234;
	case M_SS_2ESI_ESP: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.esp); //2235;
	case M_SS_2ESI_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + getMicrocode()); //2236;
	case M_SS_3ESI: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.esi); //2237;
	case M_SS_2ESI_EDI: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.edi); //2238;

	case M_SS_2EDI_EAX: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.eax); //2239;
	case M_SS_2EDI_ECX: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ecx); //2240;
	case M_SS_2EDI_EDX: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.edx); //2241;
	case M_SS_2EDI_EBX: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ebx); //2242;
	case M_SS_2EDI_ESP: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.esp); //2243;
	case M_SS_2EDI_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + getMicrocode()); //2244;
	case M_SS_2EDI_ESI: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.esi); //2245;
	case M_SS_3EDI: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.edi); //2246;

	case M_SS_5EAX: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.eax); //2247
	case M_SS_4EAX_ECX: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ecx); //2248
	case M_SS_4EAX_EDX: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.edx); //2249
	case M_SS_4EAX_EBX: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ebx); //2250
	case M_SS_4EAX_ESP: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.esp); //2251
	case M_SS_4EAX_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + getMicrocode()); //2252
	case M_SS_4EAX_ESI: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.esi); //2253
	case M_SS_4EAX_EDI: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.edi); //2254

	case M_SS_4ECX_EAX: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.eax); //2255
	case M_SS_5ECX: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ecx); //2256
	case M_SS_4ECX_EDX: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.edx); //2257
	case M_SS_4ECX_EBX: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ebx); //2258
	case M_SS_4ECX_ESP: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.esp); //2259
	case M_SS_4ECX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + getMicrocode()); //2260
	case M_SS_4ECX_ESI: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.esi); //2261
	case M_SS_4ECX_EDI: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.edi); //2262

	case M_SS_4EDX_EAX: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.eax); //2263
	case M_SS_4EDX_ECX: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ecx); //2264
	case M_SS_5EDX: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.edx); //2265
	case M_SS_4EDX_EBX: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ebx); //2266
	case M_SS_4EDX_ESP: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.esp); //2267
	case M_SS_4EDX_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + getMicrocode()); //2268
	case M_SS_4EDX_ESI: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.esi); //2269
	case M_SS_4EDX_EDI: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.edi); //2270

	case M_SS_4EBX_EAX: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.eax); //2271
	case M_SS_4EBX_ECX: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ecx); //2272
	case M_SS_4EBX_EDX: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.edx); //2273
	case M_SS_5EBX: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ebx); //2274
	case M_SS_4EBX_ESP: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.esp); //2275
	case M_SS_4EBX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + getMicrocode()); //2276
	case M_SS_4EBX_ESI: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.esi); //2277
	case M_SS_4EBX_EDI: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.edi); //2278

	case M_SS_4EBP_EAX: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.eax); //2279
	case M_SS_4EBP_ECX: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ecx); //2280
	case M_SS_4EBP_EDX: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.edx); //2281
	case M_SS_4EBP_EBX: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ebx); //2282
	case M_SS_4EBP_ESP: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.esp); //2283
	case M_SS_4EBP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + getMicrocode()); //2284
	case M_SS_4EBP_ESI: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.esi); //2285
	case M_SS_4EBP_EDI: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.edi); //2286

	case M_SS_4ESI_EAX: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.eax); //2287
	case M_SS_4ESI_ECX: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ecx); //2288
	case M_SS_4ESI_EDX: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.edx); //2289
	case M_SS_4ESI_EBX: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ebx); //2290
	case M_SS_4ESI_ESP: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.esp); //2291
	case M_SS_4ESI_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + getMicrocode()); //2292
	case M_SS_5ESI: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.esi); //2293
	case M_SS_4ESI_EDI: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.edi); //2294

	case M_SS_4EDI_EAX: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.eax); //2295;
	case M_SS_4EDI_ECX: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ecx); //2296;
	case M_SS_4EDI_EDX: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.edx); //2297;
	case M_SS_4EDI_EBX: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ebx); //2298;
	case M_SS_4EDI_ESP: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.esp); //2299;
	case M_SS_4EDI_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + getMicrocode()); //2300;
	case M_SS_4EDI_ESI: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.esi); //2301;
	case M_SS_5EDI: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.edi); //2302;

	case M_SS_9EAX: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.eax); //2303;
	case M_SS_8EAX_ECX: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ecx); //2304;
	case M_SS_8EAX_EDX: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.edx); //2305;
	case M_SS_8EAX_EBX: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ebx); //2306;
	case M_SS_8EAX_ESP: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.esp); //2307;
	case M_SS_8EAX_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + getMicrocode()); //2308;
	case M_SS_8EAX_ESI: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.esi); //2309;
	case M_SS_8EAX_EDI: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_SS_8ECX_EAX: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.eax); //2311;
	case M_SS_9ECX: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_SS_8ECX_EDX: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.edx); //2313;
	case M_SS_8ECX_EBX: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_SS_8ECX_ESP: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.esp); //2315;
	case M_SS_8ECX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_SS_8ECX_ESI: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.esi); //2317;
	case M_SS_8ECX_EDI: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_SS_8EDX_EAX: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.eax); //2319;
	case M_SS_8EDX_ECX: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ecx); //2320;
	case M_SS_9EDX: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.edx); //2321;
	case M_SS_8EDX_EBX: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ebx); //2322;
	case M_SS_8EDX_ESP: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.esp); //2323;
	case M_SS_8EDX_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + getMicrocode()); //2324;
	case M_SS_8EDX_ESI: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.esi); //2325;
	case M_SS_8EDX_EDI: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_SS_8EBX_EAX: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.eax); //2327
	case M_SS_8EBX_ECX: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ecx); //2328
	case M_SS_8EBX_EDX: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.edx); //2329
	case M_SS_9EBX: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ebx); //2330
	case M_SS_8EBX_ESP: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.esp); //2331
	case M_SS_8EBX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + getMicrocode()); //2332
	case M_SS_8EBX_ESI: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.esi); //2333
	case M_SS_8EBX_EDI: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_SS_8EBP_EAX: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.eax); //2335;
	case M_SS_8EBP_ECX: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_SS_8EBP_EDX: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.edx); //2337;
	case M_SS_8EBP_EBX: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_SS_8EBP_ESP: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.esp); //2339;
	case M_SS_8EBP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_SS_8EBP_ESI: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.esi); //2341;
	case M_SS_8EBP_EDI: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_SS_8ESI_EAX: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.eax); //2343;
	case M_SS_8ESI_ECX: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ecx); //2344;
	case M_SS_8ESI_EDX: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.edx); //2345;
	case M_SS_8ESI_EBX: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ebx); //2346;
	case M_SS_8ESI_ESP: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.esp); //2347;
	case M_SS_8ESI_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + getMicrocode()); //2348;
	case M_SS_9ESI: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.esi); //2349;
	case M_SS_8ESI_EDI: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.edi); //2350;
	    
	case M_SS_8EDI_EAX: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.eax); //2351;
	case M_SS_8EDI_ECX: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ecx); //2352;
	case M_SS_8EDI_EDX: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.edx); //2353;
	case M_SS_8EDI_EBX: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ebx); //2354;
	case M_SS_8EDI_ESP: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.esp); //2355;
	case M_SS_8EDI_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + getMicrocode()); //2356;
	case M_SS_8EDI_ESI: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.esi); //2357;
	case M_SS_9EDI: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.edi); //2358;
	    
	case M_SS_2EAX_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_SS_EAX_ECX_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_SS_EAX_EDX_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_SS_EAX_EBX_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_SS_EAX_ESP_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_SS_EAX_EBP_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_SS_EAX_ESI_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_SS_EAX_EDI_IB: return cpu.ss.getDoubleWord(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_SS_2ECX_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_SS_ECX_EDX_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_SS_ECX_EBX_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_SS_ECX_ESP_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_SS_ECX_EBP_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_SS_ECX_ESI_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_SS_ECX_EDI_IB: return cpu.ss.getDoubleWord(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_SS_2EDX_IB: return cpu.ss.getDoubleWord(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_SS_EDX_EBX_IB: return cpu.ss.getDoubleWord(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_SS_EDX_ESP_IB: return cpu.ss.getDoubleWord(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_SS_EDX_EBP_IB: return cpu.ss.getDoubleWord(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_SS_EDX_ESI_IB: return cpu.ss.getDoubleWord(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_SS_EDX_EDI_IB: return cpu.ss.getDoubleWord(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_SS_2EBX_IB: return cpu.ss.getDoubleWord(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_SS_EBX_ESP_IB: return cpu.ss.getDoubleWord(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_SS_EBX_EBP_IB: return cpu.ss.getDoubleWord(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_SS_EBX_ESI_IB: return cpu.ss.getDoubleWord(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_SS_EBX_EDI_IB: return cpu.ss.getDoubleWord(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_SS_ESP_IB: return cpu.ss.getDoubleWord(cpu.esp + getMicrocode()); //2385

	case M_SS_EBP_ESP_IB: return cpu.ss.getDoubleWord(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_SS_2EBP_IB: return cpu.ss.getDoubleWord(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_SS_EBP_ESI_IB: return cpu.ss.getDoubleWord(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_SS_EBP_EDI_IB: return cpu.ss.getDoubleWord(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_SS_ESI_ESP_IB: return cpu.ss.getDoubleWord(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_SS_2ESI_IB: return cpu.ss.getDoubleWord(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_SS_ESI_EDI_IB: return cpu.ss.getDoubleWord(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_SS_EDI_ESP_IB: return cpu.ss.getDoubleWord(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_SS_2EDI_IB: return cpu.ss.getDoubleWord(cpu.edi + cpu.edi + getMicrocode()); //2394;
	    
	case M_SS_3EAX_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_SS_2EAX_ECX_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_SS_2EAX_EDX_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_SS_2EAX_EBX_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_SS_2EAX_ESP_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_SS_2EAX_EBP_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_SS_2EAX_ESI_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_SS_2EAX_EDI_IB: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_SS_2ECX_EAX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_SS_3ECX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_SS_2ECX_EDX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_SS_2ECX_EBX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_SS_2ECX_ESP_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_SS_2ECX_EBP_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_SS_2ECX_ESI_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_SS_2ECX_EDI_IB: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_SS_2EDX_EAX_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_SS_2EDX_ECX_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_SS_3EDX_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_SS_2EDX_EBX_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_SS_2EDX_ESP_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_SS_2EDX_EBP_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_SS_2EDX_ESI_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_SS_2EDX_EDI_IB: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;
	    
	case M_SS_2EBX_EAX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_SS_2EBX_ECX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_SS_2EBX_EDX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_SS_3EBX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_SS_2EBX_ESP_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_SS_2EBX_EBP_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_SS_2EBX_ESI_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_SS_2EBX_EDI_IB: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_SS_2EBP_EAX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_SS_2EBP_ECX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_SS_2EBP_EDX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_SS_2EBP_EBX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_SS_2EBP_ESP_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_SS_3EBP_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_SS_2EBP_ESI_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_SS_2EBP_EDI_IB: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_SS_2ESI_EAX_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_SS_2ESI_ECX_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_SS_2ESI_EDX_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_SS_2ESI_EBX_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_SS_2ESI_ESP_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_SS_2ESI_EBP_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_SS_3ESI_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_SS_2ESI_EDI_IB: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_SS_2EDI_EAX_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_SS_2EDI_ECX_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_SS_2EDI_EDX_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_SS_2EDI_EBX_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_SS_2EDI_ESP_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_SS_2EDI_EBP_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_SS_2EDI_ESI_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_SS_3EDI_IB: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_SS_5EAX_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_SS_4EAX_ECX_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_SS_4EAX_EDX_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_SS_4EAX_EBX_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_SS_4EAX_ESP_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_SS_4EAX_EBP_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_SS_4EAX_ESI_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_SS_4EAX_EDI_IB: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_SS_4ECX_EAX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_SS_5ECX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_SS_4ECX_EDX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_SS_4ECX_EBX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_SS_4ECX_ESP_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_SS_4ECX_EBP_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_SS_4ECX_ESI_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_SS_4ECX_EDI_IB: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_SS_4EDX_EAX_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_SS_4EDX_ECX_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_SS_5EDX_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_SS_4EDX_EBX_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_SS_4EDX_ESP_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_SS_4EDX_EBP_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_SS_4EDX_ESI_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_SS_4EDX_EDI_IB: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_SS_4EBX_EAX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_SS_4EBX_ECX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_SS_4EBX_EDX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_SS_5EBX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_SS_4EBX_ESP_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_SS_4EBX_EBP_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_SS_4EBX_ESI_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_SS_4EBX_EDI_IB: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_SS_4EBP_EAX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_SS_4EBP_ECX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_SS_4EBP_EDX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_SS_4EBP_EBX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_SS_4EBP_ESP_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_SS_5EBP_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_SS_4EBP_ESI_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_SS_4EBP_EDI_IB: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_SS_4ESI_EAX_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_SS_4ESI_ECX_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_SS_4ESI_EDX_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_SS_4ESI_EBX_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_SS_4ESI_ESP_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_SS_4ESI_EBP_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_SS_5ESI_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_SS_4ESI_EDI_IB: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_SS_4EDI_EAX_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_SS_4EDI_ECX_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_SS_4EDI_EDX_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_SS_4EDI_EBX_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_SS_4EDI_ESP_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_SS_4EDI_EBP_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_SS_4EDI_ESI_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_SS_5EDI_IB: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_SS_9EAX_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_SS_8EAX_ECX_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_SS_8EAX_EDX_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_SS_8EAX_EBX_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_SS_8EAX_ESP_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_SS_8EAX_EBP_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_SS_8EAX_ESI_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_SS_8EAX_EDI_IB: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_SS_8ECX_EAX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_SS_9ECX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_SS_8ECX_EDX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_SS_8ECX_EBX_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_SS_8ECX_ESP_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_SS_8ECX_EBP_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_SS_8ECX_ESI_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_SS_8ECX_EDI_IB: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_SS_8EDX_EAX_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_SS_8EDX_ECX_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_SS_9EDX_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_SS_8EDX_EBX_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_SS_8EDX_ESP_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_SS_8EDX_EBP_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_SS_8EDX_ESI_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_SS_8EDX_EDI_IB: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_SS_8EBX_EAX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_SS_8EBX_ECX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_SS_8EBX_EDX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_SS_9EBX_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_SS_8EBX_ESP_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_SS_8EBX_EBP_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_SS_8EBX_ESI_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_SS_8EBX_EDI_IB: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_SS_8EBP_EAX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_SS_8EBP_ECX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_SS_8EBP_EDX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_SS_8EBP_EBX_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_SS_8EBP_ESP_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_SS_9EBP_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_SS_8EBP_ESI_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_SS_8EBP_EDI_IB: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_SS_8ESI_EAX_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_SS_8ESI_ECX_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_SS_8ESI_EDX_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_SS_8ESI_EBX_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_SS_8ESI_ESP_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_SS_8ESI_EBP_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_SS_9ESI_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_SS_8ESI_EDI_IB: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_SS_8EDI_EAX_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_SS_8EDI_ECX_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_SS_8EDI_EDX_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_SS_8EDI_EBX_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_SS_8EDI_ESP_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_SS_8EDI_EBP_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_SS_8EDI_ESI_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_SS_9EDI_IB: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_SS_EAX_ECX_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_SS_EAX_EDX_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_SS_EAX_EBX_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_SS_EAX_ESP_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_SS_EAX_EBP_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_SS_EAX_ESI_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_SS_EAX_EDI_ID: return cpu.ss.getDoubleWord(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_SS_ECX_EDX_ID: return cpu.ss.getDoubleWord(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_SS_ECX_EBX_ID: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_SS_ECX_ESP_ID: return cpu.ss.getDoubleWord(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_SS_ECX_EBP_ID: return cpu.ss.getDoubleWord(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_SS_ECX_ESI_ID: return cpu.ss.getDoubleWord(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_SS_ECX_EDI_ID: return cpu.ss.getDoubleWord(cpu.ecx + cpu.edi + getMicrocode()); //2575;
	    
	case M_SS_EDX_EBX_ID: return cpu.ss.getDoubleWord(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_SS_EDX_ESP_ID: return cpu.ss.getDoubleWord(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_SS_EDX_EBP_ID: return cpu.ss.getDoubleWord(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_SS_EDX_ESI_ID: return cpu.ss.getDoubleWord(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_SS_EDX_EDI_ID: return cpu.ss.getDoubleWord(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_SS_EBX_ESP_ID: return cpu.ss.getDoubleWord(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_SS_EBX_EBP_ID: return cpu.ss.getDoubleWord(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_SS_EBX_ESI_ID: return cpu.ss.getDoubleWord(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_SS_EBX_EDI_ID: return cpu.ss.getDoubleWord(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_SS_ESP_ID: return cpu.ss.getDoubleWord(cpu.esp + getMicrocode()); //2585

	case M_SS_EBP_ESP_ID: return cpu.ss.getDoubleWord(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_SS_EBP_ESI_ID: return cpu.ss.getDoubleWord(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_SS_EBP_EDI_ID: return cpu.ss.getDoubleWord(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_SS_ESI_ESP_ID: return cpu.ss.getDoubleWord(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_SS_ESI_EDI_ID: return cpu.ss.getDoubleWord(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_SS_EDI_ESP_ID: return cpu.ss.getDoubleWord(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_SS_3EAX_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_SS_2EAX_ECX_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_SS_2EAX_EDX_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_SS_2EAX_EBX_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_SS_2EAX_ESP_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_SS_2EAX_EBP_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_SS_2EAX_ESI_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_SS_2EAX_EDI_ID: return cpu.ss.getDoubleWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_SS_2ECX_EAX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_SS_3ECX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_SS_2ECX_EDX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_SS_2ECX_EBX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_SS_2ECX_ESP_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_SS_2ECX_EBP_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_SS_2ECX_ESI_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_SS_2ECX_EDI_ID: return cpu.ss.getDoubleWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_SS_2EDX_EAX_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_SS_2EDX_ECX_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_SS_3EDX_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_SS_2EDX_EBX_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_SS_2EDX_ESP_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_SS_2EDX_EBP_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_SS_2EDX_ESI_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_SS_2EDX_EDI_ID: return cpu.ss.getDoubleWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_SS_2EBX_EAX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_SS_2EBX_ECX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_SS_2EBX_EDX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_SS_3EBX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_SS_2EBX_ESP_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_SS_2EBX_EBP_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_SS_2EBX_ESI_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_SS_2EBX_EDI_ID: return cpu.ss.getDoubleWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_SS_2EBP_EAX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_SS_2EBP_ECX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_SS_2EBP_EDX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_SS_2EBP_EBX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_SS_2EBP_ESP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_SS_3EBP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_SS_2EBP_ESI_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_SS_2EBP_EDI_ID: return cpu.ss.getDoubleWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_SS_2ESI_EAX_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_SS_2ESI_ECX_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_SS_2ESI_EDX_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_SS_2ESI_EBX_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_SS_2ESI_ESP_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_SS_2ESI_EBP_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_SS_3ESI_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_SS_2ESI_EDI_ID: return cpu.ss.getDoubleWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_SS_2EDI_EAX_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_SS_2EDI_ECX_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_SS_2EDI_EDX_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_SS_2EDI_EBX_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_SS_2EDI_ESP_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_SS_2EDI_EBP_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_SS_2EDI_ESI_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_SS_3EDI_ID: return cpu.ss.getDoubleWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_SS_5EAX_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_SS_4EAX_ECX_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_SS_4EAX_EDX_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_SS_4EAX_EBX_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_SS_4EAX_ESP_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_SS_4EAX_EBP_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_SS_4EAX_ESI_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_SS_4EAX_EDI_ID: return cpu.ss.getDoubleWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_SS_4ECX_EAX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_SS_5ECX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_SS_4ECX_EDX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_SS_4ECX_EBX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_SS_4ECX_ESP_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_SS_4ECX_EBP_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_SS_4ECX_ESI_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_SS_4ECX_EDI_ID: return cpu.ss.getDoubleWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_SS_4EDX_EAX_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_SS_4EDX_ECX_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_SS_5EDX_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_SS_4EDX_EBX_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_SS_4EDX_ESP_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_SS_4EDX_EBP_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_SS_4EDX_ESI_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_SS_4EDX_EDI_ID: return cpu.ss.getDoubleWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_SS_4EBX_EAX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_SS_4EBX_ECX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_SS_4EBX_EDX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_SS_5EBX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_SS_4EBX_ESP_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_SS_4EBX_EBP_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_SS_4EBX_ESI_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_SS_4EBX_EDI_ID: return cpu.ss.getDoubleWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_SS_4EBP_EAX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_SS_4EBP_ECX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_SS_4EBP_EDX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_SS_4EBP_EBX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_SS_4EBP_ESP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_SS_5EBP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_SS_4EBP_ESI_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_SS_4EBP_EDI_ID: return cpu.ss.getDoubleWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_SS_4ESI_EAX_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_SS_4ESI_ECX_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_SS_4ESI_EDX_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_SS_4ESI_EBX_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_SS_4ESI_ESP_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_SS_4ESI_EBP_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_SS_5ESI_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_SS_4ESI_EDI_ID: return cpu.ss.getDoubleWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_SS_4EDI_EAX_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_SS_4EDI_ECX_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_SS_4EDI_EDX_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_SS_4EDI_EBX_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_SS_4EDI_ESP_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_SS_4EDI_EBP_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_SS_4EDI_ESI_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_SS_5EDI_ID: return cpu.ss.getDoubleWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_SS_9EAX_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_SS_8EAX_ECX_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_SS_8EAX_EDX_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_SS_8EAX_EBX_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_SS_8EAX_ESP_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_SS_8EAX_EBP_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_SS_8EAX_ESI_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_SS_8EAX_EDI_ID: return cpu.ss.getDoubleWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_SS_8ECX_EAX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_SS_9ECX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_SS_8ECX_EDX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_SS_8ECX_EBX_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_SS_8ECX_ESP_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_SS_8ECX_EBP_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_SS_8ECX_ESI_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_SS_8ECX_EDI_ID: return cpu.ss.getDoubleWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_SS_8EDX_EAX_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_SS_8EDX_ECX_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_SS_9EDX_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_SS_8EDX_EBX_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_SS_8EDX_ESP_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_SS_8EDX_EBP_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_SS_8EDX_ESI_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_SS_8EDX_EDI_ID: return cpu.ss.getDoubleWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_SS_8EBX_EAX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_SS_8EBX_ECX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_SS_8EBX_EDX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_SS_9EBX_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_SS_8EBX_ESP_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_SS_8EBX_EBP_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_SS_8EBX_ESI_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_SS_8EBX_EDI_ID: return cpu.ss.getDoubleWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_SS_8EBP_EAX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_SS_8EBP_ECX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_SS_8EBP_EDX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_SS_8EBP_EBX_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_SS_8EBP_ESP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_SS_9EBP_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_SS_8EBP_ESI_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_SS_8EBP_EDI_ID: return cpu.ss.getDoubleWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_SS_8ESI_EAX_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_SS_8ESI_ECX_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_SS_8ESI_EDX_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_SS_8ESI_EBX_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_SS_8ESI_ESP_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_SS_8ESI_EBP_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_SS_9ESI_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_SS_8ESI_EDI_ID: return cpu.ss.getDoubleWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_SS_8EDI_EAX_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_SS_8EDI_ECX_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_SS_8EDI_EDX_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_SS_8EDI_EBX_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_SS_8EDI_ESP_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_SS_8EDI_EBP_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_SS_8EDI_ESI_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_SS_9EDI_ID: return cpu.ss.getDoubleWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;
	}

	System.err.println("Unimplemented GetInt Operand: " + microcode);
	throw unimplemented;
    }

    protected final long getLongOperand()
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case IA48: return (getMicrocode() & 0xffffffffl) | ((getMicrocode() & 0xffffffffl) << 32); 
	}

	return getLongMemoryOperand(microcode);
    }

    private final long getLongMemoryOperand(int microcode)
    {
	switch (microcode) {
	case M_ES_BX_SI: return cpu.es.getQuadWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_ES_BX_DI: return cpu.es.getQuadWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_ES_BP_SI: return cpu.es.getQuadWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_ES_BP_DI: return cpu.es.getQuadWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_ES_SI: return cpu.es.getQuadWord(cpu.esi & 0xffff);
	case M_ES_DI: return cpu.es.getQuadWord(cpu.edi & 0xffff);
	case M_ES_IW: return cpu.es.getQuadWord(getMicrocode() & 0xffff);
	case M_ES_BX: return cpu.es.getQuadWord(cpu.ebx & 0xffff);
	case M_ES_BX_SI_IB: return cpu.es.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IB: return cpu.es.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IB: return cpu.es.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IB: return cpu.es.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IB: return cpu.es.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IB: return cpu.es.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IB: return cpu.es.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IB: return cpu.es.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_ES_BX_SI_IW: return cpu.es.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BX_DI_IW: return cpu.es.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_SI_IW: return cpu.es.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_BP_DI_IW: return cpu.es.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_SI_IW: return cpu.es.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_ES_DI_IW: return cpu.es.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_ES_BP_IW: return cpu.es.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_ES_BX_IW: return cpu.es.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_CS_BX_SI: return cpu.cs.getQuadWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_CS_BX_DI: return cpu.cs.getQuadWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_CS_BP_SI: return cpu.cs.getQuadWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_CS_BP_DI: return cpu.cs.getQuadWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_CS_SI: return cpu.cs.getQuadWord(cpu.esi & 0xffff);
	case M_CS_DI: return cpu.cs.getQuadWord(cpu.edi & 0xffff);
	case M_CS_IW: return cpu.cs.getQuadWord(getMicrocode() & 0xffff);
	case M_CS_BX: return cpu.cs.getQuadWord(cpu.ebx & 0xffff);
	case M_CS_BX_SI_IB: return cpu.cs.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IB: return cpu.cs.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IB: return cpu.cs.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IB: return cpu.cs.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IB: return cpu.cs.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IB: return cpu.cs.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IB: return cpu.cs.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IB: return cpu.cs.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_CS_BX_SI_IW: return cpu.cs.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BX_DI_IW: return cpu.cs.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_SI_IW: return cpu.cs.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_BP_DI_IW: return cpu.cs.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_SI_IW: return cpu.cs.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_CS_DI_IW: return cpu.cs.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_CS_BP_IW: return cpu.cs.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_CS_BX_IW: return cpu.cs.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_SS_BX_SI: return cpu.ss.getQuadWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_SS_BX_DI: return cpu.ss.getQuadWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_SS_BP_SI: return cpu.ss.getQuadWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_SS_BP_DI: return cpu.ss.getQuadWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_SS_SI: return cpu.ss.getQuadWord(cpu.esi & 0xffff);
	case M_SS_DI: return cpu.ss.getQuadWord(cpu.edi & 0xffff);
	case M_SS_IW: return cpu.ss.getQuadWord(getMicrocode() & 0xffff);
	case M_SS_BX: return cpu.ss.getQuadWord(cpu.ebx & 0xffff);
	case M_SS_BX_SI_IB: return cpu.ss.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IB: return cpu.ss.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IB: return cpu.ss.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IB: return cpu.ss.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IB: return cpu.ss.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IB: return cpu.ss.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IB: return cpu.ss.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IB: return cpu.ss.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_SS_BX_SI_IW: return cpu.ss.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BX_DI_IW: return cpu.ss.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_SI_IW: return cpu.ss.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_BP_DI_IW: return cpu.ss.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_SI_IW: return cpu.ss.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_SS_DI_IW: return cpu.ss.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_SS_BP_IW: return cpu.ss.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_SS_BX_IW: return cpu.ss.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_DS_BX_SI: return cpu.ds.getQuadWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_DS_BX_DI: return cpu.ds.getQuadWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_DS_BP_SI: return cpu.ds.getQuadWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_DS_BP_DI: return cpu.ds.getQuadWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_DS_SI: return cpu.ds.getQuadWord(cpu.esi & 0xffff);
	case M_DS_DI: return cpu.ds.getQuadWord(cpu.edi & 0xffff);
	case M_DS_IW: return cpu.ds.getQuadWord(getMicrocode() & 0xffff);
	case M_DS_BX: return cpu.ds.getQuadWord(cpu.ebx & 0xffff);
	case M_DS_BX_SI_IB: return cpu.ds.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IB: return cpu.ds.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IB: return cpu.ds.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IB: return cpu.ds.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IB: return cpu.ds.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IB: return cpu.ds.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IB: return cpu.ds.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IB: return cpu.ds.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_DS_BX_SI_IW: return cpu.ds.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BX_DI_IW: return cpu.ds.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_SI_IW: return cpu.ds.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_BP_DI_IW: return cpu.ds.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_SI_IW: return cpu.ds.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_DS_DI_IW: return cpu.ds.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_DS_BP_IW: return cpu.ds.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_DS_BX_IW: return cpu.ds.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_FS_BX_SI: return cpu.fs.getQuadWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_FS_BX_DI: return cpu.fs.getQuadWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_FS_BP_SI: return cpu.fs.getQuadWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_FS_BP_DI: return cpu.fs.getQuadWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_FS_SI: return cpu.fs.getQuadWord(cpu.esi & 0xffff);
	case M_FS_DI: return cpu.fs.getQuadWord(cpu.edi & 0xffff);
	case M_FS_IW: return cpu.fs.getQuadWord(getMicrocode() & 0xffff);
	case M_FS_BX: return cpu.fs.getQuadWord(cpu.ebx & 0xffff);
	case M_FS_BX_SI_IB: return cpu.fs.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IB: return cpu.fs.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IB: return cpu.fs.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IB: return cpu.fs.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IB: return cpu.fs.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IB: return cpu.fs.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IB: return cpu.fs.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IB: return cpu.fs.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_FS_BX_SI_IW: return cpu.fs.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BX_DI_IW: return cpu.fs.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_SI_IW: return cpu.fs.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_BP_DI_IW: return cpu.fs.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_SI_IW: return cpu.fs.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_FS_DI_IW: return cpu.fs.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_FS_BP_IW: return cpu.fs.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_FS_BX_IW: return cpu.fs.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_GS_BX_SI: return cpu.gs.getQuadWord((cpu.ebx + cpu.esi) & 0xffff);
	case M_GS_BX_DI: return cpu.gs.getQuadWord((cpu.ebx + cpu.edi) & 0xffff);
	case M_GS_BP_SI: return cpu.gs.getQuadWord((cpu.ebp + cpu.esi) & 0xffff);
	case M_GS_BP_DI: return cpu.gs.getQuadWord((cpu.ebp + cpu.edi) & 0xffff);
	case M_GS_SI: return cpu.gs.getQuadWord(cpu.esi & 0xffff);
	case M_GS_DI: return cpu.gs.getQuadWord(cpu.edi & 0xffff);
	case M_GS_IW: return cpu.gs.getQuadWord(getMicrocode() & 0xffff);
	case M_GS_BX: return cpu.gs.getQuadWord(cpu.ebx & 0xffff);
	case M_GS_BX_SI_IB: return cpu.gs.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IB: return cpu.gs.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IB: return cpu.gs.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IB: return cpu.gs.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IB: return cpu.gs.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IB: return cpu.gs.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IB: return cpu.gs.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IB: return cpu.gs.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);
	case M_GS_BX_SI_IW: return cpu.gs.getQuadWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BX_DI_IW: return cpu.gs.getQuadWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_SI_IW: return cpu.gs.getQuadWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_BP_DI_IW: return cpu.gs.getQuadWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_SI_IW: return cpu.gs.getQuadWord((cpu.esi + getMicrocode()) & 0xffff);
	case M_GS_DI_IW: return cpu.gs.getQuadWord((cpu.edi + getMicrocode()) & 0xffff);
	case M_GS_BP_IW: return cpu.gs.getQuadWord((cpu.ebp + getMicrocode()) & 0xffff);
	case M_GS_BX_IW: return cpu.gs.getQuadWord((cpu.ebx + getMicrocode()) & 0xffff);

	case M_CS_ID: return cpu.cs.getQuadWord(getMicrocode());

	case M_SS_EAX_IB: return cpu.ss.getQuadWord(cpu.eax + getMicrocode()); //90
	case M_SS_ECX_IB: return cpu.ss.getQuadWord(cpu.ecx + getMicrocode()); //91
	case M_SS_EDX_IB: return cpu.ss.getQuadWord(cpu.edx + getMicrocode()); //92
	case M_SS_EBX_IB: return cpu.ss.getQuadWord(cpu.ebx + getMicrocode()); //93
	case M_SS_EBP_IB: return cpu.ss.getQuadWord(cpu.ebp + getMicrocode()); //94
	case M_SS_ESI_IB: return cpu.ss.getQuadWord(cpu.esi + getMicrocode()); //95
	case M_SS_EDI_IB: return cpu.ss.getQuadWord(cpu.edi + getMicrocode()); //96

	case M_SS_EAX_ID: return cpu.ss.getQuadWord(cpu.eax + getMicrocode()); //97
	case M_SS_ECX_ID: return cpu.ss.getQuadWord(cpu.ecx + getMicrocode()); //98
	case M_SS_EDX_ID: return cpu.ss.getQuadWord(cpu.edx + getMicrocode()); //99
	case M_SS_EBX_ID: return cpu.ss.getQuadWord(cpu.ebx + getMicrocode()); //100
	case M_SS_EBP_ID: return cpu.ss.getQuadWord(cpu.ebp + getMicrocode()); //101
	case M_SS_ESI_ID: return cpu.ss.getQuadWord(cpu.esi + getMicrocode()); //102
	case M_SS_EDI_ID: return cpu.ss.getQuadWord(cpu.edi + getMicrocode()); //103

	case M_DS_EAX: return cpu.ds.getQuadWord(cpu.eax); //104
	case M_DS_ECX: return cpu.ds.getQuadWord(cpu.ecx); //105
	case M_DS_EDX: return cpu.ds.getQuadWord(cpu.edx); //106
	case M_DS_EBX: return cpu.ds.getQuadWord(cpu.ebx); //107
	case M_DS_ID: return cpu.ds.getQuadWord(getMicrocode()); //108
	case M_DS_ESI: return cpu.ds.getQuadWord(cpu.esi); //109
	case M_DS_EDI: return cpu.ds.getQuadWord(cpu.edi); //110

	case M_DS_EAX_IB: return cpu.ds.getQuadWord(cpu.eax + getMicrocode()); //111
	case M_DS_ECX_IB: return cpu.ds.getQuadWord(cpu.ecx + getMicrocode()); //112
	case M_DS_EDX_IB: return cpu.ds.getQuadWord(cpu.edx + getMicrocode()); //113
	case M_DS_EBX_IB: return cpu.ds.getQuadWord(cpu.ebx + getMicrocode()); //114
	case M_DS_EBP_IB: return cpu.ds.getQuadWord(cpu.ebp + getMicrocode()); //115
	case M_DS_ESI_IB: return cpu.ds.getQuadWord(cpu.esi + getMicrocode()); //116
	case M_DS_EDI_IB: return cpu.ds.getQuadWord(cpu.edi + getMicrocode()); //117

	case M_DS_EAX_ID: return cpu.ds.getQuadWord(cpu.eax + getMicrocode()); //118
	case M_DS_ECX_ID: return cpu.ds.getQuadWord(cpu.ecx + getMicrocode()); //119
	case M_DS_EDX_ID: return cpu.ds.getQuadWord(cpu.edx + getMicrocode()); //120
	case M_DS_EBX_ID: return cpu.ds.getQuadWord(cpu.ebx + getMicrocode()); //121
	case M_DS_EBP_ID: return cpu.ds.getQuadWord(cpu.ebp + getMicrocode()); //122
	case M_DS_ESI_ID: return cpu.ds.getQuadWord(cpu.esi + getMicrocode()); //123
	case M_DS_EDI_ID: return cpu.ds.getQuadWord(cpu.edi + getMicrocode()); //124

	case M_DS_2EAX: return cpu.ds.getQuadWord(cpu.eax + cpu.eax); //2156
	case M_DS_EAX_ECX: return cpu.ds.getQuadWord(cpu.eax + cpu.ecx); //2157
	case M_DS_EAX_EDX: return cpu.ds.getQuadWord(cpu.eax + cpu.edx); //2158
	case M_DS_EAX_EBX: return cpu.ds.getQuadWord(cpu.eax + cpu.ebx); //2159
	case M_DS_EAX_ESP: return cpu.ds.getQuadWord(cpu.eax + cpu.esp); //2160
	case M_DS_EAX_EBP: return cpu.ds.getQuadWord(cpu.eax + cpu.ebp); //2161
	case M_DS_EAX_ESI: return cpu.ds.getQuadWord(cpu.eax + cpu.esi); //2162
	case M_DS_EAX_EDI: return cpu.ds.getQuadWord(cpu.eax + cpu.edi); //2163

	case M_DS_2ECX: return cpu.ds.getQuadWord(cpu.ecx + cpu.ecx); //2164
	case M_DS_ECX_EDX: return cpu.ds.getQuadWord(cpu.ecx + cpu.edx); //2165
	case M_DS_ECX_EBX: return cpu.ds.getQuadWord(cpu.ecx + cpu.ebx); //2166
	case M_DS_ECX_ESP: return cpu.ds.getQuadWord(cpu.ecx + cpu.esp); //2167
	case M_DS_ECX_EBP: return cpu.ds.getQuadWord(cpu.ecx + cpu.ebp); //2168
	case M_DS_ECX_ESI: return cpu.ds.getQuadWord(cpu.ecx + cpu.esi); //2169
	case M_DS_ECX_EDI: return cpu.ds.getQuadWord(cpu.ecx + cpu.edi); //2170

	case M_DS_2EDX: return cpu.ds.getQuadWord(cpu.edx + cpu.edx); //2171
	case M_DS_EDX_EBX: return cpu.ds.getQuadWord(cpu.edx + cpu.ebx); //2172
	case M_DS_EDX_ESP: return cpu.ds.getQuadWord(cpu.edx + cpu.esp); //2173
	case M_DS_EDX_EBP: return cpu.ds.getQuadWord(cpu.edx + cpu.ebp); //2174
	case M_DS_EDX_ESI: return cpu.ds.getQuadWord(cpu.edx + cpu.esi); //2175
	case M_DS_EDX_EDI: return cpu.ds.getQuadWord(cpu.edx + cpu.edi); //2176

	case M_DS_2EBX: return cpu.ds.getQuadWord(cpu.ebx + cpu.ebx); //2177;
	case M_DS_EBX_ESP: return cpu.ds.getQuadWord(cpu.ebx + cpu.esp); //2178
	case M_DS_EBX_EBP: return cpu.ds.getQuadWord(cpu.ebx + cpu.ebp); //2179
	case M_DS_EBX_ESI: return cpu.ds.getQuadWord(cpu.ebx + cpu.esi); //2180
	case M_DS_EBX_EDI: return cpu.ds.getQuadWord(cpu.ebx + cpu.edi); //2181

	case M_DS_ESP: return cpu.ds.getQuadWord(cpu.esp); //2182

	case M_DS_EBP_ESP: return cpu.ds.getQuadWord(cpu.ebp + cpu.esp); //2183;
	case M_DS_EBP_ESI: return cpu.ds.getQuadWord(cpu.ebp + cpu.esi); //2184;
	case M_DS_EBP_EDI: return cpu.ds.getQuadWord(cpu.ebp + cpu.edi); //2185;

	case M_DS_ESI_ESP: return cpu.ds.getQuadWord(cpu.esi + cpu.esp); //2186;
	case M_DS_2ESI: return cpu.ds.getQuadWord(cpu.esi + cpu.esi); //2187;
	case M_DS_ESI_EDI: return cpu.ds.getQuadWord(cpu.esi + cpu.edi); //2188;

	case M_DS_EDI_ESP: return cpu.ds.getQuadWord(cpu.edi + cpu.esp); //2189;
	case M_DS_2EDI: return cpu.ds.getQuadWord(cpu.edi + cpu.edi); //2190;

	case M_DS_3EAX: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.eax); //2191;
	case M_DS_2EAX_ECX: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ecx); //2192;
	case M_DS_2EAX_EDX: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.edx); //2193;
	case M_DS_2EAX_EBX: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ebx); //2194;
	case M_DS_2EAX_ESP: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.esp); //2195;
	case M_DS_2EAX_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + getMicrocode()); //2196;
	case M_DS_2EAX_ESI: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.esi); //2197;
	case M_DS_2EAX_EDI: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.edi); //2198;

	case M_DS_2ECX_EAX: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.eax); //2199;
	case M_DS_3ECX: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_DS_2ECX_EDX: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.edx); //2201;
	case M_DS_2ECX_EBX: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_DS_2ECX_ESP: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.esp); //2203;
	case M_DS_2ECX_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_DS_2ECX_ESI: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.esi); //2205;
	case M_DS_2ECX_EDI: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.edi); //2206;

	case M_DS_2EDX_EAX: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.eax); //2207
	case M_DS_2EDX_ECX: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ecx); //2208
	case M_DS_3EDX: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.edx); //2209
	case M_DS_2EDX_EBX: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ebx); //2210
	case M_DS_2EDX_ESP: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.esp); //2211
	case M_DS_2EDX_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + getMicrocode()); //2212
	case M_DS_2EDX_ESI: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.esi); //2213
	case M_DS_2EDX_EDI: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.edi); //2214

	case M_DS_2EBX_EAX: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.eax); //2215;
	case M_DS_2EBX_ECX: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_DS_2EBX_EDX: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.edx); //2217;
	case M_DS_3EBX: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_DS_2EBX_ESP: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.esp); //2219;
	case M_DS_2EBX_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_DS_2EBX_ESI: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.esi); //2221;
	case M_DS_2EBX_EDI: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.edi); //2222;
	    
	case M_DS_2EBP_EAX: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.eax); //2223;
	case M_DS_2EBP_ECX: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_DS_2EBP_EDX: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.edx); //2225;
	case M_DS_2EBP_EBX: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_DS_2EBP_ESP: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.esp); //2227;
	case M_DS_2EBP_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_DS_2EBP_ESI: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.esi); //2229;
	case M_DS_2EBP_EDI: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.edi); //2230;

	case M_DS_2ESI_EAX: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.eax); //2231;
	case M_DS_2ESI_ECX: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ecx); //2232;
	case M_DS_2ESI_EDX: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.edx); //2233;
	case M_DS_2ESI_EBX: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ebx); //2234;
	case M_DS_2ESI_ESP: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.esp); //2235;
	case M_DS_2ESI_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + getMicrocode()); //2236;
	case M_DS_3ESI: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.esi); //2237;
	case M_DS_2ESI_EDI: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.edi); //2238;

	case M_DS_2EDI_EAX: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.eax); //2239;
	case M_DS_2EDI_ECX: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ecx); //2240;
	case M_DS_2EDI_EDX: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.edx); //2241;
	case M_DS_2EDI_EBX: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ebx); //2242;
	case M_DS_2EDI_ESP: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.esp); //2243;
	case M_DS_2EDI_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + getMicrocode()); //2244;
	case M_DS_2EDI_ESI: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.esi); //2245;
	case M_DS_3EDI: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.edi); //2246;

	case M_DS_5EAX: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.eax); //2247
	case M_DS_4EAX_ECX: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ecx); //2248
	case M_DS_4EAX_EDX: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.edx); //2249
	case M_DS_4EAX_EBX: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ebx); //2250
	case M_DS_4EAX_ESP: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.esp); //2251
	case M_DS_4EAX_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + getMicrocode()); //2252
	case M_DS_4EAX_ESI: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.esi); //2253
	case M_DS_4EAX_EDI: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.edi); //2254

	case M_DS_4ECX_EAX: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.eax); //2255
	case M_DS_5ECX: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ecx); //2256
	case M_DS_4ECX_EDX: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.edx); //2257
	case M_DS_4ECX_EBX: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ebx); //2258
	case M_DS_4ECX_ESP: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.esp); //2259
	case M_DS_4ECX_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + getMicrocode()); //2260
	case M_DS_4ECX_ESI: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.esi); //2261
	case M_DS_4ECX_EDI: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.edi); //2262

	case M_DS_4EDX_EAX: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.eax); //2263
	case M_DS_4EDX_ECX: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ecx); //2264
	case M_DS_5EDX: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.edx); //2265
	case M_DS_4EDX_EBX: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ebx); //2266
	case M_DS_4EDX_ESP: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.esp); //2267
	case M_DS_4EDX_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + getMicrocode()); //2268
	case M_DS_4EDX_ESI: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.esi); //2269
	case M_DS_4EDX_EDI: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.edi); //2270

	case M_DS_4EBX_EAX: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.eax); //2271
	case M_DS_4EBX_ECX: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ecx); //2272
	case M_DS_4EBX_EDX: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.edx); //2273
	case M_DS_5EBX: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ebx); //2274
	case M_DS_4EBX_ESP: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.esp); //2275
	case M_DS_4EBX_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + getMicrocode()); //2276
	case M_DS_4EBX_ESI: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.esi); //2277
	case M_DS_4EBX_EDI: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.edi); //2278

	case M_DS_4EBP_EAX: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.eax); //2279
	case M_DS_4EBP_ECX: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ecx); //2280
	case M_DS_4EBP_EDX: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.edx); //2281
	case M_DS_4EBP_EBX: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ebx); //2282
	case M_DS_4EBP_ESP: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.esp); //2283
	case M_DS_4EBP_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + getMicrocode()); //2284
	case M_DS_4EBP_ESI: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.esi); //2285
	case M_DS_4EBP_EDI: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.edi); //2286

	case M_DS_4ESI_EAX: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.eax); //2287
	case M_DS_4ESI_ECX: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ecx); //2288
	case M_DS_4ESI_EDX: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.edx); //2289
	case M_DS_4ESI_EBX: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ebx); //2290
	case M_DS_4ESI_ESP: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.esp); //2291
	case M_DS_4ESI_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + getMicrocode()); //2292
	case M_DS_5ESI: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.esi); //2293
	case M_DS_4ESI_EDI: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.edi); //2294

	case M_DS_4EDI_EAX: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.eax); //2295;
	case M_DS_4EDI_ECX: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ecx); //2296;
	case M_DS_4EDI_EDX: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.edx); //2297;
	case M_DS_4EDI_EBX: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ebx); //2298;
	case M_DS_4EDI_ESP: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.esp); //2299;
	case M_DS_4EDI_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + getMicrocode()); //2300;
	case M_DS_4EDI_ESI: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.esi); //2301;
	case M_DS_5EDI: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.edi); //2302;

	case M_DS_9EAX: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.eax); //2303;
	case M_DS_8EAX_ECX: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ecx); //2304;
	case M_DS_8EAX_EDX: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.edx); //2305;
	case M_DS_8EAX_EBX: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ebx); //2306;
	case M_DS_8EAX_ESP: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.esp); //2307;
	case M_DS_8EAX_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + getMicrocode()); //2308;
	case M_DS_8EAX_ESI: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.esi); //2309;
	case M_DS_8EAX_EDI: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_DS_8ECX_EAX: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.eax); //2311;
	case M_DS_9ECX: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_DS_8ECX_EDX: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.edx); //2313;
	case M_DS_8ECX_EBX: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_DS_8ECX_ESP: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.esp); //2315;
	case M_DS_8ECX_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_DS_8ECX_ESI: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.esi); //2317;
	case M_DS_8ECX_EDI: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_DS_8EDX_EAX: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.eax); //2319;
	case M_DS_8EDX_ECX: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ecx); //2320;
	case M_DS_9EDX: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.edx); //2321;
	case M_DS_8EDX_EBX: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ebx); //2322;
	case M_DS_8EDX_ESP: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.esp); //2323;
	case M_DS_8EDX_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + getMicrocode()); //2324;
	case M_DS_8EDX_ESI: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.esi); //2325;
	case M_DS_8EDX_EDI: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_DS_8EBX_EAX: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.eax); //2327
	case M_DS_8EBX_ECX: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ecx); //2328
	case M_DS_8EBX_EDX: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.edx); //2329
	case M_DS_9EBX: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ebx); //2330
	case M_DS_8EBX_ESP: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.esp); //2331
	case M_DS_8EBX_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + getMicrocode()); //2332
	case M_DS_8EBX_ESI: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.esi); //2333
	case M_DS_8EBX_EDI: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_DS_8EBP_EAX: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.eax); //2335;
	case M_DS_8EBP_ECX: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_DS_8EBP_EDX: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.edx); //2337;
	case M_DS_8EBP_EBX: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_DS_8EBP_ESP: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.esp); //2339;
	case M_DS_8EBP_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_DS_8EBP_ESI: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.esi); //2341;
	case M_DS_8EBP_EDI: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_DS_8ESI_EAX: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.eax); //2343;
	case M_DS_8ESI_ECX: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ecx); //2344;
	case M_DS_8ESI_EDX: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.edx); //2345;
	case M_DS_8ESI_EBX: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ebx); //2346;
	case M_DS_8ESI_ESP: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.esp); //2347;
	case M_DS_8ESI_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + getMicrocode()); //2348;
	case M_DS_9ESI: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.esi); //2349;
	case M_DS_8ESI_EDI: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.edi); //2350;

	case M_DS_8EDI_EAX: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.eax); //2351;
	case M_DS_8EDI_ECX: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ecx); //2352;
	case M_DS_8EDI_EDX: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.edx); //2353;
	case M_DS_8EDI_EBX: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ebx); //2354;
	case M_DS_8EDI_ESP: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.esp); //2355;
	case M_DS_8EDI_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + getMicrocode()); //2356;
	case M_DS_8EDI_ESI: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.esi); //2357;
	case M_DS_9EDI: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.edi); //2358;

	case M_DS_2EAX_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_DS_EAX_ECX_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_DS_EAX_EDX_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_DS_EAX_EBX_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_DS_EAX_ESP_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_DS_EAX_EBP_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_DS_EAX_ESI_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_DS_EAX_EDI_IB: return cpu.ds.getQuadWord(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_DS_2ECX_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_DS_ECX_EDX_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_DS_ECX_EBX_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_DS_ECX_ESP_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_DS_ECX_EBP_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_DS_ECX_ESI_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_DS_ECX_EDI_IB: return cpu.ds.getQuadWord(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_DS_2EDX_IB: return cpu.ds.getQuadWord(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_DS_EDX_EBX_IB: return cpu.ds.getQuadWord(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_DS_EDX_ESP_IB: return cpu.ds.getQuadWord(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_DS_EDX_EBP_IB: return cpu.ds.getQuadWord(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_DS_EDX_ESI_IB: return cpu.ds.getQuadWord(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_DS_EDX_EDI_IB: return cpu.ds.getQuadWord(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_DS_2EBX_IB: return cpu.ds.getQuadWord(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_DS_EBX_ESP_IB: return cpu.ds.getQuadWord(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_DS_EBX_EBP_IB: return cpu.ds.getQuadWord(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_DS_EBX_ESI_IB: return cpu.ds.getQuadWord(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_DS_EBX_EDI_IB: return cpu.ds.getQuadWord(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_DS_ESP_IB: return cpu.ds.getQuadWord(cpu.esp + getMicrocode()); //2385

	case M_DS_EBP_ESP_IB: return cpu.ds.getQuadWord(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_DS_2EBP_IB: return cpu.ds.getQuadWord(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_DS_EBP_ESI_IB: return cpu.ds.getQuadWord(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_DS_EBP_EDI_IB: return cpu.ds.getQuadWord(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_DS_ESI_ESP_IB: return cpu.ds.getQuadWord(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_DS_2ESI_IB: return cpu.ds.getQuadWord(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_DS_ESI_EDI_IB: return cpu.ds.getQuadWord(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_DS_EDI_ESP_IB: return cpu.ds.getQuadWord(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_DS_2EDI_IB: return cpu.ds.getQuadWord(cpu.edi + cpu.edi + getMicrocode()); //2394;

	case M_DS_3EAX_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_DS_2EAX_ECX_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_DS_2EAX_EDX_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_DS_2EAX_EBX_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_DS_2EAX_ESP_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_DS_2EAX_EBP_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_DS_2EAX_ESI_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_DS_2EAX_EDI_IB: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_DS_2ECX_EAX_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_DS_3ECX_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_DS_2ECX_EDX_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_DS_2ECX_EBX_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_DS_2ECX_ESP_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_DS_2ECX_EBP_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_DS_2ECX_ESI_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_DS_2ECX_EDI_IB: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_DS_2EDX_EAX_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_DS_2EDX_ECX_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_DS_3EDX_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_DS_2EDX_EBX_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_DS_2EDX_ESP_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_DS_2EDX_EBP_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_DS_2EDX_ESI_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_DS_2EDX_EDI_IB: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;
	    
	case M_DS_2EBX_EAX_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_DS_2EBX_ECX_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_DS_2EBX_EDX_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_DS_3EBX_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_DS_2EBX_ESP_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_DS_2EBX_EBP_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_DS_2EBX_ESI_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_DS_2EBX_EDI_IB: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_DS_2EBP_EAX_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_DS_2EBP_ECX_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_DS_2EBP_EDX_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_DS_2EBP_EBX_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_DS_2EBP_ESP_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_DS_3EBP_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_DS_2EBP_ESI_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_DS_2EBP_EDI_IB: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_DS_2ESI_EAX_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_DS_2ESI_ECX_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_DS_2ESI_EDX_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_DS_2ESI_EBX_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_DS_2ESI_ESP_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_DS_2ESI_EBP_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_DS_3ESI_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_DS_2ESI_EDI_IB: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_DS_2EDI_EAX_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_DS_2EDI_ECX_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_DS_2EDI_EDX_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_DS_2EDI_EBX_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_DS_2EDI_ESP_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_DS_2EDI_EBP_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_DS_2EDI_ESI_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_DS_3EDI_IB: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_DS_5EAX_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_DS_4EAX_ECX_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_DS_4EAX_EDX_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_DS_4EAX_EBX_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_DS_4EAX_ESP_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_DS_4EAX_EBP_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_DS_4EAX_ESI_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_DS_4EAX_EDI_IB: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_DS_4ECX_EAX_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_DS_5ECX_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_DS_4ECX_EDX_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_DS_4ECX_EBX_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_DS_4ECX_ESP_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_DS_4ECX_EBP_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_DS_4ECX_ESI_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_DS_4ECX_EDI_IB: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_DS_4EDX_EAX_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_DS_4EDX_ECX_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_DS_5EDX_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_DS_4EDX_EBX_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_DS_4EDX_ESP_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_DS_4EDX_EBP_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_DS_4EDX_ESI_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_DS_4EDX_EDI_IB: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_DS_4EBX_EAX_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_DS_4EBX_ECX_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_DS_4EBX_EDX_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_DS_5EBX_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_DS_4EBX_ESP_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_DS_4EBX_EBP_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_DS_4EBX_ESI_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_DS_4EBX_EDI_IB: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_DS_4EBP_EAX_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_DS_4EBP_ECX_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_DS_4EBP_EDX_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_DS_4EBP_EBX_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_DS_4EBP_ESP_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_DS_5EBP_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_DS_4EBP_ESI_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_DS_4EBP_EDI_IB: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_DS_4ESI_EAX_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_DS_4ESI_ECX_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_DS_4ESI_EDX_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_DS_4ESI_EBX_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_DS_4ESI_ESP_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_DS_4ESI_EBP_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_DS_5ESI_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_DS_4ESI_EDI_IB: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_DS_4EDI_EAX_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_DS_4EDI_ECX_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_DS_4EDI_EDX_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_DS_4EDI_EBX_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_DS_4EDI_ESP_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_DS_4EDI_EBP_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_DS_4EDI_ESI_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_DS_5EDI_IB: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_DS_9EAX_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_DS_8EAX_ECX_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_DS_8EAX_EDX_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_DS_8EAX_EBX_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_DS_8EAX_ESP_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_DS_8EAX_EBP_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_DS_8EAX_ESI_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_DS_8EAX_EDI_IB: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_DS_8ECX_EAX_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_DS_9ECX_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_DS_8ECX_EDX_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_DS_8ECX_EBX_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_DS_8ECX_ESP_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_DS_8ECX_EBP_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_DS_8ECX_ESI_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_DS_8ECX_EDI_IB: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_DS_8EDX_EAX_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_DS_8EDX_ECX_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_DS_9EDX_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_DS_8EDX_EBX_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_DS_8EDX_ESP_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_DS_8EDX_EBP_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_DS_8EDX_ESI_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_DS_8EDX_EDI_IB: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_DS_8EBX_EAX_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_DS_8EBX_ECX_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_DS_8EBX_EDX_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_DS_9EBX_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_DS_8EBX_ESP_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_DS_8EBX_EBP_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_DS_8EBX_ESI_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_DS_8EBX_EDI_IB: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_DS_8EBP_EAX_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_DS_8EBP_ECX_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_DS_8EBP_EDX_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_DS_8EBP_EBX_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_DS_8EBP_ESP_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_DS_9EBP_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_DS_8EBP_ESI_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_DS_8EBP_EDI_IB: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_DS_8ESI_EAX_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_DS_8ESI_ECX_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_DS_8ESI_EDX_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_DS_8ESI_EBX_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_DS_8ESI_ESP_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_DS_8ESI_EBP_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_DS_9ESI_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_DS_8ESI_EDI_IB: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_DS_8EDI_EAX_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_DS_8EDI_ECX_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_DS_8EDI_EDX_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_DS_8EDI_EBX_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_DS_8EDI_ESP_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_DS_8EDI_EBP_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_DS_8EDI_ESI_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_DS_9EDI_IB: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_DS_EAX_ECX_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_DS_EAX_EDX_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_DS_EAX_EBX_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_DS_EAX_ESP_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_DS_EAX_EBP_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_DS_EAX_ESI_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_DS_EAX_EDI_ID: return cpu.ds.getQuadWord(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_DS_ECX_EDX_ID: return cpu.ds.getQuadWord(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_DS_ECX_EBX_ID: return cpu.ds.getQuadWord(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_DS_ECX_ESP_ID: return cpu.ds.getQuadWord(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_DS_ECX_EBP_ID: return cpu.ds.getQuadWord(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_DS_ECX_ESI_ID: return cpu.ds.getQuadWord(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_DS_ECX_EDI_ID: return cpu.ds.getQuadWord(cpu.ecx + cpu.edi + getMicrocode()); //2575;

	case M_DS_EDX_EBX_ID: return cpu.ds.getQuadWord(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_DS_EDX_ESP_ID: return cpu.ds.getQuadWord(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_DS_EDX_EBP_ID: return cpu.ds.getQuadWord(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_DS_EDX_ESI_ID: return cpu.ds.getQuadWord(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_DS_EDX_EDI_ID: return cpu.ds.getQuadWord(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_DS_EBX_ESP_ID: return cpu.ds.getQuadWord(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_DS_EBX_EBP_ID: return cpu.ds.getQuadWord(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_DS_EBX_ESI_ID: return cpu.ds.getQuadWord(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_DS_EBX_EDI_ID: return cpu.ds.getQuadWord(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_DS_ESP_ID: return cpu.ds.getQuadWord(cpu.esp + getMicrocode()); //2585

	case M_DS_EBP_ESP_ID: return cpu.ds.getQuadWord(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_DS_EBP_ESI_ID: return cpu.ds.getQuadWord(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_DS_EBP_EDI_ID: return cpu.ds.getQuadWord(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_DS_ESI_ESP_ID: return cpu.ds.getQuadWord(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_DS_ESI_EDI_ID: return cpu.ds.getQuadWord(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_DS_EDI_ESP_ID: return cpu.ds.getQuadWord(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_DS_3EAX_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_DS_2EAX_ECX_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_DS_2EAX_EDX_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_DS_2EAX_EBX_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_DS_2EAX_ESP_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_DS_2EAX_EBP_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_DS_2EAX_ESI_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_DS_2EAX_EDI_ID: return cpu.ds.getQuadWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_DS_2ECX_EAX_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_DS_3ECX_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_DS_2ECX_EDX_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_DS_2ECX_EBX_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_DS_2ECX_ESP_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_DS_2ECX_EBP_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_DS_2ECX_ESI_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_DS_2ECX_EDI_ID: return cpu.ds.getQuadWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_DS_2EDX_EAX_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_DS_2EDX_ECX_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_DS_3EDX_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_DS_2EDX_EBX_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_DS_2EDX_ESP_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_DS_2EDX_EBP_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_DS_2EDX_ESI_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_DS_2EDX_EDI_ID: return cpu.ds.getQuadWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_DS_2EBX_EAX_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_DS_2EBX_ECX_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_DS_2EBX_EDX_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_DS_3EBX_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_DS_2EBX_ESP_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_DS_2EBX_EBP_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_DS_2EBX_ESI_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_DS_2EBX_EDI_ID: return cpu.ds.getQuadWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_DS_2EBP_EAX_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_DS_2EBP_ECX_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_DS_2EBP_EDX_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_DS_2EBP_EBX_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_DS_2EBP_ESP_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_DS_3EBP_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_DS_2EBP_ESI_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_DS_2EBP_EDI_ID: return cpu.ds.getQuadWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_DS_2ESI_EAX_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_DS_2ESI_ECX_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_DS_2ESI_EDX_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_DS_2ESI_EBX_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_DS_2ESI_ESP_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_DS_2ESI_EBP_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_DS_3ESI_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_DS_2ESI_EDI_ID: return cpu.ds.getQuadWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_DS_2EDI_EAX_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_DS_2EDI_ECX_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_DS_2EDI_EDX_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_DS_2EDI_EBX_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_DS_2EDI_ESP_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_DS_2EDI_EBP_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_DS_2EDI_ESI_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_DS_3EDI_ID: return cpu.ds.getQuadWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_DS_5EAX_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_DS_4EAX_ECX_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_DS_4EAX_EDX_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_DS_4EAX_EBX_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_DS_4EAX_ESP_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_DS_4EAX_EBP_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_DS_4EAX_ESI_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_DS_4EAX_EDI_ID: return cpu.ds.getQuadWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_DS_4ECX_EAX_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_DS_5ECX_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_DS_4ECX_EDX_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_DS_4ECX_EBX_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_DS_4ECX_ESP_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_DS_4ECX_EBP_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_DS_4ECX_ESI_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_DS_4ECX_EDI_ID: return cpu.ds.getQuadWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_DS_4EDX_EAX_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_DS_4EDX_ECX_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_DS_5EDX_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_DS_4EDX_EBX_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_DS_4EDX_ESP_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_DS_4EDX_EBP_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_DS_4EDX_ESI_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_DS_4EDX_EDI_ID: return cpu.ds.getQuadWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_DS_4EBX_EAX_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_DS_4EBX_ECX_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_DS_4EBX_EDX_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_DS_5EBX_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_DS_4EBX_ESP_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_DS_4EBX_EBP_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_DS_4EBX_ESI_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_DS_4EBX_EDI_ID: return cpu.ds.getQuadWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_DS_4EBP_EAX_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_DS_4EBP_ECX_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_DS_4EBP_EDX_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_DS_4EBP_EBX_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_DS_4EBP_ESP_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_DS_5EBP_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_DS_4EBP_ESI_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_DS_4EBP_EDI_ID: return cpu.ds.getQuadWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_DS_4ESI_EAX_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_DS_4ESI_ECX_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_DS_4ESI_EDX_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_DS_4ESI_EBX_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_DS_4ESI_ESP_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_DS_4ESI_EBP_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_DS_5ESI_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_DS_4ESI_EDI_ID: return cpu.ds.getQuadWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_DS_4EDI_EAX_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_DS_4EDI_ECX_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_DS_4EDI_EDX_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_DS_4EDI_EBX_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_DS_4EDI_ESP_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_DS_4EDI_EBP_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_DS_4EDI_ESI_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_DS_5EDI_ID: return cpu.ds.getQuadWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_DS_9EAX_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_DS_8EAX_ECX_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_DS_8EAX_EDX_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_DS_8EAX_EBX_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_DS_8EAX_ESP_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_DS_8EAX_EBP_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_DS_8EAX_ESI_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_DS_8EAX_EDI_ID: return cpu.ds.getQuadWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_DS_8ECX_EAX_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_DS_9ECX_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_DS_8ECX_EDX_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_DS_8ECX_EBX_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_DS_8ECX_ESP_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_DS_8ECX_EBP_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_DS_8ECX_ESI_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_DS_8ECX_EDI_ID: return cpu.ds.getQuadWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_DS_8EDX_EAX_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_DS_8EDX_ECX_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_DS_9EDX_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_DS_8EDX_EBX_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_DS_8EDX_ESP_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_DS_8EDX_EBP_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_DS_8EDX_ESI_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_DS_8EDX_EDI_ID: return cpu.ds.getQuadWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_DS_8EBX_EAX_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_DS_8EBX_ECX_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_DS_8EBX_EDX_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_DS_9EBX_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_DS_8EBX_ESP_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_DS_8EBX_EBP_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_DS_8EBX_ESI_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_DS_8EBX_EDI_ID: return cpu.ds.getQuadWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_DS_8EBP_EAX_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_DS_8EBP_ECX_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_DS_8EBP_EDX_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_DS_8EBP_EBX_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_DS_8EBP_ESP_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_DS_9EBP_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_DS_8EBP_ESI_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_DS_8EBP_EDI_ID: return cpu.ds.getQuadWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_DS_8ESI_EAX_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_DS_8ESI_ECX_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_DS_8ESI_EDX_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_DS_8ESI_EBX_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_DS_8ESI_ESP_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_DS_8ESI_EBP_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_DS_9ESI_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_DS_8ESI_EDI_ID: return cpu.ds.getQuadWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_DS_8EDI_EAX_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_DS_8EDI_ECX_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_DS_8EDI_EDX_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_DS_8EDI_EBX_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_DS_8EDI_ESP_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_DS_8EDI_EBP_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_DS_8EDI_ESI_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_DS_9EDI_ID: return cpu.ds.getQuadWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;

	case M_SS_2EAX: return cpu.ss.getQuadWord(cpu.eax + cpu.eax); //2156
	case M_SS_EAX_ECX: return cpu.ss.getQuadWord(cpu.eax + cpu.ecx); //2157
	case M_SS_EAX_EDX: return cpu.ss.getQuadWord(cpu.eax + cpu.edx); //2158
	case M_SS_EAX_EBX: return cpu.ss.getQuadWord(cpu.eax + cpu.ebx); //2159
	case M_SS_EAX_ESP: return cpu.ss.getQuadWord(cpu.eax + cpu.esp); //2160
	case M_SS_EAX_EBP: return cpu.ss.getQuadWord(cpu.eax + cpu.ebp); //2161
	case M_SS_EAX_ESI: return cpu.ss.getQuadWord(cpu.eax + cpu.esi); //2162
	case M_SS_EAX_EDI: return cpu.ss.getQuadWord(cpu.eax + cpu.edi); //2163

	case M_SS_2ECX: return cpu.ss.getQuadWord(cpu.ecx + cpu.ecx); //2164
	case M_SS_ECX_EDX: return cpu.ss.getQuadWord(cpu.ecx + cpu.edx); //2165
	case M_SS_ECX_EBX: return cpu.ss.getQuadWord(cpu.ecx + cpu.ebx); //2166
	case M_SS_ECX_ESP: return cpu.ss.getQuadWord(cpu.ecx + cpu.esp); //2167
	case M_SS_ECX_EBP: return cpu.ss.getQuadWord(cpu.ecx + cpu.ebp); //2168
	case M_SS_ECX_ESI: return cpu.ss.getQuadWord(cpu.ecx + cpu.esi); //2169
	case M_SS_ECX_EDI: return cpu.ss.getQuadWord(cpu.ecx + cpu.edi); //2170

	case M_SS_2EDX: return cpu.ss.getQuadWord(cpu.edx + cpu.edx); //2171
	case M_SS_EDX_EBX: return cpu.ss.getQuadWord(cpu.edx + cpu.ebx); //2172
	case M_SS_EDX_ESP: return cpu.ss.getQuadWord(cpu.edx + cpu.esp); //2173
	case M_SS_EDX_EBP: return cpu.ss.getQuadWord(cpu.edx + cpu.ebp); //2174
	case M_SS_EDX_ESI: return cpu.ss.getQuadWord(cpu.edx + cpu.esi); //2175
	case M_SS_EDX_EDI: return cpu.ss.getQuadWord(cpu.edx + cpu.edi); //2176

	case M_SS_2EBX: return cpu.ss.getQuadWord(cpu.ebx + cpu.ebx); //2177;
	case M_SS_EBX_ESP: return cpu.ss.getQuadWord(cpu.ebx + cpu.esp); //2178
	case M_SS_EBX_EBP: return cpu.ss.getQuadWord(cpu.ebx + cpu.ebp); //2179
	case M_SS_EBX_ESI: return cpu.ss.getQuadWord(cpu.ebx + cpu.esi); //2180
	case M_SS_EBX_EDI: return cpu.ss.getQuadWord(cpu.ebx + cpu.edi); //2181

	case M_SS_ESP: return cpu.ss.getQuadWord(cpu.esp); //2182

	case M_SS_EBP_ESP: return cpu.ss.getQuadWord(cpu.ebp + cpu.esp); //2183;
	case M_SS_EBP_ESI: return cpu.ss.getQuadWord(cpu.ebp + cpu.esi); //2184;
	case M_SS_EBP_EDI: return cpu.ss.getQuadWord(cpu.ebp + cpu.edi); //2185;

	case M_SS_ESI_ESP: return cpu.ss.getQuadWord(cpu.esi + cpu.esp); //2186;
	case M_SS_2ESI: return cpu.ss.getQuadWord(cpu.esi + cpu.esi); //2187;
	case M_SS_ESI_EDI: return cpu.ss.getQuadWord(cpu.esi + cpu.edi); //2188;

	case M_SS_EDI_ESP: return cpu.ss.getQuadWord(cpu.edi + cpu.esp); //2189;
	case M_SS_2EDI: return cpu.ss.getQuadWord(cpu.edi + cpu.edi); //2190;

	case M_SS_3EAX: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.eax); //2191;
	case M_SS_2EAX_ECX: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ecx); //2192;
	case M_SS_2EAX_EDX: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.edx); //2193;
	case M_SS_2EAX_EBX: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ebx); //2194;
	case M_SS_2EAX_ESP: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.esp); //2195;
	case M_SS_2EAX_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + getMicrocode()); //2196;
	case M_SS_2EAX_ESI: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.esi); //2197;
	case M_SS_2EAX_EDI: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.edi); //2198;

	case M_SS_2ECX_EAX: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.eax); //2199;
	case M_SS_3ECX: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ecx); //2200;
	case M_SS_2ECX_EDX: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.edx); //2201;
	case M_SS_2ECX_EBX: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ebx); //2202;
	case M_SS_2ECX_ESP: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.esp); //2203;
	case M_SS_2ECX_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + getMicrocode()); //2204;
	case M_SS_2ECX_ESI: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.esi); //2205;
	case M_SS_2ECX_EDI: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.edi); //2206;

	case M_SS_2EDX_EAX: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.eax); //2207
	case M_SS_2EDX_ECX: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ecx); //2208
	case M_SS_3EDX: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.edx); //2209
	case M_SS_2EDX_EBX: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ebx); //2210
	case M_SS_2EDX_ESP: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.esp); //2211
	case M_SS_2EDX_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + getMicrocode()); //2212
	case M_SS_2EDX_ESI: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.esi); //2213
	case M_SS_2EDX_EDI: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.edi); //2214

	case M_SS_2EBX_EAX: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.eax); //2215;
	case M_SS_2EBX_ECX: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ecx); //2216;
	case M_SS_2EBX_EDX: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.edx); //2217;
	case M_SS_3EBX: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ebx); //2218;
	case M_SS_2EBX_ESP: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.esp); //2219;
	case M_SS_2EBX_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + getMicrocode()); //2220;
	case M_SS_2EBX_ESI: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.esi); //2221;
	case M_SS_2EBX_EDI: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.edi); //2222;
	    
	case M_SS_2EBP_EAX: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.eax); //2223;
	case M_SS_2EBP_ECX: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ecx); //2224;
	case M_SS_2EBP_EDX: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.edx); //2225;
	case M_SS_2EBP_EBX: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ebx); //2226;
	case M_SS_2EBP_ESP: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.esp); //2227;
	case M_SS_2EBP_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + getMicrocode()); //2228;
	case M_SS_2EBP_ESI: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.esi); //2229;
	case M_SS_2EBP_EDI: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.edi); //2230;

	case M_SS_2ESI_EAX: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.eax); //2231;
	case M_SS_2ESI_ECX: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ecx); //2232;
	case M_SS_2ESI_EDX: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.edx); //2233;
	case M_SS_2ESI_EBX: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ebx); //2234;
	case M_SS_2ESI_ESP: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.esp); //2235;
	case M_SS_2ESI_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + getMicrocode()); //2236;
	case M_SS_3ESI: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.esi); //2237;
	case M_SS_2ESI_EDI: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.edi); //2238;

	case M_SS_2EDI_EAX: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.eax); //2239;
	case M_SS_2EDI_ECX: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ecx); //2240;
	case M_SS_2EDI_EDX: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.edx); //2241;
	case M_SS_2EDI_EBX: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ebx); //2242;
	case M_SS_2EDI_ESP: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.esp); //2243;
	case M_SS_2EDI_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + getMicrocode()); //2244;
	case M_SS_2EDI_ESI: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.esi); //2245;
	case M_SS_3EDI: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.edi); //2246;

	case M_SS_5EAX: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.eax); //2247
	case M_SS_4EAX_ECX: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ecx); //2248
	case M_SS_4EAX_EDX: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.edx); //2249
	case M_SS_4EAX_EBX: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ebx); //2250
	case M_SS_4EAX_ESP: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.esp); //2251
	case M_SS_4EAX_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + getMicrocode()); //2252
	case M_SS_4EAX_ESI: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.esi); //2253
	case M_SS_4EAX_EDI: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.edi); //2254

	case M_SS_4ECX_EAX: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.eax); //2255
	case M_SS_5ECX: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ecx); //2256
	case M_SS_4ECX_EDX: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.edx); //2257
	case M_SS_4ECX_EBX: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ebx); //2258
	case M_SS_4ECX_ESP: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.esp); //2259
	case M_SS_4ECX_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + getMicrocode()); //2260
	case M_SS_4ECX_ESI: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.esi); //2261
	case M_SS_4ECX_EDI: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.edi); //2262

	case M_SS_4EDX_EAX: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.eax); //2263
	case M_SS_4EDX_ECX: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ecx); //2264
	case M_SS_5EDX: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.edx); //2265
	case M_SS_4EDX_EBX: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ebx); //2266
	case M_SS_4EDX_ESP: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.esp); //2267
	case M_SS_4EDX_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + getMicrocode()); //2268
	case M_SS_4EDX_ESI: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.esi); //2269
	case M_SS_4EDX_EDI: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.edi); //2270

	case M_SS_4EBX_EAX: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.eax); //2271
	case M_SS_4EBX_ECX: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ecx); //2272
	case M_SS_4EBX_EDX: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.edx); //2273
	case M_SS_5EBX: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ebx); //2274
	case M_SS_4EBX_ESP: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.esp); //2275
	case M_SS_4EBX_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + getMicrocode()); //2276
	case M_SS_4EBX_ESI: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.esi); //2277
	case M_SS_4EBX_EDI: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.edi); //2278

	case M_SS_4EBP_EAX: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.eax); //2279
	case M_SS_4EBP_ECX: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ecx); //2280
	case M_SS_4EBP_EDX: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.edx); //2281
	case M_SS_4EBP_EBX: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ebx); //2282
	case M_SS_4EBP_ESP: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.esp); //2283
	case M_SS_4EBP_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + getMicrocode()); //2284
	case M_SS_4EBP_ESI: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.esi); //2285
	case M_SS_4EBP_EDI: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.edi); //2286

	case M_SS_4ESI_EAX: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.eax); //2287
	case M_SS_4ESI_ECX: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ecx); //2288
	case M_SS_4ESI_EDX: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.edx); //2289
	case M_SS_4ESI_EBX: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ebx); //2290
	case M_SS_4ESI_ESP: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.esp); //2291
	case M_SS_4ESI_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + getMicrocode()); //2292
	case M_SS_5ESI: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.esi); //2293
	case M_SS_4ESI_EDI: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.edi); //2294

	case M_SS_4EDI_EAX: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.eax); //2295;
	case M_SS_4EDI_ECX: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ecx); //2296;
	case M_SS_4EDI_EDX: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.edx); //2297;
	case M_SS_4EDI_EBX: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ebx); //2298;
	case M_SS_4EDI_ESP: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.esp); //2299;
	case M_SS_4EDI_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + getMicrocode()); //2300;
	case M_SS_4EDI_ESI: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.esi); //2301;
	case M_SS_5EDI: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.edi); //2302;

	case M_SS_9EAX: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.eax); //2303;
	case M_SS_8EAX_ECX: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ecx); //2304;
	case M_SS_8EAX_EDX: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.edx); //2305;
	case M_SS_8EAX_EBX: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ebx); //2306;
	case M_SS_8EAX_ESP: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.esp); //2307;
	case M_SS_8EAX_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + getMicrocode()); //2308;
	case M_SS_8EAX_ESI: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.esi); //2309;
	case M_SS_8EAX_EDI: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.edi); //2310;
	    
	case M_SS_8ECX_EAX: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.eax); //2311;
	case M_SS_9ECX: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ecx); //2312;
	case M_SS_8ECX_EDX: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.edx); //2313;
	case M_SS_8ECX_EBX: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ebx); //2314;
	case M_SS_8ECX_ESP: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.esp); //2315;
	case M_SS_8ECX_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + getMicrocode()); //2316;
	case M_SS_8ECX_ESI: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.esi); //2317;
	case M_SS_8ECX_EDI: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.edi); //2318;
	    
	case M_SS_8EDX_EAX: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.eax); //2319;
	case M_SS_8EDX_ECX: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ecx); //2320;
	case M_SS_9EDX: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.edx); //2321;
	case M_SS_8EDX_EBX: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ebx); //2322;
	case M_SS_8EDX_ESP: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.esp); //2323;
	case M_SS_8EDX_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + getMicrocode()); //2324;
	case M_SS_8EDX_ESI: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.esi); //2325;
	case M_SS_8EDX_EDI: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.edi); //2326;
	    
	case M_SS_8EBX_EAX: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.eax); //2327
	case M_SS_8EBX_ECX: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ecx); //2328
	case M_SS_8EBX_EDX: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.edx); //2329
	case M_SS_9EBX: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ebx); //2330
	case M_SS_8EBX_ESP: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.esp); //2331
	case M_SS_8EBX_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + getMicrocode()); //2332
	case M_SS_8EBX_ESI: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.esi); //2333
	case M_SS_8EBX_EDI: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.edi); //2334
	    
	case M_SS_8EBP_EAX: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.eax); //2335;
	case M_SS_8EBP_ECX: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ecx); //2336;
	case M_SS_8EBP_EDX: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.edx); //2337;
	case M_SS_8EBP_EBX: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ebx); //2338;
	case M_SS_8EBP_ESP: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.esp); //2339;
	case M_SS_8EBP_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + getMicrocode()); //2340;
	case M_SS_8EBP_ESI: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.esi); //2341;
	case M_SS_8EBP_EDI: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.edi); //2342;
	    
	case M_SS_8ESI_EAX: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.eax); //2343;
	case M_SS_8ESI_ECX: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ecx); //2344;
	case M_SS_8ESI_EDX: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.edx); //2345;
	case M_SS_8ESI_EBX: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ebx); //2346;
	case M_SS_8ESI_ESP: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.esp); //2347;
	case M_SS_8ESI_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + getMicrocode()); //2348;
	case M_SS_9ESI: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.esi); //2349;
	case M_SS_8ESI_EDI: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.edi); //2350;

	case M_SS_8EDI_EAX: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.eax); //2351;
	case M_SS_8EDI_ECX: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ecx); //2352;
	case M_SS_8EDI_EDX: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.edx); //2353;
	case M_SS_8EDI_EBX: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ebx); //2354;
	case M_SS_8EDI_ESP: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.esp); //2355;
	case M_SS_8EDI_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + getMicrocode()); //2356;
	case M_SS_8EDI_ESI: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.esi); //2357;
	case M_SS_9EDI: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.edi); //2358;

	case M_SS_2EAX_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.eax + getMicrocode()); //2359
	case M_SS_EAX_ECX_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.ecx + getMicrocode()); //2360
	case M_SS_EAX_EDX_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.edx + getMicrocode()); //2361
	case M_SS_EAX_EBX_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.ebx + getMicrocode()); //2362
	case M_SS_EAX_ESP_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.esp + getMicrocode()); //2363
	case M_SS_EAX_EBP_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.ebp + getMicrocode()); //2364
	case M_SS_EAX_ESI_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.esi + getMicrocode()); //2365
	case M_SS_EAX_EDI_IB: return cpu.ss.getQuadWord(cpu.eax + cpu.edi + getMicrocode()); //2366

	case M_SS_2ECX_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.ecx + getMicrocode()); //2367
	case M_SS_ECX_EDX_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.edx + getMicrocode()); //2368
	case M_SS_ECX_EBX_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.ebx + getMicrocode()); //2369
	case M_SS_ECX_ESP_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.esp + getMicrocode()); //2370
	case M_SS_ECX_EBP_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.ebp + getMicrocode()); //2371
	case M_SS_ECX_ESI_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.esi + getMicrocode()); //2372
	case M_SS_ECX_EDI_IB: return cpu.ss.getQuadWord(cpu.ecx + cpu.edi + getMicrocode()); //2373

	case M_SS_2EDX_IB: return cpu.ss.getQuadWord(cpu.edx + cpu.edx + getMicrocode()); //2374
	case M_SS_EDX_EBX_IB: return cpu.ss.getQuadWord(cpu.edx + cpu.ebx + getMicrocode()); //2375
	case M_SS_EDX_ESP_IB: return cpu.ss.getQuadWord(cpu.edx + cpu.esp + getMicrocode()); //2376
	case M_SS_EDX_EBP_IB: return cpu.ss.getQuadWord(cpu.edx + cpu.ebp + getMicrocode()); //2377
	case M_SS_EDX_ESI_IB: return cpu.ss.getQuadWord(cpu.edx + cpu.esi + getMicrocode()); //2378
	case M_SS_EDX_EDI_IB: return cpu.ss.getQuadWord(cpu.edx + cpu.edi + getMicrocode()); //2379

	case M_SS_2EBX_IB: return cpu.ss.getQuadWord(cpu.ebx + cpu.ebx + getMicrocode()); //2380
	case M_SS_EBX_ESP_IB: return cpu.ss.getQuadWord(cpu.ebx + cpu.esp + getMicrocode()); //2381
	case M_SS_EBX_EBP_IB: return cpu.ss.getQuadWord(cpu.ebx + cpu.ebp + getMicrocode()); //2382
	case M_SS_EBX_ESI_IB: return cpu.ss.getQuadWord(cpu.ebx + cpu.esi + getMicrocode()); //2383
	case M_SS_EBX_EDI_IB: return cpu.ss.getQuadWord(cpu.ebx + cpu.edi + getMicrocode()); //2384

	case M_SS_ESP_IB: return cpu.ss.getQuadWord(cpu.esp + getMicrocode()); //2385

	case M_SS_EBP_ESP_IB: return cpu.ss.getQuadWord(cpu.ebp + cpu.esp + getMicrocode()); //2386;
	case M_SS_2EBP_IB: return cpu.ss.getQuadWord(cpu.ebp + cpu.ebp + getMicrocode()); //2387;
	case M_SS_EBP_ESI_IB: return cpu.ss.getQuadWord(cpu.ebp + cpu.esi + getMicrocode()); //2388;
	case M_SS_EBP_EDI_IB: return cpu.ss.getQuadWord(cpu.ebp + cpu.edi + getMicrocode()); //2389;

	case M_SS_ESI_ESP_IB: return cpu.ss.getQuadWord(cpu.esi + cpu.esp + getMicrocode()); //2390
	case M_SS_2ESI_IB: return cpu.ss.getQuadWord(cpu.esi + cpu.esi + getMicrocode()); //2391
	case M_SS_ESI_EDI_IB: return cpu.ss.getQuadWord(cpu.esi + cpu.edi + getMicrocode()); //2392

	case M_SS_EDI_ESP_IB: return cpu.ss.getQuadWord(cpu.edi + cpu.esp + getMicrocode()); //2393;
	case M_SS_2EDI_IB: return cpu.ss.getQuadWord(cpu.edi + cpu.edi + getMicrocode()); //2394;

	case M_SS_3EAX_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2395;
	case M_SS_2EAX_ECX_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2396;
	case M_SS_2EAX_EDX_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2397;
	case M_SS_2EAX_EBX_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2398;
	case M_SS_2EAX_ESP_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2399;
	case M_SS_2EAX_EBP_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2400;
	case M_SS_2EAX_ESI_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2401;
	case M_SS_2EAX_EDI_IB: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2402;
	    
	case M_SS_2ECX_EAX_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2403;
	case M_SS_3ECX_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2404;
	case M_SS_2ECX_EDX_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2405;
	case M_SS_2ECX_EBX_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2406;
	case M_SS_2ECX_ESP_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2407;
	case M_SS_2ECX_EBP_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2408;
	case M_SS_2ECX_ESI_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2409;
	case M_SS_2ECX_EDI_IB: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2410;

	case M_SS_2EDX_EAX_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2411;
	case M_SS_2EDX_ECX_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2412;
	case M_SS_3EDX_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2413;
	case M_SS_2EDX_EBX_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2414;
	case M_SS_2EDX_ESP_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2415;
	case M_SS_2EDX_EBP_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2416;
	case M_SS_2EDX_ESI_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2417;
	case M_SS_2EDX_EDI_IB: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2418;
	    
	case M_SS_2EBX_EAX_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2419;
	case M_SS_2EBX_ECX_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2420;
	case M_SS_2EBX_EDX_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2421;
	case M_SS_3EBX_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2422;
	case M_SS_2EBX_ESP_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2423;
	case M_SS_2EBX_EBP_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2424;
	case M_SS_2EBX_ESI_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2425;
	case M_SS_2EBX_EDI_IB: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2426;
	    
	case M_SS_2EBP_EAX_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2427;
	case M_SS_2EBP_ECX_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2428;
	case M_SS_2EBP_EDX_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2429;
	case M_SS_2EBP_EBX_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2430;
	case M_SS_2EBP_ESP_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2431;
	case M_SS_3EBP_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2432;
	case M_SS_2EBP_ESI_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2433;
	case M_SS_2EBP_EDI_IB: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2434;
	    
	case M_SS_2ESI_EAX_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2435;
	case M_SS_2ESI_ECX_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2436;
	case M_SS_2ESI_EDX_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2437;
	case M_SS_2ESI_EBX_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2438;
	case M_SS_2ESI_ESP_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2439;
	case M_SS_2ESI_EBP_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2440;
	case M_SS_3ESI_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2441;
	case M_SS_2ESI_EDI_IB: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2442;
	    
	case M_SS_2EDI_EAX_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2443;
	case M_SS_2EDI_ECX_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2444;
	case M_SS_2EDI_EDX_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2445;
	case M_SS_2EDI_EBX_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2446;
	case M_SS_2EDI_ESP_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2447;
	case M_SS_2EDI_EBP_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2448;
	case M_SS_2EDI_ESI_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2449;
	case M_SS_3EDI_IB: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2450;

	case M_SS_5EAX_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2451;
	case M_SS_4EAX_ECX_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2452;
	case M_SS_4EAX_EDX_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2453;
	case M_SS_4EAX_EBX_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2454;
	case M_SS_4EAX_ESP_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2455;
	case M_SS_4EAX_EBP_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2456;
	case M_SS_4EAX_ESI_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2457;
	case M_SS_4EAX_EDI_IB: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2458;

	case M_SS_4ECX_EAX_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2459;
	case M_SS_5ECX_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2460;
	case M_SS_4ECX_EDX_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2461;
	case M_SS_4ECX_EBX_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2462;
	case M_SS_4ECX_ESP_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2463;
	case M_SS_4ECX_EBP_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2464;
	case M_SS_4ECX_ESI_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2465;
	case M_SS_4ECX_EDI_IB: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2466;

	case M_SS_4EDX_EAX_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2467
	case M_SS_4EDX_ECX_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2468
	case M_SS_5EDX_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2469
	case M_SS_4EDX_EBX_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2470
	case M_SS_4EDX_ESP_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2471
	case M_SS_4EDX_EBP_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2472
	case M_SS_4EDX_ESI_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2473
	case M_SS_4EDX_EDI_IB: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2474

	case M_SS_4EBX_EAX_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2475;
	case M_SS_4EBX_ECX_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2476;
	case M_SS_4EBX_EDX_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2477;
	case M_SS_5EBX_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2478;
	case M_SS_4EBX_ESP_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2479;
	case M_SS_4EBX_EBP_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2480;
	case M_SS_4EBX_ESI_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2481;
	case M_SS_4EBX_EDI_IB: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2482;

	case M_SS_4EBP_EAX_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2483;
	case M_SS_4EBP_ECX_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2484;
	case M_SS_4EBP_EDX_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2485;
	case M_SS_4EBP_EBX_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2486;
	case M_SS_4EBP_ESP_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2487;
	case M_SS_5EBP_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2488;
	case M_SS_4EBP_ESI_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2489;
	case M_SS_4EBP_EDI_IB: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2490;
	    
	case M_SS_4ESI_EAX_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2491;
	case M_SS_4ESI_ECX_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2492;
	case M_SS_4ESI_EDX_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2493;
	case M_SS_4ESI_EBX_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2494;
	case M_SS_4ESI_ESP_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2495;
	case M_SS_4ESI_EBP_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2496
	case M_SS_5ESI_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2497;
	case M_SS_4ESI_EDI_IB: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2498;
	    
	case M_SS_4EDI_EAX_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2499;
	case M_SS_4EDI_ECX_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2500;
	case M_SS_4EDI_EDX_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2501;
	case M_SS_4EDI_EBX_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2502;
	case M_SS_4EDI_ESP_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2503;
	case M_SS_4EDI_EBP_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2504;
	case M_SS_4EDI_ESI_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2505;
	case M_SS_5EDI_IB: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2506;

	case M_SS_9EAX_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2507;
	case M_SS_8EAX_ECX_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2508;
	case M_SS_8EAX_EDX_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2509;
	case M_SS_8EAX_EBX_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2510;
	case M_SS_8EAX_ESP_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2511;
	case M_SS_8EAX_EBP_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2512;
	case M_SS_8EAX_ESI_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2513;
	case M_SS_8EAX_EDI_IB: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2514;

	case M_SS_8ECX_EAX_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2515;
	case M_SS_9ECX_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2516;
	case M_SS_8ECX_EDX_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2517;
	case M_SS_8ECX_EBX_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2518;
	case M_SS_8ECX_ESP_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2519;
	case M_SS_8ECX_EBP_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2520;
	case M_SS_8ECX_ESI_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2521;
	case M_SS_8ECX_EDI_IB: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2522;

	case M_SS_8EDX_EAX_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2523;
	case M_SS_8EDX_ECX_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2524;
	case M_SS_9EDX_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2525;
	case M_SS_8EDX_EBX_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2526;
	case M_SS_8EDX_ESP_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2527;
	case M_SS_8EDX_EBP_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2528;
	case M_SS_8EDX_ESI_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2529;
	case M_SS_8EDX_EDI_IB: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2530;

	case M_SS_8EBX_EAX_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2531
	case M_SS_8EBX_ECX_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2532
	case M_SS_8EBX_EDX_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2533
	case M_SS_9EBX_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2534
	case M_SS_8EBX_ESP_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2535
	case M_SS_8EBX_EBP_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2536
	case M_SS_8EBX_ESI_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2537
	case M_SS_8EBX_EDI_IB: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2538

	case M_SS_8EBP_EAX_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2539;
	case M_SS_8EBP_ECX_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2540;
	case M_SS_8EBP_EDX_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2541;
	case M_SS_8EBP_EBX_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2542;
	case M_SS_8EBP_ESP_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2543;
	case M_SS_9EBP_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2544;
	case M_SS_8EBP_ESI_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2545;
	case M_SS_8EBP_EDI_IB: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2546;

	case M_SS_8ESI_EAX_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2547;
	case M_SS_8ESI_ECX_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2548;
	case M_SS_8ESI_EDX_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2549;
	case M_SS_8ESI_EBX_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2550;
	case M_SS_8ESI_ESP_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2551;
	case M_SS_8ESI_EBP_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2552;
	case M_SS_9ESI_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2553;
	case M_SS_8ESI_EDI_IB: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2554;

	case M_SS_8EDI_EAX_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2555;
	case M_SS_8EDI_ECX_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2556;
	case M_SS_8EDI_EDX_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2557;
	case M_SS_8EDI_EBX_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2558;
	case M_SS_8EDI_ESP_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2559;
	case M_SS_8EDI_EBP_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2560;
	case M_SS_8EDI_ESI_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2561;
	case M_SS_9EDI_IB: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2562;

	case M_SS_EAX_ECX_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.ecx + getMicrocode()); //2563;
	case M_SS_EAX_EDX_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.edx + getMicrocode()); //2564;
	case M_SS_EAX_EBX_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.ebx + getMicrocode()); //2565;
	case M_SS_EAX_ESP_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.esp + getMicrocode()); //2566;
	case M_SS_EAX_EBP_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.ebp + getMicrocode()); //2567;
	case M_SS_EAX_ESI_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.esi + getMicrocode()); //2568;
	case M_SS_EAX_EDI_ID: return cpu.ss.getQuadWord(cpu.eax + cpu.edi + getMicrocode()); //2569;

	case M_SS_ECX_EDX_ID: return cpu.ss.getQuadWord(cpu.ecx + cpu.edx + getMicrocode()); //2570;
	case M_SS_ECX_EBX_ID: return cpu.ss.getQuadWord(cpu.ecx + cpu.ebx + getMicrocode()); //2571;
	case M_SS_ECX_ESP_ID: return cpu.ss.getQuadWord(cpu.ecx + cpu.esp + getMicrocode()); //2572;
	case M_SS_ECX_EBP_ID: return cpu.ss.getQuadWord(cpu.ecx + cpu.ebp + getMicrocode()); //2573;
	case M_SS_ECX_ESI_ID: return cpu.ss.getQuadWord(cpu.ecx + cpu.esi + getMicrocode()); //2574;
	case M_SS_ECX_EDI_ID: return cpu.ss.getQuadWord(cpu.ecx + cpu.edi + getMicrocode()); //2575;

	case M_SS_EDX_EBX_ID: return cpu.ss.getQuadWord(cpu.edx + cpu.ebx + getMicrocode()); //2576
	case M_SS_EDX_ESP_ID: return cpu.ss.getQuadWord(cpu.edx + cpu.esp + getMicrocode()); //2577
	case M_SS_EDX_EBP_ID: return cpu.ss.getQuadWord(cpu.edx + cpu.ebp + getMicrocode()); //2578
	case M_SS_EDX_ESI_ID: return cpu.ss.getQuadWord(cpu.edx + cpu.esi + getMicrocode()); //2579
	case M_SS_EDX_EDI_ID: return cpu.ss.getQuadWord(cpu.edx + cpu.edi + getMicrocode()); //2580

	case M_SS_EBX_ESP_ID: return cpu.ss.getQuadWord(cpu.ebx + cpu.esp + getMicrocode()); //2581
	case M_SS_EBX_EBP_ID: return cpu.ss.getQuadWord(cpu.ebx + cpu.ebp + getMicrocode()); //2582
	case M_SS_EBX_ESI_ID: return cpu.ss.getQuadWord(cpu.ebx + cpu.esi + getMicrocode()); //2583
	case M_SS_EBX_EDI_ID: return cpu.ss.getQuadWord(cpu.ebx + cpu.edi + getMicrocode()); //2584

	case M_SS_ESP_ID: return cpu.ss.getQuadWord(cpu.esp + getMicrocode()); //2585

	case M_SS_EBP_ESP_ID: return cpu.ss.getQuadWord(cpu.ebp + cpu.esp + getMicrocode()); //2586;
	case M_SS_EBP_ESI_ID: return cpu.ss.getQuadWord(cpu.ebp + cpu.esi + getMicrocode()); //2587;
	case M_SS_EBP_EDI_ID: return cpu.ss.getQuadWord(cpu.ebp + cpu.edi + getMicrocode()); //2588;

	case M_SS_ESI_ESP_ID: return cpu.ss.getQuadWord(cpu.esi + cpu.esp + getMicrocode()); //2589
	case M_SS_ESI_EDI_ID: return cpu.ss.getQuadWord(cpu.esi + cpu.edi + getMicrocode()); //2590

	case M_SS_EDI_ESP_ID: return cpu.ss.getQuadWord(cpu.edi + cpu.esp + getMicrocode()); //2591

	case M_SS_3EAX_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.eax + getMicrocode()); //2592;
	case M_SS_2EAX_ECX_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ecx + getMicrocode()); //2593;
	case M_SS_2EAX_EDX_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.edx + getMicrocode()); //2594;
	case M_SS_2EAX_EBX_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ebx + getMicrocode()); //2595;
	case M_SS_2EAX_ESP_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.esp + getMicrocode()); //2596;
	case M_SS_2EAX_EBP_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.ebp + getMicrocode()); //2597;
	case M_SS_2EAX_ESI_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.esi + getMicrocode()); //2598;
	case M_SS_2EAX_EDI_ID: return cpu.ss.getQuadWord((cpu.eax << 1) + cpu.edi + getMicrocode()); //2599;
	    
	case M_SS_2ECX_EAX_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.eax + getMicrocode()); //2600;
	case M_SS_3ECX_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ecx + getMicrocode()); //2601;
	case M_SS_2ECX_EDX_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.edx + getMicrocode()); //2602;
	case M_SS_2ECX_EBX_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ebx + getMicrocode()); //2603;
	case M_SS_2ECX_ESP_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.esp + getMicrocode()); //2604;
	case M_SS_2ECX_EBP_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.ebp + getMicrocode()); //2605;
	case M_SS_2ECX_ESI_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.esi + getMicrocode()); //2606;
	case M_SS_2ECX_EDI_ID: return cpu.ss.getQuadWord((cpu.ecx << 1) + cpu.edi + getMicrocode()); //2607;
	    
	case M_SS_2EDX_EAX_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.eax + getMicrocode()); //2608;
	case M_SS_2EDX_ECX_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ecx + getMicrocode()); //2609;
	case M_SS_3EDX_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.edx + getMicrocode()); //2610;
	case M_SS_2EDX_EBX_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ebx + getMicrocode()); //2611;
	case M_SS_2EDX_ESP_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.esp + getMicrocode()); //2612;
	case M_SS_2EDX_EBP_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.ebp + getMicrocode()); //2613;
	case M_SS_2EDX_ESI_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.esi + getMicrocode()); //2614;
	case M_SS_2EDX_EDI_ID: return cpu.ss.getQuadWord((cpu.edx << 1) + cpu.edi + getMicrocode()); //2615;
	    
	case M_SS_2EBX_EAX_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.eax + getMicrocode()); //2616;
	case M_SS_2EBX_ECX_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ecx + getMicrocode()); //2617;
	case M_SS_2EBX_EDX_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.edx + getMicrocode()); //2618;
	case M_SS_3EBX_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ebx + getMicrocode()); //2619;
	case M_SS_2EBX_ESP_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.esp + getMicrocode()); //2620;
	case M_SS_2EBX_EBP_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.ebp + getMicrocode()); //2621;
	case M_SS_2EBX_ESI_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.esi + getMicrocode()); //2622;
	case M_SS_2EBX_EDI_ID: return cpu.ss.getQuadWord((cpu.ebx << 1) + cpu.edi + getMicrocode()); //2623;
	    
	case M_SS_2EBP_EAX_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.eax + getMicrocode()); //2624;
	case M_SS_2EBP_ECX_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ecx + getMicrocode()); //2625;
	case M_SS_2EBP_EDX_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.edx + getMicrocode()); //2626;
	case M_SS_2EBP_EBX_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ebx + getMicrocode()); //2627;
	case M_SS_2EBP_ESP_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.esp + getMicrocode()); //2628;
	case M_SS_3EBP_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.ebp + getMicrocode()); //2629;
	case M_SS_2EBP_ESI_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.esi + getMicrocode()); //2630;
	case M_SS_2EBP_EDI_ID: return cpu.ss.getQuadWord((cpu.ebp << 1) + cpu.edi + getMicrocode()); //2631;
	    
	case M_SS_2ESI_EAX_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.eax + getMicrocode()); //2632;
	case M_SS_2ESI_ECX_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ecx + getMicrocode()); //2633;
	case M_SS_2ESI_EDX_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.edx + getMicrocode()); //2634;
	case M_SS_2ESI_EBX_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ebx + getMicrocode()); //2635;
	case M_SS_2ESI_ESP_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.esp + getMicrocode()); //2636;
	case M_SS_2ESI_EBP_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.ebp + getMicrocode()); //2637;
	case M_SS_3ESI_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.esi + getMicrocode()); //2638;
	case M_SS_2ESI_EDI_ID: return cpu.ss.getQuadWord((cpu.esi << 1) + cpu.edi + getMicrocode()); //2639;
	    
	case M_SS_2EDI_EAX_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.eax + getMicrocode()); //2640;
	case M_SS_2EDI_ECX_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ecx + getMicrocode()); //2641;
	case M_SS_2EDI_EDX_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.edx + getMicrocode()); //2642;
	case M_SS_2EDI_EBX_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ebx + getMicrocode()); //2643;
	case M_SS_2EDI_ESP_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.esp + getMicrocode()); //2644;
	case M_SS_2EDI_EBP_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.ebp + getMicrocode()); //2645;
	case M_SS_2EDI_ESI_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.esi + getMicrocode()); //2646;
	case M_SS_3EDI_ID: return cpu.ss.getQuadWord((cpu.edi << 1) + cpu.edi + getMicrocode()); //2647;

	case M_SS_5EAX_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.eax + getMicrocode()); //2648
	case M_SS_4EAX_ECX_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ecx + getMicrocode()); //2649
	case M_SS_4EAX_EDX_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.edx + getMicrocode()); //2650
	case M_SS_4EAX_EBX_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ebx + getMicrocode()); //2651
	case M_SS_4EAX_ESP_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.esp + getMicrocode()); //2652
	case M_SS_4EAX_EBP_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.ebp + getMicrocode()); //2653
	case M_SS_4EAX_ESI_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.esi + getMicrocode()); //2654
	case M_SS_4EAX_EDI_ID: return cpu.ss.getQuadWord((cpu.eax << 2) + cpu.edi + getMicrocode()); //2655

	case M_SS_4ECX_EAX_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.eax + getMicrocode()); //2656;
	case M_SS_5ECX_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ecx + getMicrocode()); //2657;
	case M_SS_4ECX_EDX_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.edx + getMicrocode()); //2658;
	case M_SS_4ECX_EBX_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ebx + getMicrocode()); //2659;
	case M_SS_4ECX_ESP_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.esp + getMicrocode()); //2660;
	case M_SS_4ECX_EBP_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.ebp + getMicrocode()); //2661;
	case M_SS_4ECX_ESI_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.esi + getMicrocode()); //2662;
	case M_SS_4ECX_EDI_ID: return cpu.ss.getQuadWord((cpu.ecx << 2) + cpu.edi + getMicrocode()); //2663;
	    
	case M_SS_4EDX_EAX_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.eax + getMicrocode()); //2664;
	case M_SS_4EDX_ECX_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ecx + getMicrocode()); //2665;
	case M_SS_5EDX_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.edx + getMicrocode()); //2666;
	case M_SS_4EDX_EBX_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ebx + getMicrocode()); //2667;
	case M_SS_4EDX_ESP_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.esp + getMicrocode()); //2668;
	case M_SS_4EDX_EBP_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.ebp + getMicrocode()); //2669;
	case M_SS_4EDX_ESI_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.esi + getMicrocode()); //2670;
	case M_SS_4EDX_EDI_ID: return cpu.ss.getQuadWord((cpu.edx << 2) + cpu.edi + getMicrocode()); //2671;
	    
	case M_SS_4EBX_EAX_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.eax + getMicrocode()); //2672;
	case M_SS_4EBX_ECX_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ecx + getMicrocode()); //2673;
	case M_SS_4EBX_EDX_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.edx + getMicrocode()); //2674;
	case M_SS_5EBX_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ebx + getMicrocode()); //2675;
	case M_SS_4EBX_ESP_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.esp + getMicrocode()); //2676;
	case M_SS_4EBX_EBP_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.ebp + getMicrocode()); //2677;
	case M_SS_4EBX_ESI_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.esi + getMicrocode()); //2678;
	case M_SS_4EBX_EDI_ID: return cpu.ss.getQuadWord((cpu.ebx << 2) + cpu.edi + getMicrocode()); //2679;
	    
	case M_SS_4EBP_EAX_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.eax + getMicrocode()); //2680;
	case M_SS_4EBP_ECX_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ecx + getMicrocode()); //2681;
	case M_SS_4EBP_EDX_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.edx + getMicrocode()); //2682;
	case M_SS_4EBP_EBX_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ebx + getMicrocode()); //2683;
	case M_SS_4EBP_ESP_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.esp + getMicrocode()); //2684;
	case M_SS_5EBP_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.ebp + getMicrocode()); //2685;
	case M_SS_4EBP_ESI_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.esi + getMicrocode()); //2686;
	case M_SS_4EBP_EDI_ID: return cpu.ss.getQuadWord((cpu.ebp << 2) + cpu.edi + getMicrocode()); //2687;
	    
	case M_SS_4ESI_EAX_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.eax + getMicrocode()); //2688;
	case M_SS_4ESI_ECX_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ecx + getMicrocode()); //2689;
	case M_SS_4ESI_EDX_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.edx + getMicrocode()); //2690;
	case M_SS_4ESI_EBX_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ebx + getMicrocode()); //2691;
	case M_SS_4ESI_ESP_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.esp + getMicrocode()); //2692;
	case M_SS_4ESI_EBP_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.ebp + getMicrocode()); //2693;
	case M_SS_5ESI_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.esi + getMicrocode()); //2694;
	case M_SS_4ESI_EDI_ID: return cpu.ss.getQuadWord((cpu.esi << 2) + cpu.edi + getMicrocode()); //2695;

	case M_SS_4EDI_EAX_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.eax + getMicrocode()); //2696;
	case M_SS_4EDI_ECX_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ecx + getMicrocode()); //2697;
	case M_SS_4EDI_EDX_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.edx + getMicrocode()); //2698;
	case M_SS_4EDI_EBX_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ebx + getMicrocode()); //2699;
	case M_SS_4EDI_ESP_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.esp + getMicrocode()); //2700;
	case M_SS_4EDI_EBP_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.ebp + getMicrocode()); //2701;
	case M_SS_4EDI_ESI_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.esi + getMicrocode()); //2702;
	case M_SS_5EDI_ID: return cpu.ss.getQuadWord((cpu.edi << 2) + cpu.edi + getMicrocode()); //2703;

	case M_SS_9EAX_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.eax + getMicrocode()); //2704;
	case M_SS_8EAX_ECX_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ecx + getMicrocode()); //2705;
	case M_SS_8EAX_EDX_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.edx + getMicrocode()); //2706;
	case M_SS_8EAX_EBX_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ebx + getMicrocode()); //2707;
	case M_SS_8EAX_ESP_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.esp + getMicrocode()); //2708;
	case M_SS_8EAX_EBP_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.ebp + getMicrocode()); //2709;
	case M_SS_8EAX_ESI_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.esi + getMicrocode()); //2710;
	case M_SS_8EAX_EDI_ID: return cpu.ss.getQuadWord((cpu.eax << 3) + cpu.edi + getMicrocode()); //2711;

	case M_SS_8ECX_EAX_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.eax + getMicrocode()); //2712;
	case M_SS_9ECX_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ecx + getMicrocode()); //2713;
	case M_SS_8ECX_EDX_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.edx + getMicrocode()); //2714;
	case M_SS_8ECX_EBX_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ebx + getMicrocode()); //2715;
	case M_SS_8ECX_ESP_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.esp + getMicrocode()); //2716;
	case M_SS_8ECX_EBP_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.ebp + getMicrocode()); //2717;
	case M_SS_8ECX_ESI_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.esi + getMicrocode()); //2718;
	case M_SS_8ECX_EDI_ID: return cpu.ss.getQuadWord((cpu.ecx << 3) + cpu.edi + getMicrocode()); //2719;

	case M_SS_8EDX_EAX_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.eax + getMicrocode()); //2720;
	case M_SS_8EDX_ECX_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ecx + getMicrocode()); //2721;
	case M_SS_9EDX_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.edx + getMicrocode()); //2722;
	case M_SS_8EDX_EBX_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ebx + getMicrocode()); //2723;
	case M_SS_8EDX_ESP_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.esp + getMicrocode()); //2724;
	case M_SS_8EDX_EBP_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.ebp + getMicrocode()); //2725;
	case M_SS_8EDX_ESI_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.esi + getMicrocode()); //2726;
	case M_SS_8EDX_EDI_ID: return cpu.ss.getQuadWord((cpu.edx << 3) + cpu.edi + getMicrocode()); //2727;

	case M_SS_8EBX_EAX_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.eax + getMicrocode()); //2728;
	case M_SS_8EBX_ECX_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ecx + getMicrocode()); //2729;
	case M_SS_8EBX_EDX_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.edx + getMicrocode()); //2730;
	case M_SS_9EBX_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ebx + getMicrocode()); //2731;
	case M_SS_8EBX_ESP_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.esp + getMicrocode()); //2732;
	case M_SS_8EBX_EBP_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.ebp + getMicrocode()); //2733;
	case M_SS_8EBX_ESI_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.esi + getMicrocode()); //2734;
	case M_SS_8EBX_EDI_ID: return cpu.ss.getQuadWord((cpu.ebx << 3) + cpu.edi + getMicrocode()); //2735;
	    
	case M_SS_8EBP_EAX_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.eax + getMicrocode()); //2736;
	case M_SS_8EBP_ECX_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ecx + getMicrocode()); //2737;
	case M_SS_8EBP_EDX_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.edx + getMicrocode()); //2738;
	case M_SS_8EBP_EBX_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ebx + getMicrocode()); //2739;
	case M_SS_8EBP_ESP_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.esp + getMicrocode()); //2740;
	case M_SS_9EBP_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.ebp + getMicrocode()); //2741;
	case M_SS_8EBP_ESI_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.esi + getMicrocode()); //2742;
	case M_SS_8EBP_EDI_ID: return cpu.ss.getQuadWord((cpu.ebp << 3) + cpu.edi + getMicrocode()); //2743;
	    
	case M_SS_8ESI_EAX_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.eax + getMicrocode()); //2744;
	case M_SS_8ESI_ECX_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ecx + getMicrocode()); //2745;
	case M_SS_8ESI_EDX_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.edx + getMicrocode()); //2746;
	case M_SS_8ESI_EBX_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ebx + getMicrocode()); //2747;
	case M_SS_8ESI_ESP_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.esp + getMicrocode()); //2748;
	case M_SS_8ESI_EBP_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.ebp + getMicrocode()); //2749;
	case M_SS_9ESI_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.esi + getMicrocode()); //2750;
	case M_SS_8ESI_EDI_ID: return cpu.ss.getQuadWord((cpu.esi << 3) + cpu.edi + getMicrocode()); //2751;
    
	case M_SS_8EDI_EAX_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.eax + getMicrocode()); //2752;
	case M_SS_8EDI_ECX_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ecx + getMicrocode()); //2753;
	case M_SS_8EDI_EDX_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.edx + getMicrocode()); //2754;
	case M_SS_8EDI_EBX_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ebx + getMicrocode()); //2755;
	case M_SS_8EDI_ESP_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.esp + getMicrocode()); //2756;
	case M_SS_8EDI_EBP_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.ebp + getMicrocode()); //2757;
	case M_SS_8EDI_ESI_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.esi + getMicrocode()); //2758;
	case M_SS_9EDI_ID: return cpu.ss.getQuadWord((cpu.edi << 3) + cpu.edi + getMicrocode()); //2759;
	}

	System.err.println("Unimplemented GetLong Operand: " + microcode);
	throw unimplemented;
    }

    protected final void setByteOperand(byte data)
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case AL:  cpu.eax = (cpu.eax & 0xffffff00) | (data & 0xff); return;
	case AH:  cpu.eax = (cpu.eax & 0xffff00ff) | ((data << 8) & 0xff00); return;
	case CL:  cpu.ecx = (cpu.ecx & 0xffffff00) | (data & 0xff); return;
	case CH:  cpu.ecx = (cpu.ecx & 0xffff00ff) | ((data << 8) & 0xff00); return;
	case DL:  cpu.edx = (cpu.edx & 0xffffff00) | (data & 0xff); return;
	case DH:  cpu.edx = (cpu.edx & 0xffff00ff) | ((data << 8) & 0xff00); return;
	case BL:  cpu.ebx = (cpu.ebx & 0xffffff00) | (data & 0xff); return;
	case BH:  cpu.ebx = (cpu.ebx & 0xffff00ff) | ((data << 8) & 0xff00); return;
	}

	setByteMemoryOperand(microcode, data);
	return;
    }

    private final void setByteMemoryOperand(int microcode, byte data)
    {
	switch(microcode) {
	case M_ES_BX_uAL: cpu.es.setByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff, data); return;
	case M_CS_BX_uAL: cpu.cs.setByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff, data); return;
	case M_SS_BX_uAL: cpu.ss.setByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff, data); return;
	case M_DS_BX_uAL: cpu.ds.setByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff, data); return;
	case M_FS_BX_uAL: cpu.fs.setByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff, data); return;
	case M_GS_BX_uAL: cpu.gs.setByte((cpu.ebx + (0xff & cpu.eax)) & 0xffff, data); return;

	case M_ES_BX_SI: cpu.es.setByte((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_ES_BX_DI: cpu.es.setByte((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_ES_BP_SI: cpu.es.setByte((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_ES_BP_DI: cpu.es.setByte((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_ES_SI: cpu.es.setByte(cpu.esi & 0xffff, data); return;
	case M_ES_DI: cpu.es.setByte(cpu.edi & 0xffff, data); return;
	case M_ES_IW: cpu.es.setByte(getMicrocode() & 0xffff, data); return;
	case M_ES_BX: cpu.es.setByte(cpu.ebx & 0xffff, data); return;
	case M_ES_BX_SI_IB: cpu.es.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_DI_IB: cpu.es.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_SI_IB: cpu.es.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_DI_IB: cpu.es.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_SI_IB: cpu.es.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_DI_IB: cpu.es.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_IB: cpu.es.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_IB: cpu.es.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_SI_IW: cpu.es.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_DI_IW: cpu.es.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_SI_IW: cpu.es.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_DI_IW: cpu.es.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_SI_IW: cpu.es.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_DI_IW: cpu.es.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_IW: cpu.es.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_IW: cpu.es.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_CS_BX_SI: cpu.cs.setByte((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_CS_BX_DI: cpu.cs.setByte((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_CS_BP_SI: cpu.cs.setByte((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_CS_BP_DI: cpu.cs.setByte((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_CS_SI: cpu.cs.setByte(cpu.esi & 0xffff, data); return;
	case M_CS_DI: cpu.cs.setByte(cpu.edi & 0xffff, data); return;
	case M_CS_IW: cpu.cs.setByte(getMicrocode() & 0xffff, data); return;
	case M_CS_BX: cpu.cs.setByte(cpu.ebx & 0xffff, data); return;
	case M_CS_BX_SI_IB: cpu.cs.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_DI_IB: cpu.cs.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_SI_IB: cpu.cs.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_DI_IB: cpu.cs.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_SI_IB: cpu.cs.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_DI_IB: cpu.cs.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_IB: cpu.cs.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_IB: cpu.cs.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_SI_IW: cpu.cs.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_DI_IW: cpu.cs.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_SI_IW: cpu.cs.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_DI_IW: cpu.cs.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_SI_IW: cpu.cs.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_DI_IW: cpu.cs.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_IW: cpu.cs.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_IW: cpu.cs.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_SS_BX_SI: cpu.ss.setByte((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_SS_BX_DI: cpu.ss.setByte((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_SS_BP_SI: cpu.ss.setByte((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_SS_BP_DI: cpu.ss.setByte((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_SS_SI: cpu.ss.setByte(cpu.esi & 0xffff, data); return;
	case M_SS_DI: cpu.ss.setByte(cpu.edi & 0xffff, data); return;
	case M_SS_IW: cpu.ss.setByte(getMicrocode() & 0xffff, data); return;
	case M_SS_BX: cpu.ss.setByte(cpu.ebx & 0xffff, data); return;
	case M_SS_BX_SI_IB: cpu.ss.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_DI_IB: cpu.ss.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_SI_IB: cpu.ss.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_DI_IB: cpu.ss.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_SI_IB: cpu.ss.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_DI_IB: cpu.ss.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_IB: cpu.ss.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_IB: cpu.ss.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_SI_IW: cpu.ss.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_DI_IW: cpu.ss.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_SI_IW: cpu.ss.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_DI_IW: cpu.ss.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_SI_IW: cpu.ss.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_DI_IW: cpu.ss.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_IW: cpu.ss.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_IW: cpu.ss.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_DS_BX_SI: cpu.ds.setByte((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_DS_BX_DI: cpu.ds.setByte((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_DS_BP_SI: cpu.ds.setByte((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_DS_BP_DI: cpu.ds.setByte((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_DS_SI: cpu.ds.setByte(cpu.esi & 0xffff, data); return;
	case M_DS_DI: cpu.ds.setByte(cpu.edi & 0xffff, data); return;
	case M_DS_IW: cpu.ds.setByte(getMicrocode() & 0xffff, data); return;
	case M_DS_BX: cpu.ds.setByte(cpu.ebx & 0xffff, data); return;
	case M_DS_BX_SI_IB: cpu.ds.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_DI_IB: cpu.ds.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_SI_IB: cpu.ds.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_DI_IB: cpu.ds.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_SI_IB: cpu.ds.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_DI_IB: cpu.ds.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_IB: cpu.ds.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_IB: cpu.ds.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_SI_IW: cpu.ds.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_DI_IW: cpu.ds.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_SI_IW: cpu.ds.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_DI_IW: cpu.ds.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_SI_IW: cpu.ds.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_DI_IW: cpu.ds.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_IW: cpu.ds.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_IW: cpu.ds.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_FS_BX_SI: cpu.fs.setByte((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_FS_BX_DI: cpu.fs.setByte((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_FS_BP_SI: cpu.fs.setByte((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_FS_BP_DI: cpu.fs.setByte((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_FS_SI: cpu.fs.setByte(cpu.esi & 0xffff, data); return;
	case M_FS_DI: cpu.fs.setByte(cpu.edi & 0xffff, data); return;
	case M_FS_IW: cpu.fs.setByte(getMicrocode() & 0xffff, data); return;
	case M_FS_BX: cpu.fs.setByte(cpu.ebx & 0xffff, data); return;
	case M_FS_BX_SI_IB: cpu.fs.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_DI_IB: cpu.fs.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_SI_IB: cpu.fs.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_DI_IB: cpu.fs.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_SI_IB: cpu.fs.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_DI_IB: cpu.fs.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_IB: cpu.fs.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_IB: cpu.fs.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_SI_IW: cpu.fs.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_DI_IW: cpu.fs.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_SI_IW: cpu.fs.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_DI_IW: cpu.fs.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_SI_IW: cpu.fs.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_DI_IW: cpu.fs.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_IW: cpu.fs.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_IW: cpu.fs.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_GS_BX_SI: cpu.gs.setByte((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_GS_BX_DI: cpu.gs.setByte((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_GS_BP_SI: cpu.gs.setByte((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_GS_BP_DI: cpu.gs.setByte((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_GS_SI: cpu.gs.setByte(cpu.esi & 0xffff, data); return;
	case M_GS_DI: cpu.gs.setByte(cpu.edi & 0xffff, data); return;
	case M_GS_IW: cpu.gs.setByte(getMicrocode() & 0xffff, data); return;
	case M_GS_BX: cpu.gs.setByte(cpu.ebx & 0xffff, data); return;
	case M_GS_BX_SI_IB: cpu.gs.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_DI_IB: cpu.gs.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_SI_IB: cpu.gs.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_DI_IB: cpu.gs.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_SI_IB: cpu.gs.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_DI_IB: cpu.gs.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_IB: cpu.gs.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_IB: cpu.gs.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_SI_IW: cpu.gs.setByte((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_DI_IW: cpu.gs.setByte((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_SI_IW: cpu.gs.setByte((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_DI_IW: cpu.gs.setByte((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_SI_IW: cpu.gs.setByte((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_DI_IW: cpu.gs.setByte((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_IW: cpu.gs.setByte((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_IW: cpu.gs.setByte((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_ES_EAX: cpu.es.setByte(cpu.eax, data); return; //41
	case M_ES_ECX: cpu.es.setByte(cpu.ecx, data); return; //42
	case M_ES_EDX: cpu.es.setByte(cpu.edx, data); return; //43
	case M_ES_EBX: cpu.es.setByte(cpu.ebx, data); return; //44
	case M_ES_ID: cpu.es.setByte(getMicrocode(), data); return; //45
	case M_ES_ESI: cpu.es.setByte(cpu.esi, data); return; //46
	case M_ES_EDI:cpu.es.setByte(cpu.edi, data); return; //47


	case M_SS_EAX_IB: cpu.ss.setByte(cpu.eax + getMicrocode(), data); return; //90
	case M_SS_ECX_IB: cpu.ss.setByte(cpu.ecx + getMicrocode(), data); return; //91
	case M_SS_EDX_IB: cpu.ss.setByte(cpu.edx + getMicrocode(), data); return; //92
	case M_SS_EBX_IB: cpu.ss.setByte(cpu.ebx + getMicrocode(), data); return; //93
	case M_SS_EBP_IB: cpu.ss.setByte(cpu.ebp + getMicrocode(), data); return; //94
	case M_SS_ESI_IB: cpu.ss.setByte(cpu.esi + getMicrocode(), data); return; //95
	case M_SS_EDI_IB: cpu.ss.setByte(cpu.edi + getMicrocode(), data); return; //96

	case M_SS_EAX_ID: cpu.ss.setByte(cpu.eax + getMicrocode(), data); return; //97
	case M_SS_ECX_ID: cpu.ss.setByte(cpu.ecx + getMicrocode(), data); return; //98
	case M_SS_EDX_ID: cpu.ss.setByte(cpu.edx + getMicrocode(), data); return; //99
	case M_SS_EBX_ID: cpu.ss.setByte(cpu.ebx + getMicrocode(), data); return; //100
	case M_SS_EBP_ID: cpu.ss.setByte(cpu.ebp + getMicrocode(), data); return; //101
	case M_SS_ESI_ID: cpu.ss.setByte(cpu.esi + getMicrocode(), data); return; //102
	case M_SS_EDI_ID: cpu.ss.setByte(cpu.edi + getMicrocode(), data); return; //103

	case M_DS_EAX: cpu.ds.setByte(cpu.eax, data); return; //104
	case M_DS_ECX: cpu.ds.setByte(cpu.ecx, data); return; //105
	case M_DS_EDX: cpu.ds.setByte(cpu.edx, data); return; //106
	case M_DS_EBX: cpu.ds.setByte(cpu.ebx, data); return; //107
	case M_DS_ID: cpu.ds.setByte(getMicrocode(), data); return; //108
	case M_DS_ESI: cpu.ds.setByte(cpu.esi, data); return; //109
	case M_DS_EDI: cpu.ds.setByte(cpu.edi, data); return; //110

	case M_DS_EAX_IB: cpu.ds.setByte(cpu.eax + getMicrocode(), data); return; //111
	case M_DS_ECX_IB: cpu.ds.setByte(cpu.ecx + getMicrocode(), data); return; //112
	case M_DS_EDX_IB: cpu.ds.setByte(cpu.edx + getMicrocode(), data); return; //113
	case M_DS_EBX_IB: cpu.ds.setByte(cpu.ebx + getMicrocode(), data); return; //114
	case M_DS_EBP_IB: cpu.ds.setByte(cpu.ebp + getMicrocode(), data); return; //115
	case M_DS_ESI_IB: cpu.ds.setByte(cpu.esi + getMicrocode(), data); return; //116
	case M_DS_EDI_IB: cpu.ds.setByte(cpu.edi + getMicrocode(), data); return; //117

	case M_DS_EAX_ID: cpu.ds.setByte(cpu.eax + getMicrocode(), data); return; //118
	case M_DS_ECX_ID: cpu.ds.setByte(cpu.ecx + getMicrocode(), data); return; //119
	case M_DS_EDX_ID: cpu.ds.setByte(cpu.edx + getMicrocode(), data); return; //120
	case M_DS_EBX_ID: cpu.ds.setByte(cpu.ebx + getMicrocode(), data); return; //121
	case M_DS_EBP_ID: cpu.ds.setByte(cpu.ebp + getMicrocode(), data); return; //122
	case M_DS_ESI_ID: cpu.ds.setByte(cpu.esi + getMicrocode(), data); return; //123
	case M_DS_EDI_ID: cpu.ds.setByte(cpu.edi + getMicrocode(), data); return; //124

	case M_DS_2EAX: cpu.ds.setByte(cpu.eax + cpu.eax, data); return; //2156
	case M_DS_EAX_ECX: cpu.ds.setByte(cpu.eax + cpu.ecx, data); return; //2157
	case M_DS_EAX_EDX: cpu.ds.setByte(cpu.eax + cpu.edx, data); return; //2158
	case M_DS_EAX_EBX: cpu.ds.setByte(cpu.eax + cpu.ebx, data); return; //2159
	case M_DS_EAX_ESP: cpu.ds.setByte(cpu.eax + cpu.esp, data); return; //2160
	case M_DS_EAX_EBP: cpu.ds.setByte(cpu.eax + cpu.ebp, data); return; //2161
	case M_DS_EAX_ESI: cpu.ds.setByte(cpu.eax + cpu.esi, data); return; //2162
	case M_DS_EAX_EDI: cpu.ds.setByte(cpu.eax + cpu.edi, data); return; //2163

	case M_DS_2ECX: cpu.ds.setByte(cpu.ecx + cpu.ecx, data); return; //2164
	case M_DS_ECX_EDX: cpu.ds.setByte(cpu.ecx + cpu.edx, data); return; //2165
	case M_DS_ECX_EBX: cpu.ds.setByte(cpu.ecx + cpu.ebx, data); return; //2166
	case M_DS_ECX_ESP: cpu.ds.setByte(cpu.ecx + cpu.esp, data); return; //2167
	case M_DS_ECX_EBP: cpu.ds.setByte(cpu.ecx + cpu.ebp, data); return; //2168
	case M_DS_ECX_ESI: cpu.ds.setByte(cpu.ecx + cpu.esi, data); return; //2169
	case M_DS_ECX_EDI: cpu.ds.setByte(cpu.ecx + cpu.edi, data); return; //2170

	case M_DS_2EDX: cpu.ds.setByte(cpu.edx + cpu.edx, data); return; //2171
	case M_DS_EDX_EBX: cpu.ds.setByte(cpu.edx + cpu.ebx, data); return; //2172
	case M_DS_EDX_ESP: cpu.ds.setByte(cpu.edx + cpu.esp, data); return; //2173
	case M_DS_EDX_EBP: cpu.ds.setByte(cpu.edx + cpu.ebp, data); return; //2174
	case M_DS_EDX_ESI: cpu.ds.setByte(cpu.edx + cpu.esi, data); return; //2175
	case M_DS_EDX_EDI: cpu.ds.setByte(cpu.edx + cpu.edi, data); return; //2176

	case M_DS_2EBX: cpu.ds.setByte(cpu.ebx + cpu.ebx, data); return; //2177;
	case M_DS_EBX_ESP: cpu.ds.setByte(cpu.ebx + cpu.esp, data); return; //2178;
	case M_DS_EBX_EBP: cpu.ds.setByte(cpu.ebx + cpu.ebp, data); return; //2179;
	case M_DS_EBX_ESI: cpu.ds.setByte(cpu.ebx + cpu.esi, data); return; //2180;
	case M_DS_EBX_EDI: cpu.ds.setByte(cpu.ebx + cpu.edi, data); return; //2181;
	    
	case M_DS_ESP: cpu.ds.setByte(cpu.esp, data); return; //2182

	case M_DS_EBP_ESP: cpu.ds.setByte(cpu.ebp + cpu.esp, data); return; //2183;
	case M_DS_EBP_ESI: cpu.ds.setByte(cpu.ebp + cpu.esi, data); return; //2184;
	case M_DS_EBP_EDI: cpu.ds.setByte(cpu.ebp + cpu.edi, data); return; //2185;

	case M_DS_ESI_ESP: cpu.ds.setByte(cpu.esi + cpu.esp, data); return; //2186;
	case M_DS_2ESI: cpu.ds.setByte(cpu.esi + cpu.esi, data); return; //2187;
	case M_DS_ESI_EDI: cpu.ds.setByte(cpu.esi + cpu.edi, data); return; //2188;

	case M_DS_EDI_ESP: cpu.ds.setByte(cpu.edi + cpu.esp, data); return; //2189;
	case M_DS_2EDI: cpu.ds.setByte(cpu.edi + cpu.edi, data); return; //2190;

	case M_DS_3EAX: cpu.ds.setByte((cpu.eax << 1) + cpu.eax, data); return; //2191;
	case M_DS_2EAX_ECX: cpu.ds.setByte((cpu.eax << 1) + cpu.ecx, data); return; //2192;
	case M_DS_2EAX_EDX: cpu.ds.setByte((cpu.eax << 1) + cpu.edx, data); return; //2193;
	case M_DS_2EAX_EBX: cpu.ds.setByte((cpu.eax << 1) + cpu.ebx, data); return; //2194;
	case M_DS_2EAX_ESP: cpu.ds.setByte((cpu.eax << 1) + cpu.esp, data); return; //2195;
	case M_DS_2EAX_ID: cpu.ds.setByte((cpu.eax << 1) + getMicrocode(), data); return; //2196;
	case M_DS_2EAX_ESI: cpu.ds.setByte((cpu.eax << 1) + cpu.esi, data); return; //2197;
	case M_DS_2EAX_EDI: cpu.ds.setByte((cpu.eax << 1) + cpu.edi, data); return; //2198;

	case M_DS_2ECX_EAX: cpu.ds.setByte((cpu.ecx << 1) + cpu.eax, data); return; //2199;
	case M_DS_3ECX: cpu.ds.setByte((cpu.ecx << 1) + cpu.ecx, data); return; //2200;
	case M_DS_2ECX_EDX: cpu.ds.setByte((cpu.ecx << 1) + cpu.edx, data); return; //2201;
	case M_DS_2ECX_EBX: cpu.ds.setByte((cpu.ecx << 1) + cpu.ebx, data); return; //2202;
	case M_DS_2ECX_ESP: cpu.ds.setByte((cpu.ecx << 1) + cpu.esp, data); return; //2203;
	case M_DS_2ECX_ID: cpu.ds.setByte((cpu.ecx << 1) + getMicrocode(), data); return; //2204;
	case M_DS_2ECX_ESI: cpu.ds.setByte((cpu.ecx << 1) + cpu.esi, data); return; //2205;
	case M_DS_2ECX_EDI: cpu.ds.setByte((cpu.ecx << 1) + cpu.edi, data); return; //2206;

	case M_DS_2EDX_EAX: cpu.ds.setByte((cpu.edx << 1) + cpu.eax, data); return; //2207;
	case M_DS_2EDX_ECX: cpu.ds.setByte((cpu.edx << 1) + cpu.ecx, data); return; //2208;
	case M_DS_3EDX: cpu.ds.setByte((cpu.edx << 1) + cpu.edx, data); return; //2209;
	case M_DS_2EDX_EBX: cpu.ds.setByte((cpu.edx << 1) + cpu.ebx, data); return; //2210;
	case M_DS_2EDX_ESP: cpu.ds.setByte((cpu.edx << 1) + cpu.esp, data); return; //2211;
	case M_DS_2EDX_ID: cpu.ds.setByte((cpu.edx << 1) + getMicrocode(), data); return; //2212;
	case M_DS_2EDX_ESI: cpu.ds.setByte((cpu.edx << 1) + cpu.esi, data); return; //2213;
	case M_DS_2EDX_EDI: cpu.ds.setByte((cpu.edx << 1) + cpu.edi, data); return; //2214;

	case M_DS_2EBX_EAX: cpu.ds.setByte((cpu.ebx << 1) + cpu.eax, data); return; //2215;
	case M_DS_2EBX_ECX: cpu.ds.setByte((cpu.ebx << 1) + cpu.ecx, data); return; //2216;
	case M_DS_2EBX_EDX: cpu.ds.setByte((cpu.ebx << 1) + cpu.edx, data); return; //2217;
	case M_DS_3EBX: cpu.ds.setByte((cpu.ebx << 1) + cpu.ebx, data); return; //2218;
	case M_DS_2EBX_ESP: cpu.ds.setByte((cpu.ebx << 1) + cpu.esp, data); return; //2219;
	case M_DS_2EBX_ID: cpu.ds.setByte((cpu.ebx << 1) + getMicrocode(), data); return; //2220;
	case M_DS_2EBX_ESI: cpu.ds.setByte((cpu.ebx << 1) + cpu.esi, data); return; //2221;
	case M_DS_2EBX_EDI: cpu.ds.setByte((cpu.ebx << 1) + cpu.edi, data); return; //2222;
	    
	case M_DS_2EBP_EAX: cpu.ds.setByte((cpu.ebp << 1) + cpu.eax, data); return; //2223;
	case M_DS_2EBP_ECX: cpu.ds.setByte((cpu.ebp << 1) + cpu.ecx, data); return; //2224;
	case M_DS_2EBP_EDX: cpu.ds.setByte((cpu.ebp << 1) + cpu.edx, data); return; //2225;
	case M_DS_2EBP_EBX: cpu.ds.setByte((cpu.ebp << 1) + cpu.ebx, data); return; //2226;
	case M_DS_2EBP_ESP: cpu.ds.setByte((cpu.ebp << 1) + cpu.esp, data); return; //2227;
	case M_DS_2EBP_ID: cpu.ds.setByte((cpu.ebp << 1) + getMicrocode(), data); return; //2228;
	case M_DS_2EBP_ESI: cpu.ds.setByte((cpu.ebp << 1) + cpu.esi, data); return; //2229;
	case M_DS_2EBP_EDI: cpu.ds.setByte((cpu.ebp << 1) + cpu.edi, data); return; //2230;

	case M_DS_2ESI_EAX: cpu.ds.setByte((cpu.esi << 1) + cpu.eax, data); return; //2231;
	case M_DS_2ESI_ECX: cpu.ds.setByte((cpu.esi << 1) + cpu.ecx, data); return; //2232;
	case M_DS_2ESI_EDX: cpu.ds.setByte((cpu.esi << 1) + cpu.edx, data); return; //2233;
	case M_DS_2ESI_EBX: cpu.ds.setByte((cpu.esi << 1) + cpu.ebx, data); return; //2234;
	case M_DS_2ESI_ESP: cpu.ds.setByte((cpu.esi << 1) + cpu.esp, data); return; //2235;
	case M_DS_2ESI_ID: cpu.ds.setByte((cpu.esi << 1) + getMicrocode(), data); return; //2236;
	case M_DS_3ESI: cpu.ds.setByte((cpu.esi << 1) + cpu.esi, data); return; //2237;
	case M_DS_2ESI_EDI: cpu.ds.setByte((cpu.esi << 1) + cpu.edi, data); return; //2238;

	case M_DS_2EDI_EAX: cpu.ds.setByte((cpu.edi << 1) + cpu.eax, data); return; //2239;
	case M_DS_2EDI_ECX: cpu.ds.setByte((cpu.edi << 1) + cpu.ecx, data); return; //2240;
	case M_DS_2EDI_EDX: cpu.ds.setByte((cpu.edi << 1) + cpu.edx, data); return; //2241;
	case M_DS_2EDI_EBX: cpu.ds.setByte((cpu.edi << 1) + cpu.ebx, data); return; //2242;
	case M_DS_2EDI_ESP: cpu.ds.setByte((cpu.edi << 1) + cpu.esp, data); return; //2243;
	case M_DS_2EDI_ID: cpu.ds.setByte((cpu.edi << 1) + getMicrocode(), data); return; //2244;
	case M_DS_2EDI_ESI: cpu.ds.setByte((cpu.edi << 1) + cpu.esi, data); return; //2245;
	case M_DS_3EDI: cpu.ds.setByte((cpu.edi << 1) + cpu.edi, data); return; //2246;

	case M_DS_5EAX: cpu.ds.setByte((cpu.eax << 2) + cpu.eax, data); return; //2247
	case M_DS_4EAX_ECX: cpu.ds.setByte((cpu.eax << 2) + cpu.ecx, data); return; //2248
	case M_DS_4EAX_EDX: cpu.ds.setByte((cpu.eax << 2) + cpu.edx, data); return; //2249
	case M_DS_4EAX_EBX: cpu.ds.setByte((cpu.eax << 2) + cpu.ebx, data); return; //2250
	case M_DS_4EAX_ESP: cpu.ds.setByte((cpu.eax << 2) + cpu.esp, data); return; //2251
	case M_DS_4EAX_ID: cpu.ds.setByte((cpu.eax << 2) + getMicrocode(), data); return; //2252
	case M_DS_4EAX_ESI: cpu.ds.setByte((cpu.eax << 2) + cpu.esi, data); return; //2253
	case M_DS_4EAX_EDI: cpu.ds.setByte((cpu.eax << 2) + cpu.edi, data); return; //2254

	case M_DS_4ECX_EAX: cpu.ds.setByte((cpu.ecx << 2) + cpu.eax, data); return; //2255
	case M_DS_5ECX: cpu.ds.setByte((cpu.ecx << 2) + cpu.ecx, data); return; //2256
	case M_DS_4ECX_EDX: cpu.ds.setByte((cpu.ecx << 2) + cpu.edx, data); return; //2257
	case M_DS_4ECX_EBX: cpu.ds.setByte((cpu.ecx << 2) + cpu.ebx, data); return; //2258
	case M_DS_4ECX_ESP: cpu.ds.setByte((cpu.ecx << 2) + cpu.esp, data); return; //2259
	case M_DS_4ECX_ID: cpu.ds.setByte((cpu.ecx << 2) + getMicrocode(), data); return; //2260
	case M_DS_4ECX_ESI: cpu.ds.setByte((cpu.ecx << 2) + cpu.esi, data); return; //2261
	case M_DS_4ECX_EDI: cpu.ds.setByte((cpu.ecx << 2) + cpu.edi, data); return; //2262

	case M_DS_4EDX_EAX: cpu.ds.setByte((cpu.edx << 2) + cpu.eax, data); return; //2263
	case M_DS_4EDX_ECX: cpu.ds.setByte((cpu.edx << 2) + cpu.ecx, data); return; //2264
	case M_DS_5EDX: cpu.ds.setByte((cpu.edx << 2) + cpu.edx, data); return; //2265
	case M_DS_4EDX_EBX: cpu.ds.setByte((cpu.edx << 2) + cpu.ebx, data); return; //2266
	case M_DS_4EDX_ESP: cpu.ds.setByte((cpu.edx << 2) + cpu.esp, data); return; //2267
	case M_DS_4EDX_ID: cpu.ds.setByte((cpu.edx << 2) + getMicrocode(), data); return; //2268
	case M_DS_4EDX_ESI: cpu.ds.setByte((cpu.edx << 2) + cpu.esi, data); return; //2269
	case M_DS_4EDX_EDI: cpu.ds.setByte((cpu.edx << 2) + cpu.edi, data); return; //2270

	case M_DS_4EBX_EAX: cpu.ds.setByte((cpu.ebx << 2) + cpu.eax, data); return; //2271;
	case M_DS_4EBX_ECX: cpu.ds.setByte((cpu.ebx << 2) + cpu.ecx, data); return; //2272;
	case M_DS_4EBX_EDX: cpu.ds.setByte((cpu.ebx << 2) + cpu.edx, data); return; //2273;
	case M_DS_5EBX: cpu.ds.setByte((cpu.ebx << 2) + cpu.ebx, data); return; //2274;
	case M_DS_4EBX_ESP: cpu.ds.setByte((cpu.ebx << 2) + cpu.esp, data); return; //2275;
	case M_DS_4EBX_ID: cpu.ds.setByte((cpu.ebx << 2) + getMicrocode(), data); return; //2276;
	case M_DS_4EBX_ESI: cpu.ds.setByte((cpu.ebx << 2) + cpu.esi, data); return; //2277;
	case M_DS_4EBX_EDI: cpu.ds.setByte((cpu.ebx << 2) + cpu.edi, data); return; //2278;
	    
	case M_DS_4EBP_EAX: cpu.ds.setByte((cpu.ebp << 2) + cpu.eax, data); return; //2279
	case M_DS_4EBP_ECX: cpu.ds.setByte((cpu.ebp << 2) + cpu.ecx, data); return; //2280
	case M_DS_4EBP_EDX: cpu.ds.setByte((cpu.ebp << 2) + cpu.edx, data); return; //2281
	case M_DS_4EBP_EBX: cpu.ds.setByte((cpu.ebp << 2) + cpu.ebx, data); return; //2282
	case M_DS_4EBP_ESP: cpu.ds.setByte((cpu.ebp << 2) + cpu.esp, data); return; //2283
	case M_DS_4EBP_ID: cpu.ds.setByte((cpu.ebp << 2) + getMicrocode(), data); return; //2284
	case M_DS_4EBP_ESI: cpu.ds.setByte((cpu.ebp << 2) + cpu.esi, data); return; //2285
	case M_DS_4EBP_EDI: cpu.ds.setByte((cpu.ebp << 2) + cpu.edi, data); return; //2286

	case M_DS_4ESI_EAX: cpu.ds.setByte((cpu.esi << 2) + cpu.eax, data); return; //2287;
	case M_DS_4ESI_ECX: cpu.ds.setByte((cpu.esi << 2) + cpu.ecx, data); return; //2288;
	case M_DS_4ESI_EDX: cpu.ds.setByte((cpu.esi << 2) + cpu.edx, data); return; //2289;
	case M_DS_4ESI_EBX: cpu.ds.setByte((cpu.esi << 2) + cpu.ebx, data); return; //2290;
	case M_DS_4ESI_ESP: cpu.ds.setByte((cpu.esi << 2) + cpu.esp, data); return; //2291;
	case M_DS_4ESI_ID: cpu.ds.setByte((cpu.esi << 2) + getMicrocode(), data); return; //2292;
	case M_DS_5ESI: cpu.ds.setByte((cpu.esi << 2) + cpu.esi, data); return; //2293;
	case M_DS_4ESI_EDI: cpu.ds.setByte((cpu.esi << 2) + cpu.edi, data); return; //2294;

	case M_DS_4EDI_EAX: cpu.ds.setByte((cpu.edi << 2) + cpu.eax, data); return; //2295;
	case M_DS_4EDI_ECX: cpu.ds.setByte((cpu.edi << 2) + cpu.ecx, data); return; //2296;
	case M_DS_4EDI_EDX: cpu.ds.setByte((cpu.edi << 2) + cpu.edx, data); return; //2297;
	case M_DS_4EDI_EBX: cpu.ds.setByte((cpu.edi << 2) + cpu.ebx, data); return; //2298;
	case M_DS_4EDI_ESP: cpu.ds.setByte((cpu.edi << 2) + cpu.esp, data); return; //2299;
	case M_DS_4EDI_ID: cpu.ds.setByte((cpu.edi << 2) + getMicrocode(), data); return; //2300;
	case M_DS_4EDI_ESI: cpu.ds.setByte((cpu.edi << 2) + cpu.esi, data); return; //2301;
	case M_DS_5EDI: cpu.ds.setByte((cpu.edi << 2) + cpu.edi, data); return; //2302;

	case M_DS_9EAX: cpu.ds.setByte((cpu.eax << 3) + cpu.eax, data); return; //2303;
	case M_DS_8EAX_ECX: cpu.ds.setByte((cpu.eax << 3) + cpu.ecx, data); return; //2304;
	case M_DS_8EAX_EDX: cpu.ds.setByte((cpu.eax << 3) + cpu.edx, data); return; //2305;
	case M_DS_8EAX_EBX: cpu.ds.setByte((cpu.eax << 3) + cpu.ebx, data); return; //2306;
	case M_DS_8EAX_ESP: cpu.ds.setByte((cpu.eax << 3) + cpu.esp, data); return; //2307;
	case M_DS_8EAX_ID: cpu.ds.setByte((cpu.eax << 3) + getMicrocode(), data); return; //2308;
	case M_DS_8EAX_ESI: cpu.ds.setByte((cpu.eax << 3) + cpu.esi, data); return; //2309;
	case M_DS_8EAX_EDI: cpu.ds.setByte((cpu.eax << 3) + cpu.edi, data); return; //2310;
	    
	case M_DS_8ECX_EAX: cpu.ds.setByte((cpu.ecx << 3) + cpu.eax, data); return; //2311;
	case M_DS_9ECX: cpu.ds.setByte((cpu.ecx << 3) + cpu.ecx, data); return; //2312;
	case M_DS_8ECX_EDX: cpu.ds.setByte((cpu.ecx << 3) + cpu.edx, data); return; //2313;
	case M_DS_8ECX_EBX: cpu.ds.setByte((cpu.ecx << 3) + cpu.ebx, data); return; //2314;
	case M_DS_8ECX_ESP: cpu.ds.setByte((cpu.ecx << 3) + cpu.esp, data); return; //2315;
	case M_DS_8ECX_ID: cpu.ds.setByte((cpu.ecx << 3) + getMicrocode(), data); return; //2316;
	case M_DS_8ECX_ESI: cpu.ds.setByte((cpu.ecx << 3) + cpu.esi, data); return; //2317;
	case M_DS_8ECX_EDI: cpu.ds.setByte((cpu.ecx << 3) + cpu.edi, data); return; //2318;
	    
	case M_DS_8EDX_EAX: cpu.ds.setByte((cpu.edx << 3) + cpu.eax, data); return; //2319;
	case M_DS_8EDX_ECX: cpu.ds.setByte((cpu.edx << 3) + cpu.ecx, data); return; //2320;
	case M_DS_9EDX: cpu.ds.setByte((cpu.edx << 3) + cpu.edx, data); return; //2321;
	case M_DS_8EDX_EBX: cpu.ds.setByte((cpu.edx << 3) + cpu.ebx, data); return; //2322;
	case M_DS_8EDX_ESP: cpu.ds.setByte((cpu.edx << 3) + cpu.esp, data); return; //2323;
	case M_DS_8EDX_ID: cpu.ds.setByte((cpu.edx << 3) + getMicrocode(), data); return; //2324;
	case M_DS_8EDX_ESI: cpu.ds.setByte((cpu.edx << 3) + cpu.esi, data); return; //2325;
	case M_DS_8EDX_EDI: cpu.ds.setByte((cpu.edx << 3) + cpu.edi, data); return; //2326;
	    
	case M_DS_8EBX_EAX: cpu.ds.setByte((cpu.ebx << 3) + cpu.eax, data); return; //2327
	case M_DS_8EBX_ECX: cpu.ds.setByte((cpu.ebx << 3) + cpu.ecx, data); return; //2328
	case M_DS_8EBX_EDX: cpu.ds.setByte((cpu.ebx << 3) + cpu.edx, data); return; //2329
	case M_DS_9EBX: cpu.ds.setByte((cpu.ebx << 3) + cpu.ebx, data); return; //2330
	case M_DS_8EBX_ESP: cpu.ds.setByte((cpu.ebx << 3) + cpu.esp, data); return; //2331
	case M_DS_8EBX_ID: cpu.ds.setByte((cpu.ebx << 3) + getMicrocode(), data); return; //2332
	case M_DS_8EBX_ESI: cpu.ds.setByte((cpu.ebx << 3) + cpu.esi, data); return; //2333
	case M_DS_8EBX_EDI: cpu.ds.setByte((cpu.ebx << 3) + cpu.edi, data); return; //2334
	    
	case M_DS_8EBP_EAX: cpu.ds.setByte((cpu.ebp << 3) + cpu.eax, data); return; //2335;
	case M_DS_8EBP_ECX: cpu.ds.setByte((cpu.ebp << 3) + cpu.ecx, data); return; //2336;
	case M_DS_8EBP_EDX: cpu.ds.setByte((cpu.ebp << 3) + cpu.edx, data); return; //2337;
	case M_DS_8EBP_EBX: cpu.ds.setByte((cpu.ebp << 3) + cpu.ebx, data); return; //2338;
	case M_DS_8EBP_ESP: cpu.ds.setByte((cpu.ebp << 3) + cpu.esp, data); return; //2339;
	case M_DS_8EBP_ID: cpu.ds.setByte((cpu.ebp << 3) + getMicrocode(), data); return; //2340;
	case M_DS_8EBP_ESI: cpu.ds.setByte((cpu.ebp << 3) + cpu.esi, data); return; //2341;
	case M_DS_8EBP_EDI: cpu.ds.setByte((cpu.ebp << 3) + cpu.edi, data); return; //2342;
	    
	case M_DS_8ESI_EAX: cpu.ds.setByte((cpu.esi << 3) + cpu.eax, data); return; //2343;
	case M_DS_8ESI_ECX: cpu.ds.setByte((cpu.esi << 3) + cpu.ecx, data); return; //2344;
	case M_DS_8ESI_EDX: cpu.ds.setByte((cpu.esi << 3) + cpu.edx, data); return; //2345;
	case M_DS_8ESI_EBX: cpu.ds.setByte((cpu.esi << 3) + cpu.ebx, data); return; //2346;
	case M_DS_8ESI_ESP: cpu.ds.setByte((cpu.esi << 3) + cpu.esp, data); return; //2347;
	case M_DS_8ESI_ID: cpu.ds.setByte((cpu.esi << 3) + getMicrocode(), data); return; //2348;
	case M_DS_9ESI: cpu.ds.setByte((cpu.esi << 3) + cpu.esi, data); return; //2349;
	case M_DS_8ESI_EDI: cpu.ds.setByte((cpu.esi << 3) + cpu.edi, data); return; //2350;
	    
	case M_DS_8EDI_EAX: cpu.ds.setByte((cpu.edi << 3) + cpu.eax, data); return; //2351;
	case M_DS_8EDI_ECX: cpu.ds.setByte((cpu.edi << 3) + cpu.ecx, data); return; //2352;
	case M_DS_8EDI_EDX: cpu.ds.setByte((cpu.edi << 3) + cpu.edx, data); return; //2353;
	case M_DS_8EDI_EBX: cpu.ds.setByte((cpu.edi << 3) + cpu.ebx, data); return; //2354;
	case M_DS_8EDI_ESP: cpu.ds.setByte((cpu.edi << 3) + cpu.esp, data); return; //2355;
	case M_DS_8EDI_ID: cpu.ds.setByte((cpu.edi << 3) + getMicrocode(), data); return; //2356;
	case M_DS_8EDI_ESI: cpu.ds.setByte((cpu.edi << 3) + cpu.esi, data); return; //2357;
	case M_DS_9EDI: cpu.ds.setByte((cpu.edi << 3) + cpu.edi, data); return; //2358;
	    
	case M_DS_2EAX_IB: cpu.ds.setByte(cpu.eax + cpu.eax + getMicrocode(), data); return; //2359
	case M_DS_EAX_ECX_IB: cpu.ds.setByte(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2360
	case M_DS_EAX_EDX_IB: cpu.ds.setByte(cpu.eax + cpu.edx + getMicrocode(), data); return; //2361
	case M_DS_EAX_EBX_IB: cpu.ds.setByte(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2362
	case M_DS_EAX_ESP_IB: cpu.ds.setByte(cpu.eax + cpu.esp + getMicrocode(), data); return; //2363
	case M_DS_EAX_EBP_IB: cpu.ds.setByte(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2364
	case M_DS_EAX_ESI_IB: cpu.ds.setByte(cpu.eax + cpu.esi + getMicrocode(), data); return; //2365
	case M_DS_EAX_EDI_IB: cpu.ds.setByte(cpu.eax + cpu.edi + getMicrocode(), data); return; //2366

	case M_DS_2ECX_IB: cpu.ds.setByte(cpu.ecx + cpu.ecx + getMicrocode(), data); return; //2367;
	case M_DS_ECX_EDX_IB: cpu.ds.setByte(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2368;
	case M_DS_ECX_EBX_IB: cpu.ds.setByte(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2369;
	case M_DS_ECX_ESP_IB: cpu.ds.setByte(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2370;
	case M_DS_ECX_EBP_IB: cpu.ds.setByte(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2371;
	case M_DS_ECX_ESI_IB: cpu.ds.setByte(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2372;
	case M_DS_ECX_EDI_IB: cpu.ds.setByte(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2373;

	case M_DS_2EDX_IB: cpu.ds.setByte(cpu.edx + cpu.edx + getMicrocode(), data); return; //2374
	case M_DS_EDX_EBX_IB: cpu.ds.setByte(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2375
	case M_DS_EDX_ESP_IB: cpu.ds.setByte(cpu.edx + cpu.esp + getMicrocode(), data); return; //2376
	case M_DS_EDX_EBP_IB: cpu.ds.setByte(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2377
	case M_DS_EDX_ESI_IB: cpu.ds.setByte(cpu.edx + cpu.esi + getMicrocode(), data); return; //2378
	case M_DS_EDX_EDI_IB: cpu.ds.setByte(cpu.edx + cpu.edi + getMicrocode(), data); return; //2379

	case M_DS_2EBX_IB: cpu.ds.setByte(cpu.ebx + cpu.ebx + getMicrocode(), data); return; //2380
	case M_DS_EBX_ESP_IB: cpu.ds.setByte(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2381
	case M_DS_EBX_EBP_IB: cpu.ds.setByte(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2382
	case M_DS_EBX_ESI_IB: cpu.ds.setByte(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2383
	case M_DS_EBX_EDI_IB: cpu.ds.setByte(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2384

	case M_DS_ESP_IB: cpu.ds.setByte(cpu.esp + getMicrocode(), data); return; //2385

	case M_DS_EBP_ESP_IB: cpu.ds.setByte(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2386;
	case M_DS_2EBP_IB: cpu.ds.setByte(cpu.ebp + cpu.ebp + getMicrocode(), data); return; //2387;
	case M_DS_EBP_ESI_IB: cpu.ds.setByte(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2388;
	case M_DS_EBP_EDI_IB: cpu.ds.setByte(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2389;

	case M_DS_ESI_ESP_IB: cpu.ds.setByte(cpu.esi + cpu.esp + getMicrocode(), data); return; //2390;
	case M_DS_2ESI_IB: cpu.ds.setByte(cpu.esi + cpu.esi + getMicrocode(), data); return; //2391;
	case M_DS_ESI_EDI_IB: cpu.ds.setByte(cpu.esi + cpu.edi + getMicrocode(), data); return; //2392;
	    
	case M_DS_EDI_ESP_IB: cpu.ds.setByte(cpu.edi + cpu.esp + getMicrocode(), data); return; //2393;
	case M_DS_2EDI_IB: cpu.ds.setByte(cpu.edi + cpu.edi + getMicrocode(), data); return; //2394;
	    
	case M_DS_3EAX_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2395;
	case M_DS_2EAX_ECX_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2396;
	case M_DS_2EAX_EDX_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2397;
	case M_DS_2EAX_EBX_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2398;
	case M_DS_2EAX_ESP_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2399;
	case M_DS_2EAX_EBP_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2400;
	case M_DS_2EAX_ESI_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2401;
	case M_DS_2EAX_EDI_IB: cpu.ds.setByte((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2402;
	    
	case M_DS_2ECX_EAX_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2403;
	case M_DS_3ECX_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2404;
	case M_DS_2ECX_EDX_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2405;
	case M_DS_2ECX_EBX_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2406;
	case M_DS_2ECX_ESP_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2407;
	case M_DS_2ECX_EBP_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2408;
	case M_DS_2ECX_ESI_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2409;
	case M_DS_2ECX_EDI_IB: cpu.ds.setByte((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2410;

	case M_DS_2EDX_EAX_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2411;
	case M_DS_2EDX_ECX_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2412;
	case M_DS_3EDX_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2413;
	case M_DS_2EDX_EBX_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2414;
	case M_DS_2EDX_ESP_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2415;
	case M_DS_2EDX_EBP_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2416;
	case M_DS_2EDX_ESI_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2417;
	case M_DS_2EDX_EDI_IB: cpu.ds.setByte((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2418;
	    
	case M_DS_2EBX_EAX_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2419;
	case M_DS_2EBX_ECX_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2420;
	case M_DS_2EBX_EDX_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2421;
	case M_DS_3EBX_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2422;
	case M_DS_2EBX_ESP_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2423;
	case M_DS_2EBX_EBP_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2424;
	case M_DS_2EBX_ESI_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2425;
	case M_DS_2EBX_EDI_IB: cpu.ds.setByte((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2426;
	    
	case M_DS_2EBP_EAX_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2427;
	case M_DS_2EBP_ECX_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2428;
	case M_DS_2EBP_EDX_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2429;
	case M_DS_2EBP_EBX_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2430;
	case M_DS_2EBP_ESP_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2431;
	case M_DS_3EBP_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2432;
	case M_DS_2EBP_ESI_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2433;
	case M_DS_2EBP_EDI_IB: cpu.ds.setByte((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2434;
	    
	case M_DS_2ESI_EAX_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2435;
	case M_DS_2ESI_ECX_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2436;
	case M_DS_2ESI_EDX_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2437;
	case M_DS_2ESI_EBX_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2438;
	case M_DS_2ESI_ESP_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2439;
	case M_DS_2ESI_EBP_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2440;
	case M_DS_3ESI_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2441;
	case M_DS_2ESI_EDI_IB: cpu.ds.setByte((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2442;
	    
	case M_DS_2EDI_EAX_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2443;
	case M_DS_2EDI_ECX_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2444;
	case M_DS_2EDI_EDX_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2445;
	case M_DS_2EDI_EBX_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2446;
	case M_DS_2EDI_ESP_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2447;
	case M_DS_2EDI_EBP_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2448;
	case M_DS_2EDI_ESI_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2449;
	case M_DS_3EDI_IB: cpu.ds.setByte((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2450;
	    
	case M_DS_5EAX_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2451;
	case M_DS_4EAX_ECX_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2452;
	case M_DS_4EAX_EDX_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2453;
	case M_DS_4EAX_EBX_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2454;
	case M_DS_4EAX_ESP_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2455;
	case M_DS_4EAX_EBP_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2456;
	case M_DS_4EAX_ESI_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2457;
	case M_DS_4EAX_EDI_IB: cpu.ds.setByte((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2458;

	case M_DS_4ECX_EAX_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2459;
	case M_DS_5ECX_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2460;
	case M_DS_4ECX_EDX_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2461;
	case M_DS_4ECX_EBX_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2462;
	case M_DS_4ECX_ESP_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2463;
	case M_DS_4ECX_EBP_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2464;
	case M_DS_4ECX_ESI_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2465;
	case M_DS_4ECX_EDI_IB: cpu.ds.setByte((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2466;

	case M_DS_4EDX_EAX_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2467
	case M_DS_4EDX_ECX_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2468
	case M_DS_5EDX_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2469
	case M_DS_4EDX_EBX_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2470
	case M_DS_4EDX_ESP_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2471
	case M_DS_4EDX_EBP_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2472
	case M_DS_4EDX_ESI_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2473
	case M_DS_4EDX_EDI_IB: cpu.ds.setByte((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2474

	case M_DS_4EBX_EAX_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2475;
	case M_DS_4EBX_ECX_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2476;
	case M_DS_4EBX_EDX_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2477;
	case M_DS_5EBX_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2478;
	case M_DS_4EBX_ESP_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2479;
	case M_DS_4EBX_EBP_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2480;
	case M_DS_4EBX_ESI_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2481;
	case M_DS_4EBX_EDI_IB: cpu.ds.setByte((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2482;

	case M_DS_4EBP_EAX_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2483;
	case M_DS_4EBP_ECX_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2484;
	case M_DS_4EBP_EDX_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2485;
	case M_DS_4EBP_EBX_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2486;
	case M_DS_4EBP_ESP_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2487;
	case M_DS_5EBP_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2488;
	case M_DS_4EBP_ESI_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2489;
	case M_DS_4EBP_EDI_IB: cpu.ds.setByte((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2490;
	    
	case M_DS_4ESI_EAX_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2491;
	case M_DS_4ESI_ECX_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2492;
	case M_DS_4ESI_EDX_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2493;
	case M_DS_4ESI_EBX_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2494;
	case M_DS_4ESI_ESP_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2495;
	case M_DS_4ESI_EBP_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2496
	case M_DS_5ESI_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2497;
	case M_DS_4ESI_EDI_IB: cpu.ds.setByte((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2498;
	    
	case M_DS_4EDI_EAX_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2499;
	case M_DS_4EDI_ECX_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2500;
	case M_DS_4EDI_EDX_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2501;
	case M_DS_4EDI_EBX_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2502;
	case M_DS_4EDI_ESP_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2503;
	case M_DS_4EDI_EBP_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2504;
	case M_DS_4EDI_ESI_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2505;
	case M_DS_5EDI_IB: cpu.ds.setByte((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2506;

	case M_DS_9EAX_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2507;
	case M_DS_8EAX_ECX_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2508;
	case M_DS_8EAX_EDX_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2509;
	case M_DS_8EAX_EBX_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2510;
	case M_DS_8EAX_ESP_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2511;
	case M_DS_8EAX_EBP_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2512;
	case M_DS_8EAX_ESI_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2513;
	case M_DS_8EAX_EDI_IB: cpu.ds.setByte((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2514;
	    
	case M_DS_8ECX_EAX_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2515;
	case M_DS_9ECX_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2516;
	case M_DS_8ECX_EDX_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2517;
	case M_DS_8ECX_EBX_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2518;
	case M_DS_8ECX_ESP_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2519;
	case M_DS_8ECX_EBP_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2520;
	case M_DS_8ECX_ESI_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2521;
	case M_DS_8ECX_EDI_IB: cpu.ds.setByte((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2522;
	    
	case M_DS_8EDX_EAX_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2523;
	case M_DS_8EDX_ECX_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2524;
	case M_DS_9EDX_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2525;
	case M_DS_8EDX_EBX_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2526;
	case M_DS_8EDX_ESP_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2527;
	case M_DS_8EDX_EBP_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2528;
	case M_DS_8EDX_ESI_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2529;
	case M_DS_8EDX_EDI_IB: cpu.ds.setByte((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2530;
	    
	case M_DS_8EBX_EAX_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2531
	case M_DS_8EBX_ECX_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2532
	case M_DS_8EBX_EDX_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2533
	case M_DS_9EBX_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2534
	case M_DS_8EBX_ESP_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2535
	case M_DS_8EBX_EBP_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2536
	case M_DS_8EBX_ESI_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2537
	case M_DS_8EBX_EDI_IB: cpu.ds.setByte((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2538
	    
	case M_DS_8EBP_EAX_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2539;
	case M_DS_8EBP_ECX_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2540;
	case M_DS_8EBP_EDX_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2541;
	case M_DS_8EBP_EBX_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2542;
	case M_DS_8EBP_ESP_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2543;
	case M_DS_9EBP_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2544;
	case M_DS_8EBP_ESI_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2545;
	case M_DS_8EBP_EDI_IB: cpu.ds.setByte((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2546;
	    
	case M_DS_8ESI_EAX_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2547;
	case M_DS_8ESI_ECX_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2548;
	case M_DS_8ESI_EDX_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2549;
	case M_DS_8ESI_EBX_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2550;
	case M_DS_8ESI_ESP_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2551;
	case M_DS_8ESI_EBP_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2552;
	case M_DS_9ESI_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2553;
	case M_DS_8ESI_EDI_IB: cpu.ds.setByte((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2554;
	    
	case M_DS_8EDI_EAX_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2555;
	case M_DS_8EDI_ECX_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2556;
	case M_DS_8EDI_EDX_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2557;
	case M_DS_8EDI_EBX_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2558;
	case M_DS_8EDI_ESP_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2559;
	case M_DS_8EDI_EBP_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2560;
	case M_DS_8EDI_ESI_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2561;
	case M_DS_9EDI_IB: cpu.ds.setByte((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2562;

    //SIB + ID Operands
	case M_DS_EAX_ECX_ID: cpu.ds.setByte(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2563;
	case M_DS_EAX_EDX_ID: cpu.ds.setByte(cpu.eax + cpu.edx + getMicrocode(), data); return; //2564;
	case M_DS_EAX_EBX_ID: cpu.ds.setByte(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2565;
	case M_DS_EAX_ESP_ID: cpu.ds.setByte(cpu.eax + cpu.esp + getMicrocode(), data); return; //2566;
	case M_DS_EAX_EBP_ID: cpu.ds.setByte(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2567;
	case M_DS_EAX_ESI_ID: cpu.ds.setByte(cpu.eax + cpu.esi + getMicrocode(), data); return; //2568;
	case M_DS_EAX_EDI_ID: cpu.ds.setByte(cpu.eax + cpu.edi + getMicrocode(), data); return; //2569;
	    
	case M_DS_ECX_EDX_ID: cpu.ds.setByte(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2570;
	case M_DS_ECX_EBX_ID: cpu.ds.setByte(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2571;
	case M_DS_ECX_ESP_ID: cpu.ds.setByte(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2572;
	case M_DS_ECX_EBP_ID: cpu.ds.setByte(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2573;
	case M_DS_ECX_ESI_ID: cpu.ds.setByte(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2574;
	case M_DS_ECX_EDI_ID: cpu.ds.setByte(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2575;
	    
	case M_DS_EDX_EBX_ID: cpu.ds.setByte(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2576
	case M_DS_EDX_ESP_ID: cpu.ds.setByte(cpu.edx + cpu.esp + getMicrocode(), data); return; //2577
	case M_DS_EDX_EBP_ID: cpu.ds.setByte(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2578
	case M_DS_EDX_ESI_ID: cpu.ds.setByte(cpu.edx + cpu.esi + getMicrocode(), data); return; //2579
	case M_DS_EDX_EDI_ID: cpu.ds.setByte(cpu.edx + cpu.edi + getMicrocode(), data); return; //2580

	case M_DS_EBX_ESP_ID: cpu.ds.setByte(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2581;
	case M_DS_EBX_EBP_ID: cpu.ds.setByte(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2582;
	case M_DS_EBX_ESI_ID: cpu.ds.setByte(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2583;
	case M_DS_EBX_EDI_ID: cpu.ds.setByte(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2584;
	    
	case M_DS_ESP_ID: cpu.ds.setByte(cpu.esp + getMicrocode(), data); return; //2585
	    
	case M_DS_EBP_ESP_ID: cpu.ds.setByte(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2586;
	case M_DS_EBP_ESI_ID: cpu.ds.setByte(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2587;
	case M_DS_EBP_EDI_ID: cpu.ds.setByte(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2588;
	    
	case M_DS_ESI_ESP_ID: cpu.ds.setByte(cpu.esi + cpu.esp + getMicrocode(), data); return; //2589;
	case M_DS_ESI_EDI_ID: cpu.ds.setByte(cpu.esi + cpu.edi + getMicrocode(), data); return; //2590;
	    
	case M_DS_EDI_ESP_ID: cpu.ds.setByte(cpu.edi + cpu.esp + getMicrocode(), data); return; //2591;
	    
	case M_DS_3EAX_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2592;
	case M_DS_2EAX_ECX_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2593;
	case M_DS_2EAX_EDX_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2594;
	case M_DS_2EAX_EBX_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2595;
	case M_DS_2EAX_ESP_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2596;
	case M_DS_2EAX_EBP_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2597;
	case M_DS_2EAX_ESI_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2598;
	case M_DS_2EAX_EDI_ID: cpu.ds.setByte((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2599;
	    
	case M_DS_2ECX_EAX_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2600;
	case M_DS_3ECX_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2601;
	case M_DS_2ECX_EDX_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2602;
	case M_DS_2ECX_EBX_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2603;
	case M_DS_2ECX_ESP_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2604;
	case M_DS_2ECX_EBP_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2605;
	case M_DS_2ECX_ESI_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2606;
	case M_DS_2ECX_EDI_ID: cpu.ds.setByte((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2607;
	    
	case M_DS_2EDX_EAX_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2608;
	case M_DS_2EDX_ECX_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2609;
	case M_DS_3EDX_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2610;
	case M_DS_2EDX_EBX_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2611;
	case M_DS_2EDX_ESP_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2612;
	case M_DS_2EDX_EBP_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2613;
	case M_DS_2EDX_ESI_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2614;
	case M_DS_2EDX_EDI_ID: cpu.ds.setByte((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2615;
	    
	case M_DS_2EBX_EAX_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2616;
	case M_DS_2EBX_ECX_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2617;
	case M_DS_2EBX_EDX_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2618;
	case M_DS_3EBX_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2619;
	case M_DS_2EBX_ESP_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2620;
	case M_DS_2EBX_EBP_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2621;
	case M_DS_2EBX_ESI_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2622;
	case M_DS_2EBX_EDI_ID: cpu.ds.setByte((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2623;
	    
	case M_DS_2EBP_EAX_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2624;
	case M_DS_2EBP_ECX_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2625;
	case M_DS_2EBP_EDX_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2626;
	case M_DS_2EBP_EBX_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2627;
	case M_DS_2EBP_ESP_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2628;
	case M_DS_3EBP_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2629;
	case M_DS_2EBP_ESI_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2630;
	case M_DS_2EBP_EDI_ID: cpu.ds.setByte((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2631;
	    
	case M_DS_2ESI_EAX_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2632;
	case M_DS_2ESI_ECX_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2633;
	case M_DS_2ESI_EDX_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2634;
	case M_DS_2ESI_EBX_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2635;
	case M_DS_2ESI_ESP_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2636;
	case M_DS_2ESI_EBP_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2637;
	case M_DS_3ESI_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2638;
	case M_DS_2ESI_EDI_ID: cpu.ds.setByte((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2639;
	    
	case M_DS_2EDI_EAX_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2640;
	case M_DS_2EDI_ECX_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2641;
	case M_DS_2EDI_EDX_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2642;
	case M_DS_2EDI_EBX_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2643;
	case M_DS_2EDI_ESP_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2644;
	case M_DS_2EDI_EBP_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2645;
	case M_DS_2EDI_ESI_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2646;
	case M_DS_3EDI_ID: cpu.ds.setByte((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2647;
	    
	case M_DS_5EAX_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2648
	case M_DS_4EAX_ECX_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2649
	case M_DS_4EAX_EDX_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2650
	case M_DS_4EAX_EBX_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2651
	case M_DS_4EAX_ESP_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2652
	case M_DS_4EAX_EBP_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2653
	case M_DS_4EAX_ESI_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2654
	case M_DS_4EAX_EDI_ID: cpu.ds.setByte((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2655

	case M_DS_4ECX_EAX_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2656;
	case M_DS_5ECX_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2657;
	case M_DS_4ECX_EDX_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2658;
	case M_DS_4ECX_EBX_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2659;
	case M_DS_4ECX_ESP_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2660;
	case M_DS_4ECX_EBP_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2661;
	case M_DS_4ECX_ESI_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2662;
	case M_DS_4ECX_EDI_ID: cpu.ds.setByte((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2663;
	    
	case M_DS_4EDX_EAX_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2664;
	case M_DS_4EDX_ECX_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2665;
	case M_DS_5EDX_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2666;
	case M_DS_4EDX_EBX_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2667;
	case M_DS_4EDX_ESP_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2668;
	case M_DS_4EDX_EBP_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2669;
	case M_DS_4EDX_ESI_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2670;
	case M_DS_4EDX_EDI_ID: cpu.ds.setByte((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2671;
	    
	case M_DS_4EBX_EAX_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2672;
	case M_DS_4EBX_ECX_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2673;
	case M_DS_4EBX_EDX_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2674;
	case M_DS_5EBX_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2675;
	case M_DS_4EBX_ESP_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2676;
	case M_DS_4EBX_EBP_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2677;
	case M_DS_4EBX_ESI_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2678;
	case M_DS_4EBX_EDI_ID: cpu.ds.setByte((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2679;
	    
	case M_DS_4EBP_EAX_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2680;
	case M_DS_4EBP_ECX_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2681;
	case M_DS_4EBP_EDX_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2682;
	case M_DS_4EBP_EBX_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2683;
	case M_DS_4EBP_ESP_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2684;
	case M_DS_5EBP_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2685;
	case M_DS_4EBP_ESI_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2686;
	case M_DS_4EBP_EDI_ID: cpu.ds.setByte((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2687;
	    
	case M_DS_4ESI_EAX_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2688;
	case M_DS_4ESI_ECX_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2689;
	case M_DS_4ESI_EDX_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2690;
	case M_DS_4ESI_EBX_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2691;
	case M_DS_4ESI_ESP_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2692;
	case M_DS_4ESI_EBP_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2693;
	case M_DS_5ESI_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2694;
	case M_DS_4ESI_EDI_ID: cpu.ds.setByte((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2695;

	case M_DS_4EDI_EAX_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2696;
	case M_DS_4EDI_ECX_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2697;
	case M_DS_4EDI_EDX_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2698;
	case M_DS_4EDI_EBX_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2699;
	case M_DS_4EDI_ESP_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2700;
	case M_DS_4EDI_EBP_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2701;
	case M_DS_4EDI_ESI_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2702;
	case M_DS_5EDI_ID: cpu.ds.setByte((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2703;	    

	case M_DS_9EAX_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2704;
	case M_DS_8EAX_ECX_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2705;
	case M_DS_8EAX_EDX_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2706;
	case M_DS_8EAX_EBX_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2707;
	case M_DS_8EAX_ESP_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2708;
	case M_DS_8EAX_EBP_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2709;
	case M_DS_8EAX_ESI_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2710;
	case M_DS_8EAX_EDI_ID: cpu.ds.setByte((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2711;

	case M_DS_8ECX_EAX_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2712;
	case M_DS_9ECX_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2713;
	case M_DS_8ECX_EDX_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2714;
	case M_DS_8ECX_EBX_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2715;
	case M_DS_8ECX_ESP_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2716;
	case M_DS_8ECX_EBP_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2717;
	case M_DS_8ECX_ESI_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2718;
	case M_DS_8ECX_EDI_ID: cpu.ds.setByte((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2719;

	case M_DS_8EDX_EAX_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2720;
	case M_DS_8EDX_ECX_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2721;
	case M_DS_9EDX_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2722;
	case M_DS_8EDX_EBX_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2723;
	case M_DS_8EDX_ESP_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2724;
	case M_DS_8EDX_EBP_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2725;
	case M_DS_8EDX_ESI_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2726;
	case M_DS_8EDX_EDI_ID: cpu.ds.setByte((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2727;

	case M_DS_8EBX_EAX_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2728;
	case M_DS_8EBX_ECX_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2729;
	case M_DS_8EBX_EDX_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2730;
	case M_DS_9EBX_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2731;
	case M_DS_8EBX_ESP_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2732;
	case M_DS_8EBX_EBP_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2733;
	case M_DS_8EBX_ESI_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2734;
	case M_DS_8EBX_EDI_ID: cpu.ds.setByte((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2735;
	    
	case M_DS_8EBP_EAX_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2736;
	case M_DS_8EBP_ECX_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2737;
	case M_DS_8EBP_EDX_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2738;
	case M_DS_8EBP_EBX_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2739;
	case M_DS_8EBP_ESP_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2740;
	case M_DS_9EBP_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2741;
	case M_DS_8EBP_ESI_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2742;
	case M_DS_8EBP_EDI_ID: cpu.ds.setByte((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2743;
	    
	case M_DS_8ESI_EAX_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2744;
	case M_DS_8ESI_ECX_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2745;
	case M_DS_8ESI_EDX_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2746;
	case M_DS_8ESI_EBX_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2747;
	case M_DS_8ESI_ESP_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2748;
	case M_DS_8ESI_EBP_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2749;
	case M_DS_9ESI_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2750;
	case M_DS_8ESI_EDI_ID: cpu.ds.setByte((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2751;
    
	case M_DS_8EDI_EAX_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2752;
	case M_DS_8EDI_ECX_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2753;
	case M_DS_8EDI_EDX_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2754;
	case M_DS_8EDI_EBX_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2755;
	case M_DS_8EDI_ESP_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2756;
	case M_DS_8EDI_EBP_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2757;
	case M_DS_8EDI_ESI_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2758;
	case M_DS_9EDI_ID: cpu.ds.setByte((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2759;

	case M_SS_2EAX: cpu.ss.setByte(cpu.eax + cpu.eax, data); return; //2156
	case M_SS_EAX_ECX: cpu.ss.setByte(cpu.eax + cpu.ecx, data); return; //2157
	case M_SS_EAX_EDX: cpu.ss.setByte(cpu.eax + cpu.edx, data); return; //2158
	case M_SS_EAX_EBX: cpu.ss.setByte(cpu.eax + cpu.ebx, data); return; //2159
	case M_SS_EAX_ESP: cpu.ss.setByte(cpu.eax + cpu.esp, data); return; //2160
	case M_SS_EAX_EBP: cpu.ss.setByte(cpu.eax + cpu.ebp, data); return; //2161
	case M_SS_EAX_ESI: cpu.ss.setByte(cpu.eax + cpu.esi, data); return; //2162
	case M_SS_EAX_EDI: cpu.ss.setByte(cpu.eax + cpu.edi, data); return; //2163

	case M_SS_2ECX: cpu.ss.setByte(cpu.ecx + cpu.ecx, data); return; //2164
	case M_SS_ECX_EDX: cpu.ss.setByte(cpu.ecx + cpu.edx, data); return; //2165
	case M_SS_ECX_EBX: cpu.ss.setByte(cpu.ecx + cpu.ebx, data); return; //2166
	case M_SS_ECX_ESP: cpu.ss.setByte(cpu.ecx + cpu.esp, data); return; //2167
	case M_SS_ECX_EBP: cpu.ss.setByte(cpu.ecx + cpu.ebp, data); return; //2168
	case M_SS_ECX_ESI: cpu.ss.setByte(cpu.ecx + cpu.esi, data); return; //2169
	case M_SS_ECX_EDI: cpu.ss.setByte(cpu.ecx + cpu.edi, data); return; //2170

	case M_SS_2EDX: cpu.ss.setByte(cpu.edx + cpu.edx, data); return; //2171
	case M_SS_EDX_EBX: cpu.ss.setByte(cpu.edx + cpu.ebx, data); return; //2172
	case M_SS_EDX_ESP: cpu.ss.setByte(cpu.edx + cpu.esp, data); return; //2173
	case M_SS_EDX_EBP: cpu.ss.setByte(cpu.edx + cpu.ebp, data); return; //2174
	case M_SS_EDX_ESI: cpu.ss.setByte(cpu.edx + cpu.esi, data); return; //2175
	case M_SS_EDX_EDI: cpu.ss.setByte(cpu.edx + cpu.edi, data); return; //2176

	case M_SS_2EBX: cpu.ss.setByte(cpu.ebx + cpu.ebx, data); return; //2177;
	case M_SS_EBX_ESP: cpu.ss.setByte(cpu.ebx + cpu.esp, data); return; //2178;
	case M_SS_EBX_EBP: cpu.ss.setByte(cpu.ebx + cpu.ebp, data); return; //2179;
	case M_SS_EBX_ESI: cpu.ss.setByte(cpu.ebx + cpu.esi, data); return; //2180;
	case M_SS_EBX_EDI: cpu.ss.setByte(cpu.ebx + cpu.edi, data); return; //2181;
	    
	case M_SS_ESP: cpu.ss.setByte(cpu.esp, data); return; //2182

	case M_SS_EBP_ESP: cpu.ss.setByte(cpu.ebp + cpu.esp, data); return; //2183;
	case M_SS_EBP_ESI: cpu.ss.setByte(cpu.ebp + cpu.esi, data); return; //2184;
	case M_SS_EBP_EDI: cpu.ss.setByte(cpu.ebp + cpu.edi, data); return; //2185;

	case M_SS_ESI_ESP: cpu.ss.setByte(cpu.esi + cpu.esp, data); return; //2186;
	case M_SS_2ESI: cpu.ss.setByte(cpu.esi + cpu.esi, data); return; //2187;
	case M_SS_ESI_EDI: cpu.ss.setByte(cpu.esi + cpu.edi, data); return; //2188;

	case M_SS_EDI_ESP: cpu.ss.setByte(cpu.edi + cpu.esp, data); return; //2189;
	case M_SS_2EDI: cpu.ss.setByte(cpu.edi + cpu.edi, data); return; //2190;

	case M_SS_3EAX: cpu.ss.setByte((cpu.eax << 1) + cpu.eax, data); return; //2191;
	case M_SS_2EAX_ECX: cpu.ss.setByte((cpu.eax << 1) + cpu.ecx, data); return; //2192;
	case M_SS_2EAX_EDX: cpu.ss.setByte((cpu.eax << 1) + cpu.edx, data); return; //2193;
	case M_SS_2EAX_EBX: cpu.ss.setByte((cpu.eax << 1) + cpu.ebx, data); return; //2194;
	case M_SS_2EAX_ESP: cpu.ss.setByte((cpu.eax << 1) + cpu.esp, data); return; //2195;
	case M_SS_2EAX_ID: cpu.ss.setByte((cpu.eax << 1) + getMicrocode(), data); return; //2196;
	case M_SS_2EAX_ESI: cpu.ss.setByte((cpu.eax << 1) + cpu.esi, data); return; //2197;
	case M_SS_2EAX_EDI: cpu.ss.setByte((cpu.eax << 1) + cpu.edi, data); return; //2198;

	case M_SS_2ECX_EAX: cpu.ss.setByte((cpu.ecx << 1) + cpu.eax, data); return; //2199;
	case M_SS_3ECX: cpu.ss.setByte((cpu.ecx << 1) + cpu.ecx, data); return; //2200;
	case M_SS_2ECX_EDX: cpu.ss.setByte((cpu.ecx << 1) + cpu.edx, data); return; //2201;
	case M_SS_2ECX_EBX: cpu.ss.setByte((cpu.ecx << 1) + cpu.ebx, data); return; //2202;
	case M_SS_2ECX_ESP: cpu.ss.setByte((cpu.ecx << 1) + cpu.esp, data); return; //2203;
	case M_SS_2ECX_ID: cpu.ss.setByte((cpu.ecx << 1) + getMicrocode(), data); return; //2204;
	case M_SS_2ECX_ESI: cpu.ss.setByte((cpu.ecx << 1) + cpu.esi, data); return; //2205;
	case M_SS_2ECX_EDI: cpu.ss.setByte((cpu.ecx << 1) + cpu.edi, data); return; //2206;

	case M_SS_2EDX_EAX: cpu.ss.setByte((cpu.edx << 1) + cpu.eax, data); return; //2207;
	case M_SS_2EDX_ECX: cpu.ss.setByte((cpu.edx << 1) + cpu.ecx, data); return; //2208;
	case M_SS_3EDX: cpu.ss.setByte((cpu.edx << 1) + cpu.edx, data); return; //2209;
	case M_SS_2EDX_EBX: cpu.ss.setByte((cpu.edx << 1) + cpu.ebx, data); return; //2210;
	case M_SS_2EDX_ESP: cpu.ss.setByte((cpu.edx << 1) + cpu.esp, data); return; //2211;
	case M_SS_2EDX_ID: cpu.ss.setByte((cpu.edx << 1) + getMicrocode(), data); return; //2212;
	case M_SS_2EDX_ESI: cpu.ss.setByte((cpu.edx << 1) + cpu.esi, data); return; //2213;
	case M_SS_2EDX_EDI: cpu.ss.setByte((cpu.edx << 1) + cpu.edi, data); return; //2214;

	case M_SS_2EBX_EAX: cpu.ss.setByte((cpu.ebx << 1) + cpu.eax, data); return; //2215;
	case M_SS_2EBX_ECX: cpu.ss.setByte((cpu.ebx << 1) + cpu.ecx, data); return; //2216;
	case M_SS_2EBX_EDX: cpu.ss.setByte((cpu.ebx << 1) + cpu.edx, data); return; //2217;
	case M_SS_3EBX: cpu.ss.setByte((cpu.ebx << 1) + cpu.ebx, data); return; //2218;
	case M_SS_2EBX_ESP: cpu.ss.setByte((cpu.ebx << 1) + cpu.esp, data); return; //2219;
	case M_SS_2EBX_ID: cpu.ss.setByte((cpu.ebx << 1) + getMicrocode(), data); return; //2220;
	case M_SS_2EBX_ESI: cpu.ss.setByte((cpu.ebx << 1) + cpu.esi, data); return; //2221;
	case M_SS_2EBX_EDI: cpu.ss.setByte((cpu.ebx << 1) + cpu.edi, data); return; //2222;
	    
	case M_SS_2EBP_EAX: cpu.ss.setByte((cpu.ebp << 1) + cpu.eax, data); return; //2223;
	case M_SS_2EBP_ECX: cpu.ss.setByte((cpu.ebp << 1) + cpu.ecx, data); return; //2224;
	case M_SS_2EBP_EDX: cpu.ss.setByte((cpu.ebp << 1) + cpu.edx, data); return; //2225;
	case M_SS_2EBP_EBX: cpu.ss.setByte((cpu.ebp << 1) + cpu.ebx, data); return; //2226;
	case M_SS_2EBP_ESP: cpu.ss.setByte((cpu.ebp << 1) + cpu.esp, data); return; //2227;
	case M_SS_2EBP_ID: cpu.ss.setByte((cpu.ebp << 1) + getMicrocode(), data); return; //2228;
	case M_SS_2EBP_ESI: cpu.ss.setByte((cpu.ebp << 1) + cpu.esi, data); return; //2229;
	case M_SS_2EBP_EDI: cpu.ss.setByte((cpu.ebp << 1) + cpu.edi, data); return; //2230;

	case M_SS_2ESI_EAX: cpu.ss.setByte((cpu.esi << 1) + cpu.eax, data); return; //2231;
	case M_SS_2ESI_ECX: cpu.ss.setByte((cpu.esi << 1) + cpu.ecx, data); return; //2232;
	case M_SS_2ESI_EDX: cpu.ss.setByte((cpu.esi << 1) + cpu.edx, data); return; //2233;
	case M_SS_2ESI_EBX: cpu.ss.setByte((cpu.esi << 1) + cpu.ebx, data); return; //2234;
	case M_SS_2ESI_ESP: cpu.ss.setByte((cpu.esi << 1) + cpu.esp, data); return; //2235;
	case M_SS_2ESI_ID: cpu.ss.setByte((cpu.esi << 1) + getMicrocode(), data); return; //2236;
	case M_SS_3ESI: cpu.ss.setByte((cpu.esi << 1) + cpu.esi, data); return; //2237;
	case M_SS_2ESI_EDI: cpu.ss.setByte((cpu.esi << 1) + cpu.edi, data); return; //2238;

	case M_SS_2EDI_EAX: cpu.ss.setByte((cpu.edi << 1) + cpu.eax, data); return; //2239;
	case M_SS_2EDI_ECX: cpu.ss.setByte((cpu.edi << 1) + cpu.ecx, data); return; //2240;
	case M_SS_2EDI_EDX: cpu.ss.setByte((cpu.edi << 1) + cpu.edx, data); return; //2241;
	case M_SS_2EDI_EBX: cpu.ss.setByte((cpu.edi << 1) + cpu.ebx, data); return; //2242;
	case M_SS_2EDI_ESP: cpu.ss.setByte((cpu.edi << 1) + cpu.esp, data); return; //2243;
	case M_SS_2EDI_ID: cpu.ss.setByte((cpu.edi << 1) + getMicrocode(), data); return; //2244;
	case M_SS_2EDI_ESI: cpu.ss.setByte((cpu.edi << 1) + cpu.esi, data); return; //2245;
	case M_SS_3EDI: cpu.ss.setByte((cpu.edi << 1) + cpu.edi, data); return; //2246;

	case M_SS_5EAX: cpu.ss.setByte((cpu.eax << 2) + cpu.eax, data); return; //2247
	case M_SS_4EAX_ECX: cpu.ss.setByte((cpu.eax << 2) + cpu.ecx, data); return; //2248
	case M_SS_4EAX_EDX: cpu.ss.setByte((cpu.eax << 2) + cpu.edx, data); return; //2249
	case M_SS_4EAX_EBX: cpu.ss.setByte((cpu.eax << 2) + cpu.ebx, data); return; //2250
	case M_SS_4EAX_ESP: cpu.ss.setByte((cpu.eax << 2) + cpu.esp, data); return; //2251
	case M_SS_4EAX_ID: cpu.ss.setByte((cpu.eax << 2) + getMicrocode(), data); return; //2252
	case M_SS_4EAX_ESI: cpu.ss.setByte((cpu.eax << 2) + cpu.esi, data); return; //2253
	case M_SS_4EAX_EDI: cpu.ss.setByte((cpu.eax << 2) + cpu.edi, data); return; //2254

	case M_SS_4ECX_EAX: cpu.ss.setByte((cpu.ecx << 2) + cpu.eax, data); return; //2255
	case M_SS_5ECX: cpu.ss.setByte((cpu.ecx << 2) + cpu.ecx, data); return; //2256
	case M_SS_4ECX_EDX: cpu.ss.setByte((cpu.ecx << 2) + cpu.edx, data); return; //2257
	case M_SS_4ECX_EBX: cpu.ss.setByte((cpu.ecx << 2) + cpu.ebx, data); return; //2258
	case M_SS_4ECX_ESP: cpu.ss.setByte((cpu.ecx << 2) + cpu.esp, data); return; //2259
	case M_SS_4ECX_ID: cpu.ss.setByte((cpu.ecx << 2) + getMicrocode(), data); return; //2260
	case M_SS_4ECX_ESI: cpu.ss.setByte((cpu.ecx << 2) + cpu.esi, data); return; //2261
	case M_SS_4ECX_EDI: cpu.ss.setByte((cpu.ecx << 2) + cpu.edi, data); return; //2262

	case M_SS_4EDX_EAX: cpu.ss.setByte((cpu.edx << 2) + cpu.eax, data); return; //2263
	case M_SS_4EDX_ECX: cpu.ss.setByte((cpu.edx << 2) + cpu.ecx, data); return; //2264
	case M_SS_5EDX: cpu.ss.setByte((cpu.edx << 2) + cpu.edx, data); return; //2265
	case M_SS_4EDX_EBX: cpu.ss.setByte((cpu.edx << 2) + cpu.ebx, data); return; //2266
	case M_SS_4EDX_ESP: cpu.ss.setByte((cpu.edx << 2) + cpu.esp, data); return; //2267
	case M_SS_4EDX_ID: cpu.ss.setByte((cpu.edx << 2) + getMicrocode(), data); return; //2268
	case M_SS_4EDX_ESI: cpu.ss.setByte((cpu.edx << 2) + cpu.esi, data); return; //2269
	case M_SS_4EDX_EDI: cpu.ss.setByte((cpu.edx << 2) + cpu.edi, data); return; //2270

	case M_SS_4EBX_EAX: cpu.ss.setByte((cpu.ebx << 2) + cpu.eax, data); return; //2271;
	case M_SS_4EBX_ECX: cpu.ss.setByte((cpu.ebx << 2) + cpu.ecx, data); return; //2272;
	case M_SS_4EBX_EDX: cpu.ss.setByte((cpu.ebx << 2) + cpu.edx, data); return; //2273;
	case M_SS_5EBX: cpu.ss.setByte((cpu.ebx << 2) + cpu.ebx, data); return; //2274;
	case M_SS_4EBX_ESP: cpu.ss.setByte((cpu.ebx << 2) + cpu.esp, data); return; //2275;
	case M_SS_4EBX_ID: cpu.ss.setByte((cpu.ebx << 2) + getMicrocode(), data); return; //2276;
	case M_SS_4EBX_ESI: cpu.ss.setByte((cpu.ebx << 2) + cpu.esi, data); return; //2277;
	case M_SS_4EBX_EDI: cpu.ss.setByte((cpu.ebx << 2) + cpu.edi, data); return; //2278;
	    
	case M_SS_4EBP_EAX: cpu.ss.setByte((cpu.ebp << 2) + cpu.eax, data); return; //2279
	case M_SS_4EBP_ECX: cpu.ss.setByte((cpu.ebp << 2) + cpu.ecx, data); return; //2280
	case M_SS_4EBP_EDX: cpu.ss.setByte((cpu.ebp << 2) + cpu.edx, data); return; //2281
	case M_SS_4EBP_EBX: cpu.ss.setByte((cpu.ebp << 2) + cpu.ebx, data); return; //2282
	case M_SS_4EBP_ESP: cpu.ss.setByte((cpu.ebp << 2) + cpu.esp, data); return; //2283
	case M_SS_4EBP_ID: cpu.ss.setByte((cpu.ebp << 2) + getMicrocode(), data); return; //2284
	case M_SS_4EBP_ESI: cpu.ss.setByte((cpu.ebp << 2) + cpu.esi, data); return; //2285
	case M_SS_4EBP_EDI: cpu.ss.setByte((cpu.ebp << 2) + cpu.edi, data); return; //2286

	case M_SS_4ESI_EAX: cpu.ss.setByte((cpu.esi << 2) + cpu.eax, data); return; //2287;
	case M_SS_4ESI_ECX: cpu.ss.setByte((cpu.esi << 2) + cpu.ecx, data); return; //2288;
	case M_SS_4ESI_EDX: cpu.ss.setByte((cpu.esi << 2) + cpu.edx, data); return; //2289;
	case M_SS_4ESI_EBX: cpu.ss.setByte((cpu.esi << 2) + cpu.ebx, data); return; //2290;
	case M_SS_4ESI_ESP: cpu.ss.setByte((cpu.esi << 2) + cpu.esp, data); return; //2291;
	case M_SS_4ESI_ID: cpu.ss.setByte((cpu.esi << 2) + getMicrocode(), data); return; //2292;
	case M_SS_5ESI: cpu.ss.setByte((cpu.esi << 2) + cpu.esi, data); return; //2293;
	case M_SS_4ESI_EDI: cpu.ss.setByte((cpu.esi << 2) + cpu.edi, data); return; //2294;

	case M_SS_4EDI_EAX: cpu.ss.setByte((cpu.edi << 2) + cpu.eax, data); return; //2295;
	case M_SS_4EDI_ECX: cpu.ss.setByte((cpu.edi << 2) + cpu.ecx, data); return; //2296;
	case M_SS_4EDI_EDX: cpu.ss.setByte((cpu.edi << 2) + cpu.edx, data); return; //2297;
	case M_SS_4EDI_EBX: cpu.ss.setByte((cpu.edi << 2) + cpu.ebx, data); return; //2298;
	case M_SS_4EDI_ESP: cpu.ss.setByte((cpu.edi << 2) + cpu.esp, data); return; //2299;
	case M_SS_4EDI_ID: cpu.ss.setByte((cpu.edi << 2) + getMicrocode(), data); return; //2300;
	case M_SS_4EDI_ESI: cpu.ss.setByte((cpu.edi << 2) + cpu.esi, data); return; //2301;
	case M_SS_5EDI: cpu.ss.setByte((cpu.edi << 2) + cpu.edi, data); return; //2302;

	case M_SS_9EAX: cpu.ss.setByte((cpu.eax << 3) + cpu.eax, data); return; //2303;
	case M_SS_8EAX_ECX: cpu.ss.setByte((cpu.eax << 3) + cpu.ecx, data); return; //2304;
	case M_SS_8EAX_EDX: cpu.ss.setByte((cpu.eax << 3) + cpu.edx, data); return; //2305;
	case M_SS_8EAX_EBX: cpu.ss.setByte((cpu.eax << 3) + cpu.ebx, data); return; //2306;
	case M_SS_8EAX_ESP: cpu.ss.setByte((cpu.eax << 3) + cpu.esp, data); return; //2307;
	case M_SS_8EAX_ID: cpu.ss.setByte((cpu.eax << 3) + getMicrocode(), data); return; //2308;
	case M_SS_8EAX_ESI: cpu.ss.setByte((cpu.eax << 3) + cpu.esi, data); return; //2309;
	case M_SS_8EAX_EDI: cpu.ss.setByte((cpu.eax << 3) + cpu.edi, data); return; //2310;
	    
	case M_SS_8ECX_EAX: cpu.ss.setByte((cpu.ecx << 3) + cpu.eax, data); return; //2311;
	case M_SS_9ECX: cpu.ss.setByte((cpu.ecx << 3) + cpu.ecx, data); return; //2312;
	case M_SS_8ECX_EDX: cpu.ss.setByte((cpu.ecx << 3) + cpu.edx, data); return; //2313;
	case M_SS_8ECX_EBX: cpu.ss.setByte((cpu.ecx << 3) + cpu.ebx, data); return; //2314;
	case M_SS_8ECX_ESP: cpu.ss.setByte((cpu.ecx << 3) + cpu.esp, data); return; //2315;
	case M_SS_8ECX_ID: cpu.ss.setByte((cpu.ecx << 3) + getMicrocode(), data); return; //2316;
	case M_SS_8ECX_ESI: cpu.ss.setByte((cpu.ecx << 3) + cpu.esi, data); return; //2317;
	case M_SS_8ECX_EDI: cpu.ss.setByte((cpu.ecx << 3) + cpu.edi, data); return; //2318;
	    
	case M_SS_8EDX_EAX: cpu.ss.setByte((cpu.edx << 3) + cpu.eax, data); return; //2319;
	case M_SS_8EDX_ECX: cpu.ss.setByte((cpu.edx << 3) + cpu.ecx, data); return; //2320;
	case M_SS_9EDX: cpu.ss.setByte((cpu.edx << 3) + cpu.edx, data); return; //2321;
	case M_SS_8EDX_EBX: cpu.ss.setByte((cpu.edx << 3) + cpu.ebx, data); return; //2322;
	case M_SS_8EDX_ESP: cpu.ss.setByte((cpu.edx << 3) + cpu.esp, data); return; //2323;
	case M_SS_8EDX_ID: cpu.ss.setByte((cpu.edx << 3) + getMicrocode(), data); return; //2324;
	case M_SS_8EDX_ESI: cpu.ss.setByte((cpu.edx << 3) + cpu.esi, data); return; //2325;
	case M_SS_8EDX_EDI: cpu.ss.setByte((cpu.edx << 3) + cpu.edi, data); return; //2326;
	    
	case M_SS_8EBX_EAX: cpu.ss.setByte((cpu.ebx << 3) + cpu.eax, data); return; //2327
	case M_SS_8EBX_ECX: cpu.ss.setByte((cpu.ebx << 3) + cpu.ecx, data); return; //2328
	case M_SS_8EBX_EDX: cpu.ss.setByte((cpu.ebx << 3) + cpu.edx, data); return; //2329
	case M_SS_9EBX: cpu.ss.setByte((cpu.ebx << 3) + cpu.ebx, data); return; //2330
	case M_SS_8EBX_ESP: cpu.ss.setByte((cpu.ebx << 3) + cpu.esp, data); return; //2331
	case M_SS_8EBX_ID: cpu.ss.setByte((cpu.ebx << 3) + getMicrocode(), data); return; //2332
	case M_SS_8EBX_ESI: cpu.ss.setByte((cpu.ebx << 3) + cpu.esi, data); return; //2333
	case M_SS_8EBX_EDI: cpu.ss.setByte((cpu.ebx << 3) + cpu.edi, data); return; //2334
	    
	case M_SS_8EBP_EAX: cpu.ss.setByte((cpu.ebp << 3) + cpu.eax, data); return; //2335;
	case M_SS_8EBP_ECX: cpu.ss.setByte((cpu.ebp << 3) + cpu.ecx, data); return; //2336;
	case M_SS_8EBP_EDX: cpu.ss.setByte((cpu.ebp << 3) + cpu.edx, data); return; //2337;
	case M_SS_8EBP_EBX: cpu.ss.setByte((cpu.ebp << 3) + cpu.ebx, data); return; //2338;
	case M_SS_8EBP_ESP: cpu.ss.setByte((cpu.ebp << 3) + cpu.esp, data); return; //2339;
	case M_SS_8EBP_ID: cpu.ss.setByte((cpu.ebp << 3) + getMicrocode(), data); return; //2340;
	case M_SS_8EBP_ESI: cpu.ss.setByte((cpu.ebp << 3) + cpu.esi, data); return; //2341;
	case M_SS_8EBP_EDI: cpu.ss.setByte((cpu.ebp << 3) + cpu.edi, data); return; //2342;
	    
	case M_SS_8ESI_EAX: cpu.ss.setByte((cpu.esi << 3) + cpu.eax, data); return; //2343;
	case M_SS_8ESI_ECX: cpu.ss.setByte((cpu.esi << 3) + cpu.ecx, data); return; //2344;
	case M_SS_8ESI_EDX: cpu.ss.setByte((cpu.esi << 3) + cpu.edx, data); return; //2345;
	case M_SS_8ESI_EBX: cpu.ss.setByte((cpu.esi << 3) + cpu.ebx, data); return; //2346;
	case M_SS_8ESI_ESP: cpu.ss.setByte((cpu.esi << 3) + cpu.esp, data); return; //2347;
	case M_SS_8ESI_ID: cpu.ss.setByte((cpu.esi << 3) + getMicrocode(), data); return; //2348;
	case M_SS_9ESI: cpu.ss.setByte((cpu.esi << 3) + cpu.esi, data); return; //2349;
	case M_SS_8ESI_EDI: cpu.ss.setByte((cpu.esi << 3) + cpu.edi, data); return; //2350;
	    
	case M_SS_8EDI_EAX: cpu.ss.setByte((cpu.edi << 3) + cpu.eax, data); return; //2351;
	case M_SS_8EDI_ECX: cpu.ss.setByte((cpu.edi << 3) + cpu.ecx, data); return; //2352;
	case M_SS_8EDI_EDX: cpu.ss.setByte((cpu.edi << 3) + cpu.edx, data); return; //2353;
	case M_SS_8EDI_EBX: cpu.ss.setByte((cpu.edi << 3) + cpu.ebx, data); return; //2354;
	case M_SS_8EDI_ESP: cpu.ss.setByte((cpu.edi << 3) + cpu.esp, data); return; //2355;
	case M_SS_8EDI_ID: cpu.ss.setByte((cpu.edi << 3) + getMicrocode(), data); return; //2356;
	case M_SS_8EDI_ESI: cpu.ss.setByte((cpu.edi << 3) + cpu.esi, data); return; //2357;
	case M_SS_9EDI: cpu.ss.setByte((cpu.edi << 3) + cpu.edi, data); return; //2358;
	    
	case M_SS_2EAX_IB: cpu.ss.setByte(cpu.eax + cpu.eax + getMicrocode(), data); return; //2359
	case M_SS_EAX_ECX_IB: cpu.ss.setByte(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2360
	case M_SS_EAX_EDX_IB: cpu.ss.setByte(cpu.eax + cpu.edx + getMicrocode(), data); return; //2361
	case M_SS_EAX_EBX_IB: cpu.ss.setByte(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2362
	case M_SS_EAX_ESP_IB: cpu.ss.setByte(cpu.eax + cpu.esp + getMicrocode(), data); return; //2363
	case M_SS_EAX_EBP_IB: cpu.ss.setByte(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2364
	case M_SS_EAX_ESI_IB: cpu.ss.setByte(cpu.eax + cpu.esi + getMicrocode(), data); return; //2365
	case M_SS_EAX_EDI_IB: cpu.ss.setByte(cpu.eax + cpu.edi + getMicrocode(), data); return; //2366

	case M_SS_2ECX_IB: cpu.ss.setByte(cpu.ecx + cpu.ecx + getMicrocode(), data); return; //2367;
	case M_SS_ECX_EDX_IB: cpu.ss.setByte(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2368;
	case M_SS_ECX_EBX_IB: cpu.ss.setByte(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2369;
	case M_SS_ECX_ESP_IB: cpu.ss.setByte(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2370;
	case M_SS_ECX_EBP_IB: cpu.ss.setByte(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2371;
	case M_SS_ECX_ESI_IB: cpu.ss.setByte(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2372;
	case M_SS_ECX_EDI_IB: cpu.ss.setByte(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2373;

	case M_SS_2EDX_IB: cpu.ss.setByte(cpu.edx + cpu.edx + getMicrocode(), data); return; //2374
	case M_SS_EDX_EBX_IB: cpu.ss.setByte(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2375
	case M_SS_EDX_ESP_IB: cpu.ss.setByte(cpu.edx + cpu.esp + getMicrocode(), data); return; //2376
	case M_SS_EDX_EBP_IB: cpu.ss.setByte(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2377
	case M_SS_EDX_ESI_IB: cpu.ss.setByte(cpu.edx + cpu.esi + getMicrocode(), data); return; //2378
	case M_SS_EDX_EDI_IB: cpu.ss.setByte(cpu.edx + cpu.edi + getMicrocode(), data); return; //2379

	case M_SS_2EBX_IB: cpu.ss.setByte(cpu.ebx + cpu.ebx + getMicrocode(), data); return; //2380
	case M_SS_EBX_ESP_IB: cpu.ss.setByte(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2381
	case M_SS_EBX_EBP_IB: cpu.ss.setByte(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2382
	case M_SS_EBX_ESI_IB: cpu.ss.setByte(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2383
	case M_SS_EBX_EDI_IB: cpu.ss.setByte(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2384

	case M_SS_ESP_IB: cpu.ss.setByte(cpu.esp + getMicrocode(), data); return; //2385

	case M_SS_EBP_ESP_IB: cpu.ss.setByte(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2386;
	case M_SS_2EBP_IB: cpu.ss.setByte(cpu.ebp + cpu.ebp + getMicrocode(), data); return; //2387;
	case M_SS_EBP_ESI_IB: cpu.ss.setByte(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2388;
	case M_SS_EBP_EDI_IB: cpu.ss.setByte(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2389;

	case M_SS_ESI_ESP_IB: cpu.ss.setByte(cpu.esi + cpu.esp + getMicrocode(), data); return; //2390;
	case M_SS_2ESI_IB: cpu.ss.setByte(cpu.esi + cpu.esi + getMicrocode(), data); return; //2391;
	case M_SS_ESI_EDI_IB: cpu.ss.setByte(cpu.esi + cpu.edi + getMicrocode(), data); return; //2392;
	    
	case M_SS_EDI_ESP_IB: cpu.ss.setByte(cpu.edi + cpu.esp + getMicrocode(), data); return; //2393;
	case M_SS_2EDI_IB: cpu.ss.setByte(cpu.edi + cpu.edi + getMicrocode(), data); return; //2394;
	    
	case M_SS_3EAX_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2395;
	case M_SS_2EAX_ECX_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2396;
	case M_SS_2EAX_EDX_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2397;
	case M_SS_2EAX_EBX_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2398;
	case M_SS_2EAX_ESP_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2399;
	case M_SS_2EAX_EBP_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2400;
	case M_SS_2EAX_ESI_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2401;
	case M_SS_2EAX_EDI_IB: cpu.ss.setByte((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2402;
	    
	case M_SS_2ECX_EAX_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2403;
	case M_SS_3ECX_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2404;
	case M_SS_2ECX_EDX_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2405;
	case M_SS_2ECX_EBX_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2406;
	case M_SS_2ECX_ESP_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2407;
	case M_SS_2ECX_EBP_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2408;
	case M_SS_2ECX_ESI_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2409;
	case M_SS_2ECX_EDI_IB: cpu.ss.setByte((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2410;

	case M_SS_2EDX_EAX_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2411;
	case M_SS_2EDX_ECX_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2412;
	case M_SS_3EDX_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2413;
	case M_SS_2EDX_EBX_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2414;
	case M_SS_2EDX_ESP_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2415;
	case M_SS_2EDX_EBP_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2416;
	case M_SS_2EDX_ESI_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2417;
	case M_SS_2EDX_EDI_IB: cpu.ss.setByte((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2418;
	    
	case M_SS_2EBX_EAX_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2419;
	case M_SS_2EBX_ECX_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2420;
	case M_SS_2EBX_EDX_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2421;
	case M_SS_3EBX_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2422;
	case M_SS_2EBX_ESP_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2423;
	case M_SS_2EBX_EBP_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2424;
	case M_SS_2EBX_ESI_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2425;
	case M_SS_2EBX_EDI_IB: cpu.ss.setByte((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2426;
	    
	case M_SS_2EBP_EAX_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2427;
	case M_SS_2EBP_ECX_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2428;
	case M_SS_2EBP_EDX_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2429;
	case M_SS_2EBP_EBX_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2430;
	case M_SS_2EBP_ESP_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2431;
	case M_SS_3EBP_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2432;
	case M_SS_2EBP_ESI_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2433;
	case M_SS_2EBP_EDI_IB: cpu.ss.setByte((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2434;
	    
	case M_SS_2ESI_EAX_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2435;
	case M_SS_2ESI_ECX_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2436;
	case M_SS_2ESI_EDX_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2437;
	case M_SS_2ESI_EBX_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2438;
	case M_SS_2ESI_ESP_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2439;
	case M_SS_2ESI_EBP_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2440;
	case M_SS_3ESI_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2441;
	case M_SS_2ESI_EDI_IB: cpu.ss.setByte((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2442;
	    
	case M_SS_2EDI_EAX_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2443;
	case M_SS_2EDI_ECX_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2444;
	case M_SS_2EDI_EDX_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2445;
	case M_SS_2EDI_EBX_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2446;
	case M_SS_2EDI_ESP_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2447;
	case M_SS_2EDI_EBP_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2448;
	case M_SS_2EDI_ESI_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2449;
	case M_SS_3EDI_IB: cpu.ss.setByte((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2450;
	    
	case M_SS_5EAX_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2451;
	case M_SS_4EAX_ECX_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2452;
	case M_SS_4EAX_EDX_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2453;
	case M_SS_4EAX_EBX_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2454;
	case M_SS_4EAX_ESP_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2455;
	case M_SS_4EAX_EBP_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2456;
	case M_SS_4EAX_ESI_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2457;
	case M_SS_4EAX_EDI_IB: cpu.ss.setByte((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2458;

	case M_SS_4ECX_EAX_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2459;
	case M_SS_5ECX_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2460;
	case M_SS_4ECX_EDX_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2461;
	case M_SS_4ECX_EBX_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2462;
	case M_SS_4ECX_ESP_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2463;
	case M_SS_4ECX_EBP_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2464;
	case M_SS_4ECX_ESI_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2465;
	case M_SS_4ECX_EDI_IB: cpu.ss.setByte((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2466;

	case M_SS_4EDX_EAX_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2467
	case M_SS_4EDX_ECX_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2468
	case M_SS_5EDX_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2469
	case M_SS_4EDX_EBX_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2470
	case M_SS_4EDX_ESP_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2471
	case M_SS_4EDX_EBP_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2472
	case M_SS_4EDX_ESI_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2473
	case M_SS_4EDX_EDI_IB: cpu.ss.setByte((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2474

	case M_SS_4EBX_EAX_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2475;
	case M_SS_4EBX_ECX_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2476;
	case M_SS_4EBX_EDX_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2477;
	case M_SS_5EBX_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2478;
	case M_SS_4EBX_ESP_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2479;
	case M_SS_4EBX_EBP_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2480;
	case M_SS_4EBX_ESI_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2481;
	case M_SS_4EBX_EDI_IB: cpu.ss.setByte((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2482;

	case M_SS_4EBP_EAX_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2483;
	case M_SS_4EBP_ECX_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2484;
	case M_SS_4EBP_EDX_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2485;
	case M_SS_4EBP_EBX_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2486;
	case M_SS_4EBP_ESP_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2487;
	case M_SS_5EBP_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2488;
	case M_SS_4EBP_ESI_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2489;
	case M_SS_4EBP_EDI_IB: cpu.ss.setByte((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2490;
	    
	case M_SS_4ESI_EAX_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2491;
	case M_SS_4ESI_ECX_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2492;
	case M_SS_4ESI_EDX_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2493;
	case M_SS_4ESI_EBX_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2494;
	case M_SS_4ESI_ESP_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2495;
	case M_SS_4ESI_EBP_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2496
	case M_SS_5ESI_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2497;
	case M_SS_4ESI_EDI_IB: cpu.ss.setByte((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2498;
	    
	case M_SS_4EDI_EAX_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2499;
	case M_SS_4EDI_ECX_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2500;
	case M_SS_4EDI_EDX_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2501;
	case M_SS_4EDI_EBX_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2502;
	case M_SS_4EDI_ESP_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2503;
	case M_SS_4EDI_EBP_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2504;
	case M_SS_4EDI_ESI_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2505;
	case M_SS_5EDI_IB: cpu.ss.setByte((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2506;

	case M_SS_9EAX_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2507;
	case M_SS_8EAX_ECX_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2508;
	case M_SS_8EAX_EDX_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2509;
	case M_SS_8EAX_EBX_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2510;
	case M_SS_8EAX_ESP_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2511;
	case M_SS_8EAX_EBP_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2512;
	case M_SS_8EAX_ESI_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2513;
	case M_SS_8EAX_EDI_IB: cpu.ss.setByte((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2514;
	    
	case M_SS_8ECX_EAX_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2515;
	case M_SS_9ECX_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2516;
	case M_SS_8ECX_EDX_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2517;
	case M_SS_8ECX_EBX_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2518;
	case M_SS_8ECX_ESP_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2519;
	case M_SS_8ECX_EBP_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2520;
	case M_SS_8ECX_ESI_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2521;
	case M_SS_8ECX_EDI_IB: cpu.ss.setByte((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2522;
	    
	case M_SS_8EDX_EAX_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2523;
	case M_SS_8EDX_ECX_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2524;
	case M_SS_9EDX_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2525;
	case M_SS_8EDX_EBX_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2526;
	case M_SS_8EDX_ESP_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2527;
	case M_SS_8EDX_EBP_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2528;
	case M_SS_8EDX_ESI_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2529;
	case M_SS_8EDX_EDI_IB: cpu.ss.setByte((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2530;
	    
	case M_SS_8EBX_EAX_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2531
	case M_SS_8EBX_ECX_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2532
	case M_SS_8EBX_EDX_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2533
	case M_SS_9EBX_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2534
	case M_SS_8EBX_ESP_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2535
	case M_SS_8EBX_EBP_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2536
	case M_SS_8EBX_ESI_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2537
	case M_SS_8EBX_EDI_IB: cpu.ss.setByte((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2538
	    
	case M_SS_8EBP_EAX_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2539;
	case M_SS_8EBP_ECX_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2540;
	case M_SS_8EBP_EDX_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2541;
	case M_SS_8EBP_EBX_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2542;
	case M_SS_8EBP_ESP_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2543;
	case M_SS_9EBP_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2544;
	case M_SS_8EBP_ESI_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2545;
	case M_SS_8EBP_EDI_IB: cpu.ss.setByte((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2546;
	    
	case M_SS_8ESI_EAX_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2547;
	case M_SS_8ESI_ECX_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2548;
	case M_SS_8ESI_EDX_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2549;
	case M_SS_8ESI_EBX_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2550;
	case M_SS_8ESI_ESP_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2551;
	case M_SS_8ESI_EBP_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2552;
	case M_SS_9ESI_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2553;
	case M_SS_8ESI_EDI_IB: cpu.ss.setByte((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2554;
	    
	case M_SS_8EDI_EAX_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2555;
	case M_SS_8EDI_ECX_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2556;
	case M_SS_8EDI_EDX_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2557;
	case M_SS_8EDI_EBX_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2558;
	case M_SS_8EDI_ESP_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2559;
	case M_SS_8EDI_EBP_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2560;
	case M_SS_8EDI_ESI_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2561;
	case M_SS_9EDI_IB: cpu.ss.setByte((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2562;

    //SIB + ID Operanss
	case M_SS_EAX_ECX_ID: cpu.ss.setByte(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2563;
	case M_SS_EAX_EDX_ID: cpu.ss.setByte(cpu.eax + cpu.edx + getMicrocode(), data); return; //2564;
	case M_SS_EAX_EBX_ID: cpu.ss.setByte(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2565;
	case M_SS_EAX_ESP_ID: cpu.ss.setByte(cpu.eax + cpu.esp + getMicrocode(), data); return; //2566;
	case M_SS_EAX_EBP_ID: cpu.ss.setByte(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2567;
	case M_SS_EAX_ESI_ID: cpu.ss.setByte(cpu.eax + cpu.esi + getMicrocode(), data); return; //2568;
	case M_SS_EAX_EDI_ID: cpu.ss.setByte(cpu.eax + cpu.edi + getMicrocode(), data); return; //2569;
	    
	case M_SS_ECX_EDX_ID: cpu.ss.setByte(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2570;
	case M_SS_ECX_EBX_ID: cpu.ss.setByte(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2571;
	case M_SS_ECX_ESP_ID: cpu.ss.setByte(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2572;
	case M_SS_ECX_EBP_ID: cpu.ss.setByte(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2573;
	case M_SS_ECX_ESI_ID: cpu.ss.setByte(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2574;
	case M_SS_ECX_EDI_ID: cpu.ss.setByte(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2575;
	    
	case M_SS_EDX_EBX_ID: cpu.ss.setByte(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2576
	case M_SS_EDX_ESP_ID: cpu.ss.setByte(cpu.edx + cpu.esp + getMicrocode(), data); return; //2577
	case M_SS_EDX_EBP_ID: cpu.ss.setByte(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2578
	case M_SS_EDX_ESI_ID: cpu.ss.setByte(cpu.edx + cpu.esi + getMicrocode(), data); return; //2579
	case M_SS_EDX_EDI_ID: cpu.ss.setByte(cpu.edx + cpu.edi + getMicrocode(), data); return; //2580

	case M_SS_EBX_ESP_ID: cpu.ss.setByte(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2581;
	case M_SS_EBX_EBP_ID: cpu.ss.setByte(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2582;
	case M_SS_EBX_ESI_ID: cpu.ss.setByte(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2583;
	case M_SS_EBX_EDI_ID: cpu.ss.setByte(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2584;
	    
	case M_SS_ESP_ID: cpu.ss.setByte(cpu.esp + getMicrocode(), data); return; //2585
	    
	case M_SS_EBP_ESP_ID: cpu.ss.setByte(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2586;
	case M_SS_EBP_ESI_ID: cpu.ss.setByte(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2587;
	case M_SS_EBP_EDI_ID: cpu.ss.setByte(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2588;
	    
	case M_SS_ESI_ESP_ID: cpu.ss.setByte(cpu.esi + cpu.esp + getMicrocode(), data); return; //2589;
	case M_SS_ESI_EDI_ID: cpu.ss.setByte(cpu.esi + cpu.edi + getMicrocode(), data); return; //2590;
	    
	case M_SS_EDI_ESP_ID: cpu.ss.setByte(cpu.edi + cpu.esp + getMicrocode(), data); return; //2591;
	    
	case M_SS_3EAX_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2592;
	case M_SS_2EAX_ECX_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2593;
	case M_SS_2EAX_EDX_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2594;
	case M_SS_2EAX_EBX_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2595;
	case M_SS_2EAX_ESP_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2596;
	case M_SS_2EAX_EBP_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2597;
	case M_SS_2EAX_ESI_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2598;
	case M_SS_2EAX_EDI_ID: cpu.ss.setByte((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2599;
	    
	case M_SS_2ECX_EAX_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2600;
	case M_SS_3ECX_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2601;
	case M_SS_2ECX_EDX_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2602;
	case M_SS_2ECX_EBX_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2603;
	case M_SS_2ECX_ESP_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2604;
	case M_SS_2ECX_EBP_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2605;
	case M_SS_2ECX_ESI_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2606;
	case M_SS_2ECX_EDI_ID: cpu.ss.setByte((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2607;
	    
	case M_SS_2EDX_EAX_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2608;
	case M_SS_2EDX_ECX_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2609;
	case M_SS_3EDX_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2610;
	case M_SS_2EDX_EBX_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2611;
	case M_SS_2EDX_ESP_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2612;
	case M_SS_2EDX_EBP_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2613;
	case M_SS_2EDX_ESI_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2614;
	case M_SS_2EDX_EDI_ID: cpu.ss.setByte((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2615;
	    
	case M_SS_2EBX_EAX_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2616;
	case M_SS_2EBX_ECX_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2617;
	case M_SS_2EBX_EDX_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2618;
	case M_SS_3EBX_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2619;
	case M_SS_2EBX_ESP_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2620;
	case M_SS_2EBX_EBP_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2621;
	case M_SS_2EBX_ESI_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2622;
	case M_SS_2EBX_EDI_ID: cpu.ss.setByte((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2623;
	    
	case M_SS_2EBP_EAX_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2624;
	case M_SS_2EBP_ECX_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2625;
	case M_SS_2EBP_EDX_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2626;
	case M_SS_2EBP_EBX_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2627;
	case M_SS_2EBP_ESP_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2628;
	case M_SS_3EBP_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2629;
	case M_SS_2EBP_ESI_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2630;
	case M_SS_2EBP_EDI_ID: cpu.ss.setByte((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2631;
	    
	case M_SS_2ESI_EAX_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2632;
	case M_SS_2ESI_ECX_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2633;
	case M_SS_2ESI_EDX_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2634;
	case M_SS_2ESI_EBX_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2635;
	case M_SS_2ESI_ESP_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2636;
	case M_SS_2ESI_EBP_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2637;
	case M_SS_3ESI_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2638;
	case M_SS_2ESI_EDI_ID: cpu.ss.setByte((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2639;
	    
	case M_SS_2EDI_EAX_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2640;
	case M_SS_2EDI_ECX_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2641;
	case M_SS_2EDI_EDX_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2642;
	case M_SS_2EDI_EBX_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2643;
	case M_SS_2EDI_ESP_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2644;
	case M_SS_2EDI_EBP_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2645;
	case M_SS_2EDI_ESI_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2646;
	case M_SS_3EDI_ID: cpu.ss.setByte((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2647;
	    
	case M_SS_5EAX_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2648
	case M_SS_4EAX_ECX_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2649
	case M_SS_4EAX_EDX_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2650
	case M_SS_4EAX_EBX_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2651
	case M_SS_4EAX_ESP_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2652
	case M_SS_4EAX_EBP_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2653
	case M_SS_4EAX_ESI_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2654
	case M_SS_4EAX_EDI_ID: cpu.ss.setByte((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2655

	case M_SS_4ECX_EAX_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2656;
	case M_SS_5ECX_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2657;
	case M_SS_4ECX_EDX_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2658;
	case M_SS_4ECX_EBX_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2659;
	case M_SS_4ECX_ESP_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2660;
	case M_SS_4ECX_EBP_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2661;
	case M_SS_4ECX_ESI_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2662;
	case M_SS_4ECX_EDI_ID: cpu.ss.setByte((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2663;
	    
	case M_SS_4EDX_EAX_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2664;
	case M_SS_4EDX_ECX_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2665;
	case M_SS_5EDX_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2666;
	case M_SS_4EDX_EBX_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2667;
	case M_SS_4EDX_ESP_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2668;
	case M_SS_4EDX_EBP_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2669;
	case M_SS_4EDX_ESI_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2670;
	case M_SS_4EDX_EDI_ID: cpu.ss.setByte((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2671;
	    
	case M_SS_4EBX_EAX_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2672;
	case M_SS_4EBX_ECX_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2673;
	case M_SS_4EBX_EDX_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2674;
	case M_SS_5EBX_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2675;
	case M_SS_4EBX_ESP_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2676;
	case M_SS_4EBX_EBP_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2677;
	case M_SS_4EBX_ESI_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2678;
	case M_SS_4EBX_EDI_ID: cpu.ss.setByte((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2679;
	    
	case M_SS_4EBP_EAX_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2680;
	case M_SS_4EBP_ECX_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2681;
	case M_SS_4EBP_EDX_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2682;
	case M_SS_4EBP_EBX_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2683;
	case M_SS_4EBP_ESP_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2684;
	case M_SS_5EBP_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2685;
	case M_SS_4EBP_ESI_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2686;
	case M_SS_4EBP_EDI_ID: cpu.ss.setByte((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2687;
	    
	case M_SS_4ESI_EAX_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2688;
	case M_SS_4ESI_ECX_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2689;
	case M_SS_4ESI_EDX_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2690;
	case M_SS_4ESI_EBX_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2691;
	case M_SS_4ESI_ESP_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2692;
	case M_SS_4ESI_EBP_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2693;
	case M_SS_5ESI_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2694;
	case M_SS_4ESI_EDI_ID: cpu.ss.setByte((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2695;

	case M_SS_4EDI_EAX_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2696;
	case M_SS_4EDI_ECX_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2697;
	case M_SS_4EDI_EDX_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2698;
	case M_SS_4EDI_EBX_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2699;
	case M_SS_4EDI_ESP_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2700;
	case M_SS_4EDI_EBP_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2701;
	case M_SS_4EDI_ESI_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2702;
	case M_SS_5EDI_ID: cpu.ss.setByte((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2703;	    

	case M_SS_9EAX_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2704;
	case M_SS_8EAX_ECX_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2705;
	case M_SS_8EAX_EDX_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2706;
	case M_SS_8EAX_EBX_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2707;
	case M_SS_8EAX_ESP_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2708;
	case M_SS_8EAX_EBP_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2709;
	case M_SS_8EAX_ESI_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2710;
	case M_SS_8EAX_EDI_ID: cpu.ss.setByte((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2711;

	case M_SS_8ECX_EAX_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2712;
	case M_SS_9ECX_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2713;
	case M_SS_8ECX_EDX_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2714;
	case M_SS_8ECX_EBX_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2715;
	case M_SS_8ECX_ESP_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2716;
	case M_SS_8ECX_EBP_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2717;
	case M_SS_8ECX_ESI_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2718;
	case M_SS_8ECX_EDI_ID: cpu.ss.setByte((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2719;

	case M_SS_8EDX_EAX_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2720;
	case M_SS_8EDX_ECX_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2721;
	case M_SS_9EDX_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2722;
	case M_SS_8EDX_EBX_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2723;
	case M_SS_8EDX_ESP_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2724;
	case M_SS_8EDX_EBP_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2725;
	case M_SS_8EDX_ESI_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2726;
	case M_SS_8EDX_EDI_ID: cpu.ss.setByte((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2727;

	case M_SS_8EBX_EAX_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2728;
	case M_SS_8EBX_ECX_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2729;
	case M_SS_8EBX_EDX_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2730;
	case M_SS_9EBX_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2731;
	case M_SS_8EBX_ESP_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2732;
	case M_SS_8EBX_EBP_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2733;
	case M_SS_8EBX_ESI_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2734;
	case M_SS_8EBX_EDI_ID: cpu.ss.setByte((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2735;
	    
	case M_SS_8EBP_EAX_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2736;
	case M_SS_8EBP_ECX_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2737;
	case M_SS_8EBP_EDX_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2738;
	case M_SS_8EBP_EBX_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2739;
	case M_SS_8EBP_ESP_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2740;
	case M_SS_9EBP_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2741;
	case M_SS_8EBP_ESI_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2742;
	case M_SS_8EBP_EDI_ID: cpu.ss.setByte((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2743;
	    
	case M_SS_8ESI_EAX_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2744;
	case M_SS_8ESI_ECX_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2745;
	case M_SS_8ESI_EDX_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2746;
	case M_SS_8ESI_EBX_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2747;
	case M_SS_8ESI_ESP_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2748;
	case M_SS_8ESI_EBP_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2749;
	case M_SS_9ESI_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2750;
	case M_SS_8ESI_EDI_ID: cpu.ss.setByte((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2751;
    
	case M_SS_8EDI_EAX_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2752;
	case M_SS_8EDI_ECX_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2753;
	case M_SS_8EDI_EDX_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2754;
	case M_SS_8EDI_EBX_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2755;
	case M_SS_8EDI_ESP_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2756;
	case M_SS_8EDI_EBP_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2757;
	case M_SS_8EDI_ESI_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2758;
	case M_SS_9EDI_ID: cpu.ss.setByte((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2759;
	}

	System.err.println("Unimplemented SetByte Operand: " + microcode);
	throw unimplemented;
    }

    protected final void setShortOperand(short data)
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case AX:  cpu.eax = (cpu.eax & 0xffff0000) | (data & 0xffff); return;
	case CX:  cpu.ecx = (cpu.ecx & 0xffff0000) | (data & 0xffff); return;
	case DX:  cpu.edx = (cpu.edx & 0xffff0000) | (data & 0xffff); return;
	case BX:  cpu.ebx = (cpu.ebx & 0xffff0000) | (data & 0xffff); return;
	case SP:  cpu.esp = (cpu.esp & 0xffff0000) | (data & 0xffff); return;
	case BP:  cpu.ebp = (cpu.ebp & 0xffff0000) | (data & 0xffff); return;
	case SI:  cpu.esi = (cpu.esi & 0xffff0000) | (data & 0xffff); return;
	case DI:  cpu.edi = (cpu.edi & 0xffff0000) | (data & 0xffff); return;

	case ES: loadSegment(ES, 0xffff & data); return;
	case CS: loadSegment(CS, 0xffff & data); return;
	case SS: loadSegment(SS, 0xffff & data); return;
	case DS: loadSegment(DS, 0xffff & data); return;
	case FS: loadSegment(FS, 0xffff & data); return;
	case GS: loadSegment(GS, 0xffff & data); return;
	}

	setShortMemoryOperand(microcode, data);
	return;
    }

    private final void setShortMemoryOperand(int microcode, short data)
    {
	switch (microcode) {
	case M_ES_BX_SI: cpu.es.setWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_ES_BX_DI: cpu.es.setWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_ES_BP_SI: cpu.es.setWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_ES_BP_DI: cpu.es.setWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_ES_SI: cpu.es.setWord(cpu.esi & 0xffff, data); return;
	case M_ES_DI: cpu.es.setWord(cpu.edi & 0xffff, data); return;
	case M_ES_IW: cpu.es.setWord(getMicrocode() & 0xffff, data); return;
	case M_ES_BX: cpu.es.setWord(cpu.ebx & 0xffff, data); return;
	case M_ES_BX_SI_IB: cpu.es.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_DI_IB: cpu.es.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_SI_IB: cpu.es.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_DI_IB: cpu.es.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_SI_IB: cpu.es.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_DI_IB: cpu.es.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_IB: cpu.es.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_IB: cpu.es.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_SI_IW: cpu.es.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_DI_IW: cpu.es.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_SI_IW: cpu.es.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_DI_IW: cpu.es.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_SI_IW: cpu.es.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_DI_IW: cpu.es.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_IW: cpu.es.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_IW: cpu.es.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_CS_BX_SI: cpu.cs.setWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_CS_BX_DI: cpu.cs.setWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_CS_BP_SI: cpu.cs.setWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_CS_BP_DI: cpu.cs.setWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_CS_SI: cpu.cs.setWord(cpu.esi & 0xffff, data); return;
	case M_CS_DI: cpu.cs.setWord(cpu.edi & 0xffff, data); return;
	case M_CS_IW: cpu.cs.setWord(getMicrocode() & 0xffff, data); return;
	case M_CS_BX: cpu.cs.setWord(cpu.ebx & 0xffff, data); return;
	case M_CS_BX_SI_IB: cpu.cs.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_DI_IB: cpu.cs.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_SI_IB: cpu.cs.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_DI_IB: cpu.cs.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_SI_IB: cpu.cs.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_DI_IB: cpu.cs.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_IB: cpu.cs.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_IB: cpu.cs.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_SI_IW: cpu.cs.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_DI_IW: cpu.cs.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_SI_IW: cpu.cs.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_DI_IW: cpu.cs.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_SI_IW: cpu.cs.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_CS_DI_IW: cpu.cs.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_CS_BP_IW: cpu.cs.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_CS_BX_IW: cpu.cs.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_SS_BX_SI: cpu.ss.setWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_SS_BX_DI: cpu.ss.setWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_SS_BP_SI: cpu.ss.setWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_SS_BP_DI: cpu.ss.setWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_SS_SI: cpu.ss.setWord(cpu.esi & 0xffff, data); return;
	case M_SS_DI: cpu.ss.setWord(cpu.edi & 0xffff, data); return;
	case M_SS_IW: cpu.ss.setWord(getMicrocode() & 0xffff, data); return;
	case M_SS_BX: cpu.ss.setWord(cpu.ebx & 0xffff, data); return;
	case M_SS_BX_SI_IB: cpu.ss.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_DI_IB: cpu.ss.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_SI_IB: cpu.ss.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_DI_IB: cpu.ss.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_SI_IB: cpu.ss.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_DI_IB: cpu.ss.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_IB: cpu.ss.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_IB: cpu.ss.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_SI_IW: cpu.ss.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_DI_IW: cpu.ss.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_SI_IW: cpu.ss.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_DI_IW: cpu.ss.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_SI_IW: cpu.ss.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_SS_DI_IW: cpu.ss.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_SS_BP_IW: cpu.ss.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_SS_BX_IW: cpu.ss.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_DS_BX_SI: cpu.ds.setWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_DS_BX_DI: cpu.ds.setWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_DS_BP_SI: cpu.ds.setWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_DS_BP_DI: cpu.ds.setWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_DS_SI: cpu.ds.setWord(cpu.esi & 0xffff, data); return;
	case M_DS_DI: cpu.ds.setWord(cpu.edi & 0xffff, data); return;
	case M_DS_IW: cpu.ds.setWord(getMicrocode() & 0xffff, data); return;
	case M_DS_BX: cpu.ds.setWord(cpu.ebx & 0xffff, data); return;
	case M_DS_BX_SI_IB: cpu.ds.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_DI_IB: cpu.ds.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_SI_IB: cpu.ds.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_DI_IB: cpu.ds.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_SI_IB: cpu.ds.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_DI_IB: cpu.ds.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_IB: cpu.ds.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_IB: cpu.ds.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_SI_IW: cpu.ds.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_DI_IW: cpu.ds.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_SI_IW: cpu.ds.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_DI_IW: cpu.ds.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_SI_IW: cpu.ds.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_DS_DI_IW: cpu.ds.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_DS_BP_IW: cpu.ds.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_DS_BX_IW: cpu.ds.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_FS_BX_SI: cpu.fs.setWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_FS_BX_DI: cpu.fs.setWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_FS_BP_SI: cpu.fs.setWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_FS_BP_DI: cpu.fs.setWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_FS_SI: cpu.fs.setWord(cpu.esi & 0xffff, data); return;
	case M_FS_DI: cpu.fs.setWord(cpu.edi & 0xffff, data); return;
	case M_FS_IW: cpu.fs.setWord(getMicrocode() & 0xffff, data); return;
	case M_FS_BX: cpu.fs.setWord(cpu.ebx & 0xffff, data); return;
	case M_FS_BX_SI_IB: cpu.fs.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_DI_IB: cpu.fs.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_SI_IB: cpu.fs.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_DI_IB: cpu.fs.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_SI_IB: cpu.fs.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_DI_IB: cpu.fs.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_IB: cpu.fs.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_IB: cpu.fs.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_SI_IW: cpu.fs.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_DI_IW: cpu.fs.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_SI_IW: cpu.fs.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_DI_IW: cpu.fs.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_SI_IW: cpu.fs.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_FS_DI_IW: cpu.fs.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_FS_BP_IW: cpu.fs.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_FS_BX_IW: cpu.fs.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_GS_BX_SI: cpu.gs.setWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_GS_BX_DI: cpu.gs.setWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_GS_BP_SI: cpu.gs.setWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_GS_BP_DI: cpu.gs.setWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_GS_SI: cpu.gs.setWord(cpu.esi & 0xffff, data); return;
	case M_GS_DI: cpu.gs.setWord(cpu.edi & 0xffff, data); return;
	case M_GS_IW: cpu.gs.setWord(getMicrocode() & 0xffff, data); return;
	case M_GS_BX: cpu.gs.setWord(cpu.ebx & 0xffff, data); return;
	case M_GS_BX_SI_IB: cpu.gs.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_DI_IB: cpu.gs.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_SI_IB: cpu.gs.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_DI_IB: cpu.gs.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_SI_IB: cpu.gs.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_DI_IB: cpu.gs.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_IB: cpu.gs.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_IB: cpu.gs.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_SI_IW: cpu.gs.setWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_DI_IW: cpu.gs.setWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_SI_IW: cpu.gs.setWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_DI_IW: cpu.gs.setWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_SI_IW: cpu.gs.setWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_GS_DI_IW: cpu.gs.setWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_GS_BP_IW: cpu.gs.setWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_GS_BX_IW: cpu.gs.setWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;

	case M_ES_EAX: cpu.es.setWord(cpu.eax, data); return; //41
	case M_ES_ECX: cpu.es.setWord(cpu.ecx, data); return; //42
	case M_ES_EDX: cpu.es.setWord(cpu.edx, data); return; //43
	case M_ES_EBX: cpu.es.setWord(cpu.ebx, data); return; //44
	case M_ES_ID: cpu.es.setWord(getMicrocode(), data); return; //45
	case M_ES_ESI: cpu.es.setWord(cpu.esi, data); return; //46
	case M_ES_EDI:cpu.es.setWord(cpu.edi, data); return; //47

	case M_SS_EAX_IB: cpu.ss.setWord(cpu.eax + getMicrocode(), data); return; //90
	case M_SS_ECX_IB: cpu.ss.setWord(cpu.ecx + getMicrocode(), data); return; //91
	case M_SS_EDX_IB: cpu.ss.setWord(cpu.edx + getMicrocode(), data); return; //92
	case M_SS_EBX_IB: cpu.ss.setWord(cpu.ebx + getMicrocode(), data); return; //93
	case M_SS_EBP_IB: cpu.ss.setWord(cpu.ebp + getMicrocode(), data); return; //94
	case M_SS_ESI_IB: cpu.ss.setWord(cpu.esi + getMicrocode(), data); return; //95
	case M_SS_EDI_IB: cpu.ss.setWord(cpu.edi + getMicrocode(), data); return; //96

	case M_SS_EAX_ID: cpu.ss.setWord(cpu.eax + getMicrocode(), data); return; //97
	case M_SS_ECX_ID: cpu.ss.setWord(cpu.ecx + getMicrocode(), data); return; //98
	case M_SS_EDX_ID: cpu.ss.setWord(cpu.edx + getMicrocode(), data); return; //99
	case M_SS_EBX_ID: cpu.ss.setWord(cpu.ebx + getMicrocode(), data); return; //100
	case M_SS_EBP_ID: cpu.ss.setWord(cpu.ebp + getMicrocode(), data); return; //101
	case M_SS_ESI_ID: cpu.ss.setWord(cpu.esi + getMicrocode(), data); return; //102
	case M_SS_EDI_ID: cpu.ss.setWord(cpu.edi + getMicrocode(), data); return; //103

	case M_DS_EAX: cpu.ds.setWord(cpu.eax, data); return; //104
	case M_DS_ECX: cpu.ds.setWord(cpu.ecx, data); return; //105
	case M_DS_EDX: cpu.ds.setWord(cpu.edx, data); return; //106
	case M_DS_EBX: cpu.ds.setWord(cpu.ebx, data); return; //107
	case M_DS_ID: cpu.ds.setWord(getMicrocode(), data); return; //108
	case M_DS_ESI: cpu.ds.setWord(cpu.esi, data); return; //109
	case M_DS_EDI: cpu.ds.setWord(cpu.edi, data); return; //110

	case M_DS_EAX_IB: cpu.ds.setWord(cpu.eax + getMicrocode(), data); return; //111
	case M_DS_ECX_IB: cpu.ds.setWord(cpu.ecx + getMicrocode(), data); return; //112
	case M_DS_EDX_IB: cpu.ds.setWord(cpu.edx + getMicrocode(), data); return; //113
	case M_DS_EBX_IB: cpu.ds.setWord(cpu.ebx + getMicrocode(), data); return; //114
	case M_DS_EBP_IB: cpu.ds.setWord(cpu.ebp + getMicrocode(), data); return; //115
	case M_DS_ESI_IB: cpu.ds.setWord(cpu.esi + getMicrocode(), data); return; //116
	case M_DS_EDI_IB: cpu.ds.setWord(cpu.edi + getMicrocode(), data); return; //117

	case M_DS_EAX_ID: cpu.ds.setWord(cpu.eax + getMicrocode(), data); return; //118
	case M_DS_ECX_ID: cpu.ds.setWord(cpu.ecx + getMicrocode(), data); return; //119
	case M_DS_EDX_ID: cpu.ds.setWord(cpu.edx + getMicrocode(), data); return; //120
	case M_DS_EBX_ID: cpu.ds.setWord(cpu.ebx + getMicrocode(), data); return; //121
	case M_DS_EBP_ID: cpu.ds.setWord(cpu.ebp + getMicrocode(), data); return; //122
	case M_DS_ESI_ID: cpu.ds.setWord(cpu.esi + getMicrocode(), data); return; //123
	case M_DS_EDI_ID: cpu.ds.setWord(cpu.edi + getMicrocode(), data); return; //124

	case M_DS_2EAX: cpu.ds.setWord(cpu.eax + cpu.eax, data); return; //2156
	case M_DS_EAX_ECX: cpu.ds.setWord(cpu.eax + cpu.ecx, data); return; //2157
	case M_DS_EAX_EDX: cpu.ds.setWord(cpu.eax + cpu.edx, data); return; //2158
	case M_DS_EAX_EBX: cpu.ds.setWord(cpu.eax + cpu.ebx, data); return; //2159
	case M_DS_EAX_ESP: cpu.ds.setWord(cpu.eax + cpu.esp, data); return; //2160
	case M_DS_EAX_EBP: cpu.ds.setWord(cpu.eax + cpu.ebp, data); return; //2161
	case M_DS_EAX_ESI: cpu.ds.setWord(cpu.eax + cpu.esi, data); return; //2162
	case M_DS_EAX_EDI: cpu.ds.setWord(cpu.eax + cpu.edi, data); return; //2163

	case M_DS_2ECX: cpu.ds.setWord(cpu.ecx + cpu.ecx, data); return; //2164
	case M_DS_ECX_EDX: cpu.ds.setWord(cpu.ecx + cpu.edx, data); return; //2165
	case M_DS_ECX_EBX: cpu.ds.setWord(cpu.ecx + cpu.ebx, data); return; //2166
	case M_DS_ECX_ESP: cpu.ds.setWord(cpu.ecx + cpu.esp, data); return; //2167
	case M_DS_ECX_EBP: cpu.ds.setWord(cpu.ecx + cpu.ebp, data); return; //2168
	case M_DS_ECX_ESI: cpu.ds.setWord(cpu.ecx + cpu.esi, data); return; //2169
	case M_DS_ECX_EDI: cpu.ds.setWord(cpu.ecx + cpu.edi, data); return; //2170

	case M_DS_2EDX: cpu.ds.setWord(cpu.edx + cpu.edx, data); return; //2171
	case M_DS_EDX_EBX: cpu.ds.setWord(cpu.edx + cpu.ebx, data); return; //2172
	case M_DS_EDX_ESP: cpu.ds.setWord(cpu.edx + cpu.esp, data); return; //2173
	case M_DS_EDX_EBP: cpu.ds.setWord(cpu.edx + cpu.ebp, data); return; //2174
	case M_DS_EDX_ESI: cpu.ds.setWord(cpu.edx + cpu.esi, data); return; //2175
	case M_DS_EDX_EDI: cpu.ds.setWord(cpu.edx + cpu.edi, data); return; //2176

	case M_DS_2EBX: cpu.ds.setWord(cpu.ebx + cpu.ebx, data); return; //2177;
	case M_DS_EBX_ESP: cpu.ds.setWord(cpu.ebx + cpu.esp, data); return; //2178;
	case M_DS_EBX_EBP: cpu.ds.setWord(cpu.ebx + cpu.ebp, data); return; //2179;
	case M_DS_EBX_ESI: cpu.ds.setWord(cpu.ebx + cpu.esi, data); return; //2180;
	case M_DS_EBX_EDI: cpu.ds.setWord(cpu.ebx + cpu.edi, data); return; //2181;
	    
	case M_DS_ESP: cpu.ds.setWord(cpu.esp, data); return; //2182

	case M_DS_EBP_ESP: cpu.ds.setWord(cpu.ebp + cpu.esp, data); return; //2183;
	case M_DS_EBP_ESI: cpu.ds.setWord(cpu.ebp + cpu.esi, data); return; //2184;
	case M_DS_EBP_EDI: cpu.ds.setWord(cpu.ebp + cpu.edi, data); return; //2185;

	case M_DS_ESI_ESP: cpu.ds.setWord(cpu.esi + cpu.esp, data); return; //2186;
	case M_DS_2ESI: cpu.ds.setWord(cpu.esi + cpu.esi, data); return; //2187;
	case M_DS_ESI_EDI: cpu.ds.setWord(cpu.esi + cpu.edi, data); return; //2188;

	case M_DS_EDI_ESP: cpu.ds.setWord(cpu.edi + cpu.esp, data); return; //2189;
	case M_DS_2EDI: cpu.ds.setWord(cpu.edi + cpu.edi, data); return; //2190;
	    
	case M_DS_3EAX: cpu.ds.setWord((cpu.eax << 1) + cpu.eax, data); return; //2191;
	case M_DS_2EAX_ECX: cpu.ds.setWord((cpu.eax << 1) + cpu.ecx, data); return; //2192;
	case M_DS_2EAX_EDX: cpu.ds.setWord((cpu.eax << 1) + cpu.edx, data); return; //2193;
	case M_DS_2EAX_EBX: cpu.ds.setWord((cpu.eax << 1) + cpu.ebx, data); return; //2194;
	case M_DS_2EAX_ESP: cpu.ds.setWord((cpu.eax << 1) + cpu.esp, data); return; //2195;
	case M_DS_2EAX_ID: cpu.ds.setWord((cpu.eax << 1) + getMicrocode(), data); return; //2196;
	case M_DS_2EAX_ESI: cpu.ds.setWord((cpu.eax << 1) + cpu.esi, data); return; //2197;
	case M_DS_2EAX_EDI: cpu.ds.setWord((cpu.eax << 1) + cpu.edi, data); return; //2198;

	case M_DS_2ECX_EAX: cpu.ds.setWord((cpu.ecx << 1) + cpu.eax, data); return; //2199;
	case M_DS_3ECX: cpu.ds.setWord((cpu.ecx << 1) + cpu.ecx, data); return; //2200;
	case M_DS_2ECX_EDX: cpu.ds.setWord((cpu.ecx << 1) + cpu.edx, data); return; //2201;
	case M_DS_2ECX_EBX: cpu.ds.setWord((cpu.ecx << 1) + cpu.ebx, data); return; //2202;
	case M_DS_2ECX_ESP: cpu.ds.setWord((cpu.ecx << 1) + cpu.esp, data); return; //2203;
	case M_DS_2ECX_ID: cpu.ds.setWord((cpu.ecx << 1) + getMicrocode(), data); return; //2204;
	case M_DS_2ECX_ESI: cpu.ds.setWord((cpu.ecx << 1) + cpu.esi, data); return; //2205;
	case M_DS_2ECX_EDI: cpu.ds.setWord((cpu.ecx << 1) + cpu.edi, data); return; //2206;

	case M_DS_2EDX_EAX: cpu.ds.setWord((cpu.edx << 1) + cpu.eax, data); return; //2207;
	case M_DS_2EDX_ECX: cpu.ds.setWord((cpu.edx << 1) + cpu.ecx, data); return; //2208;
	case M_DS_3EDX: cpu.ds.setWord((cpu.edx << 1) + cpu.edx, data); return; //2209;
	case M_DS_2EDX_EBX: cpu.ds.setWord((cpu.edx << 1) + cpu.ebx, data); return; //2210;
	case M_DS_2EDX_ESP: cpu.ds.setWord((cpu.edx << 1) + cpu.esp, data); return; //2211;
	case M_DS_2EDX_ID: cpu.ds.setWord((cpu.edx << 1) + getMicrocode(), data); return; //2212;
	case M_DS_2EDX_ESI: cpu.ds.setWord((cpu.edx << 1) + cpu.esi, data); return; //2213;
	case M_DS_2EDX_EDI: cpu.ds.setWord((cpu.edx << 1) + cpu.edi, data); return; //2214;

	case M_DS_2EBX_EAX: cpu.ds.setWord((cpu.ebx << 1) + cpu.eax, data); return; //2215;
	case M_DS_2EBX_ECX: cpu.ds.setWord((cpu.ebx << 1) + cpu.ecx, data); return; //2216;
	case M_DS_2EBX_EDX: cpu.ds.setWord((cpu.ebx << 1) + cpu.edx, data); return; //2217;
	case M_DS_3EBX: cpu.ds.setWord((cpu.ebx << 1) + cpu.ebx, data); return; //2218;
	case M_DS_2EBX_ESP: cpu.ds.setWord((cpu.ebx << 1) + cpu.esp, data); return; //2219;
	case M_DS_2EBX_ID: cpu.ds.setWord((cpu.ebx << 1) + getMicrocode(), data); return; //2220;
	case M_DS_2EBX_ESI: cpu.ds.setWord((cpu.ebx << 1) + cpu.esi, data); return; //2221;
	case M_DS_2EBX_EDI: cpu.ds.setWord((cpu.ebx << 1) + cpu.edi, data); return; //2222;
	    
	case M_DS_2EBP_EAX: cpu.ds.setWord((cpu.ebp << 1) + cpu.eax, data); return; //2223;
	case M_DS_2EBP_ECX: cpu.ds.setWord((cpu.ebp << 1) + cpu.ecx, data); return; //2224;
	case M_DS_2EBP_EDX: cpu.ds.setWord((cpu.ebp << 1) + cpu.edx, data); return; //2225;
	case M_DS_2EBP_EBX: cpu.ds.setWord((cpu.ebp << 1) + cpu.ebx, data); return; //2226;
	case M_DS_2EBP_ESP: cpu.ds.setWord((cpu.ebp << 1) + cpu.esp, data); return; //2227;
	case M_DS_2EBP_ID: cpu.ds.setWord((cpu.ebp << 1) + getMicrocode(), data); return; //2228;
	case M_DS_2EBP_ESI: cpu.ds.setWord((cpu.ebp << 1) + cpu.esi, data); return; //2229;
	case M_DS_2EBP_EDI: cpu.ds.setWord((cpu.ebp << 1) + cpu.edi, data); return; //2230;

	case M_DS_2ESI_EAX: cpu.ds.setWord((cpu.esi << 1) + cpu.eax, data); return; //2231;
	case M_DS_2ESI_ECX: cpu.ds.setWord((cpu.esi << 1) + cpu.ecx, data); return; //2232;
	case M_DS_2ESI_EDX: cpu.ds.setWord((cpu.esi << 1) + cpu.edx, data); return; //2233;
	case M_DS_2ESI_EBX: cpu.ds.setWord((cpu.esi << 1) + cpu.ebx, data); return; //2234;
	case M_DS_2ESI_ESP: cpu.ds.setWord((cpu.esi << 1) + cpu.esp, data); return; //2235;
	case M_DS_2ESI_ID: cpu.ds.setWord((cpu.esi << 1) + getMicrocode(), data); return; //2236;
	case M_DS_3ESI: cpu.ds.setWord((cpu.esi << 1) + cpu.esi, data); return; //2237;
	case M_DS_2ESI_EDI: cpu.ds.setWord((cpu.esi << 1) + cpu.edi, data); return; //2238;

	case M_DS_2EDI_EAX: cpu.ds.setWord((cpu.edi << 1) + cpu.eax, data); return; //2239;
	case M_DS_2EDI_ECX: cpu.ds.setWord((cpu.edi << 1) + cpu.ecx, data); return; //2240;
	case M_DS_2EDI_EDX: cpu.ds.setWord((cpu.edi << 1) + cpu.edx, data); return; //2241;
	case M_DS_2EDI_EBX: cpu.ds.setWord((cpu.edi << 1) + cpu.ebx, data); return; //2242;
	case M_DS_2EDI_ESP: cpu.ds.setWord((cpu.edi << 1) + cpu.esp, data); return; //2243;
	case M_DS_2EDI_ID: cpu.ds.setWord((cpu.edi << 1) + getMicrocode(), data); return; //2244;
	case M_DS_2EDI_ESI: cpu.ds.setWord((cpu.edi << 1) + cpu.esi, data); return; //2245;
	case M_DS_3EDI: cpu.ds.setWord((cpu.edi << 1) + cpu.edi, data); return; //2246;

	case M_DS_5EAX: cpu.ds.setWord((cpu.eax << 2) + cpu.eax, data); return; //2247
	case M_DS_4EAX_ECX: cpu.ds.setWord((cpu.eax << 2) + cpu.ecx, data); return; //2248
	case M_DS_4EAX_EDX: cpu.ds.setWord((cpu.eax << 2) + cpu.edx, data); return; //2249
	case M_DS_4EAX_EBX: cpu.ds.setWord((cpu.eax << 2) + cpu.ebx, data); return; //2250
	case M_DS_4EAX_ESP: cpu.ds.setWord((cpu.eax << 2) + cpu.esp, data); return; //2251
	case M_DS_4EAX_ID: cpu.ds.setWord((cpu.eax << 2) + getMicrocode(), data); return; //2252
	case M_DS_4EAX_ESI: cpu.ds.setWord((cpu.eax << 2) + cpu.esi, data); return; //2253
	case M_DS_4EAX_EDI: cpu.ds.setWord((cpu.eax << 2) + cpu.edi, data); return; //2254

	case M_DS_4ECX_EAX: cpu.ds.setWord((cpu.ecx << 2) + cpu.eax, data); return; //2255
	case M_DS_5ECX: cpu.ds.setWord((cpu.ecx << 2) + cpu.ecx, data); return; //2256
	case M_DS_4ECX_EDX: cpu.ds.setWord((cpu.ecx << 2) + cpu.edx, data); return; //2257
	case M_DS_4ECX_EBX: cpu.ds.setWord((cpu.ecx << 2) + cpu.ebx, data); return; //2258
	case M_DS_4ECX_ESP: cpu.ds.setWord((cpu.ecx << 2) + cpu.esp, data); return; //2259
	case M_DS_4ECX_ID: cpu.ds.setWord((cpu.ecx << 2) + getMicrocode(), data); return; //2260
	case M_DS_4ECX_ESI: cpu.ds.setWord((cpu.ecx << 2) + cpu.esi, data); return; //2261
	case M_DS_4ECX_EDI: cpu.ds.setWord((cpu.ecx << 2) + cpu.edi, data); return; //2262

	case M_DS_4EDX_EAX: cpu.ds.setWord((cpu.edx << 2) + cpu.eax, data); return; //2263
	case M_DS_4EDX_ECX: cpu.ds.setWord((cpu.edx << 2) + cpu.ecx, data); return; //2264
	case M_DS_5EDX: cpu.ds.setWord((cpu.edx << 2) + cpu.edx, data); return; //2265
	case M_DS_4EDX_EBX: cpu.ds.setWord((cpu.edx << 2) + cpu.ebx, data); return; //2266
	case M_DS_4EDX_ESP: cpu.ds.setWord((cpu.edx << 2) + cpu.esp, data); return; //2267
	case M_DS_4EDX_ID: cpu.ds.setWord((cpu.edx << 2) + getMicrocode(), data); return; //2268
	case M_DS_4EDX_ESI: cpu.ds.setWord((cpu.edx << 2) + cpu.esi, data); return; //2269
	case M_DS_4EDX_EDI: cpu.ds.setWord((cpu.edx << 2) + cpu.edi, data); return; //2270

	case M_DS_4EBX_EAX: cpu.ds.setWord((cpu.ebx << 2) + cpu.eax, data); return; //2271;
	case M_DS_4EBX_ECX: cpu.ds.setWord((cpu.ebx << 2) + cpu.ecx, data); return; //2272;
	case M_DS_4EBX_EDX: cpu.ds.setWord((cpu.ebx << 2) + cpu.edx, data); return; //2273;
	case M_DS_5EBX: cpu.ds.setWord((cpu.ebx << 2) + cpu.ebx, data); return; //2274;
	case M_DS_4EBX_ESP: cpu.ds.setWord((cpu.ebx << 2) + cpu.esp, data); return; //2275;
	case M_DS_4EBX_ID: cpu.ds.setWord((cpu.ebx << 2) + getMicrocode(), data); return; //2276;
	case M_DS_4EBX_ESI: cpu.ds.setWord((cpu.ebx << 2) + cpu.esi, data); return; //2277;
	case M_DS_4EBX_EDI: cpu.ds.setWord((cpu.ebx << 2) + cpu.edi, data); return; //2278;
	    
	case M_DS_4EBP_EAX: cpu.ds.setWord((cpu.ebp << 2) + cpu.eax, data); return; //2279
	case M_DS_4EBP_ECX: cpu.ds.setWord((cpu.ebp << 2) + cpu.ecx, data); return; //2280
	case M_DS_4EBP_EDX: cpu.ds.setWord((cpu.ebp << 2) + cpu.edx, data); return; //2281
	case M_DS_4EBP_EBX: cpu.ds.setWord((cpu.ebp << 2) + cpu.ebx, data); return; //2282
	case M_DS_4EBP_ESP: cpu.ds.setWord((cpu.ebp << 2) + cpu.esp, data); return; //2283
	case M_DS_4EBP_ID: cpu.ds.setWord((cpu.ebp << 2) + getMicrocode(), data); return; //2284
	case M_DS_4EBP_ESI: cpu.ds.setWord((cpu.ebp << 2) + cpu.esi, data); return; //2285
	case M_DS_4EBP_EDI: cpu.ds.setWord((cpu.ebp << 2) + cpu.edi, data); return; //2286

	case M_DS_4ESI_EAX: cpu.ds.setWord((cpu.esi << 2) + cpu.eax, data); return; //2287;
	case M_DS_4ESI_ECX: cpu.ds.setWord((cpu.esi << 2) + cpu.ecx, data); return; //2288;
	case M_DS_4ESI_EDX: cpu.ds.setWord((cpu.esi << 2) + cpu.edx, data); return; //2289;
	case M_DS_4ESI_EBX: cpu.ds.setWord((cpu.esi << 2) + cpu.ebx, data); return; //2290;
	case M_DS_4ESI_ESP: cpu.ds.setWord((cpu.esi << 2) + cpu.esp, data); return; //2291;
	case M_DS_4ESI_ID: cpu.ds.setWord((cpu.esi << 2) + getMicrocode(), data); return; //2292;
	case M_DS_5ESI: cpu.ds.setWord((cpu.esi << 2) + cpu.esi, data); return; //2293;
	case M_DS_4ESI_EDI: cpu.ds.setWord((cpu.esi << 2) + cpu.edi, data); return; //2294;

	case M_DS_4EDI_EAX: cpu.ds.setWord((cpu.edi << 2) + cpu.eax, data); return; //2295;
	case M_DS_4EDI_ECX: cpu.ds.setWord((cpu.edi << 2) + cpu.ecx, data); return; //2296;
	case M_DS_4EDI_EDX: cpu.ds.setWord((cpu.edi << 2) + cpu.edx, data); return; //2297;
	case M_DS_4EDI_EBX: cpu.ds.setWord((cpu.edi << 2) + cpu.ebx, data); return; //2298;
	case M_DS_4EDI_ESP: cpu.ds.setWord((cpu.edi << 2) + cpu.esp, data); return; //2299;
	case M_DS_4EDI_ID: cpu.ds.setWord((cpu.edi << 2) + getMicrocode(), data); return; //2300;
	case M_DS_4EDI_ESI: cpu.ds.setWord((cpu.edi << 2) + cpu.esi, data); return; //2301;
	case M_DS_5EDI: cpu.ds.setWord((cpu.edi << 2) + cpu.edi, data); return; //2302;

	case M_DS_9EAX: cpu.ds.setWord((cpu.eax << 3) + cpu.eax, data); return; //2303;
	case M_DS_8EAX_ECX: cpu.ds.setWord((cpu.eax << 3) + cpu.ecx, data); return; //2304;
	case M_DS_8EAX_EDX: cpu.ds.setWord((cpu.eax << 3) + cpu.edx, data); return; //2305;
	case M_DS_8EAX_EBX: cpu.ds.setWord((cpu.eax << 3) + cpu.ebx, data); return; //2306;
	case M_DS_8EAX_ESP: cpu.ds.setWord((cpu.eax << 3) + cpu.esp, data); return; //2307;
	case M_DS_8EAX_ID: cpu.ds.setWord((cpu.eax << 3) + getMicrocode(), data); return; //2308;
	case M_DS_8EAX_ESI: cpu.ds.setWord((cpu.eax << 3) + cpu.esi, data); return; //2309;
	case M_DS_8EAX_EDI: cpu.ds.setWord((cpu.eax << 3) + cpu.edi, data); return; //2310;
	    
	case M_DS_8ECX_EAX: cpu.ds.setWord((cpu.ecx << 3) + cpu.eax, data); return; //2311;
	case M_DS_9ECX: cpu.ds.setWord((cpu.ecx << 3) + cpu.ecx, data); return; //2312;
	case M_DS_8ECX_EDX: cpu.ds.setWord((cpu.ecx << 3) + cpu.edx, data); return; //2313;
	case M_DS_8ECX_EBX: cpu.ds.setWord((cpu.ecx << 3) + cpu.ebx, data); return; //2314;
	case M_DS_8ECX_ESP: cpu.ds.setWord((cpu.ecx << 3) + cpu.esp, data); return; //2315;
	case M_DS_8ECX_ID: cpu.ds.setWord((cpu.ecx << 3) + getMicrocode(), data); return; //2316;
	case M_DS_8ECX_ESI: cpu.ds.setWord((cpu.ecx << 3) + cpu.esi, data); return; //2317;
	case M_DS_8ECX_EDI: cpu.ds.setWord((cpu.ecx << 3) + cpu.edi, data); return; //2318;
	    
	case M_DS_8EDX_EAX: cpu.ds.setWord((cpu.edx << 3) + cpu.eax, data); return; //2319;
	case M_DS_8EDX_ECX: cpu.ds.setWord((cpu.edx << 3) + cpu.ecx, data); return; //2320;
	case M_DS_9EDX: cpu.ds.setWord((cpu.edx << 3) + cpu.edx, data); return; //2321;
	case M_DS_8EDX_EBX: cpu.ds.setWord((cpu.edx << 3) + cpu.ebx, data); return; //2322;
	case M_DS_8EDX_ESP: cpu.ds.setWord((cpu.edx << 3) + cpu.esp, data); return; //2323;
	case M_DS_8EDX_ID: cpu.ds.setWord((cpu.edx << 3) + getMicrocode(), data); return; //2324;
	case M_DS_8EDX_ESI: cpu.ds.setWord((cpu.edx << 3) + cpu.esi, data); return; //2325;
	case M_DS_8EDX_EDI: cpu.ds.setWord((cpu.edx << 3) + cpu.edi, data); return; //2326;
	    
	case M_DS_8EBX_EAX: cpu.ds.setWord((cpu.ebx << 3) + cpu.eax, data); return; //2327
	case M_DS_8EBX_ECX: cpu.ds.setWord((cpu.ebx << 3) + cpu.ecx, data); return; //2328
	case M_DS_8EBX_EDX: cpu.ds.setWord((cpu.ebx << 3) + cpu.edx, data); return; //2329
	case M_DS_9EBX: cpu.ds.setWord((cpu.ebx << 3) + cpu.ebx, data); return; //2330
	case M_DS_8EBX_ESP: cpu.ds.setWord((cpu.ebx << 3) + cpu.esp, data); return; //2331
	case M_DS_8EBX_ID: cpu.ds.setWord((cpu.ebx << 3) + getMicrocode(), data); return; //2332
	case M_DS_8EBX_ESI: cpu.ds.setWord((cpu.ebx << 3) + cpu.esi, data); return; //2333
	case M_DS_8EBX_EDI: cpu.ds.setWord((cpu.ebx << 3) + cpu.edi, data); return; //2334
	    
	case M_DS_8EBP_EAX: cpu.ds.setWord((cpu.ebp << 3) + cpu.eax, data); return; //2335;
	case M_DS_8EBP_ECX: cpu.ds.setWord((cpu.ebp << 3) + cpu.ecx, data); return; //2336;
	case M_DS_8EBP_EDX: cpu.ds.setWord((cpu.ebp << 3) + cpu.edx, data); return; //2337;
	case M_DS_8EBP_EBX: cpu.ds.setWord((cpu.ebp << 3) + cpu.ebx, data); return; //2338;
	case M_DS_8EBP_ESP: cpu.ds.setWord((cpu.ebp << 3) + cpu.esp, data); return; //2339;
	case M_DS_8EBP_ID: cpu.ds.setWord((cpu.ebp << 3) + getMicrocode(), data); return; //2340;
	case M_DS_8EBP_ESI: cpu.ds.setWord((cpu.ebp << 3) + cpu.esi, data); return; //2341;
	case M_DS_8EBP_EDI: cpu.ds.setWord((cpu.ebp << 3) + cpu.edi, data); return; //2342;
	    
	case M_DS_8ESI_EAX: cpu.ds.setWord((cpu.esi << 3) + cpu.eax, data); return; //2343;
	case M_DS_8ESI_ECX: cpu.ds.setWord((cpu.esi << 3) + cpu.ecx, data); return; //2344;
	case M_DS_8ESI_EDX: cpu.ds.setWord((cpu.esi << 3) + cpu.edx, data); return; //2345;
	case M_DS_8ESI_EBX: cpu.ds.setWord((cpu.esi << 3) + cpu.ebx, data); return; //2346;
	case M_DS_8ESI_ESP: cpu.ds.setWord((cpu.esi << 3) + cpu.esp, data); return; //2347;
	case M_DS_8ESI_ID: cpu.ds.setWord((cpu.esi << 3) + getMicrocode(), data); return; //2348;
	case M_DS_9ESI: cpu.ds.setWord((cpu.esi << 3) + cpu.esi, data); return; //2349;
	case M_DS_8ESI_EDI: cpu.ds.setWord((cpu.esi << 3) + cpu.edi, data); return; //2350;
	    
	case M_DS_8EDI_EAX: cpu.ds.setWord((cpu.edi << 3) + cpu.eax, data); return; //2351;
	case M_DS_8EDI_ECX: cpu.ds.setWord((cpu.edi << 3) + cpu.ecx, data); return; //2352;
	case M_DS_8EDI_EDX: cpu.ds.setWord((cpu.edi << 3) + cpu.edx, data); return; //2353;
	case M_DS_8EDI_EBX: cpu.ds.setWord((cpu.edi << 3) + cpu.ebx, data); return; //2354;
	case M_DS_8EDI_ESP: cpu.ds.setWord((cpu.edi << 3) + cpu.esp, data); return; //2355;
	case M_DS_8EDI_ID: cpu.ds.setWord((cpu.edi << 3) + getMicrocode(), data); return; //2356;
	case M_DS_8EDI_ESI: cpu.ds.setWord((cpu.edi << 3) + cpu.esi, data); return; //2357;
	case M_DS_9EDI: cpu.ds.setWord((cpu.edi << 3) + cpu.edi, data); return; //2358;
	    
	case M_DS_2EAX_IB: cpu.ds.setWord(cpu.eax + cpu.eax + getMicrocode(), data); return; //2359
	case M_DS_EAX_ECX_IB: cpu.ds.setWord(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2360
	case M_DS_EAX_EDX_IB: cpu.ds.setWord(cpu.eax + cpu.edx + getMicrocode(), data); return; //2361
	case M_DS_EAX_EBX_IB: cpu.ds.setWord(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2362
	case M_DS_EAX_ESP_IB: cpu.ds.setWord(cpu.eax + cpu.esp + getMicrocode(), data); return; //2363
	case M_DS_EAX_EBP_IB: cpu.ds.setWord(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2364
	case M_DS_EAX_ESI_IB: cpu.ds.setWord(cpu.eax + cpu.esi + getMicrocode(), data); return; //2365
	case M_DS_EAX_EDI_IB: cpu.ds.setWord(cpu.eax + cpu.edi + getMicrocode(), data); return; //2366

	case M_DS_2ECX_IB: cpu.ds.setWord(cpu.ecx + cpu.ecx + getMicrocode(), data); return; //2367;
	case M_DS_ECX_EDX_IB: cpu.ds.setWord(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2368;
	case M_DS_ECX_EBX_IB: cpu.ds.setWord(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2369;
	case M_DS_ECX_ESP_IB: cpu.ds.setWord(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2370;
	case M_DS_ECX_EBP_IB: cpu.ds.setWord(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2371;
	case M_DS_ECX_ESI_IB: cpu.ds.setWord(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2372;
	case M_DS_ECX_EDI_IB: cpu.ds.setWord(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2373;

	case M_DS_2EDX_IB: cpu.ds.setWord(cpu.edx + cpu.edx + getMicrocode(), data); return; //2374
	case M_DS_EDX_EBX_IB: cpu.ds.setWord(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2375
	case M_DS_EDX_ESP_IB: cpu.ds.setWord(cpu.edx + cpu.esp + getMicrocode(), data); return; //2376
	case M_DS_EDX_EBP_IB: cpu.ds.setWord(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2377
	case M_DS_EDX_ESI_IB: cpu.ds.setWord(cpu.edx + cpu.esi + getMicrocode(), data); return; //2378
	case M_DS_EDX_EDI_IB: cpu.ds.setWord(cpu.edx + cpu.edi + getMicrocode(), data); return; //2379

	case M_DS_2EBX_IB: cpu.ds.setWord(cpu.ebx + cpu.ebx + getMicrocode(), data); return; //2380
	case M_DS_EBX_ESP_IB: cpu.ds.setWord(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2381
	case M_DS_EBX_EBP_IB: cpu.ds.setWord(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2382
	case M_DS_EBX_ESI_IB: cpu.ds.setWord(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2383
	case M_DS_EBX_EDI_IB: cpu.ds.setWord(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2384

	case M_DS_ESP_IB: cpu.ds.setWord(cpu.esp + getMicrocode(), data); return; //2385

	case M_DS_EBP_ESP_IB: cpu.ds.setWord(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2386;
	case M_DS_2EBP_IB: cpu.ds.setWord(cpu.ebp + cpu.ebp + getMicrocode(), data); return; //2387;
	case M_DS_EBP_ESI_IB: cpu.ds.setWord(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2388;
	case M_DS_EBP_EDI_IB: cpu.ds.setWord(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2389;

	case M_DS_ESI_ESP_IB: cpu.ds.setWord(cpu.esi + cpu.esp + getMicrocode(), data); return; //2390;
	case M_DS_2ESI_IB: cpu.ds.setWord(cpu.esi + cpu.esi + getMicrocode(), data); return; //2391;
	case M_DS_ESI_EDI_IB: cpu.ds.setWord(cpu.esi + cpu.edi + getMicrocode(), data); return; //2392;
	    
	case M_DS_EDI_ESP_IB: cpu.ds.setWord(cpu.edi + cpu.esp + getMicrocode(), data); return; //2393;
	case M_DS_2EDI_IB: cpu.ds.setWord(cpu.edi + cpu.edi + getMicrocode(), data); return; //2394;
	    
	case M_DS_3EAX_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2395;
	case M_DS_2EAX_ECX_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2396;
	case M_DS_2EAX_EDX_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2397;
	case M_DS_2EAX_EBX_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2398;
	case M_DS_2EAX_ESP_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2399;
	case M_DS_2EAX_EBP_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2400;
	case M_DS_2EAX_ESI_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2401;
	case M_DS_2EAX_EDI_IB: cpu.ds.setWord((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2402;
	    
	case M_DS_2ECX_EAX_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2403;
	case M_DS_3ECX_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2404;
	case M_DS_2ECX_EDX_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2405;
	case M_DS_2ECX_EBX_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2406;
	case M_DS_2ECX_ESP_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2407;
	case M_DS_2ECX_EBP_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2408;
	case M_DS_2ECX_ESI_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2409;
	case M_DS_2ECX_EDI_IB: cpu.ds.setWord((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2410;

	case M_DS_2EDX_EAX_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2411;
	case M_DS_2EDX_ECX_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2412;
	case M_DS_3EDX_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2413;
	case M_DS_2EDX_EBX_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2414;
	case M_DS_2EDX_ESP_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2415;
	case M_DS_2EDX_EBP_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2416;
	case M_DS_2EDX_ESI_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2417;
	case M_DS_2EDX_EDI_IB: cpu.ds.setWord((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2418;
	    
	case M_DS_2EBX_EAX_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2419;
	case M_DS_2EBX_ECX_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2420;
	case M_DS_2EBX_EDX_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2421;
	case M_DS_3EBX_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2422;
	case M_DS_2EBX_ESP_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2423;
	case M_DS_2EBX_EBP_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2424;
	case M_DS_2EBX_ESI_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2425;
	case M_DS_2EBX_EDI_IB: cpu.ds.setWord((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2426;
	    
	case M_DS_2EBP_EAX_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2427;
	case M_DS_2EBP_ECX_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2428;
	case M_DS_2EBP_EDX_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2429;
	case M_DS_2EBP_EBX_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2430;
	case M_DS_2EBP_ESP_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2431;
	case M_DS_3EBP_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2432;
	case M_DS_2EBP_ESI_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2433;
	case M_DS_2EBP_EDI_IB: cpu.ds.setWord((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2434;
	    
	case M_DS_2ESI_EAX_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2435;
	case M_DS_2ESI_ECX_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2436;
	case M_DS_2ESI_EDX_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2437;
	case M_DS_2ESI_EBX_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2438;
	case M_DS_2ESI_ESP_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2439;
	case M_DS_2ESI_EBP_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2440;
	case M_DS_3ESI_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2441;
	case M_DS_2ESI_EDI_IB: cpu.ds.setWord((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2442;
	    
	case M_DS_2EDI_EAX_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2443;
	case M_DS_2EDI_ECX_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2444;
	case M_DS_2EDI_EDX_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2445;
	case M_DS_2EDI_EBX_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2446;
	case M_DS_2EDI_ESP_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2447;
	case M_DS_2EDI_EBP_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2448;
	case M_DS_2EDI_ESI_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2449;
	case M_DS_3EDI_IB: cpu.ds.setWord((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2450;
	    
	case M_DS_5EAX_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2451;
	case M_DS_4EAX_ECX_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2452;
	case M_DS_4EAX_EDX_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2453;
	case M_DS_4EAX_EBX_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2454;
	case M_DS_4EAX_ESP_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2455;
	case M_DS_4EAX_EBP_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2456;
	case M_DS_4EAX_ESI_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2457;
	case M_DS_4EAX_EDI_IB: cpu.ds.setWord((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2458;

	case M_DS_4ECX_EAX_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2459;
	case M_DS_5ECX_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2460;
	case M_DS_4ECX_EDX_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2461;
	case M_DS_4ECX_EBX_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2462;
	case M_DS_4ECX_ESP_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2463;
	case M_DS_4ECX_EBP_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2464;
	case M_DS_4ECX_ESI_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2465;
	case M_DS_4ECX_EDI_IB: cpu.ds.setWord((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2466;

	case M_DS_4EDX_EAX_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2467
	case M_DS_4EDX_ECX_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2468
	case M_DS_5EDX_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2469
	case M_DS_4EDX_EBX_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2470
	case M_DS_4EDX_ESP_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2471
	case M_DS_4EDX_EBP_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2472
	case M_DS_4EDX_ESI_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2473
	case M_DS_4EDX_EDI_IB: cpu.ds.setWord((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2474

	case M_DS_4EBX_EAX_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2475;
	case M_DS_4EBX_ECX_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2476;
	case M_DS_4EBX_EDX_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2477;
	case M_DS_5EBX_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2478;
	case M_DS_4EBX_ESP_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2479;
	case M_DS_4EBX_EBP_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2480;
	case M_DS_4EBX_ESI_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2481;
	case M_DS_4EBX_EDI_IB: cpu.ds.setWord((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2482;

	case M_DS_4EBP_EAX_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2483;
	case M_DS_4EBP_ECX_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2484;
	case M_DS_4EBP_EDX_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2485;
	case M_DS_4EBP_EBX_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2486;
	case M_DS_4EBP_ESP_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2487;
	case M_DS_5EBP_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2488;
	case M_DS_4EBP_ESI_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2489;
	case M_DS_4EBP_EDI_IB: cpu.ds.setWord((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2490;
	    
	case M_DS_4ESI_EAX_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2491;
	case M_DS_4ESI_ECX_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2492;
	case M_DS_4ESI_EDX_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2493;
	case M_DS_4ESI_EBX_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2494;
	case M_DS_4ESI_ESP_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2495;
	case M_DS_4ESI_EBP_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2496
	case M_DS_5ESI_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2497;
	case M_DS_4ESI_EDI_IB: cpu.ds.setWord((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2498;
	    
	case M_DS_4EDI_EAX_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2499;
	case M_DS_4EDI_ECX_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2500;
	case M_DS_4EDI_EDX_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2501;
	case M_DS_4EDI_EBX_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2502;
	case M_DS_4EDI_ESP_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2503;
	case M_DS_4EDI_EBP_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2504;
	case M_DS_4EDI_ESI_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2505;
	case M_DS_5EDI_IB: cpu.ds.setWord((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2506;

	case M_DS_9EAX_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2507;
	case M_DS_8EAX_ECX_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2508;
	case M_DS_8EAX_EDX_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2509;
	case M_DS_8EAX_EBX_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2510;
	case M_DS_8EAX_ESP_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2511;
	case M_DS_8EAX_EBP_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2512;
	case M_DS_8EAX_ESI_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2513;
	case M_DS_8EAX_EDI_IB: cpu.ds.setWord((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2514;
	    
	case M_DS_8ECX_EAX_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2515;
	case M_DS_9ECX_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2516;
	case M_DS_8ECX_EDX_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2517;
	case M_DS_8ECX_EBX_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2518;
	case M_DS_8ECX_ESP_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2519;
	case M_DS_8ECX_EBP_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2520;
	case M_DS_8ECX_ESI_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2521;
	case M_DS_8ECX_EDI_IB: cpu.ds.setWord((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2522;
	    
	case M_DS_8EDX_EAX_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2523;
	case M_DS_8EDX_ECX_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2524;
	case M_DS_9EDX_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2525;
	case M_DS_8EDX_EBX_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2526;
	case M_DS_8EDX_ESP_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2527;
	case M_DS_8EDX_EBP_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2528;
	case M_DS_8EDX_ESI_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2529;
	case M_DS_8EDX_EDI_IB: cpu.ds.setWord((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2530;
	    
	case M_DS_8EBX_EAX_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2531
	case M_DS_8EBX_ECX_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2532
	case M_DS_8EBX_EDX_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2533
	case M_DS_9EBX_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2534
	case M_DS_8EBX_ESP_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2535
	case M_DS_8EBX_EBP_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2536
	case M_DS_8EBX_ESI_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2537
	case M_DS_8EBX_EDI_IB: cpu.ds.setWord((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2538
	    
	case M_DS_8EBP_EAX_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2539;
	case M_DS_8EBP_ECX_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2540;
	case M_DS_8EBP_EDX_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2541;
	case M_DS_8EBP_EBX_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2542;
	case M_DS_8EBP_ESP_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2543;
	case M_DS_9EBP_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2544;
	case M_DS_8EBP_ESI_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2545;
	case M_DS_8EBP_EDI_IB: cpu.ds.setWord((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2546;
	    
	case M_DS_8ESI_EAX_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2547;
	case M_DS_8ESI_ECX_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2548;
	case M_DS_8ESI_EDX_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2549;
	case M_DS_8ESI_EBX_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2550;
	case M_DS_8ESI_ESP_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2551;
	case M_DS_8ESI_EBP_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2552;
	case M_DS_9ESI_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2553;
	case M_DS_8ESI_EDI_IB: cpu.ds.setWord((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2554;
	    
	case M_DS_8EDI_EAX_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2555;
	case M_DS_8EDI_ECX_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2556;
	case M_DS_8EDI_EDX_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2557;
	case M_DS_8EDI_EBX_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2558;
	case M_DS_8EDI_ESP_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2559;
	case M_DS_8EDI_EBP_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2560;
	case M_DS_8EDI_ESI_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2561;
	case M_DS_9EDI_IB: cpu.ds.setWord((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2562;

    //SIB + ID Operands
	case M_DS_EAX_ECX_ID: cpu.ds.setWord(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2563;
	case M_DS_EAX_EDX_ID: cpu.ds.setWord(cpu.eax + cpu.edx + getMicrocode(), data); return; //2564;
	case M_DS_EAX_EBX_ID: cpu.ds.setWord(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2565;
	case M_DS_EAX_ESP_ID: cpu.ds.setWord(cpu.eax + cpu.esp + getMicrocode(), data); return; //2566;
	case M_DS_EAX_EBP_ID: cpu.ds.setWord(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2567;
	case M_DS_EAX_ESI_ID: cpu.ds.setWord(cpu.eax + cpu.esi + getMicrocode(), data); return; //2568;
	case M_DS_EAX_EDI_ID: cpu.ds.setWord(cpu.eax + cpu.edi + getMicrocode(), data); return; //2569;
	    
	case M_DS_ECX_EDX_ID: cpu.ds.setWord(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2570;
	case M_DS_ECX_EBX_ID: cpu.ds.setWord(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2571;
	case M_DS_ECX_ESP_ID: cpu.ds.setWord(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2572;
	case M_DS_ECX_EBP_ID: cpu.ds.setWord(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2573;
	case M_DS_ECX_ESI_ID: cpu.ds.setWord(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2574;
	case M_DS_ECX_EDI_ID: cpu.ds.setWord(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2575;
	    
	case M_DS_EDX_EBX_ID: cpu.ds.setWord(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2576
	case M_DS_EDX_ESP_ID: cpu.ds.setWord(cpu.edx + cpu.esp + getMicrocode(), data); return; //2577
	case M_DS_EDX_EBP_ID: cpu.ds.setWord(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2578
	case M_DS_EDX_ESI_ID: cpu.ds.setWord(cpu.edx + cpu.esi + getMicrocode(), data); return; //2579
	case M_DS_EDX_EDI_ID: cpu.ds.setWord(cpu.edx + cpu.edi + getMicrocode(), data); return; //2580

	case M_DS_EBX_ESP_ID: cpu.ds.setWord(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2581;
	case M_DS_EBX_EBP_ID: cpu.ds.setWord(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2582;
	case M_DS_EBX_ESI_ID: cpu.ds.setWord(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2583;
	case M_DS_EBX_EDI_ID: cpu.ds.setWord(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2584;
	    
	case M_DS_ESP_ID: cpu.ds.setWord(cpu.esp + getMicrocode(), data); return; //2585
	    
	case M_DS_EBP_ESP_ID: cpu.ds.setWord(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2586;
	case M_DS_EBP_ESI_ID: cpu.ds.setWord(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2587;
	case M_DS_EBP_EDI_ID: cpu.ds.setWord(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2588;
	    
	case M_DS_ESI_ESP_ID: cpu.ds.setWord(cpu.esi + cpu.esp + getMicrocode(), data); return; //2589;
	case M_DS_ESI_EDI_ID: cpu.ds.setWord(cpu.esi + cpu.edi + getMicrocode(), data); return; //2590;
	    
	case M_DS_EDI_ESP_ID: cpu.ds.setWord(cpu.edi + cpu.esp + getMicrocode(), data); return; //2591;
	    
	case M_DS_3EAX_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2592;
	case M_DS_2EAX_ECX_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2593;
	case M_DS_2EAX_EDX_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2594;
	case M_DS_2EAX_EBX_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2595;
	case M_DS_2EAX_ESP_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2596;
	case M_DS_2EAX_EBP_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2597;
	case M_DS_2EAX_ESI_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2598;
	case M_DS_2EAX_EDI_ID: cpu.ds.setWord((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2599;
	    
	case M_DS_2ECX_EAX_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2600;
	case M_DS_3ECX_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2601;
	case M_DS_2ECX_EDX_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2602;
	case M_DS_2ECX_EBX_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2603;
	case M_DS_2ECX_ESP_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2604;
	case M_DS_2ECX_EBP_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2605;
	case M_DS_2ECX_ESI_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2606;
	case M_DS_2ECX_EDI_ID: cpu.ds.setWord((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2607;
	    
	case M_DS_2EDX_EAX_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2608;
	case M_DS_2EDX_ECX_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2609;
	case M_DS_3EDX_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2610;
	case M_DS_2EDX_EBX_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2611;
	case M_DS_2EDX_ESP_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2612;
	case M_DS_2EDX_EBP_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2613;
	case M_DS_2EDX_ESI_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2614;
	case M_DS_2EDX_EDI_ID: cpu.ds.setWord((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2615;
	    
	case M_DS_2EBX_EAX_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2616;
	case M_DS_2EBX_ECX_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2617;
	case M_DS_2EBX_EDX_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2618;
	case M_DS_3EBX_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2619;
	case M_DS_2EBX_ESP_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2620;
	case M_DS_2EBX_EBP_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2621;
	case M_DS_2EBX_ESI_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2622;
	case M_DS_2EBX_EDI_ID: cpu.ds.setWord((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2623;
	    
	case M_DS_2EBP_EAX_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2624;
	case M_DS_2EBP_ECX_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2625;
	case M_DS_2EBP_EDX_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2626;
	case M_DS_2EBP_EBX_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2627;
	case M_DS_2EBP_ESP_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2628;
	case M_DS_3EBP_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2629;
	case M_DS_2EBP_ESI_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2630;
	case M_DS_2EBP_EDI_ID: cpu.ds.setWord((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2631;
	    
	case M_DS_2ESI_EAX_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2632;
	case M_DS_2ESI_ECX_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2633;
	case M_DS_2ESI_EDX_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2634;
	case M_DS_2ESI_EBX_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2635;
	case M_DS_2ESI_ESP_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2636;
	case M_DS_2ESI_EBP_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2637;
	case M_DS_3ESI_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2638;
	case M_DS_2ESI_EDI_ID: cpu.ds.setWord((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2639;
	    
	case M_DS_2EDI_EAX_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2640;
	case M_DS_2EDI_ECX_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2641;
	case M_DS_2EDI_EDX_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2642;
	case M_DS_2EDI_EBX_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2643;
	case M_DS_2EDI_ESP_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2644;
	case M_DS_2EDI_EBP_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2645;
	case M_DS_2EDI_ESI_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2646;
	case M_DS_3EDI_ID: cpu.ds.setWord((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2647;
	    
	case M_DS_5EAX_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2648
	case M_DS_4EAX_ECX_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2649
	case M_DS_4EAX_EDX_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2650
	case M_DS_4EAX_EBX_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2651
	case M_DS_4EAX_ESP_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2652
	case M_DS_4EAX_EBP_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2653
	case M_DS_4EAX_ESI_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2654
	case M_DS_4EAX_EDI_ID: cpu.ds.setWord((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2655

	case M_DS_4ECX_EAX_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2656;
	case M_DS_5ECX_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2657;
	case M_DS_4ECX_EDX_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2658;
	case M_DS_4ECX_EBX_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2659;
	case M_DS_4ECX_ESP_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2660;
	case M_DS_4ECX_EBP_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2661;
	case M_DS_4ECX_ESI_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2662;
	case M_DS_4ECX_EDI_ID: cpu.ds.setWord((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2663;
	    
	case M_DS_4EDX_EAX_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2664;
	case M_DS_4EDX_ECX_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2665;
	case M_DS_5EDX_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2666;
	case M_DS_4EDX_EBX_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2667;
	case M_DS_4EDX_ESP_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2668;
	case M_DS_4EDX_EBP_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2669;
	case M_DS_4EDX_ESI_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2670;
	case M_DS_4EDX_EDI_ID: cpu.ds.setWord((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2671;
	    
	case M_DS_4EBX_EAX_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2672;
	case M_DS_4EBX_ECX_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2673;
	case M_DS_4EBX_EDX_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2674;
	case M_DS_5EBX_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2675;
	case M_DS_4EBX_ESP_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2676;
	case M_DS_4EBX_EBP_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2677;
	case M_DS_4EBX_ESI_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2678;
	case M_DS_4EBX_EDI_ID: cpu.ds.setWord((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2679;
	    
	case M_DS_4EBP_EAX_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2680;
	case M_DS_4EBP_ECX_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2681;
	case M_DS_4EBP_EDX_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2682;
	case M_DS_4EBP_EBX_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2683;
	case M_DS_4EBP_ESP_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2684;
	case M_DS_5EBP_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2685;
	case M_DS_4EBP_ESI_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2686;
	case M_DS_4EBP_EDI_ID: cpu.ds.setWord((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2687;
	    
	case M_DS_4ESI_EAX_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2688;
	case M_DS_4ESI_ECX_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2689;
	case M_DS_4ESI_EDX_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2690;
	case M_DS_4ESI_EBX_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2691;
	case M_DS_4ESI_ESP_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2692;
	case M_DS_4ESI_EBP_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2693;
	case M_DS_5ESI_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2694;
	case M_DS_4ESI_EDI_ID: cpu.ds.setWord((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2695;

	case M_DS_4EDI_EAX_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2696;
	case M_DS_4EDI_ECX_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2697;
	case M_DS_4EDI_EDX_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2698;
	case M_DS_4EDI_EBX_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2699;
	case M_DS_4EDI_ESP_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2700;
	case M_DS_4EDI_EBP_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2701;
	case M_DS_4EDI_ESI_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2702;
	case M_DS_5EDI_ID: cpu.ds.setWord((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2703;
	    
	    
	case M_DS_9EAX_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2704;
	case M_DS_8EAX_ECX_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2705;
	case M_DS_8EAX_EDX_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2706;
	case M_DS_8EAX_EBX_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2707;
	case M_DS_8EAX_ESP_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2708;
	case M_DS_8EAX_EBP_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2709;
	case M_DS_8EAX_ESI_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2710;
	case M_DS_8EAX_EDI_ID: cpu.ds.setWord((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2711;

	case M_DS_8ECX_EAX_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2712;
	case M_DS_9ECX_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2713;
	case M_DS_8ECX_EDX_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2714;
	case M_DS_8ECX_EBX_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2715;
	case M_DS_8ECX_ESP_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2716;
	case M_DS_8ECX_EBP_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2717;
	case M_DS_8ECX_ESI_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2718;
	case M_DS_8ECX_EDI_ID: cpu.ds.setWord((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2719;

	case M_DS_8EDX_EAX_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2720;
	case M_DS_8EDX_ECX_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2721;
	case M_DS_9EDX_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2722;
	case M_DS_8EDX_EBX_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2723;
	case M_DS_8EDX_ESP_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2724;
	case M_DS_8EDX_EBP_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2725;
	case M_DS_8EDX_ESI_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2726;
	case M_DS_8EDX_EDI_ID: cpu.ds.setWord((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2727;

	case M_DS_8EBX_EAX_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2728;
	case M_DS_8EBX_ECX_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2729;
	case M_DS_8EBX_EDX_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2730;
	case M_DS_9EBX_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2731;
	case M_DS_8EBX_ESP_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2732;
	case M_DS_8EBX_EBP_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2733;
	case M_DS_8EBX_ESI_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2734;
	case M_DS_8EBX_EDI_ID: cpu.ds.setWord((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2735;
	    
	case M_DS_8EBP_EAX_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2736;
	case M_DS_8EBP_ECX_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2737;
	case M_DS_8EBP_EDX_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2738;
	case M_DS_8EBP_EBX_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2739;
	case M_DS_8EBP_ESP_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2740;
	case M_DS_9EBP_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2741;
	case M_DS_8EBP_ESI_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2742;
	case M_DS_8EBP_EDI_ID: cpu.ds.setWord((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2743;
	    
	case M_DS_8ESI_EAX_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2744;
	case M_DS_8ESI_ECX_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2745;
	case M_DS_8ESI_EDX_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2746;
	case M_DS_8ESI_EBX_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2747;
	case M_DS_8ESI_ESP_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2748;
	case M_DS_8ESI_EBP_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2749;
	case M_DS_9ESI_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2750;
	case M_DS_8ESI_EDI_ID: cpu.ds.setWord((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2751;
    
	case M_DS_8EDI_EAX_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2752;
	case M_DS_8EDI_ECX_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2753;
	case M_DS_8EDI_EDX_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2754;
	case M_DS_8EDI_EBX_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2755;
	case M_DS_8EDI_ESP_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2756;
	case M_DS_8EDI_EBP_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2757;
	case M_DS_8EDI_ESI_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2758;
	case M_DS_9EDI_ID: cpu.ds.setWord((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2759;

	case M_SS_2EAX: cpu.ss.setWord(cpu.eax + cpu.eax, data); return; //2156
	case M_SS_EAX_ECX: cpu.ss.setWord(cpu.eax + cpu.ecx, data); return; //2157
	case M_SS_EAX_EDX: cpu.ss.setWord(cpu.eax + cpu.edx, data); return; //2158
	case M_SS_EAX_EBX: cpu.ss.setWord(cpu.eax + cpu.ebx, data); return; //2159
	case M_SS_EAX_ESP: cpu.ss.setWord(cpu.eax + cpu.esp, data); return; //2160
	case M_SS_EAX_EBP: cpu.ss.setWord(cpu.eax + cpu.ebp, data); return; //2161
	case M_SS_EAX_ESI: cpu.ss.setWord(cpu.eax + cpu.esi, data); return; //2162
	case M_SS_EAX_EDI: cpu.ss.setWord(cpu.eax + cpu.edi, data); return; //2163

	case M_SS_2ECX: cpu.ss.setWord(cpu.ecx + cpu.ecx, data); return; //2164
	case M_SS_ECX_EDX: cpu.ss.setWord(cpu.ecx + cpu.edx, data); return; //2165
	case M_SS_ECX_EBX: cpu.ss.setWord(cpu.ecx + cpu.ebx, data); return; //2166
	case M_SS_ECX_ESP: cpu.ss.setWord(cpu.ecx + cpu.esp, data); return; //2167
	case M_SS_ECX_EBP: cpu.ss.setWord(cpu.ecx + cpu.ebp, data); return; //2168
	case M_SS_ECX_ESI: cpu.ss.setWord(cpu.ecx + cpu.esi, data); return; //2169
	case M_SS_ECX_EDI: cpu.ss.setWord(cpu.ecx + cpu.edi, data); return; //2170

	case M_SS_2EDX: cpu.ss.setWord(cpu.edx + cpu.edx, data); return; //2171
	case M_SS_EDX_EBX: cpu.ss.setWord(cpu.edx + cpu.ebx, data); return; //2172
	case M_SS_EDX_ESP: cpu.ss.setWord(cpu.edx + cpu.esp, data); return; //2173
	case M_SS_EDX_EBP: cpu.ss.setWord(cpu.edx + cpu.ebp, data); return; //2174
	case M_SS_EDX_ESI: cpu.ss.setWord(cpu.edx + cpu.esi, data); return; //2175
	case M_SS_EDX_EDI: cpu.ss.setWord(cpu.edx + cpu.edi, data); return; //2176

	case M_SS_2EBX: cpu.ss.setWord(cpu.ebx + cpu.ebx, data); return; //2177;
	case M_SS_EBX_ESP: cpu.ss.setWord(cpu.ebx + cpu.esp, data); return; //2178;
	case M_SS_EBX_EBP: cpu.ss.setWord(cpu.ebx + cpu.ebp, data); return; //2179;
	case M_SS_EBX_ESI: cpu.ss.setWord(cpu.ebx + cpu.esi, data); return; //2180;
	case M_SS_EBX_EDI: cpu.ss.setWord(cpu.ebx + cpu.edi, data); return; //2181;
	    
	case M_SS_ESP: cpu.ss.setWord(cpu.esp, data); return; //2182

	case M_SS_EBP_ESP: cpu.ss.setWord(cpu.ebp + cpu.esp, data); return; //2183;
	case M_SS_EBP_ESI: cpu.ss.setWord(cpu.ebp + cpu.esi, data); return; //2184;
	case M_SS_EBP_EDI: cpu.ss.setWord(cpu.ebp + cpu.edi, data); return; //2185;

	case M_SS_ESI_ESP: cpu.ss.setWord(cpu.esi + cpu.esp, data); return; //2186;
	case M_SS_2ESI: cpu.ss.setWord(cpu.esi + cpu.esi, data); return; //2187;
	case M_SS_ESI_EDI: cpu.ss.setWord(cpu.esi + cpu.edi, data); return; //2188;

	case M_SS_EDI_ESP: cpu.ss.setWord(cpu.edi + cpu.esp, data); return; //2189;
	case M_SS_2EDI: cpu.ss.setWord(cpu.edi + cpu.edi, data); return; //2190;
	    
	case M_SS_3EAX: cpu.ss.setWord((cpu.eax << 1) + cpu.eax, data); return; //2191;
	case M_SS_2EAX_ECX: cpu.ss.setWord((cpu.eax << 1) + cpu.ecx, data); return; //2192;
	case M_SS_2EAX_EDX: cpu.ss.setWord((cpu.eax << 1) + cpu.edx, data); return; //2193;
	case M_SS_2EAX_EBX: cpu.ss.setWord((cpu.eax << 1) + cpu.ebx, data); return; //2194;
	case M_SS_2EAX_ESP: cpu.ss.setWord((cpu.eax << 1) + cpu.esp, data); return; //2195;
	case M_SS_2EAX_ID: cpu.ss.setWord((cpu.eax << 1) + getMicrocode(), data); return; //2196;
	case M_SS_2EAX_ESI: cpu.ss.setWord((cpu.eax << 1) + cpu.esi, data); return; //2197;
	case M_SS_2EAX_EDI: cpu.ss.setWord((cpu.eax << 1) + cpu.edi, data); return; //2198;

	case M_SS_2ECX_EAX: cpu.ss.setWord((cpu.ecx << 1) + cpu.eax, data); return; //2199;
	case M_SS_3ECX: cpu.ss.setWord((cpu.ecx << 1) + cpu.ecx, data); return; //2200;
	case M_SS_2ECX_EDX: cpu.ss.setWord((cpu.ecx << 1) + cpu.edx, data); return; //2201;
	case M_SS_2ECX_EBX: cpu.ss.setWord((cpu.ecx << 1) + cpu.ebx, data); return; //2202;
	case M_SS_2ECX_ESP: cpu.ss.setWord((cpu.ecx << 1) + cpu.esp, data); return; //2203;
	case M_SS_2ECX_ID: cpu.ss.setWord((cpu.ecx << 1) + getMicrocode(), data); return; //2204;
	case M_SS_2ECX_ESI: cpu.ss.setWord((cpu.ecx << 1) + cpu.esi, data); return; //2205;
	case M_SS_2ECX_EDI: cpu.ss.setWord((cpu.ecx << 1) + cpu.edi, data); return; //2206;

	case M_SS_2EDX_EAX: cpu.ss.setWord((cpu.edx << 1) + cpu.eax, data); return; //2207;
	case M_SS_2EDX_ECX: cpu.ss.setWord((cpu.edx << 1) + cpu.ecx, data); return; //2208;
	case M_SS_3EDX: cpu.ss.setWord((cpu.edx << 1) + cpu.edx, data); return; //2209;
	case M_SS_2EDX_EBX: cpu.ss.setWord((cpu.edx << 1) + cpu.ebx, data); return; //2210;
	case M_SS_2EDX_ESP: cpu.ss.setWord((cpu.edx << 1) + cpu.esp, data); return; //2211;
	case M_SS_2EDX_ID: cpu.ss.setWord((cpu.edx << 1) + getMicrocode(), data); return; //2212;
	case M_SS_2EDX_ESI: cpu.ss.setWord((cpu.edx << 1) + cpu.esi, data); return; //2213;
	case M_SS_2EDX_EDI: cpu.ss.setWord((cpu.edx << 1) + cpu.edi, data); return; //2214;

	case M_SS_2EBX_EAX: cpu.ss.setWord((cpu.ebx << 1) + cpu.eax, data); return; //2215;
	case M_SS_2EBX_ECX: cpu.ss.setWord((cpu.ebx << 1) + cpu.ecx, data); return; //2216;
	case M_SS_2EBX_EDX: cpu.ss.setWord((cpu.ebx << 1) + cpu.edx, data); return; //2217;
	case M_SS_3EBX: cpu.ss.setWord((cpu.ebx << 1) + cpu.ebx, data); return; //2218;
	case M_SS_2EBX_ESP: cpu.ss.setWord((cpu.ebx << 1) + cpu.esp, data); return; //2219;
	case M_SS_2EBX_ID: cpu.ss.setWord((cpu.ebx << 1) + getMicrocode(), data); return; //2220;
	case M_SS_2EBX_ESI: cpu.ss.setWord((cpu.ebx << 1) + cpu.esi, data); return; //2221;
	case M_SS_2EBX_EDI: cpu.ss.setWord((cpu.ebx << 1) + cpu.edi, data); return; //2222;
	    
	case M_SS_2EBP_EAX: cpu.ss.setWord((cpu.ebp << 1) + cpu.eax, data); return; //2223;
	case M_SS_2EBP_ECX: cpu.ss.setWord((cpu.ebp << 1) + cpu.ecx, data); return; //2224;
	case M_SS_2EBP_EDX: cpu.ss.setWord((cpu.ebp << 1) + cpu.edx, data); return; //2225;
	case M_SS_2EBP_EBX: cpu.ss.setWord((cpu.ebp << 1) + cpu.ebx, data); return; //2226;
	case M_SS_2EBP_ESP: cpu.ss.setWord((cpu.ebp << 1) + cpu.esp, data); return; //2227;
	case M_SS_2EBP_ID: cpu.ss.setWord((cpu.ebp << 1) + getMicrocode(), data); return; //2228;
	case M_SS_2EBP_ESI: cpu.ss.setWord((cpu.ebp << 1) + cpu.esi, data); return; //2229;
	case M_SS_2EBP_EDI: cpu.ss.setWord((cpu.ebp << 1) + cpu.edi, data); return; //2230;

	case M_SS_2ESI_EAX: cpu.ss.setWord((cpu.esi << 1) + cpu.eax, data); return; //2231;
	case M_SS_2ESI_ECX: cpu.ss.setWord((cpu.esi << 1) + cpu.ecx, data); return; //2232;
	case M_SS_2ESI_EDX: cpu.ss.setWord((cpu.esi << 1) + cpu.edx, data); return; //2233;
	case M_SS_2ESI_EBX: cpu.ss.setWord((cpu.esi << 1) + cpu.ebx, data); return; //2234;
	case M_SS_2ESI_ESP: cpu.ss.setWord((cpu.esi << 1) + cpu.esp, data); return; //2235;
	case M_SS_2ESI_ID: cpu.ss.setWord((cpu.esi << 1) + getMicrocode(), data); return; //2236;
	case M_SS_3ESI: cpu.ss.setWord((cpu.esi << 1) + cpu.esi, data); return; //2237;
	case M_SS_2ESI_EDI: cpu.ss.setWord((cpu.esi << 1) + cpu.edi, data); return; //2238;

	case M_SS_2EDI_EAX: cpu.ss.setWord((cpu.edi << 1) + cpu.eax, data); return; //2239;
	case M_SS_2EDI_ECX: cpu.ss.setWord((cpu.edi << 1) + cpu.ecx, data); return; //2240;
	case M_SS_2EDI_EDX: cpu.ss.setWord((cpu.edi << 1) + cpu.edx, data); return; //2241;
	case M_SS_2EDI_EBX: cpu.ss.setWord((cpu.edi << 1) + cpu.ebx, data); return; //2242;
	case M_SS_2EDI_ESP: cpu.ss.setWord((cpu.edi << 1) + cpu.esp, data); return; //2243;
	case M_SS_2EDI_ID: cpu.ss.setWord((cpu.edi << 1) + getMicrocode(), data); return; //2244;
	case M_SS_2EDI_ESI: cpu.ss.setWord((cpu.edi << 1) + cpu.esi, data); return; //2245;
	case M_SS_3EDI: cpu.ss.setWord((cpu.edi << 1) + cpu.edi, data); return; //2246;

	case M_SS_5EAX: cpu.ss.setWord((cpu.eax << 2) + cpu.eax, data); return; //2247
	case M_SS_4EAX_ECX: cpu.ss.setWord((cpu.eax << 2) + cpu.ecx, data); return; //2248
	case M_SS_4EAX_EDX: cpu.ss.setWord((cpu.eax << 2) + cpu.edx, data); return; //2249
	case M_SS_4EAX_EBX: cpu.ss.setWord((cpu.eax << 2) + cpu.ebx, data); return; //2250
	case M_SS_4EAX_ESP: cpu.ss.setWord((cpu.eax << 2) + cpu.esp, data); return; //2251
	case M_SS_4EAX_ID: cpu.ss.setWord((cpu.eax << 2) + getMicrocode(), data); return; //2252
	case M_SS_4EAX_ESI: cpu.ss.setWord((cpu.eax << 2) + cpu.esi, data); return; //2253
	case M_SS_4EAX_EDI: cpu.ss.setWord((cpu.eax << 2) + cpu.edi, data); return; //2254

	case M_SS_4ECX_EAX: cpu.ss.setWord((cpu.ecx << 2) + cpu.eax, data); return; //2255
	case M_SS_5ECX: cpu.ss.setWord((cpu.ecx << 2) + cpu.ecx, data); return; //2256
	case M_SS_4ECX_EDX: cpu.ss.setWord((cpu.ecx << 2) + cpu.edx, data); return; //2257
	case M_SS_4ECX_EBX: cpu.ss.setWord((cpu.ecx << 2) + cpu.ebx, data); return; //2258
	case M_SS_4ECX_ESP: cpu.ss.setWord((cpu.ecx << 2) + cpu.esp, data); return; //2259
	case M_SS_4ECX_ID: cpu.ss.setWord((cpu.ecx << 2) + getMicrocode(), data); return; //2260
	case M_SS_4ECX_ESI: cpu.ss.setWord((cpu.ecx << 2) + cpu.esi, data); return; //2261
	case M_SS_4ECX_EDI: cpu.ss.setWord((cpu.ecx << 2) + cpu.edi, data); return; //2262

	case M_SS_4EDX_EAX: cpu.ss.setWord((cpu.edx << 2) + cpu.eax, data); return; //2263
	case M_SS_4EDX_ECX: cpu.ss.setWord((cpu.edx << 2) + cpu.ecx, data); return; //2264
	case M_SS_5EDX: cpu.ss.setWord((cpu.edx << 2) + cpu.edx, data); return; //2265
	case M_SS_4EDX_EBX: cpu.ss.setWord((cpu.edx << 2) + cpu.ebx, data); return; //2266
	case M_SS_4EDX_ESP: cpu.ss.setWord((cpu.edx << 2) + cpu.esp, data); return; //2267
	case M_SS_4EDX_ID: cpu.ss.setWord((cpu.edx << 2) + getMicrocode(), data); return; //2268
	case M_SS_4EDX_ESI: cpu.ss.setWord((cpu.edx << 2) + cpu.esi, data); return; //2269
	case M_SS_4EDX_EDI: cpu.ss.setWord((cpu.edx << 2) + cpu.edi, data); return; //2270

	case M_SS_4EBX_EAX: cpu.ss.setWord((cpu.ebx << 2) + cpu.eax, data); return; //2271;
	case M_SS_4EBX_ECX: cpu.ss.setWord((cpu.ebx << 2) + cpu.ecx, data); return; //2272;
	case M_SS_4EBX_EDX: cpu.ss.setWord((cpu.ebx << 2) + cpu.edx, data); return; //2273;
	case M_SS_5EBX: cpu.ss.setWord((cpu.ebx << 2) + cpu.ebx, data); return; //2274;
	case M_SS_4EBX_ESP: cpu.ss.setWord((cpu.ebx << 2) + cpu.esp, data); return; //2275;
	case M_SS_4EBX_ID: cpu.ss.setWord((cpu.ebx << 2) + getMicrocode(), data); return; //2276;
	case M_SS_4EBX_ESI: cpu.ss.setWord((cpu.ebx << 2) + cpu.esi, data); return; //2277;
	case M_SS_4EBX_EDI: cpu.ss.setWord((cpu.ebx << 2) + cpu.edi, data); return; //2278;
	    
	case M_SS_4EBP_EAX: cpu.ss.setWord((cpu.ebp << 2) + cpu.eax, data); return; //2279
	case M_SS_4EBP_ECX: cpu.ss.setWord((cpu.ebp << 2) + cpu.ecx, data); return; //2280
	case M_SS_4EBP_EDX: cpu.ss.setWord((cpu.ebp << 2) + cpu.edx, data); return; //2281
	case M_SS_4EBP_EBX: cpu.ss.setWord((cpu.ebp << 2) + cpu.ebx, data); return; //2282
	case M_SS_4EBP_ESP: cpu.ss.setWord((cpu.ebp << 2) + cpu.esp, data); return; //2283
	case M_SS_4EBP_ID: cpu.ss.setWord((cpu.ebp << 2) + getMicrocode(), data); return; //2284
	case M_SS_4EBP_ESI: cpu.ss.setWord((cpu.ebp << 2) + cpu.esi, data); return; //2285
	case M_SS_4EBP_EDI: cpu.ss.setWord((cpu.ebp << 2) + cpu.edi, data); return; //2286

	case M_SS_4ESI_EAX: cpu.ss.setWord((cpu.esi << 2) + cpu.eax, data); return; //2287;
	case M_SS_4ESI_ECX: cpu.ss.setWord((cpu.esi << 2) + cpu.ecx, data); return; //2288;
	case M_SS_4ESI_EDX: cpu.ss.setWord((cpu.esi << 2) + cpu.edx, data); return; //2289;
	case M_SS_4ESI_EBX: cpu.ss.setWord((cpu.esi << 2) + cpu.ebx, data); return; //2290;
	case M_SS_4ESI_ESP: cpu.ss.setWord((cpu.esi << 2) + cpu.esp, data); return; //2291;
	case M_SS_4ESI_ID: cpu.ss.setWord((cpu.esi << 2) + getMicrocode(), data); return; //2292;
	case M_SS_5ESI: cpu.ss.setWord((cpu.esi << 2) + cpu.esi, data); return; //2293;
	case M_SS_4ESI_EDI: cpu.ss.setWord((cpu.esi << 2) + cpu.edi, data); return; //2294;

	case M_SS_4EDI_EAX: cpu.ss.setWord((cpu.edi << 2) + cpu.eax, data); return; //2295;
	case M_SS_4EDI_ECX: cpu.ss.setWord((cpu.edi << 2) + cpu.ecx, data); return; //2296;
	case M_SS_4EDI_EDX: cpu.ss.setWord((cpu.edi << 2) + cpu.edx, data); return; //2297;
	case M_SS_4EDI_EBX: cpu.ss.setWord((cpu.edi << 2) + cpu.ebx, data); return; //2298;
	case M_SS_4EDI_ESP: cpu.ss.setWord((cpu.edi << 2) + cpu.esp, data); return; //2299;
	case M_SS_4EDI_ID: cpu.ss.setWord((cpu.edi << 2) + getMicrocode(), data); return; //2300;
	case M_SS_4EDI_ESI: cpu.ss.setWord((cpu.edi << 2) + cpu.esi, data); return; //2301;
	case M_SS_5EDI: cpu.ss.setWord((cpu.edi << 2) + cpu.edi, data); return; //2302;

	case M_SS_9EAX: cpu.ss.setWord((cpu.eax << 3) + cpu.eax, data); return; //2303;
	case M_SS_8EAX_ECX: cpu.ss.setWord((cpu.eax << 3) + cpu.ecx, data); return; //2304;
	case M_SS_8EAX_EDX: cpu.ss.setWord((cpu.eax << 3) + cpu.edx, data); return; //2305;
	case M_SS_8EAX_EBX: cpu.ss.setWord((cpu.eax << 3) + cpu.ebx, data); return; //2306;
	case M_SS_8EAX_ESP: cpu.ss.setWord((cpu.eax << 3) + cpu.esp, data); return; //2307;
	case M_SS_8EAX_ID: cpu.ss.setWord((cpu.eax << 3) + getMicrocode(), data); return; //2308;
	case M_SS_8EAX_ESI: cpu.ss.setWord((cpu.eax << 3) + cpu.esi, data); return; //2309;
	case M_SS_8EAX_EDI: cpu.ss.setWord((cpu.eax << 3) + cpu.edi, data); return; //2310;
	    
	case M_SS_8ECX_EAX: cpu.ss.setWord((cpu.ecx << 3) + cpu.eax, data); return; //2311;
	case M_SS_9ECX: cpu.ss.setWord((cpu.ecx << 3) + cpu.ecx, data); return; //2312;
	case M_SS_8ECX_EDX: cpu.ss.setWord((cpu.ecx << 3) + cpu.edx, data); return; //2313;
	case M_SS_8ECX_EBX: cpu.ss.setWord((cpu.ecx << 3) + cpu.ebx, data); return; //2314;
	case M_SS_8ECX_ESP: cpu.ss.setWord((cpu.ecx << 3) + cpu.esp, data); return; //2315;
	case M_SS_8ECX_ID: cpu.ss.setWord((cpu.ecx << 3) + getMicrocode(), data); return; //2316;
	case M_SS_8ECX_ESI: cpu.ss.setWord((cpu.ecx << 3) + cpu.esi, data); return; //2317;
	case M_SS_8ECX_EDI: cpu.ss.setWord((cpu.ecx << 3) + cpu.edi, data); return; //2318;
	    
	case M_SS_8EDX_EAX: cpu.ss.setWord((cpu.edx << 3) + cpu.eax, data); return; //2319;
	case M_SS_8EDX_ECX: cpu.ss.setWord((cpu.edx << 3) + cpu.ecx, data); return; //2320;
	case M_SS_9EDX: cpu.ss.setWord((cpu.edx << 3) + cpu.edx, data); return; //2321;
	case M_SS_8EDX_EBX: cpu.ss.setWord((cpu.edx << 3) + cpu.ebx, data); return; //2322;
	case M_SS_8EDX_ESP: cpu.ss.setWord((cpu.edx << 3) + cpu.esp, data); return; //2323;
	case M_SS_8EDX_ID: cpu.ss.setWord((cpu.edx << 3) + getMicrocode(), data); return; //2324;
	case M_SS_8EDX_ESI: cpu.ss.setWord((cpu.edx << 3) + cpu.esi, data); return; //2325;
	case M_SS_8EDX_EDI: cpu.ss.setWord((cpu.edx << 3) + cpu.edi, data); return; //2326;
	    
	case M_SS_8EBX_EAX: cpu.ss.setWord((cpu.ebx << 3) + cpu.eax, data); return; //2327
	case M_SS_8EBX_ECX: cpu.ss.setWord((cpu.ebx << 3) + cpu.ecx, data); return; //2328
	case M_SS_8EBX_EDX: cpu.ss.setWord((cpu.ebx << 3) + cpu.edx, data); return; //2329
	case M_SS_9EBX: cpu.ss.setWord((cpu.ebx << 3) + cpu.ebx, data); return; //2330
	case M_SS_8EBX_ESP: cpu.ss.setWord((cpu.ebx << 3) + cpu.esp, data); return; //2331
	case M_SS_8EBX_ID: cpu.ss.setWord((cpu.ebx << 3) + getMicrocode(), data); return; //2332
	case M_SS_8EBX_ESI: cpu.ss.setWord((cpu.ebx << 3) + cpu.esi, data); return; //2333
	case M_SS_8EBX_EDI: cpu.ss.setWord((cpu.ebx << 3) + cpu.edi, data); return; //2334
	    
	case M_SS_8EBP_EAX: cpu.ss.setWord((cpu.ebp << 3) + cpu.eax, data); return; //2335;
	case M_SS_8EBP_ECX: cpu.ss.setWord((cpu.ebp << 3) + cpu.ecx, data); return; //2336;
	case M_SS_8EBP_EDX: cpu.ss.setWord((cpu.ebp << 3) + cpu.edx, data); return; //2337;
	case M_SS_8EBP_EBX: cpu.ss.setWord((cpu.ebp << 3) + cpu.ebx, data); return; //2338;
	case M_SS_8EBP_ESP: cpu.ss.setWord((cpu.ebp << 3) + cpu.esp, data); return; //2339;
	case M_SS_8EBP_ID: cpu.ss.setWord((cpu.ebp << 3) + getMicrocode(), data); return; //2340;
	case M_SS_8EBP_ESI: cpu.ss.setWord((cpu.ebp << 3) + cpu.esi, data); return; //2341;
	case M_SS_8EBP_EDI: cpu.ss.setWord((cpu.ebp << 3) + cpu.edi, data); return; //2342;
	    
	case M_SS_8ESI_EAX: cpu.ss.setWord((cpu.esi << 3) + cpu.eax, data); return; //2343;
	case M_SS_8ESI_ECX: cpu.ss.setWord((cpu.esi << 3) + cpu.ecx, data); return; //2344;
	case M_SS_8ESI_EDX: cpu.ss.setWord((cpu.esi << 3) + cpu.edx, data); return; //2345;
	case M_SS_8ESI_EBX: cpu.ss.setWord((cpu.esi << 3) + cpu.ebx, data); return; //2346;
	case M_SS_8ESI_ESP: cpu.ss.setWord((cpu.esi << 3) + cpu.esp, data); return; //2347;
	case M_SS_8ESI_ID: cpu.ss.setWord((cpu.esi << 3) + getMicrocode(), data); return; //2348;
	case M_SS_9ESI: cpu.ss.setWord((cpu.esi << 3) + cpu.esi, data); return; //2349;
	case M_SS_8ESI_EDI: cpu.ss.setWord((cpu.esi << 3) + cpu.edi, data); return; //2350;
	    
	case M_SS_8EDI_EAX: cpu.ss.setWord((cpu.edi << 3) + cpu.eax, data); return; //2351;
	case M_SS_8EDI_ECX: cpu.ss.setWord((cpu.edi << 3) + cpu.ecx, data); return; //2352;
	case M_SS_8EDI_EDX: cpu.ss.setWord((cpu.edi << 3) + cpu.edx, data); return; //2353;
	case M_SS_8EDI_EBX: cpu.ss.setWord((cpu.edi << 3) + cpu.ebx, data); return; //2354;
	case M_SS_8EDI_ESP: cpu.ss.setWord((cpu.edi << 3) + cpu.esp, data); return; //2355;
	case M_SS_8EDI_ID: cpu.ss.setWord((cpu.edi << 3) + getMicrocode(), data); return; //2356;
	case M_SS_8EDI_ESI: cpu.ss.setWord((cpu.edi << 3) + cpu.esi, data); return; //2357;
	case M_SS_9EDI: cpu.ss.setWord((cpu.edi << 3) + cpu.edi, data); return; //2358;
	    
	case M_SS_2EAX_IB: cpu.ss.setWord(cpu.eax + cpu.eax + getMicrocode(), data); return; //2359
	case M_SS_EAX_ECX_IB: cpu.ss.setWord(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2360
	case M_SS_EAX_EDX_IB: cpu.ss.setWord(cpu.eax + cpu.edx + getMicrocode(), data); return; //2361
	case M_SS_EAX_EBX_IB: cpu.ss.setWord(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2362
	case M_SS_EAX_ESP_IB: cpu.ss.setWord(cpu.eax + cpu.esp + getMicrocode(), data); return; //2363
	case M_SS_EAX_EBP_IB: cpu.ss.setWord(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2364
	case M_SS_EAX_ESI_IB: cpu.ss.setWord(cpu.eax + cpu.esi + getMicrocode(), data); return; //2365
	case M_SS_EAX_EDI_IB: cpu.ss.setWord(cpu.eax + cpu.edi + getMicrocode(), data); return; //2366

	case M_SS_2ECX_IB: cpu.ss.setWord(cpu.ecx + cpu.ecx + getMicrocode(), data); return; //2367;
	case M_SS_ECX_EDX_IB: cpu.ss.setWord(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2368;
	case M_SS_ECX_EBX_IB: cpu.ss.setWord(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2369;
	case M_SS_ECX_ESP_IB: cpu.ss.setWord(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2370;
	case M_SS_ECX_EBP_IB: cpu.ss.setWord(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2371;
	case M_SS_ECX_ESI_IB: cpu.ss.setWord(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2372;
	case M_SS_ECX_EDI_IB: cpu.ss.setWord(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2373;

	case M_SS_2EDX_IB: cpu.ss.setWord(cpu.edx + cpu.edx + getMicrocode(), data); return; //2374
	case M_SS_EDX_EBX_IB: cpu.ss.setWord(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2375
	case M_SS_EDX_ESP_IB: cpu.ss.setWord(cpu.edx + cpu.esp + getMicrocode(), data); return; //2376
	case M_SS_EDX_EBP_IB: cpu.ss.setWord(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2377
	case M_SS_EDX_ESI_IB: cpu.ss.setWord(cpu.edx + cpu.esi + getMicrocode(), data); return; //2378
	case M_SS_EDX_EDI_IB: cpu.ss.setWord(cpu.edx + cpu.edi + getMicrocode(), data); return; //2379

	case M_SS_2EBX_IB: cpu.ss.setWord(cpu.ebx + cpu.ebx + getMicrocode(), data); return; //2380
	case M_SS_EBX_ESP_IB: cpu.ss.setWord(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2381
	case M_SS_EBX_EBP_IB: cpu.ss.setWord(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2382
	case M_SS_EBX_ESI_IB: cpu.ss.setWord(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2383
	case M_SS_EBX_EDI_IB: cpu.ss.setWord(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2384

	case M_SS_ESP_IB: cpu.ss.setWord(cpu.esp + getMicrocode(), data); return; //2385

	case M_SS_EBP_ESP_IB: cpu.ss.setWord(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2386;
	case M_SS_2EBP_IB: cpu.ss.setWord(cpu.ebp + cpu.ebp + getMicrocode(), data); return; //2387;
	case M_SS_EBP_ESI_IB: cpu.ss.setWord(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2388;
	case M_SS_EBP_EDI_IB: cpu.ss.setWord(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2389;

	case M_SS_ESI_ESP_IB: cpu.ss.setWord(cpu.esi + cpu.esp + getMicrocode(), data); return; //2390;
	case M_SS_2ESI_IB: cpu.ss.setWord(cpu.esi + cpu.esi + getMicrocode(), data); return; //2391;
	case M_SS_ESI_EDI_IB: cpu.ss.setWord(cpu.esi + cpu.edi + getMicrocode(), data); return; //2392;
	    
	case M_SS_EDI_ESP_IB: cpu.ss.setWord(cpu.edi + cpu.esp + getMicrocode(), data); return; //2393;
	case M_SS_2EDI_IB: cpu.ss.setWord(cpu.edi + cpu.edi + getMicrocode(), data); return; //2394;
	    
	case M_SS_3EAX_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2395;
	case M_SS_2EAX_ECX_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2396;
	case M_SS_2EAX_EDX_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2397;
	case M_SS_2EAX_EBX_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2398;
	case M_SS_2EAX_ESP_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2399;
	case M_SS_2EAX_EBP_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2400;
	case M_SS_2EAX_ESI_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2401;
	case M_SS_2EAX_EDI_IB: cpu.ss.setWord((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2402;
	    
	case M_SS_2ECX_EAX_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2403;
	case M_SS_3ECX_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2404;
	case M_SS_2ECX_EDX_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2405;
	case M_SS_2ECX_EBX_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2406;
	case M_SS_2ECX_ESP_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2407;
	case M_SS_2ECX_EBP_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2408;
	case M_SS_2ECX_ESI_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2409;
	case M_SS_2ECX_EDI_IB: cpu.ss.setWord((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2410;

	case M_SS_2EDX_EAX_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2411;
	case M_SS_2EDX_ECX_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2412;
	case M_SS_3EDX_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2413;
	case M_SS_2EDX_EBX_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2414;
	case M_SS_2EDX_ESP_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2415;
	case M_SS_2EDX_EBP_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2416;
	case M_SS_2EDX_ESI_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2417;
	case M_SS_2EDX_EDI_IB: cpu.ss.setWord((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2418;
	    
	case M_SS_2EBX_EAX_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2419;
	case M_SS_2EBX_ECX_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2420;
	case M_SS_2EBX_EDX_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2421;
	case M_SS_3EBX_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2422;
	case M_SS_2EBX_ESP_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2423;
	case M_SS_2EBX_EBP_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2424;
	case M_SS_2EBX_ESI_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2425;
	case M_SS_2EBX_EDI_IB: cpu.ss.setWord((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2426;
	    
	case M_SS_2EBP_EAX_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2427;
	case M_SS_2EBP_ECX_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2428;
	case M_SS_2EBP_EDX_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2429;
	case M_SS_2EBP_EBX_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2430;
	case M_SS_2EBP_ESP_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2431;
	case M_SS_3EBP_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2432;
	case M_SS_2EBP_ESI_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2433;
	case M_SS_2EBP_EDI_IB: cpu.ss.setWord((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2434;
	    
	case M_SS_2ESI_EAX_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2435;
	case M_SS_2ESI_ECX_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2436;
	case M_SS_2ESI_EDX_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2437;
	case M_SS_2ESI_EBX_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2438;
	case M_SS_2ESI_ESP_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2439;
	case M_SS_2ESI_EBP_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2440;
	case M_SS_3ESI_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2441;
	case M_SS_2ESI_EDI_IB: cpu.ss.setWord((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2442;
	    
	case M_SS_2EDI_EAX_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2443;
	case M_SS_2EDI_ECX_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2444;
	case M_SS_2EDI_EDX_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2445;
	case M_SS_2EDI_EBX_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2446;
	case M_SS_2EDI_ESP_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2447;
	case M_SS_2EDI_EBP_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2448;
	case M_SS_2EDI_ESI_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2449;
	case M_SS_3EDI_IB: cpu.ss.setWord((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2450;
	    
	case M_SS_5EAX_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2451;
	case M_SS_4EAX_ECX_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2452;
	case M_SS_4EAX_EDX_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2453;
	case M_SS_4EAX_EBX_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2454;
	case M_SS_4EAX_ESP_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2455;
	case M_SS_4EAX_EBP_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2456;
	case M_SS_4EAX_ESI_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2457;
	case M_SS_4EAX_EDI_IB: cpu.ss.setWord((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2458;

	case M_SS_4ECX_EAX_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2459;
	case M_SS_5ECX_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2460;
	case M_SS_4ECX_EDX_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2461;
	case M_SS_4ECX_EBX_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2462;
	case M_SS_4ECX_ESP_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2463;
	case M_SS_4ECX_EBP_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2464;
	case M_SS_4ECX_ESI_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2465;
	case M_SS_4ECX_EDI_IB: cpu.ss.setWord((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2466;

	case M_SS_4EDX_EAX_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2467
	case M_SS_4EDX_ECX_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2468
	case M_SS_5EDX_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2469
	case M_SS_4EDX_EBX_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2470
	case M_SS_4EDX_ESP_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2471
	case M_SS_4EDX_EBP_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2472
	case M_SS_4EDX_ESI_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2473
	case M_SS_4EDX_EDI_IB: cpu.ss.setWord((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2474

	case M_SS_4EBX_EAX_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2475;
	case M_SS_4EBX_ECX_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2476;
	case M_SS_4EBX_EDX_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2477;
	case M_SS_5EBX_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2478;
	case M_SS_4EBX_ESP_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2479;
	case M_SS_4EBX_EBP_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2480;
	case M_SS_4EBX_ESI_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2481;
	case M_SS_4EBX_EDI_IB: cpu.ss.setWord((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2482;

	case M_SS_4EBP_EAX_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2483;
	case M_SS_4EBP_ECX_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2484;
	case M_SS_4EBP_EDX_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2485;
	case M_SS_4EBP_EBX_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2486;
	case M_SS_4EBP_ESP_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2487;
	case M_SS_5EBP_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2488;
	case M_SS_4EBP_ESI_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2489;
	case M_SS_4EBP_EDI_IB: cpu.ss.setWord((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2490;
	    
	case M_SS_4ESI_EAX_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2491;
	case M_SS_4ESI_ECX_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2492;
	case M_SS_4ESI_EDX_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2493;
	case M_SS_4ESI_EBX_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2494;
	case M_SS_4ESI_ESP_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2495;
	case M_SS_4ESI_EBP_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2496
	case M_SS_5ESI_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2497;
	case M_SS_4ESI_EDI_IB: cpu.ss.setWord((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2498;
	    
	case M_SS_4EDI_EAX_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2499;
	case M_SS_4EDI_ECX_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2500;
	case M_SS_4EDI_EDX_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2501;
	case M_SS_4EDI_EBX_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2502;
	case M_SS_4EDI_ESP_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2503;
	case M_SS_4EDI_EBP_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2504;
	case M_SS_4EDI_ESI_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2505;
	case M_SS_5EDI_IB: cpu.ss.setWord((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2506;

	case M_SS_9EAX_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2507;
	case M_SS_8EAX_ECX_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2508;
	case M_SS_8EAX_EDX_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2509;
	case M_SS_8EAX_EBX_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2510;
	case M_SS_8EAX_ESP_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2511;
	case M_SS_8EAX_EBP_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2512;
	case M_SS_8EAX_ESI_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2513;
	case M_SS_8EAX_EDI_IB: cpu.ss.setWord((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2514;
	    
	case M_SS_8ECX_EAX_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2515;
	case M_SS_9ECX_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2516;
	case M_SS_8ECX_EDX_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2517;
	case M_SS_8ECX_EBX_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2518;
	case M_SS_8ECX_ESP_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2519;
	case M_SS_8ECX_EBP_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2520;
	case M_SS_8ECX_ESI_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2521;
	case M_SS_8ECX_EDI_IB: cpu.ss.setWord((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2522;
	    
	case M_SS_8EDX_EAX_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2523;
	case M_SS_8EDX_ECX_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2524;
	case M_SS_9EDX_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2525;
	case M_SS_8EDX_EBX_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2526;
	case M_SS_8EDX_ESP_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2527;
	case M_SS_8EDX_EBP_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2528;
	case M_SS_8EDX_ESI_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2529;
	case M_SS_8EDX_EDI_IB: cpu.ss.setWord((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2530;
	    
	case M_SS_8EBX_EAX_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2531
	case M_SS_8EBX_ECX_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2532
	case M_SS_8EBX_EDX_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2533
	case M_SS_9EBX_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2534
	case M_SS_8EBX_ESP_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2535
	case M_SS_8EBX_EBP_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2536
	case M_SS_8EBX_ESI_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2537
	case M_SS_8EBX_EDI_IB: cpu.ss.setWord((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2538
	    
	case M_SS_8EBP_EAX_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2539;
	case M_SS_8EBP_ECX_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2540;
	case M_SS_8EBP_EDX_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2541;
	case M_SS_8EBP_EBX_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2542;
	case M_SS_8EBP_ESP_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2543;
	case M_SS_9EBP_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2544;
	case M_SS_8EBP_ESI_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2545;
	case M_SS_8EBP_EDI_IB: cpu.ss.setWord((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2546;
	    
	case M_SS_8ESI_EAX_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2547;
	case M_SS_8ESI_ECX_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2548;
	case M_SS_8ESI_EDX_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2549;
	case M_SS_8ESI_EBX_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2550;
	case M_SS_8ESI_ESP_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2551;
	case M_SS_8ESI_EBP_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2552;
	case M_SS_9ESI_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2553;
	case M_SS_8ESI_EDI_IB: cpu.ss.setWord((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2554;
	    
	case M_SS_8EDI_EAX_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2555;
	case M_SS_8EDI_ECX_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2556;
	case M_SS_8EDI_EDX_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2557;
	case M_SS_8EDI_EBX_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2558;
	case M_SS_8EDI_ESP_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2559;
	case M_SS_8EDI_EBP_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2560;
	case M_SS_8EDI_ESI_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2561;
	case M_SS_9EDI_IB: cpu.ss.setWord((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2562;

    //SIB + ID Operanss
	case M_SS_EAX_ECX_ID: cpu.ss.setWord(cpu.eax + cpu.ecx + getMicrocode(), data); return; //2563;
	case M_SS_EAX_EDX_ID: cpu.ss.setWord(cpu.eax + cpu.edx + getMicrocode(), data); return; //2564;
	case M_SS_EAX_EBX_ID: cpu.ss.setWord(cpu.eax + cpu.ebx + getMicrocode(), data); return; //2565;
	case M_SS_EAX_ESP_ID: cpu.ss.setWord(cpu.eax + cpu.esp + getMicrocode(), data); return; //2566;
	case M_SS_EAX_EBP_ID: cpu.ss.setWord(cpu.eax + cpu.ebp + getMicrocode(), data); return; //2567;
	case M_SS_EAX_ESI_ID: cpu.ss.setWord(cpu.eax + cpu.esi + getMicrocode(), data); return; //2568;
	case M_SS_EAX_EDI_ID: cpu.ss.setWord(cpu.eax + cpu.edi + getMicrocode(), data); return; //2569;
	    
	case M_SS_ECX_EDX_ID: cpu.ss.setWord(cpu.ecx + cpu.edx + getMicrocode(), data); return; //2570;
	case M_SS_ECX_EBX_ID: cpu.ss.setWord(cpu.ecx + cpu.ebx + getMicrocode(), data); return; //2571;
	case M_SS_ECX_ESP_ID: cpu.ss.setWord(cpu.ecx + cpu.esp + getMicrocode(), data); return; //2572;
	case M_SS_ECX_EBP_ID: cpu.ss.setWord(cpu.ecx + cpu.ebp + getMicrocode(), data); return; //2573;
	case M_SS_ECX_ESI_ID: cpu.ss.setWord(cpu.ecx + cpu.esi + getMicrocode(), data); return; //2574;
	case M_SS_ECX_EDI_ID: cpu.ss.setWord(cpu.ecx + cpu.edi + getMicrocode(), data); return; //2575;
	    
	case M_SS_EDX_EBX_ID: cpu.ss.setWord(cpu.edx + cpu.ebx + getMicrocode(), data); return; //2576
	case M_SS_EDX_ESP_ID: cpu.ss.setWord(cpu.edx + cpu.esp + getMicrocode(), data); return; //2577
	case M_SS_EDX_EBP_ID: cpu.ss.setWord(cpu.edx + cpu.ebp + getMicrocode(), data); return; //2578
	case M_SS_EDX_ESI_ID: cpu.ss.setWord(cpu.edx + cpu.esi + getMicrocode(), data); return; //2579
	case M_SS_EDX_EDI_ID: cpu.ss.setWord(cpu.edx + cpu.edi + getMicrocode(), data); return; //2580

	case M_SS_EBX_ESP_ID: cpu.ss.setWord(cpu.ebx + cpu.esp + getMicrocode(), data); return; //2581;
	case M_SS_EBX_EBP_ID: cpu.ss.setWord(cpu.ebx + cpu.ebp + getMicrocode(), data); return; //2582;
	case M_SS_EBX_ESI_ID: cpu.ss.setWord(cpu.ebx + cpu.esi + getMicrocode(), data); return; //2583;
	case M_SS_EBX_EDI_ID: cpu.ss.setWord(cpu.ebx + cpu.edi + getMicrocode(), data); return; //2584;
	    
	case M_SS_ESP_ID: cpu.ss.setWord(cpu.esp + getMicrocode(), data); return; //2585
	    
	case M_SS_EBP_ESP_ID: cpu.ss.setWord(cpu.ebp + cpu.esp + getMicrocode(), data); return; //2586;
	case M_SS_EBP_ESI_ID: cpu.ss.setWord(cpu.ebp + cpu.esi + getMicrocode(), data); return; //2587;
	case M_SS_EBP_EDI_ID: cpu.ss.setWord(cpu.ebp + cpu.edi + getMicrocode(), data); return; //2588;
	    
	case M_SS_ESI_ESP_ID: cpu.ss.setWord(cpu.esi + cpu.esp + getMicrocode(), data); return; //2589;
	case M_SS_ESI_EDI_ID: cpu.ss.setWord(cpu.esi + cpu.edi + getMicrocode(), data); return; //2590;
	    
	case M_SS_EDI_ESP_ID: cpu.ss.setWord(cpu.edi + cpu.esp + getMicrocode(), data); return; //2591;
	    
	case M_SS_3EAX_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.eax + getMicrocode(), data); return; //2592;
	case M_SS_2EAX_ECX_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.ecx + getMicrocode(), data); return; //2593;
	case M_SS_2EAX_EDX_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.edx + getMicrocode(), data); return; //2594;
	case M_SS_2EAX_EBX_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.ebx + getMicrocode(), data); return; //2595;
	case M_SS_2EAX_ESP_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.esp + getMicrocode(), data); return; //2596;
	case M_SS_2EAX_EBP_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.ebp + getMicrocode(), data); return; //2597;
	case M_SS_2EAX_ESI_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.esi + getMicrocode(), data); return; //2598;
	case M_SS_2EAX_EDI_ID: cpu.ss.setWord((cpu.eax << 1) + cpu.edi + getMicrocode(), data); return; //2599;
	    
	case M_SS_2ECX_EAX_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.eax + getMicrocode(), data); return; //2600;
	case M_SS_3ECX_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.ecx + getMicrocode(), data); return; //2601;
	case M_SS_2ECX_EDX_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.edx + getMicrocode(), data); return; //2602;
	case M_SS_2ECX_EBX_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.ebx + getMicrocode(), data); return; //2603;
	case M_SS_2ECX_ESP_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.esp + getMicrocode(), data); return; //2604;
	case M_SS_2ECX_EBP_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.ebp + getMicrocode(), data); return; //2605;
	case M_SS_2ECX_ESI_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.esi + getMicrocode(), data); return; //2606;
	case M_SS_2ECX_EDI_ID: cpu.ss.setWord((cpu.ecx << 1) + cpu.edi + getMicrocode(), data); return; //2607;
	    
	case M_SS_2EDX_EAX_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.eax + getMicrocode(), data); return; //2608;
	case M_SS_2EDX_ECX_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.ecx + getMicrocode(), data); return; //2609;
	case M_SS_3EDX_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.edx + getMicrocode(), data); return; //2610;
	case M_SS_2EDX_EBX_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.ebx + getMicrocode(), data); return; //2611;
	case M_SS_2EDX_ESP_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.esp + getMicrocode(), data); return; //2612;
	case M_SS_2EDX_EBP_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.ebp + getMicrocode(), data); return; //2613;
	case M_SS_2EDX_ESI_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.esi + getMicrocode(), data); return; //2614;
	case M_SS_2EDX_EDI_ID: cpu.ss.setWord((cpu.edx << 1) + cpu.edi + getMicrocode(), data); return; //2615;
	    
	case M_SS_2EBX_EAX_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.eax + getMicrocode(), data); return; //2616;
	case M_SS_2EBX_ECX_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.ecx + getMicrocode(), data); return; //2617;
	case M_SS_2EBX_EDX_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.edx + getMicrocode(), data); return; //2618;
	case M_SS_3EBX_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.ebx + getMicrocode(), data); return; //2619;
	case M_SS_2EBX_ESP_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.esp + getMicrocode(), data); return; //2620;
	case M_SS_2EBX_EBP_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.ebp + getMicrocode(), data); return; //2621;
	case M_SS_2EBX_ESI_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.esi + getMicrocode(), data); return; //2622;
	case M_SS_2EBX_EDI_ID: cpu.ss.setWord((cpu.ebx << 1) + cpu.edi + getMicrocode(), data); return; //2623;
	    
	case M_SS_2EBP_EAX_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.eax + getMicrocode(), data); return; //2624;
	case M_SS_2EBP_ECX_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.ecx + getMicrocode(), data); return; //2625;
	case M_SS_2EBP_EDX_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.edx + getMicrocode(), data); return; //2626;
	case M_SS_2EBP_EBX_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.ebx + getMicrocode(), data); return; //2627;
	case M_SS_2EBP_ESP_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.esp + getMicrocode(), data); return; //2628;
	case M_SS_3EBP_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.ebp + getMicrocode(), data); return; //2629;
	case M_SS_2EBP_ESI_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.esi + getMicrocode(), data); return; //2630;
	case M_SS_2EBP_EDI_ID: cpu.ss.setWord((cpu.ebp << 1) + cpu.edi + getMicrocode(), data); return; //2631;
	    
	case M_SS_2ESI_EAX_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.eax + getMicrocode(), data); return; //2632;
	case M_SS_2ESI_ECX_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.ecx + getMicrocode(), data); return; //2633;
	case M_SS_2ESI_EDX_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.edx + getMicrocode(), data); return; //2634;
	case M_SS_2ESI_EBX_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.ebx + getMicrocode(), data); return; //2635;
	case M_SS_2ESI_ESP_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.esp + getMicrocode(), data); return; //2636;
	case M_SS_2ESI_EBP_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.ebp + getMicrocode(), data); return; //2637;
	case M_SS_3ESI_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.esi + getMicrocode(), data); return; //2638;
	case M_SS_2ESI_EDI_ID: cpu.ss.setWord((cpu.esi << 1) + cpu.edi + getMicrocode(), data); return; //2639;
	    
	case M_SS_2EDI_EAX_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.eax + getMicrocode(), data); return; //2640;
	case M_SS_2EDI_ECX_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.ecx + getMicrocode(), data); return; //2641;
	case M_SS_2EDI_EDX_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.edx + getMicrocode(), data); return; //2642;
	case M_SS_2EDI_EBX_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.ebx + getMicrocode(), data); return; //2643;
	case M_SS_2EDI_ESP_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.esp + getMicrocode(), data); return; //2644;
	case M_SS_2EDI_EBP_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.ebp + getMicrocode(), data); return; //2645;
	case M_SS_2EDI_ESI_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.esi + getMicrocode(), data); return; //2646;
	case M_SS_3EDI_ID: cpu.ss.setWord((cpu.edi << 1) + cpu.edi + getMicrocode(), data); return; //2647;
	    
	case M_SS_5EAX_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.eax + getMicrocode(), data); return; //2648
	case M_SS_4EAX_ECX_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.ecx + getMicrocode(), data); return; //2649
	case M_SS_4EAX_EDX_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.edx + getMicrocode(), data); return; //2650
	case M_SS_4EAX_EBX_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.ebx + getMicrocode(), data); return; //2651
	case M_SS_4EAX_ESP_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.esp + getMicrocode(), data); return; //2652
	case M_SS_4EAX_EBP_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.ebp + getMicrocode(), data); return; //2653
	case M_SS_4EAX_ESI_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.esi + getMicrocode(), data); return; //2654
	case M_SS_4EAX_EDI_ID: cpu.ss.setWord((cpu.eax << 2) + cpu.edi + getMicrocode(), data); return; //2655

	case M_SS_4ECX_EAX_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.eax + getMicrocode(), data); return; //2656;
	case M_SS_5ECX_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.ecx + getMicrocode(), data); return; //2657;
	case M_SS_4ECX_EDX_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.edx + getMicrocode(), data); return; //2658;
	case M_SS_4ECX_EBX_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.ebx + getMicrocode(), data); return; //2659;
	case M_SS_4ECX_ESP_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.esp + getMicrocode(), data); return; //2660;
	case M_SS_4ECX_EBP_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.ebp + getMicrocode(), data); return; //2661;
	case M_SS_4ECX_ESI_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.esi + getMicrocode(), data); return; //2662;
	case M_SS_4ECX_EDI_ID: cpu.ss.setWord((cpu.ecx << 2) + cpu.edi + getMicrocode(), data); return; //2663;
	    
	case M_SS_4EDX_EAX_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.eax + getMicrocode(), data); return; //2664;
	case M_SS_4EDX_ECX_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.ecx + getMicrocode(), data); return; //2665;
	case M_SS_5EDX_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.edx + getMicrocode(), data); return; //2666;
	case M_SS_4EDX_EBX_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.ebx + getMicrocode(), data); return; //2667;
	case M_SS_4EDX_ESP_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.esp + getMicrocode(), data); return; //2668;
	case M_SS_4EDX_EBP_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.ebp + getMicrocode(), data); return; //2669;
	case M_SS_4EDX_ESI_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.esi + getMicrocode(), data); return; //2670;
	case M_SS_4EDX_EDI_ID: cpu.ss.setWord((cpu.edx << 2) + cpu.edi + getMicrocode(), data); return; //2671;
	    
	case M_SS_4EBX_EAX_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.eax + getMicrocode(), data); return; //2672;
	case M_SS_4EBX_ECX_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.ecx + getMicrocode(), data); return; //2673;
	case M_SS_4EBX_EDX_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.edx + getMicrocode(), data); return; //2674;
	case M_SS_5EBX_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.ebx + getMicrocode(), data); return; //2675;
	case M_SS_4EBX_ESP_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.esp + getMicrocode(), data); return; //2676;
	case M_SS_4EBX_EBP_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.ebp + getMicrocode(), data); return; //2677;
	case M_SS_4EBX_ESI_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.esi + getMicrocode(), data); return; //2678;
	case M_SS_4EBX_EDI_ID: cpu.ss.setWord((cpu.ebx << 2) + cpu.edi + getMicrocode(), data); return; //2679;
	    
	case M_SS_4EBP_EAX_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.eax + getMicrocode(), data); return; //2680;
	case M_SS_4EBP_ECX_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.ecx + getMicrocode(), data); return; //2681;
	case M_SS_4EBP_EDX_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.edx + getMicrocode(), data); return; //2682;
	case M_SS_4EBP_EBX_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.ebx + getMicrocode(), data); return; //2683;
	case M_SS_4EBP_ESP_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.esp + getMicrocode(), data); return; //2684;
	case M_SS_5EBP_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.ebp + getMicrocode(), data); return; //2685;
	case M_SS_4EBP_ESI_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.esi + getMicrocode(), data); return; //2686;
	case M_SS_4EBP_EDI_ID: cpu.ss.setWord((cpu.ebp << 2) + cpu.edi + getMicrocode(), data); return; //2687;
	    
	case M_SS_4ESI_EAX_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.eax + getMicrocode(), data); return; //2688;
	case M_SS_4ESI_ECX_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.ecx + getMicrocode(), data); return; //2689;
	case M_SS_4ESI_EDX_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.edx + getMicrocode(), data); return; //2690;
	case M_SS_4ESI_EBX_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.ebx + getMicrocode(), data); return; //2691;
	case M_SS_4ESI_ESP_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.esp + getMicrocode(), data); return; //2692;
	case M_SS_4ESI_EBP_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.ebp + getMicrocode(), data); return; //2693;
	case M_SS_5ESI_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.esi + getMicrocode(), data); return; //2694;
	case M_SS_4ESI_EDI_ID: cpu.ss.setWord((cpu.esi << 2) + cpu.edi + getMicrocode(), data); return; //2695;

	case M_SS_4EDI_EAX_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.eax + getMicrocode(), data); return; //2696;
	case M_SS_4EDI_ECX_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.ecx + getMicrocode(), data); return; //2697;
	case M_SS_4EDI_EDX_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.edx + getMicrocode(), data); return; //2698;
	case M_SS_4EDI_EBX_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.ebx + getMicrocode(), data); return; //2699;
	case M_SS_4EDI_ESP_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.esp + getMicrocode(), data); return; //2700;
	case M_SS_4EDI_EBP_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.ebp + getMicrocode(), data); return; //2701;
	case M_SS_4EDI_ESI_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.esi + getMicrocode(), data); return; //2702;
	case M_SS_5EDI_ID: cpu.ss.setWord((cpu.edi << 2) + cpu.edi + getMicrocode(), data); return; //2703;
	    
	    
	case M_SS_9EAX_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.eax + getMicrocode(), data); return; //2704;
	case M_SS_8EAX_ECX_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.ecx + getMicrocode(), data); return; //2705;
	case M_SS_8EAX_EDX_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.edx + getMicrocode(), data); return; //2706;
	case M_SS_8EAX_EBX_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.ebx + getMicrocode(), data); return; //2707;
	case M_SS_8EAX_ESP_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.esp + getMicrocode(), data); return; //2708;
	case M_SS_8EAX_EBP_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.ebp + getMicrocode(), data); return; //2709;
	case M_SS_8EAX_ESI_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.esi + getMicrocode(), data); return; //2710;
	case M_SS_8EAX_EDI_ID: cpu.ss.setWord((cpu.eax << 3) + cpu.edi + getMicrocode(), data); return; //2711;

	case M_SS_8ECX_EAX_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.eax + getMicrocode(), data); return; //2712;
	case M_SS_9ECX_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.ecx + getMicrocode(), data); return; //2713;
	case M_SS_8ECX_EDX_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.edx + getMicrocode(), data); return; //2714;
	case M_SS_8ECX_EBX_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.ebx + getMicrocode(), data); return; //2715;
	case M_SS_8ECX_ESP_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.esp + getMicrocode(), data); return; //2716;
	case M_SS_8ECX_EBP_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.ebp + getMicrocode(), data); return; //2717;
	case M_SS_8ECX_ESI_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.esi + getMicrocode(), data); return; //2718;
	case M_SS_8ECX_EDI_ID: cpu.ss.setWord((cpu.ecx << 3) + cpu.edi + getMicrocode(), data); return; //2719;

	case M_SS_8EDX_EAX_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.eax + getMicrocode(), data); return; //2720;
	case M_SS_8EDX_ECX_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.ecx + getMicrocode(), data); return; //2721;
	case M_SS_9EDX_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.edx + getMicrocode(), data); return; //2722;
	case M_SS_8EDX_EBX_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.ebx + getMicrocode(), data); return; //2723;
	case M_SS_8EDX_ESP_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.esp + getMicrocode(), data); return; //2724;
	case M_SS_8EDX_EBP_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.ebp + getMicrocode(), data); return; //2725;
	case M_SS_8EDX_ESI_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.esi + getMicrocode(), data); return; //2726;
	case M_SS_8EDX_EDI_ID: cpu.ss.setWord((cpu.edx << 3) + cpu.edi + getMicrocode(), data); return; //2727;

	case M_SS_8EBX_EAX_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.eax + getMicrocode(), data); return; //2728;
	case M_SS_8EBX_ECX_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.ecx + getMicrocode(), data); return; //2729;
	case M_SS_8EBX_EDX_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.edx + getMicrocode(), data); return; //2730;
	case M_SS_9EBX_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.ebx + getMicrocode(), data); return; //2731;
	case M_SS_8EBX_ESP_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.esp + getMicrocode(), data); return; //2732;
	case M_SS_8EBX_EBP_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.ebp + getMicrocode(), data); return; //2733;
	case M_SS_8EBX_ESI_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.esi + getMicrocode(), data); return; //2734;
	case M_SS_8EBX_EDI_ID: cpu.ss.setWord((cpu.ebx << 3) + cpu.edi + getMicrocode(), data); return; //2735;
	    
	case M_SS_8EBP_EAX_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.eax + getMicrocode(), data); return; //2736;
	case M_SS_8EBP_ECX_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.ecx + getMicrocode(), data); return; //2737;
	case M_SS_8EBP_EDX_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.edx + getMicrocode(), data); return; //2738;
	case M_SS_8EBP_EBX_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.ebx + getMicrocode(), data); return; //2739;
	case M_SS_8EBP_ESP_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.esp + getMicrocode(), data); return; //2740;
	case M_SS_9EBP_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.ebp + getMicrocode(), data); return; //2741;
	case M_SS_8EBP_ESI_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.esi + getMicrocode(), data); return; //2742;
	case M_SS_8EBP_EDI_ID: cpu.ss.setWord((cpu.ebp << 3) + cpu.edi + getMicrocode(), data); return; //2743;
	    
	case M_SS_8ESI_EAX_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.eax + getMicrocode(), data); return; //2744;
	case M_SS_8ESI_ECX_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.ecx + getMicrocode(), data); return; //2745;
	case M_SS_8ESI_EDX_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.edx + getMicrocode(), data); return; //2746;
	case M_SS_8ESI_EBX_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.ebx + getMicrocode(), data); return; //2747;
	case M_SS_8ESI_ESP_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.esp + getMicrocode(), data); return; //2748;
	case M_SS_8ESI_EBP_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.ebp + getMicrocode(), data); return; //2749;
	case M_SS_9ESI_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.esi + getMicrocode(), data); return; //2750;
	case M_SS_8ESI_EDI_ID: cpu.ss.setWord((cpu.esi << 3) + cpu.edi + getMicrocode(), data); return; //2751;
    
	case M_SS_8EDI_EAX_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.eax + getMicrocode(), data); return; //2752;
	case M_SS_8EDI_ECX_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.ecx + getMicrocode(), data); return; //2753;
	case M_SS_8EDI_EDX_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.edx + getMicrocode(), data); return; //2754;
	case M_SS_8EDI_EBX_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.ebx + getMicrocode(), data); return; //2755;
	case M_SS_8EDI_ESP_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.esp + getMicrocode(), data); return; //2756;
	case M_SS_8EDI_EBP_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.ebp + getMicrocode(), data); return; //2757;
	case M_SS_8EDI_ESI_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.esi + getMicrocode(), data); return; //2758;
	case M_SS_9EDI_ID: cpu.ss.setWord((cpu.edi << 3) + cpu.edi + getMicrocode(), data); return; //2759;

	case M_GS_2EBX_ID: cpu.gs.setWord(cpu.ebx + cpu.ebx + getMicrocode(), data); return;

	}

	System.err.println("Unimplemented SetShort Operand: " + microcode);
	throw unimplemented;
    }

    protected final void setIntOperand(int data)
    {
	int microcode = getMicrocode();
	switch (microcode) {
	case EAX: cpu.eax = data; return;
	case ECX: cpu.ecx = data; return;
	case EDX: cpu.edx = data; return;
	case EBX: cpu.ebx = data; return;
	case ESP:  cpu.esp = data; return;
	case EBP:  cpu.ebp = data; return;
	case ESI:  cpu.esi = data; return;
	case EDI:  cpu.edi = data; return;

	case CR0: cpu.setCR0(data); return;

	case DR0: cpu.setDR0(data); return;
	case DR1: cpu.setDR1(data); return;
	case DR2: cpu.setDR2(data); return;
	case DR3: cpu.setDR3(data); return;
	case DR6: cpu.setDR6(data); return;
	case DR7: cpu.setDR7(data); return;

	case ES: loadSegment(ES, data); return;
	case CS: loadSegment(CS, data); return;
	case SS: loadSegment(SS, data); return;
	case DS: loadSegment(DS, data); return;
	case FS: loadSegment(FS, data); return;
	case GS: loadSegment(GS, data); return;
	}

	setIntMemoryOperand(microcode, data);
	return;
    }

    private final void setIntMemoryOperand(int microcode, int data)
    {
	switch (microcode) {
	case M_ES_BX_SI: cpu.es.setDoubleWord((cpu.ebx + cpu.esi) & 0xffff, data); return;
	case M_ES_BX_DI: cpu.es.setDoubleWord((cpu.ebx + cpu.edi) & 0xffff, data); return;
	case M_ES_BP_SI: cpu.es.setDoubleWord((cpu.ebp + cpu.esi) & 0xffff, data); return;
	case M_ES_BP_DI: cpu.es.setDoubleWord((cpu.ebp + cpu.edi) & 0xffff, data); return;
	case M_ES_SI: cpu.es.setDoubleWord(cpu.esi & 0xffff, data); return;
	case M_ES_DI: cpu.es.setDoubleWord(cpu.edi & 0xffff, data); return;
	case M_ES_IW: cpu.es.setDoubleWord(getMicrocode() & 0xffff, data); return;
	case M_ES_BX: cpu.es.setDoubleWord(cpu.ebx & 0xffff, data); return;
	case M_ES_BX_SI_IB: cpu.es.setDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_DI_IB: cpu.es.setDoubleWord((cpu.ebx + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_SI_IB: cpu.es.setDoubleWord((cpu.ebp + cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_DI_IB: cpu.es.setDoubleWord((cpu.ebp + cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_SI_IB: cpu.es.setDoubleWord((cpu.esi + getMicrocode()) & 0xffff, data); return;
	case M_ES_DI_IB: cpu.es.setDoubleWord((cpu.edi + getMicrocode()) & 0xffff, data); return;
	case M_ES_BP_IB: cpu.es.setDoubleWord((cpu.ebp + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_IB: cpu.es.setDoubleWord((cpu.ebx + getMicrocode()) & 0xffff, data); return;
	case M_ES_BX_SI_IW: cpu.es.setDoubleWord((cpu.ebx + cpu.esi + getMicrocode()) & 0x