/*
 * md5sum.c	- Generate/check MD5 Message Digests
 *
 * Compile and link with md5.c.  Wildcards can now be used on the commandline by
 * default.
 *
 * Written March 1993 by Branko Lankester
 * Modified June 1993 by Colin Plumb for altered md5.c.
 * Modified August 2005 by Blair Campbell for FreeDOS project and enhanced features
 */
#include <stdio.h>						/* Standard input and output */
#include <dos.h>						/* DOS function headers */
#include <string.h>						/* String-manipulation header */
#include <process.h>						/* Exit function */
#include <ctype.h>						/* Toupper function */
#include "md5.h"						/* MD5 header */
#include "kitten.h"						/* Translation header */

#ifdef UNIX							/* Defines for other OSes */
#define	FOPRTXT	"r"
#define	FOPRBIN	"r"
#else
#ifdef VMS
#define	FOPRTXT	"r","ctx=stm"
#define	FOPRBIN	"rb","ctx=stm"
#else								/* If not UNIX or VMS */
#define	FOPRTXT	"r"
#define	FOPRBIN	"rb"
#endif
#endif

int mdfile(FILE *fp, unsigned char *digest);			/* Define variables and functs */
int do_check(FILE *chkf);

short verbose = 0;
short bin_mode = 1;
short check = 0;

void
usage(void)							/* The help function */
{								/* Print help messages */
	printf("\n%s",  kittengets(0,0,"Usage: MD5SUM [/TBV] [/C [file]] | [file...]"));
	printf("\n%s",  kittengets(0,1,"Generates or checks MD5 Message Digests"));
	printf("\n%s",  kittengets(0,2,"    /C  check message digests (default is generate)"));
	printf("\n%s",  kittengets(0,3,"    /V  verbose, print file names when checking"));
	printf("\n%s",  kittengets(0,4,"    /T  read files in text mode"));
	printf("\n%s",  kittengets(0,5,"    /B  read files in binary mode (default; overrides /T)"));
	printf("\n%s",  kittengets(0,6,"The input for /C should be the list of message digests and file names"));
	printf("\n%s\n",kittengets(0,7,"that is printed on stdout by this program when it generates digests."));
	kittenclose();						/* Close catalog and exit */
	exit(2);
}

void
print_digest(unsigned char *p)
{
	int i;

	for (i = 0; i < 16; ++i)				/* Print digest of file */
		printf("%02x", *p++);
}

void
checkarg(const char *opt)
{
	short i, len;						/* Define variables */
	
	len=strlen(opt);					/* Find out the length of opt */
	if(len==1)						/* opt is not an arg, exit*/
	{
		printf("\n%s %s", kittengets(1,0,"Invalid option in"), opt);
		kittenclose();					/* Close catalog before exit */
		exit(1);
	}
	for(i=1;i<=len;i++)					/* Parse options */
	{
		switch(toupper(opt[i]))				/* Make opt upper case */
		{
			case 'H':				/* The recognized help opts */
			case '?':
				usage();
				break;
			case 'C':				/* Check digests, not generate */
				check = 1;
				break;
			case 'V':				/* Verbose mode */
				verbose = 1;
				break;
			case 'T':				/* Text mode instead of binary */
				bin_mode = 0;
				break;
			case 'B':				/* Use bin mode (default) */
				bin_mode = 1;
				break;
		}
	}
}

int
main(int argc, char **argv)
{
	short i = 1, rc = 0, c = NULL;				/* Define variables */
	FILE *fp;
	struct _find_t fileinfo;
	int filename;
	unsigned char digest[16];
	
	kittenopen("md5sum");					/* Open the message catalog */
	
	if (argc <= 1) usage();					/* No arguments */
	while(i<argc && (argv[i][0]=='/' || argv[i][0]=='-'))   /* Scan for options */
	{
		checkarg(argv[i]);				/* Run the scanning function */
		if(check)					/* Check option found */
		{
			i++;					/* The file is the next arg */
			c = i;
			if (strcmp(argv[c], NULL) == 0) fp = stdin;
			else					/* If no arg found, use stdin */
			{					/* Otherwise, use the file */
				if ((fp = fopen(argv[c], FOPRTXT)) == NULL)
				{
					printf("\n%s: %s", kittengets(1,1,"File does not exist"), (argv[c]));
					kittenclose();		/* Close catalog before exit */
					exit(2);
				}
			}
			exit(do_check(fp));			/* Check the files and exit */
		}
		i++;
	}
	if (argc < i)						/* Use stdin if no next arg */
	{							/* (Not functional yet) */
		if (mdfile(stdin, digest))			/* Generate digest from stdin */
		{
			printf("\nmd5sum: %s", kittengets(1,2,"read error on stdin"));
			kittenclose();				/* Close catalog before exit */
			exit(2);
		}
		print_digest(digest);				/* Print the digest */
		printf("\n");					/* Print newline */
		kittenclose();					/* Close catalog before exit */
		exit(0);
	}
	for (; argc >= i; i++)
	{							/* Arguments found, find files */
		filename = _dos_findfirst(strupr(argv[i]), _A_NORMAL|_A_SYSTEM|_A_HIDDEN|_A_RDONLY|_A_ARCH, &fileinfo);
		while(filename == 0)				/* Find all files matching arg */
		{
			if (bin_mode) fp = fopen(fileinfo.name, FOPRBIN);
			else fp = fopen(fileinfo.name, FOPRTXT);/* Generate according to mode */
			if (fp == NULL)				/* Check file existance */
			{
				printf("\n%s: %s", kittengets(1,1,"File does not exist"), fileinfo.name);
				rc = 2;				/* Set exit code and continue */
				continue;
			}
			if (mdfile(fp, digest))			/* Check if digest failed */
			{
				printf("\nmd5sum: %s %s", kittengets(1,3,"error reading"), fileinfo.name);
				rc = 2;				/* Set exit code and continue */
			}
			else
			{
				print_digest(digest);		/* Digest worked, print digest */
				printf(" %c%s\n", bin_mode ? '*' : ' ', fileinfo.name);
			}
			fclose(fp);				/* Close the file */
			filename = _dos_findnext(&fileinfo);	/* Find next file matching arg */
		}
		_dos_findclose(&fileinfo);			/* All files found, close find */
	}
	kittenclose();						/* Close the message catalog */
	exit(rc);						/* Exit with preset code rc */
	return 0;						/* Use return function for int */
}

int
mdfile(FILE *fp, unsigned char *digest)
{
	unsigned char buf[1024];				/* Define variables */
	MD5_CTX ctx;
	int n;

	MD5Init(&ctx);
	while ((n = fread(buf, 1, sizeof(buf), fp)) > 0)	/* Read the file */
		MD5Update(&ctx, buf, n);
	MD5Final(digest, &ctx);					/* Produce digest */
	if (ferror(fp))						/* Check for file error */
		return -1;
	return 0;						/* Return to previous function */
}

int
hex_digit(int c)						/* Binary digest */
{
	if (c >= '0' && c <= '9')
		return c - '0';
	if (c >= 'a' && c <= 'f')
		return c - 'a' + 10;
	return -1;
}

int
get_md5_line(FILE *fp, unsigned char *digest, char *file)
{
	char buf[1024];						/* Define variables*/
	int i, d1, d2, rc;
	char *p = buf;

	if (fgets(buf, sizeof(buf), fp) == NULL)		/* Read file; check for error */
		return -1;

	for (i = 0; i < 16; ++i) {				/* Generate digest */
		if ((d1 = hex_digit(*p++)) == -1)
			return 0;
		if ((d2 = hex_digit(*p++)) == -1)
			return 0;
		*digest++ = d1*16 + d2;
	}
	if (*p++ != ' ')
		return 0;
	/*
	 * next char is an attribute char, space means text file
	 * if it's a '*' the file should be checked in binary mode.
	 */
	if (*p == ' ')
		rc = 1;
	else if (*p == '*')
		rc = 2;
	else {							/* Error found, print error */
		printf("\nmd5sum: %s: %s", kittengets(1,4,"unrecognized line"), buf);
		return 0;					/* Return */
	}
	++p;
	i = strlen(p);
	if (i < 2 || i > 255)
		return 0;
	p[i-1] = '\0';
	strcpy(file, p);
	return rc;						/* Return exit code */
}

int
do_check(FILE *chkf)						/* Check digests */
{
	int rc, ex = 0, failed = 0, checked = 0;		/* Define variables */
	unsigned char chk_digest[16], file_digest[16];
	char filename[256];
	FILE *fp;
	int flen = 14;
								/* Read file with digests */
	while ((rc = get_md5_line(chkf, chk_digest, filename)) >= 0) {
		if (rc == 0)					/* not an md5 line */
			continue;
		if (verbose) {					/* Print verbose message */
			if (strlen(filename) > flen)
				flen = strlen(filename);
			printf("%-*s ", flen, filename);
		}
		if (bin_mode || rc == 2)			/* If bin mode is defined */
			fp = fopen(filename, FOPRBIN);
		else						/* If text mode is defined */
			fp = fopen(filename, FOPRTXT);
		if (fp == NULL) {				/* File to check doesn't exist */
			printf("\nmd5sum: %s %s", kittengets(1,5,"can't open"), filename);
			ex = 2;					/* Exit code to return */
			continue;
		}
		if (mdfile(fp, file_digest)) {			/* Error reading file to check */
			printf("\nmd5sum: %s %s", kittengets(1,4,"error reading"), filename);
			ex = 2;					/* Exit code to return */
			fclose(fp);				/* Close the file */
			continue;
		}
		fclose(fp);					/* Close file after digesting */
		if (memcmp(chk_digest, file_digest, 16) != 0) {	/* Digest does not match */
			if (verbose)				/* Print verbose message */
				printf("\n%s", kittengets(1,6,"FAILED"));
			else					/* Print non-verbose message */
				printf("\nmd5sum: %s '%s'", kittengets(1,7,"MD5 check failed for"), filename);
			++failed;				/* Increment failed */
		} else if (verbose)
			printf("OK\n");				/* Print verbose message */
		++checked;					/* Increment checked */
	}
	if (verbose && failed)					/* Print verbose if failed */
		printf("\nmd5sum: %d %s %d %s", failed, kittengets(1,8, "of"), checked, kittengets(1,9,"file(s) failed MD5 check"));
	if (!checked) {						/* If no files were checked */
		printf("\nmd5sum: %s", kittengets(1,10,"no files checked"));
		kittenclose();					/* Close message catalog */
		return 3;					/* Exit the function */
	}
	if (!ex && failed) ex = 1;				/* Change exit code if failed */
	kittenclose();						/* Close message catalog */
	return ex;						/* Exit the function with 'ex' */
}
