(**********************************************************************
** LinkLib 0.9                                                       **
** ----------------------------------------------------------------- **
** Interface to deal with FreeDOS symbolic links                     **
** ----------------------------------------------------------------- **
** Cronos, 1992,1999-2000                                            **
** This unit is distributed with NO WARRANTY, under the terms of     **
** the GNU-GPL (ver 2.0 or later) license                            **
** ----------------------------------------------------------------- **
** Original author: Aitor Santamara Merino (0.9)                    **
** Modifications: <NONE>                                             **
**********************************************************************)

UNIT LinkLib;

INTERFACE

(********************************************************************
** checklink: checks if a fname or fname.vlk is a valid link       **
** --------------------------------------------------------------- **
** INPUT:  fname:   filename to be tested (with ot without .ext)   **
** OUTPUT: TRUE if it is really a link, FALSE otherwise            **
********************************************************************)
FUNCTION CheckLink(var fname: string): boolean;


(********************************************************************
** linkgetdir: gets the starting dir for the target of the link    **
** --------------------------------------------------------------- **
** INPUT:  lname:   link name                                      **
** OUTPUT: the working path of the link                            **
********************************************************************)
FUNCTION linkgetdir (lname: string): string;


(********************************************************************
** linkgettarget: gets the file pointed by the symbolic link       **
** --------------------------------------------------------------- **
** INPUT:  lname:   link name                                      **
** OUTPUT: the target of the link                                  **
********************************************************************)
FUNCTION linkgettarget (lname: string): string;


(********************************************************************
** linkgeticon32: gets the icon32 for the symbolic link            **
** --------------------------------------------------------------- **
** INPUT:  lname:   link name                                      **
** OUTPUT: the name of the icon file                               **
********************************************************************)
FUNCTION linkgeticon32 (lname: string): string;


(********************************************************************
** startlink:  EXECUTES a link                                     **
** --------------------------------------------------------------- **
** INPUT:  lname:   link name                                      **
** OUTPUT: 1: link error; otherwise, DOSExitCode result            **
********************************************************************)
FUNCTION  startlink (lname: string): integer;


(********************************************************************
** createlink:  creates a new link                                 **
** --------------------------------------------------------------- **
** INPUT:  lname: new link name                                    **
**         target: complete or relative path to the pointed file   **
**         startdir: the working path of the link (can be '')      **
**         icon32: 32-pixel icon (can be '')                       **
** OUTPUT: success status                                          **
** NOTES: the existence of the target file is NOT checked          **
********************************************************************)
FUNCTION createlink (lname, target, startdir, icon32: string): boolean;


IMPLEMENTATION

USES
    DOS, PasExt, RegFS;

CONST
     Lext = '.lnk';


(* checks if fname or fname.<lext> exist *)
FUNCTION TrueFName (fname: string): string;
BEGIN
     IF NOT FileExist (fname) THEN
        IF NOT FileExist(fname+Lext)
           THEN fname := ''
           ELSE fname := fname+Lext;
     TrueFName := fname
END;

{$IFNDEF FPC}
FUNCTION SUpCase (str: string): string;
VAR
   I : byte;
BEGIN
     IF length(str)>0 THEN
     FOR I:=1 to length(str) DO
         str[i] := upcase(str[i]);
     SUpCase := str
END;
{$ELSE}
FUNCTION SUpCase (str: string): string;
BEGIN
     SUpCase := upcase(str)
END;
{$ENDIF}


(* checks if fname or fname.<lext> is a link *)
FUNCTION CheckLink(var fname: string): boolean;
BEGIN
     FName := TrueFName (fname);
     IF fname=''
        THEN CheckLink := FALSE
        ELSE CheckLink := supcase(lext)=supcase(Copy(fname,length(fname)-3,4))
END;


(* removes initial and last CHARACTER, supposed to be " *)
function NoQuotes (p: string): string;
Begin
     NoQuotes := Copy (p,2,length(p)-2)
End;


(* gets the starting directory of the link *)
FUNCTION linkgetdir (lname: string): string;
VAR
   retdir: string;
BEGIN
     IF CheckLink (lname) THEN BEGIN
        retdir := getlabel (lname, 'definition', 'workpath', 1);
        if retdir='' then linkgetdir := ''
                     else linkgetdir := noquotes (retdir)
     END
       ELSE linkgetdir := ''
END;

        
(* gets the target of a link *)
FUNCTION linkgettarget (lname: string): string;
VAR
   retdir: string;
BEGIN
     IF CheckLink (lname) THEN BEGIN
        retdir := getlabel (lname, 'definition', 'link', 1);
        if retdir='' then linkgettarget := ''
                     else linkgettarget := noquotes (retdir)
     END
       ELSE linkgettarget := ''
END;


(* gets the icon32 of a link *)
FUNCTION linkgeticon32 (lname: string): string;
VAR
   retdir: string;
BEGIN
     IF CheckLink (lname) THEN BEGIN
        retdir := getlabel (lname, 'definition', 'icon32', 1);
        if retdir='' then linkgeticon32 := ''
                     else linkgeticon32 := noquotes (retdir)
     END
       ELSE linkgeticon32 := ''
END;


(* opens a link *)
FUNCTION  startlink (lname: string): integer;
VAR
   sd,fn: string;
BEGIN
     while checklink(lname) do begin
         sd := linkgetdir (lname);
         IF sd<>'' THEN chdir (sd);
         fn := linkgettarget (lname);
         lname := fn
     end;

     (* Code to be REPLACED in the future by using "START "+fn *)
     IF fn<>''
        THEN BEGIN Exec (fn,'');
                   startlink := DOSExitCode
             END
        ELSE startlink := 1
END;


(* creates a link with the information specified *)
FUNCTION createlink (lname, target, startdir, icon32: string): boolean;
VAR
   F: text;
BEGIN
     Assign (F, lname +lext);
     {$I-}
     Reset (F);
     IF IOResult<>0 then createLink := FALSE
     else begin
     {$I+}
       WriteLn (F,'[definition]');
       WriteLn (F,'link="',target,'"');
       IF startdir<>'' THEN WriteLn (F,'workpath="',startdir,'"');
       IF icon32<>'' THEN WriteLn (F,'icon32="',icon32,'"');
       Close(F);
       createlink := TRUE
    end;
END;

BEGIN
END.
