;-------------------------------------------------------------------------------
; FIN -- Free-DOS file information utility
;
; (c) Copyright 1993-1997 by K. Heidenstrom.
;
; This program is free software.  You may redistribute it and/or
; modify it under the terms of the GNU General Public License as
; published by the Free Software Foundation; either version 2 of
; the License, or (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	In no
; event will the author be liable for any damages of any kind
; related to the use of this program.  See the GNU General Public
; License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;
; If you find any bugs or make significant improvements, please
; consider sending the details to the author so that other Free-DOS
; users may benefit.  Contact details are:
;
; Internet:  kheidens@actrix.gen.nz
; Snail:     K. Heidenstrom c/- P.O. Box 27-103, Wellington, New Zealand
;
;-------------------------------------------------------------------------------
;
; This program must be assembled with Borland's TASM.  The syntax is:
;
;	tasm /ml /t /w2 /denglish fin;
;	tlink /t fin, fin, nul
;	touch -cf fin.com
;
; The /denglish option to TASM specifies the language for which FIN is to
; be assembled.  National language support is implemented via language
; specific include files which are INCLUDEd via the INCLUDE file FIN.NLS.
; These language specific files are named FIN.Ixx where 'xx' indicates the
; language.  Currently only FIN.IEN (English) is defined.  Refer to FIN.NLS
; for information on creating new language-specific include files for FIN.
;
;-------------------------------------------------------------------------------
;
; Modified:
;
; KH.19930122.001  0.0.0  Started
; KH.19930123.002  1.0.0  Working version, optimised CRC calculation slightly
; KH.19930124.003  1.0.1  Generate CRC table at runtime, optimised table layout
;		   1.0.2  Added -A option to enable hidden/system files
; KH.19930126.004  1.1.0  Changed name from CRC to FIN, added file info options
; KH.19930127.005  1.1.1  Added display of total and free clusters on drive,
;			  fixed bug where file time is always shown incorrectly
; KH.19930214.006  1.1.2  Changed format of disk cluster display message
; KH.19930513.007  1.2.0  Display total clusters, histogram, 'Each' option
; KH.19930522.008  1.2.1  Fixed bug in mainline, showed histogram if -C given!
; KH.19930615.009	  Removed C source comment from histogram display func.
; KH.19931224.010  1.2.2  Fixed total clusters display
; KH.19940418.011  1.2.3  Display total bytes in all files if -D option given
; KH.19940725.012  1.2.4  Don't display total bytes in all files if no files!
;			  Don't space pad total of files found - looks dumb.
; KH.19940726.013  1.2.5  Bug fix and change to space padding stuff
; KH.19940730.014  1.2.6  Use library print-decimal routine in PRDEC32.ASM
; KH.19940814.015  1.2.7  Fixed bug for ConvDec16, wasn't loading CX with 5
; KH.19940815.016  1.3.0  Added options to -C for byte- and word-wise checksums
; KH.19940818.017  1.3.1  Bug fix, if -H specified, program would terminate.
; KH.19941117.018  1.3.2  Combine total clusters and bytes if both -S and -D
; KH.19950111.019  1.3.3  Begun conversion for Free-DOS
; KH.19950121.020	  More work
; KH.19950125.021	  More work
; KH.19950126.022	  Word counting stuff started
; KH.19950128.023	  Work on word counting stuff, two classification tables
; KH.19950130.024	  More work on documentation and tidying up
; KH.19950201.025	  More work on pathspec qualifying stuff
; KH.19950202.026  1.4.0  Finished pathspec stuff, fixed bug in histogram output
; KH.19950219.027  1.4.1  Split QualPathspec function into separate file
; KH.19950521.028  1.4.2  Internationalisation via FIN.NLS and FIN.Ixx files
; KH.19950715.029  1.4.3  Added -P option
; KH.19950716.030  1.4.4  Added -L option
; KH.19950717.031  1.4.5  Default to '*.*' if no pathspec, if switches given
; KH.19951217.032  1.4.6  Added -F option (filename only, suppress paths)
; KH.19960218.033  1.4.7  Fixed bug - was checking CFlag instead of SFlag on
;			  line after "jz NoTotB", causing FIN -DCRC to report
;			  "... in 0 clusters..." and FIN -DS to report clusters
;			  separately instead of same line as bytes/files totals,
;			  Also added '6' and '3' sub-options for -S option to
;			  support 16 and 32K cluster sizes
; KH.19960806.034  1.4.8  Added -9 option (reports non-ISO9660-compliant file
;			  names).
; KH.19970602.035  1.4.9  Increased space for pathname from 65 to 120 chars
; KH.19970908.036  1.5.0  Reassembled after correction to FIN.IEN and to
;			  QUALPATH.ASM to (to handle invalid drive letters);
;			  also has copyright year ranges updated to 1997

; Idea - if displaying total bytes and total files, also calculate and display
; total bytes _allocated_ based on cluster size of drive where files reside.
;
Ver		EQU	1
SubVer		EQU	5
ModVer		EQU	0
VerDate		EQU	"19970908"		; YYYYMMDD format

;-------------------------------------------------------------------------------

		PAGE	,132
		IDEAL
		%TITLE	"FIN -- Free-DOS file information utility"
		NAME	FIN

		INCLUDE	"FIN.NLS"	; Include appropriate language

; Notes ------------------------------------------------------------------------

; This program requires include files QUALPATH.ASM and PRDEC32.ASM

; Documentation ----------------------------------------------------------------
;
; This program displays various pieces of information about a file or a list
; of files.  The program can recurse subdirectories if the -R switch is given.
; Any combination of the following can be displayed:
;
;	Directory information (date, time, and size)		(-D switch)
;	Attributes (read-only, archive, system, hidden)		(-A switch)
;	Number of clusters occupied				(-Sc switch)
;	CRC or checksum						(-Cxx switch)
;	Word and line counts					(-Wn switch)
;	Histogram (frequency of values)				(-H switch)
;
; If more than two files are processed, totals are displayed for:
;
;	Size
;	Number of clusters
;	Word and line counts
;	Histogram
;
; The CRC section of this program was based on a program called CRC, written
; in C by Gary S. Brown, Copyright 1986, which I believe was made free for
; general use.	The code performs a 32-bit CRC calculation, known as ANSI
; X3.66 CRC, on a set of files.  This implementation is rewritten from
; scratch in assembly language and also performs the CRC table generation
; at runtime, rather than having the table hard-coded into the source.
;
; The following description is quoted from the documentation in CRC.C:
;
; "This CRC is used as the frame check sequence in ADCCP (ANSI X3.66, also
; known as FIPS PUB 71 and FED-STD-1003, the U.S. versions of CCITT's X.25
; link-level protocol).  The 32-bit FCS was added via the Federal Register
; 1 June 1982, p.23798.  I presume but don't know for certain that this
; polynomial is or will be included in CCITT V.41, which defines the 16-bit
; CRC (often called CRC-CCITT) polynomial.  FIPS PUB 78 says that the 32-bit
; FCS reduces otherwise undetected errors by a factor of 10^-5 over 16-bit FCS.
;
; "First, the polynomial itself and its table of feedback terms.  The polynomial
; is X^32+X^26+X^23+X^22+X^16+X^12+X^11+X^10+X^8+X^7+X^5+X^4+X^2+X^1+X^0.
; Note that we take it "backwards" and put the highest-order term in the
; lowest-order bit.  The X^32 term is "implied"; the LSB is the X^31 term,
; etc.	The X^0 term (usually shown as "+1") results in the MSB being 1.
; The usual hardware shift register implementation, which is what we're using
; (we're merely optimizing it by doing eight-bit chunks at a time) shifts bits
; into the lowest-order term.  In this implementation, that means shifting
; towards the right.  This is done because the calculated CRC must be
; transmitted in order from highest-order to lowest-order term.  UARTs
; transmit characters in order from LSB to MSB, so the CRC is stored this
; way and is sent to the UART low-byte to high-byte, the UART sends each
; byte low-bit to hight-bit, and the result is transmission bit by bit
; from highest-order to lowest-order term without requiring bit shuffling".
;
; The above description was quoted from CRC.C by Gary S. Brown, Copyright 1986.
;
; This program INCLUDEs the file PRDEC32.ASM which contains a nice little
; generic routine to display a 32-bit number in decimal representation.
; See that file for the details.
;
; ------------------------------------------------------------------------------

; Equates

AppName		EQU	"FIN"

TestQual	=	0		; Debugging conditional

CharClasses	=	2		; Number of char class tables

NDTAs		=	32		; Number of nested DTAs for recurse
InBufSize	=	52*1024		; Size of input buffer - MUST BE EVEN!!!

; Offsets into DTA

DTA_Attrib	=	21		; BYTE	Attribute of file found
DTA_Time	=	22		; WORD	Time stamp word
DTA_Date	=	24		; WORD	Date stamp word
DTA_FSize	=	26		; DWORD File size, DWord
DTA_FName	=	30		; CHARS Filename, ASCIIZ, 13 characters
DTASize		=	43		; Size of DTA
DTAAlloc	=	44		; Number of bytes to allocate for a DTA

; Shorthand

MACRO	point	Reg,Label
		mov	Reg,OFFSET Label
ENDM

MACRO	clr	Regs
		IRP	Reg,<Regs>
		xor	Reg,Reg
		ENDM
ENDM

MACRO	cje	Arg1,Arg2,Label
		cmp	Arg1,Arg2
		je	Label
ENDM

MACRO	WCST	NewState,Words,Lines
		DW	NewState
		DB	(Words SHL 1) + Lines
ENDM

MACRO	prexp	Text1,Exp,Text2		; Invoke in MASM mode only
		%OUT	Text1 &Exp Text2
ENDM

NL		EQU	13,10

;-------------------------------------------------------------------------------

; Program

		SEGMENT	ComFile
		ASSUME	cs:ComFile,ds:ComFile,es:nothing,ss:nothing

		ORG	0100h		; Com-type file
PROC	Main		near
		jmp	Main2
ENDP	Main

Signature	DB	"SIG: Free-DOS ",AppName,".COM version ",Ver+"0","."
		DB	SubVer+"0",".",ModVer+"0",", ",VerDate,": ",LangName,"; "
;---------------
;!! Change the following line if you create a derivative version of this program
		DB	"Original"
;---------------
		DB	"  ",26		; Ctrl-Z
SigLen		=	$ - Main	; Length of program signature

; Transient messages -----------------------------------------------------------

Transient_Msgs				; Macro from the language specific
					;   include file
IntErrE		DB	227,0		; Internal error - no message is issued

; Transient tables -------------------------------------------------------------

; Info for data format, per date format number (from DOS country info)

DateHTbl	DW	DateFormat0
		DW	DateFormat1
		DW	DateFormat2

DateFormat1	DW	DateDay
		DW	DateMonth
DateFormat2	DW	DateYear
DateFormat0	DW	DateMonth
		DW	DateDay
		DW	DateYear

DateSeps	DB	"--."		; Date separator chars: US, Euro, Jap.

StarDotStar	DB	"*.*",13

; Option switches --------------------------------------------------------------

; Initial switch letters and sub-option handler addresses - note, if switches
; are changed, also change the switch flag variables starting at 'AFlag'.

Switches	DB	"9ACDFHLPRSW"	; Switches - 'c', 's', 'w' have subopts
NumSwitches	=	$ - Switches	; Number of switches

SwitchSubs	DW	SwSub_None	; No sub-options for '9'
		DW	SwSub_None	; No sub-options for 'A'
		DW	SwSub_C		; Sub-option handler for 'C'
		DW	SwSub_None	; No sub-options for 'D'
		DW	SwSub_None	; No sub-options for 'F'
		DW	SwSub_None	; No sub-options for 'H'
		DW	SwSub_None	; No sub-options for 'L'
		DW	SwSub_None	; No sub-options for 'P'
		DW	SwSub_None	; No sub-options for 'R'
		DW	SwSub_S		; Sub-option handler for 'S'
		DW	SwSub_W		; Sub-option handler for 'W'

		IF	($ - SwitchSubs) NE (NumSwitches SHL 1)
		ERR	"Number of switch subroutines does not match number of switch letters"
		ENDIF

; This table gives the number of times that '1' should be shifted left for each
; character from "1" to "8" parsed following the '-S' option.  Numbers 1, 2, 4,
; 5, and 8 correspond to 1K, 2K, 4K, 512-byte, and 8K cluster sizes.  Entries
; 3, 6, and 7 are invalid, and will be shifted 16 times so that the '1' shifts
; into carry and an error will be flagged.

NumToBPCShift	DB	10,11,15,12,9,14,16,13

; Sub-options to '-C' switch with associated CRC function pointers

CRCTypes	DW	"CR","SB","SW","XB","XW"
CRCFuncs	DW	Pass_CRC32,Pass_Chk_BS,Pass_Chk_WS
		DW	Pass_Chk_BX,Pass_Chk_WX

; Here are the five character classes used by the word counting functions.
; They are numbered in increments of three because each entry in the state
; tables occupies three bytes (two-byte pointer plus one byte for bitflags).

CC_EOF		=	0	; Character 26 only
CC_CR		=	3	; Character 13 only
CC_LF		=	6	; Character 10 only
CC_Text		=	9	; Characters that will comprise 'words'
CC_White	=	12	; Characters that will comprise whitespace

; The following tables are simple RLE-encoded representations of the character
; classification tables.  The tables are generated from this information at
; run-time.  There are two classification tables - table 1 is the 'text' char
; classification table, which defines characters "0"-"9", "A"-"Z", "a"-"z",
; 128-154, and 160-167 as text and others as whitespace, and table 2 is the
; 'string' classification table, which defines "!" - 0FFh excluding "-" as
; text and others as whitespace.  In both tables, characters 10, 13, and 26
; are specially defined as LF, CR, and EOF respectively.
; One of the two tables will be used by the word count function, according to
; the WCClass variable, which in turn is set according to the 'n' value given
; in the -Wn option on the command line.

; 'Text' classification table

CharClassRLE	DB	10,CC_White	; 0-9		Whitespace
		DB	 1,CC_LF	; 10		Linefeed
		DB	 2,CC_White	; 11,12		Whitespace
		DB	 1,CC_CR	; 13		Carriage return
		DB	12,CC_White	; 14-25		Whitespace
		DB	 1,CC_EOF	; 26		End of file
		DB	21,CC_White	; 27-47		Whitespace
		DB	10,CC_Text	; "0"-"9"	Text
		DB	 7,CC_White	; ":"-"@"	Whitespace
		DB	26,CC_Text	; "A"-"Z"	Text
		DB	 6,CC_White	; "["-"`"	Whitespace
		DB	26,CC_Text	; "a"-"z"	Text
		DB	 5,CC_White	; "{"-Del	Whitespace
		DB	27,CC_Text	; ""-""	Text
		DB	 5,CC_White	; ""-""	Whitespace
		DB	 8,CC_Text	; ""-""	Text (?)
		DB	88,CC_White	; ""-0FFh	Whitespace

; 'String' classification table

		DB	10,CC_White	; 0-9		Whitespace
		DB	 1,CC_LF	; 10		Linefeed
		DB	 2,CC_White	; 11,12		Whitespace
		DB	 1,CC_CR	; 13		Carriage return
		DB	12,CC_White	; 14-25		Whitespace
		DB	 1,CC_EOF	; 26		End of file
		DB	 6,CC_White	; 27-32		Whitespace
		DB	12,CC_Text	; "!"-","	Text
		DB	 1,CC_White	; "-"		Whitespace
		DB	210,CC_Text	; "."-0FFh	Text
		DB	0		; End of table

; Word counting state tables - see notes just before Pass_WordCount for details
;
; Ŀ
; 	   -- EOF --  -- CR ---  -- LF ---  - Text --  - White - 
;  Name   WL	State  WL  State  WL	State  WL  State  WL	State 
; Ĵ
;   Zer    00  EOF    01  CR1    01  LF1    10  Txt    00  Wht  
;   Wht    01  EOF    01  CR1    01  LF1    10  Txt    00  Wht  
;   Txt    01  EOF    01  CR1    01  LF1    00  Txt    00  Wht  
;   CR1    00  EOF    01  CR1    00  Zer    10  Txt    00  Wht  
;   LF1    00  EOF    00  Zer    01  LF1    10  Txt    00  Wht  
;   EOF    00  EOF    00  EOF    00  EOF    00  EOF    00  EOF  
; 

WCInitialState	= $				; Initial state table is here

WCState_Zer:
		WCST	WCState_EOF,0,0
		WCST	WCState_CR1,0,1
		WCST	WCState_LF1,0,1
		WCST	WCState_Txt,1,0
		WCST	WCState_Wht,0,0

WCState_Wht:
		WCST	WCState_EOF,0,1
		WCST	WCState_CR1,0,1
		WCST	WCState_LF1,0,1
		WCST	WCState_Txt,1,0
		WCST	WCState_Wht,0,0

WCState_Txt:
		WCST	WCState_EOF,0,1
		WCST	WCState_CR1,0,1
		WCST	WCState_LF1,0,1
		WCST	WCState_Txt,0,0
		WCST	WCState_Wht,0,0

WCState_CR1:
		WCST	WCState_EOF,0,0
		WCST	WCState_CR1,0,1
		WCST	WCState_Zer,0,0
		WCST	WCState_Txt,1,0
		WCST	WCState_Wht,0,0

WCState_LF1:
		WCST	WCState_EOF,0,0
		WCST	WCState_Zer,0,0
		WCST	WCState_LF1,0,1
		WCST	WCState_Txt,1,0
		WCST	WCState_Wht,0,0

WCState_EOF:
		WCST	WCState_EOF,0,0
		WCST	WCState_EOF,0,0
		WCST	WCState_EOF,0,0
		WCST	WCState_EOF,0,0
		WCST	WCState_EOF,0,0

A_CtrlZ		DB	1Ah		; Ctrl-Z EOF to ensure that WC finishes
DirSearchSpec	DB	"*.*",0		; Filespec to use for subdir recurse

; Transient data ---------------------------------------------------------------

; Note - the switch flags must appear in the same order as the characters at
; 'Switches' above; the parsing code will set the flag at the appropriate
; offset into SwitchFlags.  Options 'c', 's', and 'w' have sub-option chars
; and are handled via the SwitchSubs table earlier.

SwitchFlags	= $			; Start of switch flags
Flag9		DB	0		; Mark non-ISO9660-compliant names flag
AFlag		DB	0		; Include hidden/system files flag
CFlag		DB	0		; Do CRCs flag
DFlag		DB	0		; Display directory info flag
FFlag		DB	0		; Filename only flag
HFlag		DB	0		; Do Histograms flag
LFlag		DB	0		; Listing file flag
PFlag		DB	0		; Pause after each page flag
RFlag		DB	0		; Recurse subdirectory flag
SFlag		DB	0		; Display size in clusters flag
WFlag		DB	0		; Display word and line counts

		IF	($ - SwitchFlags) NE NumSwitches
		ERR	"Number of switch flags does not match number of switch letters"
		ENDIF

; General variables

Files		DW	0		; Counter for files processed
DateFPtr	DW	0		; Pointer to date format table
DateSep		DB	0		; Date separator char
InHandle	DW	0		; Input file handle (CRC, hist, wordcnt)
FindFunc	DB	0		; Findfirst/next function number
DTAPtr		DW	0		; DTA pointer for file being processed
Bad9660		DB	0		; Flag whether filename is non-ISO9660
Spaced		DB	0		; Flag used to control field spacing
LineCount	DB	23		; Reverse line counter, used for -P
AnySwitches	DB	0		; Count of switches processed
DriveTot	DW	0		; Total clusters on drive if -Sd
DriveAvl	DW	0		; Available clusters on drive if -Sd

		ALIGN	2

; Directory information (most displayed information is taken from the DTA)

TotBytesL	DW	0		; Total bytes in all files
TotBytesH	DW	0		; Hiword of same

; Cluster count variables

UserBPC		DW	0		; BPC specified (or zero for default)
BytesPerClust	DW	0		; Actual BPC to use
TotClustL	DW	0		; Loword of total clusters
TotClustH	DW	0		; Hiword of total clusters

; Checksum/CRC variables

CRCFunc		DW	0		; Pointer to checksum/CRC function or 0
Checksum	DW	0		; 16-bit checksum
CRC		DD	0		; 32-bit CRC

; Word and line counts

WCClass		DB	0,0		; Word counter char classification type
WC_State	DW	0		; Pointer to current state table
WordCntL	DW	0		; Word count - loword
WordCntH	DW	0		; Word count - hiword
LineCntL	DW	0		; Line count - loword
LineCntH	DW	0		; Line count - hiword
WordTotL	DW	0		; Word total - loword
WordTotH	DW	0		; Word total - hiword
LineTotL	DW	0		; Line total - loword
LineTotH	DW	0		; Line total - hiword

; Histogram variables

HistLimL	DW	0		; Limit for max-value scan, hist output
HistLimH	DW	0		; Hiword of same
HistMaxL	DW	0		; Maximum value in scan, hist output
HistMaxH	DW	0		; Hiword of same
FHistTotL	DW	0		; Total bytes in file histogram - loword
FHistTotH	DW	0		; Hiword of same
THistTotL	DW	0		; Total bytes in all files for histogram
THistTotH	DW	0		; Hiword of same

; Transient code ===============================================================

; The mainline code checks the DOS version and amount of memory, and relocates
; the stack, then sets up the country number.
; It then creates the CRC table, which is stored as four separate 256-byte
; tables (CRCTbl0-3), and decodes the character class table(s), and clears the
; combined histogram (histogram for all files combined).
; Next, the command tail is parsed.  Switches are checked in a loop, and must
; begin with '-' or the current DOS switchar.  Some switches are more than one
; character long, but each switch is always a known length; this allows two or
; more switches to be concatenated (e.g. -D -R becomes -DR) without ambiguity.
; Once the first non-switch is found, switch processing ceases and pathspec
; processing begins.
; Each pathspec on the command line is processed in the following way.	First
; it is expanded into a fully qualified drive/path specification, with '*.*'
; appended if appropriate.  This is performed by the QualPathspec subroutine.
; Then if the -S switch was given, BytesPerClust is set up: if UserBPC is zero
; (use default for drive where file resides), the drive information for the
; disk specified in Searchspec is requested and the number of bytes per cluster
; is stored to BytesPerClust.  Otherwise, UserBPC is copied to BytesPerClust.
; Then the filespec part of the pathspec is copied to Filespec for later use by
; other subroutines.
; Now, the Main3 function is called, with DX initially pointing to the first
; DTA block.  This function is recursive if the -R option was given, and will
; use the DTAs in ascending order as it recurses further and further through
; the directories.
; Each time Main3 finds a matching file, it calls the HandleFile subroutine,
; which is responsible for generating all the appropriate information about
; the file, according to the option switches provided.	It is also responsible
; for updating all relevant global variables that will be displayed in the
; totals message later.
; When Main3 completes, the whitespace that followed the parameter on the
; command tail is skipped, and if another parameter follows, the same process
; is performed for that parameter, and so on.
; Finally when the C/R at the command tail end is found, the DisplayTotals
; subroutine is called, and displays one or more messages giving totals for
; the requested pieces of information.	This routine will only display the
; totals if two or more files were processed.

PROC	Main2		near
		mov	ah,30h
		int	21h
		cmp	al,2		; Expect DOS 2.0 or later
		jae	DOS_Ok
		mov	dx,OFFSET DOSVersMsg
		mov	ah,9
		int	21h
		int	20h

BadUsage:	point	bx,UsageEM	; Incorrect usage

; Errexit code - aborts the program with a specified message and errorlevel.
; On entry, BX points to a control string within the current code segment,
; which consists of a one-byte errorlevel followed by a null-terminated error
; message which may be blank.  This code assumes DOS version 2.0 handle
; functions are supported.  After writing the error message to StdErr, DOS
; function 4Ch (terminate with return code) is used to terminate the program.

ErrExit:	push	[WORD cs:bx]	; Errorlevel onto stack
		inc	bx		; Point to error message
		call	ErrWriteMsg	; Display
		pop	ax		; Errorlevel
ErrExitQ:	mov	ah,4Ch
		int	21h		; Terminate with errorlevel in AL

ErrWriteMsg:	mov	cx,2		; Handle of Standard Error device
WriteMsg:	push	cs
		pop	ds		; Make DS valid
		mov	dx,bx		; Point DX for later
Err_Parse1:	inc	bx		; Bump pointer
		cmp	[BYTE ds:bx-1],0 ; Hit null terminator yet?
		jnz	Err_Parse1	; If not, loop
		xchg	cx,bx		; Get address of null terminator
		sub	cx,dx		; Subtract offset of start of text
		dec	cx		; Adjust to correct number of chars
		jz	Err_Parse2	; If no text present
		mov	ah,40h		; Function to write to file or device
		int	21h		; Write error message to StdErr
Err_Parse2:	ret			; Return to exit code or whoever

; Check amount of available memory ---------------------------------------------

DOS_Ok:		mov	ax,cs		; Get current segment
		DB	5		; ADD AX,nnnn (avoid TASM warning!)
		DW	MemParas	; Get paragraph past end of program
		cmp	ax,[WORD ds:2]	; Check against paragraph of mem top
		mov	bx,OFFSET MemoryEM ; Prepare for error
		jae	ErrExit		; If not enough memory
		point	sp,WorkStackTop	; Relocate stack to safe area

; Get country-specific information ---------------------------------------------

		point	dx,DTAs		; Point at area to store country info
		mov	ax,3800h
		int	21h		; Request current country-dependent info
		mov	bx,[WORD DTAs]	; Get date format code
		cmp	bx,3		; Ensure within limits
		jb	CountryOK	; If so
		xor	bx,bx		; If not, use American format
CountryOK:	mov	al,[DateSeps+bx] ; Get date separator character
		mov	[DateSep],al	; Store
		shl	bx,1		; Convert to table index
		mov	ax,[DateHTbl+bx] ; Get pointer to date format table
		mov	[DateFPtr],ax	; Store pointer to date format table
		mov	ah,30h
		int	21h		; Get DOS version
		cmp	al,3
		jb	NoDateSep	; If not 3.0 or later
		mov	al,[DTAs+11]	; If 3.0 or later, get separator
		mov	[DateSep],al	; Store it

; Generate CRC table -----------------------------------------------------------

NoDateSep:	mov	di,0FFh		; Entry number
NextEntry:	xor	dx,dx		; Zero hiword of CRC
		mov	ax,di		; Get entry number to loword of CRC
		mov	cx,8		; Bit counter
ShiftLoop:	shr	dx,1		; Shift CRC right one
		rcr	ax,1		; Both words
		jnc	NextShift	; If bit not set, no XORs
		xor	dx,1110110110111000b ; XOR CRC with reversed polynomial
		xor	ax,1000001100100000b ; Both words
NextShift:	loop	ShiftLoop	; Loop for all eight bits
		mov	[CRCTbl0+di],al	; Store byte 0
		mov	[CRCTbl1+di],ah	; Store byte 1
		mov	[CRCTbl2+di],dl	; Store byte 2
		mov	[CRCTbl3+di],dh	; Store byte 3
		dec	di		; Next entry
		jns	NextEntry	; Loop for all entries

; Generate character class table(s) --------------------------------------------

		point	si,CharClassRLE	; Point to encoded data to expand
		point	di,CharClass0	; Point to first translation table
ExpandCCTbl:	lodsb			; Get count
		clr	ah		; Zero hibyte
		xchg	ax,cx		; To CX
		jcxz	ExpandCCDone	; If finished
		lodsb			; Get value
		rep	stosb		; Store to array
		jmp	SHORT ExpandCCTbl ; Continue
ExpandCCDone:	cmp	di,OFFSET CharClass0 + (CharClasses * 256) ; Filled tbl?
		je	ExpandCCOK	; If so, alright
		point	bx,IntErrE	; Internal error!
		jmp	ErrExit		; Go to error exit stuff

; Zero combined file histogram -------------------------------------------------

ExpandCCOK:	clr	ax		; Zero
		point	di,THistTbl	; Point at histogram table
		mov	cx,512		; Number of words to clear
		rep	stosw		; Zero them

; Command tail parsing ---------------------------------------------------------

; Support both the hyphen ('-') and the currently active DOS Switchar character
; to indicate an option switch.  The switches are processed first, and must have
; been typed first on the command line.  Once a non-switch is found, the switch
; processing is completed, and any switches that appear later will be regarded
; as pathspecs.

		cld
		mov	si,81h		; Prepare for command-line parsing
		mov	ax,3700h
		int	21h		; Get switchar to DL
Parse:		lodsb			; Next character
		cje	al,13,TestNoFiles ; Check for end of command tail
		cmp	al," "		; Check for space or other whitespace
		jbe	Parse		; Skip it
		cje	al,"-",Switch	; Check for '-' explicitly
		cmp	al,dl
		jne	FirstFile	; Check for switchar too

; Found a switch

Switch:		inc	[AnySwitches]	; Increment count of switches found
		lodsb			; Get char following '-' or switchar
		call	ConvertUC	; Convert to upper case
		point	di,Switches	; Point to option switch table
		mov	cx,NumSwitches	; Get number of switches to scan
		repne	scasb		; Look for switch
		jne	GoBadUsage1	; If not found
		sub	di,OFFSET Switches + 1 ; Convert into switch number
		mov	[BYTE SwitchFlags+di],1 ; Set appropriate switch flag
		shl	di,1		; Shift for word sized pointers
		call	[SwitchSubs+di]	; Call subswitch handler, if any
		cmp	[BYTE si]," "	; Are options run together?
		ja	Switch		; If so, process next switch
		jmp	SHORT Parse	; Loop

TestNoFiles:	point	si,StarDotStar+1 ; Prepare to default to '*.*'
		cmp	[AnySwitches],0	; Get any switches?
		jnz	FirstFile	; If so, use default of '*.*'

GoBadUsage1:	jmp	BadUsage	; Syntax error

; Pathspec found on command tail -----------------------------------------------

FirstFile:

; Report cluster information if -Sd used ---------------------------------------

; The cluster size, total clusters, and free clusters are reported.

		cmp	[DrivePatch],"?" ; Display a message?
		je	File		; If not

		push	si
		point	di,OutLine	; Point to area to store text
		point	si,ClustMsg1	; First text message
		call	CopyStr		; Copy string
		mov	ax,[UserBPC]	; Number of bytes per cluster
		call	ConvDec16NP	; Convert to decimal
		point	si,ClustMsg2	; Second text message
		call	CopyStr		; Copy string
		mov	ax,[DriveTot]	; Get total clusters
		call	ConvDec16NP	; Convert to decimal
		point	si,ClustMsg3	; Third text message
		call	CopyStr		; Copy string
		mov	ax,[DriveAvl]	; Get number of available clusters
		call	ConvDec16NP	; Convert to decimal
		point	si,ClustMsg4	; Fourth text message
		call	CopyStr		; Copy string
		call	WriteOutLine	; Output the line
		pop	si

File:		dec	si		; Point to start of pathspec found
		point	di,SearchSpec	; Point to destination area

		call	QualPathspec	; Produce a fully qualified pathspec

		IF	TestQual
		 push	si
		 point	si,SearchSpec	; String
		 point	di,OutLine
		 call	CopyStr
		 call	WriteOutLine	; Display it
		 pop	si
		 jmp	SHORT SkipWhite	; Next parameter
		ENDIF

; We now have a full pathspec, in the form 'd:\dir\dir\filespec' at Searchspec.
; If -S switch given, set up BytesPerClust - either from UserBPC, or from drive
; information for the drive specified for this file.

		cmp	[SFlag],0	; S flag?
		jz	NoSFlag1	; If not
		mov	ax,[UserBPC]	; Get user-specified BPC
		test	ax,ax		; User-specified or default?
		jnz	GotBPC		; If user-specified
		mov	al,[SearchSpec]	; Get drive letter
		call	GetDriveInfo	; Get drive info
GotBPC:		mov	[BytesPerClust],ax ; Store cluster size for this pass

; Take a copy of the final portion of Searchspec, i.e. the file specification,
; to Filespec.

NoSFlag1:	push	si		; Preserve pointer into command tail
		mov	si,di		; Point to past end of specification
FindFilespec:	dec	si		; Back up
		cmp	[BYTE si-1],"\"	; Hit final backslash yet?
		jne	FindFilespec	; If not, keep looking
		push	si		; Keep pointer
		point	di,Filespec	; Point to area to store file spec into
		call	CopyStr		; Copy string including null terminator
		pop	di		; Restore 'append pointer'
		point	dx,DTAs		; Point to start of DTA area

; We now have a fully qualified pathspec at Searchspec and a copy of the
; filespec portion in Filespec.  Now call Main3, which is an optionally
; recursive function, to find and process each file matching the filespec.
; If the -R (recurse) option was specified, then Main3 will recurse through
; all the subdirectories.  This is why Filespec is required - as Main3
; recurses through the subdirectories, it tacks each new directory name and
; name of each file it finds, onto the string at Searchspec, and it uses
; Filespec to reinstate the file mask after appending each new directory or
; file name to Searchspec.
;
; During operation of Main3, the following registers contain recurse
; nesting related information:
;
; DX points to the current DTA
; DI points to the 'append' point in Pathspec; i.e. where the new
;	directory (if found) will be appended.

		call	Main3		; Do processing

		pop	si		; Restore command tail pointer
SkipWhite:	lodsb			; Next char
		cje	al,13,DoneFiles	; If C/R, have end of command tail
		cmp	al," "		; Whitespace?
		jbe	SkipWhite	; If so, skip it
		jmp	File		; Handle next filespec

; End of file processing - display totals messages and terminate

DoneFiles:	call	DisplayTotals
		mov	ax,4C00h	; Errorlevel zero
		int	21h		; Terminate with errorlevel
ENDP	Main2

; Switch sub-option handlers ---------------------------------------------------

; The -Cxx option specifies CRC or checksum calculation on the file.
; There are five forms:
;
; -CRC		CRC (32-bit, ANSI X3.66)
; -CBS		Checksum, byte-wise summed
; -CWS		Checksum, word-wise summed
; -CBX		Checksum, byte-wise XORed
; -CWX		Checksum, word-wise XORed

PROC	SwSub_C		near
		lodsw			; Get two chars following the '-C'
		and	ax,0101111101011111b ; Convert both chars to upper case
		mov	cx,5		; Number of sub-options
		point	di,CRCTypes	; Point at start
		repne	scasw		; Look for a match
		jne	GoBadUsage2	; If failed
		mov	ax,[di+CRCFuncs-CRCTypes-2] ; Get function pointer
		mov	[CRCFunc],ax	; Store pointer to CRC function
		ret			; Continue parsing
ENDP	SwSub_C

GoBadUsage2:	jmp	BadUsage	; If error

; The -S option specifies that the size of the file, in clusters, should be
; calculated.  There are nine forms:
;
; -S*		Use size of clusters on drive where the file resides
; -Sd		Use size of clusters on disk in drive d:
; -S5		Cluster size 512 bytes
; -S1		Cluster size 1K
; -S2		Cluster size 2K
; -S4		Cluster size 4K
; -S8		Cluster size 8K
; -S6		Cluster size 16K
; -S3		Cluster size 32K
;
; If the second form is used, the DrivePatch character is patched from '?' to
; the drive letter.  This is detected by the mainline, which displays an
; informative message giving the characteristics of the disk in drive d.

PROC	SwSub_S		near
		lodsb			; Get character following -S
		cje	al,"*",SwSub_None ; If default
		push	ax		; Keep the character for a second
		sub	al,"1"		; Convert "1"-"8" to 0-7
		cmp	al,8		; Check within range
		cmc			; Carry will be set if outside range
		jc	Not1_8		; If not within range
		point	bx,NumToBPCShift ; Point to table
		xlat			; Get number of shifts
		xchg	ax,cx		; To CL
		mov	dx,1		; Initial '1'
		shl	dx,cl		; Shift it

; At this point, carry will be set if the character following the '-S' was
; not 1, 2, 4, 5, 8, 6 or 3.  If carry is clear, DX contains the user BPC
; (bytes per cluster) value.

Not1_8:		pop	ax		; Restore character
		mov	[UserBPC],dx	; Store possible bytes per cluster
		jnc	SwSub_None	; If was 1, 2, 4, 5, or 8

; Did not match 1, 2, 4, 5, 8, 6 or 3.	Check for a drive letter.

		call	ConvertUC	; Convert to upper case
		cmp	al,"A"
		jb	GoBadUsage2	; If not a letter
		cmp	al,"Z"
		ja	GoBadUsage2	; Ditto

; The '-S' option subtype was a drive letter

		mov	[DrivePatch],al	; Set up character in informative msg
		call	GetDriveInfo	; Get drive information
		mov	[UserBPC],ax	; Keep bytes per sector value
		mov	[DriveTot],bp	; Store total clusters on drive
		mov	[DriveAvl],bx	; Store number of clusters available
SwSub_None:	ret			; Continue parsing
ENDP	SwSub_S

PROC	SwSub_W		near
		lodsb			; Get character following -W
		sub	al,"1"		; Convert "1"-"9" to 0-8
		cmp	al,CharClasses	; Check within range
		jae	GoBadUsage2	; If not
		mov	[WCClass],al	; Store class number
		ret
ENDP	SwSub_W

; Qualify Pathspec -------------------------------------------------------------

; See documentation contained in QUALPATH.ASM
;
; The behaviour of this function can be checked if the TestQual conditional
; is set to 1.	This causes the program to simply display the qualified path
; specification resulting from QualPathspec instead of performing the file
; finding and processing functions.

		INCLUDE	"QUALPATH.ASM"

; Main file processing ---------------------------------------------------------

; The Main3 function processes a single path specification.  If the -R flag
; is set, it recurses subdirectories, calling itself recursively.  Each time
; it is called, four bytes of stack space are required - two for the append
; pointer which is pushed onto the stack during the recursive call, and two
; for the return address.  Note that the stack allocation at the end of the
; program includes four bytes per nesting level for the maximum nesting depth.

PROC	Main3		near		; Possibly recursive subroutine
;				Func:	Perform desired action on specified
;					files, recurse subdirectories if desired
;				In:	[DS:DX] = Working DTA
;					[DS:DI] = Filespec append point
;					[RFlag] = Flag whether to recurse or not
;					Filespec contains text of file spec part
;				Out:	None
;				Lost:	AX BX CX SI BP
		mov	ah,1Ah
		int	21h		; Set Disk Transfer Area to current ptr
		mov	[FindFunc],4Eh	; Set up find-first/next for first
NextFile:	point	si,Filespec	; Point to filespec to be searched for
		call	CopyStrR	; Append filespec to current path
		cmp	[AFlag],1	; Set carry if flag is clear
		cmc			; Set carry if flag is set
		sbb	cl,cl		; Zero if flag was clear, 0FFh if set
		and	cx,0000000000000110b ; Allow hidden/system if flag set
		call	FindFirstNext	; Find a matching file
		jc	NoneFound	; If no more files
		mov	si,dx		; Point SI back to DTA
		add	si,DTA_FName	; Point to filename found
		push	si		; Preserve pointer
		call	CopyStrR	; Append filename onto end of path
		pop	si		; Restore pointer
		mov	[Bad9660],0	; Prepare for no ISO9660 violations
		cmp	[Flag9],0	; Should we check it?
		jz	NoCheck9660	; If not
Check9660:	lodsb			; Get char
		test	al,al		; Done?
		jz	NoCheck9660	; If so
		cmp	al,"_"		; Underline is allowed
		je	Check9660	; If so, try next
		cmp	al,"."		; Dot (one of them!) is allowed
		je	Check9660	; If so, try next
		cmp	al,"0"		; Check for numeric
		jb	Error9660	; If too low
		cmp	al,"9"		; Check for numeric
		jbe	Check9660	; If alright, check next
		cmp	al,"A"		; Check for alphabetic
		jb	Error9660	; If too low
		cmp	al,"Z"		; Check for alphabetic
		jbe	Check9660	; If alright, check next
Error9660:	mov	[Bad9660],1	; Flag error
NoCheck9660:	call	HandleFile	; Do specified actions on this file
		jmp	SHORT NextFile	; Loop

NoneFound:	cmp	[RFlag],0	; Should we recurse subdirectories?
		jz	NoRecurse	; If not
		mov	[FindFunc],4Eh	; Set up find-first/next for first
NextDir:	mov	ah,1Ah
		int	21h		; Reset Disk Transfer Area pointer
		point	si,DirSearchSpec ; Now try to find subdirectories
		call	CopyStrR	; Append '*.*' to working pathname
		mov	cl,00010000b	; Attribute for subdirectory
FindSubDirs:	call	FindFirstNext	; Try to find a subdirectory
		jc	NoRecurse	; If none, finished this recursion
		test	[BYTE si+DTA_Attrib],00010000b ; Was it a subdirectory?
		jz	FindSubDirs	; If not, keep looking
		push	di		; Found a subdirectory - recurse into it
		add	si,DTA_FName	; Point to start of subdirectory name
		call	CopyStr		; Append subdir name to working path
		mov	[WORD di],"\"	; Add trailing backslash and null
		inc	di		; Point to append point
		add	dx,DTAAlloc	; Allocate a new DTA above old one
		call	Main3		; Recurse to find files in this one
		sub	dx,DTAAlloc	; Point back to our DTA
		pop	di		; Restore pointer to path append point
		jmp	SHORT NextDir	; Reset DTA and continue search
ENDP	Main3

; Find first / find next -------------------------------------------------------

; This function is used by Main3 to find the first or next file matching the
; search specification at SearchSpec.  Before calling, FindFunc must be set
; to 4Eh (find-first).	This function sets FindFunc to 4Fh (find-next) after
; the initial call.  On calling, CX must contain the attribute mask to be
; passed to the DOS function(s).
; DOS reports no error if there are no more files to be found; this is just
; indicated by a return code of 18.  This function returns carry set if there
; were no files, a file error, or no more files to be found, otherwise it
; returns carry clear, indicating a file has been found.  This function also
; filters out the 'fules' ("." and "..") that appear at the start of every
; subdirectory; if a found file begins with '.' then another search is made.
; The caller is responsible for allocating and specifying (via function 1Ah)
; the DTA to be used.

PROC	FindFirstNext	near
		push	dx
FindAnother:	point	dx,SearchSpec
		mov	ah,[FindFunc]	; Function number
		mov	[FindFunc],4Fh	; Set up function for find-next
		int	21h		; Find first/next
		jc	PopDXRet	; If error, don't test for fake dirs
		cje	ax,18,CMCPopDXRet ; If error code 18, carry may be clear
		cmp	[BYTE si+DTA_FName],"." ; If got a name, ignore if name is
		je	FindAnother	;   a fake directory name '.' or '..'
		stc			; If got a non-fake name, flag no error
CMCPopDXRet:	cmc
PopDXRet:	pop	dx
NoRecurse:	ret
ENDP	FindFirstNext

; Process a found file ---------------------------------------------------------

; This function is called by Main3 for every matching file that is found.
; When called, the full pathname of the file is at Searchspec, and the
; directory information is in the current DTA, which is pointed to by DX
; on entry.
;
; This function is responsible for performing all necessary actions specified
; by the command line switches, generating one line of text (or possibly an
; entire histogram dump) for the file, and updating total values.
;
; If the -Cxx, -H, or -Wn options were specified, the file must be opened
; and read into the deblocking buffer, so that the CRC/checksum, histogram,
; and/or word-count functions can operate on it.  This is handled by the
; ExamineFile function, which is called by this function if one or more of
; these switches are set.
;
; After examining the file, if necessary, this function calls the Output_xxx
; functions to build up an output line containing the following information,
; in this order, each item only being generated if the appropriate flag is set:
;
; CRC or checksum			(-Cxx)
; Cluster count				(-Sc)
; Size, date, and time			(-D)
; File attributes			(-A)
; Word and line count			(-Wn)
; File name (fully qualified, or filename only if -F option given)
; The text "  (non-ISO9660)" if the filename is not ISO9660 compatible   (-9)
;
; The above items are all placed on a single line.  If -H (histograms) was
; specified, the multiple-line histogram dump follows this line.
;
; As each piece of information is stored into the output line, the appropriate
; totals values are updated.  Totals that have not been requested (i.e. the
; appropriate switches were not set) are not updated.
;
; If the -L option was specified, instead of the variable-format output line
; being generated, a simple narrow list-type line is generated, and the other
; option switches are ignored.	In this case, total bytes and count of files
; will also be displayed.  If requested, filenames will be checked for ISO9660
; compliance and the text "  (non-ISO9660)" will be appended to the line if the
; name does not comply with ISO9660 file name requirements.

PROC	HandleFile	near
;				Func:	Handle one file, generate output line
;				In:	SearchSpec = file name
;					DS:DX -> DTA containing file information
;				Out:	None
;				Lost:	AX BX CX DX SI DI
;				Note:	Must not destroy DX or DI
		push	di
		push	dx		; Preserve these regs
		mov	[DTAPtr],dx	; Store DTA pointer for use by handlers
		inc	[Files]		; Increment counter of files handled

; Check for, and handle, listfile format output first.

		cmp	[LFlag],0	; Check for listing format
		jnz	IsListFormat	; If so
		jmp	NotListFormat	; If not
IsListFormat:	point	di,SearchSpec	; Filename
		xor	al,al		; Null-terminator
		mov	cx,70		; Max. length to scan
		repne	scasb		; Scan for terminator
		dec	di		; Back up to terminator
		std			; Backwards now
		mov	cx,70		; Max. length again
		mov	al,"\"		; Scan for last '\'
		repne	scasb		; Scan for it
		cld			; Back to normal string direction!
		lea	si,[di+2]	; Point to final part of pathname
		point	di,OutLine-1	; Just before start of output line
CopyListF1:	stosb			; Store char into line
		lodsb			; Get char from filename
		test	al,al		; Hit null-terminator?
		jz	PadFull		; If so
		cmp	al,"."		; Hit the dot?
		jne	CopyListF1	; If not, loop
		DB	3Ch		; Skip next 1-byte instruction
PadFirstPart:	stosb			; Store character
		mov	al," "		; Space padding
		cmp	di,OFFSET OutLine+8 ; Done primary part of name?
		jb	PadFirstPart	; If not, loop
		mov	al,"."		; Put the dot at the eighth column
CopyListF2:	stosb			; Stick it in there
		lodsb			; Get char from extension
		test	al,al		; Hit null-terminator?
		jnz	CopyListF2	; Loop if not
		DB	3Ch		; Skip next 1-byte instruction
PadSecondPart:	stosb			; Store character
PadFull:	mov	al," "		; Space padding
		cmp	di,OFFSET OutLine+12 ; Done full filename?
		jb	PadSecondPart	; If not, loop
		stosb			; One space after filename

; Date

		mov	si,[DTAPtr]	; Point to DTA
		mov	ax,[si+DTA_FSize+0] ; Loword of file size
		mov	dx,[si+DTA_FSize+2] ; Hiword of file size
		add	[TotBytesL],ax	; Add into total accumulator loword
		adc	[TotBytesH],dx	; Hiword too, with carry also
		mov	cx," "*256+8	; Eight digits, space padding
		call	Mach32_DecASC	; Convert to decimal
		mov	al," "		; Space
		stosb			; Store to output
		mov	si,[DTAPtr]	; Point to DTA
		mov	ax,[si+DTA_Date] ; Get date word
		push	ax		; Keep date word on stack
		call	DateYear	; Output the year
		pop	ax		; Restore date word
		push	ax		; Back onto stack
		call	DateMonth	; Output the month
		pop	ax		; Restore date word
		call	DateDay		; Output the day
		mov	al," "		; Single space
		stosb			; Store it

; Time

		mov	ax,[si+DTA_Time] ; Get time word
		push	ax		; Onto stack for later
		mov	cl,11		; Shift count
		shr	ax,cl		; Shift it
		and	ax,00011111b	; Limit to 31 (0-23)
		call	ConvertDec2	; Convert to decimal
		pop	ax		; Restore time word
		mov	cl,5		; Shift count
		shr	ax,cl		; Shift it
		and	ax,00111111b	; Limit to 63 (0-59)
		call	ConvertDec2	; Convert to decimal
		mov	al," "		; Single space
		stosb			; Store it
		call	Output_9660	; Add "  (non-ISO9660)" if appropriate
		call	WriteOutLine	; Write output line
		jmp	NoHist		; Don't do any other special output

; Perform checksum/CRC, histogram, and/or word count if options specified

NotListFormat:	mov	al,[CFlag]	; If checksum/CRC flag, or
		or	al,[HFlag]	; Histogram flag, or
		or	al,[WFlag]	; Word count flag...
		jz	NoExamine	; If not
		call	ExamineFile	; If so

; Generate output line according to options requested

NoExamine:	mov	[Spaced],0	; Flag not spaced yet
		point	di,OutLine	; Point to output line text
		cmp	[CFlag],0
		jz	NoCRC		; If not doing CRCs
		call	Output_CksumCRC	; Store file checksum/CRC into OutLine
NoCRC:		cmp	[SFlag],0
		jz	NoClust		; If not doing clusters calculation
		call	Output_Clust	; Generate cluster information
NoClust:	cmp	[DFlag],0
		jz	NoDir		; If not doing directory information
		call	Output_Dir	; Generate directory information
NoDir:		cmp	[AFlag],0
		jz	NoAttr		; If not doing attribute display
		call	Output_Attr	; Generate attribute information
NoAttr:		cmp	[WFlag],0
		jz	NoWordcount	; If not doing word and line counts
		call	Output_WC	; Generate word and line count info
NoWordcount:	call	Spacer		; Do spacer

; Display file name here - either the fully qualified name at SearchSpec, or
; if -F given, just the last part (excluding the drive and path).

		point	si,SearchSpec	; Point to full pathname
		cmp	[FFlag],0	; Use full pathname?
		jz	GotFNamePtr	; If so
Scan1:		lodsb			; Next char
		test	al,al		; Null-terminator?
		jnz	Scan1		; Loop
Scan2:		dec	si		; Back up
		cmp	[BYTE si-1],"\"	; Final backslash?
		jne	Scan2		; If not, keep looking
GotFNamePtr:	call	CopyStr		; Add it to the line
		call	Output_9660	; Add "  (non-ISO9660)" if appropriate
		call	WriteOutLine	; Write text at OutLine
		cmp	[HFlag],0
		jz	NoHist		; If no histogram output requested

; Announce total bytes in histogram

		call	WriteNewline	; Blank line
		point	di,OutLine	; Output line
		mov	ax,[FHistTotL]	; Get histogram total loword
		mov	dx,[FHistTotH]	; Get histogram total hiword
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal ASCII
		point	si,TotHistMsg	; " bytes total in file histogram"
		call	CopyStr		; Store to line
		call	WriteOutLine	; Write output line
		call	WriteNewline	; Blank line
		point	bp,FHistTbl	; Point to local (per-file) histogram
		call	ShowHist	; Display histogram
NoHist:		pop	dx
		pop	di		; Restore regs
		ret			; Return to Main3
ENDP	HandleFile

GoErrExit:	jmp	ErrExit		; Go to error exit stuff

; File contents ----------------------------------------------------------------

; The ExamineFile function is called by HandleFile for every file found, if
; one or more of the -Cxx, -H, or -Wn options were provided.  It reads the
; file contents into InBuffer in chunks, calling the appropriate function(s)
; for each chunk, according to the switches from the command line.
; Once processing is completed, the global histogram is updated from the
; local (per-file) histogram, and the totals for words and lines are updated.
;
; Processing functions Pass_Hist and Pass_WordCount and one of five checksum/
; CRC pass functions are called.
;
; Note that some actions (e.g. initialisation of tables and values) are
; performed regardless of whether the appropriate option is enabled; it is
; simpler to just do them, rather than checking option flags all the time.

; Exit point if a read error occurs

ReadError:	mov	ah,3Eh
		int	21h		; Close the file
		point	bx,ReadEM	; Error reading file

PROC	ExamineFile	near
;				Func:	Read file contents and calculate the
;					CRC/checksum, histogram, and/or word
;					and line count.  (Does not display
;					the results of these calculations).
;				In:	SearchSpec = Pathname of file
;					[CRCFunc] points to CRC update function
;					or zero if no CRC requested
;				Out:	Sets FHistTbl table, CRC, WordCntL,H,
;					LineCntL,H variables as appropriate
;				Lost:	AX BX CX DX
		point	dx,SearchSpec	; Point to file name found
		mov	ax,3D00h	; Open file for read
		int	21h		; Do it
		point	bx,CantOpenEM	; Prepare for error
		jc	GoErrExit	; If error opening file
		mov	[InHandle],ax	; Store file handle

; Initialise word and line counts, histogram, checksum, and CRC

		clr	ax		; Get zero
		mov	[WordCntL],ax	; Zero loword
		mov	[WordCntH],ax	; Zero hiword
		mov	[LineCntL],ax	; Zero loword
		mov	[LineCntH],ax	; Zero hiword

		mov	[WC_State],OFFSET WCInitialState ; Init starting state

		point	di,FHistTbl	; Point at file histogram table
		mov	cx,512		; Number of words to clear
		rep	stosw		; Zero them
		mov	[FHistTotL],ax	; Zero total byte count too
		mov	[FHistTotH],ax	; Both words

		mov	[Checksum],ax	; Set checksum to zero
		dec	ax		; Precondition value is 0FFFFFFFFh
		mov	[WORD CRC+0],ax
		mov	[WORD CRC+2],ax	; Precondition CRC to -1

; Read a chunk of data from the file

ReadBlock:	mov	bx,[InHandle]	; Get handle
		mov	cx,InBufSize	; Max size of block to read
		point	dx,InBuffer	; Location to read into
		mov	ah,3Fh		; Read block of data
		int	21h
		jc	ReadError	; If error
		xchg	ax,cx		; Count to CX for update
		jcxz	DoneUpdate	; If no bytes read

; Call appropriate checksum or CRC routine

		cmp	[CFlag],0	; CRC or checksum requested?
		jz	DoneCRC		; If not
		push	cx		; Keep count
		point	si,InBuffer	; Point to buffer
		call	[CRCFunc]	; Do checksum or CRC on this block
		pop	cx		; Restore count

; Update histogram

DoneCRC:	cmp	[HFlag],0	; Doing histograms at all?
		jz	NoCalcHist	; If not
		push	cx		; Keep count
		point	si,InBuffer	; Point to buffer
		call	Pass_Hist	; Do a histogram pass
		pop	cx		; Restore count

; Do word and line count

NoCalcHist:	cmp	[WFlag],0	; Do word and line count?
		jz	DoneUpdate	; If not
		push	cx		; Keep count
		point	si,InBuffer	; Point to buffer
		call	Pass_WordCount	; If so
		pop	cx		; Restore count

DoneUpdate:	cmp	cx,InBufSize	; Did we read a full buffer?
		je	ReadBlock	; If so, loop for more
		mov	bx,[InHandle]	; Get handle
		mov	ah,3Eh
		int	21h		; Close the file

; If word counting, process an extra Ctrl-Z, in case there was no EOF character
; in the file.	This applies only to word counting; the dummy EOF character is
; not included in any CRC, checksum, or histogram calculation.	It ensures that
; the end of the file is regarded as whitespace again.

		cmp	[WFlag],0	; Doing word counts?
		jz	NoWC_CtrlZ	; If not
		point	si,A_CtrlZ	; Point to a Ctrl-Z
		mov	cx,1
		call	Pass_WordCount	; Process the Ctrl-Z

; Complement the CRC

NoWC_CtrlZ:	not	[WORD CRC+0]	; Complement loword
		not	[WORD CRC+2]	; Complement hiword

; Update global histogram from histogram info from this file

		point	si,FHistTbl	; Point to local histogram
		mov	cx,256		; Number of entries to update
HistAddLoop:	lodsw			; Get loword from local histogram
		add	[WORD si+THistTbl-FHistTbl-2],ax ; Add lowords
		lodsw			; Get hiword from local histogram
		adc	[WORD si+THistTbl-FHistTbl-2],ax ; Add hiwords
		loop	HistAddLoop

		mov	ax,[FHistTotL]	; Get total bytes - loword
		mov	dx,[FHistTotH]	; Get hiword
		add	[THistTotL],ax	; Add into combined total
		adc	[THistTotH],dx	; Add hiwords too

; Update total word and line counters

		mov	ax,[WordCntL]	; Word count loword
		mov	dx,[WordCntH]	; Hiword
		add	[WordTotL],ax	; Update total
		adc	[WordTotH],dx	; Hiword

		mov	ax,[LineCntL]	; Line count loword
		mov	dx,[LineCntH]	; Hiword
		add	[LineTotL],ax	; Update total
		adc	[LineTotH],dx	; Hiword

		ret			; Done

ENDP	ExamineFile

; File pass functions ----------------------------------------------------------

; The following functions performs one update pass on the data in InBuffer.
; When called, SI points to the data (InBuffer) and CX contains the count of
; data bytes to be processed, which is guaranteed to be non-zero.
; The functions are allowed to destroy all registers.
;
; The count in CX is also guaranteed to be even, except on the final pass of
; the file.  This is unimportant for most functions, but is relevant to the
; word-wise checksum functions.
;
; The first set of functions are five CRC/checksum functions, which are called
; through the CRCFunc vector by ExamineFile.

PROC	Pass_CRC32	near
		mov	bx,[WORD CRC+0]	; Load CRC into working registers
		mov	dx,[WORD CRC+2]
		clr	ax		; Ensure AH remains zero during loop

UpdateLoop:	lodsb			; Get byte
		xor	al,bl		; XOR byte with lobyte of CRC
		mov	di,ax		; Point to table entry
		mov	bl,bh
		mov	bh,dl		;	CRC = CRCTable[(CRC ^ Byte)
		mov	dl,dh		;		& 0xFF] ^ (CRC >> 8);
		xor	bl,[CRCTbl0+di]
		xor	bh,[CRCTbl1+di]
		xor	dl,[CRCTbl2+di]
		mov	dh,[CRCTbl3+di]
		loop	UpdateLoop	; Loop for all data bytes

		mov	[WORD CRC+0],bx
		mov	[WORD CRC+2],dx	; Store updated CRC
		ret
ENDP	Pass_CRC32

PROC	Pass_Chk_BS	near
		mov	dx,[Checksum]	; Get checksum
		clr	ax		; Ensure AH remains zero during loop
ChkBSLoop:	lodsb			; Byte from file
		add	dx,ax		; Accumulate
		loop	ChkBSLoop	; Loop
		mov	[Checksum],dx	; Store back
		ret
ENDP	Pass_Chk_BS

PROC	Pass_Chk_WS	near
		mov	dx,[Checksum]	; Get checksum
		shr	cx,1		; Calculate number of words in block
		pushf			; Keep carry, if any
ChkWSLoop:	lodsw			; Word from file
		add	dx,ax		; Accumulate
		loop	ChkWSLoop	; Loop
		popf			; Check for odd byte
		jnc	NoOdd_WS	; If not
		lodsb			; Get the odd byte
		clr	ah		; Zero hibyte
		add	dx,ax		; Accumulate
NoOdd_WS:	mov	[Checksum],dx	; Store back
		ret
ENDP	Pass_Chk_WS

PROC	Pass_Chk_BX	near
		mov	ah,[BYTE Checksum] ; Get checksum
ChkBXLoop:	lodsb			; Byte from file
		xor	ah,al		; XOR into checksum
		loop	ChkBXLoop	; Loop
		mov	[BYTE Checksum],ah ; Store back
		ret
ENDP	Pass_Chk_BX

PROC	Pass_Chk_WX	near
		mov	dx,[Checksum]	; Get checksum
		shr	cx,1		; Calculate number of words in block
		pushf			; Keep carry, if any
ChkWXLoop:	lodsw			; Word from file
		xor	dx,ax		; XOR into checksum
		loop	ChkWXLoop	; Loop
		popf			; Check for odd byte
		jnc	NoOdd_WX	; If not
		lodsb			; Get the odd byte
		clr	ah		; Zero hibyte
		xor	dx,ax		; XOR into checksum
NoOdd_WX:	mov	[Checksum],dx	; Store back
		ret
ENDP	Pass_Chk_WX

; This function updates the local (per-file) histogram table.

PROC	Pass_Hist	near
		add	[FHistTotL],cx	; Add into histogram total bytes
		adc	[FHistTotH],0	; Carry into hiword
		clr	ax		; AH remains zero during loop
HistUpdLoop:	lodsb			; Get byte
		mov	di,ax		; To DI
		shl	di,1		; Shift for word
		shl	di,1		; Shift for dword
		inc	[FHistTbl+di]	; Increment loword
		jz	HistCarried	; If wrapped around
HistLoopCont:	loop	HistUpdLoop	; Do for all
		ret
HistCarried:	inc	[FHistTbl+2+di]	; Carry into hiword
		jmp	SHORT HistLoopCont ; Continue
ENDP	Pass_Hist

; Word counting is implemented through character classification tables and
; state tables.  Each character is read from the file and converted into a
; character class, using the desired character class translation table.  At
; present there are two classification tables.	They start at CharClass0.
;
; More classification tables may be added if desired.  The character classes
; are listed near to the definition of the encoded classification table data
; at CharClassRLE.
;
; Each state table has an entry for each possible character class.  Each entry
; specifies which state to go to, and also whether to increment the counters
; for words and/or lines upon switching to that state, if a character of that
; class is read from the file.
;
; Each entry in each state table is three bytes long.  The first two bytes
; are a pointer to base of the new state table.  The third byte is allocated
; as follows:
;
; 7 6 5 4 3 2 1 0
; * * * * * * . .  Not used
; . . . . . . * .  Increment Word Count flag
; . . . . . . . *  Increment Line Count flag
;
; See the WCST macro and state table around 'InitialState' for details.
;
; During processing:
;
; BX -> Classification table
; CX = Character counter
; SI -> Characters from file buffer
; DI = State table pointer
;
; This function is called once for each chunk of data read from the file.
; On entry, SI points to the start of the buffer and CX contains the
; count of characters in the buffer.  The ExamineFile function, which calls
; this function, will call this function once more with a single Ctrl-Z
; once the file has ended, to make sure that the end-of-file has registered.
;
; The loop starting at NextChar: is fairly tight.  I optimised it a bit more
; and improved the speed from about 1.683MB/sec to 1.814MB/sec (on RAM drive
; on my 486DX2-66) but this was messy, so it currently gets about 1.683MB/sec.
; An assembler optimisation wizard could probably improve this significantly.

PROC	Pass_WordCount	near
		mov	di,[WC_State]	; Get initial state
		cmp	di,OFFSET WCState_EOF ; Already in EOF state?
		je	PassedEOF	; If so, don't waste time here
		jcxz	PassedEOF	; If no bytes to process!

		point	bx,CharClass0	; Point to classification table
		add	bh,[WCClass]	; Index to appropriate table
		mov	dx,[LineCntL]	; Get line count loword
		mov	bp,[WordCntL]	; Get word count loword
		clr	ax		; AH will remain zero throughout

NextChar:	lodsb			; Get char from file buffer
		xlat			; Get its classification (0/3/6/9/12)
		add	di,ax		; Point to new state
		mov	al,[di+2]	; Get flags
		mov	di,[di]		; Get new state pointer
		shr	al,1		; Get "increment the line count" flag
		adc	dx,0		; Increment loword if set
		jc	IncLineH	; If carry into hiword
ContLine:	shr	al,1		; Get "increment the word count" flag
		adc	bp,0		; Increment loword if set
		jc	IncWordH	; If carry into hiword
ContWord:	dec	cx		; Quicker than LOOP
		jnz	NextChar
		mov	[WC_State],di	; Save state for next buffer's-worth
		mov	[LineCntL],dx	; Restore line count loword
		mov	[WordCntL],bp	; Restore word count loword
PassedEOF:	ret			; Done this buffer

IncLineH:	inc	[LineCntH]	; Carry into hiword
		jmp	SHORT ContLine	; Continue
IncWordH:	inc	[WordCntH]	; Carry into hiword
		jmp	SHORT ContWord	; Continue
ENDP	Pass_WordCount

; Output functions -------------------------------------------------------------

; These functions are called by ProcessFile to insert appropriate bits of
; information into the line of information being generated for a file.
; The functions are only called if the appropriate option flags are set.

PROC	Output_CksumCRC	near
;				Func:	Display checksum or CRC text value on
;					output line
;				In:	DI -> Position in output text line
;				Out:	Output text line is modified
;				Lost:	AX BX CX DX SI DI
		call	Spacer		; Do space padding
		cmp	[CRCFunc],OFFSET Pass_CRC32 ; Doing 32-bit CRCs?
		jne	DoingChksums	; If not, we are doing checksums
		mov	ax,[WORD CRC+2]	; Hiword
		call	ConvertHex	; Convert to hex
		mov	ax,[WORD CRC+0]	; Loword
		jmp	SHORT ConvertHex ; Convert to hex
DoingChksums:	mov	ax,[Checksum]	; Get checksum
PROC	ConvertHex	near
		push	ax
		mov	al,ah		; Get hibyte
		call	ConvertHexByte	; Convert it
		pop	ax		; And fall through
PROC	ConvertHexByte	near
		push	ax
		shr	al,1
		shr	al,1
		shr	al,1
		shr	al,1
		call	PrintNibble
		pop	ax
PrintNibble:	and	al,0Fh
		add	al,90h
		daa
		adc	al,40h
		daa
PrintNibbl1:	stosb
		ret
ENDP	ConvertHexByte
ENDP	ConvertHex
ENDP	Output_CksumCRC

PROC	Output_Clust	near
;				Func:	Calculate number of clusters required
;					for file, given file size and cluster
;					size, store into output line, update
;					total cluster count
;				In:	[DTAPtr] -> DTA containing file info
;					DI -> Position in output text line
;				Out:	Output text line is modified
;					TotClustL,H are modified
;				Lost:	AX BX CX DX DI
		mov	si,[DTAPtr]	; Point to DTA
		call	Spacer		; Do space padding
		mov	ax,[si+DTA_FSize+0] ; Loword of file size
		mov	dx,[si+DTA_FSize+2] ; Hiword of file size
		mov	bx,[BytesPerClust] ; Get number of bytes per cluster
		dec	bx		; Get bytes per cluster minus one
		add	ax,bx		; Add to file size
		adc	dx,0		; Carry into hiword
		inc	bx		; Back to bytes per cluster
		mov	cx,16		; Limit shift count
ShiftClust:	shr	bx,1		; Shift down bytes per cluster
		jc	GotClustCnt	; If carried, have divided correctly
		shr	dx,1		; Shift down hiword
		rcr	ax,1		; Shift down loword
		loop	ShiftClust	; Loop for division
GotClustCnt:	add	[TotClustL],ax	; Add to total clusters loword
		adc	[TotClustH],dx	; Carry into hiword
		mov	cx," "*256+5	; Maximum of 5 digits required
		jmp	SHORT ConvDec16PNP ; Continue
ConvDec16NP:	mov	cx,5		; Maximum of 5 digits required
ConvDec16PNP:	clr	dx		; Max clusters is 32M / 512 bytes = 64K

MACRO	PrDec_Output
		stosb
ENDM

		INCLUDE	"PRDEC32.ASM"	; Decimal output routine from this file

ENDP	Output_Clust

PROC	Output_Dir	near
;				Func:	Generate file size, date, and time text
;					strings into output line, update
;					TotBytesL and TotBytesH accumulators.
;				In:	[DTAPtr] -> DTA containing file info
;					DI -> Position in output text line
;				Out:	Output text line is modified
;					TotBytesL,H are updated
;				Lost:	AX BX CL DX SI DI
		mov	si,[DTAPtr]	; Point to DTA
		call	Spacer		; Do space padding
		mov	ax,[si+DTA_FSize+0] ; Loword of file size
		mov	dx,[si+DTA_FSize+2] ; Hiword of file size
		add	[TotBytesL],ax	; Add into total accumulator loword
		adc	[TotBytesH],dx	; Hiword too, with carry also
		mov	cx," "*256+10	; Ten digits, space padding
		call	Mach32_DecASC	; Convert to decimal
		call	Spacer		; Do space padding
		push	[WORD si+DTA_Time] ; Get time word onto stack
		mov	ax,[si+DTA_Date] ; Get date word
		mov	si,[DateFPtr]	; Point to date format table
		push	ax		; Keep date word on stack
		call	[WORD si+0]	; Call handler for first digit group
		mov	al,[DateSep]	; Separator
		stosb			; Store it
		pop	ax		; Restore date word
		push	ax		; Back onto stack
		call	[WORD si+2]	; Call handler for second digit group
		mov	al,[DateSep]	; Separator
		stosb			; Store it
		pop	ax		; Restore date word
		call	[WORD si+4]	; Call handler for third digit group
		call	Spacer		; Do space padding
		pop	ax		; Restore time word
		push	ax		; Back onto stack
		mov	cl,11		; Shift count
		shr	ax,cl		; Shift it
		and	ax,00011111b	; Limit to 31 (0-23)
		call	ConvertDec2	; Convert to decimal
		mov	al,":"		; Separator
		stosb			; Store it
		pop	ax		; Restore time word
		mov	cl,5		; Shift count
		shr	ax,cl		; Shift it
		and	ax,00111111b	; Limit to 63 (0-59)
		jmp	SHORT ConvertDec2 ; Convert to decimal and return
ENDP	Output_Dir

PROC	DateDay		near
		and	ax,00011111b	; Get day
		jmp	SHORT ConvertDec2 ; Go to convert to decimal
ENDP	DateDay

PROC	DateMonth	near
		mov	cl,5		; Shift count
		shr	ax,cl		; Shift it
		and	ax,00001111b	; Limit to 15
		jmp	SHORT ConvertDec2
ENDP	DateMonth

PROC	DateYear	near
		mov	cl,9		; Shift count
		shr	ax,cl		; Shift it
		and	ax,01111111b	; Limit to 127
		add	ax,180		; Convert to 1980 upwards
AdjustYear:	sub	ax,100		; Adjust
		cmp	ax,100
		jae	AdjustYear	; Make sure it's less than 100
ConvertDec2:	aam			; Split digits into AH and AL
		xchg	al,ah		; Swap characters
		add	ax,"00"		; Convert to ASCII
		stosw			; Store characters
		ret
ENDP	DateYear

PROC	Output_Attr	near
;				Func:	Generate file attribute characters,
;					store into output line
;				In:	[DTAPtr] -> DTA containing file information
;					DI -> Position in output text line
;				Out:	Output text line is modified
;				Lost:	AX DI
		mov	si,[DTAPtr]	; Point to DTA
		call	Spacer		; Do space padding
		mov	ah,[si+DTA_Attrib] ; Get file attribute
		mov	al,"r"		; Read-only flag
		test	ah,00000001b	; Test the flag
		call	SingleAttr	; Generate the appropriate character
		mov	al,"a"		; Archive flag
		test	ah,00100000b	; Test the flag
		call	SingleAttr	; Generate the appropriate character
		mov	al,"s"		; System flag
		test	ah,00000100b	; Test the flag
		call	SingleAttr	; Generate the appropriate character
		mov	al,"h"		; Hidden flag
		test	ah,00000010b	; Test the flag
SingleAttr:	jz	GotAttr		; If attribute wasn't set
		and	al,01011111b	; If attribute set, convert to U/C
GotAttr:	stosb			; Store character into output line
		ret
ENDP	Output_Attr

PROC	Output_WC	near
;				Func:	Store word and line count values into
;					output line, update WordTotL,H and
;					LineTotL,H accumulators
;				In:	WordCntL,H = Count of words found
;					LineCntL,H = Count of lines found
;					DI -> Position in output text line
;				Out:	Output text line is modified
;					WordCntL,H and LineCntL,H updated
;				Lost:	AX DI
		call	Spacer		; Do space padding
		mov	ax,[WordCntL]	; Loword of word count
		mov	dx,[WordCntH]	; Hiword of word count
		add	[WordTotL],ax	; Add into total
		adc	[WordTotH],dx	; Hiword
		mov	cx," "*256+8	; Eight digits, space padding
		call	Mach32_DecASC	; Convert to decimal
		call	Spacer		; Do space padding
		mov	ax,[LineCntL]	; Loword of line count
		mov	dx,[LineCntH]	; Hiword of line count
		add	[LineTotL],ax	; Add into total
		adc	[LineTotH],dx	; Hiword
		mov	cx," "*256+8	; Eight digits, space padding
		jmp	Mach32_DecASC	; Convert to decimal
ENDP	Output_WC

PROC	Output_9660	near
		cmp	[Bad9660],0	; Did filename violate ISO9660?
		jz	NoOutput_9660	; If not
		point	si,Text9660	; If so
		jmp	CopyStr		; Add it to the line
NoOutput_9660:	ret
ENDP	Output_9660

; Final totals display ---------------------------------------------------------

; This function does nothing unless at least two files were found.
;
; If the -L option was given, it displays the total number of bytes and the
; total number of files, and nothing else.  If -L was not given, it checks
; the other parameters and behaves as follows.
;
; If directory information displayed, display total bytes in all files.
; If clusters displayed, display total clusters.  If both displayed,
; use a combined message for both.
; If word and line counts displayed, display totals for these.
; Finally, if histograms were displayed, display a combined histogram.

PROC	DisplayTotals	near
		cmp	[Files],2	; Processed two or more files?
		jae	DispTotalCont	; If so, continue
		ret			; Just return
DispTotalCont:	cmp	[LFlag],0	; Did we use -L?
		jnz	JustTotBytes	; If so, output total bytes and files
		cmp	[DFlag],0
		jz	NoTotB		; If no directory info requested
		cmp	[SFlag],0	; Size in clusters requested too?
		jz	JustTotBytes	; If not, just display total bytes

; Display total bytes and clusters on a single line if both requested

		point	di,OutLine	; Point to start of output line
		point	si,BothTot1Msg	; Point to "Total of " text
		call	CopyStr		; Copy into output line
		mov	ax,[TotBytesL]	; Get total bytes, loword
		mov	dx,[TotBytesH]	; Get hiword too
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal ASCII
		point	si,BothTot2Msg	; Point to " bytes occupying " text
		call	CopyStr		; Copy into output line
		mov	ax,[TotClustL]	; Get loword of total clusters
		mov	dx,[TotClustH]	; Get hiword of total clusters
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal in OutLine
		point	si,BothTot3Msg	; Point to " clusters in " text
		call	CopyStr		; Copy into output line
		mov	ax,[Files]	; Get number of files handled
		clr	dx		; Zero hiword
		mov	cx,5		; Five digits, no padding
		call	Mach32_DecASC	; Convert to decimal ASCII
		point	si,FilesMsg	; Point to " files" text
		call	CopyStr
		call	WriteOutLine	; Write line to output, with newline
		jmp	SHORT DoneClustSum ; Skip following check

JustTotBytes:	point	di,OutLine	; Point to start of output line
		mov	ax,[TotBytesL]	; Get total bytes, loword
		mov	dx,[TotBytesH]	; Get hiword too
		mov	cx," "*256+10	; Ten digits, space padding
		call	Mach32_DecASC	; Convert to decimal ASCII
		point	si,TotBytesMsg	; Point to " total bytes in " text
		call	CopyStr		; Copy into output line
		mov	ax,[Files]	; Get number of files handled
		clr	dx		; Zero hiword
		mov	cx,5		; Five digits, no padding
		call	Mach32_DecASC	; Convert to decimal ASCII
		point	si,FilesMsg	; Point to " files" text
		call	CopyStr
		call	WriteOutLine	; Write line to output, with newline
		cmp	[LFlag],0	; Did we use -L?
		jnz	DispTotalFin	; If so, we're done

; If number of clusters were displayed, display the total clusters

NoTotB:		cmp	[SFlag],0	; Cluster counts?
		jz	DoneClustSum	; If not
		point	di,OutLine	; Point to start of output line
		point	si,TotalMsg	; Point to "Total clusters: " text
		call	CopyStr		; Copy into output line
		mov	ax,[TotClustL]	; Get loword of total clusters
		mov	dx,[TotClustH]	; Get hiword of total clusters
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal in OutLine
		call	WriteOutLine	; Write output line

; Display total words and lines if appropriate

DoneClustSum:	cmp	[WFlag],0	; Word counts?
		jz	NoWordCount	; If not
		point	di,OutLine	; Point to start of output line
		point	si,TotWordsMsg	; Point to "Total words: " text
		call	CopyStr		; Copy into output line
		mov	ax,[WordTotL]	; Loword of total word count
		mov	dx,[WordTotH]	; Hiword
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal
		point	si,TotLinesMsg	; Point to "  Total lines: " text
		call	CopyStr		; Copy into output line
		mov	ax,[LineTotL]	; Loword of total line count
		mov	dx,[LineTotH]	; Hiword
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal
		call	WriteOutLine	; Write output line

NoWordCount:	cmp	[HFlag],0
		jz	DispTotalFin	; If no histograms requested, finish up

; Display total bytes and combined file histogram

		call	WriteNewline	; Blank line
		point	di,OutLine	; Output line
		mov	ax,[THistTotL]	; Get histogram total loword
		mov	dx,[THistTotH]	; Get histogram total hiword
		mov	cx,10		; Ten digits, no padding
		call	Mach32_DecASC	; Convert to decimal ASCII
		point	si,CombHistMsg	; Point to " bytes total in combined histogram for all files:"
		call	CopyStr		; Store to line
		call	WriteOutLine	; Write output line
		call	WriteNewline	; Blank line
		point	bp,THistTbl	; Point to total (all files) histogram
		call	ShowHist	; Display histogram for combined files
DispTotalFin:	ret
ENDP	DisplayTotals

PROC	ShowHist	near
;				Func:	Generate output showing result of the
;					file histogram, from the FHistTbl or
;					THistTbl table, write to StdOut
;				In:	BP -> Hist. tbl (FHistTbl or THistTbl)
;					F/THistTbl = Histogram table
;				Out:	Writes to StdOut
;				Lost:	AX BX CX DX SI DI
		mov	[HistLimL],-1	; Look for the very highest value
		mov	[HistLimH],-1
HistLoop1:	mov	si,bp		; Start at start of histogram
		mov	cx,256		; Number of entries
		mov	[HistMaxL],0	; Looking for highest - reset max value
		mov	[HistMaxH],0
HistLoop2:	mov	ax,[si]		; Get loword of histogram value
		mov	dx,[si+2]	; Get hiword too
		mov	bx,dx		; Hiword to BX (will be destroyed)
		cmp	ax,[HistMaxL]	; Compare loword to max for this pass
		sbb	bx,[HistMaxH]	; Compare hiword
		jb	NotMax		; If not greater or equal
		mov	bx,dx		; Hiword to BX (will be destroyed)
		cmp	ax,[HistLimL]	; Compare loword to limit
		sbb	bx,[HistLimH]	; Compare hiword
		jae	NotMax		; If greater than limit value
		mov	[HistMaxL],ax	; Store new maximum for this pass
		mov	[HistMaxH],dx	; Hiword too
NotMax:		add	si,4		; Next entry
		loop	HistLoop2	; Loop for all entries

		mov	ax,[HistMaxL]	; Get count for pass just done
		mov	dx,[HistMaxH]	; Hiword too
		mov	[HistLimL],ax	; Store as limit for next pass
		mov	[HistLimH],dx	; Hiword too
		or	ax,dx		; Test for zero
		jz	WriteNewline	; If zero

; Now have highest histogram count less than previous pass - generate a list.

		mov	si,bp		; Start at start of histogram
		mov	cx,256		; Count of entries to scan
HistLoop3:	mov	ax,[si]		; Get loword of histogram value
		mov	dx,[si+2]	; Get hiword too
		cmp	ax,[HistMaxL]	; Lowords match?
		jne	NotCount	; If not
		cmp	dx,[HistMaxH]	; Hiwords match?
		jne	NotCount	; If not
		push	si		; Keep pointer
		mov	bx,si		; Get pointer
		sub	bx,bp		; Get byte value x 4
		shr	bx,1		; Get byte value x 2
		shr	bx,1		; Get byte value
		push	cx		; Keep count
		call	Hist_DispRec	; Display record
		pop	cx		; Restore count
		pop	si		; Restore pointer
NotCount:	add	si,4		; Next entry
		loop	HistLoop3	; Loop for all entries with this count
		jmp	HistLoop1	; Loop for more entries
ENDP	ShowHist

; This function is called with BX = byte value and DX|AX = count of occurrences.
; It displays a single histogram dump line, listing the count of occurrences
; and the value in decimal and hex, and ASCII if appropriate.
; The SI and CX registers will be destroyed by the subroutine.

PROC	Hist_DispRec	near
		push	bx		; Keep byte value
		mov	cx," "*256+7	; Seven digits, space padding
		point	di,OutLine	; Output line
		call	Mach32_DecASC	; Convert to decimal in output line
		point	si,OfMsg	; Point to " of byte value" text
		call	CopyStr		; Add to line
		pop	ax		; Get value back
		push	ax		; Back onto stack
		cwd			; Zero DX
		mov	cx," "*256+5	; Five chars, space pad (only 3 used)
		call	Mach32_DecASC	; Convert to decimal in output line
		mov	ax,"  "		; Get "  " characters
		stosw			; Into output line
		mov	al,"0"		; Leading zero for hex
		stosb			; Into output line
		pop	ax		; Restore byte value again
		push	ax		; Back onto stack
		call	ConvertHexByte	; In hex format
		mov	al,"h"		; Hex suffix
		stosb			; Into output line
		mov	ax,"  "		; Separator spaces
		stosw			; Into output line
		pop	ax		; Byte value back again
		cmp	al," "		; Control character?
		jb	NotPrintable	; If so
		cmp	al,7Fh		; High-ASCII or Del?
		jae	NotPrintable	; If so
		mov	ah,al		; To AH
		mov	al,"'"		; Quote
		stosw			; Store quote and character
		stosb			; Store closing quote
NotPrintable:	jmp	SHORT WriteOutLine ; Write output line
ENDP	Hist_DispRec

; The general text output method is to point DI to 'OutLine' and build up a
; line of text in this buffer.	When the line is complete, WriteOutLine is
; called, with DI pointing just past the last character stored into the line.
; WriteOutLine appends a CR/LF pair to the text, calculates the length of the
; line, and sends it to the standard output (handle #1).

PROC	WriteNewline	near
		point	di,OutLine	; Empty OutLine - just the CR/LF
PROC	WriteOutLine	near
;				Func:	Write string of text to standard output
;					with CR/LF appended
;				In:	OutLine = Text string to be output
;					DI -> End of text in OutLine
;				Out:	None
;				Lost:	AX BX CX DX DI
		mov	ax,0A0Dh	; CR/LF
		stosw			; Store
		point	dx,OutLine	; Point to line of text
		mov	cx,di		; Get pointer past end of text
		sub	cx,dx		; Calculate string length
		mov	bx,1		; Standard output handle
		mov	ah,40h
		int	21h		; Write to standard output
		cmp	[PFlag],0	; Pause after each page?
		jz	NoPause		; If not
		dec	[LineCount]	; Decrement counter of lines remaining
		jnz	NoPause		; If there are more
		mov	[LineCount],23	; Reset line count
		point	dx,PressKeyM	; Point to message
		mov	cx,PressKeyL	; Get length
		mov	bx,2		; Write it to StdErr
		mov	ah,40h
		int	21h		; Display prompt
WaitKeypress:	mov	ah,08h
		int	21h		; Wait for keypress (supports Ctrl-C)
		push	ax		; Keep keypress
		point	dx,WipeKeyM	; Point to message to wipe old one
		mov	cx,WipeKeyL	; Get length
		mov	ah,40h
		int	21h		; Wipe out the old message
		pop	ax		; Restore key
		cmp	al,13		; Was it C/R?
		jne	NoPause		; If not
		mov	[LineCount],1	; If so, only display one more line
NoPause:	ret
ENDP	WriteOutLine
ENDP	WriteNewline

; The Spacer function handles spacing the various parameters along the line
; as a file information line is generated.  Initially the Spaced flag is
; clear, and the first call to Spacer will have no effect, so that the first
; column is hard up against the left of the screen in the output.  But once
; Spacer has been called, the Spaced flag is set, and subsequent calls to
; Spacer will cause two spaces to be inserted into the line being built up
; in the OutLine buffer area.  This keeps the columns separate.

PROC	Spacer		near
;				Func:	Generate two spaces in output line
;					if appropriate
;				In:	DI -> Position in output line
;					[Spaced] = Spacing control flag
;				Out:	DI -> Updated position in output line
;					[Spaced] is modified
;				Lost:	AX DI
		cmp	[Spaced],0
		jz	NoSpc		; If not time to do spacing now
		mov	ax,"  "		; Two spaces
		stosw			; Store into output line
NoSpc:		mov	[Spaced],1	; Flag ready for next time
		ret
ENDP	Spacer

PROC	CopyStr		near
;				Func:	Copy null-terminated string, return
;					destination pointer pointing to null
;					terminator just copied
;				In:	[DS:SI] = String to be copied
;					[ES:DI] = Destination for copy
;				Out:	[ES:DI] -> Null-terminator of copy
;				Lost:	AL=0 SI DI
CopyStrLp:	lodsb			; Copy bytes from source to destination
		stosb
		test	al,al
		jnz	CopyStrLp	; Loop until copied the null-terminator
		dec	di		; Destination pointer to null-terminator
		ret
ENDP	CopyStr

PROC	CopyStrR	near
;				Func:	Copy null-terminated string, restore
;					destination pointer after copy
;				In:	[DS:SI] = String to be copied
;					[ES:DI] = Destination for copy
;				Out:	SI is set to DX (DTA Pointer)
;				Lost:	AL=0 SI
		push	di		; Preserve destination pointer
		call	CopyStr		; Copy the string including terminator
		pop	di		; Restore destination pointer
		mov	si,dx		; Set up SI to point to DTA
		ret
ENDP	CopyStrR

PROC	ConvertUC	near
;				Func:	Convert character in AL to upper case
;				In:	AL = Character to be converted
;				Out:	AL = Converted character
;				Lost:	AL
		cmp	al,"a"
		jb	NoConvertUC
		cmp	al,"z"
		ja	NoConvertUC
		sub	al,"a"-"A"
NoConvertUC:	ret
ENDP	ConvertUC

; Get drive info ---------------------------------------------------------------

; This function accepts a drive letter (in AL) and returns for that drive,
; the number of bytes per cluster (in AX), the total clusters (in BP) and
; the number of free clusters (in BX).
; DOS function 36h reports this information, but will not issue a critical
; error if there is no disk in the drive (I don't know why), so the returned
; number of sectors per cluster (in AX) is checked for negative; if negative,
; then DOS function (36h) has failed, so this function terminates with an
; appropriate error message.
;
; Info from Ralf Brown's Interrupt List: "lost clusters" are considered to be
; in use, and according to Dave Williams' MS-DOS reference, the value in DX
; (total clusters) is incorrect for non-default drives after ASSIGN is run.

PROC	GetDriveInfo	near
		mov	[BadDriveLtr],al ; Store in case of error
		sub	al,"A"-1	; Convert 'A' to 1
		xchg	ax,dx		; To DL
		mov	ah,36h		; Will return:	AX = sectors per cluster
		int	21h		;		BX = free clusters
		test	ax,ax		;		CX = bytes per sector
		jns	GotDriveInfo	;		DX = total clusters
		point	bx,BadDriveEM	; If failed
		jmp	ErrExit		; Go to error exit
GotDriveInfo:	mov	bp,dx		; Total clusters to BP
		mul	cx		; Get total bytes per cluster to AX
		ret
ENDP	GetDriveInfo

; Working stack area -----------------------------------------------------------

		ALIGN	2

WorkStack	DB	512 DUP(?)	; Working stack
		DB	(4 * NDTAs) DUP(?) ; Extra stack space for recurse
WorkStackTop	= $

; Uninitialised transient data -------------------------------------------------

SearchSpec	DB	2 DUP(?)	; Drive letter and colon
Pathspec	DB	120 DUP(?)	; Pathspec contents
Filespec	DB	15 DUP(?)	; Filespec
		DB	2 DUP(?)	; Gap
OutLine		DB	150 DUP(?)	; Output text line

		ALIGN	2

CRCTbl0		DB	256 DUP(?)	; CRC table byte 0
CRCTbl1		DB	256 DUP(?)	; CRC table byte 1
CRCTbl2		DB	256 DUP(?)	; CRC table byte 2
CRCTbl3		DB	256 DUP(?)	; CRC table byte 3

FHistTbl	DW	512 DUP(?)	; Per-file histogram count table

THistTbl	DW	512 DUP(?)	; Same table for all files combined

CharClass0	DB	(CharClasses * 256) DUP(?) ; Character class xlat tables

QualPathDTA	DB	DTAAlloc DUP(?)	; DTA for QualPathspec function

DTAs		DB	(NDTAs * DTAAlloc) DUP(?) ; DTA storage

InBuffer	DB	InBufSize DUP(?) ; File deblocking storage

; End of transient portion -----------------------------------------------------

		MASM
FreeSpace	=	$
MemParas	=	(OFFSET (FreeSpace-@curseg+15) SHR 4)
		prexp	<Transient size:> %(MemParas SHL 4) < bytes>
		IDEAL

; Note - if the transient size starts approaching 65536, reduce InBufSize.

		ENDS	ComFile
		END	Main

;-------------------------------------------------------------------------------
