
/*

   mTCP FtpSrv.cpp
   Copyright (C) 2009-2011 Michael B. Brutman (mbbrutman@gmail.com)
   mTCP web page: http://www.brutman.com/mTCP


   This file is part of mTCP.

   mTCP is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   mTCP is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mTCP.  If not, see <http://www.gnu.org/licenses/>.


   Description: FTP Server

   Changes:

   2011-05-27: Initial release as open source software

*/



// RFC  765 - File Transfer Protocol specification
// RFC 1579 - Firewall-Friendly FTP


#include <bios.h>
#include <ctype.h>
#include <direct.h>
#include <dos.h>
#include <io.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>

#include "types.h"

#include "utils.h"
#include "timer.h"
#include "packet.h"
#include "arp.h"
#include "tcp.h"
#include "tcpsockm.h"
#include "telnet.h"
#include "ftpcl.h"





// Data structures used for interpreting directory entries

typedef union {
  unsigned short us;
  struct {
    unsigned short twosecs : 5; /* seconds / 2 */
    unsigned short minutes : 6; /* minutes (0,59) */
    unsigned short hours : 5;   /* hours (0,23) */
  } fields;
} ftime_t;

typedef union {
  unsigned short us;
  struct {
    unsigned short day : 5;   /* day (1,31) */
    unsigned short month : 4; /* month (1,12) */
    unsigned short year : 7;  /* 0 is 1980 */
  } fields;
} fdate_t;

static char *Months[] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};





// FTP return codes and text messages
//
// If a msg string name ends in _v then it has printf specifiers in it
// and it needs to be used with addToOutput_var.

#define _NL_ "\r\n"

static char Msg_150_Send_File_List[] =       "150 Sending file list" _NL_;
static char Msg_200_Port_OK[] =              "200 PORT command successful" _NL_;
static char Msg_200_Noop_OK[] =              "200 NOOP command successful" _NL_;
static char Msg_202_No_Alloc_needed[] =      "202 No storage allocation necessary" _NL_;
static char Msg_215_System_Type[] =          "215 DOS Type: L8" _NL_;
static char Msg_220_ServerStr[] =            "220 mTCP FTP Server" _NL_;
static char Msg_221_Closing[] =              "221 Server closing connection" _NL_;
static char Msg_226_Transfer_Complete[] =    "226 Transfer complete" _NL_;
static char Msg_226_ABOR_Complete[] =        "226 ABOR complete" _NL_;
static char Msg_230_User_Logged_In[] =       "230 User logged in" _NL_;
static char Msg_331_UserOkSendPass[] =       "331 User OK, send Password" _NL_;
static char Msg_421_Service_Not_Avail[] =    "421 Service not available, try back later" _NL_;
static char Msg_425_Cant_Open_Conn[] =       "425 Cant open connection - please try again" _NL_;
static char Msg_425_Send_Port[] =            "425 Send PORT first or try passive mode" _NL_;
static char Msg_425_Transfer_In_Progress[] = "425 Transfer already in progress" _NL_;
static char Msg_426_Request_term[] =         "426 Request terminated" _NL_;

static char Msg_500_Parm_Missing_v[] =       "500 %s command requires a parameter" _NL_;
static char Msg_500_Syntax_Error_v[] =       "500 Syntax error: %s" _NL_;
static char Msg_501_Unknown_Option_v[] =     "501 Unrecognized option for %s: %s" _NL_;
static char Msg_502_Not_Implemented[] =      "502 Command not implemented" _NL_;
static char Msg_503_Already_Logged_In[] =    "503 You are already logged in" _NL_;
static char Msg_503_Send_RNFR_first[] =      "503 Send RNFR first" _NL_;
static char Msg_504_Unsupp_Option_v[] =      "504 Unsupported option for %s: %s" _NL_;
static char Msg_530_Login_Incorrect[] =      "530 Login incorrect" _NL_;
static char Msg_530_Please_Login[] =         "530 Please login" _NL_;
static char Msg_550_Bad_Path_Or_File[] =     "550 Bad path or filename" _NL_;
static char Msg_550_Filesystem_error[] =     "550 Filesystem error" _NL_;


static char *Msg_214_Help[] = {
  "214-Welcome to the mTCP FTP server, Beta Version: " __DATE__ _NL_,
  " USER  PASS  REIN  ACCT  REST" _NL_, 
  " RNFR  RNTO  DELE" _NL_, 
  " CWD   XCWD  CDUP  XCUP  PWD   XPWD  MKD   XMKD  RMD   XRMD" _NL_, 
  " PASV  PORT  ABOR  LIST  NLST  RETR  STOR  STOU  APPE" _NL_, 
  " MODE  STRU  TYPE  HELP  ALLO  FEAT  MDTM  NOOP  STAT  SYST SITE" _NL_,
  "214 OK" _NL_,
  NULL
};



static const char ASCII_str[] = "ASCII";
static const char BIN_str[] = "BINARY";


// Function prototypes

static int  readConfigParms( void );
static void shutdown( int rc );


static void processNewConnection( TcpSocket *newSocket );
static void serviceClient( FtpClient *client );


static void doAbort( FtpClient *client );
static void doCwd ( FtpClient *client, char *target );
static void doDele ( FtpClient *client, char *target );
static void doHelp ( FtpClient *client );
static void doRnfr ( FtpClient *client, char *target );
static void doRnto ( FtpClient *client, char *target );
static void doMkd ( FtpClient *client, char *target );
static void doRmd ( FtpClient *client, char *target );
static void doDataXfer( FtpClient *client, char *filespec );
static void doMode( FtpClient *client, char *nextTokenPtr );
static void doMdtm( FtpClient *client, char *nextTokenPtr );
static void doPort( FtpClient *client, char *nextTokenPtr );
static void doPasv( FtpClient *client );
static void doSize( FtpClient *client, char *nextTokenPtr );
static void doSite( FtpClient *client, char *nextTokenPtr );
static void doStat( FtpClient *client, char *nextTokenPtr );
static void doStru( FtpClient *client, char *nextTokenPtr );
static void doType( FtpClient *client, char *nextTokenPtr );
static void doXfer( FtpClient *client, char *nextTokenPtr, FtpClient::DataXferType listType );

static void doSiteStats( FtpClient *client );
static void doSiteWho( FtpClient *client );
static void doSiteDiskFree( FtpClient *client, char *nextTokenPtr );

static int formFullPath( FtpClient *client, char *outBuffer, int outBufferLen, char *userPart );
static int formFullPath_DataXfers( FtpClient *client, char *outBuffer, int outBufferLen, char *userPart );

static void endDataTransfers( FtpClient *client, char *msg );
static void endSession( FtpClient *client );


int normalizeDir( char *buffer, int bufferLen );
void convertToDosForm( char *buffer_p );
void convertForDisplay( int unixstyle, char *buffer_p );





// Path and file helper functions


int isAbsolute( const char *path ) {
  if ( path == NULL ) return 0;
  return ( (path[0] == '/') || (path[0] == '\\') || ( isalpha(path[0]) && path[1]==':' && (path[2] == '/' || path[2] == '\\') ));
}


int startsWithDriveLetter( const char *path ) {
  if ( path == NULL ) return 0;
  return (isalpha(path[0])) && (path[1]==':');
}



int isDirectory( const char *fullPath ) {

  if ( fullPath == NULL ) return 0;

  // We might pass in a drive letter and colon and ask if it is a directory.
  // This happens when the user is at the root of a drive; the backslash
  // gets stripped off.  If this happens make a small correction.

  char newPath[4];
  if ( isalpha(fullPath[0]) && fullPath[1] == ':' && fullPath[2] == 0 ) {
    newPath[0] = fullPath[0]; newPath[1] = fullPath[1];
    newPath[2] = '\\'; newPath[3] = 0;
    fullPath = newPath;
  }

  struct stat statbuf;
  int rc = stat( fullPath, &statbuf );
  if ( rc == 0 ) {
    return S_ISDIR(statbuf.st_mode);
  }
  return 0;
}



int isFile( const char *fullPath ) {

  struct stat statbuf;
  int rc = stat( fullPath, &statbuf );
  if ( rc == 0 ) {
    return S_ISREG(statbuf.st_mode);
  }
  return 0;
}


int doesExist( const char *fullPath ) {
  struct stat statbuf;
  int rc = stat( fullPath, &statbuf );
  if ( rc == 0 ) {
    return 1;
  }
  return 0;
}







// Trap Ctrl-Break and Ctrl-C so that we can unhook the timer interrupt
// and shutdown cleanly.

// Check this flag once in a while to see if the user wants out.
volatile uint8_t CtrlBreakDetected = 0;

void ( __interrupt __far *oldCtrlBreakHandler)( );

void __interrupt __far ctrlBreakHandler( ) {
  CtrlBreakDetected = 1;
}

void __interrupt __far ctrlCHandler( ) {
  CtrlBreakDetected = 1;
}




// Globals

TcpSocket *ListeningSocket = NULL;

char PasswordFilename[DOS_MAX_PATHFILE_LENGTH];

char LogFilename[DOS_MAX_PATHFILE_LENGTH];
FILE *LogFile = NULL;

clockTicks_t FtpSrv_timeoutTicks = 3276; // 180 seconds at 18.2 ticks per second

uint16_t FtpSrv_Clients = 3;

uint16_t FtpSrv_Control_Port = 21;

IpAddr_t Pasv_IpAddr;
uint16_t Pasv_Base = 2048;
uint16_t Pasv_Ports = 1024;


// Fixme: This might roll over on us!
uint16_t Current_Year;


uint32_t SessionCounter = 0;

uint32_t Stat_SessionTimeouts = 0;

uint32_t Stat_LIST = 0, Stat_NLST = 0;
uint32_t Stat_RETR = 0, Stat_APPE = 0, Stat_STOR = 0, Stat_STOU = 0;

char StartTime[26];



// Toggles
//
uint8_t Sound = 1;




void addToLog( char *fmt, ... ) {

  char lineBuffer[240];

  DosTime_t currentTime;
  gettime( &currentTime );

  DosDate_t currentDate;
  getdate( &currentDate );

  fprintf( LogFile, "%04d-%02d-%02d %02d:%02d:%02d.%02d ",
           currentDate.year, currentDate.month, currentDate.day,
           currentTime.hour, currentTime.minute, currentTime.second,
           currentTime.hsecond );

  va_list ap;
  va_start( ap, fmt );
  vsprintf( lineBuffer, fmt, ap );
  va_end( ap );

  fprintf( LogFile, "%s", lineBuffer );
  printf( "%s", lineBuffer );

  // If tracing is turned on put it there automatically.
  TRACE(( "Ftp %s", lineBuffer ));

  // flushall( );

}



// initSrv
//
// Returns 0 on succesful startup
// Returns 1 on failure

int initSrv( void ) {

  // Read parameters and initialize
  if ( Utils::parseEnv( ) ) {
    return 1;
  }

  // Once our IP address is known set our default address that will
  // be advertised on the PASV command.  This might be overridden
  // when we read our application specific config parms.
  Ip::copy( Pasv_IpAddr, MyIpAddr );


  if ( readConfigParms( ) || FtpUser::init( PasswordFilename ) ) {
    return 1;
  }


  // See if we can open the log file for append.
  LogFile = fopen( LogFilename, "ac" );
  if ( LogFile == NULL ) {
    puts( "\nCan't open logfile for writing." );
    return 1;
  }

  fprintf( LogFile, "mTCP FtpSrv version(" __DATE__ ") starting\n" );

  if ( FtpClient::initClients( FtpSrv_Clients ) ) {
    puts( "\nFailed to initialize clients" );
    return 1;
  }


  // For small numbers of clients (5 and under) allocate 3 sockets per client
  // plus one more for a listening socket.  After five clients start to only
  // give one sockets per client.

  uint16_t requestedSockets;
  uint16_t requestedTcpBuffers;

  if ( FtpSrv_Clients < 6 ) {
    requestedSockets = FtpSrv_Clients * 3 + 1;
    requestedTcpBuffers = FtpSrv_Clients * 5;
  }
  else {
    requestedSockets = 16 + (FtpSrv_Clients-5);
    requestedTcpBuffers = TCP_MAX_XMIT_BUFS;
  }


  if ( Utils::initStack( requestedSockets, requestedTcpBuffers ) ) {
    puts( "\nFailed to initialize TCP/IP - exiting" );
    return 1;
  }


  // From this point forward you have to call the shutdown( ) routine to
  // exit because we have the timer interrupt hooked.

  // Save off the oldCtrlBreakHander and put our own in.  Shutdown( ) will
  // restore the original handler for us.
  oldCtrlBreakHandler = getvect( 0x1b );
  setvect( 0x1b, ctrlBreakHandler);

  // Get the Ctrl-C interrupt too, but do nothing.
  setvect( 0x23, ctrlCHandler);


  // Note our starting time
  struct tm time_of_day;
  time_t tmpTime;
  time( &tmpTime );
  _localtime( &tmpTime, &time_of_day );
  _asctime( &time_of_day, StartTime );
  StartTime[24] = 0; // Get rid of unwanted carriage return

  // Make a note of the current year - we use this for directory listings.
  DosDate_t currentDate;
  getdate( &currentDate );
  Current_Year = currentDate.year;



  char lineBuffer[80];

  sprintf( lineBuffer, "Clients: %u, TCP sockets: %u, TCP send bufs: %u, Session timeout: %lu secs\n",
          FtpSrv_Clients, requestedSockets, requestedTcpBuffers, (FtpSrv_timeoutTicks / 18ul) );
  addToLog( "%s", lineBuffer );

  sprintf( lineBuffer, "Control port: %u, Pasv ports: %u-%u\n", FtpSrv_Control_Port, Pasv_Base, (Pasv_Base+Pasv_Ports-1) );
  addToLog( "%s", lineBuffer );

  sprintf( lineBuffer, "Pasv response IP addr: %d.%d.%d.%d\n", Pasv_IpAddr[0], Pasv_IpAddr[1], Pasv_IpAddr[2], Pasv_IpAddr[3] );
  addToLog( "%s", lineBuffer );

  puts( "\nPress [Ctrl-C] to end the server\n" );

  return 0;

}


static char CopyrightMsg1[] = "mTCP FtpSrv by M Brutman (mbbrutman@gmail.com) (C)opyright 2010-2011\n";
static char CopyrightMsg2[] = "Version: " __DATE__ "\n\n";

int main( int argc, char *argv[] ) {

  printf( "%s  %s", CopyrightMsg1, CopyrightMsg2 );

  if ( initSrv( ) ) {
    puts( "\nServer can not start - exiting" );
    exit( 1 );
  }


  // If you get to here you must use shutdown to end the program because we
  // have the timer interrupt, Ctrl-Break and Ctrl-C hooked.


  // Setup our listening socket

  ListeningSocket = TcpSocketMgr::getSocket( );
  ListeningSocket->listen( FtpSrv_Control_Port, 512 );


  clockTicks_t lastTimeoutSweep = TIMER_GET_CURRENT( );
  clockTicks_t lastKeyboardCheck = TIMER_GET_CURRENT( );


  // Main loop

  uint8_t shuttingDown = 0;

  while ( 1 ) {

    PACKET_PROCESS_MULT( PACKET_BUFFERS );
    Arp::driveArp( );
    Tcp::drivePackets( );


    // Check for client inactivity every 10 seconds

    if ( Timer_diff( lastTimeoutSweep, TIMER_GET_CURRENT( ) ) > TIMER_MS_TO_TICKS( 10000 ) ) {

      lastTimeoutSweep = TIMER_GET_CURRENT( );

      for ( uint16_t i=0; i < FtpClient::activeClients; i++ ) {

        FtpClient *client = FtpClient::activeClientsTable[i];

        if ( client->state != FtpClient::Closed ) {

          // Get the newer of the control socket last activity time and the
          // data socket last activity time, if it is in use.

          clockTicks_t last = client->cs->lastActivity;
          if ( client->ds != NULL && client->ds->lastActivity > last ) last = client->ds->lastActivity;

          clockTicks_t diff = Timer_diff( last, TIMER_GET_CURRENT( ) );

          // End them if latest activity is greater than timeout value.
          if ( diff > FtpSrv_timeoutTicks ) {
            Stat_SessionTimeouts++;
            endSession( client );
          }

        }

      } // end for

    }



    // Things to do once a second:
    //
    // - Check to see if we are finished shutting down
    // - Check for Ctrl-Break or Ctrl-C
    // - Read keyboard input

    if ( Timer_diff( lastKeyboardCheck, TIMER_GET_CURRENT( ) ) > TIMER_MS_TO_TICKS( 1000 ) ) {

      lastKeyboardCheck = TIMER_GET_CURRENT( );

      if ( shuttingDown ) {
        // Waiting for shutdown to complete
        if ( FtpClient::activeClients == 0 ) break;
      }
      else {

        int shutdownRequested = 0;

        // Check the keyboard

        if ( CtrlBreakDetected ) {
          shutdownRequested++;
        }
        else {

          if ( bioskey(1) != 0 ) {
            char c = bioskey(0);
            if ( c == 3 ) {
              shutdownRequested++;
            }
            else if ( c == 7 ) {
              Sound = !Sound;
            }
          }

        }

        if ( shutdownRequested ) {
          addToLog( "Shutdown requested\n" );
          // Start an involuntary close on everything
          for ( uint16_t i=0; i < FtpClient::activeClients; i++ ) {
            FtpClient *client = FtpClient::activeClientsTable[i];
            endSession( client );
          }
          shuttingDown++;
        }

      } // !shutting down.

    }


    if ( !shuttingDown ) {
      // Check for new connections
      TcpSocket *tmpSocket = TcpSocketMgr::accept( );
      if ( tmpSocket != NULL ) {
        processNewConnection( tmpSocket );
      }
    }



    // Service active FTP clients

    for ( uint16_t i=0; i < FtpClient::activeClients; i++ ) {

      FtpClient *client = FtpClient::activeClientsTable[i];

      // If it is in the active list and it went to Closed, then recycle it.
      if ( client->state == FtpClient::Closed ) {

        addToLog( "(%lu) Disconnect: %d.%d.%d.%d:%u\n", client->sessionId,
          client->cs->dstHost[0], client->cs->dstHost[1],
          client->cs->dstHost[2], client->cs->dstHost[3],
          client->cs->dstPort );

        client->cleanupSession( );

        // Remove from active list and put back on free list
        FtpClient::removeFromActiveList( client );
        FtpClient::returnFreeClient( client );

        // Break the loop and start from scratch because we changed the
        // order of entries in the Client table
        break;

      } // Did client completely close?

      else {
        // Service the socket
        serviceClient( client );
      }

    } // end service active clients


  } // end while


  shutdown( 0 );

  return 0;
}







void shutdown( int rc ) {

  addToLog( "Stats: Sessions: %lu  Timeouts: %lu\n", SessionCounter, Stat_SessionTimeouts );
  addToLog( "       LIST: %lu  NLST: %lu  RETR: %lu\n", Stat_LIST, Stat_NLST, Stat_RETR );
  addToLog( "       STOR: %lu  STOU: %lu  APPE: %lu\n", Stat_STOR, Stat_STOU, Stat_APPE );
  addToLog( "=== Server shutdown === \n" );

  Utils::endStack( );
  Utils::dumpStats( stdout );
  fclose( TrcStream );

  fclose( LogFile );
  exit( rc );
}




void processNewConnection( TcpSocket *newSocket ) {

  TRACE(( "Ftp Connect on port %u from %d.%d.%d.%d:%u\n",
          newSocket->srcPort,
          newSocket->dstHost[0], newSocket->dstHost[1],
          newSocket->dstHost[2], newSocket->dstHost[3],
          newSocket->dstPort ));

  // If this is a new connection to our control port create a new client.

  if ( newSocket->srcPort == FtpSrv_Control_Port ) {

    FtpClient *client = FtpClient::getFreeClient( ); 

    if ( client != NULL ) {
      client->startNewSession( newSocket, SessionCounter++ );
      client->addToOutput( Msg_220_ServerStr );
      newSocket = NULL;
    }
    else {
      // Could not get a new client.  Fall through to close the socket.
      // (newSocket not being NULL will trigger that.)
      newSocket->send( (uint8_t *)Msg_421_Service_Not_Avail, strlen(Msg_421_Service_Not_Avail) );
    }

  }
  else {

    // Could be a data socket.  If so, find the listening client.

    for ( uint16_t i=0; i < FtpClient::activeClients; i++ ) {

      FtpClient *client = FtpClient::activeClientsTable[i];

      // The client has to be listening and the address has to match perfectly.
      //
      // If the client does something stupid like try to connect twice we will
      // take the first one but not match on the second one because we won't
      // be listening anymore.

      if ( (client->ls != NULL) && (client->pasvPort == newSocket->srcPort) && Ip::isSame( client->cs->dstHost, newSocket->dstHost ) ) {

        // Great, it's a match.  Close the listening socket and set the
        // data socket.

        client->ls->close( );
        TcpSocketMgr::freeSocket( client->ls );
        client->ls = NULL;

        TRACE(( "Ftp (%lu) Close listening socket\n", client->sessionId ));


        // Ensure that there is not a data socket open already or we will
        // lose it.  This should only happen if the client does a PASV,
        // makes a connection, doesn't use it, and then does another PASV.
        // We'll guard against that by forcing data connections closed if
        // somebody does a PASV or PORT command while a data socket is already
        // present but a transfer is not in progress.

        if ( client->ds != NULL ) {
          // This is an error.  Force it closed before taking a new one.
          TRACE_WARN(( "Ftp (%lu) Closing data connection that was never used\n", client->sessionId ));
          client->ds->close( );
          client->ds = NULL;
        }

        client->ds = newSocket;

        // Set this to NULL to signify that we used it
        newSocket = NULL;
        break;
      }

    }

  }


  // If nobody claimed it close it

  if ( newSocket ) {
    TRACE(( "Ftp Nobody claimed the new socket - closing it\n" ));
    newSocket->close( );
    TcpSocketMgr::freeSocket( newSocket );
  }

}








// - If there was an active data connection service it.
// - If there is pending output push it out
// - If we are supposed to be closing up don't process any more user input


void serviceClient( FtpClient *client ) {


  // Did they drop on us?
  if ( client->cs->isRemoteClosed( ) ) {

    switch ( client->state ) {

      case FtpClient::ClosingPushOutput: {
        // If we were trying to force final output to the user don't bother.
        client->clearOutput( );
        break;
      }

      case FtpClient::Closing: {
        // Do nothing; we are already waiting for them to close
        break;
      }

      default: {
        // Unexpected close.
        
        TRACE(( "Ftp (%lu) Control socket dropped: %d.%d.%d.%d:%u\n", client->sessionId,
          client->cs->dstHost[0], client->cs->dstHost[1],
          client->cs->dstHost[2], client->cs->dstHost[3],
          client->cs->dstPort ));

        endSession( client );
      }

    } // end switch

  }



  // Send output on control socket.
  //
  // If there was any pending output try to send it out.  If we can't send it
  // then don't bother doing anything else; we don't want to overflow the
  // buffer.

  if ( client->pendingOutput( ) ) {
    client->sendOutput( );
    return;
  }



  // Handle directory listings that are in flight
  //
  // We have to do this to detect if they were closing and to process the
  // close.  This also cleans up the data structures and sockets
  
  if ( client->dataXferState != FtpClient::DL_NotActive ) {
    doDataXfer( client, NULL );
  }



  // If we were trying to push out final output on the control socket and
  // we got here, then we suceeded.  Now we can start the close process on
  // the control socket.

  if ( client->state == FtpClient::ClosingPushOutput ) {
    TRACE(( "Ftp (%lu) Last output pushed out, moving to Closing\n", client->sessionId ));
    client->cs->closeNonblocking( );
    client->state = FtpClient::Closing;
    return;
  }



  // Were we waiting for the sockets to close?  If all of our sockets
  // have closed then we can clean up.

  if ( client->state == FtpClient::Closing ) {

    // If dataXferState is DL_NotActive then those sockets are properly closed.
    // If our control socket is closed too then we are done.

    if ( (client->dataXferState == FtpClient::DL_NotActive) && (client->cs->isCloseDone( )) ) {
      TRACE(( "Ftp (%lu) All sockets closed\n", client->sessionId ));
      client->state = FtpClient::Closed;
      // Harvesting this client will be done in the main loop.
    }

    // Whether we are done or not, return because we don't want to process
    // more user input.
    return;
  }


  if ( client->statCmdActive ) { doStat( client, NULL ); return; }


  // Check for new input on the socket

  {
    int16_t bytesToRead = INPUTBUFFER_SIZE - client->inputBufferIndex;

    int16_t bytesRead = client->cs->recv( (uint8_t *)(client->inputBuffer + client->inputBufferIndex), bytesToRead );
    if ( bytesRead < 0 ) {
      TRACE(( "Ftp (%lu) error reading socket!\n", client->sessionId ));
      return;
    }
    client->inputBufferIndex += bytesRead;


    // Did we get a full line of input?

    if ( client->inputBufferIndex < 2 ) {
      // Not even a CR/LF pair fits here
      return;
    }

    int fullLine = 0;
    for ( int i=0; i < (client->inputBufferIndex)-1; i++ ) {
      if ( client->inputBuffer[i] == '\r' && client->inputBuffer[i+1] == '\n' ) {
        fullLine = 1;
        break;
      }
    }

    if ( fullLine ) {

      if ( client->eatUntilNextCrLf ) {
        client->inputBufferIndex = 0;
        client->eatUntilNextCrLf = 0;
        return;
      }

      // Reset for the next read.  Get rid of CR/LF too.
      client->inputBuffer[client->inputBufferIndex-2] = 0;
      client->inputBufferIndex = 0;
    }
    else {

      // Need to read some more.  But before we try again, check to make
      // sure that there is room in the buffer

      if ( client->inputBufferIndex == INPUTBUFFER_SIZE ) {
        TRACE_WARN(( "Ftp (%lu) Input buffer overflow on control socket\n", client->sessionId ));
        client->addToOutput_var( Msg_500_Syntax_Error_v, "Line too long" );
        client->inputBufferIndex = 0;
        client->eatUntilNextCrLf = 1;
      }

      // Read some more, picking up where we left off
      return;
    }
  }



  // By this point we have a full line of input.

  TRACE(( "Ftp (%lu) State: %d  Input from %d.%d.%d.%d:%u: %s\n",
    client->sessionId, client->state,
    client->cs->dstHost[0], client->cs->dstHost[1],
    client->cs->dstHost[2], client->cs->dstHost[3],
    client->cs->dstPort, client->inputBuffer ));


  // If the first char is a Telnet IAC then we should interpret the sequence.
  //
  // Unix does things correctly by sending IAC before each telnet command.
  // Windows XP appears not to.  Be sloppy - if the first char is IAC then
  // assume the ABOR is coming sometime later ..

  if ( client->inputBuffer[0] == TEL_IAC ) {
    int i;
    for ( i=0; i < strlen(client->inputBuffer); i++ ) {
      if ( client->inputBuffer[i] < 128 ) break;
    }

    memmove( client->inputBuffer, client->inputBuffer+i, (strlen(client->inputBuffer)-i)+1 );

    TRACE(( "TEL_IAC detected: removed %u chars, cmd is now: %s---\n", i, client->inputBuffer ));
  }



  char command[COMMAND_MAX_LEN];
  char *nextTokenPtr = Utils::getNextToken( client->inputBuffer, command, COMMAND_MAX_LEN );

  if ( *command == 0 ) return;

  strupr( command );


  switch ( client->state ) {

    case FtpClient::UserPrompt: {

      if ( strcmp(command, "USER") == 0 ) {

        client->loginAttempts++;

        if ( client->loginAttempts > 3 ) {
          // Disconnect them for security.
          endSession( client );
          break;
        }

        char tmpUserName[USERNAME_LEN];
        Utils::getNextToken( nextTokenPtr, tmpUserName, USERNAME_LEN );
        if ( tmpUserName[0] ) {

          // Lookup in pw file
          int rc = FtpUser::getUserRec( tmpUserName, &client->user );

          if ( rc == 1 ) {

            // Send password prompt

            if ( strcmp( client->user.userPass, "[email]" ) == 0 ) {
              client->addToOutput( "331 Anonymous ok, send your email addr as the password" _NL_ );
            }
            else {
              client->addToOutput( Msg_331_UserOkSendPass );
            }

            client->state = FtpClient::PasswordPrompt;

          }
          else {
            client->addToOutput( "530 I dont like your name" _NL_ );
            addToLog( "(%lu) Bad userid: %s\n", client->sessionId, tmpUserName );
          }

        }
        else {
          // Missing parm
          client->addToOutput_var( Msg_500_Parm_Missing_v, "USER" );
        }

      }
      else {
        // Bogus command
        client->addToOutput( Msg_530_Please_Login );
      }

      break;
    }

    case FtpClient::PasswordPrompt: {

      if ( strcmp(command, "PASS") == 0 ) {

        // tmpPassword has to be long enough to accomodate reasonable
        // email addresses.

        char tmpPassword[50];
        Utils::getNextToken( nextTokenPtr, tmpPassword, 50 );

        if ( tmpPassword[0] ) {

          // Check password here

          if ( strcmp( client->user.userPass, "[email]" ) == 0 ) {
            addToLog( "(%lu) Anon user: %s, email: %s from %d.%d.%d.%d:%u\n",
                      client->sessionId, client->user.userName, tmpPassword,
                      client->cs->dstHost[0],  client->cs->dstHost[1],
                      client->cs->dstHost[2],  client->cs->dstHost[3],
                      client->cs->dstPort );
          }
          else {

            if ( strcmp( client->user.userPass, tmpPassword ) != 0 ) {
              client->addToOutput( "530 Bad password" _NL_ );
              client->state = FtpClient::UserPrompt;
              addToLog( "(%lu) Failed password attempt user %s at %d.%d.%d.%d:%u\n",
                        client->sessionId, client->user.userName,
                        client->cs->dstHost[0],  client->cs->dstHost[1], 
                        client->cs->dstHost[2],  client->cs->dstHost[3],
                        client->cs->dstPort );
              break;
            }

            addToLog( "(%lu) User %s is signed in from %d.%d.%d.%d:%u\n",
                      client->sessionId, client->user.userName,
                      client->cs->dstHost[0],  client->cs->dstHost[1],
                      client->cs->dstHost[2],  client->cs->dstHost[3],
                      client->cs->dstPort );
          }

          // Ok, tell them connected
          client->addToOutput( Msg_230_User_Logged_In );
          client->state = FtpClient::CommandLine;

          if ( Sound ) { sound(500); delay(100); sound(1000); delay(100); nosound( ); }

          // Per user housekeeping to do

          if ( strcmp( client->user.sandbox, "[NONE]" ) == 0 ) {
            // No sandbox - this user gets DOS style directory paths
            client->ftproot[0] = 0;
            client->loggedDrive = _getdrive( ) + 'A' - 1;
            client->cwd[0] = client->loggedDrive; client->cwd[1] = ':';
            client->cwd[2] = '\\'; client->cwd[3] = 0;
          }
          else {
            // Sandbox - this user gets Unix style directory paths
            strcpy( client->ftproot, client->user.sandbox );
            client->loggedDrive = 0;
            client->cwd[0] = '/'; client->cwd[1] = 0;
          }

        }
        else {
          // No password
          client->addToOutput( Msg_530_Login_Incorrect );
          client->state = FtpClient::UserPrompt;

        }
      }
      else {
        // Bogus command
        client->addToOutput( Msg_530_Please_Login );
        client->state = FtpClient::UserPrompt;
      }

      break;
    }

    case FtpClient::RnfrSent: {

      // Going back to command line no matter what.
      client->state = FtpClient::CommandLine;

      if ( strcmp(command, "RNTO") == 0 ) {
        doRnto( client, nextTokenPtr );
        break;
      }

      // Fall through to CommandLine if it wasn't RNTO.  Not terribly valid
      // but we will tolerate it.
    }

    case FtpClient::CommandLine: {

      if ( strcmp(command, "QUIT") == 0 ) {

        // We really want this to make it out before the socket closes,
        // but we are not going to make an extraordinary effort to do it.
        client->addToOutput( Msg_221_Closing );
        client->sendOutput( );

        endSession( client );
      }


      // Path related
      else if ( strcmp(command, "DELE") == 0 ) { doDele( client, nextTokenPtr ); }
      else if ( strcmp(command, "RNFR") == 0 ) { doRnfr( client, nextTokenPtr ); }
      else if ( strcmp(command, "RNTO") == 0 ) { client->addToOutput( Msg_503_Send_RNFR_first ); }
      else if ( strcmp(command, "CWD" ) == 0 || strcmp(command, "XCWD") == 0 ) { doCwd( client, nextTokenPtr ); }
      else if ( strcmp(command, "CDUP") == 0 || strcmp(command, "XCUP") == 0 ) { doCwd( client, ".." ); }
      else if ( strcmp(command, "PWD" ) == 0 || strcmp(command, "XPWD") == 0 ) { client->addToOutput_var( "257 \"%s\" is current directory" _NL_, client->cwd ); }
      else if ( strcmp(command, "MKD" ) == 0 || strcmp(command, "XMKD") == 0 ) { doMkd( client, nextTokenPtr ); }
      else if ( strcmp(command, "RMD" ) == 0 || strcmp(command, "XRMD") == 0 ) { doRmd( client, nextTokenPtr ); }

      // Data transfer
      else if ( strcmp(command, "PASV") == 0 ) { doPasv( client ); }
      else if ( strcmp(command, "PORT") == 0 ) { doPort( client, nextTokenPtr ); }
      else if ( strcmp(command, "ABOR") == 0 ) { doAbort( client ); }
      else if ( strcmp(command, "LIST") == 0 ) { doXfer( client, nextTokenPtr, FtpClient::List ); }
      else if ( strcmp(command, "NLST") == 0 ) { doXfer( client, nextTokenPtr, FtpClient::Nlist ); }
      else if ( strcmp(command, "RETR") == 0 ) { doXfer( client, nextTokenPtr, FtpClient::Retr ); }
      else if ( strcmp(command, "STOR") == 0 ) { doXfer( client, nextTokenPtr, FtpClient::Stor ); }
      else if ( strcmp(command, "APPE") == 0 ) { doXfer( client, nextTokenPtr, FtpClient::StorA ); }
      else if ( strcmp(command, "STOU") == 0 ) { doXfer( client, nextTokenPtr, FtpClient::StorU ); }

      // Environment selection
      else if ( strcmp(command, "MODE") == 0 ) { doMode( client, nextTokenPtr ); }
      else if ( strcmp(command, "STRU") == 0 ) { doStru( client, nextTokenPtr ); }
      else if ( strcmp(command, "TYPE") == 0 ) { doType( client, nextTokenPtr ); }

      // Misc
      else if ( strcmp(command, "HELP") == 0 ) { doHelp( client ); }
      else if ( strcmp(command, "ALLO") == 0 ) { client->addToOutput( Msg_202_No_Alloc_needed ); }
      else if ( strcmp(command, "FEAT") == 0 ) { client->addToOutput( "211-mTCP FTP server features:" _NL_ " MDTM" _NL_ "211 End" _NL_ ); }
      else if ( strcmp(command, "MDTM") == 0 ) { doMdtm( client, nextTokenPtr ); }
      else if ( strcmp(command, "NOOP") == 0 ) { client->addToOutput( Msg_200_Noop_OK ); }
      else if ( strcmp(command, "STAT") == 0 ) { doStat( client, nextTokenPtr ); }
      else if ( strcmp(command, "SYST") == 0 ) { client->addToOutput( Msg_215_System_Type ); }

      else if ( strcmp(command, "SITE") == 0 ) { doSite( client, nextTokenPtr ); }

      else if ( (strcmp(command, "USER") == 0) || (strcmp(command, "PASS") == 0) ) {
        client->addToOutput( Msg_503_Already_Logged_In );
      }

      else if ( (strcmp(command, "REIN") == 0) || (strcmp(command, "ACCT") == 0) || (strcmp(command, "REST") == 0) )
      {
        client->addToOutput( Msg_502_Not_Implemented );
      }

      else {
        client->addToOutput_var( Msg_500_Syntax_Error_v, command );
        TRACE_WARN(( "Ftp: unknown command: %s\n", client->inputBuffer ));
      }

      break;
    }


  } // end switch 


}



void doHelp( FtpClient *client ) {

  int i=0;
  while ( Msg_214_Help[i] != NULL ) {
    client->addToOutput( Msg_214_Help[i] );
    i++;
  }

}



// doStat
//
// With no parameters it just returns some basic status.
//
// If given a parameter it does a directory list on the parameter.  Unlike
// the standard directory list, all output flows back over the control
// connection.  I expect this to be a fairly rare usage of the command
// so it's not optimized for speed in anyway.

void doStat( FtpClient *client, char *nextTokenPtr ) {

  // If this is the first time here parse the input.  If there is a
  // parameter setup to start sending directory entries back.

  if ( client->statCmdActive == 0 ) {

    char parm[ DOS_MAX_PATH_LENGTH ];
    Utils::getNextToken( nextTokenPtr, parm, DOS_MAX_PATH_LENGTH );

    if ( parm[0] == 0 ) {
      client->addToOutput_var( "211-Status of mTCP FTP Server" _NL_ " Logged in as %s" _NL_, client->user.userName );
      if ( client->ds ) {
        client->addToOutput( " Active data connection" _NL_ );
      }
      else {
        client->addToOutput( " No active data connection" _NL_ );
      }
      client->addToOutput_var( " Type: %s Structure: File, Mode: Stream" _NL_, (client->asciiMode?"ASCII":"IMAGE") );
      client->addToOutput( "211 End of status" _NL_ );
      return;
    }


    // This is going to be longer than we thought.
    client->statCmdActive = 1;

    client->addToOutput_var( "211-Status of %s" _NL_, parm );

    char fullpath[DOS_MAX_PATHFILE_LENGTH];
    if ( formFullPath( client, fullpath, DOS_MAX_PATHFILE_LENGTH, parm ) ) {
      client->addToOutput( "211 End of Status" _NL_ );
      client->statCmdActive = 0;
      return;
    }

    // Stat it.  If it is a directory then add the *.* to the end.
    // If it's not valid don't worry about it - they will get an empty
    // listing.

    if ( strlen(fullpath) < DOS_MAX_PATH_LENGTH ) {
      if ( isDirectory( fullpath ) ) {
        strcat( fullpath, "\\*.*" );
      }
    }

    client->noMoreData = _dos_findfirst( fullpath, (_A_NORMAL | _A_SUBDIR), &client->fileinfo);

    if ( client->noMoreData ) {
      _dos_findclose( &client->fileinfo );
      client->addToOutput( "211 End of Status" _NL_ );
      client->statCmdActive = 0;
    }


    // Return from this function without doing any real work; not terribly
    // efficient but it cuts down on code duplication.  We will pick up where
    // we left off on the next call to this code.
  }


  else {

    // We don't get here if the client isn't done sending previously queued
    // data.  We also don't care too much about the performance of this
    // command, so keep things simple and just send one line at a time.

    // Fixme: do a small optimization by doing two lines of output at a time.

    // Format file attributes
    char attrs[] = "-rwxrwxrwx"; // Default
    if ( client->fileinfo.attrib & _A_SUBDIR ) { attrs[0] = 'd'; }
    if ( client->fileinfo.attrib & _A_RDONLY ) { attrs[2] = attrs[5] = attrs[8] = '-'; }

    ftime_t ft;
    ft.us = client->fileinfo.wr_time;

    fdate_t fd;
    fd.us = client->fileinfo.wr_date;

    if ( fd.fields.year + 1980 != Current_Year ) {
      client->addToOutput_var( " %s 1 ftp ftp %10lu %s %2d  %4d %s" _NL_,
                 attrs, client->fileinfo.size, Months[fd.fields.month-1],
                 fd.fields.day, (fd.fields.year + 1980),
                 client->fileinfo.name );
    }
    else {
      client->addToOutput_var( " %s 1 ftp ftp %10lu %s %2d %02d:%02d %s" _NL_,
                 attrs, client->fileinfo.size, Months[fd.fields.month-1],
                 fd.fields.day, ft.fields.hours, ft.fields.minutes,
                 client->fileinfo.name );
    }

    if ( (client->noMoreData = _dos_findnext( &client->fileinfo )) ) {
      _dos_findclose( &client->fileinfo );
      client->addToOutput( "211 End of Status" _NL_ );
      client->statCmdActive = 0;
    }

  }

}


void doSite( FtpClient *client, char *nextTokenPtr ) {

  char siteCmd[ 10 ];
  nextTokenPtr = Utils::getNextToken( nextTokenPtr, siteCmd, 10 );

  if ( stricmp( siteCmd, "stats" ) == 0 ) {
    doSiteStats( client );
  }
  else if ( stricmp( siteCmd, "who" ) == 0 ) {
    doSiteWho( client );
  }
  else if ( stricmp( siteCmd, "help" ) == 0 ) {
    client->addToOutput( "211 Site commands: HELP DISKFREE STATS WHO" _NL_ );
  }
  else if ( stricmp( siteCmd, "diskfree" ) == 0 ) {
    doSiteDiskFree( client, nextTokenPtr );
  }
  else {
    client->addToOutput( "500 Unknown SITE command" _NL_ );
  }

}



void doSiteStats( FtpClient *client ) {

  client->addToOutput_var(
    "211-Stats: Started: %s" _NL_ " Sessions: %lu  Active: %u  Timeouts: %lu" _NL_,
    StartTime, SessionCounter, FtpClient::activeClients, Stat_SessionTimeouts
  );

  client->addToOutput_var(
    " LIST: %lu  NLST: %lu  RETR: %lu" _NL_ " STOR: %lu  STOU: %lu  APPE: %lu" _NL_,
    Stat_LIST, Stat_NLST, Stat_RETR, Stat_STOR, Stat_STOU, Stat_APPE
  );

  client->addToOutput_var( " Tcp Sockets used: %d free: %d" _NL_,
    TcpSocketMgr::getActiveSockets( ), TcpSocketMgr::getFreeSockets( )
  );

  client->addToOutput_var(
    " Tcp: Sent %lu Rcvd %lu Retrans %lu Seq/Ack errs %lu Dropped %lu" _NL_,
    Tcp::Packets_Sent, Tcp::Packets_Received, Tcp::Packets_Retransmitted,
    Tcp::Packets_SeqOrAckError, Tcp::Packets_DroppedNoSpace
  );

  client->addToOutput_var(
    " Packets: Sent: %lu Rcvd: %lu Dropped: %lu LowFreeBufCount: %u" _NL_,
    Packets_sent, Packets_received, Packets_dropped, Buffer_lowFreeCount
  );

  client->addToOutput( "211 OK" _NL_ );

}


void doSiteWho( FtpClient *client ) {

  client->addToOutput("200- Online users" _NL_ " UserId     IpAddr:port" _NL_ );

  for ( uint16_t i=0; i < FtpClient::activeClients; i++ ) {

    FtpClient *tmpClient = FtpClient::activeClientsTable[i];

    if ( tmpClient->state != FtpClient::Closed ) {

      client->addToOutput_var( " %-10s %d.%d.%d.%d:%u" _NL_,
        tmpClient->user.userName,
        tmpClient->cs->dstHost[0], tmpClient->cs->dstHost[1],
        tmpClient->cs->dstHost[2], tmpClient->cs->dstHost[3],
        tmpClient->cs->dstPort );

    }

  }

  client->addToOutput("200 OK" _NL_ );

}

void doSiteDiskFree( FtpClient *client, char *nextTokenPtr ) {

  char driveLetter[2];
  Utils::getNextToken( nextTokenPtr, driveLetter, 2 );

  if ( driveLetter[0] == 0 ) {
    client->addToOutput( "211 Please specify a drive letter" _NL_ );
    return;
  }

  if ( islower( driveLetter[0] ) ) {
    driveLetter[0] = toupper( driveLetter[0] );
  }

  if ( !isupper( driveLetter[0] ) ) {
    client->addToOutput( "211 Bad drive letter" _NL_ );
    return;
  }

  int dl = driveLetter[0] - 'A' + 1;

  if ( dl == 1 || dl == 2 ) {
    client->addToOutput_var( "211 Sorry, not going to touch floppy drives" _NL_ );
    return;
  }

  struct diskfree_t disk_data;

  if ( _dos_getdiskfree( dl, &disk_data ) == 0 ) {
    uint32_t freeSpace = (uint32_t)disk_data.avail_clusters *
                         (uint32_t)disk_data.sectors_per_cluster *
                         (uint32_t)disk_data.bytes_per_sector;

    client->addToOutput_var( "211 Disk %s has %lu free bytes" _NL_, driveLetter, freeSpace );
  }
  else {
    client->addToOutput_var( "211 Error reading free space on Disk %s" _NL_, driveLetter );
  }

}




void doType( FtpClient *client, char *nextTokenPtr ) {

  char datatype[20];
  Utils::getNextToken( nextTokenPtr, datatype, 20 );

  if ( *datatype == 0 ) {
    client->addToOutput_var( Msg_500_Parm_Missing_v, "TYPE" );
    return;
  }

  if ( *datatype == 'a' || *datatype == 'A' ) {
    client->asciiMode = 1;
    client->addToOutput( "200 Type set to A" _NL_ );
  }
  else if ( *datatype == 'i' || *datatype == 'I' ) {
    client->asciiMode = 0;
    client->addToOutput( "200 Type set to I" _NL_ );
  }
  else {
    client->addToOutput_var( "500 TYPE %s not understood or supported" _NL_, datatype );
  }

}


void doStru( FtpClient *client, char *nextTokenPtr ) {

  char struType[20];
  Utils::getNextToken( nextTokenPtr, struType, 20 );

  if ( *struType == 0 ) {
    client->addToOutput_var( Msg_500_Parm_Missing_v, "STRU" );
    return;
  }

  if ( *struType == 'f' || *struType == 'F' ) {
    client->addToOutput( "200 STRU set to F" _NL_ );
  }
  else if ( *struType == 'r' || *struType == 'R' ||  *struType == 'p' || *struType == 'P' ) {
    client->addToOutput_var( Msg_504_Unsupp_Option_v, "STRU", struType );
  }
  else {
    client->addToOutput_var( Msg_501_Unknown_Option_v, "STRU", struType );
  }

}



void doMode( FtpClient *client, char *nextTokenPtr ) {

  char modeType[20];
  Utils::getNextToken( nextTokenPtr, modeType, 20 );

  if ( *modeType == 0 ) {
    client->addToOutput_var( Msg_500_Parm_Missing_v, "MODE" );
    return;
  }

  if ( *modeType == 's' || *modeType == 'S' ) {
    client->addToOutput( "200 MODE set to S" _NL_ );
  }
  else if ( *modeType == 'b' || *modeType == 'B' ||  *modeType == 'c' || *modeType == 'C' ) {
    client->addToOutput_var( Msg_504_Unsupp_Option_v, "MODE", modeType );
  }
  else {
    client->addToOutput_var( Msg_501_Unknown_Option_v, "MODE", modeType );
  }

}





void doPort( FtpClient *client, char *nextTokenPtr ) {

  // If we have a data transfer going already don't honor a PORT command.
  // PORT isn't much of a problem - it just caches information for the next
  // command which is probably a data transfer.  But it's possible that they
  // had done PASV even connected (but not started transfering data) and we
  // want to clean up the data socket in preparation for another transfer.

  if ( client->dataXferState != FtpClient::DL_NotActive ) {
    TRACE_WARN(( "Ftp (%lu) doPort: Transfer already in progress\n", client->sessionId ));
    client->addToOutput( Msg_425_Transfer_In_Progress );
    return;
  }


  // Ok, no transfers in progress at the moment.  If PASV had been used and
  // we had a listening socket open we need to close it.  If the user had
  // also connected the data socket (but not started a transfer), then
  // kill the data socket too.  This prevents us from losing the socket later.

  if ( client->ls != NULL ) {
    TRACE(( "Ftp (%lu) PORT command supercedes PASV, closing listening socket\n", client->sessionId ));
    client->ls->close( );
    TcpSocketMgr::freeSocket( client->ls );
    client->ls = NULL;
  }

  if ( client->ds != NULL ) {
    // This is an error.  No active data transfer, so it's safe to just hit
    // it over the head.
    TRACE_WARN(( "Ftp (%lu) doPort: Closing data connection that was never used\n", client->sessionId ));
    client->ds->close( );
    client->ds = NULL;
  }

  uint16_t t0, t1, t2, t3, t4, t5;

  int rc = sscanf( nextTokenPtr, "%d,%d,%d,%d,%d,%d",  &t0, &t1, &t2, &t3, &t4, &t5 );

  if ( rc != 6 ) {
    client->addToOutput( "501 Illegal PORT command" _NL_ );
    return;
  }


  client->dataTarget[0] = t0;  client->dataTarget[1] = t1;  client->dataTarget[2] = t2;  client->dataTarget[3] = t3;
  client->dataPort = (t4<<8) + t5;

  // Ok, tell them we liked that
  client->addToOutput( Msg_200_Port_OK );
}


void doPasv( FtpClient *client ) {

  // If we have a data transfers going already don't honor a PASV command.
  // This probably never happens but we can lose sockets if we start listening
  // for a socket when one is already open.

  if ( client->dataXferState != FtpClient::DL_NotActive ) {
    client->addToOutput( Msg_425_Transfer_In_Progress );
    return;
  }


  // Ok, no data transfers were active.  If we were listening because of a
  // previous PASV command then close that socket.

  // If we have a listening socket already close it.  
  if ( client->ls != NULL ) {
    TRACE(( "Ftp (%lu) Closing previously opened listening socket\n", client->sessionId ));
    client->ls->close( );
    TcpSocketMgr::freeSocket( client->ls );
    client->ls = NULL;
  }

  if ( client->ds != NULL ) {
    // This is an error.  Force it closed before taking a new one.
    TRACE_WARN(( "Ftp (%lu) doPasv: Closing data connection that was never used\n", client->sessionId ));
    client->ds->close( );
    client->ds = NULL;
  }

        
  // Open a listening socket immediately, even before pushing a response
  // on the control connection.  This prevents timing problems; the
  // client might be very fast to open the data connection.

  client->ls = TcpSocketMgr::getSocket( );
        
  if ( client->ls == NULL ) {
    TRACE_WARN(( "Ftp (%lu) Could not get listening socket for PASV\n", client->sessionId ));
    client->addToOutput( Msg_425_Cant_Open_Conn );
    return;
  }


  client->pasvPort = (rand( ) % Pasv_Ports) + Pasv_Base;

  uint16_t hiByte = client->pasvPort / 256;
  uint16_t loByte = client->pasvPort - hiByte*256;


  // Fixme: check the return code, we might have a collsion on a port.
  if ( client->ls->listen( client->pasvPort, DATA_RCV_BUF_SIZE ) ) {
    client->addToOutput( Msg_425_Cant_Open_Conn );
    return;
  }

  client->addToOutput_var( "227 Entering Passive Mode (%d,%d,%d,%d,%d,%d)" _NL_,
           client->pasvAddr[0], client->pasvAddr[1], client->pasvAddr[2], client->pasvAddr[3], hiByte, loByte );
        
  TRACE(( "Ftp (%lu) Waiting for data connection on %u\n", client->sessionId, client->pasvPort ));

}



// RFC 3659
//
void doMdtm( FtpClient *client, char *nextTokenPtr ) {

  // Form the full path first.
  char userPart[DOS_MAX_PATHFILE_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATHFILE_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATHFILE_LENGTH, userPart ) ) {
    client->addToOutput( Msg_550_Bad_Path_Or_File );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  if ( rc == 0 ) {

    // Ok, it exists

    if ( S_ISREG(statbuf.st_mode) ) {

      ftime_t ft;
      ft.us = client->fileinfo.wr_time;

      fdate_t fd;
      fd.us = client->fileinfo.wr_date;

      client->addToOutput_var( "213 %4d%02d%02d%02d%02d%02d" _NL_,
                               fd.fields.year + 1980, fd.fields.month, fd.fields.day,
                               ft.fields.hours, ft.fields.minutes, ft.fields.twosecs<1 );
    }
    else {
      client->addToOutput_var( "550 %s: not a plain file" _NL_, userPart );
    }

  }
  else {
    client->addToOutput_var( "550 %s: bad file" _NL_, userPart );
  }

}


void doDele( FtpClient *client, char *nextTokenPtr ) {

  // Permission check
  if ( client->user.cmd_DELE == 0 ) {
    client->addToOutput( "550 permission denied" _NL_ );
    return;
  }

  // Form the full path first.
  char userPart[DOS_MAX_PATHFILE_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATHFILE_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATHFILE_LENGTH, userPart ) ) {
    client->addToOutput( "550 Bad file" _NL_ );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  if ( rc == 0 ) {

    // Ok, it exists

    if ( S_ISREG(statbuf.st_mode) ) {
      if ( unlink( fullpath ) ) {
        client->addToOutput_var( "550 Error removing %s" _NL_, userPart );
      }
      else {
        client->addToOutput( "250 DELE command successful" _NL_ );
        addToLog( "(%lu) DELE %s\n", client->sessionId, fullpath );
      }
    }
    else {
      client->addToOutput_var( "550 %s: not a plain file" _NL_, userPart );
    }

  }
  else {
    client->addToOutput_var( "550 %s: bad file" _NL_, userPart );
  }

}




void doRmd( FtpClient *client, char *nextTokenPtr ) {

  // Permission check
  if ( client->user.cmd_RMD == 0 ) {
    client->addToOutput( "550 permission denied" _NL_ );
    return;
  }

  // Form the full path first.
  char userPart[DOS_MAX_PATH_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATH_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATH_LENGTH, userPart ) ) {
    client->addToOutput( "550 Bad path" _NL_ );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  if ( rc == 0 ) {

    // Ok, it exists

    if ( S_ISDIR(statbuf.st_mode) ) {
      if ( rmdir( fullpath ) ) {
        client->addToOutput_var( "550 Error removing %s" _NL_, userPart );
      }
      else {
        client->addToOutput( "250 RMD command successful" _NL_ );
        addToLog( "(%lu) RMD %s\n", client->sessionId, fullpath );
      }
    }
    else {
      client->addToOutput_var( "550 %s: not a directory" _NL_, userPart );
    }

  }
  else {
    client->addToOutput_var( "550 %s: bad file" _NL_, userPart );
  }

}

void doMkd( FtpClient *client, char *nextTokenPtr ) {

  // Permission check
  if ( client->user.cmd_MKD == 0 ) {
    client->addToOutput( "550 permission denied" _NL_ );
    return;
  }

  // Form the full path first.
  char userPart[DOS_MAX_PATH_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATH_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATH_LENGTH, userPart ) ) {
    client->addToOutput( "550 Bad path" _NL_ );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  int ftpRootLen = strlen(client->ftproot);
  if ( ftpRootLen ) ftpRootLen++;

  if ( rc != 0 ) {

    // Does not exist yet

    if ( mkdir( fullpath ) ) {
      client->addToOutput_var( "550 Error creating %s" _NL_, (fullpath+ftpRootLen) );
    }
    else {
      client->addToOutput_var( "257 %s created" _NL_, (fullpath+ftpRootLen) );
      addToLog( "(%lu) MKD %s\n", client->sessionId, fullpath );
    }

  }
  else {
    client->addToOutput_var( "550 %s: already exists" _NL_, (fullpath+ftpRootLen) );
  }

}




void doRnfr( FtpClient *client, char *nextTokenPtr ) {

  // Permission check
  if ( client->user.cmd_RNFR == 0 ) {
    client->addToOutput( "550 permission denied" _NL_ );
    return;
  }

  // Form the full path first.
  char userPart[DOS_MAX_PATHFILE_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATHFILE_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATHFILE_LENGTH, userPart ) ) {
    client->addToOutput( "550 Bad path or file" _NL_ );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  if ( rc == 0 ) {
    // Ok, it exists
    strcpy( client->filespec, fullpath );
    client->addToOutput( "350 File or directory exists, ready for destination name" _NL_ );
    client->state = FtpClient::RnfrSent;
  }
  else {
    client->addToOutput_var( "550 %s: bad file or directory" _NL_, userPart );
  }

}



void doRnto( FtpClient *client, char *nextTokenPtr ) {

  // Form the full path first.
  char userPart[DOS_MAX_PATHFILE_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATHFILE_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATHFILE_LENGTH, userPart ) ) {
    client->addToOutput( "550 Bad path or file" _NL_ );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  if ( rc != 0 ) {

    // Good, it does not exist yet

    if ( rename( client->filespec, fullpath ) == 0 ) {
      addToLog( "(%lu) RNTO %s to %s\n", client->sessionId, client->filespec, fullpath );
      client->addToOutput( "250 Rename successful" _NL_ );
    }
    else {
      client->addToOutput( "550 Rename failed" _NL_ );
    }

  }
  else {
    client->addToOutput_var( "550 %s: already exists" _NL_, userPart );
  }

}







// RFC 3659
//
// Might need to remove this because we really don't want to scan files
// to see how they are going to change when we do ASCII vs. BIN transfers.

void doSize( FtpClient *client, char *nextTokenPtr ) {

  // Form the full path first.
  char userPart[DOS_MAX_PATHFILE_LENGTH];
  Utils::getNextToken( nextTokenPtr, userPart, DOS_MAX_PATHFILE_LENGTH );

  if ( *userPart == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  char fullpath[DOS_MAX_PATHFILE_LENGTH];
  if ( formFullPath( client, fullpath, DOS_MAX_PATHFILE_LENGTH, userPart ) ) {
    client->addToOutput( "550 Bad file" _NL_ );
    return;
  }

  struct stat statbuf;
  int rc = stat( fullpath, &statbuf );

  if ( rc == 0 ) {

    // Ok, it's a file.

    if ( S_ISREG(statbuf.st_mode) ) {
      client->addToOutput_var( "213 %lu" _NL_, statbuf.st_size );
    }                          
    else {
      client->addToOutput_var( "550 %s: not a plain file" _NL_, userPart );
    } 
    
  }
  else {
    client->addToOutput_var( "550 %s: bad file" _NL_, userPart );
  }
  
}




// DOS has a path length maximum of 67 chars in the following format
//
//   1 + 1 + 63 + 1 + 1
// drive_letter + colon + path + / + null
//
// We need to leave room for a last / to be able to append a filename
// onto a filespec.
//
// It's possible that the FTPROOT is NULL so that would leave the full
// path available.


// If not in sandbox ..
//
// - The CWD always starts with a drive leter and / (an absolute path).
//
// - If they use a drive letter they have to use an absolute path.
//   We don't remember anything except the path for the current drive.
//
// - If they use a path starting with a / then be nice and put the
//   current drive letter in front of it.
//
// - If they give us anything else it is a relative path; it gets
//   appended to the CWD.

// If in the sandbox ..
//
// - Their CWD always starts with / (an absolute path).
//
// - Don't help them by putting a drive letter in front of a /.


void doCwd( FtpClient *client, char *nextTokenPtr ) {

  char parm[ DOS_MAX_PATH_LENGTH ];
  Utils::getNextToken( nextTokenPtr, parm, DOS_MAX_PATH_LENGTH );
  strupr( parm );

  if ( parm[0] == 0 ) {
    client->addToOutput( "501 Invalid number of arguments" _NL_ );
    return;
  }

  uint8_t isSandbox = ( client->ftproot[0] != 0 );
  uint8_t startsWithDrive = startsWithDriveLetter( parm );
  uint8_t isAbs = isAbsolute( parm );


  // Length should be DOS_MAX_PATH_LENGTH, but we need to leave a little
  // extra in case they have a full path and want to use a ".." to back up.
  // We will ensure it is small enough for DOS later.

  char newpath[ DOS_MAX_PATHFILE_LENGTH + 20 ];
  newpath[0] = 0;

  if ( isSandbox ) {
    if ( startsWithDrive ) {
      client->addToOutput_var( "550 No drive letters for you!" _NL_, parm );
      return;
    }
  }
  else {
    if ( startsWithDrive ) {
      if ( !isAbs ) {
        client->addToOutput_var( "550 If you use a drive letter you must use absolute paths" _NL_, parm );
        return;
      }
    }
    else {
      if ( isAbs ) {
        // Started with a '/' but no drive letter.  Put it in there for them to make processing easier down the road.
        newpath[0] = client->loggedDrive; newpath[1] = ':'; newpath[2] = 0;
      }
    }
  }


  // If not absolute prepend the current working directory.
  if ( !isAbs ) {
    strcat( newpath, client->cwd );
  }


  if ( strlen(newpath) + strlen(parm) > (DOS_MAX_PATHFILE_LENGTH+20-1) ) {
    client->addToOutput( "550 Path too long" _NL_ );
    return;
  }

  // Now add the user input
  strcat( newpath, parm );


  // By this point we have the full path as the user sees it.  If the user
  // wasn't in a sandbox and missed the drive letter we were nice and gave
  // it to them.
  //
  // Now validate it and parse out any . or .. components in the path.
  // This keeps a user in a sandbox from backing up too far out of their
  // sandbox.

  if ( normalizeDir( newpath, DOS_MAX_PATHFILE_LENGTH ) ) {
    client->addToOutput_var( "550 \"%s\": Bad path" _NL_, parm );
    return;
  }


  // Ok, it's a sane path at least.  If the user is in a sandbox we need
  // to prepend the root directory.

  char fullpath[DOS_MAX_PATH_LENGTH];
  fullpath[0] = 0;

  // If the user is in a sandbox prepend the sandbox directory.
  if ( isSandbox ) {
    strcpy( fullpath, client->ftproot );
  }


  // If the path is too long throw an error.  Silently truncating it is
  // going to lead to user confusion.  We need to ensure there is room
  // for a trailing '/' and a terminating null.

  if ( strlen(fullpath) + strlen(newpath) > (DOS_MAX_PATH_LENGTH-2) ) {
    client->addToOutput( "550 Path too long" _NL_ );
    return;
  }

  strcat( fullpath, newpath );


  // Convert unix '/' style delimeters to DOS style.  Strip off any trailing
  // delimeters too.
  convertToDosForm( fullpath );

  if ( isDirectory( fullpath ) ) {

    // Their input was good.  Remember it as the new path.

    if ( !isSandbox ) {
      // Remember logged drive
      client->loggedDrive = fullpath[0];

      // Might have added the drive letter to the front, so use fullpath not newpath
      // This will expose the DOS backslashes that we use under the covers.
      strcpy( client->cwd, fullpath );
      convertForDisplay( 0, client->cwd );

      // This is a directory so make sure it ends in a trailing \.
      // We left room for it above. when we called convertToDosForm
      if ( client->cwd[ strlen(client->cwd)-1 ] != '\\' ) { strcat( client->cwd, "\\" ); }
    }
    else {
      strcpy( client->cwd, newpath );

      convertForDisplay( 1, client->cwd );

      // This is a directory so make sure it ends in a trailing /.
      // We left room for it above. when we called convertToDosForm
      if ( client->cwd[ strlen(client->cwd)-1 ] != '/' ) { strcat( client->cwd, "/" ); }
    }

    client->addToOutput( "250 CWD command successful" _NL_ );

  }
  else {
    client->addToOutput_var( "550 \"%s\": No such directory" _NL_, parm );
  }

}






static void cleanupDataXferStructs( FtpClient *client ) {

  switch ( client->dataXferType ) {

    case FtpClient::List:
    case FtpClient::Nlist: {
      _dos_findclose( &client->fileinfo );
      break;
    }

    case FtpClient::Retr:
    case FtpClient::Stor:
    case FtpClient::StorA:
    case FtpClient::StorU: {
      if (client->file != NULL) {
        fclose( client->file );
        client->file = NULL;
      }
      break;
    }
  }

  client->dataXferType = FtpClient::NoDataXfer;
}




// formFullPath
//
// Given a current working directory and a filespec form the full path for
// the filespec.  If the filespec was absolute or included ".." directories
// those get handled too.
// 
// Returns 0 if the full path and filespec can be constructed.  This doesn't
// mean it is a valid file or directory; only that it is a legal path and
// filespec.
//
// Returns:
//
//   1 - illegal use of drive letters
//   2 - need to use absolute path with a drive letter
//   3 - path too long
//   4 - syntax error in the path

int formFullPath( FtpClient *client, char *outBuffer, int outBufferLen, char *filespec ) {

  uint8_t isSandbox = ( client->ftproot[0] != 0 );
  uint8_t startsWithDrive = startsWithDriveLetter( filespec );
  uint8_t isAbs = isAbsolute( filespec );

  // Back this a little bigger than max to accomodate things like ".."
  char newpath[ DOS_MAX_PATHFILE_LENGTH + 20 ];
  newpath[0] = 0;

  if ( isSandbox ) {
    if ( startsWithDrive ) {
      return 1;
    }
  }
  else {
    if ( startsWithDrive ) {
      if ( !isAbs ) {
        return 2;
      }
    }
    else {
      if ( isAbs ) {
        // Started with a '/' but no drive letter.  Put it in there for them to make processing easier down the road.
        newpath[0] = client->loggedDrive; newpath[1] = ':'; newpath[2] = 0;
      }
    }
  }

  // If not absolute prepend the current working directory.
  if ( !isAbs ) {
    strcat( newpath, client->cwd );
  }

  // Add the user filespec
  if ( strlen(newpath) + strlen(filespec) > (DOS_MAX_PATHFILE_LENGTH + 20 - 1) ) {
    return 3;
  }
  strcat( newpath, filespec );
  


  // By this point we have the full path as the user sees it.  If the user
  // wasn't in a sandbox and missed the drive letter we were nice and gave
  // it to them.
  //
  // Now validate it and parse out any . or .. components in the path.
  // This keeps a user in a sandbox from backing up too far out of their
  // sandbox.

  if ( normalizeDir( newpath, DOS_MAX_PATHFILE_LENGTH ) ) {
    return 4;
  }

  // Ok, it's a sane path at least.  If the user is in a sandbox we need
  // to prepend the root directory.

  outBuffer[0] = 0;

  // If the user is in a sandbox prepend the sandbox directory.
  if ( isSandbox ) {
    strcpy( outBuffer, client->ftproot );
  }

  if ( outBufferLen - strlen(outBuffer) <= strlen(newpath) ) {
    return 3;
  }

  strcat( outBuffer, newpath );

  // Convert unix '/' style delimeters to DOS style.  Strip off any trailing
  // delimeters too.
  convertToDosForm( outBuffer );

  return 0;
}



//   1 - illegal use of drive letters
//   2 - need to use absolute path with a drive letter
//   3 - path too long
//   4 - syntax error in the path

int formFullPath_DataXfer( FtpClient *client, char *outBuffer, int outBufferLen, char *filespec ) {

  int rc = formFullPath( client, outBuffer, outBufferLen, filespec );

  switch ( rc ) {

    case 0: { break; }
    case 1: { endDataTransfers( client, "550 No drive letters for you!" _NL_ ); break; }
    case 2: { endDataTransfers( client, "550 If you use a drive letter you must use absolute paths" _NL_ ); break; }
    case 3: { endDataTransfers( client, "550 Path too long" _NL_ ); break; }
    case 4: { endDataTransfers( client, "550 Bad path" _NL_ ); break; }

    default: { endDataTransfers( client, "550 Unknown error" _NL_ ); break; }
  }

  return rc;
}




void doXfer( FtpClient *client, char *nextTokenPtr, FtpClient::DataXferType listType ) {

  // Permission checks

  if ( ( (listType == FtpClient::Stor)  && (client->user.cmd_STOR == 0) )
    || ( (listType == FtpClient::StorA) && (client->user.cmd_APPE == 0) )
    || ( (listType == FtpClient::StorU) && (client->user.cmd_STOU == 0) ) )
  {
    client->addToOutput( "550 permission denied" _NL_ );
    return;
  }

  if ( (listType == FtpClient::Stor) || (listType == FtpClient::StorA) || (listType == FtpClient::StorU) ) {

    if ( (strcmp(client->user.uploaddir, "[ANY]") != 0) && stricmp( client->user.uploaddir, client->cwd ) != 0 ) {
      client->addToOutput_var( "550 You need to be in the %s directory to upload" _NL_, client->user.uploaddir );
      return;
    }

  }

  if ( client->dataXferState == FtpClient::DL_NotActive ) {
    client->dataXferState = FtpClient::DL_Init;
    client->dataXferType = listType;
    doDataXfer( client, nextTokenPtr );
  }
  else {
    client->addToOutput( Msg_425_Cant_Open_Conn );
  }

}


// If there is any activity on the data socket then this code gets run.
//
// - If the data sockets are being closed for either natural or unnatural
//   reasons we will detect that just wait for them to go closed.
// - If this is the first time in we'll initialize everything.
// - If they closed their side we'll start the close process
// - If we had unsent output, we'll send it.
// - If we need to generate more output and send it, we'll do that too.
// - If we run out of output, we'll start the close process.


void doDataXfer( FtpClient *client, char *parms ) {

  if ( client->dataXferState == FtpClient::DL_Closing ) {

    // We are supposed to be cleaning up.  Wait for both sockets to close.
    // After they close cleanup the fileinfo structure or open file pointer.
    // If a data transfer ever got past the Init stage then is should come
    // through here to clean up.

    // If the client had sent a PORT command wipe out the port so that we
    // know // they are forced to set it again for the next transfer.  We
    // don't need to wipe out the whole address - having a port of zero is
    // enough of an indicator.

    client->dataPort = 0;

    if ( client->ds != NULL ) {

      if ( client->ds->isCloseDone( ) ) {

        // Great, return the socket and close up.
        TcpSocketMgr::freeSocket( client->ds );
        client->ds = NULL;

        cleanupDataXferStructs( client );
        client->dataXferState = FtpClient::DL_NotActive;
      }

    }
    else {
      // Socket was never allocated so we don't wait for it to close.
      cleanupDataXferStructs( client );
      client->dataXferState = FtpClient::DL_NotActive;
    }

  } // end if DL_Closing



  else if ( client->dataXferState == FtpClient::DL_Init ) {

    client->connectStarted = time( NULL );
    Utils::getNextToken( parms, client->filespec, DOS_MAX_PATH_LENGTH );


    // If it is a RETR and the file doesn't exist, cut them off early.
    // If it is a STOR and the file does exist, cut them off early.
    // We don't check APPE because it doesn't matter if the file exists or
    // not.  We don't check STOU either because it doesn't look at the
    // filename.

    char fullpath[DOS_MAX_PATHFILE_LENGTH];

    if ( client->dataXferType == FtpClient::Retr ) {
      if ( formFullPath_DataXfer( client, fullpath, DOS_MAX_PATHFILE_LENGTH, client->filespec ) ) {
        return;
      }
      if ( !isFile( fullpath ) ) {
        endDataTransfers( client, "550 File not found" _NL_ );
        return;
      }
    }
    else if ( client->dataXferType == FtpClient::Stor ) {
      if ( formFullPath_DataXfer( client, fullpath, DOS_MAX_PATHFILE_LENGTH, client->filespec ) ) {
        return;
      }
      if ( doesExist( fullpath ) ) {
        endDataTransfers( client, "553 File exists already" _NL_ );
        return;
      }
    }
      

    // If we have a listening socket open already then we must be in PASV
    // mode and waiting for a client to connect.  Move to the next state
    // and continue to wait for the connection.
    //
    // If it was PASV and they connected already do the same thing.

    if ( client->ls != NULL ) {
      // Listening socket is open, still waiting for a data connection.
      client->activeConnect = 0;
      client->dataXferState = FtpClient::DL_Connecting;
    }
    else if ( client->ds != NULL ) {
      // Listening socket is not open, but we have a data socket already.
      client->activeConnect = 0;
      client->dataXferState = FtpClient::DL_Connected;
    }
    else {

      // Not listening and not connected yet.  That means we should try
      // to connect.

      if ( client->dataPort == 0 ) {
        endDataTransfers( client, Msg_425_Send_Port );
        return;
      }

      client->ds = TcpSocketMgr::getSocket( );
      if ( client->ds == NULL ) {
        TRACE_WARN(( "Ftp (%lu) Could not allocate a data socket\n", client->sessionId ));
        endDataTransfers( client, Msg_425_Cant_Open_Conn );
        return;
      }

      if ( client->dataXferType == FtpClient::Stor ||
           client->dataXferType == FtpClient::StorA ||
           client->dataXferType == FtpClient::StorU )
      {

        // Setup receive buffer for the socket.  Fixme: This is a waste
        // for dir listings and file sends; 
        if ( client->ds->setRecvBuffer( DATA_RCV_BUF_SIZE ) ) {
          TRACE_WARN(( "Ftp (%lu) Could not allocate data socket receive buffer\n", client->sessionId ));
          endDataTransfers( client, Msg_425_Cant_Open_Conn );
          return;
        }

      }

      // Start a non-blocking connect
      int16_t rc = client->ds->connectNonBlocking( (FtpSrv_Control_Port-1), client->dataTarget, client->dataPort );
      if ( rc ) {
        TRACE(( "Ftp (%lu) Initial connect call on data socket failed\n", client->sessionId ));
        endDataTransfers( client, Msg_425_Cant_Open_Conn );
        return;
      }

      client->activeConnect = 1;

      // Have to wait for the connect to complete now.
      client->dataXferState = FtpClient::DL_Connecting;
    }

  }




  else if ( client->dataXferState == FtpClient::DL_Connecting ) {

    // We are waiting for the data connection.  Check for timeout.

    if ( client->activeConnect == 0 ) {
      if ( client->ls != NULL ) {
        if ( time( NULL ) - client->connectStarted > 10 ) {
          TRACE(( "Ftp (%lu) Passive data connection timed out\n", client->sessionId ));
          endDataTransfers( client, Msg_425_Cant_Open_Conn );
        }
        // If you made it here you are still waiting and not timed out yet.
      }
      else {
        // We have our data connection - move to next state.
        client->dataXferState = FtpClient::DL_Connected;
      }

    }
    else {
      // This was a nonblocking connect that we started.
      if ( client->ds->isConnectComplete( ) ) {
        client->dataXferState = FtpClient::DL_Connected;
      }
      else {
        if ( time( NULL ) - client->connectStarted > 10 ) {
          TRACE(( "Ftp (%lu) Nonblocking connected for data socket timed out\n", client->sessionId ));
          endDataTransfers( client, Msg_425_Cant_Open_Conn );
        }
        // If you made it here you are still waiting and not timed out yet.
      }

    }

  }




  else if ( client->dataXferState == FtpClient::DL_Connected ) {

    // Ok, we have a data connection now.  Setup to actually start
    // transferring data.

    const char *dataTypeStr = BIN_str;
    if ( client->asciiMode ) {
      dataTypeStr = ASCII_str;
    }

    switch ( client->dataXferType ) {

      case FtpClient::List:
      case FtpClient::Nlist: {

        client->addToOutput( Msg_150_Send_File_List );

        char fullpath[DOS_MAX_PATHFILE_LENGTH];
        if ( formFullPath_DataXfer( client, fullpath, DOS_MAX_PATHFILE_LENGTH, client->filespec ) ) {
          return;
        }

        // Stat it.  If it is a directory then add the *.* to the end.
        // If it's not valid don't worry about it - they will get an empty
        // listing.

        if ( strlen(fullpath) < DOS_MAX_PATH_LENGTH ) {
          if ( isDirectory( fullpath ) ) {
            strcat( fullpath, "\\*.*" );
          }
        }

        client->noMoreData = _dos_findfirst( fullpath, (_A_NORMAL | _A_SUBDIR), &client->fileinfo);

        break;
      }

      case FtpClient::Retr: {

        // Fix me at some point.  We moved this code earlier, but have no way
        // pass the full filename here.  Just do it again, and it should not
        // fail.

        char fullpath[DOS_MAX_PATHFILE_LENGTH];
        if ( formFullPath_DataXfer( client, fullpath, DOS_MAX_PATHFILE_LENGTH, client->filespec ) ) {
          return;
        }

        if ( isFile( fullpath ) ) {
          client->addToOutput_var( "150 %s type File RETR started" _NL_, dataTypeStr );
        }
        else {
          endDataTransfers( client, "550 File not found" _NL_ );
          return;
        }

        if ( client->asciiMode ) {
          client->file = fopen( fullpath, "r" );
        }
        else {
          client->file = fopen( fullpath, "rb" );
        }

        if ( client->file == NULL ) {
          endDataTransfers( client, Msg_550_Filesystem_error );
          return;
        }

        addToLog( "(%lu) %s RETR started for %s\n", client->sessionId, dataTypeStr, fullpath );

        client->noMoreData = 0;
        break;
      }


      case FtpClient::Stor:
      case FtpClient::StorA: {

        char fullpath[DOS_MAX_PATHFILE_LENGTH];
        if ( formFullPath_DataXfer( client, fullpath, DOS_MAX_PATHFILE_LENGTH, client->filespec ) ) {
          return;
        }

        if ( client->dataXferType == FtpClient::Stor ) {

          if ( doesExist( fullpath ) ) {
            endDataTransfers( client, "550 File exists already" _NL_ );
            return;
          }

          client->addToOutput_var( "150 %s type File STOR started" _NL_, dataTypeStr );

        }
        else {

          // If it exists, then it must be a file.  If it doesn't exist that is ok too.
          // I assume that STAT will pick up the special filenames; if not, this doesn't
          // work.

          if ( doesExist(fullpath) && !isFile(fullpath) ) {
            endDataTransfers( client, "550 Target exists but is not a normal file" _NL_ );
            return;
          }

          client->addToOutput_var( "150 %s type File APPE started" _NL_, dataTypeStr );
        }

        char filemode[3];
        if ( client->dataXferType == FtpClient::Stor ) { filemode[0] = 'w'; } else { filemode[0] = 'a'; }
        if ( client->asciiMode ) { filemode[1] = 't'; } else { filemode[1] = 'b'; }
        filemode[2] = 0;

        client->file = fopen( fullpath, filemode );

        if ( client->file == NULL ) {
          endDataTransfers( client, Msg_550_Filesystem_error );
          return;
        }

        addToLog( "(%lu) %s STOR or APPE started for %s\n", client->sessionId, dataTypeStr, fullpath );

        client->noMoreData = 0;

        break;
      }


      case FtpClient::StorU: {

        // Need to create a unique filename in the selected directory.

        char filename[13] = "U0000000.QUE";
        char fullpath[DOS_MAX_PATHFILE_LENGTH];

        int attempts = 0;

        while ( attempts < 5 ) {

          // Generate a semi-random filename
          for ( int i=1; i < 8; i++ ) { filename[i] = (rand( ) % 10) + 48; }

          // Should not fail
          if ( formFullPath_DataXfer( client, fullpath, DOS_MAX_PATHFILE_LENGTH, filename ) ) {
            return;
          }

          // Stat it to see if it is unique
          struct stat statbuf;
          int rc = stat( fullpath, &statbuf );
          if ( rc ) break;

          attempts++;
        }

        if ( attempts == 5 ) {
          endDataTransfers( client, "550 Cant generate a unique name" _NL_ );
          return;
        }

        char filemode[3];
        filemode[0] = 'w';
        if ( client->asciiMode ) { filemode[1] = 't'; } else { filemode[1] = 'b'; }
        filemode[2] = 0;

        client->file = fopen( fullpath, filemode );

        if ( client->file == NULL ) {
          endDataTransfers( client, Msg_550_Filesystem_error );
          return;
        }

        client->addToOutput_var( "150 %s type STOU started, Filename is %s%s" _NL_, dataTypeStr, client->cwd, filename );

        client->noMoreData = 0;

        addToLog( "(%lu) %s STOU started for %s\n", client->sessionId, dataTypeStr, fullpath );

        break;
      }

    } // end switch


    // Common initialization for all of the data transfer types

    client->dataXferState = FtpClient::DL_Active;
    client->bytesSent = 0;
    client->fileBufferIndex = 0;

    // Used only by receive path
    client->bytesRead = 0;
    client->bytesToRead = FILEBUFFER_SIZE;

  }



  else if ( client->dataXferState == FtpClient::DL_Active ) {

    // Did the data socket close on us?
    //
    // If we were sending data (LIST, NLST or RETR) and they closed the
    // connection it is an error and there is no need to wait.  If we
    // were receiving data (STOR) then this means end of file, but we
    // have to wait until all data is read from the socket.

    if ( client->ds->isRemoteClosed( ) ) {

      if ( (client->dataXferType != FtpClient::Stor) && (client->dataXferType != FtpClient::StorA) && (client->dataXferType != FtpClient::StorU) ) {
        TRACE(( "(%lu) Data socket closed on us\n", client->sessionId ));
        endDataTransfers( client, Msg_426_Request_term );
      }
      else {
        client->noMoreData = 1;
      }

    }



    // We are primed and ready to read our first directory entries, or we have
    // re-entered because we gave up control to give somebody else a chance.


    if ( (client->dataXferType != FtpClient::Stor) && (client->dataXferType != FtpClient::StorA) && (client->dataXferType != FtpClient::StorU) ) {

      // Are there leftover bytes to send from the last time we were called?
      if ( client->fileBufferIndex ) {
        client->bytesSent += client->ds->send( (uint8_t *)(client->fileBuffer) + client->bytesSent, client->fileBufferIndex - client->bytesSent );
        // client->bytesSent += client->ds->send( (uint8_t *)(client->fileBuffer) + client->bytesSent, 1 );
        if ( client->bytesSent == client->fileBufferIndex ) {
          // Good - we cleared out the previous data.
          client->bytesSent = 0;
          client->fileBufferIndex = 0;
        }
        else {
          // Still blocked.  Give somebody else a chance.
          return;
        }
      }


      // By this point any previous data that needed to be sent has been sent.
      // Build up a new string to send.


      switch ( client->dataXferType ) {

        case FtpClient::Nlist:
        case FtpClient::List: {

          if ( !client->noMoreData ) {

            // Loop until we fill the buffer or there are no more directory entries.
            while ( 1 ) {

                if ( strcmp( client->fileinfo.name, "." ) != 0 && strcmp( client->fileinfo.name, ".." ) != 0 ) {

                  // Format file attributes
                  char attrs[] = "-rwxrwxrwx"; // Default
                  if ( client->fileinfo.attrib & _A_SUBDIR ) { attrs[0] = 'd'; }
                  if ( client->fileinfo.attrib & _A_RDONLY ) { attrs[2] = attrs[5] = attrs[8] = '-'; }

                  ftime_t ft;
                  ft.us = client->fileinfo.wr_time;

                  fdate_t fd;
                  fd.us = client->fileinfo.wr_date;

                  uint16_t rc = 0;;
                  if ( client->dataXferType == FtpClient::List ) {
                    if ( fd.fields.year + 1980 != Current_Year ) {
                      rc = sprintf( (char *)(client->fileBuffer + client->fileBufferIndex), "%s 1 ftp ftp %10lu %s %2d  %4d %s" _NL_,
                             attrs, client->fileinfo.size, Months[fd.fields.month-1], fd.fields.day, (fd.fields.year + 1980),
                             client->fileinfo.name );
                    }
                    else {
                      rc = sprintf( (char *)(client->fileBuffer + client->fileBufferIndex), "%s 1 ftp ftp %10lu %s %2d %02d:%02d %s" _NL_,
                             attrs, client->fileinfo.size, Months[fd.fields.month-1], fd.fields.day, ft.fields.hours, ft.fields.minutes,
                             client->fileinfo.name );
                    }
                  }
                  else {
                    rc = sprintf( (char *)(client->fileBuffer + client->fileBufferIndex), "%s" _NL_, client->fileinfo.name );
                  }

                  client->fileBufferIndex += rc;

                } // end if . or ..


              if ( (client->noMoreData = _dos_findnext( &client->fileinfo )) ) {
                _dos_findclose( &client->fileinfo );
                break;
              }

              if ( (FILEBUFFER_SIZE - client->fileBufferIndex) < 80 ) {
                break;
              }

            } // end build string up

          } // end no more dir entries

          break;
        }

        case FtpClient::Retr: {

          int rc = fread( client->fileBuffer, 1, FILEBUFFER_SIZE, client->file );
          if ( rc ) {
            client->fileBufferIndex = rc;
          }
          if ( feof( client->file ) ) client->noMoreData = 1;

          break;
        }


      } // end switch on data xfer type

      // Send the bytes out?
      if ( client->fileBufferIndex ) {
        client->bytesSent = client->ds->send( (uint8_t *)(client->fileBuffer), client->fileBufferIndex );
        // client->bytesSent = client->ds->send( (uint8_t *)(client->fileBuffer), 1 );
        if ( client->bytesSent == client->fileBufferIndex ) {
          // Good - we cleared out the previous data.
          client->bytesSent = 0;
          client->fileBufferIndex = 0;
        }
        else {
          // Still blocked.  Give somebody else a chance.
          return;
        }
      }

    } // end if sending data

    else {  // Receiving

      int16_t recvRc;

      while ( recvRc = client->ds->recv( client->fileBuffer+client->bytesRead, client->bytesToRead ) ) {

        if ( recvRc > 0 ) {

          client->bytesRead += recvRc;
          client->bytesToRead -= recvRc;

          if ( client->bytesToRead == 0 ) {
            size_t rc = fwrite( client->fileBuffer, 1, client->bytesRead, client->file );
            if ( rc != client->bytesRead ) {
              endDataTransfers( client, Msg_550_Filesystem_error );
              return;
            }
            client->bytesToRead = FILEBUFFER_SIZE;
            client->bytesRead = 0;

          }

        }
        else if ( recvRc < 0 ) {
          endDataTransfers( client, Msg_550_Filesystem_error );
          return;
        }

      } // end while

      // Flush remaining bytes
      if ( client->noMoreData && recvRc == 0 ) {
        int rc = fwrite( client->fileBuffer, 1, client->bytesRead, client->file );
        if (rc != client->bytesRead ) {
          endDataTransfers( client, Msg_550_Filesystem_error );
          return;
        }

      }

    }


    // If you got here there is no leftover data to send.  If there are no more
    // dir entries, pack up and go home.

    if ( client->noMoreData ) {
      client->ds->closeNonblocking( );

      switch ( client->dataXferType ) {
        case FtpClient::List: { Stat_LIST++; break; }
        case FtpClient::Nlist: { Stat_NLST++; break; }
        case FtpClient::Retr:  { Stat_RETR++; addToLog( "(%lu) RETR completed\n", client->sessionId ); break; }
        case FtpClient::Stor:  { Stat_STOR++; addToLog( "(%lu) STOR completed\n", client->sessionId ); break; }
        case FtpClient::StorA: { Stat_APPE++; addToLog( "(%lu) APPE completed\n", client->sessionId ); break; }
        case FtpClient::StorU: { Stat_STOU++; addToLog( "(%lu) STOU completed\n", client->sessionId ); break; }
      }

      client->dataXferState = FtpClient::DL_Closing;
      client->addToOutput( Msg_226_Transfer_Complete );
    }

  }

}




// doAbort needs to terminate any current data transfer, including the
// listening socket if it is in use.

void doAbort( FtpClient *client ) {

  TRACE(( "Ftp (%lu) doAbort\n" ));
  endDataTransfers( client, Msg_426_Request_term );
  client->addToOutput( Msg_226_ABOR_Complete );

}



// endDataTransfers
//
// If a data transfer was active this will start the close process and
// send a message to the control connection.
//
// If we were only in PASV state listening for a connection then it goes
// away without a message.

void endDataTransfers( FtpClient *client, char *msg ) {

  TRACE(( "Ftp (%lu) endDataTransfers  cs: (%08lx)  ds: (%08lx)  ls: (%08lx)\n",
          client->sessionId, client->cs, client->ds, client->ls ));

  // If there was a socket open for listening close it and return it.
  if ( client->ls != NULL ) {
    client->ls->close( ); // Should be immediate; nonBlocking is not needed.
    TcpSocketMgr::freeSocket( client->ls );
    client->ls = NULL;
  }


  // There might be a data connection even if there is no transfer in progress.
  // This happens when the client has sent PASV and has made the data
  // connection, but has not send a command that uses the data socket.

  if ( client->ds != NULL ) {
    // Throw away any data that might come in from this point forward
    client->ds->shutdown( TCP_SHUT_RD );
    client->ds->closeNonblocking( );

    // Ensure the state goes to closing so that we will drive it to completion.
    // This is done down below.
  }


  // If the user had started a data transfer send them the cancelled msg.

  if ( client->dataXferState != FtpClient::DL_NotActive ) {
    client->addToOutput( msg );
  }

  client->dataXferState = FtpClient::DL_Closing;
}




// endSession
//
// Mark the session as ending.  Besides ending the session we need to
// end any background processing like file transfers and directory
// listings.
//
// The main loop will wait for all of the client sockets to close and then
// recycle the client data structure.

void endSession( FtpClient *client ) {

  TRACE(( "Ftp (%lu) endSession\n", client->sessionId ));
  endDataTransfers( client, Msg_426_Request_term );

  // Mark the client as closing.
  client->state = FtpClient::ClosingPushOutput;

  // Now we just need to wait for everything to close
}





static char DosChars[] = "!@#$%^&()-_{}`'~*?";

static int isValidDosChar( char c ) {
  if ( isalnum( c ) || (c>127) ) return 1;
  for ( int i=0; i < 18; i++ ) {
    if ( c == DosChars[i] ) return 1;
  }
  return 0;
}


static int isValidDosFilename( const char *filename ) {

  if (filename==NULL) return 0;

  // Special case - check for . and ..
  if ( strcmp( filename, "." ) == 0 || strcmp( filename, ".." ) == 0 ) return 1;

  int len = strlen(filename);

  if ( len == 0 ) return 0;

  if ( !isValidDosChar( filename[0] ) ) return 0;

  int i;
  for ( i=1; (i<8) && (i<len) ; i++ ) {
    if ( filename[i] == '.' ) break;
    if ( !isValidDosChar( filename[i] ) ) return 0;
  }

  if ( i == len ) return 1;

  if ( filename[i] != '.' ) return 0;

  i++;
  int j;
  for ( j=0; (j+i) < len; j++ ) {
    if ( !isValidDosChar( filename[j+i] ) ) return 0;
  }

  if ( j > 3 ) return 0;

  return 1;
}






// Normalize takes a full path and breaks it up into components.
// Along the way it checks each component for validity.  At the end
// if everything is valid it rewrites the normalized path.
//
// The output never has a trailing /, as we don't know if this is
// a path or a path with a filename at the end.  The caller should
// add a '/' to the end if they want to denote it is a directory.


static char component[32][13];

int normalizeDir( char *buffer, int bufferLen ) {

  char driveLetter = 0;

  int top = 0;

  char tmp[13];

  int bufferIndex = 0;

  // Special case - is there a drive letter up front?
  if ( strlen( buffer ) > 1 ) {

    if ( buffer[1] == ':' ) {
      if ( isalpha( buffer[0] ) ) {
        driveLetter = buffer[0];
        bufferIndex = 2;
      }
      else {
        // Not a drive letter but the colon is present.
        return 1;
      }
    }
  }

  // Enforce a leading slash
  if ( buffer[bufferIndex] != '/' && buffer[bufferIndex] != '\\' ) {
    return 1;
  }

  bufferIndex++;

  while ( 1 ) {

    if ( top == 20 ) return 1;

    // Read next component from the path

    char tmp[13];
    int tmpIndex = 0;
    while ( 1 ) {

      if ( buffer[bufferIndex] == 0 ) {
        // Out of data
        break;
      }

      if ( buffer[bufferIndex] == '/' || buffer[bufferIndex] == '\\' ) {
        bufferIndex++;
        break;
      }

      if ( tmpIndex > 12 ) {
        return 1;
      }

      tmp[tmpIndex++] = buffer[bufferIndex++];

    }
    tmp[tmpIndex] = 0;

    if ( tmp[0] == 0 ) {
      if ( buffer[bufferIndex] == 0 ) {
        // Empty component and end of input ..  end main loop.
        break; 
      }
      else {
        // Empty component, but not end of input.  Must have been
        // back to back slashes.  We don't tolerate this.
        return 1;
      }
    }

    if ( !isValidDosFilename( tmp ) ) {
      return 1;
    }

    if ( strcmp( tmp, ".." ) == 0 ) {
      if ( top > 0 ) {
        top--;
      }
    }
    else if ( strcmp( tmp, "." ) == 0 ) {
      // Do nothing
    }
    else if ( tmp[0] ) {
      // Add it to the stack
      strcpy(component[top], tmp );
      top++;
    }

  }


  buffer[0] = 0;

  int ni = 0;
  if ( driveLetter != ' ' ) {
    buffer[ni++] = driveLetter;
    buffer[ni++] = ':';
    buffer[ni++] = 0;
  }

  if ( top == 0 ) {
    strcat( buffer, "/" );
  }
  else {
    for ( int i=0; i < top; i++ ) {
      strcat( buffer, "/" );
      strcat( buffer, component[i] );
    }
  }

  return 0;
}



// convertToDosForm fixes a path for use by DOS, including removing
// a trailing \.  It is used right before touching the file or dir.

void convertToDosForm( char *buffer_p ) {

  char *buffer = buffer_p;
  while ( *buffer ) {
    if ( *buffer == '/' ) *buffer = '\\';
    buffer++;
  }

  // Remove trailing  /
  int l = strlen( buffer_p );

  if ( l && buffer_p[l-1] == '\\' ) buffer_p[l-1] = 0;
}



// convertForDisplay ensures that a path looks like the Unix form,
// using /.  It is what the client will always see.
//
// Not anymore!  (April)

void convertForDisplay( int unixstyle, char *buffer_p ) {

  char *buffer = buffer_p;

  // Make everything lowercase to be consistent
  while ( *buffer ) {
    if ( islower( *buffer ) ) *buffer -= 32;
    buffer++;
  }

  buffer = buffer_p;
  
  if ( unixstyle ) {
    while ( *buffer ) {
      if ( *buffer == '\\' ) *buffer = '/';
      buffer++;
    }
  }
  else {
    while ( *buffer ) {
      if ( *buffer == '/' ) *buffer = '\\';
      buffer++;
    }
  }

}




static int readConfigParms( void ) {

  Utils::openCfgFile( );


  // Password file is required.

  if ( Utils::getAppValue( "FTPSRV_PASSWORD_FILE", PasswordFilename, DOS_MAX_PATHFILE_LENGTH ) ) {
    puts( "Need to specify FTPSRV_PASSWORD_FILE in mTCP config file" );
    return 1;
  }


  // Logfile is required.

  if ( Utils::getAppValue( "FTPSRV_LOG_FILE", LogFilename, DOS_MAX_PATHFILE_LENGTH ) ) {
    puts( "Need to specify FTPSRV_LOG_FILE in mTCP config file" );
    return 1;
  }


  char tmpBuffer[20];
  uint16_t tmpVal;

  if ( Utils::getAppValue( "FTPSRV_SESSION_TIMEOUT", tmpBuffer, 10 ) == 0 ) {
    tmpVal = atoi( tmpBuffer );
    if ( (tmpVal > 59) && tmpVal < 1201 ) {
      FtpSrv_timeoutTicks = tmpVal;
      FtpSrv_timeoutTicks = FtpSrv_timeoutTicks * 18ul;
    }
    else {
      puts( "FTPSRV_SESSION_TIMEOUT must be between 60 and 1200 seconds" );
      return 1;
    }
  }

  if ( Utils::getAppValue( "FTPSRV_CONTROL_PORT", tmpBuffer, 10 ) == 0 ) {
    tmpVal = atoi( tmpBuffer );
    if ( tmpVal > 0 ) {
      FtpSrv_Control_Port = tmpVal;
    }
    else {
      puts( "FTPSRV_CONTROL_PORT must be greater than 0" );
      return 1;
    }
  }


  if ( Utils::getAppValue( "FTPSRV_EXT_IPADDR", tmpBuffer, 20 ) == 0 ) {
    uint16_t tmp1, tmp2, tmp3, tmp4;
    int rc = sscanf( tmpBuffer, "%d.%d.%d.%d\n", &tmp1, &tmp2, &tmp3, &tmp4 );
    if ( rc != 4 ) {
      puts( "Bad IP address format on FTPSRV_EXT_IPADDR" );
      return 1;
    }
    Pasv_IpAddr[0] = tmp1; Pasv_IpAddr[1] = tmp2;
    Pasv_IpAddr[2] = tmp3; Pasv_IpAddr[3] = tmp4;
  }

  if ( Utils::getAppValue( "FTPSRV_PASV_BASE", tmpBuffer, 10 ) == 0 ) {
    tmpVal = atoi( tmpBuffer );
    if ( tmpVal > 1023 && tmpVal < 32768u ) {
      Pasv_Base = tmpVal;
    }
    else {
      puts( "FTPSRV_PASV_BASE must be between 1024 and 32768" );
      return 1;
    }
  }

  if ( Utils::getAppValue( "FTPSRV_PASV_PORTS", tmpBuffer, 10 ) == 0 ) {
    tmpVal = atoi( tmpBuffer );
    if ( tmpVal > 255 && tmpVal < 10241 ) {
      Pasv_Ports = tmpVal;
    }
    else {
      puts( "FTPSRV_PASV_PORTS must be between 256 and 10240" );
      return 1;
    }
  }

  if ( Utils::getAppValue( "FTPSRV_CLIENTS", tmpBuffer, 10 ) == 0 ) {
    tmpVal = atoi( tmpBuffer );
    if ( tmpVal > 0 && tmpVal <= FTP_MAX_CLIENTS ) {
      FtpSrv_Clients = tmpVal;
    }
    else {
      printf( "FTPSRV_CLIENTS must be between 1 and %u\n", FTP_MAX_CLIENTS );
      return 1;
    }
  }


  Utils::closeCfgFile( );

  return 0;
}
