
/*

   mTCP Screen.cpp
   Copyright (C) 2008-2011 Michael B. Brutman (mbbrutman@yahoo.com)
   mTCP web page: http://www.brutman.com/mTCP


   This file is part of mTCP.

   mTCP is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   mTCP is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mTCP.  If not, see <http://www.gnu.org/licenses/>.


   Description: Screen handling code for IRCjr

   Changes:

   2011-05-27: Initial release as open source software

*/



#include <bios.h>
#include <ctype.h>
#include <conio.h>
#include <dos.h>
#include <malloc.h>
#include <string.h>

#include "screen.h"
#include "session.h"



uint8_t Screen::colorCard;
uint8_t far *Screen::screenBase;
uint8_t Screen::rows;
uint8_t Screen::separatorRow;
uint8_t Screen::outputRows;
uint8_t far *Screen::separatorRowAddr;
uint8_t far *Screen::inputAreaStart;

uint8_t Screen::input_x;
uint8_t Screen::input_y;
uint16_t Screen::input_len;
char Screen::tmpInputBuffer[SCBUFFER_MAX_INPUT_LEN];
char *Screen::userInputBuffer;
uint8_t *Screen::switchToSession;



int8_t Screen::init( char *userInputBuffer_p, uint8_t *switchToSession_p ) {

  // This always works:
  unsigned char mode = *((unsigned char far *)MK_FP( 0x40, 0x49 ));

  if ( mode == 7 ) {
    colorCard = 0;
    screenBase = (uint8_t far *)MK_FP( 0xb000, 0 );
  }
  else {
    colorCard = 1;
    screenBase = (uint8_t far *)MK_FP( 0xb800, 0 );
  }

  // Call int 10, ah=12 for EGA/VGA config

  union REGS inregs, outregs;
  struct SREGS segregs;

  inregs.h.ah = 0x12;
  inregs.h.bl = 0x10;
  int86x( 0x10, &inregs, &outregs, &segregs );

  if ( outregs.h.bl == 0x10 ) {
    // Failed.  Must be MDA or CGA
    rows = 25;
  }
  else {
    rows = *((unsigned char far *)MK_FP( 0x40, 0x84 )) + 1;
  }

  separatorRow = rows - INPUT_ROWS - 1;
  outputRows = rows - (INPUT_ROWS + 1);

  separatorRowAddr = screenBase + ( separatorRow * BYTES_PER_ROW );
  inputAreaStart = separatorRowAddr + BYTES_PER_ROW;

  input_x = 0;
  input_y = separatorRow + 1;
  input_len = 0;

  userInputBuffer = userInputBuffer_p;
  switchToSession = switchToSession_p;


  // Draw initial screen
  fillUsingWord( screenBase, 7<<8|32, outputRows * 80 );
  fillUsingWord( separatorRowAddr, 7<<8|196, 80 );
  fillUsingWord( inputAreaStart, 7<<8|32, INPUT_ROWS * BYTES_PER_ROW );

  return 0;
}





void Screen::clearInputArea( void ) {

  fillUsingWord( inputAreaStart, (7<<8|32), INPUT_ROWS * BYTES_PER_ROW );

  input_x = 0;
  input_y = separatorRow + 1;
  input_len = 0;

  // Put the cursor back in the right spot to be pretty.
  gotoxy( input_x, input_y );

}





Screen::InputActions Screen::getInput2( void ) {

  gotoxy( input_x, input_y );

  uint8_t ch = getch( );

  if ( ((ch > 31) && (ch < 127)) || (ch > 127) ) {

    if ( input_len < (SCBUFFER_MAX_INPUT_LEN-2) ) {

      gotoxy( input_x, input_y );
      putch( ch );

      tmpInputBuffer[input_len] = ch;
      input_len++;

      input_x++;
      if ( input_x > 79 ) {
        input_x = 0;
        input_y++;
      }

      return NoAction;
    }
    else { BEEP( ); }

  }
  else if ( iscntrl(ch) ) {

    if ( ch == 0 ) {
      ch = getch( );
      if ( ch == 73 ) return BackScroll;      // PgUp
      if ( ch == 81 ) return ForwardScroll;   // PgDn
      if ( ch == 31 ) return Stats;           // Alt-S
      if ( ch == 45 ) return EndProgram;      // Alt-X
      if ( ch == 46 ) return CloseWindow;     // Alt-C
      if ( ch == 48 ) return BeepToggle;      // Alt-B
      if ( ch == 35 ) return Help;            // Alt-H
      if ( ch == 19 ) return ShowRawToggle;   // Alt-R
      if ( ch == 20 ) return TimestampToggle; // Alt-T
      if ( ch == 38 ) return LoggingToggle;   // Alt-L

      if ( ch == 129 ) {      // Alt-0 = ServerSession
        *switchToSession = 0;
        return SwitchSession;
      }

      if ( ch >= 120 && ch <= 128 ) {
        *switchToSession = ch - 119;
        return SwitchSession;
      }

    }
    else {

      if ( ch == 8 ) {

        if ( input_len ) {

          input_len--;

          if ( input_x == 0 ) {
            input_x = 80;
            input_y--;
          }

          input_x--;

          gotoxy( input_x, input_y );
          putch( ' ' );
          gotoxy( input_x, input_y );

        }
        else {
          BEEP( );
        }

      }

      else if ( ch == 13 ) {
        tmpInputBuffer[input_len] = 0;
        strcpy( userInputBuffer, tmpInputBuffer );
        clearInputArea( );
        return InputReady;
      }

    }

  }

  return NoAction;

}



void Screen::writeOnConsole( uint8_t attr, uint8_t x, uint8_t y, char *msg ) {

  uint16_t far *start = (uint16_t far *)(screenBase + (y*80+x)*2);

  uint16_t len = strlen( msg );

  for ( uint16_t i = 0; i < len; i++ ) {
    *start++ = ( attr << 8 | msg[i] );
  }

}


