
/*

   mTCP FtpUsr.cpp
   Copyright (C) 2009-2011 Michael B. Brutman (mbbrutman@yahoo.com)
   mTCP web page: http://www.brutman.com/mTCP


   This file is part of mTCP.

   mTCP is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   mTCP is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mTCP.  If not, see <http://www.gnu.org/licenses/>.


   Description: Code for FTP server user file management

   Changes:

   2011-05-27: Initial release as open source software

*/




#include <stdio.h>
#include <string.h>
#include <sys/stat.h>


#include "utils.h"
#include "ftpusr.h"




extern int normalizeDir( char *buffer, int bufferLen );
extern void convertToDosForm( char *buffer_p );
extern void convertForDisplay( int sandbox, char *buffer_p );



// Ftp User design
//
// The general idea is to store all of the ftp users in a file and to leave
// the file open for reading while the program is running.  When we need to
// find a user we'll do a linear scan through the file.
//
// It is quite probable that the file will be fully resident in memory so
// that no disk I/O actually occurs.  If the disk I/O does become a problem
// implement a small cache of the 5 most commonly used users, and use an LRU
// algorithm.  That will cover most of our needs quite easily.



FILE *FtpUser::userFile = NULL;




int FtpUser::sanityCheck( char *errMsgOut ) {

  if ( strcmp( sandbox, "[NONE]" ) != 0 ) {
    if ( normalizeDir( sandbox, DOS_MAX_PATH_LENGTH ) ) {
      strcpy( errMsgOut, "sandbox field not valid" );
      return 1;
    }

    // Try to stat the sandbox
    struct stat statbuf;
    int rc = stat( sandbox, &statbuf );

    if ( (rc != 0) || ( (rc == 0) && (S_ISDIR(statbuf.st_mode) == 0) ) ) {
      strcpy( errMsgOut, "sandbox is not a directory" );
      return 1;
    }
  }


  if ( strcmp( uploaddir, "[ANY]" ) != 0 ) {
    if ( normalizeDir( uploaddir, DOS_MAX_PATH_LENGTH ) ) {
      strcpy( errMsgOut, "uploaddir field not valid" );
      return 1;
    }
  }

  return 0;
}



int FtpUser::init( const char *userFilename ) {

  printf( "Opening password file at %s\n", userFilename );

  userFile = fopen( userFilename, "r" );
  if ( userFile == NULL ) {
    printf( "  Error reading user file\n" );
    return 1;
  }


  // Scan file for a sanity check

  char lineBuffer[256];
  int lineNo = 0, errors = 0;
  while ( 1 ) {
    
    if ( feof( userFile ) ) break;
    
    char *rc = fgets( lineBuffer, 255, userFile );
    lineNo++;
    if ( rc == NULL ) break; // EOF or read error

    // Wipe out trailing newline char if it is there.
    int l = strlen( lineBuffer );
    if ( lineBuffer[l] == '\n' ) lineBuffer[l] = 0;
    
    l = strlen( lineBuffer );
    if ( l == 0 || ( (l) && (lineBuffer[0] == '#') ) ) continue; // Comment or blank line
    
    char tmpUserName[ USERNAME_LEN ];
    char *nextTokenPtr = Utils::getNextToken( lineBuffer, tmpUserName, USERNAME_LEN );
    if ( *tmpUserName == 0 ) continue; // Blank line


    // Try to create the user rec and sanity check it.

    FtpUser buffer;
    char errMsg[40];

    switch ( createUserRec( lineBuffer, &buffer ) ) {

      case 0: { 
        // Ok, sanity check it
        if ( buffer.sanityCheck( errMsg ) ) {
          printf( "  Error on line: %d, Error: %s\n", lineNo, errMsg );
          errors++;
        }
        break;
      }
      case 1: {
        printf( "  Missing fields on line: %d\n", lineNo );
        errors++;
        break;
      }
      case 2: {
        printf( "  Unrecognized permissions text on line: %d\n", lineNo );
        errors++;
        break;
      }

    } // end switch.

  }

  if ( errors ) {
    printf( "  Total errors found: %d\n", errors );
    return 1;
  }

  puts( "  Password file looks reasonable.\n" );
  return 0;
}




// Returns 0 if not found, 1 if found

int FtpUser::getUserRec( const char *targetUser, FtpUser *buffer ) {

  // Position to beginning
  fseek( userFile, 0, SEEK_SET );

  // Start looking for our user

  char lineBuffer[256];
  while ( 1 ) {

    if ( feof( userFile ) ) break;

    char *rc = fgets( lineBuffer, 255, userFile );
    if ( rc == NULL ) break; // EOF or error

    // Wipe out trailing newline char if it is there.
    int l = strlen( lineBuffer );
    if ( lineBuffer[l] == '\n' ) lineBuffer[l] = 0;

    l = strlen( lineBuffer );
    if ( l == 0 || ( (l) && (lineBuffer[0] == '#') ) ) continue; // Comment or blank line

    char tmpUserName[ USERNAME_LEN ];
    char *nextTokenPtr = Utils::getNextToken( lineBuffer, tmpUserName, USERNAME_LEN );
    if ( *tmpUserName == 0 ) continue; // Blank line

    if ( stricmp( targetUser, tmpUserName ) == 0 ) {
      if ( createUserRec( lineBuffer, buffer ) == 0 ) {
        return 1; // Return found
      }
      else {
        return 0; // Was found, but in error - return not found
      }

    }

  }

  return 0;

}




// Returns 0 if successful, 1 if missing a field, 2 if unrecognized
// permission field.

int FtpUser::createUserRec( char *input, FtpUser *buffer ) {

  // Ensure the target buffer is clean
  buffer->wipe( );

  // If the input is null for some reason exit early.
  if ( input[0] == 0 ) return 1;

  char *nextTokenPtr = Utils::getNextToken( input, buffer->userName, USERNAME_LEN );
  if ( nextTokenPtr == NULL || buffer->userName[0] == 0 ) return 1;

  nextTokenPtr = Utils::getNextToken( nextTokenPtr, buffer->userPass, USERPASS_LEN );
  if ( nextTokenPtr == NULL || buffer->userPass[0] == 0 ) return 1;

  nextTokenPtr = Utils::getNextToken( nextTokenPtr, buffer->sandbox, DOS_MAX_PATH_LENGTH );
  if ( nextTokenPtr == NULL || buffer->sandbox[0] == 0 ) return 1;
  strupr( buffer->sandbox );

  nextTokenPtr = Utils::getNextToken( nextTokenPtr, buffer->uploaddir, DOS_MAX_PATH_LENGTH );
  if ( buffer->uploaddir[0] == 0 ) return 1;
  strupr( buffer->uploaddir );

  // If the sandbox is not "[NONE]" then convert slashes to what DOS uses
  // and remove any trailing slashes.

  if ( strcmp( buffer->sandbox, "[NONE]" ) != 0 ) {
    convertToDosForm( buffer->sandbox );
  }


  // If the uploaddir is not "[ANY]" then convert slashes to Unix style
  // (which is what we use for display) and tack on a trailing slash to
  // match how we display the current working directory.

  if ( strcmp( buffer->uploaddir, "[ANY]" ) != 0 ) {
    convertForDisplay( 1, buffer->uploaddir );
    int l = strlen( buffer->uploaddir );
    if ( (l < DOS_MAX_PATH_LENGTH-1 ) && (buffer->uploaddir[l-1] != '/') && (buffer->uploaddir[l-1] != '\\') ) {
      buffer->uploaddir[l] = '/';
      buffer->uploaddir[l+1] = 0;
    }
  }


  // Read permissions

  char tmpPermToken[10];

  // Permissions are optional.  If you don't give a user any explicit
  // permissions then they are only to do reads and other non-filesystem
  // altering operations.

  while ( nextTokenPtr != NULL ) {

    nextTokenPtr = Utils::getNextToken( nextTokenPtr, tmpPermToken, 10 );
    if ( tmpPermToken[0] == 0 ) break;

    if ( stricmp( tmpPermToken, "all" ) == 0 ) {
      buffer->cmd_DELE = 1; buffer->cmd_MKD = 1; buffer->cmd_RMD = 1; buffer->cmd_RNFR = 1;
      buffer->cmd_STOR = 1; buffer->cmd_APPE = 1; buffer->cmd_STOU = 1;
    }
    else if ( stricmp( tmpPermToken, "DELE" ) == 0 ) { buffer->cmd_DELE = 1; }
    else if ( stricmp( tmpPermToken, "MKD" ) == 0 )  { buffer->cmd_MKD = 1; }
    else if ( stricmp( tmpPermToken, "RMD" ) == 0 )  { buffer->cmd_RMD = 1; }
    else if ( stricmp( tmpPermToken, "RNFR" ) == 0 ) { buffer->cmd_RNFR = 1; }
    else if ( stricmp( tmpPermToken, "STOR" ) == 0 ) { buffer->cmd_STOR = 1; }
    else if ( stricmp( tmpPermToken, "APPE" ) == 0 ) { buffer->cmd_APPE = 1; }
    else if ( stricmp( tmpPermToken, "STOU" ) == 0 ) { buffer->cmd_STOU = 1; }
    else {
      return 2;
    }

  } // end while

  return 0;
}
