     
     		Introduction to XSI Curses Interface
     
     The X/Open Group has identified a strong need for a generic terminal
     interface for applications that wish to be independent of terminal
     hardware and connection method.
     
     This interface should allow the attachment of character and
     block-oriented terminals.  Furthermore, it should not put any
     constraints on how the terminals are attached (e.g., Local Area
     Networks, PADs on X.25, etc.). 
     
     The curses library interfaces provides the user with a method of
     updating screens with reasonable optimisation. 
     
     The X/Open group has found it impossible to define a totally portable
     set of curses interface routines that cover asynchronous, networked,
     and synchronous terminals.  The functions are oriented towards locally
     connected asynchronous terminals.  For such terminals, applications
     conforming to this interface are portable.  The interface routines
     curses may, however, also be used with synchronous and networked
     terminals, provided the restrictions below are considered. 
     
     These functions have been included been included in the X/Open
     definition in the "optional" category.  This means that although they
     are likely to appear on many X/Open compliant systems, they are not
     guaranteed to be on all systems.  Where they are supported, they will
     conform to the given definition. 
     
     
     ----------------------------------------------------------------------
     
     	    Synchronous and Networked Asynchronous Terminals
     
     These notes indicate to the application writer some considerations to
     be borne in mind when driving synchronous, networked asynchronous
     (NWA) or non-standard directly connected asynchronous terminals.
     
     Such terminals are often used in a mainframe environment and
     communicatie to the host in block mode.  That is, the user types
     characters at the terminal then presses a special key to initiate
     transmission of the characters to the host. 
     
     Frequently, although it may be possible to send arbitrary sized blocks
     to the host, it may not be possible or desireable to cause a character
     to be transmitted with only a single keystroke. 
     
     This can cause severe problems to an application wishing to make use
     of single character input. 
     
     
     
     OUTPUT
     
     The curses package can be used in the normal way for all operations
     pertaining to output to the terminal, with the possible exception that
     on some terminals the refresh() routine may have to redraw the entire
     screen contents in order to perform any update.
     
     If it is additionally necessary to clear the screen before each such
     operation, the result could be unacceptable. 
     
     
     INPUT
     
     Because of the nature of operation of synchronous (block-mode) and NWA
     terminals, it may not be possible to support all or any of the curses
     input functions.  In particular, the following points should be noted:
     
     * Single-character input may not be possible.  It may be necessary to
       press a special key to cause all characters typed at the terminal to
       be transmitted to the host. 
     
     * It may not be possibel to disable echo.  Character echo may be performed
       directly by the terminal.  On terminals that behave this way, any curses
       application that performs input should be aware that any characters
       type will appear on the screen wherever the cursor is positioned.
       This may not necessarily correspond to the position of the cursor in
       the window.
     
     
     ----------------------------------------------------------------------
     
     		Data Types and the <curses.h> Header
     
     The data types supported by curses are described in this section.
     
     As the library supports a procedural interface to the data types,
     actual structure contents are not described.  All curses data are
     manipulated using the routines provided. 
     
     
     THE <curses.h> HEADER
     
     The <curses.h> header defines various constants and declares the data types
     that are available to the application.
     
     
     DATA TYPES
     
     The following data types are declared:
     
     	WINDOW*		pointer to screen representation
     	SCREEN*		pointer to terminal descriptor
     	bool		boolean data type
     	chtype		representation of a character in a window
     
     The actual WINDOW and SCREEN objects used to store information are
     created by the corresponding routines and a pointer to them is
     provided.  All manipulation is through that pointer. 
     
     
     CONSTANTS
     
     The following constants are defined. 
     
     
     GENERAL
     
     	COLS		number of columns on terminal screen
     	ERR		value returned on error condition
     	FALSE		boolean false value
     	LINES		number of lines on terminal screen
     	OK		value returned on successful completion
     	NULL		zero pointer value
     	TRUE		boolean true value
     
     VIDEO ATTRIBUTES
     
     	A_BLINK		blinking
     	A_BOLD		extra bright or bold
     	A_DIM		half bright
     	A_REVERSE	reverse video
     	A_STANDOUT	terminal's best highlighting mode
     	A_UNDERLINE	underlining
     	A_ATTRIBUTES	bit-mask to extract attributes
     	A_CHARTEXT	bit-mask to extract a character
     
     Normally, attributres are a property of the character. 
     
     
     INPUT VALUES
     
     The following constants might be returned by getch() if keypad() has
     been enabled.  Note that not all of these may be supported on a
     particular terminal if the terminal does not transmit a unique code
     when the key is pressed or the definition for the key is not present
     in the underlying table of terminal capabilities. 
     
     	KEY_BREAK	break key
     	KEY_DOWN	the four arrow keys
     	KEY_UP
     	KEY_LEFT
     	KEY_RIGHT
     	KEY_HOME	home key (upward+left arrow)
     	KEY_BACKSPACE	backspace
     	KEY_F0		function keys; space for 64 keys is reserved
     	KEY_F(n)	(KEY_F0+(n))
     	KEY_DL		delete line
     	KEY_IL		insert line
     	KEY_DC		delete character
     	KEY_IC		insert character
     	KEY_EIC		exit insert character mode
     	KEY_CLEAR	clear screen
     	KEY_EOS		clear to end of screen
     	KEY_EOL		clear to end of line
     	KEY_SF		scroll 1 line forwards
     	KEY_SR		scroll 1 line backwards (reverse)
     	KEY_NPAGE	next page
     	KEY_PPAGE	previous page
     	KEY_STAB	set tab
     	KEY_CTAB	clear tab
     	KEY_CATAB	clear all tabs
     	KEY_ENTER	enter or send
     	KEY_SRESET	soft (partial) reset
     	KEY_RESET	reset or hard reset
     	KEY_PRINT	print or copy
     	KEY_LL		home down or bottom (lower left)
     	KEY_A1		upper left of virtual keypad
     	KEY_A3		upper right of virtual keypad
     	KEY_B2		centre of virtual keypad
     	KEY_C1		lower left of virtual keypad
     	KEY_C3		lower right of virtual keypad
     
     The virtual keypad is arranged like this:
     
     	A1	up	A3
     	left	B2	right
     	C1	down	C3
     
     FUNCTIONS
     
     	The following table lists each curses routine and the name of the 
     	manual page on which it is described. This list is based on
     	System V R4 curses:
     
     	   Curses Function        Manual Page Name
     
     		addch                 addch
     		addchnstr             addchstr
     		addchstr              addchstr
     		addnstr               addstr
     		addstr                addstr
     		attroff               attr
     		attron                attr
     		baudrate              termattr
     		beep                  beep
     		bkgd                  bkgd
     		bkgdset               bkgd
     		border                border
     		box                   border
     		can_change_color      color
     		cbreak                inopts
     		clear                 clear
     		clearok               outopts
     		clrtobot              clear
     		clrtoeol              clear
     		color_content         *** color
     		copywin               overlay
     		curs_set              kernel
     		def_prog_mode         kernel
     		def_shell_mode        kernel
     		del_curterm           *** terminfo
     		delay_output          util
     		delch                 delch
     		deleteln              deleteln
     		delscreen             *** initscr
     		delwin                window
     		derwin                window
     		doupdate              refresh
     		dupwin                window
     		echo                  inopts
     		echochar              addch
     		endwin                initscr
     		erase                 clear
     		erasechar             termattr
     		filter                *** util
     		flash                 beep
     		flushinp              util
     		getbegyx              getyx
     		getch                 getch
     		getmaxyx              getyx
     		getparyx              getyx
     		getstr                getstr
     		getsyx                *** kernel
     		getwin                *** util
     		getyx                 getyx
     		halfdelay             *** inopts
     		has_colors            color
     		has_ic                termattr
     		has_il                termattr
     		hline                 border
     		idcok                 outopts
     		idlok                 outopts
     		immedok               outopts
     		inch                  inch
     		inchnstr              inchstr
     		inchstr               inchstr
     		init_color            color
     		init_pair             color
     		initscr               initscr
     		innstr                instr
     		insch                 insch
     		insdelln              deleteln
     		insertln              deleteln
     		insnstr               innstr
     		insstr                innstr
     		instr                 instr
     		intrflush             inopts
     		is_linetouched        touch
     		is_wintouched         touch
     		isendwin              *** initscr
     		keyname               util
     		keypad                inopts
     		killchar              termattr
     		leaveok               outopts
     		longname              termattr
     		meta                  inopts
     		move                  move
     		mvaddch               addch
     		mvaddchnstr           addchstr
     		mvaddchstr            addchstr
     		mvaddnstr             addstr
     		mvaddstr              addstr
     		mvcur                 terminfo
     		mvdelch               delch
     		mvderwin              window
     		mvgetch               getch
     		mvgetstr              getstr
     		mvinch                inch
     		mvinchnstr            inchstr
     		mvinchstr             inchstr
     		mvinnstr              instr
     		mvinsch               insch
     		mvinsnstr             insstr
     		mvinsstr              insstr
     		mvinstr               instr
     		mvprintw              printw
     		mvscanw               scanw
     		mvwaddch              addch
     		mvwaddchnstr          addchstr
     		mvwaddchstr           addchstr
     		mvwaddnstr            addstr
     		mvwaddstr             addstr
     		mvwdelch              delch
     		mvwgetch              getch
     		mvwgetstr             getstr
     		mvwin                 window
     		mvwinch               inch
     		mvwinchnstr           inchstr
     		mvwinchstr            inchstr
     		mvwinnstr             instr
     		mvwinsch              insch
     		mvwinsnstr            insstr
     		mvwinsstr             insstr
     		mvwinstr              instr
     		mvwprintw             printw
     		mvwscanw              scanw
     		napms                 kernel
     		newpad                pad
     		newterm               *** initscr
     		newwin                window
     		nl                    outopts
     		nocbreak              inopts
     		nodelay               inopts
     		noecho                inopts
     		nonl                  outopts
     		noqiflush             *** inopts
     		noraw                 inopts
     		notimeout             inopts
     		overlay               overlay
     		overwrite             overlay
     		pair_content          color
     		pechochar             pad
     		pnoutrefresh          pad
     		prefresh              pad
     		printw                printw
     		putp                  *** terminfo
     		putwin                *** util
     		qiflush               *** inopts
     		raw                   inopts
     		raw_output            outopts
     		redrawwin             refresh
     		refresh               refresh
     		reset_prog_mode       kernel
     		reset_shell_mode      kernel
     		resetty               kernel
     		restartterm           *** terminfo
     		ripoffline            *** kernel
     		savetty               kernel
     		scanw                 scanw
     		scr_dump              *** scr_dump
     		scr_init              *** scr_dump
     		scr_restore           *** scr_dump
     		scr_set               *** scr_dump
     		scrl                  scroll
     		scroll                scroll
     		scrollok              outopts
     		set_term              initscr
     		setscrreg             outopts
     		setsyx                *** kernel
     		setterm               *** terminfo
     		setupterm             *** terminfo
     		slk_attroff           *** slk
     		slk_attron            *** slk
     		slk_attrset           *** slk
     		slk_clear             *** slk
     		slk_init              *** slk
     		slk_label             *** slk
     		slk_noutrefresh       *** slk
     		slk_refresh           *** slk
     		slk_restore           *** slk
     		slk_set               *** slk
     		slk_touch             *** slk
     		standend              attr
     		standout              attr
     		start_color           color
     		subpad                pad
     		subwin                window
     		syncok                *** window
     		termattrs             termattrs
     		termname              termattrs
     		tgetent               *** termcap
     		tgetflag              *** termcap
     		tgetnum               *** termcap
     		tgetstr               *** termcap
     		tgoto                 *** termcap
     		tigetflag             *** terminfo
     		tigetnum              *** terminfo
     		tigetstr              *** terminfo
     		timeout               *** inopts
     		touchline             touch
     		touchwin              touch
     		tparm                 *** terminfo
     		tputs                 *** terminfo
     		tputs                 *** termcap
     		trace_on              pdcdebug
     		typeahead             inopts
     		unctrl                util
     		ungetch               getch
     		untouchwin            touch
     		use_env               *** util
     		vidattr               *** terminfo
     		vidputs               *** terminfo
     		vline                 border
     		vwprintw              *** printw
     		vwscanw               *** scanw
     		waddch                addch
     		waddchnstr            addchstr
     		waddchstr             addchstr
     		waddnstr              addstr
     		waddstr               addstr
     		wattroff              attr
     		wattron               attr
     		wattrset              attr
     		wbkgd                 bkgd
     		wbkgdset              bkgd
     		wborder               border
     		wclear                clear
     		wclrtobot             clear
     		wclrtoeol             clear
     		wcursyncup            *** window
     		wdelch                delch
     		wdeleteln             deleteln
     		wechochar             addch
     		werase                clear
     		wgetch                getch
     		wgetnstr              getstr
     		wgetstr               getstr
     		whline                border
     		winch                 inch
     		winchnstr             inchstr
     		winchstr              inchstr
     		winnstr               instr
     		winsch                insch
     		winsdelln             deleteln
     		winsertln             deleteln
     		winsnstr              insstr
     		winsstr               insstr
     		winstr                instr
     		wmove                 move
     		wnoutrefresh          refresh
     		wordchar              termattr
     		wprintw               printw
     		wredrawln             refresh
     		wrefresh              refresh
     		wscanw                scanw
     		wscrl                 scroll
     		wsetscrreg            outopts
     		wstandend             attr
     		wstandout             attr
     		wsyncdown             *** window
     		wsyncup               *** window
     		wtimeout              *** inopts
     		wtouchln              touch
     		wvline                border
     
     	The following table lists each private curses routine and the name 
     	of the manual page on which it is described. These routines should
     	not be used directly if portability is to be considered.
     
     	   Curses Function        Manual Page Name
     
     		PDC_backchar          pdcdisp
     		PDC_chadd             pdcdisp
     		PDC_check_bios_key    pdckbd
     		PDC_chg_attrs         pdcdisp
     		PDC_chg_attr_pair     pdcdisp
     		PDC_chins             pdcdisp
     		PDC_clr_scrn          pdcdisp
     		PDC_clr_update        pdcdisp
     		PDC_copy_win          pdcwin
     		PDC_cursor_off        pdcdisp
     		PDC_cursor_on         pdcdisp
     		PDC_debug             pdcdebug
     		PDC_fix_cursor        pdcdisp
     		PDC_getch             getch
     		PDC_get_attribute     pdcgetsc
     		PDC_get_bios_key      pdckbd
     		PDC_get_columns       pdcgetsc
     		PDC_get_ctrl_break    pdckbd
     		PDC_get_cursor_mode   pdcgetsc
     		PDC_get_cursor_pos    pdcgetsc
     		PDC_get_cur_col       pdcgetsc
     		PDC_get_cur_row       pdcgetsc
     		PDC_get_font          pdcgetsc
     		PDC_get_rows          pdcgetsc
     		PDC_get_scrn_mode     pdcgetsc
     		PDC_gotoxy            pdcdisp
     		PDC_makenew           pdcwin
     		PDC_memmove           pdcutil
     		PDC_newline           pdcdisp
     		PDC_print             pdcprint
     		PDC_putc              pdcdisp
     		PDC_putctty           pdcdisp
     		PDC_query_adapter_type  pdcgetsc
     		PDC_rawgetch          pdckbd
     		PDC_sanity_check      pdcgetsc
     		PDC_scrn_modes_equal  pdcscrn      
     		PDC_scroll            pdcdisp
     		PDC_scr_close         pdcscrn
     		PDC_scr_open          pdcscrn
     		PDC_set_80x25         pdcsetsc
     		PDC_set_ctrl_break    pdckbd
     		PDC_set_cursor_mode   pdcsetsc
     		PDC_set_font          pdcsetsc
     		PDC_set_rows          pdcsetsc
     		PDC_set_scrn_mode     pdcsetsc
     		PDC_sysgetch          pdckbd
     		PDC_transform_line    pdcdisp
     		PDC_usleep            pdcutil
     		PDC_validchar         pdckbd
     		PDC_vsscanf           pdcutil
     
     	NOTE: Functions prefixed with '***' above and on each manual page are 
     		functions that have not been implemented.
     



     --------------------------------------------------------------------------





     
     			Curses Overview
     
     The X/Open Curses Interface Definition describes a set of C-Language
     functions that provide screen-handling and updating, which are
     collectively known as the curses library.
     
     The curses library permits manipulation of data structures called
     windows which may be thought of as two-dimensional arrays of
     characters representing all or part of a terminal's screen.  The
     windows are manipulated using a procedural interface described
     elsewhere.  The curses package maintains a record of what characters
     are on the screen.  At the most basic level, manipulation is done with
     the routines move() and addch() which are used to "move" the curses
     around and add characters to the default window, stdscr, which
     represents the whole screen.
     
     An application may use these routines to add data to the window in any
     convenient order.  Once all data have been added, the routine
     refresh() is called.  The package then determines what changes have
     been made which affect the screen.  The screen contents are then
     changed to reflect those characters now in the window. using a
     sequence of operations optimised for the type of terminal in use. 
     
     At a higher level routines combining the actions of move() and addch()
     are defined, as are routines to add whole strings and to perform
     format conversions in the manner of printf(). 
     
     Interfaces are alse defined to erase the entire window and to specify
     the attributes of individual characters in the winodw.  Attributes
     such as inverse video, underline and blink can be used on a
     per-character basis. 
     
     New windows can be created by allowing the application to build
     several images of the screen and display the appropriate one very
     quickly.  New windows are created using the routine newwin().  For
     each routine that manipulates the default window, stdscr, there is a
     corresponding routine prefixed with w to manipulate the contents of a
     specified window; for example, move() and wmove().  In fact, move(...)
     is functionally equivalent to wmove( stdscr, ...).  This is similar to
     the interface offered by printf(...) and fprintf(stdout, ...). 
     
     Windows do not have to correspond to the entire screen.  It is
     possible to create smaller windows, and also to indicate that the
     window is only partially visible on the screen.  Furthermore, large
     windows or pads, which are bigger than the actual screen size, may be
     created. 
     
     The routine newterm() may be called to "open" additional terminals by
     large applications wishing to manipulate several terminals at once.
     The set_term() function is used to select the terminal whose screen is
     to be updated by the next refresh(). 
     
     Interfaces are also defined to allow input character manipulation and
     to disable and enable many input attributes: character echo, single
     character input with or without signal processing (cbreak or raw
     modes), carriage returns mapping to newlines, screen scrolling, etc. 
     



     --------------------------------------------------------------------------





.......=================================== 
.......PDCurses Definitions and Variables:	
.......=================================== 
     
     All defines are "defined" here.  All compiler and environment
     specific definitions are defined into generic class defines.
     These defines are to be given values so that the code can
     rely on #if, rather than a complicated set of #if defined() or
     #ifdefs...
     
     PDCurses definitions list:  (Only define those needed)
     
     	REGISTERWINDOWS True for auto window update registery.
     	FAST_VIDEO      True if display is memory mapped, or
     	                we can utilize the fast video update routines.
     	DOS             True if compiling for DOS.
     	OS2             True if compiling for OS/2.
     	FLEXOS          True if compiling for Flexos.
     	HC              True if using a Metaware compiler.
     	TC              True if using a Borland compiler.
     	MSC             True if using a Microsoft compiler.
     	ANSI            True if the compiler supports ANSI C and
     	                (full or mixed) prototypes.
     	CPLUSPLUS       True if the compiler supports C++.
     
     PDCurses portable platform definitions list:
     
     	PDCurses        Enables access to PDCurses-only routines.
     	XOPEN           Always true.
     	SYSV            True if you are compiling for SYSV portability.
     	BSD             True if you are compiling for BSD portability.
     	INTERNAL        Enables access to internal PDCurses routines.
     	PDCURSES_WCLR   Makes behaviour of wclrtoeol() and wclrtoeof()
     	                unique to PDCurses. By default Unix behavior is set.
     	                See notes in wclrtoeol() and wclrtoeof().



     --------------------------------------------------------------------------
     
     PDCurses Text Attributes:
     
     To include colour in PDCurses, a number of things had to be sacrificed
     from the strict Unix and System V support.
     The main problem is fitting all character attributes and colour into
     an unsigned char (all 8 bits!). On System V, chtype is a long on
     PDCurses it is a short int.
     
     The following is the structure of a win->_attrs chtype:
     
     -------------------------------------------------
     |15|14|13|12|11|10| 9| 8| 7| 6| 5| 4| 3| 2| 1| 0|
     -------------------------------------------------
      colour number |  attrs |   character eg 'a'
     
     the high order char is an index into an array of physical colours
     (defined in INITPAIR.c). 32 (5 bits) foreground/background colour
     combinations combined with 8 (3 bits) attribute modifiers are
     available.
     
     The available attribute enhancers are bold, reverse and blink.
     All other Unix attributes have no effect as attributes. This
     limitation may be overcome in future releases by expanding chtype
     to a long.
     



     --------------------------------------------------------------------------





.......============================ 
.......Portable PDCurses Functions:		
.......============================ 
     
       Name:                                                       overlay
     
       Synopsis:
       	int	overlay(WINDOW *src_w, WINDOW *dst_w)
       	int	overwrite(WINDOW *src_w, WINDOW *dst_w)
       	int	copywin(WINDOW *src_w, WINDOW *dst_w, int src_tr,
       			int src_tc, int dst_tr, int dst_tc, int dst_br, int dst_bc,
       			bool overlay)
     
       X/Open Description:
      	The overlay() and overwrite() functions overlay src_w on top of 
      	dst_w; that is, all text in src_w is copied into dst_w. The windows
      	src_w and dst_w are not required to be the same size. The
      	copy starts at (0, 0) on each window. The difference between
      	the two functions is that overlay() is non-destructive
      	(blanks are not copied) while overwrite() is destructive
      	(blanks are copied).
     
      	copywin() is similar to overwrite() and overlay() but copywin()
      	does not require that the two windows overlap.
      	The arguments src_tc and src_tr specify the top left corner of the
      	region to be copied to the destination window.
      	The arguments dst_tc,dst_tr,dst_br,dst_bc specify the region within
      	the destination window to where the copy is made.
      	The argument overlay, if TRUE, indicates that the copy is done
      	non-destructively (as in overlay()). Blanks in the source window
      	are not copied to the destination window. When overlay is FALSE,
      	(as in overwrit()), the copy is destructive; blanks are copied
      	to the destination window.
     
       PDCurses Description:
      	The above description for overlay() and overwrite() is misleading 
      	in the actual behaviour exhibited by both SysV and BSD curses. 
      	The above implies that the character in 0,0 of the source window 
      	is copied to 0,0 of the destination window. What actually happens 
      	is that those characters in the source window that intersect with 
      	characters in the destination window RELATIVE TO ABSOLUTE 0,0 ON 
      	THE SCREEN, are copied to the destination window so that the 
      	characters appear in the same physical position on the screen.
     
      	Thanks to Andreas Otte (venn@@uni-paderborn.de) for the correction
      	and code changes.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           overlay                               Y        Y       Y
           overwrite                             Y        Y       Y
           copywin                               -        -      3.0
     



     --------------------------------------------------------------------------
     
       Name:                                                         clear
     
       Synopsis:
       	int clear(void);
       	int wclear(WINDOW *win);
       	int erase(void);
       	int werase(WINDOW *win);
       	int clrtobot(void);
       	int wclrtobot(WINDOW *win);
       	int clrtoeol(void);
       	int wclrtoeol(WINDOW *win);
     
       X/Open Description:
       	The erase() and werase() functions copy blanks to every position
       	of the window.
     
       	The clear() and wclear() functions are similar to erase() and
       	werase() except they also call clearok() to ensure that the
       	the screen is cleared on the next call to wrefresh() for that
       	window.
     
       	The clrtobot() and wclrtobot() functions clear the screen from
       	the current cursor position to the end of the current line and
       	all remaining lines in the window.
     
       	The clrtoeol() and wclrtoeol() functions clear the screen from
       	the current cursor position to the end of the current line only.
     
      	NOTE: clear(), wclear(), erase(), clrtobot(), and clrtoeol()
      	are implemented as macros
     
       PDCurses Description:
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       NOTE:
      	The behaviour of Unix curses is to clear the line with a space
      	and attributes of A_NORMAL. PDCurses clears the line with the
      	window's current attributes (including current colour). To get
      	the behaviour of PDCurses, #define ODCURSES_WCLR in curses.h or
      	add -DPDCURSES_WCLR to the compile switches.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           clear                                 Y        Y       Y
           wclear                                Y        Y       Y
           erase                                 Y        Y       Y
           werase                                Y        Y       Y
           clrtobot                              Y        Y       Y
           wclrtobot                             Y        Y       Y
           clrtoeol                              Y        Y       Y
           wclrtoeol                             Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                          beep
     
       Synopsis:
       	int beep(void);
       	int flash(void);
     
       X/Open Description:
      	These routines are used to signal the terminal user.  The beep()
      	function will sound the audible bell on the terminal, if possible
      	and if not, will flash the screen (visible bell), if possible.
      	The flash() function will flash the screen, and if that is not
      	possible, will sound the audible signal.  If neither signal is
      	possible, nothing will happen.  Nearly all terminals have an
      	audible signal (bell or beep), but only some can flash the screen.
     
       X/Open Return Value:
      	These functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for these functions.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           beep                                  Y        Y       Y
           flash                                 Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                          inch
     
       Synopsis:
       	chtype inch(void);
       	chtype winch(WINDOW *win);
       	chtype mvinch(int y, int x);
       	chtype mvwinch(WINDOW *win, int y, int x);
     
       X/Open Description:
     
      	NOTE: All these functions are implemented as macros.
     
       PDCurses Description:
      	Depending upon the state of the raw character output, 7- or
      	8-bit characters will be output.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           inch                                  Y        Y       Y
           winch                                 Y        Y       Y
           mvinch                                Y        Y       Y
           mvwinch                               Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                        getstr
     
       Synopsis:
       	int getstr(char *str);
       	int wgetstr(WINDOW *win, char *str);
       	int mvgetstr(int y, int x, char *str);
       	int mvwgetstr(WINDOW *win, int y, int x, char *str);
       	int wgetnstr(WINDOW *win, char *str, int ch);
     
       X/Open Description:
      	The effect of getstr() is as though a series of calls to getch()
      	were made, until a newline or carriage return is received. The
      	resulting value is placed in the area pointed to by *str. The user's
      	erase and kill characters are interpreted, as well as any special
      	keys; such as function keys.
     
      	With wgetnstr(), a series of characters are read until a newline 
      	or carriage return is received.  The resulting value is placed 
      	in the area pointed to by the character pointer str.  The user's 
      	erase and kill characters are interpreted.  This differs from 
      	wgetstr() in that the number of characters read is limited by a passed
      	argument.
     
      	NOTE: getstr(), mvgetstr() and mvwgetstr() are implemented as macros.
     
      	WARNING:  There is no way to know how long the buffer passed to
      		wgetstr() is, so it is possible to overwrite wrong
      		memory or code!! This is the reason for the wgetnstr()
      		function :-)
     
       X/Open Return Value:
      	This functions return ERR on failure or any other value on success.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           getstr                                Y        Y       Y
           wgtdch                                Y        Y       Y
           mvgetstr                              Y        Y       Y
           mvwgetstr                             Y        Y       Y
           wgetnstr                              -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                         color
     
       Synopsis:
       	int start_color(void);
       	int init_pair(short pair, short fg, short bg);
       ***	int init_color(short color, short red, short green, short blue);
       	bool has_colors(void);
       	bool can_change_color(void);
       ***	int color_content(short color, short *redp, short *greenp, short bluep);
       	int pair_content(short pair, short *fgp, short *bgp);
     
       X/Open Description:
      	To use these routines, start_color() must be called, usually
      	immediately after initscr(). Colors are always used in pairs refered
      	to as color-pairs. A color-pair consists of a foreground color and
      	a background color. A color-pair is initialized with init_pair().
      	After it has been initialized, COLOR_PAIR(n), a macro defined in
      	<curses.h>, can be used like any other video attribute.
     
      	start_color() initializes eight basic colors (black,red,green,yellow,
      	blue,magenta,cyan, and white), and two global variables; COLORS
      	and COLOR_PAIRS (respectively defining the maximum number of
      	colors and color-pairs the terminal is capable of displaying).
     
      	init_pair() changes the definitions of a color-pair.
      	The routine takes three arguments: the number of the color-pair
      	to be redefined, and the new values of the foreground and
      	background colors.
      	The value of color-pair must be between 1 and COLOR_PAIRS-1.
      	The values of foreground and background must be between 0 and
      	COLORS-1 (this is a PDCurses abberation; normally it is 0 and
      	COLORS)
      	If the color pair was previously initialized, the screen is refreshed
      	and all occurrences of that color-pair are changed to the new
      	definition.
     
      	has_colors() indicates if the terminal supports, and can maniplulate
      	color. It returns TRUE or FALSE.
     
      	can_change_color() indicates if the terminal has the capability 
      	to change the definition of its colors. Although this is possible,
      	at least with VGA monitors, this function always returns FALSE.
     
      	pair_content() is used to determine what the colors of a given 
      	color-pair consist of.
     
      	NOTE: has_colors() is implemented as a macro.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error except for
      	has_colors() and can_change_colors() which return TRUE or FALSE.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           start_color                           -        -      3.2
           init_pair                             -        -      3.2
           init_color                            -        -      3.2
           has_colors                            -        -      3.2
           can_change_color                      -        -      3.2
           color_content                         -        -      3.2
           pair_content                          -        -      3.2
     



     --------------------------------------------------------------------------
     
       Name:                                                         touch
     
       Synopsis:
       	int touchwin(WINDOW *win);
       	int touchline(WINDOW *win, int start,int count);
       	int untouchwin(WINDOW *win);
       	int wtouchln(WINDOW *win, int y, int n, int changed);
       	int is_linetouched(WINDOW *win,int line);
       	int is_wintouched(WINDOW *win);
     
       X/Open Description:
      	The touchwin() and touchline() functions throw away all optimisation 
      	information about which parts of the window have been touched, 
      	by pretending that the entire window has been drawn on.  
      	This is sometimes necessary when using overlapping
      	windows, since a change to one window will affect the other window,
      	but the records of which lines have been changed in the other
      	window will not reflect the change.
     
      	The untouchwin() routine marks all lines in the window as unchanged
      	since the last call to wrefresh().
     
      	The wtouchln() routine makes n lines in the window, starting at 
      	line y, look as if they have (changed=1) or have not (changed=0) 
      	been changed since the last call to wrefresh().
     
      	The is_linetouched() routine returns TRUE if the specified line 
      	in the specified window has been changed since the last call to 
      	wrefresh(). If the line has not changed, FALSE is returned.
     
      	The is_wintouched() routine returns TRUE if the specified window 
      	has been changed since the last call to wrefresh(). If the window 
      	has not changed, FALSE is returned.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error except
      	is_wintouched() and is_linetouched().
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           touchwin                              Y        Y       Y
           touchline                             Y        -      3.0
           untouchwin                            -        -      4.0
           wtouchln                              Y        Y       Y
           is_linetouched                        -        -      4.0
           is_wintouched                         -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        inopts
     
       Synopsis:
       	int cbreak(void);
       	int nocbreak(void);
       	int echo(void);
       	int noecho(void);
       ***	int halfdelay(int tenths);
       	int intrflush(WINDOW *win, bool bf);
       	int keypad(WINDOW *win, bool bf);
       	int meta(WINDOW *win, bool bf);
       	int nodelay(WINDOW *win, bool bf);
       	int notimeout(WINDOW *win, bool bf);
       	int raw(void);
       	int noraw(void);
       ***	void noqiflush(void);
       ***	void qiflush(void);
       ***	int timeout(int delay);
       ***	int wtimeout(WINDOW *win, int delay);
       	int typeahead(int fildes);
     
       X/Open Description:
      	cbreak() and nocbreak() puts the terminal into and out of cbreak
      	mode. In cbreak mode, characters typed by the user are immediately
      	available to the program and erase/kill character processing is
      	not performed.  When out of cbreak mode, the terminal driver
      	will buffer characters typed until a newline or carriage return
      	is typed.  Interrupt and flow control characters are unaffected
      	by this mode.  Initially the terminal may or may not need be
      	in cbreak mode.
     
      	echo() and noecho() control whether characters typed by the user
      	are echoed by the input routine.  Initially, input characters
      	are echoed.  Subsequent calls to echo() and noecho() do not
      	flush type-ahead.
     
      	If the intrflush() option is enabled (bf is TRUE), and an interrupt
      	is pressed on the keyboard (INTR, BREAK, or QUIT) all output in
      	the terminal driver queue will be flushed, giving the effect
      	of faster response to the interrupt but causing curses to have
      	the wrong idea of what is on the screen.  Disabling the option
      	prevents the flush.  The default for the option is inherited
      	from the terminal driver settings.  The window argument is
      	ignored.
     
      	The keypad() function changes the keypad option of the user's terminal.
      	If enabled (bf is TRUE), the user can press a function key (such
      	as the left arrow key) and getch() will return a single value
      	that represents the KEY_LEFT function key.  (See Section 11.3.3,
      	Input Values.)  If disabled, curses will not treat function keys
      	as special keys and the program has to interpret the escape
      	sequences itself.  If the keypad is enabled, the terminal keypad
      	is turned on before input begins.
     
      	The meta() function forces the user's terminal to return 7 or 8
      	significant bits on input.  To force 8 bits to be returned,
      	invoke meta() with bf as TRUE.  To force 7 bits to be returned,
      	invoke meta() with bf as FALSE.
      	The window argument is always ignored, but it must still be a
      	valid window to avoid compiler errors.
     
      	The nodelay() function controls whether wgetch() is a non-blocking
      	call. If the option is enabled, and no input is ready, wgetch()
      	will return ERR. If disabled, wgetch() will hang until input
      	is ready.
     
      	While interpreting an input escape sequence, wgetch sets a timer while
      	waiting for the next character.  If notimeout(win,TRUE) is called, then
      	wgetch does not set a timer.  The purpose of the timeout is to
      	differentiate between sequences received from a function key and those
      	typed by a user.
     
      	With raw() and noraw(), the terminal in placed into or out of raw 
      	mode.  Raw mode is similar to cbreak mode, in that characters typed 
      	are immediately passed through to the user program.  The differences
      	are that in raw mode, the INTR, QUIT, SUSP, and STOP characters are 
      	passed through without being interpreted, and without generating a
      	signal.  The behaviour of the BREAK key depends on other
      	parameters of the terminal drive that are not set by curses.
     
      	The curses package does the "line-breakout optimisation" by
      	looking for type-ahead periodically while updating the screen.
      	If input is found, the current update will be postponed until
      	refresh() or doupdate() are called again.  This allows faster
      	response to commands typed in advance.  Normally, the input FILE
      	pointer passed to newterm(), or stdin in the case when initscr()
      	was called, will be used to do this type-ahead checking.  The
      	typeahead() routine specified that the file descriptor fd is to
      	be used to check for type-ahead instead.  If fd is -1, then no
      	type-ahead checking will be done.
     
       PDCurses Description:
      	The meta() function is provided for portability.  By default, 8 bits
      	are returned.
     
      	notimeout() is a no-op in PDCurses.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           cbreak                                Y        Y       Y
           nocbreak                              Y        Y       Y
           echo                                  Y        Y       Y
           noecho                                Y        Y       Y
           halfdelay                             Y        Y       Y
           intrflush                             Y        Y       Y
           keypad                                Y        Y       Y
           meta                                  Y        Y       Y
           nodelay                               Y        Y       Y
           notimeout                             Y        Y       Y
           raw                                   Y        Y       Y
           noraw                                 Y        Y       Y
           noqiflush                             Y        Y       Y
           qiflush                               Y        Y       Y
           timeout                               Y        Y       Y
           wtimeout                              Y        Y       Y
           typeahead                             Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                       outopts
     
       Synopsis:
       	int clearok(WINDOW *win, bool bf);
       	int idlok( WINDOW *win, bool bf );
       	int idcok( WINDOW *win, bool bf );
       ***	int immedok( WINDOW *win, bool bf );
       	int leaveok( WINDOW *win, bool bf );
       	int setscrreg(int top, int bot);
       	int wsetscrreg(WINDOW *win, int top, int bot);
       	int scrollok(WINDOW *win, bool bf);
       	int nl(void);
       	int nonl(void);
     
       	int raw_output(bool bf);
     
       X/Open Description:
      	With the clearok() routine, if bf is TRUE, the next call to 
      	wrefresh() with this window will clear the screen completely 
      	and redraw the entire screen.
     
      	If idlok() is enabled (bf is TRUE), curses will use the insert/delete
      	line feature of terminals so equipped.  If disabled, curses will not
      	use this feature.  (The insert/delete character feature is always
      	used.)  This option should be enabled only if the application
      	needs insert/delete line; for example, for a screen editor.  It
      	is disabled by default because insert/delete line tends to be
      	visually annoying when used in applications where it isn't really
      	needed.  If insert/delete line cannot be used, curses will redraw
      	the changed portions of all lines.
     
      	With the idcok() routine, if enabled (bf is TRUE), curses considers
      	using the hardware insert/delete character feature of terminals so 
      	equipped. This is enabled by default.
     
      	Normally, the hardware cursor is left at the location of the
      	window being refreshed.  leaveok() allows the cursor to be
      	left whereever the update happens to leave it.  It is useful
      	for applications where the cursor is not used, since it reduces
      	the need for cursor motions.  If possible, the cursor is made
      	invisible when this option is enabled.
     
      	The setscrreg() and wsetscrreg() functions allow the user to set 
      	a software scrolling region in a window.  The parameters 'top' 
      	and 'bot' are the line numbers of the top and bottom margin of the 
      	scrolling region.  (Line 0 is the top line of the window.)  If this 
      	option and scrollok() are enabled, an attempt to move off the bottom 
      	margin will cause all lines in the scrolling region to scroll up one 
      	line.  Only the text of the window is scrolled.
     
      	The nl() function enables the translation of newline into a
      	carriage return and a line-feed on output, and a carriage return
      	is translated into a newline on input.  Initially, the translations
      	do occur.  By disabling these translations, curses is able to
      	make better use of the line-feed capability, resulting in faster
      	cursor motion.
     
      	The nonl() function disables the translation of newline into a
      	carriage return and a line-feed on output, and a carriage return
      	is translated into a newline on input.  Initially, the translations
      	do occur.  By disabling these translations, curses is able to
      	make better use of the line-feed capability, resulting in faster
      	cursor motion.
     
       PDCurses Description:
      	idlok() and idcok() are provided for portability.  
      	This may have some value for the FLEXOS platform, but there is 
      	no code in PDCurses at this time (910725) that takes advantage 
      	of this.
     
      	Though not explicitly stated, the cursor will be made visible
      	again, if leaveok() is passed a FALSE value.
      			(Frotz:911226 Comments?)
     
      	The raw_output() routine enables the output of raw characters using 
      	the 'standard' *add* and *ins* curses functions.
      	To force 8 bit output and no translation of control characters,
      	invoke raw_output() with bf as TRUE.  To force 7 bit output and
      	translation of control characters, invoke raw_output() with bf
      	as FALSE.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           clearok                               Y        Y       Y
           idlok                                 Y        Y       Y
           idcok                                 -        -      4.0
           immedok                               -        -      4.0
           leaveok                               Y        Y       Y
           setscrreg                             Y        Y       Y
           wsetscrreg                            Y        Y       Y
           scrollok                              Y        Y       Y
           nl                                    Y        Y       Y
           nonl                                  Y        Y       Y
           raw_output                            -        -       -
     



     --------------------------------------------------------------------------
     
       Name:                                                        window
     
       Synopsis:
       	WINDOW *newwin(int nlines, int ncols, int begy, int begx);
       	int delwin(WINDOW *win);
       	int mvwin(WINDOW *win, int y, int x);
       	WINDOW *subwin(WINDOW* orig,int nlines,int ncols,
       			int begin_y,int begin_x);
       	WINDOW *derwin(WINDOW* orig,int nlines,int ncols,
       			int begin_y,int begin_x);
       	int mvderwin(WINDOW *win, int par_y, int par_x);
       	int dupwin(WINDOW *win);
       ***	void wsyncup(WINDOW *win);
       ***	int syncok(WINDOW *win, bool bf);
       ***	void wcursyncup(WINDOW *win);
       ***	void wsyncdown(WINDOW *win);
     
       	WINDOW *resize_window(WINDOW *w, int lins, int cols);
     
       X/Open Description:
      	newwin() creates a new window with the given number of lines, 
      	nlines and columns, ncols. The upper left corner of the window is 
      	at line begy, column begx. If either nlines or ncols is zero,
      	they will be defaulted to LINES - begy and COLS - begx. A
      	new full-screen window is created by calling newwin(0, 0, 0, 0).
     
      	delwin() deletes the named window, freeing all memory associated 
      	with it. In the case of overlapping windows, subwindows should be 
      	deleted before the main window.
     
      	mvwin() moves the window so that the upper left-hand corner is at
      	position (y,x). If the move would cause the window to be off the
      	screen, it is an error and the window is not moved. Moving subwindows
      	is allowed, but should be avoided. (I don't know why ?).
     
      	subwin() creates a new sub-window within a window.  The
      	dimensions of the sub-window are nlines lines and ncols
      	columns.  The sub-window is at position (begin_y, begin_x) on
      	the screen.  This position is relative to the screen, and not
      	to the window orig.
      	The sub-window is made in the middle of the window orig, so
      	that changes made to either window will affect both.  When
      	using this routine, it will often be necessary to call
      	touchwin() before calling wrefresh().
     
      	derwin() is the same as subwin(), except that begin_y and
      	begin_x are relative to the origin of the window orig rather than
      	the screen.  There is no difference between subwindows and derived
      	windows.
     
      	mvderwin() moves a derived window (or subwindow) inside its
      	parent window.  The screen-relative parameters of the window are not
      	changed.  This routine is used to dispay different parts of the parent
      	window at the same physical position on the screen.
     
      	dupwin() creates an exact duplicate of the window win.
     
       PDCurses Description:
      	resize_window() does necessary initializations for the PDCurses 
      	package when doing screen size changes. The user is responsible for
      	deleting and/or resizing windows after this call is made.
     
      	See the call PDC_resize_win().
     
      	WARNING: This routine deallocated the existing stdscr, curscr
      	and modifies LINES, COLS and other internal PDCurses
      	variables.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       PDCurses Errors:
      	It is an error to call resize_window()function before calling initscr().
      	Also, an error will be generated if we fail to create a newly
      	sized replacement window for curscr, or stdscr.
      	This will typically happen when increasing the window size.
     
      	NOTE:  If this happens, the previously successfully allocated
      	windows are left alone.  i.e. The resize is NOT cancelled for
      	those windows.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           newwin                                Y        Y       Y
           delwin                                Y        Y       Y
           mvwin                                 Y        Y       Y
           subwin                                Y        Y       Y
           derwin                                Y        -       Y
           mvderwin                              Y        -       Y
           dupwin                                -        -      4.0
           wsyncup                               -        -      4.0
           syncok                                -        -      4.0
           wcursyncup                            -        -      4.0
           wsyncdown                             -        -      4.0
           resize_window                         -        -       -
     



     --------------------------------------------------------------------------
     
       Name:                                                        kernel
     
       Synopsis:
       	int def_prog_mode(void);
       	int def_shell_mode(void);
       	int reset_prog_mode(void);
       	int reset_shell_mode(void);
       	int resetty(void);
       	int savetty(void);
       ***	int getsyx(int y, int x);
       ***	int setsyx(int y, int x);
       ***	int ripoffline(int line, init(*int)(WINDOW *,int));
       	int curs_set(int visibility);
       	int napms(int ms);
     
       X/Open Description:
      	The def_prog_mode() and def_shell_mode() functions save the 
      	current terminal modes as the "program" (in curses) or
      	"shell" (not in curses) state for use by the reset_prog_mode()
      	and reset_shell_mode() functions.  This is done automatically by
      	initscr().
     
      	The reset_prog_mode() and reset_shell_mode() functions restore 
      	the terminal to "program" (in curses) or "shell" (not in curses)
      	state.  These are done automatically by endwin()
      	and doupdate() after an endwin(), so they would normally not
      	be called before these functions.
     
      	The savetty() and resetty() routines save and restore the state of 
      	the terminal modes. The savetty() function saves the current state 
      	in a buffer and resetty() restores the state to what it was at the 
      	last call to savetty().
     
       PDCurses Description:
      	FYI: It is very unclear whether savetty() and resetty() functions
      	are a duplication of the reset_prog_mode() and reset_shell_mode() 
      	functions or whether this is a backing store type of operation.  
      	At this time, they are implemented similar to the reset_*_mode() 
      	routines.
     
      	The curs_set() routine is used to set the visibility of the cursor.
      	The cursor can be made invisible, normal or highly visible by setting
      	the parameter to 0, 1 or 2 respectively. If an invalid value is passed
      	the function will set the cursor to "normal".
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error except curs_set()
      	which returns the previous visibility.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           def_prog_mode                         Y        Y       Y
           def_shell_mode                        Y        Y       Y
           reset_prog_mode                       Y        Y       Y
           reset_shell_mode                      Y        Y       Y
           resetty                               Y        Y       Y
           savetty                               Y        Y       Y
           getsyx                                -        -      3.0
           setsyx                                -        -      3.0
           ripoffline                            -        -      3.0
           curs_set                              -        -      3.0
           napms                                 Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                          bkgd
     
       Synopsis:
       	int bkgd(chtype ch);
       	void bkgdset(chtype ch);
       	int wbkgd(WINDOW *win, chtype ch);
       	void wbkgdset(WINDOW *win, chtype ch);
     
       UNIX System V (Rel 3.2 or 4) Description:
      	The bkgdset() and wbkgdset() routines manipulate the backgound of 
      	the named window.  Background is a chtype consisting of any combination 
      	of attributes and non-blank characters that are written into the window
      	with waddch().  Both the character and attribute parts of the background
      	are combined with the blank characters.  The background becomes a 
      	property of the character and moves with the character through any 
      	scrolling and insert/delete line/character operations.  To the extent
      	possible on a particular terminal, the attribute part of the background
      	is displayed as the graphic rendition of the character put on the
      	screen.
     
      	The bkgd() and wbkgd() routines combine the new background with every
      	position in the window.  Background is any combination of attributes
      	and a character.  Only the attribute part is used to set the background
      	of non-blank characters, while both character and attributes are used
      	for blank positions.  To the extent possible on a particular terminal,
      	the attribute part of the background is displayed as the graphic
      	rendition of the character put on the screen.
     
       PDCurses Description:
      	The attributes that are defined with the attrset()/attron() set of 
      	functions take precedence over the background attributes if there is a
      	conflict (e.g., different color pairs).
     
       System V Return Value:
      	bkgd() and wbkgd() return the integer OK, or a non-negative integer, if
      	immedok() is set.
     
       PDCurses Return Value:
      	bkgd() and wbkgd() return the integer OK ( the function immedok() is
      	currently not supported ).
     
       System V Notes:
      	Note that bkgdset() and bkgd() may be macros.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           bkgd                                  -        -      4.0
           bkgdset                               -        -      4.0
           wbkgd                                 -        -      4.0
           wbkgdset                              -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                         getyx
     
       Synopsis:
       	void getyx(WINDOW *win, int y, int x);
       	void getparyx(WINDOW *win, int y, int x);
       	void getbegyx(WINDOW *win, int y, int x);
       	void getmaxyx(WINDOW *win, int y, int x);
     
       	int getbegy(WINDOW *win);
       	int getbegx(WINDOW *win);
       	int getpary(WINDOW *win);
       	int getparx(WINDOW *win);
       	int getmaxy(WINDOW *win);
       	int getmaxx(WINDOW *win);
     
       X/Open Description:
      	With the getyx() macro, the cursor position of the window is placed
      	in the two integer variables y and x. getbegyx() and getmaxyx()
      	return the current beginning coordinates and size of the specified
      	window respectively. getparyx() returns the beginning coordinates 
      	of the parent's window if the specified window is a sub-window
      	otherwise -1 is returned.
     
      	These functions are implemented as macros, so their return values
      	are undefined. Also as these functions are implemented as macros,
      	no '&' is required before the variables y and x.
     
      	The functions getbegy(), getbegx(), getmaxy(), getmaxx() ,
      	getpary(), and getparx() do return the beginning coordinate or 
      	size values, so they can be specified on the right-hand side of 
      	an assignment statement. They are, however, implemented as macros.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           getyx                                 Y        Y       Y
           getparyx                              -        -      4.0
           getbegyx                              -        -      3.0
           getmaxyx                              -        -      3.0
           getbegy                               -        -       -
           getbegx                               -        -       -
           getpary                               -        -       -
           getparx                               -        -       -
           getmaxy                               -        -       -
           getmaxx                               -        -       -
     



     --------------------------------------------------------------------------
     
       Name:                                                       initscr
     
       Synopsis:
       	WINDOW *initscr(void);
       	int endwin(void);
       ***	int isendwin(void);
       ***	SCREEN *newterm(char *type, FILE *outfd, FILE *infd);
       	SCREEN *set_term(SCREEN *new);
       ***	void delscreen(SCREEN *sp);
     
       ***	int resize_screen(int nlines);
     
       X/Open Description:
      	The first curses routine called should be initscr().  This will
      	determine the terminal type and initialize all curses data
      	structures.  The initscr() function also arranges that the
      	first call to refresh() will clear the screen.  If errors
      	occur, initscr() will write an appropriate error message to
      	standard error and exit.  If the program wants an indication
      	of error conditions, newterm() should be used instead of
      	initscr().
     
      	A program should always call endwin() before exiting or
      	escaping from curses mode temporarily.  This routine will
      	restore tty modes, move the cursor to the lower left corner
      	of the screen and reset the terminal into the proper non-visual
      	mode.  To resume curses after a temporary escape, refresh() or
      	doupdate() should be called.
     
      	A program which outputs to more than one terminal should use
      	newterm() for each terminal instead of initscr().  The newterm()
      	function should be called once for each terminal.  It returns a 
      	value of type SCREEN* which should be saved as a reference to that
      	terminal. The arguments are the type of of terminal to be used
      	in place of TERM (environment variable), a file pointer for output
      	to the terminal and another file pointer for input from the terminal.
      	The program must also call endwin() for each terminal no longer being
      	used.
     
      	This function is used to switch between different terminals.
      	The screen reference 'new' becomes the new current terminal.
      	The previous terminal is returned by the routine.  This is the
      	only routine which manipulates SCREEN pointers; all other routines
      	affect only the current terminal.
     
       PDCurses Description:
      	Due to the fact that newterm() does not yet exist in PDCurses,
      	there is no way to recover from an error in initscr().
     
       X/Open Return Value:
      	All functions return NULL on error, except endwin(), which
      	returns ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           initscr                               Y        Y       Y
           endwin                                Y        Y       Y
           isendwin                              -        -      3.0
           newterm                               -        -       Y
           set_term                              -        -       Y
           delscreen                             -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        border
     
       Synopsis:
       	int border(chtype ls, chtype rs, chtype ts, chtype bs, chtype tl, 
       	           chtype tr, chtype bl, chtype br);
       	int wborder(WINDOW *win, chtype ls, chtype rs, chtype ts, 
       	            chtype bs, chtype tl, chtype tr, chtype bl, chtype br);
       	int box(WINDOW *win, chtype verch, chtype horch);
       	int hline(chtype ch, int n);
       	int vline(chtype ch, int n);
       	int whline(WINDOW *win, chtype ch, int n);
       	int wvline(WINDOW *win, chtype ch, int n);
     
       X/Open Description:
      	The border(), wborder(), and box() routines, a border is drawn
      	around the edge of the window. If any of the arguments is zero,
      	an appropriate default is used. These default values are specified
      	in <curses.h>. The arguments and defaults to these functions are:
      		ls	left side of border		ACS_VLINE
      		rs	right side of border		ACS_VLINE
      		ts	top side of border		ACS_HLINE
      		bs	bottom side of border		ACS_HLINE
      		tl	top left corner of border	ACS_ULCORNER
      		tr	top right corner of border	ACS_URCORNER
      		bl	bottom left corner of border	ACS_BLCORNER
      		br	bottom right corner of border	ACS_BLCORNER
     
      	The hline() and whline() functions draw a left to right line
      	using ch starting from the current cursor position. The cursor
      	position does not change. The line is at most n characters long
      	or as many as will fit in the window.
     
      	The vline() and wvline() functions draw a top to bottom line
      	using ch starting from the current cursor position. The cursor
      	position does not change. The line is at most n characters long
      	or as many as will fit in the window.
     
      	border(), box(), hline(), and vline() are implemented as macros.
     
       X/Open Return Value:
      	These functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for these functions.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           border                                -        -      4.0
           wborder                               -        -      4.0
           box                                   Y        Y       Y
           hline                                 -        -      4.0
           whline                                -        -      4.0
           vline                                 -        -      4.0
           wvline                                -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                         addch
     
       Synopsis:
       	int addch(chtype ch);
       	int waddch(WINDOW *win, chtype ch);
       	int mvaddch(int y, int x, chtype ch);
       	int mvwaddch(WINDOW *win, int y, int x, chtype ch);
       	int echochar(chtype ch);
       	int wechochar(WINDOW *win, chtype ch);
     
       X/Open Description:
      	The routine addch() inserts the character ch into the default
      	window at the current cursor position and the window cursor is
      	advanced.  The character is of the type chtype as containing
      	both data and attributes.
     
      	The routine waddch() inserts the character ch into the specified
      	window at the current cursor position.  The cursor position is
      	advanced.
     
      	The routine mvaddch() moves the cursor to the specified (y, x)
      	position and inserts the character ch into the default window.
      	The cursor position is advanced after the character has been
      	inserted.
     
      	The routine mvwaddch() moves the cursor to the specified (y, x)
      	position and inserts the character ch into the specified
      	window.  The cursor position is advanced after the character
      	has been inserted.
     
      	The routine echochar() inserts the character ch into stdscr
      	at the current cursor position and a refresh() is called.  
      	The cursor position is advanced.
     
      	The routine wechochar() inserts the character ch into the
      	specified window at the current cursor position and a wrefresh() 
      	is called. The cursor position is advanced.
     
      	All these routines are similar to putchar().  The following
      	information applies to all the routines.
     
      	If the cursor moves on to the right margin, an automatic
      	newline is performed.  If scrollok is enabled, and a character
      	is added to the bottom right corner of the screen, the
      	scrolling region will be scrolled up one line.  If scrolling
      	is not allowed, ERR will be returned.
     
      	If ch is a tab, newline, or backspace, the cursor will be
      	moved appropriately within the window.  If ch is a newline,
      	the clrtoeol routine is called before the cursor is moved to
      	the beginning of the next line.  If newline mapping is off,
      	the cursor will be moved to the next line, but the x
      	coordinate will be unchanged.  If ch is a tab the cursor is
      	moved to the next tab position within the window.  If ch is
      	another control character, it will be drawn in the ^X
      	notation.  Calling the inch() routine after adding a control
      	character returns the representation of the control character,
      	not the control character.
     
      	Video attributes can be combined with a character by ORing
      	them into the parameter.  This will result in these attributes
      	being set.  The intent here is that text, including
      	attributes, can be copied from one place to another using inch()
      	and addch().
     
      	NOTE: All these functions are implemented as macros.
     
       PDCurses Description:
      	Depending upon the state of the raw character output, 7- or
      	8-bit characters will be output.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           addch                                 Y        Y       Y
           waddch                                Y        Y       Y
           mvaddch                               Y        Y       Y
           mvwaddch                              Y        Y       Y
           echochar                              -        -      3.0
           wechochar                             -        -      3.0
     



     --------------------------------------------------------------------------
     
       Name:                                                         delch
     
       Synopsis:
       	int delch(void);
       	int wdelch(WINDOW *win);
       	int mvdelch(int y, int x);
       	int mvwdelch(WINDOW *win, int y, int x);
     
       X/Open Description:
      	The character under the cursor in the window is deleted.  All
      	characters to the right on the same line are moved to the left
      	one position and the last character on the line is filled with
      	a blank.  The cursor position does not change (after moving to
      	y, x if coordinates are specified).
     
      	NOTE: delch(), mvdelch(), and mvwdelch() are implemented as macros.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       NOTE:
      	The behaviour of Unix curses is to display a blank in the last
      	column of the window with the A_NORMAL attribute. PDCurses
      	displays the blank with the window's current attributes 
      	(including current colour). To get the behaviour of PDCurses,
      	#define PDCURSES_WCLR in curses.h or add -DPDCURSES_WCLR to the 
      	compile switches.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           delch                                 Y        Y       Y
           wdelch                                Y        Y       Y
           mvdelch                               Y        Y       Y
           mvwdelch                              Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                         insch
     
       Synopsis:
       	int insch(chtype ch);
       	int winsch(WINDOW *win, chtype ch);
       	int mvinsch(int y, int x, chtype ch);
       	int mvwinsch(WINDOW *win, int y, int x, chtype ch);
     
       X/Open Description:
      	The routine insch() inserts the character ch into the default
      	window at the current cursor position and the window cursor is
      	advanced.  The character is of the type chtype as containing
      	both data and attributes.
     
      	The routine winsch() inserts the character ch into the specified
      	window at the current cursor position.  The cursor position is
      	advanced.
     
      	The routine mvinsch() moves the cursor to the specified (y, x)
      	position and inserts the character ch into the default window.
      	The cursor position is advanced after the character has been
      	inserted.
     
      	The routine mvwinsch() moves the cursor to the specified (y, x)
      	position and inserts the character ch into the specified
      	window.  The cursor position is advanced after the character
      	has been inserted.
     
      	The routine echochar() inserts the character ch into stdscr
      	at the current cursor position and a refresh() is called.  
      	The cursor position is advanced.
     
      	The routine wechochar() inserts the character ch into the
      	specified window at the current cursor position and a wrefresh() 
      	is called. The cursor position is advanced.
     
      	All these routines are similar to putchar().  The following
      	information applies to all the routines.
     
      	If the cursor moves on to the right margin, an automatic
      	newline is performed.  If scrollok is enabled, and a character
      	is added to the bottom right corner of the screen, the
      	scrolling region will be scrolled up one line.  If scrolling
      	is not allowed, ERR will be returned.
     
      	If ch is a tab, newline, or backspace, the cursor will be
      	moved appropriately within the window.  If ch is a newline,
      	the clrtoeol routine is called before the cursor is moved to
      	the beginning of the next line.  If newline mapping is off,
      	the cursor will be moved to the next line, but the x
      	coordinate will be unchanged.  If ch is a tab the cursor is
      	moved to the next tab position within the window.  If ch is
      	another control character, it will be drawn in the ^X
      	notation.  Calling the inch() routine after adding a control
      	character returns the representation of the control character,
      	not the control character.
     
      	Video attributes can be combined with a character by ORing
      	them into the parameter.  This will result in these attributes
      	being set.  The intent here is that text, including
      	attributes, can be copied from one place to another using inch()
      	and insch().
     
      	NOTE: All these functions are implemented as macros.
     
       PDCurses Description:
      	Depending upon the state of the raw character output, 7- or
      	8-bit characters will be output.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           insch                                 Y        Y       Y
           winsch                                Y        Y       Y
           mvinsch                               Y        Y       Y
           mvwinsch                              Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                        printw
     
       Synopsis:
       	int printw(char *fmt, ...);
       	int wprintw(WINDOW *win, char *fmt, ...);
       	int mvprintw(int y, int x, char *fmt, ...);
       	int mvwprintw(WINDOW *win, int y, int x, char *fmt,...);
       ***	int vwprintw(WINDOW *win, char *fmt, va_list varglist);
     
       X/Open Description:
      	The printw() routine adds a string to the default window
      	starting at the current cursor position.  This routine causes
      	the string that would normally be output by printf() to be
      	output by addstr().
     
      	The routine wprintw() adds a string to the specified window
      	starting at the current cursor position.  This routine causes
      	the string that would normally be output by printf() to be
      	output by waddstr().
     
      	The routine mvprintw() adds a string to the default window
      	starting at the specified cursor position.  This routine
      	causes the string that would normally be output by printf() to
      	be output by addstr().
     
      	The routine mvwprintw() adds a string to the specified window
      	starting at the specified cursor position.  This routine
      	causes the string that would normally be output by printf() to
      	be output by waddstr().
     
      	All these routines are analogous to printf().  It is advisable
      	to use the field width options of printf() to avoid leaving
      	unwanted characters on the screen from earlier calls.
     
       PDCurses Description:
      	The old Bjorn Larssen code for the 68K platform has been removed
      	from this module.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           printw                                Y        Y       Y
           wprintw                               Y        Y       Y
           mvprintw                              Y        Y       Y
           mvwprintw                             Y        Y       Y
           vwprintw                              -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                         scanw
     
       Synopsis:
       	int scanw(char *fmt, ...);
       	int wscanw(WINDOW *win, char *fmt, ...);
       	int mvscanw(int y, int x, char *fmt, ...);
       	int mvwscanw(WINDOW *win, int y, int x, char *fmt,...);
       ***	int vwscanw(WINDOW *win, char *fmt, va_list varglist);
     
       X/Open Description:
      	These routines correspond to scanf(). The function scanw() reads
      	input from the default window. The function wscanw() reads
      	input from the specified window. The function mvscanw() moves
      	the cursor to the specified position and then reads input from
      	the default window. The function mvwscanw() moves the cursor to
      	the specified position and then reads input from the specified
      	window.
     
      	For all the functions, the routine wgetstr() is called to get a
      	string from the window, and the resulting line is used as
      	input for the scan.  All character interpretation is carried
      	out according to the scanf function rules.
     
       PDCurses Description:
      	The old Bjorn Larssen code for the 68K platform has been removed
      	from this module.
     
       X/Open Return Value:
      	Upon successful completion, the scanw, mvscanw, mvwscanw and
      	wscanw functions return the number of items successfully
      	matched.  On end-of-file, they return EOF.  Otherwise they
      	return ERR.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           scanw                                 Y        Y       Y
           wscanw                                Y        Y       Y
           mvscanw                               Y        Y       Y
           mvwscanw                              Y        Y       Y
           vwscanw                               -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        instr
     
       Synopsis:
       	int instr(char *str);
       	int innstr(char *str, int n);
       	int winstr(WINDOW *win, char *str);
       	int winnstr(WINDOW *win, char *str, int n);
       	int mvinstr(int y, int x, char *str);
       	int mvinnstr(int y, int x, char *str, int n);
       	int mvwinstr(WINDOW *, int y, int x, char *str);
       	int mvwinnstr(WINDOW *, int y, int x, char *str, int n);
     
       System V Curses Description:
      	These routines return a string of characters in str, starting at the
      	current cursor position in the named window and ending at the right
      	margin of the window.  Attributes are stripped from the characters.
      	The four functions with n as the last argument return the string at most
      	n characters long.
     
      	Note:  instr(), innstr(), winstr(), mvinstr(), mvinnstr(),
      		mvwinstr() and mvwinnstr() are all macros.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           instr                                 -        -      4.0
           winstr                                -        -      4.0
           mvinstr                               -        -      4.0
           mvwinstr                              -        -      4.0
           innstr                                -        -      4.0
           winnstr                               -        -      4.0
           mvinnstr                              -        -      4.0
           mvwinnstr                             -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        insstr
     
       Synopsis:
       	int insstr(char *str);
       	int insnstr(char *str, int n);
       	int winsstr(WINDOW *win, char *str);
       	int winsnstr(WINDOW *win, char *str, int n);
       	int mvinsstr(int y, int x, char *str);
       	int mvinsnstr(int y, int x, char *str, int n);
       	int mvwinsstr(WINDOW *, int y, int x, char *str);
       	int mvwinsnstr(WINDOW *, int y, int x, char *str, int n);
     
       System V Curses Description:
       	With these routines, a character string (as many characters as 
       	will fit on the line) is inserted before the character under 
       	the cursor.  All characters to the right of the cursor are moved 
       	to the right, with the possibility of the rightmost characters 
       	on the line being lost.  The cursor position does not change 
       	(after moving to y,x if specified).  The four routines with n as 
       	the last argument insert at most n characters.  If n<=0, then 
       	the entire string is inserted.
     
      	NOTE:	insstr(), mvinsstr(), and mvwinsstr() are implemented as macros.
      		insnstr(), mvinsnstr(), and mvwinsnstr() are implemented as macros.
     
       PDCurses Description:
      	The *raw*() routines output 8 bit values.  These contrast to their
      	normal counterparts which output 7 bit values and convert control
      	character to the ^X notation.
     
      	str is a standard 8 bit character string WITHOUT embedded attributes.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           insstr                                -        -      4.0
           winsstr                               -        -      4.0
           mvinsstr                              -        -      4.0
           mvwinsstr                             -        -      4.0
           insnstr                               -        -      4.0
           winsnstr                              -        -      4.0
           mvinsnstr                             -        -      4.0
           mvwinsnstr                            -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                      deleteln
     
       Synopsis:
       	int deleteln(void);
       	int wdeleteln(WINDOW *win);
       	int insdelln(int n);
       	int winsdelln(WINDOW *win, int n);
       	int insertln(void);
       	int winsertln(WINDOW *win);
     
       X/Open Description:
      	With the deleteln() and wdelteln() functions,
      	the line under the cursor in the window is deleted.  All
      	lines below the current line are moved up one line.  The
      	bottom line of the window is cleared.  The cursor position
      	does not change.
     
      	With the insertln() and winsertn() functions,
      	a blank line is inserted above the current line and the bottom
      	line is lost.
     
      	NOTE: deleteln() and insertln() are implemented as macros.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       NOTE:
      	The behaviour of Unix curses is to clear the line with a space
      	and attributes of A_NORMAL. PDCurses clears the line with the
      	window's current attributes (including current colour). To get
      	the behaviour of PDCurses, #define PDCURSES_WCLR in curses.h or
      	add -DPDCURSES_WCLR to the compile switches.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           deleteln                              Y        Y       Y
           wdeleteln                             Y        Y       Y
           insdelln                              -        -      4.0
           winsdelln                             -        -      4.0
           insertln                              Y        Y       Y
           winsertln                             Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                          util
     
       Synopsis:
       	char *unctrl(chtype c);
       	char *keyname(int key);
       ***	int filter(void);
       ***	void use_env(void);
       ***	int putwin(WINDOW *win, FILE *filep);
       ***	WINDOW *getwin(FILE *filep);
       	int delay_output( int ms );
       	int flushinp(void);
     
       X/Open Description:
      	The unctrl() routine expands the character c into a character
      	string which is a printable representation of the character.
      	Control characters are displayed in the ^X notation.  Printing
      	characters are displayed normally.
     
      	The keyname() function returns a pointer to a character string 
      	containing a symbolic name corresponding to that specified in 
      	the argument key. key may be any key returned by wgetch().
     
      	The delay_output() function inserts ms millisecond pause in output.
      	On some systems, this has no effect.
     
      	The flushinp() routine throws away any type-ahead that has been 
      	typed by the user and has not yet been read by the program.
     
       PDCurses Description:
      	The conversion from a control character to a two-character
      	sequence is done by the unctrl() function. In the BSD version
      	of curses it is done by a macro, which uses a publicly
      	available translation table. Some ill-behaved application
      	programs use the table directly, and since it does not exist
      	in this curses version such application will link with an
      	error message complaining about undefined symbols.
     
      	If the PDCurses library is compiled under DOS with the FAST_VIDEO
      	define true, then we will poke the BIOS keyboard buffer head and
      	tail pointers, resetting the typeahead to implement flushinp().
      	If this is not true, then we will be unable to reliably flush
      	the typeahead.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           unctrl                                Y        Y       Y
           keyname                               Y        -      3.0
           filter                                Y        -      3.0
           use_env                               -        -      4.0
           put_win                               -        -      4.0
           get_win                               -        -      4.0
           delay_output                          Y        Y       Y
           flushinp                              Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                        addchstr
     
       Synopsis:
       	int addchstr(chtype *ch);
       	int addchnstr(chtype *ch, int n);
       	int waddchstr(WINDOW *win, chtype *ch);
       	int waddchnstr(WINDOW *win, chtype *ch, int n);
       	int mvaddchstr(int y, int x, chtype *ch);
       	int mvaddchnstr(int y, int x, chtype *ch, int n);
       	int mvwaddchstr(WINDOW *, int y, int x, chtype *ch);
       	int mvwaddchnstr(WINDOW *, int y, int x, chtype *ch, int n);
     
       X/Open Description:
      	These routines write a chtype directly into the window structure
      	starting at the current position.
      	The four routines with n as the last argument copy at most n
      	elements, but no more than will fit on the line.
      	If n=-1 then the whole string is copied, to the maximum number
      	that will fit on the line.
     
      	The cursor position is not advanced. These routines do not check for
      	newline or other special characters, no does any line wrapping occur.
     
      	NOTE:	addchstr(), mvaddchstr(), mvwaddchstr() addchnstr(), 
      		mvaddchnstr(), and mvwaddchnstr() are implemented as macros.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           addchstr                              -        -      4.0
           waddchstr                             -        -      4.0
           mvaddchstr                            -        -      4.0
           mvwaddchstr                           -        -      4.0
           addchnstr                             -        -      4.0
           waddchnstr                            -        -      4.0
           mvaddchnstr                           -        -      4.0
           mvwaddchnstr                          -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        inchstr
     
       Synopsis:
       	int inchstr(chtype *ch);
       	int inchnstr(chtype *ch, int n);
       	int winchstr(WINDOW *win, chtype *ch);
       	int winchnstr(WINDOW *win, chtype *ch, int n);
       	int mvinchstr(int y, int x, chtype *ch);
       	int mvinchnstr(int y, int x, chtype *ch, int n);
       	int mvwinchstr(WINDOW *, int y, int x, chtype *ch);
       	int mvwinchnstr(WINDOW *, int y, int x, chtype *ch, int n);
     
       X/Open Description:
      	These routines read a chtype string directly from the window structure
      	starting at the current position and ending at the right margin.
      	The four routines with n as the last argument copy at most n
      	elements, but no more than will fit on the line.
     
      	NOTE:	inchstr(), mvinchstr(), mvwinchstr() inchnstr(), 
      		mvinchnstr(), and mvwinchnstr() are implemented as macros.
     
       X/Open Return Value:
      	All functions return ERR on error and an integer value other than
      	ERR on success. 
     
       X/Open Errors:
      	Normally the value returned indicates the number of chtypes
      	returned.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           inchstr                               -        -      4.0
           winchstr                              -        -      4.0
           mvinchstr                             -        -      4.0
           mvwinchstr                            -        -      4.0
           inchnstr                              -        -      4.0
           winchnstr                             -        -      4.0
           mvinchnstr                            -        -      4.0
           mvwinchnstr                           -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        addstr
     
       Synopsis:
       	int addstr(char *str);
       	int addnstr(char *str, int n);
       	int waddstr(WINDOW *win, char *str);
       	int waddnstr(WINDOW *win, char *str, int n);
       	int mvaddstr(int y, int x, char *str);
       	int mvaddnstr(int y, int x, char *str, int n);
       	int mvwaddstr(WINDOW *, int y, int x, char *str);
       	int mvwaddnstr(WINDOW *, int y, int x, char *str, int n);
     
       X/Open Description:
      	These routines write all the characters of the null-terminated
      	string str on the given window.  The functionality is equivalent
      	to calling waddch() once for each character in the string.  The four
      	routines with n as the last argument write at most n characters.  If n
      	is negative, then the entire string will be added.
     
      	NOTE:	addstr(), mvaddstr(), and mvwaddstr() are implemented as macros.
      		addnstr(), mvaddnstr(), and mvwaddnstr() are implemented as macros.
     
       PDCurses Description:
      	The *raw*() routines output 8 bit values.  These contrast to their
      	normal counterparts which output 7 bit values and convert control
      	character to the ^X notation.
     
      	str is a standard 8 bit character string WITHOUT embedded attributes.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           addstr                                Y        Y       Y
           waddstr                               Y        Y       Y
           mvaddstr                              Y        Y       Y
           mvwaddstr                             Y        Y       Y
           addnstr                               -        -      4.0
           waddnstr                              -        -      4.0
           mvaddnstr                             -        -      4.0
           mvwaddnstr                            -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                     termattrs
     
       Synopsis:
       	int baudrate(void);
       	char erasechar(void);
       	bool has_ic(void);
       	bool has_il(void);
       	char killchar(void);
       	char *longname(void);
       	chtype termattrsr(void);
       	char *termname(void);
     
       	char	wordchar(void);
     
       X/Open Description:
      	baudrate() returns the output speed of the terminal.  The number 
      	returned is bits per second, for example 9600, and is an integer.
     
      	The user's current ERASE character is returned from a call to
      	erasechar();
     
      	The has_ic function will return TRUE if the terminal has insert 
      	character and delete character capabilities.
     
      	The has_il() function will return TRUE if the terminal has insert
      	line and delete line capabilities.
     
      	The user's current KILL character is returned from a call to
      	killchar();
     
      	longname() returns a pointer to a static area containing a
      	verbose description of the current terminal.  The maximum length
      	of the string is 128 characters.  It is defined only after the
      	call to initscr() or newterm().  The area is overwritten by each
      	call to newterm() and is not restored by set_term().  The value
      	should therefore be saved between calls to newterm(), if
      	longname() is going to be used with multiple terminals.
     
      	termname() returns a pointer to a static area containing a
      	short description of the current terminal (14 characters).
     
      	termattrs() returns a logical OR of all video attributes
      	supported by the terminal.
     
       PDCurses Description:
      	If FAST_VIDEO is true, then this is the largest possible
      	(portable) int value (INT_MAX from limits.h) IF direct
      	video is possible, OR the approximate guess at BIOS speeds,
      	19200.
      	If FAST_VIDEO is false, this is an approximate guess at BIOS
      	speeds, 19200.
     
      	erasechar(), killchar() and wordchar() all return values that are 
      	hardcoded at this time.  There may be future development
      	to allow applications an easy way to specify these value.
     
      	For DOS, has_ic() and has_il() always return TRUE, as the speed of
      	the DIRECT VIDEO or even BIOS writes is quicker than on a
      	serial terminal.
      	For FLEXOS, this value has meaning and depends on the terminal,
      	though at this time, it is a constant TRUE.
     
      	In addition to the above definition for longname(), the form of 
      	this string is the adapter name (or video card name) and the text 
      	resolution. This may also be followed by the notation that the video 
      	card may be a clone, which indicates that the card identification
      	maps to more than one unique card.
      	e.g. The MDS Genius and the Quadram QuadHPG identify themselves
      	in the same manner, but are vastly different in maximum resolution.
     
      	The user's current WORD character is returned from a call to
      	wordchar();
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           baudrate                              Y        Y       Y
           erasechar                             Y        Y       Y
           has_ic                                Y        Y       Y
           has_il                                Y        Y       Y
           killchar                              Y        Y       Y
           longname                              Y        Y       Y
           termattrs                             Y        Y       Y
           termname                              Y        Y       Y
           wordchar                              -        -       -
     



     --------------------------------------------------------------------------
     
       Name:                                                         getch
     
       Synopsis:
       	int getch(void);
       	int wgetch(WINDOW *win);
       	int mvgetch(int y, int x);
       	int mvwgetch(WINDOW *win, int y, int x);
       	int ungetch(int ch);
     
       X/Open Description:
      	With the getch(), wgetch(), mvgetch(), and mvwgetch() functions, 
      	a character is read from the terminal associated with the window. 
      	In nodelay mode, if there is no input
      	waiting, the value ERR is returned. In delay mode, the program will
      	hang until the system passes text through to the program.
      	Depending on the setting of cbreak(), this will be after one
      	character or after the first newline.  Unless noecho() has
      	been set, the character will also be echoed into the designated
      	window.
     
      	If keypad() is TRUE, and a function key is pressed, the token for
      	that function key will be returned instead of the raw characters.
      	Possible function keys are defined in <curses.h> with integers
      	beginning with 0401, whose names begin with KEY_.  If a character
      	is received that could be the beginning of a function key (such as
      	escape), curses will set a timer.  If the remainder of the sequence
      	does not come in within the designated time, the character will be
      	passed through, otherwise the function key value will be returned.
      	For this reason, on many terminals, there will be a delay after a
      	user presses the escape key before the escape is returned to the
      	program.  (Use by a programmer of the escape key for a single
      	character function is discouraged.)
     
      	If nodelay(win,TRUE) has been called on the window and no input is
      	waiting, the value ERR is returned.
     
      	The ungetch() function places ch back onto the input queue to be
      	returned by the next call to wgetch().
     
      	NOTE: getch(), mvgetch() and mvwgetch() are implemented as macros.
     
       PDCurses Description:
      	Given the nature of the PC, there is no such timer set for an
      	incoming ESCAPE value, because function keys generate unique
      	scan codes that are not prefixed with the ESCAPE character.
     
      	Also, note that the getch() definition will conflict  with
      	many DOS compiler's runtime libraries.
     
       X/Open Return Value:
      	This functions return ERR or the value of the character, meta 
      	character or function key token.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           getch                                 Y        Y       Y
           wgetch                                Y        Y       Y
           mvgetch                               Y        Y       Y
           mvwgetch                              Y        Y       Y
           ungetch                               Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                          attr
     
       Synopsis:
       	int attroff(chtype attrs);
       	int wattroff(WINDOW *win, chtype attrs);
       	int attron(chtype attrs);
       	int wattron(WINDOW *win, chtype attrs);
       	int attrset(chtype attrs);
       	int wattrset(WINDOW *win, chtype attrs);
       	int standend(void);
       	int wstandend(WINDOW *win);
       	int standout(void);
       	int wstandout(WINDOW *win);
     
       X/Open Description:
      	These functions manipulate the current attributes of the named
      	window.  These attributes can be any combination of A_STANDOUT,
      	A_REVERSE, A_BOLD, A_DIM, A_BLINK, A_UNDERLINE.
     
      	These constants are defined in <curses.h> and can be combined
      	with the bitwise-OR operator (|).
     
      	The current attributes of a window are applied to all characters
      	that are written into the window with waddch().  Attributes are
      	a property of the character, and move with the character
      	through any scrolling and insert/delete line/character operations.
      	To the extent possible on the particular terminal, they will be
      	displayed as the graphic rendition of characters put on the
      	screen.
     
      	The attrset() function sets the current attributes of the given
      	window to attrs.  The attroff() function turns off the named
      	attributes without turning on or off any other attributes.  The
      	attron() function turs on the named attriutes without affecting
      	any others.  The standout() function is the same as
      	attron( A_STANDOUT ).  The standend() function is the same as
      	attrset( A_NORMAL ); that is it turns off all attributes.
     
      	NOTE:	attroff(), attron(), attrset(), standend(), standout(),
      		wstandend(), and wstandout() are defined as macros.
     
       PDCurses Description:
      	System V compatible color support is included.
      	See <curses.h> for further details.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	It is an error to call this function with a NULL window pointer.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           attroff                               Y        Y       Y
           wattroff                              Y        Y       Y
           attron                                Y        Y       Y
           wattron                               Y        Y       Y
           attrset                               Y        Y       Y
           wattrset                              Y        Y       Y
           standend                              Y        Y       Y
           wstandend                             Y        Y       Y
           standout                              Y        Y       Y
           wstandout                             Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                      terminfo
     
       Synopsis:
       	int mvcur(int oldrow, int oldcol, int newrow, int newcol)
     
       X/Open Description:
      	The mvcur() function controls low-level cursor motion with 
      	optimization.
     
      	NOTE: The remainer of the terminfo functions have not been
      			implemented.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           mvcur                                 Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                          move
     
       Synopsis:
       	int move(int y, int x);
       	int wmove(WINDOW *win, int y, int x);
     
       X/Open Description:
      	The cursor associated with the window is moved to the given
      	location.  This does not move the physical cursor of the
      	terminal until refresh() is called.  The position specified is
      	relative to the upper left corner of the window, which is (0,0).
     
      	NOTE: move() is a macro.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           move                                  Y        Y       Y
           wmove                                 Y        Y       Y
     



     --------------------------------------------------------------------------
     
       Name:                                                       refresh
     
       Synopsis:
       	int	refresh(void);
       	int	wrefresh(WINDOW *win);
       	int	wnoutrefresh(register WINDOW *win);
       	int	doupdate(void);
       	int	redrawwin(WINDOW *win);
       	int	redrawln(WINDOW *win, int beg_line, int num_lines);
     
       X/Open Description:
      	The routine wrefresh() copies the named window to the physical
      	terminal screen, taking into account what is already there in
      	order to optimize cursor movement.
      	The routine refresh() does the same, using stdscr as a default
      	screen.
      	These routines must be called to get any output on the
      	terminal, as other routines only manipulate data structures.
      	Unless leaveok has been enabled, the physical cursor of the
      	terminal is left at the location of the window's cursor.
     
      	The wnoutrefresh() and doupdate() routines allow multiple updates
      	with more efficiency than wrefresh() alone.  In addition to all 
      	of the window structures representing the terminal screen: a physical
      	screen, describing what is actually on the screen and a virtual screen,
      	describing what the programmer wants to have on the screen.
     
      	The wrefresh() function works by first calling wnoutrefresh(),
      	which copies the named window to the virtual screen.  It then
      	calls doupdate(), which compares the virtual screen to the
      	physical screen and does the actual update.  If the programmer
      	wishes to output several windows at once, a series of cals to
      	wrefresh() will result in alternating calls to wnoutrefresh()
      	and doupdate(), causing several bursts of output to the
      	screen.  By first calling wnoutrefresh() for each window, it
      	is then possible to call doupdate() once.  This results in
      	only one burst of output, with probably fewer total characters
      	transmitted and certainly less CPU time used.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           refresh                               Y        Y       Y
           wrefresh                              Y        Y       Y
           wnoutrefresh                          Y        Y       Y
           doupdate                              Y        Y       Y
           redrawwin                             -        -      4.0
           wredrawln                             -        -      4.0
     



     --------------------------------------------------------------------------
     
       Name:                                                           pad
     
       Synopsis:
       	WINDOW* newpad( int nlines, int ncols );
       	WINDOW* subpad(WINDOW* orig,int nlines,int ncols,int begin_y,int begin_x);
       	int prefresh(WINDOW* win,int py,int px,int sy1,int sx1,int sy2,int sx2);
       	int pnoutrefresh(WINDOW* w,int py,int px,int sy1,int sx1,int sy2,int sx2);
       	int pechochar(WINDOW *pad, chtype ch);
     
       X/Open Description:
      	newpad() creates a new pad data structure.  A pad is a special case
      	of a window, which is not restricted by the screen size, and is not
      	necessarily associated with a particular part of the screen.  A
      	pad can be used when a large window is needed, and only a part
      	of the window will be on the screen at one tme.  Automatic
      	refreshes of pads (e.g., from scrolling or echoing of input) do
      	not occur.  It is not legal to call refresh() with a pad as an
      	argument; the routines prefresh() or pnoutrefresh() should be
      	called instead.  Note that these routines require additional
      	parameters to specify the part of the pad to be displayed and
      	the location on the screen to be used for display.
     
      	The subpad() routine creates a new sub-pad within a pad.  The
      	dimensions of the sub-pad are nlines lines and ncols
      	columns.  The sub-pad is at position (begin_y, begin_x) in the
      	the parent pad.  This position is relative to the pad, and not
      	to the screen like with subwin.
      	The sub-pad is made in the middle of the pad orig, so
      	that changes made to either pad will affect both.  When
      	using this routine, it will often be necessary to call
      	touchwin before calling prefresh.
     
      	The prefresh() routine copies the specified pad to the physical
      	terminal screen.  It takes account of what is already
      	displayed on the screen to optimize cursor movement.
      	The pnoutrefresh() routine copies the named pad to the virtual
      	screen. It then compares the virtual screen with the physical
      	screen and performs the actual update.
      	These routines are analogous to the routines wrefresh() and
      	wnoutrefresh() except that pads, instead of windows, are
      	involved.  Additional parameters are also needed to indicate
      	what part of the pad and screen are involved. The upper left
      	corner of the part of the pad to be displayed is specified by
      	py and px.  The coordinates sy1, sx1, sy2, and sx2 specify the
      	edges of the screen rectangle that will contain the selected
      	part of the pad.
      	The lower right corner of the pad rectangle to be displayed is
      	calculated from the screen co-ordinates.  This ensures that
      	the screen rectangle and the pad rectangle are the same size.
      	Both rectangles must be entirely contained within their
      	respective structures.
     
      	The pechochar() is functionally equivalent to adch() followed
      	by a call to refresh().
     
       PDCurses Description:
      	PDCurses (as a library) provides the developer with the ability to
      	hook in their own malloc debugging  package.  See the details in
      	INITSCR.C for details on how to accomplish this.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           newpad                                Y        -       Y
           subpad                                Y        -       Y
           prefresh                              Y        -       Y
           pnoutrefresh                          Y        -       Y
           pechochar                             -        -      3.0
     



     --------------------------------------------------------------------------
     
       Name:                                                        scroll
     
       Synopsis:
       	int scroll(WINDOW *win);
       	int scrl(int n);
       	int wscrl(WINDOW *win, int n);
     
       X/Open Description:
      	scroll() causes the window to scroll up one line.  This involves 
      	moving the lines in the window data strcture.
     
      	With the scrl() and wscrl() routines, for positive n scroll the 
      	window up n lines (line i+n becomes i); otherwise scroll the 
      	window down n lines.
     
      	For these functions to work, scrolling must be enabled via 
      	scrollok().
     
      	Note that scrl() and scroll() may be macros.
     
      	Note also that scrolling is not allowed if the supplied window 
      	is a PAD.
     
       X/Open Return Value:
      	All functions return OK on success and ERR on error.
     
       X/Open Errors:
      	No errors are defined for this function.
     
       NOTE:
      	The behaviour of Unix curses is to clear the line with a space
      	and attributes of A_NORMAL. PDCurses clears the line with the
      	window's current attributes (including current colour). To get
      	the behaviour of PDCurses, #define PDCURSES_WCLR in curses.h or
      	add -DPDCURSES_WCLR to the compile switches.
     
       Portability                             X/Open    BSD    SYS V
                                               Dec '88
           scroll                                Y        Y       Y
           scrl                                  -        -      4.0
           wscrl                                 -        -      4.0
     



     --------------------------------------------------------------------------





.......============================ 
.......Internal PDCurses Functions:		
.......============================ 
     
       PDC_debug()	- Write debugging info to log file.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
       PDCurses Return Value:
      	No return value.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	void PDC_debug( char *,... );
     



     --------------------------------------------------------------------------
     
       PDC_backchar()	- Visually erase character in window
     
       PDCurses Description:
      	This is a private PDCurses function
     
      	This routine will visually erase a character.  It is called by
      	the PDCurses character I/O routines.
     
       PDCurses Return Value:
      	This routine will return OK upon success and otherwise ERR will be
      	returned.
     
       PDCurses Errors:
      	It is an error to pass a NULL WINDOW pointer.
     
       Portability:
      	PDCurses	int	PDC_backchar( WINDOW* w, char* ch, int* len );
     



     --------------------------------------------------------------------------
     
       PDC_chg_attr_pair()	- Writes character and attribute to physical screen
     
       PDCurses Description:
      	This is a private PDCurses function.
     
      	Writes a single character 'chr' with attribute 'attr' to the
      	current cursor location.
     
      	NOTE:	Though passed as 16 bit quantities, only the lower 8 bits
      		will be used to create a character/attribute pair.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function under DOS.
     
      	An ERR may be returned under FLEXOS if s_copy() fails.  See the
      	Flexos Programmer's Reference Manual for details on the error.
     
       Portability:
      	PDCurses	int PDC_chg_attr_pair( chtype chr, chtype attr );
     



     --------------------------------------------------------------------------
     
       PDC_chadd()      - Low level; Put a character to a window
     
       PDCurses Description:
             This is a private PDCurses function.
     
             This routine will insert the character 'c' at the current cursor
             position in the passed window.
     
             If 'xlat' is TRUE, PDC_chadd() will handle things in a cooked
             manner (tabs, newlines, carriage returns, etc).  If 'xlat' is
             FALSE, the characters are simply output directly.
     
             If 'advance' is TRUE, PDC_chadd() will move the current cusor position
             appropriately. The *addch functions call PDC_chadd() with advance TRUE,
             while the *insch functions call PDC_chadd() with advance FALSE.
     
             The normal curses routines (non-raw-output-mode) call PDC_chadd()
             with 'xlat' TRUE.
     
       PDCurses Return Value:
             This function returns OK on success and ERR on error.
     
       PDCurses Errors:
             It is an error to call this function with a NULL window pointer.
     
       Portability:
             PDCurses        int PDC_chadd( WINDOW* win, chtype ch, bool xlat, bool advance );
     



     --------------------------------------------------------------------------
     
       PDC_chg_attrs()	- Change attributes in a rectangle
     
       PDCurses Description:
      	This routine will change the attribute(s) from a starting (y,x)
      	position to an ending (y,x) position to the specified attribute.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	It is an error to call this function with a NULL window pointer.
      	It is also an error to pass rectangular coordinates that lay
      	outside of window.
     
       Portability:
      	PDCurses	int PDC_chg_attrs( WINDOW* w, chtype attr,
      					int sy, int sx,
      					int ey, int ex );
     



     --------------------------------------------------------------------------
     
       PDC_chins()	- Low-level insert character in window
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	This routine provides the basic functionality for the X/Open
      	[mv][w]insch() routines.  The xlat flag indicates that normal
      	character translation is performed or not.  If not, then the
      	character is output as is.
     
      	The 'xlat' flag is TRUE for the normal curses routines.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	It is an error to call this function with a NULL window pointer.
     
       Portability:
      	PDCurses	int PDC_chins( WINDOW* win, chtype c, bool xlat );
     



     --------------------------------------------------------------------------
     
       PDC_clr_scrn()	- Clears the physical screen and homes the cursor.
     
       PDCurses Description:
      	This is an internal routine called by the doupdate() routines.
     
       PDCurses Return Value:
      	This routine always returns OK.
     
       Portability:
      	PDCurses	int PDC_clr_scrn( WINDOW* win );
     



     --------------------------------------------------------------------------
     
       PDC_clr_update()	- Updates the screen with a full redraw.
     
       PDCurses Description:
      	Updates the screen by clearing it and then redraw it in its
      	entirety. If _cursvar.refrbrk is TRUE, and there is pending
      	input characters, the update will be prematurely terminated.
     
       PDCurses Return Value:
      	This routine returns ERR if it is unable to accomplish it's task.
      	This return value is ONLY under FLEXOS.
     
      	The return value OK is returned if there were no errors.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_clr_update( WINDOW* s );
     



     --------------------------------------------------------------------------
     
       PDC_cursor_on()	- Turns on the hardware cursor.
     
       PDCurses Description:
      	Turns on the hardware curses, it does nothing if it is already on.
     
       PDCurses Return Value:
      	Returns OK upon success, ERR upon failure.
     
       Portability:
      	PDCurses	int PDC_cursor_on( void );
     



     --------------------------------------------------------------------------
     
       PDC_cursor_off()	- Turns off the hardware cursor.
     
       PDCurses Description:
      	Turns off the hardware curses, it does nothing if it is already off.
     
       PDCurses Return Value:
      	Returns OK upon success, ERR upon failure.
     
       PDCurses Errors:
      	ERR will be returned (in the case of FLEXOS) if the hardware cursor
      	can not be disabled.
     
       Portability:
      	PDCurses	int PDC_cursor_off( void );
     



     --------------------------------------------------------------------------
     
       PDC_fix_cursor()	- Fix the cursor start and stop scan lines (if necessary)
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	This routine will fix the cursor shape for certain video adapters.
      	Normally, the values used are correct, but some adapters choke.
      	The most noticable choke is on a monochrome adapter.  The "correct"
      	scan lines will result in the cursor being set in the middle of the
      	character cell, rather than at the bottom.
     
      	The passed flag indicates whether the cursor is visible or not.
     
      	This only applies to the DOS platform.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_fix_cursor( int flag );
     



     --------------------------------------------------------------------------
     
       PDC_gotoxy()	- position hardware cursor at (x, y)
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Moves the physical cursor to the desired address on the
      	screen. We don't optimize here -- on a PC, it takes more time
      	to optimize than to do things directly.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_gotoxy( int row, int col );
     



     --------------------------------------------------------------------------
     
       PDC_newline()	- Advances 1 newline from supplied line number.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Does line advance and returns the new cursor line.  If error,
      	return -1.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_newline( WINDOW* win, int lin );
     



     --------------------------------------------------------------------------
     
       PDC_putc()	- Output a character in the current attribute.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Outputs character 'chr' to screen in tty fashion. If a colour
      	mode is active, the character is written with colour 'colour'.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_putc( chtype character, chtype color );
     



     --------------------------------------------------------------------------
     
       PDC_putctty()	- Output a character and attribute in TTY fashion.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Outputs character 'chr' to screen in tty fashion. If a colour
      	mode is active, the character is written with colour 'colour'.
     
      	This function moves the physical cursor after writing so the
      	screen will scroll if necessary.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_putctty( chtype character, chtype color );
     



     --------------------------------------------------------------------------
     
       PDC_scroll()	- low level screen scroll
     
       PDCurses Description:
      	Scrolls a window in the current page up or down. Urow, lcol,
      	lrow, rcol are the window coordinates.	Lines is the number of
      	lines to scroll. If 0, clears the window, if < 0 scrolls down,
      	if > 0 scrolls up.  Blanks areas that are left, and sets
      	character attributes to attr. If in a colour graphics mode,
      	fills them with the colour 'attr' instead.
     
       PDCurses Return Value:
      	The PDC_scroll() function returns OK on success otherwise ERR is returned.
     
       PDCurses Errors:
      	An error will only be returned on the Flexos platform if s_copy()
      	fails.
     
       Portability:
      	PDCurses	int PDC_scroll( int urow, int lcol, int rcol,
      				     int nlines, chtype attr );
     



     --------------------------------------------------------------------------
     
       PDC_transform_line()	- display a physical line of the screen
     
       PDCurses Description:
      	This is a private PDCurses function.
     
      	Updates the given physical line to look like the corresponding
      	line in _curscr.
     
       PDCurses Return Value:
      	This routine returns TRUE if a premature refresh end
      	is allowed, and there is an input character pending.  Otherwise,
      	FALSE is returned.
     
       PDCurses Errors:
      	No errors are defined for this routine.
     
       Portability:
      	PDCurses	bool	PDC_transform_line( int lineno );
     



     --------------------------------------------------------------------------
     
       PDC_get_cursor_pos()	- return current cursor position
     
       PDCurses Description:
      	This is a private PDCurses function
     
      	Gets the cursor position in video page 0.  'row' and 'column'
      	are the cursor address.  At this time, there is no support for
      	use of multiple screen pages.
     
       PDCurses Return Value:
      	This routine will return OK upon success and otherwise ERR will be
      	returned.
     
       PDCurses Errors:
      	There are no defined errors for this routine.
     
       Portability:
      	PDCurses	int	PDC_get_cursor_pos( int* row, int* col );
     



     --------------------------------------------------------------------------
     
       PDC_get_cur_col()	- get current column position of cursor
     
       PDCurses Description:
      	This is a private PDCurses function
     
      	This routine returns the current column position of the cursor on
      	screen.
     
       PDCurses Return Value:
      	This routine returns the current column position of the cursor. No
      	error is returned.
     
       PDCurses Errors:
      	There are no defined errors for this routine.
     
       Portability:
      	PDCurses	int	PDC_get_cur_col( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_cur_row()	- get current row position of cursor
     
       PDCurses Description:
      	This is a private PDCurses function
     
      	This routine returns the current row position of the cursor on
      	screen.
     
       PDCurses Return Value:
      	This routine returns the current row position of the cursor. No
      	error is returned.
     
       PDCurses Errors:
      	There are no defined errors for this routine.
     
       Portability:
      	PDCurses	int	PDC_get_cur_row( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_attribute()	- Get attribute at current cursor
     
       PDCurses Description:
      	This is a private PDCurses function
     
      	Return the current attr at current cursor position on the screen.
     
       PDCurses Return Value:
      	This routine will return OK upon success and otherwise ERR will be
      	returned.
     
       PDCurses Errors:
      	There are no defined errors for this routine.
     
       Portability:
      	PDCurses	int	PDC_get_attribute( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_columns()	- return width of screen/viewport.
     
       PDCurses Description:
      	This is a private PDCurses function
     
      	This function will return the width of the current screen.
     
       PDCurses Return Value:
      	This routine will return OK upon success and otherwise ERR will be
      	returned.
     
       PDCurses Errors:
      	There are no defined errors for this routine.
     
       Portability:
      	PDCurses	int	PDC_get_columns( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_cursor_mode()	- Get the cursor start and stop scan lines.
     
       PDCurses Description:
      	Gets the cursor type to begin in scan line startrow and end in
      	scan line endrow.  Both values should be 0-31.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_get_cursor_mode( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_font()	- Get the current font size
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	This function returns the current font size.  This function only
      	works if the #define FAST_VIDEO is true.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	An ERR will be returned if FAST_VIDEO is not true.
     
       Portability:
      	PDCurses	int PDC_get_font( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_rows()	- Return number of screen rows.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Returns the maximum number of rows supported by the display.
      	e.g.  25, 28, 43, 50, 60, 66...
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_get_rows( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_scrn_mode()	- Return the current BIOS video mode
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
     
       PDCurses Return Value:
      	Returns the current BIOS Video Mode Number.
     
       PDCurses Errors:
      	The FLEXOS version of this routine returns an ERR.
      	The UNIX version of this routine returns an ERR.
      	The EMXVIDEO version of this routine returns an ERR.
     
       Portability:
      	PDCurses	int PDC_get_scrn_mode( void );
     



     --------------------------------------------------------------------------
     
       PDC_query_adapter_type()	- Determine PC video adapter type
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Thanks to Jeff Duntemann, K16RA for providing the impetus
      	(through the Dr. Dobbs Journal, March 1989 issue) for getting
      	the routines below merged into Bjorn Larsson's PDCurses 1.3...
      		-- frotz@dri.com	900730
     
       PDCurses Return Value:
      	This function returns a macro identifier indicating the adapter
      	type.  See the list of adapter types in CURSPRIV.H.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_query_adapter_type( void );
     



     --------------------------------------------------------------------------
     
       PDC_sanity_check() - A video adapter identification sanity check
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	This routine will force sane values for various control flags.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_sanity_check( int adapter );
     



     --------------------------------------------------------------------------
     
       PDC_breakout()	- check for type-ahead
     
       X/Open Description:
      	The curses package does the "line-breakout optimisation" by
      	looking for type-ahead periodically while updating the screen.
      	If input is found, the current update will be postponed until
      	refresh() or doupdate() are called again.  This allows faster
      	response to commands typed in advance.  Normally, the input FILE
      	pointer passed to newterm(), or stdin in the case when initscr()
      	was called, will be used to do this type-ahead checking.  This routine
      	will do the actual check for PDcurses to see if input is pending.
     
       PDCurses Description:
      	This routine does what the PDcurses typeahead routine used to do.  But
      	to make that routine consistent with its System V counterpart, all it
      	does now is set whether or not typeahead checking is performed.
     
       X/Open Return Value:
      	The PDC_breakout() routine returns TRUE if keyboard input is pending
      	otherwise FALSE is returned.
     
       Portability:
      	PDCurses	bool PD_breakout( void );
     



     --------------------------------------------------------------------------
     
       PDC_check_bios_key()	- Check BIOS key data area for input
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	This routine will check the BIOS for any indication that
      	keystrokes are pending.
     
       PDCurses Return Value:
      	Returns 1 if a keyboard character is available, 0 otherwise.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	bool	PDC_check_bios_key( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_bios_key()	- Returns the next key available from the BIOS.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Returns the next key code struck at the keyboard. If the low 8
      	bits are 0, the upper bits contain the extended character
      	code. If bit 0-7 are non-zero, the upper bits = 0.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_get_bios_key( void );
     



     --------------------------------------------------------------------------
     
       PDC_get_ctrl_break()	- return OS control break state
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Returns the current OS Control Break Check state.
     
       PDCurses Return Value:
      	DOS:
      		This function returns TRUE on if the Control Break
      		Check is enabled otherwise FALSE is returned.
     
      	FLEXOS:
      		This function returns TRUE on if the Keyboard Mode
      		allows a break to bre reported otherwise FALSE is returned.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	bool	PDC_get_ctrl_break( void );
     



     --------------------------------------------------------------------------
     
       PDC_rawgetch()	- Returns the next uninterpreted character (if available).
     
       PDCurses Description:
      	Gets a character without any interpretation at all and returns
      	it. If keypad mode is active for the designated window,
      	function key translation will be performed.  Otherwise,
      	function keys are ignored.  If nodelay mode is active in the
      	window, then PDC_rawgetch() returns -1 if no character is
      	available.
     
      	WARNING:  It is unknown whether the FUNCTION key translation
      		  is performed at this level. --Frotz 911130 BUG
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int	PDC_rawgetch( void );
     



     --------------------------------------------------------------------------
     
       PDC_set_ctrl_break()	- Enables/Disables the host OS BREAK key check.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Enables/Disables the host OS BREAK key check. This function toggles
      	the BREAK setting. If it was on, it turns itoff; if it was aff it turns
      	it on.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_set_ctrl_break( bool setting );
     



     --------------------------------------------------------------------------
     
       PDC_sysgetch()	- Return a character using default system routines.
     
       PDCurses Description:
      	This is a private PDCurses function.
     
      	Gets a character without normal ^S, ^Q, ^P and ^C interpretation
      	and returns it.  If keypad mode is active for the designated
      	window, function key translation will be performed. Otherwise,
      	function keys are ignored. If nodelay mode is active in the
      	window, then sysgetch() returns -1 if no character is
      	available.
     
       PDCurses Return Value:
      	This function returns OK upon success otherwise ERR is returned.
     
       PDCurses Errors:
      	No errors are defined for this routine.
     
       Portability:
      	PDCurses	int	PDC_sysgetch( void );
     



     --------------------------------------------------------------------------
     
       PDC_validchar()	- validate/translate passed character
       
       PDCurses Description:
       	This is a private PDCurses function.
       
       	Checks that 'c' is a valid character, and if so returns it,
       	with function key translation applied if 'w' has keypad mode
       	set.  If char is invalid, returns -1.
       
       PDCurses Return Value:
       	This function returns -1 if the passed character is invalid, or
       	the WINDOW* 'w' is NULL, or 'w's keypad is not active.
       
       	This function returns 0 under Flexos if the passed character
       	is 0x0300.  (-:Forget why.  --Frotz	911130:-)
       
       	Otherwise, this function returns the PDCurses equivalent of the
       	passed character.  See the function key and key macros in
       	<curses.h>
       
       PDCurses Errors:
       	There are no errors defined for this routine.
       
       Portability:
       	PDCurses	int	PDC_validchar( int c );
     



     --------------------------------------------------------------------------
     
       PDC_print()	- Provides primitive access to the BIOS printer functions
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Implements write/init/read printer services at the BIOS level.
     
      	This provides the basic support that PDCurses needs to dump the
      	contents of windows or pads to the printer attached to the BIOS
      	printer port.
     
       PDCurses Return Value:
      	See the BIOS INT 0x17 specifications.
     
       PDCurses Errors:
      	See the BIOS INT 0x17 specifications.
     
       Portability:
      	PDCurses	int PDC_print( int cmd, int byte, int port );
     



     --------------------------------------------------------------------------
     
       PDC_scr_close()	- Internal low-level binding to close the physical screen
     
       PDCurses Description:
      	This function provides a low-level binding for the Flexos
      	platform which must close the screen before writing to it.
      	This is a nop for the DOS platform.
     
      	This function is provided in order to reset the FlexOS 16 bit
      	character set for input rather than the limited input
      	character set associated with the VT52.
     
       PDCurses Return Value:
      	This function returns OK on success, otherwise an ERR is returned.
     
       PDCurses Errors:
      	The DOS platform will never fail.  The Flexos platform may fail
      	depending on the ability to close the current virtual console in
      	8 (as opposed to 16) bit mode.
     
       Portability:
      	PDCurses	int	PDC_scr_close( void );
     



     --------------------------------------------------------------------------
     
       PDC_scrn_modes_equal()	- Decide if two screen modes are equal
     
       PDCurses Description:
      	Mainly required for OS/2. It decides if two screen modes
             (VIOMODEINFO structure) are equal. Under DOS it just compares
             two integers
     
       PDCurses Return Value:
      	This function returns TRUE if equal else FALSe.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_scrn_modes_equal( int mode1, int mode2 );
      	OS2 PDCurses	int PDC_scrn_modes_equal( VIOMODEINFO mode1, VIOMODEINFO mode2 );
     



     --------------------------------------------------------------------------
     
       PDC_scr_open()	- Internal low-level binding to open the physical screen
     
       PDCurses Description:
      	This function provides a low-level binding for the Flexos
      	platform which must open the screen before writing to it.
     
      	This function is provided in order to access the FlexOS 16 bit
      	character set for input rather than the limited input
      	character set associated with the VT52.
     
       PDCurses Return Value:
      	This function returns OK on success, otherwise an ERR is returned.
     
       PDCurses Errors:
      	The DOS platform will never fail.  The Flexos platform may fail
      	depending on the ability to open the current virtual console in
      	8 (as opposed to 16) bit mode.
     
       Portability:
      	PDCurses	int	PDC_scr_open( SCREEN* internal, bool echo );
     



     --------------------------------------------------------------------------
     
       PDC_set_80x25()	- force a known screen state: 80x25 text mode.
     
       PDCurses Description:
      	This is a private PDCurses function.
     
      	Forces the appropriate 80x25 alpha mode given the display adapter.
     
      	Since we currently do not support changing the virtual console size,
      	this routine is a NOP under Flexos.
     
       PDCurses Return Value:
      	This function returns OK upon success otherwise ERR is returned.
     
       PDCurses Errors:
      	No errors are defined for this routine.
     
       Portability:
      	PDCurses	int	PDC_set_80x25( void );
     



     --------------------------------------------------------------------------
     
       PDC_set_cursor_mode()	- Set the cursor start and stop scan lines.
     
       PDCurses Description:
      	Sets the cursor type to begin in scan line startrow and end in
      	scan line endrow.  Both values should be 0-31.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_set_cursor_mode( int startrow, int endrow );
     



     --------------------------------------------------------------------------
     
       PDC_set_font()	- sets the current font size
     
       PDCurses Description:
      	This is a private PDCurses function.
     
      	This routine sets the current font size, if the adapter allows
      	such a change.
     
       PDCurses Return Value:
      	This function returns OK upon success otherwise ERR is returned.
     
       PDCurses Errors:
      	It is an error to attempt to change the font size on a "bogus"
      	adapter.  The reason for this is that we have a known video
      	adapter identity problem.  e.g. Two adapters report the same
      	identifying characteristics.
     
      	It is also an error to attempt to change the size of the Flexos
      	console (as there is currently no support for that).
     
       Portability:
      	PDCurses	int	PDC_set_font( int size );
     



     --------------------------------------------------------------------------
     
       PDC_set_rows()	- sets the physical number of rows on screen
     
       PDCurses Description:
      	This is a private PDCurses function.
     
      	This routine attempts to set the number of rows on the physical
      	screen to the passed value.
     
       PDCurses Return Value:
      	This function returns OK upon success otherwise ERR is returned.
     
       PDCurses Errors:
      	It is an error to attempt to change the screen size on a "bogus"
      	adapter.  The reason for this is that we have a known video
      	adapter identity problem.  e.g. Two adapters report the same
      	identifying characteristics.
     
      	It is also an error to attempt to change the size of the Flexos
      	console (as there is currently no support for that).
     
       Portability:
      	PDCurses	int	PDC_set_rows( int rows );
     



     --------------------------------------------------------------------------
     
       PDC_set_scrn_mode()	- Set BIOS Video Mode
     
       PDCurses Description:
      	Sets the BIOS Video Mode Number ONLY if it is different from
      	the current video mode.  This routine is for DOS systems only.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_set_scrn_mode( int new_mode );
     



     --------------------------------------------------------------------------
     
       PDC_memmove()	- Replacement memmove() for lacking compliers.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Move a portion of memory taking consideration of overlapping memory
      	areas.
     
       PDCurses Return Value:
      	This function returns NULL if no temporary memory could be allocated.
     
       Portability:
      	PDCurses	int PDC_memmove(void *s1,const void *s2,size_t n);
     



     --------------------------------------------------------------------------
     
       PDC_usleep()	- waits for specified number of microseconds
     
       PDCurses Description:
      	This routine is intended to provide a mechanism to wait the
      	specified number of microseconds. This routine is provided for
      	users of the Microsoft compilers under DOS as the default
      	library does not provide a suitable function.
      	Under any other Operating System/Compiler option, this function
      	does nothing.
     
       Portability:
      	PDCurses	void PDC_usleep( clock_t );
     
       Acknowledgement
      	PDC_usleep() was written by John Steele  (jsteele@netcom.com)
     



     --------------------------------------------------------------------------
     
       PDC_vsscanf()	- Internal routine to parse and format an input buffer.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Scan a series of input fields. Each field is formatted according to
      	a supplied format string and the formatted input is stored in the
      	variable number of addresses passed.
     
       PDCurses Return Value:
      	This function returns the number of input fields or EOF on error.
     
       PDCurses Errors:
      	If the supplied data is invalid or an incorrect number of arguments
      	are passed, EOF is returned as an error.
     
       Portability:
      	PDCurses	int PDC_vsscanf(char *buf,const char *fmt,va_list arg_ptr);
     



     --------------------------------------------------------------------------
     
       PDC_copy_win()	- Common routine for copywin(), overlay() and overwrite()
      	functions.
     
       PDCurses Description:
      	This function copies the region of the source window specified
      	over the specified region of the destination window. All validation
      	of limits are done by the calling function.
     
      	Thanks to Andreas Otte (venn@uni-paderborn.de) for the code changes.
     
       PDCurses Errors:
      	ERR is returned if either src or dst windows are NULL;
     
       Portability:
      	PDCurses	int	PDC_copy_win( WINDOW* src_w, WINDOW* dst_w
      			int src_tr,int src_tc,int src_br,int src_bc,
      			int dst_tr,int dst_tc,int dst_br,int dst_bc,bool overlay);
     



     --------------------------------------------------------------------------
     
       PDC_makenew()	- Create a WINDOW* (sans line allocation)
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Allocates all data for a new WINDOW* except the actual lines
      	themselves.
     
       PDCurses Return Value:
      	This function returns a valid WINDOW* on success and NULL on error.
     
       PDCurses Errors:
      	If PDC_makenew() is unable to allocate memory for the window
      	structure, it will free all allocated memory and return
      	a NULL pointer.
     
       Portability:
      	PDCurses	WINDOW* PDC_makenew( int num_lines, int num_columns,
      					 int begy, int begx );
     



     --------------------------------------------------------------------------





.......========================== 
.......PDCurses Flexos Functions:		
.......========================== 
     
       _flexos_16bitmode()	- Puts console 16 bit mode.
     
       PDCurses Description:
      	This routine sets 16 bit mode (8 bit character +
      	8 bit attribute) for Flexos.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	Under FLEXOS, an ERR will be returned if the s_get() call
      	fails, or if the s_set() call fails.  You are referred to
      	the Flexos Programmer's Reference Guide for details on the
      	actual error.
     
       Portability:
      	PDCurses	int _flexos_16bitmode( void );
     



     --------------------------------------------------------------------------
     
       _flexos_8bitmode()	- Puts console into 8 bit mode.
     
       PDCurses Description:
      	This routine sets 8 bit mode (no 8 bit attribute) for Flexos.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	Under FLEXOS, an ERR will be returned if the s_get() call
      	fails, or if the s_set() call fails.  You are referred to
      	the Flexos Programmer's Reference Guide for details on the
      	actual error.
     
       Portability:
      	PDCurses	int _flexos_16bitmode( void );
     



     --------------------------------------------------------------------------
     
       _flexos_gname()	- FLEXOS: Return virtual console name
     
       PDCurses Description:
      	This is a Flexos platform PDCurses function.
     
      	This routine returns the name of the Flexos virtual console.
      	The VIRCON vir structure is initialized within the initscr()
      	calltree.
     
       PDCurses Return Value:
      	This routine returns a character pointer.
     
       PDCurses Errors:
      	No errors are defined for this routine.
     
       Portability:
      	PDCurses	char*	_flexos_gname( void );
     



     --------------------------------------------------------------------------





.......========================== 
.......PDCurses Unix Functions:		
.......========================== 
     
       PDC_kbhit()	- Check if a character has been entered on keyboard.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Outputs character 'chr' to screen in tty fashion. If a colour
      	mode is active, the character is written with colour 'colour'.
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_putc( chtype character, chtype attr );
     



     --------------------------------------------------------------------------
     
       PDC_setup_keys()	- Setup function key translations.
     
       PDCurses Description:
      	This is a private PDCurses routine.
     
      	Sets up the array of key sequences and their associated curses key number
      	for use by the getch() function when determining if a sequence of
      	characters is a key escape sequence.
     
     
       PDCurses Return Value:
      	This function returns OK on success and ERR on error.
     
       PDCurses Errors:
      	No errors are defined for this function.
     
       Portability:
      	PDCurses	int PDC_setup_keys( char *keyptr, int keynum );
     



     --------------------------------------------------------------------------





.......========================== 
.......PDCurses Panel Functions:		
.......========================== 
     
       panels	- panel package for curses
     
       PDCurses Description:
      	The panel library is built using the curses library and any program
      	using panels routines must call one of the curses initialization
      	routines such as initscr(). A program using these routines must be
      	linked with the panels and curses libraries.
     
      	The panels package gives the applications programmer a way to have
      	depth relationships between curses windows; a curses window is
      	associated with every panel. The panels routines allow curses windows
      	to overlap without making visible the overlapped portions of 
      	underlying windows. The initial curses window, stdscr, lies beneath
      	all panels. The set of currently visible panels is the 'deck' of
      	panels.
     
      	The panels package allows the applications programmer to create
      	panels, fetch and set their associated windows, shuffle panels in 
      	the deck, and manipulate panels in other ways.
     
       PDCurses Return Value:
      	Each panels routine that returns a pointer to an object returns NULL
      	if an error occurs. Each panel routine that returns an integer, 
      	returns OK if it executes successfully and ERR if it does not.
     
       Notes
      	The header <panel.h> includes the header <curses.h>.
     
       Credits
      	Original Author - Warren Tucker N4HGF
      	{gatech,emory}!n4hgf!wht -or- wht@n4hgf.Mt-Park.GA.US
     



     --------------------------------------------------------------------------
     
       bottom_panel	- puts panel at bottom of deck
     
       PDCurses Description:
      	This function places pan at the bottom of the deck. The size, location
      	and contents of the panel are unchanged.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int bottom_panel( PANEL *pan );
      	SYS V Curses	int bottom_panel( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       del_panel	- deletes a panel
     
       PDCurses Description:
      	This function deletes pan but not its associated winwow.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int del_panel( PANEL *pan );
      	SYS V Curses	int del_panel( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       hide_panel	- removes a panel from the deck
     
       PDCurses Description:
      	This function removes a panel from the deck and thus hides it from
      	view.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int hide_panel( PANEL *pan );
      	SYS V Curses	int hide_panel( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       move_panel	- move a window on the virtual screen
     
       PDCurses Description:
      	This function move the curses window associated with pan so that
      	its upper lefthand corner is at the supplied coordinates. Do not
      	use mvwin() on the window.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL or an error occurs when
      	trying to move the curses window.
     
       Portability:
      	PDCurses	int move_panel( PANEL *pan, int starty, int startx);
      	SYS V Curses	int move_panel( PANEL *pan, int starty, int startx);
     



     --------------------------------------------------------------------------
     
       new_panel	- create a new panel
     
       PDCurses Description:
      	This function creates a new panel associated with win and returns
      	the panel pointer. The new panel is placed at the top of the deck.
     
       PDCurses Return Value:
      	Returns pointer to new panel, or NULL if an error occurs.
     
       PDCurses Errors:
      	Returns NULL if an error occurs.
     
       Portability:
      	PDCurses	PANEL *new_panel( WINDOW *win );
      	SYS V Curses	PANEL *new_panel( WINDOW *win );
     



     --------------------------------------------------------------------------
     
       panel_above	- return pointer to panel above
     
       PDCurses Description:
      	This function returns a pointer to the panel in the deck above
      	pan. If the value of pan passed is NULL, this function returns
      	a pointer to the bottom panel in the deck.
     
       PDCurses Return Value:
      	Returns pointer to panel above pan, or NULL if pan is the top
      	panel.
     
       PDCurses Errors:
      	Returns NULL if an error occurs.
     
       Portability:
      	PDCurses	PANEL *panel_above( PANEL *pan );
      	SYS V Curses	PANEL *panel_above( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       panel_below	- return pointer to panel below
     
       PDCurses Description:
      	This function returns a pointer to the panel in the deck below
      	pan. If the value of pan passed is NULL, this function returns
      	a pointer to the top panel in the deck.
     
       PDCurses Return Value:
      	Returns pointer to panel below pan, or NULL if pan is the bottom
      	panel.
     
       PDCurses Errors:
      	Returns NULL if an error occurs.
     
       Portability:
      	PDCurses	PANEL *panel_below( PANEL *pan );
      	SYS V Curses	PANEL *panel_below( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       panel_hidden	- indicates if panel is hidden
     
       PDCurses Description:
      	This function returns TRUE if pan is hidden and FALSE if it is not.
     
       PDCurses Return Value:
      	TRUE or FALSE.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int panel_hidden( PANEL *pan );
      	SYS V Curses	int panel_hidden( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       panel_userptr	- return user information
     
       PDCurses Description:
      	Each panel has a user pointer available for maintaining relevant
      	information. This function returns a pointer to that information
      	previously set up by set_panel_userptr().
     
       PDCurses Return Value:
      	Returns pointer to user information.
     
       PDCurses Errors:
      	Returns NULL if pan is NULL or no user information exists.
     
       Portability:
      	PDCurses	char *panel_userptr( PANEL *pan );
      	SYS V Curses	char *panel_userptr( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       panel_window	- returns pointer to curses window
     
       PDCurses Description:
      	This function returns a pointer to the curses window associated
      	with the panel.
     
       PDCurses Return Value:
      	Pointer to panel's window.
     
       PDCurses Errors:
      	Return NULL on error.
     
       Portability:
      	PDCurses	WINDOW *panel_window(PANEL *);
      	SYS V Curses	WINDOW *panel_window(PANEL *);
     



     --------------------------------------------------------------------------
     
       replace_panel	- set curses window contents
     
       PDCurses Description:
      	This function replaces the current window of pan with win.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int replace_panel( PANEL *pan, WINDOW *win );
      	SYS V Curses	int replace_panel( PANEL *pan, WINDOW *win );
     



     --------------------------------------------------------------------------
     
       set_panel_userptr	- sets user information for a panel
     
       PDCurses Description:
      	Each panel has a user pointer available for maintaining relevant
      	information. This function sets the value of that information.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int set_panel_userptr( PANEL *pan, char *uptr );
      	SYS V Curses	int set_panel_userptr( PANEL *pan, char *uptr );
     



     --------------------------------------------------------------------------
     
       show_panel	- displays a panel
     
       PDCurses Description:
      	This function makes a previously hidden panel visible and places
      	it back in the deck on top.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int show_panel( PANEL *pan );
      	SYS V Curses	int show_panel( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       top_panel	- puts panel on top of deck
     
       PDCurses Description:
      	This function places pan on the top of the deck. The size, location
      	and contents of the panel are unchanged.
     
       PDCurses Return Value:
      	Returns OK or ERR.
     
       PDCurses Errors:
      	Returns ERR if pan is NULL.
     
       Portability:
      	PDCurses	int top_panel( PANEL *pan );
      	SYS V Curses	int top_panel( PANEL *pan );
     



     --------------------------------------------------------------------------
     
       update_panels	- panels virtual screen refresh routine
     
       PDCurses Description:
      	This function refreshes the virtual screen to reflect the depth
      	relationships between the panels in the deck. The user must use
      	doupdate() to refresh the physical screen.
     
       PDCurses Return Value:
      	None
     
       PDCurses Errors:
      	None
     
       Portability:
      	PDCurses	void update_panels( void )
      	SYS V Curses	void update_panels( void )
     



     --------------------------------------------------------------------------





.......======================== 
.......Xwindows considerations:		
.......======================== 
     
     Introduction:
     
     	To use the Xwindows port of PDCurses (Xcurses), you need to add one
     	line to the main module of your curses program:
     	
     	char *XPDCurses_program_name = "name_of_program";
     
     	where you substitute "name_of_program" with the name of your program.
     
     X Resources:
     
     	Xcurses uses various resource settings.
     



     --------------------------------------------------------------------------





