/*==============================================================*/
/*   Graphic functions for EGA,VGA, and several SVGA            */
/*==============================================================*/

/* Return codes from function : 1 = O.K.  else = Error          */

// Conditional translation for Hi color modes (1-yes, 0-no)

#ifdef HICOLOR
#define HI_COLOR  1
#else
#define HI_COLOR  0
#endif

// Xlopif with virtual videoram
#ifndef XTVERSION
#define VIRT_SCR
#endif
// Max. number of virt. screens
#define MAXVIRTUAL 15

// Constants for 2,16,256,HiColor (only bits 5..7!!)
#define MM_2      0x20
#define MM_16     0x40
#define MM_256    0x80
#define MM_Hic    0x60

/*##-------------------- x_grf_mod -------- Bios----------------*/
/* Sets graphical mode given by no. xmod. Interesting modes:    */
/* Names: EGA - 640 x 350 x 16                                  */
/*        VGA - 640 x 480 x 16                                  */
/*        VGA2- 320 x 200 x 256                                 */
/*        A   - 800 x 600 x 16                                  */
/*        B   - 640 x 480 x 256                                 */
/*        C   - 600 x 800 x 256                                 */
/*        D   -1024 x 768 x 16                                  */
/*        E   -1024 x 768 x 256                                 */
/*        BCGA- 640 x 200 x 2                                   */

/*        03 - implicit text mode on EGA/VGA                    */

/* Hi-Col Hi16: 16Bit (65536 colors)                            */
/*        Hi15: 15Bit (32768 colors)                            */
/*        I   - 640 x 480 x 15|16                               */
/*        J   - 800 x 600 x 15|16                               */
/*        K   -1024 x 768 x 15|16                               */

// Example:     Hi16.j = 65536 colors x 800 x 600
// Description fci spec. for Hi-col under:  #if HI_COLOR

/*##-------------------- x_grf_mod  ----------------------------*/
int x_grf_mod(int xmod);
/* xmod - real number of mode according to graphical card       */

/*##-------------------- x_setnomode ---------------------------*/
// pri nomode != 0 doopravdy nenastavi grf. mod
// tr.: with nomode != 0 does not really set graph. mod
void x_setnomode(int nomode);

/*##-------------------- x_SetXmsEms ---------------------------*/
int SetXmsEms(int Mode);
// Mode - 1-XMS, 2-EMS
// ret    1-OK,  0-ERROR

/*##-------------------- mem_all_xmem ------------------------*/
long mem_all_xmem(void);
// vrati ceklove naalokovanych KB XMS pomoci alloc_xmem()
// tr.: returns as a whole allocated KB XMS with the help of alloc_xmem()

int  mem_xmem(unsigned *n1, unsigned *n2);

/*##-------------------- hinf_xmem ---------------------------*/
// K handlu XMS zjisti kolik KB alokuje a volny pocet handlu
// tr.: for handle XMS finds out how many KB allocates a free number of handles
int hinf_xmem(int hxms, int *Kb, int *FreeH);
// hxms - handle xms   (in)
//   kb - kolik KB patri k tomuto handlu (out)
//   tr.: kb - how many KB belong ti this handle (out)
// FreeH- zbyvajici pocet handlu XMS
// tr.: FreeH- left number of handles XMS


/*##-------------------- x_rea_svga ----------------------------*/
//  Zjisti dle jmena konkretni graf. mod
//  tr.: Finds out concrete graphic mode according to name 
int x_rea_svga(char *path, char *g_jmeno, int *mod);

/*path   - does not have any meaning                            */
/*g_jmeno- name of card+mode :                                  */
/*         names of standard graph. modes are EGA,VGA,VGA2.     */
/*         names of SVGA graph. modes follow the pattern:       */
/*         nameSVGA.name_mode  e.g.: TSG3.B , OAK.C etc.        */
/*         see table of names at x_grf_mod()                    */
/*mod    - real number of graphical mode                        */

/*       return: 1 - O.K.                                       */
/*               2 - file does not exist (cannot be opened)     */
/*               4 - unknown graph. card, syntax error          */
/*               6 - mode is not supported by given card        */


/*##-------------------- x_getgrfmode ------ Bios --------------*/
/* Returns number of set graphic mode                           */

int x_getgrfmode(void);


/*##------------------- x_map_pal------------ Bios --------------*/
/* Maps 16 registers of 256 for 16 color VGA modes               */
/* Is called only once, before x_palett is used                  */
/* After initialization no registers are set !!!                 */
/* !! Do not call for EGA mode                                   */

void x_map_pal(char *mapreg);

/* mapreg - field with pal. register, that will be further used  */


/*##------------------- x_palett ---------- Bios ---------------*/
/* Sets colour palette for given mode. 16 or 256 colours        */
/* After initialization the implicit palette is set             */
/* Note: for EGA mode, palette is first converted to EGA        */
/*       colours by rounding. See fce x_set_spc_ega().          */

void x_palett(int len, char *paleta);

/* len    - length of palette (16,256)                          */
/* paleta - field with palette, that contains R,G,B parts for   */
/*          each colour. Length of field = len * 3 bytes        */


/*##-------------------- x_pal_1 ---------- Bios ---------------*/
/* Sets palette for 1 colour in 16/256 colour modes             */
/* Note: for EGA mode, palette is first converted to EGA        */
/*       colours by rounding. See fce x_set_spc_ega().          */

void x_pal_1(int n_pal, char *pal_1);

/* n_pal  - which colour (index)                                */
/* pal_1  - field with palette, that contains R,G,B parts       */


/*##-------------------- x_getpalette --------------------------*/
/* Vrati nastavenou paletu (delka 16,256) v poli, ktere obsahuje          */
/* pro kazdou polozku palety 3 byty (RGB)                                 */
/* Delka palety je dana nastavenym graf. modem.                           */
/* Pozn: V EGA modu jsou cisla barev prevedeny do RGB slozek              */
/*       tak ze bity 0..3 jsou vzdy 0, bity 4,5 jsou nastaveny.           */
/* tr.: Returns the defined palette (length 16,256) in a field that       */
/*      contains for each element of the palette 3 bytes (RGB)            */
/*      The length of the palette is given by the graph. mode setting.    */
/*      Note: In EGA mode the colour numbers are converted to RGB parts.  */
/*            So bits 0..3 are always 0, bits 4,5 are set.                */

void x_getpalette(char *pal3);

/* pal3 - field with palette                                    */

/*##-------------------- x_set_spc_ega -------------------------*/
/* Ma vyznam jen pro EGA mod. Nastavi zpusob ziskani EGA barvy  */
/* z RGB slozek. Pouzit pred x_palett,x_pal_1.                  */
/* tr.: Has meaning only for EGA mode. Defines the way of       */
/*      getting EGA colours from RGB parts. To be used before   */
/*      x_palett,x_pal_1.                                       */

void x_set_spc_ega(int flag);

/* flag = 0 - ega colour is selected by rounding from RGB parts */
/*            implicite value                                   */
/* flag = 1 - in the VGA palette the spec. EGA palette is       */
/*            located in the 2 highest bits of the RGB elements */

/*##-------------------- x_setcolor ---- Bios-------------------*/
/* Nastavi barvu (index) pro dalsi pouziti v kreslicich funkcich*/
/* Pozn: Pro x_bar() se nastavi barva pomoci x_setfill. Pro     */
/*       psani textu je tato barva barvou textu.                */
/* Po x_grf_ini() je nastavena barva 7.                         */
/* tr.: Sets colour (index) for further use in drawing functions  */
/*      Note: For x_bar() the colour is set with x_setfill. For   */
/*            writing text this colour is the colour of the text. */
/*      After/according to x_grf_ini() colour is set to 7.        */

void x_setcolor(int color);

/* color  - which colour (index)                                */


/*##-------------------- x_getmaxcol ---------------------------*/
/* Vrati maximalni cislo barvy (1,15,255).                      */
/* tr.: Returns maximum colour number (1,15,255).               */

int x_getmaxcol(void);


/*##-------------------- x_setfill -----------------------------*/
/* Nastavi barvu (index) pro dalsi pouziti ve funkci x_bar a    */
/* pro pozadi fce x_textxy() a x_text_ib()                      */
/* Po x_grf_ini() je nastavena barva 0.                         */
/* tr.: Sets colour (index) for further use in function x_bar   */
/*      and for background fce x_textxy() and x_text_ib()       */
/*      After/according to x_grf_ini() colour is set to 0.      */

void x_setfill(int vypln, int color);

/* vypln  - zpusob vyplneni obdelnika (ma vyznam jen pro        */
/* standartni mody, pro ostatni se vyplni barvou)               */
/* color  - ktera barva (index)                                 */
/* tr.: vypln - way of filling the rectangle (has meaning only for */  
/*         standard mode, for other modes it is filled with colour */
/*      color - which colour (index)                               */

/*##-------------------- x_setpattern --------------------------*/
/* Nastavi vzorek a barvu (index) pro dalsi pouziti ve funkci   */
/* x_bar(). Zavolanim fce x_setfill se kresleni uzivatelskym    */
/* vrorem zrusi.                                                */
/* tr.: Sets up pattern and colour (index) for further use in function   */
/*      x_bar(). Calling of fce x_setfill deletes drawing with user's    */
/*      pattern.                                                         */

void x_setpattern(char *patt, int color);

/* patt - pole 8 bytu dlouhe ve kterem je definovan vzor 8x8    */
/* color - barva pro kresleni vzorem  na pozadi barvou 0        */
/* tr.: patt - 8 bytes long field, that defines pattern 8x8     */
/*      color - colour for drawing with the pattern on          */
/*         background with colour 0                             */

/*##-------------------- x_setcircf ----------------------------*/
/* Nastavi zda kruznice, ci elipsy (x_circ,x_elipse) se budou   */
/* kreslit nevyplnene nebo vyplnene. Vyplnene se kresli barvou  */
/* nastavenou pres x_setfill. Vyplnovat lze plnou barvou nebo   */
/* nastavenym vzorkem.                                          */
/* tr.: Defines whether cirles or ellipses (x_circ,x_elipse) will    */
/*      be filled or not. If filled they are drawn with the colour   */
/*      set through x_setfill. Filling can be done fully or with     */
/*      a defined pattern.                                           */

void x_setcircf(int flag);

/* flag - 0 = Unfilled circles (ellipses), 1 = Filled circles */

/*##-------------------- x_getcol -------------------------------*/
/* Vrati cislo nastavene barvy                                   */
/* tr.: returns number of defined colour                         */
int x_getcol(void);

/*##-------------------- x_getfill ------------------------------*/
/* Vrati cislo nastavene barvy  vyplne                           */
/* tr.: returns number of defined filling colour                 */
int x_getfill(void);

/*##-------------------- x_setlinstyle --------------------------*/
/* Nastavi typ cary, definuje caru a nastavi tloustku cary       */
/* pro dalsi pouziti.                                            */
/* Pozn: Po x_grf_ini() je nastavena plna cara.                  */
/* tr.: Defines type of line, defines line and sets up strength  */
/*      of line for further use.                                 */
/*      Note: according to/after x_grf_ini() is defined          */
/*              a full/filled line.                              */

void x_setlinstyle(int typ, int user, int width);

/* typ  - preddefinovany typ cary PLNA,CARKOVANA,atd             */
/* user - ve dvou bytech definice cary (16 pixlu)                */
/* width- tloustka cary (zatim jen tenka)                        */
/* tr.: typ  - predefined type of line FILLED, DOTTED. etc.      */
/*      user - line definition in two bytes (16 pixels)          */
/*      width - strength of line cary (until now only thin)      */


/*##-------------------- x_setratio -----------------------------*/
/* Funkce nastavujici aspectratio, tj pomer velikosti pixlu ve   */
/* smeru X a Y. (Aby kruznice byla vzdy kruznice)                */
/* tr.: Function setting the aspectratio, ie. relation of pixel  */
/*      size in direction X and Y (in order to make a circle     */
/*      always look as a circle)                                 */

void  x_setratio( int xasp, int yasp );


/*##-------------------- x_getratio -----------------------------*/
/* Funkce vraci nastavene aspectratio, tj pomer velikosti        */
/* pixlu ve smeru X a Y. (Aby kruznice byla vzdy kruznice)       */
/* tr.: Function returns the defined aspectratio, ie. relation   */
/*      of pixel size in direction X and Y (in order to make     */
/*      a circle always look as a circle)                        */

void  x_getratio( int *xasp, int *yasp );


/*##-------------------- x_setview ------------------------------*/
/* Funkce nastavujici viewport                                   */
/* tr.: Function that sets the viewport                          */

void  x_setview(int x1, int y1, int x2, int y2, int clip);

/* x1,y1,x2,y2 - viewport                                        */
/* clip        - 0/1 vypnuti,zapnuti orezavani ve viewportu      */
/* clip        - 0/1 switch off,on cutting off in viewport       */


/*##-------------------- x_cleardev ----- VGA--------------------*/
/* clears the screen                                             */

void x_cleardev(void);


/*##-------------------- x_clearview ---- VGA--------------------*/
/* clears the viewport                                           */

void x_clearview(void);


/*##------------------- wrt_video ---------- VGA ----------------*/
/* Writes to the videoram buffer with picture                    */

int wrt_video(char *buf, int xz,int yz, int ncol,int nrow, int pix);

/* buf  - field containing pixels of picture, in one byte can     */
/*        also be more than one pixel (2,4,8)                     */
/*        Note: max. size of buffer is 64KB.                      */
/* xz   - beginning of picture horizontally in pixels             */
/* yz   - beginning of picture vertically in pixels               */
/* ncol - number of columns of picture                            */
/* nrow - number of rows of picture                               */
/* pix  - number if pixels in one byte (1,2,4,8).                 */
/*        For 256 colours pix=1                                   */


/*##------------------ x_line ------------------------------------*/
/* Draws section [x1,y1]-->>[x2,y2]                               */

void x_line(int x1, int y1, int x2, int y2 );


/*##------------------ x_poly ------------------------------------*/
/* Nakresli polygon o nump vrcholech. Souradnice polygonu jsou v  */
/* poli pole[]. Neuzavira se automaticky !                        */
/* tr.: Draws polygon with nump corners. Coordinates of polygonu  */
/*      are in field pole[]. It is not closed automatically!      */

void x_poly(int nump, int *pole );


/*##------------------ x_circle ----------------------------------*/
/* Nakresli kruznici o stredu [x1,y1] a polomeru r.               */
/* tr.: Draws circle with center [x1,y1] and radius r.            */

void x_circle(int x1, int y1, int r );


/*##------------------ x_ellipse ---------------------------------*/
/* Nakresli elipsu o stredu [x1,y1] a polomerech rx a ry.         */
/* tr.: Draws ellipse with center [x1,y1] and radii rx a ry.      */

void x_ellipse(int x1, int y1, int rx, int ry);


/*##------------------ x_arc -------------------------------------*/
/* Nakresli oblouk o stredu [x1,y1] uhlu od sa do ea o polomeru r */
/* tr.: Draws arch with center [x1,y1] angle from sa to ea and    */
/*      radius r                                                  */

void x_arc(int x1, int y1, int sa, int ea, int r );


/*##------------------ x_rect ------------------------------------*/
/* Nakresli nevyplneny obdelnik barvou z x_setcolor()             */
/* Pozn: pomoci fce x_wrtmode() lze nastavit psani s XOR          */
/* tr.: Draws unfilled rectangle in the colour from x_setcolor()  */
/*      Note: with the help of fce x_wrtmode() you can switch     */
/*            on writing with XOR                                 */

void x_rect(int xz, int yz, int xk, int yk);
                                                                                                  
/* xz   - beginning of rectangle horizontally                     */
/* yz   - beginning of rectangle vertically                       */
/* xk   - end horizontally                                        */
/* yk   - end vertically                                          */


/*##------------------ x_bar -------------------------------------*/
/* Nakresli vyplneny obdelnik  barvou z x_setfill()               */
/* Pozn: Pro standatr. mody pouziva Borland fce bar()             */
/*       a pro Super_VGA mody primo psani do VGA_reg.             */
/* tr.: Draws filled rectangle in the colour from x_setfill()     */
/*      Note: For standatr. modes it uses Borland fce bar() and   */
/*            for Super_VGA modes directly writing into VGA_reg.  */

void x_bar(int xz, int yz, int xk, int yk);

/* xz   - beginning of rectangle horizontally                     */
/* yz   - beginning of rectangle vertically                       */
/* xk   - end horizontally                                        */
/* yk   - end vertically                                          */

/*##------------------ x_fillpoly --------------------------------*/
/* Nakresli polygon vyplneny vzorem.                              */
/* tr.: Draws polygon filled with pattern.                        */

void x_fillpoly(int nump, int *pole );


/*##------------------ x_floodfill --------------------------------*/
/* Vyplni oblast zadanou barvou v x_setfill().                     */
/* tr.: fills area with colour given in x_setfill().               */

int  x_floodfill(int x, int y, int flag);

/* x,y  - startovni bod vyplnovani                                 */
/* flag - bit 0: 0->videoram,  1->interni bit. rovina xf_bitmap    */
/*        bit 1: 0->nelze dale pouzit x_flood_2, 1->lze pouzit     */
/* tr.: x,y  - point where the filling starts                          */
/*      flag - bit 0: 0->videoram,  1->internal bit level xf_bitmap    */
/*             bit 1: 0->x_flood_2 cannot be used any longer           */
/*                    1->can be used                                   */

/* Vyplneni barvou ci vzorem po predchozim volani x_floodfill      */
/* tr.: Filling with colour or pattern after calling x_floodfill   */

int  x_flood_2(int dx, int dy);

/* dx, dy - posune vyplnenou oblast o dx,dy                        */
/* tr.: dx, dy - moves filled area by dx,dy                        */

/* Uvolneni pameti : Ma vyznam pouze kdyz param. flag != 0         */
/* tr.: Releasing memory : Has meaning only if param. flag != 0    */

void x_flood_free(void);

/* Kazde nove volani x_floodfill() znovu samo uvolnuje a prideluje */
/* pamet. Ale x_flood_2() ne. Lze je tedy volat nekolikrat po      */
/* jednom x_floodfill().                                           */
/* tr.: Whereas every new call of x_floodfill() releases and allocates */
/*      memory again, x_flood_2() does not. So it can be called        */
/*      several times again after a single call of x_floodfill().      */

/*##----------- x_putpix -----------------------------------------*/
/* Writes pixel to videoram                                       */

void x_putpix(int x, int y, int color);


/*##------------- x_getpix ---------------------------------------*/
/* Loads pixel from videoram                                      */

int x_getpix(int x, int y);


/*##----------------- x_fnt_load ---------------------------------*/
/* Nahraje font pro psani v grafickem modu pro fce x_textxy()     */
/* a x_text_ib()                                                  */
/* Soubor s fontem je ve tvaru pro program FEDIT, ktery je        */
/* soucasti softwaru pro S_VGA kartu.                             */
/* tr.: Loads font for writing in graphic mode for fce x_textxy()        */
/*      and x_text_ib()                                                  */
/*      The file with the font has the format for the program FEDIT,     */
/*      that is part of the software for the S_VGA card.                 */

int x_fnt_load(char *fnt_file, int num, int mod);

/* fnt_file - soubor s fontem                                     */
/* num      - pocet radku textu na stranku (pro mod 0)            */
/* mod      - 0 - nahrati do BIOSU, pak lze pouzit fci x_textxy() */
/*                Pro tento mod se x_fnt_load() nemusi volat.     */
/*            1 - nahrati jen do vlastniho bufru pro x_text_ib()  */
/*                Pozn: Bufer zabira cca 4-64kB                   */
/*            2 - oboji (0 a 1)                                   */
/* Pozn:      Pro proporcni fonty (WINDOWS) se provede vzdy pouze */ 
/*            mod 1.  Nelze je vnutit BIOSU !!!                   */
/* tr.: fnt_file - file that contains the font                             */
/*      num      - number of text rows on a page (for mod 0)               */
/*      mod      - 0 - loading into BIOSU, then fci x_textxy() can be used */
/*                 For this mod it is not necessary to call x_fnt_load() . */
/*                 1 - loading only into a special buffer for x_text_ib()  */
/*                 Note: Buffer occupies cca 4-64kB                        */
/*                 2 - both (0 and 1)                                      */
/*      Note:      For proportional fonts (WINDOWS) will be executed       */
/*                 always only mod 1. They cannot be forced to BIOSU !!!   */

/*##----------------- x_fnt_cls ----------------------------------*/
/* Uvolni font z pameti (xg_fbuf), pripadne uvolni XMS, zavre     */
/* soubor s fontem. Neni-li zadny font v pameti, neprovede se nic */
/* tr.: Releases font from memory (xg_fbuf), releases XMS, closes font  */
/*      file. If there is no font in memory, it does nothing            */

int x_fnt_cls(void);

// return - 1   - O.K.
//          2,4 - did not succeed in releasing XMS, close font file


// Naalokuje buffer xg_fbuf pro praci s fontem. Tento buffer se
// bude pouzivat stale dokola pro ruzne fonty. Implicitne se vzdy
// pro kazdy font znovu alokuje novy bufer dle velikosti fontu.
// tr.: Allocates buffer xg_fbuf for work with font. This buffer will 
//      be used again and again for different fonts. For every new font will
//      be allocated as default a new buffer according to font size.

int x_fnt_alloc(long Size);
// size - max. velikost fontu v Bytech. Urci se tak, ze vezmeme nejdelsi
// soubor s fontem a od nej odecteme 1280.
// Zadame-li Size=0, bude se alokovat pro kazdy font znovu pamet.
// Pozn: Volani zrusi jiz nacteny font.
// Pozn: Max. velikost Size je 64kb !!!
// tr.: size - max. font size in Bytes. We get it by taking the
//      longest file with the font and subtracting 1280. 
//      If we enter Size=0, for every font memory will be allocated again.
//      Note: Calling deletes font that has allready been loaded.
//      Note: Max. size Size is 64kb !!!

// ret : 1 - O.K.,  0-Error (no memory)

/*##----------------- x_fnt_initxms -----------------------------*/
// Inicializuje tabulky a XMS pro uchovavani fontu v XMS.
// Funkce x_fnt_load() pak umisti nacteny font do XMS, a pri dalsim volani
// x_fnt_load() se stejmym jmenem ho necte z disku ale z XMS. Dojde-li
// misto v tabulce fontu, nebo XMS pak se tabulka vyprazdni a zacne se
// znova plnit pri volani x_fnt_load().
// Pozn: max. delka jmena pro font je 31 znaku !!!
//       volat nekde pri init xlopifu po x_fnt_alloc() !!! (pouziva-li se)
// Jeden font umisteny v XMS zabira navic 38B v konvencni pameti, tabulka
// se alokuje v x_fnt_initxms(). XMS se alokuje take zde a ma velikost
// Maxfnt * SIZE_XMS_FNT bytu (prumerne 10kB na font viz #define SIZE...).
// Pro uvolneni XMS je nutno na konci programu voloat x_fnt_cls().
// tr.: Initializes tables and XMS in order to save font in XMS.
//      Function x_fnt_load() then places loaded font into XMS, and
//      at the next call of x_fnt_load() with the same name it will not 
//      be read from disk, but from XMS. If the font table or XMS runs out 
//      of space, then the table will be emptied and it will be filled
//      again at call of x_fnt_load()
//      Note: max. length of name for font is 31 characters !!!
//            to call somewhere during init of xlopif after x_fnt_alloc() !!!
//            (provided that it is used)
//      One font placed in XMS takes 38B more in conventional memory, the 
//      table is allocated in x_fnt_initxms(). XMS is also allocated here
//      and has the size Maxfnt * SIZE_XMS_FNT bytes (average 10kB for
//      one font cf. #define SIZE...).
//      In order to release XMS it is necessary to call x_fnt_cls() at the
//      end of the program.

int  x_fnt_initxms(int Maxfnt);
// Maxfnt - max. pocet fontu soucasne v XMS. 0,-1 znamena zadne fonty
//          v XMS(jako drive).
// tr.: Maxfnt - max. number of fonts, being in XMS at one time.
//               0,-1 means no fonts in XMS(as before/previously).

/*##------------------ x_textxy -------------- Bios --------------*/
/* Napise text v grafickem modu. Tvar textu lze menit zmenou      */
/* fontu pres fci x_fnt_load. Pise pouze vodorovne. Zarovnani ve  */
/* smeru x se nastavi pres fci x_settextjusty(). Pise se barvou   */
/* nastavenou v x_setcolor() a na pozadi nastavene pres funkci    */
/* x_setffill().  Souradnice se zadavaji v pixlech a jsou pred    */
/* psanim zaokrouhleny na radky a sloupce textu dle velikosti     */
/* fontu. Pocet radku a sloupcu na obrazovce  zavisi na velikosti */
/* velikosti fontu. Typicky znak je velky 8x8,8x14,8x16 pixlu.    */
/* Pozn: Tato funkce vyuziva pro psani textu BIOS.                */
/* tr.: Writes text in graphic mode. The shape of the text can be changed */
/*      by changing the font with the help of fci x_fnt_load. It writes   */
/*      only horizontally. Alignment in x direction is set with the help  */
/*      of fci x_settextjusty(). It writes with the colour set in         */
/*      x_setcolor() on a background set with function x_setffill().      */
/*      Coordinates are entered in pixels and are rounded to rows and     */
/*      columns according to the font size before writing. The number of  */
/*      rows and columns on the screen depends on the font size.          */
/*      A typical character has the size 8x8,8x14,8x16 pixels.            */
/*      Note: This function uses BIOS for writing text.                   */

void x_textxy(int xp, int yp, char *text);

/* xp,yp - coordinates of the beginning of text in pixels           */
/* text  - written text                                             */


/*##------------------ x_text_ib ------------- VGA reg -----------*/
/* Napise text v grafickem modu. Tvar textu lze menit zmenou      */
/* fontu pres fci x_fnt_load. Pise pouze vodorovne. Zarovnani ve  */
/* smeru x,y se nastavi pres fci x_settextjusty(). Pise se barvou */
/* nastavenou v x_setcolor() a na pozadi nastavene pres funkci    */
/* x_setffill().  Souradnice se zadavaji v pixlech a text lze     */
/* na rozdil od fce x_textxy() umistit od libovolneho pixlu.      */
/* Pozn: Pro 16 bar. mody je x_text_ib() asi 2x pomalejsi nez     */
/*       x_textxy(). Pro 256 bar. mody je asi 2x rychlejsi.       */
/* tr.: Writes text in graphic mode. The shape of the text can be changed */
/*      by changing the font with the help of fci x_fnt_load. It writes   */
/*      only horizontally. Alignment in x direction is set with the help  */
/*      of fci x_settextjusty(). It writes with the colour set in         */
/*      x_setcolor() on a background set with function x_setffill().      */
/*      Coordinates are entered in pixels and the text can be placed -    */
/*      differently from fce  x_textxy() - at any pixel on the screen.    */
/*      Note: For 16 colour mode x_text_ib() is about 2x slower than      */
/*            x_textxy(). For 256 colour mode it is about 2x faster.      */

int x_text_ib(int xp, int yp, unsigned char *text);

/* xp,yp - coordinates of the beginning of text in pixels         */
/* text  - written text                                           */
/* return  1 - text O.K., 2- character from XMS|DISK could not be read */


/*##----------------- x_settextjusty -----------------------------*/
/* Sets way of text alignment                                     */
/* After initialization it is set to: left,top                    */

void x_settextjusty(int horiz, int vert);

/* horiz - 0=left, 1=center, 2=right                              */
/* vert  - 0=bott, 1=center, 2=top                                */


/*##----------------- x_txwidth,x_txheight -----------------------*/
/* Returns length and height of text in pixels (according to current font) */

int x_txwidth (char *string);
int x_txheight(char *string);

/* string - for x_txheight() it has no meaning                       */

/*##----------------- x_charmax ----------------------------------*/
/*  Urci kolik znaku se vejde do zadane sirky v pixlech           */
/*  tr.: finds out how many characters fit to the given width in pixels  */

int x_charmax(unsigned char *string, int dxpixlu);

//  string - string with text
//  dxpixlu- width of window in pixels
//  ret    - max. number of characters to given width
//           (mainly for prop. fonts)
//           and if it is smaller, length of string in characters

/*##----------------- x_charmod ----------------------------------*/
/*  Nastavi, zda se u textu bude kreslit i pozadi (pusvitny text) */
/*  tr.: Defines, whether, together with text will also be drawn  */
/*       background (transparent text)                            */

void x_charmod(int chrmod);

// chrmod - 0 - bude se kreslit pozadi (implicitni po startu)
//          1 - nebude se kreslit pozadi - prusvitny text
// tr.: chrmod - 0 - background will be drawn (default after start)
//               1 - background will not be drawn - transparent text

int x_getcharmod(void);   // returns type of background

/*##----------------- x_text_zoom --------------------------------*/
/* Nastavi zvetseni textu pro x_text_ib(), Pouze: 2x nebo 4x      */
/* tr.: Sets enlargement of text for x_text_ib(), Only: 2x or 4x  */

void x_text_zoom(int zoom);

/* zoom - velikost textu (pouze 1,2,4)                             */

/*##---------------- x_getsize -----------------------------------*/
/* Zjisti velikost bufru pro uschovu obrazku                      */
/* Je buffer >= 64K vrati 0xFFFF, jinak potrebnou velikost        */
/* tr.: Finds out size of buffer for saving image                 */
/*      If buffer >= 64K it returns 0xFFFF, otherwise the required size  */

unsigned int x_getsize(int x1, int y1, int x2, int y2);

/* x1,y1,x2,y2 - window in pixels                                  */


/*##---------------- x_set_planes ---------------------------------*/
/* Nasatvi pocet bitovych rovin pro GETIMG,PUTIMG (16 barevne mody)*/
/* Implicitne 4 bit. roviny, rozsah 1..4 roviny                    */

void x_set_planes(int planes);

/* planes - pocet bitovych rovin (1..4) */
/* tr.: planes - number of bit levels (1..4) */

/*##---------------- x_getimg ---------------------- VGA_reg-------*/
/* Uschova cast obrazovky zadane oknem do bufru                    */
/* Pozn: Pro 16 bar.mody je obraz ve tvaru bit. rovin po radcich.  */
/*       Pro 256 bar. mody primo ctenim videoram. Bufer na zacatku */
/*       obsahuje pocet_slopcu(2B) a pocet_radku(2B). Dale pixly   */
/*       obrazu ulozene po radcich. 1 pixel = 1B.                  */
/* tr.: Saves the part of the screen that is given by the window   */
/*      into buffer                                                */
/*      Note: For 16 colour mode the image is in the form of bit   */
/*            levels according to rows.                            */
/*            For 256 colour modes directly by reading the         */
/*            videoram. The buffer contains at the beginning       */
/*            the number of columns pocet_slopcu(2B) and the       */
/*            number of rows pocet_radku(2B). Followed by the      */
/*            pixels of the image saved according to rows.         */
/*            1 pixel = 1B.                                        */

void x_getimg(int x1, int y1, int x2, int y2, char *bitmap);

/* x1,y1,x2,y2 - window on the screen                              */
/* bitmap      - buffer with image                                 */


/*##---------------- x_putimg ---------------------- VGA_reg-------*/
/* Zobrazi v danem miste cast obrazovky, ktera byla uschovana pres */
/* funkci x_getimg.                                                */
/* Pozn: Pro 16 bar. mody je obraz zapsan pres Borland fci putimage*/
/*       Pro 256 bar. mody zapisem primo do videoram.              */
/* tr.: Displays at the given place the part of the screen that has     */
/*      been saved by the function x_getimg.                            */
/*      Note: For 16 colour modes the image is written through Borland  */
/*            fci putimage. For 256 colour modes it is written directly */
/*            to the videoram.                                          */

void x_putimg(int xz,int yz, char *bitmap, int op);

/* xz,yz - beginning of the immage                                  */
/* bitmap- buffer with the immage                                   */
/* op    - see: putimg(), only 16 colour modes                      */


/*##------------- x_maxx, x_maxy -----------------------------------*/
/* Finds out the maximum size of coordinates for active graph. mode */

int x_maxx(void);       /* In direction X */
#ifdef CALDERA
#define x_maxy() my_x_maxy()	// Screen size excludes S/W keyboard
#else
int x_maxy(void);       /* In direction Y */
#endif // CALDERA


/*##------------- x_yncurs -----------------------------------------*/
/* Zapne (vypne) zobrazovani kursoru, zobrazi kursor v pocatecnim   */
/* bode a nastavi jeho barvu.                                       */
/* tr.: Switches on (off) cursor display, displays cursor at the    */
/*      initial point and sets its colour.                          */

void x_yncurs(int on, int x, int y, int col);

/* on     = 1 -> zapnuti kursoru, = 0 -> vypnuti kursoru            */
/* Pozn: Vsechny dalsi parametry maji vyznam pouze pro zapnuti !!   */
/* x,y  - misto pocatecniho zobrazeni kursoru                       */
/* col  - barva kursoru                                             */
/* tr.: on     = 1 -> switch cursor on, = 0 -> switch cursor off         */
/*      Note: All other parameters have meaning only for switching on!!  */
/*      x,y  - place of initial display of the cursor                    */
/*      col  - colour of the cursor                                      */


/*##------------- x_cursor -----------------------------------------*/
/* Simulace kurzoru 16x16 pixlu pro 256 barevne mody                */
/* Tvar kursoru se definuje vyplnenim tzv. screen a cursor masky.   */
/* Jde o bitova pole 16 * 16 bitu (viz. popis GMOUSE )              */
/* viz. fce x_defcurs()                                             */
/* Na kazde zavolani kurzor v starem miste zmizi a objevi se v      */
/* novem. Pri zadavani stejnych souradnic fce. x_cursor nedela nic. */
/* tr.: Simulation of the cursor of 16x16 pixels for 256 colour modes  */
/*      The shape of the cursor is defined by filling of a so-called   */
/*      and a cursor mask. It is a field of 16x16 bits (cf. the        */
/*      desription of GMOUSE)                                          */
/*      cf. fce x_defcurs()                                            */
/*      With every calling the cursor disappears at the former place   */
/*      and appears at the new location. If the entered coordinates    */
/*      are identical fce. x_cursor does nothing.                      */

void x_cursor(int x, int y);

/* x,y - coordinates for cursor display                                */


/*##------------- x_defcurs ----------------------------------------*/
/* Definuje tvar kursoru pomoci screen a cursor poli (jako v GMOUSE)*/
/* tr.: Defines the shape of the cursor with the help of screen     */
/*      and a cursor field (as in GMOUSE)                           */

void x_defcurs(short *screen, short *cursor, int color);

/* int screen[16], cursor[16] - bit masks 16x16 cursor definition   */


/*##-------------------- x_wrtmode -----------------------------*/
/* Specificates the way of writing (overwriting/XOR) into videoram */
/* for functions: x_rect(), x_line(); ...                          */

void x_wrtmode(int wrtmode);

/* wrtmode   -  0=overwriting(default), 1=XOR                      */

/*##-------------------- x_id_ellip ----------------------------*/
/* Identifikace hranice elipsy(kruznice) oknem                  */
/* tr.: Identification of the border of an ellipse (circle) by a window */

int  x_id_ellip( int okno[4],       // identification window (I/O)
                 int x1, int y1,    // center
                 int rx, int ry );  // radii

/* Return : 0/1 - Identified NO/YES                             */
/*          if YES: in the window [0],[1] identif. point        */

/*##--------------------- img_to_file --------------------------*/
/* Creates file .OBR by saving a part of the screen             */

int img_to_file(int x1, int y1, int x2, int y2, char *file);

// x1..y2 - window on the screen
// file   - file name
// It is presumed that this is called only in graph. mode
// Note: Numbers are rounded to multiples of 8, In order to save memory 
//       it writes on single lines (not very fast!)
//       4kB local variables (STACK !)

/*##-------------------- x_detect ------------------------------*/
/* Detection of super VGA card                                  */

int x_detect(char *svga, int *kby);

//  svag - string according to IBASE (vystup, TSG3,TSG4,OAK,TRIDENT...3)
//  kby  - memory of SVGA in kB   (output) - only 64..256kB
//  ret  - 0 - no known one was detected 
//         1 - was detected

/*##-------------------- x_strcmp ------------------------------*/
/* Hledani substringu ve stringu (stringy mohou obsahovat bin 0)*/
/* tr.: Searching a substring in string (strings can contain bin 0) */

char *x_strcmp(char *buf, int lenb, char *str, int lens);

//  buf - string that we search through, lenb- its length 
//  str - strign that we are looking for, lens- its length
//  return - NULL - not found, other - address str in buf


//##--------- Setting parameters for line break ---------------
void x_set_zalom(int istyle, int ostyle, float MaxMez, float MaxRozt);

/*
istyle -   0 => respektovat vicenasobne mezery
	   1 => vicenas.mezera = 1 mezera
ostyle -   bit 0:  0 => bez zarovnavani (="na praporek")
		   1 => zarovnavani na pravy okraj
	   bit 1:  0 => nenechavat "sirotky" na konci radku
		   1 => nechavat sirotky (sirotek= 1 osamele pismeno:
		       a..z,A..Z,nebo jakakoliv otviraci zavorka)
MaxMez -   maximalni pripustna sirka mezery - zadava se jako nasobek
	   sirky jedne mezery, musi byt >= 1.0; hodne velke => zarovnava
	   se jen rozsirovanim mezer mezi slovy; doporucena hodnota:
	   <2.0 .. 3.0>
MaxRozt -  maximalni pripustna pomocna mezera mezi znaky - nasobek
	   prumerne sirky znaku. Je-li <= 0.0 => zadne mezery se
	   nebudou vkladat, jinak se roztec vzdy zaokrouhlli alespon
	   na 1 pixel; doporucena hodnota: <0.1 .. 0.4>

tr.: istyle -   0 => respect multiple space
                1 => multiple space = 1 space
     ostyle -   bit 0:  0 => without alignment (="flag shape")
                        1 => right aligned
                bit 1:  0 => do not leave "orphans" at end of line
                        1 => leave orphans (orphan= 1 lonely letter:
                           a..z,A..Z,or any opening parenthese)

[This is a specific rule in Czech orthography: Czech language knows
several one-letter-prepositions. If they occur as the last word in a
line ("lonely letter") they should rather be moved to the beginning
of the next line.]

     MaxMez -   maximum accepted width of spaces - is entered as a
                multiple of the width of one space, must be >= 1.0;
                very large => is aligned only by enlarging spaces
                between words; recommended value: <2.0 .. 3.0>
     MaxRozt -  maximum accepted auxiliary space bewenn characters -
                multiple of average width of character. If it is <=0.0
                => no spaces will be inserted, otherwise the spacing
                will always be rounded to at least 1 pixel; recommended
                value: <0.1 .. 0.4>
*/
//##------- Nastaveni psani do vide/XMS pro vektorove fce (x_line...)
//##------- tr.: Sets up writing into vide/XMS for vector fce (x_line...)
void x_video_XMS(int vidXMS, int bincol);

//vidXMS - 0-psani na obrazovku(default), 1-do XMS(disk)
//bincol - barva pro psani do XMS pro binar (0-cerne,1-bile)
//tr.: vidXMS - 0-writing to the screen(default), 1-to XMS(disk)
//     bincol - colour for writing into XMS for binar (0-black,1-white)

/*====================== Internal function(s) ============================*/
void x_text_1(int col, int xz, int yz, char *text, int xr);
/* col  - colour of text                                           */
/* xz   - beginning of text horizontally                           */
/* yz   - beginning of text vertically                             */
/* text - written text                                             */
/* xr   - for 16 colour modes 0/1 = overwrite/XOR                  */
/*        for 256 colour modes - background colour                 */

void x_putpix_v(int x, int y, int color);
int  x_getpix_v(int x, int y);

int z_xmsputpix(int x, int y, int color);
int z_xmsgetpix(int x, int y);
int z_xmsbar(int x1, int y1, int x2, int y2, int col);

void rea_w256(char  *buf, int xz, int yz, int dx, int dy);
void rea_bincga(char  *buf, int xz, int yz, int dx, int dy);
void x_bar256(int col, int xz, int yz, int dx, int dy);
void x_putpix_s(int x, int y, int color);
int  x_getpix_s(int x, int y);
void x_rect_s(int col, int xz, int yz, int dx, int dy, int xr);
void wrt_chr(char  *buf, int xz, int yz, int dx, int dy,
	     int col1, int col2);
int  wrt_video1(char *buf, int xz,int yz, int ncol,int nrow, int pix);
void wrt_bincga(char  *buf, int xz, int yz, int dx, int dy,
		int LxMask, int First);
int  x_viewline(int *x1, int *y1, int *x2, int *y2 );
void pal_vga_ega(char *pal1, int *ega_col, int jak); /* conversions EGA <--> VGA */
void pal_ega_vga(int ega_col, char *pal1);
void x_bar_patt(int x1,int y1,int dx,int dy);
void x_gimg16(int x1,int y1,int x2,int y2,char *buf,int plan);  //GETIMAGE
void x_pimg16(int x1,int y1,char *buf, int op, int plan);       //PUTIMAGE
void x_b16(int x1,int y1,int x2,int y2,int col,int mask);       //16BAR

/*====================== External variables of lopif ===================*/

extern int xg_mod;                   /* Graph. mod   */
extern int xg_256;                   /* 0/1 selected 256 colour mode      */
extern int xg_color;                 /* Selected colour (rect,text)       */
extern int xg_fillc;                 /* Selected colour fort x_bar()      */
extern int xg_wrt;                   /* 0=overwrite, 1=XOR                */
extern int xg_style;                 /* Definition of line                */
extern int xg_xr,xg_yr;              /* ratio                             */
extern int xg_view[4];               /* Viewport                          */
extern int xg_xfnt,xg_yfnt;          /* Font size in pixels               */
extern int xg_tjustx;                /* Text alignment direction X        */
extern int xg_tjusty;                /* Alignment direction Y             */
extern int xg_clip;                  /* whether to cut off in viewport    */
extern int xg_fbyt;                  /* Number of bytes for one font char */
extern int xg_flag;                  /* Types: bit 0 - spec EGA pal       */
                                     /*        bit 1 - active patt        */
extern unsigned char xg_upatt[8];    /* Pattern 8x8 for filling           */
extern char *xg_fbuf;                /* Font bufffer                      */

#if HI_COLOR
extern char xg_curs[520];            /* Image under cursor           */
#else
extern char xg_curs[260];            /* Image under cursor           */
#endif

extern char xg_f_cur;
extern unsigned int xg_c_col;       /* On/Off,cursor colour     */
extern int  xg_x_cur,xg_y_cur;      /* Last coordinate          */
extern short xg_s1[16];   /* Screen mask  */
extern short xg_s2[16];   /* Cursor mask  */
extern int xg_svga;     /* Bit map for SVGA modes       */
                        /* lower byte : bit 0..3 line length 320..1024 */
                        /*              bit 7    0=16/1=256 colours    */
                        /* higher byte : 0..6 type of graphic card (7=VESA) */
extern int xg_intern;   /* Internal number of graphic mode 0..7 */
extern char xg_egapal[17];  /* Palette for EGA */
extern unsigned short xg_and16[68];   // Masks for 16 colour cursor
extern unsigned short xg_or16[68];
extern int xg_fnt_zoo;      /* For zooming of text 1 | 2 */
extern int xg_col_plan;     // Number of bit levels
extern int xg_notview;      // 0-to vieport, 1-absolutely (only text)

//---- News 930319 for WINDOWS fonts ------------------------
extern unsigned char xg_fonlen[256];  // Width of character for prop. fonts
extern long  int     xg_fonadr[256];  // Beginning of char for prop. fonts
extern unsigned char xg_foncon;       // Flag - const/prop font [0/average]
extern unsigned char xg_fonmem;       // Where is the font: 0-MEM(xg_fbuf),1-XMS,2-DISK
extern int           xg_fonhan;       // Handle for XMS/DISK
extern unsigned int  xg_lbfnt;        // length of buffer with font data
extern char          xg_fnt_akt[64];  // Name of active font
// Note: for fonts in XMS|DISK xv_fbuf 4K for one character
extern long          xg_fntalloc;     // Fixed length of buffer for font

//---- More fonts in XMS 971001
#define  SIZE_XMS_FNT  10000   // average size of fonts in XMS

struct FNTXTAB             // for one font in memory
{
#ifdef POSIX
  void          *ptr;
  void          *fbuf;
#else
   long         Offset;    // offset buffer with font in XMS
   unsigned int Size;      // length of font in XMS
#endif
   char         Name[80];  // font name
};

struct FNTMBUF             // buffer for writing/reading font in XMS
{ unsigned int  lbfnt;
  int           xfnt;
  int           yfnt;
  int           fbyt;
  int           foncon;
  int           fonmem;
  int           fonhan;
  unsigned char fonlen[256];
  long     int  fonadr[256];
};

extern int  xg_fnt_max;       // max number of fonts in XMS
extern int  xg_fnt_fre;       // first free in table
extern int  xg_fnt_xms;       // handle XMS
extern long xg_fnt_xlen;      // whole length in XMS
extern long xg_fnt_xoff;      // free space in XMS
extern struct FNTXTAB *xg_fnt_xtab;   // font table

//---- Zalamovani textu ----------------------------------------
//---- tr.: line break  ----------------------------------------
//extern int   xg_istyle;               // Parameters of function IZALOM(...)
//extern int   xg_ostyle;
//extern float xg_MaxMez;
//extern float xg_MaxRozt;

extern int   xg_31yn;                 // Zda znaky <32 kreslit/nebo jen posun
          // tr.: whether charcters <32 draw/or only move/shift
extern int   xg_chrmod;               // Kresleni pozadi textu
          // tr.: drawing background of text
extern int   xg_no_mode;              // !=0-> doopravdy se grf.mod nenastavi
          // tr.: !=0-> really/in reality graphic mode is not set
//---- for VIRT video -----------------------------
extern int   xg_video_XMS;            // 0-kresli se do video, 1-do VIRT
                              // tr.: 0-draws to video, 1-do VIRT
extern int   xg_bincol;               // barva pro kresleni do binarni VIRT
                              // tr.: colour for drawing to binary VIRT
//---- Writing coloured text (allocate field and fill with colours)
extern unsigned char *xg_chr1c;       // Foreground
extern unsigned char *xg_chr2c;       // Background


//---- Hi color modes ------------------------------
#if HI_COLOR
// Conversion to hi-col from palette (0..63)
#ifdef POSIX
#define RGBHI15(R,G,B)  (((unsigned short)R>>1)<<10)|(((unsigned short)G>>1)<<5)|((unsigned short)B>>1)
#define RGBHI16(R,G,B)  (((unsigned short)R>>1)<<11)|((unsigned short)G<<5)|((unsigned short)B>>1)
#else
#define RGBHI15(R,G,B)  (((unsigned)R>>1)<<10)|(((unsigned)G>>1)<<5)|((unsigned)B>>1)
#define RGBHI16(R,G,B)  (((unsigned)R>>1)<<11)|((unsigned)G<<5)|((unsigned)B>>1)
#endif
// Fce "C":
//##---- Conversion of buffer with RGB triplets to Hi-col (2B)
void xh_RgbToHi(unsigned char *Rgb, unsigned char *Hi,
	    int Pixs, int Rows, int LenLin);
//Rgb   - in - Buffer se vstupnimi radky obrazu v RGB (0..255)
//Hi    - out- Buffer s vystupnim obrazem v Hi-col
//Pixs  - in - Pocet pixlu na radek
//Rows  - in - Pocet radku obrazu (Rows < 0 : zrcadleni radku)
//LenLin- in - Delka vstupniho radku v bytech
// tr.: Rgb   - in - Buffer with input rows of picture in RGB (0..255)
//      Hi    - out- Buffer with output picture in Hi-col
//      Pixs  - in - Number of pixels in one row 
//      Rows  - in - Number of rows in picture (Rows < 0 : inverting row)
//      LenLin- in - Length of input row in bytes

//##---- Conversion of buffer with byte (palette) picture to Hi-col
void xh_ByteToHi(unsigned char *Ib1, unsigned char *Hi,
	    short Pixs, short Rows, short LenLine);
//Ib1   - in - Buffer se vstupnimi radky obrazu: pixel=byte(0..255)
//Hi    - out- Buffer s vystupnim obrazem v Hi-col
//Pixs  - in - Pocet pixlu na radek
//Rows  - in - Pocet radku obrazu (Rows < 0 : zrcadleni radku)
//LenLin- in - Delka vstupniho radku v bytech
//tr.:  Ib1   - in - Buffer with input rows of picture: pixel=byte(0..255)
//      Hi    - out- Buffer with output picture in Hi-col
//      Pixs  - in - Number of pixels in one row
//      Rows  - in - Number of rows in picture (Rows < 0 : inverting row)
//      LenLin- in - Length of input row in bytes

//##---- Prevod R,G,B (paleta 0..63) do Hi-col
unsigned short xh_RgbHiPal(unsigned char R, unsigned char G, unsigned char B);
// R,G,B - in - polozky palety R,G,B v rozsahu 0..63
// return  :    Hi-color pixel

//##---- Conversion of Hi-col pixels to R,G,B components (0..63)
void     xh_HiPalRgb(unsigned int Hi, unsigned char *rgb);
//Hi   - in - pixel in Hi-color (e.g. from x_getpix() )
//rgb  - out- components R,G,B pixels
//Pozn:  xh_RgbHiPal() a xh_HiPalRgb() nejsou invezni fce. Pri
//       prevodu dochazi ke ztrate presnosti v jednom bitu na slozku.
//tr.: Note: xh_RgbHiPal() and xh_HiPalRgb() are not invertable fce. Through 
//           conversion accuracy is lost in one bit in each component.

//##---- Nastavi zpusob ziskani delky radku pri kresleni fci wrt_video()
//##---- tr.: Sets up the way of getting the length of row while
//            drawing fci wrt_video()
int      xh_SetRounding(int Round);
//round - in - How to get the length of row in bytes:
//    0 - length of row is calculated by simply multiplying pixels and bits/pixel
//    1 - length of row is rounded up to a multiple of 4 bytes (eg.BMP)
//    return : original round
//    default : 0

//##---- Nastavi zpusob prace s paletou v Hi-color modech
//##---- tr.: Sets up the way how to work with the palette in Hi-color modes
int      xh_SetPalMode(int Mode);
//Mode - in - How to work with the palette
//   0 - All colours x_setcol(),... are entered as index into the palette
//       that needs, however, to be set up first
//   1 - Colours are entered as Hi-col RGB, no need to set up palette 
//    return : original mode
//    default : 0

// Fce "ASM" -> internal
int  xh_getpix(int x, int y);
void xh_putpix(int x, int y, int color);
int  xh_read(unsigned char *buf,int xpix, int ypix, int ncol, int nrow);
void xh_barx(unsigned int col, int xz, int yz, int dx, int dy);
int  xh_wrtchr(char *buf,int xpix, int ypix, int ncol, int nrow, int col1, int col2);
int  xh_write(unsigned char *Buf, int x, int y, int dx, int dy);

// Externals
extern int xg_hi16;    // mode : 16bit=1,15bit=0
extern int xg_xgr;     // order of RGB in triplet for true color, 0=BGR,1=RGB
extern int xg_hipalmod;// x_setcolor(), x_setfill():0=index, 1=direct hicolor
extern int xg_round;   // for calculating length of row in B (0=according ncol, 1-mult. four(BMP))
extern unsigned char xg_hipal[768];  // Pal for HiCol mode (set up through x_setpal())
extern unsigned short  xg_hival[256]; // Hi-col values belonging to the palette
#endif

#ifdef VIRT_SCR
//  *******  Fce for managing with virtual videoram: ********
//  Virtual videoram is in XMS(EMS) or disk file and simulate
//  normal videoram.
//  Supported types : 1bit/pixel, 8bit/pixel,
//  Max number of virtual videoram in program : 3

//  Using:
//  1. Create or open virtual videoram(s), set active virt.videoram
//  2. Call x_video_XMS(1,...), All output from xlopif->virt.mem.
//  3. Call xlopif function, x_bar(), x_text_ib(),...
//  4. Copy virt. videoram to screen
//  5. Close (delete) virt. videoram

//##--- Create new virtual videoram
int xv_new_virt(char *filenam, // File name for disk file
	      int dx, int dy,  // Size in pixels
	      int col,         // Default color
	      int bitpix,      // 3-1bit/pixel, 0-8bit/pixel, -1-16bit/pixel
	      int npal,        // Length of palette
	      char *pal,       // Palette, range RGB 0..63, max. 256 entries
	      int Index,       // Index 0..5, number of virtual videoram
	      int Typ);        // 0-XMS or DISK, 1-XMS, 2-DISK

//  return   1 - O.K. XMS,
//           3 - O.K. Disk, else error

//##--- Open file .OBR as virtual videoram
int  xv_opn_virt(char *filnam,    // File name of .OBR
		int Index,        // number of virtual videoram
		int Typ);         // 0-XMS or DISK, 1-XMS, 2-DISK
// return   1 = O.K XMS
//          3 = O.K disk
//          2 = Open error .OBR,   4 = Read error
//          6 = Memory DOS error,  8 = Write error (XMS)

//##---- Close virtual videoram .OBR vytvoreny xv_opn_file,xv_new_obr
int  xv_cls_virt(int Xwrt,        // 0-only free XMS, 1-write XMS on disk
		int Index);       // number of virtual videoram
// return   1 = O.K. else error

//##---- Set active virtual videoram for next operation
int xv_set_actvirt(int Index);    // 0..2, default 0

//##---- Copy rect from active virt.videoram to screen
int  xv_to_scr(int xs, int ys, int xo, int yo, int dx, int dy);
// xs,ys - left upper point in virt.videoram
// xo,yo - left upper point in screen
// dx,dy - size in pixels
// return - 1 - O.K. else error

//##---- Copy rect from screen to active virt. vieoram
int  xv_to_virt(int xo, int yo, int xs, int ys, int dx, int dy);
// xo,yo - left upper point in screen
// xs,ys - left upper point in virt.videoram
// dx,dy - size in pixels
// return - 1 - O.K. else error

//##--- Read rect to buffer from virt.videoram in format getimage()
int  xv_int_rea(int xs, int ys, int dx, int dy, char *buf);
//   xs,ys - upper left point
//   dx,dy - size in pixels
//   buf   - output buffer with format getimage(), max. 64KB !!!

//##--- Write rect to virt mem. in format putimage() -----
int  xv_int_wrt(int xs, int ys, char *buf);
//   xs,ys - upper left point
//   buf   - input buffer with format getimage(), max. 64KB !!!

//##--- Write text to virt. screen
int xv_text_virt(int xp, int yp, char *text);
//   xp,yp - point in pixels
//   text  - buffer with text;

//##--- Write rect to virtual screen
int xv_wrt_virt(char *Buf, int xz, int yz, int ncol, int nrow, int pix);

// WARNING:
// getimage,putimage,cursor working always with real videoram

// RESTRICTION:
// 1. Not implemented XOR, only COPY for putpixel, line, rect, bar ...
// 2. Not implemented fill pattern for bar, only solid fill

//-------------------- Global var. for vv -------------------
typedef struct tagXV_VV
{
int  xv_XMS;           // For extended memory ( -1 = disk)
int  xv_file;          // Open file .OBR (handle)
int  xv_bits;          // For shift 8bit=0,4bit=1,1bit=3
int  xv_len_col;       // Number of bytes in a row
int  xv_rows;          // Number of rows
int  xv_zmap;          // beginning of the image in the file (512,1024)
//!!mp: reduced
char xv_File[16];
} XV_VV;

extern XV_VV xv_vv[MAXVIRTUAL];
extern int   xv_act;    // Actual vv
extern int   xv_NoDisk; // 1-only XMS, 0-Disk|XMS
#endif
/*==================================================================*/
