/*
    LHMAIN.C - program to load a DOS executable into upper memory
    Copyright (C) 1995 Svante Frey
 
    Compiler assumptions: Structures are packed (byte-aligned).

    This file should normally not be run directly, since that may
    cause memory fragmentation problems. This file should be compiled
    and linked to a .COM file, and then appended to the LOADHI.COM file.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>       /* must have those MK_FP() macros */
#include "dosstruc.h"  /* contains structures used by DOS */
#include "dosfunc.h"   /* contains prototypes for the assembler functions */
#include "loadhi.h"    /* contains macros, global variables, etc */

/* These files contains parts of the program */
#include "getargs.h"
#include "findfile.h"
#include "chkfile.h"
#include "allocmem.h"

/*  This program tries to load another program into the upper memory.
 *
 *  Brief functional description of the program:
 *   1. Copy environment info, search for upper memory regions
 *   2. Parse arguments
 *   3. Identify file to load
 *   4. If EXE, read EXE header
 *   5. Calculate memory requirements
 *   6. Allocate memory blocks
 *   7. Create environment copy
 *   8. Create new PSP
 *   9. Load file
 *  10. Execute program
 *  11. Exit
 */

int main(void)
{
   if (initialise() == OK)

     if ((rc = parseArgs(myPSP->commandLine)) == OK)
     /* command line was OK - try to find the file */
 
        if ((rc = findfile()) == OK)
        /* a file was found: load it */
            
           rc = loadFileHigh();


   /* if any error occurred, rc will hold the error code */
   if (rc) {
       error(rc);
       exitcode = 255;
   }

   cleanup();

   return exitcode;
}

int initialise(void)
{
   old_link = DosSetUMBLink(1);
   old_strat = DosSetStrategy(0);

   myPSP = MK_FP(getCurrentPSP(), 0);

   if (!(umbRegion = malloc(64 * sizeof(*umbRegion))))
       return err_prog_mem;

   if (!(regionOrder = malloc(64 * sizeof(*regionOrder))))
       return err_prog_mem;

   if ((rc = readEnv(myPSP->environSeg)) == OK)
       rc = findUMBRegions();

   return rc;
}

void cleanup(void)
{
   /* Restore UMB link state and DOS malloc strategy.
      These variables were set by the startup code. */
   DosSetUMBLink(old_link);
   DosSetStrategy(old_strat);
 
   if (!rc && sw[sw_verbose]) {
       /* print exit messages */
       printf ("\nLOADHI: %s %s, exit code = %d\n",
                filename, termName[exittype], exitcode);
       printf ("LOADHI: exiting...\n");
   }
}

/* readEnv(): 
 *
 * read this program's environment info, and copy it into near memory.
 * If no environment is found (the environment segment in the PSP is zero),
 * a default environment consisting of only an empty string will be created.
 */

int readEnv(WORD env_seg)
{
   char *envp;

   myEnvSize = 1;

   if (env_seg) {
       struct MCB far *mcb = MK_FP(env_seg - 1, 0);
       myEnvSize = mcb->size << 4;
   }

   if (!(myEnv = malloc(myEnvSize)))
       return err_alloc_env;

   if (!env_seg) myEnv[0] = 0;
   else farmemcpy(myEnv, MK_FP(env_seg, 0), myEnvSize);

   envp = myEnv;

   /* get the size of the environment strings */
   while (*envp) envp += strlen(envp) + 1;
   myEnvSize = envp - myEnv + 1;
   return OK;
}

/* error(): print error messages to stderr */

void error(int errcode)
{
   if (errcode > err_help)
       fprintf (stderr, "LOADHI: error - ");

   switch(errcode)
   {
      case err_invalid_cmdline:
        fprintf (stderr, "invalid command line\n");
        /* fall-through to show usage */
      case err_help:
        usage();
        break;
      case err_not_found:
        fprintf (stderr, "file not found: %s\n", filename);
        break;
      case err_env_create:
        fprintf (stderr, "couldn't create the environment copy\n");
        break;
      case err_open:
        fprintf (stderr, "error opening file %s\n", filename);
        break;
      case err_dos:
        fprintf (stderr, "DOS error %d\n", dos_error);
        break;
      case err_invalid_file:
        fprintf (stderr, "file %s is not a valid EXE file\n", filename);
        break;
      case err_psp_create:
        fprintf (stderr, "error creating new PSP\n");
        break;
      case err_loading_file:
        fprintf (stderr, "error loading file %s\n", filename);
        break;
      case err_prog_mem:
        fprintf (stderr, "couldn't allocate enough memory for the program\n");
        break;
      case err_bat_file:
        fprintf (stderr, "%s is a batch file, start it from the command line\n",
                 filename);
        break;
     case err_invalid_name:
        fprintf (stderr, "invalid file name: %s\n", filename);
        break;
     case err_invalid_mem:
        fprintf (stderr, "illegal memory size arguments\n");
        break;
     case err_invalid_path:
        fprintf (stderr, "invalid path\n");
        break;
     case err_mcb_chain: /* error while searching the MCB chain */
        fprintf (stderr, "MCB chain corrupt, or MS-DOS incompatible system\n");
        break;
     default:
        fprintf (stderr, "error code %d\n", errcode);
   }
}

void usage(void)
{
   fprintf (stderr, "\nLOADHI - Loads a program into upper memory.\n");
   fprintf (stderr, " (C) 1995 Svante Frey\n\n");
   fprintf (stderr, "Usage: LOADHI [options] filename [arguments to <filename>]\n\n");
   fprintf (stderr, " /L:m[,n][;m[,n]]...  Specify which UMB regions the program can use\n");
   fprintf (stderr, " /S                   Shrink the regions to minimal size before loading\n"); 
   fprintf (stderr, " /E                   Create no environment copy for the program\n");
   fprintf (stderr, " /V                   Verbose mode - print informational output\n");
   fprintf (stderr, " /@XXXX               Load program at segment address XXXX\n");
   fprintf (stderr, " /M=min[,max]         Specify minimal and maximal memory requirements\n");
   fprintf (stderr, "                      of the program.\n");
}

/* loadFileHigh():
 *
 * This is the 'main' procedure, that does most of the work.
 */

int loadFileHigh(void)
{
   int i;
   
   if (sw[sw_verbose])
       printf ("LOADHI: Loading file %s\n", filename);
       
   if ((rc = checkFile()) != OK)   /* check if file valid, determine type of file */
        return rc;

   /* Check if the 'max' argument has a resonable value */
   if (maxMem && (maxMem < memNeeded || maxMem < minMem))
       return err_invalid_mem;

   if (minMem > memNeeded) memNeeded = minMem;
   if (minMem > memWanted) memWanted = minMem;
   if (maxMem) memWanted = maxMem;
 
   /* allocate memory for the program */
   if ((rc = allocateMemory()) != OK)
        return rc;

   /* create new PSP, complete with command line, environment & FCBs */
   if (createClientPSP() != OK)
       return err_psp_create;

   if (sw[sw_verbose])
       printf ("LOADHI: New PSP created, loading file...\n");   

   calcRelocs();   /* calculate relocation factor & some other stuff */

   if ((rc = loadFile()) != OK)  /* read the file into memory */
       return rc;

   markMCBs();   /* make the new program owner of its MCB:s */
  
   if (inUMB) DosSetStrategy(0x80);

   if (sw[sw_verbose])
       printf ("LOADHI: Loading successful...executing program at %04x:%04x\n",
               init.cs, init.ip);

   /* execute the file */
   rc = execute();

   /* free any memory that was allocated to prevent the program from using it */
   for (i = 0; i < allocatedBlocks; i++)
        DosFree(block[i]);

   return rc;
}

int createClientPSP(void)
{
   /* It is important that the exec file handle is not open at this 
      However, this inheritance gurantees that I/O redirection commands 
      will work as expected. */

   if (createChildPSP(pspSeg, pspSeg + memSize) != OK)
       return err_psp_create;

   newPSP = MK_FP(pspSeg, 0);

   /* copy environment info */
   if (envSeg) {
       char far *envp = MK_FP(envSeg, 0);

       farmemcpy(envp, myEnv, myEnvSize);   /* copy the strings */
       envp += myEnvSize;
       *((int far *)envp)++ = 1;                /* extra word count = 1 */
       farmemcpy(envp, filename, strlen(filename) + 1); /* copy filename */
   }

   /* write environment and command line info to new PSP */
   newPSP->environSeg = envSeg;
   farmemcpy(newPSP->commandLine, newcmdline, *newcmdline + 2);

   /* make default FCBs of the two first arguments */
   makeDefaultFCBs();

   return OK;
} 

/* makeDefaultFCBs():
 *
 * The routine parses the two first argument of the new command line into
 * FCBs, by calling the DOS function to do this.
 */

int makeDefaultFCBs(void)
{
   char args[2][128];         /* buffers for the arguments */
   int currArg, i;
   char *c = newcmdline + 1;
   char *argstart;

   /* Copy the default, empty FCB to use if no arguments found */
   farmemcpy(newPSP->fcb[0], &dummyFCB, 0x10);
   farmemcpy(newPSP->fcb[1], &dummyFCB, 0x10);

   /* Copy the two arguments to ASCIIZ strings */
   for (currArg = 0; currArg < 2; currArg++)
   {
       for (; *c != '\r'; c++)
            if (*c == ' ' || *c == '\t') break;
     
       if (*c == '\r') break;    /* end of command line */

       argstart = c;
       while (*c != ' ' && *c != '\t' && *c != '\r') c++;
 
       memcpy(args[currArg], argstart, c - argstart);
       args[currArg][c - argstart] = 0;
   }
  
   /* Use the DOS parse FCB function (int 21h, ah = 29h) 
      to parse the arguments. The program's initial AX value shall 
      reflect if the default FCB:s contain invalid drive letters.    */

   for (i = 0; i < currArg; i++) {
        if (DosParseFCB(args[i], &dummyFCB) == 0xff)
            init.ax |= i? 0xff00 : 0xff;
    
        farmemcpy(newPSP->fcb[i], &dummyFCB, 0x10);
   }
}

/* calcRelocs(): Calculate the relocation factor to use, 
   and where the file image will be loaded in memory. 
   Fill in the program's initial register values.
*/

void calcRelocs(void)
{
   relocFactor = pspSeg;
   
   /* If the file is to be loaded high, the file data will be loaded
      at the top of the allocated memory block, otherwise it will be loaded
      right after the PSP. */

   if (loadhighFlag)
       relocFactor += memSize - topara(prgSize);
   else if (exeType == t_exe) relocFactor += 0x10;
   
   /* imageSeg = the segment where the file image will be loaded
      relocFactor = the value to add to segment references */
   imageSeg = relocFactor;
   if (exeType == t_com) imageSeg += 0x10;

   /* now, complete the program's initial register values */
   init.cs += relocFactor;
   init.ss += relocFactor;
   init.ds = pspSeg;
   init.es = pspSeg;
   init.cx = (WORD) prgSize; /* bx:cx = number of bytes read from the file */
   init.bx = prgSize >> 16;
   init.dx = 0;
   
   /* For a COM program, SP will be set to the end of the memory block,
      if it's < 64K, or to FFFE, if it's >= 64K */
   if (exeType == t_com)
       init.sp = memSize >= 0x1000? 0xfffe : (memSize << 4) - 2;
}

/*   The PSP memory block and the environment memory block of the new program
 *   must be marked as owned by the new process.
 *   DOS will free them when the program terminates.
 */

void markMCBs(void)
{
   /* Pointers to MCB:s */
   struct MCB far *envMCB = MK_FP(envSeg - 1, 0);   
   struct MCB far *pspMCB = MK_FP(pspSeg - 1, 0);
 
   /* Mark these memory blocks as owned by the client process */
   pspMCB->owner = pspSeg;
   if (envSeg) envMCB->owner = pspSeg;

   newmodNameLen = getModuleName(newModuleName, filename);
   if (newmodNameLen < 8) newmodNameLen++;

   farmemcpy(pspMCB->name, newModuleName, newmodNameLen);   
}

/*  getModuleName():
 *
 *  This routine finds the module name in a filename (the part between the 
 *  path and the extension).
 */

size_t getModuleName(char *dest, char *src)
{
   char  *nameEnd = strrchr(src, '.');
   char  *nameStart = strrchr(src, '\\');

   if (!nameStart) 
       nameStart = src[1] == ':'? src + 2 : src;
   else nameStart++;

   if (!nameEnd || nameEnd < nameStart) 
       nameEnd = src + strlen(src);

   memcpy(dest, nameStart, nameEnd - nameStart);
   dest[nameEnd - nameStart] = 0;

   return nameEnd - nameStart;
}

/* findUMBRegions():
 *
 * This routine scans the MCB chain to find all active memory regions.
 * Info about the regions will be written to the array "umbRegions". 
 *
 * Each continous region is numbered. Region 0 is the 
 * conventional memory.
 */

int findUMBRegions(void)
{

  struct UMBREGION *region = umbRegion;
  struct MCB far *mcb = MK_FP(GetFirstMCB(), 0);
  char sig;
  int i;

  /* First, find the end of the conventional memory:
     Turn UMB link off, and track the MCB chain to the end. */

  DosSetUMBLink(0);

  region->start = FP_SEG(mcb);

  while (mcb->sig == 'M') 
      FP_SEG(mcb) += mcb->size + 1;
      
  if (mcb->sig != 'Z') return err_mcb_chain;

  /* If the last memory block in conventional memory is a "reserved" one,
     conventional memory ends at the paragraph before the block. If the
     last block is an ordininary one, conventional memory ends at the
     last paragraph of the block. */

  if (mcb->owner == 8 && !farmemcmp(mcb->name, "SC", 3))
      region->end = FP_SEG(mcb) - 1;
  else region->end = FP_SEG(mcb) + mcb->size;
     
  region++;
  region->start = 0;

  /* Turn UMB link on. If MS-DOS UMBs are available, the signature of
     the last conventional memory block will change from 'Z' to 'M'. */

  DosSetUMBLink(1);

  if (mcb->sig == 'M') {            /* UMBs are available */
      FP_SEG(mcb) += mcb->size + 1;   /* go to next block */

      do {
         sig = mcb->sig;
          
         if (mcb->owner == 8 && !farmemcmp(mcb->name, "SC", 2)) {
             /* this is a 'hole' in memory */
             if (region->start) {
                 region->end = FP_SEG(mcb) - 1;
                 region++;
                 region->start = 0;
             }
         } else { 
             struct MCB far *umb_mcb = mcb;
             FP_SEG(umb_mcb)--;
    
             if (umb_mcb->sig == 'Z' || umb_mcb->sig == 'M') 
                 if (!farmemcmp(umb_mcb->name, "UMB     ", 8)) {
                     /* This is how MS-DOS marks UMB regions */    
                     
                     mcb = umb_mcb;
                     region->start = mcb->owner;
                     region->end = mcb->owner + mcb->size - 1;
                     if ((sig = mcb->sig) == 'M') region->end--;
                     region++;
                     region->start = 0;
                     FP_SEG(mcb) += mcb->size;
                     if (sig == 'Z') break;
                     continue;
                  }
             if (!region->start) 
                 region->start = FP_SEG(mcb);
         } 

         if (sig == 'Z') {
             region->end = FP_SEG(mcb) + mcb->size;
             region++;
         }

         FP_SEG(mcb) += mcb->size + 1;
      } while (sig == 'M');
     
      if (sig != 'Z')
          return err_mcb_chain;
  }
  umbRegions = region - umbRegion;

  /* By default, the program will have access to all areas. This may 
     be modified by command-line arguments. */

  for (i = 0; i < umbRegions; i++) {
       umbRegion[i].access = 1;
       umbRegion[i].minSize = 0xffff;
       if (i > 0) regionOrder[i - 1] = i;       
  }
  regionOrder[umbRegions - 1] = 0;
  return OK;
}

/* loadFile():
 *
 * This functions loads the file, by calling the DOS load
 * overlay function. 
 */

int loadFile(void)
{
   struct LoadOverlay lo;

   lo.loadSeg = imageSeg;
   lo.relocFactor = relocFactor;

   return DosLoadOverlay(filename, &lo);
}
