/*
 * RCEDIT.C - Resource Editor
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <io.h>
#include <dos.h>
#include <dir.h>
#include <time.h>
#include <alloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ifsmgr.h>
#include <dialog.h>
#include <errno.h>

#define RCVERSION	0x0100
#define MAXOBJECT	64
#define MAXWINDOW	(25*80)
#define MAXALLOCSIZE	(16 + (MAXWINDOW * 2) + (MAXOBJECT * 16))
#define MAXNAMELEN	32

#define MAX_X		79
#define MAX_Y		24
#define MAX_ROW		25
#define MAX_COL		80

#define MAXATTRIB	(8+16+16)

#define NAMEOFFSET	25
#define TOBJ_DELETED	0xFFFF

#define _A_MLINE	0x0001
#define _A_UTIME	0x0002
#define _A_UDATE	0x0004
#define _A_MENUS	0x0008
#define _A_SLINE	0x0010
#define _A_MOUSE	0x0020
#define _A_COLOR	0x0040

typedef struct {
	unsigned Saved		: 1;
	unsigned Resource	: 1;
	unsigned Code		: 1;
	unsigned Color		: 1;
	unsigned NewDialog	: 1;
	unsigned NewObject	: 1;
	unsigned OpenDialog	: 1;
	unsigned Unused		: 9;
      } rcflag;

typedef struct {
	WORD	version;
	WORD	flag;
	COLOR   dlg;
	COLOR	dos;
      } RCFG;

typedef struct {
	WORD	memsize;        /* alloc size */
	FOBJ	dialog;
	FOBJ	object[MAXOBJECT];
	WCHR	wz[MAXWINDOW + 8];
      } CRES;

typedef struct {
	unsigned char x;
	unsigned char y;
	unsigned char at;
	unsigned char ch;
     }  MSXY;

typedef struct {
	RECT 	rc;
	int  	key;
      } stinfo;

typedef struct {
	int 	flag;
	COLOR *	clp;
	char 	lcount;
	char 	old_lcount;
	COBJ 	old_cursor;
	WCHR *	old_screen;
      } AOBJ;

#define isOpen()	(dialog.flag & _D_DOPEN)
#define isVisible()	(dialog.flag & _D_ONSCR)
#define isDeleted(o)	(o->flag == TOBJ_DELETED)

const char copyright[] =
"Resource Editor Version 1.0 Copyright 1995, Hjort Nidudsson\n\n";

const char cp_usage[] =
"RCEDIT accepts the following command line syntax:\n"
" RCEDIT [Options] [file]\n\n"

" Options may be preceded by either / or -. Valid options are as follows:\n"
"    -N     Create a new dialog\n"
"    -O#    Open dialog # from [file]\n"
"    -G     Use graphical color [265:16] (default is standard MSDOS color)\n";

const char cp_statusline[] =
"&F&1 Help  &F&2 New  &F&3 Open  &F&4 Save  &F&5 Menu  "
"&F&6 DOBJ  &F&7 Color         &E&s&c Exit";

char filename_c[WMAXPATH];
char filename_t[WMAXPATH];
char filename_b[WMAXPATH];

char ext_c[] = ".c";
char ext_t[] = ".tmp";
char ext_b[] = ".bak";

int _end = 0;

RCFG config = {
	RCVERSION, 0, {
		{	0x00,0x10,0x20,0x30,0x40,0x50,0x60,0x70	},
		{	0x00,0x0D,0x0F,0x09,0x0F,0x07,0x08,0x07,
			0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F	},
		{	0,136,7,7,4,63,3,7,56,59,0,0,4,3,63,63	}
	},{
		{	0x00,0x10,0x70,0x70,0x40,0x30,0x30,0x70	},
		{	0x00,0x0A,0x0F,0x0B,0x0F,0x07,0x08,0x07,
			0x08,0x0B,0x00,0x00,0x0F,0x0A,0x0F,0x0F	},
		{	0,1,2,3,4,5,20,7,56,57,58,59,60,61,62,63 }
	}
};

AOBJ api = { _A_MLINE|_A_SLINE|_A_MOUSE|_A_UTIME|_A_COLOR, &config.dlg, 24, };
rcflag cflag = { 1, 0, 0, 0, 0, };
CRES res;
WCHR window[MAXWINDOW + 80 + 50 + 32];

stinfo st_line[] = {
	{ { 1,24, 7, 1}, F1 },
	{ {10,24, 6, 1}, F2 },
	{ {18,24, 7, 1}, F3 },
	{ {27,24, 7, 1}, F4 },
	{ {36,24,12, 1}, F5 },
	{ {50,24, 9, 1}, F6 },
	{ {62,24, 7, 1}, F7 },
	{ {71,24, 7, 1}, ESC },
};

#define STLCOUNT	(sizeof(st_line) / sizeof(stinfo))
#define ALTX		0x2D00

char dialogname[128];
char identifier[128];
TOBJ object[MAXOBJECT];
char objectname[MAXOBJECT][128];
DOBJ dialog = {
	_D_DMOVE|_D_SHADE|_D_RESAT, 0, 0, { 10,7,10,60 }, NULL, object };

extern ROBJ *IDD_EVENT;
extern ROBJ *IDD_QUIK_MENU;
extern ROBJ *IDD_EXIT;
extern ROBJ *IDD_CHILD;
extern ROBJ *IDD_SAVE_DIALOG;
extern ROBJ *IDD_HELP;
extern ROBJ *IDD_DOBJECT;
extern ROBJ *IDD_BACKGROUND;
extern ROBJ *IDD_Color;
extern ROBJ *IDD_FOREGROUND;
extern ROBJ *IDD_Frametype;
extern ROBJ *IDD_TOBJECT;
extern ROBJ *IDD_DialogOpen;

enum {
	ID_COLOR1,
	ID_COLOR2,
	ID_COLOR3,
	ID_COLOR4,
	ID_COLOR5,
	ID_COLOR6,
	ID_USECOLOR,
	ID_EDITATTRIB,
	ID_EDITPAL,
	ID_STANDARD,
	ID_LOAD,
	ID_SAVE,
	ID_QUIT,
     };

extern unsigned char tclrascii;

int asetlcount(AOBJ *ap, int lcount)
{
	if (lcount == *__egaline)
		return 0;
	__mshide();
	if (lcount == 49) {
		bios_scanline400();
		bios_loadfont8x8();
	} else {
		bios_settextmode();
		if (ap->flag & _A_COLOR)
			loadpal(ap->clp->palett);
	}
	__msshow();
	return (ap->lcount = *__egaline);
}

int aopen(AOBJ *ap)
{
	int at;

	ap->old_lcount = *__egaline;
	ap->old_screen = scpush(ap->old_lcount + 1);
	getcursor(&ap->old_cursor);
	if (ap->flag & _A_MOUSE)
		mouseinit();
	asetlcount(ap, ap->lcount);
	if (ap->flag & _A_COLOR) {
		memcpy(at_background, ap->clp, sizeof(COLOR));
		console |= CON_COLOR;
		loadpal(at_palett);
	}
	if (ap->flag & _A_MLINE) {
		at = at_Menus;
		scputc(0, 0, at, 80, ' ');
		if (ap->flag & _A_UTIME) {
			console |= CON_UTIME;
			scputfg(72, 0, 8, 8);
		}
		if (ap->flag & _A_UDATE) {
			console |= CON_UDATE;
			scputfg(63, 0, 8, 8);
		}
	}
	if (ap->flag & _A_MOUSE)
		mouseon();
	return 1;
}

void aclose(AOBJ *ap)
{
	if (ap->flag & _A_MOUSE)
		mouseoff();
	scpop(ap->old_screen, ap->old_lcount + 1);
	if (ap->lcount != ap->old_lcount)
		asetlcount(ap, ap->old_lcount);
	setcursor(ap->old_cursor);
	if (ap->flag & _A_COLOR)
		resetpal();
}

int asetcolor(AOBJ *ap, COLOR *cp)
{
	ap->clp = cp;
	memcpy(at_background, cp->backgr, 8);
	memcpy(at_foreground, cp->foregr, 16);
	memcpy(at_palett, cp->palett, 16);
	if (ap->flag & _A_COLOR)
		loadpal(ap->clp->palett);
	else if (console & 0x80)
		resetpal();
	return 1;
}

static WCHR apbuf[80];

void apushstl(const char *cp)
{
	wcpushst(apbuf, cp);
}

void apopstl(void)
{
	wcpopst(apbuf);
}

void rc_loadattrib(void)
{
	if (cflag.Color == 0) {
		api.flag &= ~_A_COLOR;
		asetcolor(&api, &config.dos);
	} else {
		api.flag |= _A_COLOR;
		asetcolor(&api, &config.dlg);
	}
}

void rc_initscreen(void)
{
	int q;

	rc_loadattrib();
	scputa(0, 1, 80*23, at_Desktop);
	scputa(0, 0, 80, at_Menus);
	scputa(0,24, 80, at_Menus);
	wcputs(getxyp(1, 24), 80, 0, cp_statusline);
	for (q = 0; q < 8; q++) {
		scputa(1 + q, 1, 1, q | q << 4);
		scputa(1 + q, 22, 1, q | q << 4);
	}
	for (q = 8; q < 16; q++) {
		scputa(1 + q, 1, 1, q | 0x10);
		scputa(1 + q, 22, 1, q | 0x10);
	}
}

void rc_putscreen(void)
{
	int  q,z;
	RECT rc;

	scputc(0, 1, 0, 80 * 25, ' ');
	scputs(2, 0, 0, 0, "Resource Edit:");
	scputs(57, 0, 0, 0, "[00:00][00:00]");
	rc.x = 0;
	rc.y = 1;
	rc.col = 80;
	rc.row = 23;
	rcframe(rc, getxyp(0, 0), 80, 0);
	rc.col = 18;
	rc.row = 23;
	rcframe(rc, getxyp(0, 0), 80, 0x0300);
	rc.y = 18;
	rc.col = 18;
	rc.row = 4;
	rcframe(rc, getxyp(0, 0), 80, 0x0300);
	scputc(78, 12, 0, 2, '-');
	scputc(39, 1, 0, 1, 179);
	scputc(39, 23, 0, 1, 179);
	scputc(1, 1, 0, 16, 219);
	scputc(1, 22, 0, 16, 219);
	for (q = 0; q < 16; q++) {
		for (z = 0; z < 16; z++)
			scputc(1 + z, 2 + q, 0, 1, (16 * q) + z);
	}
	for (q = '0'; q <= '9'; q++) {
		scputc(20 + q - '0', 1, 0, 1, q);
	}
	for (q = 'A'; q <= 'Z'; q++) {
		scputc(42 + q - 'A', 1, 0, 1, q);
		scputc(42 + q - 'A', 23, 0, 1, q | 0x20);
	}
}

int event_handler(void)
{
	switch (rsmodal(IDD_EVENT)) {
	case 1: return UP;
	case 2: return ESC;
	case 3: return ENTER;
	case 4: return DOWN;
	}
	return 0;
}

int event_child(void)
{
	switch (rsmodal(IDD_CHILD)) {
	case 4: return _C_REOPEN; /* Rewrite dialog and continue*/
	case 3: return _C_ESCAPE; /* Close dialog and return 0	*/
	case 2: return _C_RETURN; /* Close dialog and return 1	*/
	case 1: return _C_NORMAL; /* Continue dialog		*/
	}
	return 0;
}

void setdefault(int q)
{
	object[q].rc.x   = 4;
	object[q].rc.y   = 2;
	object[q].rc.col = 8;
	object[q].rc.row = 1;
	object[q].count  = 0;
	object[q].ascii  = 0;
	object[q].flag   = 0xFFFF;//_O_STATE;
	object[q].data	 = objectname[q];
	object[q].proc	 = event_handler;
	objectname[q][0] = 0;
}

void rcdefault(void)
{
	int q;

	dialog.flag   = _D_STDDLG;
	dialog.rc.x   = 10;
	dialog.rc.y   = 7;
	dialog.rc.row = 10;
	dialog.rc.col = 60;
	dialog.count  = 0;
	dialog.index  = 0;
	dialog.object = object;
	for (q = 0; q < MAXOBJECT; q++)
		setdefault(q);
}

void rcputidd(const char *n)
{
	if (dialog.rc.y) {
		scputc(15, 0, 0, 32, ' ');
		scputf(15, 0, 0, 32, ": IDD_%s", n);
	}
}

int opendlg(void)
{
	int color;
	RECT rc;

	if (dialog.flag & _D_DOPEN)
		return twshow(&dialog);

	switch(dialog.flag & 0x7000) {
		case _D_STERR: color = at_Error; break;
		case _D_MENUS: color = at_Menus; break;
		case _D_MUSER: color = at_Desktop; break;
		default:
			color = at_Dialog;
			break;
	}

	if (dialog.flag & _D_MENUS) {
		if ((dialog.wp = (WCHR *)rcopen(
			dialog.rc,
			dialog.flag,
			color,
			NULL,
			dialog.wp)) == NULL) {
			return 0;
		}
		rc = dialog.rc;
		rc.x = 0;
		rc.y = 1;
		rc.row--;
		rcframe(rc, dialog.wp, rc.col, 0x0000);
	} else {
		if ((dialog.wp = (WCHR *)rcopen(
			dialog.rc,
			dialog.flag,
			color,
			dialogname,
			dialog.wp)) == NULL) {
			return 0;
		}
	}

	dialog.flag |= _D_DOPEN;
	twshow(&dialog);
	return 1;
}

int closedlg(void)
{
	int result;

	result = twclose(&dialog);
	rcdefault();

	cflag.Saved = 0;
	cflag.Resource = 0;
	cflag.NewDialog = 1;

	return result;
}

const char cp_msstart	[] = "Click on start point..";
const char cp_msend  	[] = "Click on end point..";
const char cp_msinvalid	[] = "Ivalid point...";

int msinvalid(void)
{
	apushstl(cp_msinvalid);
	while (mousep())
		;
	apopstl();
	return 0;
}

int getxy(int *sx, int *sy, const char *msg)
{
	int 	x = *sx,
		y = *sy,
		end = 0;

	apushstl(msg);
	while (mousep());

	while (end == 0) {
		gotoxy(x, y);
		switch (getevent()) {
		case UP:
			if (y == 0)
				break;
			y--;
			break;
		case LEFT:
			if (x == 0)
				break;
			x--;
			break;
		case RIGHT: x++; break;
		case DOWN:  y++; break;
		case MOUSECMD:
			x = mousex();
			y = mousey();
		case ENTER:
			end = 1;
			break;
		case ESC:
			end = ESC;
			break;
		}
	}
	*sx = x;
	*sy = y;
	apopstl();
	return (end != ESC);
}

int msgetevent(const char *msg)
{
	int event;

	apushstl(msg);
	while (mousep());
	event = getevent();
	apopstl();
	if (event != MOUSECMD)
		return 0;
	return 1;
}

int msgetrc(RECT dl, RECT *rp)
{
	RECT rc;
	int x,y;

	x = dl.x;
	y = dl.y;
	if (getxy(&x, &y, cp_msstart) == 0)
		return 0;
	rc = dl;
	if (rcxyrow(rc, x, y) == 0)
		return msinvalid();
	rc.x = x;
	rc.y = y;
	if (getxy(&x, &y, cp_msend) == 0)
		return msinvalid();
	if (y == rc.y && x == rc.x)
		return msinvalid();
	if (y < rc.y) {
		rc.row = rc.y - y + 1;
		rc.y = y;
	} else {
		rc.row = y - rc.y + 1;
	}
	if (x < rc.x) {
		rc.col = rc.x - x + 1;
		rc.x = x;
	} else {
		rc.col = x - rc.x + 1;
	}
	if (rcinside(dl, rc) != 0)
		return msinvalid();
	*rp = rc;
	return 1;
}

int msget(MSXY *ms, const char *cp)
{
	int x,y;

	if (msgetevent(cp) == 0)
		return msinvalid();
	x = mousex();
	y = mousey();
	ms->x = x;
	ms->y = y;
	ms->at = getxya(x, y);
	ms->ch = getxyc(x, y);
	return 1;
}

/* Read resource from C-file into memory */

char inc_resourse[] =
"#if !defined(__INC__DTYPES)\n"
" #include <_dtypes.h>\n"
"#endif\n\n"
"#define\t_RCEDIT_\n\n";

char res_begin	[] = "/******** Resource begin %s *\n";
char res_fobj	[] = "\t{ 0x%04X, %3d, %3d, {%2d,%2d,%2d,%2d} },\n";
char res_data	[] = "********* Resource data  *******************/\n"
			     "static int %s_RC[] = {\n";
char res_ifdef	[] = "#ifdef _RCEDIT_\n";
char res_endif	[] = "#endif\n\t";
char res_bsize	[] = "}; //\t%d byte\n";
char res_robj	[] = "ROBJ *IDD_%s = (ROBJ *)&%s_RC;\n";
char res_end	[] = "/******** Resource end   %s */\n";

int line_id = 0;

int c_cwgetline(FILE *fp, char *b)
{
	if (fgets(b, 256, fp) == NULL)
		return 0;
	if (b[0] == '#') {
		if (fgets(b, 256, fp) == NULL)
			return 0;
	}
	strcpy(b, strstart(b));
	strtrim(b);
	return (b[0] >= '0' && b[0] <= '9');
}

int c_getcw(FILE *fp, char *b, void *wp)
{
	WORD *w = (WORD*)wp;

	if (b[0] < '0' || b[0] > '9') {
		if (c_cwgetline(fp, b) == 0)
			return 0;
	}
	*w = xtol(&b[2]);
	strcpy(b, b + 7);
	return 1;
}

int c_getresource(FILE *fp, char *b)
{
	while (fgets(b, 256, fp) != NULL) {
		// /******** Resource begin %s\n";
		if (strnicmp(b, res_begin, NAMEOFFSET) == 0)
			return 1;
	}
	return 0;
}

int c_testname(char *b, char *n)
{	// /******** Resource begin %s *\n";
	char *p;
	char namea[MAXNAMELEN];
	char nameb[MAXNAMELEN];

	if (strnicmp(b, res_begin, NAMEOFFSET) != 0)
		return 0;
	if ((p = strchr(b + NAMEOFFSET, ' ')) == NULL)
		return 0;
	*p = 0;
	strcpy(namea, b + NAMEOFFSET);
	*p = ' ';
	strcpy(nameb, n);
	strrev(nameb);
	strrev(namea);
	if (stricmp(nameb, namea) == 0)
		return 1;
	return 0;
}

int c_getdialog(FILE *fp, char *b, char *name)
{
	while (c_getresource(fp, b)) {
		if (c_testname(b, name))
			return 1;
	}
	return 0;
}

int c_getdlgdata(FILE *fp, char *b)
{
	while (fgets(b, 256, fp) != NULL) {
		// '********* Resource data  ********'
		if (strnicmp(b, res_data, 32) == 0) {
			if (fgets(b, 256, fp) == NULL)
				return 0;
			return 1;
		}
	}
	return 0;
}

int c_getdlgend(FILE *fp, char *b)
{
	while (fgets(b, 256, fp) != NULL) {
		// /******** Resource end %s\n";
		if (strnicmp(b, res_end, NAMEOFFSET) == 0)
			return 1;
	}
	return 0;
}

int c_read(char *n)
{
	FILE *fp;
	int *w;
	CRES *p;
	DOBJ dl;
	char b[256];
	int z,q,count;

	if ((fp = fopen(filename_c, "r")) == NULL)
		return 0;
	if (c_getdialog(fp, b, n) == 0 || c_getdlgdata(fp, b) == 0) {
		fclose(fp);
		return 0;
	}

	p = &res;
	w = (int*)p;
	count = 0;
	while (c_getcw(fp, b, w++)) {
		if (++count == sizeof(CRES))
			break;
	}
	fclose(fp);

	if (count < 8 || count == sizeof(CRES) ||
	    res.dialog.rc.x > 	MAX_X ||
	    res.dialog.rc.y > 	MAX_Y ||
	    res.dialog.rc.row > MAX_ROW ||
	    res.dialog.rc.col > MAX_COL ||
	    res.dialog.count > 	MAXOBJECT) {

		ermsg(NULL, "Error reading the resource\n'%s'", n);
		return 0;
	}

	q = 5 + (p->dialog.count * 4);
	w = (int*)p + q;
	z = res.dialog.rc.row * res.dialog.rc.col;
	if (res.dialog.flag & _D_RESAT)
		z |= 0x8000;
	wcunzip(&window, w, z);
	memcpy(&dl, &res.dialog, 8);
	dl.wp = window;
	dl.flag |= _D_MYBUF;
	memcpy(&dialog, &dl, 8);
	dialog.flag = (dl.flag & _D_RCSAVE);
	dialog.object = object;
	dialog.wp = rcopen(dialog.rc, dialog.flag, 0, NULL, NULL);
	if (dialog.wp == NULL) {
		return 0;
	}
	twshow(&dl);
	rcread(dialog.rc, dialog.wp);
	wcstrcpy(dialogname, rcsprc(dialog.rc), dialog.rc.col);
	if (dialogname[0] <= 0)
		strcpy(dialogname, n);
	strcpy(identifier, n);
	rcputidd(n);
	if (dl.count) {
		for (q = 0; q < dl.count; q++) {
			memcpy(&object[q], &res.object[q], 8);
			wcstrcpy(object[q].data, rcsprcrc(dialog.rc, object[q].rc), object[q].rc.col);
			if (object[q].flag & _O_CHILD)
				object[q].proc = event_child;
		}
	}
	twclose(&dl);
	dialog.flag |= _D_DOPEN;
	twshow(&dialog);

	cflag.Saved = 0;
	cflag.NewDialog = 0;
	cflag.Resource = 1;
	return 1;
}

#define ID_STATIC	(4 + 1)

void c_rename(void)
{
	remove(filename_b);
	rename(filename_c, filename_b);
	rename(filename_t, filename_c);
}

void c_printdw(FILE *fp, WCHR w)
{
	fprintf(fp, "0x%04X,", w);
	if (++line_id > 7) {
		line_id = 0;
		fputs("\n\t", fp);
	}
}

void c_print8b(FILE *fp, void *b)
{
	WCHR *p = b;

	c_printdw(fp, p[0]);
	c_printdw(fp, p[1]);
	c_printdw(fp, p[2]);
	c_printdw(fp, p[3]);
}

int c_write(FILE *fp, char *n)
{
	WCHR *w;
	DOBJ *p;
	TOBJ *o;
	char *a;
	int q;
	int flag;
	int count;
	int wc_count;

	p = &dialog;
	flag = p->flag;
	p->flag = _D_MYBUF | (p->flag & _D_RCSAVE);
	w = res.wz;
	count = 1 + p->count;
	wc_count = wczip(w, p->wp, p->rc.col * p->rc.row);

	fprintf(fp, res_begin, n);
	fprintf(fp, res_fobj, p->flag,
		count - 1, p->index, p->rc.x, p->rc.y, p->rc.col, p->rc.row);
	if (count > 1) {
		o = &object[0];
		for (q = 0; q < count - 1; q++, o++) {
			fprintf(fp, "\t{ 0x%04X, %3d, ", o->flag, o->count);
			if (o->ascii == 0)
				fprintf(fp, "  0,");
			else
				fprintf(fp, "'%c',", o->ascii);
			fprintf(fp, " {%2d,%2d,%2d,%2d} },\n",
				o->rc.x, o->rc.y, o->rc.col, o->rc.row);
		}
	}

	fprintf(fp, res_data, n);
	if (p->flag & _D_MENUS)
		fputs(res_ifdef, fp);
	fprintf(fp, "\t0x%04X,", tdmemsize(p));
	line_id = 1;
	c_print8b(fp, p);
	if (count > 1) {
		o = &object[0];
		for (q = 0; q < count - 1; q++, o++) {
			c_print8b(fp, o);
		}
	}
	if (p->flag & _D_MENUS) {
		line_id = 0;
		fputs("\n", fp);
		fputs(res_endif, fp);
	}
	for (q = 0; q < wc_count; q++)
		c_printdw(fp, w[q]);
	if (line_id != 0) {
		fputs("\n", fp);
	} else {
		if (++fp->cnt < fp->bufsize)
			fp->bp--;
		else
			fp->cnt--;
	}
	fprintf(fp, res_bsize, (2 + (count * 8) + (wc_count * 2)));
	fprintf(fp, res_robj, n, n);
	p->flag = flag;
	fprintf(fp, res_end, n);
	return wc_count;
}

int save_static(char *n)
{
	FILE *fp;
	TOBJ *o;
	char *p;
	WORD *w;
	int q,x,c;

	if ((fp = fopen(n, "a")) == NULL)
		return stderror(n, EMEROPEN, 0);

	fprintf(fp, "_DATA\t\tSEGMENT WORD PUBLIC 'DATA'\n");
	fprintf(fp, "_DATA\t\tENDS\n\t\tDGOUP\tGROUP _DATA\n");
	fprintf(fp, "_DATA\t\tSEGMENT\n");
	fprintf(fp, "\t\tPUBLIC\tDLG%s\n", identifier);
	fprintf(fp, "DLGWINDOW\tDW\t");
	tdhide(&dialog);

	x = 1;
	w = (WORD *)dialog.wp;
	c = dialog.rc.col * dialog.rc.row;
	fprintf(fp, "%04Xh", w[0]);
	for (q = 1; q < c; q++,x++) {
		if (x == 8) {
			fprintf(fp, "\n\t\tDW\t");
			x = 0;
		} else {
			fprintf(fp, ",");
		}
		fprintf(fp, "%04Xh", w[q]);
	}
	fprintf(fp, "\n");
	c = 1 + dialog.count;
	if (c > 1) {
		o = &object[0];
		fprintf(fp, "DLGOBJECT\tLABEL WORD\n");
		for (q = 0; q < c - 1; q++, o++) {
			fprintf(fp, "\t\tDW\t%lbB\n", o->flag);
			fprintf(fp, "\t\tDB\t%d\n", o->count);
			if (o->ascii == 0)
				fprintf(fp, "\t\tDB\t0\n");
			else
				fprintf(fp, "\t\tDB\t'%c'\n", o->ascii);
			fprintf(fp, "\t\tDB\t%d,%d,%d,%d\n",
				o->rc.x, o->rc.y, o->rc.col, o->rc.row);
			fprintf(fp, "\t\tDD\t00000000h\n");
			fprintf(fp, "\t\tDD\t00000000h\n");
		}
	}

	fprintf(fp, "DLG%s\tDW\t%lbB\n",
		identifier, _D_MYBUF | _D_DOPEN | (dialog.flag & (_D_DMOVE|_D_SHADE)));
	fprintf(fp, "\t\tDB\t%d\n", dialog.count);
	fprintf(fp, "\t\tDB\t%d\n", dialog.index);
	fprintf(fp, "\t\tDB\t%d,%d,%d,%d\n",
		dialog.rc.x,
		dialog.rc.y,
		dialog.rc.col,
		dialog.rc.row);
	fprintf(fp, "\t\tDD\tDGOUP:DLGWINDOW\n");
	if (c > 1)
		fprintf(fp, "\t\tDD\tDGOUP:DLGOBJECT\n");
	else
		fprintf(fp, "\t\tDD\t00000000h\n");
	fprintf(fp, "_DATA\t\tENDS\n\n\t\tEND\n");
	fclose(fp);
	return 1;
}

int c_save(char *n)
{
	FILE *	fp;
	FILE *	fp_res;
	int	x,result;
	char 	b[256];

	if ((fp = fopen(filename_t, "w")) == NULL) {
		return stderror(filename_t, EMEROPEN, 0);
	}
	if ((fp_res = fopen(filename_c, "r")) == NULL) {
		if ((fp_res = fopen(filename_c, "w+")) == NULL) {
			fclose(fp);
			return stderror(filename_c, EMEROPEN, 0);
		}
		fputs(inc_resourse, fp_res);
		fclose(fp_res);
		if ((fp_res = fopen(filename_c, "r")) == NULL) {
			fclose(fp);
			ermsg(NULL, "Error open %s", filename_c);
			return 0;
		}
	}
	result = 0;
	while ( 1 ) {
		if (fgets(b, 256, fp_res) == NULL)
			break;
		if (c_testname(b, n)) {
			result = 1;
			break;
		}
		fputs(b, fp);
	}
	x = tdhide(&dialog);
	c_write(fp, n);
	if (x != 0)
		tdshow(&dialog);
	if (result == 0) {
		fclose(fp_res);
		fclose(fp);
		c_rename();
		return 1;
	}
	if (c_getdlgend(fp_res, b)) {
		do {
			if (fgets(b, 256, fp_res) == NULL)
				break;
			fputs(b, fp);
		} while ( 1 );
	}
	fclose(fp_res);
	fclose(fp);
	c_rename();
	return 1;
}

int c_delete(char *n)
{
	FILE *fp;
	FILE *fp_res;
	int result;
	char b[256];

	if ((fp = fopen(filename_t, "w+")) == NULL)
		return 0;
	if ((fp_res = fopen(filename_c, "r")) == NULL) {
		fclose(fp);
		return 0;
	}
	result = 0;
	while ( 1 ) {
		if (fgets(b, 256, fp_res) == NULL)
			break;
		if (c_testname(b, n)) {
			result = 1;
			break;
		}
		fputs(b, fp);
	}
	if (result == 0) {
		fclose(fp_res);
		fclose(fp);
		remove(filename_t);
		return 0;
	}
	result = c_getdlgend(fp_res, b);
	if (result == 0) {
		fputs("\n/** Error delete: missing end section [", fp);
		fputs(n, fp);
		fputs("] **/\n", fp);
	} else {
		while (fgets(b, 256, fp_res) != NULL)
			fputs(b, fp);
	}
	fclose(fp_res);
	fclose(fp);
	if (result) {
		remove(filename_b);
		rename(filename_c, filename_b);
		rename(filename_t, filename_c);
	}
	return result;
}

int rsinitfile_c(char *fn)
{
	char n[WMAXPATH];

	strcpy(n, fn);
	if (filexist(n) == 1) {
		strcpy(filename_c, wlongpath(n, NULL));
	} else {
		if (fullpath(filename_c, 0) == NULL)
			return 0;
		strcpy(filename_c, wlongpath(filename_c, NULL));
		strfcat(filename_c, NULL, n);
	}
	if (strrchr(n, '.') == NULL) {
		strcat(filename_c, ext_c);
	} else {
		strcpy(strrchr(filename_c, '.'), ext_c);
	}
	return 1;
}

int rsinitfilenames(void)
{
	strcpy(filename_t, filename_c);
	strcpy(filename_b, filename_c);
	strcpy(strrchr(filename_t, '.'), ext_t);
	strcpy(strrchr(filename_b, '.'), ext_b);
	return 1;
}

int cmdsave(void)
{
	DOBJ *p;
	int result;

	if (cflag.Saved)
		return 0;
	if ((dialog.flag & _D_DOPEN) == 0)
		return 0;
	if ((p = rsopen(IDD_SAVE_DIALOG)) == NULL)
		return 0;

	strcpy(p->object[0].data, identifier);
	strcpy(p->object[1].data, filename_c);
	tdinit(p);
	result = tdevent(p);
	if (result == 0) {
		twclose(p);
		return 0;
	}
	strcpy(identifier, p->object[0].data);
	if (result == ID_STATIC) {
		twhide(p);
		save_static(p->object[1].data);
		twclose(p);
		return 0;
	}
	rcputidd(identifier);
	if (stricmp(filename_c, p->object[1].data) != 0) {
		rsinitfile_c(strfn(p->object[1].data));
		rsinitfilenames();
	}
	twclose(p);

	cflag.Saved = 1;
	result = c_save(identifier);

	return result;
}

#define CLEARSTATE	0x03FF

enum {  ID_LISTU,
	ID_LISTD = 11,
	ID_OPEND,
	ID_CANCEL,
	ID_OPENF,
	ID_MOUSEU,
	ID_MOUSED
};

#define OCOUNT	ID_OPEND

LOBJ list;

int open_initlist(void)
{
	int x;

	for (x = 0; x < OCOUNT; x++)
		tdialog->object[x].data = object[list.index + x].data;
	tdinit(tdialog);

	return 1;
}

int open_event_openfile(void)
{
	int x;
	char *p;
	TOBJ *o;
	FILE *fp;
	char b[256];

	list.index = 0;
	list.count = 0;
	list.dcount = OCOUNT;
	memset(objectname, 0, sizeof(objectname));
	if ((fp = fopen(filename_c, "r")) != NULL) {
		while (c_getresource(fp, b)) {
			if ((p = strchr(b + NAMEOFFSET, ' ')) != NULL) {
				*p = 0;
				strcpy(object[list.count++].data, b + NAMEOFFSET);
			}
		}
		fclose(fp);
		rsinitfilenames();
	}
	list.numcel = MIN(list.count, OCOUNT);
	o = tdialog->object;
	for (x = 0; x < list.numcel; x++)
		o[x].flag &= ~_O_STATE;
	for (; x < OCOUNT; x++)
		o[x].flag |= _O_STATE;
	open_initlist();
	tdinit(tdialog);
	if (list.count == 0)
		stdmsg("No object", "There is no dialogs saved in\n%s", filename_c);
	return (list.count != 0);
}

int ll_eventopen(void)
{
	if (list.celoff >= list.dcount) {
		stdmsg("TDialog", "No item selected\n");
		return _C_NORMAL;
	}
	return _C_RETURN;
}


DOBJ *open_initdlg(void)
{
	int x;
	TOBJ *o;
	DOBJ *d;

	if ((d = rsopen(IDD_DialogOpen)) == NULL)
		return NULL;

	list.numcel = 0;
	list.dcount = OCOUNT;
	list.celoff = OCOUNT;
	list.proc   = open_initlist;
	o = d->object;
	o[ID_OPEND].proc  = ll_eventopen;
	o[ID_OPENF].proc  = open_event_openfile;
	o[ID_OPENF].data  = filename_c;
	o[ID_OPENF].count = (WMAXPATH >> 4);
	return d;
}

int cmdopen(void)
{
	DOBJ *d;
	int ax;

	cmdsave();
	closedlg();
	tclrascii = 250;
	if ((d = open_initdlg()) == NULL)
		return 0;
	tdialog = d;
	open_event_openfile();
	ax = tdlevent(d, &list);
	tdclose(d);

	if (ax == 0 || list.celoff >= OCOUNT)
		return 0;
	if (c_read(object[list.index + list.celoff].data) == 0)
		return 0;
	return 1;
}

enum {	OBJ_PBUTT,
	OBJ_RBUTT,
	OBJ_CHBOX,
	OBJ_XCELL,
	OBJ_TEDIT,
	OBJ_MENUS,
	OBJ_XHTML,
	OBJ_TYPE7,
	OBJ_TYPE8,
	OBJ_TYPE9,
	OBJ_GLCMD,
	OBJ_DHELP,
	OBJ_TYPE11,
	OBJ_TYPE12,
	OBJ_TYPE13,
	OBJ_TYPE14,

	OBJ_FLAGA,
	OBJ_FLAGB,
	OBJ_FLAGC,
	OBJ_FLAGD,
	OBJ_RADIO,
	OBJ_DEXIT,
	OBJ_CURID,
	OBJ_FLAGE,
	OBJ_FLAGF,
	OBJ_EVENT,
	OBJ_CHILD,
	OBJ_STATE,

	OBJ_RCX,
	OBJ_RCY,
	OBJ_RCCOL,
	OBJ_RCROW,

	OBJ_ASCII,
	OBJ_BCOUNT,
	OBJ_INDEX,
	OBJ_NAME,
	OBJ_OK,
	OBJ_COPY,
	OBJ_DELETE,
	OBJ_CANCEL,
	OBJ_OBJCOUNT
};

DOBJ *tobj_dp;

void objxchg(TOBJ *a, TOBJ *b)
{
	void *p = a->data;

	a->data = b->data;
	b->data = p;
	memxchg(a->data, b->data, 128);
	memxchg(a, b, sizeof(TOBJ));
}

int tobj_moveup(void)
{
	int id;

	if ((id = dialog.index) == 0)
		return 0;

	objxchg(&dialog.object[id], &dialog.object[id - 1]);
	dialog.index--;
	return 1;
}

int tobj_movedown(void)
{
	int id;

	if ((id = dialog.index) == dialog.count - 1)
		return 0;

	objxchg(&dialog.object[id], &dialog.object[id + 1]);
	dialog.index++;
	return 1;
}

void init_rectdlg(TOBJ *o, RECT rc)
{
	sprintf(o[0].data, "%d", rc.x);
	sprintf(o[1].data, "%d", rc.y);
	sprintf(o[2].data, "%d", rc.col);
	sprintf(o[3].data, "%d", rc.row);
}

RECT save_rectdlg(TOBJ *o)
{
	RECT rc;
	rc.x   = atoi(o[0].data);
	rc.y   = atoi(o[1].data);
	rc.col = atoi(o[2].data);
	rc.row = atoi(o[3].data);
	return rc;
}

void init_objectdlg(TOBJ *o, TOBJ *src)
{
	int c,q;
	char *p;

	tdsetbitflag(o + 16, 12, _O_FLAGB, src->flag >> 4);
	q = (src->flag & 15);
	o[q].flag |= _O_RADIO;
	q = src->count;
	q <<= 4;
	sprintf(o[OBJ_BCOUNT].data, "%d", q);
	p = o[OBJ_ASCII].data;
	p[0] = src->ascii;
	p[1] = 0;
	init_rectdlg(&o[OBJ_RCX], src->rc);
}

int tdgetrdid(DOBJ *d)
{
	int x,s;
	TOBJ *o = d->object;

	for (x = 0; x < d->count; x++) {
		if ((o[x].flag & 15) == _O_RBUTT)
			break;
	}
	s = x;
	for (; x < d->count; x++) {
		if (o[x].flag & _O_RADIO)
			return x - s;
	}
	return 0;
}

void save_objectdlg(TOBJ *o, TOBJ *src)
{
	int q,x;
	char *p;

	src->flag = (tdgetbitflag(o + 16, 12, _O_FLAGB) << 4);
	src->flag |= tdgetrdid(tobj_dp);
	src->rc = save_rectdlg(&o[OBJ_RCX]);
	p = o[OBJ_ASCII].data;
	src->ascii = *p;
	p = o[OBJ_BCOUNT].data;
	if ((q = atoi(p)) != 0) {
		x = (q & 0x000F);
		q >>= 4;
		if (x != 0)
			q++;
	}
	src->count = q;
}

int update_object(TOBJ *o)
{
	int	at;
	int	att;
	WCHR *	sw;
	char *	cp;
	RECT	rc;
	int 	result;

	mouseoff();
	rc = o->rc;
	cp = o->data;
	sw = rcsprcrc(dialog.rc, rc);
	at = getxya(dialog.rc.x, dialog.rc.y + 1);
	att = sw->at;

	switch (O_type(o)) {
	case _O_PBUTT:
		wcputw(sw, rc.col, MKW(at, ' '));
		wcputw(sw + rc.col, 1, MKW(at, ' '));
		wcputw(sw + 81, rc.col, MKW(at, ' '));
		wcpbutt(sw, 80, rc.col, cp);
		break;
	case _O_RBUTT:
		wcputw(sw, rc.col, MKW(at, ' '));
		if (cp != NULL)
			wcputs(sw, 0, 0, cp);
		break;
	case _O_CHBOX:
		wcputw(sw, rc.col, MKW(at, ' '));
		if (cp != NULL)
			wcputs(sw, 0, 0, cp);
		break;
	case _O_TEDIT:
		for (at = 0; at < rc.row; at++, sw += 80)
			wcputw(sw, rc.col, MKW(att, tclrascii));
		break;
	case _O_MENUS:
	case _O_XCELL:
		if (cp != NULL && *cp) {
			wcputw(sw, rc.col, MKW(att, ' '));
			wcputs(sw + 1, 0, 0, cp);
		}
		break;
	case _O_XHTML:
		if (cp != NULL && *cp) {
			wcputw(sw, rc.col, MKW(att, ' '));
			wcputs(sw, 0, 0, cp);
		}
		break;
	default:
		break;
	}
	mouseon();

	return result;
}

int event_index(void)
{
	int id;

	id = atoi(tobj_dp->object[OBJ_INDEX].data);
	if (id < dialog.index) {
		while (id < dialog.index) {
			if (tobj_moveup() == 0)
				break;
		}
		if (id < dialog.index)
			ermsg("TObject", "Index out of range: %d", id);
	} else if (id > dialog.index) {
		while (id > dialog.index) {
			if (tobj_movedown() == 0)
				break;
		}
		if (id > dialog.index)
			ermsg("TObject", "Index out of range: %d", id);
	}
	sprintf(tobj_dp->object[OBJ_INDEX].data, "%d", dialog.index);
	return _C_REOPEN;
}

int cmddelete(int id)
{
	int a,q;
	RECT rc;
	TOBJ *o;

	o  = object;
	a  = getxya(dialog.rc.x, dialog.rc.y + 1);
	rc = rcaddrc(dialog.rc, o[id].rc);

	for (q = 0; q < rc.row; q++)
		scputc(rc.x, rc.y + q, a, rc.col, ' ');
	if ((o[id].flag & 15) == _O_PBUTT) {
		scputc(rc.x + rc.col, rc.y, 0, 1, ' ');
		scputc(rc.x, rc.y + 1, 0, rc.col + 2, ' ');
	}

	o[id].flag = TOBJ_DELETED;
	for (q = id + 1; q < dialog.count; q++) {
		if (o[q].flag != TOBJ_DELETED)
			objxchg(&o[q - 1], &o[q]);
	}
	if (dialog.count)
		dialog.count--;
	if (dialog.index >= dialog.count)
		dialog.index--;
	return 1;
}

int getobjid(void)
{
	int id;

	if (++dialog.count >= MAXOBJECT) {
		dialog.count--;
		return -1;
	}
	for (id = 0; id < MAXOBJECT; id++) {
		if (object[id].flag == TOBJ_DELETED)
			break;
	}
	if (id >= MAXOBJECT)
		return -1;
	return id;
}

int cmdeditobj(int id)
{
	TOBJ *o;
	TOBJ *q;
	int result;

	if (isOpen() == 0 || (tobj_dp = rsopen(IDD_TOBJECT)) == NULL)
		return 0;

	dialog.index = id;
	if (dialog.count == 0)
		cflag.NewObject = 1;

	o = tobj_dp->object;
	o[OBJ_INDEX].proc = event_index;
	o[OBJ_PBUTT].data = "TObject";

	q = &object[id];
	strcpy(o[OBJ_NAME].data, object[id].data);
	sprintf(o[OBJ_INDEX].data, "%d", dialog.index);
	init_objectdlg(o, q);
	tdinit(tobj_dp);
	twshow(tobj_dp);
	result = tdevent(tobj_dp);

	if (result == 0) {
		twclose(tobj_dp);
		return 0;
	}
	if (result == OBJ_DELETE + 1) {
		twclose(tobj_dp);
		cmddelete(dialog.index);
		return -1;
	}
	if (result == OBJ_COPY + 1) {
		if ((result = getobjid()) == -1) {
			twclose(tobj_dp);
			return 0;
		}
		dialog.index = result;
		strcpy(object[result].data, o[OBJ_NAME].data);
		save_objectdlg(o, &object[result]);
		twclose(tobj_dp);
		update_object(&object[result]);
		return cmdeditobj(result);
	}

	q = &object[dialog.index];
	strcpy(q->data, o[OBJ_NAME].data);
	save_objectdlg(o, q);
	twclose(tobj_dp);
	update_object(q);
	return 1;
}

enum {	ID_ADDCANCEL,
	ID_ADDPBUTT,
	ID_ADDRBUTT,
	ID_ADDCHBOX,
	ID_ADDTEDIT,
	ID_ADDXCELL,
	ID_ADDMENUS	};

int cmdaddobj(void)
{
	int id;
	int flag;
	int result;

	flag = _O_PBUTT;
	if ((id = getobjid()) == -1)
		return 0;
	setdefault(id);
	object[id].flag = flag;
	result = cmdeditobj(id);
	if (result == 0) {
		object[id].flag = TOBJ_DELETED;
		dialog.count--;
	}
	return result;
}

int cmdtext(void)
{
	int  x,y;
	RECT rc;
	WCHR sc[80];
	char cb[80];

	apushstl("Click on start point..");
	while (mousep());
	x = tgetevent();
	apopstl();
	if (x != MOUSECMD)
		return 0;

	rc = dialog.rc;
	x = mousex();
	y = mousey();
	if (rcxyrow(rc, x, y) == 0)
		return 0;

	rc.col -= (x - rc.x);
	rc.x = x;
	rc.y = y;
	rc.row = 1;
	cb[0] = 0;
	rcread(rc, sc);
	if (tdedit(cb, rc, 80) != ENTER) {
		rcwrite(rc, sc);
		return 0;
	}
	rcwrite(rc, sc);
	mouseoff();
	wcputs(rcsprc(rc), 80, 0, cb);
	mouseon();
	cflag.Saved = 0;
	return 1;
}

int cmdresize(void)
{
	RECT rc;
	WCHR *b;
	WCHR *p;
	WCHR *wp;
	int ex,x;
	int ey,y;
	int q,flag;

	cflag.Saved = 0;
	flag = dialog.flag;
	dialog.flag |= _D_CLEAR|_D_FOREG|_D_BACKG;
	while (mousep() == 1) {
		x = mousex();
		y = mousey();
		rc = dialog.rc;
		ex = rc.x + rc.col - 1;
		ey = rc.y + rc.row - 1;
		if (x != ex || y != ey) {
			if (x < rc.x + 10 || y < rc.y + 2) {
				mousewait(x, y, 1);
				continue;
			}
			wp = (WCHR *)rcopen(rc, dialog.flag & _D_SHADE, 0, 0, wp);
			twclose(&dialog);//closedlg();
			if (x != ex)
				dialog.rc.col = x - dialog.rc.x + 1;
			if (y != ey)
				dialog.rc.row = y - dialog.rc.y + 1;
			opendlg();
			p  = rcsprc(dialog.rc) + 80;
			b  = wp + rc.col;
			rc.y++;
			rc.row = MIN(dialog.rc.row, rc.row) - 1;
			ex = MIN(dialog.rc.col, rc.col) * 2;
			__mshide();
			for (q = 0; q < rc.row; q++, p += 80, b += rc.col)
				memcpy(p, b, ex);
			__msshow();
			rcclose(rc, _D_DOPEN, wp);
			mousewait(x, y, 1);
		}
	}
	dialog.flag = flag;
	return 0;
}

int cmdmoveobj(int i)
{
	int ex,ey,flag,end;
	RECT rc;
	WCHR *wp;

	cflag.Saved = 0;
	if (dialog.rc.y + dialog.rc.row < 25)
		apushstl("Use Mouse or [Left/Right/Up/Down] to move object..");
	flag = _D_DMOVE|_D_CLEAR|_D_COLOR;
	rc = rcaddrc(dialog.rc, object[i].rc);
	if ((object[i].flag & 15) == _O_PBUTT) {
		rc.row++;
		rc.col++;
	}
	wp = (WCHR *)rcopen(rc, flag, getxya(dialog.rc.x, dialog.rc.y + 1), 0, 0);
	flag |= (_D_DOPEN|_D_ONSCR);
	end  = 0;

	while (end == 0) {
		gotoxy(rc.x, rc.y);
		if (mousep() == 1) {
			rc = rcmsmove(rc, wp, flag);
			break;
		} else switch (getkey()) {
		case UP:
			rc = rcmove(rc, wp, flag, rc.x, rc.y - 1);
			break;
		case DOWN:
			rc = rcmove(rc, wp, flag, rc.x, rc.y + 1);
			break;
		case LEFT:
			rc = rcmove(rc, wp, flag, rc.x - 1, rc.y);
			break;
		case RIGHT:
			rc = rcmove(rc, wp, flag, rc.x + 1, rc.y);
			break;
		case ENTER:
			end = 1;
			break;
		case ESC:
			end = ESC;
			break;
		}
	}
	if (dialog.rc.y + dialog.rc.row < 25)
		apopstl();
	if (end == ESC ||
	    rc.x < dialog.rc.x ||
	    rc.y < dialog.rc.y ||
	    rc.x + rc.col > dialog.rc.x + dialog.rc.col ||
	    rc.y + rc.row > dialog.rc.y + dialog.rc.row) {
		rcxchg(rc, wp);
		rc.x = object[i].rc.x + dialog.rc.x;
		rc.y = object[i].rc.y + dialog.rc.y;
		rcclose(rc, _D_DOPEN|_D_ONSCR, wp);
		return 0;
	}
	object[i].rc.x = rc.x - dialog.rc.x;
	object[i].rc.y = rc.y - dialog.rc.y;
	rcclose(rc, _D_DOPEN, wp);
	cflag.Saved = 0;
	return 1;
}

int cmdmovetext(void)
{
	int x,y,flag,end;
	RECT rc;
	WCHR *wp;

	cflag.Saved = 0;
	if (msgetrc(dialog.rc, &rc) == 0)
		return msinvalid();

	flag = _D_DMOVE|_D_CLEAR|_D_COLOR;
	wp = (WCHR *)rcopen(rc, flag, getxya(dialog.rc.x, dialog.rc.y + 1), 0, 0);
	flag |= (_D_DOPEN|_D_ONSCR);
	end  = 0;

	if (dialog.rc.y + dialog.rc.row < 25)
		apushstl("Mouseclick on first line, or [Left/Right/Up/Down] to move object..");
	while (mousep() == 1);
	while (end == 0) {
		gotoxy(rc.x, rc.y);
		if (mousep() == 1) {
			rc = rcmsmove(rc, wp, flag);
			break;
		} else switch (getkey()) {
		case UP:
			rc = rcmove(rc, wp, flag, rc.x, rc.y - 1);
			break;
		case DOWN:
			rc = rcmove(rc, wp, flag, rc.x, rc.y + 1);
			break;
		case LEFT:
			rc = rcmove(rc, wp, flag, rc.x - 1, rc.y);
			break;
		case RIGHT:
			rc = rcmove(rc, wp, flag, rc.x + 1, rc.y);
			break;
		case ENTER:
			end = 1;
			break;
		case ESC:
			end = ESC;
			break;
		}
	}
	if (dialog.rc.y + dialog.rc.row < 25)
		apopstl();
	rcclose(rc, _D_DOPEN, wp);
	cflag.Saved = 0;
	return 1;
}

int cmdframe(void)
{
	RECT rc;
	int type;

	if ((type = rsmodal(IDD_Frametype)) == 0)
		return msinvalid();

	if (msgetrc(dialog.rc, &rc) == 0)
		return msinvalid();
	__mshide();
	type--;
	rcframe(rc, getxyp(0, 0), 80, type << 8);
	__msshow();
	cflag.Saved = 0;
	return 1;
}

int cmdfground(void)
{
	int x,y;
	int color;
	RECT rc;

	x = rsmodal(IDD_FOREGROUND);
	if (x == 0)
		return 0;
	rc = dialog.rc;
	color = x - 1;
	if (rc.y + rc.row < 25)
		apushstl("Click to set foreground..");
	while (mousep());
	while ( 1 ) {
		if (tgetevent() != MOUSECMD)
			break;
		x = mousex();
		y = mousey();
		if (mousep() == 1 && rcxyrow(rc, x, y)) {
			scputfg(x, y, 1, color);
		} else if (mousep() == 2) {
			color = getxya(x, y);
		}
		mousewait(x, y, 1);
	}
	if (rc.y + rc.row < 25)
		apopstl();
	cflag.Saved = 0;
	return 1;
}

int cmdclearfg(void)
{
	int x,y,a,n;
	MSXY ms;

	if (isOpen() == 0)
		return 0;
	if (isVisible() == 0)
		twshow(&dialog);
	if (msget(&ms, "Click on foreground color to change..") == 0)
		return 0;
	a = (ms.at & 0x0F);
	if (msget(&ms, "Click to get new foreground color to set..") == 0)
		return 0;
	n = (ms.at & 0x0F);
	twhide(&dialog);
	for (x = 0; x < dialog.rc.col * dialog.rc.row; x++) {
		y = dialog.wp[x].at;
		if ((y & 0x0F) == a) {
			dialog.wp[x].at &= 0xF0;
			dialog.wp[x].at |= n;
		}
	}
	twshow(&dialog);
	cflag.Saved = 0;
	return 1;
}

int child_editattrib(void)
{
	editattrib();
	return _C_NORMAL;
}

int child_editpal(void)
{
	editpal();
	return _C_NORMAL;
}

int cmdcolor(void)
{
	DOBJ *d;
	int result;

	if ((d = rsopen(IDD_Color)) == NULL)
		return 0;
	tdinit(d);
	d->object[ID_EDITATTRIB].proc = child_editattrib;
	d->object[ID_EDITPAL].proc = child_editpal;

	result = rsevent(IDD_Color, d);
	tdclose(d);

	return result;
}

int cmdcenter(void)
{
	int x = 40 - (dialog.rc.col / 2);
	int y = 12 - (dialog.rc.row / 2);

	twhide(&dialog);
	dialog.rc.x = x;
	dialog.rc.y = y;
	twshow(&dialog);
	cflag.Saved = 0;

	return 1;
}

int cmdbground(void)
{
	int x,y;
	int color;
	RECT rc;

	x = rsmodal(IDD_BACKGROUND);
	if (x == 0)
		return 0;
	rc = dialog.rc;
	color = (x - 1) << 4;
	if (rc.y + rc.row < 25)
		apushstl("Left-Click to set/Right-Click to get background..");
	while ( 1 ) {
		if (tgetevent() != MOUSECMD)
			break;
		x = mousex();
		y = mousey();
		if (mousep() == 1 && rcxyrow(rc, x, y)) {
			scputbg(x, y, 1, color);
		} else if (mousep() == 2) {
			color = getxya(x, y);
		}
		mousewait(x, y, 1);
	}
	if (rc.y + rc.row < 25)
		apopstl();
	cflag.Saved = 0;
	return 1;
}

int cmdclearbg(void)
{
	int x,a,n;
	MSXY ms;

	if (isOpen() == 0)
		return 0;
	if (isVisible() == 0)
		twshow(&dialog);
	if (msget(&ms, "Click on color to change..") == 0)
		return 0;
	if (rcxyrow(dialog.rc, ms.x, ms.y) == 0)
		return 0;
	a = (ms.at & 0xF0);
	if (msget(&ms, "Click to get new color to set..") == 0)
		return 0;
	n = (ms.at & 0xF0);
	twhide(&dialog);
	for (x = 0; x < dialog.rc.col * dialog.rc.row; x++) {
		if ((dialog.wp[x].at & 0xF0) == a) {
			dialog.wp[x].at &= 0x0F;
			dialog.wp[x].at |= n;
		}
	}
	twshow(&dialog);
	cflag.Saved = 0;
	return 1;
}

int cmdclearat(void)
{
	int b,f;
	MSXY ms;

	if (isOpen() == 0)
		return 0;
	if (isVisible() == 0)
		twshow(&dialog);

	apushstl("Select background color of attrib..");
	if ((b = rsmodal(IDD_BACKGROUND)) == 0)
		return 0;
	b = (b - 1) << 4;
	apushstl("Select foreground color of attrib..");
	if ((f = rsmodal(IDD_FOREGROUND)) == 0)
		return 0;
	apopstl();
	b |= (f - 1);
	twhide(&dialog);
	wcputa(dialog.wp, dialog.rc.col * dialog.rc.row, b);
	twshow(&dialog);
	cflag.Saved = 0;
	return 1;
}

int cmdascii(void)
{
	int ch;
	int q,z;
	int x,y;
	RECT rc;

	apushstl("Left-Click to set/Right-Click to get ascii..");
	while (mousep());
	q = tgetevent();
	if (q != MOUSECMD) {
		apopstl();
		return 0;
	}
	x = mousex();
	y = mousey();
	ch = getxyc(x, y);
	rc = dialog.rc;
	if (rc.y + rc.row > 23)
		apopstl();
	while ( 1 ) {
		if (tgetevent() != MOUSECMD)
			break;
		x = mousex();
		y = mousey();
		if (mousep() == 1 && rcxyrow(rc, x, y)) {
			scputc(x, y, 0, 1, ch);
		} else if (mousep() == 2) {
			ch = getxyc(x, y);
		}
		mousewait(x, y, 1);
	}
	if (rc.y + rc.row < 24)
		apopstl();
	cflag.Saved = 0;
	return 1;
}

int cmdtitle(void)
{
	int result;

	if ((result = tgetline("Dialogname title", dialogname, 32, 128)) != 0) {
		if (dialog.flag & _D_DOPEN) {
			RECT rc = dialog.rc;
			result = tdhide(&dialog);
			wctitle(dialog.wp, rc.col, dialogname);
			if (result)
				tdshow(&dialog);
		}
	}
	return _C_NORMAL;
}

enum {	ID_PBUTT,
	ID_RBUTT,
	ID_CHBOX,
	ID_XCELL,
	ID_TEDIT,
	ID_MENUS,
	ID_XHTML,
	ID_GLCMD,

	ID_RADIO,
	ID_DEXIT,
	ID_FLAGB,
	ID_CURID,
	ID_EXTRN,
	ID_EVENT,
	ID_CHILD,
	ID_STATE,

	ID_NAME,
	ID_BCOUNT,
	ID_ASCII,
	ID_RCX,
	ID_RCY,
	ID_RCCOL,
	ID_RCROW,

	ID_OBJCOUNT
};

enum DObject {
	ID_SETAT = ID_OBJCOUNT,
	ID_SETBG,
	ID_SETFG,
	ID_TITLE,
	ID_OK,
	ID_DCANCEL,
	ID_COUNT,
};

int cmdeditdlg(void)
{
	TOBJ *o;
	DOBJ *d;
	int result;

	if ((d = rsopen(IDD_DOBJECT)) == NULL)
		return 0;
	o = d->object;
	strcpy(o[ID_NAME].data, identifier);
	tdsetbitflag(o, 16, _O_FLAGB, dialog.flag);
	sprintf(o[ID_BCOUNT].data, "%d", dialog.count);
	sprintf(o[ID_ASCII ].data, "%d", dialog.index);
	init_rectdlg(&o[ID_RCX], dialog.rc);
	tdinit(d);
	if ((result = tdevent(d)) == 0) {
		twclose(d);
		return 0;
	}
	strcpy(identifier, o[ID_NAME].data);
	dialog.flag  = tdgetbitflag(o, 16, _O_FLAGB);
	dialog.rc    = save_rectdlg(&o[ID_RCX]);
	dialog.index = atoi(o[ID_ASCII ].data);
	twclose(d);
	rcputidd(identifier);
	if (dialog.index >= dialog.count)
		dialog.index = 0;

	switch (result - 1) {
	case ID_SETAT: cmdclearat(); break;
	case ID_SETBG: cmdclearbg(); break;
	case ID_SETFG: cmdclearfg(); break;
	case ID_TITLE: cmdtitle();
		break;
	}
	cflag.Saved = 0;
	return 1;
}

int cmdnewdlg(void)
{
	cmdsave();
	closedlg();
	cflag.Saved = 0;
	cflag.NewDialog = 1;
	cflag.Resource = 0;
	strcpy(identifier, "Dialogname");
	strcpy(dialogname, identifier);
	if (cmdeditdlg() == 0)
		return 0;
	opendlg();
	return 1;
}

int cmdhelp(void)
{
	return rsmodal(IDD_HELP);
}

int cmdexit(void)
{
	if (rsmodal(IDD_EXIT))
	       _end = 1;
	return _end;
}

int update_clock(void)
{
	static int  _mousex = 0;
	static int  _mousey = 0;
	static int  _ocount = 0;
	static int  _oindex = 0;

	if (dialog.rc.y	== 0)
		return 0;

	if (_oindex != dialog.index) {
		_oindex = dialog.index;
		scputf(58, 0, 0, 0, "%02d", _oindex);
	}
	if (_ocount != dialog.count) {
		_ocount = dialog.count;
		scputf(61, 0, 0, 0, "%02d", _ocount);
	}
	if (_mousex != mousex()) {
		_mousex = mousex();
		scputf(65, 0, 0, 0, "%02d", _mousex);
	}
	if (_mousey != mousey()) {
		_mousey = mousey();
		scputf(68, 0, 0, 0, "%02d", _mousey);
	}
	tupdtime();
	return 0;
}

int cmdmenu(void)
{
	int x,y;
	DOBJ *p;

	if ((dialog.flag & _D_ONSCR) == 0)
		return 0;
	if ((p = rsopen(IDD_QUIK_MENU)) == NULL)
		return 0;

	if (mousep() == 2) {
		x = mousex();
		if (x < 60)
			p->rc.x = x;
		else
			p->rc.x = 59;
		p->rc.y = mousey();
	}
	twshow(p);
	while (mousep());

	switch (tdmodal(p)) {
	case 8:	tdinit(&dialog);
		return tdevent(&dialog);
	case 7: return cmdcenter();
	case 6: return cmdbground();
	case 5: return cmdfground();
	case 4: return cmdframe();
	case 3: return cmdascii();
	case 2: return cmdtext();
	case 1: return cmdaddobj();
	default:
		return 0;
	}
}

int mousevent(void)
{
	RECT rc;
	int  x,y,q;

	x = mousex();
	y = mousey();

	if (y == 24) {
		if (isVisible() && dialog.rc.y + dialog.rc.row >= y) {
		} else {
			for (q = 0; q < STLCOUNT; q++) {
				if (rcxyrow(st_line[q].rc, x, y)) {
					mousewait(x, y, 1);
					return st_line[q].key;
				}
			}
			mousewait(x, y, 1);
			return 0;
		}
	}
	if (isVisible() == 0)
		return 0;

	if (rcxyrow(dialog.rc, x, y) == 1) {
		twmove(&dialog);
		cflag.Saved = 0;
		return 0;
	}

	if (y == dialog.rc.y + dialog.rc.row - 1
		&& x == dialog.rc.x + dialog.rc.col - 1) {
		cmdresize();
		return 0;
	}

	for (q = 0; q < dialog.count; q++) {
		rc = rcaddrc(dialog.rc, object[q].rc);
		if (object[q].flag != TOBJ_DELETED) {
			if (rcxyrow(rc, x, y) == 1) {
				if (mousep() == 1)
					cmdmoveobj(q);
				else
					cmdeditobj(q);
				return 1;
			}
		}
	}
	if (mousep() == 2)
		cmdmenu();

	return 0;
}

int rc_event(int event)
{
	RECT rc;
	int result;

	switch (event) {
	case F1: return cmdhelp();
	case F2: return cmdnewdlg();
	case F3: return cmdopen();
	case F4: return cmdsave();
	case F5: return cmdmenu();
	case F6: return cmdeditdlg();
	case F7: return cmdcolor();
	case CTRLF7:
		result = twhide(&dialog);
		mouseoff();
		if (console & 0x80) {
			cflag.Color = 0;
			rc_initscreen();
		} else {
			cflag.Color = 1;
			rc_initscreen();
		}
		mouseon();
		if (result)
			twshow(&dialog);
		break;
	case ESC: return cmdexit();
	case ALTX:
		_end = 1;
		return ALTX;
	}
	if (isVisible() == 0)
		return 0;
	if (event == CTRLF8)
		cmdmovetext();
	if (dialog.count == 0)
		return 0;

	switch (event) {
	case UP:
		if (dialog.index)
			dialog.index--;
		else if (dialog.count > 1)
			dialog.index = dialog.count - 1;
		break;
	case TAB:
	case DOWN:
		if (dialog.index == dialog.count - 1)
			dialog.index = 0;
		else if (dialog.count > 1)
			dialog.index++;
		break;
	case ENTER:
		cmdeditobj(dialog.index);
		break;
	case F8:
		cmdmoveobj(dialog.index);
		break;
	case F9:  cmdtext();	break;
	case F10: cmdcenter();	break;
	case F11: cmdfground();	break;
	case F12: cmdbground();	break;
	case CTRLF4:
		cmdascii();
		break;
	case CTRLF5:
		cmdframe();
		break;
	case CTRLF6:
		tdinit(&dialog);
		tdevent(&dialog);
		break;
	}

	return 0;
}

int modal(void)
{
	COBJ cu;
	RECT rc;
	int result;

	cu.flag = 0x0B01;
	while (_end == 0) {
		if (isVisible() && dialog.count) {
			rc = rcaddrc(dialog.rc, object[dialog.index].rc);
			cu.xpos = rc.x;
			cu.ypos = rc.y;
			setcursor(cu);
		} else {
			cursoroff();
		}
		while (0 == (result = getkey())) {
			tupdate();
			if (mousep()) {
				result = mousevent();
				break;
			}
		}
		if (rc_event(result) == ALTX)
			break;
	}

	closedlg();
	return 0;
}

int rsreadconfig(void)
{
	int h;
	RCFG cl;
	char fn[WMAXPATH];

	strcpy(fn, _argv[0]);
	strcpy(strrchr(fn, '.'), ".cfg");
	if ((h = osopen(fn, _A_NORMAL, M_RDONLY, A_OPEN)) == -1)
		return 0;
#ifndef DEBUG
	if (osread(h, &cl, sizeof(RCFG)) == sizeof(RCFG) ||
	    cl.version == RCVERSION)
		memcpy(&config, &cl, sizeof(RCFG));
#endif
	close(h);
	return 1;
}

int rswriteconfig(void)
{
	int h;
	char fn[WMAXPATH];

	strcpy(fn, _argv[0]);
	strcpy(strrchr(fn, '.'), ".cfg");
	if ((h = osopen(fn, _A_NORMAL, M_WRONLY, A_CREATE|A_TRUNC)) == -1)
		return 0;
	if (cflag.Color)
		memcpy(&config.dlg, at_background, sizeof(COLOR));
	else
		memcpy(&config.dos, at_background, sizeof(COLOR));
	oswrite(h, &config, sizeof(RCFG));
	close(h);
	return 1;
}

int rcedit(/*char *dialog_name*/void)
{
	int result;

	console |= CON_UTIME;
	if (cflag.NewDialog == 1) {
		if (tgetline("Resource filename", filename_b, 30, 256) == 0)
			return 0;
		if (rsinitfile_c(filename_b) == 0)
			return 0;
	} else {
		if (rsinitfile_c(filename_c) == 0)
			return 0;
	}
	rsinitfilenames();
	rsreadconfig();

	rcdefault();
	rc_loadattrib();
	aopen(&api);
	rc_putscreen();
	rc_initscreen();
	scpath(17, 0, 32, filename_c);
	tupdate = update_clock;
	if (cflag.NewDialog) {
		cmdnewdlg();
	} else {
		if (cmdopen() == 0)
			closedlg();
	}

	result = modal();
	rswriteconfig();
	aclose(&api);
	resetpal();
	return (result != 0);
}

int main(int argc, char **argv)
{
	int c;
	char arg_dialog[MAXNAMELEN];

	printf(copyright);
	if (argc == 1) {
		printf(cp_usage);
		return 1;
	}

//	cflag.Color = 1;
	arg_dialog[0] = 0;
	filename_c[0] = 0;
	for (c = 1; c < argc; c++) {
		switch (argv[c][0]) {
		case '?':
			puts(cp_usage);
			return 0;
		case '/':
		case '-':
			switch (argv[c][1] | 0x20) {
			case 'n':
				cflag.NewDialog = 1;
				break;
			case 'o':
				cflag.OpenDialog = 1;
				strcpy(arg_dialog, &argv[c][2]);
				break;
			case 'g':
				cflag.Color = 1;
				break;
			default:
				puts(cp_usage);
				return 0;
			}
			break;
		default:
			strcpy(filename_c, argv[c]);
			break;
		}
	}
	if (cflag.NewDialog == 0 && filename_c[0] == 0) {
		cflag.NewDialog = 1;
	}
	return rcedit(/*arg_dialog*/);
}

/******** Resource begin EVENT *
	{ 0x041C,   5,   4, {10, 8,68,10} },
	{ 0x0000,   0, 'U', {48, 2,13, 1} },
	{ 0x0000,   0, 'R', {48, 4,13, 1} },
	{ 0x0000,   0, 'E', {48, 6,13, 1} },
	{ 0x0000,   0, 'D', {48, 8,13, 1} },
	{ 0x0A00,   0, 'C', { 5, 8,13, 1} },
********* Resource data  *******************/
static int EVENT_RC[] = {
	0x0606,0x041C,0x0405,0x080A,0x0A44,0x0000,0x5500,0x0230,
	0x010D,0x0000,0x5200,0x0430,0x010D,0x0000,0x4500,0x0630,
	0x010D,0x0000,0x4400,0x0830,0x010D,0x0A00,0x4300,0x0805,
	0x010D,0x44F0,0xF050,0x2A74,0x05F0,0x5C50,0x07F0,0x2050,
	0x37F0,0xF02A,0x200D,0x36F0,0xF02A,0x5005,0xF05C,0x5007,
	0xF020,0x2A37,0x0DF0,0xF020,0x2A36,0x05F0,0x5C50,0x07F0,
	0x2050,0x37F0,0xF02A,0x200D,0x0BF0,0xF02A,0x5005,0xF05C,
	0x5007,0xF020,0x2A1D,0x05F0,0x5C50,0x07F0,0x2050,0x0CF0,
	0xF02A,0x200D,0x1EF0,0xF02A,0x200D,0x06F0,0xF02A,0x201F,
	0x7645,0x6E65,0xF074,0x2096,0x2E31,0x5520,0xF070,0x2006,
	0xF0DC,0x200C,0x7845,0x6574,0x6E72,0x4520,0x6576,0x746E,
	0x4820,0x6E61,0x6C64,0x7265,0x17F0,0xF020,0xDF0D,0x38F0,
	0x3220,0x202E,0x6552,0x7574,0x6E72,0x2020,0xF0DC,0x2037,
	0x0DF0,0xF0DF,0x2038,0x2E33,0x4520,0x6373,0x7061,0x2065,
	0xDC20,0x37F0,0xF020,0xDF0D,0x0DF0,0x3020,0x202E,0x6143,
	0x636E,0x6C65,0x2020,0xF0DC,0x201F,0x2E34,0x4420,0x776F,
	0xF06E,0x2004,0xF0DC,0x200C,0x0DF0,0xF0DF,0x201E,0x0DF0,
	0xF0DF,0x2006,0x2A2A,
}; //	278 byte
ROBJ *IDD_EVENT = (ROBJ *)&EVENT_RC;
/******** Resource end   EVENT */

/******** Resource begin QUIK_MENU *
	{ 0x4C1C,   8,   0, {31, 7,18,10} },
	{ 0x0006,   0, 'O', { 1, 1,16, 1} },
	{ 0x0006,   0, 'T', { 1, 2,16, 1} },
	{ 0x0006,   0, 'A', { 1, 3,16, 1} },
	{ 0x0006,   0, 'R', { 1, 4,16, 1} },
	{ 0x0006,   0, 'F', { 1, 5,16, 1} },
	{ 0x0006,   0, 'B', { 1, 6,16, 1} },
	{ 0x0006,   0, 'M', { 1, 7,16, 1} },
	{ 0x0006,   0, 'E', { 1, 8,16, 1} },
********* Resource data  *******************/
static int QUIK_MENU_RC[] = {
	0x021C,0x4C1C,0x0008,0x071F,0x0A12,0x0006,0x4F00,0x0101,
	0x0110,0x0006,0x5400,0x0201,0x0110,0x0006,0x4100,0x0301,
	0x0110,0x0006,0x5200,0x0401,0x0110,0x0006,0x4600,0x0501,
	0x0110,0x0006,0x4200,0x0601,0x0110,0x0006,0x4D00,0x0701,
	0x0110,0x0006,0x4500,0x0801,0x0110,0x14F0,0x7470,0x11F0,
	0x7470,0x11F0,0x7470,0x12F0,0x7470,0x10F0,0x7470,0x11F0,
	0x7470,0x11F0,0x7470,0x11F0,0x7470,0x21F0,0xDA70,0x10F0,
	0xBFC4,0x20B3,0x624F,0x656A,0x7463,0x09F0,0xB320,0x20B3,
	0x6554,0x7478,0x0BF0,0xB320,0x20B3,0x7341,0x6963,0xF069,
	0x200A,0xB3B3,0x4620,0x6172,0x656D,0x0AF0,0xB320,0x20B3,
	0x6F46,0x6572,0x7267,0x756F,0x646E,0x05F0,0xB320,0x20B3,
	0x6142,0x6B63,0x7267,0x756F,0x646E,0x05F0,0xB320,0x20B3,
	0x6F4D,0x6576,0x7420,0x206F,0x6563,0x746E,0x7265,0xB320,
	0x20B3,0x7845,0x6365,0x7475,0x2065,0x6964,0x6C61,0x676F,
	0xB320,0xF0C0,0xC410,0x70D9,
}; //	232 byte
ROBJ *IDD_QUIK_MENU = (ROBJ *)&QUIK_MENU_RC;
/******** Resource end   QUIK_MENU */

/******** Resource begin EXIT *
	{ 0x001C,   2,   0, {26, 7,28, 6} },
	{ 0x0000,   0, 'O', { 4, 4, 6, 1} },
	{ 0x0200,   0, 'C', {14, 4,10, 1} },
********* Resource data  *******************/
static int EXIT_RC[] = {
	0x01A6,0x001C,0x0002,0x071A,0x061C,0x0000,0x4F00,0x0404,
	0x0106,0x0200,0x4300,0x040E,0x010A,0x1CF0,0xF070,0x4758,
	0x7070,0xF07F,0x7003,0xF040,0x4703,0x7070,0xF07F,0x7007,
	0xF040,0x4708,0x06F0,0xF040,0x4704,0x0AF0,0xF040,0x4703,
	0x0CF0,0x4520,0x6978,0xF074,0x202F,0x7845,0x7469,0x7020,
	0x6F72,0x7267,0x6D61,0x3F20,0x29F0,0x4F20,0x206B,0xDC20,
	0x05F0,0x4320,0x6E61,0x6563,0x206C,0xDC20,0x08F0,0xF020,
	0xDF06,0x04F0,0xF020,0xDF0A,0x03F0,0x4720,
}; //	124 byte
ROBJ *IDD_EXIT = (ROBJ *)&EXIT_RC;
/******** Resource end   EXIT */

/******** Resource begin CHILD *
	{ 0x081C,   5,   3, {10, 8,68,10} },
	{ 0x0A00,   0,   0, { 5, 8,13, 1} },
	{ 0x0000,   0, 'R', {48, 4,13, 1} },
	{ 0x0000,   0, 'E', {48, 6,13, 1} },
	{ 0x0000,   0, 'O', {48, 8,13, 1} },
	{ 0x0A00,   0,   0, { 5, 8,13, 1} },
********* Resource data  *******************/
static int CHILD_RC[] = {
	0x0606,0x081C,0x0305,0x080A,0x0A44,0x0A00,0x0000,0x0805,
	0x010D,0x0000,0x5200,0x0430,0x010D,0x0000,0x4500,0x0630,
	0x010D,0x0000,0x4F00,0x0830,0x010D,0x0A00,0x0000,0x0805,
	0x010D,0x44F0,0xF050,0x2A74,0x05F0,0x5C50,0x07F0,0x2050,
	0x37F0,0xF02A,0x200D,0x36F0,0xF02A,0x5005,0xF05C,0x5007,
	0xF020,0x2A37,0x0DF0,0xF020,0x2A36,0x05F0,0x5C50,0x07F0,
	0x2050,0x37F0,0xF02A,0x200D,0x0BF0,0xF02A,0x5005,0xF05C,
	0x5007,0xF020,0x2A1D,0x07F0,0x5C50,0x05F0,0x2050,0x0CF0,
	0xF02A,0x200D,0x1EF0,0xF02A,0x200D,0x06F0,0xF02A,0x201F,
	0x6843,0x6C69,0xF064,0x2069,0x435F,0x4E5F,0x524F,0x414D,
	0x204C,0x4320,0x6E6F,0x6974,0x756E,0x2065,0x6944,0x6C61,
	0x676F,0x13F0,0x3120,0x202E,0x6F4E,0x6D72,0x6C61,0x2020,
	0xF0DC,0x200B,0x435F,0x525F,0x5445,0x5255,0x204E,0x4320,
	0x6F6C,0x6573,0x6420,0x6169,0x6F6C,0x2067,0x6E61,0x2064,
	0x6572,0x7574,0x6E72,0x3120,0x08F0,0xF020,0xDF0D,0x0BF0,
	0x5F20,0x5F43,0x5345,0x4143,0x4550,0x2020,0x6C43,0x736F,
	0x2065,0x6964,0x6C61,0x676F,0x6120,0x646E,0x7220,0x7465,
	0x7275,0x206E,0xF030,0x2009,0x2E32,0x5220,0x7465,0x7275,
	0x206E,0xDC20,0x0BF0,0x5F20,0x5F43,0x4552,0x504F,0x4E45,
	0x2020,0x6552,0x7277,0x7469,0x2065,0x6964,0x6C61,0x676F,
	0x6120,0x646E,0x6320,0x6E6F,0x6974,0x756E,0xF065,0x2006,
	0x0DF0,0xF0DF,0x2038,0x2E33,0x4520,0x6373,0x7061,0x2065,
	0xDC20,0x37F0,0xF020,0xDF0D,0x0DF0,0x3020,0x202E,0x6143,
	0x636E,0x6C65,0x2020,0xF0DC,0x201F,0x2E34,0x5220,0x6F65,
	0x6570,0x206E,0xDC20,0x0CF0,0xF020,0xDF0D,0x1EF0,0xF020,
	0xDF0D,0x06F0,0x2A20,
}; //	406 byte
ROBJ *IDD_CHILD = (ROBJ *)&CHILD_RC;
/******** Resource end   CHILD */

/******** Resource begin SAVE_DIALOG *
	{ 0x081C,   5,   2, {16, 9,49, 9} },
	{ 0x0004,  16, 'I', { 3, 3,42, 1} },
	{ 0x0004,  16, 'F', { 3, 5,42, 1} },
	{ 0x0000,   0, 'S', { 3, 7, 8, 1} },
	{ 0x0A00,   0, 'C', {14, 7,10, 1} },
	{ 0x0000,   0, 'A', {37, 7, 7, 1} },
********* Resource data  *******************/
static int SAVE_DIALOG_RC[] = {
	0x0613,0x081C,0x0205,0x0910,0x0931,0x0004,0x4910,0x0303,
	0x012A,0x0004,0x4610,0x0503,0x012A,0x0000,0x5300,0x0703,
	0x0108,0x0A00,0x4300,0x070E,0x010A,0x0000,0x4100,0x0725,
	0x0107,0x31F0,0xF070,0x6A34,0xF06E,0x6A30,0x2AF0,0xF009,
	0x6A07,0xF06E,0x6A30,0x2AF0,0xF009,0x6A38,0x7070,0xF07F,
	0x7005,0x6A60,0x706A,0x7F70,0x07F0,0x6070,0x0CF0,0x706A,
	0x7F70,0x04F0,0x6070,0x08F0,0xF06A,0x6008,0x03F0,0xF06A,
	0x600A,0x0DF0,0xF06A,0x6007,0x04F0,0xF06A,0x2013,0x6153,
	0x6576,0x4420,0x6169,0x6F6C,0xF067,0x2047,0x4449,0x2044,
	0x614E,0x656D,0x5AF0,0x4620,0x6C69,0x6E65,0x6D61,0xF065,
	0x208D,0x6153,0x6576,0x2020,0xF0DC,0x2004,0x6143,0x636E,
	0x6C65,0x2020,0xF0DC,0x200E,0x5341,0x204D,0xDC20,0x08F0,
	0xF020,0xDF08,0x03F0,0xF020,0xDF0A,0x0DF0,0xF020,0xDF07,
	0x04F0,0x0920,
}; //	212 byte
ROBJ *IDD_SAVE_DIALOG = (ROBJ *)&SAVE_DIALOG_RC;
/******** Resource end   SAVE_DIALOG */

/******** Resource begin HELP *
	{ 0x041C,   1,   0, { 8, 4,65,17} },
	{ 0x0000,   0,   0, {29,15, 6, 1} },
********* Resource data  *******************/
static int HELP_RC[] = {
	0x0923,0x041C,0x0001,0x0408,0x1141,0x0000,0x0000,0x0F1D,
	0x0106,0x41F0,0xF050,0x7044,0x7F7F,0x15F0,0xF070,0x7F03,
	0x27F0,0x7F70,0xF07F,0x7015,0x03F0,0xF07F,0x7027,0x7F7F,
	0x3FF0,0x7F70,0xF07F,0x7015,0x07F0,0xF07F,0x7023,0x7F7F,
	0x15F0,0xF070,0x7F07,0x23F0,0x7F70,0xF07F,0x7015,0x07F0,
	0xF07F,0x7023,0x7F7F,0x15F0,0xF070,0x7F07,0x23F0,0x7F70,
	0xF07F,0x7015,0x07F0,0xF07F,0x7023,0x7F7F,0x15F0,0xF070,
	0x7F03,0x27F0,0xF070,0x7F03,0x14F0,0xF070,0x7F05,0x02F1,
	0x6070,0x6F60,0x03F0,0xF060,0x705F,0x1EF0,0x4820,0x6C65,
	0xF070,0x2063,0x3146,0x04F0,0x4820,0x6C65,0xF070,0x200D,
	0x3146,0xF031,0x2007,0x6553,0x2074,0x6F66,0x6572,0x7267,
	0x756F,0x646E,0x6320,0x6C6F,0x726F,0x0CF0,0x4620,0xF032,
	0x2004,0x654E,0xF077,0x200E,0x3146,0xF032,0x2007,0x6553,
	0x2074,0x6162,0x6B63,0x7267,0x756F,0x646E,0x6320,0x6C6F,
	0x726F,0x0CF0,0x4620,0xF033,0x2004,0x704F,0x6E65,0x37F0,
	0x4620,0xF034,0x2004,0x6153,0x6576,0x0DF0,0x4320,0x7274,
	0x2B6C,0x3446,0x03F0,0x4920,0x736E,0x7265,0x2074,0x7361,
	0x6963,0xF069,0x2014,0x3546,0x04F0,0x4D20,0x6E65,0xF075,
	0x200D,0x7443,0x6C72,0x462B,0xF035,0x2003,0x6E49,0x6573,
	0x7472,0x6620,0x6172,0x656D,0x14F0,0x4620,0xF036,0x2004,
	0x7250,0x706F,0x7265,0x6974,0x7365,0x07F0,0x4320,0x7274,
	0x2B6C,0x3646,0x03F0,0x4520,0x6578,0x7563,0x6574,0x6420,
	0x6169,0x6F6C,0xF067,0x2012,0x3746,0x04F0,0x4320,0x6C6F,
	0x726F,0x0CF0,0x4320,0x7274,0x2B6C,0x3746,0x03F0,0x4620,
	0x696C,0x2070,0x6F63,0x6F6C,0x2D72,0x6F6D,0x6564,0x2820,
	0x3631,0x322F,0x3635,0xF029,0x2008,0x3846,0x04F0,0x4D20,
	0x766F,0x2065,0x626F,0x656A,0x7463,0x06F0,0x4320,0x7274,
	0x2B6C,0x3846,0x03F0,0x4D20,0x766F,0x2065,0x6574,0x7478,
	0x7266,0x6D61,0xF065,0x2012,0x3946,0x04F0,0x4920,0x736E,
	0x7265,0x2074,0x6574,0x7478,0x696C,0x656E,0x2020,0x7345,
	0xF063,0x2007,0x7845,0x7469,0x3F20,0x2820,0x6964,0x6C61,
	0x676F,0xF029,0x2011,0x3146,0xF030,0x2003,0x6F4D,0x6576,
	0x7420,0x206F,0x6563,0x746E,0x7265,0x03F0,0x4120,0x746C,
	0x582B,0x05F0,0x5120,0x6975,0xF074,0x205D,0x6F4D,0x7375,
	0x6365,0x696C,0x6B63,0x6F20,0x206E,0x6F6C,0x6577,0x2D72,
	0x6972,0x6867,0x2074,0x203D,0x6572,0x6973,0x657A,0x202E,
	0x7355,0x2065,0x6972,0x6867,0x2074,0x6F6D,0x7375,0x2065,
	0x6F74,0x7320,0x7465,0xF02C,0x2006,0x656C,0x7466,0x7420,
	0x206F,0x6567,0x2074,0x7361,0x6963,0x2F69,0x6F63,0x6F6C,
	0x2E72,0x45F0,0x4F20,0x206B,0xDC20,0x3BF0,0xF020,0xDF06,
	0x1DF0,0x7020,
}; //	644 byte
ROBJ *IDD_HELP = (ROBJ *)&HELP_RC;
/******** Resource end   HELP */

/******** Resource begin DOBJECT *
	{ 0x041C,  29,  16, {18, 3,44,19} },
	{ 0x8002,   0, 'D', {14, 4,12, 1} },
	{ 0x8002,   0,   0, {14, 5,12, 1} },
	{ 0x0002,   0, 'M', {14, 6,12, 1} },
	{ 0x0002,   0, 'S', {14, 7,12, 1} },
	{ 0x0002,   0,   0, {14, 8,12, 1} },
	{ 0x0002,   0, 'R', {14, 9,12, 1} },
	{ 0x0002,   0, 'K', {14,10,14, 1} },
	{ 0x0002,   0, 'G', {14,11,14, 1} },
	{ 0x0002,   0, 'L', {29, 4,12, 1} },
	{ 0x0002,   0,   0, {29, 5,12, 1} },
	{ 0x0002,   0,   0, {29, 6,12, 1} },
	{ 0x0002,   0,   0, {29, 7,12, 1} },
	{ 0x0002,   0,   0, {29, 8,12, 1} },
	{ 0x0002,   0,   0, {29, 9,12, 1} },
	{ 0x0002,   0,   0, {29,10,12, 1} },
	{ 0x0002,   0,   0, {29,11,12, 1} },
	{ 0x0004,   4, 'N', { 9, 2,32, 1} },
	{ 0x0004,   1, 'U', { 9, 5, 4, 1} },
	{ 0x0004,   1, 'I', { 9, 6, 4, 1} },
	{ 0x0004,   1, 'X', { 9, 8, 3, 1} },
	{ 0x0004,   1, 'Y', { 9, 9, 3, 1} },
	{ 0x0004,   1, 'O', { 9,10, 3, 1} },
	{ 0x0004,   1, 'W', { 9,11, 3, 1} },
	{ 0x0000,   0, 'A', { 3,13,20, 1} },
	{ 0x0000,   0, 'B', { 3,15,20, 1} },
	{ 0x0000,   0, 'F', { 3,17,20, 1} },
	{ 0x0000,   0, 'T', {30,13,10, 1} },
	{ 0x0000,   0, 'O', {30,15,10, 1} },
	{ 0x0200,   0, 'C', {30,17,10, 1} },
********* Resource data  *******************/
static int DOBJECT_RC[] = {
	0x0958,0x041C,0x101D,0x0312,0x132C,0x8002,0x4400,0x040E,
	0x010C,0x8002,0x0000,0x050E,0x010C,0x0002,0x4D00,0x060E,
	0x010C,0x0002,0x5300,0x070E,0x010C,0x0002,0x0000,0x080E,
	0x010C,0x0002,0x5200,0x090E,0x010C,0x0002,0x4B00,0x0A0E,
	0x010E,0x0002,0x4700,0x0B0E,0x010E,0x0002,0x4C00,0x041D,
	0x010C,0x0002,0x0000,0x051D,0x010C,0x0002,0x0000,0x061D,
	0x010C,0x0002,0x0000,0x071D,0x010C,0x0002,0x0000,0x081D,
	0x010C,0x0002,0x0000,0x091D,0x010C,0x0002,0x0000,0x0A1D,
	0x010C,0x0002,0x0000,0x0B1D,0x010C,0x0004,0x4E04,0x0209,
	0x0120,0x0004,0x5501,0x0509,0x0104,0x0004,0x4901,0x0609,
	0x0104,0x0004,0x5801,0x0809,0x0103,0x0004,0x5901,0x0909,
	0x0103,0x0004,0x4F01,0x0A09,0x0103,0x0004,0x5701,0x0B09,
	0x0103,0x0000,0x4100,0x0D03,0x0114,0x0000,0x4200,0x0F03,
	0x0114,0x0000,0x4600,0x1103,0x0114,0x0000,0x5400,0x0D1E,
	0x010A,0x0000,0x4F00,0x0F1E,0x010A,0x0200,0x4300,0x111E,
	0x010A,0x2CF0,0xF050,0x7011,0xF078,0x701D,0xF07F,0x7005,
	0x20F0,0xF007,0x7006,0xF07F,0x7036,0x09F0,0xF078,0x701A,
	0xF07F,0x7003,0x04F0,0x7007,0x09F0,0xF078,0x7018,0xF07F,
	0x7005,0x04F0,0xF007,0x7052,0x707F,0x03F0,0xF007,0x7027,
	0x707F,0x03F0,0xF007,0x7026,0x707F,0xF070,0x0703,0x27F0,
	0x7F70,0xF070,0x0703,0x4FF0,0xF070,0x5008,0xF05C,0x500B,
	0x07F0,0x5070,0x5C50,0x07F0,0xF050,0x7033,0x08F0,0x5C50,
	0x0BF0,0xF050,0x7007,0x5050,0xF05C,0x5007,0x33F0,0xF070,
	0x5008,0xF05C,0x500B,0x07F0,0x5070,0x5C50,0x07F0,0xF050,
	0x7030,0x10F0,0x4420,0x424F,0x3A4A,0x4420,0x6169,0x6F6C,
	0xF067,0x203F,0x614E,0x656D,0x28F0,0x4620,0x616C,0x2067,
	0x05F0,0xC2C4,0x1BF0,0xBFC4,0x04F0,0xDA20,0x0AF0,0xB4C4,
	0x205B,0x205D,0x4F44,0x4550,0xF04E,0x2006,0x205B,0x205D,
	0x4C43,0x4145,0xF052,0x2003,0xF0B3,0x2004,0x63B3,0x756F,
	0x746E,0x05F0,0xB320,0x205B,0x205D,0x4E4F,0x4353,0xF052,
	0x2006,0x205B,0x205D,0x4854,0x4C45,0xF050,0x2003,0xF0B3,
	0x2004,0x69B3,0x646E,0x7865,0x05F0,0xB320,0x205B,0x205D,
	0x4D44,0x564F,0xF045,0x2006,0x205B,0x205D,0x4552,0x4153,
	0xF054,0x2003,0xF0B3,0x2004,0xF0C3,0xC40A,0x5BB4,0x5D20,
	0x5320,0x4148,0x4544,0x06F0,0x5B20,0x5D20,0x4C20,0x494C,
	0x5453,0x03F0,0xB320,0x04F0,0xB320,0x04F0,0x7820,0x05F0,
	0xB320,0x205B,0x205D,0x594D,0x5542,0xF046,0x2006,0x205B,
	0x205D,0x5453,0x5245,0xF052,0x2003,0xF0B3,0x2004,0xF0B3,
	0x2004,0xF079,0x2005,0x5BB3,0x5D20,0x5220,0x4E43,0x5745,
	0x06F0,0x5B20,0x5D20,0x4D20,0x4E45,0x5355,0x03F0,0xB320,
	0x04F0,0xB320,0x2020,0x6F63,0xF06C,0x2005,0x5BB3,0x5D20,
	0x4220,0x4341,0x474B,0x06F0,0x5B20,0x5D20,0x4D20,0x5355,
	0x5245,0x03F0,0xB320,0x04F0,0xB320,0x2020,0x6F72,0xF077,
	0x2005,0x5BB3,0x5D20,0x4620,0x524F,0x4745,0x06F0,0x5B20,
	0x5D20,0x3C20,0xF03E,0x2006,0xF0B3,0x2004,0xF0C0,0xC40A,
	0xF0C1,0xC41B,0xF0D9,0x2007,0x6C43,0x6165,0x2072,0x7441,
	0x7274,0x6269,0x06F0,0xDC20,0x08F0,0x5420,0x7469,0x656C,
	0x03F0,0xDC20,0x07F0,0xF020,0xDF14,0x07F0,0xF020,0xDF0A,
	0x08F0,0x4320,0x656C,0x7261,0x4220,0x6361,0x676B,0x6F72,
	0x6E75,0x2064,0xDC20,0x08F0,0x4F20,0xF06B,0x2006,0xF0DC,
	0x2007,0x14F0,0xF0DF,0x2007,0x0AF0,0xF0DF,0x2008,0x6C43,
	0x6165,0x2072,0x6F46,0x6572,0x7267,0x756F,0x646E,0x2020,
	0xF0DC,0x2008,0x6143,0x636E,0x6C65,0x2020,0xF0DC,0x2007,
	0x14F0,0xF0DF,0x2007,0x0AF0,0xF0DF,0x2003,0x7070,
}; //	862 byte
ROBJ *IDD_DOBJECT = (ROBJ *)&DOBJECT_RC;
/******** Resource end   DOBJECT */

/******** Resource begin BACKGROUND *
	{ 0x041C,   8,   0, {18, 6,45,13} },
	{ 0x0000,   0, 'T', { 4, 3, 5, 1} },
	{ 0x0000,   0, 'D', { 4, 5, 5, 1} },
	{ 0x0000,   0, 'L', { 4, 7, 5, 1} },
	{ 0x0000,   0, 'M', { 4, 9, 5, 1} },
	{ 0x0000,   0, 'I', {24, 3, 5, 1} },
	{ 0x0000,   0, 'M', {24, 5, 5, 1} },
	{ 0x0000,   0, 'E', {24, 7, 5, 1} },
	{ 0x0000,   0, 'S', {24, 9, 5, 1} },
********* Resource data  *******************/
static int BACKGROUND_RC[] = {
	0x0567,0x041C,0x0008,0x0612,0x0D2D,0x0000,0x5400,0x0304,
	0x0105,0x0000,0x4400,0x0504,0x0105,0x0000,0x4C00,0x0704,
	0x0105,0x0000,0x4D00,0x0904,0x0105,0x0000,0x4900,0x0318,
	0x0105,0x0000,0x4D00,0x0518,0x0105,0x0000,0x4500,0x0718,
	0x0105,0x0000,0x5300,0x0918,0x0105,0x2DF0,0xF050,0x705E,
	0x5050,0x505C,0xF050,0x700F,0x5050,0x505C,0xF050,0x7003,
	0x05F0,0xF074,0x700C,0xF07A,0x7013,0xF07A,0x7018,0x5050,
	0x505C,0xF050,0x7003,0x07F0,0xF071,0x7005,0x5050,0x505C,
	0xF050,0x7003,0x05F0,0xF075,0x700B,0x7A7A,0x13F0,0x7A70,
	0x17F0,0x7A70,0x5050,0x505C,0xF050,0x7003,0x06F0,0xF072,
	0x7006,0x5050,0x505C,0xF050,0x7003,0x07F0,0xF076,0x7009,
	0x7A7A,0x13F0,0x7A70,0x13F0,0x7A70,0x04F0,0x5070,0x5C50,
	0x5050,0x0FF0,0x5070,0x5C50,0x5050,0x1AF0,0xF070,0x7A0A,
	0x7A70,0x7A70,0xF07A,0x706E,0x11F0,0x4220,0x6361,0x676B,
	0x6F72,0x6E75,0xF064,0x2041,0xF0DA,0xC413,0xF0C2,0xC413,
	0xF0BF,0x2004,0xF0B3,0x2003,0x2030,0xDC20,0x2020,0x6C42,
	0x6361,0xF06B,0x2005,0xF0B3,0x2003,0x2034,0xDC20,0x2020,
	0x7245,0x6F72,0xF072,0x2005,0xF0B3,0x2004,0x20B3,0xF020,
	0xDF05,0x0CF0,0xB320,0x2020,0x05F0,0xF0DF,0x200C,0xF0B3,
	0x2004,0xF0B3,0x2003,0x2031,0xDC20,0x2020,0x6544,0x6B73,
	0x6F74,0xF070,0x2003,0xF0B3,0x2003,0x2035,0xDC20,0x2020,
	0x6954,0x6C74,0xF065,0x2005,0xF0B3,0x2004,0x20B3,0xF020,
	0xDF05,0x0CF0,0xB320,0x2020,0x05F0,0xF0DF,0x200C,0xF0B3,
	0x2004,0xF0B3,0x2003,0x2032,0xDC20,0x2020,0x6944,0x6C61,
	0x676F,0x04F0,0xB320,0x03F0,0x3620,0x2020,0x20DC,0x5220,
	0x7665,0x7265,0x6573,0x03F0,0xB320,0x04F0,0xB320,0x2020,
	0x05F0,0xF0DF,0x200C,0x20B3,0xF020,0xDF05,0x0CF0,0xB320,
	0x04F0,0xB320,0x03F0,0x3320,0x2020,0x20DC,0x4D20,0x6E65,
	0x7375,0x05F0,0xB320,0x03F0,0x3720,0x2020,0x20DC,0x4720,
	0x6172,0xF079,0x2006,0xF0B3,0x2004,0x20B3,0xF020,0xDF05,
	0x0CF0,0xB320,0x2020,0x05F0,0xF0DF,0x200C,0xF0B3,0x2004,
	0xF0C0,0xC413,0xF0C1,0xC413,0xF0D9,0x202F,0x505C,
}; //	510 byte
ROBJ *IDD_BACKGROUND = (ROBJ *)&BACKGROUND_RC;
/******** Resource end   BACKGROUND */

/******** Resource begin Color *
	{ 0x081C,  13,  12, { 9, 4,52,11} },
	{ 0x0001,   0, 'B', { 4, 3, 8, 1} },
	{ 0x0001,   0, 'G', { 4, 4, 9, 1} },
	{ 0x0001,   0, 'Y', { 4, 5,10, 1} },
	{ 0x0001,   0, 'R', {18, 3, 7, 1} },
	{ 0x0001,   0, 'T', {18, 4,10, 1} },
	{ 0x0101,   0, 'D', {18, 5,11, 1} },
	{ 0x0002,   0, 'U', { 4, 7,24, 1} },
	{ 0x4000,   0, 'A', {33, 3,16, 1} },
	{ 0x4000,   0, 'P', {33, 5,16, 1} },
	{ 0x0000,   0, 's', { 3, 9,12, 1} },
	{ 0xC000,   0, 'L', {19, 9, 8, 1} },
	{ 0xC000,   0, 'v', {30, 9, 8, 1} },
	{ 0x0A00,   0, 'Q', {41, 9, 8, 1} },
********* Resource data  *******************/
static int Color_RC[] = {
	0x05A0,0x081C,0x0C0D,0x0409,0x0B34,0x0001,0x4200,0x0304,
	0x0108,0x0001,0x4700,0x0404,0x0109,0x0001,0x5900,0x0504,
	0x010A,0x0001,0x5200,0x0312,0x0107,0x0001,0x5400,0x0412,
	0x010A,0x0101,0x4400,0x0512,0x010B,0x0002,0x5500,0x0704,
	0x0118,0x4000,0x4100,0x0321,0x0110,0x4000,0x5000,0x0521,
	0x0110,0x0000,0x7300,0x0903,0x010C,0xC000,0x4C00,0x0913,
	0x0108,0xC000,0x7600,0x091E,0x0108,0x0A00,0x5100,0x0929,
	0x0108,0x34F0,0xF050,0x2A70,0xF02E,0x2A0D,0xF02E,0x2A0A,
	0x07F0,0x5C50,0x08F0,0x2050,0x0AF0,0x2E2A,0x10F0,0x2E2A,
	0x08F0,0xF02A,0x2010,0x0AF0,0x2E2A,0x0DF0,0x2E2A,0x0AF0,
	0xF02A,0x5007,0xF05C,0x5008,0xF020,0x2A24,0x10F0,0xF020,
	0x2A0A,0xF02E,0x2A62,0x5050,0xF05C,0x5009,0xF020,0x2A03,
	0x5858,0xF05C,0x5805,0x2A20,0xF02A,0x5804,0xF05C,0x5803,
	0x2A20,0x502A,0x5C50,0x05F0,0x2050,0x06F0,0xF02A,0x200C,
	0x04F0,0xF02A,0x2008,0x03F0,0xF02A,0x2008,0x03F0,0xF02A,
	0x2008,0x2A2A,0x17F0,0x4320,0x6C6F,0x726F,0x4EF0,0xDA20,
	0x20C4,0x7453,0x6E61,0x6164,0x6472,0x6320,0x6C6F,0x726F,
	0xF020,0xC40A,0xF0BF,0x2017,0x20B3,0x2028,0x2029,0x6C42,
	0x6575,0x06F0,0x2820,0x2920,0x5220,0x6465,0x05F0,0xB320,
	0x04F0,0x4520,0x6964,0x2074,0x7441,0x7274,0x6269,0x03F0,
	0xDC20,0x04F0,0xB320,0x2820,0x2920,0x4720,0x6572,0x6E65,
	0x05F0,0x2820,0x2920,0x4C20,0x7061,0x6F74,0x2070,0xB320,
	0x03F0,0xF020,0xDF10,0x04F0,0xB320,0x2820,0x2920,0x5920,
	0x6C65,0x6F6C,0xF077,0x2004,0x2028,0x2029,0x6544,0x6166,
	0x6C75,0x2074,0xF0B3,0x2004,0x6445,0x7469,0x5020,0x6C61,
	0x7465,0x6574,0x2020,0xF0DC,0x2004,0xF0C0,0xC41B,0xF0D9,
	0x2003,0x10F0,0xF0DF,0x2006,0x205B,0x205D,0x7355,0x2065,
	0x7267,0x7061,0x6968,0x6163,0x206C,0x6F63,0x6F6C,0x7372,
	0x51F0,0x5320,0x6174,0x646E,0x7261,0x2064,0xDC20,0x05F0,
	0x4C20,0x616F,0x2064,0xDC20,0x04F0,0x5320,0x7661,0x2065,
	0xDC20,0x04F0,0x5120,0x6975,0x2074,0xDC20,0x06F0,0xF020,
	0xDF0C,0x04F0,0xF020,0xDF08,0x03F0,0xF020,0xDF08,0x03F0,
	0xF020,0xDF08,0x2020,0x2A2A,
}; //	520 byte
ROBJ *IDD_Color = (ROBJ *)&Color_RC;
/******** Resource end   Color */

/******** Resource begin DialogOpen *
	{ 0x041C,  17,  14, {24, 2,41,20} },
	{ 0x8043,   0,   0, { 5, 5,31, 1} },
	{ 0x8043,   0,   0, { 5, 6,31, 1} },
	{ 0x8043,   0,   0, { 5, 7,31, 1} },
	{ 0x8043,   0,   0, { 5, 8,31, 1} },
	{ 0x8043,   0,   0, { 5, 9,31, 1} },
	{ 0x8043,   0,   0, { 5,10,31, 1} },
	{ 0x8043,   0,   0, { 5,11,31, 1} },
	{ 0x8043,   0,   0, { 5,12,31, 1} },
	{ 0x8043,   0,   0, { 5,13,31, 1} },
	{ 0x8043,   0,   0, { 5,14,31, 1} },
	{ 0x8043,   0,   0, { 5,15,31, 1} },
	{ 0x8043,   0,   0, { 5,16,31, 1} },
	{ 0x0000,   0, 'O', { 4,18, 8, 1} },
	{ 0x0200,   0, 'C', {16,18,10, 1} },
	{ 0x4004,   0, 'F', {10, 2,26, 1} },
	{ 0x8008,   0,   0, { 3, 4,35, 1} },
	{ 0x8009,   0,   0, { 3,17,35, 1} },
********* Resource data  *******************/
static int DialogOpen_RC[] = {
	0x07D7,0x041C,0x0E11,0x0218,0x1429,0x8043,0x0000,0x0505,
	0x011F,0x8043,0x0000,0x0605,0x011F,0x8043,0x0000,0x0705,
	0x011F,0x8043,0x0000,0x0805,0x011F,0x8043,0x0000,0x0905,
	0x011F,0x8043,0x0000,0x0A05,0x011F,0x8043,0x0000,0x0B05,
	0x011F,0x8043,0x0000,0x0C05,0x011F,0x8043,0x0000,0x0D05,
	0x011F,0x8043,0x0000,0x0E05,0x011F,0x8043,0x0000,0x0F05,
	0x011F,0x8043,0x0000,0x1005,0x011F,0x0000,0x4F00,0x1204,
	0x0108,0x0200,0x4300,0x1210,0x010A,0x4004,0x4600,0x020A,
	0x011A,0x8008,0x0000,0x0403,0x0123,0x8009,0x0000,0x1103,
	0x0123,0x29F0,0xF060,0x7028,0xF07A,0x7003,0xF07F,0x7024,
	0xF07A,0x7003,0xF07F,0x7024,0xF07A,0x7028,0xF07A,0x7028,
	0xF07A,0x7028,0xF07A,0x7028,0xF07A,0x7028,0xF07A,0x7028,
	0xF07A,0x7028,0xF07A,0x7028,0xF07A,0x7028,0xF07A,0x7028,
	0xF07A,0x7028,0xF07A,0x7028,0xF07A,0x7028,0xF07A,0x7028,
	0xF07A,0x7004,0x6060,0xF06F,0x6005,0x04F0,0x6070,0x6F60,
	0x07F0,0xF060,0x700E,0xF07A,0x7028,0xF07A,0x2012,0x704F,
	0x6E65,0x3FF0,0x4620,0x6C69,0x3A65,0x5B20,0x1AF0,0x5D20,
	0x07F0,0x5220,0x7365,0x756F,0x6372,0x3A65,0x21F0,0xDA20,
	0x1FF0,0xBFC4,0x08F0,0xB320,0x1FF0,0xB320,0x08F0,0xB320,
	0x1FF0,0xB320,0x08F0,0xB320,0x1FF0,0xB320,0x08F0,0xB320,
	0x1FF0,0xB320,0x08F0,0xB320,0x1FF0,0xB320,0x08F0,0xB320,
	0x1FF0,0xB320,0x08F0,0xB320,0x1FF0,0xB320,0x08F0,0xB320,
	0x1FF0,0xB320,0x08F0,0xB320,0x1FF0,0xB320,0x08F0,0xB320,
	0x1FF0,0xB320,0x08F0,0xB320,0x1FF0,0xB320,0x08F0,0xB320,
	0x1FF0,0xB320,0x08F0,0xC020,0x1FF0,0xD9C4,0x0AF0,0x4F20,
	0x6570,0x206E,0xDC20,0x05F0,0x4320,0x6E61,0x6563,0x206C,
	0xDC20,0x13F0,0xF020,0xDF08,0x04F0,0xF020,0xDF0A,0x0EF0,
	0x7020,
}; //	434 byte
ROBJ *IDD_DialogOpen = (ROBJ *)&DialogOpen_RC;
/******** Resource end   DialogOpen */

/******** Resource begin FOREGROUND *
	{ 0x4C1C,  16,   0, { 2, 2,19,19} },
	{ 0x0006,   0, 'B', { 2, 2,15, 1} },
	{ 0x0006,   0, 'L', { 2, 3,15, 1} },
	{ 0x0006,   0, 'E', { 2, 4,15, 1} },
	{ 0x0006,   0, 'C', { 2, 5,15, 1} },
	{ 0x0006,   0, 'R', { 2, 6,15, 1} },
	{ 0x0006,   0, 'M', { 2, 7,15, 1} },
	{ 0x0006,   0, 'W', { 2, 8,15, 1} },
	{ 0x0006,   0, 'G', { 2, 9,15, 1} },
	{ 0x0006,   0, 'K', { 2,10,15, 1} },
	{ 0x0006,   0, 'D', { 2,11,15, 1} },
	{ 0x0006,   0, 'I', { 2,12,15, 1} },
	{ 0x0006,   0, 'N', { 2,13,15, 1} },
	{ 0x0006,   0, 'T', { 2,14,15, 1} },
	{ 0x0006,   0, 'S', { 2,15,15, 1} },
	{ 0x0006,   0, 'O', { 2,16,15, 1} },
	{ 0x0006,   0, 'U', { 2,17,15, 1} },
********* Resource data  *******************/
static int FOREGROUND_RC[] = {
	0x0419,0x4C1C,0x0010,0x0202,0x1313,0x0006,0x4200,0x0202,
	0x010F,0x0006,0x4C00,0x0302,0x010F,0x0006,0x4500,0x0402,
	0x010F,0x0006,0x4300,0x0502,0x010F,0x0006,0x5200,0x0602,
	0x010F,0x0006,0x4D00,0x0702,0x010F,0x0006,0x5700,0x0802,
	0x010F,0x0006,0x4700,0x0902,0x010F,0x0006,0x4B00,0x0A02,
	0x010F,0x0006,0x4400,0x0B02,0x010F,0x0006,0x4900,0x0C02,
	0x010F,0x0006,0x4E00,0x0D02,0x010F,0x0006,0x5400,0x0E02,
	0x010F,0x0006,0x5300,0x0F02,0x010F,0x0006,0x4F00,0x1002,
	0x010F,0x0006,0x5500,0x1102,0x010F,0x13F0,0xF050,0x2016,
	0xF02E,0x2013,0xF02E,0x2009,0x2121,0x08F0,0x2E20,0x08F0,
	0x2220,0xF022,0x2006,0xF02E,0x200A,0x2323,0x06F0,0x2E20,
	0x0AF0,0x2420,0xF024,0x2006,0xF02E,0x200A,0x2525,0x09F0,
	0x2E20,0x07F0,0x2620,0xF026,0x2006,0xF02E,0x2015,0xF02E,
	0x2007,0x2828,0x06F0,0x2E20,0x0AF0,0x2920,0xF029,0x2007,
	0xF02E,0x2009,0x2A2A,0x08F0,0x2E20,0x08F0,0x2B20,0xF02B,
	0x2006,0xF02E,0x200A,0x2C2C,0x08F0,0x2E20,0x08F0,0x2F20,
	0xF02F,0x200A,0xF02E,0x2006,0x2E2E,0x09F0,0x2E20,0x07F0,
	0x2F20,0xF02F,0x2016,0x04F0,0x4620,0x726F,0x6765,0x6F72,
	0x6E75,0xF064,0x2006,0xF0DA,0xC40F,0x20BF,0xB320,0x4220,
	0x616C,0x6B63,0x06F0,0xDB20,0x20DB,0x20B3,0xB320,0x4220,
	0x756C,0xF065,0x2007,0xDBDB,0xB320,0x2020,0x20B3,0x7247,
	0x6565,0xF06E,0x2006,0xDBDB,0xB320,0x2020,0x20B3,0x7943,
	0x6E61,0x07F0,0xDB20,0x20DB,0x20B3,0xB320,0x5220,0x6465,
	0x08F0,0xDB20,0x20DB,0x20B3,0xB320,0x4D20,0x6761,0x6E65,
	0x6174,0x04F0,0xDB20,0x20DB,0x20B3,0xB320,0x4220,0x6F72,
	0x6E77,0x06F0,0xDB20,0x20DB,0x20B3,0xB320,0x4720,0x6172,
	0xF079,0x2007,0xDBDB,0xB320,0x2020,0x20B3,0x6144,0x6B72,
	0x7247,0x7961,0x03F0,0xDB20,0x20DB,0x20B3,0xB320,0x4420,
	0x7365,0x746B,0x706F,0x04F0,0xDB20,0x20DB,0x20B3,0xB320,
	0x4420,0x6169,0x6F6C,0xF067,0x2005,0xDBDB,0xB320,0x2020,
	0x20B3,0x654D,0x756E,0xF073,0x2006,0xDBDB,0xB320,0x2020,
	0x20B3,0x6954,0x6C74,0x4B65,0x7965,0x03F0,0xDB20,0x20DB,
	0x20B3,0xB320,0x4420,0x7365,0x746B,0x706F,0x654B,0x2079,
	0xDBDB,0xB320,0x2020,0x20B3,0x6944,0x6C61,0x676F,0x654B,
	0x2079,0xDB20,0x20DB,0x20B3,0xB320,0x4D20,0x6E65,0x7375,
	0x654B,0xF079,0x2003,0xDBDB,0xB320,0x2020,0xF0C0,0xC40F,
	0x20D9,0xB320,
}; //	580 byte
ROBJ *IDD_FOREGROUND = (ROBJ *)&FOREGROUND_RC;
/******** Resource end   FOREGROUND */

/******** Resource begin Frametype *
	{ 0x0C1C,   4,   0, {40, 5,28,11} },
	{ 0x0000,   0, 'S', { 3, 2,14, 1} },
	{ 0x0000,   0, 'D', { 3, 4,14, 1} },
	{ 0x0000,   0, 'V', { 3, 6,14, 1} },
	{ 0x0000,   0, 'H', { 3, 8,14, 1} },
********* Resource data  *******************/
static int Frametype_RC[] = {
	0x02E8,0x0C1C,0x0004,0x0528,0x0B1C,0x0000,0x5300,0x0203,
	0x010E,0x0000,0x4400,0x0403,0x010E,0x0000,0x5600,0x0603,
	0x010E,0x0000,0x4800,0x0803,0x010E,0x1CF0,0xF050,0x2A1F,
	0x5050,0xF05C,0x500B,0xF020,0x2A0E,0x0EF0,0xF020,0x2A0D,
	0x5050,0xF05C,0x500B,0xF020,0x2A0E,0x0EF0,0xF020,0x2A0D,
	0x5050,0xF05C,0x500B,0xF020,0x2A0E,0x0EF0,0xF020,0x2A0D,
	0x5050,0xF05C,0x500B,0xF020,0x2A0E,0x0EF0,0xF020,0x2A26,
	0x06F0,0x5320,0x6C65,0x6365,0x2074,0x7246,0x6D61,0x7465,
	0x7079,0xF065,0x2027,0x6953,0x676E,0x6C65,0x06F0,0xDC20,
	0x03F0,0xDA20,0xC4C4,0xF0BF,0x2007,0x0EF0,0xF0DF,0x2003,
	0xC4C0,0xD9C4,0x08F0,0x4420,0x626F,0x6562,0xF06C,0x2006,
	0xF0DC,0x2003,0xCDC9,0xBBCD,0x07F0,0xF020,0xDF0E,0x03F0,
	0xC820,0xCDCD,0xF0BC,0x2008,0x6556,0x7472,0x6369,0x6C61,
	0x04F0,0xDC20,0x03F0,0xC320,0xC4C4,0xF0B4,0x2007,0x0EF0,
	0xF0DF,0x2003,0xC4C3,0xB4C4,0x08F0,0x4820,0x726F,0x7A69,
	0x6E6F,0x6174,0x206C,0xDC20,0x03F0,0xC220,0xC4C4,0xF0C2,
	0x2007,0x0EF0,0xF0DF,0x2003,0xC4C1,0xC1C4,0x1FF0,0x2A20,
}; //	272 byte
ROBJ *IDD_Frametype = (ROBJ *)&Frametype_RC;
/******** Resource end   Frametype */

/******** Resource begin TOBJECT *
	{ 0x041C,  40,  36, {14, 2,60,21} },
	{ 0x0001,   0, 'U', {40, 2,16, 1} },
	{ 0x0001,   0, 'R', {40, 3,16, 1} },
	{ 0x0001,   0, 'K', {40, 4,16, 1} },
	{ 0x0001,   0, 'E', {40, 5,16, 1} },
	{ 0x0001,   0, 'T', {40, 6,16, 1} },
	{ 0x0001,   0, 'M', {40, 7,16, 1} },
	{ 0x0001,   0, 'H', {40, 8,16, 1} },
	{ 0x0001,   0, 'S', {40, 9,16, 1} },
	{ 0x0001,   0,   0, {40,10,16, 1} },
	{ 0x0001,   0,   0, {40,11,16, 1} },
	{ 0x0001,   0,   0, {40,12,16, 1} },
	{ 0x0001,   0,   0, {40,13,16, 1} },
	{ 0x0001,   0,   0, {40,14,16, 1} },
	{ 0x0001,   0,   0, {40,15,16, 1} },
	{ 0x0001,   0,   0, {40,16,16, 1} },
	{ 0x0001,   0,   0, {40,17,16, 1} },
	{ 0x0002,   0,   0, {28, 4,10, 1} },
	{ 0x0002,   0,   0, {28, 5,10, 1} },
	{ 0x0002,   0,   0, {28, 6,10, 1} },
	{ 0x0002,   0,   0, {28, 7,10, 1} },
	{ 0x0002,   0,   0, {28, 8,10, 1} },
	{ 0x0002,   0,   0, {28, 9,10, 1} },
	{ 0x0002,   0,   0, {28,10,10, 1} },
	{ 0x0002,   0,   0, {28,11,10, 1} },
	{ 0x0002,   0,   0, {28,12,10, 1} },
	{ 0x0002,   0,   0, {28,13,10, 1} },
	{ 0x0002,   0,   0, {28,14,10, 1} },
	{ 0x0002,   0,   0, {28,15,10, 1} },
	{ 0x0004,   1, 'X', {21, 6, 3, 1} },
	{ 0x0004,   1, 'Y', {21, 7, 3, 1} },
	{ 0x0004,   1, 'L', {21, 8, 3, 1} },
	{ 0x0004,   1, 'W', {21, 9, 3, 1} },
	{ 0x0004,   1, 'A', {22,11, 2, 1} },
	{ 0x0004,   1, 'B', {19,12, 5, 1} },
	{ 0x4004,   1, 'I', {21,14, 3, 1} },
	{ 0x0004,   4, 'N', {10,17,27, 1} },
	{ 0x0000,   0, 'O', { 3,19, 6, 1} },
	{ 0x0000,   0, 'P', {13,19, 8, 1} },
	{ 0x0000,   0, 'D', {25,19,10, 1} },
	{ 0x0A00,   0, 'C', {39,19,10, 1} },
********* Resource data  *******************/
static int TOBJECT_RC[] = {
	0x0D7C,0x041C,0x2428,0x020E,0x153C,0x0001,0x5500,0x0228,
	0x0110,0x0001,0x5200,0x0328,0x0110,0x0001,0x4B00,0x0428,
	0x0110,0x0001,0x4500,0x0528,0x0110,0x0001,0x5400,0x0628,
	0x0110,0x0001,0x4D00,0x0728,0x0110,0x0001,0x4800,0x0828,
	0x0110,0x0001,0x5300,0x0928,0x0110,0x0001,0x0000,0x0A28,
	0x0110,0x0001,0x0000,0x0B28,0x0110,0x0001,0x0000,0x0C28,
	0x0110,0x0001,0x0000,0x0D28,0x0110,0x0001,0x0000,0x0E28,
	0x0110,0x0001,0x0000,0x0F28,0x0110,0x0001,0x0000,0x1028,
	0x0110,0x0001,0x0000,0x1128,0x0110,0x0002,0x0000,0x041C,
	0x010A,0x0002,0x0000,0x051C,0x010A,0x0002,0x0000,0x061C,
	0x010A,0x0002,0x0000,0x071C,0x010A,0x0002,0x0000,0x081C,
	0x010A,0x0002,0x0000,0x091C,0x010A,0x0002,0x0000,0x0A1C,
	0x010A,0x0002,0x0000,0x0B1C,0x010A,0x0002,0x0000,0x0C1C,
	0x010A,0x0002,0x0000,0x0D1C,0x010A,0x0002,0x0000,0x0E1C,
	0x010A,0x0002,0x0000,0x0F1C,0x010A,0x0004,0x5801,0x0615,
	0x0103,0x0004,0x5901,0x0715,0x0103,0x0004,0x4C01,0x0815,
	0x0103,0x0004,0x5701,0x0915,0x0103,0x0004,0x4101,0x0B16,
	0x0102,0x0004,0x4201,0x0C13,0x0105,0x4004,0x4901,0x0E15,
	0x0103,0x0004,0x4E04,0x110A,0x011B,0x0000,0x4F00,0x1303,
	0x0106,0x0000,0x5000,0x130D,0x0108,0x0000,0x4400,0x1319,
	0x010A,0x0A00,0x4300,0x1327,0x010A,0x3CF0,0xF050,0x703F,
	0xF07F,0x7029,0xF03F,0x703A,0xF07F,0x703F,0xF07F,0x7039,
	0xF07F,0x7020,0x707F,0x03F0,0xF007,0x7014,0xF07F,0x7022,
	0x707F,0x03F0,0xF007,0x7014,0xF07F,0x7022,0x707F,0x03F0,
	0xF007,0x7015,0xF07F,0x7021,0x707F,0x03F0,0xF007,0x7017,
	0xF07F,0x704D,0xF07F,0x7010,0x0707,0x29F0,0x7F70,0x0DF0,
	0xF070,0x0705,0x65F0,0x7F70,0x0FF0,0xF070,0x0703,0xA1F0,
	0x7F70,0x04F0,0xF070,0x071B,0x56F0,0x5070,0x5C50,0x03F0,
	0xF050,0x7004,0x04F0,0x5C50,0x03F0,0xF050,0x7004,0x5050,
	0xF05C,0x5007,0x04F0,0x5070,0x5C50,0x07F0,0xF050,0x7047,
	0x14F0,0x5420,0x424F,0x3A4A,0x4420,0x6169,0x6F6C,0x2067,
	0x624F,0x656A,0x7463,0x3BF0,0xDA20,0x11F0,0xBFC4,0x08F0,
	0x5420,0x7079,0x2065,0x6228,0x7469,0x3020,0x332D,0xF029,
	0x2005,0x0EF0,0x30C4,0x2820,0x2920,0x5020,0x7375,0x6268,
	0x7475,0x6F74,0x206E,0xB320,0x08F0,0x4620,0x616C,0x2067,
	0x6228,0x7469,0x3420,0x312D,0x2935,0x04F0,0xF020,0x4603,
	0xF030,0xC40A,0x20B4,0x2028,0x2029,0x6152,0x6964,0x626F,
	0x7475,0x6F74,0x206E,0xF0B3,0x201B,0x3030,0x2031,0x205B,
	0x205D,0x4C46,0x4741,0x2041,0x20B3,0x2028,0x2029,0x6843,
	0x6365,0x626B,0x786F,0x04F0,0xB320,0x1BF0,0x3020,0x3230,
	0x5B20,0x5D20,0x4620,0x414C,0x4247,0xB320,0x2820,0x2920,
	0x5820,0x6543,0x6C6C,0x07F0,0xB320,0x08F0,0x5220,0x6365,
	0xF074,0x200A,0xF078,0x2006,0x20B3,0x205B,0x205D,0x4C4C,
	0x5349,0x2054,0x20B3,0x2028,0x2029,0x6554,0x7478,0x6465,
	0x7469,0x04F0,0xB320,0x16F0,0x7920,0x06F0,0xB320,0x5B20,
	0x5D20,0x3C20,0xF03E,0x2004,0x20B3,0x2028,0x2029,0x654D,
	0x756E,0xF073,0x2007,0xF0B3,0x2014,0x6F63,0xF06C,0x2006,
	0x20B3,0x205B,0x205D,0x4152,0x4944,0x204F,0x20B3,0x2028,
	0x2029,0x4858,0x6D74,0xF06C,0x2007,0xF0B3,0x2014,0x6F72,
	0xF077,0x2006,0x20B3,0x205B,0x205D,0x4544,0x4958,0x2054,
	0x20B3,0x2028,0x2029,0x6F4D,0x7375,0x2065,0x7261,0x6165,
	0x2020,0xF0B3,0x201B,0x3430,0x2030,0x205B,0x205D,0x4844,
	0x4C45,0x2050,0x20B3,0x2028,0x2029,0x494C,0x5453,0xF055,
	0x2007,0xF0B3,0x2008,0x7341,0x6963,0x2069,0x6873,0x726F,
	0x6B74,0x7965,0x07F0,0xB320,0x5B20,0x5D20,0x4320,0x5255,
	0x4449,0xB320,0x2820,0x2920,0x4C20,0x5349,0x4454,0x07F0,
	0xB320,0x08F0,0x4220,0x6675,0x6566,0x2072,0x6973,0x657A,
	0x0AF0,0xB320,0x5B20,0x5D20,0x4720,0x434C,0x444D,0xB320,
	0x2820,0x2920,0x0DF0,0x4120,0x1DF0,0xB320,0x5B20,0x5D20,
	0x4520,0x4556,0x544E,0xB320,0x2820,0x2920,0x0DF0,0x4220,
	0x08F0,0x4920,0x646E,0x7865,0x10F0,0xB320,0x5B20,0x5D20,
	0x4320,0x4948,0x444C,0xB320,0x2820,0x2920,0x0DF0,0x4320,
	0x1BF0,0x3820,0x3030,0x5B20,0x5D20,0x5320,0x4154,0x4554,
	0xB320,0x2820,0x2920,0x0DF0,0x4420,0x1DF0,0xC020,0x0BF0,
	0xB4C4,0x2820,0x2920,0x0DF0,0x4520,0x08F0,0x4E20,0x6D61,
	0xF065,0x201D,0x2046,0x2028,0xF029,0x200D,0xF0B3,0x2029,
	0xF0C0,0xC411,0xF0D9,0x2008,0x6B4F,0x2020,0xF0DC,0x2005,
	0x6F43,0x7970,0x2020,0xF0DC,0x2005,0x6544,0x656C,0x6574,
	0x2020,0xF0DC,0x2005,0x6143,0x636E,0x6C65,0x2020,0xF0DC,
	0x200E,0x06F0,0xF0DF,0x2004,0x08F0,0xF0DF,0x2004,0x0AF0,
	0xF0DF,0x2004,0x0AF0,0xF0DF,0x200A,0x7070,
}; //	1132 byte
ROBJ *IDD_TOBJECT = (ROBJ *)&TOBJECT_RC;
/******** Resource end   TOBJECT */
