;	TEDIT.ASM			Version 1.0, June 15, 1995

;	Tiny full screen, line oriented, text editor.

;	Inspired from TED.ASM (PC Magazine, Tom Kihlken, 11/15/88).

;	Assemble, link, and convert to a .COM file.

;	Invoke by using:	tedit [/s or /m or /l] [filename]
; where the /parameter allows editing files of up to 16k bytes (/s for small), 
; 32k bytes (/m for medium), or 64k bytes (/l for large).  When no /parameter
; is given, or it is erroneous, a default of 8k bytes (tiny) is used.

;	The program assumes that a new file is to be created unless invoked
; with the name of the file to be edited.  It creates automatically a *.bak
; file when saving a modifed file.  If desired, the modified file can be saved
; with a new name.

;	The program takes on its-own about 3k bytes of memory.  Additional
; space is taken by the file buffer and a paste buffer of the same size.  Thus
; total memory requirements varies from about 19k to about 131k depending on
; the /parameter used.

;	The editor is full screen in that the cursor keys, PageUp, PageDown,
; Home (of line), and End (of line) keys allows cursor movement throughout the
; screen.  The Ins key toggles the Insert/Overstrike mode and the Delete key
; deletes the character at the cursor.

;	The key combinations CTRL PageUp and CTRL PageDown allows to position
; the cursor at the beginning and end of the file respectively.

;	The editor is line oriented in that each line must be terminated by a
; carriage return.  However, a line can consist of up to 254 characters.  A
; line which is longer than the number of columns on the screen has a diamond
; (ASCII 4) as its right most displayed character.  CTRL -> and CTRL <- shift
; the screen by 8 columns so that the hidden portions of the long lines can be
; displayed/edited.

;	The editor defaults to a screen of 80 columns by 25 lines but can also
; work with other standard DOS text modes (e.g. 40 x 25, 80 x 30, or 80 x 43)
; if supported by the hardware and set before starting the program.

;	The bottom line of the screen displays the meaning of the ten function
; keys.  The action of the function keys is straight forward with the exception
; of F4 (mark; toggle mark state i.e. block definition on/off), F5 (cut; moves
; the marked text to the paste buffer), F3 (print; print the marked text) and
; F6 (paste; insert the contents of the paste buffer at the cursor).  The
; equivalent of a copy function is performed by cutting the marked text and
; then pasting it without moving the cursor, then pasting it again at the
; desired copy location (the paste buffer remains the same until a new cut is
; made).

;
CSEG	SEGMENT
	ASSUME CS:CSEG, DS:CSEG, ES:CSEG
	ORG 100h

start:	JMP	begin
;
; Local data area
;
TAB	EQU	 9
CR	EQU	13
LF	EQU	10
;
copyright	DB	CR,LF,"TEDIT 1.0"
		DB	CR,LF,"Adaptation of TED 1.0"
		DB	"(c) Ziff Communications Co."
		DB	CR,LF,"PC Magazine ", 254," Tom Kihlken$"
		DB	1Ah

memory_error	DB	"Not enough memory$"
file_too_big	DB	"File too big$"
read_err_mess	DB	"Read error$"

verify_mess	DB	" Lose changes (y/n)?",00h
save_mess	DB	" Save as: ",00h
dot_$$$		DB	".$$$",00h
dot_bak		DB	".bak",00h

prompt_string	DB	"1Abort", 00h
		DB	"2Undo", 00h
		DB	"3Print", 00h
		DB	"4Mark", 00h
		DB	"5Cut", 00h
		DB	"6Paste", 00h
		DB	"7Exit", 00h
		DB	"8Del ->line", 00h
		DB	"9Del L", 00h
		DB	"10Udel L", 00h, 00h
prompt_length	= $ - OFFSET prompt_string

dirty_bits	DB	01h
insert_mode	DB	0FFh
mark_mode	DB	00h
line_flag	DB	00h
normal		DB	07h
inverse		DB	70h
left_margin	DB	00h
margin_count	DB	00h
save_column	DB	00h
save_row	DB	00h
rows		DB	23
	
		Even
file_seg_size	DW	0200h		; Default segment size of 8192 bytes
file_max_size	DW	01FFEh		; Default size of 8190 bytes+CR+LF+EOF
name_pointer	DW	0081h
name_end	DW	0081h
video_seg	DW	0B000h
status_reg	DW	?
paste_seg	DW	?
paste_size	DW	0000h
line_length	DW	0000h
undo_length	DW	0000h
cursor		DW	0000h
cur_posn	DW	0000h
mark_start	DW	0FFFFh
mark_end	DW	0000h
mark_home	DW	0000h
top_of_screen	DW	0000h
last_char	DW	0000h
columnsb	LABEL	BYTE
columns		DW	?
page_proc	DW	?
oldint24	DD	?
;
dispatch_table	DW	OFFSET abort
		DW	OFFSET undo
		DW	OFFSET print
		DW	OFFSET mark
		DW	OFFSET cut
		DW	OFFSET paste
		DW	OFFSET exit
		DW	OFFSET del_eol
		DW	OFFSET del_l
		DW	OFFSET udel_l
		DW	OFFSET bad_key
		DW	OFFSET bad_key
		DW	OFFSET home
		DW	OFFSET up
		DW	OFFSET pgup
		DW	OFFSET bad_key
		DW	OFFSET left
		DW	OFFSET bad_key
		DW	OFFSET right
		DW	OFFSET bad_key
		DW	OFFSET endd
		DW	OFFSET down
		DW	OFFSET pgdn
		DW	OFFSET insert
		DW	OFFSET del_char
;
; The following machine instruction removes the desnow delay.  It is inserted
;into the code for EGA, VGA, and MONO displays.
;
no_desnow = 0EBh + (OFFSET write_it - OFFSET hwait - 2) * 256
;
; Initialize the display.  Then parse the command line for a /parameter, if
;one found, decode and allocate the requested memory for the file and paste
;segments.  Parse the remaining of the command line for a filename, if one was 
;input, read in the file.  Finally, set the INT 23 and 24 vectors.
;

begin:	XOR	AX,AX
	MOV	DS,AX			; Segment 0000h
	ASSUME	DS:nothing
	MOV	AH,12H
	MOV	BL,10H
	INT	10H			; Get EGA information
	CMP	BL,10H
	JE	not_ega			; If BL did not change, not EGA
	TEST	BYTE PTR DS:[0487h],08h ; Is EGA active?
	JNZ	not_ega
	MOV	WORD PTR CS:hwait,no_desnow	; Get rid of desnow
	MOV	AX,DS:[0484h]		; Get number of rows
	DEC	AL			; Last row is for prompt line
	MOV	CS:[rows],AL		; Save the number of rows
not_ega:
	MOV	AX,DS:[044Ah]		; Get number of columns
	MOV	CS:columns,ax		; and store it
	MOV	AX,DS:[0463h]		; Address of display card
	ADD	AX,6			; For status port
	PUSH	CS
	POP	DS
	ASSUME	DS:CSEG
	MOV	status_reg,ax
	CMP	AX,3BAh			; Is this a MONO display
	JNE	color			; If not, must be CGA
	MOV	WORD PTR hwait,no_desnow	; Get rid of desnow
	JMP	SHORT move_stack
color:	MOV	video_seg,0B800h	; Segment for color card
	XOR	BH,BH			; Get current attribute
	MOV	AH,08h
	INT	10h
	MOV	normal,AH
	XOR	AH,1110111b		; Flip the color bits
	MOV	inverse,AH
move_stack:
	MOV	BX,OFFSET new_stack	; Move the stack downward
	MOV	SP,BX
	ADD	BX,15
	MOV	CL,04h
	SHR	BX,CL			; Convert size to paragraphs
	MOV	AH,4Ah
	INT	21h			; Deallocate unused memory
get_pars:
	MOV	SI,80h			; Point to command line parameters
	MOV	CL,[SI]			; Get number of characters
	XOR	CH,CH			; Convert to word
	INC	SI			; Point to first character
	PUSH	SI
	ADD	SI,CX			; Point to one after last character
	MOV	BYTE PTR [SI],00H	; Make it an ASCIIZ string
	MOV	name_end,SI		; Save pointer to last character
	POP	SI			; Get back pointer to filename
	MOV	name_pointer,SI		; Save (temporary) pointer to filename
	CLD				; Forward direction
	MOV	BX,file_seg_size	; Default file segment size
	JCXZ	do_size			; If no parameters, just default
skip_blanks:
	LODSB				; Get character into AL
	CMP	AL," "			; Blank?
	JNE	found_char
	LOOP	skip_blanks
found_char:
	CMP	AL,"/"			; /size parameter?
	JNE	adjust_SI		; Must be filename!
	LODSB				; Get next character
	CMP	AL,"s"			; Small?
	JE	s_small
	CMP	AL,"S"
	JNE	not_small
s_small:
	MOV	BX,0400h		; 16 384 bytes file segment
	JMP	SHORT do_size
not_small:
	CMP	AL,"m"			; Medium?
	JE	s_medium
	CMP	AL,"M"
	JNE	not_medium
s_medium:
	MOV	BX,0800h		; 32 768 bytes file segment
	JMP	SHORT do_size
not_medium:
	CMP	AL,"l"			; Large?
	JE	s_large
	CMP	AL,"L"
	JNE	adjust_SI		; None of the above, so filename!
s_large:
	MOV	BX,1000h		; 65 536 bytes file segment
	JMP	SHORT do_size
adjust_SI:
	DEC	SI			; Adjust SI back one character
do_size:
	MOV	name_pointer,SI		; Store (temporary) pointer to file name
	MOV	file_seg_size,BX	; Store file segment size
	MOV	AH,48h
	INT	21h			; File segment memory request
	JC	not_enough
	MOV	ES,AX			; Stash into ES for immediate use
	ASSUME	ES:file_seg
	MOV	AH,48h
	INT	21h			; Paste buffer memory request
	JC	not_enough
	MOV	paste_seg,AX		; Stash into storage for future use
	MOV	CL,04h
	SHL	BX,CL			; Convert size to bytes
	DEC	BX
	DEC	BX			; Calculate maximum file size
	MOV	file_max_size,BX	; and store it
	JMP	SHORT get_filename

not_enough:
	MOV	DX,OFFSET memory_error
err_exit:
	PUSH	CS
	POP	DS			; Repoint DS: to this segment
	MOV	AH,09h			; Write the error message @[DX]
	INT	21h
	JMP	exit_to_dos		; Exit on allocation or file error

get_filename:
	MOV	CX,name_end		; Get back pointer to last character
	MOV	SI,name_pointer		; Get back (temporary) ptr to filename
	SUB	CX,SI			; Re-calculate number of characters
	CLD				; Forward direction
	JCXZ	no_filename		; If no filename, no need to read
del_spaces:
	LODSB				; Get character into AL
	CMP	AL," "			; Blank?
	JNE	found_letter
	LOOP	del_spaces
found_letter:
	DEC	SI			; Backup to first letter
	MOV	name_pointer,SI		; Save (final) pointer to filename
	MOV	DX,SI
	MOV	AX,3D00h		; Setup to open file
	INT	21h
	JC	no_filename		; If can't open, must be a new file
file_opened:
	PUSH	ES
	POP	DS			; DS has file segment also
	ASSUME	DS:file_seg
	MOV	BX,AX			; Get file handle into BX
	XOR	DX,DX			; Point to file buffer
	MOV	AH,3Fh			; Read service
	MOV	CX,file_max_size	; Read up to maximum file size
	INT	21h
	MOV	DI,AX			; Actual number of bytes read in
	JNC	look_for_eof		; If no error, take jump
	MOV	DX,OFFSET read_err_mess ; Display error
	JMP	SHORT err_exit
look_for_eof:
	CMP	BYTE PTR [DI-1],1Ah	; Was last character an EOF?
	JNE	no_eof			; If not, must be ok
	DEC	DI			; Backup to last character
no_eof: MOV	last_char,DI		; Save the file size
	CMP	CX,AX			; Did the buffer fill?
	MOV	DX,OFFSET file_too_big	; Yes, file too big
	JE	err_exit
	MOV	AH,3Eh			; Close file service
	INT	21h
no_filename:
	PUSH	ES
	PUSH	ES			; Stash file segment
	MOV	AX,3524h		; Get INT 24h service
	INT	21h
	MOV	WORD PTR oldint24,BX	; Store offset
	MOV	WORD PTR oldint24+2,ES	; and segment
	PUSH	CS
	POP	DS
	MOV	DX,OFFSET newint24	; Point to new vector
	MOV	AX,2524h		; Change INT 24h service
	INT	21h
	MOV	DX,OFFSET newint23	; Change INT 24h service
	MOV	AX,2523h
	INT	21h
	POP	ES			; Unstash file segment
	POP	DS
	ASSUME	DS:file_seg, ES:file_seg
	CALL	redo_prompt		; Draw the prompt line
;
; Main program loop; updating the screen then reading a keystroke and again
;
read_a_key:
	CMP	mark_mode,00h		; If the mark state on?
	JE	mark_off		; No, skip
	OR	dirty_bits,04h		; Refresh the current row
	MOV	DX,cur_posn
	CMP	save_row,DH		; Are we on the save row?
	JE	same_row		; Yes, refresh the row
	MOV	dirty_bits,01h		; No, refresh the whole screen
same_row:
	MOV	AX,cursor		; Get cursor location
	MOV	BX,mark_home		; Get the anchor mark position
	CMP	AX,BX			; Moving backward in file?
	JAE	s1
	MOV	mark_start,AX		; Yes, switch start and end positions
	MOV	mark_end,BX
	JMP	SHORT mark_off
s1:	MOV	mark_end,AX		; Store start and end marks
	MOV	mark_start,BX
mark_off:
	MOV	DX,cur_posn
	MOV	save_row,DH
	CALL	set_cursor		; Position the cursor
	TEST	dirty_bits,01h		; Check for "dirty" screen
	JZ	screen_ok		; Zero, OK
	MOV	AH,01h			; Get keyboard status
	INT	16h			; Any key ready?
	JNZ	current_ok		; Yes, skip update
	CALL	display_screen		; No, refresh screen
	MOV	dirty_bits,00h		; Clear "dirty"
screen_ok:
	TEST	dirty_bits,04h		; Is the current line dirty?
	JZ	current_ok		; No, skip update
	CALL	display_current		; Resfresh screen
	MOV	dirty_bits,00h		; Clear "dirty"
current_ok:
	XOR	AH,AH			; Read the next key
	INT	16h
	OR	AL,AL			; Extended code?
	JZ	extended_code
	CMP	Ah,0Eh			; Backspace?
	JE	back_space
	CALl	insert_key		; Character
	JMP	read_a_key		; Continue getting keystrokes
back_space:
	CMP	cursor,00h		; At start of file?
	JE	bad_key			; Yes, can't
	CALL	left			; Move left one space
	CALL	del_char		; and delete character to right
	JMP	read_a_key		; Continue getting keystrokes
extended_code:
	CMP	AH,84h			; CTRL PgUP?
	JNE	not_top
	CALL	top
	JMP	read_a_key
not_top:
	CMP	AH,76h			; CTRL PgDOWN?
	JNE	not_bottom
	CALL	bottom
bad_key:
	JMP	read_a_key
not_bottom:
	CMP	AH,73h			; CTRL Left?
	JE	sh_left
	CMP	AH,74h			; CTRL Right?
	JE	sh_right
	CMP	AH,53h			; Skip high numbered keys
	JA	bad_key
	XCHG	AH,AL
	SUB	Al,3Bh			; and low numbered keys
	JC	bad_key
	SHL	AX,1h			; From byte to offset
	MOV	BX,AX			; As an index
	CALL	CS:dispatch_table[BX]	; and dispatch function
	JMP	read_a_key		; Continue reading keystrokes
;
; Shift the display right or left according to the length of the line over 80
; columns
;
sh_right	Proc	Near
	CMP	left_margin,255 - 8	; Past maximum right margin?
	JAE	no_shift
	ADD	left_margin,08h		; Move margin over
sh_return:
	CALL	cursor_col		; Compute column for cursor
	MOV	DX,cur_posn
	MOV	save_column,DL		; Save the current column
	MOV	dirty_bits,01h		; Redraw the screen
no_shift:
	JMP	read_a_key
sh_right	Endp
;
sh_left		Proc	Near
	CMP	left_margin,00h		; At start of line?
	JE	no_shift
	SUB	left_margin,08h		; Move margin (window) over
	JMP	sh_return
sh_left		Endp
;
; This routine moves the cursor to the top of the file
;
top		Proc	Near
	XOR	AX,AX			; Zero
	MOV	cursor,AX		; Cursor to start of file
	MOV	top_of_screen,AX	; Top line
	MOV	left_margin,AL		; Left column
	MOV	dirty_bits,01h		; Force a screen refresh
	MOV	cur_posn,AX		; Home the cursor
	MOV	save_column,AL		; Save current column
	RET
top		Endp
;
; This routine moves the cursor to the bottom of the file
;
bottom		Proc	Near
	MOV	DH,rows			; Get screen size
	MOV	SI,last_char		; Point to last character
	MOV	left_margin,00h		; Set window to start of line
	CALL	locate			; Adjust screen position
	CALL	home			; Move cursor to start of line
	MOV	dirty_bits,01h		; Force a screen refresh
	RET
bottom		Endp
;
; This routine deletes from the cursor position to the end of line
;
del_eol		Proc	Near
	MOV	line_flag,00h
	PUSH	cursor			; Save starting cursor location
	CALL	endd			; Move to the end of line
	POP	SI			; Get back starting cursor
	MOV	CX,cursor		; Offset of the end of line
	MOV	cursor,SI		; Restore starting cursor
	JMP	del_end			; Delete characters to end
del_eol Endp
;
; This routine deletes a line, placing it in the line buffer
;
del_l		Proc	Near
	MOV	line_flag,01h
	CALL	find_start		; Find start of this line
	MOV	cursor,SI		; This will be the new cursor
	PUSH	SI			; Stash it
	CALL	find_next		; Find the next line
	MOV	CX,SI			; CX will hold line length
	POP	SI			; Unstash cursor
del_end:
	SUB	CX,SI			; Number of bytes on line
	OR	CH,CH			; Is line too long to fit?
	JZ	not_too_long
	MOV	CX,100h			; Only save 256 characters
not_too_long:
	MOV	line_length,CX		; Store length of deleted line
	JCXZ	no_del_l
	MOV	DI,OFFSET line_buffer	; Buffer for deleted line
	PUSH	CX
	PUSH	ES
	PUSH	CS
	POP	ES			; Line buffer is in CSEG
	REP	MOVSB			; Put deleted line in buffer
	POP	ES			; Get back into file segment
	POP	AX
	MOV	CX,last_char		; Get the file size
	SUB	last_char,AX		; Substract the deleted line
	MOV	SI,cursor		; Get new cursor position
	MOV	DI,SI
	ADD	SI,AX			; SI points to end of file
	SUB	CX,SI			; Length of remaining file
	JCXZ	no_del_l
	REP	MOVSB			; Shift remainder of file up
no_del_l:
	MOV	DX,cur_posn		; Get cursor row/column
	MOV	SI,cursor		; Get cursor offset
	CALL	locate			; Adjust the screen position
	MOV	dirty_bits,01h		; Force a screen refresh
	RET
del_l		Endp
;
; This undeletes a line by copying it from the line buffer into the file
;
udel_l		Proc	Near
	CMP	line_flag,00h		; Is this an end of line only?
	JE	udel_eol		; Yes, don't home the cursor
	CALL	home
udel_eol:
	MOV	AX,line_length		; Length of deleted line
	MOV	SI,OFFSET line_buffer
	JMP	insert_string
udel_l		Endp
;
; These routines moves the cursor left and right
;
left		Proc	Near
	CMP	cursor,00h		; At start of file?
	JZ	lr_no_change		; Yes, can't move
	MOV	DX,cur_posn
	OR	DL,DL			; At first column?
	JZ	move_up			; Yes, move up one!
	DEC	cursor			; Shift the cursor offset
lr_return:
	CALL	cursor_col		; Compute column for cursor
	MOV	save_column,DL		; Save the cursor column
lr_no_change:
	MOV	undo_length,00h
	RET
move_up:
	CALL	up
	JMP	SHORT ENDD
left	Endp
;
right		Proc	Near
	MOV	SI,cursor
	CMP	SI,last_char		; At end of file?
	JE	lr_no_change		; Yes, can't move
	CMP	BYTE PTR [SI],CR	; At end of a line?
	JE	move_down		; Yes, move down one!
	INC	cursor			; Advance cursor
	JMP	lr_return
move_down:
	CALL	home			; Move to start of line
	JMP	down			; And move down one row
right		Endp
;
; This moves the cursor to the start of the current line
;
home		Proc	Near
	CALL	find_start		; Find start of line
	MOV	cursor,SI		; Save the new cursor
	MOV	save_column,00h		; Save the cursor column
	MOV	BYTE PTR cur_posn,00h	; Store column number
	RET
home		Endp
;
; This moves the cursor to the end of the current line
;
endd		Proc	Near
	MOV	SI,cursor
	CALL	find_eol		; Find end of line
	MOV	cursor,SI		; Save the new cursor
	CALL	cursor_col		; Compute correct column
	MOV	save_column,DL		; Save the cursor column
	RET
endd		Endp
;
; This moves the cursor up one row.  If the cursor is at the first row,
; the screen is scrolled down.
up		Proc	Near
	MOV	undo_length,00h
	MOV	DX,cur_posn
	MOV	SI,cursor
	OR	DH,DH			; At top row already?
	JZ	screen_dn		; Yes, scroll down
	DEC	DH			; Move cursor one row
	CALL	find_cr			; Find the beginning of this row
	MOV	cursor,SI
	CALL	find_start		; Find the start of this row
	MOV	cursor,SI
	CALL	shift_right		; Skip over the current column
at_top: RET
screen_dn:
	MOV	SI,top_of_screen
	OR	SI,SI			; At top of file?
	JZ	at_top			; Yes, do nothing
	CALL	find_previous		; Find the preceeding line
	MOV	top_of_screen,SI	; Save new top of screen
	MOV	SI,cursor
	CALL	find_previous		; Find the preceeding line
	MOV	cursor,SI		; This is the new cursor
shift_ret:
	MOV	dirty_bits,01h		; Force screen redraw
	MOV	SI,cursor
	MOV	DX,cur_posn
	JMP	shift_right		; Move cursor to current column
up		Endp
;
; This moves the cursor down one row.  When the last row is reached,
; the screen is shifted up one row.
;
down		Proc	Near
	MOV	undo_length,00h
	MOV	DX,cur_posn
	CMP	DH,rows			; At bottom row already?
	MOV	SI,cursor		; Get position in file
	JE	screen_up		; If at bottom, then scroll up
	CALL	find_next		; Find start of next line
	JC	down_ret		; Return if no more lines
	MOV	cursor,SI
	INC	DH			; Advance cursor to next row
	CALL	shift_right		; Move cursor to current column
down_ret:
	RET
screen_up:
	CMP	SI,last_char		; Get cursor offset
	JE	down_ret
	CALL	find_start		; Find the start of this line
	MOV	cursor,SI		; This is the new cursor
	CALL	find_next		; Find the offset of next line
	JC	shift_ret		; Return if no more lines
	MOV	cursor,SI		; This is the new cursor
	MOV	SI,top_of_screen	; Get start of the top row
	CALL	find_next		; And find the next line
	MOV	top_of_screen,SI	; Store the new top of screen
	JMP	shift_ret
down	Endp
;
; These two routines move th screen one page at a time by calling the UP
; and DOWN pprocedures.
;
pgdn		Proc	Near
	MOV	page_proc,OFFSET down
page_up_dn:
	MOV	CL,rows			; Get length of the screen
	SUB	CL,5			; Don't page a full screen
	XOR	CH,CH			; Convert to word
page_loop:
	PUSH	CX
	CALL	page_proc		; Move the cursor down
	POP	CX
	LOOP	page_loop		; Loop for one page length
	RET
pgdn		Endp

pgup		Proc	Near
	MOV	page_proc,OFFSET up	; Same story for up
	JMP	page_up_dn
pgup		Endp
;
; This toggles the insert/overstrike mode.
;
insert		Proc	Near
	NOT	insert_mode		; Toggle the switch
	JMP	redo_prompt		; Refresh the insrt status
insert		Endp
;
; This deletes the character at the cursor by shifting the remaining
; characters forward.
;
del_char	Proc	Near
	MOV	CX,last_char
	MOV	SI,cursor
	MOV	DI,SI
	CMP	SI,CX			; At end of file?
	JAE	no_del			; Yes, can't delete
	LODSB
	CALL	save_char		; Save if for UNDO function
	MOV	AH,[SI]			; Look at the next character also
	PUSH	AX			; Save character were deleting
	DEC	last_char		; Shorten file by one character
	SUB	CX,SI
	REP	MOVSB			; Move file down by one
	POP	AX			; Get back character just deleted
	CMP	AL,CR			; Deleted a CR?
	JE	combine			; Combine the two lines
	OR	dirty_bits,04h		; Force a refresh of current line
no_del: RET
combine:
	CMP	AH,LF			; Was next character a LF?
	JNE	no_del_lf
	CALL	del_char		; Delete the line feed
no_del_lf:
	CALL	display_bottom		; Refresh the bottom of the screen
	MOV	DX,cur_posn
	MOV	save_column,DL		; Save the cursor column
	RET
del_char	Endp
;
; This toggles the mark state and resets the past buffer pointers.
;
mark		Proc	Near
	XOR	AX,AX
	NOT	mark_mode		; Toggle the mode flag
	CMP	mark_mode,AL		; Turning it on?
	JNE	mark_on
	MOV	dirty_bits,01h		; Force a refresh of the screen
	MOV	mark_start,0FFFFh	; Re-intialize upper mark boundary
	JMP	SHORT mark_ret
mark_on:
	MOV	AX,cursor		; Get the cursor offset
	MOV	mark_start,AX		; Start of marked range
mark_ret:
	MOV	mark_end,AX		; End of marked range
	MOV	mark_home,AX		; Center of marked range
	RET
mark		Endp
;
; This removes the marked text and places it in the paste buffer.
;
cut		Proc	Near
	CMP	mark_mode,00h		; Is the mark mode on?
	JE	no_mark			; No, do nothing
	MOV	CX,mark_end		; Get end of mark region
	MOV	SI,mark_start		; Get start of mark region
	SUB	CX,SI			; Number of bytes selected
	MOV	paste_size,CX
	JCXZ	no_mark
	XOR	DI,DI			; Point to past buffer
	PUSH	CX
	PUSH	ES
	MOV	ES,paste_seg		; Get the paste segment
	REP	MOVSB			; Put deleted text in buffer
	POP	ES
	POP	AX
	MOV	CX,last_char
	SUB	last_char,AX		; Shorten the file this muchs
	MOV	DI,mark_start
	MOV	SI,mark_end
	SUB	CX,SI
	JCXZ	no_delete
	REP	MOVSB			; Shorten the file
no_delete:
	MOV	DX,cur_posn
	MOV	SI,mark_start
	CALL	locate			; Adjust the screen position
	CALL	mark			; This turns off the mark mode
no_mark:	RET
cut		Endp
;
; This copies the paste buffer into the file at the cursor location.
;
paste		Proc	Near
	MOV	AX,paste_size		; Number of characters in buffer
	OR	AX,AX			; Any?
	JZ	no_paste		; No, do nothing
	MOV	SI,cursor		; Get cursor location
	PUSH	AX
	PUSH	SI
	CALL	open_space		; Make room for new characters
	POP	DI
	POP	CX
	JC	no_paste		; If no room, just exit
	XOR	SI,SI			; Point to paste buffer
	PUSH	DS
	MOV	DS,paste_seg		; Segment of past buffer
	REP	MOVSB			; Copy in the new characters
	POP	DS
	MOV	SI,DI
	MOV	cursor,SI		; Cursor moved to end of insert
	MOV	DX,cur_posn		; Get current cursor row
	CALL	locate			; Adjust the screen position
	MOV	dirty_bits,01h		; Force a refresh of the screen
no_paste:
	RET
paste		Endp
;
; This prints the marked text.  It is canceled if printer fails.
;
print		Proc	 Near
	CMP	mark_mode,00h		; Mark mode on?
	JE	print_ret		; No, do nothing
	MOV	CX,mark_end		; End of marked region
	MOV	SI,mark_start		; Start of marked region
	SUB	CX,SI			; Number of bytes selected
	JCXZ	print_done		; If nothing to print, return
	MOV	AH,02h
	XOR	DX,DX			; Select printer 0
	INT	17h			; Get printer status
	TEST	AH,10000000B		; Is busy bit set?
	JZ	print_done
	TEST	AH,00100000B		; Is printer out of paper?
	JNZ	print_done
print_loop:
	LODSB
	XOR	AH,AH
	INT	17h			; Print the character
	ROR	AH,1			; Check time out bit
	JC	print_done		; Set? quit printing
	LOOP	print_loop
	MOV	AL,CR			; Finish with a CR
	XOR	AH,00h
	INT	17h
print_done:
	CALL	 mark			; Turn off the mark state
print_ret:
	RET
print		Endp
;
; This command restore any characters which have been currently been deleted.
;
undo	 	Proc	 Near
	XOR	AX,AX
	XCHG	AX,undo_length		; Get buffer length
	MOV	SI,OFFSEt undo_buffer
	JMP	insert_string
undo		Endp
;
; This inserts AX characters from CS:SI into the file.
;
insert_string	Proc	 Near
	PUSH	SI			; Save string buffer
	MOV	SI,cursor		; Get cursor offset
	PUSH	AX			; Save length of string
	PUSH	SI
	CALL	open_space		; Make space to insert string
	POP	DI			; Get back cursor position
	POP	CX			; Get back string length
	POP	SI			; Get back string buffer
	JC	no_space		; No space?, exit
	PUSH	DS
	PUSH	CX
	POP	DS
	ASSUME	DS:CSEG
	REP	MOVSB			; Copy the characters in the file
	MOV	SI,cursor		; Get the new cursor offset
	MOV	DX,cur_posn		; Also get the current row
	MOV	dirty_bits,01h		; Force a screen refresh
	POP	DS
	ASSUME	DS:nothing
	CALL	locate			; Adjust screen position
no_space:
	RET
insert_string	Endp
;
; This adds a character to the undo buffer.
;
save_char	Proc	 Near
	MOV	BX,undo_length
	OR	BH,BH			; Is buffer filled?
	JNZ	no_save			; No space?, exit
	INC	undo_length
	MOV	BYTE PTR CS:undo_buffer[BX],AL
no_save:
	RET
save_char	Endp
;
; This prompts for a verify keystroke then exits without saving the file
;
abort		Proc	 Near
	ASSUME	DS:CSEG
	PUSH	CS
	POP	DS
	MOV	DH,rows			; Last row on display
	INC	DH			; Bottom row of screen
	XOR	DL,DL			; First column
	MOV	SI,OFFSET verify_mess
	CALL	tty_string		; Display verify message
	XOR	AH,AH			; Read the next key
	INT	16h			; BIOS read key routine
	OR	AL,32			; Cover to lower case
	CMP	AL,"y"			; Was yes?
	JE	finished		; Yes, then finished
	CALL	redo_prompt		; No, refresh the prompt
	PUSH	ES
	POP	DS			; Set DS back to file segment
	RET
finished:
	MOV	DH,rows			; Move the last row on screen
	XOR	DL,DL			; And column zero
	CALL	set_cursor
	INC	DH
	CALL	erase_eol		; Erase the last row
exit_to_dos:
	PUSH	CS
	POP	DS			; Point to code segment
	MOV	DX,OFFSET copyright
	MOV	AH,09h			; Display string
	INT	21h
	MOV	AX,4C00h		; Exit!
	INT	21h
abort		Endp
;
; This prompts for a filename then writes the file.  The original file is
; renamed to filename.bak.  If an invalid filename is entered, the speaker
; is beeped.
;
exit	 	Proc	 Near
	PUSH	DS
	PUSH	ES
	MOV	AX,CS
	MOV	DS,AX
	MOV	ES,AX
	ASSUME	DS:CSEG, ES:CSEG
next_letter:
	MOV	DH,rows
	INC	DH			; Last row on the screen
	XOR	DL,DL			; First column
	MOV	SI,OFFSET save_mess
	PUSH	DX
	CALL	tty_string		; Display file name prompt
	POP	DX
	ADD	DL,09h			; Move right 9 spaces
	MOV	SI,name_pointer
	CALL	tty_string		; Display the filename
	XOR	AH,AH			; Read the next key
	INT	16h
	MOV	DI,name_end		; This points to last letter
	OR	AL,AL			; Is it a real character?
	JZ	next_letter		; Ignore extended keys
	CMP	AL,27			; ESC?
	JNE	not_escape
	MOV	dirty_bits,01h		; Force a screen refresh
	POP	ES
	POP	DS			; Get back file segments
	JMP	redo_prompt		; Refresh the prompt
not_escape:
	CMP	AL,13			; CR? (0Dh?)
	JE	got_name
	CMP	AL,08h			; BS?
	JNE	normal_letter
	CMP	DI,name_pointer		; At first letter?
	JLE	next_letter		; Yes, don't erase it
	MOV	BYTE PTR [DI-1],00h
	DEC	name_end
	JMP	next_letter
normal_letter:
	CMP	DI,81h + 65		; Too many letters?
	JG	next_letter		; Yes, ignore them
	XOR	AH,AH
	STOSW				; Store the new letter
	INC	name_end		; Name is one character longer
	JMP	next_letter		; Read another keystroke
got_name:
	MOV	DX,name_pointer		; Point to the filename
	MOV	AX,4300h		; Get the file attributes
	INT	21h
	JNC	name_ok			; No error, must be good name
	CMP	AX,03h			; Path not found?
	JE	bad_name		; Yes, bad file name
name_ok:
	MOV	SI,OFFSET dot_$$$	; Point to the ".$$$"
	MOV	DI,OFFSET name_dot_$$$
	CALL	chg_extension		; Add the new extension
	MOV	DX,OFFSET name_dot_$$$	; Point to the temp filename
	MOV	AH,3Ch			; Create file function
	MOV	CX,0020h		; Attributes for new file
	INT	21h			; Attempt creation
	JNC	name_was_ok		; No problem, continue?
bad_name:
	MOV	AX,0E07h		; Sound the bell via teletype write
	INT	10h
	JMP	next_letter		; Get another letter
write_error:
	MOV	AH,3Eh			; Close file function
	INT	21h
	JMP	bad_name		; File name must be bad
name_was_ok:
	XOR	DX,DX			; This is the file buffer
	MOV	CX,last_char		; Number of bytes in file
	MOV	DI,CX
	MOV	BX,AX			; This is the file handle
	MOV	AH,40h			; Write file
	POP	DS			; Unstash buffer segment
	CMP	CX,file_max_size	; Buffer already full (0xxxEh)?
	JA	dont_add_eof		; Yes, don't add EOF marker
	INC	CX			; Plus one character for EOF marker
	MOV	BYTE PTR [DI],1Ah	; Add it
dont_add_eof:
	INT	21h			; Write the buffer contents
	POP	DS
	JC	write_error		; Exit on write error
	CMP	AX,CX			; Was all file written?
	JNE	write_error		; No, exit
	PUSH	CS
	POP	DS			; Get the code segment
	MOV	AH,3Eh			; Close file function (temporary file)
	INT	21h
	MOV	SI,OFFSET dot_bak	; Point to the ".BAK"
	MOV	DI,OFFSET name_dot_bak
	CALL	chg_extension		; Make the backup filename
	MOV	DX,OFFSET name_dot_bak	; Point to the backup name
	MOV	AH,41h			; Delete file function (backup)
	INT	21h
	MOV	DI,OFFSET name_dot_bak
	MOV	DX,name_pointer
	MOV	AH,56h
	INT	21h
	MOV	DI,name_pointer		; Point to new filename
	MOV	DX,OFFSET name_dot_$$$	; Point to temporary file
	MOV	AH,56h			; Rename file function
	INT	21h
	POP	AX			; Restore stack
	POP	AX
	JMP	finished
exit		Endp
;
; This subroutine displays a character by writing directly to the screen
; buffer.  To avoid screen noise (snow) on the color card, the horizontal
; retrace has to be monitored.
;
write_inverse	Proc	 Near
	ASSUME	DS:file_seg, ES:file_seg
	MOV	BH,inverse		; Attribute for inverse video
	JMP	SHORT write_screen
write_normal:
	MOV	BH,normal
write_screen:
	MOV	BL,AL			; Save character
	PUSH	ES
	MOV	DX,status_reg		; Retrieve status register
	MOV	ES,video_seg		; Point to video buffer segment
hwait:	IN	AL,DX			; Get video status
	ROR	AL,1			; Look for horizontal retrace
	JNC	hwait			; Wait for it!
write_it:
	MOV	AX,BX			; Get the character/attribute
	STOSW				; Write it!
	POP	ES
	RET
write_inverse	Endp
;
; This moves the cursor to the row/column in DX.
;
set_cursor	Proc	 Near
	XOR	BH,BH			; Forces page zero (first page)
	MOV	AH,02h			; BIOS set cursor function
	INT	10h
	RET
set_cursor	Endp
;
; This computes the video offset for the row/column in DX.
;
position	Proc	 Near
	MOV	AX,columns		; Take columns per row
	MUL	DH			; Times row number
	XOR	DH,DH
	ADD	AX,DX			; Add in the column number
	SHL	AX,1			; Times 2 for offset
	MOV	DI,AX			; Return in DI
	RET
position	Endp
;
; This erases from the location in DX to the right edge of the screen.
;
erase_eol	Proc	 Near
	CALL	position		; Find screen offset
	MOV	CX,columns		; Get screen size
	SUB	CL,DL			; Subtract current position
	JCXZ	no_clear
erase_loop:
	MOV	AL," "			; Write blanks to erase
	CALL	write_normal		; Display it!
	LOOP	erase_loop		; All?
no_clear:
	RET
erase_eol	Endp
;
; This displays the function keys prompt and insert mode state.
;
redo_prompt	Proc	 Near
	ASSUME	DS:nothing, ES:nothing
	PUSH	DS
	PUSH	CS
	POP	DS
	ASSUME	DS:CSEG
	MOV	DH,rows			; Put prompt at last row
	INC	DH
	XOR	DL,DL			; and column 0
	CALL	position		; Convert to screen offset
	MOV	SI,OFFSET prompt_string
key_loop:
	MOV	AL,"F"			; Display an "F"
	CALL	write_normal
	LODSB
	OR	AL,AL			; Last key in prompt?
	JZ	prompt_done
	CALL	write_normal
	CMP	BYTE ptr CS:[SI],"0"	; Is if F10?
	JNE	text_loop
	LODSB
	CALL	write_normal
text_loop:
	LODSB
	OR	AL,AL			; Last letter in word?
	JNZ	write_char
	MOV	AL," "			; Display a space
	CALL	write_normal
	JMP	key_loop
write_char:
	CALL	write_inverse		; Display the letter
	JMP	text_loop		; Do the next letter
prompt_done:
	MOV	DH,rows
	INC	DH			; Get the last row on screen
	MOV	DL,prompt_length+9
	CALL	erase_eol		; Erase to the end of this row
	MOV	AL,"O"			; Write an "O"
	CMP	insert_mode,00h		; Is insert mode?
	JE	overstrike
	MOV	AL,"I"			; Write an "I"
overstrike:
	DEC	DI			; Backup one character position
	DEC	DI
	CALL	write_normal
	POP	DS
	RET
redo_prompt	Endp
;
; This displays the file buffer on the screen.
;
display_screen	Proc	 Near
	ASSUME	DS:file_seg, ES:file_seg
	MOV	SI,top_of_screen	; Point to first char on screen
	XOR	DH,DH			; Start at first row
	JMP	SHORT next_row
display_bottom:
	CALL	find_start		; Find first character on this row
	MOV	DX,cur_posn		; Get current cursor row
next_row:
	PUSH	DX
	CALL	display_line		; Display a line
	POP	DX
	INC	DH			; Move to the next row
	CMP	DH,rows			; At end of screen yet?
	JBE	next_row		; Do all the rows
	RET
display_screen	Endp
;
; This subroutine displays a single line to the screen.  DH holds the row
; number, SI has the offset into the file buffer.  Tabs are expanded.
; Adjustment is made for side shift.
;
display_current Proc	 Near
	CALL	find_start
	MOV	DX,cur_posn
display_line:
	XOR	DL,DL			; Start at column zero
	MOV	margin_count,DL
	MOV	CX,DX			; Use CL to count the columns
	CALL	position		; Compute offset into video
next_char:
	CMP	SI,last_char		; At end of file?
	JAE	line_done
	LODSB				; Get next character
	CMP	AL,CR			; CR?
	JE	found_cr		; Yes, done!
	CMP	AL,TAB			; TB?
	JE	expand_tab		; Yes, expand!
	CALL	put_char		; Put character onto screen
tab_done:
	CMP	CL,columnsb		; At right edge of screen?
	JB	next_char
	CMP	BYTE PTR [SI],CR	; CR?
	JE	not_beyound
	DEC	DI			; Backup one character
	DEC	DI
	MOV	AL,4			; Show a diamond
	CALL	write_inverse		; in inverse video
not_beyound:
	JMP	find_next		; Find start of next line
found_cr:
	LODSB				; Look at the next character
	CMP	AL,LF			; LF?
	JE	line_done
	DEC	SI
line_done:
	MOV	DX,CX
	JMP	erase_eol		; Erase the rest of the line
expand_tab:
	MOV	AL," "			; Convert TB to spaces
	CALL	put_char
	MOV	AL,margin_count
	ADD	AL,CL
	TEST	AL,00000111b		; At even multiple of eight?
	JNZ	expand_tab		; No, keep expanding
	JMP	tab_done
display_current Endp
;
; This displays a single character to the screen.  If the character is marked,
; it is shown in inverse video.  Characters outside the current margin are
; not displayed.  Characters left of the margin are skipped.
;
put_char	Proc	 Near
	MOV	BL,margin_count		; Get distance to left margin
	CMP	BL,left_margin		; Are we inside left margin
	JAE	in_window		; Yes, show the character
	INC	BL
	MOV	margin_count,BL
	RET
in_window:
	CMP	SI,mark_start		; Marked character?
	JBE	not_marked
	CMP	SI,mark_end
	JA	not_marked
	CALL	write_inverse		; Marked characters shown inverse
	JMP	SHORT next_col
not_marked:
	CALL	write_normal
next_col:
	INC	CL			; Increment the column count
	RET
put_char	Endp
;
; This routine adds a character into the file.  In insert mode, remaining
; characters are pushed forward.  If a CR is inserted, a LF is added also.
;
insert_key	Proc	 Near
	MOV	SI,cursor
	CMP	AL,CR			; Was CR?
	JE	new_line		; Yes, generate new line
	MOV	SI,cursor
	CMP	insert_mode,00h		; In insert mode?
	JNE	insert_char
	CMP	BYTE PTR [SI],CR	; At end of line?
	JE	insert_char
	CMP	SI,last_char		; At end of file?
	JE	insert_char
	MOV	DI,SI
	XCHG	DS:[SI],AL		; Switch new character for old one
	CALL	save_char		; Store the old character
	JMP	SHORT advance
insert_char:
	PUSH	SI
	PUSH	AX			; Save the new character
	MOV	AX,1
	CALL	open_space		; Make room for it
	POP	AX			; Get back the new character
	POP	DI
	JC	file_full
	STOSB				; Insert it in file buffer
advance:
	OR	dirty_bits,04h		; Force a refresh of current line
	PUSH	undo_length
	CALL	right			; Move cursor to next letter
	POP	undo_length
file_full:
	RET
new_line:
	PUSH	 SI
	MOV	AX,2
	CALL	open_space		; Make space for CR and LF
	POP	DI			; Get back old cursor location
	JC	file_full
	MOV	AX,LF*256+CR
	STOSW				; Store the CR and LF
	CALL	display_bottom		; Refresh bottom of the screen
	CALL	home			; Cursor to start of line
	JMP	down			; Move down to the new line
insert_key	Endp
;
; This subroutine inserts spaces into the file buffer.  On entry AX contains
; the number of spaces to be inserted.  On return, CF=1 if there was not
; enough space in the file buffer.
;
open_space	Proc	Near
	MOV	CX,last_char		; Last character in the file
	MOV	SI,CX
	MOV	DI,CX
	ADD	DI,AX			; Offset for new end of file
	JC	no_room			; Carry! No more room for 64k segment
	CMP	DI,file_seg_size	; No more room? (8, 16, or 32k)
	JA	no_room			; Check if carry for JA?
	MOV	last_char,DI		; Save offset of end of file
	SUB	CX,cursor		; Number of characters to shift
	DEC	DI
	DEC	SI
	STD				; String moves goes forward
	REP	MOVSB			; Do it, shifting file upward
	CLD				; Restore forward direction
	CLC
no_room:
	RET
open_space	Endp
;
; This subroutine adjusts the cursor position ahead of the saved cursor
; column.  On entry DH has the cursor row.
;
shift_right	Proc	Near
	MOV	CL,save_column		; Keep the saved cursor offset
	XOR	CH,CH
	MOV	BP,CX			; Keep the saved cursor position
	ADD	CL,left_margin		; Shift into visible window also
	ADC	CH,00h
	XOR	DL,DL
	MOV	cur_posn,DX		; Get cursor row/column
	JCXZ	no_change
right_again:
	PUSH	CX
	CMP	BYTE PTR [SI],CR	; At end of line?
	JE	dont_move		; Stop moving
	CALL	right			; Move right one character
dont_move:
	POP	CX
	MOV	Al,save_column
	XOR	AH,AH
	CMP	AX,CX			; Is cursor still in margin?
	JL	in_margin		; Yes, keep moving
	MOV	DX,cur_posn		; Get cursor position again
	XOR	DH,DH
	CMP	DX,BP			; At saved cursor position
	JE	right_done		; Yes, done
	JA	right_too_far		; Too far?
in_margin:
	LOOP	right_again
right_done:
	MOV	CX,BP
	MOV	save_column,CL		; Get back left one place
no_change:
	RET
right_too_far:
	CALL	left			; Move back left one place
	MOV	CX,BP
	MOV	save_column,CL		; Get back saved cursor position
	RET
shift_right	Endp
;
; This subroutine skips past the CR and LF at SI.  SI returns new offset.
;
skip_cr_lf	Proc	Near
	CMP	SI,last_char		; At last character in file?
	JAE	no_skip			; Yes, can't skip
	CMP	BYTE PTR [SI],CR	; This first character is CR?
	JNE	no_skip			; No, can't skip
	INC	SI			; Examine next character
	CMP	SI,last_char		; At last character in file?
	JAE	no_skip			; Yes, can't skip
	CMP	BYTE PTR [SI],LF	; This second character is LF?
	JNE	no_skip			; No, can't skip
	INC	SI
no_skip:
	RET
skip_cr_lf	Endp
;
; This subroutine finds the beginning of the previous line.
;
find_previous	Proc	Near
	PUSH	cursor			; Save the cursor location
	CALL	find_cr			; Find start of this line
	MOV	cursor,SI		; Save the new cursor
	CALL	find_start		; Find start of this line
	POP	cursor			; Get back starting cursor
	RET
find_previous	Endp
;
; This searches for the previous carriage return.  Search starts as SI.
;
find_cr		Proc	Near
	PUSH	CX
	MOV	AL,CR			; Looking for CR
	MOV	DI,SI
	MOV	CX,SI
	JCXZ	at_beginning
	DEC	DI
	STD				; Search backwards
	REPNE	SCASB			; Scan for CR
	CLD				; Restore forward direction
	INC	DI
	MOV	SI,DI
at_beginning:
	POP	CX
	RET
find_cr		Endp
;
; The subroutine computes the location of the start of current line.  Returns
; SI pointing to the first character of the current line.
;
find_start	Proc	Near
	MOV	SI,cursor		; Get the current cursor
	OR	SI,SI			; At start of the file?
	JZ	at_start		; Yes, done!
	CALL	find_cr			; Find the CR
	CALL	skip_cr_lf
at_start:
	RET
find_start	Endp
;
; This finds the offset of the start of the next line.  The search is
; started at location ES:SI.  On return CF=1 if no CR was found.
;
find_next	Proc	Near
	PUSH	CX
	CALL	find_eol		; Find the end of this line
	JC	at_next			; If at end of file, return
	CALL	skip_cr_lf		; Skip past CR and LF
	CLC				; Indicate end of line found
at_next:
	POP	CX
	RET
find_next	Endp
;
; This searches for the next CR in the file.  The search starts at the
; offset in register SI.
;
find_eol	Proc	Near
	MOV	AL,CR			; Look for a CR
	MOV	CX,last_char		; Last character in the file
	SUB	CX,SI			; Count for the search
	MOV	DI,SI
	JCXZ	at_end			; If nothing to search, done
	REPNE	SCASB			; Scan for CR
	MOV	SI,DI			; Return the location just found
	JCXZ	at_end			; Not found, just return
	DEC	SI
	CLC				; Indicate CR was found!
	RET
at_end: STC				; Indicate CR was NOT found!
	RET
find_eol	Endp
;
; This subroutine positions the screen with the cursor at the row selected
; in register DH.  On entry, SI holds the cursor offset.
;
locate		Proc	Near
	MOV	CL,DH
	XOR	CH,CH
	MOV	cursor,SI
	XOR	DX,DX			; Start at top of the screen
	OR	SI,SI			; At start of buffer?
	JZ	locate_first
	CALL	find_start		; Get start of this row
	XOR	DX,DX			; Start at top of the screen
	OR	SI,SI
	JZ	locate_first		; Done if locating top row!
	JCXZ	locate_first
find_top:
	PUSH	SI
	PUSH	CX
	CALL	find_cr			; Find previous row
	POP	CX
	POP	AX
	CMP	BYTE PTR [SI],CR
	JNE	locate_first
	CMP	SI,AX			; Any change?
	JE	locate_done		; No, no need to keep trying
	INC	DH			; Cursor moves to next row
	LOOP	find_top
locate_done:
	PUSH	cursor
	MOV	cursor,SI
	CALL	find_start		; Find start of top of screen
	POP	cursor
locate_first:
	MOV	top_of_screen,SI
	MOV	cur_posn,DX
	CALL	cursor_col
	MOV	save_column,DL
	RET
locate		Endp
;
; This subroutine computes the correct column for the cursor.  No inputs.
; On exit, cur_posn is set and DX has the row/column.
;
cursor_col	Proc	Near
	MOV	SI,cursor		; Get cursor offset
	CALL	find_start		; Find start of this line
	MOV	CX,cursor
	SUB	CX,SI
	MOV	DX,cur_posn		; Get current row
	XOR	DL,DL			; Start at column zero
	MOV	margin_count,DL		; Count past the left margin
	JCXZ	col_done
cursor_loop:
	LODSB				; Get the next character
	CMP	AL,CR			; End of line?
	JE	col_done		; Yes, done!
	CMP	AL,TAB			; TAB?
	JNE	not_a_tab
	MOV	BL,margin_count
	OR	BL,00000111b
	MOV	margin_count,BL
	CMP	BL,left_margin		; Inside visible window yet?
	JB	not_a_tab		; No, don't advance cursor
	OR	DL,00000111b		; Move to multiple of eight
not_a_tab:
	MOV	BL,margin_count
	INC	BL
	MOV	margin_count,BL
	CMP	Bl,left_margin
	JBE	out_of_window
	INC	DL			; Now at next column
out_of_window:
	LOOP	cursor_loop
col_done:
	CMP	DL,columnsb		; Past end of display?
	JB	column_ok		; No, still ok
	MOV	DL,columnsb
	DEC	DL			; Leave cursor at last column
column_ok:
	MOV	cur_posn,DX		; Store the row/column
	RET
cursor_col	Endp
;
; This displays the string at CS:SI at the location DX.  The remainder of
; the row is erased.  Cursor is put at the end of the line.
;
tty_string	Proc	Near
	ASSUME	DS:CSEG
	PUSH	DX
	CALL	position		; Compute offset into video
	POP	DX
tty_loop:
	LODSB
	OR	AL,AL			; At end of string?
	JZ	tty_done
	INC	DL
	PUSH	DX
	CALL	write_inverse		; Write in inverse video
	POP	DX
	JMP	tty_loop
tty_done:
	CALL	set_cursor		; Move cursor to end of string
	JMP	erase_eol		; Erase the rest of line
tty_string	Endp
;
; This copies the input filename to CS:DI and changes the extension.
;
chg_extension	Proc	Near
	ASSUME	DS:CSEG, ES:CSEG
	PUSH	SI
	MOV	SI,name_pointer
chg_loop:
	LODSB
	CMP	AL,"."			; Look for the dot
	JE	found_dot
	OR	AL,AL
	JZ	found_dot
	STOSB				; Copy a character
	JMP	chg_loop
found_dot:
	MOV	CX,5			; Five characters in extension!
	POP	SI
	REP	MOVSB			; Copy new extension
	RET
chg_extension	Endp
;
; This is the control break handler.  It ignores the control break.
;
newint23	Proc	Far
	ASSUME	DS:nothing, ES:nothing
	MOV	CS:dirty_bits,01h	; Force a screen refresh
	CLC				; Tells DOS to ignore the break
	IRET
newint23	Endp
;
; This is the severe error handler.  It homes the cursor before processing
; the error.
;
newint24	Proc	Far
	ASSUME	DS:nothing, ES:nothing
	PUSHF
	PUSH	AX
	PUSH	BX
	PUSH	DX
	MOV	CS:dirty_bits,01h	; Force a screen refresh
	XOR	DX,DX
	CALL	set_cursor		; Put cursor at home
	POP	DX
	POP	BX
	POP	AX
	POPF
	JMP	CS:oldint24
newint24	Endp
;
	Even
name_dot_$$$	EQU	$
name_dot_bak	EQU	$ +  80h
undo_buffer	EQU	$ + 100h
line_buffer	EQU	$ + 200h
new_stack	EQU	$ + 500h
CSEG	ENDS
;
file_seg	SEGMENT
file_seg	ENDS
	END	start
