Program EditFont;

{
              FFE, an 8x16 bitmap font editor for FreeDOS

                Copyright (C) 2006 by Francesco Zamblera
                  under the GNU General Public License

                           vilnergoy@yahoo.it


    This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

    This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
}

USES     crt;

CONST    StartX           = 4;
         StartY           = 7;
         EndX             = 11;
         EndY             = 22;
         DisplayX         = 15 ;
         DisplayMenuY     = 11 ;
         DisplayCharY     = 8;
         DisplayStateY    = 9;
         DisplayFontNameY = 7;

         DefaultExt       : String[4]   = '.dat';

TYPE     States   = (_write, _erase, _skip);
         Dirs     = (_up, _down, _left, _right);

VAR      Font8x16         : array [0..2047] of char;
         Grid8x16         : array [0..7,0..15] of boolean;
         CurrentFileName  : String;
         CurrentChar      : Byte;
         FontFile         : File of char;
         State            : States;
         CharEdited       : Boolean;
         FontEdited       : Boolean;
         X,Y              : Byte;

Procedure Message (s:string);
begin
  GoToXY (StartX, EndY + 2);
  Write('                                     ');
  GoToXY (StartX, EndY + 2);
  Write(s);
  GoToXY (X,Y)
end;

Procedure PromptMessage(s: string);
begin
 GoToXY (StartX, EndY + 2);
 Write('                                     ');
 GoToXY (StartX, EndY + 2);
 Write(s)
end;

Procedure Error(s: String);
var answ: char;
begin
 PromptMessage (s); Write ('Continue? ');
 if answ in ['y','Y'] then halt(0)
                      else Message ('')
end;

Function Power (Base, Exp: Integer): Integer;
var i,n: integer;
begin
 n := 1;
 for i := 1 to Exp do n := n * base;
 Power := n
end;

Procedure DisplayFileName (fn: string);
begin
 GoToXY(DisplayX,DisplayFontNameY);
 write('Font: ',fn);
 GoToXY(X,Y)
end;

Procedure DisplayGlyph;
var i,j: byte; SaveX, SaveY: byte;
begin
 SaveX := X; SaveY := Y;
 X := StartX; Y := StartY;
 for j := 0 to 15 do begin
                      GoToXY(X,Y+j);
                      for i := 0 to 7 do
                          if grid8x16[i,j] then write('') else write('.')
                    end;
 X := SaveX; Y := SaveY;
 GoToxY(X,Y)
end;

Procedure WriteStartGrid;
var i: byte;
begin
 Writeln('  This is FontEdit 1.0');
 Writeln('  Copyright (C) 2006 Francesco Zamblera');
 Writeln('  under the GNU GPL');
 GoToxY(Startx,StartY-1 );
 Write('01234567');
 for i := 0 to 15 do
        begin
         GoToXY(StartX-2,StartY+i);
         Write(i)
        end;
 DisplayGlyph ;
end;

Procedure DisplayCharNumber (CP: byte);
begin
 GoToXY(DisplayX, DisplayCharY);
 Write('Character: ',CP + 128 );
 GoToxY (X,Y)
end;

Procedure ImportGlyph (CP: byte);
var n,i,j: byte; CharPos: Integer;
begin
 CharPos := CP * 16;
 for j := 0 to 15
     do begin
         n := ord (Font8x16 [CharPos+j]);
         for i := 0 to 7 do grid8x16 [i,j] := false;
         i := 7;
         while n > 0 do begin
                         grid8x16 [i,j] := n mod 2 = 1;
                         n := n div 2;
                         i := i - 1
                        end
        end ;
 DisplayGlyph

end;

Function Bool2bit (b: boolean) : byte;
begin
 if b then Bool2bit := 1 else Bool2bit := 0
end;

Procedure StoreGlyph (CP: byte);
var     n,i,j: byte;
        CharPos: Integer;
begin
 CharPos := CP * 16;
 for j := 0 to 15
     do begin
         n := 0;
         for i := 0 to 7 do n := n + power(2,7-i) * Bool2bit (grid8x16[i,j]);
         Font8x16 [CharPos+j] := chr (n)
        end;
 CharEdited := false;
 FontEdited := true
end;

Procedure InitGrid;
var i,j: byte;
begin
 for i := 0 to 7
     do for j := 0 to 15
        do grid8x16[i,j] := FALSE
end;

Procedure ShiftLeft;
var i,j: byte;
begin
 for i := 1 to 7 do
        for j := 0 to 15 do
                grid8x16[i-1,j] := grid8x16[i,j];
 for j := 0 to 15 do grid8x16[7,j] := false;
 DisplayGlyph;
 CharEdited := true
end;


Procedure ShiftRight;
var i,j: byte;
begin
 for i := 7 downto 1 do
        for j := 0 to 15 do
                grid8x16[i,j] := grid8x16[i-1,j];
 for j := 0 to 15 do grid8x16[0,j] := false;
 DisplayGlyph;
 CharEdited := true
end;


Procedure ShiftUp;
var i,j: byte;
begin
 for i := 0 to 7 do
        for j := 1 to 15 do
                grid8x16[i,j-1] := grid8x16[i,j];
 for i := 0 to 7 do grid8x16[i,15] := false;
 DisplayGlyph;
 CharEdited := true
end;

Procedure ShiftDown;
var i,j: byte;
begin
 for i := 0 to 7 do
        for j := 15 downto 1 do
                grid8x16[i,j] := grid8x16[i,j-1];
  for i := 0 to 7 do grid8x16[i,0] := false;
 DisplayGlyph;
 CharEdited := true
end;

Procedure Move (Dir: Dirs; State: States);
var DisplaceX, DisplaceY : integer;
    i,j: byte;
begin
 DisplaceX := 0; DisplaceY := 0;
 case Dir of
      _up:    if Y > StartY then DisplaceY := -1;
      _down:  if Y < EndY   then DisplaceY := 1;
      _right: if X < EndX   then DisplaceX := 1;
      _left:  if X > StartX then DisplaceX := -1
     end;
 if (DisplaceX <> 0) or (DisplaceY <> 0)
    then begin
          if State <> _skip then CharEdited := true;
          X := X + DisplaceX; Y := Y + DisplaceY;
          GoToXY(X,Y);
          i := X - StartX;
          j := Y - StartY;
          Case State of
               _write: begin
                        write('');
                        GoToXY(X,Y);
                        Grid8x16[i,j] := true
                       end;
               _erase: begin
                        write('.');
                        GoToXY(X,Y);
                        Grid8x16[i,j] := false
                       end;
               end
        end
end;

Procedure DisplayMenu;
begin
 GoToXY(DisplayX, DisplayMenuY);
 write('MENU');
 GoToXY(DisplayX, DisplayMenuY+1);
 write('w : enter WRITE mode');
 GoToXY(DisplayX, DisplayMenuY+2);
 write('e : enter ERASE mode');
 GoToXY(DisplayX, DisplayMenuY+3);
 write('s : enter SKIP mode');
 GoToXY(DisplayX, DisplayMenuY+4);
 write('L : Load file');
 GoToXY(DisplayX, DisplayMenuY+5);
 write('W : Write file');
 GoToXY(DisplayX, DisplayMenuY+6);
 write('g : Go to character ');
 GoToXY(DisplayX, DisplayMenuY+7);
 write('p : Previous character');
 GoToXY(DisplayX, DisplayMenuY+8);
 write('n : Next  character');
 GoToXY(DisplayX, DisplayMenuY+9);
 write('S : Store  character');
 GoToXY(DisplayX, DisplayMenuY+10);
 write('q : Quit');
 GoToXY(DisplayX, DisplayMenuY+11);
 write('Other: see documentation');
end;

Procedure DisplayState(s: States);
begin
 GoToXY(DisplayX, DisplayStateY);
 Write('Mode: ');
 case s of      _write: write('WRITE');
                _skip:  write('SKIP ');
                _erase: write('ERASE');
               end;
 GoToXY(X,Y)
end;

Procedure UpDateState (s: States);
begin
 State := s;
 DisplayState (s)
end;

Procedure AddExtension (var fn: String);
begin
 if pos('.', fn) = 0 then fn := fn + DefaultExt
end;

Procedure WriteFile;
var i: integer;
begin
 if CurrentFileName = ''
        then begin
                PromptMessage('Input filename: ');
                readln(CurrentFileName);
                Message ('');
                AddExtension (CurrentFileName);
                Assign(FontFile, CurrentFileName);
             end;
 rewrite (FontFile);
 for i := 0 to 2047 do write (FontFile, Font8x16[i]);
 close (FontFile);
 FontEdited := false;
 CharEdited := false
end;

Procedure PromptToSave;
var answ : char;
begin
 PromptMessage ('Font modified. Save? '); readln(answ);
 if answ in ['Y','y'] then WriteFile;
 Message ('');
 GoToXY (X,Y)
end;

Procedure ReadFile (fn: String);
var i: integer;
begin
 assign(FontFile, fn);
 reset(FontFile);
 i := 0;
 while not eof(FontFile) do
       begin
        read(FontFile,Font8x16[i]);
        inc(i);
       end;
 close(FontFile)
end;

Procedure GetFile;
begin
 AddExtension (CurrentFileName);
 ReadFile (CurrentFileName);
 FontEdited := false; CharEdited := false;
 DisplayFileName (CurrentFileName);
 CurrentChar := 0;
 ImportGlyph (CurrentChar);
 DisplayCharNumber (CurrentChar)
end;

Procedure ImportFile;
begin
 if FontEdited or CharEdited then PromptToSave;
 FontEdited := false;
 CharEdited := false;
 PromptMessage('Input Filename: '); readln(CurrentFileName);
 Message('');
 GetFile
end;

Procedure PromptToStore;
var c: char;
begin
 PromptMessage('Current glyph modified. Store? ');
 readln(c);
 if c in ['y','Y'] then StoreGlyph (CurrentChar);
 Message ('');
 CharEdited := false
end;

Procedure GoToChar (var CC: byte);
var n: integer;
begin
 If CharEdited then PromptToStore;
 repeat
  PromptMessage('Number (0 to exit): '); read (n)
 until n in [0,128..255];
 Message('');
 if n <> 0 then begin
                 CC := n - 128;
                 ImportGlyph (CC);
                 DisplayCharNumber (CC)
                end
end;

Procedure PreviousChar;
begin
 if CurrentChar > 0
        then begin
                If CharEdited then PromptToStore;
                dec (CurrentChar);
                ImportGlyph (CurrentChar);
                DisplayCharNumber (CurrentChar)
             end
end;

Procedure NextChar;
begin
 if CurrentChar < 127 then begin
                            If CharEdited then PromptToStore;
                            inc(CurrentChar);
                            ImportGlyph (CurrentChar);
                            DisplayCharNumber (CurrentChar)
                           end
end;

Procedure ImportGlyphNumber;
var n,i,j: byte; CharPos: integer;
begin
 repeat
        PromptMessage('Input glyph number: ');
        read(n);
 until n in [128..255];
 Message ('');
 ImportGlyph (n-128);
 DisplayGlyph;
 CharEdited := true
end;

Procedure Save_as;
var NewName: String;
begin
 PromptMessage ('Save as: '); readln(CurrentFileName);
 AddExtension (CurrentFileName);
 WriteFile;
 Message('');
 DisplayFileName (CurrentFileName)
end;

Procedure Edit;
var c: char;
begin
 repeat
   c := ReadKey;
   case c of
       'L': ImportFile;
       'W': WriteFile;
       'A': Save_as;
       'S': StoreGlyph (CurrentChar);
       'g': GoToChar (CurrentChar);
       'p': PreviousChar;
       'n': NextChar;
       'w': UpDateState(_write);
       'e': UpDateState(_erase);
       's': UpDateState(_skip);
       #0 : begin
             c := ReadKey;
             case c of
                  #72: Move(_up,State);
                  #80: Move(_down,State);
                  #75: Move(_left,State);
                  #77: Move(_right,State);
                  #38: {Alt-l} ShiftLeft;
                  #19: {Alt-r} ShiftRight;
                  #22: {Alt-u} ShiftUp;
                  #32: {Alt-d} ShiftDown;
                  #23: {Alt-i} ImportGlyphNumber
                 end
            end
  end
 until c in [#27,'q','Q'];
 if CharEdited then PromptToStore;
 if FontEdited then PromptToSave
end;

procedure writescancode;
var c: char;
begin
 repeat
  c := readkey;
  writeln(ord(c));
 until c = #27
end;

Procedure InitFont;
var i: integer;
begin
 for i := 0 to 2047 do Font8x16[i] := #0
end;

Procedure Init;
begin
  CurrentFileName := ParamStr(1);
  CurrentChar := 0;
  CharEdited := false;
  FontEdited := false;
  InitFont;
  InitGrid;
  WriteStartGrid;
  DisplayMenu;
  if CurrentFileName <> '' then GetFile;
  State := _skip;
  DisplayState(State);
  GoToXY (StartX,StartY);
  X := StartX; Y := StartY
end;

begin
 TextMode (C40);
 init;
 Edit;
 TextMode (C80)
end.

