{===========================================================================
||  FreeDOS Keyb  2.0                                                     ||
||  (Prototype 3 - 23.June.2003)                                          ||
|| ---------------------------------------------------------------------- ||
||  Open Source keyboard driver for DOS                                   ||
||  License:       GNU-GPL 2.0  (see copying.txt)                         ||
||  Platform:      DOS v.1.0 or later                                     ||
||                 PC XT/AT with keyboard interface at int9h/int15h       ||
||  Compiler:      Borland Turbo Pascal 7.0                               ||
|| ---------------------------------------------------------------------- ||
|| (C) Aitor SANTAMARIA MERINO                                            ||
|| (aitor.sm@wanadoo.es)                                                  ||
|| ---------------------------------------------------------------------- ||
|| Contributions:                                                         ||
||   - Dietmar  HOEHMANN   (Xkeyb API and some basic code)                ||
||   - Matthias PAUL       (reboot and flush cache code, general help)    ||
||   - Axel     FRINKE     (header of int15h, general help and ideas)     ||
===========================================================================}



{$A-,B-,D-,E-,F-,I-,L-,N-,O-,R-,S-,V-,Y-}
{$M $2000,0,0}

PROGRAM Keyb;

USES DOS;


CONST
        {Version constants}
        VerS          = '2.0 (Prototype3)';
        Version       = $0100 + 93;

        E0Prefixed    = 2;   {flags in KStatusByte1}

        { Installable KEYB functions}
        KF_ExtInt9    = 1;
        KF_StoreKey   = 2;
        KF_APMFunc    = 3;
        KF_DoCommand  = 4;

        { APMProc function }
        APM_FlushCache = 0;
        APM_WarmReboot = 1;
        APM_ColdReboot = 2;
        APM_PowerOff   = 3;
        APM_Suspend    = 4;

TYPE
        SimpleProc     = procedure;     { parameter-less callable function }
        PtrRec         = record         { pointer record }
                            Ofs,Seg : Word;
                         End;

{=== BIOS Data segment variables ===}
VAR
        KStatusFlags1     : Byte absolute 0:$417;  { Listing active keys }
                                            { 0 : Right SHIFT pressed}
                                            { 1 : Left SHIFT  pressed}
                                            { 2 : CTRL  pressed}
                                            { 3 : ALT  pressed}
                                            { 4 : SCROLL LOCK active }
                                            { 5 : NUM LOCK active }
                                            { 6 : CAPS LOCK  active}
                                            { 7 : INSERT active}
       KStatusFlags2     : Byte absolute 0:$418;  { Currently pressed modifier keys. }
       AltInput          : Byte absolute 0:$419;  { Char entered by Alt and numberkeys. }
       KStatusByte1      : Byte absolute 0:$496;  { Status of right ALT and Control. }
       KStatusByte2      : Byte absolute 0:$497;  { Status of LEDs. }
       BufferStart       : Word absolute 0:$480;  { Start address of key buffer. }
       BufferEnd         : Word absolute 0:$482;  { End address of key buffer. }
       BufferHead        : Word absolute 0:$41C;  { Next free char in buffer. }
       BufferTail        : Word absolute 0:$41A;  { Next char to read. }


    {------- XKEYB-style layout information ------------------------------}
    {------- To be REPLACED in prototype 4  ------------------------------}
    Type
         TransTabTyp = Array[1..256,0..5] of Byte;  {127 scancodes, E0+127 scancodes}
         CombTabTyp  = Array[0..190] of Byte;
         FileNameTyp = String[64];

         DR0 = record     {DR without the XStuff}
               ConfigFile  : FileNameTyp;           { Name of Config file. }
               CombTab     : CombTabTyp;            { Table for combinated chars. }
               TransTable  : TransTabTyp;           { Maximally 128 keys + 128 extended keys }
         end;

         DR = Record                                 { All data needed after installation. }
               ConfigFile  : FileNameTyp;           { Name of Config file. }
               CombTab     : CombTabTyp;            { Table for combinated chars. }
               TransTable  : TransTabTyp;           { Maximally 128 keys + 128 extended keys }
                                                    { 5 key levels (0-4): normal, shift, ctrl, alt, ctrl+alt (alt gr). }
                                                    { Level 5 contains administration information for every key: }
                                                    { Bits 0-2 tell, whether the key is influenced by Num, Caps or Scroll. }
                                                    { Bits 3-7 tell for every level whether it's mapped to an XStr. }
               XStrBufSize : Word;                  { Size of buffer for xstrings. }
               XStrings    : Array[0..$C8C8] of Byte;{ Up to 50K of Kbdextensions.
                                               Aitor's note: reduced from 63Kb}
         End;
         PDR    = ^DR;
    {------- END ---------------------------------------------------------}


CONST
        {======= KEYB GLOBAL DATA SEGMENT (later accessed through CS) ====}

        { Old interrupt vectors }
        OldInt9h       : Pointer = NIL;         { Chain for int9h. }
        OldInt2Fh      : Pointer = NIL;         { Chain for multiplexer interrupt. }
        OldInt16h      : Pointer = NIL;         { Chain for Int 16h. }
        OldInt15h      : Pointer = NIL;         { Chain for Int 15h. }

       {Layout pointers}
       {NOTE: these point to xkeyb-style structures.
       They are structures to be updated in prototype 4}
       LayoutList      : PDR  = NIL;
       LastXStr        : Byte = 0;              { Number of the last defined XString. }
       CurrentLayout   : PDR  = NIL;            { Pointer to the DR (ancestor of the future layout block) }

       {Other global variables}
       CombStat        : Word = 0;  { Status of char combination. 0=No combination signs in work. }
                                    { Else pointer to combination table. }
                                    { will probably be integrated into COMBI procedure }

       { Installable functions vectors }
       { NOTE: if ExtInt9Proc is MOVED to other position in the table, you
         should also modify CallKeybFunc }
       ExtInt9Proc : SimpleProc   = NIL;    { Int9h management extensions }
       CurStoreKey : SimpleProc   = NIL;    { procedure used to Store Key }
       APMProcedure: SimpleProc   = NIL;    { APM (power) procedure       }
       DoCommands  : SimpleProc   = NIL;    { perform commands 100..199   }

       { End of the KEYB Global data segment }
       EoDS        : Byte = 0;


{------------------------------------------------------------------------
---- KEYB 2 CORE ROUTINES (non-discardable)                         -----
------------------------------------------------------------------------}


{*** SPACE for the KEYB Global Data Segment, accessible through CS ***}
Procedure GlobalDS;
Begin
   ASM
   DD 0,0,0,0,0,0,0,0,0,0
   End;
End;


{************************************************************************
** CallKEYBFunc:   calls a KEYB installable function                   **
** ------------------------------------------------------------------- **
** IN:   CX: Function code (see constants above)                       **
**       CS: KEYB Global data segment                                  **
** OUT:  CF set if function not found                                  **
*************************************************************************}
procedure CallKEYBFunc; assembler;
label CallFunc;
asm
        { CX = FirstFunction + (CX-1)*4 }
        DEC    CX
        SHL    CX, 2
        ADD    CX, Offset ExtInt9Proc

        { Preserve DI, BX }
        PUSH   DI
        PUSH   BX

        { CS:DI -> Pointer to function; load and test if zero }
        MOV    DI, CX
        MOV    BX,CS:[DI]
        OR     BX,CS:[DI+2]

        { Recover registers and test if zero }
        TEST   BX,$FF
        POP    BX
        JNZ    CallFunc
        POP    DI

        { function not found: exit }
        STC
        RET

        { function found: call it and return }
CallFunc:
        CALL   DWORD PTR CS:[DI]
        POP    DI
        CLC
end;


{************************************************************************
** MultiplexHandler: handler for the DOS-MuX Keyb services             **
** ------------------------------------------------------------------- **
** INT:  services interrupt 2Fh (AX=AD80h)                             **
*************************************************************************}
procedure MultiplexHandler; assembler;
label our;
asm
      CMP  AX, $AD80
      JE   Our
      JMP  CS:[OldInt2Fh]
Our:
      MOV AX,$FFFF
      PUSH CS
      POP  ES
      MOV  DI,ES:[Offset CurrentLayout]
      MOV  DX,Version
      IRET
end;



{FORWARDS of discardable functions}
{TODO: replace the direct references with indirect references
       (using CallKEYBFunc)}
function TranslateScanCode (s: byte): byte; forward;


{StoreKey: the old xkeyb procedure to do all preprocessing before sending the
           key to buffer (includes Scancode translation and PAUSE management
 TODOs:    - replace all StoreKey references within code by EfStoreKey if
           possible.
           - possibly integrate it in the int15h }
Procedure StoreKey(A,S : Byte);
Begin
      {*** Scancode processing ***}

      s := TranslateScanCode (s);       { by function }

      if (KStatusByte1 AND E0Prefixed)>0 then begin   { "manual" }
         if A=0 then A := $E0
                else if S<>255 then S := $E0
      end;

      {*** Pause or Store ***}

      If (KStatusFlags2 and 8)=8 Then          { Pause ? }
      ASM                                      { Then end Pause. }
          Mov AL,ES:[Offset KStatusFlags2]      { ES still remains from the check for 0000! }
          And AL,$F7
          Mov ES:[Offset KStatusFlags2],AL
      End
      Else                                     { Else store pressed key. }
      asm
         Mov Ah,S                 { Scancode. }
         Mov Al,A                 { ASCII-Value. }

         call CurStoreKey;
      end;
End;

{ TableLookUp:  the old xkeyb/KEYB2 mixed code lookUp table routine,
                at KEYB's heart.
                From scancode in AL, it either produces a character
                or performs a special function
  TODO: This is the main routine to be completely replaced in prototype4
        (newer features) and prototype5 (adding codepage support)
        This is to be integrated in the int 15 handler below }
procedure TableLookUp;
label Ende, Skp1, Skp1_2, Skp1_4, Skp1_5, Lop2, Skp5, NoCombine, X, DefaultX,
      StartCombi, Lop3, Skp7, Store, Done, Reject, IsMakeCode, DoCommandLabel;

var
   a  : byte;
   NCS: boolean;
   scancode: byte;
   row     : byte;

begin
     ASM
        PUSH BX
        PUSH CX
        PUSH DX
        PUSH DI
        PUSH SI
        PUSH DS
        PUSH ES


        {**** CS -> DS }
        PUSH CS
        POP  DS

        CMP AL, 128         { process only MakeCodes }
        JB  IsMakeCode
        CMP AL, $E0         { and E0h, extended prefix }
        JNE Reject

        { it's E0, so update LED }
        MOV CL, ES:[Offset KStatusByte1]      { CL := KStatusByte1 }
        OR  CL, E0Prefixed   { Put E0Prefixed flag }
        MOV ES:[Offset KStatusByte1], CL
        JMP Reject           { even though, let it continue processing }

IsMakeCode:       

    CMP AL,83       { TEMPORARY IN PROTOTYPE 3: DEL IS ALWAYS BYPASSED!! }
    JE  Reject

        Mov scancode, AL
     END;


{========= This piece in Pascal comes directly from xkeyb =======}
     
    If (CurrentLayout = NIL) then begin   {temporary in prots. 3 & 4: Ctrl+Alt+F2 to reenable}
       if (scancode = 60) and ((KStatusFlags1 and 12)=12) then CurrentLayout := LayoutList;
       asm
          mov al, scancode
          jmp reject
       end;
    end;

{The following piece is for the Alt+nnn inputs, could be rejected in some
special circumstances}

   If ((KStatusFlags1 and $F) = 8 ) and ((KStatusByte1 AND E0Prefixed)=0) and           { ALT pressed? }
      (ScanCode>=$47) and (ScanCode<=$52) and
      (Scancode<>$4A) and (scancode<>$4E)   { Is key influenced by NUM? }
      Then
      Begin
        If (KStatusFlags2 and 8)=8 Then               { Pause ? }
           ASM                                        { Then end Pause. }
             Mov AL,ES:[Offset KStatusFlags2]         { ES still remains from the check for 0000! }
             And AL,$F7
             Mov ES:[Offset KStatusFlags2],AL
           End
        else begin
          {translate scancode->number}
         case Scancode of $47..$49 : Scancode := Scancode - $40;
                          $4B..$4D : Scancode := Scancode - $47;
                          $4F..$51 : Scancode := Scancode - $4E;
                          Else       Scancode := 0;
         end;
         AltInput:=AltInput*10 + ScanCode; { -> append pressed number to the existing value. }
        end;
        goto done;
      End;


   { ********** Compute the row ********** }

   scancode := scancode + ((KStatusByte1 AND E0Prefixed) SHL 6);
        { if E0-prefixed, then scancode adds 128}

   NCS:=(( CurrentLayout^.TransTable[ScanCode,5] and      { Check, whether Num, Caps or Scroll are on }
          (KStatusFlags1 shr 4) and               { and whether the pressed key is influenced by them. }
          $7
        ) > 0);                { Extended Shift key clears NCS! }


  if (KStatusFlags1 AND $0C)=4 then row := 2   {ctrl}
  else if (KStatusFlags1 and $03)>0 then row := byte(NCS) xor 1  {shift}
  else if (KStatusFlags1 and $08)>0 then row := 3 + ord((KStatusByte1 AND $08)>0)  {alt or AltGr}
  else row := byte (NCS);
 
{========= This piece in Assembler comes directly from xkeyb =======}

   ASM
      Mov Al,ScanCode             { ScanCode. }
      Dec Al                      { Array starts at 1. Correcting that. }
      Xor AH,AH
      SHL AX,1
      Mov BX,Ax
      Shl Ax,1
      Add Ax,Bx                   { 6 times value -> pointer in TransTable. }
      LES Di,CurrentLayout           { Start address of table. }
      ADD Di, offset DR.TransTable
      Add Di,Ax                   { Address of table entry. }
      Mov Al,ES:[DI+5]            { Read status byte. }
      Mov Bl,ScanCode
      AND BL,$7F                  { disable the effect of +128 for E0-prefixed ones}

      Mov Cl,Row                  { Line. }
      Inc Cl
      Shl Al,Cl                   { The XStr-Bit corresponding to row gets shifted to Carry. }
      JNC Skp1                    { Jump if not occupied by XStr. }
      Mov BL,$FF                  { Replace scancode by signature FFh }

Skp1: Mov Cl,Row
      Xor Ch,Ch
      Add Di,CX
      Mov Al,ES:[DI]              { ASCII value or XStr number from Transtable. }

      TEST AL,AL                  { if not found, }
      JNZ  Skp1_2
      MOV  AL, BL                 { return scancode to AL }
      JMP  Reject                 { and reject it }


{========= Mixed xkeyb/KEYB2 code =======}

{====================== WE HAVE Scan, ASCII ===========}

Skp1_2:
      CMP BL,$FF      { XOperation: command, XString or COMBI }
      JE  X

{====================== PART 1: non-XOperation ========}

      CMP AL,$E0          { if ASCII=$E0k,$F0, then ScanCode=0 }
      JE  Skp1_4
      CMP AL, $F0
      JNE Skp1_5
Skp1_4:
      XOR BL,BL

Skp1_5:
      CMP CombStat, 0     { pending COMBI operation? }
      JZ  Store           { NO, store the ASCII,SCANCODE }

    {===== PART 1-1: COMBINE ==} 

      Mov Di,CombStat
      Mov CombStat,0

      Mov Si,Di
      Mov CL,DS:[DI]              { Number of chars to check. }
      Xor Ch,Ch
      Inc Di
Lop2: Cmp Al,DS:[DI]              { Found 2nd char of combination? }
      JNZ Skp5                    { No. }

      Inc Di
      Mov Al,DS:[Di]              { Replacement char. }
      XOR BX,BX                   { ScanCode should be 0 with COMBI!}
      Jmp Store

Skp5: Inc Di                      { Next entry. }
      Inc Di
      Dec Cx                      { Another candidate? }
      JNZ Lop2                    { Then go on. }

NoCombine:
      Push Ax                     { All that was a failure. }
      Push BX                     { Now throw both chars into the key buffer seperately. }
      Mov Al,DS:[Si-1]            { First char. }
      Xor Bl,Bl                   { Scancode forgotten. So we put 0. }
      Push Ax
      Push Bx
      Call StoreKey               { Store first char. }
      Call StoreKey               { Store 2nd char. }
      Jmp Done

{====================== PART 2: XOperation ========}

X:
      TEST      AL,$FF                    { 0: CHAIN (continue) }
      JZ        Reject

      CMP       AL,160                     { 160: NOP (absorve) }
      JE        Done

      CMP       AL,100                     { 1..99:     XString }
      JB        Store

      CMP       AL,199                  { 200..255:     COMBIs  }
      JA        StartCombi

      XOR       DX, DX
      MOV       DL, AL                  { 100..199: other comms }

      MOV       CX, KF_DoCommand
      CALL      CallKeybFunc

      JMP       Done

StartCombi:
      LES  DI, CurrentLayout
      ADD  DI, Offset DR.CombTab
Lop3: Cmp Byte Ptr DS:[Di],0      { End of list? }
      Jz  Store                   { Yes. }
      Cmp Al,200                  { Is this the char? }
      JE  Skp7                    { Yes. }
      Inc Di
      Mov Cl,DS:[Di]
      Xor Ch,Ch
      Shl Cx,1
      Add Di,Cx
      Inc Di                      { Next char. }
      Dec Al
      Jmp Lop3

Skp7: Inc Di
      Mov CombStat,Di
      Jmp Done

{==================== FINAL: Store the key in AX, BX ========}

Store:
      PUSH Ax
      Push Bx
      Call StoreKey


Done:                   { final if everything was right! }
      Mov  AL, Scancode
      CLC
      JMP  Ende

Reject:                 { reject scancode, so that it continues processing }
      STC

Ende:

        POP ES
        POP DS
        POP SI
        POP DI
        POP DX
        POP CX
        POP BX

End; {ASM}
end;

{************************************************************************
** Int15h: handler for int15h (Axel Frinke)                            **
** ------------------------------------------------------------------- **
** INT:  services interrupt 15h (AH=4Fh)                               **
*************************************************************************}
procedure Int15h; assembler;
label IsFunc4F, KeyboardHandler;
asm

    PUSHF             { preserve flags! }

    CMP   AH, $4F     { is it function 4Fh? }
    JE    IsFunc4F    { check functions }
    POPF
    JMP   DWORD PTR CS:[OldInt15h]

IsFunc4F:

    CALL  DWORD PTR CS:[OldInt15h] { the old one first! }
    JC    KeyboardHandler          { if Carry clear, }
    RETF  2                        { return and preserve flags! }

KeyboardHandler:
    CALL  TableLookUp;             { call the old one:
                                     it will be pasted here, once all of it
                                     is turned into assembler }
    RETF  2

end;




{------------------------------------------------------------------------
---- KEYB 2 DISCARDABLE FUNCTIONS BEGIN HERE                        -----
------------------------------------------------------------------------}


{------------------------------------------------------------------------
---- Command processing functions (func4)                           -----
-------------------------------------------------------------------------
-- DISCARD: if no commands 100-199 are used (excluded 160)             --
------------------------------------------------------------------------}


{TODO: XBufferTail=XBufferHead in break}
{************************************************************************
** ProcessCommands: executes KEYB commands 100-199 (excluded 160)      **
** ------------------------------------------------------------------- **
** IN:   DL: KEYB command number (see KEYB documentation)              **
** OUT:  Many registers are probably trashed                           **
**       On APMCommands, may never return                              **
*************************************************************************}
{**** List of commands already implemented ****
100: disable keyb
140: int5h
141: int19h
142: int1bh
150..154: APMcommand (n-150)
161: set pause bit
164: simulate INS }
procedure ProcessCommands ; far; assembler;
label  no100, no140, no141, no142, no161, no164, endPC;
ASM
         push       es
         xor        ax, ax
         mov        es, ax

         {---- individual quick commands ---}
         cmp        dl, 100
         jne        no100

         {100: disable KEYB}
         xor        cx,cx
         mov        [cs:Offset CurrentLayout], cx
         mov        [cs:Offset CurrentLayout+2], cx
         jmp        endPC

no100:   cmp        dl, 140
         jne        no140

         {140: int 5h}
         int        $5
         jmp        endPC

no140:   cmp        dl, 141
         jne        no141

         {141: int 19h}
         int        $19
         jmp        endPC

no141:   cmp        dl, 142
         jne        no142

         {142: int 1Bh}
         mov        al, [es: offset BufferHead]
         mov        [es: offset BufferTail], al
         int        $1b

no142:   cmp        dl, 161
         jne        no161

         {161: set pause}
         mov        al, [es: offset KStatusFlags2]
         or         al, 8
         mov        [es: offset KStatusFlags2], al
         jmp        endPC

no161:   cmp        dl, 164
         jne        no164

         {164: simulate INS}
         mov        al, [es:offset KStatusFlags1]
         xor        al, $80
         mov        [es:offset KStatusFlags1], al
         xor        al,al
         Mov        ah,82
         mov        cx, KF_StoreKey
         call       CallKeybFunc
         jmp        endPC

         {----  APM Commands (150-154) ----}
no164:   cmp        dl, 150
         jb         endPC
         cmp        dl,154
         ja         endPC
         sub        dl,150
         mov        cx, KF_APMFunc
         call       CallKeybFunc


endPC:   pop        es

end;

{------------------------------------------------------------------------
---- Scancode translation functions (Func?)                         -----
-------------------------------------------------------------------------
---- DISCARD: if no affected keys are remapped                         --
------------------------------------------------------------------------}


{ TranslateScancode: translates scancode for those keys that need it }
{ This function is to be converted to ASSEMBLER (possibly in prot. 4)}
function TranslateScanCode (s: byte): byte;
begin
     case s of
              2..13 :  if (KStatusFlags1 AND 8)>0 { upper numeric keys, with Alt changes s }
                         then s := s+118;
              55    :  if (KStatusFlags1 AND 4)>0 { ^Print }
                         then s := 114;
              59..68:  { F1..F10 }
                       if (KStatusFlags1 and 8)>0 then S:=s+45           {Lalt or Ralt}
                       else if (KStatusFlags1 and 4)>0 then S:=s+35      {ctrl}
                       else if (KStatusFlags1 and 3)>0 then S:=s+25;     {Lshift and RShift}
              71..83:  { ^NumericPAD }
                       If ((KStatusFlags1 and $C) = 4) then
                       Case S of
                                71 : S:=119;         { ^Home. }
                                72 : S:=141;         { ^Cursor Up }
                                73 : S:=132;         { ^Pg Up. }
                                75 : S:=115;         { ^Cursor left. }
                                76 : S:=143;         { ^numericPad 5 }
                                77 : S:=116;         { ^Cursor right. }
                                79 : S:=117;         { ^End. }
                                80 : S:=145;         { ^Cursor Down }
                                81 : S:=118;         { ^Pg Down. }
                                82 : S:=146;         { ^Ins      }
                                83 : S:=147;         { ^Del }
                       End;
              87..88:  { F11, F12 }
                       if (KStatusFlags1 and 8)>0 then S:=s+52           {Lalt or Ralt}
                       else if (KStatusFlags1 and 4)>0 then S:=s+50      {ctrl}
                       else if (KStatusFlags1 and 3)>0 then S:=s+48      {Lshift and RShift}
                       else s:=s+46;
     end;
     TranslateScancode := s
end;

{------------------------------------------------------------------------
---- PC/XT keyboard buffer storage functions (Func2)                -----
-------------------------------------------------------------------------
-- DISCARD: in AT Class machines (in favour of int11h / )              --
------------------------------------------------------------------------}


{************************************************************************
** EfStoreKey1: Stores AX into BIOS KEYB buffer                        **
** ------------------------------------------------------------------- **
** IN:   AX: word to be stored                                         **
** OUT:  ES is zeroed                                                  **
**       DI, SI: trashed                                               **
*************************************************************************}
Procedure EfStoreKey1; far; assembler;
label ende;
asm
         CLI

         Xor Si,Si
         Mov Es,Si
         Mov DI,ES:BufferHead     { Pointer to end of buffer. }
         Mov ES:[DI+$400],AX      { Key in [40:BufferHead]. }

         Add Di,2                 { Pointer to next entry. }
         Cmp DI,ES:BufferEnd      { Reached end of buffer ? }
         Jne ende                 { No -> Proceed in Text. }
         Mov DI,ES:BufferStart    { Pointer back to start. }

ende:
         Mov ES:BufferHead,DI

         STI
end;


{==================  /9- MODE LIMIT (/9* STARTS HERE) =============}


{------------------------------------------------------------------------
---- Advanced Power Management Functions (Func3)                    -----
-------------------------------------------------------------------------
-- DISCARD: if no APM commands are used                                --
------------------------------------------------------------------------}

{************************************************************************
** APMProc: process an APM command                                     **
** ------------------------------------------------------------------- **
** IN:   DL: (0-based) APM command number (see constants above)        **
** OUT:  Many registers are probably trashed                           **
**       On some commands, may never return                            **
*************************************************************************}
procedure APMProc; far; assembler;
label FlushNLCACHE, NLCACHE_callf, NLCACHE_farentry, NLCACHE_reentry,
      FlushSMARTDRV, FlushCacheForce, FlushCacheExit,
      EndAPMProc, jmp1, jmp2, jmp3, doReset, Skp1, Skp2, Skp3;

const CallReboot : pointer = ptr ($FFFF,$0000);

asm

        push dx { place parameter on DX }

(*  BEGIN FlushCache code (by Matthias Paul)
== Last edit: 2001-06-18 MPAUL
;
; - Taken from Axel C. Frinkes & Matthias Pauls FreeKEYB sources
;   with some modifications to possibly work as a drop-in replacement
;   in 4DOS.
; - While the implied actions are different for SMARTDRV 4.0+
;   and NWCACHE 1.00+, the result is the same for both of
;   them - the cache will be flushed unconditionally.
; - Works around a problem with DBLSPACE loaded, where DBLSPACE
;   may terminate the current process with an error message, that
;   it would not be compatible with SMARTDRV before 4.10+.
; - Works around a problem, where the cache would not be flushed
;   with NWCACHE 1.00+, if the CX register accidently happened
;   to be 0EDCh.
; - Is enabled to continue to work with future issues of NWCACHE,
;   which might no longer flush the cache just on calling
;   SMARTDRV's install check...
; - Supports NetWare Lite's NLCACHE and Personal NetWare's
;   NWCACHE sideband API to asynchronously flush the cache.
;   This ensures system integrity even when the NetWare Lite or
;   Personal NetWare SERVER is loaded on the machine.
; - Furthermore, under some conditions on pre-386 machines
;   NWCACHE cannot intercept a reboot broadcast by itself, and
;   hence it must be called explicitely before a possible reboot.  *)


FlushNLCACHE:
        mov     ax,0D8C0h       { NLCACHE/NWCACHE install check  }
        mov     cl,ah           { (sanity check: preset CL >= 10h) }
        push    cs              { (preset to ourselves for safety) }
        pop     es              
        mov     di,offset NLCACHE_farentry
        int     2Fh             { (NLCACHE/NWCACHE modify AL,CL,DX,DI,ES)}

        cmp     ax,0D8FFh       { cache installed? (AL = FFh)             }
        jne    FlushSMARTDRV

        cmp     cl,al           { CL=FFh? (workaround for NWCACHE before  }
         je     NLCACHE_callf   { BETA 17 1993-09-28, CL=FFh,00h,01h)     }
        cmp     cl,10h          { (sanity check: CL < 10h on return,      }
         jae    FlushSMARTDRV   { only CL=01h,02h,03h are defined so far) }

NLCACHE_callf:
        xor     bx,bx           { BX=0: asynch. flush request from server }
        push    cs              { push return address on stack            }
        mov     ax,offset NLCACHE_reentry
        push    ax              
        push    es              { push ES:DI -> entry point into far API  }
        push    di              
        clc                     { assume pure cache                       }
NLCACHE_farentry:               { (dummy entry point)                     }
        retf                    { simulate a CALLF into sideband function }

NLCACHE_reentry:                { return from sideband (AX/flags modified)}
         jc     FlushNLCACHE    { if error retry because still dirty cache}
        test    ax,ax           { CF clear and AX=0000h?                  }
         jnz    FlushNLCACHE    { if not, retry until everything is OK    }

(*NLCACHE/NWCACHE is pure now, so it would be safe to exit here.
; However, it doesn't harm to play it extra safe and
; just fall through into the normal SMARTDRV flush sequence...
; Who knows, multiple caches might be loaded at the same time...  *)

FlushSMARTDRV:
        mov     ax,4A10h        { SMARTDRV 4.00+ API                      }
        xor     bx,bx           { install check (BX=0)                    }
        mov     cx,0EBABh       { mimic SMARTDRV 4.10+ (CX=EBABh)         }
                                { to workaround DBLSPACE problem.         }
                                { CX<>0EDCh to avoid NWCACHE's /FLUSH:OFF }
                                { special case! Flush regardless          }
                                { of NWCACHE's configuration setting.     }
        int     2Fh             { (modifies AX,BX,CX,DX,DI,SI,BP,DS;ES?)  }
                                { NWCACHE 1.xx has flushed its buffers    }
                                { now, but we should not rely on this.    }
        cmp     ax,6756h        { NWCACHE 1.00+ magic return?             }
         je     FlushCacheForce {  (extra-safe for future NWCACHE 2.00+)  }
        cmp     ax,0BABEh       { SMARTDRV 4.0+ magic return?             }
         jne    FlushCacheExit  {  nothing we can do                      }
         jcxz   FlushCacheExit  { any dirty cache elements?               }
FlushCacheForce:
        mov     cx,ax           { CX<>0EDCh to avoid NWCACHE special case,}
                                { hence we preset with magic return for   }
                                { possible future broadcast expansion.    }
        mov     ax,4A10h        { SMARTDRV 4.00+ API                      }
        mov     bx,0001h        { force synchronous cache flush           }
        push    cx              
        int     2Fh             { (modifies BP???)                        }
        pop     cx              { (safety only, not necessary)            }
        cmp     cx,6756h        { retry for any cache but NWCACHE         }
        jne    FlushSMARTDRV    { probably obsolete, but safer            }
                                { at the risk of a possible deadlock in   }
                                { case some hyphothetical SMARTDRV        }
                                { clone would not support the CX return   }
FlushCacheExit:

(** END FlushCache code (by Matthias Paul)  **)

        pop ax
        test ax,ax
        clc
        jz EndAPMProc

        push   es
        xor    cx,cx
        mov    es,cx

        dec al
        jnz skp1

        mov    cx,$1234        { YES: warm reboot }
        jmp    doReset

        stc
        jmp EndAPMProc

skp1:
        dec al
        jnz skp2

        doReset:mov    [es:$472],cx    { YES: cold reboot }
        call   callReboot

        stc
        jmp EndAPMProc

skp2:
        pop  es
        push ax

     MOV AX, $5301      {Real Mode interface connect}
     XOR BX, BX
     INT $15

     MOV AX, $530F      {Engage power management}
     MOV BX, 1
     MOV CX, BX
     INT $15

     MOV AX, $5308      {Enable APM for all devices}
     MOV BX, 1
     MOV CX, BX
     INT $15

     MOV AX, $530E      {force version 1.1}
     XOR BX, BX
     MOV CX, $0101
     INT $15

        pop ax
        dec al
        jnz skp3

     MOV AX, $5307              {First attempt: switch all off}
     MOV BX, 1
     MOV CX, 3
     INT $15

     MOV AX, $5307              {Second attempt: system bios}
     XOR BX, BX
     MOV CX, 3
     INT $15
     
        stc
        jmp EndAPMProc

skp3:

     MOV AH,$2
     MOV DL,'H'
     INT $21

     MOV AX, $5307
     MOV BX, 1
     MOV CX, 2
     INT $15
     clc

EndAPMProc:

end;


{------------------------------------------------------------------------
---- Basic int9h management Functions (Int9h)                       -----
-------------------------------------------------------------------------
-- DISCARD: if Enhanced(AT) keyboard AND int9 no management requested  --
------------------------------------------------------------------------}

CONST
        { keyboard commands }
        DisableKeyboard = $AD;
        EnableKeyboard  = $AE;

{************************************************************************
** EnableKeyboardProc: enable keyboard hardware                        **
** ------------------------------------------------------------------- **
** IN:   -                                                             **
** OUT:  AX and CX are trashed                                         **
*************************************************************************}
Procedure EnableKeyboardProc; assembler;
label WaitReady2;
asm
        XOR    CX, CX       { counter to 65535! }
WaitReady2:
        IN     AL, $64      { read status register }
        TEST   AL, 2        { bit 1 set: input buffer full? }
        LOOPNZ WaitReady2   { loop until timeout or bit 1 clear }

        MOV AL, EnableKeyboard
        OUT $64, AL         { send the enable command }

        RET
end;


procedure ProcessPause; forward;


{************************************************************************
** Int9H: Basic int9 management                                        **
** ------------------------------------------------------------------- **
** INT:  services interrupt 9h (keyboard IRQ)                          **
*************************************************************************}
procedure Int9H; assembler;
Label  IsActive, Start, CallExtInt9, Reenable, LeaveInt,
       WaitReady1, ProcessIt, ChainToOld;
ASM

    {******* SECTION 1: COMMON INT9h ************}

        {-- Workaround for APL software --}
        JMP    Start
IsActive:DW    1             { modified by APL software}

        {-- Save the registers --}
Start:  PUSH   AX            { scancode }
        PUSH   BX            { boolExtInt9 }
        PUSH   CX            { counter }

        {-- Compute if we should handle the extended --}
        MOV  BX, CS:[Offset IsActive]

        {-- Disable the interrupts and the keyboard --}
        CLI                  { disable hardware ints }
        XOR    CX, CX        { counter to 65535! }
WaitReady1:
        IN     AL, $64       { read status register }
        TEST   AL, 2         { bit 1 set: input buffer full? }
        LOOPNZ WaitReady1    { loop until timeout or bit 1 clear }
        MOV    AL, DisableKeyboard 
        OUT    $64, AL       { send the disable command }

        {-- Read and authenticate scancode --}
        IN      AL, $60      { get scancode to AL }
        MOV     Ah,$4F       { Authenticate scancode}
        STC
        INT     15h          { Return: CF clear if key needs no further processing }

        {-- What to do next? --}
        JC      ProcessIt    { No further processing of pressed keys if CARRY cleared! }
        XOR     BX, BX       { no extended processing }
        JMP     Reenable     { reenable all }

ProcessIt:
        TEST    BX,$FF       { is driver active? }
        JNZ     CallExtInt9  { NO: chain to older driver } 

    {******* SECTION 2: CHAIN TO PREVIOUS INT9 HANDLER ************}

ChainToOld:
        CALL   EnableKeyboardProc
        PUSHF
        STI
        CALL   DWORD PTR CS:[Offset OldInt9h]
        JMP    LeaveInt

    {******* SECTION 3: CALLING EXTINT9 ************}

CallExtInt9:
        MOV    CX, KF_ExtInt9
        CALL   CallKEYBFunc
        JNC    Reenable
        XOR    BX, BX        { if function not found: no longer extended handling }
        JMP    ChainToOld    { whenever NO function, try previous }

    {******* SECTION 4: REENABLE EVERYTHING (process PAUSE if needed) ************}

Reenable:
        {-- Reenable Interrupt controller, Interrupts and Keyboard --}
        CALL   EnableKeyboardProc  { reenable keyboard }
        MOV    AL, $20       { report End of Interrupt to interrupt controller }
        OUT    $20, AL
        STI                  { restore interrupts }

        {-- Process PAUSE, if needed (this is EXTENDED handling) --}
        TEST   BX, $FF
        JZ     LeaveInt
        CALL   ProcessPause

    {******* SECTION 5: LEAVE INTERRUPT ROUTINE ************}
LeaveInt:

        POP CX
        POP BX
        POP AX
        IRET
end;


{==================  /9* MODE LIMIT (/9+ STARTS HERE) =============}


{------------------------------------------------------------------------
---- Extended (full) int9h management functions (Func1)             -----
-------------------------------------------------------------------------
-- DISCARD: if no full int9h management is requested                   --
------------------------------------------------------------------------}


{************************************************************************
** ExtInt9Data: variables for extended int9h management                **
*************************************************************************}
Procedure ExtInt9Data;
Begin
   ASM
     DB   0                              { LoopRunning: locks access to loop running }
     DB   54, 42, 29, 56, 70, 69, 58, 82 { ShiftKeys }
   End;
End;


{************************************************************************
** ProcessPause: process the system pause                              **
** ------------------------------------------------------------------- **
** IN:   -                                                             **
** OUT:  -                                                             **
*************************************************************************}
procedure ProcessPause; assembler;
label Lop1, EndProcessPause;
asm
        MOV AL, CS:[Offset ExtInt9Data]  {LoopRunning}
        TEST AL, $FF               { is Pause loop running?}
        JNZ EndProcessPause        { yes: leave interrupt! }

        PUSH ES
        MOV AL, 1                  { lock this region }
        MOV CS:[Offset ExtInt9Data], AL   {LoopRunning}
        XOR AX, AX                 { see Pause bit in KStatusFlags2}
        MOV ES, AX
Lop1:   TEST ES:KStatusFlags2, 8   { in Pause? then continue loop }
        JNZ Lop1
        XOR AX, AX
        MOV CS:[Offset ExtInt9Data], AL { exit locked region }  {LoopRunning}
        POP ES

EndProcessPause:
end;

{ DoShiftKeys: updates BIOS variables according to the SHIFT key that has
  been pressed }
procedure DoShiftKeys (shiftkey: byte; isbreak: boolean);  {0..7}
var
   shiftedbit, v1, v2 : byte;
begin
    shiftedbit := 1 SHL shiftkey;
    case shiftkey of
                    0..1: {shift}
                          if isBreak then KStatusFlags1 := KStatusFlags1 XOR shiftedbit
                                     else KStatusFlags1 := KStatusFlags1 OR shiftedbit;
                    2..3: begin {ctrl,alt}
                            if (KStatusByte1 AND E0Prefixed)>0 then begin
                                if isBreak then KStatusByte1 := KStatusByte1 XOR shiftedbit
                                           else KStatusByte1 := KStatusByte1 OR shiftedbit
                            end else begin
                                shiftedbit := shiftedbit SHR 2;
                                if isBreak then KStatusFlags2 := KStatusFlags2 XOR shiftedbit
                                           else KStatusFlags2 := KStatusFlags2 OR shiftedbit
                            end;
                            v1 := (KStatusByte1 OR (KStatusFlags2 SHL 2)) AND $0C;
                            v2 := KStatusFlags1 AND $F3;
                            KStatusFlags1 := v1 OR v2;

                            if (shiftkey=3) AND isBreak AND (AltInput<>0)
                               then begin
                                 StoreKey (AltInput,0);
                                 AltInput := 0
                               end;

                          end;
                    4..6: begin  {caps, num, scroll}
                              if isBreak then KStatusFlags2 := KStatusFlags2 XOR shiftedbit
                                         else KStatusFlags2 := KStatusFlags2 OR shiftedbit;
                              if not isBreak then KStatusFlags1 := KStatusFlags1 XOR shiftedbit
                          end;
                    7   : if (KStatusFlags2 AND $0C)=0 then begin  {no ctrl, alt}
                               v1 := ord ( (KStatusFlags2 AND $03)>0 );
                               v2 := (KStatusFlags2 AND $20) SHR 5;
                               if (v1 xor v2)=0 then begin  {no shifted!}
                                  if isBreak then KStatusFlags2 := KStatusFlags2 XOR shiftedbit
                                             else KStatusFlags2 := KStatusFlags2 OR shiftedbit;
                                  if not isBreak then begin
                                      KStatusFlags1 := KStatusFlags1 XOR shiftedbit;
                                      storekey (0, 82)
                                  end;
                               end;
                          end
    end;
end;

{ This forward is to be removed, because XStrData belongs to another module }
procedure XStrData; forward;

CONST
        { keyboard signals }
        AckSignal    = $FA;
        NakSignal    = $FE;


{************************************************************************
** ExtInt9h: extended int 9h management                                **
** ------------------------------------------------------------------- **
** IN:   AL: scancode                                                  **
** OUT:  Several registers trashed                                     **
*************************************************************************}
procedure ExtInt9h; far; assembler; 
label  Jmp1, Jmp2, Jmp3, Jmp4, NoBreak, Lop2, ShiftFound, Jmp5, Jmp6,
       NoPause, SimplyStore, Ende;
asm
        PUSH DX
        PUSH DI
        PUSH SI
        PUSH DS
        PUSH ES

        XOR    CX, CX 
        MOV    ES, CX        { we annihilate ES, for the extended handling }

        PUSH CS       {CS->DS}
        POP  DS


        {**** Test for Ack, Nak, XKey }

        CMP AL, $E0          { E0? then jump to Ende, and there E0prefixed will be updated }
        JE  Ende
        CMP AL, AckSignal    { Acknowledge signal? }
        JNE Jmp1             
        MOV CL, $10
        JMP Jmp2
Jmp1:   CMP AL, NakSignal    { No-acknowledge signal? }
        JNE Jmp3
        MOV CL, $20
Jmp2:   OR  ES:KStatusByte2, CL   { Ack or Nak signals:  }
        JMP Ende                  { update extended keyboard state }


        {**** Determine if it's break or make code }
        { Fill CX: BreakCode? }

Jmp3:
        XOR  CX, CX          { this will store BreakCode }
        TEST AL, $80         { break or make? }
        JZ   Jmp4            { if zero -> make }
        MOV  CL,1            { breakcode}
        AND  AL, $7F         { disable the break bit}

        { ********** Test for Break ********** }

Jmp4:
        TEST CL,$FF         { if NOT breakcode }
        JNZ  NoBreak
        MOV  BL, ES:[Offset KStatusByte1]
        TEST BL, E0Prefixed { and it is Extended }
        JZ   NoBreak
        CMP  AL, 70         { scancode 70 (break) }
        JNE  NoBreak

        MOV BX, CS:[offset XStrData+6]  {XBufferHead}
        MOV CS:[offset XStrData+8], BX  {XBufferTail}
        MOV BX, ES:[Offset BufferHead]
        MOV ES:BufferTail, BX
        INT $1B
        JMP Ende

        { ********** Scan the shifting keys ********** }


NoBreak:
        XOR BX, BX
        MOV DI, offset ExtInt9Data + 1
Lop2:   MOV DL, CS:[DI]
        CMP DL, AL
        JE  ShiftFound
        INC DI
        INC BL
        CMP BL, 8
        JB  Lop2
        JMP Jmp5
ShiftFound:
        PUSH BX         { FOUND: call DoShiftKeys with appropriate params }
        PUSH CX
        CALL DoShiftKeys
        JMP  Ende

        { ********** BreakCodes are no longer processed ********** }


Jmp5:   TEST CL, $FF    { no more breakcodes processed }
        JNZ  Ende 


        { ********** Test for Pause (=Ctrl+NumLock) ********** }

        CMP BL, 5                   { IF NumLock }
        JNE NoPause
        XOR AX, AX
        MOV ES, AX
        TEST ES:KStatusFlags1, 4    { AND Control... }
        JZ  NoPause                 { then PAUSE!! }
        OR  ES:KStatusFlags2,8      { Set pause bit. }
        JMP Ende                    { Goto final waiting loop. }


         { ********** If we reached this point, simply store with ASCII=0 ********** }

NoPause:

         CMP   AL, 83         { first: Ctrl+Alt+Del }
         JNE   SimplyStore
         MOV   BL, [ES:KStatusFlags1]
         AND   BL, 12
         CMP   BL, 12
         JNE   SimplyStore
         MOV   DL, APM_WarmReboot  { warm reboot! }
         CALL  APMProc
         JMP   Ende            { just in case }
         
         
SimplyStore:
         XOR   BL, BL         { otherwise, StoreKey ( 0, AL) }
         PUSH  BX
         PUSH  AX
         Call  StoreKey

Ende:

        {===== update the E0 flag in KStatusByte1 (0040h:0096h) ==}
        XOR CX,CX
        MOV ES,CX
        MOV CL, ES:[Offset KStatusByte1]      { CL := KStatusByte1 }
        AND CL, $FD          { Clear E0Prefixed flag }
        CMP AL, $E0          { AL=Scancode = E0h ? }
        JNE Jmp6
        OR  CL, E0Prefixed   { Put E0Prefixed flag }
Jmp6:   MOV ES:[Offset KStatusByte1], CL

        POP ES
        POP DS
        POP SI
        POP DI
        POP DX

end;

{------------------------------------------------------------------------
---- XStrings management functions (FuncX)                          -----
-------------------------------------------------------------------------
-- DISCARD: if no XStrings are required                                --
------------------------------------------------------------------------}

{************************************************************************
** ExtInt9Data: variables for XStrings management                      **
*************************************************************************}
procedure XStrData;
Begin
   ASM
     DD   0               { XBuffer: starts on 0 }
     DW   0,0             { XBufferHead and XBufferTail }
     DB   0               { PutXStr (bool): XString in process? }
     DD   0               { XString (ptr):  being processed }
     DB   0               { XStrPos (byte): position in string }
   End;
End;


{************************************************************************
** EfStoreKey2: store a key in the secondary buffer                    **
** ------------------------------------------------------------------- **
** IN:   AX: scancode/char pair to be stored                           **
** OUT:  BX, ES:DI are trashed                                         **
*************************************************************************}
Procedure EfStoreKey2; far; assembler;  { effectively store key in XBuffer. }
asm
        MOV  BX,CS:[Offset XStrData+4]   { buffer head to BX }

        LES  DI,CS:[Offset XStrData]
        PUSH BX                          { Increase to DI the Head offset }
        SHL  BX,1                        { (it is counted in words) }
        ADD  DI,BX
        POP  BX
        MOV  [ES:DI],AX                  { store the value }

        INC  BX                          { increase pointer, maximally is 64 }
        AND  BX,63
        MOV  [CS:Offset XStrdata+4], BX  { store again the buffer head }
end;


{************************************************************************
** Int16Handler: int 16h prologue                                      **
** ------------------------------------------------------------------- **
** INT:  before int16h is served, transfers keys from secondary to     **
**       primary/BIOS buffer                                           **
*************************************************************************}

Procedure Int16Handler; assembler;
label Skp1,Lop1,FindXStr,Lop2,Skp2,NextEntry,PutXStrChar,Skp3,
      ContinueLoop,NoAction;

ASM
      {** 1.- Push the registers **}
      PUSHF
      PushA       {to be replaced by the appropriate code!!!}
      Push DS
      Push CS
      Pop  DS
      Push ES

      {** 2.- Compute free space in primary buffer **}

      Xor Bx,Bx
      Mov Es,Bx
      Mov BX,ES:BufferEnd
      Sub BX,ES:BufferStart                 { Buffer size in Byte. }
      Mov AX,ES:BufferHead
      Sub ax,ES:BufferTail                  { Number of Bytes in buffer. }
      JNC Skp1                              { If BX negative, -BX is the free space! }
      Xor Bx,Bx
Skp1: Sub Bx,ax                             { AX = Free Bytes. }
      Shr Bx,1                              { AX = Free Entries. }
      Dec BX                                { One entry stays empty for administration. }

      {** 3.- Loop while room in buffer (BX) **}
Lop1: OR  bx,bx
      push bx
      JZ  NoAction

      {** INSIDE LOOP on free primary buffer space **}

           { if string being put, get character from there}
           mov al,CS:[offset XStrData+8]
           test al,$FF
           jne PutXStrChar

           { if secondary buffer is empty, exit the loop directly}
           Mov Bx,CS:[offset XStrData+4]   {XBufferHead}
           Cmp Bx,CS:[offset XStrData+6]   {XBufferTail}
           JE  NoAction

           { == No string being processed, secondary buffer non empty ==}

           { Get a key from secondary buffer to AX }
           Push ES
           Mov AX,CS:[offset XStrData+6] {XBufferTail}
           Shl Ax,1
           LES DI,CS:[offset XStrData]  {XBuffer}
           Add Di,Ax                   { Address of next key. }
           Mov Ax,ES:[DI]              { Read key. }
           Pop ES

           { if scancode is $FF, then it is XString, else store it }
           Cmp  Ah,$FF
           Je   FindXStr

           Call EfStoreKey1

           Jmp  NextEntry

FindXStr:  { find the XString in the buffer }
               cmp AL, LastXStr            { AL is the string number }
               ja  NextEntry               { if above: IGNORE }
           
               LES DI,CurrentLayout        { ES:DI -> XStrings }
               ADD DI,Offset DR.XStrings

               { skip n-1 XStrings }
               xor cx, cx        
               mov cl, al
               dec cl
               jz skp2

Lop2:          Mov Al,ES:[DI]              { Address of next XStr. }
               Xor Ah,Ah
               Inc AX
               Add DI,Ax
               LOOP Lop2
Skp2:

               { store address of XString }
               Mov [CS:offset XStrData+9],DI   {Word Ptr XString}
               Mov [CS:offset XStrData+11],ES  {Word Ptr XString+2}
               CMP Byte Ptr ES:[DI],0     { length of string = 0 ? }
               JZ  NextEntry                           { Then end. }

               MOV AL, 1
               MOV CS:[offset XStrData+13],AL {XStrPos}            { pointer in XString = 1 }
               MOV CS:[offset XStrData+8],AL     {PutXStr}     { processing XString = TRUE }

           { Move on the secondary buffer pointer }
NextEntry: Mov AX,CS:[offset XStrData+6]  {XBufferTail}
           Inc Ax
           And Ax,63
           Mov CS:[offset XStrData+6],Ax          { Pointer to next entry. } {XBufferTail}

           { Are we processing a XString? }
           mov al,CS:[offset XStrData+8]
           test al,$ff
           je  ContinueLoop

           { Put a character from the XString buffer }
PutXStrChar:
           Mov Al,CS:[offset XStrData+13]  {XStrPos}
           Xor Ah,AH
           Mov Di,Ax
           inc al
           mov CS:[offset XStrData+13],al
           Add Di,Word Ptr offset XStrData+9 {XString}    {!}
           mov al,[di]

           call EfStoreKey1

           { PutXStr:= XStrPos <= Length(XString^) }
           xor  al,al
           mov  di, [Offset XStrData+9]  {[Offset XString]}   {!}
           mov  ah, [di]
           cmp  ah, CS:[offset XStrData+13] {XStrPos}
           jb   Skp3
           inc  al
Skp3:      mov  CS:[offset XStrData+8],al   {PutXStr}

ContinueLoop:
      {** End of loop **}
      pop  bx
      dec  bx
      jmp  lop1

NoAction:
      pop  BX

      {** 4.- END: recover registers}
      Pop  ES
      Pop  DS
      PopA
      PopF

      JMP CS:OldInt16h
End;


{------------------------------------------------------------------------
---- KEYB 2 LAYOUT DATA INFORMATION, FIRST BLOCK                    -----
------------------------------------------------------------------------}

Procedure Data;                               {  about 2'75K free memory. }
Begin ASM
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0

DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
DD 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
End; End;

{------------------------------------------------------------------------
---- KEYB 2 INITIALIZATION STUB BEGINS HERE                         -----
-------------------------------------------------------------------------
---- From here onwards, code is no longer clean, as it is mostly       --
---- coming from xkeyb, and will mostly be replaced/removed in         --
---- prototype 4                                                       --
------------------------------------------------------------------------}

TYPE
       ActionTyp   = (Install,Uninstall,OverLoad,GetInfo,WrongVers,OtherDrv,FastHlp);



Procedure Keep(EndP : Pointer; Code : Byte);     { Terminate program but stay resident in memory. }

type   pWord       = ^Word;
Var    Laenge      : Word;                       { !The program will stay in memory only until the given address! }
       vES         : word;
Begin

   {free the environment block}
   vES := PWord (ptr(PrefixSeg,$2C))^;

   asm
      MOV ES, vES
      MOV AX, $4900
      INT $21
   end;

   {compute size and exit}
   Laenge:=ptrRec(Endp).Seg-PrefixSeg +     { Calculate program length. }
           (ptrrec(Endp).Ofs+15) Div 16;

   ASM
      MOV AH, 49
      MOV AL, Code
      MOV DX, Laenge
      INT $21
   END;
End;



Procedure Error(B : Byte);
Begin
   Case B of
      1 : Writeln('Different Version of Keyb installed.');
      2 : Writeln('Incompatible keyboard driver installed.');
      3 : Writeln('Resident part of keyb was NOT installed.');
      4 : Writeln('Specified file could not be opened.');
      5 : Writeln('The resident part of Keyb could not be removed.');
      6 : Writeln('Internal failure: Global memory space too small');
      7 : Writeln('Keyb requires (still!) an AT/286 or better');
{      8 : Writeln('Required configuration file PC437.KEY not found');}
      9 : Writeln('Overloading is not supported in this version');  
   End;
   Halt(b);
End;


{***********************************************************
 Load translation table.
 ***********************************************************}



{ +++++++++++ Special keys for XStrings +++++++++++ }

Const  KeyNames    : Array[1..18] of String[5] =
                     ('HOME','END','PU','PD','CL','CR','CU','CD','DEL','INS',
                      'CHOME' ,'CEND','CPU','CPD','CCL','CCR','F11','F12');
       KeyCodes    : Array[1..18] of Byte =
                     (71,79,73,81,75,77,72,80,83,82,119,117,132,118,115,116,87,88);

{$S+,I+,R+}

Var    XStrEnd     : Word;
       XStrs       : Array[1..100] of String; { The XStrs will first be deposited here and only later }
                                              { be copied to their data region. So we don't have to move }
                                              { all the following XStrs if we insert one. If XStrs exist already, }
                                              { they will be copied here first! }
       Int9Type    : byte;    {0=none, 1=chain, 2=extended}
const
       ShiftKeys   : Array[0..7] of Byte = (54, 42, 29, 56, 70, 69, 58, 82);   { ScanCodes of Shift keys. }


Function ProgramName:String;                { Finds out name and path of the current program. }
Var    EnvSeg,                              { Replaces ParamStr(0) as that won't work if environment is empty. }
       EnvOfs      : Word;
       S           : String;
Begin
   EnvSeg:=MemW[PrefixSeg:$2C];             { Environment Adress. }
   EnvOfs:=0;
   While MemW[EnvSeg:EnvOfs]>0 Do           { Seek end of environment. }
      Inc(EnvOfs);
   Inc(EnvOfs,4);
   S:='';
   While Mem[EnvSeg:EnvOfs]>0 Do            { Program name. Terminated by zero. }
   Begin
      S:=S+Char(Mem[EnvSeg:EnvOfs]);
      Inc(EnvOfs);
   End;
   ProgramName:=S;                          { Let's give it back ... }
End;



{keeps trailing \}
Function ProgramPath: string;
var
   s: string;
begin
     s := ProgramName;
     while s[length(s)]<>'\' do
        delete (s,length(s),1);
     ProgramPath := s
End; {ProgramPath}



Function GetNumber(Var Line:String; Var LinePtr:Byte):Byte; { Read byte from line. }
Var    S           : String[8];
       Num         : Byte;
       err         : Integer;
Begin
   S:='0';
   err := 0;

   While Line[LinePtr]=' ' Do               { Skip eventual spaces. }
      Inc(LinePtr);

   {regular case}
   While (Line[LinePtr]>='0') and           { until no more numbers follow or end of line reached. }
         (Line[LinePtr]<='9') and
         (LinePtr<=Length(Line)) Do
   Begin
      S:=S+Line[LinePtr];
      Inc(LinePtr);
   End;
   Val(S,Num,err);                          { Calculate value. }

   {Send result}
   if err=0 then GetNumber:=Num
            else GetNumber:=0
End;


Procedure SetKey(Var Line:String);          { Read sections with the name KEYS. }
Var    KeyNum      : Byte;
       LinePtr     : Byte;
       C           : Char;
       B           : Byte;
       flag       : boolean;
Begin
   LinePtr:=1;

   While Line[LinePtr]=' ' Do               { Skip eventual spaces. }
      Inc(LinePtr);

   {Add Aitor 1.10: Exx to read the extended scancodes}
   flag := upcase(line[lineptr])='E';
   if flag then inc(LinePtr);

   KeyNum:=GetNumber(Line,LinePtr)+128*ord(flag);      { Read keynumber. }

   If (KeyNum=0) or (KeyNum>256) Then    { Keynumber legal? }
   Begin
      Writeln('Illegal key number:');
      Writeln(Line);
   End;

   LayoutList^.TransTable[KeyNum,5]:=0;             { Key attribute = 0. }

   C:=Line[LinePtr];
   While C<>' ' Do                       { After key number, NCS follows optionally, then space. }
   with LayoutList^ do
   Begin
      Case UpCase(C) Of
         'N' : Inc(TransTable[KeyNum,5],2);   { Set Num Lock Bit. }
         'C' : Inc(TransTable[KeyNum,5],4);   { Caps. }
         'S' : Inc(TransTable[KeyNum,5],1);   { Scroll. }
      End;
      Inc(LinePtr);                      { Next char. }
      C:=Line[LinePtr];
   End;

   For B:=0 To 4 Do                      { Mappings for 5 levels. }
   with LayoutList^ do
   Begin
      While Line[LinePtr]=' ' Do
         Inc(LinePtr);                   { Skip space. }
      Case Line[LinePtr] of
         '#' : Begin                     { #+ASCII-Value of char. }
                  Inc(LinePtr);
                  TransTable[KeyNum,B]:=GetNumber(Line,LinePtr);
               End;
         '!' : Begin                     { Key mapped to XStr. }
                  Inc(LinePtr);
                    flag := upcase(line[lineptr])='C';  {!C means 199+ (for COMBI)}
                    if flag then inc(LinePtr);
                  TransTable[KeyNum,B]:=GetNumber(Line,LinePtr)+199*ord(flag);
                  Inc(TransTable[KeyNum,5],$80 shr B);  { Set XStr Bit for actual level. }
               End;
         Else  Begin                     { Normal mapping with chars. }
                  TransTable[KeyNum,B]:=Byte(Line[LinePtr]);
                  Inc(LinePtr);
               End;
      End;
   End;
End;

Procedure SetShifts(Var Line:String);       { Read scancodes of shift keys from config file. }
Var    B           : Byte;
       LinePtr     : Byte;
Begin
   LinePtr:=1;
   B:=0;
   While (LinePtr<=Length(Line)) and (B<=7) Do     { All scancodes need to be in one row! }
   Begin
      ShiftKeys[B]:=GetNumber(Line,LinePtr);       { And it needs to contain at least 8 scancodes! }
      Inc(B);
   End;

   If B<=7 Then                          { Less than 8 scancodes found? }
   Begin
      Writeln('Warning:');
      Writeln('Some shift keys remain undefined!');
      Writeln(Line);
   End;
End;


Function GetKeyByName(Var Line:String; Var LinePtr:Byte):String;
Var    KeyName     : String[10];            { For processing XStr definitions. }
       ShiftOffset : Byte;                  { Gets the scancode of a key by its name. }
       B           : Byte;                  { Names of keys are in KeyNames. }
       I           : Integer;               { The corresponding scancodes are in KeyCodes. }
Begin
   GetKeyByName:='';
   KeyName:='';

   While (LinePtr<Length(Line)) and         { Read name of key. }
         (Line[LinePtr]<>']') and
         (Length(KeyName)<7) Do
   Begin                                    { Copy char until ']', end of line or 7 chars read. }
      KeyName:=KeyName+UpCase(Line[LinePtr]);
      Inc(LinePtr);
   End;

   If Line[LinePtr]<>']' Then               { Key not terminated by ']' ? }
   Begin
      Writeln;
      Writeln('"]" missing in xstring definition:');
      Writeln(Line);
      Exit;
   End
   Else Inc(LinePtr);

   B:=0;
   Repeat                                   { Search key in list specified at the top of this program section. }
      Inc(B);
   Until (B=17) or (KeyName=KeyNames[B]);
   If B<17 Then                             { Key found => return its scancode. }
   Begin
      GetKeyByName:=#0+Char(KeyCodes[B]);   { We return a zero-Byte + the scancode. }
                                            { Returning string may be appended directly to the XStr. }
   End
   Else
   Begin                                    { Key not found. May be function key. }
      ShiftOffset:=0;
      Case KeyName[1] of                    { Scancodes of function keys (F) are number + an offset. }
         'S' : ShiftOffset:=83;             { Offset for Shift. }
         'C' : ShiftOffset:=93;             { Control. }
         'A' : ShiftOffset:=103;            { Alt. }
         'F' : ShiftOffset:=58;             { No modifying key. }
      End;
      If (ShiftOffset=0) or
         (
            (ShiftOffset>58) and
            (KeyName[2]<>'F')
         ) Then
      Begin                                 { Name of key is invalid. }
         Writeln;
         Writeln('Invalid keyname ',KeyName,' in xstring definition:');
         Writeln(Line);
         Exit;
      End;

      If ShiftOffset>58 Then Delete(KeyName,1,2) { Isolate function key number. }
                        Else Delete(KeyName,1,1);

      Val(KeyName,B,I);                     { and turn into value. }
      If (B<1) or (B>10) Then               { Only values 1-10 are allowed. }
      Begin
         Writeln;
         Writeln('Invalid function key F',B,' in xstring definition:');
         Writeln(Line);
         Exit;
      End;

      GetKeyByName:=#0+Char(B+ShiftOffset); { The scancode is key number + the level independent offset. }
   End;
End;

Function GetSpecKey(Var Line:String; Var LinePtr : Byte):String;
Var    Len         : Byte;                  { This function manages if a '\' appears in the XStr definition. }
       Number      : Byte;                  { If needed it calls the function GetKeyByName. }
       Select      : Char;
Begin
   Len:=Length(Line);
   GetSpecKey:='';

   If LinePtr>Len Then                      { Was '\' the last char in the XStr? }
   Begin
      Writeln;                              { Yes -> Error. }
      Writeln('Unexpected end of xstring definition:');
      Writeln(Line);
      Exit;
   End;

   Select:=UpCase(Line[LinePtr]);           { Evaluate char after \ . }
   Case Select of
      'N' : Begin
               GetSpecKey:=#13;             { \n = CR. }
               Inc(LinePtr);
            End;
      '\' : Begin
               GetSpecKey:='\';             { \\ = \. }
               Inc(LinePtr);
            End;
      'A',
      'S' : Begin                           { \Axxx = Chr(xxx) ; \Sxxx = Key(xxx). }
               If Lineptr+3<Len Then        { The number has got max. 3 digits. }
                  Byte(Line[0]):=           { Shorten length of line }
                  LinePtr+3;                { to ignore digits following eventually upon the number! }
               Inc(LinePtr);
               Number:=GetNumber(Line,LinePtr);
               Byte(Line[0]):=Len;          { Restore original line length. }

               If Select='A'
                  Then GetSpecKey:=Char(Number)
                  Else GetSpecKey:=#0+Char(Number);
            End;
      '[' : Begin                           { In [] we have the name of a key. }
               Inc(LinePtr);                { GetKeyByName calculates the corresponding code. }
               GetSpecKey:= GetKeyByName(Line,LinePtr);
            End;
      Else Begin                            { Illegal char after \ -> Error. }
              Writeln;
              Writeln('Syntax error in xstring definition:');
              Writeln(Line);
           End;
   End;
End;

Procedure ParseXStr(Var Line:String; Var LinePtr:Byte; Var Dest:String);
Begin                                       { This routine translates an XStr. }
   Dest:='';
   While LinePtr<=Length(Line) Do           { Process all chars until end of line. }
   Begin
      If Line[LinePtr]='\' Then             { Is char a backslash ? }
      Begin
         Inc(LinePtr);                      { Yes -> special char. }
         Dest:=Dest+GetSpecKey(Line,LinePtr);    { Meaning will be calculated by GetSpecKey. }
      End
      Else
      Begin                                 { No -> Take char unmodified. }
         Dest:=Dest+Line[LinePtr];
         Inc(LinePtr);
      End;
   End;
End;


Procedure SetXStr(Var Line : String);       { Process sections with label [XSTRINGS] }
Var    LinePtr     : Byte;
       XStrNum     : Byte;
Begin
   LinePtr:=1;
   XStrNum:=GetNumber(Line,LinePtr);     { Line starts with number of the XStr. }

   If (XStrNum=0) or (XStrNum>99) Then  { Number valid ? }
   Begin
      Writeln;
      Writeln('illegal xstring number:');
      Writeln(Line);
      Writeln('legal xstring numbers: 1-99.')
   End
   Else
   Begin
      If XStrNum>LastXStr Then LastXStr:=XStrNum;  { Number greater than LastXStr? -> Set new LastXStr. }
      Inc(LinePtr);                      { Exactly ONE space follows. Skip that. }

      XStrs[XStrNum]:='';
      ParseXStr(Line,LinePtr,XStrs[XStrNum]); { Let XStr translate by ParseXStr. }
   End;
End;

Procedure ReadCombis(Var S : String);
{ S[1]=First char of combinations of this list. }
{ S[2]=Number of combinations in this list. Needs to be 0 at the calling of this routine. }
Var    Loop        : Byte;
Begin
   With LayoutList^ do
   Begin
      Loop:=0;
      While (CombTab[Loop]<>0) and (CombTab[Loop]<>Byte(S[1])) Do
      Begin                                 { Search combination char. }
         Loop:=Loop+2*CombTab[Loop+1]+2;
      End;

      If CombTab[Loop]>0 Then               { Found combinations with this char. }
      Begin
         Byte(S[0]):=CombTab[Loop+1]*2+2;
         Move(CombTab[Loop],S[1],Byte(S[0])); { Copy existing combinations. }

         Move(CombTab[Loop+Byte(S[0])],CombTab[Loop],192-Loop);
                                            { The read combinations will be erased! }
                                            { Eventually reentered later in other form. }
      End;
   End;
End;

Procedure WriteCombis(Var S : String);
Var    Loop        : Byte;
Begin
   Loop:=0;
   With LayoutList^ Do
   Begin
      While CombTab[Loop]>0 Do
         Loop:=Loop+2*CombTab[Loop+1]+2;

      If Loop+Length(S)>190 Then
      Begin
         Writeln(#10'Warning: Overflow of combination char table. ');
         Writeln('combinations with char ',S[1],' inactive.');
      End
      Else
      Begin
         Move(S[1],CombTab[Loop],Length(S)); { Insert the crap. }
         CombTab[Loop+Length(S)]:=0;
      End;
   End;
End;


Procedure SetCombi(Var Line : String);
Var    CombiChars  : String;
       LinePtr     : Byte;
Begin
   LinePtr:=1;
   While (Line[LinePtr]=' ') and (LinePtr<=Length(Line)) Do
      Inc(LinePtr);                         { Skip blanks. }
   If LinePtr>Length(Line) Then Exit;       { Nothing in the row? }

   CombiChars:=Line[LinePtr]+#0;            { First char of the combination after CombiChars. }
   ReadCombis(CombiChars);                  { Read eventually existing combinations with this char. }

   Inc(LinePtr);
   While LinePtr<=Length(Line) Do           { Process line. }
   Begin
      If Line[LinePtr]<>' ' Then            { Blank ? *shiver* }
      Begin
         If Line[LinePtr]='#' Then          { ASCII-Value for char after #. }
         Begin
            Inc(LinePtr);                   { Get ASCII-Value and translate into char. }
            CombiChars:=CombiChars+Char(GetNumber(Line,LinePtr));
         End
         Else
         If Line[LinePtr]='!' Then          { ! erases an existing definition. }
         Begin
            CombiChars[0]:=#2;
            CombiChars[2]:=#0;
         End
         Else
         Begin
            CombiChars:=CombiChars+Line[LinePtr];
            Inc(LinePtr);                   { Else copy chars directly. }
         End;
      End
      Else
         Inc(LinePtr);                      { Skip blanks. }
   End;

   If Odd(Length(CombiChars)) Then
   Begin                                    { Here we have just a halve pair of chars -> nonsense. }
      Writeln(#10'Warning: Halve pair of chars at combination char:');
      Writeln(#10,Line);
      Writeln(#10'Line was shortened to full pairs of chars!');
      Dec(CombiChars[0]);
   End;

   CombiChars[2]:=
      Char(Length(CombiChars) Shr 1 -1);    { Half length of list minus 1 is number of combinations. }
   If CombiChars[2]>#0 Then
      WriteCombis(CombiChars);              { Save combinations. }
End;



Procedure CopyXStrs(BufSize:Word);           { Shorten XStrings and save to buffer. }
Var    B           : Byte;
       BufferFull  : Boolean;
Begin
   XStrEnd:=0;
   BufferFull:=False;

   With LayoutList^ Do
   Begin
      B:=1;
      While (B<=LastXStr) and not BufferFull Do  { Until all XStrs are inserted or buffer is full. }
      Begin
         If XStrEnd+Length(XStrs[B])>=BufSize Then    { Will the string still fit into the buffer ? }
         Begin
            Writeln;                                  { No. }
            Writeln('Not enough buffer space for xstring declaration:');
            Writeln(XStrs[B]);
            XStrs[B]:='';                             { Erase XString. }
            BufferFull:=(BufSize-XStrEnd)=0;          { Buffer completely full ? => Exit loop. }
         End
         Else
         Begin
            Move(XStrs[B],XStrings[XStrEnd],Byte(XStrs[B][0])+1); { Copy XStr into the buffer and }
            XStrEnd:=XStrEnd+Byte(XStrs[B][0])+1;     { recalculate end of occupied memory. }
            Inc(B);
         End;
      End;
      LastXStr:=B-1;                        { If not all strings could be inserted. }

      If XStrBufSize=0 Then
         XStrBufSize:=XStrEnd;              { Buffer size not declared => minimize. }
   End;
End;

Procedure GetOldXStrs;                      { Read XStrings from resident copy of Keyb. }
Var    B           : Byte;
       W           : Word;
Begin
   With LayoutList^ Do                          { The data region of the resident copy will be used! }
   Begin
      W:=0;
      For B:=1 To LastXStr Do               { Go through all XStrs. }
      Begin
         Move(XStrings[W],XStrs[B],XStrings[W]+1);    { Copy XStr to other data region. }
         W:=W+XStrings[W]+1;                { Calculate address of next XStr. }
      End;
   End;
End;




Procedure ExpandFileName(Var Name : String);{ If needed, extend name of config file. }
Begin
  { first, we are loading a .KEY file }
   If Pos('.',Name)=0 Then    { filename has no extension -> add .KEY }
      Name:=Name+'.Key';

   If (Pos('\',Name)=0) and
      (Pos(':',Name)=0) Then  { filename contains no path -> add program path }
   Begin
      Name := FSearch (Name, ProgramPath+';'+GetEnv('PATH'));  {RQ 1.6-1.7}
      Name := FExpand (Name);
   End;
End;



Type   SectionTyp  = (Keys,Shifts,XStrings,Comment,List,Continue,Combi);

Function GetSection(Var Line : String):SectionTyp;    { Evaluate section name. }
Var    B           : Byte;
Begin
   B:=2;
   While (B<Length(Line)) and               { Make line uppercase. }
         (Line[B]<>']') Do                  { Ignore all chars after ']'. }
   Begin
      Line[B]:=UpCase(Line[B]);
      Inc(B);
   End;
   Line[0]:=Char(B);                        { Ignore chars after ] . }

   If Line='[KEYS]' Then GetSection:=Keys
   Else If Line='[SHIFTS]' Then GetSection:=Shifts
   Else If Line='[XSTRINGS]' Then GetSection:=XStrings
   Else If Line='[COMMENT]' Then GetSection:=Comment
   Else If Line='[LIST]' Then GetSection:=List
   Else If Line='[CONTINUE]' Then GetSection:=Continue
   Else If Line='[COMBI]' Then GetSection:=Combi
   Else
   Begin
      Writeln('Warning:');
      Writeln('Unknown section ',Line,' found.');
      Writeln('Skipping section.');
      GetSection:=Comment;
   End;
End;

Procedure ReadConfigFile(Name:String; BufSize : Word);   { Read config file. }
Var    B           : Byte;
       S           : String;
       Section     : SectionTyp;
       Line        : String;
       ConfigFile  : Text;
       FileN       : byte;    {number of file being parsed}
       SingleName  : string;
       TrueFile    : boolean;
Label  Cont;
Begin
   For B:=1 To 100 Do                       { Clear XString workspace. }
      XStrs[B]:='';

   If BufSize=$FFFF Then GetOldXStrs;       { Read old xstrings. BufSize=FFFFh -> XKeyb resident installiert. }

   TrueFile := FALSE;
   SingleName := 'PC437';

Cont:
   ExpandFileName(SingleName);
   Assign(ConfigFile,SingleName);
{$I-}
   Reset(ConfigFile);
{$I+}
   If IOResult<>0 Then begin
     if truefile then Error(4)
                 else begin
                        truefile   := TRUE;
                        SingleName := Name;
                        goto cont
                 end
   end;

   If TrueFile then LayoutList^.ConfigFile := SingleName;
   Section:=Comment;

   While not Eof(ConfigFile) Do             { Read whole file. }
   Begin
      Readln(ConfigFile,Line);

      If (Length(Line)>0) and (Line[1]<>';') Then                { Ignore empty lines. }
         If Line[1]='[' Then Section:=GetSection(Line)
         Else
            Case Section of
               Keys     : SetKey(Line);
               Shifts   : SetShifts(Line);
               XStrings : SetXStr(Line);
               List     : Writeln(Line);
               Combi    : SetCombi(Line);   { Define combination chars. }
               Continue : Begin             { Proceed with next file. }
                             Close(ConfigFile);
                             SingleName:=Line;
                             Goto Cont;
                          End;   {en FIleN>1}
            End
      Else If Section=List Then Writeln;
   End;
   Close(ConfigFile);
   if trueFile then begin
       WriteLn ('Installed ',SingleName);
   end else begin
       truefile   := TRUE;
       SingleName := Name;
       goto cont
   end;

   With LayoutList^ Do
      If BufSize<$FFFF Then                 { No resident installation yet. }
      Begin
         XStrBufSize:=BufSize;
         If BufSize>1024 Then BufSize:=1024;{ For installation max. 1K buffer, else we overwrite our code! }
         If BufSize=0 Then BufSize:=1024;   { No size given? -> minimal, up to 1K. }
      End
      Else BufSize:=XStrBufSize;            { If installed already, resident copy declares buffer size. }

   CopyXStrs(BufSize);                      { Put XStrs to their data region. }

End;


{***********************************************************
 Main program & Check for already installed driver.
 ***********************************************************}

Type   BCDString   = String[2];

Function BCD(B : Byte) : BCDString;         { Translate BCD number into string. }
Begin
   If B>15 Then BCD:=Char(B shr 4 + 48)+Char(B and 15 + 48)
           Else BCD:=Char(B+48);
End;

Function VS(W : Word) : String;             { Returns version number as string. }
Var    S           : String;
Begin
   S:=BCD(LO(W));                           { Turn second value into string. }
   If Length(S)<2 Then S:='0'+S;            { Eventually add leading zero. }
   VS:=BCD(Hi(W))+'.'+S;                    { Add first value and point. }
End;

Function TestInstallation : Byte;
{ Check whether a copy of XKeyb is already installed. }
{ Result:   0 -> No keyboard driver installed. }
{           1 -> Identical version of XKeyb installed. }
{                >> LayoutList will be set to data region of resident copy. }
{           2 -> Different version of XKeyb installed. }
{           3 -> Different keyboard driver installed. }
label  NoKeyb, OtherKeyb, OtherVersion, EndTest;
var    r: byte;
begin

  asm
      mov  ax, $AD80
      xor  cx,cx
      mov  bx,cx
      int  $2F

      cmp  al, $FF      {=== Part 1: any-KEYB? ==}
      jne  NoKeyb

      mov  ax,1

      test bx,$FFFF     {=== Part 2: which KEYB? ==}
      jnz  OtherKeyb
      test cx,$FFFF
      jnz  OtherKeyb

      cmp  dx,Version   {=== Part 3: which version? ==}
      jne  OtherVersion

      mov  [ds:offset LayoutList],di
      mov  [ds:offset LayoutList+2],es
      jmp  endTest      { ax=1 already! }

OtherKeyb:
      inc  ax

OtherVersion:
      inc  ax
      jmp  endTest
      
NoKeyb:
      xor  ax,ax

endTest:
      mov r, al
  end;
  TestInstallation := r
end;


Procedure Remove;                           { Remove Keyb from memory. }
Var    IntVec      : Array[Byte] of Pointer absolute 0:0;
       MultiHand   : Pointer;
       Int16Hand   : Pointer;
       Int9Hand    : Pointer;
       Int15Hand   : Pointer;
       s           : word;
Begin
{ Check whether removing is possible. }

   MultiHand:=LayoutList;
   ptrrec(MultiHand).Ofs := Ofs(MultiplexHandler);
   Int16Hand:=LayoutList;
   ptrrec(Int16Hand).Ofs := Ofs(Int16Handler);
   Int9Hand :=LayoutList;
   ptrrec(Int9Hand).Ofs  := Ofs(Int9h);
   Int15Hand :=LayoutList;
   ptrrec(Int15Hand).Ofs := Ofs(Int15h);

   if not (
         ((IntVec[$09]=Int9Hand) or (not assigned(OldInt9h))) and
         ((IntVec[$16]=Int16Hand) or (LastXStr=0)) and
         (IntVec[$2F]=MultiHand) and
         (IntVec[$15]=Int15Hand) )

   Then Error(5);      { Removing impossible. Interrupt vectors were changed by another program. }

{ Uninstall it. }
      if assigned (OldInt9h) then SetIntVec ($9, OldInt9h);
      if LastXStr>0 then SetIntVec ($16, OldInt16h);
      SetIntVec ($15, OldInt15h);
      SetIntVec ($2F, OldInt2Fh);



      s := PtrRec(Int15Hand).seg-16;           { Program segment. }

      asm
         mov  ah,$49
         mov  es,s
         int $21
      end;

   Writeln('Resident part of Keyb removed.');
End;

Procedure ShowInfo;
Begin
      Writeln('Active definition file     : ',LayoutList^.ConfigFile);
      Writeln('Number of XStrings defined : ',Byte( Ptr(Seg(LayoutList^{DatSeg^}) , Ofs(LastXStr))^ ));
      Writeln('XString buffer size        : ',{DatSeg}LayoutList^.XStrBufSize,' Bytes');
End;

Function PerformParam:ActionTyp;            { Evaluate parameters. }
Var    B           : Byte;
       I           : Integer;
       S           : String;
       ConFileName : String;
       Action      : ActionTyp;
       Installed   : Byte;
       XStrBufSize : Word;
       FastDatSeg  : Word;
       T           : text;
       SChar       : char;
Begin
   FastDatSeg:=CSeg;
   ConFileName:='';
   XStrBufSize:=0;
   Action:=GetInfo;
   Installed:=TestInstallation;

   If Installed=1 Then                      { If already installed get certain values from resident GlobalDS. }
      Move(Ptr(ptrrec(LayoutList).Seg , Ofs(OldInt9h))^,
           OldInt9h,21);

   asm
      mov  ax,$3700
      int  $21
      mov  SChar,dl
   end;

   For B:=1 To ParamCount Do                { Process all parameters. }
   Begin
      S:=ParamStr(B);
      If S[1]=SChar Then            { parameter starts with switch char. }
      Begin
         Case Upcase(S[2]) of
            {/X was removed here, because you no longer can insert XStrings via the API }
            'U' : Begin                     { Uninstall.}
                     Action:=Uninstall;
                  End;
            'Q' : Begin                     { Quit. Ignore LIST sections. }
                     Close(OutPut);
                     Assign(OutPut,'Nul');
                     Rewrite(OutPut);
                  End;
            'I' : Begin                     { Install. Ignore other driver. }
                     Installed:=0;
                     Action:=Install;
                  End;
            '9' : case s[3] of
                              '+': Int9Type := 2;
                              '*': Int9Type := 1;
                              '-': Int9Type := 0;
                  else begin
                         Writeln;
                         Writeln('Invalid modifier -  ',S);
                       end;
                  end;
            '?' : begin
                        Action := FastHlp;        { Show the fast help }
                  end;
            Else
            Begin                           { Unknown option required -> Error. }
               Writeln;
               Writeln('Invalid modifier -  ',S);  {Aitor 1.7}
            End;
         End;
      End
      Else
      Begin
         ConFileName:=S;      { a parameter without / is the .KEY file name }
         Case Installed of
            0:Action:=Install;
            1:Action:=OverLoad;
         End;
      End;
   End;

   if action<>FastHlp then
   Case Installed of
      2:Action:=WrongVers;
      3:Action:=OtherDrv;
   End;

   if action=OverLoad then begin   {when file not found, abort! }
       ExpandFileName(S);
       Assign(t,S);
       {$I-}
       Reset(t);
       {$I+}
       If IOResult<>0
          then error(4)
          else close(t);
   end;

   Case Action of
      Install          : ReadConfigFile(ConFileName,XStrBufSize);
      OverLoad         : Error(9);  {ReadConfigFile(ConFileName,$FFFF);}
      GetInfo          : If Installed=1 Then ShowInfo Else Error(3);
      Uninstall        : If Installed=1 Then Remove Else Error(3);
      WrongVers        : Error(1);
      OtherDrv         : Error(2);
   End;

(*
   If Installed=1 Then  { If already installed, write certain values into resident GlobalDS. }
      Move(OldInt9h,
           Ptr(ptrrec(LayoutList).Seg , Ofs(OldInt9h))^, 20);
*)

   PerformParam:=Action;
End;

procedure ShowFastHelp;
begin
    Writeln ('("Taurus") Second generation keyboard driver for FreeDOS');
    WriteLn ('License:  GNU-GPL 2.0 or later');
    WriteLn ('(c) Aitor Santamara  - 2003');
    WriteLn;
    WriteLn ('KEYB  layoutFile [/Q] [/I] [/9{+|*|-}]');
    WriteLn ('KEYB  /U');
    WriteLn ('KEYB  /?');
    WriteLn;
    WriteLn ('LayoutFile  File containing the information for your keyboard');
    WriteLn ('/Q          (Quiet) Avoid Listing information on the layout file');
    WriteLn ('/I          (force Install) Install keyb unconditionally');
    WriteLn ('/U          (uninstall) unloads the driver from memory');
    WriteLn ('/9{+|*|-}   int 9h handler mode ');
    WriteLn ('            + Full handler ');
    WriteLn ('            * Chained handler ');
    WriteLn ('            - No handler ');
    WriteLn ('/?          shows this help');
    WriteLn;
    WriteLn ('More information: NOTES.TXT')
end;


type pbyte=^byte;


Var  Action      : ActionTyp;
     CutHere     : pointer;
     pb          : pByte;
     pp          : ^pointer;


Begin

   If Ofs(EoDS)>Ofs(CallKEYBFunc) Then Error(6);

   WriteLn ('KEYB ',VerS,': keyboard driver for FreeDOS (/?: more info)');


   {*** Variable initialisation ***}
   {XBuffer}
   pp  := @XStrData;
   pp^ := Ptr(PrefixSeg,128);

   { Layout pointers }
   LayoutList   := @DR(@Data^);              { Pointer to Layout information }
                                            { If necessary, this setting will be changed by TestInstallation. }

   { KEYB Functions }
   APMProcedure := APMProc;
   DoCommands   := ProcessCommands;
   CutHere      := NIL;
   Int9Type     := 0;

   {*** Parse commandline, act accordingly ***}

   Action:=PerformParam;                    { Evaluate command line parameters. }

   {*** Embryo of module discardation ***}
   if LastXStr>0 then begin         { OPTION 1: With XStrings }
       ExtInt9Proc := ExtInt9h;
       CurStoreKey := EfStoreKey2;  { done! }
   end else begin
       CurStoreKey := EfStoreKey1;
       case Int9Type of
                       0:  begin { NO INT9 handler! }
                              CutHere  := @APMProc;
                              APMProcedure := NIL
                           end;
                       1:  begin  {chained INT9handler}
                              CutHere  := @ExtInt9Data;
                              ExtInt9Proc := NIL
                           end;
                       else begin  {full int9h}
                              CutHere      := @XStrData;
                              ExtInt9Proc  := ExtInt9h;
                       end;
       end
   end;


   {  now efectively move the data }
   if assigned (CutHere) then begin
       Move ( pbyte(@data)^, pbyte(CutHere)^, sizeof(DR0)+2+LayoutList^.XStrBufSize );
       LayoutList := CutHere;
   end;

   CurrentLayout := LayoutList;

   {*** Do the actions ***}

   If Action=FastHlp Then
      ShowFastHelp
   Else
   If Action=Install Then
   With LayoutList^ Do
   Begin
      XStrBufSize:=                         { Round up end of buffer to segment border. }
         XStrBufSize
           +(
               16
              -Ofs(XStrings[XStrBufSize]) and $F
            )
            and $F;

      GetIntVec($2F,OldInt2Fh);
      if @CurStoreKey=@EfStoreKey2 then GetIntVec($16,OldInt16h);
      GetIntVec($15,OldInt15h);
      if int9Type>0 then GetIntVec($9 ,OldInt9h);

      Move(Ptr(DSeg,0)^,Ptr(CSeg,0)^,Ofs(EoDS)); { Copy data into resident GlobalDS. }

      if Int9Type>1 then begin {transfer ShiftKeys}
         Move ( ShiftKeys, Ptr(CSeg, ofs(ExtInt9Data)+1)^, 8);
         pb := ptr (CSeg, ofs(ExtInt9Data));
         pb^ := 0;
      end;

      { Set the vectors }
      SetIntVec($2F,@MultiPlexHandler);
      if @CurStoreKey=@EfStoreKey2 then SetIntVec($16,@Int16Handler);
      SetIntVec($15,@Int15h);
      if int9Type>0 then SetIntVec($9, @Int9H);


      { End, make it resident }
      Keep(@XStrings[XStrBufSize],0);
 
      GlobalDS;                              { Avoids removal of GlobalDS by the linker. }
      ExtInt9Data;
      XStrData;
   End;
END.

