;~~
; UUDMA -- Universal UDMA driver   Written 15-Nov-2003 by Jack R. Ellis
;
; UUDMA is free software.  You can redistribute and/or modify it, under
; the terms of the GNU General Public License (hereafter called GPL) as
; published by the Free Software Foundation, either version 2 of GPL or
; any later versions, at your option.  UUDMA is distributed in the hope
; that it will be useful, but WITHOUT ANY WARRANTY and without even the
; implied warranties of MERCHANTABILITY nor of FITNESS FOR A PARTICULAR
; PURPOSE!  See the GPL for details.  You ought to have received a copy
; of the GPL with these UDMA files.  If not, write to the Free Software
; Foundation Inc., 59 Temple Place Ste. 330, Boston, MA 02111-1307 USA.
; Their Internet address is <http://www.gnu.org/>
;
; Special thanks to Luchezar I. Georgiev for his INESTIMABLE advice and
; help in research, revisions, enhancement and testing of this driver!!
;
; This is a DOS driver designed to handle 1 to 4 UltraDMA hard-disks on
; PC motherboards having a VIA 8235 or equivalent chipset.   The driver
; will determine which of the IDE units (primary-master, primary-slave,
; secondary-master, secondary-slave) are in fact UltraDMA hard-disks at
; initialization and will run all such disks.	All UltraDMA disks from
; mode 0 ATA-16 to mode 7 ATA-166 may be used.	 ATAPI devices or other
; low-speed IDE units (CDROM, ZIP etc.) should not be mixed on the same
; IDE channel with UltraDMA disks, so the BIOS does not need to set the
; disk to a lower "common" UltraDMA mode, as ATA/ATAPI specs require.
;
; An UltraDMA disk is assumed to support full LBA mode (63 sectors, 255
; heads, and a specified cylinder count).   "LBA mode" I-O requests are
; supported for FreeDOS, MS-DOS V7.xx or other systems that allow them.
; LBA values over 28 bits cause the driver to use "Read/Write Extended"
; DMA commands and require an ATA-6 or newer hard-disk.	  LBA values of
; 28 bits or less will use regular DMA commands.   24-bit "CHS mode" is
; also supported for MS-DOS V6.xx and earlier.	 Data accessed with CHS
; calls must be located in the initial 8-GB of the disk.
;
; The driver intercepts BIOS INT13 read or write requests only.	  Other
; INT13 requests (including seeks) and read/write requests with invalid
; parameters will be "passed" back to the BIOS or some other driver for
; handling.   If a user I-O buffer is not DWORD aligned, crosses a 64K-
; boundary or fails a VDS "lock", the I-O request will use a 64K buffer
; in XMS memory with UltraDMA to or from the buffer, to avoid "passing"
; these requests to the BIOS for execution in slow PIO mode!   Although
; UltraDMA specifies word-aligned buffers, ERRATA in some chipsets does
; require DWORD alignment and avoiding a 64K DMA address boundary!
;
; Revision History:
; ----------------
;   V1.0    6-Nov-03   JRE   Initial release (had been named UDMA-E)
;   V1.1    7-Nov-03   JRE   Used 80386 test from V5.9 UDMA
;   V1.2   12-Nov-03   JRE   No "timeout error", other size reductions
;   V1.3   13-Nov-03   JRE   "DoIO" does ALL I-O, "XMS error" now 0FFh
;   V1.4   14-Nov-03   JRE   Fixed DMA-status reset
;   V1.5   15-Nov-03   JRE   Added all UDMA init functions but ctlr. name

; General Program Equations

%define VER 'V1.5, 15-Nov-2003.'
SECSCYL equ	255*63		; LBA sectors per cylinder
HEADS	equ	255		; LBA heads
SECSHD	equ	63		; LBA sectors per head
RDYTO	equ	7		; 100-msec ready timeout
RBYTES	equ	(2*1024*65536*12/14318180+1)*512 ; Read-test byte count
	; ( = microseconds per tick, but adjusted for binary megabytes)
	; Number of reads this length per tick = transfer rate in MB/s
BIOSTMR equ	46Ch		; BIOS "tick" timer address
HDISKS	equ	475h		; BIOS hard-disk count address
VDSFLAG equ	47Bh		; BIOS "Virtual DMA" flag address
DMAERR	equ	8		; DMA-error return code
HERROR	equ	0E0h		; IDE-error return code
				; (XMS-error return code is 0FFh)
CR	equ	0Dh		; ASCII carriage-return
LF	equ	0Ah		; ASCII line-feed

; IDE Controller Register Definitions

CDATA	equ	1F0h		; Data port
CSUBCM	equ	CDATA+1		; Subcommand register
CSECCT	equ	CDATA+2		; I-O sector count
CDSEL	equ	CDATA+6		; Disk-select and upper LBA
CCMD	equ	CDATA+7		; Command register
CSTAT	equ	CDATA+7		; Primary status register
CSTAT2	equ	CDATA+206h	; Alternate status register

; Controller Status and Command Definitions

BSY	equ	80h		; IDE controller is busy
RDY	equ	40h		; IDE disk is "ready"
FLT	equ	20h		; IDE disk has a "fault"
DRQ	equ	8		; IDE data request
ERR	equ	1		; IDE general error flag
DMI	equ	4		; DMA interrupt has occured
DME	equ	2		; DMA error has occurred
DRCMD	equ	0C8h		; DMA read command (write is 0CAh,
				;     LBA-48 commands are 25h/35h)
SETM	equ	3		; Set Mode subcommand
SETF	equ	0EFh		; Set Features command
LBABITS equ	0E0h		; Fixed high-order LBA commands

; LBA "Device Address Packet" Layout

struc	DAP
DapPL	resb	1		; Packet length
	resb	1		; (Reserved)
DapSC	resb	1		; I-O sector count
	resb	1		; (Reserved)
DapBuf	resd	1		; I-O buffer address (roffset & segment)
DapLBA	resd	2		; Disk logical block address
endstruc

; DOS "Request Packet" layout

struc	RP
	resb	2		; (Unused by us)
RPOp	resb	1		; Opcode
RPStat	resw	1		; Status word
	resb	9		; (Unused by us)
RPSize	resd	1		; Resident driver size
endstruc
RPERR	equ	8003h		; "Strategy" packet error flags
RPDON	equ	100h		; "Strategy" packet done flag

; DOS Driver Device Header

@	dd	0FFFFFFFFh	; Link to next device-header block
	dw	8000h		; Driver "device attributes"
	dw	Strat		; "Strategy" routine offset
PCICm	equ	$-2		; (PCI UDMA command address, after init)
	dw	DevInt		; "Device-Interrupt" routine offset
ChIDE	equ	$-2		; (IDE channel "offset", after init)
ChPCI	equ	$-1		; (PCI channel "offset", after init)
	db	16,16,'UUDMA$',0; Driver name (arrows avoid user errors)

; Resident Driver Variables

VLF	db	0		; VDS "lock" flag
Units	dd	0FFFFFFFFh	; IDE "active units" table  (set by init)
PRDAd	dd	IOAdr		; PRD command-list address  (set by init)
	db	0		; IDE "upper" sector count  (always zero)
LBAHi	db	0, 0, 0		; IDE "upper" LBA bits 24-47
SecCt	db	0		; IDE "lower" sector count  (always used)
LBALo	db	0, 0, 0		; IDE "lower" LBA bits 0-23
DSCmd	db	0		; IDE disk-select and LBA commands
IOCmd	db	0		; IDE command byte
Handl	dw	0		; XMS buffer handle number  (set by init)
XMSOf	dd	0		; XMS 32-bit buffer offset  (set by init)
IOLen	dd	0		; XMS and VDS I-O byte count
XMSSH	dw	0		; XMS source block handle   (00h if DS:SI)
XMSSA	dd	0		; XMS 32-bit source address (may be DS:SI)
XMSDH	dw	0		; XMS dest. block handle    (00h if ES:DI)
IOAdr	dd	0		; XMS dest. & VDS/DMA addr. (may be ES:DI)
VDSOf	equ	XMSSH		; VDS parameters all SHARE the XMS block!
VDSSg	equ	XMSSA+2
DMALn	dd	80000000h	; DMA byte count and "end" flag

; Driver Main Routine.	 For CHS requests, at entry the registers contain:
;
;   AH	    Command code.  We handle only 2 read and 3 write
;   AL	    I-O sector count
;   CH	    Lower 8 bits of starting cylinder number
;   CL	    Starting sector number and upper 2 bits of cylinder number
;   DH	    Starting head number
;   DL	    Unit number.   We handle UltraDMA hard-disks of 80h and up
;   ES:BX   I-O buffer address
;
; For LBA requests, at entry the registers contain:
;
;   AH	    Command code.  We handle only 42h read and 43h write
;   DL	    Unit number.   We handle UltraDMA hard-disks of 80h and up
;   DS:SI   Pointer to Device Address Packet ("DAP"), described above

NotUs	popf			; Request not for us - reload CPU flags
	pop	es		; Reload CPU registers
	popa
	jmp	0000:0000	; "Pass" request back to INT13 chain
@PrvI13 equ	$-4		; (Previous INT13 vector, set by Init)
Entry	pusha			; Driver entry - save CPU registers
	push	es
	pushf			; Save CPU flags
	mov	bp,4		; Reset active-units table index
NxtUnit dec	bp		; Any more active units to check?
	js	NotUs		; No, exit fast!
	cmp	dl,[cs:bp+Units-@] ; Does request unit match our table?
	jne	NxtUnit		; No, see if more table entries remain
	mov	dl,ah		; Get command code in DL-register
	shr	dl,1		; Shift out write command bit
	dec	dl		; Is this a CHS read or write request?
	jz	CheckSC		; Yes, validate sector count
	cmp	dl,20h		; Is this an LBA read or write request?
	jne	NotUs		; No, exit fast!
	cmp	dword [si+DapBuf],byte -1 ; 64-bit I-O buffer address?
	je	NotUs		; Yes?	Let BIOS handle this request!
	mov	al,[si+DapSC]	; Get "DAP" sector count
	les	bx,[si+DapBuf]	; Get "DAP" I-O buffer address
	mov	di,[si+DapLBA+4]; Get "DAP" logical-block address
	mov	dx,[si+DapLBA+2]
	mov	si,[si+DapLBA]
CheckSC dec	al		; Is sector count from 1 to 128?
	js	NotUs		; No?  Let BIOS handle this request!
	popf			; Valid request - discard entry flags
	sti			; Enable CPU interrupts
	push	ds		; Save DS-register
	inc	ax		; Restore sector count
	xchg	ax,cx		; Save command byte and sector count
	shl	ch,1		; Is this an LBA request?
	js	ReqExe		; Yes, go execute request below
	mov	si,SECSHD	; CHS - get starting sector in SI-reg
	and	si,ax
	mov	di,dx		; Get starting head in DI-reg
	shr	al,6		; Get cylinder number in AX-reg
	xchg	al,ah
	mov	dx,SECSCYL	; Convert cylinder to sectors
	mul	dx
	xchg	ax,di		; Swap low-order and head number
	mov	al,SECSHD	; Convert head to sectors
	mul	ah
	add	di,ax		; Add to cylinder sectors
	adc	dl,dh
	dec	si		; Get starting sector number -1
	add	si,di		; Add to cylinder/head sectors
	adc	dl,dh
	xor	di,di		; Reset upper LBA address bits
ReqExe	push	cs		; Set our DS-reg
	pop	ds
	xchg	ax,bx		; Get user buffer offset in AX-reg
	xor	bx,bx		; Zero BX-reg. for relative commands
	mov	[bx+SecCt-@],cl ; Set I-O sector count
	mov	[bx+LBALo-@],si ; Set disk LBA (bits 24-31 set below)
	mov	[bx+LBALo+2-@],dl
	mov	[bx+LBAHi+1-@],di
	shr	bp,1		; Get slave-select bit in carry
	jz	ChOffs		; Secondary channel I-O request?
	mov	bp,880h		; Yes, get offsets for secondary
ChOffs	mov	[bx+ChIDE-@],bp ; Set IDE and PCI channel offsets
	mov	dl,(LBABITS/32) ; Initialize LBA command byte
	rcl	dl,5
	shl	ch,3		; Get read/write request as 20h/30h
	or	di,di		; Anything in LBA bits 32-47?
	jnz	GetCmd		; Yes, get LBA-48 IDE command
	test	dh,0F0h		; Anything in LBA bits 28-31?
	jnz	GetCmd		; Yes, get LBA-48 IDE command
	or	dl,dh		; Put LBA bits 24-27 in LBA command
	mov	dh,0		; Reset IDE byte 1 (LBA bits 24-31)
	shr	ch,3		; Get read/write request as 04h/06h
	or	ch,(DRCMD+1)	; Get LBA-28 IDE read/write byte + 5
GetCmd	xor	ch,5		; Get final IDE command byte
	mov	[bx+LBAHi-@],dh ; Set disk LBA bits 24-31
	mov	dh,ch		; Set LBA and IDE command bytes
	mov	[bx+DSCmd-@],dx
	mov	ch,0		; Set I-O and DMA byte counts
	shl	cx,1
	mov	[bx+IOLen+1-@],cx
	mov	[bx+DMALn+1-@],cx
	mov	[VDSOf],ax	; Set 32-bit VDS offset
	mov	[bx+VDSOf+2-@],bx
	mov	[bx+VDSSg-@],es ; Set 16-bit VDS segment
	mov	bp,sp		; Set BP-reg. to current stack pointer
	test	al,3		; Is user's I-O buffer DWORD aligned?
	jnz	UseBuf		; No, use buffered I-O logic below
	mov	[bx+IOAdr-@],es ; Initialize 20-bit buffer segment
	mov	es,bx		; Point ES-reg. to low memory
	cli			; Avoid interrupts during VDS tests
	test	byte [es:VDSFLAG],20h ; Are "VDS services" active?
	jz	NoVDS		; No, set 20-bit DMA buffer address
	stc			; Do VDS "lock" of user I-O buffer
	mov	ax,8103h	; (Return carry if no VDS, we HOPE!)
	mov	dx,0Ch
	call	VDSLock
	sti			; Re-enable CPU interrupts
	jc	UseBuf		; VDS error - use buffered logic
	inc	byte [bx+VLF-@] ; Set VDS "lock" flag
Chk64K	mov	ax,[IOLen]	; Get low-order ending DMA address
	dec	ax		; (IOLen - 1 + IOAdr)
	add	ax,[bx+IOAdr-@] ; Will this I-O cross a 64K boundary?
	jc	NoLock		; Yes, use buffered I-O logic below
	call	DoIO		; Do direct DMA I-O with user's buffer
Done	xor	ah,ah		; Success!  Zero error code and carry
Error	mov	sp,bp		; Discard any leftover stack data
	mov	[bp+19],ah	; Post error code in exiting AH-reg
	rcr	byte [bp+24],1	; Post error flag in exiting carry
	rol	byte [bp+24],1
	call	VDSUnlk		; If needed, "unlock" user I-O buffer
	pop	ds		; Reload all CPU registers and exit
	pop	es
	popa
	iret
BufOut	call	XMSMove		; Buffered output - move data to XMS
	call	BufIO		; Output all data from XMS buffer
	jmp	Done		; Done - go post "success" and exit
NoVDS	sti			; No VDS - re-enable CPU interrupts
	mov	[bx+IOAdr+2-@],bx ; Get 20-bit buffer segment
	shl	dword [bx+IOAdr-@],4
	add	[bx+IOAdr-@],ax ; Set 20-bit DMA buffer address
	adc	[bx+IOAdr+2-@],bx
	jmp	Chk64K		; Go check if I-O will cross 64K
NoLock	call	VDSUnlk		; Buffered I-O - "unlock" user buffer
UseBuf	shl	dword [bx+VDSOf-@],16 ; Convert to XMS handle/offset
	test	byte [bx+IOCmd-@],12h ; Is this a write request?
	jnz	BufOut		; Yes, use output logic above
	call	BufIO		; Input all data to XMS buffer
	call	XMSMove		; Move XMS data to user's buffer
	jmp	Done		; Done - go post "success" and exit

; Subroutine to move data to and from the driver's XMS buffer
; NOTE:	 Before entering here, the main routine has converted
;   our user-buffer offset (VDSOf) to a "null" handle and hi-
;   order offset, and so the XMS source field ALREADY has the
;   user-buffer address needed by XMS moves, which simplifies
;   the following logic!

XMSMove push	ds		; Point ES-reg. to our data
	pop	es
	cld			; Ensure FORWARD "string" commands!
	mov	di,XMSDH	; Point to XMS destination field
	jnz	XMSOut		; If output, just set XMS destination!
	mov	si,XMSSH	; Point to user-buffer address
	movsw			; Move user-buffer address from
	movsw			;  XMS source to XMS destination
	movsw
	mov	di,XMSSH	; Point to XMS source field
XMSOut	mov	si,Handl	; Set XMS buffer handle/offset as the
	movsw			;  input source or output destination
	movsw
	movsw
	mov	ah,0Bh		; Move data to or from our XMS buffer
	mov	si,IOLen	; (The XMS driver will manage the A20
	call	0000:0000	;   line, as HIMEM.SYS and others do!)
@XEntry equ	$-4		; (XMS "entry" address, set by init)
	xor	bx,bx		; Zero BX-reg. for relative commands
	dec	ax		; Any errors during XMS move?
	jmp	ChkErr		; Go check result below, then exit

; Subroutine to do VDS "lock" and "unlock" functions

VDSUnlk shr	byte [bx+VLF-@],1 ; Was user buffer "locked" by VDS?
	jnc	VDSExit		; No, go exit
	mov	ax,8104h	; Do VDS "unlock" of user I-O buffer
	xor	dx,dx
VDSLock push	ds		; Point to VDS parameter block
	pop	es
	mov	di,IOLen
	int	4Bh		; Execute desired VDS function
VDSExit ret			; Exit

; Subroutine to execute an I-O request

BufIO	mov	dword [bx+IOAdr-@],0 ; Buffered I-O - use XMS buffer
@XBufAd equ	$-4		; (XMS buffer address, set by init)
DoIO	mov	dx,[bx+PCICm-@] ; Get DMA command-register address
	xor	dl,[bx+ChPCI-@]
	in	al,dx		; Ensure any previous DMA is stopped!
	and	al,0FEh		; (See comments below in driver-init)
	out	dx,al
	push	dx		; Save command-register address
	mov	al,[DSCmd]	; Select our desired disk
	and	al,0F0h
	mov	dx,CDSEL
	xor	dl,[bx+ChIDE-@]
	out	dx,al
	mov	di,dx		; Save IDE drive-select address
	mov	ah,HERROR	; Preset hard-error return code
	mov	ch,FLT		; Check only disk fault after ready
	call	SetTmr		; Go set timer and wait for ready
	cld			; Ensure FORWARD "string" commands!
	shr	byte [bp+19],1	; Get write request bit in carry
	cmc			; Invert carry so 1 = read, 0 = write
	rcl	al,4		; Set DMA "read/write" byte
ResetCR pop	dx		; Reload DMA command-register address
	out	dx,al		; Reset command register and set mode
	inc	ax		; Set Start/Stop in "read/write" byte
	push	ax		; Save DMA "read/write" byte
	push	dx		; Save DMA command-register address
	inc	dx		; Point to DMA status register
	inc	dx
	in	al,dx		; Reset DMA status register
	or	al,6		; (Done this way so we do NOT alter
	out	dx,al		;   the "DMA capable" status bits!)
	inc	dx		; Set PRD pointer to our DMA address
	inc	dx
	mov	si,PRDAd
	outsd
	mov	cx,1F7h		; Set IDE parameter-output flags
NxtPar	lea	dx,[di+CSECCT-CDSEL-1] ; Point to IDE sector count -1
IDEPar	inc	dx		; Output LBA-48 IDE parameter bytes
	outsb			; (If LBA-28, 1st 4 get overwritten!)
	shr	cx,1		; More parameters to go in this group?
	jc	IDEPar		; Yes, loop back and output next one
	jnz	NxtPar		; If first 4 output, go do last 6
SetTmr	mov	es,bx		; Point to BIOS timer in low-memory
	mov	si,BIOSTMR
	mov	cl,RDYTO	; Set I-O timeout limit in CL-reg
	add	cl,[es:si]
	or	ch,ch		; Is this an initial check for ready?
	jnz	ChkRdy		; Yes, go check only IDE controller
	mov	dh,3		; Get IDE alternate-status address
	dec	dx		; (Primary-status address | 300h - 1)
ChkDRQ	cmp	cl,[es:si]	; Too long without 1st data-request?
	je	Kaput		; Yes?	Exit and return IDE error!
	in	al,dx		; Read IDE alternate status
	test	al,DRQ		; Has 1st data-request arrived?
	jz	ChkDRQ		; No, wait
	pop	dx		; Reload DMA command-register address
	pop	ax		; Reload DMA "read/write" byte
	out	dx,al		; Set DMA Start/Stop (starts DMA)
	mov	ch,FLT+ERR	; Check fault and error after I-O end
ChkDMA	cmp	cl,[es:si]	; Has a DMA transfer timed out?
	je	HltDMA		; Yes?	Halt DMA & return DMA error!
	inc	dx		; Read DMA controller status
	inc	dx
	in	al,dx
	dec	dx
	dec	dx
	shl	al,6		; Any DMA errors during this I-O?
	js	HltDMA		; Yes?	Halt DMA & return DMA error!
	jnc	ChkDMA		; If no DMA interrupt, go check again
	in	al,dx		; Reset DMA Start/Stop bit
	and	al,0FEh
	out	dx,al
	inc	dx		; Reread DMA controller status
	inc	dx
	in	al,dx
	test	al,DME		; Any "late" DMA error after DMA end?
	jnz	BadDMA		; Yes?	Exit and return DMA error!
ChkRdy	cmp	cl,[es:si]	; Too long without becoming ready?
	je	Kaput		; Yes?	Exit and return IDE error!
	lea	dx,[di+CSTAT-CDSEL] ; Read IDE primary status
	in	al,dx
	test	al,BSY+RDY	; Controller or disk still busy?
	jle	ChkRdy		; Yes, loop back and check again
	and	al,ch		; Disk-fault or IDE error at I-O end?
ChkErr	jnz	Kaput		; Yes?	Exit and return IDE error!
	ret			; All is well - exit
HltDMA	in	al,dx		; Ensure DMA is stopped when we exit!
	and	al,0FEh
	out	dx,al
BadDMA	mov	ah,DMAERR	; Set DMA error/timeout return code
Kaput	stc			; Set carry flag to denote "error"
	jmp	Error		; Go set stack return codes and exit
	align	16
ResEnd	equ	$		; End of resident driver

; Initialization Variables

IVDSLen dd	ResEnd		; Initialization VDS parameters
IVDSOfs dd	0
IVDSSeg dd	0
IVDSAdr dd	0
Packet	dd	0		; "Init" request packet address
BufAddr dd	0		; XMS buffer-address "bucket"
HDNames dw	PMMsg		; Table of hard-disk "name" pointers
	dw	PSMsg
	dw	SMMsg
	dw	SSMsg
Modes	db	'16. '		; Mode 0 = ATA-16  UltraDMA mode table
	db	'25. '		; Mode 1 = ATA-25
	db	'33. '		; Mode 2 = ATA-33
	db	'44. '		; Mode 3 = ATA-44  (Rare but possible)
	db	'66. '		; Mode 4 = ATA-66
	db	'100.'		; Mode 5 = ATA-100
	db	'133.'		; Mode 6 = ATA-133
	db	'166.'		; Mode 7 = ATA-166
ITbl	db	80h,8Ah,0FAh	; Interface byte table
ITEnd	equ	$
HDCount db	0		; BIOS hard-disk count
HDUnit	db	80h		; Current BIOS unit number
HDNibbl db	LBABITS		; IDE drive-select "nibble"
HDIndex db	0		; IDE "index" number
HDOffs	db	0		; IDE channel "offset"

; "Strategy" routine - At entry, ES:BX points to the DOS initialization
;   request packet, which is saved for processing below

Strat	mov	[cs:Packet],bx	; Save DOS request-packet address
	mov	[cs:Packet+2],es
	retf			; Exit - await DOS "Device Interrupt"

; "Device-Interrupt" routine - This routine initializes the driver

DevInt	pushf			; Entry - save all registers
	push	ds
	push	es
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	cs		; Set our DS-reg
	pop	ds
	les	bx,[Packet]	; Point to DOS request packet
	xor	ax,ax		; Get a zero for below
	cmp	[es:bx+RPOp],al ; Is this an "Init" packet?
	je	I_Title		; Yes, display our title message
	jmp	I_BadP		; Go post errors and exit quick!
I_Title mov	ah,9		; Display driver "title" message
	mov	dx,TTLMsg
	int	21h
	pushf			; 80386 test - save CPU flags
	push	sp		; See if CPU is an 80286 or newer
	pop	ax
	cmp	ax,sp		; 80286+ push SP, then decrement it
	jne	I_Junk		; CPU is below 80286 - cannot use it!
	push	7000h		; 80286 or newer - try to set NT|IOPL
	popf
	pushf
	pop	ax
	test	ah,70h		; Did any NT|IOPL bits get set?
	jnz	I_80386		; Yes, CPU is at least an 80386
I_Junk	popf			; Reload starting CPU flags
	mov	ah,9		; Display "Not an 80386" message
	mov	dx,PRMsg
	int	21h
	jmp	I_Quit		; Go display suffix and exit quick!
I_80386 popf			; Reload starting CPU flags
	pop	di		; Reload all 16-bit registers
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pushad			; Save all 32-bit registers
	call	SetPCI		; Find and set our UDMA PCI address
	jc	near I_ErOut	; Error - display message and exit!
	mov	word [XMSSH],XEMsg ; Set XMS error-message pointer
	mov	ah,9		; Request 128K of XMS memory
	mov	dx,128
	call	far [@XEntry]
	cmp	ax,1		; Did we get our buffer memory?
	jne	near I_XMErr	; No?  Display error message & exit!
	mov	[Handl],dx	; Save our buffer handle
	mov	ah,0Ch		; Lock our XMS memory buffer
	call	far [@XEntry]
	cmp	ax,1		; Did buffer get locked?
	jne	near I_XLErr	; No?  Display error message & exit!
	mov	[BufAddr],bx	; Save 32-bit XMS buffer address
	mov	[BufAddr+2],dx
	mov	eax,[BufAddr]	; Get XMS buffer address
	add	eax,65536	; Find 1st 64K boundary after start
	xor	ax,ax
	mov	[@XBufAd],eax	; Set XMS buffer address and offset
	sub	eax,[BufAddr]
	mov	[XMSOf],eax
	xor	eax,eax		; Point ES-reg. to low memory
	mov	es,ax
	mov	ax,cs		; Set our code segment in VDS block
	mov	[IVDSSeg],ax
	shl	eax,4		; Get 20-bit driver virtual address
	cli			; Avoid interrupts during VDS tests
	test	byte [es:VDSFLAG],20h ; Are "VDS services" active?
	jz	I_SetA		; No, set 20-bit virtual addresses
	push	cs		; Point to VDS parameter block
	pop	es
	mov	di,IVDSLen
	stc			; "Lock" this driver into memory
	mov	ax,8103h	; (Return carry if no VDS, we HOPE!)
	mov	dx,0Ch
	int	4Bh
	jc	near I_VErr	; Error?  Display error msg. & exit!
	mov	eax,[IVDSAdr]	; Get 32-bit starting driver address
I_SetA	sti			; Re-enable CPU interrupts
	add	[PRDAd],eax	; Set relocated 32-bit PRD address
	mov	dx,[PCICm]	; Ensure any previous DMA is stopped
	in	al,dx		; (On some older chipsets, if DMA is
	and	al,0FEh		;   running, reading an IDE register
	out	dx,al		;   causes the controller to HANG!!)
	add	dx,8		; Stop secondary-channel DMA, also!
	in	al,dx
	and	al,0FEh
	out	dx,al
	xor	di,di		; Set our BIOS hard-disk count
	mov	es,di
	mov	al,[es:HDISKS]
	mov	[HDCount],al
	cmp	al,0		; Did BIOS find any hard-disks?
	je	near I_None	; No?  Display message and exit!
	mov	ah,41h		; See if this system has an EDD BIOS
	mov	bx,55AAh
	mov	dl,80h
	int	13h
	jc	near I_1Disk	; Error - check for only 1 hard-disk
	cmp	bx,0AA55h	; Did BIOS "reverse" our entry code?
	jne	near I_1Disk	; No?  Check for only 1 hard-disk
	test	cx,4		; Does BIOS support the EDD subset?
	jnz	I_Next		; Yes, search for hard-disks to use
I_1Disk cmp	byte [HDCount],1; Do we have exactly ONE hard-disk?
	je	I_ChNam		; Yes, test it as the primary-master
I_NoEDD mov	dx,EBMsg	; Point to "No EDD BIOS" message
	jmp	I_Err		; Go display error message and exit!
I_Next	mov	ah,48h		; Get next BIOS disk's EDD parameters
	mov	dl,[HDUnit]
	mov	si,EDDBuff
	int	13h
	jc	I_NoEDD		; Error - display message and exit!
	cmp	dword [si+26],byte -1 ; Valid DPTE pointer?
	je	I_More		; No, check for more BIOS disks
	les	si,[si+26]	; Get this disk's DPTE pointer
	mov	bx,15		; Calculate DPTE checksum
	mov	al,0
I_CkSum add	al,[es:bx+si]
	dec	bx
	jns	I_CkSum
	cmp	al,0		; Is DPTE valid (checksum = 0)?
	jne	I_NoEDD		; No?  Display message and exit!
	mov	al,[es:si+4]	; Save disk's device-select "nibble"
	mov	[HDNibbl],al
	shr	al,4		; Initialize IDE index and offset
	and	ax,1
	mov	[HDIndex],ax
	mov	ax,[es:si]	; Get disk's IDE base address
	cmp	ax,CDATA	; Is this a primary-channel disk?
	je	I_ChNam		; Yes, display its "channel" name
	cmp	ax,(CDATA-80h)	; Is this a secondary-channel disk?
	jne	I_More		; No?  Ignore it - WEIRD situation!
	add	word [HDIndex],8002h ; Adjust for secondary channel
I_ChNam movzx	bx,[HDIndex]	; Get "channel name" message index
	shl	bx,1
	mov	ah,9		; Display disk's IDE "channel name"
	mov	dx,[bx+HDNames] ; ("Primary master", etc.)
	int	21h
	mov	ah,8		; Get BIOS parameters for this disk
	mov	dl,[HDUnit]
	int	13h
	xchg	ax,dx		; Set AX-reg. with head-number value
	mov	dx,LEMsg	; Point to "not in LBA mode" message
	jc	I_NotU		; Error - display msg. & ignore disk
	and	cl,SECSHD	; Clear cylinder bits
	cmp	cl,SECSHD	; Sectors per cylinder = 63?
	jne	I_NotU		; No, display message & ignore disk
	cmp	ah,HEADS-1	; Heads = 255 (max. head = 254)?
	jne	I_NotU		; No, display message & ignore disk
	call	I_ChkD		; See if this is an UltraDMA disk
	jc	I_NotU		; Any errors during UltraDMA tests?
	movzx	bx,[HDIndex]	; No, activate disk in unit table
	mov	al,[HDUnit]
	mov	[bx+Units],al
	jmp	short I_More	; Go check for more BIOS disks
I_NotU	mov	ah,9		; Display error for this disk
	int	21h
	mov	ah,9		; Display error-message suffix
	mov	dx,CRMsg
	int	21h
I_More	inc	byte [HDUnit]	; Increment BIOS unit number
	dec	byte [HDCount]	; More BIOS disks to check?
	jnz	I_Next		; Yes, loop back and do next one
I_None	mov	dx,NDMsg	; Point to "No disks" message
	cmp	dword [Units],byte -1 ; Any active UltraDMA disks?
	je	I_Err		; No?  "Say Goodnight, Gracie!"
	mov	ax,3513h	; Get current INT13 vector
	int	21h
	mov	[@PrvI13],bx	; Save vector for "passed" requests
	mov	[@PrvI13+2],es
	mov	ax,2513h	; "Hook" this driver into INT13
	mov	dx,Entry
	int	21h
	les	bx,[Packet]	; Post driver size & success code
	mov	word [es:bx+RPSize],ResEnd
	mov	[es:bx+RPSize+2],cs
	mov	word [es:bx+RPStat],RPDON
	popad			; Reload all CPU registers and exit
	pop	es
	pop	ds
	popf
	retf
I_VErr	sti			; VDS "lock" error!  Enable interrupts
	mov	word [XMSSH],VEMsg ; Point to VDS "lock" error message
	jmp	I_XUnlk		; Go get rid of our XMS memory
I_Err	mov	[XMSSH],dx	; Save error message pointer
	push	cs		; Point to VDS parameter block
	pop	es
	mov	di,IVDSLen
	mov	ax,8104h	; Do VDS "unlock" of this driver
	xor	dx,dx
	int	4Bh
I_XUnlk mov	ah,0Dh		; Unlock our XMS memory buffer
	mov	dx,[Handl]
	call	far [@XEntry]
I_XLErr mov	ah,0Ah		; Free our XMS memory buffer
	mov	dx,[Handl]
	call	far [@XEntry]
I_XMErr mov	dx,[XMSSH]	; Reload error message pointer
I_ErOut mov	ah,9		; Display error message
	int	21h
	popad			; Reload all 32-bit registers
	push	ax		; Save all 16-bit registers
	push	bx
	push	cx
	push	dx
	push	si
	push	di
I_Quit	mov	ah,9		; Display message suffix
	mov	dx,Suffix
	int	21h
	les	bx,[Packet]	; Post "null" driver size
	xor	ax,ax
	mov	[es:bx+RPSize],ax
	mov	[es:bx+RPSize+2],cs
I_BadP	mov	ax,RPDON+RPERR	; Post "error" in init packet
	mov	[es:bx+RPStat],ax
	pop	di		; Reload all CPU registers and exit
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	es
	pop	ds
	popf
	retf

; Subroutine to test for UDMA hard-disks and set their UDMA mode

I_ChkD	mov	al,[HDNibbl]	; Select master or slave disk
	mov	dx,CDSEL
	xor	dl,[HDOffs]
	out	dx,al
	mov	al,0ECh		; Issue "Identify Device" command
	call	I_Cmd
	jnc	I_PIO		; If no error, get "identify" data
I_AErr	mov	dx,AEMsg	; Absent or non-ATA!   Point to msg
	stc			; Set carry flag (error!) and exit
	ret
I_PIO	mov	dx,CDATA	; Point to controller PIO data reg
	xor	dl,[HDOffs]
	in	ax,dx		; Read I.D. bytes 0 and 1
	xchg	ax,si		; Save "ATA/ATAPI" flag word
	mov	cx,26		; Skip I.D. bytes 2-53
I_Skip0 in	ax,dx
	loop	I_Skip0
	cld			; Ensure FORWARD "string" commands!
	push	ds		; Point to disk-name message
	pop	es
	mov	di,DiskNam
	mov	cl,20		; Read & swap disk name into message
I_RdNam in	ax,dx		; (I.D. bytes 54-93)
	xchg	ah,al		; Swap its bytes
	stosw
	loop	I_RdNam
	mov	cl,6		; Skip I.D. bytes 94-105
I_Skip1 in	ax,dx
	loop	I_Skip1
	in	ax,dx		; Read I.D. bytes 106 and 107
	mov	bh,al		; Save "DMA valid" flag byte
	mov	cl,34		; Skip I.D. bytes 108-175
I_Skip2 in	ax,dx
	loop	I_Skip2
	in	ax,dx		; Read I.D. bytes 176 and 177
	mov	bl,ah		; Save "UDMA selected" flag byte
	mov	cl,167		; Skip remaining I.D. data
I_Skip3 in	ax,dx
	loop	I_Skip3
	shl	si,1		; Is this an "ATA" hard-disk?
	jc	I_AErr		; No?  Display message & exit!
	test	bh,4		; Are UltraDMA flag bits valid?
	jz	I_DErr		; No?  Display message & exit!
	mov	di,Modes	; Point to UDMA mode table
	mov	al,'0'		; Initialize "current mode" value
	mov	cl,2		; Set rotating mode-check bit
	cmp	bl,1		; Will disk do UDMA mode 0?
	jae	I_NxtM		; Yes, find its best UDMA mode
I_DErr	mov	dx,DEMsg	; Not a UDMA disk!   Point to message
	stc			; Set carry flag (error!) and exit
	ret
I_NxtM	cmp	bl,cl		; Will disk do next UDMA mode?
	jb	I_GotM		; No, use previous mode
	inc	ax		; Set up for next UDMA mode
	add	di,4
	shl	cl,1		; More UDMA modes to check?
	jnz	I_NxtM		; Yes, loop back
I_GotM	mov	[CurMode],al	; Update "current mode" value
	mov	eax,[di]	; Post UDMA mode in set-mode message
	mov	[DspMode],eax
	mov	dx,CSUBCM	; Set mode-select subcode
	xor	dl,[HDOffs]
	mov	al,SETM
	out	dx,al
	inc	dx
	mov	al,[CurMode]	; Set desired UltraDMA mode value
	add	al,10h
	out	dx,al
	mov	al,SETF		; Issue set-features command to disk
	call	I_Cmd
	jnc	I_ScanN		; If no errors, skip disk name spaces
	mov	dx,SEMsg	; Set-mode error!  Point to message
	stc			; Set carry flag (error!) and exit
	ret
I_ScanN mov	di,DNamEnd	; Point to end of disk name
I_NextN cmp	di,DiskNam	; Are we at the disk-name start?
	je	I_NullN		; Yes, disk has no name!
	cmp	byte [di-1],' ' ; Is preceding byte a space?
	jne	I_TermN		; No, terminate disk name message
	dec	di		; Decrement disk name pointer
	jmp	I_NextN		; Go see if next byte is a space
I_NullN mov	dword [di],"unna" ; Set "unnamed" as disk name
	mov	dword [di+4],"med "
	add	di,7
I_TermN mov	word [di],".$"	; Set message terminators after name
	call	I_DvrRd		; Read disk block 1 using BIOS
	jc	I_RFail		; Error - display message and exit!
	push	ax		; Save checksum from BIOS read
	mov	ax,3513h	; Get current Int 13h vector
	int	21h
	mov	[@PrvI13],bx	; Save vector for passed requests
	mov	[@PrvI13+2],es
	mov	ax,2513h	; Hook our driver into Int 13h
	mov	dx,Entry
	int	21h
	call	I_DvrRd		; Read disk block 1 with our driver
	push	ax		; Save checksum from driver read
	pushf			; Save CPU flags
	mov	ax,2513h	; Set back the old Int 13h vector
	lds	dx,[@PrvI13]
	int	21h
	push	cs		; Reload our DS-register
	pop	ds
	popf			; Reload CPU flags
	pop	cx		; Reload checksum from driver read
	pop	ax		; Reload checksum from BIOS read
	jc	I_RFail		; Error - display message and exit!
	cmp	ax,cx		; Same checksum from both reads?
	jne	I_RFail		; No?  Display message and exit!
	mov	eax,[@XBufAd]	; Set DMA buffer address
	mov	[IOAdr],eax
	mov	ax,(RBYTES/512) ; Set input sector count
	mov	[SecCt],al
	shl	ax,1		; Set DMA byte count
	mov	[DMALn+1],ax
	xor	ax,ax		; Set "null" logical block address
	mov	[LBALo],ax
	mov	[LBALo+2],al
	mov	al,[HDNibbl]	; Set LBA and IDE command bytes
	mov	ah,DRCMD
	mov	[DSCmd],ax
	call	I_Read		; Do initial read for synchronization
	jnc	I_RSetC		; If O.K., set up 4-pass read test
I_RFail mov	dx,TEMsg	; Point to "FAILED read test" message
	stc			; Set carry flag (error!) and exit
	ret
I_RSetC xor	ax,ax		; Clear read counter
	mov	cl,[es:si]	; Load current timer tick count LSB
I_RWait cmp	cl,[es:si]	; Next tick arrived?
	je	I_RWait		; No, keep waiting
	add	cl,1+4		; Yes, update and prepare for 4 passes
I_RNext inc	ax		; Yes, count reads up
	push	ax		; Save counters
	push	cx
	call	I_Read		; Read RBYTES bytes
	pop	cx		; Recover counters
	pop	ax
	jc	I_RFail		; Read error? Exit
	cmp	cl,[es:si]	; Next timer interrupt arrived?
	jne	I_RNext		; No, read once more
	shr	ax,2		; Save average rate for 4 passes
	push	ax
	mov	dx,DNamMsg	; Display disk "name" message
	mov	ah,9
	int	21h
	mov	dx,MSMsg	; Display "Set to mode" message
	mov	ah,9
	int	21h
	pop	ax		; Reload average read rate
	mov	di,DspRate+4	; Point to read-rate digits message
	mov	byte [di],'0'	; Initialize read rate digits to 0
	or	ax,ax		; Did the disk read NOTHING?
	jz	I_DspRR		; Yes, display read rate = 0
	mov	cx,10		; CX = divisor
I_ItoA	xor	dx,dx		; DX:AX = dividend
	div	cx
	xchg	dx,ax		; DX = quotient, AX = remainder
	add	al,'0'		; convert to ASCII
	mov	[di],al
	dec	di
	xchg	dx,ax		; AX = quotient
	or	ax,ax		; zero?
	jnz	I_ItoA		; no, continue
	lea	dx,[di+1]	; Display read-rate message
I_DspRR mov	ah,9
	int	21h
	clc			; Clear carry (no error) and exit
	ret			; Exit

; Subroutine to do initialization buffered-input commands

I_Read	mov	dx,[PCICm]	; Get DMA command-register address
	mov	al,[HDOffs]
	shr	al,4
	xor	dl,al
	in	al,dx		; Ensure any previous DMA is stopped!
	and	al,0FEh		; (See comments below in driver-init)
	out	dx,al
	push	dx		; Save command-register address
	mov	al,[HDNibbl]	; Select our desired disk
	mov	dx,CDSEL
	xor	dl,[HDOffs]
	out	dx,al
	mov	di,dx		; Save IDE drive-select address
	xor	si,si		; Point to BIOS timer in low-memory
	mov	es,si
	mov	si,BIOSTMR
	mov	ah,RDYTO	; Set I-O timeout limit in CL-reg
	add	ah,[es:si]
	mov	cl,FLT		; Check only disk fault after ready
	call	I_CkRdy		; Await controller- and disk-ready
	pop	dx		; Reload DMA command-register address
	jc	I_Kaput		; If ready error, set carry and exit!
	mov	al,8		; Reset command register and set mode
	out	dx,al
	push	dx		; Save DMA command-register address
	inc	dx		; Point to DMA status register
	inc	dx
	in	al,dx		; Reset DMA status register
	or	al,6		; (Done this way so we do NOT alter
	out	dx,al		;   the "DMA capable" status bits!)
	inc	dx		; Set PRD pointer to our DMA address
	inc	dx
	cld			; Ensure FORWARD "string" commands!
	mov	si,PRDAd
	outsd
	add	si,4		; Skip high-order IDE parameters
	lea	dx,[di+CSECCT-CDSEL] ; Point to IDE sector count
	mov	cx,6		; Output our 6 IDE parameter bytes
I_Parms outsb
	inc	dx
	loop	I_Parms
	pop	cx		; Reload DMA command-register address
	mov	si,BIOSTMR	; Point to BIOS timer in low-memory
I_CkDRQ cmp	ah,[es:si]	; Too long without 1st data-request?
	je	I_Kaput		; Yes?	Set carry flag and exit!
	lea	dx,[di+CSTAT2-CDSEL] ; Read IDE alternate status
	in	al,dx		; Read IDE alternate status
	test	al,DRQ		; Has 1st data-request arrived?
	jz	I_CkDRQ		; No, wait
	mov	dx,cx		; Get DMA command-register address
	mov	al,9		; Set DMA Start/Stop (starts DMA)
	out	dx,al
I_CkDMA cmp	ah,[es:si]	; Has a DMA transfer timed out?
	je	I_NoDMA		; Yes?	Halt DMA & return DMA error!
	inc	dx		; Read DMA controller status
	inc	dx
	in	al,dx
	dec	dx
	dec	dx
	shl	al,6		; Any DMA errors during this I-O?
	js	I_NoDMA		; Yes?	Halt DMA & return DMA error!
	jnc	I_CkDMA		; If no DMA interrupt, go check again
	in	al,dx		; Reset DMA Start/Stop bit
	and	al,0FEh
	out	dx,al
	inc	dx		; Reread DMA controller status
	inc	dx
	in	al,dx
	test	al,DME		; Any "late" DMA error after DMA end?
	jnz	I_Kaput		; Yes?	Set carry flag and exit!
	mov	cl,FLT+ERR	; Check fault and error after I-O end

; Subroutine to do initialization controller- and disk-ready checks

I_CkRdy cmp	ah,[es:si]	; Too long without becoming ready?
	je	I_Kaput		; Yes?	Set carry flag and exit!
	lea	dx,[di+CSTAT-CDSEL] ; Read IDE primary status
	in	al,dx
	test	al,BSY+RDY	; Controller or disk still busy?
	jle	I_CkRdy		; Yes, loop back and check again
	and	al,cl		; Disk-fault or IDE error at I-O end?
	jnz	I_Kaput		; Yes?	Set carry flag and exit!
	ret			; All is well - exit
I_NoDMA in	al,dx		; Ensure DMA is stopped when we exit!
	and	al,0FEh
	out	dx,al
I_Kaput stc			; Set carry flag to denote "error"
	ret			; Exit

; Subroutine to read block 1 of the current disk and "checksum" it

I_DvrRd push	ds		; Point to our read buffer
	pop	es
	mov	bx,RBuffer
	mov	ax,201h		; Read block 1 of the current disk
	mov	cx,1
	xor	dx,dx
	mov	dl,[HDUnit]
	int	13h
	mov	cx,256		; Set 256-word checksum count
	mov	si,RBuffer	; Point to our input buffer
	xor	ax,ax		; Reset "checksum" bucket
I_Accum add	ax,[si]		; Add next input word to checksum
	adc	ax,byte 0	; Add back "carryouts", too
	inc	si		; Increment to next input word
	inc	si
	loop	I_Accum		; If more words to add, loop back
	ret			; Exit - AX-reg. has our "checksum"

; Subroutine to issue initialization commands to our disks

I_Cmd	mov	dx,CCMD		; Issue desired init command
	xor	dl,[HDOffs]
	out	dx,al
	xor	si,si		; Point to BIOS timer in low-memory
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,RDYTO	; Set I-O timeout limit in CL-reg
	add	cl,[es:si]
I_CmdW	cmp	cl,[es:si]	; Has our command timed out?
	je	I_CmdE		; Yes, set CPU carry flag & exit
	mov	dx,CSTAT	; Get IDE controller status
	xor	dl,[HDOffs]
	in	al,dx
	test	al,BSY+RDY	; Controller or disk still busy?
	jle	I_CmdW		; Yes, loop back and check again
	test	al,ERR		; Did command cause any errors?
	jz	I_CmdX		; No, leave carry flag off & exit
I_CmdE	stc			; Error - set CPU carry flag on
I_CmdX	ret			; Exit

; Start of EDD and Read-Test Input Buffers, put here so they "overlay"
;   code and messages which are not needed, when the buffers are used!

	align	4
EDDBuff dd	30
RBuffer equ	EDDBuff+3	; ("Odd" address forces use of XMS)

; Subroutine to find and set the PCI address of our UltraDMA controller
;   All "PCI logic" in this routine is courtesy of Luchezar I. Georgiev

SetPCI	xor	edi,edi		; Get PCI BIOS "I.D." code
	mov	ax,0B101h
	int	1Ah
	cmp	edx,'PCI '	; Is PCI BIOS V2.0C or later?
	je	BiosOK		; Yes, continue search
	mov	dx,PEMsg	; Baaad news!  Point to error message
	stc			; Set carry flag (error!) and exit
	ret
BiosOK	mov	si,ITbl		; Point to interface byte table
	cld			; Ensure we do "forward" string commands!
GetDev	mov	ecx,10100h	; Look for class 1 (storage) subclass 1 (IDE)
	lodsb			; Get next interface byte
	mov	cl,al
	push	si		; Find PCI class code
	mov	ax,0B103h	; (Returns bus/device/function in BX-reg.)
	xor	si,si
	int	1Ah
	pop	si
	jnc	GotDev		; Found our boy!  Go process it
	cmp	si,ITEnd	; More interface bytes to try?
	jb	GetDev		; Yes, try next one
	mov	dx,NEMsg	; Baaad news!  Point to error message
	stc			; Set carry flag (error!) and exit
	ret
GotDev	push	bx		; Save bus/device/function
	mov	ax,0B108h	; Get low-order command byte
	mov	di,4
	int	1Ah
	pop	bx		; Reload bus/device/function
	and	cl,5		; Mask Bus-Master and I-O Space bits
	cmp	cl,5		; Are these bits what we found?
	je	BaseAd		; Yes, get our PCI base address
	mov	dx,MEMsg	; Baaad news!  Point to error message
	stc			; Set carry flag (error!) and exit
	ret
BaseAd	push	bx		; Get PCI base address (register 4)
	mov	ax,0B109h
	mov	di,32
	int	1Ah
	pop	bx
	xchg	ax,cx		; Post run-time PCI UDMA address
	and	al,0FCh
	mov	[PCICm],ax
	mov	cx,4		; Set hex address in display message
	mov	si,DspAddr
	call	HexConv
	push	bx		; Get Vendor and Device I.D.
	mov	di,0
	mov	ax,0B10Ah
	int	1Ah
	pop	bx
	xchg	eax,ecx		; Save Vendor and Device I.D.
	push	eax
	mov	cx,4		; Set vendor I.D. in display message
	mov	si,DspVID
	call	HexConv
	pop	eax		; Reload Vendor and Device I.D.
	shr	eax,16		; Set Device I.D. in display message
	mov	cl,4
	mov	si,DspDID
	call	HexConv
	mov	ah,bh		; Set PCI bus number in message
	mov	cl,2
	mov	si,DspBus
	call	HexConv
	mov	ah,bl		; Set PCI device number in message
	shr	ah,3
	mov	cl,2
	mov	si,DspDev
	call	HexConv
	and	bl,7		; Set PCI function number in message
	or	bl,30h
	mov	[DspFnc],bl
	mov	ax,4300h	; Inquire about an XMS driver
	int	2Fh
	cmp	al,80h		; Is an XMS driver installed?
	je	I_SvXMS		; Yes, save its entry address
	mov	dx,NXMsg	; Baaad news!  Point to error message
	stc			; Set carry flag (error!) and exit
	ret
I_SvXMS mov	ax,4310h	; Save XMS driver "entry" address
	int	2Fh
	mov	[@XEntry],bx
	mov	[@XEntry+2],es
	mov	ah,9		; Display "UDMA controller" message
	mov	dx,PCMsg
	int	21h
	clc			; Clear carry (no error) and exit
	ret

; Subroutine to convert a number from hex to ASCII for messages
;   At entry, the message address is in the SI-reg.   A 2-digit
;   value is in the AH-reg. and the CX-reg. is equal to 2, or a
;   4-digit value is in the AX-reg. and the CX-reg. is set to 4

HexConv rol	ax,4		; Rotate next hex digit to low-order
	push	ax		; Save hex address
	and	al,0Fh		; Mask off next hex digit
	cmp	al,9		; Is digit 0-9?
	jbe	HexCnv1		; Yes, convert to ASCII
	add	al,7		; Add A-F offset
HexCnv1 add	al,30h		; Convert digit to ASCII
	mov	[si],al		; Set next ASCII hex digit in message
	inc	si		; Bump message address
	pop	ax		; Reload hex address
	loop	HexConv		; If more digits to convert, loop back
	ret			; Exit

; Initialization Messages

TTLMsg	db	CR,LF,'Universal UltraDMA Disk Driver '
	db	VER,CR,LF,'$'
PRMsg	db	'No 80386 CPU$'
MEMsg	db	'Bus-Master ERROR$'
NEMsg	db	'No PCI UltraDMA controller$'
PEMsg	db	'PCI BIOS below V2.0C$'
NXMsg	db	'No XMS driver$'
XEMsg	db	'XMS memory error$'
VEMsg	db	'VDS lock error$'
PCMsg	db	'UltraDMA controller found at PCI address '
DspAddr db	'0000h.',CR,LF,'    Bus '
DspBus	db	'00h, device '
DspDev	db	'00h, function '
DspFnc	db	'0, vendor ID '
DspVID	db	'0000h, device ID '
DspDID	db	'0000h'
CRMsg	db	'.',CR,LF,'$'
EBMsg	db	'No EDD BIOS data$'
NDMsg	db	'No UltraDMA hard disks$'
PMMsg	db	'Primary-master disk $'
PSMsg	db	'Primary-slave disk $'
SMMsg	db	'Secondary-master disk $'
SSMsg	db	'Secondary-slave disk $'
DNamMsg db	'is '
DiskNam db	'                                        '
DNamEnd db	'.$'
MSMsg	db	CR,LF,'    Set to UltraDMA mode '
CurMode db	'0, ATA-'
DspMode db	'16.    Read test = $'
DspRate db	'    0 MB/sec.',CR,LF,'$'
AEMsg	db	'absent or non-ATA$'
DEMsg	db	'is not UltraDMA$'
LEMsg	db	'is not in LBA mode$'
SEMsg	db	'set-mode error$'
TEMsg	db	'FAILED read test$'
Suffix	db	'; driver NOT loaded!',CR,LF,'$'
