;			       UDMA Driver for DOS

;Version  Date	Description
; v2.0	 1/9/3: Base buffered driver (MASM 5 version) written by Jack R. Ellis.
; v2.1	 4/9/3: NASM port, PCI port detection, LBA, options (Luchezar Georgiev)
; v2.2	 6/9/3: Bug fix, code cleanup and optimisation (Jack Ellis &L.Georgiev)
; v2.3	 7/9/3: Load descriptor table pointer reg on each transfer, bugfix (JE)
; v2.4	 9/9/3: Send DMAAddr/Len after command/control regs reset, cleanup (JE)
; v2.5	12/9/3: Reduced Ready subroutine, Go bit now reset on next request (JE)
; v2.6	14/9/3: Check DRQ before starting DMA, detect VDS via call 8102 (JE&LG)
; v2.7	15/9/3: Save initial VDS lock state, save EAX & do CLI on VDS call (JE)
; v2.8	16/9/3: Ready now waits not only Active low but Interrupt high too (LG)
; v2.9	17/9/3: Code cleanup and optimisation, the 'W' test option removed (JE)

%define Version 'v2.9'

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;	This program is distributed in the hope that it will be useful,
;	but WITHOUT ANY WARRANTY; without even the implied warranty of
;	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;	GNU General Public License for more details.
; You should have received a copy of the GNU General Public License along
; with this program; if not, write to the Free Software Foundation, Inc.,
; 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  (www.gnu.org)

; General Program Equations:

BUFSIZ	equ	4096		; Driver I/O buffer size
SECSBUF equ	BUFSIZ/512	; Sectors per I/O buffer
SECSCYL equ	255*63		; LBA sectors per cylinder
SECSHD	equ	63		; LBA sectors per head
RDYTO	equ	7		; 384 msec ready timeout
HERROR	equ	0FF00h		; Hard-error return code
TIMOUT	equ	0BB00h		; Timed-out  return code
BIOSSEG equ	40h		; BIOS segment value
BIOSTMR equ	6Ch		; BIOS "tick" timer offset
BIOSVDS equ	7Bh		; BIOS VDS flag-byte offset
CR	equ	0Dh		; ASCII carriage-return
LF	equ	0Ah		; ASCII line-feed
%define CRLF	CR,LF

; IDE Controller Register Definitions:

CDATA	equ	1F0h		; Data port
CSUBCM	equ	CDATA+1		; Subcommand register
CSECCT	equ	CDATA+2		; I/O sector count
CCMD	equ	CDATA+7		; Command register
CSTAT	equ	CDATA+7		; Status register
CSTAT2	equ	CDATA+206h	; Alternate status register

; IDE Controller Status bits:

BUSY	equ	80h		; Controller is busy
DSKRDY	equ	40h		; Disk is ready
FAULT	equ	20h		; Disk has a fault
ONCYL	equ	10h		; Device seek complete
DRQ	equ	8		; Data request by disk
ERROR	equ	1		; General error flag

; IDE Command Definitions:

SETM	equ	3		; Set Mode subcommand
SETF	equ	0EFh		; Set Features command
DRCMD	equ	0C8h		; DMA read command (write is 0CAh)

; UDMA Controller status and miscellaneous bits:

INTRPT	equ	4		; Interrupt has occured
ACTIVE	equ	1		; Bus master IDE active
LBABITS equ	0E0h		; Fixed MSB LBA bits: LBA = 1, DEV = 0

; Word and Doubleword Layouts:

struc dbt
	.lb	resb	1
	.hb	resb	1
endstruc
struc dwd
	.lw	resw	1
	.hw	resw	1
endstruc

; Device Address (LBA) Packet Layout:

struc	DAP
	.DASize resb	1	; Packet size in bytes (16 for IA-32)
	.DARes1 resb	1	; Reserved, must be 0
	.DANum	resb	1	; Number of blocks to transfer
	.DARes2 resb	1	; Reserved, must be 0
	.DABuf	resd	1	; Address of transfer buffer
	.DABeg	resd	2	; Starting logical block address
endstruc

; DOS "Request Packet" Layout:

struc	RP
	.RPlenU resb	2	; (Unused by us)
	.RPOp	resb	1	; Opcode
	.RPStat resw	1	; Status word
	.RPResN resb	9	; (Unused by us)
	.RPSize resd	1	; Resident driver size
	.RPCmdl resd	1	; Pointer to command line arguments
endstruc

RPERR	equ	8003h		; "Strategy" packet error flags
RPDON	equ	100h		; "Strategy" packet done flag

; DOS Driver Device Header:

	dd	-1		; Link to next device-header block
	dw	0C000h		; Driver "device attributes"
	dw	Strat		; "Strategy" routine offset
	dw	DevInt		; "Device-Interrupt" routine offset
	db	'UDMA$',0	; Driver name

; Resident Driver Variables:

PCIAddr dw	0		; PCI Base UDMA Address (set by init)
Flags	db	0		; Driver flags:
				;   Bit 7 = VDS "locked" at init
Params	equ	$		; I/O Parameters:
SectCt	db	0		;   Sector count
LBA	dd	0		;   Logical Block Address and commands
IOCmd	db	0		;   Command byte
Sectors db	0		; Remaining I/O sector count
VDSLock db	0		; VDS "lock" flag
ChkErrs db	FAULT		; "Check I/O errors" flag
UserBuf dd	0		; "User" buffer pointer and segment
PRDAddr dd	DMAAddr		; PRD Address (always points to DMAAddr)
BufBase dd	Buff		; 32-bit buffer base address
VDS	equ	$		; VDS Parameter Block:
VDSLen	dd	ResEnd		;   Buffer length
VDSOffs dd	0		;   Buffer offset (32-bit!)
VDSSeg	dd	0		;   Buffer segment (LSW) and ID (MSW)
VDSAddr dd	-1		;   Buffer address (32-bit)
DMAAddr dd	0		; DMA buffer address (32-bit)
DMALen	dd	80000000h	; DMA buffer length and "end" flag

; Initialization Variables:
;	NOTE - All memory from Buff through BuffEnd is the driver I/O buffer
;	after init, which places all run-time data (variables and buffer) in
;	one place and takes only ONE init check for crossing a 64K boundary!

Buff	equ	$		; Start of driver I/O buffer
MaxMode db	'6'		; Maximum possible UDMA mode (command option)

; Messages:
@:
TTLMsg	db	'UDMA Hard Disk Driver ', Version, CRLF,'$'
DFMsg	db	CRLF,'Default '
SMMsg	db	CRLF,'UDMA mode '
CurMode db	'2, ATA-'
DspMode db	'33. ',CRLF,'$'
BEMsg	db	CRLF,'64K DMA boundary cross buffer$'
SEMsg	db	CRLF,'Set-mode$'
VEMsg	db	CRLF,'VDS lock$'
Suffix	db	' error - driver NOT loaded',CRLF,'$'
notsup	db	CR,'PCI BIOS 2.0c+ required$'
cntrlr	db	'UDMA controller $'
notfnd	db	'not found$'
busstr	db	'found at bus $'
devstr	db	', device $'
funstr	db	', function $'
adrstr	db	', base $'
busmer	db	CRLF,'Bus master$'

; "Strategy" routine. This routine saves our DOS "Init" request packet
; for handling below. At entry, ES:BX points to the packet.

Strat	mov	[cs:UserBuf+dwd.lw],bx; Save DOS request-packet address
	mov	[cs:UserBuf+dwd.hw],es
	retf			; Exit - await DOS "Device Interrupt"

; "Device-Interrupt" routine. This routine initializes the driver.

DevInt	pushf			; Entry - save used registers
	pushad
	push	ds
	push	es
	push	cs		; Set our DS
	pop	ds
	les	bx,[UserBuf]	; Point to DOS request packet
	cmp	byte [es:bx+RP.RPOp],0; Is this an "Init" packet?
	jne	near I_BadP	; No? Post errors and exit quick!
	mov	dl,TTLMsg-@	; Display our "title" message
	call	putstr
	les	bx,[es:bx+RP.RPCmdl]; Point to the command line tail
GetNext mov	al,[es:bx]	; Get a command line character
	inc	bx
	or	al,al		; Terminate if end of line reached
	jz	FindIt
	cmp	al,CR
	je	FindIt
	cmp	al,LF
	je	FindIt
	cmp	al,'2'		; Mode digit?
	jb	GetNext		; No, continue
	cmp	al,'6'
	ja	GetNext
SetMode mov	[MaxMode],al	; Yes, store the maximum allowable UDMA mode
FindIt	call	find_it		; Find the UDMA controller PCI base port
	jc	near I_Quit	; Not found? Quit!
	mov	[PCIAddr],ax	; Save command register address ( = base)
	mov	dx,CCMD-1	; Issue "Identify Device" command
	mov	al,0A0h
	out	dx,al
	mov	al,0ECh
	call	I_Cmd
	jc	I_DefM		; Error - display "Default" message
	mov	dx,CDATA	; Point to controller PIO data register
	mov	cx,53		; Skip ID bytes 0-105
I_Skp1	in	ax,dx
	loop	I_Skp1
	in	ax,dx		; Read ID bytes 106 and 107
	mov	bh,al		; Save "DMA valid" flag byte
	mov	cl,34		; Skip ID bytes 108-175
I_Skp2	in	ax,dx
	loop	I_Skp2
	in	al,dx		; Read ID bytes 176 and 177
	mov	bl,al		; Save "DMA mode" flag byte
	mov	cl,167		; Skip remaining ID data
I_Skp3	in	ax,dx
	loop	I_Skp3
	test	bh,4		; Are UltraDMA flag bits valid?
	jnz	I_Pick		; Yes, select best UltraDMA mode
I_DefM	mov	dx,DFMsg	; Set up to display "Default" message
	jmp	short I_SelM	; Go display message and set UDMA mode 2
I_Pick	mov	al,[CurMode]	; Get "current mode" value
	cmp	al,[MaxMode]	; Maximum allowable mode reached?
	jae	I_GotM		; Yes, use mode 2
	cmp	bl,8		; No, will disk do UDMA mode 3?
	jb	I_GotM		; No, use mode 2
	inc	ax		; Set up for UDMA mode 3
	mov	word [DspMode],'44'
	cmp	al,[MaxMode]	; Maximum allowable mode reached?
	jae	I_GotM		; Yes, use mode 3
	cmp	bl,10h		; No, will disk do UDMA mode 4?
	jb	I_GotM		; No, use mode 3
	inc	ax		; Set up for UDMA mode 4
	mov	word [DspMode],'66'
	cmp	al,[MaxMode]	; Maximum allowable mode reached?
	jae	I_GotM		; Yes, use mode 4
	cmp	bl,20h		; No, will disk do UDMA mode 5?
	jb	I_GotM		; No, use mode 4
	inc	ax		; Set up for UDMA mode 5
	mov	dword [DspMode],'100.'
	cmp	al,[MaxMode]	; Maximum allowable mode reached?
	jae	I_GotM		; Yes, use mode 5
	cmp	bl,40h		; No, will disk do UDMA mode 6?
	jb	I_GotM		; No, use mode 5
	inc	ax		; Set up for UDMA mode 6
	mov	word [DspMode+1],'33'
I_GotM	mov	[CurMode],al	; Update "current mode" value
	mov	dl,SMMsg-@	; Display "Setting mode" message
I_SelM	call	putstr
	mov	dx,CSUBCM	; Set mode-select subcode
	mov	al,SETM
	out	dx,al
	inc	dx
	mov	al,[CurMode]	; Set desired UltraDMA mode value
	add	al,10h
	out	dx,al
	mov	al,SETF		; Issue set-features command to disk
	call	I_Cmd
	jc	near I_SErr	; Error - display message and exit!
	pushf			; Save CPU flags
	cli			; Disable interrupts during VDS tests
	xor	eax,eax		; Get driver code segment
	mov	ax,cs
	mov	[VDSSeg+dwd.lw],ax; Set code segment in VDS block
	shl	eax,4		; Save 20-bit driver virtual address
	push	eax
	mov	ax,8102h
	xor	dx,dx
	stc			; Prevent false VDS detection
	int	4Bh		; See if VDS services are active
	pop	eax		; (Restore 20-bit virtual address)
	jc	I_SetV		; No VDS - set 20-bit virtual addresses
	push	cs		; Point to VDS descriptor block
	pop	es
	mov	di,VDS
	mov	ax,8103h	; "Lock" this driver into memory
	mov	dx,0Ch
	int	4Bh
	jc	I_VErr		; Failed? Display error message and exit
	cmp	word [VDSAddr+dwd.hw],byte -1; REALLY got a 32-bit address?
	je	I_VErr		; No? Display error message and exit
	or	byte [Flags],80h; Set initialization VDS "lock" flag
	mov	eax,[VDSAddr]	; Get 32-bit starting driver address
I_SetV	popf			; Restore CPU flags
	add	[PRDAddr],eax	; Set 32-bit PRD address
	add	[BufBase],eax	; Set 32-bit buffer base address
	mov	word [VDSLen+dwd.lw],0; Reset lower word of VDS length
	mov	eax,BUFSIZ+7	; Calculate ending 32-bit buffer addr.
	add	eax,[PRDAddr]	; (+7 includes DMAAddr/DMALen too!)
	xor	ax,ax		; Clear low-order ending address
	cmp	eax,[PRDAddr]	; Does buffer cross a 64K boundary?
	jna	I_Vect		; No, do INT13 vector "hook"
	test	byte [Flags],80h; Cannot load! Was driver "locked"?
	jz	I_BErr		; No, display error message and exit
	push	cs		; Point to VDS descriptor block
	pop	es
	mov	di,VDS
	mov	ax,8104h	; "Unlock" this driver from memory
	xor	dx,dx
	int	4Bh
I_BErr	mov	dx,BEMsg	; Boundary error! Point to message
	jmp	short I_Err	; Go display error message and exit
I_VErr	popf			; VDS "lock" error! Restore CPU flags
	mov	dx,VEMsg	; Point to VDS error message
	jmp	short I_Err	; Go display error message and exit
I_Vect	mov	ax,3513h	; Get current INT13 vector
	int	21h
	mov	[Int13V+dwd.lw],bx; Save vector for "passed" requests
	mov	[Int13V+dwd.hw],es
	mov	ax,2513h	; "Hook" this driver into INT13
	mov	dx,Entry
	int	21h
	les	bx,[UserBuf]	; Post driver size and success code
	mov	word [es:bx+RP.RPSize+dwd.lw],ResEnd
	mov	[es:bx+RP.RPSize+dwd.hw],cs
	mov	word [es:bx+RP.RPStat],RPDON
	jmp	short I_Exit	; Go reload registers and exit
I_SErr	mov	dl,SEMsg-@	; Set-mode error! Point to message
I_Err	call	putstr		; Display error message and suffix
I_Quit	mov	dl,Suffix-@
	call	putstr
	les	bx,[UserBuf]	; Post 0 size and error status
	mov	word [es:bx+RP.RPSize+dwd.lw],0
	mov	[es:bx+RP.RPSize+dwd.hw],cs
I_BadP	mov	word [es:bx+RP.RPStat],RPERR; "Unknown command"
I_Exit	pop	es		; Reload registers and exit
	pop	ds
	popad
	popf
	retf

; Subroutine to issue initialization commands to our disk

I_Cmd	mov	dx,CCMD		; Issue desired initialization command
	out	dx,al
	mov	ax,BIOSSEG	; Point to BIOS timer
	mov	es,ax
	mov	di,BIOSTMR
	mov	cl,RDYTO	; Set up ID command time limit
	add	cl,[es:di]	; (3MSB not needed)
I_CmdW	cmp	cl,[es:di]	; Has our ID command timed out?
	je	I_CmdE		; Yes, set CPU carry flag and exit
	in	al,dx		; Get controller alternate status
	test	al,BUSY		; Is our controller still busy?
	jnz	I_CmdW		; Yes, loop back and keep checking
	test	al,ERROR	; Did ID command cause any errors?
	jz	I_CmdX		; No, leave carry flag off and exit
I_CmdE	stc			; Error - set CPU carry flag on
I_CmdX	ret			; Exit

hprint	call	_8p		; Stanislav's tightest but cryptic hex output
_8p	xchg	al,ah
_8print call	_4p
_4p	ror	al,4
_4print push	ax
	and	al,0Fh
	cmp	al,0Ah
	sbb	al,69h
	das
	int	29h
	pop	ax
	ret

putstr	push	ax
	mov	dh,0
	add	dx,@
	mov	ah,9		; display string in DX
	int	21h
	pop	ax
	ret

; Interface byte table - busmaster, legacy mode (bit writable or not):

itable	db	80h,8Ah,0FAh
it_end	equ	$

; Find the UltraDMA controller. Return its base I/O address in AX (CF on error)

find_it mov	dl,cntrlr-@
	call	putstr
	xor	edi,edi
	mov	ax,0B101h
	int	1Ah		; PCI BIOS v2.0c+ installation check
	cmp	edx,'PCI '	; BIOS OK?
	je	biosok		; yes, continue
	mov	dx,notsup	; no, BIOS too old
	jmp	short finderr
biosok	mov	si,itable
	cld			; ensure SI will be increased
findev	mov	ecx,10100h	; class 1 (storage), subclass 1 (IDE)
	lodsb			; interface byte
	mov	cl,al
	mov	ax,0B103h	; find PCI class code
	push	si
	xor	si,si		; prepare device index (0)
	int	1Ah		; BH: bus, BL: bit 7-3: device, 2-0: function
	pop	si
	jnc	found		; if found, break loop
	cmp	si,it_end	; end of table?
	jne	findev		; no, continue searching
	mov	dx,notfnd	; yes - not found, show error and exit
	jmp	short finderr
found	mov	dl,busstr-@
	call	putstr
	mov	ax,bx		; get bus/device/function for display
	xchg	ah,al
	call	_8print		; show bus
	mov	dl,devstr-@
	call	putstr
	rol	ax,5		; move bits 15:8 to 4:0,15:13
	and	al,11111b
	call	_8print		; show device
	mov	dl,funstr-@
	call	putstr
	rol	ax,3		; move bits 15:13 to 2:0
	and	al,111b
	call	_4print		; show function
	mov	di,4		; get command low byte
	mov	ax,0B108h	; read configuration byte
	push	bx		; bus/device/function may be distroyed
	int	1Ah
	pop	bx
	and	cl,101b		; mask Bus Master and I/O Space bits
	cmp	cl,101b		; asked bits supported?
	je	bitsup		; yes, continue
	mov	dl,busmer-@	; no, exit
finderr call	putstr
	stc			; error return
	ret

bitsup	mov	di,20h		; point to base address register 4
	mov	ax,0B109h	; read configuration word
	int	1Ah
	xchg	cx,ax		; base address now in AX
	and	ax,0FFFCh	; mask non-signinficant 2LSb
	mov	dl,adrstr-@	; no, continue
	call	putstr
	call	hprint		; show base address
	clc
	ret			; base in AX

	times Buff+BUFSIZ-$ nop ; "Pad" remainder of I/O buffer: DON'T REMOVE!
				; (forces RAM occupation at load time)
BuffEnd equ	$		; End of init logic and driver I/O buffer

; Main driver routine, invoked by Int 13h (Disk Services) calls.
; For a CHS request, the registers contain the following at entry:
;
;   AH	   Command code - we handle only 2 (read) and 3 (write)
;   AL	   I/O sector count
;   CH	   Lower 8 bits of starting cylinder number
;   CL	   Starting sector number and upper 2 bits of cylinder number
;   DH	   Starting head number
;   DL	   Unit number - we handle only 80h (hard disk 0)
;   ES:BX  I/O buffer address
;
; For an LBA request, the registers contain the following at entry:
;
;   AH	   Command code - we handle only 42h (read) and 43h (write)
;   DL	   Unit number	- we handle only 80h (hard disk 0)
;   DS:SI  Pointer to Device Address Packet ("DAP"), described above

Entry	pushf			; Driver entry point - save CPU flags
	cmp	dl,80h		; Is this request for our disk?
	je	CheckRW		; Yes, check for read or write command
NotUs	popf			; Request NOT for us - reload CPU flags
	db	0EAh		; "Pass" request to BIOS or other drivers
Int13V	dd	-1		; Previous INT13 vector, set by Init
CheckRW push	ax		; Save command
	and	ah,0BEh		; Mask out LBA (bit 6) and write (bit 0)
	cmp	ah,2		; Is command a read or write?
	pop	ax		; Reload command
	jne	NotUs		; NOT read or write - exit quick!
	popf			; Valid request - discard entry flags
	sti			; Enable CPU interrupts
	push	eax		; Save needed CPU registers
	push	cx		; (Unusual order, for exit logic, and
	push	dx		;  so DS:SI "DAP" pointer is on stack)
	push	di
	push	es
	push	ds
	push	si
	push	bp
	push	cs		; Set our DS
	pop	ds
	mov	bp,sp		; Point BP to our "stack frame"
	and	byte [bp+22+dbt.lb],0FEh; Reset exiting carry bit (error flag)
	mov	dl,ah		; Save opcode for LBA test below
	xchg	al,ah		; Swap opcode and CHS sector count
	and	al,0BFh		; Mask Int 13 extension bit (6)
	shl	al,1		; Get DMA command byte
	add	al,DRCMD-4
	mov	[IOCmd],ax	; Save DMA command and CHS sector count
	shl	dl,1		; New-style LBA request?
	jns	CalcLBA		; No, calculate the LBA from CHS values
	les	di,[bp+2]	; Get "DAP" pointer in ES:DI
	mov	al,[es:di+DAP.DANum]; Save I/O sector count
	mov	[Sectors],al
	mov	si,[es:di+DAP.DABeg+dwd.lw]; Get LBA value in SI and DX
	mov	dx,[es:di+DAP.DABeg+dwd.hw]
	les	cx,[es:di+DAP.DABuf]; Get user I/O buffer address
	jmp	short SaveBuf	; Go save buffer address and LBA value
CalcLBA mov	si,SECSHD	; CHS - save starting sector in SI
	and	si,cx
	mov	di,dx		; Save starting head in DI
	shr	cl,6		; Get cylinder number in CX
	xchg	cl,ch
	mov	ax,SECSCYL	; Convert cylinder to sectors
	mul	cx
	xchg	ax,di		; Swap low-order and head number
	mov	al,SECSHD	; Convert head to sectors
	mul	ah
	add	di,ax		; Add to cylinder sectors
	adc	dl,dh
	dec	si		; Get starting sector number - 1
	add	si,di		; Add to cylinder/head sectors
	adc	dl,dh
	mov	cx,bx		; Set buffer offset in CX
SaveBuf mov	[UserBuf+dwd.lw],cx; Save user I/O buffer address
	mov	[UserBuf+dwd.hw],es
	mov	[LBA+dwd.lw],si ; Save lower LBA value in I/O parameters
	and	dh,0Fh		; Ensure maximum 28-bit LBA value
	or	dh,LBABITS	; "Or" in DEV/LBA bits
	mov	[LBA+dwd.hw],dx ; Save upper LBA value in I/O parameters
	mov	word [VDSAddr+dwd.hw],-1; Invalidate DMA address
	test	cl,3		; Is user's buffer DWORD-aligned?
	jnz	UseBuf		; No, use driver-buffered logic below
	movzx	ax,[Sectors]	; Set I/O sector count
	mov	[SectCt],al
	shl	ax,1		; Set VDS buffer length
	mov	[VDSLen+1],ax
	mov	[VDSOffs+dwd.lw],cx; Save buffer address in VDS block
	mov	ax,es
	mov	[VDSSeg+dwd.lw],ax
	cli			; Disable CPU interrupts during VDS tests
	test	byte [Flags],80h; Was driver "locked" during init?
	jnz	DoLock		; Yes, VDS services must be active
	mov	ax,8102h
	xor	dx,dx
	stc			; Prevent false detection
	int	4Bh		; Are VDS services active?
	jc	NoVDS		; No, set up 20-bit DMA address
DoLock	push	ds		; Point to VDS descriptor block
	pop	es
	mov	di,VDS
	mov	ax,8103h	; Do VDS "lock" on our I/O buffer
	mov	dx,0Ch
	int	4Bh
	sti			; Re-enable CPU interrupts
	jc	UseBuf		; If error, use buffered logic below
	cmp	word [VDSAddr+dwd.hw],byte -1; Got a valid DMA address?
	je	UseBuf		; No, use driver-buffered logic below
	inc	byte [VDSLock]	; Set VDS "lock" flag
DirDMA	mov	eax,[VDSLen]	; Get I/O buffer length - 1
	dec	eax
	add	eax,[VDSAddr]	; Calculate ending DMA address
	xor	ax,ax		; Clear low-order ending address
	cmp	eax,[VDSAddr]	; Will this I/O cross a 64K boundary?
	ja	UseBuf		; Yes, use driver-buffered logic below
	call	Ready		; Await controller- and disk-ready
	mov	eax,[VDSAddr]	; Do DMA input direct to user buffer
	call	DoIO
	jmp	short GoodIO	; All done - post "success" and exit
NoVDS	sti			; No VDS - re-enable CPU interrupts
	mov	eax,[VDSSeg]	; Set 20-bit DMA address
	shl	eax,4
	add	eax,[VDSOffs]
	mov	[VDSAddr],eax
	jmp	DirDMA		; Go do direct DMA to/from user buffer
UseBuf	call	Ready		; Await controller- and disk-ready
	align	4
Next	mov	al,[Sectors]	; Get remaining I/O sector count
	cmp	al,SECSBUF	; Will next I/O be a full buffer?
	jb	NextSC		; No, use remaining sector count
	mov	al,SECSBUF	; Use full-buffer sector count
NextSC	mov	[SectCt],al	; Set next I/O sector count
	cmp	byte [IOCmd],DRCMD; Is this a read request?
	jne	Output		; No, use output logic below
	call	BufIO		; Input next block of sectors
	movzx	cx,[SectCt]	; Set move data length
	shl	cx,7		; (32-bit word count)
	mov	si,Buff		; Set driver and user buffer pointers
	les	di,[UserBuf]
	cli			; Move driver data to user's buffer
	rep	movsd
	sti
	mov	[UserBuf+dwd.lw],di; Update user buffer offset
	jmp	short More	; Go check for more I/O sectors
Output	push	ds		; Output - save our DS
	pop	es		; Set driver buffer pointer
	mov	di,Buff
	lds	si,[UserBuf]	; Set user buffer pointer
	movzx	cx,al		; Set move data length
	shl	cx,7		; (32-bit word count)
	cli			; Move user data to driver's buffer
	rep	movsd
	sti
	push	es		; Reload our DS
	pop	ds
	mov	[UserBuf+dwd.lw],si; Update user buffer offset
	call	BufIO		; Output next block of sectors
More	sub	byte [Sectors],SECSBUF; More I/O sectors left to go?
	ja	Next		; Yes, loop back
GoodIO	mov	byte [bp+14+dbt.hb],0; Success! Reset exiting AH
Done	pop	bp		; Reload BP and SI (not needed below)
	pop	si
	shr	byte [VDSLock],1; Was I/O buffer "locked" through VDS?
	jnc	Exit		; No, reload remaining registers
	push	ds		; Point to VDS descriptor block
	pop	es
	mov	di,VDS
	mov	ax,8104h	; Do VDS "unlock" on input buffer
	xor	dx,dx
	int	4Bh
Exit	pop	ds		; Reload remaining registers and exit
	pop	es
	pop	di
	pop	dx
	pop	cx
	pop	eax
	iret

; Subroutine to start I/O. This logic exits by "falling through" to
; the "Ready" routine below, nasty but FAST assembly-language code!

BufIO	mov	eax,[BufBase]	; Buffered - get driver buffer address
DoIO	push	eax		; Save 32-bit DMA address for below
	mov	al,[IOCmd]	; If reading, get DMA "read" command
	add	al,6
	and	al,8
	mov	dx,[PCIAddr]	; Reset DMA command register
	out	dx,al
	xchg	ax,di		; Save DMA "read/write" byte
	inc	dx		; Reset DMA control register
	inc	dx
	mov	ax,26h
	out	dx,al
	pop	dword [DMAAddr] ; Set DMA buffer address
	mov	al,[SectCt]	; Set DMA buffer length
	shl	ax,1		; (AH reset by "mov ax,26h" above)
	mov	[DMALen+1],ax
	inc	dx		; Set PRD pointer to our DMAAddr
	inc	dx
	mov	eax,[PRDAddr]
	out	dx,eax
	mov	dx,CSECCT	; Output our 6 I/O parameters
	mov	si,Params
DMAPar	outsb
	inc	dx
	cmp	dx,CCMD+1
	jb	DMAPar
ChkDRQ	mov	dx,CSTAT2	; Read IDE alternate status
	in	al,dx
	test	al,DRQ		; Has 1st DRQ arrived?
	jz	ChkDRQ		; No, wait
	xchg	ax,di		; Reload DMA "read/write" byte
	mov	dx,[PCIAddr]	; Set DMA "GO" command bit
	inc	ax
	out	dx,al
	add	dword [LBA],byte SECSBUF; Update logical block address
	mov	byte [ChkErrs],FAULT+ERROR; Check fault and error at I/O end

; Subroutine to await controller- and disk-ready

Ready	cld			; Ensure we do "forward" string commands!
	mov	di,BIOSSEG	; Point to BIOS timer
	mov	es,di
	mov	di,BIOSTMR
	mov	cx,RDYTO	; Set up command timeout limit
	add	cx,[es:di]
	mov	ah,FAULT	; Load/reset "Check I/O Errors" flag
	xchg	ah,[ChkErrs]
	test	ah,ERROR	; Checking one of our I/O requests?
	jz	ChkRdy		; No, do NOT check DMA status
ChkDMA	cmp	cx,[es:di]	; Has a DMA transfer timed out?
	je	TimedO		; Yes? Post timeout error and exit!
	mov	dx,[PCIAddr]	; Read DMA controller status
	inc	dx
	inc	dx
	in	al,dx
	xor	al,INTRPT	; Get "no-interrupt" status
	and	al,INTRPT+ACTIVE; DMA still in progress, and no interrupt?
	jnz	ChkDMA		; Yes, loop back and check again
	xchg	cx,cx		; (Unused alignment "filler")
ChkRdy	cmp	cx,[es:di]	; Too long without becoming ready?
	je	TimedO		; Yes? Post timeout error and exit!
	mov	dx,CSTAT	; Read IDE controller primary status
	in	al,dx
	xor	al,DSKRDY+ONCYL ; Get "not-ready" and "off-cylinder"
	test	al,BUSY+DSKRDY+ONCYL; Controller or disk still busy?
	jnz	ChkRdy		; Yes, loop back and check again
	and	al,ah		; Error or disk fault at I/O end?
	mov	ax,HERROR	; (Load hard-error code if so)
	jnz	BadIO		; Yes? Post hard-error and exit!
	ret			; All is well - exit
TimedO	mov	ax,TIMOUT	; Timed out! Get error code
BadIO	pop	dx		; Discard subroutine exit address
	mov	[bp+14],ax	; Set error and reset exiting sector count
	inc	byte [bp+22+dbt.lb]; Set exiting carry bit (error!)
	jmp	Done		; Go see about VDS "unlock" and exit

	align	64		; Cache line alignment
ResEnd	equ	$		; Resident driver end
