;
; UDMA2S.ASM    Written 28-Sep-2005 by Jack R. Ellis
;
; UDMA2S is free software.  You can redistribute and/or modify it under
; the terms of the GNU General Public License (hereafter called GPL) as
; published by the Free Software Foundation, either version 2 of GPL or
; any later version at your option.   UDMA2S is distributed in the hope
; that it will be useful, but WITHOUT ANY WARRANTY and without even the
; implied warranties of MERCHANTABILITY nor of FITNESS FOR A PARTICULAR
; PURPOSE!  See the GPL for details.  You ought to have received a copy
; of the GPL with your UDMA2S files; if not, write to the Free Software
; Foundation Inc., 59 Temple Place Ste. 330, Boston, MA 02111-1307 USA.
; http://www.gnu.org/licenses/
;
; Special thanks to Luchezar I. Georgiev for his INESTIMABLE advice and
; help in research, revising, enhancing, and test of the original UDMA!
;
; This is a DOS driver for UltraDMA hard-disks on PC motherboards using
; a VIA VT8235 or similar chipset.   On loading, the driver finds which
; IDE drives are UltraDMA hard-disks and will run all such disks.   Any
; UltraDMA disks may be used.   LBA and CHS mode requests are accepted.
; "Read/Write Extended" DMA commands are issued for disk addresses over
; 28 bits and require ATA-6 or newer hard-disks.   Disk addresses of 28
; bits or less use normal DMA commands.   The driver handles BIOS Int13
; read or write requests only.   Other Int13 requests (including seeks)
; and I-O requests having bad parameters are "passed" back to the BIOS.
; If an input buffer is not 32-bit aligned, crosses a 64K DMA boundary,
; or fails VDS "lock", data goes through a 64K XMS memory buffer, using
; UltraDMA to/from the buffer.   If XMS is not available, such requests
; will also be "passed" back to the BIOS.
;
; UDMA2S switch options, specified in the CONFIG.SYS command that loads
; the driver, are as follows:
;
;   /L   Limits DMA to "low memory" below 640K.   /L is REQUIRED to use
;          the UMBPCI upper-memory driver, or any similar drivers whose
;          upper-memory areas do not support DMA.   If /L is specified,
;          UDMA2S must be loaded in LOW memory so DMA command-lists can
;          be "fetched", or driver loading will ABORT!    /L causes any
;          I-O request past 640K to go through the driver's XMS buffer.
;          If /B is also used, or if XMS is unavailable, such I-O shall
;          be "passed" to the BIOS for execution.
;
;   /B   Causes XMS memory to be IGNORED, for "backward compatibility".
;          /B forces "DMA only" mode and affects /L as noted above.
;
;   /Mn  Specifies the MAXIMUM UltraDMA "mode" to be used by all disks,
;          where  n  is a number between 0 and 7, as follows:
;              0 = ATA-16, 16 MB/sec.     4 = ATA-66,   66 MB/sec.
;              1 = ATA-25, 25 MB/sec.     5 = ATA-100, 100 MB/sec.
;              2 = ATA-33, 33 MB/sec.     6 = ATA-133, 133 MB/sec.
;              3 = ATA-44, 44 MB/sec.     7 = ATA-166, 166 MB/sec.
;          Disks designed to a "mode" LESS than the given value will be
;          limited to their own highest "mode".
;
;   /S   Enables the driver's local-stack for all I-O requests.
;
; For each switch, a dash may replace the slash, and lower-case letters
; may also be used.
;
; On exit from successful I-O requests, the AH-register is zero and the
; carry flag is reset.	 If an error occurs, the carry flag is SET, and
; the AH-register contains one of the following codes:
;
;    Code 08h - DMA timed out
;	  0Fh - DMA error
;	  20h - Controller busy before I-O
;	  21h - Controller busy after I-O
;	  80h - First DRQ timed out
;	  AAh - Disk not ready before I-O
;	  ABh - Disk not ready after I-O
;	  CCh - Disk FAULT before I-O
;	  CDh - Disk FAULT after I-O
;	  E0h - Hard error at I-O end
;	  FEh - BIOS/driver read MISMATCH (init only)
;	  FFh - XMS memory error
;
;
; Revision History:
; ----------------
;  V2.6  28-Sep-05   JE   Put in David Muller's ALi class-codes and EDD
;                           30-byte validity check.   THANK YOU, David!
;  V2.5  19-Apr-05   JE   Fixed "Math ERROR" in XMS buffer alignment
;  V2.4  11-Apr-05   JE   Revised to use "DVRCORE.ASM" core logic
;  V2.3  28-Mar-05   JE   SERIOUS init problem fixed; "read tests" OUT!
;  V2.2  20-Mar-05   JE   /S enables a local-stack, other minor changes
;  V2.1  19-Feb-05   JE   Fixed LBA bits 32-47 and switch-option errors
;  V2.0   7-Jan-05   JE   Added /L for UMBPCI, fixed no-read-test flag
;  V1.9  18-Dec-04   JE   Fixed DRQ timeout code
;  V1.8  11-Dec-04   JE   Fixed FOOLISH disk-select bug (My apologies!)
;  V1.7  10-Dec-04   JE   No EDD BIOS error abort, more code reductions
;  V1.6	  2-Dec-04   JE   Initial release derived from the UDMA2 "core"
;
;
; Program-Specific Equations.
;
%define	DVRNAME	'UDMA2S$',0	    ;Driver name.
%define VERSION	'V2.6, 28-Sep-2005' ;Version name for title.
%define XMINPUT	XMSMov		    ;XMS input pointer.
%define	VLENGTH	ResEnd		    ;Initial VDS length.
STACK	equ	312		    ;Driver local-stack size.
%include "DVRCORE.ASM"
;
; Local-Stack Entry and Exit Routines.   This logic is "dismissed"
;   during driver initialization if a local-stack is not requested.
;
LSExit	pushf			;I-O exit -- save error flag (carry).
	call	VDUnlk		;If needed, "unlock" user I-O buffer.
	popf			;Reload error flag into carry bit.
LSExit1	cli			;"Pass" exit -- disable interrupts.
	mov	bp,sp		;Put exit address in stacked BP-reg.
	pop	word [bp+10]
	inc	byte [bp+BPFlags]  ;Reset driver "busy" flag.
	pop	es		   ;Reload all CPU registers.
	pop	ds
	popa
	lss	sp,[cs:CStack]	;Switch to caller's stack and exit.
	jmp	bp
LSEntry	cli			;Entry -- disable CPU interrupts.
	sar	byte [cs:Flags],1  ;Busy handling a prior I-O request?
	jc	InvalF		;Yes?  Post "invalid function" & exit!
	mov	[cs:CStack],sp	;Save caller's stack pointer.
	mov	[cs:CStack+2],ss
	mov	sp,cs		;Switch to this driver's stack.
	mov	ss,sp
	mov	sp,CStack
	not	byte [cs:Flags]	;Post driver "busy" flag.
	sti			;Re-enable CPU interrupts.
	pusha			;Save all CPU registers.
	push	ds
	push	es
	jmp	TestRW		;Go check for read or write request.
InvalF	mov	ah,001h		;We're BUSY, you fool!  Post "invalid
	jmp	GetOut		;  function" code and get out, quick!
CStack	equ	$+STACK		;Caller's saved stack pointer.
BPFlags	equ	Flags-CStack+22	;Exiting "Flags" offset via BP-reg.
LSEnd	equ	CStack+4	;End of resident local-stack driver.
;
; Initialization Tables And Variables.
;
HDNames dw	PMMsg		;Table of hard-disk "name" pointers.
	dw	PSMsg
	dw	SMMsg
	dw	SSMsg
Modes	db	'16. '		;Mode 0 = ATA-16  UltraDMA mode table.
	db	'25. '		;Mode 1 = ATA-25.
	db	'33. '		;Mode 2 = ATA-33.
	db	'44. '		;Mode 3 = ATA-44  (Rare but possible).
	db	'66. '		;Mode 4 = ATA-66.
	db	'100.'		;Mode 5 = ATA-100.
	db	'133.'		;Mode 6 = ATA-133.
	db	'166.'		;Mode 7 = ATA-166.
NonXLen	dw	NonXEnd		;Non-XMS length ("LSEnd-@" for stack).
MaxMode	db	0FFh		;Maximum desired UltraDMA "mode".
IgnXMS	db	0		;"Ignore XMS memory" flag.
BiosHD	db	0		;BIOS hard-disk count.
EDDFlag db	0		;"EDD BIOS present" flag.
HDUnit	db	0		;Current BIOS unit number.
HDIndex db	0		;IDE "index" number.
EDDBuff equ	$		;Start of 30-byte EDD input buffer.
;
; Initialization Messages.
;
UDMsg	db	CR,LF,'UDMA2S Disk Driver ',VERSION
CRMsg	db	'.',CR,LF,'$'
PRMsg	db	'CPU is below an 80386$'
PEMsg	db	'PCI BIOS is not V2.0C+$'
NEMsg	db	'No controller found$'
MEMsg	db	'Bus-Master setup BAD$'
NXMsg	db	'No XMS manager$'
XEMsg	db	'XMS init error$'
XDMsg	db	'/B, XMS ignored$'
NBMsg	db	'; using only DMA I-O.',CR,LF,'$'
VEMsg	db	'VDS lock error$'
UMMsg	db	'CANNOT use /L if residing "high"$'
UNMsg	db	'unnamed$'
DNMsg	db	'is '
DName	equ	$		;Disk names overlay the following.
DNEnd	equ	DName+40
PCMsg	db	'UltraDMA controller at I-O address '
DspAd	db	'0000h, Vendor/Device I.D. '
DspID	db	'0000h/0000h.',CR,LF,'$'
NDMsg	db	'No disks to use$'
EBMsg	db	'EDD error!  BIOS unit ignored$'
HOMsg	db	'Hardware-only disk scan:',CR,LF,'$'
PMMsg	db	'Primary-master disk $'
PSMsg	db	'Primary-slave disk $'
SMMsg	db	'Secondary-master disk $'
SSMsg	db	'Secondary-slave disk $'
MSMsg	db	', ATA-'
DMode	db	'16. ',CR,LF,'$'
AEMsg	db	'absent or non-ATA$'
DEMsg	db	'is not UltraDMA$'
NPMsg	db	'CHS-values error$'
SEMsg	db	'Set-Mode error$'
Suffix	db	'; driver NOT loaded!',CR,LF,'$'
;
; Local-Stack "Overlay" Table.   The code in this table overlays
;   the resident driver when our local-stack is requested.
;
LS_Ovl	db	1		;"Entry" overlay length & address.
	dw	Entry
	push	bp		;Driver entry -- Save BP-register.
	db	3		;"@LS1" overlay length & address.
	dw	@LS1
	jmp	$+LSEntry-@LS1	;Jump to local-stack entry routine.
	db	PasEnd-PasBeg	;"Pass" overlay length & address.
	dw	Pass
PasBeg	call	$+LSExit1-Pass	;Call local-stack exit routine.	
	mov	bp,sp		;Reload CPU flags saved by Int 13h.
	push	word [bp+6]
	popf
	pop	bp		;Reload BP-register.
PasEnd	equ	$
	db	LS2End-LS2Beg	;"@LS2" overlay length & address.
	dw	@LS2
LS2Beg	call	$+LSExit-@LS2	;Call local-stack exit routine.
	mov	bp,sp		;Point to caller's saved BP-reg.
	rcr	byte [bp+6],1	;Set error flag in exiting carry
	rol	byte [bp+6],1	;  bit, in flags saved by Int 13h.
	pop	bp		;Reload BP-register.
LS2End	equ	$
	db	0		;End of local-stack overlay table.
;
; "Strategy" Routine -- At entry, ES:BX points to the DOS init request
;   packet, whose address is saved for processing below.
;
Strat	mov	[cs:LBA+1],bx	;Save DOS request-packet address.
	mov	[cs:LBA+3],es
	retf			;Exit & await DOS "Device Interrupt".
;
; "Device-Interrupt" Routine.   This logic initializes the driver.
;
DevInt	pushf			;Entry -- save CPU flags.
	push	ds		;Save all 16-bit CPU registers.
	push	es
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	cs		;Set our DS-register.
	pop	ds
	les	bx,[LBA+1]	;Point to DOS request packet.
	cmp	byte [es:bx+RPOp],0  ;Is this an "Init" packet?
	je	I_CmdL		;Yes, scan command-line parameters.
	jmp	I_BadP		;Go post errors and exit quick!
I_CmdL	les	bx,[es:bx+RPCL]	;Get command-line data pointer.
I_NxtC	mov	ax,[es:bx]	;Get next 2 command-line bytes.
	inc	bx		;Bump pointer past first byte.
	cmp	al,0		;Is byte the command-line terminator?
	je	I_TTL		;Yes, display driver "title" message.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je	I_TTL		;Yes, display driver "title" message.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je	I_TTL		;Yes, display driver "title" message.
	cmp	al,'/'		;Is byte a slash?
	je	I_ChkB		;Yes, see if "B" or "b" is next.
	cmp	al,'-'		;Is byte a dash?
	jne	I_NxtC		;No, check next command-line byte.
I_ChkB	mov	al,ah		;Get byte after the slash or dash.
	and	al,0DFh		;Mask out lower-case bit (020h).
	cmp	al,'B'		;Is byte a "B" or "b"?
	jne	I_ChkL		;No, go see if byte is "L" or "l".
	mov	[IgnXMS],al	;Set "Ignore XMS" flag below.
I_Nxt1	inc	bx		;Point to next command-line byte.
	jmp	short I_NxtC	;Continue scanning for a terminator.
I_ChkL	cmp	al,'L'		;Is byte an "L" or "l"?
	jne	I_ChkM		;No, go see if byte is "M" or "m".
	mov	byte [@DMALmt],009h  ;Set 640K "DMA limit" above.
	jmp	short I_Nxt1	;Continue scanning for a terminator.
I_ChkM	cmp	al,'M'		;Is byte an "M" or "m"?
	jne	I_ChkS		;No, go see if byte is "S" or "s".
	inc	bx		;Get byte following the "M" or "m".
	mov	al,[es:bx]
	cmp	al,'0'		;Is byte below a zero?
	jb	I_NxtC		;Yes, see if byte is a terminator.
	cmp	al,'7'		;Is byte above a seven?
	ja	I_NxtC		;Yes, see if byte is a terminator.
	mov	[MaxMode],al	;Set maximum UltraDMA "mode" value.
	jmp	short I_Nxt1	;Continue scanning for a terminator.
I_ChkS	cmp	al,'S'		;Is byte an "S" or "s"?
	jne	I_NxtC		;No, see if byte is a terminator.
	mov	ax,LSEnd	;Set "local-stack" driver length.
	mov	[NonXLen],ax
	mov	[VDSLn],ax
	cld			;Ensure FORWARD "string" commands!
	push	cs		;Point ES-register to this driver.
	pop	es
	mov	si,LS_Ovl	;Point to local-stack overlay table.
I_LSOvl	lodsb			;Get next overlay length.
	movzx	cx,al		;At the end of our overlay table?
	jcxz	I_Nxt1		;Yes, continue terminator scan.
	lodsw			;Overlay next resident-code section.
	xchg	ax,di
	rep	movsb
	jmp	short I_LSOvl	;Go get next overlay length.
I_TTL	mov	dx,UDMsg	;Display driver "title" message.
	call	I_Dspl
	pushf			;80386 test -- save CPU flags.
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax
	cmp	ax,sp		;(80286+ push SP, then decrement it)
	jne	I_Junk		;CPU is 8086/80186 -- cannot USE it!
	push	07000h		;80286 and up -- try to set NT|IOPL.
	popf
	pushf
	pop	ax
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz	I_386		;Yes, CPU is at least an 80386.
I_Junk	popf			;Reload starting CPU flags.
	mov	dx,PRMsg	;Display "No 80386 CPU" message.
	call	I_Dspl
	jmp	I_Quit		;Go display suffix and exit quick!
I_386	popf			;Reload starting CPU flags.
	pop	di		;Reload all 16-bit registers.
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pushad			;Save all 32-bit registers.
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	ax,0B101h
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	cmp	edx,"PCI "	;Is PCI BIOS V2.0C or newer?
	mov	dx,PEMsg	;(Get error message pointer if not).
	jne	I_PCEr		;No?  Go display message and exit!
	mov	si,SecCt2	;Point to interface byte table.
	cld			;Ensure FORWARD "string" commands!
I_GetD	mov	ecx,000010100h	;We want class 1 storage, subclass 1 IDE.
	lodsb			;Get next "valid" PCI interface byte.
	mov	cl,al
	push	si		;Search for our PCI class code.
	mov	ax,0B103h	;(Returns bus/device/function in BX).
	xor	si,si
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	si
	jnc	I_GotD		;Found our boy!  Go process it.
	cmp	si,byte (LBA+1)	;More interface bytes to try?
	jb	I_GetD		;Yes, go try next one.
	mov	dx,NEMsg	;BAAAD News!  Point to error message.
	jmp	short I_PCEr	;Go display error message and exit.
I_GotD	push	bx		;Save PCI bus/device/function.
	mov	ax,0B108h	;Get low-order PCI command byte.
	mov	di,4
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	not	cl		;Mask Bus-Master and I-O Space bits.
	and	cl,005h		;Is this how our controller is set up?
	jz	I_Base		;Yes, get our PCI base address.
	mov	dx,MEMsg	;Cannot USE it -- point to error msg.
I_PCEr	jmp	I_EOut		;Go display "INVALID" message & exit!
I_Base	push	bx		;Save PCI bus/device/function.
	mov	ax,0B109h	;Get PCI base address (register 4).
	mov	di,32
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	xchg	ax,cx		;Post our DMA controller address.
	and	al,0FCh
	mov	[DMAAd],ax
	mov	[@DMALo1],al	;Set low-order DMA address bytes.
	add	[@DMALo2],al
	mov	si,DspAd	;Set hex address in display message.
	call	I_Hex4
	mov	ax,0B10Ah	;Get Vendor and Device I.D.
	xor	di,di
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	xchg	eax,ecx		;Set vendor I.D. in display message.
	mov	si,DspID
	call	I_Hex4
	shr	eax,16		;Set Device I.D. in display message.
	inc	si
	inc	si
	call	I_Hex4
	mov	dx,PCMsg	;Display all controller information.
	call	I_Dspl
	mov	dx,XDMsg	;Get "XMS ignored" message pointer.
	mov	ax,04300h	;Get "XMS present" request code.
	cmp	al,[IgnXMS]	;Is XMS memory to be ignored?
	jne	I_XErr		;Yes, display message & disable XMS.
	int	02Fh		;Inquire about an XMS manager.
	push	cs		;Reload our DS-register.
	pop	ds
	mov	dx,NXMsg	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_XErr		;No, display message & disable XMS.
	mov	ax,04310h	;Get XMS manager "entry" address.
	int	02Fh
	push	cs		;Reload our DS-register.
	pop	ds
	push	es		;Save XMS manager "entry" address.
	push	bx
	pop	dword [@XEntry]
	mov	ah,009h		;Ask XMS manager for 128K of memory.
	mov	dx,128
	call	XMCall
	jc	I_XMErr		;If error, display msg. & disable XMS.
	mov	[XMHdl],dx	;Save XMS buffer handle number.
	mov	ah,00Ch		;"Lock" our XMS memory (direct call,
	call	far [@XEntry]	;  as our subroutine ZEROS BX-reg.!).
	push	cs		;Reload our DS-register.
	pop	ds
	dec	ax		;Did XMS memory get locked?
	jnz	I_RidX		;No?  Unuseable -- get RID of it!
	shl	edx,16		;Get unaligned XMS buffer address.
	mov	dx,bx
	mov	eax,edx
	or	ax,ax		;Is buffer ALREADY on a 64K boundary?
	jz	I_XBAd		;Yes, use XMS buffer address as-is.
	add	eax,65536	;Find 1st 64K boundary after start.
	xor	ax,ax
I_XBAd	mov	[@XBufAd],eax	;Set final XMS 32-bit buffer address.
	sub	eax,edx		;Initialize "offset" into XMS memory.
	mov	[XMOfs],ax
	mov	ah,005h		;Local-enable "A20 line" FOREVER!
	call	XMCall		;("A20" CANNOT turn off during DMA!).
	jnc	I_Stop		;If no error, ensure DMA is stopped.
	mov	ah,00Dh		;Unuseable!  Unlock our XMS memory.
	mov	dx,[XMHdl]
	call	XMCall
I_RidX	xor	dx,dx		;Load & reset our XMS buffer handle.
	xchg	dx,[XMHdl]
	mov	ah,00Ah		;Free our XMS memory.
	call	XMCall
I_XMErr mov	dx,XEMsg	;Point to "XMS setup error" message.
I_XErr	call	I_Dspl		;Display desired XMS error message.
	mov	dx,NBMsg	;Display "no buffered I-O" message.
	call	I_Dspl
	mov	ax,Pass-(GoBuf+3)  ;Reject buffered I-O using a
	mov	[GoBuf+1],ax	   ;  "jmp Pass" at label GoBuf.
	mov	ax,[NonXLen]	   ;Set non-XMS driver length.
	mov	[VDSLn],ax
I_Stop	mov	dx,[DMAAd]	;Ensure any previous DMA is stopped.
	in	al,dx		;(See "DoDMA" routine notes below).
	and	al,0FEh
	out	dx,al
	add	dx,byte 8	;Stop secondary-channel DMA, also!
	in	al,dx
	and	al,0FEh
	out	dx,al
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	[SecCt2],al     ;Reset IDE upper sector count.
	mov	es,ax		;Point ES-reg. to low memory.
	or	al,[es:HDISKS]	;Did BIOS find any hard-disks?
	jz	near I_RScn	;No?  Display "No disk" and exit!
	mov	[BiosHD],al	;Save BIOS hard-disk count.
	mov	ax,cs		;Set our code segment in VDS block.
	mov	[VDSSg],ax
	shl	eax,4		;Set 20-bit driver virtual address.
	mov	[IOAdr],eax
	cli			;Avoid interrupts during VDS tests.
	test	byte [es:VDSFLAG],020h ;Are "VDS services" active?
	jz	I_SetA		;No, set 20-bit virtual addresses.
	mov	ax,08103h	;"Lock" this driver into memory.
	mov	dx,0000Ch
	call	VDLock
	jc	near I_VErr	;Error?  Display error msg. & exit!
	inc	byte [VDSOf]	;Set initialization VDS "lock" flag.
	mov	eax,[IOAdr]	;Get 32-bit starting driver address.
I_SetA	sti			;Re-enable CPU interrupts.
	add	[PRDAd],eax	;Set relocated 32-bit PRD address.
	movsx	ax,[@DMALmt]	;Get DMA memory limit (if any).
	inc	ax		;Has /L limited DMA to low-memory?
	jz	I_EDD		;No, see if we have an EDD BIOS.
	mov	dx,UMMsg	;Point to upper-memory error message.
	cmp	ax,[IOAdr+2]	;Are we loaded in upper-memory?
	jbe	near I_Err	;Yes?  Display error message & exit!
I_EDD	mov	ah,041h		;See if we have an EDD BIOS.
	mov	bx,055AAh
	mov	dl,080h
	int	013h
	push	cs		;(Reload our DS-register).
	pop	ds
	jc	I_Scan		;No, scan for disks without EDD.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne	I_Scan		;No, scan for disks without EDD.
	xchg	ax,cx		;Set "EDD BIOS present" flag.
	and	al,004h
	mov	[EDDFlag],al
	jmp	short I_Scan	;Go scan for UltraDMA disks to use.
I_RScn	mov	dx,NDMsg	;Point to "No disk to use" message.
	xor	ax,ax		;Load & reset EDD BIOS flag.
	xchg	al,[EDDFlag]
	or	ax,ax		;Were we scanning v.s. DPTE data?
	jz	near I_Err	;No?  Display "No disk" and exit!
I_Scan	mov	ax,00080h	;Reset hard-disk unit number & index.
	mov	[HDUnit],ax
	mov	al,[BiosHD]	;Reset remaining hard-disk count.
	mov	[Timer],al
	cmp	ah,[EDDFlag]	;Will disk scan use the EDD BIOS?
	jne	I_Next		;Yes, go start with BIOS unit 80h.
	mov	dx,HOMsg	;Display "hardware-only" message.
	call	I_Dspl
I_Next	movzx	bx,[HDIndex]	;Get disk unit-number index.
	cmp	bh,[EDDFlag]	;Are we using DPTE data from BIOS?
	je	I_ChMS		;No, check disk at "fixed" addresses.
	mov	si,EDDBuff	;Point to EDD input buffer.
	mov	word [si],30	;From David Muller:  Set 30-byte size.
	mov	ah,048h		;Get next BIOS disk's EDD parameters.
	mov	dl,[HDUnit]
	int	013h
	push	cs
	pop	ds
	jc      I_ErEDD		       ;Error?  Ignore this BIOS unit!
	cmp	word [si],byte 30      ;From David Muller:  30+ bytes?
	jb	I_ErEDD		       ;No?  Ignore this BIOS unit!
	cmp	dword [si+26],byte -1  ;Valid DPTE pointer?
	je	near I_More	;No, ignore unit & check for more.
	les	si,[si+26]	;Get this disk's DPTE pointer.
	mov	bx,15		;Calculate DPTE checksum.
	xor	cx,cx
I_CkSm	add	cl,[es:bx+si]
	dec	bx
	jns	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
I_ErEDD	mov	dx,EBMsg	;Display "EDD error" and ignore unit!
	jmp	short I_ErrD
I_EDOK	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,[es:si+4]
	shr	bl,4
	mov	ax,[es:si]	;Get disk's IDE base address.
	cmp	ax,CDATA	;Is this a primary-channel disk?
	je	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust for secondary channel.
	inc	bx
	cmp	ax,(CDATA-080h)	;Is this a secondary-channel disk?
	jne	I_More		;No, ignore unit & check for more.
I_ChMS	mov	ax,bx		;Separate channel and master/slave.
	shr	al,1
	mov	ah,(LBABITS/32)	;Get drive-select "nibble".
	rcl	ah,5
	ror	al,1		;Set channel offset (2nd = 080h).
	mov	[IDEAd],al
	mov	dx,CDSEL	;Get IDE disk-select address.
	xor	dl,al
	mov	al,ah		;Select master or slave disk.
	out	dx,al
	shl	bx,2		;Save disk's "Units" table index.
	push	bx
	shr	bx,1		;Get "channel name" message index.
	mov	dx,[bx+HDNames] ;Display disk's IDE channel name.
	call	I_Dspl		;("Primary master", etc.).
	mov	ah,008h		;Get BIOS CHS values for this disk.
	mov	dl,[HDUnit]
	int	013h
	push	cs		;Reload our DS-register.
	pop	ds
	mov	al,dh		;Set head-number value in AL-reg.
	pop	bx		;Reload disk's unit number index.
	mov	dx,NPMsg	;Point to "parameter ERROR" message.
	jc	I_ErrD		;Error -- display msg. & ignore disk!
	and	cl,03Fh		;Get sectors per head (lower 6 bits).
	inc	al		;Get heads/cylinder (BIOS value + 1).
	mul	cl		;Calculate sectors per cylinder.
	or	ax,ax		;Is either CHS parameter zero?
	jz	I_ErrD		;Yes?  Display message & ignore disk!
	mov	[bx+Units+1-@],cl  ;Save this disk's CHS parameters.
	mov	[bx+Units+2-@],ax
	mov	al,[HDUnit]	;Activate this disk in main driver.
	mov	[bx+Units-@],al
	push	bx		;Validate this UltraDMA disk.
	call	I_ValD
	pop	bx
	jnc	I_More		;If no errors, check for more disks.
	mov	byte [bx+Units-@],0FFh  ;DELETE disk in main driver!
I_ErrD	call	I_Dspl		;Display error for this disk.
	mov	dx,CRMsg	;Display error-message suffix.
	call	I_Dspl
I_More	add	word [HDUnit],00101h  ;Bump BIOS unit & disk index.
	cmp	word [EDDFlag],08400h ;No EDD and all 4 units tested?
	je	I_AnyD		;Yes, see if we found any disks.
	dec	byte [Timer]	;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyD	mov	bx,16		;Set up to scan for last unit.
I_ChkU	sub	bx,byte 4	;Any more active units to check?
	js	near I_RScn	;No, see if we should do a re-scan.
	cmp	byte [bx+Units-@],0FFh  ;Is this unit active?
	je	I_ChkU		;No, loop back and check next unit.
	cld			;Clear driver local-stack memory.
	mov	cx,STACK+4	;(Helps debug if unused stack = 0!).
	mov	di,CStack-STACK
	push	cs
	pop	es
	xor	ax,ax
	rep	stosb
	mov	ax,03513h	;Get current Int 13h vector.
	call	I_In21
	push	es		;Save previous Int 13h vector.
	push	bx
	pop	dword [@PrvI13]
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,Entry
	call	I_In21
	popad			;Reload all 32-bit registers.
	push	ax		;Save all 16-bit registers.
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	xor	ax,ax		;Load & reset driver length.
	xchg	ax,[VDSLn]
	mov	dx,RPDON	;Get "init" packet success flag.
	jmp	short I_Exit	;Go post "init" packet results & exit.
I_VErr	mov	dx,VEMsg	;VDS "lock" error!  Point to message.
I_Err	push	dx		;Save error message pointer.
	shr	byte [VDSOf],1	;Was driver "locked" by VDS?
	jnc	I_XUnl		;No, get rid of our XMS memory.
	call	VDUnlX		;"Unlock" this driver from memory.
I_XUnl	mov	cx,[XMHdl]	;Did we reserve any XMS memory?
	jcxz	I_LdMP		;No, display desired error message.
	mov	ah,00Dh		;Unlock our XMS memory buffer.
	mov	dx,cx
	push	dx
	call	XMCall
	mov	ah,00Ah		;Free our XMS memory buffer.
	pop	dx
	call	XMCall
	mov	ah,006h		;Do local-disable of "A20 line".
	call	XMCall
I_LdMP	pop	dx		;Reload error message pointer.
I_EOut	call	I_Dspl		;Display desired error message.
	popad			;Reload all 32-bit registers.
	push	ax		;Save all 16-bit registers.
	push	bx
	push	cx
	push	dx
	push	si
	push	di
I_Quit	mov	dx,Suffix	;Display message suffix.
	call	I_Dspl
I_BadP	xor	ax,ax		;Get "null" length & error flags.
	mov	dx,RPDON+RPERR
I_Exit	les	bx,[LBA+1]	;Post results in "init" packet.
	mov	[es:bx+RPSize],ax
	mov	[es:bx+RPSize+2],cs
	mov	[es:bx+RPStat],dx
	pop	di		;Reload all CPU registers and exit.
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	es
	pop	ds
	popf
	retf
;
; Subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD	mov	al,0ECh		;Issue "Identify Device" command.
	call	I_Cmd
	jnc	I_PIO		;If no error, get "identify" data.
I_AErr	mov	dx,AEMsg	;Absent or non-ATA!   Point to msg.
	stc			;Set carry flag (error!) and exit.
	ret
I_PIO	add	dx,byte (CDATA-CCMD)  ;Point to PIO data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	xchg	ax,si		;Save "ATA/ATAPI" flag word.
	mov	cx,26		;Skip I.D. bytes 2-53.
I_Skp1	in	ax,dx
	loop	I_Skp1
	cld			;Ensure FORWARD "string" commands!
	push	cs		;Point to disk-name message.
	pop	es
	mov	di,DName
	mov	cl,26		;Read & swap disk name into message.
I_RdNm	in	ax,dx		;(I.D. bytes 54-93.  Bytes 94-105
	xchg	ah,al		;  are also read, but are ignored).
	stosw
	loop	I_RdNm
	in	ax,dx		;Read I.D. bytes 106 and 107.
	mov	bh,al		;Save "DMA valid" flag byte.
	mov	cl,35		;Skip I.D. bytes 108-175 and
I_Skp2	in	ax,dx		;  read I.D. bytes 176-177.
	loop	I_Skp2
	mov	bl,ah		;Save "UltraDMA selected" flag byte.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3	in	ax,dx
	loop	I_Skp3
	shl	si,1		;Is this an "ATA" hard-disk?
	jc	I_AErr		;No?  Exit & display message!
	test	bh,004h		;Are UltraDMA flag bits valid?
	jz	I_DErr		;No?  Exit & display message!
	mov	di,Modes	;Point to UltraDMA mode table.
	mov	al,'0'		;Initialize "current mode" value.
	mov	cl,002h		;Set rotating mode-check bit.
	cmp	bl,001h		;Will disk do UltraDMA mode 0?
	jae	I_NxtM		;Yes, find its best UltraDMA mode.
I_DErr	mov	dx,DEMsg	;Not an UltraDMA disk!   Point to msg.
I_SErr	stc			;Set carry flag (error!) and exit.
	ret
I_NxtM	cmp	al,[MaxMode]	;Are we at the desired MAXIMUM "mode"?
	jae	I_GotM		;Yes, use current mode.
	cmp	bl,cl		;Will disk do next UltraDMA mode?
	jb	I_GotM		;No, use current mode.
	inc	ax		;Set up for next UltraDMA mode.
	add	di,byte 4
	shl	cl,1		;More UltraDMA modes to check?
	jnz	I_NxtM		;Yes, loop back.
I_GotM	push	ax		;Save "current mode" value.
	mov	eax,[di]	;Set UltraDMA mode in set-mode message.
	mov	[DMode],eax
	inc	dx		;Set mode-select subcode.
	mov	al,SETM
	out	dx,al
	pop	ax		;Set desired UltraDMA mode.
	add	al,010h
	inc	dx
	out	dx,al
	mov	al,SETF		;Issue set-features command to disk.
	call	I_Cmd
	mov	dx,SEMsg	;Point to "Set-mode" error message.
	jc	I_SErr		;If error, set carry flag and exit.
	mov	dx,UNMsg	;Point to "unnamed" message.
	mov	di,DNEnd	;Point to end of disk name.
I_NxtN	cmp	di,DName	;Are we at the disk-name start?
	je	I_Unnm		;Yes, disk name is all spaces!
	cmp	byte [di-1],' '	;Is preceding byte a space?
	jne	I_TrmN		;No, terminate disk name message.
	dec	di		;Decrement disk name pointer.
	jmp	short I_NxtN	;Go see if next byte is a space.
I_TrmN	mov	byte [di],'$'	;Set message terminator after name.
	mov	dx,DNMsg	;Display disk "name" message.
I_Unnm	call	I_Dspl
	mov	dx,MSMsg	;Point to disk "mode" message.
I_Dspl	mov	ah,009h		;Get DOS "display string" command.
I_In21	int	021h		;Issue desired Int 021h request.
	push	cs		;Reload our DS-register.
	pop	ds
	clc			;Clear carry (no errors!) and exit.
	ret
;
; Subroutine to issue disk initialization commands.
;
I_Cmd	mov	dx,CCMD		;Issue desired init command.
	xor	dl,[IDEAd]
	out	dx,al
	xor	si,si		;Point to low-memory BIOS timer.
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,RDYTO	;Set I-O timeout limit in CL-reg.
	add	cl,[es:si]
I_CmdW	cmp	cl,[es:si]	;Has our command timed out?
	je	I_CmdE		;Yes, set CPU carry flag & exit.
	in	al,dx		;Get IDE controller status.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle	I_CmdW		;Yes, loop back and check again.
	test	al,ERR		;Did command cause any errors?
	jz	I_CmdX		;No, leave carry flag off & exit.
I_CmdE	stc			;Error!  Set CPU carry flag.
I_CmdX	ret			;Exit.
;
; Subroutine to convert hex digits to ASCII for output messages.
;   At entry the AX-reg. has the data, the CX-reg. has the digit
;   count, and the SI-reg. has the message pointer.
;
I_Hex4	mov	cx,4		;Set normal 4-digit count.
I_Hex	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe	I_HexA		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexA	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Set next ASCII digit in message.
	inc	si		;Bump message address.
	pop	ax		;Reload remaining digits.
	loop	I_Hex		;If more digits to go, loop back.
	ret			;Exit.
