/****************************************************************************

  findfile: Generic FindFirst and FindNext routines for various
            compilers.  Initially Borland's TC2.01, TC/C++ 1.01,
            TC/C++ 3.0, BC/C++ 3.1, BCC5.5.1; Microsoft's Visual C/C++ 5;
            Dave Dunfield's Micro-C 3.14,3.15,3.21;
            With plans to support DJGPP 2, POSIX compliant compilers,
            Microsoft C v6 (DOS compiler), generic Win32 (C api) compilers, 
            openWatcom, and BCC 4.x and 5.
            (Note: BCC 4.x and 5 should work as-is, but untested)

  Written by: Kenneth J. Davis
  Date:       January/February, 2001
  Updated:    
  Contact:    jeremyd@computer.org


Copyright (c): Public Domain [United States Definition]

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR AUTHORS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

****************************************************************************/

/**
 * Several macros or functions are defined to implement various file
 * searching functions with a consistant interface across compilers.
 *
 * One (and only one) of the source files should define the macro
 * FF_INCLUDE_FUNCTION_DEFS to ensure that when functions are required
 * for a given compiler instead of just macros, the function is (only)
 * included once for linking.  If it is not defined at least once
 * before including this file, then on some compilers the linker may
 * report undefined symbols.
 *
 *
 * getCurrentDrive(): returns current drive as integer where A=0,B=1,C=2,...
 * setCurrentDrive(drive): set current drive (input matches getCurrentDrive)
 * getCurrentDirectory(): returns char * with current directory,
 *         the pointer should be freed by calling free on it when done,
 *         a NULL is returned on error, current drive is assumed
 * getCurrentDirectoryEx(buffer, size): same as getCurrentDirectory(), except
 *         the buffer for the directory (and its size) is specified by
 *         the user, NULL is returned on error (such as size of user
 *         supplied buffer is too small), FF_MAXPATH can be used for max size
 * getCWD(drive): same as getCurrentDirectory, except
 *         the user specifies the drive (0=current, 1=A, 2=B,...)
 *         a NULL is returned on error otherwise the returned pointer
 *         should be freed when it is no longer needed
 *
 * FF_MAXPATH: defines the maximum size of complete path and filename
 *
 * getVolumeLabel(drive, label, maxsize): returns 0 if it was able to get 
 *         the volume label, and nonzero if an error occurred, such as the 
 *         drive is invalid.  The drive may be NULL to use current drive, 
 *         but if specified must refer to root directory, eg c:\ or e:\ 
 *         NOT c: or e:.  label is a char * to a user supplied buffer
 *         of maxsize length.  label should be non NULL and maxsize
 *         should be large enough to hold the volume label.
 *         If successful, but there is no volume label, then an empty
 *         string ("") is returned as the label.  If the actual label
 *         is longer than the buffer supplied, the label is truncated.
 *         This function should be used instead of directly using findfirst 
 *         variants.  It handles cases where LFN entries return false 
 *         values (part of Win9x long filename entries) and cases where 
 *         findfirst will not return entries with label attribute set.
 *
 * findFirst: performs initial search for file(s) or directory
 * findNext: continues a previous search, finds next file matching criteria
 * findClose: ends search, frees resources if necessary
 *
 * findFirst and findNext return -1 on error, 0 on success, with errno
 *         set to the operating system specific error code.
 *
 * FFDATA: a structure containing filefind information, use macros to access
 *
 * FF_GetFileName(ADDRESSOF(FFDATA ffdata)): returns '\0' terminated 
 *         char * to filename (a C string)
 *
 * FF_MAXDRIVE: defines the max size of drive component, usually 3, "?:\0"
 * FF_MAXFILEDIR: defines the maximum size of directory component
 * FF_MAXFILENAME: defines the maximum size of filename (without path)
 * FF_MAXFILEEXT: defines the maximum size of the filename extension
 *
 * MAXFILENAME: defines the maximum size of a filename and extension,
 *         this is usually either FF_MAXPATH or (FF_MAXFILENAME+FF_MAXFILEEXT)
 *
 * FF_GetAttributes(ADDRESSOF(FFDATA ffdata)): returns integer attributes 
 *         that can can be anded (&) with defines below to determine 
 *         file's attribute
 *
 * FF_GetFileSize(FFDATA * ffdata): defined for all but Micro-C/PC which
 *         does not support long int: returns a FSIZE_T (an unsigned long)
 *         with the size of the file.  The type defined by the compiler
 *         which usually ends up being either long or unsigned long
 *         is available as a _fsize_t
 *
 * The attributes of the matching filename returned
 * FF_A_LABEL: file specifies disk's label; findFirst/findNext may NOT
 *        return any files with volume label attribute set even if
 *        a file exists as label or a file is part of a LFN entry
 *        (any find based on Win32 FindFirstFile/FindNextFile under
 *        Windows NT 2000pro, at least, exhibits this problem);
 *        under DOS, may return multiple files if LFN entries exist.
 * FF_A_DIRECTORY: file is a directory (folder)
 * FF_A_ARCHIVE: file has archive bit set (modified since bit last cleared)
 * FF_A_READONLY: file is read only (can not be modified)
 * FF_A_HIDDEN: file is not normally visible to directory searches
 * FF_A_SYSTEM: file is a system file
 * FF_A_NORMAL: none of the other attributes are set, a normal file
 *
 * To work with Micro-C/PC which implements structures different from expected,
 * it seems to always treat struct variables as pointer to struct variables,
 * you should create a FFDATA instance, and use the ADDRESSOF macro when
 * passing the instance to any of the find routines.  If you do not
 * care about supporting Micro-C/PC and its many quirks then you may
 * use the address of operator (&) as normal, do not use the ADDRESSOF
 * macro when you are using pointers to a FFDATA instance.
 *
 * sample usage:
 *         #include <stdio.h>
 *         #include "findfile.h"
 *         int main(void)
 *         {
 *           FFDATA ffdata;
 *           if (findFirst("*.c", ADDRESSOF(ffdata)))
 *             printf("No files matching *.c in current directory.\n");
 *           else
 *           {
 *             printf("DIR ARC RDO HID SYS Name\n");
 *             do 
 *             {
 *               printf(" %c   %c   %c   %c   %c  %s\n",
 *                 (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_DIRECTORY) ? 'Y' : 'N',
 *                 (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_ARCHIVE) ? 'Y' : 'N',
 *                 (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_READONLY) ? 'Y' : 'N',
 *                 (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_HIDDEN) ? 'Y' : 'N',
 *                 (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_SYSTEM) ? 'Y' : 'N',
 *                 FF_GetFileName(ADDRESSOF(ffdata))
 *               );
 *             } while (findNext(ADDRESSOF(ffdata)) == 0);
 *             findClose(ADDRESSOF(ffdata));
 *           }
 *           return 0;
 *         }
 */

#ifndef FINDFILE_H
#define FINDFILE_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * MICRO-C/PC does not understand #if defined or #elif, so we are doing:
 * #ifdef MICROC
 *   ... Micro-C/PC specifics
 * #else
 *   ... all other compilers
 *   #if defined XXX
 *   #elif defined YYY
 *   #else 
 *     #error Unsupported configuration
 *   #endif 
 * #endif
 */
#ifdef MICROC  

/* #include <stdlib.h> Doesn't exist */
#include <stdio.h>
#include <file.h>

#define FF_MAXPATH PATH_SIZE     /* 65 under MC3.14, MC3.15, MC3.21 */
#define FF_MAXDRIVE 3            /* drive colon EOS, ?:\0, eg "C:"  */
#define FF_MAXFILEDIR PATH_SIZE  /* should be similar to max path   */
#define FF_MAXFILENAME 9         /* 8.3 convention, 8 + '\0' fname  */
#define FF_MAXFILEEXT 5          /* .3 + '\0' for extension         */

#define MAXFILENAME (FF_MAXFILENAME + FF_MAXFILEEXT)

#define getCurrentDrive() get_drive()
#define setCurrentDrive(drive) set_drive(drive)

#define getCurrentDirectory() getCurrentDirectoryEx(NULL, FF_MAXPATH)

#ifndef FF_INCLUDE_FUNCTION_DEFS
char * getCurrentDirectoryEx(char *buffer, int size);
#else
char * getCurrentDirectoryEx(char *buffer, int size)
{
  char currentDir[FF_MAXPATH];
  register int pathLenNeeded;

  /* get the current drive and prepend it */
  currentDir[0] = 'A' + getCurrentDrive();
  currentDir[1] = ':';
  currentDir[2] = '\\';
  currentDir[3] = '\0';

  /* get current directory, on error return NULL */
  if (getdir(currentDir+3))
    return NULL;  /* error getting cwd */

  /* determine size of path returned, +1 for '\0' */
  pathLenNeeded = strlen(currentDir) + 1;

  /* if user did not give us a buffer allocate one using malloc */
  if (buffer == NULL)
  {
    pathLenNeeded = (pathLenNeeded < size) ? size : pathLenNeeded;
    if ((buffer = (char *)malloc(pathLenNeeded * sizeof(char))) == NULL)
      return NULL;  /* error allocating memory */
  }
  else /* verify buffer given is large enough, if not return NULL */
  {
    if (pathLenNeeded > size) 
      return NULL;  /* error buffer to small */
  }
  
  /* copy contents over */
  strcpy(buffer, currentDir);

  return buffer;
}
#endif

#define FF_LACKS_GETCWD


/* Find File macros */
#define FF_A_LABEL     VOLUME
#define FF_A_DIRECTORY DIRECTORY
#define FF_A_ARCHIVE   ARCHIVE
#define FF_A_READONLY  READONLY
#define FF_A_HIDDEN    HIDDEN
#define FF_A_SYSTEM    SYSTEM
#define FF_A_NORMAL    0x00

#define FF_FINDFILE_ATTR (FF_A_DIRECTORY | FF_A_ARCHIVE | FF_A_READONLY | FF_A_HIDDEN | FF_A_SYSTEM | FF_A_NORMAL)

#define FFDATA struct FF_block

/* work around for Micro-C/PC thinking a struct FF_block is a pointer to a struck FF_BLOCK */
#define ADDRESSOF(x) (x)

/* define errno */
#ifndef FF_INCLUDE_FUNCTION_DEFS
extern int errno;
#else
int errno;
#endif

#define findFirst(filespec, ffdata, attr) (((errno=findfirst(filespec, ffdata, FF_FINDFILE_ATTR)) == 0) ? 0 : -1)
#define findFirstEx(filespec, ffdata, attr) (((errno=findfirst(filespec, ffdata, attr)) == 0) ? 0 : -1)
#define findNext(ffdata) (((errno=findnext(ffdata)) == 0) ? 0 : -1)
#define findClose(ffdata)

#define FF_GetFileName(ffdata) (ffdata)->FF_name
#define FF_GetAttributes(ffdata) (ffdata)->FF_attrib

#define FF_GETVOLLBL_WITH_FINDFIRST


#else /* All non Micro-C/PC compilers */

/* work around for Micro-C/PC thinking a struct FF_block is a pointer to a struck FF_BLOCK */
#define ADDRESSOF(x) &(x)


#if defined __BORLANDC__ || __TURBOC__

#include <stdlib.h>
#include <dir.h>
#include <errno.h>

#ifdef __MSDOS__
#include <dos.h>
#endif

#define FF_MAXPATH MAXPATH      /* 260 under BCC55, 80 under TC10&TC30 */
#define FF_MAXDRIVE MAXDRIVE    /*   3 under BCC55,  3 under TC10&TC30 */
#define FF_MAXFILEDIR MAXDIR    /* 256 under BCC55, 66 under TC10&TC30 */
#define FF_MAXFILENAME MAXFILE  /* 256 under BCC55,  9 under TC10&TC30 */
#define FF_MAXFILEEXT MAXEXT    /* 256 under BCC55,  5 under TC10&TC30 */

#if (FF_MAXFILENAME + FF_MAXFILEEXT) > FF_MAXPATH
#define MAXFILENAME FF_MAXPATH
#else
#define MAXFILENAME (FF_MAXFILENAME + FF_MAXFILEEXT)
#endif

#define getCurrentDrive() getdisk()
#define setCurrentDrive(drive) setdisk(drive)

#define getCurrentDirectory() getcwd(NULL, FF_MAXPATH)
#define getCurrentDirectoryEx(buffer, size) getcwd(buffer, size)
#if (__TURBOC__ < 0x300)
#ifndef FF_INCLUDE_FUNCTION_DEFS
char * getCWD(int drive);
#else
char * getCWD(int drive)
{
  char *cwd;

  /* Allocate memory for current working directory of max dir length */
  if ((cwd = (char *)malloc(FF_MAXPATH * sizeof(char))) == NULL)
    return NULL;

  /* get the current drive and prepend it */
  cwd[0] = 'A' + getCurrentDrive();
  cwd[1] = ':';
  cwd[2] = '\\';
  cwd[3] = '\0';

  /* Get the directory, cleaning up on error */
  if (getcurdir(drive, cwd+3) != 0)
  {
    free(cwd);
    return NULL;
  }
  return cwd;
}
#endif
#else /* Turbo C/C++ 3.0 or above */
#include <direct.h>
#define getCWD(drive) _getdcwd((drive), NULL, FF_MAXPATH)
#endif

#define FF_A_LABEL     FA_LABEL
#define FF_A_DIRECTORY FA_DIREC
#define FF_A_ARCHIVE   FA_ARCH
#define FF_A_READONLY  FA_RDONLY
#define FF_A_HIDDEN    FA_HIDDEN
#define FF_A_SYSTEM    FA_SYSTEM
#if (__TURBOC__ < 0x300) || (__BORLANDC__ < 0x300)
#define FF_A_NORMAL    0x00
#else
#define FF_A_NORMAL    FA_NORMAL
#endif

#define FF_FINDFILE_ATTR (FF_A_DIRECTORY | FF_A_ARCHIVE | FF_A_READONLY | FF_A_HIDDEN | FF_A_SYSTEM | FF_A_NORMAL)

/* Find File macros */
typedef struct ffblk FFDATA;

#define findFirst(filespec, ffdata) findFirstEx(filespec, ffdata, FF_FINDFILE_ATTR)
#define findFirstEx(filespec, ffdata, attr) findfirst(filespec, ffdata, attr)
#define findNext(ffdata) findnext(ffdata)
#define findClose(ffdata)

#define FF_GetFileName(ffdata)   (ffdata)->ff_name
#define FF_GetAttributes(ffdata) (ffdata)->ff_attrib

typedef long _fsize_t;
typedef unsigned long FSIZE_T;
#define FF_GetFileSize(ffdata) (FSIZE_T)((ffdata)->ff_fsize)

#ifdef _WIN32
#define FF_GETVOLLBL_WITH_WIN32GETVOLINFO
#else
#define FF_GETVOLLBL_WITH_FINDFIRST
#endif

#elif defined _MSC_VER || defined __MSC /* Below only tested with MS Visual C/C++ 5 */

#include <stdlib.h>
#define FF_MAXPATH _MAX_PATH      /* 260 under VC5 */
#define FF_MAXDRIVE _MAX_DRIVE    /*   3 under VC5 */
#define FF_MAXFILEDIR _MAX_DIR    /* 256 under VC5 */
#define FF_MAXFILENAME _MAX_FNAME /* 256 under VC5 */
#define FF_MAXFILEEXT _MAX_EXT    /* 256 under VC5 */

#if (FF_MAXFILENAME + FF_MAXFILEEXT) > FF_MAXPATH
#define MAXFILENAME FF_MAXPATH
#else
#define MAXFILENAME (FF_MAXFILENAME + FF_MAXFILEEXT)
#endif

#include <direct.h>
#define getCurrentDrive() (_getdrive() - 1)
#define setCurrentDrive(drive) _chdisk(drive+1)

#define getCurrentDirectory() _getcwd(NULL, FF_MAXPATH)
#define getCurrentDirectoryEx(buffer, size) _getcwd(buffer, size)
#define getCWD(drive) _getdcwd(drive, NULL, FF_MAXPATH)

/* Find File macros */
#include <io.h>

#ifndef _A_VOLID
#define FF_A_LABEL     0x08
#else
#define FF_A_LABEL     _A_VOLID
#endif
#define FF_A_DIRECTORY _A_SUBDIR
#define FF_A_ARCHIVE   _A_ARCH
#define FF_A_READONLY  _A_RDONLY
#define FF_A_HIDDEN    _A_HIDDEN
#define FF_A_SYSTEM    _A_SYSTEM
#define FF_A_NORMAL    _A_NORMAL

/* #define FF_FINDFILE_ATTR (FF_A_DIRECTORY | FF_A_ARCHIVE | FF_A_READONLY | FF_A_HIDDEN | FF_A_SYSTEM | FF_A_NORMAL) */

typedef struct FFDATA
{
  long hFile;
  struct _finddata_t ffd;
} FFDATA;

#define findFirst(filespec, ffdata) ((((ffdata)->hFile = _findfirst(filespec, &((ffdata)->ffd))) == -1L) ? -1 : 0)
/* #define findFirstEx(filespec, ffdata, attr) Unsupported */
#define findNext(ffdata)  ((_findnext((ffdata)->hFile, &((ffdata)->ffd)) != 0) ? -1 : 0)
#define findClose(ffdata) _findclose((ffdata)->hFile)

#define FF_GetFileName(ffdata) (ffdata)->ffd.name
#define FF_GetAttributes(ffdata) (ffdata)->ffd.attrib

/* typedef unsigned long _fsize_t; */
typedef unsigned long FSIZE_T;
#define FF_GetFileSize(ffdata) (FSIZE_T)((ffdata)->ffd.size)

#ifdef _WIN32
#define FF_GETVOLLBL_WITH_WIN32GETVOLINFO
#else
#define FF_GETVOLLBL_WITH_FINDFIRST
#endif

#elif defined WIN32  /* Uses the Win32 API for FindFirst/FindNext/FindClose/etc. */

#define FF_GETVOLLBL_WITH_WIN32GETVOLINFO
#error Win32 support has not been added yet.

#elif defined POSIX  /* Uses the POSIX defined OpenDir/ReadDir/CloseDir/etc. */

#error POSIX support has not been added yet.

#else
#error Platform (OS) or compiler is not supported.
#endif
#endif /* All non Micro-C/PC compilers */

/**
 * This section contains macros/functions common to all/most compilers.
 * This includes Micro-C/PC, so limited preprocessor support.
 */


#ifdef FF_LACKS_GETCWD
#ifndef FF_INCLUDE_FUNCTION_DEFS
char * getCWD(int drive);
#else
char * getCWD(int drive)
{
  char *cwd;
  int curdrive;

  curdrive = getCurrentDrive(); /* store current drive */

  if (drive == 0)               /* correct requested drive */
    drive = curdrive;           /* use current drive if 0 given */
  else
  {
    drive--;                    /* convert from 1=A to 0=A,... */
  } 

  setCurrentDrive(drive);       /* change to drive to get cwd from */

/* setting current drive does not return error status so we must        *
 * manually check if drive changed, but only if not using current drive */
  if (drive != curdrive)
    if (getCurrentDrive() != drive) return NULL; /* drive change failed */

  cwd = getCurrentDirectory();  /* Get the directory */
  setCurrentDrive(curdrive);    /* restore initial drive */

  return cwd;                   /* returns malloc'd path or NULL */
}
#endif
#endif

#ifdef FF_GETVOLLBL_WITH_FINDFIRST
/* Use FindFirst/FindNext to get file with volume label attribute set. */

#ifndef MICROC
#define CONST const
#else
#if MICROC >= 321
#define CONST const
#else
#define CONST
#endif
#endif

#ifndef FF_INCLUDE_FUNCTION_DEFS
int getVolumeLabel(CONST char *drive, char *label, int bufferSize);
#else

#ifndef MICROC
#include <string.h>
#include <ctype.h>
#endif

int getVolumeLabel(CONST char *drive, char *label, int bufferSize)
{
  char volSearch[FF_MAXDRIVE+4]; /* hold drive (c:), root dir (\), and *.* eg c:\*.* */
  FFDATA ffdata;
  register char *tcharptr;

  if ((label == NULL) || (bufferSize < 1)) return -1;

  if (drive == NULL)
    strcpy(volSearch, "\\*.*");
  else
  {
    /* validate drive if given (not given means use current so is valid) */
    if (drive[1] == ':')
    {
      /* simple test, if we can get the cwd of drive, assume valid. */
      tcharptr = getCWD(toupper(drive[0]) - 'A' + 1);
      if (tcharptr == NULL)
        return -1;    /* invalid drive or at least unable to get cwd */
      free(tcharptr); /* we don't actually need this, so free it */
    }

    /* validated, so now perform search */
    strcpy(volSearch, drive);
    strcat(volSearch, "*.*");
  }

  /* Initially clear label */
  *label = '\0';  

  /* loop until no more files or other error */
  if (findFirstEx(volSearch, ADDRESSOF(ffdata), FF_A_LABEL) == 0)
  {
    do
    {
      /* if found entry, and entry has only label bit set then valid label (MS LFN docs), but also accept archive bit */
      if ((FF_GetAttributes(ADDRESSOF(ffdata)) & ~FF_A_ARCHIVE) == FF_A_LABEL)
      {
        /* Remove the . since volumes don't use it */
        if (strlen(FF_GetFileName(ADDRESSOF(ffdata))) > 8)
        {
          tcharptr = FF_GetFileName(ADDRESSOF(ffdata)) + 8;  /* point to . */
          if (*tcharptr == '.')
          {
            *tcharptr = *(tcharptr + 1);  tcharptr++;
            *tcharptr = *(tcharptr + 1);  tcharptr++;
            *tcharptr = *(tcharptr + 1);  tcharptr++;
            *tcharptr = '\0';
          }
        }
        
        strncpy(label, FF_GetFileName(ADDRESSOF(ffdata)), bufferSize);
        label[bufferSize-1] = '\0';  /* Ensure string terminated! */
        return 0; /* Found and set label :) */
      }
    } while (findNext(ADDRESSOF(ffdata)) == 0);
    findClose(ADDRESSOF(ffdata));    
  }

  /* 0x12 == No more files, may also return 0x02 == File not found */
  if ((errno == 0x12) || (errno == 0x02))  /* No label, but valid drive. */
    return 0;
  else
    return -1; /* Invalid drive */
}
#endif /* FF_INCLUDE_FUNCTION_DEFS    */ 
#endif /* FF_GETVOLLBL_WITH_FINDFIRST */


#ifdef FF_GETVOLLBL_WITH_WIN32GETVOLINFO
/* Use Win32 API's GetVolumeInformation to get volume label. */

#define WIN32_LEAN_AND_MEAN
#include <windows.h>  /* necessary since winbase.h needs, but may not directly include */
#include <winbase.h>  /* GetVolumeInformation */

#define getVolumeLabel(drive, label, bufferSize) ((GetVolumeInformation(drive ,label, bufferSize, NULL, NULL, NULL, NULL, 0) == 0) ? -1 : 0)

#endif /* FF_GETVOLLBL_WITH_WIN32GETVOLINFO */


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* FINDFILE_H */


/**
 * The rest of this file is for demonstration and testing purposes.
 * To test/demonstrate, create a c source file and simply include
 * the three lines:
 *        #define FF_TEST
 *        #define FF_INCLUDE_FUNCTION_DEFS
 *        #include "findfile.h"
 * then compile and run.
 */
#ifdef FF_TEST

#ifndef MICROC
#include <stdio.h>  /* Do not include FindFile.h and stdio.h when using Micro-C/PC */
#endif 
/* 
#define FF_INCLUDE_FUNCTION_DEFS
#include "findfile.h" 
*/

int main(void)
{
  FFDATA ffdata;
  /* char *curDir; */
  char curDir[FF_MAXPATH];

  printf("Current drive is %c:\n", 'A' + getCurrentDrive());

  if (getCurrentDirectoryEx(curDir, FF_MAXPATH) == NULL)
  /* if ((curDir = getCurrentDirectory()) == NULL) */
  {
    printf("Error getting current directory!\n");
    return 1;
  }

  printf("Looking in %s for files matching *.c\n\n", curDir);

  if (findFirst("*.c", ADDRESSOF(ffdata)))
    printf("No files matching *.c found.\n");
  else
  {
    printf("DIR ARC RDO HID SYS  Name\n");
    do 
    {
      printf(" %c   %c   %c   %c   %c  %s\n",
        (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_DIRECTORY) ? 'Y' : 'N',
        (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_ARCHIVE) ? 'Y' : 'N',
        (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_READONLY) ? 'Y' : 'N',
        (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_HIDDEN) ? 'Y' : 'N',
        (FF_GetAttributes(ADDRESSOF(ffdata)) & FF_A_SYSTEM) ? 'Y' : 'N',
        FF_GetFileName(ADDRESSOF(ffdata))
      );
    } while (findNext(ADDRESSOF(ffdata)) == 0);
    findClose(ADDRESSOF(ffdata));
  }

  /* if (curDir != NULL) free(curDir); */

  return 0;
}

#endif /* FF_TEST */
