/***************************************************************************/
/* REPLACE                                                                 */
/* A replacement for the MS-DOS(tm) REPLACE-command.                       */
/* Compilable with Borland Turbo C v1.0 or higher.                         */
/*-------------------------------------------------------------------------*/
/* history:                                                                */
/* v1.0 (2001-03-01) - initial release                                     */
/*-------------------------------------------------------------------------*/
/* (C)opyright 2001 by Rene Ableidinger (rene.ableidinger@gmx.at)          */
/* This program is protected under the GNU General Public License v2.x.    */
/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>

/*-------------------------------------------------------------------------*/
/* GLOBAL TYPES/VARIABLES                                                  */
/*-------------------------------------------------------------------------*/
typedef struct s_file {
          struct s_file *next;
          char name[MAXFILE + MAXEXT];
        };
struct s_file *src_files = NULL;
char opt_add = 0,
     opt_prompt = 0,
     opt_readonly = 0,
     opt_subdir = 0,
     opt_update = 0,
     opt_wait = 0,
     dest_drive;
int file_counter = 0;

/*-------------------------------------------------------------------------*/
/* PROTOTYPES                                                              */
/*-------------------------------------------------------------------------*/
void classify_args(char argc, char *argv[],
                   char *fileargc, char *fileargv[],
                   char *optargc, char *optargv[]);
void help(void);
void exit_fn(void);
void get_src_files(char src_pathname[], char src_filemask[]);
void replace_files(char src_pathname[], char dest_pathname[]);
void replace_file(char src_filename[], char dest_filename[]);

/*-------------------------------------------------------------------------*/
/* MAIN-PROGRAM                                                            */
/*-------------------------------------------------------------------------*/
int main(char argc, char *argv[]) {
  char fileargc,
       *fileargv[255],
       optargc,
       *optargv[255],
       cur_pathname[MAXPATH] = "",
       src_pathname[MAXPATH] = "",
       src_filemask[MAXPATH] = "",
       dest_pathname[MAXPATH] = "",
       *ptr,
       option[255] = "",
       i;


  classify_args(argc, argv, &fileargc, fileargv, &optargc, optargv);

  if (fileargc < 1 || optargv[0] == "?") {
    help();
    exit(-1);
  }

  /* activate termination function */
  /* (writes no. of replaced/added files at exit) */
  atexit(exit_fn);

  /* store current path */
  getcwd(cur_pathname, MAXPATH);

  /* get source-pathname (with trailing backspace) and filename/-mask */
  strcpy(src_pathname, fileargv[0]);
  ptr = strrchr(src_pathname, '\\');
  if (ptr == NULL) {
    /* no source-path specified -> use current */
    strcpy(src_pathname, cur_pathname);
    strcpy(src_filemask, fileargv[0]);
  }
  else {
    /* source-path specified -> check it */
    strcpy(src_filemask, ptr + 1);
    *ptr = '\0';
    if (chdir(src_pathname) != 0) {
      printf("Source-path not found - %s\n", src_pathname);
      exit(-1);
    }
    getcwd(src_pathname, MAXPATH);
  }
  strcat(src_pathname, "\\");
  /* restore path */
  chdir(cur_pathname);

  /* get destination-pathname (with trailing backspace) */
  if (fileargc < 2) {
    /* no destination-path specified -> use current */
    strcpy(dest_pathname, cur_pathname);
  }
  else {
    /* destination-path specified -> check it */
    strcpy(dest_pathname, fileargv[1]);
    if (strlen(dest_pathname) > 2) {
      ptr = &dest_pathname[strlen(dest_pathname) - 1];
      if (*ptr == '\\') {
        /* remove trailing backspace */
        *ptr = '\0';
      }
    }
    if (chdir(dest_pathname) != 0) {
      printf("Destination-path not found - %s\n", dest_pathname);
      exit(-1);
    }
    getcwd(dest_pathname, MAXPATH);
  }
  strcat(dest_pathname, "\\");
  /* restore path */
  chdir(cur_pathname);

  /* check if source- and destination-path are equal */
  if (strcmp(src_pathname, dest_pathname) == 0) {
    printf("Source- and destination-path must not be the same\n");
    exit(-1);
  }

  /* get destination-drive (1 = A, 2 = B, 3 = C, ...) */
  dest_drive = toupper(dest_pathname[0]) - 64;

  /* get options */
  for (i = 0; i < optargc; i++) {
    strcpy(option, optargv[i]);
    strupr(option);
    if (strcmp(option, "A") == 0)
      opt_add = -1;
    else if (strcmp(option, "P") == 0)
      opt_prompt = -1;
    else if (strcmp(option, "R") == 0)
      opt_readonly = -1;
    else if (strcmp(option, "S") == 0)
      opt_subdir = -1;
    else if (strcmp(option, "U") == 0)
      opt_update = -1;
    else if (strcmp(option, "W") == 0)
      opt_wait = -1;
    else {
      printf("Invalid option - %s\n", optargv[i]);
      exit(-1);
    }
  }
  if ((opt_add && opt_subdir) ||
      (opt_add && opt_update)) {
    printf("Invalid combination of parameters\n");
    exit(-1);
  }

  if (opt_wait) {
    printf("Press any key to continue...\n");
    getch();
    fflush(stdin);
  }

  get_src_files(src_pathname, src_filemask);
  replace_files(src_pathname, dest_pathname);

  exit(0);

  /* This line is never reached and exists only to remove the */
  /* compiler-warning! */
  return 0;
}

/*-------------------------------------------------------------------------*/
/* SUB-PROGRAMS                                                            */
/*-------------------------------------------------------------------------*/
void classify_args(char argc, char *argv[],
                   char *fileargc, char *fileargv[],
                   char *optargc, char *optargv[]) {
/*-------------------------------------------------------------------------*/
/* Sorts the program-parameters into file- and option-parameters.          */
/*-------------------------------------------------------------------------*/
  char i,
       *argptr;


  *fileargc = 0;
  *optargc = 0;
  for (i = 1; i < argc; i++) {
    argptr = argv[i];
    if (argptr[0] == '/' || argptr[0] == '-') {
      /* first character of parameter is '/' or '-' -> option-parameter */
      optargv[*optargc] = argptr + 1;
      *optargc = *optargc + 1;
    }
    else {
      /* file-parameter */
      fileargv[*fileargc] = argptr;
      *fileargc = *fileargc + 1;
    }
  }
}

void help(void) {
  printf("Replaces files.\n"
         "\n"
         "REPLACE [drive1:][path1]filename [drive2:][path2] [/A] [/P] [/R] [/W]\n"
         "REPLACE [drive1:][path1]filename [drive2:][path2] [/P] [/R] [/S] [/U] [/W]\n"
         "\n"
         "  [drive1:][path1]filename    Specifies the source-file(s)\n"
         "  [drive2:][path2]            Specifies the directory where files are to be\n"
	 "                              replaced.\n"
         "  /A        Adds new files to destination directory. Can not be used together\n"
         "            with options /A or /U.\n"
         "  /P        Prompts for confirmation.\n"
         "  /R        Replaces read-only and unprotected files.\n"
         "  /S        Replaces also files in sub-directories of destination directory.\n"
         "            Can not be used together with option /A.\n"
         "  /W        Waits for you to insert a disk before copying.\n"
         "  /U        Replaces (updates) only those files on the destination directory\n"
         "            that are older than those in the source directory. Can not be used\n"
         "            together with option /A.\n");
}

void exit_fn(void) {
/*-------------------------------------------------------------------------*/
/* Gets called by the "exit"-command and is used to free the memory for    */
/* the pointer-chain "src_files" and to write a status-message.            */
/*-------------------------------------------------------------------------*/
  struct s_file *file,
                *file_prev;


  /* remove pointer-chain from memory */
  file = src_files;
  while (file != NULL) {
    file_prev = file;
    file = file->next;
    free(file_prev);
  }

  if (opt_add)
    printf("%d file(s) added\n", file_counter);
  else
    printf("%d file(s) replaced\n", file_counter);
}

void get_src_files(char src_pathname[], char src_filemask[]) {
/*-------------------------------------------------------------------------*/
/* Reads the names of all files in the specified source-directory which    */
/* match the specified filemask (wildcards possible) into the global       */
/* pointer-chain "src_files".                                              */
/*-------------------------------------------------------------------------*/
  char filemask[MAXPATH];
  struct s_file *file,
                *file_prev;
  struct ffblk fileblock;
  int done;


  src_files = NULL;
  strcpy(filemask, src_pathname);
  strcat(filemask, src_filemask);
  done = findfirst(filemask , &fileblock, 0);
  while (! done) {
    /* allocate memory for filename in pointer-chain */
    file = (struct s_file *) malloc(sizeof(struct s_file));
    if (file == NULL) {
      printf("Not enough memory\n");
      exit(1);
    }

    strcpy(file->name, fileblock.ff_name);

    if (src_files == NULL)
      /* first entry in pointer-chain */
      src_files = file;
    else
      /* n-th entry in pointer-chain -> chain with previous entry */
      file_prev->next = file;
    file_prev = file;

    done = findnext(&fileblock);
  }
  file->next = NULL;
}

void replace_files(char src_pathname[], char dest_pathname[]) {
/*-------------------------------------------------------------------------*/
/* Copies all files specified in the pointer-chain "src_files" from the    */
/* source- into the destination-directory (and its subdirectories)         */
/* depending on the specified option-parameters.                           */
/*-------------------------------------------------------------------------*/
  char filemask[MAXPATH],
       new_dest_pathname[MAXPATH],
       src_filename[MAXPATH],
       dest_filename[MAXPATH];
  struct s_file *file,
                *file_prev;
  struct ffblk fileblock;
  int done;


  file = src_files;
  while (file != NULL) {
    /* rebuild source-filename */
    strcpy(src_filename, src_pathname);
    strcat(src_filename, file->name);

    /* build destination-filename */
    strcpy(dest_filename, dest_pathname);
    strcat(dest_filename, file->name);

    replace_file(src_filename, dest_filename);

    file = file->next;
  }

  if (opt_subdir) {
    /* replace files in sub-directories too */
    strcpy(filemask, dest_pathname);
    strcat(filemask, "*.*");
    done = findfirst(filemask, &fileblock, FA_DIREC);
    while (! done) {
      if (fileblock.ff_attrib == FA_DIREC &&
          strcmp(fileblock.ff_name, ".") != 0 &&
          strcmp(fileblock.ff_name, "..") != 0) {
        /* build destination-pathname */
        strcpy(new_dest_pathname, dest_pathname);
        strcat(new_dest_pathname, fileblock.ff_name);
        strcat(new_dest_pathname, "\\");

        replace_files(src_pathname, new_dest_pathname);
      }

      done = findnext(&fileblock);
    }
  }
}

void replace_file(char src_filename[], char dest_filename[]) {
/*-------------------------------------------------------------------------*/
/* Copies the source- into the destination-file including file-attributes  */
/* and timestamp depending on the specified option-parameters.             */
/*-------------------------------------------------------------------------*/
  FILE *src_file,
       *dest_file;
  char buffer[BUFSIZ],
       ch,
       msg_prompt[255];
  int buffersize,
      dest_file_exists,
      fileattrib;
  struct stat src_statbuf,
              dest_statbuf;
  struct ftime filetime;
  struct dfree disktable;
  unsigned long free_diskspace;


  /* check source-file for read-permission */
  /* (only usefull under an OS with the ability to deny read-access) */
  if (access(src_filename, 4) != 0) {
    printf("Read-access denied - %s\n", src_filename);
    exit(-1);
  }

  /* get info of source- and destination-file */
  stat(src_filename, &src_statbuf);
  dest_file_exists = ! stat(dest_filename, &dest_statbuf);

  /* get amount of free disk-space in destination-drive */
  getdfree(dest_drive, &disktable);
  free_diskspace = (unsigned long) disktable.df_avail *
                   disktable.df_sclus * disktable.df_bsec;

  if (opt_add) {
    /* file can be added only if it doesn't exist */
    if (dest_file_exists) {
      return;
    }

    /* check free space on destination-disk */
    if (src_statbuf.st_size > free_diskspace) {
      printf("Insufficient disk-space in destination-path - %s\n", dest_filename);
      exit(-1);
    }
  }
  else {
    /* file can be replaced only if it already exists */
    if (! dest_file_exists) {
      return;
    }

    if (opt_update) {
      /* check if source-file is older than destination-file */
      if (src_statbuf.st_mtime <= dest_statbuf.st_mtime) {
        return;
      }
    }

    /* check free space on destination-disk */
    if (src_statbuf.st_size > free_diskspace - dest_statbuf.st_size) {
      printf("Insufficient disk-space in destination-path - %s\n", dest_filename);
      exit(-1);
    }

    /* check destination-file for write-permission */
    if (access(dest_filename, 2) != 0) {
      if (! opt_readonly) {
        printf("Write-access denied - %s\n", dest_filename);
        exit(-1);
      }

      /* remove readonly-attribute */
      fileattrib = _chmod(dest_filename, 0);
      fileattrib = fileattrib ^ FA_RDONLY;
      _chmod(dest_filename, 1, fileattrib);
    }
  }

  if (opt_prompt) {
    /* ask for confirmation */
    if (opt_add) {
      strcpy(msg_prompt, "Add %s (Y/N) ");
    }
    else {
      strcpy(msg_prompt, "Replace %s (Y/N) ");
    }

    do {
      printf(msg_prompt, dest_filename);
      scanf("%c", &ch);
      fflush(stdin);
      ch = toupper(ch);
    } while (ch != 'Y' && ch != 'N');
    if (ch != 'Y') {
      return;
    }
  }

  /* open source-file */
  src_file = fopen(src_filename, "rb");
  if (src_file == NULL) {
    printf("Can't read source-file - %s\n", src_filename);
    exit(-1);
  }

  /* open destination-file */
  dest_file = fopen(dest_filename, "wb");
  if (dest_file == NULL) {
    printf("Can't overwrite destination-file - %s\n", dest_filename);
    fclose(src_file);
    exit(-1);
  }

  if (opt_add) {
    printf("Adding %s\n", dest_filename);
  }
  else {
    printf("Replacing %s\n", dest_filename);
  }

  /* copy file-data */
  buffersize = fread(buffer, sizeof(char), BUFSIZ, src_file);
  while (buffersize > 0) {
    if (fwrite(buffer, sizeof(char), buffersize, dest_file) != buffersize) {
      printf("Write error on destination-file - %s\n", dest_filename);
      fclose(src_file);
      fclose(dest_file);
      exit(-1);
    }
    buffersize = fread(buffer, sizeof(char), BUFSIZ, src_file);
  }

  /* copy file-timestamp */
  getftime(fileno(src_file), &filetime);
  setftime(fileno(dest_file), &filetime);

  /* close files */
  fclose(src_file);
  fclose(dest_file);

  /* copy file-attributes */
  fileattrib = _chmod(src_filename, 0);
  _chmod(dest_filename, 1, fileattrib);

  file_counter = file_counter + 1;
}
