/* Partial implementation of the DOS MODE command by Eric Auer 2003 ..., */
/* for inclusion in MODECON (by Aitor Merino), which does the codepages! */
/* This code is primarily for Aitor, but you may use it as is under the  */
/* terms of the GNU GPL (v2) license if you want to, see www.gnu.org ... */
/* If you have questions, mail me: eric%coli.uni-sb.de (replace % by @). */

/* This file: drivers for several aspects of CON */
/* We just call modecp functions for codepage things */

#include "mode.h"

#define VIDMAXROW peekb(0x40,0x84)

static union REGS r;

/* ------------------------------------------------ */

/* set number of screen lines */
int set_lines(int lines)
{
  int havevga = 0;
  int oldlines = VIDMAXROW+1; /* only valid for EGa or better */

  r.h.ah = 0x12; /* ah=0x12, bl=0x10: get EGA info */
  r.x.bx = 0xff10; /* init bh to -1 */
  r.x.cx = 0xffff; /* init cx to -1 */
  int86(0x10, &r, &r);
  if ( (r.x.cx == 0xffff) || (r.h.bh == 0xff) ) {
    if (lines == 25) {
#ifdef DMODE
      printf("Not EGA or better, assuming 25 lines.\r\n");
#endif
      return 0; /* 25 lines is tautologic for less-than-EGA */
    }
    printf("Only EGA or better allows more than 25 lines!\r\n");
    return 1;
  }

  r.x.ax = 0x1a00; /* get VGA display combination - VGA install check */
  int86(0x10, &r, &r);
  /* returned active / alternate display in BL / BH (use al=1 to SET...) */
  /* 0 none, 1 mono, 2 CGA, 4..5 EGA 6 PGA 7..8 VGA a..c MCGA */
  if (r.h.al == 0x1a)
    havevga = 1; /* VGA found */

  if ( (lines == 50) && (havevga == 0) ) { /* 50 lines requires VGA */
    printf("Only VGA or better allows more than 43 lines!\r\n");
    return 1;
  }

  if (oldlines == lines) {
#ifdef DMODE
    printf("Number of lines already as requested.\r\n");
#endif
    return 0; /* no action needed */
  }

  /* only reached for EGA, VGA or better */
  switch (lines) {
    case 25:
#if 0
      r.h.ah = 0x0f; /* get current video mode */
      int86(0x10, &r, &r);
      r.h.ah = 0;
      /* AL: simply re-activate current mode */
      if (r.h.al > 3)
        r.h.al = 3; /* better use only known modes like 80x25 color */
      int86(0x10, &r, &r);
#endif
      r.x.ax = (havevga==1) ? 0x1114 : 0x1111;
      /* activate 8x16 / 8x14 VGA / EGA (or MONO) default font */
      /* use AL 1x for full mode reset, 0x for font load only */
      r.h.bl = 0; /* font bank 0 */
      int86(0x10, &r, &r);
      break;
    case 43:
    case 50:
      if (havevga==1) { /* we need extra 43 <-> 50 line switching in VGA */
        r.h.ah = 0x12; /* set resolution (with BL 0x30) */
        r.h.bl = 0x30;
        r.h.al = (lines==43) ? 1 : 2; /* 1 means 350 pixels rows, 2 400 */
        int86(0x10, &r, &r);
        if (r.h.al==0x12) { /* did the call succeed? */
          r.h.ah = 0x0f; /* get current video mode */
          int86(0x10, &r, &r);
          r.h.ah = 0; /* set mode again, to let resolution change take effect! */
          r.h.al |= 0x80; /* mode |= flag "do not clear screen" */
          int86(0x10, &r, &r);
#ifdef DMODE
          printf("Selecting vertical resolution %d on VGA.\r\n",
            (lines==43) ? 350 : 400);
#endif
        } else {
          printf("Could not select ???x%d pixel resolution, using default.\r\n",
            (lines==43) ? 350 : 400);
        }
      }
      r.x.ax = 0x1112; /* activate 8x8 default font */
      /* use AL 1x for full mode reset, 0x for font load only */
      r.h.bl = 0; /* font bank 0 */
      int86(0x10, &r, &r);
  } /* switch */

  oldlines = lines; /* desired value */
  lines = VIDMAXROW+1; /* value from BIOS */
  if (lines < 25)
    lines = 25;
  if (lines!=oldlines) {
    printf("Could only set %d rows, not %d.\r\n", lines, oldlines);
  }

  return 0; /* success */
} /* set_lines */

/* ------------------------------------------------ */

/* sub-parser for CODEPAGE commands: REFRESH, SELECT=n,  */
/* PREPARE=((...) file) and (void) which displays status */
/* "..." can be 1 or more "," separated numbers, empty   */
/* numbers are possible: ((123,,456) foo.txt)            */
int modeconcp(char * what)
{
  unsigned int cp = 0xffff;

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
     return ShowStatus();
  }

  if ( !strncmp(what, "REFRESH", 7) ) {
     cp = GetCodePage();
     if ((cp == 0xffff) || (cp == 0)) {
       printf("No codepage active - refresh impossible");
       return -1;
     }
     printf("Refreshing codepage number %d\n", cp);
     return CodePageSelect(cp);
  }

  cp = grabarg(what, "SELECT=");
  if (!cp) cp = grabarg(what, "SEL="); /* SEL, not SP */
  if (!cp) cp = grabarg(what, "SELECT");
  if (!cp) cp = grabarg(what, "SEL");
  if (cp) {
#ifdef DMODE
    printf("Calling CodePageSelect(%d);\n", cp);
#endif    
    return CodePageSelect(cp);
  }

  if (  (!strncmp(what, "PREPARE=", 8))
     || (!strncmp(what, "PREP=",    5)) 
     || (!strncmp(what, "PREPARE",  7))
     || (!strncmp(what, "PREP",     4)) ) {
    char * numbers = strchr(what, '(') + 1;
    char * fname;
    char * eol;
    if (numbers == NULL) {
      printf("Syntax error: Opening '(' missing!\n");
      return -1;
    }
    numbers = skipspace(numbers);
    fname = skipspace(strchr(numbers,')')+1);
    if (fname == NULL) {
      printf("Syntax error: Closing ')' of (numbers,...) missing!\n");
      return -1;
    }
    eol = strchr(numbers,')') + 1;
    if (eol[0]!=' ') {
      printf("Syntax error: space expected before filename.\n");
      return -1;
    }
    eol[0] = '\0'; /* split string after first ')' */
    eol = strrchr(fname, ')');
    if (eol == NULL) {
      printf("Syntax error: Closing ')' missing after filename!\n");
      return -1;
    }
    eol[0] = '\0'; /* remove trailing ')' */
    /* assumes no spaces between filename and trailing ')' */
    if ( (strchr(fname, ' ') != NULL) || (strchr(fname, '\t') != NULL) ) {
      printf("Syntax error: No spaces allowed in filename!\n");
    }
#ifdef DMODE
    printf("Calling CodepagePrepare('%s', '%s');\n", numbers, fname);
#endif    
    return CodePagePrepare(numbers, fname);
  }

  printf("Unexpected CODEPAGE command - syntax error?");
  return -1;

} /* modeconcp */

/* ------------------------------------------------ */

/* handle console related commands */
/* show status if no argument or argument "/STA..." */
/* chains to CODEPAGE code if first argument "CP" or "CODEPAGE" */
/* otherwise, keyboard repeat OR screen size can be set: */
/* style 1: cols=... lines=... (40/80 and 25/43/50) */
/* style 2: rate=... delay=... (1..32 and 1..4) */
int console(char * what)
{
  unsigned int cols, lines, rate, delay, i;

  if ( (what == NULL) || (!strncmp(what, "/STA", 4)) ) {
    unsigned int j;
    printf("*** CONSOLE STATUS ***\r\n");
    r.x.ax = 0x0306; /* get keyboard repeat settings */
    r.x.bx = 0xffff; /* to notice if function not possible (common!) */
    int86(0x16, &r, &r);
    if (r.x.bx == 0xffff) {
      printf("Keyboard repeat rate and delay could not be read.\r\n");
    } else {
      i = r.h.bh;
      j = r.h.bl;
      printf("Keyboard repeat after %u msec, rate %u (0=2/s...32=30/s).\r\n",
        (i+1)*250, 32-j);
    }
    lines = VIDMAXROW+1;
    if (lines < 25) /* CGA/Mono does not set this value */
      lines = 25;
    r.h.ah = 0x0f; /* get current video mode */
    int86(0x10, &r, &r);
    cols = r.h.ah;
    i = r.h.al & 0x7f; /* current video mode, without "no clear" flag. */
    printf("Screen size: %d columns, %d rows, mode %d.\r\n",
      cols, lines, i);
    return 0;
  }

  if ( !strncmp(what, "CP", 2) ) {
#ifdef DMODE
    printf("MODE CON CODEPAGE '%s'\r\n", skipspace(what+2));
#endif
    return modeconcp(skipspace(what+2));
  }

  if ( !strncmp(what, "CODEPAGE", 8) ) {
#ifdef DMODE
    printf("MODE CON CODEPAGE '%s'\r\n", skipspace(what+8));
#endif
    return modeconcp(skipspace(what+8));
  }

  cols  = grabarg(what, "COLS=");
  lines = grabarg(what, "LINES=");
  rate  = grabarg(what, "RATE=");
  delay = grabarg(what, "DELAY=");

  if ( (cols!=0) && (cols!=40) && (cols!=80) ) {
    printf("Columns must be 40 or 80.\r\n");
    return 1; /* failed */
  }

  if ( (lines!=0) && (lines!=25) && (lines!=43) && (lines!=50) ) {
    printf("Lines must be 25, 43 or 50.\r\n");
    return 1; /* failed */
  }

  if ( (rate<0) || (rate>32) ) {
    printf("Repeat rate code must be between 1 and 32.\r\n");
    return 1; /* failed */
  }

  if ( (delay<0) || (delay>4) ) {
    printf("Repeat delay must be between 1 and 4 (unit is 0.25 seconds).\r\n");
    return 1; /* failed */
  }

  if ( (delay==0) && (rate==0) && (cols==0) && (lines==0) ) {
    printf("Syntax error in 'MODE CON ...', please check 'MODE /?'.\r\n");
    return 1;
  }

#ifdef DMODE
  printf("MODE CON cols=%d lines=%d rate=%d delay=%d\r\n",
    cols, lines, rate, delay);
#endif

  if ( (rate!=0) || (delay!=0) ) {
    if (rate>0) {
      rate = 32-rate; /* turn 1..32 into 31..0 */
    } else {
      rate = 12; /* default: 20 chars per second */
    }
    if (delay==0)
      delay = 2; /* default: 2/4 seconds */
    r.x.ax = 0x0305; /* set keyboard repeat */
    r.h.bh = delay-1; /* unit 250ms, value 0 means 250ms */
    r.h.bl = rate; /* 0 is 30/sec, default 12 is 10/sec, 31 is 2/sec */
    int86(0x16, &r, &r);
  }

  if ( (cols!=0) || (lines!=0) ) {
    r.h.ah = 0x0f; /* get video mode */
    int86(0x10, &r, &r);
    if (cols != r.h.ah) { /* current column count not as desired? */
      i = r.h.al & 0x7f;  /* current video mode */
      r.h.al &= 0x80;
      if ( (i < 2) && (cols==80) ) { /* 40x... mode but 80x... requested? */
        i = (i==1) ? 3 : 2; /* mode 2/3 work on MONO, too. */
      }
      if ( (i > 1) && (cols==40) ) { /* 80x... mode but 40x... requested? */
        i = ( (i==2) || (i==7) ) ? 0 : 1; /* 0 if mode was 2 or 7, 1 else */
      }
      r.h.ah = 0;  /* set video mode */
      r.h.al |= i; /* the new mode */
      int86(0x10, &r, &r);
    } /* mode change needed for column selection */

    return set_lines(lines); /* SET NUMBER OF TEXT ROWS */
  }

  return 0;
} /* console */

/* ------------------------------------------------ */

/* find out whether a CRTC with editable cursor register exists at port. */
/* CGA/MDA/HGC 6845 has registers 0..0x11 (only 0x0e..0x0f readable)     */
/* 0x0e/0x0f: cursor location (high, low!) (12bit in 6845)               */
/* EGA: 0x0c..0x0f readable, VGA: 0..7 write-protectable, regs 0..0x18   */
int testCRTC(unsigned int port)
{
  unsigned char one, other;
  disable(); /* block IRQs during test */
  outportb(port,0x0f);     /* 0x0f is cursor position register, lower byte */
  one = inportb(port+1);   /* current value */
  outportb(port,0x0f);
  other = one ^ 1;
  outportb(port+1,other);  /* a modified value */
  outportb(port,0x0f);
  other = inportb(port+1); /* did new value get stored? */
  outportb(port,0x0f);
  outportb(port+1,one);    /* restore value */
  enable();  /* enable IRQs again */
  return (one != other);   /* if we could modify the value, success! */
} /* testCRTC */

/* ------------------------------------------------ */

/* handle video mode related commands */
/* optional argument: ",rows" (25/43/50) */
int screen(int mode, char * what)
{
  static volatile unsigned char far * sysinfop =
    (unsigned char far *)MK_FP(0x40, 0x10);
  unsigned int lines = 0;
  /* unsigned int flags = SYSINFO & 0x30; ... */
  unsigned int havevga = 0;

  if (mode==4) /* not 0..3? */
    mode=7; /* adjust MONO mode number */

  if (mode == 7) {
    if ( testCRTC(0x3b4) && testCRTC(0x3d4) ) { /* 2 active CRTCs found? */
      sysinfop[0] |= 0x30; /* force MONO mode (if mono CARD installed!) */
      /* 0x00: card-with-BIOS, 0x10 CGA40, 0x20 CGA80, 0x30 MONO */
      /* the sysinfop trick is needed to switch if TWO cards exist... */
    } /* dual monitor */
    /* other special MONO preparations needed if single monitor? */
  } else {
    if ( testCRTC(0x3b4) && testCRTC(0x3d4) ) { /* 2 active CRTCs found? */
      r.x.ax = 0x1a00; /* get VGA display combination - VGA install check */
      int86(0x10, &r, &r);
      /* returned active / alternate display in BL / BH (use al=1 to SET...) */
      /* 0 none, 1 mono, 2 CGA, 4..5 EGA 6 PGA 7..8 VGA a..c MCGA */
      if (r.h.al == 0x1a)
        havevga = 1; /* VGA found */
      sysinfop[0] = (sysinfop[0] & ~0x30) |
        ( (havevga==1) ? 0 : 0x20 ); /* switch to EGA/VGA BIOS or CGA80. */
        /* EGA seems to work fine in CGA80 mode - we only check for VGA. */
    } /* dual monitor */
    /* other special COLOR preparations needed if single monitor? */
  }

  /* how do we detect cases where ONLY mono or ONLY color works? */
  /* single monitor VGA can be in either mode but still can switch! */

  r.h.ah = 0; /* set mode */
  r.h.al = mode;
  int86(0x10, &r, &r);

#ifdef DMODE
  printf("MODE: screen mode %d, lines %d\r\n", mode, lines);
  if ( testCRTC(0x3b4) && testCRTC(0x3d4) ) { /* 2 active CRTCs found? */
    printf("Dual monitor system detected (MONO plus CGA or newer).\r\n");
  } /* dual monitor */
#endif

  if ((what!=NULL) && (what[0]==',')) {
    what++;
    lines = atoi(what);
    if ( (lines!=25) && (lines!=43)  && (lines!=50) ) {
        printf("Rows must be 25, 43 or 80.\r\n");
        return 1;
    }

    return set_lines(lines); /* SET NUMBER OF TEXT ROWS */
  } /* argument given */

  return 0;
} /* screen */

