/*  Mem

    Copyright (c) Express Software 1997.
    All Rights Reserved.

    Created by: Joseph Cosentino.
    With help from Michal Meller for the convmemfree routine.

    Clean up and bug fixes March 2001, Bart Oldeman, with a patch
    applied from Martin Stromberg.
    Thanks to Arkady V.Belousov for suggestions.

    April 2001, tom ehlert
    moved to small memory model to cure some strange bugs
    change largest executable program size from mem_free to
        largest free MCB
        
	Added (for Win98) with the help of Alain support for INT2F AX=4309 (Win98)

    Bart: moved to tiny model and added malloc checks. Then moved
    it back to the small model.
*/

#pragma inline  /* Use TASM to assemble inline ASM */
#if defined(__TURBOC__)
#pragma option -a-   /*  Be sure to compile with word alignment OFF !!! */
#endif

// I N C L U D E S //////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <dos.h>
#include <mem.h>
#include <string.h>
#include <ctype.h>
#include <bios.h>

// D E F I N E S ////////////////////////////////////////////////////////////

#define BYTE   unsigned char
#define uchar  unsigned char
#define ushort unsigned int
#define ulong  unsigned long

#define FALSE 0
#define TRUE  1

#define CARRY  1

#define MT_NONE    0
#define MT_SYSCODE 1
#define MT_SYSDATA 2
#define MT_PROGRAM 3
#define MT_DEVICE  4
#define MT_ENV     5
#define MT_DATA    6
#define MT_FREE    7
#define MT_MAP     8

// S T R U C T U R E S //////////////////////////////////////////////////////

typedef struct device_header
{
    struct device_header far*next;
    ushort attr;
    ushort strategy_rutin;
    ushort interrupt_rutin;
    uchar name[8];
} DEVICE_HEADER;

typedef struct devinfo
{
    struct device_header far *addr;
    uchar devname[9];
    uchar progname[9];
    ushort attr;
    uchar firstdrive, lastdrive;
    struct devinfo *next;
} DEVINFO;

typedef struct dpb
{
    uchar drive_num;
    uchar unit_number;
    ushort bytes_in_sector;
    uchar lsec_num;
    uchar log_base;
    ushort reserved_num;
    uchar FAT_num;
    ushort rootentry_num;
    ushort first_sector;
    ushort largest_cluster;
    ushort sectors_in_FAT;
    ushort root_firstsector;
    DEVICE_HEADER far*device_addr;
    uchar media_desc;
    uchar block_flag;
    struct dpb far*next_dpb;
} DPB;

typedef struct
{
    uchar type;
    ushort start;
    ushort size;
    uchar unused[3];
    uchar name[8];
} SD_HEADER;

typedef struct
{
    uchar type;
    ushort owner;
    ushort size;
    uchar unused[3];
    uchar name[8];
} MCB;

typedef struct minfo
{
    uchar type;
    ushort seg;
    ushort owner;
    ushort environment;
    uchar name[10];
    ulong size;
    uchar vecnum;
    uchar *vectors;
    struct minfo *next;
} MINFO;

typedef struct ems_handle
{
    ushort handle;
    ushort pages;
} EMS_HANDLE;

typedef struct xms_handle
{
    ushort handle;
    ulong size;
    ushort locks;
    struct xms_handle *next;
} XMS_HANDLE;

struct MCBheader {
    char type;
    unsigned int pid;
    unsigned int length;
    char reserved[3];
    char progname[8];
};

typedef struct {
    uchar vermajor, verminor;
    ushort totalhandle, freehandle, usehandle, frame;
    ulong size, free;
    EMS_HANDLE *handles;
} EMSINFO;

typedef struct {
    ulong total, free, largest;
    uchar freehandle, hma;
    uchar vermajor, verminor, a20;
    XMS_HANDLE *handles;
} XMSINFO;

typedef struct upperinfo {
    ulong total, free, largest;
} UPPERINFO;
    
// P R O T O T Y P E S //////////////////////////////////////////////////////

 EMSINFO *check_ems(void);
 XMSINFO *check_xms(void);
 UPPERINFO *check_upper(MINFO *mlist);
 uchar get_upperlink(void);
 int set_upperlink(uchar);
 void search_vectors(MINFO *);
 MINFO *search_sd(MCB far*, MINFO *mlist);
 MINFO *register_mcb(MCB far*, MINFO *mlist);
 int is_psp(MCB far*);
 ushort env_seg(MCB far*);
 MINFO *make_mcb_list(void);
 DEVINFO *make_dev_list(MINFO *mlist);
 void normal_list(char *argv0);
 void upper_list(void);
 void full_list(void);
 void device_list(void);
 void ems_list(void);
 void xms_list(void);
 EMSINFO *ems_error(EMSINFO *ems);
 void convert(char *format, long num);
 char *get_os(void);
 
/* void fmemcpy(void far *dest, void far *src, unsigned len);
 void fmemcmp(void far *s1,   void far *s2,  unsigned len);
*/ 
void fmemcpy(unsigned char far *dest, unsigned char far *src, unsigned len);
int  fmemcmp(unsigned char far *s1,   unsigned char far *s2,  unsigned len);
 

// F U N C T I O N S ////////////////////////////////////////////////////////

void fatal(char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  vprintf(format, ap);
  va_end(ap);
  exit(1);
}

void *xmalloc (size_t size)
{
  register void *value = malloc (size);
  if (value == NULL)
    fatal ("Out of memory.\n");
  return value;
}

void *xcalloc (size_t nitems, size_t size)
{
  register void *value = calloc (nitems, size);
  if (value == NULL)
    fatal ("Out of memory.\n");
  return value;
}

void convert(char *format, long num)
{
    char tmp;
    int c=0, n;
    char des[4*sizeof(long)];
    
    des[c++]='\0';
    do {
        if (c%4==0 && c>0)
            des[c++]=',';

        des[c++]=(char)(num%10)+'0';
        num/=10;
    } while (num > 0);

    /* flip array */
    for (n=0;n<c-1-n;n++) { 
        tmp=des[n];
        des[n]=des[c-1-n];
        des[c-1-n]=tmp;
    } // end else.

    printf(format, des);
} // end convert.

/////////////////////////////////////////////////////////////////////////////

static char* get_os()
{
    union REGS regs;
    BYTE OEMNumber, DOSMajor30;

    regs.h.ah = 0x30;                  // Get DOS version.
    regs.h.al = 0x00;                  // Get OEM number.
    intdos(&regs, &regs);
    OEMNumber = regs.h.bh;
    DOSMajor30 = regs.h.al;

    if (DOSMajor30 == 0x00)
        DOSMajor30 = 0x01;              // DOS 1.x.

    switch (OEMNumber)
        {
        case 0xFD:
            return "FreeDOS";
        case 0xFF:
            if (DOSMajor30 <= 6)
                return "MS-DOS";
            else
                return "Windows";
        case 0x00:
            return "PC-DOS";
        case 0xEE:
            return "DR-DOS";
        case 0xEF:
            return "Novell";
        case 0x66:
            return "PTS-DOS";
        case 0x5E:
            return "RxDOS";
        default:
            return "An unknown operating system";

        } // end switch.

} // end Check_DOS_Version.

/////////////////////////////////////////////////////////////////////////////

static EMSINFO *ems_error(EMSINFO *ems)
{
    printf("EMS INTERNAL ERROR.\n");
    free(ems);
    return NULL;
}

#define EMSNAME "EMMXXXX0"
static EMSINFO *check_ems(void)
{
    char far *int67;
    union REGS regs;
    struct SREGS sregs;
    static EMSINFO *ems = NULL;
    
    int67=(char far *)getvect(0x67);
    if (int67 == NULL || ems != NULL || fmemcmp(MK_FP(FP_SEG(int67),10), EMSNAME, 8) != 0)
        return(ems);
    
    ems = xcalloc(1, sizeof(EMSINFO));
    
    regs.h.ah = 0x41;
    int86(0x67, &regs, &regs);
    if (regs.h.ah)
        return ems_error(ems);
    ems->frame = regs.x.bx;

    regs.h.ah = 0x46;
    int86(0x67, &regs, &regs);
    if (regs.h.ah)
        return ems_error(ems);
    ems->vermajor = regs.h.al >> 4;
    ems->verminor = regs.h.al & 0xf;

    regs.h.ah = 0x42;
    int86(0x67, &regs, &regs);
    if (regs.h.ah)
        return ems_error(ems);
    ems->size = regs.x.dx*16384L;
    ems->free = regs.x.bx*16384L;

    regs.h.ah = 0x4B;
    int86x(0x67, &regs, &regs, &sregs);
    if (regs.h.ah)
        return ems_error(ems);
    ems->handles=xmalloc(regs.x.bx*sizeof(EMS_HANDLE));

    sregs.es = FP_SEG(ems->handles);
    regs.x.di = FP_OFF(ems->handles);
    regs.h.ah = 0x4D;
    int86x(0x67, &regs, &regs, &sregs);
    if (regs.h.ah)
        return ems_error(ems);
    ems->usehandle=regs.x.bx;

    if (ems->vermajor >= 4)
        {
        regs.x.ax = 0x5402;
        int86(0x67, &regs, &regs);
        if (regs.h.ah)
            return ems_error(ems);
        ems->totalhandle = regs.x.bx;
        } // end if.
    else
        {
        ems->totalhandle=ems->usehandle;
        } // end else.

    ems->freehandle=ems->totalhandle - ems->usehandle;
    return(ems);
} // end check_ems.

/////////////////////////////////////////////////////////////////////////////

void call_xms_driver(ushort *pax, ushort *pbx, ushort *pdx)
{
    static void far (*xms_drv)(void) = NULL;
    
    if (xms_drv == NULL)
    {
        struct SREGS sregs;
        union REGS regs;
        regs.x.ax = 0x4310;
        int86x(0x2f, &regs, &regs, &sregs);
        xms_drv = MK_FP(sregs.es, regs.x.bx);
    }
    asm {
	push si;
	mov si, pax;
	mov ax, [si]
	mov si, pbx
	mov bx, [si]
	mov si, pdx
	mov dx, [si]
	call dword ptr [xms_drv]
	mov si, pax;
	mov [si], ax
	mov si, pbx
	mov [si], bx
	mov si, pdx
	mov [si], dx
	pop si
    }
}

/////////////////////////////////////////////////////////////////////////////

static XMSINFO *check_xms(void)
{
    union REGS regs;
    struct
    {
        ulong baselow;
        ulong basehigh;
        ulong lenlow;
        ulong lenhigh;
        ulong type;
    } e820map;
    ulong total = 0;
    ushort ax, bx, dx;
    static XMSINFO *xms = NULL;
 
    if (xms != NULL) return xms;

    regs.x.ax = 0x4300;
    int86(0x2f, &regs, &regs);
    if (regs.h.al != 0x80)
        return NULL;

#define SMAP  0x534d4150

    /* check for 386+ before doing e820 stuff */
    /* load value 0x7000 into FLAGS register */
    /* then check whether all bits are set */
    asm {
            pushf
            pushf
            pop ax    
            or ax, 7000h
            push ax
            popf
            pushf
            pop ax
            popf    
            and ax, 7000h
            cmp ax, 7000h
            jne no386
        } /* yes: we have a 386! and can use ax=0xe820 for int15 */

    /* and now we can use some help from the linux kernel */
    /* adapted from v 2.4.1 linux/arch/i386/boot/setup.S */
    asm {
        .386
        push    ds
        pop     es
	lea     di, e820map
	xor     ebx, ebx                        /* continuation counter */
        xor     esi, esi
        }
jmpe820:
    asm {
        mov     eax, 0000e820h                  // e820, upper word zeroed
        mov     edx, SMAP                       // ascii 'SMAP'
        mov     ecx, 20                         // size of the e820map
        int     15h                             // make the call
        jc      no386                           // fall to e801 if it fails
        cmp     eax, SMAP                       // check the return is `SMAP`
        jne     no386                           // fall to e801 if it fails

        cmp	byte ptr [di+10h], 1	        // has to be system memory
        jne	againe820
        cmp     dword ptr [di], 1024*1024       // above 1 Meg.
        jb	againe820
        add     esi, [di+8]
    }
againe820:
    asm {
	or	ebx, ebx			 // check to see if
        jnz     jmpe820                          // ebx is set to EOF
        mov	total, esi
        .8086                                                      
    }
    
no386:
    
    regs.x.ax = 0xe801;
    int86(0x15, &regs, &regs);
    if ((regs.x.flags & CARRY) || (regs.x.bx == 0)) {
        regs.x.ax = 0x8800;
        int86(0x15, &regs, &regs);
        if ((regs.x.flags & CARRY) || (regs.x.ax == 0)) {
            /* Try the CMOS approach required by Alain from Ralf */
            regs.x.ax = 0;
            if (*(unsigned char far *)MK_FP(0xF000, 0xFFFE) == 0xFC) { /*is_AT*/
                outportb(0x70, 0x17);
                regs.h.al = inportb(0x71);
                outportb(0x70, 0x18);
                regs.h.ah = inportb(0x71);
            }
        }
        regs.x.bx = 0;
    }
    xms = xcalloc(1, sizeof(XMSINFO));

    xms->total=regs.x.ax * 1024L + regs.x.bx * 65536L;
    if (total > 0)
        xms->total = total;

    ax=0x800;
    call_xms_driver(&ax, &bx, &dx);
    xms->largest=ax*1024L;
    xms->free=dx*1024L;

    ax=0;
    call_xms_driver(&ax, &bx, &dx);
    xms->vermajor=ax >> 8;
    xms->verminor=ax & 0xff;
    xms->hma=dx & 0xff;

    return(xms);
} // end check_xms.

/////////////////////////////////////////////////////////////////////////////

static uchar get_upperlink(void)
{
    union REGS regs;

    regs.x.ax = 0x5802;
    intdos(&regs, &regs);
    return(regs.h.al);
} // end get_upperlink.

/////////////////////////////////////////////////////////////////////////////

static int set_upperlink(uchar link)
{
    union REGS regs;

    regs.x.ax = 0x5803;
    regs.h.bh = 0;
    regs.h.bl = link;
    intdos(&regs, &regs);
    if(!(regs.x.flags & CARRY))
        return(1);

    if (regs.x.ax == 1)
        return(0);

    return(-1);
} // end set_upperlink.

/////////////////////////////////////////////////////////////////////////////

UPPERINFO *check_upper(MINFO *mlist)
{
    static UPPERINFO *upper = NULL;
    uchar origlink;
    ushort lastconseg;

    if (upper!=NULL)
        return upper;
    
    origlink=get_upperlink();
    switch (set_upperlink(1))
        {
        case 1:
            set_upperlink(origlink);
            break;

        case 0:
            return upper;
            
        case -1:
            fatal("SYSTEM MEMORY TRASHED!\n");

        } // end switch.

    upper = xcalloc(1, sizeof(UPPERINFO));
    lastconseg = biosmemory()*64 - 1;
    while (mlist!=NULL && mlist->seg != lastconseg)
        mlist=mlist->next;
    
    if (mlist==NULL)
        fatal("MEM: UMB Corruption.\n");
    
    mlist=mlist->next;
    while (mlist!=NULL) {
        if (mlist->type == MT_FREE)
        {
            upper->free+=mlist->size;
            if (mlist->size > upper->largest)
                upper->largest=mlist->size;
            
        } // end if.
        upper->total += mlist->size + 16;
        mlist=mlist->next;
    }
    return(upper);
}

/////////////////////////////////////////////////////////////////////////////

char far *dos_list_of_lists(void)
{
    union REGS regs;
    struct SREGS sregs;
    // Get pointer to LL struct.
    regs.h.ah=0x52;
    intdosx(&regs, &regs, &sregs);
    return MK_FP(sregs.es, regs.x.bx);
}

/////////////////////////////////////////////////////////////////////////////

ulong convmemfree(char progname[8])
{
    struct MCBheader far *currentmcbblock;
    unsigned ulong freemem = 0;
    ushort convtopseg = biosmemory()*64;

    // In LL in offset -02 there's pointer to first mem block (segment only).
    currentmcbblock = (struct MCBheader far *)
        MK_FP((*(ushort far*)(dos_list_of_lists()-2)), 0);

    do {
        if (FP_SEG(currentmcbblock) < convtopseg &&
            ((currentmcbblock->pid==0) || fmemcmp(currentmcbblock->progname, progname, min(strlen(progname),8))==0)) 
            freemem += ((long)currentmcbblock->length+1) * 16;
        
        if (currentmcbblock->type == 'Z')  // Z means last memory block.
            break;
        
        currentmcbblock = MK_FP(FP_SEG(currentmcbblock) + 1 + currentmcbblock->length, 0);
    } // end do.
    while (1);
    
    return freemem;

} // end convmemfree.

/////////////////////////////////////////////////////////////////////////////

int is_psp(MCB far*mcb)
{
    return (*(ushort far*)MK_FP(FP_SEG(mcb)+1, FP_OFF(mcb)) == 0x20CD);
} // end is_psp.

/////////////////////////////////////////////////////////////////////////////

ushort env_seg(MCB far*mcb)
{
    ushort env;
    env = *(ushort far *)MK_FP(FP_SEG(mcb)+1, 0x2C);
    return (((MCB far *)MK_FP(env-1,0))->owner==FP_SEG(mcb)+1 ? env : 0);
} // end env_seg.

/////////////////////////////////////////////////////////////////////////////

void search_vectors(MINFO *m)
{
    ushort i;
    ulong begin, end, iv;
    uchar far *ivp;
    uchar vectors[256];

    begin=(ulong)(m->seg + 1) << 4;
    end=begin + m->size;
    for (i=0;i<256;i++)
        {
        ivp = *(uchar far **)MK_FP(0, i*4);
        iv=((ulong)(FP_SEG(ivp) + 1) << 4) + (ulong)FP_OFF(ivp);
        if ((iv > begin) && (iv < end))
            vectors[m->vecnum++]=(uchar)i;
	} // end for.
    if (m->vecnum != 0)
	{
	m->vectors = xmalloc(m->vecnum);
	memcpy(m->vectors, vectors, m->vecnum);
	}
} // end search_vectors.

/////////////////////////////////////////////////////////////////////////////

MINFO *search_sd(MCB far *mcb, MINFO *mlist)
{
    ushort begin, end;
    SD_HEADER far *sd;

    sd=MK_FP(FP_SEG(mcb) + 1, 0);
    begin=FP_SEG(mcb);
    end=FP_SEG(mcb) + mcb->size;
    while ((FP_SEG(sd) > begin) && (FP_SEG(sd) < end))
        {
        mlist->next = xcalloc(1, sizeof(MINFO));
        mlist = mlist->next;
        mlist->seg=sd->start;
        mlist->size=(ulong)sd->size << 4;
        switch (sd->type)
            {
            case 'D':
            case 'I':
                mlist->name[0]=' ';
                fmemcpy(&mlist->name[1], sd->name, 8);
                strupr(mlist->name);
                mlist->type=MT_DEVICE;
                break;

            case 'F':
                strcpy(mlist->name, " FILES");
                mlist->type=MT_DATA;
                break;

            case 'X':
                strcpy(mlist->name, " FCBS");
                mlist->type=MT_DATA;
                break;

            case 'C':
            case 'B':
                strcpy(mlist->name, " BUFFERS");
                mlist->type=MT_DATA;
                break;

            case 'L':
                strcpy(mlist->name, " LASTDRV");
                mlist->type=MT_DATA;
                break;

            case 'S':
                strcpy(mlist->name, " STACKS");
                mlist->type=MT_DATA;
                break;

            default:
                strcpy(mlist->name, " ??????");
                mlist->type=MT_DATA;
                break;

            } // end switch.

        sd=MK_FP(sd->start + sd->size, 0);
        } // end while.
    return(mlist);
} // end search_sd.

/////////////////////////////////////////////////////////////////////////////

void check_name(uchar *name)
{
    for (;*name && *name >= ' ';name++)
        ;
    *name = '\0';
} // end check_name.

/////////////////////////////////////////////////////////////////////////////

MINFO *register_mcb(MCB far*mcb, MINFO *mlist)
{
    mlist->seg=FP_SEG(mcb);
    mlist->owner=mcb->owner;
    mlist->size=(ulong)mcb->size << 4;
    if (mlist->seg <= biosmemory()*64 - 1)
        {
        if (is_psp(mcb))
            {
            fmemcpy(mlist->name, mcb->name, 8);
            check_name(mlist->name);
            strupr(mlist->name);
            mlist->environment=env_seg(mcb);
            mlist->type=MT_PROGRAM;
            } // end if.

        if (!mcb->owner)
            {
            mlist->type=MT_FREE;
            } // end if.
        else if (mcb->owner <= 0x0008)
            {
            strcpy(mlist->name, "DOS");
            if (!fmemcmp(mcb->name, "SD", 2))
                {
                mlist->type=MT_SYSDATA;
                mlist=search_sd(mcb, mlist);
                } // end if.
            else if (!fmemcmp(mcb->name, "SC", 2))
                {
                mlist->type=MT_SYSCODE;
                } // end else.
            else
                mlist->type=MT_SYSCODE;

            } // end else.

        } // end if.
    else
        {
        if (!mcb->owner)
            {
            mlist->type=MT_FREE;
            } // end if.
        else if (mcb->owner <= 0x0008)
            {
            strcpy(mlist->name, "DOS");
            if (!fmemcmp(mcb->name, "SD", 2))
                {
                mlist->type=MT_SYSDATA;
                search_sd(mcb, mlist);
                } // end if.
            else if (!fmemcmp(mcb->name, "SC", 2))
                {
                mlist->type=MT_SYSCODE;
                } // end else.

            } // end else.
        else if (mcb->owner > 0x0008)
            {
            mlist->environment=env_seg(mcb);
            mlist->type=MT_PROGRAM;
            fmemcpy(mlist->name, mcb->name, 8);
            strupr(mlist->name);
            } // end else.

        } // end else.
    return mlist;
    
} // end register_mcb.

/////////////////////////////////////////////////////////////////////////////

MINFO *make_mcb_list()
{
    MCB far *cur_mcb;
    uchar origlink;
    MINFO *mlistj, *mlist,*mlisttemp;
    static MINFO *mlistroot = NULL;

    if(mlistroot!=NULL)
        return(mlistroot);

    origlink=get_upperlink();
    set_upperlink(1);

    mlistroot = mlist = xcalloc(1, sizeof(MINFO));
    cur_mcb=MK_FP(*(ushort far *)(dos_list_of_lists()-2), 0);

    while(cur_mcb->type == 'M')
    {
        mlist=register_mcb(cur_mcb, mlist);
        cur_mcb=(MCB far *)MK_FP(FP_SEG(cur_mcb) + cur_mcb->size + 1, 0);
        mlist->next=xcalloc(1, sizeof(MINFO));
        mlist=mlist->next;
    } // end while.
    if (cur_mcb->type != 'Z')
        fatal("MEM: The MCB chain is corrupted.\n");
    mlist=register_mcb(cur_mcb, mlist);
    set_upperlink(origlink);

    for (mlist=mlistroot; mlist!=NULL; mlist=mlist->next)
        {
        if (mlist->type == MT_PROGRAM)
            for(mlistj=mlistroot;mlistj!=NULL;mlistj=mlistj->next)
               if ((mlist->seg != mlistj->seg) && (mlistj->owner == mlist->seg+1))
                   {
                   strcpy(mlistj->name, mlist->name);
                   mlistj->type=(mlist->environment == mlistj->seg+1) ? MT_ENV : MT_DATA;
                   } // end if.

        if (mlist->type != MT_SYSDATA)
            search_vectors(mlist);

        } // end for.


    // now find MEM himself and environment and mark this as MT_FREE

    for (mlist=mlistroot; mlist!=NULL ; mlist=mlist->next)
        {
        if (mlist->seg+1 == _psp)       // thats MEM himself
            {
            mlist->type=MT_FREE;
            mlist->name[0] = 0;
            
            for (mlistj=mlistroot; mlistj!=NULL ; mlistj=mlistj->next)
                {
                if (mlistj->seg+1 == mlist->environment)    // MEM environment
                    {
                    mlistj->type=MT_FREE;
                    mlistj->name[0] = 0;
                    }
                }
            break;
            }
        }
        
    // merge free blocks        
    for (mlist=mlistroot; mlist->next !=NULL ;)
        {
        if (mlist->type       != MT_FREE ||       
            mlist->next->type != MT_FREE)
            {
            mlist=mlist->next;
            continue;
            }
                                          // both free, merge

        mlist->size += mlist->next->size + 16;

        mlisttemp = mlist->next;
        mlist->next = mlist->next->next;     
        free(mlisttemp);
        }                                     
    
    return(mlistroot);
} // end make_mcb_list.

/*
    return largest possible segment size
*/

ulong mcb_largest(void)
{
    MINFO *mlist;
    ulong largest = 0;

    for (mlist=make_mcb_list(); mlist->next !=NULL ;mlist = mlist->next)
        {
        if (mlist->type == MT_FREE)
            if (mlist->size > largest)
                largest = mlist->size;
        }
    return largest;
    
}    


/////////////////////////////////////////////////////////////////////////////

DEVINFO *make_dev_list(MINFO *mlist)
{
    ushort i, j;
    DEVICE_HEADER far *cur_dev;
    DPB far*cur_dpb;
    DEVINFO *dlistroot, *dlist;
    MINFO *mlistroot = mlist;
    
    dlist = dlistroot = xcalloc(1, sizeof(DEVINFO));

    cur_dev = (DEVICE_HEADER far *)(dos_list_of_lists() + 0x22);
    
    while (FP_OFF(cur_dev) != 0xFFFF) {
        dlist->addr=cur_dev;
        dlist->attr=cur_dev->attr;
        fmemcpy(dlist->devname, cur_dev->name, 8);
        check_name(dlist->devname);
        strupr(dlist->devname);
        cur_dev=cur_dev->next;
        if (FP_OFF(cur_dev) != 0xFFFF) {
            dlist->next=xcalloc(1, sizeof(DEVINFO));
            dlist=dlist->next;
        }
    } // end while.


    for (dlist=dlistroot;dlist!=NULL;dlist=dlist->next)
        for (mlist=mlistroot;mlist!=NULL;mlist=mlist->next)
            if (mlist->seg == FP_SEG(dlist->addr))
                strcpy(dlist->progname, (mlist->name[0] == ' ') ?
                       &mlist->name[1] : mlist->name);

    for  (cur_dpb = *((DPB far *far*)dos_list_of_lists());
          FP_OFF(cur_dpb) != 0xFFFF; cur_dpb=cur_dpb->next_dpb)
        {
        for (dlist=dlistroot;dlist!=NULL && dlist->addr != cur_dpb->device_addr;
                 dlist=dlist->next)
            ;
        
        if (dlist!=NULL)
        {
            if (dlist->firstdrive==0)
                dlist->firstdrive=cur_dpb->drive_num+'A';
            else
                dlist->lastdrive=cur_dpb->drive_num+'A';
            }
        } // end if

    for (dlist=dlistroot;dlist!=NULL;dlist=dlist->next)
        {
        if ((dlist->attr & 0x8000) == 0)
            dlist->devname[0]='\0';

        if (dlist->firstdrive != 0)
            {
            if (dlist->lastdrive == 0)
                sprintf(dlist->devname, "%c:", dlist->firstdrive);
            else
                sprintf(dlist->devname, "%c: - %c:", dlist->firstdrive,
                        dlist->lastdrive);

            } // end if.
        } // end for

    return dlistroot;
} // end make_dev_list.

/////////////////////////////////////////////////////////////////////////////

void normal_list(char *argv0)
{
    ushort i, num;
    ushort memory, memused;
    ulong memfree;
    XMSINFO *xms;
    EMSINFO *ems;
    UPPERINFO *upper;
    union REGS regs;
    ulong largest_executable;

    ems=check_ems();
    xms=check_xms();
    upper=check_upper(make_mcb_list());
    memory=biosmemory();
    i = strlen(argv0);
    while (argv0[i] != '\\' && argv0[i] != '.' && argv0[i] != ':' && i>0) i--;
    if (argv0[i] == '.') {
        argv0[i] = '\0';
        while (argv0[i] != '\\' && argv0[i] != '.' && argv0[i] != ':' && i>0) i--;
    }
    if (i>0) i++;
    memfree=convmemfree(&argv0[i]);
    memused=(ulong)memory - (memfree+512)/1024;
    printf("\nMemory Type        Total        Used        Free\n"
	   "----------------  --------   ---------    --------\n"
           "Conventional %12uK %10uK %10luK\n", memory, memused, (memfree+512)/1024);
    if (upper != NULL) {
        upper->free=(upper->free+512)/1024;
        upper->total=(upper->total+512)/1024;
        printf("Upper        %12luK %10luK %10luK\n",
               upper->total, upper->total-upper->free, upper->free);
    }
    if (xms != NULL) {
        convert("Extended (XMS) %10sK ", (xms->total+512)/1024);
        convert("%10sK ", (xms->total+512)/1024-(xms->free+512)/1024);
        convert("%10sK\n", (xms->free+512)/1024);
    }
    printf(       
	   "----------------  --------   ---------    --------\n\n");
    if (ems != NULL) {
        printf("Total Expanded (EMS) ");
        convert("%19sK ", (ems->size+512)/1024);
        convert("(%10s bytes)\n", ems->size);
        printf("Free Expanded (EMS)  ");
        convert("%19sK ", (ems->free+512)/1024);
        convert("(%10s bytes)\n\n", ems->free);
    }

    largest_executable = mcb_largest();

    
    printf("Largest executable program size %8luK ", (largest_executable+512)/1024);
    convert("(%10s bytes)\n", largest_executable);
    if (upper != NULL) {
        printf("Largest free upper memory block %8luK ", (upper->largest+512)/1024);
        convert("(%10s bytes)\n", upper->largest);
    }

    regs.x.ax = 0x3306;
    regs.x.dx = 0;
    intdos(&regs, &regs);
    if (regs.h.dh & 0x10)
        printf("%s is resident in the high memory area.\n", get_os());
    else
        printf("%s is not resident in the high memory area.\n", get_os());
    
} // end normal_list.


/////////////////////////////////////////////////////////////////////////////

void print_entry(MINFO *entry)
{
    static uchar *typenames[]= { "", "system code", "system data",
                                 "program", "device driver", "environment",
                                 "data area", "free", "MT_MAP" };

    printf("  %04X   %6lu   %-9s   %-13s\n",
           entry->seg, entry->size, entry->name, typenames[entry->type]);
}

/////////////////////////////////////////////////////////////////////////////

void upper_list()
{
    MINFO *ml;

    printf("\nSegment   Size       Name         Type\n");
    printf(  "------- --------  ----------  -------------\n");
    for (ml=make_mcb_list();ml!=NULL;ml=ml->next)
        if ((ml->type == MT_SYSCODE) || (ml->type == MT_SYSDATA) ||
            (ml->type == MT_FREE) || (ml->type == MT_PROGRAM) ||
            (ml->type == MT_DEVICE))
                print_entry(ml);
} // end upper_list.

/////////////////////////////////////////////////////////////////////////////

void full_list()
{
    MINFO *ml;
    
    printf("\nSegment   Size       Name         Type\n");
    printf(  "------- --------  ----------  -------------\n");
    for (ml=make_mcb_list();ml!=NULL;ml=ml->next)
/*        if (ml->type != MT_MAP) */
            print_entry(ml);
} // end full_list.

/////////////////////////////////////////////////////////////////////////////

void device_list()
{
    DEVINFO *dl;

    printf("\n   Address     Attr    Name       Program\n");
    printf(  " -----------  ------ ----------  ----------\n");
          //  XXXX:XXXX    XXXX   XXXXXXXX    XXXXXXXX
    for (dl=make_dev_list(make_mcb_list());dl!=NULL;dl=dl->next)
         printf("  %Fp    %04X   %-8s    %-8s\n", dl->addr, dl->attr,
                dl->devname, dl->progname);
} // end device_list.

/////////////////////////////////////////////////////////////////////////////

void ems_list()
{
    EMSINFO *ems;
    ushort i;
    uchar handlename[9];
    union REGS regs;
    struct SREGS sregs;
    char format[] = "  %-20s";
    
    ems=check_ems();
    if (ems==NULL)
        {
        printf("  EMS driver not installed in system.  \n");
        } // end if.
    else
	{
        printf(format, "\nEMS driver version");
        printf("%1i.%1i\n", (int)ems->vermajor, (int)ems->verminor);
        printf(format, "EMS page frame");
        printf("%04X\n", ems->frame);
        printf(format, "Total EMS memory");
        printf("%lu bytes\n", ems->size);
        printf(format, "Free EMS memory");
        printf("%lu bytes\n", ems->free);
        printf(format, "Total handles");
        printf("%u\n", ems->totalhandle);
        printf(format, "Free handles");
        printf("%u\n", ems->freehandle);

        printf("\n  Handle   Pages    Size       Name\n");
        printf("   -------- ------  --------   ----------\n");
        for (i=0;i<ems->usehandle;i++)
            {
            memset(handlename, 0, sizeof(handlename));
            if (ems->vermajor >= 4)
                {
                if (ems->handles[i].handle == 0)
                    {
                    strcpy(handlename, "SYSTEM");
                    } // end if.
                else
                    {
                    regs.x.dx=ems->handles[i].handle;
                    sregs.es=FP_SEG(&handlename);
                    regs.x.di=FP_OFF(&handlename);
                    regs.x.ax=0x5300;
                    int86x(0x67, &regs, &regs, &sregs);
                    } // end else.

                strupr(handlename);
                } // end if.

            printf("%9u %6u %9lu   %-9s\n", ems->handles[i].handle,
                   ems->handles[i].pages, (ulong)ems->handles[i].pages * 16384L, handlename);
            } // end for.

        } // end else.

} // end ems_list.

/////////////////////////////////////////////////////////////////////////////

// 01/4/27 tom + alain
//
// although the 'old' method to search the handle table should be OK,
// it crashes somehow and for unknown reason under Win98. So, a 'new' method to
// query all handles was implemented, using INT 2F, AX=4309

// structures for INT 2F AX=4309
typedef struct{
    unsigned char flag;
    unsigned char locks;
    unsigned long xmsAddrKb;
    unsigned long xmsBlkSize;
}XMS_HANDLE_DESCRIPTOR;
typedef struct{
    unsigned char checkByte;
    unsigned char sizeOfDesc;
    unsigned short numbOfHandles;
    XMS_HANDLE_DESCRIPTOR far* xmsHandleDescr;
}XMS_HANDLE_TABLE;

void xms_list()
{
    UPPERINFO *upper;
    XMSINFO *xms;
    XMS_HANDLE *handle = NULL;
    ushort i, ax, bx, dx;
    char format[] = "%-26s";
    union REGS regs;
    struct SREGS sregs;
    uchar free_handles_tmp;
    XMS_HANDLE_TABLE far* xmsHanTab;

    xms = check_xms();
    
    if (xms==NULL)
        {
        printf("XMS driver not installed in system.\n");
        return;
        } // end if.

    printf("\nTesting XMS memory ...\n");

    ax = 0x700;
    call_xms_driver(&ax, &bx, &dx);
    xms->a20 = ax & 0xff;
    if ((bx&0xff)!=0) {
	printf("XMS INTERNAL ERROR.\n");
	return;
    }

	// new : test support for INT2F AX=4309 first    
    regs.x.ax=0x4309;
    int86x(0x2f, &regs, &regs, &sregs);
    if ( regs.h.al==0x43                                 && // test returned OK
	(xmsHanTab=MK_FP(sregs.es,regs.x.bx)) != NULL    &&
	 xmsHanTab->sizeOfDesc == sizeof(XMS_HANDLE_DESCRIPTOR)) // assert correct size
    {
	XMS_HANDLE_DESCRIPTOR far* descr = xmsHanTab->xmsHandleDescr;

	printf("INT 2F AX=4309 supported\n");

	    for (i=0;i<xmsHanTab->numbOfHandles;i++,descr++)
	      {
		if (descr->flag != 0x01 && // not free
		    descr->xmsBlkSize != 0)   // and takes memory
		{
		  if (handle==NULL)
		      xms->handles=handle=xmalloc(sizeof(XMS_HANDLE));
		  else {
		      handle->next=xmalloc(sizeof(XMS_HANDLE));
		      handle=handle->next;
		  }
		  handle->handle=FP_OFF(descr);
		  handle->size=descr->xmsBlkSize*1024L;
		  handle->locks=descr->locks;
		  handle->next=NULL;
	        }
	      }
	}
    else
    {
								// old method
								// query all handles 0..0xffff

	    for (i=0;i<65535;i++)
	    {
	        ax=0xE00;			// Get handle information
	        dx=i;
	        call_xms_driver(&ax, &bx, &dx);
	        if (ax != 0) {
	            free_handles_tmp = bx & 0xff;
	            if (handle==NULL)
	                xms->handles=handle=xmalloc(sizeof(XMS_HANDLE));
	            else {
	                handle->next=xmalloc(sizeof(XMS_HANDLE));
	                handle=handle->next;
	            }
	            handle->handle=i;
	            handle->size=dx*1024L;
	            handle->locks=bx >> 8;
	            handle->next=NULL;
	        } // end if.
	
	        if (xms->freehandle < free_handles_tmp)
	        {
	            xms->freehandle = free_handles_tmp;
	        }
	    } // end for.
	} // end old method    

    /* First try to get a handle of our own. */
    ax = 0x900; // "Allocate extended memory block"
    dx = 1;     /* First we try 1kB. I'm not sure if XMS driver
                must support a zero sized allocate. */
    call_xms_driver(&ax, &bx, &dx);
    i = dx;
    if (ax==0) {
        /* Now try a zero sized allocate just in case there was no free memory. */
        ax = 0x900; // "Allocate extended memory block"
        dx = 0;
        call_xms_driver(&ax, &bx, &dx);
        /* If AX==0, nothing worked out. Use whatever we got from the loop above. */
        /* We can't do much if the free call fails, so it ends here. */
    }
    if (ax!=0) {
        ax = 0xe00; 			// Get handle information
        call_xms_driver(&ax, &bx, &dx);
        if (ax!=0)
            /* Hey! We got some info. Put it in a safe place. */
            xms->freehandle = (bx & 0xff) + 1;
        /* Add one for the handle we have allocated. */
        dx = i;
        ax = 0xa00;			// "Free extended memory block"
        call_xms_driver(&ax, &bx, &dx);
    }
    
    printf(format, "XMS driver version");
    printf("%i.%i\n", (ushort)xms->vermajor, (ushort)xms->verminor);
    printf(format, "HMA state");
    printf("%s\n", (xms->hma) ? "exists" : "not exists");
    printf(format, "A20 line state");
    printf("%s\n", (xms->a20) ? "enabled" : "disabled");
    printf(format, "Free XMS memory");
    printf("%lu bytes\n", xms->free);
    printf(format, "Largest free XMS block");
    printf("%lu bytes\n", xms->largest);
    printf(format, "Free handles");
    printf("%u\n\n", xms->freehandle);
    if (xms->handles != NULL)
    {
        printf(" Block   Handle     Size     Locks\n");
        printf("------- --------  --------  -------\n");
        for (i=0, handle=xms->handles;handle!=NULL;handle=handle->next, i++)
            printf("%7u %8u  %8lu  %7u\n", i, handle->handle,
                   handle->size, handle->locks);
        
    } // end if.
    
    upper = check_upper(make_mcb_list());
    if (upper != NULL)
    {
        printf(format, "Free upper memory");
        printf("%lu bytes\n", upper->free);
        printf(format, "Largest upper block");
        printf("%lu bytes\n", upper->largest);
    } // end if.
    else
    {
        printf("Upper memory not available\n");
    } // end else.
    
} // end xms_list.

/////////////////////////////////////////////////////////////////////////////

int classify_args(int narg, char *rawargs[], char *optargs[])
{
    int index, kndex;
    char *argptr;

    for (index=0,kndex=0;index<narg;index++)
        {
        argptr = rawargs[index];
        if (*argptr == '/')
            {
            argptr++;
            optargs[kndex++] = argptr;
            } // end if.
        } // end for.

   return kndex;
} // end classify_args.

/////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char **optargs = xmalloc(argc*sizeof(char *));
    int n_options, n_files, index, help_flag=0;

    n_options = classify_args(argc, argv, optargs);
    for (index=0;index<n_options && help_flag==0;index++)
        switch(toupper(optargs[index][0])) {
        case '?':
            help_flag=1;
            break;
        case 'U':
            upper_list();
            break;
        case 'F':
            full_list();
            break;
        case 'D':
            device_list();
            break;
        case 'E':
            ems_list();
            break;
        case 'X':
            xms_list();
            break;
        default:
            fatal("Invalid parameter - /%s\n", strupr(optargs[index]));
        } // end switch.
        
        
    if (help_flag)
        {
	printf("Displays the amount of used and free memory in your system.\n\n"
               "Syntax: MEM [/E] [/F] [/D] [/U] [/X] [/?]\n"
               "  /E  Reports all information about Expanded Memory\n"
               "  /F  Full list of memory blocks\n"
               "  /D  List of device drivers currently in memory\n"
               "  /U  List of programs in conventional and upper memory\n"
               "  /X  Reports all information about Extended Memory\n"
               "  /?  Displays this help message\n");
        return 1;
        } // end if.

    normal_list(argv[0]);
    return 0;

} // end main.

/*
    unfortunately not part of TC small model runtime library
*/
void fmemcpy(unsigned char far *dest, unsigned char far *src, unsigned len)
{
    for ( ; len; len--)
        *dest++ = *src++;
}
int  fmemcmp(unsigned char far *s1,   unsigned char far *s2,  unsigned len)
{
    for ( ; len; s1++,s2++,len--)
        if (*s1 != *s2) 
            return *s1 - *s2;

    return 0;            
}
 
