/*  Mem

    Copyright (c) Express Software 1997.
    All Rights Reserved.

    Created by: Joseph Cosentino.
    With help from Michal Meller for the convmemfree routine.

    Clean up and bug fixes March 2001, Bart Oldeman, with a patch
    applied from Martin Stromberg.

*/

// I N C L U D E S //////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <mem.h>
#include <string.h>
#include <bios.h>

// D E F I N E S ////////////////////////////////////////////////////////////

#define BYTE   unsigned char
#define uchar  unsigned char
#define ushort unsigned int
#define ulong  unsigned long

#define FALSE 0
#define TRUE  1

#define NUMLEN 4*sizeof(long)
#define ENVSTRLEN 40
#define CARRY  1

#define MAX_DRIVE  20
#define MAX_HANDLE 255
#define MT_NONE    0
#define MT_SYSCODE 1
#define MT_SYSDATA 2
#define MT_PROGRAM 3
#define MT_DEVICE  4
#define MT_ENV     5
#define MT_DATA    6
#define MT_FREE    7
#define MT_MAP     8

// S T R U C T U R E S //////////////////////////////////////////////////////

typedef struct device_header
{
    struct device_header *next;
    ushort attr;
    ushort strategy_rutin;
    ushort interrupt_rutin;
    uchar name[8];
} DEVICE_HEADER;

typedef struct devinfo
{
    struct device_header *addr;
    uchar devname[9];
    uchar progname[9];
    ushort attr;
    uchar drive_num;
    uchar drives[MAX_DRIVE];
    struct devinfo *next;
} DEVINFO;

typedef struct dpb
{
    uchar drive_num;
    uchar unit_number;
    ushort bytes_in_sector;
    uchar lsec_num;
    uchar log_base;
    ushort reserved_num;
    uchar FAT_num;
    ushort rootentry_num;
    ushort first_sector;
    ushort largest_cluster;
    ushort sectors_in_FAT;
    ushort root_firstsector;
    DEVICE_HEADER *device_addr;
    uchar media_desc;
    uchar block_flag;
    struct dpb *next_dpb;
} DPB;

typedef struct
{
    uchar type;
    ushort start;
    ushort size;
    uchar unused[3];
    uchar name[8];
} SD_HEADER;

typedef struct
{
    uchar type;
    ushort owner;
    ushort size;
    uchar unused[3];
    uchar name[8];
} MCB;

typedef struct minfo
{
    uchar type;
    ushort seg;
    ushort owner;
    ushort environment;
    uchar name[10];
    ulong size;
    uchar vecnum;
    uchar *vectors;
    struct minfo *next;
} MINFO;

typedef struct
{
    ushort handle;
    ushort pages;
} EMS_HANDLE;

typedef struct
{
    ushort handle;
    ulong size;
    ushort locks;
} XMS_HANDLE;

struct MCBheader {
    char type;
    unsigned int pid;
    unsigned int length;
    char reserved[3];
    char progname[8];
};

typedef struct {
    uchar installed, vermajor, verminor;
    ushort totalhandle, freehandle, usehandle, frame;
    ulong size, free;
    EMS_HANDLE handles[MAX_HANDLE];
} EMSINFO;

typedef struct {
    uchar installed;
    void far (*drv)(void);
    ulong total, free, largest;
    ushort usehandle;
    uchar freehandle, hma;
    uchar vermajor, verminor, a20;
    XMS_HANDLE handles[MAX_HANDLE];
} XMSINFO;

typedef struct upperinfo {
    uchar installed;
    ulong total, free, largest;
} UPPERINFO;
    
// P R O T O T Y P E S //////////////////////////////////////////////////////

static EMSINFO *check_ems(void);
static XMSINFO *check_xms(void);
static int check_upper(void);
static uchar get_upperlink(void);
static int set_upperlink(uchar);
static void search_vectors(MINFO *);
static MINFO *search_sd(MCB *, MINFO *mlist);
static MINFO *register_mcb(MCB *, MINFO *mlist);
static int is_psp(MCB *);
static ushort env_seg(MCB *);
static MINFO *make_mcb_list(UPPERINFO *upper);
static DEVINFO *make_dev_list(MINFO *mlist);
static void normal_list(char *argv0);
static void upper_list(void);
static void full_list(void);
static void device_list(void);
static void ems_list(void);
static void xms_list(void);

// F U N C T I O N S ////////////////////////////////////////////////////////

void convert(long num, char *des)
{
    char tmp;
    int c=0, n;
    
    des[c++]='\0';
    do {
        if (c%4==0 && c>0)
            des[c++]=',';

        des[c++]=(char)(num%10)+'0';
        num/=10;
    } while (num > 0);

    /* flip array */
    for (n=0;n<c-1-n;n++) { 
        tmp=des[n];
        des[n]=des[c-1-n];
        des[c-1-n]=tmp;
    } // end else.

} // end convert.

/////////////////////////////////////////////////////////////////////////////

void get_os(char *EnvString)
{
    union REGS regs;
    BYTE OEMNumber, DOSMajor30;

    regs.h.ah = 0x30;                  // Get DOS version.
    regs.h.al = 0x00;                  // Get OEM number.
    intdos(&regs, &regs);
    OEMNumber = regs.h.bh;
    DOSMajor30 = regs.h.al;

    if (DOSMajor30 == 0x00)
        DOSMajor30 = 0x01;              // DOS 1.x.

    switch (OEMNumber)
        {
        case 0xFD:
            strcpy(EnvString, "FreeDOS");
            break;
        case 0xFF:
            if (DOSMajor30 <= 6)
                strcpy(EnvString, "MS-DOS");
            else
                strcpy(EnvString, "Windows");
            break;
        case 0x00:
            strcpy(EnvString, "PC-DOS");
            break;
        case 0xEE:
            strcpy(EnvString, "DR-DOS");
            break;
        case 0xEF:
            strcpy(EnvString, "Novell");
            break;
        case 0x66:
            strcpy(EnvString, "PTS-DOS");
            break;
        case 0x5E:
            strcpy(EnvString, "RxDOS");
            break;
        default:
            strcpy(EnvString, "An unknown operating system");

        } // end switch.

} // end Check_DOS_Version.

/////////////////////////////////////////////////////////////////////////////

static void ems_error()
{
    puts("EMS INTERNAL ERROR.");
    exit(1);
}

#define EMSNAME "EMMXXXX0"
static EMSINFO *check_ems()
{
    char far *int67;
    union REGS regs;
    struct SREGS sregs;
    EMSINFO *ems = calloc(1, sizeof(EMSINFO));
    
    int67=(char *)getvect(0x67);
    if (int67 == NULL)
        return(ems);
    
    if (memcmp(MK_FP(FP_SEG(int67),10), EMSNAME, 8) != 0)
        return(ems);

    ems->installed = TRUE;

    regs.h.ah = 0x41;
    int86(0x67, &regs, &regs);
    if (regs.h.ah)
        ems_error();
    ems->frame = regs.x.bx;
    
    regs.h.ah = 0x46;
    int86(0x67, &regs, &regs);
    if (regs.h.ah)
        ems_error();
    ems->vermajor = regs.h.al >> 4;
    ems->verminor = regs.h.al & 0xf;

    regs.h.ah = 0x42;
    int86(0x67, &regs, &regs);
    if (regs.h.ah)
        ems_error();
    ems->size = regs.x.dx*16384L;
    ems->free = regs.x.bx*16384L;

    sregs.es = FP_SEG(&ems->handles);
    regs.x.di = FP_OFF(&ems->handles);
    regs.h.ah = 0x4D;
    int86x(0x67, &regs, &regs, &sregs);
    if (regs.h.ah)
        ems_error();
    ems->usehandle=regs.x.bx;

    if (ems->vermajor >= 4)
        {
        regs.x.ax = 0x5402;
        int86(0x67, &regs, &regs);
        if (regs.h.ah)
            ems_error();
        ems->totalhandle = regs.x.bx;
        } // end if.
    else
        {
        ems->totalhandle=ems->usehandle;
        } // end else.

    ems->freehandle=ems->totalhandle - ems->usehandle;
    return(ems);
} // end check_ems.

/////////////////////////////////////////////////////////////////////////////

static XMSINFO *check_xms(void)
{
    union REGS regs;
    struct SREGS sregs;
    struct
    {
        ulong baselow;
        ulong basehigh;
        ulong lenlow;
        ulong lenhigh;
        ulong type;
    } e820map;
    ulong total = 0;
    static XMSINFO *xms = NULL;

    if (xms != NULL) return xms;

    xms = calloc(1, sizeof(XMSINFO));
    regs.x.ax = 0x4300;
    int86(0x2f, &regs, &regs);
    if (regs.h.al != 0x80)
        return(FALSE);

    xms->installed = TRUE;

#define SMAP  0x534d4150

    /* check for 386+ before doing e820 stuff */
    /* load value 0x7000 into FLAGS register */
    /* then check whether all bits are set */
    asm {
            mov ax, 7000h
            push ax
            popf
            pushf
            pop ax
            and ax, 7000h
            cmp ax, 7000h
            jne no386
        } /* yes: we have a 386! and can use ax=0xe820 for int15 */

    /* and now we can use some help from the linux kernel */
    /* adapted from v 2.4.1 linux/arch/i386/boot/setup.S */
    _ES = FP_SEG(&e820map);
    _DI = FP_OFF(&e820map);
    asm {
        .386
        xor     ebx, ebx                        /* continuation counter */
        xor     esi, esi
        }
jmpe820:
    asm {
        mov     eax, 0000e820h                  // e820, upper word zeroed
        mov     edx, SMAP                       // ascii 'SMAP'
        mov     ecx, 20                         // size of the e820map
        int     15h                             // make the call
        jc      no386                           // fall to e801 if it fails
        cmp     eax, SMAP                       // check the return is `SMAP`
        jne     no386                           // fall to e801 if it fails

	cmp	 byte ptr es:[di+10h], 1	// has to be system memory
	jne	 againe820
        cmp      dword ptr es:[di], 1024*1024   // above 1 Meg.
	jb	 againe820
        add      esi, es:[di+8]
    }
againe820:
asm {
	or	 ebx, ebx			 // check to see if
        jnz      jmpe820                         // ebx is set to EOF
	mov	 total, esi
        .8086                                                      
    }
    
no386:
    
    regs.x.ax = 0xe801;
    int86x(0x15, &regs, &regs, &sregs);
    if ((regs.x.flags & CARRY) || (regs.x.bx == 0)) {
        regs.h.ah = 0x88;
        int86x(0x15, &regs, &regs, &sregs);
        regs.x.bx = 0;
    }
    xms->total=regs.x.ax * 1024L + regs.x.bx * 65536L;
    if (total > 0)
        xms->total = total;

    regs.x.ax = 0x4310;
    int86x(0x2f, &regs, &regs, &sregs);
    xms->drv = MK_FP(sregs.es, regs.x.bx);

    _AH=8;
    (*xms->drv)();
    regs.x.ax = _AX;
    regs.x.dx = _DX;
    xms->largest=regs.x.ax*1024L;
    xms->free=regs.x.dx*1024L;

    _AH=0;
    (*xms->drv)();
    regs.x.ax = _AX;
    regs.x.dx = _DX;
    xms->vermajor=regs.h.ah;
    xms->verminor=regs.h.al;
    xms->hma=regs.h.dl;

    return(xms);
} // end check_xms.

/////////////////////////////////////////////////////////////////////////////

static uchar get_upperlink(void)
{
    union REGS regs;

    regs.x.ax = 0x5802;
    intdos(&regs, &regs);
    return(regs.h.al);
} // end get_upperlink.

/////////////////////////////////////////////////////////////////////////////

static int set_upperlink(uchar link)
{
    union REGS regs;

    regs.x.ax = 0x5803;
    regs.h.bh = 0;
    regs.h.bl = link;
    intdos(&regs, &regs);
    if(!(regs.x.flags & CARRY))
        return(1);

    if (regs.x.ax == 1)
        return(0);

    return(-1);
} // end set_upperlink.

/////////////////////////////////////////////////////////////////////////////

static int check_upper()
{
    uchar origlink;
    
    origlink=get_upperlink();
    switch (set_upperlink(1))
        {
        case 1:
            set_upperlink(origlink);
            return(TRUE);

        case -1:
            puts("SYSTEM MEMORY TRASHED!");
            exit(1);

        } // end switch.
    return(FALSE); /* case 0 */
} // end check_upper.

/////////////////////////////////////////////////////////////////////////////

long convmemfree(char progname[8])
{
    struct MCBheader far *firstmcbpointer, far *currentmcbblock;
    unsigned int far *temp;
    unsigned long freemem;

    freemem = 0;

    // Get pointer to LL struct.
    _AH=0x52;
    geninterrupt(0x21);

    // In LL in offset -02 there's pointer to first mem block (segment only).
    temp = MK_FP(_ES, _BX - 0x2);
    firstmcbpointer = (struct MCBheader far *)MK_FP(*temp, 0);
    currentmcbblock = firstmcbpointer;
    do
	{
        if (FP_SEG(currentmcbblock) < 0xA000 &&
            ((currentmcbblock->pid==0) || strncmp(currentmcbblock->progname, progname, 8)==0)) 
            freemem += ((long)currentmcbblock->length+1) * 16;
        
        if (currentmcbblock->type == 'Z')  // Z means last memory block.
            break;

        currentmcbblock = MK_FP(FP_SEG(currentmcbblock) + 1 + currentmcbblock->length, 0);
        } // end do.
    while (1);

    return ((long)freemem);

} // end convmemfree.

/////////////////////////////////////////////////////////////////////////////

static int is_psp(MCB *mcb)
{
    return (*(ushort *)MK_FP(FP_SEG(mcb)+1, FP_OFF(mcb)) == 0x20CD);
} // end is_psp.

/////////////////////////////////////////////////////////////////////////////

static ushort env_seg(MCB *mcb)
{
    ushort env;
    env = *(ushort *)MK_FP(FP_SEG(mcb)+1, 0x2C);
    return (((MCB *)MK_FP(env-1,0))->owner==FP_SEG(mcb)+1 ? env : 0);
} // end env_seg.

/////////////////////////////////////////////////////////////////////////////

static void search_vectors(MINFO *m)
{
    ushort i;
    ulong begin, end, iv;
    uchar far *ivp;
    uchar vectors[256];

    begin=(ulong)(m->seg + 1) << 4;
    end=begin + m->size;
    for (i=0;i<256;i++)
        {
        ivp = *(uchar far **)MK_FP(0, i*4);
        iv=((ulong)(FP_SEG(ivp) + 1) << 4) + (ulong)FP_OFF(ivp);
        if ((iv > begin) && (iv < end))
            vectors[m->vecnum++]=(uchar)i;
        } // end for.
    m->vectors = malloc(m->vecnum);
    memcpy(m->vectors, vectors, m->vecnum);
} // end search_vectors.

/////////////////////////////////////////////////////////////////////////////

static MINFO *search_sd(MCB *mcb, MINFO *mlist)
{
    ushort begin, end;
    SD_HEADER *sd;

    sd=MK_FP(FP_SEG(mcb) + 1, 0);
    begin=FP_SEG(mcb);
    end=FP_SEG(mcb) + mcb->size;
    while ((FP_SEG(sd) > begin) && (FP_SEG(sd) < end))
        {
        mlist->next = calloc(1, sizeof(MINFO));
        mlist = mlist->next;
        mlist->seg=sd->start;
        mlist->size=(ulong)sd->size << 4;
        switch (sd->type)
            {
            case 'D':
            case 'I':
                mlist->name[0]=' ';
                strncpy(&mlist->name[1], sd->name, 8);
                strupr(mlist->name);
                mlist->type=MT_DEVICE;
                break;

            case 'F':
                strcpy(mlist->name, " FILES");
                mlist->type=MT_DATA;
                break;

            case 'X':
                strcpy(mlist->name, " FCBS");
                mlist->type=MT_DATA;
                break;

            case 'C':
            case 'B':
                strcpy(mlist->name, " BUFFERS");
                mlist->type=MT_DATA;
                break;

            case 'L':
                strcpy(mlist->name, " LASTDRV");
                mlist->type=MT_DATA;
                break;

            case 'S':
                strcpy(mlist->name, " STACKS");
                mlist->type=MT_DATA;
                break;

            default:
                strcpy(mlist->name, " ??????");
                mlist->type=MT_DATA;
                break;

            } // end switch.

        sd=MK_FP(sd->start + sd->size, 0);
        } // end while.
    return(mlist);
} // end search_sd.

/////////////////////////////////////////////////////////////////////////////

void check_name(uchar *name)
{
    ushort i;

    for (i=0;name[i];i++)
        if (name[i] < ' ')
            {
            name[i] = '\0';
            break;
            } // end if.

} // end check_name.

/////////////////////////////////////////////////////////////////////////////

static MINFO *register_mcb(MCB *mcb, MINFO *mlist)
{
    mlist->seg=FP_SEG(mcb);
    mlist->owner=mcb->owner;
    mlist->size=(ulong)mcb->size << 4;
    if (mlist->seg <= biosmemory()*64 - 1)
        {
        if (is_psp(mcb))
            {
            strncpy(mlist->name, mcb->name, 8);
            check_name(mlist->name);
            strupr(mlist->name);
            mlist->environment=env_seg(mcb);
            mlist->type=MT_PROGRAM;
            } // end if.

        if (!mcb->owner)
            {
            mlist->type=MT_FREE;
            } // end if.
        else if (mcb->owner <= 0x0008)
            {
            strcpy(mlist->name, "DOS");
            if (!strncmp(mcb->name, "SD", 2))
                {
                mlist->type=MT_SYSDATA;
                mlist=search_sd(mcb, mlist);
                } // end if.
            else if (!strncmp(mcb->name, "SC", 2))
                {
                mlist->type=MT_SYSCODE;
                } // end else.
            else
                mlist->type=MT_SYSCODE;

            } // end else.

        } // end if.
    else
        {
        if (!mcb->owner)
            {
            mlist->type=MT_FREE;
            } // end if.
        else if (mcb->owner <= 0x0008)
            {
            strcpy(mlist->name, "DOS");
            if (!strncmp(mcb->name, "SD", 2))
                {
                mlist->type=MT_SYSDATA;
                search_sd(mcb, mlist);
                } // end if.
            else if (!strncmp(mcb->name, "SC", 2))
                {
                mlist->type=MT_SYSCODE;
                } // end else.

            } // end else.
        else if (mcb->owner > 0x0008)
            {
            mlist->environment=env_seg(mcb);
            mlist->type=MT_PROGRAM;
            strncpy(mlist->name, mcb->name, 8);
            strupr(mlist->name);
            } // end else.

        } // end else.
    return mlist;
    
} // end register_mcb.

/////////////////////////////////////////////////////////////////////////////

static MINFO *make_mcb_list(UPPERINFO *upper)
{
    union REGS regs;
    struct SREGS sregs;
    ushort i, j;
    MCB *cur_mcb;
    uchar origlink;
    MINFO *mlistj, *mlist;
    static MINFO *mlistroot = NULL;
    static UPPERINFO *supper;

    if(mlistroot!=NULL) {
        memcpy(upper, supper, sizeof(UPPERINFO));
        return(mlistroot);
    }   

    supper = upper;
    upper->installed=check_upper();
    regs.h.ah = 0x52;
    intdosx(&regs, &regs, &sregs);

    if (upper->installed)
    {
        origlink=get_upperlink();
        set_upperlink(1);
    } // end if.

    mlistroot = mlist = calloc(1, sizeof(MINFO));
    cur_mcb=MK_FP(*(ushort *)MK_FP(sregs.es, regs.x.bx-2), 0);
    while(cur_mcb->type == 'M')
    {
        mlist=register_mcb(cur_mcb, mlist);
        cur_mcb=(MCB *)MK_FP(FP_SEG(cur_mcb) + cur_mcb->size + 1, 0);
        mlist->next=calloc(1, sizeof(MINFO));
        mlist=mlist->next;
    } // end while.
    if (cur_mcb->type != 'Z') {
        printf("MEM: The MCB chain is corrupted.\n");
        exit(1);
    }
    mlist=register_mcb(cur_mcb, mlist);
    
    if (upper->installed) {
        j = biosmemory()*64 - 1;
        for (mlist=mlistroot;mlist!=NULL && mlist->seg != j;mlist=mlist->next)
            ;
            
        if (mlist==NULL) {
            printf("UMB Corruption\n");
            exit(1);
        }
        
        mlist=mlist->next;
        while (mlist!=NULL) {
            if (mlist->type == MT_FREE)
            {
                upper->free+=mlist->size;
                if (mlist->size > upper->largest)
                    upper->largest=mlist->size;

            } // end if.
            upper->total += mlist->size + 16;
            mlist=mlist->next;
        }
        set_upperlink(origlink);
    }

    for (mlist=mlistroot; mlist!=NULL; mlist=mlist->next)
        {
        if (mlist->type == MT_PROGRAM)
            for(mlistj=mlistroot;mlistj!=NULL;mlistj=mlistj->next)
               if ((mlist->seg != mlistj->seg) && (mlistj->owner == mlist->seg+1))
                   {
                   strcpy(mlistj->name, mlist->name);
                   mlistj->type=(mlist->environment == mlistj->seg+1) ? MT_ENV : MT_DATA;
                   } // end if.

        if (mlist->type != MT_SYSDATA)
            search_vectors(mlist);

        } // end for.

    for (mlist=mlistroot; mlist!=NULL; mlist=mlist->next)
        if (mlist->seg+1 == _psp)
            {
            mlist->next->size+=mlist->size + 16;
            mlist->type=MT_MAP;
            for (mlistj=mlistroot; mlist!=NULL; mlist=mlist->next)
                if (mlistj->seg+1 == mlist->environment)
                    {
                    if (mlistj->next == mlist)
                        {
                        mlistj->type=MT_MAP;
                        } // end if.
                    else
                        {
                        mlistj->type=MT_FREE;
                        mlistj->name[0]='\0';
                        } // end else.

                    break;
                    } // end if.

            break;
            } // end if.
    return(mlistroot);
} // end make_mcb_list.

/////////////////////////////////////////////////////////////////////////////

static DEVINFO *make_dev_list(MINFO *mlist)
{
    union REGS regs;
    struct SREGS sregs;
    ushort i, j;
    DEVICE_HEADER *cur_dev;
    DPB *cur_dpb;
    DEVINFO *dlistroot, *dlist;
    MINFO *mlistroot = mlist;

    dlist = dlistroot = calloc(1, sizeof(DEVINFO));

    regs.h.ah = 0x52;
    intdosx(&regs, &regs, &sregs);

    cur_dev=MK_FP(sregs.es, regs.x.bx + 0x22);
    while ((FP_OFF(cur_dev) != 0xFFFF))
        {
        dlist->addr=cur_dev;
        dlist->attr=cur_dev->attr;
        strncpy(dlist->devname, cur_dev->name, 8);
        check_name(dlist->devname);
        strupr(dlist->devname);
        cur_dev=cur_dev->next;
        if (FP_OFF(cur_dev) == 0xFFFF) break;
        dlist->next=calloc(1, sizeof(DEVINFO));
        dlist=dlist->next;
        } // end while.

    for (dlist=dlistroot;dlist!=NULL;dlist=dlist->next) {
        for (mlist=mlistroot;mlist!=NULL;mlist=mlist->next){
            if (mlist->seg == FP_SEG(dlist->addr)) {
                strcpy(dlist->progname, (mlist->name[0] == ' ') ?
                       &mlist->name[1] : mlist->name);
                exit(1);
            }
        }
    }

    regs.h.ah = 0x52;
    intdosx(&regs, &regs, &sregs);
    cur_dpb=*((DPB **)MK_FP(sregs.es, regs.x.bx));

    while (FP_OFF(cur_dpb) != 0xFFFF)
        {
        for (dlist=dlistroot;dlist!=NULL;dlist=dlist->next)
            if (dlist->addr == cur_dpb->device_addr)
                {
                dlist->drives[dlist->drive_num++]=cur_dpb->drive_num+'A';
                break;
                } // end if.

        cur_dpb=cur_dpb->next_dpb;
        } // end while.

    for (dlist=dlistroot;dlist!=NULL;dlist=dlist->next)
        {
        if ((dlist->attr & 0x8000) == 0)
            dlist->devname[0]='\0';

        if (dlist->drive_num)
            {
            if (dlist->drive_num == 1)
                sprintf(dlist->devname, "%c:", dlist->drives[0]);
            else
                sprintf(dlist->devname, "%c: - %c:", dlist->drives[0],
                        dlist->drives[dlist->drive_num-1]);

            } // end if.

        } // end for
    
    return dlistroot;
} // end make_dev_list.

/////////////////////////////////////////////////////////////////////////////

static void normal_list(char *argv0)
{
    char EnvString[ENVSTRLEN];
    char buffer[NUMLEN], xms_buffer[NUMLEN], ems_buffer[NUMLEN], xms_lbuffer[NUMLEN],
         xms_ubuffer[NUMLEN], bytes_free[NUMLEN], upper_buffer[NUMLEN];
    ushort i, num;
    unsigned memory, memused;
    long temp;
    XMSINFO *xms;
    EMSINFO *ems;
    UPPERINFO *upper = calloc(1, sizeof(UPPERINFO));

    ems=check_ems();
    xms=check_xms();
    make_mcb_list(upper);
    get_os(EnvString);
    memory=biosmemory();
    convert((xms->free+512)/1024, xms_buffer);
    convert((xms->total+512)/1024, xms_lbuffer);
    convert((xms->total+512)/1024-(xms->free+512)/1024, xms_ubuffer);
    convert(upper->largest, upper_buffer);
    i = strlen(argv0);
    while (argv0[i] != '\\' && argv0[i] != '.' && argv0[i] != ':' && i>0) i--;
    if (argv0[i] == '.') {
        argv0[i] = '\0';
        while (argv0[i] != '\\' && argv0[i] != '.' && argv0[i] != ':' && i>0) i--;
    }
    if (i>0) i++;
    temp=convmemfree(&argv0[i]);
    upper->free=(upper->free+512)/1024;
    upper->largest=(upper->largest+512)/1024;
    upper->total=(upper->total+512)/1024;
    memused=memory - (temp+512)/1024;
    printf("\nMemory Type        Total        Used        Free\n"
	   "----------------  --------   ---------    --------\n"
           "Conventional %12uK %10uK %10luK\n", memory, memused, (temp+512)/1024);
    printf("Upper        %12luK %10luK %10luK\n"
           "Extended (XMS) %10sK %10sK %10sK\n"
	   "----------------  --------   ---------    --------\n\n", upper->total, upper->total-upper->free, upper->free, xms_lbuffer, xms_ubuffer, xms_buffer);

    convert(ems->size, buffer);
    printf("Total Expanded (EMS) %18luM (%s bytes)\n", (ems->size+1024*512)/2048/512, buffer);
    convert(ems->free, ems_buffer);
    printf("Free Expanded (EMS)  %18luM (%s bytes)\n\n", (ems->free+1024*512)/2048/512, ems_buffer);
    convert(temp, bytes_free);
    printf("Largest executable program size %8luK (%s bytes)\n", (temp+512)/1024, bytes_free);
    printf("Largest free upper memory block %8luK (%s bytes)\n", upper->largest, upper_buffer);
    if (xms->hma)
        printf("%s: the high memory area exists.\n", EnvString);
    else
        printf("%s: the high memory area does not exist.\n", EnvString);
        
} // end normal_list.


/////////////////////////////////////////////////////////////////////////////

static void print_entry(MINFO *entry)
{
    static uchar *typenames[]= { "", "system code", "system data",
                                 "program", "device driver", "environment",
                                 "data area", "free", };

    printf("  %04x   %6lu   %-9s   %-13s\n",
           entry->seg, entry->size, entry->name, typenames[entry->type]);
}

/////////////////////////////////////////////////////////////////////////////

static void upper_list()
{
    MINFO *ml;
    UPPERINFO *upper = calloc(1, sizeof(UPPERINFO));

    for (ml=make_mcb_list(upper);ml!=NULL;ml=ml->next)
	if ((ml->type == MT_SYSCODE) || (ml->type == MT_SYSDATA) ||
            (ml->type == MT_FREE) || (ml->type == MT_PROGRAM) ||
            (ml->type == MT_DEVICE))
                print_entry(ml);
} // end upper_list.

/////////////////////////////////////////////////////////////////////////////

static void full_list()
{
    MINFO *ml;
    UPPERINFO *upper = calloc(1, sizeof(UPPERINFO));
    
    puts("segment   size       name         type");
    puts("------- --------  ----------  -------------");
    for (ml=make_mcb_list(upper);ml!=NULL;ml=ml->next)
        if (ml->type != MT_MAP)
            print_entry(ml);

} // end full_list.

/////////////////////////////////////////////////////////////////////////////

static void device_list()
{
    DEVINFO *dl;
    UPPERINFO *upper = calloc(1, sizeof(UPPERINFO));

    puts("   address     attr    name       program   ");
    puts(" -----------  ------ ----------  ---------- ");
        //  XXXX:XXXX    XXXX   XXXXXXXX    XXXXXXXX
    for (dl=make_dev_list(make_mcb_list(upper));dl!=NULL;dl=dl->next)
         printf("  %Fp   %04x   %-8s   %-8s  \n", dl->addr, dl->attr,
                dl->devname, dl->progname);

} // end device_list.

/////////////////////////////////////////////////////////////////////////////

static void ems_list()
{
    EMSINFO *ems;
    ushort i;
    uchar handlename[9];
    union REGS regs;
    struct SREGS sregs;
    char format[] = "  %-20s";
    
    ems=check_ems();
    if (!ems->installed)
        {
        puts("  EMS driver not installed in system.  ");
        } // end if.
    else
	{
        printf(format, "EMS driver version");
        printf("%1i.%1i\n", (int)ems->vermajor, (int)ems->verminor);
        printf(format, "EMS page frame");
        printf("%04X\n", ems->frame);
        printf(format, "Total EMS memory");
        printf("%lu bytes\n", ems->size);
        printf(format, "Free EMS memory");
        printf("%lu bytes\n", ems->free);
        printf(format, "Total handles");
        printf("%u\n", ems->totalhandle);
        printf(format, "Free handles");
        printf("%u\n", ems->freehandle);

        puts("                                       ");
        puts("  Handle   Pages   Size      Name     ");
        puts(" -------- ------  ------   ---------- ");
        for (i=0;i<ems->usehandle;i++)
            {
            memset(handlename, 0, sizeof(handlename));
            if (ems->vermajor >= 4)
                {
                if (ems->handles[i].handle == 0)
                    {
                    strcpy(handlename, "SYSTEM");
                    } // end if.
                else
		    {
                    regs.x.dx=ems->handles[i].handle;
                    sregs.es=FP_SEG(&handlename);
                    regs.x.di=FP_OFF(&handlename);
                    regs.x.ax=0x5300;
                    int86x(0x67, &regs, &regs, &sregs);
                    } // end else.

                strupr(handlename);
                } // end if.

            printf("     %-7u %-6u %-9lu %-9s  \n", ems->handles[i].handle, ems->handles[i].pages, (ulong)ems->handles[i].pages * 16384L, handlename);
            } // end for.

        } // end else.

} // end ems_list.

/////////////////////////////////////////////////////////////////////////////

static void xms_list()
{
    UPPERINFO *upper = calloc(1, sizeof(UPPERINFO));
    XMSINFO *xms = calloc(1, sizeof(XMSINFO));
    ushort i;
    char format[] = "%-26s";
    union REGS regs;
    uchar free_handles_tmp;
    void far (*xms_drv)(void);

    xms=check_xms();
    make_mcb_list(upper);
    if (!xms->installed)
        {
        puts("XMS driver not installed in system.");
        return;
        } // end if.

    printf("Testing XMS memory ...\n");

    xms_drv = xms->drv;

    _AH=7;
    (*xms_drv)();
    regs.h.bl = _BL;
    xms->a20=_AL;
    if (regs.h.bl!=0) {
        puts("XMS INTERNAL ERROR.");
        exit(1);
    }

    memset(xms->handles, 0, sizeof(xms->handles));
    xms->usehandle=0;

    for (i=0;i<65535;i++)
        {
        _AH=0xE;
        _DX=i;
        (*xms_drv)();
        if (_AX == 0) continue;
        regs.h.bh=_BH;
        regs.x.dx=_DX;
        free_handles_tmp = _BL;
        if (xms->usehandle < MAX_HANDLE)
            {
            xms->handles[xms->usehandle].handle=i;
            xms->handles[xms->usehandle].size=regs.x.dx*1024L;
            xms->handles[xms->usehandle].locks=regs.h.bh;
            xms->total += xms->handles[xms->usehandle].size;
            xms->usehandle++;
            } // end if.

            if (xms->freehandle < free_handles_tmp)
                {
                xms->freehandle = free_handles_tmp;
                }
        } // end for.

        /* First try to get a handle of our own. */
        _AH = 9;
        _DX = 1; /* First we try 1kB. I'm not sure if XMS driver
                    must support a zero sized allocate. */
        (*xms_drv)();
        i = _DX;
        if (_AX==0) {
            /* Now try a zero sized allocate just in case there was no free memory. */
            _AH = 9;
            _DX = 0;
            (*xms_drv)();
            /* If AX==0, nothing worked out. Use whatever we got from the loop above. */
            /* We can't do much if the free call fails, so it ends here. */
        }
        if (_AX!=0) {
            _AH = 0xe;
            (*xms_drv)();
            regs.h.bl = _BL;
            if (_AX!=0)
                /* Hey! We got some info. Put it in a safe place. */
                xms->freehandle = regs.h.bl + 1;
                /* Add one for the handle we have allocated. */
            _DX = i;
            _AH = 0xa;
            (*xms_drv)();
        }
            
        printf(format, "XMS driver version");
        printf("%i.%i\n", (ushort)xms->vermajor, (ushort)xms->verminor);
        printf(format, "HMA state");
        printf("%s\n", (xms->hma) ? "exists" : "not exists");
        printf(format, "A20 line state");
        printf("%s\n", (xms->a20) ? "enabled" : "disabled");
        printf(format, "Free XMS memory");
        printf("%lu bytes\n", xms->free);
        printf(format, "Largest free XMS block");
        printf("%lu bytes\n", xms->largest);
        printf(format, "Free handles");
        printf("%u\n\n", xms->freehandle);
        if (xms->usehandle)
            {
            puts(" Block   Handle   Size       Locks");
            puts("------- -------- ------    ---------");
            for (i=0;i<xms->usehandle;i++)
                printf("   %-6u %-7u %-9lu %-12u\n", i, xms->handles[i].handle,
                       xms->handles[i].size, xms->handles[i].locks);

            } // end if.

        puts("                                          ");
        if (upper->installed)
            {
            printf(format, "Free upper memory");
            printf("%lu bytes\n", upper->free);
            printf(format, "Largest upper block");
            printf("%lu bytes\n", upper->largest);
            } // end if.
        else
            {
            puts("Upper memory            not available   ");
            } // end else.

} // end xms_list.

/////////////////////////////////////////////////////////////////////////////

int classify_args(int narg, char *rawargs[], char *fileargs[], char *optargs[])
{
    int index, jndex, kndex;
    char *argptr;

    for (index=0,jndex=0,kndex=0;index<narg;index++)
        {
        argptr = rawargs[index];
        if (*argptr == '/')
            {
            argptr++;
            optargs[kndex++] = argptr;
            } // end if.
        else
            {
            fileargs[jndex++] = argptr;
            } // end else.
            
        } // end for.

   return kndex;

} // end classify_args.

/////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    char *fileargs[64], *optargs[64];
    int n_options, n_files, index, help_flag=0;

    n_options = classify_args(argc, argv, fileargs, optargs);
    for (index=0;index<n_options;index++)
        switch(toupper(optargs[index][0])) {
        case '?':
            help_flag=1;
            break;
        case 'U':
            upper_list();
            break;
        case 'F':
            full_list();
            break;
        case 'D':
            device_list();
            break;
        case 'E':
            ems_list();
            break;
        case 'X':
            xms_list();
            break;
        default:
            printf("Invalid parameter - /%s\n", strupr(optargs[index]));
            exit(1);
        } // end switch.
        
    if (help_flag)
        {
	printf("Displays the amount of used and free memory in your system.\n\n"
               "Syntax: MEM [/E] [/F] [/D] [/U] [/X] [/?]\n"
               "  /E  Reports all information about Expanded Memory\n"
               "  /F  Full list of memory blocks\n"
               "  /D  List of device drivers currently in memory\n"
               "  /U  List of programs in conventional and upper memory\n"
               "  /X  Reports all information about Extended Memory\n"
               "  /?  Displays this help message\n");
        exit(1);
        } // end if.

    normal_list(argv[0]);
    return 0;

} // end main.
