/* $Id: inst.c,v 1.5 2000/01/04 03:46:19 jhall Exp $ */

/* Functions to install a disk series, or a package */

/* Copyright (C) 1998 Jim Hall, jhall1@isd.net */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>			/* for system(), free() */

#ifdef unix
#include "conio.h"
#else
#include <conio.h>
#endif /* unix */

#include "bargraph.h"			/* for bargraph() */
#include "dat.h"
#include "dir.h"
#include "getch_yn.h"
#include "inst.h"
#include "isfile.h"
#include "lsm.h"			/* Linux LSM files */
#include "repaint.h"
#include "unz.h"			/* for UzpMain() */


#define STRLEN 80


inst_t
set_install (const char *diskset, char *fromdir, char *destdir)
{
  /* Variables */

  inst_t ret;				/* return: no. of errors,warnings */
  inst_t this;				/* no. of errors,warnings */
  char endfile[STRLEN];			/* marks end of series */
  char descfile[STRLEN];		/* description file */
  char datfile[STRLEN];			/* current DAT file */
  int disknum = 0;			/* current disk number */
  int ch;

  /* Create the filenames */

  sprintf (endfile, "%s%c%s.end", fromdir, DIR_CHAR, diskset);
  sprintf (descfile, "%s%c%s.txt", fromdir, DIR_CHAR, diskset);

  /* Print the name of the series we are working on */

  repaint_empty();
  gotoxy (1, 3);
  cputs ("Installing series: ");
  gotoxy (1, 4);
  cputs (diskset);

  /* Install while we have disks to work from.  Since we will reach an
     exit condition within the loop, we use an infinite loop here. */

  ret.errors = 0;
  ret.warnings = 0;

  while (1) {
    /* Set the DAT file name */

    sprintf (datfile, "%s%c%s.%d", fromdir, DIR_CHAR, diskset, ++disknum);

    /* Load the first disk */

    /* First check that the datfile exists.  If it doesn't, check if
       the endfile was found. */

    if (!isfile (datfile)) {
      /* Does the endfile exist? */

      if (isfile (endfile)) {
	repaint_empty();
	gotoxy (1, 10);
	cputs ("Done installing this disk series.");
	gotoxy (1, 15);
	cputs ("If you are installing other disk series, please insert");
	gotoxy (1, 16);
	cputs ("disk #1 of the next series in the drive now.");

	gotoxy (1, 25);
	cputs ("Press any key to continue");
	getch();

	return (ret);
      }

      /* The endfile was not found, so we know there is at least one
         more disk left to do.  Keep asking the user to insert the
         next disk. */

      do {
	gotoxy (1, 10);
	cputs ("Can't find data file for this install disk!");
	gotoxy (1, 11);
	cputs (datfile);

	gotoxy (1, 15);
	cputs ("You may not have the right install floppy in the drive.");
	gotoxy (1, 16);
	cputs ("Double check that you have the right disk and try again.");

	gotoxy (1, 25);
	cputs ("Continue installing this disk? [yn]");

	ch = getch_yn();

	if ((ch != 'y') && (ch != 'Y'))
	  {
	    /* user has decided to quit this series */
	    return (ret);
	  }
      } while (!isfile (datfile));

    } /* if no datfile */

    /* Install files from this disk */

    this = disk_install (datfile, descfile, fromdir, destdir);
    ret.errors += this.errors;
    ret.warnings += this.warnings;
  } /* while (1) */
}

inst_t
disk_install(const char *datfile, const char *descfile,
	     char *fromdir, char *destdir)
{
  dat_t *dat_ary;			/* the DAT array */
  int dat_size = 30;			/* malloc size of the dat array */
  int dat_count;			/* size of the dat array */

  inst_t this;				/* return: no. of errors,warnings */
  char lsmfile[STRLEN];			/* Linux software map file */
  int ret;
  int ch;
  int i;

  /* Initialize variables */

  this.errors = 0;
  this.warnings = 0;

  /* Read dat file */

  dat_ary = malloc (sizeof (dat_t) * dat_size);
  if (dat_ary == NULL)
    {
      fprintf (stderr, "Error!\n");
      fprintf (stderr, "Unable to allocate enough memory for install floppy data file!\n");
      gotoxy (1, 25);
      cputs ("Press any key to continue");
      getch();

      return (this);
    }

  dat_count = dat_read (datfile, dat_ary, dat_size);
  if (dat_count < 1)
    {
      fprintf (stderr, "Error!\n");
      fprintf (stderr, "The install floppy data file is empty!\n");
      gotoxy (1, 25);
      cputs ("Press any key to continue");
      getch();

      free (dat_ary);
      return (this);
    }

  /* Run the install */

  for (i = 0; i < dat_count; i++) {
    /* Generate the lsmfile name */

    sprintf (lsmfile, "%s%c%s.lsm", fromdir, DIR_CHAR, dat_ary[i].name);

    /* Print the screen and progress bargraph */

    repaint_empty();
    gotoxy (15, 20);
    bargraph (i, dat_count, 50 /* width */);

    /* Print the package name */

    gotoxy (1, 5);
    cputs ("Package: ");
    cputs (dat_ary[i].name);

    /* Show the package description */

    lsm_description (8, 1, 10, lsmfile);

    /* Find out which ones the user wants to install */

    gotoxy (1, 23);
    switch (dat_ary[i].rank) {
    case 'n':
    case 'N':
      /* Do not install */

      cputs ("SKIPPED");
      break;

    case 'y':
    case 'Y':
      /* Always install */

      cputs ("REQUIRED");

      ret = unzip_file (dat_ary[i].name, fromdir, destdir);

      if (ret != 0) {
	/* Print an error message */

	cputs ("ERROR!  Failed to install REQUIRED package.");

	/* Return an error */

	this.errors++;

	/* Does the user want to continue anyway? */

	gotoxy (1, 25);
	cputs ("Continue installing this disk? [yn]");

	ch = getch_yn();

	if ((ch != 'y') && (ch != 'Y'))
	  {
	    return (this);
	  }
      }
      break;

    default:
      /* Optional */

      cputs ("OPTIONAL");

      /* Ask the user if you want to install it */

      gotoxy (1, 23);
      cputs ("Install this package? [yn]");

      ch = getch_yn();

      if ((ch == 'y') || (ch == 'Y'))
	{
	  ret = unzip_file (dat_ary[i].name, fromdir, destdir);

	  if (ret != 0)
	    {
	      /* Print a warning message */

	      gotoxy (1, 23);
	      cputs ("WARNING!  Failed to install OPTIONAL package.");

	      gotoxy (1, 25);
	      cputs ("Press any key to continue");
	      getch();

	      this.warnings++;
	    }
	}
      break;

    } /* switch */
  } /* for */

  /* Free memory for this disk */

  free (dat_ary);
  return (this);
}
