/* $Id: inst.c,v 1.2 1999/02/15 00:28:45 jhall Exp $ */

/* Functions to install a disk series, or a package */

/* Copyright (C) 1998 Jim Hall, jhall1@isd.net */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>			/* for system(), free() */

#ifdef unix
#include "conio.h"
#else
#include <conio.h>
#endif /* unix */

#include "dat.h"
#include "dir.h"
#include "getch_yn.h"
#include "inst.h"
#include "isfile.h"
#include "lsm.h"			/* Linux LSM files */
#include "unz.h"			/* for UzpMain() */


#define STRLEN 80


inst_t
set_install (const char *diskset, char *fromdir, char *destdir)
{
  /* Variables */

  inst_t ret;				/* return: no. of errors,warnings */
  inst_t this;				/* no. of errors,warnings */
  char endfile[STRLEN];			/* marks end of series */
  char descfile[STRLEN];		/* description file */
  char datfile[STRLEN];			/* current DAT file */
  int disknum = 0;			/* current disk number */
  int ch;

  /* Create the filenames */

  sprintf (endfile, "%s%c%s.END", fromdir, DIR_CHAR, diskset);
  sprintf (descfile, "%s%c%s.TXT", fromdir, DIR_CHAR, diskset);

  /* Print the name of the series we are working on */

  clrscr();
  gotoxy (10, 10);
  cputs ("Installing series: ");
  cputs (diskset);

  /* Install while we have disks to work from.  Since we will reach an
     exit condition within the loop, we use an infinite loop here. */

  ret.errors = 0;
  ret.warnings = 0;

  while (1) {
    /* Set the DAT file name */

    sprintf (datfile, "%s%c%s.%d", fromdir, DIR_CHAR, diskset, ++disknum);

    /* Load the first disk */

    /* First check that the datfile exists.  If it doesn't, check if
       the endfile was found. */

    if (!isfile (datfile)) {
      /* Does the endfile exist? */

      if (isfile (endfile)) {
	gotoxy (1, 24);
	cputs ("Found the end file: ");
	cputs (endfile);

	gotoxy (1, 25);
	cputs ("Press any key to continue");
	getch();

	return (ret);
      }

      /* The endfile was not found, so we know there is at least one
         more disk left to do.  Keep asking the user to insert the
         next disk. */

      do {
	gotoxy (1, 24);
	cputs ("Can't find dat file: ");
	cputs (datfile);

	gotoxy (1, 25);
	cputs ("Continue installing this disk? [yn]");
#if 0
	cputs ("Press `y\' to continue, or any other key to quit");
#endif /* disabled code */

	ch = getch_yn();

	if ((ch != 'y') && (ch != 'Y'))
	  {
	    /* user has decided to quit this series */
	    return (ret);
	  }
      } while (!isfile (datfile));

    } /* if no datfile */

    /* Install files from this disk */

    this = disk_install (datfile, descfile, fromdir, destdir);
    ret.errors += this.errors;
    ret.warnings += this.warnings;
  } /* while (1) */
}

inst_t
disk_install(const char *datfile, const char *descfile,
	     char *fromdir, char *destdir)
{
  dat_t *dat_ary;			/* the DAT array */
  int dat_size = 30;			/* malloc size of the dat array */
  int dat_count;			/* size of the dat array */

  inst_t this;				/* return: no. of errors,warnings */
  char lsmfile[STRLEN];			/* Linux software map file */
  int ret;
  int ch;
  int i;

  /* Initialize variables */

  this.errors = 0;
  this.warnings = 0;

  /* Read dat file */

  dat_ary = malloc (sizeof (dat_t) * dat_size);
  if (dat_ary == NULL)
    {
      fprintf (stderr, "Error!  Unable to allocate memory for dat file!\n");
      gotoxy (1, 25);
      cputs ("Press any key to continue");
      getch();

      return (this);
    }

  dat_count = dat_read (datfile, dat_ary, dat_size);
  if (dat_count < 1)
    {
      fprintf (stderr, "Error!  Could not read dat file!\n");
      gotoxy (1, 25);
      cputs ("Press any key to continue");
      getch();

      free (dat_ary);
      return (this);
    }

  /* Run the install */

  for (i = 0; i < dat_count; i++) {
    /* Generate the lsmfile name */

    sprintf (lsmfile, "%s%c%s.LSM", fromdir, DIR_CHAR, dat_ary[i].name);

    /* Print the package name */

    clrscr();
    gotoxy (1, 1);
    cputs ("Package: ");
    cputs (dat_ary[i].name);

    /* Show the package description */

    lsm_description (8, 1, 10, lsmfile);

    /* Find out which ones the user wants to install */

    gotoxy (1, 2);
    switch (dat_ary[i].rank) {
    case 'n':
    case 'N':
      /* Do not install */

      cputs ("SKIPPED");
      break;

    case 'y':
    case 'Y':
      /* Always install */

      cputs ("REQUIRED");

      ret = unzip_file (dat_ary[i].name, fromdir, destdir);

      if (ret != 0) {
	/* Print an error message */

	gotoxy (1, 24);
	cputs ("ERROR!  Failed to install REQUIRED package.");

	/* Return an error */

	this.errors++;

	/* Does the user want to continue anyway? */

	gotoxy (1, 25);
	cputs ("Continue installing this disk? [yn]");

	ch = getch_yn();

	if ((ch != 'y') && (ch != 'Y'))
	  {
	    return (this);
	  }
      }
      break;

    default:
      /* Optional */

      cputs ("OPTIONAL");

      /* Ask the user if you want to install it */

      gotoxy (1, 24);
      cputs ("Install this package? [yn]");

      ch = getch_yn();

      if ((ch == 'y') || (ch == 'Y'))
	{
	  ret = unzip_file (dat_ary[i].name, fromdir, destdir);

	  if (ret != 0)
	    {
	      /* Print a warning message */

	      gotoxy (1, 24);
	      cputs ("WARNING!  Failed to install OPTIONAL package.");

	      gotoxy (1, 25);
	      cputs ("Press any key to continue");
	      getch();

	      this.warnings++;
	    }
	}
      break;

    } /* switch */
  } /* for */

  /* Free memory for this disk */

  free (dat_ary);
  return (this);
}
