#define _INST_C

/* $Id: inst.c,v 1.20 1998/08/18 03:15:43 jhall Exp $ */

/* Functions to install a disk series, or a package */

/* Note to self: package info file are introduced in version 2.0.
   This will allow me to add package management in verison 4.0.  The
   text description files (.txt files) are gone as of version 3.0. */

/* Copyright (C) 1998 Jim Hall, jhall1@isd.net */

/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdio.h>
#include <stdlib.h>				/* for system(), free() */

#include "dat.h"
#include "file_io.h"
#include "fs.h"
#include "pause.h"

#include "dsc.h"				/* Debian dsc files */
#include "lsm.h"				/* Linux LSM files */
#include "rph.h"				/* Red Hat RPH files */

#include "unzip.h"				/* for UzpMain() */


/* Symbolic constants */

#define STRLEN 80


/* Globals */

unsigned g_errs = 0;				/* no. total errors */
unsigned g_warns = 0;				/* no. total warnings */


/* Functions */

/* Installs a package file into dir "exdir" using "unzip".  Returns
   the exit value of the archive extractor program, or -1 if the file
   is not found. Files are located in the path "from" */

int
instpkg (const char *pkg, const char *from, char *exdir)
{
  char zipfile[STRLEN];				/* name of a zipfile */
  int ret;

  char *argv[7] = {"instpkg()", "-qq", "-o", "_zipfile_", "-d", "_dir_", NULL};
  						/* set 3=zipfile, 5=dir */

  /* Create the zip file name - .ZIP extension not needed as Unzip
     will add it if not there.  Also makes this more flexible for
     non-zip formats. */

  sprintf (zipfile, "%s%c%s", from, FS_DIR_SEP, pkg);

  /* "-d exdir" extracs into directory exdir.  "-o" forces overwrite
     of same files. */

  /* I chose NOT to use -q or -qq in versions 1.x because it may help
     the user to see these files being extracted, so they know the
     system isn't hung.  Especially if we ever have a 1MB package. */

  /* In version 3.0, I will use -qq to make it very quiet, so my
     screen i/o won't get messed . */

  /* set 3=zipfile, 5=dir */

  argv[3] = zipfile;				/* pointer assignment */
  argv[5] = exdir;				/* pointer assignment */
  ret = UzpMain (7, argv);			/* the Unzip code */

  /* Done */

  return (ret);
}

/* Installs all files from the named datfile, using the descfile to
   print descriptions.  Returns the number of files installed, or 0 if
   an error occurred and we need to abort. */

int
instdisk (const char *datfile, const char *descfile,
	  char *from, char *exdir)
{
  dat_t *thisdisk;				/* the DAT array */
  int nfiles;					/* no. files installed */
  int ret;
  int i;
  unsigned num_errors;				/* no. of fatal errors */
  unsigned num_warns;				/* no. of nonfatal warnings */
  char rphfile[STRLEN];				/* Red Hat Package Header */
  char dscfile[STRLEN];				/* Debian description file */
  char lsmfile[STRLEN];				/* Linux software map file */

  /* Read the package list */

  if (!(thisdisk = dat_read (datfile))) {
    /* Failed */

    fprintf (stderr, "install: Can't find %s\n", datfile);
    g_errs++;
    return (0);
  }

  nfiles = dat_ary_size;

  /* Run the installation. */

  num_errors = 0;
  num_warns = 0;

  for (i = 0; i < nfiles; i++) {
    /* Generate the rphfile name */

    sprintf (rphfile, "%s%c%s.rph", from, FS_DIR_SEP, thisdisk[i].fname);
    sprintf (dscfile, "%s%c%s.dsc", from, FS_DIR_SEP, thisdisk[i].fname);
    sprintf (lsmfile, "%s%c%s.lsm", from, FS_DIR_SEP, thisdisk[i].fname);

    /* Show the description */

    printf ("\n\nPackage: %s\n", thisdisk[i].fname);
    if (rph_description (rphfile) < 1) {

      /* If not found, print the debian description */
      if (dsc_description (dscfile) < 1) {

	/* If not found, print the LSM file description */
	if (lsm_description (lsmfile) < 1) {

	  /* At this point, there is no description for the package */
	  printf ("(no description for this package)\n");

	} /* lsm */
      } /* dsc */
    } /* rph */

    /* Find out which ones the user wants to install */

    switch (thisdisk[i].rank) {
    case 'n':
    case 'N':
      /* Do not install */

      printf ("%s [SKIPPED]\n", thisdisk[i].fname);
      break;

    case 'y':
    case 'Y':
      /* Always install */

      printf ("%s [REQUIRED]\n", thisdisk[i].fname);
      ret = instpkg (thisdisk[i].fname, from, exdir);

      if (ret != 0) {
	fprintf (stderr, "Error: failed to install REQUIRED package!\n");
	g_errs++;
	num_errors++;

	/* Does the user want to continue anyway? */

	printf ("\aContinue installing this disk?\n");
	if (!pause_yn (1)) {
	  return (0);
	}
      }
      break;

    default:
      /* Optional */

      printf ("%s [OPTIONAL]\n", thisdisk[i].fname);

      /* Ask the user if you want to install it */

      printf ("\aInstall package %s? [yn] ", thisdisk[i].fname);
      if (pause_yn (0)) {
	ret = instpkg (thisdisk[i].fname, from, exdir);

	if (ret != 0) {
	  fprintf (stderr, "Warning: failed to install OPTIONAL package\n");
	  g_warns++;
	  num_warns++;

	  /* Continue anyway.  This is non-fatal. */

	  printf ("Continuing anyway..\n");
	}
      }
      break;

    } /* switch */
  } /* for */

  /* Free memory for this disk */

  free (thisdisk);

  /* Print summary information and return */

  if ((num_errors > 0) || (num_warns > 0)) {
    fprintf (stderr, "\n\n");;
    fprintf (stderr, "There were %u warnings and %u errors with this disk\n",
	     num_warns, num_errors);
  }

  return (nfiles);
}

/* Installs a disk series into dir "exdir", using the entry from the
   DAT file.  Files are located in the path "from".  Returns the
   number of disks installed from the series.  We know the series is
   finished when we find 'endfile' */

int
instser (const char *basename, char *from, char *exdir)
{
  /* Variables */

  char endfile[STRLEN];				/* marks end of series */
  char descfile[STRLEN];			/* description file */
  char datfile[STRLEN];				/* current DAT file */
  int ndisks = 0;				/* no. disks installed */

  /* Create the filenames */

  sprintf (endfile, "%s%c%s.END", from, FS_DIR_SEP, basename);
  sprintf (descfile, "%s%c%s.TXT", from, FS_DIR_SEP, basename);

  /* Say hello */

  printf ("\n\n");
  printf (">> Installing series %s <<\n", basename);

  /* Install while we have disks to work from.  Since we will reach an
     exit condition within the loop, we use an infinite loop here. */

  while (1) {
    /* Set the DAT file name */

    sprintf (datfile, "%s%c%s.%d", from, FS_DIR_SEP, basename, ++ndisks);

    /* Load the first disk */

    /* First check that the datfile exists.  If it doesn't check if
       the endfile was found. */

    if (!isfile (datfile)) {
      /* Does the endfile exist? */

      if (isfile (endfile)) {
	printf ("\nendfile %s found\n", endfile);

	return (ndisks);
      }

      /* The endfile was not found, so we know there is at least one
         more disk left to do.  Keep asking the user to insert the
         next disk. */

      do {
	printf ("\n\n");
	printf ("data file %s not found\n", datfile);
	printf ("\aLoad disk \'%s-%d\' in %s\n", basename, ndisks, from);

	if (!pause_yn (1)) {
	  /* user has decided to quit this series */
	  return (ndisks);
	} /* if pause_yn */

      } while (!isfile (datfile));

    } /* if no datfile */

    /* Install files from this disk */

    printf ("\ninstalling \'%s-%d\'\n", basename, ndisks);
    instdisk (datfile, descfile, from, exdir);

  } /* while (1) */
}
