/*  EXPANDVARS - A sample of how to expand shell-style variables
Copyright (C) 1998 Steffen Kaiser

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: expandv.c,v $
$Locker:  $	$Name: VER_2_0 $	$State: Exp $

char *expandVars(char *str, char **keys, char **values, int items)

Expand variables in a string str. The returned string is
the dynamically allocated, expanded string.

Both arrays 'keys' and 'values' must contain at least 'items'
elements.
'keys' are matched case-insensitive.

There are three methods of expansion supported:
UNIXVARS:	Detects variables of the form:	$name or ${name} or $?
DOSVARS:	Detects variables of the form:	%name%
LETTER:	Detects variables of the form: %?

The $name variant takes all letters and/or digits following the
dollar sign. If no alnum() is present, take the next character.
The ${name} and %name% variants also substitute the closing mark.
The %? variant takes _any_ following character as the variable name.

Because no quoting mechanism is implemented, here is how to
insert:
UNIXVARS: a dollar sign: ${}
DOSVARS: a percent sign: %%
LETTERS: a percent sign: %%

Return:
NULL: if malloc() failed
else: expanded string

Target compilers:
+ Borland C v5.2

It compiles under and emits the expected results:
+ Micro-C v3.14

$Log: expandv.c,v $
Revision 1.1  1998/03/10 03:37:30  jhall
Included Steffan Kaiser's EXPANDVARS function.  This is version 1.1, which
includes a fix for non-DOS systems.  It is generally cleaner code, too.

Revision 1.1  1998/02/04 19:14:04  jhall
Initial revision

Revision 1.4  1998/02/01 07:11:50  ska
bugfix: elimited core dump because of modifying (const char) arrays
add: memicmp() function, is stricmp() or strcasecmp()

Revision 1.3  1998/02/01 06:43:46  ska
bugfix: if no variable scan mode is defined, an error is displayed
fix: some typos
chg: #include of global headers to be more ANSI-compliant
chg: local functions app_() scn() are static now

Revision 1.2  1998/01/21 07:48:09  ska
fix: LETTERS mode: at EOS the varible name is empty -> default entry added

Revision 1.1  1998/01/21 07:45:10  ska
Initial revision

*/

#include <stdio.h>

#ifndef _MICROC_
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __GNUC__
#define NO_MEMICMP
#endif

#else		/* _MICROC_ */
#define const
#define size_t int

#endif

/* #define DOSVARS			/* which variable scanning mode? */
/* #define UNIXVARS */

#if !defined(DOSVARS) && !defined(UNIXVARS) && !defined(LETTERS)
#  define LETTERS
#endif					/* default variable scanning mode */


#ifdef RCS_Version
static const char rcsid[] =
"$Id: expandv.c,v 1.1 1998/03/10 03:37:30 jhall Exp $";
#endif

/*	When the variable is not found within the supplied array,
the default array 'dfltKeys' is searched. This way a) default
variables can be defined, and b) the "empty" variable mechanism
is implemented to work as the quoting mechanism. */
#ifdef __VARMODE
#undef __VARMODE
#endif
#ifdef UNIXVARS
#define VARCHAR '$'
static const char *dfltKeys[] = {""};			/* empty variable */
static const char *dfltValues[] = {"$"};			/* the variable sign */
static const int dfltItems = sizeof(dfltKeys) / sizeof(dfltKeys[0]);
#define __VARMODE
#endif

#ifdef DOSVARS
#ifdef __VARMODE
#error "More than one variable mode specified!"
#endif
#define VARCHAR '%'
static const char *dfltKeys[] = {""};			/* empty variable */
static const char *dfltValues[] = {"%"};			/* the variable sign */
static const int dfltItems = sizeof(dfltKeys) / sizeof(dfltKeys[0]);
#define __VARMODE
#endif

#ifdef LETTERS
#ifdef __VARMODE
#error "More than one variable mode specified!"
#endif
#define VARCHAR '%'
static const char *dfltKeys[] = {"%", ""};			/* %%, end of string */
static const char *dfltValues[] = {"%", "%"};		/* the variable sign */ static const int dfltItems = sizeof(dfltKeys) / sizeof(dfltKeys[0]);
#define __VARMODE
#endif

#ifndef __VARMODE
#error "Specify one variable scan mode: DOSVARS, UNIXVARS or LETTERS"
#endif


#ifdef NO_MEMICMP
static int memicmp(const void *s1, const void *s2, size_t length)
{	int d;
register unsigned char *p, *q;

d = 0;
p = (unsigned char*)s1;
q = (unsigned char*)s2;
while(length-- && (d = toupper(*p++) - toupper(*q++)) == 0);

return d;
}
#endif


/*
app(char **buf, size_t *buflen, char *str, size_t size)

append str with length size to dynamically allocated *buf with
length *buflen.
*/
#define APP(a,b,c,d) { if(app_(&(a),&(b),(c),(d))) return NULL; }
static int app_(char **buf, size_t *buflen, const char *str, size_t size) {	char *h;

if(!size) return 0;

if((h = (char*)realloc(*buf, *buflen + size + 1)) == NULL) {
free(*buf);
return 1;
}

*buf = h;
memcpy(&h[*buflen], str, size);
h[*buflen += size] = '\0';
return 0;
}

/*
char *scn(char *var, size_t len, char **keys, char **values, int items)

Searches for the variable var with the length len in the array
keys.
The search is performed case-insensitively.

Return values:
NULL: not found
else: the value of the variable
*/
static char *scn(const char *var, size_t len, const char *keys[]
, const char *values[], int items)
{	char *h;
int i;

h = NULL;			/* default return value */
for(i = 0; i < items; ++i)
if(keys[i][len] == '\0' && memicmp(var, keys[i], len) == 0) {
/* found !! */
h = (char*)values[i];
break;
}

return h;
}

char *expandVars(const char *str		/* string to be expanded */
, const char *keys[]			/* defined variables */
, const char *values[] 		/* their values */
, int items)						/* size of the arrays */
{	const char *p, *q, *h;
char *buf;							/* dynamical buffer */
size_t buflen;						/* its size */
size_t l;

buf = NULL;
buflen = 0;
p = str - 1;
while((p = strchr(q = p + 1, VARCHAR)) != NULL) {
/* variable sign found */

/* duplicate the string before the sign */
APP(buf, buflen, q, p - q)

/* extract the variable name */
/* q will point to the start of the name,
l contains the length of the name,
p will point to the end of the name */
if(!*(q = ++p)) {		/* end of string -> empty variable name */
--p;
l = 0;
}
else {				/* non-empty name so far */
#ifdef LETTERS	/* the immediately following character */
l = 1;
#endif	/* LETTERS */
#ifdef DOSVARS
p = strchr(p, VARCHAR);
if(!p) {			/* ill-formed variable -> take all the line */
l = strlen(q);
p = q + l - 1;
}
else {
l = p - q;
}
#endif	/* DOSVARS */
#ifdef UNIXVARS
/* check for parenthized name */
if(*q == '{') {			/* yes, take all til '}' */
p = strchr(++q, '}');
if(!p) {		/* ill-formed variable -> take all the line */
l = strlen(q);
p = q + l - 1;
}
else {
l = p - q;
}
}
else {					/* take all alnum's */
while(isalnum(*p)) ++p;
if(p == q)
l = 1;
else
l = p-- - q;
}
#endif	/* UNIXVARS */
}

/* Now p, q,& l are updated -> search for the name */
if((h = scn(q, l, keys, values, items)) == NULL	/* not found */
&& (h = scn(q, l, dfltKeys, dfltValues, dfltItems)) == NULL)
h = "";		/* no found at all -> ignore */

APP(buf, buflen, h, strlen(h))
}

APP(buf, buflen, q, strlen(q))

return buf;
}

#ifdef TEST
#ifdef LETTERS
char *k1[] = {"!", "%", "q", "Never accessable because key is too long", ""};

#else
char *k1[] = {"var1", "", "var2", "var3", "|", "Never reached, because 'items' too small"};
#endif
char *v1[] = {"VALUE1", "Because this valus is modified, you don't see any '%' here", "VALUE2", "VALUE3", "VALUE4"}; int i1 = 5;

main(void)
{
char string[] = "unzip -o %d %f";
char *k[] = {"d", "f"};
char *v[] = {"/opt/freedos", "cls.zip"};
int size = 2;

printf ("%s:\n", string);
printf ("  %s\n", expandVars (string, k, v, size));
printf(">>%s\n", expandVars("var !: %!\nno var: %%\n", k1, v1, i1));
}
#endif
