#define _INST_C

/* Functions to install a disk series, or a package */

/* Copyright (C) 1997, James Hall <jhall1@isd.net> */

/* This is free software.  See the file COPYING for details */

#include <stdio.h>
#include <stdlib.h>				/* for system(), free() */

#include "dat.h"
#include "expandv.h"
#include "file_io.h"
#include "fs.h"
#include "pause.h"
#include "showdesc.h"

/* Symbolic constants */

#define STRLEN 256


/* Globals */

unsigned g_errs = 0;				/* no. total errors */
unsigned g_warns = 0;				/* no. total warnings */

char *g_macro = (char *) NULL;			/* external de-archiver */


/* Functions */

/* Installs a package file into dir "exdir" using "unzip".  Returns
   the exit value of the archive extractor program, or -1 if the file
   is not found. Files are located in the path "from" */

int
instpkg (const char *pkg, const char *from, const char *exdir)
{
  char cmd[STRLEN];				/* the system() string */
  char zipfile[STRLEN];				/* name of a zipfile */
  char *z = zipfile;
  int ret;

  char *key[] = {"d",   "f"    };		/* support for expandVars */
  char *val[] = {exdir,  z     };		/* support for expandVars */
  int size = 2;					/* support for expandVars */


  /* Create the zip file name - is it really necessary to append the
     .ZIP extension?  Maybe a later version of the program could read
     the full name (basename plus .ZIP extension) from the DAT
     file. */

  sprintf (zipfile, "%s%c%s.ZIP", from, FS_DIR_SEP, pkg);

  /* Check if the file is there */

  if (!isfile (zipfile)) {
    /* Not there - return failure */

    printf ("install: %s not found\n", zipfile);
    g_errs++;
    return (-1);
  }

  /* Add support for Steffan's expandVars, to support using a
     third-party de-archiver program. */

  if (g_macro != (char *) NULL) {
    /* g_macro is a global flag telling me if I should use expandVars
       to create an exec string.  if non NULL, use the supplied
       method.  value is read from an EXTRACT.DAT file.  macro should
       look like this: "unfoo --extract-to %d %f" */

    ret = system (expandVars (g_macro, key, val, size));
  }

  else {
    /* "-d exdir" extracs into directory exdir.  "-o" forces overwrite
       of same files. */

    /* I chose NOT to use -q or -qq because it may help the user to
       see these files being extracted, so they know the system isn't
       hung.  Especially if we ever have a 1MB package. */

    /* In version 2.0, I will call the unzip code directly. */

    sprintf (cmd, "unzip -o %s -d%s", zipfile, exdir);
    ret = system (cmd);
  }

  /* Done */

  return (ret);
}

/* Installs all files from the named datfile, using the descfile to
   print descriptions.  Returns the number of files installed, or 0 if
   an error occurred and we need to abort. */

int
instdisk (const char *datfile, const char *descfile, const char *from,
	  const char *exdir)
{
  dat_t *thisdisk;				/* the DAT array */
  int nfiles;					/* no. files installed */
  int ret;
  int i;
  unsigned num_errors;				/* no. of fatal errors */
  unsigned num_warns;				/* no. of nonfatal warnings */

  /* Read the package list */

  if (!(thisdisk = dat_read (datfile))) {
    /* Failed */

    fprintf (stderr, "install: Can't find %s\n", datfile);
    g_errs++;
    return (0);
  }

  nfiles = dat_ary_size;

  /* Run the installation. */

  num_errors = 0;
  num_warns = 0;

  for (i = 0; i < nfiles; i++) {
    /* Show the description */

    printf ("\n\n");
    showdesc (thisdisk[i].fname, descfile);

    /* Find out which ones the user wants to install */

    switch (thisdisk[i].rank) {
    case 'n':
    case 'N':
      /* Do not install */

      printf ("%s [SKIPPED]\n", thisdisk[i].fname);
      break;

    case 'y':
    case 'Y':
      /* Always install */

      printf ("%s [REQUIRED]\n", thisdisk[i].fname);
      ret = instpkg (thisdisk[i].fname, from, exdir);
#ifdef DEBUG
      fprintf (stderr, "system = %d\n", ret);
#endif

      if (ret != 0) {
	fprintf (stderr, "Error: failed to install REQUIRED package!\n");
	g_errs++;
	num_errors++;

	/* Does the user want to continue anyway? */

	printf ("Continue installing this disk?\n");
	if (!pause_yn (1)) {
	  return (0);
	}
      }
      break;

    default:
      /* Optional */

      printf ("%s [OPTIONAL]\n", thisdisk[i].fname);

      /* Ask the user if you want to install it */

      printf ("Install package %s? [yn] ", thisdisk[i].fname);
      if (pause_yn (0)) {
	ret = instpkg (thisdisk[i].fname, from, exdir);
#ifdef DEBUG
      fprintf (stderr, "system = %d\n", ret);
#endif

	if (ret != 0) {
	  fprintf (stderr, "Warning: failed to install OPTIONAL package\n");
	  g_warns++;
	  num_warns++;

	  /* Continue anyway.  This is non-fatal. */

	  printf ("Continuing anyway..\n");
	}
      }
      break;

    } /* switch */
  } /* for */

  /* Free memory for this disk */

  free (thisdisk);

  /* Print summary information and return */

  if ((num_errors > 0) || (num_warns > 0)) {
    fprintf (stderr, "\n\n");;
    fprintf (stderr, "There were %u warnings and %u errors with this disk\n",
	     num_warns, num_errors);
  }

  return (nfiles);
}

/* Installs a disk series into dir "exdir", using the entry from the
   DAT file.  Files are located in the path "from".  Returns the
   number of disks installed from the series.  We know the series is
   finished when we find 'endfile' */

int
instser (const char *base, const char *from, const char *exdir)
{
  /* Variables */

  char endfile[STRLEN];				/* marks end of series */
  char descfile[STRLEN];			/* description file */
  char datfile[STRLEN];				/* current DAT file */
  int ndisks = 0;				/* no. disks installed */

  /* Create the filenames */

  sprintf (endfile, "%s%c%s.END", from, FS_DIR_SEP, base);
  sprintf (descfile, "%s%c%s.TXT", from, FS_DIR_SEP, base);

  /* Say hello */

  printf ("\n\n");
  printf (">> Installing series %s <<\n", base);

  /* Install while we have disks to work from.  Since we will reach an
     exit condition within the loop, we use an infinite loop here. */

  while (1) {
    /* Set the DAT file name */

    sprintf (datfile, "%s%c%s.%d", from, FS_DIR_SEP, base, ++ndisks);

    /* Load the first disk */

    /* First check that the datfile exists.  If it doesn't check if
       the endfile was found. */

    if (!isfile (datfile)) {

      /* Does the endfile exist? */

      if (isfile (endfile)) {
	printf ("\n%s found\n", endfile);
	return (ndisks);
      }

      /* The endfile was not found, so we know there is at least one
         more disk left to do.  Keep asking the user to insert the
         next disk. */

      do {
	printf ("\n\n");
	printf ("Load disk \'%s-%d\' in %s\n", base, ndisks, from);

	if (!pause_yn (1)) {
	  /* user has decided to quit this series */
	  return (ndisks);
	} /* if pause_yn */

      } while (!isfile (datfile));

    } /* if no datfile */

    /* Install files from this disk */

    printf ("\ninstalling \'%s-%d\'\n", base, ndisks);
    instdisk (datfile, descfile, from, exdir);

  } /* while (1) */
}
