/*
// Program:  Format
// Version:  0.91n
// (0.90b/c/d - warnings fixed, buffers far, boot sector IO - EA 2003)
// (0.91b..j / k... - more fixes - Eric Auer 2003 / 2004)
// Written By:  Brian E. Reifsnyder
// Copyright:  2002-2004 under the terms of the GNU GPL, Version 2
// Module Name:  driveio.c
// Module Description:  Functions specific to accessing a disk.
*/

#define DIO

#include "driveio.h"
#include "format.h"
#include "userint.h" /* Critical_* */

unsigned int FloppyBootReadWrite(char, int, ULONG,  void far *, unsigned);



/* Clear Huge Sector Buffer */
void Clear_Huge_Sector_Buffer(void)
{
  memset(huge_sector_buffer_0, 0, sizeof(huge_sector_buffer_0));
  memset(huge_sector_buffer_1, 0, sizeof(huge_sector_buffer_1));
}



/* Clear Sector Buffer */
void Clear_Sector_Buffer(void)
{
  memset(sector_buffer_0, 0, sizeof(sector_buffer_0));
  memset(sector_buffer_1, 0, sizeof(sector_buffer_1));
}



/* 0.91l - Win9x / DOS 7+: LOCK (lock TRUE) / UNLOCK (lock FALSE) drive */
void Lock_Unlock_Drive(int lock)
{
  /* *** 0.91l try to LOCK the drive (logical volume) *** */
  if (_osmajor >= 7) /* dos.h: Major DOS version number. 7 is "Win9x" */
    {
      if (debug_prog==TRUE) printf("[DEBUG]  DOS 7+ detected, %sing drive.\n",
        lock ? "LOCK" : "UNLOCK" ); /* (for UNLOCK, BH/DX are irrelevant) */

      regs.x.cx = lock ? 0x084a : 0x086a; /* (un)LOCK logical FAT1x volume */
      do
        {
          regs.x.ax = 0x440d; /* IOCTL */
          if ((!lock) && (param.fat_type == FAT32)) regs.h.ch = 0x48; /* FAT32 */
          regs.h.bh = 4;      /* Windows shell uses this lock level */
          regs.h.bl = param.drive_number + 1; /* A: is 1 etc. */
          regs.x.dx = 7;      /* 1 allow writes + 4 lock for formatting + 2 */
          regs.x.cflag = 0;   /* (2 is block new file mappings) */
          intdos(&regs, &regs); /* call int 0x21 DOS API */
          regs.h.ch += 0x40;    /* try FAT32 (etc.) mode next */
          if (!lock) regs.h.ch = 0x77; /* give up if earlier if unlocking */
        } while ( (regs.h.ch <= 0x48) && regs.x.cflag );
      if (regs.x.cflag && lock) /* ignore unlock errors */
        {
          unsigned int lockerrno = regs.x.ax;

          regs.x.ax = 0x3000; /* get full DOS version */
          intdos(&regs, &regs);
          if (regs.h.bh == 0xfd)
            {
              printf(" Ignoring FreeDOS logical drive locking error 0x%x. Old kernel?\n",
                lockerrno);
            }
          else
            {
              printf(" Could not lock logical drive (error 0x%x)! Aborting.\n",
                lockerrno);
              exit(1);
            }
        } /* LOCK error */

      /* for int 13 writes you have to lock the physical volume, too */
      if (param.drive_type == FLOPPY)
        {

          regs.x.ax = 0x440d;  /* IOCTL */
          regs.x.cx = lock ? 0x084b : 0x086b; /* (un)LOCK physical FAT1x volume */
          if ((!lock) && (param.fat_type == FAT32)) regs.h.ch = 0x48; /* FAT32 */
          regs.h.bh = 3;       /* highest lock level (DX / BL still set) */
          regs.x.cflag = 0;
          intdos(&regs, &regs); /* call int 0x21 DOS API */
          if (regs.x.cflag && lock) /* ignore unlock errors */
            {
              unsigned int lockerrno = regs.x.ax;

              regs.x.ax = 0x3000; /* get full DOS version */
              intdos(&regs, &regs);
              if (regs.h.bh == 0xfd)
                {
                  printf(" Ignoring FreeDOS physical drive locking error 0x%x. Old kernel?\n",
                    lockerrno);
                }
              else
                {
                  printf(" Could not lock physical floppy drive for formatting (error 0x%x)!?\n",
                    lockerrno);
                  /* exit(1); */
                }
            } /* LOCK error */

        } /* extra FLOPPY treatment */
    } /* DOS 7.x or newer */
}   /* *** drive is now LOCKed if lock, else UNLOCKed *** */



/* changed 0.91k: if number of sectors is negative, do not abort on fail */
int Drive_IO(int command,unsigned long sector_number,int number_of_sectors)
{
  unsigned int return_code;
  int error_counter = 0;
  int allowfail = (number_of_sectors < 0);
  if (allowfail) number_of_sectors =  -number_of_sectors;
  if (!number_of_sectors)
    {
      printf("Drive_IO: zero sized read/write request!? Ignored.\n");
      return -1;
    }

  retry:

  if (param.fat_type != FAT32)
    {
    return_code =
       TE_AbsReadWrite(param.drive_number,number_of_sectors,sector_number,
	   (number_of_sectors==1) ? sector_buffer : huge_sector_buffer,
	   command);
    }
  else
    {
    return_code =
       FAT32_AbsReadWrite(param.drive_number,number_of_sectors,sector_number,
	   (number_of_sectors==1) ? sector_buffer : huge_sector_buffer,
	   command);
    /* return code: lo is int 24 / DI code, hi is bitmask: */
    /* 1 bad command 2 bad address mark 3 (int 26 only) write protection */
    /* 4 sector not found 8 dma failure 10 crc error 20 controller fail  */
    /* 40 seek fail 80 timeout, no response */
    }

  if ( (return_code==0x0207 /* 0x0408 */) &&
    ((param.drive_number & 0x80)==0) /* avoid fishy retry */
    ) /* changed -ea */
    {
    /* As per RBIL, if 0x0408 is returned, retry with high bit of AL set. */
    /* MY copy of RBIL mentions 0x0207 for that case!? -ea */
    param.drive_number = param.drive_number + 0x80; /* set high bit. */
    error_counter++;
/*
 *  if (debug_prog==TRUE) printf("[DEBUG]  Retrying drive access.  High bit of AL is set.\n");
 */
    if (debug_prog==TRUE) printf("+"); /* give some hint about the change -ea */
    goto retry;
    }

  if ( (return_code!=0) && (error_counter<3) )
    {
    error_counter++;
/*
 *  if (debug_prog==TRUE) printf("[DEBUG]  Retrying drive access.  Retry number->  %2d\n",error_counter);
 */
    if (debug_prog==TRUE) printf("*"); /* give some problem indicator -ea */
    goto retry;
    }

  if (return_code!=0)
    {
    if ( ((param.drive_number & 0x7f)>=2) ||
         (command==WRITE) || (sector_number!=0L)  )
      {
      if (allowfail)
        {
        if (debug_prog==TRUE)
          printf("* bad sector(s): %ld (code 0x%x) on %s *\n",
            sector_number, return_code, (command==WRITE) ? "WRITE" : "READ");
        else
          printf("#");
        }
      else
        {
        /* Added more details -ea */
        printf("Drive_IO( command=%s sector=%ld count=%d ) [%s] [drive %c%c]\n",
          (command==WRITE) ? "WRITE" : "READ", sector_number, number_of_sectors,
          (param.fat_type==FAT32) ? "FAT32" : "FAT12/16",
          'A' + (param.drive_number & 0x7f),
          (param.drive_number & 0x80) ? '*' : ':' );
        Critical_Error_Handler(DOS, return_code);
        }
      }
    else
      {
      /* Only HARDDISK and WRITE and READ-beyond-BOOTSECTOR errors   */
      /* are CRITICAL, others are handled by the CALLER instead! -ea */
      printf("#");
      }
    }

  param.drive_number = param.drive_number & 0x7f; /* unset high bit. */
  return (return_code);
}



/* Do not confuse this with DOS 7.0+ drive locking */
void Enable_Disk_Access(void) /* DOS 4.0+ drive access flag / locking */
{
  unsigned char category_code;
  unsigned long error_code=0;

  category_code = (param.fat_type == FAT32) ? 0x48 : 0x08;

  /* Get the device parameters for the logical drive */

  regs.h.ah=0x44;                     /* IOCTL Block Device Request      */
  regs.h.al=0x0d;                     /* IOCTL */
  regs.h.bl=param.drive_number + 1;
  regs.h.ch=category_code;            /* 0x08 if !FAT32, 0x48 if FAT32   */
  regs.h.cl=0x67;                     /* Get Access Flags                */
  regs.x.dx=FP_OFF(&access_flags);	/* only 2 bytes: 0 and the flag */
  sregs.ds =FP_SEG(&access_flags);
  access_flags.special_function = 0;

  intdosx(&regs, &regs, &sregs);

  error_code = regs.h.al;

  if (regs.x.cflag)
    {
    /* BO: if invalid function: try to format anyway maybe access
	   flags do not work this way in this DOS (e.g. DRDOS 7.03) */
    if (error_code == 0x1 || error_code == 0x16)
      {
        printf("This DOS does not use access flags (IOCTL int 21.440d.x8.67)\n");
	return;
      }

    /* Add error trapping here */
    printf("\nFatal error obtaining disk access flags... format terminated.\n");
    printf("Error Code:  %02x\n",error_code);
    exit(1);
    }

  if (access_flags.disk_access==0) /* access not yet enabled? */
    {
    access_flags.disk_access++;
    access_flags.special_function = 0;

    regs.h.ah = 0x44;                     /* IOCTL Block Device Request          */
    regs.h.al = 0x0d;                     /* IOCTL */
    regs.h.bl = param.drive_number + 1;
    regs.h.ch = category_code;            /* 0x08 if !FAT32, 0x48 if FAT32       */
    regs.h.cl = 0x47;                     /* Set access flags                    */
    regs.x.dx = FP_OFF(&access_flags);
    sregs.ds  = FP_SEG(&access_flags);
    intdosx( &regs, &regs, &sregs);

    error_code = regs.h.al;

    if (regs.x.cflag)
	{
      /* Add error trapping here */
	printf("\nFatal error: Could not obtain disk access... format terminated.\n");
	printf("Error Code:  %02x\n",error_code);
	exit(1);
	}
    }

  if (debug_prog==TRUE)
    printf("[DEBUG]  Disk access flag checking completed.\n");
}



/* FAT32_AbsReadWrite() is modified from TE_AbsReadWrite(). */
unsigned int FAT32_AbsReadWrite(char DosDrive, int count, ULONG sector,
  void far * buffer, unsigned ReadOrWrite)
{
    unsigned diskReadPacket_seg;
    unsigned diskReadPacket_off;

    void far * diskReadPacket_p;

    struct {
	unsigned long  sectorNumber;
	unsigned short count;
	void far *address;
	} diskReadPacket;
    union REGS regs;

    struct {
	unsigned direction  : 1 ; /* low bit */
	unsigned reserved_1 : 12;
	unsigned write_type : 2 ;
	unsigned reserved_2 : 1 ;
	} mode_flags;

    diskReadPacket.sectorNumber = sector;
    diskReadPacket.count        = count;
    diskReadPacket.address      = buffer;

    diskReadPacket_p =& diskReadPacket;

    diskReadPacket_seg = FP_SEG(diskReadPacket_p);
    diskReadPacket_off = FP_OFF(diskReadPacket_p);

    mode_flags.reserved_1 = 0;
    mode_flags.write_type = 0;
    mode_flags.direction  = (ReadOrWrite == READ) ? 0 : 1;
    mode_flags.reserved_2 = 0;

    DosDrive++;

    /* no inline asm for Turbo C 2.01! -ea */
    /* Turbo C also complains about packed bitfield structures -ea */
    {
      struct SREGS s;
      regs.x.ax = 0x7305;
      regs.x.bx = diskReadPacket_off;
      s.ds = diskReadPacket_seg;
      regs.x.cx = 0xffff;
      regs.h.dl = DosDrive;
      regs.x.si = mode_flags.direction; /* | (mode_flags.write_type << 13); */
      intdosx(&regs, &regs, &s);
      return (regs.x.cflag ? regs.x.ax : 0);
    }
    
}

/* TE_AbsReadWrite() is written by Tom Ehlert. */
unsigned int TE_AbsReadWrite(char DosDrive, int count, ULONG sector,
  void far * buffer, unsigned ReadOrWrite)
{
    struct {
	unsigned long  sectorNumber;
	unsigned short count;
	void far *address;
	} diskReadPacket;
    void far * dRP_p;
    union REGS regs;
    struct SREGS sregs;


    if ( (count==1) && (sector==0) && ((DosDrive & 0x7f) < 2) )
      {
      /* ONLY use lowlevel function if only a floppy boot sector is     */
      /* affected: maybe DOS simply does not know filesystem params yet */
      /* the lowlevel function also tells DOS about the new fs. params! */
#if 0
//      if (debug_prog==TRUE)
//        printf("[DEBUG]  Using lowlevel boot sector access...\n");
#endif
        regs.x.ax = FloppyBootReadWrite(DosDrive & 0x7f, 1, 0,
          buffer, ReadOrWrite); /* (F.B.R.W. by -ea) */

        if ((debug_prog==TRUE) && (regs.h.al != 0))
#if 0
//          printf("[DEBUG] INT 13 status (hex):   %2.2x\n", regs.h.al);
#else
            printf("[%2.2x]", regs.h.al);
#endif

        switch (regs.h.al) { /* translate BIOS error code to DOS code */
                           /* AL is the AH that int 13h returned... */
          case 0: return 0; /* no error */
          case 3: return 0x0300; /* write protected */
          case 0x80: return 0x0002; /* drive not ready */
          case 0x0c: return 0x0007; /* unknown media type */
          case 0x04: return 0x000b; /* read fault */
          default: return 0x0808; /* everything else: "sector not found" */
          /* 02 - address mark not found, 07 - drive param error */
        } /* case */
        /* return 0x1414; *//* never reached */
      }

    diskReadPacket.sectorNumber = sector;
    diskReadPacket.count        = count;
    diskReadPacket.address      = buffer;

    dRP_p = &diskReadPacket;

    regs.h.al = DosDrive; /* 0 is A:, 1 is B:, ... */
    /* in "int" 25/26, the high bit of AL is an extra flag */
    regs.x.bx = FP_OFF(dRP_p);
    sregs.ds = FP_SEG(dRP_p);
    regs.x.cx = 0xffff;

    switch(ReadOrWrite)
	{
	    case READ:  int86x(0x25,&regs,&regs,&sregs); break;
	    case WRITE: int86x(0x26,&regs,&regs,&sregs); break;
	    default:
		printf("TE_AbsReadWrite wrong called %02x\n", ReadOrWrite);
		exit(1);
	 }

    
    return regs.x.cflag ? regs.x.ax : 0;
}

/* TE_* not usable while boot sector invalid, so... */
unsigned int FloppyBootReadWrite(char DosDrive, int count, ULONG sector,
  void far * buffer, unsigned ReadOrWrite)
{

    union REGS regs;
    struct SREGS sregs;
    int i = 0;
    int theerror = 0;

    if ((DosDrive > 1) || ((DosDrive & 0x80) != 0))
      return 0x0808; /* sector not found (DOS) */
    if ((sector != 0) || (count != 1))
      return 0x0808; /* sector not found (DOS) */

    do
      {
      if (ReadOrWrite == WRITE)
        {
        regs.x.ax = 0x0301; /* read 1 sector */
        }
      else
        {
        regs.x.ax = 0x0201; /* write 1 sector */
        }
      regs.x.bx = FP_OFF(buffer);
      regs.x.cx = 1; /* 1st sector, 1th (0th) track */
      regs.h.dl = DosDrive; /* drive ... */
      regs.h.dh = 0; /* 1st (0th) side */
      sregs.es = FP_SEG(buffer);
      int86x(0x13,&regs,&regs,&sregs);
      
      if (regs.h.ah != 0)
        {
        i++; /* error count */
        theerror = regs.h.ah; /* BIOS error code */
        }
      else
        {

#if 0	/* better do this where Drive_IO is CALLED, not implicitly! */
//        regs.h.ah = 0x0d;
//        intdos(&regs,&regs); /* flush buffers, reset disk system */
//
//        regs.h.ah = 0x32;
//        regs.h.dl = DosDrive + 1; /* 0 default 1 A: 2 B: ... */
//        intdosx(&regs,&regs,&sregs); /* force DOS to re-read boot sector */
//        /* we ignore the returned DS:BX pointer to the new DPB (if AL=0) */
//        segread(&sregs); 	/* restore defaults */
#endif

        return 0; /* WORKED */
        }
      }
      while (i < 5); /* do ... while */

    return theerror; /* return ERROR code */
    /* fixed in 0.91b: theerror / AH, not AX is the code! */
}

