;------------------------------------------------------------------------------
;		F R E E - D O S		X M S - D R I V E R
;------------------------------------------------------------------------------
;
; Improvments by Martin Strmberg.
; Copyright 2001, Martin Strmberg.
; 
; Martin can be reached at <ams@ludd.luth.se>. Please start the subject line 
; with "FDXMS". If you have a bug report, use bugtrack on the FreeDOS site,
; <http://www.freedos.org>. Make sure to include the FDXMS.SYS version number.
;
;
; Written by Till Gerken for the Free-DOS project.
;
; If you would like to use parts of this driver in one of your projects, please
; check up with me first.
;
; Till can be reached at:	till@tantalo.net (Internet)
;
; For questions concerning Free-DOS, mail the coordinator
; Morgan "Hannibal" Toal <hannibal@iastate.edu>
;
; Comments and bug reports are always appreciated.
;
; Copyright (c) 1995, Till Gerken
;------------------------------------------------------------------------------
; -- IMPLEMENTATION NOTES --
;
; - The Protected Mode handling is very simple, it fits only for the least
;   needs
; - I didn't care about reentrancy. If this thing should be used in
;   Multitasking Environments, well, somebody got to take a look at the code.
; - INT15h, Func. 87h (Move Block) has been re-implemented to preserve
;   the state of A20. (very slow!)
; - INT15h, Func. 88h (Extended Memory Size) has been re-implemented to
;   return 0.
; - Function 0Bh (Move Block) uses it's own Protected Mode handling.
;   It doesn't provide Interrupt windows.
; - The code is not very optimised, I just wrote it down for now. Later, when
;   everything is tested, I'm going to see what can be done
; - Some ideas were taken from the original XMS driver written by
;   Mark E. Huss (meh@bis.adp.com), but everything has been completely
;   rewritten, so if there are bugs, they are mine, not his. ;)
;------------------------------------------------------------------------------

ideal					; switch on ideal mode syntax
P386					; 386 instructions this time
jumps

include "fdxms.inc"			; this file contains structures,
					; constants and the like

;------------------------------------------------------------------------------
; 16-bit resident code and data
;------------------------------------------------------------------------------

segment code16
assume cs:code16, ds:code16, es:nothing

;------------------------------------------------------------------------------
; device driver header

		dd	-1			; last driver in list
		dw	8000h			; driver flags
		dw	offset strategy		; pointer to strategy routine
		dw	offset interrupt	; pointer to interrupt handler
		db	'XMSXXXX0'		; device driver name

;------------------------------------------------------------------------------
; global data

request_ptr	dd	?			; pointer to request header

initialized	db	0			; contains one if driver has
						; been initialised

xms_size	dw	?			; size of XMS in kbytes

gdt32		dw	gdt_size,dummy,0
dummy		dq	0
code16dsc	db	0ffh,0ffh,0,0,0,9ah,0,0
core32dsc	db	0ffh,0ffh,0,0,0,92h,0cfh,0
gdt_size=$-(offset dummy)

code16idx	=	08h
core32idx	=	10h

bios_gdt:
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
		dw	0ffffh			; Bits 0-15 of source segment length.
bios_src_low	dw	0			; Bits 0-15 of source address.
bios_src_middle	db	0			; Bits 16-23 of source address.
		db	93h			; Source access rights.
		db	0fh			; More type bits and bits 16-19 of source segment length.
bios_src_high	db	0			; Bits 24-31 of source address.
		dw	0ffffh			; Bits 0-15 of dest. segment length.
bios_dst_low	dw	0			; Bits 0-15 of dest. address.
bios_dst_middle	db	0			; Bits 16-23 of dest. address.
		db	93h			; Dest. access rights.
		db	0fh			; More type bits and bits 16-19 of dest segment length.
bios_dst_high	db	0			; Bits 24-31 of dest. address.
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
		db	0,0,0,0,0,0,0,0		; Dummy entries for BIOS.
	

old_int15	dd	?			; old INT15h vector
old_int2f	dd	?			; old INT2fh vector

hma_used	db	0			; set if HMA is used
hma_min		dw	0			; minimal space in HMA that
						; has to be requested
a20_locks	dw	0			; internal A20 lock count

xms_num_handles	dw	32			; number of available handles
trace		dd	0
debug		db	0
bios		db	0
		
;------------------------------------------------------------------------------
; strategy routine. is called by DOS to initialize the driver once.
; only thing to be done here is to store the address of the device driver
; request block.
; In:	ES:BX - address of request header
; Out:	nothing

proc	strategy	far
	mov	[word cs:request_ptr+2],es	; store segment addr
	mov	[word cs:request_ptr],bx	; store offset addr
	ret					; far return here!
endp	strategy

;------------------------------------------------------------------------------
; interrupt routine. called by DOS right after the strategy routine to
; process the incoming job. also used to initialize the driver.

proc	interrupt	far
	push	di es

	mov	di,cs
	mov	ds,di

	les	di,[request_ptr]		; load address of request header

	cmp	[es:di+request_hdr.cmd],CMD_INIT; do we have to initialize?
	jne	@@done
	cmp	[initialized],0			; do we have initialized
						; already?
	jne	@@done
	call	initialize			; no, do it now!
@@done:

	lds	si,[request_ptr]		; return this to DOS

	pop	es di
	ret					; far return here!
endp	interrupt

;------------------------------------------------------------------------------
; just delays a bit

; *** OLD CODE ***
;proc	delay
;	jmp	short $+2
;	jmp	short $+2
;	ret
;endp	delay
; *** OLD CODE ***

proc delay
@@delay_start:
     in al, 64h
     jmp @@delay_check
@@delay_check:
     and al, 2
     jnz @@delay_start
     ret
endp delay


;------------------------------------------------------------------------------
; empties the keyboard processor's command queue

; *** OLD CODE ***
;proc	empty_8042
;	call	delay			; delay a bit
;	in	al,64h
;	test	al,1			; is there something to be read?
;	jz	no_output		; no, go on
;	call	delay			; yes, first delay a bit
;	in	al,60h			; then read the output buffer
;	jmp	short empty_8042	; and try again
;no_output:
;	test	al,2			; has it finished processing?
;	jnz	empty_8042		; no, try again
;	ret				; yes, done
;endp	empty_8042
; *** OLD CODE ***

;------------------------------------------------------------------------------
; enables the A20 address line

; *** OLD CODE ***
;proc	enable_a20
;	call	empty_8042		; wait 'til he's done
;	mov	al,0d1h			; write cmd
;	out	64h,al
;	call	empty_8042
;	mov	al,0dfh			; switch on A20
;	out	60h,al
;	call	empty_8042
;	ret
;endp	enable_a20
; *** OLD CODE ***

proc enable_a20
     mov al,0d1h
     out 64h,al
     call delay
     mov al,0dfh
     out 60h,al
     call delay
     mov al,0ffh
     out 64h,al
     call delay
     ret
endp enable_a20

;------------------------------------------------------------------------------
; disables the A20 address line

; *** OLD CODE ***
;proc	disable_a20
;	call	empty_8042		; wait 'til he's done
;	mov	al,0d1h			; write cmd
;	out	64h,al
;	call	empty_8042
;	mov	al,0ddh			; switch off A20
;	out	60h,al
;	call	empty_8042
;	ret
;endp	disable_a20
; *** OLD CODE ***

proc disable_a20
     mov al,0d1h
     out 64h,al
     call delay
     mov al,0ddh
     out 60h,al
     call delay
     mov al,0ffh
     out 64h,al
     call delay
     ret
endp disable_a20

;------------------------------------------------------------------------------
; tests if the A20 address line is enabled.
; compares 256 bytes at 0:0 with ffffh:10h
; Out:	ZF=0 - A20 enabled
;	ZF=1 - A20 disabled

proc	test_a20
	push	cx si di ds es

	xor	si,si
	mov	ds,si

	mov	di,0ffffh
	mov	es,di
	mov	di,10h

	mov	cx,100h/4
	rep	cmpsd

	pop	es ds di si cx
	ret
endp	test_a20

;------------------------------------------------------------------------------
; checks if VDISK is already installed
; note: HMA check is skipped because of speed and some other (weird) reasons.
; In:	nothing
; Out:	ZF=0 -> VDISK is installed
;	ZF=1 -> VDISK not installed

vdisk_id	db	VDISK_IDSTR

proc	check_vdisk
	push	ax bx cx si di es

;	call	enable_a20		; enable A20 for this

	xor	ax,ax			; get interrupt vector 19h
	mov	es,ax
	les	bx,[es:19h*4]

	mov	di,VDISK_IDOFS
	mov	si,offset vdisk_id
	mov	cx,VDISK_IDLEN
	rep	cmpsb			; is VDISK here?
;	jnz	no_vdisk
;
;	mov	ax,0ffffh
;	mov	es,ax
;	mov	di,VDISK_IDOFS+1
;	mov	si,offset vdisk_id
;	mov	cx,VDISK_IDLEN
;	rep	cmpsb
;
;no_vdisk:
;
;	pushf				; save flags (thanks Yury!)
;	call	disable_a20
;	popf

	pop	es di si cx bx ax
	ret
endp	check_vdisk

;------------------------------------------------------------------------------
; Interrupt handlers
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; new INT15h handler

a20state	db	?		; keeps A20 state across INT15h call

proc	int15_handler
	cmp	ah,87h				; is it a block move request?
	je	@@do_move
	cmp	ah,88h				; is it a ext. mem size req.?
	je	@@ext_mem_size
	jmp	[cs:old_int15]			; jump to old handler
@@do_move:
	call	test_a20			; check if A20 is on or off
	jz	@@a20disabled
	mov	[cs:a20state],1			; preserve state
	jmp	@@call_old_mover
@@a20disabled:
	mov	[cs:a20state],0
@@call_old_mover:
	pushf					; simulate INT call
	call	[cs:old_int15]
	pushf					; save flags for return
	push	ax
	cmp	[cs:a20state],0			; see if A20 has to be switched
	jz	@@disable_it
	call	enable_a20
	jmp	@@move_done
@@disable_it:
	call	disable_a20
@@move_done:
	pop	ax
	popf
	iret
@@ext_mem_size:
	xor	ax,ax				; no memory available
	clc					; no error
	iret
endp	int15_handler

;------------------------------------------------------------------------------
; new INT2Fh handler. Catches Func. 4300h+4310h

proc	int2f_handler
	pushf
	cmp	ax,4300h			; is it "Installation Check"?
	jne	@@driver_address
	mov	al,80h				; yes, we are installed ;)
	popf
	iret
@@driver_address:
	cmp	ax,4310h			; is it "Get Driver Address"?
	jne	@@call_old2f
	mov	bx,cs
	mov	es,bx
	mov	bx,offset xms_dispatcher
	popf
	iret
@@call_old2f:
	popf
	jmp	[cs:old_int2f]			; jump to old handler
endp	int2f_handler

;------------------------------------------------------------------------------
; XMS functions
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; returns XMS version number
; In:	AH=0
; Out:	AX=XMS version number
;	BX=internal revision number
;	DX=1 if HMA exists, 0 if not

proc	xms_get_version
	mov	ax,INTERFACE_VER
	mov	bx,ax
	mov	dx,1				; HMA is always available
	popf
	retf
endp	xms_get_version

;------------------------------------------------------------------------------
; requests HMA
; In:	AH=1
;	DX=space needed in HMA (0ffffh if application tries to request HMA)
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function not implemented (implemented here ;) )
;	  BL=81h -> VDISK is detected
;	  BL=90h -> HMA does not exist
;	  BL=91h -> HMA already in use
;	  BL=92h -> DX less than HMA_MIN

proc	xms_request_hma
	cmp	[cs:hma_used],0			; is HMA already used?
	mov	bl,XMS_HMA_IN_USE
	jnz	xrh_err
	cmp	dx,[cs:hma_min]			; is request big enough?
	mov	bl,XMS_HMAREQ_TOO_SMALL
	jb	xrh_err
	mov	[cs:hma_used],1			; assign HMA to caller
	mov	ax,1
	xor	bl,bl
	popf
	retf
xrh_err:
	xor	ax,ax
	popf
	retf
endp	xms_request_hma

;------------------------------------------------------------------------------
; releases HMA
; In:	AH=2
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function not implemented
;	  BL=81h -> VDISK is detected
;	  BL=90h -> HMA doesn't exist
;	  BL=93h -> HMA wasn't allocated

proc	xms_release_hma
	cmp	[cs:hma_used],0			; is HMA used?
	mov	bl,XMS_HMA_NOT_USED
	jz	xrh_err
	mov	[cs:hma_used],0			; now release it
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_release_hma

;------------------------------------------------------------------------------
; global A20 address line enable
; In:	AH=3
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function is not implemented
;	  BL=81h -> VDISK is detected
;	  BL=82h -> A20 failure

proc	xms_global_enable_a20
	call	enable_a20			; enable A20
	call	test_a20			; is it really enabled?
	jz	xge_a20_err
	mov	ax,1
	xor	bl,bl
	popf
	retf
xge_a20_err:
	xor	ax,ax
	mov	bl,XMS_A20_FAILURE
	popf
	retf
endp	xms_global_enable_a20

;------------------------------------------------------------------------------
; global A20 address line disable
; In:	AH=4
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=80h -> function is not implemented
;	  BL=81h -> VDISK is detected
;	  BL=82h -> A20 failure
;	  BL=84h -> A20 still enabled

proc	xms_global_disable_a20
	call	disable_a20			; disable A20
	call	test_a20			; is it really disabled?
	jnz	xge_a20_err
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_global_disable_a20

;------------------------------------------------------------------------------
; enables A20 locally
; In:	AH=5
; Out:	AX=1 if A20 is enabled, 0 otherwise
;	BL=80h -> function not implemented
;	BL=81h -> VDISK is detected
;	BL=82h -> A20 failure

proc	xms_local_enable_a20
	inc	[cs:a20_locks]			; increase lock counter
	call	enable_a20			; enable it
	call	test_a20			; test if it's really enabled
	jz	xge_a20_err
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_local_enable_a20

;------------------------------------------------------------------------------
; disables A20 locally
; In:	AH=6
; Out:	AX=1 if A20 is disabled, 0 otherwise
;	BL=80h -> function not implemented
;	BL=81h -> VDISK is detected
;	BL=82h -> A20 failure

proc	xms_local_disable_a20
	dec	[cs:a20_locks]			; decrease lock counter
	jnz	@@xld_dont_disable		; disable only if needed
	call	disable_a20			; disable it
	call	test_a20			; test if it's really disabled
	jnz	xge_a20_err
@@xld_dont_disable:
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_local_disable_a20

;------------------------------------------------------------------------------
; returns the state of A20
; In:	AH=7
; Out:	AX=1 if A20 is physically enabled, AX=0 if not
;	BL=00h -> function was successful
;	BL=80h -> function is not implemented
;	BL=81h -> VDISK is detected

proc	xms_query_a20
	xor	ax,ax			; suppose A20 is disabled
	call	test_a20
	jz	@@xqa_a20dis
	mov	ax,1
@@xqa_a20dis:
	xor	bl,bl
	popf
	retf
endp	xms_query_a20

;------------------------------------------------------------------------------
; searches a/next free XMS memory block
; In:	DS=CS
;	BX - offset of start handle (if search is continued)
;	CX - remaining handles (if search is continued)
; Out:	CY=1 - no free block
;	  BX - offset of end of handle table
;	CY=0 - free block found
;	  BX - offset of free handle
;	  CX - number of remaining handles

proc	xms_find_free_block
	mov	bx,offset driver_end	; start at the beginning of the table
	mov	cx,[xms_num_handles]	; check all handles
@@find_free_block:
	cmp	[bx+xms_handle.used],0	; is it used?
	jnz	xms_find_next_free_block; yes, go on
	cmp	[bx+xms_handle.xbase],0	; assigned memory block or just blank?
	jnz	@@found_block		; assigned, return it
xms_find_next_free_block:
	add	bx,size xms_handle	; skip to next handle
	loop	@@find_free_block	; check next handle
	stc				; no free block found, error
	ret
@@found_block:
	clc				; no error, return
	ret
endp	xms_find_free_block

;------------------------------------------------------------------------------
; searches a/next free XMS memory handle
; In:	DS=CS
;	BX - offset of start handle (if search is continued)
;	CX - remaining handles (if search is continued)
; Out:	CY=1 - no free handle
;	  BX - offset of end of handle table
;	CY=0 - free handle found
;	  BX - offset of free handle
;	  CX - number of remaining handles

proc	xms_find_free_handle
	mov	bx,offset driver_end	; start at the beginning of the table
	mov	cx,[xms_num_handles]	; check all handles
@@find_free_handle:
	cmp	[bx+xms_handle.used],0		; is it used?
	jnz	xms_find_next_free_handle	; yes, go on
	cmp	[bx+xms_handle.xbase],0		; really blank handle?
	jz	@@found_handle			; found a blank handle
xms_find_next_free_handle:
	add	bx,size xms_handle	; skip to next handle
	loop	@@find_free_handle	; check next handle
	stc				; no free block found, error
	ret
@@found_handle:
	clc				; no error, return
	ret
endp	xms_find_free_handle

;------------------------------------------------------------------------------
; Verifies that a handle is valid and in use.
; In:	DX - handle to verify
; Out:	CY=1 - not a valid handle
;	  BL=0xa2 - XMS_INVALID_HANDLE
;	  AX=0 - Error return.
;	CY=0 - valid handle
;	  AX=modified

	
proc	xms_handle_valid
	mov	ax, dx
	sub	ax, offset driver_end		; Is the handle below start of handles?
	jb	@@not_valid
	
	push	dx
	push	bx
	xor	dx, dx
	mov	bx, size xms_handle
	div	bx
	or	dx, dx				; Is the handle aligned on a handle boundary?
	pop	bx
	pop	dx
	jnz	@@not_valid
	
	cmp	ax, [ cs:xms_num_handles ]	; Is the handle number less than the number of handles?
	jae	@@not_valid
	
	; If we come here, the handle is a valid handle
	mov	ax, bx
	mov	bx, dx
	cmp	[ cs:bx+xms_handle.used ], 1	; Is the handle in use?
	mov	bx, ax
	jne	@@not_valid

	; Handle is valid.
	clc
	ret
	
@@not_valid:
	; Handle is not valid
	mov	bl, XMS_INVALID_HANDLE
	xor	ax, ax
	stc
	ret
endp	xms_handle_valid
	
;------------------------------------------------------------------------------
; returns free XMS
; In:	AH=8
; Out:	AX=size of largest free XMS block in kbytes
;	DX=total amount of free XMS in kbytes
;	BL=0 if ok
;	BL=080h -> function not implemented
;	BL=081h -> VDISK is detected
;	BL=0a0h -> all XMS is allocated

proc	xms_query_free_xms
	push	bx cx si ds

	mov	ax,cs
	mov	ds,ax

	xor	ax,ax				; contains largest free block
	xor	dx,dx				; contains total free XMS

	call	xms_find_free_block		; search free block
	jc	@@no_free_xms

@@check_next:
	mov	si,[bx+xms_handle.xsize]	; get size
	add	dx,si				; update total amount
	cmp	si,ax				; check if larger than largest
	jbe	@@not_larger
	mov	ax,si				; larger, update
@@not_larger:
	call	xms_find_next_free_block
	jnc	@@check_next

	pop	ds si cx bx
	xor	bl,bl
	popf
	retf

@@no_free_xms:
	pop	ds si cx bx
	mov	bl,XMS_ALREADY_ALLOCATED
	popf
	retf
endp	xms_query_free_xms

;------------------------------------------------------------------------------
; allocates an XMS block
; In:	AH=9
;	DX=amount of XMS being requested in kbytes
; Out:	AX=1 if successful
;	  DX=handle
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> no free handles left

proc	xms_alloc_xms
	push	bx cx si ds

	mov	ax,cs
	mov	ds,ax

	call	xms_find_free_block	; see if there's a free block
	jnc	@@check_size		; if it is, go on
@@no_free_mem:
	pop	ds si cx bx
	xor	ax,ax
	mov	bl,XMS_ALREADY_ALLOCATED
	popf
	retf

@@get_next_block:
	call	xms_find_next_free_block
	jc	@@no_free_mem
@@check_size:
	cmp	dx,[bx+xms_handle.xsize]	; check if it's large enough
	ja	@@get_next_block			; no, get next block

	mov	si,bx			; save handle address
	mov	[bx+xms_handle.used],1	; this block is used from now on

	call	xms_find_free_handle	; see if there's a blank handle
	jc	@@perfect_fit		; no, there isn't, alloc all mem left
	mov	ax,[si+xms_handle.xsize]	; get size of old block
	sub	ax,dx				; calculate resting memory
	jz	@@perfect_fit			; if it fits perfectly, go on
	mov	[bx+xms_handle.xsize],ax	; store sizes of new blocks
	mov	[si+xms_handle.xsize],dx
	mov	ax,[si+xms_handle.xbase]	; get base address of old block
	add	ax,dx				; calculate new base address
	mov	[bx+xms_handle.xbase],ax	; store it in new handle
	mov	[bx+xms_handle.locks],0		; no locks on this block

@@perfect_fit:
	mov	[si+xms_handle.locks],0		; no locks on this block

	mov	dx,si				; return handle in DX

	pop	ds si cx bx
	mov	ax,1
	xor	bl,bl
	popf
	retf
endp	xms_alloc_xms

;------------------------------------------------------------------------------
; frees an XMS block
; In:	AH=0ah
;	DX=handle to allocated block that should be freed
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid
;	  BL=0abh -> handle is locked

proc	xms_free_xms
	call	xms_handle_valid
	jc	@@xms_free_not_valid
	
	push	bx cx dx si ds

	mov	ax,cs
	mov	ds,ax

	mov	si,dx				; get handle into SI

	cmp	[si+xms_handle.locks],0		; is the block locked?
	jz	@@not_locked			; no, go on
	pop	ds si dx cx bx
	xor	ax,ax
	mov	bl,XMS_BLOCK_LOCKED
	popf
	retf

@@not_locked:
	cmp	[si+xms_handle.xsize],0		; is it a zero-length handle?
	jnz	@@normal_handle
	mov	[si+xms_handle.xbase],0		; blank handle
	jmp	@@xms_free_done

@@normal_handle:
	mov	ax,[si+xms_handle.xbase]	; get base address
	add	ax,[si+xms_handle.xsize]	; calculate end-address

	call	xms_find_free_block		; check free blocks
	jc	@@xms_free_loop_2		; no, was last handle
@@try_concat:
	cmp	ax,[bx+xms_handle.xbase]	; is it adjacent to old block?
	jne	@@not_adjacent
	mov	dx,[bx+xms_handle.xsize]	; concat
	add	ax,dx
	add	[si+xms_handle.xsize],dx
	mov	[bx+xms_handle.xbase],0		; blank handle
	mov	[bx+xms_handle.xsize],0
@@not_adjacent:
	call	xms_find_next_free_block	; see if there are other blks
	jnc	@@try_concat

@@xms_free_loop_2:		; Now check if the previous block is free too.
	call	xms_find_free_block		; check free blocks
	jc	@@xms_free_done			; no, was last handle

@@try_concat_2:
	cmp	ax, [bx+xms_handle.xbase]	; is it adjacent to old block?
	add	ax, [bx+xms_handle.xsize]
	cmp	ax, [si+xms_handle.xbase]
	jne	@@not_adjacent_2
	
	mov	ax, [si+xms_handle.xsize]	; concat
	add	[bx+xms_handle.xsize], ax
	mov	[si+xms_handle.xbase], 0	; blank handle
	mov	[si+xms_handle.xsize], 0
	mov	[si+xms_handle.used], 0		; Not in use anymore.
	mov	si, bx
	jmp	@@xms_free_loop_2
@@not_adjacent_2:
	call	xms_find_next_free_block	; see if there are other blks
	jnc	@@try_concat_2
	
@@xms_free_done:
	mov	[si+xms_handle.used],0		; handle isn't used anymore
	pop	ds si dx cx bx
	mov	ax,1
	xor	bl,bl
	
@@xms_free_not_valid:
	popf
	retf
endp	xms_free_xms

;------------------------------------------------------------------------------
; calculates the move address
; In:	BX - handle (0 if EDX should be interpreted as seg:ofs value)
;	EDX - offset
; Out:	EBX - absolute move address
;	CY=1 - not valid handle.
;	CY=0 - valid handle.
	  
; Modifies: ECX, EDX

proc	xms_get_move_addr
	or	bx,bx			; translate address in EDX?
	jnz	@@dont_translate
	movzx	ecx,dx			; save offset
	xor	dx,dx			; clear lower word
	shr	edx,12			; convert segment to absolute address
	add	edx,ecx			; add offset
	xor	ebx,ebx			; we're using conventional memory
	jmp	@@no_handle
	
@@dont_translate:
	push	dx
	mov	dx, bx
	call	xms_handle_valid
	pop	dx
	jc	@@not_valid

	movzx	ebx,[cs:bx+xms_handle.xbase]	; get block base address
	shl	ebx,10				; convert from kb to absolute
	
@@no_handle:
	add	ebx,edx				; add offset into block
	
@@not_valid:	
	ret
endp	xms_get_move_addr

;------------------------------------------------------------------------------
; moves an XMS block
; In:	AH=0bh
;	DS:SI=pointer to XMS move structure
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=082h -> A20 failure
;	  BL=0a3h -> source handle is invalid
;	  BL=0a4h -> source offset is invalid
;	  BL=0a5h -> destination handle is invalid
;	  BL=0a6h -> destination offset is invalid
;	  BL=0a7h -> length is invalid
;	  BL=0a8h -> move has invalid overlap
;	  BL=0a9h -> parity error

proc	xms_move_xms
	push	eax 
	push	ebx ecx edx edi esi
	push	ds es
	cli					; no interrupts

	call	test_a20			; get A20 state
	pushf					; save it for later
	call	enable_a20			; now enable it!

;	mov	eax,cr0
;	test	al,1				; are we already in PM?
;	jnz	@@a20_failure			; yes, simulate A20 failure

	mov	bx,[si+xms_move_strc.dest_handle]
	mov	edx,[si+xms_move_strc.dest_offset]
	call	xms_get_move_addr		; get move address
	jc	@@dest_not_valid
	mov	edi,ebx				; store in destination index

	mov	bx,[si+xms_move_strc.src_handle]
	mov	edx,[si+xms_move_strc.src_offset]
	call	xms_get_move_addr		; get move address
	jc	@@src_not_valid

	mov	ecx,[si+xms_move_strc.len]	; get length

	mov	esi,ebx				; store in source index

	test	cl,1				; Is the length even?
	jnz	@@invalid_length

	;; Check if we're in protected mode already.
	mov	eax, cr0
	test	al, 1
	jnz	@@do_bios_move	; Already in PM.

	cmp	[ bios ], 0		; Do we want to always use the BIOS?
	jne	@@do_bios_move		; Yes.
	
	lgdt	[fword cs:gdt32]		; load GDTR
	mov	eax,cr0
	or	al,1				; set PE bit
	mov	cr0,eax				; shazamm!
	db	0eah				; JMP FAR
	dw	offset to_pm,code16idx		; flush IPQ and load CS sel.
to_pm:

	mov	ax,core32idx
	mov	ds,ax
	mov	es,ax

	shr	ecx,2				; get number of DWORDS to move
	jnc	@@dword_boundary		; is length a DWORD multiple?
	movs	[word esi],[word edi]		; no, move first word to adjust
@@dword_boundary:
	rep	movs [dword esi],[dword edi]	; now move the main block

	mov	eax,cr0
	and	al,not 1			; clear PE bit
	mov	cr0,eax				; shazomm!

	db	0eah				; JMP FAR
	dw	offset to_rm
code_seg dw	?				; flush IPQ and load CS sel.

to_rm:
	
@@xms_move_ok:	
	popf				; get A20 state
	jnz	@@a20_was_enabled	; if it was enabled, don't disable
	call	disable_a20		; it was disabled, so restore state
	
@@a20_was_enabled:
	pop	es ds
	pop	esi edi edx ecx ebx eax	; restore everything
	popf
	mov	ax,1				; success
	retf

@@do_bios_move:
	mov	edx, ecx
	shr	edx, 1		; Length in words.
	
@@bios_move_loop:
	mov	ecx, edx
	cmp	ecx, 8000h
	jbe	@@length_ok

	mov	ecx, 8000h

@@length_ok:
	;; Fill in source entries.
	mov	eax, esi
	mov	[ cs:bios_src_low ], ax
	shr	eax, 10h
	mov	[ cs:bios_src_middle ], al
	shr	eax, 8
	mov	[ cs:bios_src_high ], al

	;; Fill in destination entries.
	mov	eax, edi
	mov	[ cs:bios_dst_low ], ax
	shr	eax, 10h
	mov	[ cs:bios_dst_middle ], al
	shr	eax, 8
	mov	[ cs:bios_dst_high ], al

	push	ecx
	push	esi
	mov	ax, cs
	mov	es, ax

	lea	si, [ bios_gdt ]

	clc
	mov	ah, 87h
	cli
	int	15h
	sti

	pop	esi
	pop	ecx

	jc	@@a20_failure

	lea	edi, [ edi + 2*ecx ]	; dest += copied length.
	lea	esi, [ esi + 2*ecx ]	; src += copied length.
;	add	edi, ecx
;	add	edi, ecx
;	add	esi, ecx
;	add	esi, ecx
	sub	edx, ecx		; length -= copied length.
	jnz	@@bios_move_loop

	jmp	@@xms_move_ok

@@src_not_valid:
	mov	al, XMS_INVALID_SOURCE_HANDLE
	jmp	@@xms_move_failure_end
		
@@dest_not_valid:
	mov	al, XMS_INVALID_DESTINATION_HANDLE
	jmp	@@xms_move_failure_end
	
@@a20_failure:
	mov	al, XMS_A20_FAILURE 
	jmp	@@xms_move_failure_end

@@invalid_length:
	mov	al, XMS_INVALID_LENGTH
	;  Fall through.
	
@@xms_move_failure_end:	
	popf			; Saved a20 state.
	pop	es ds
	pop	esi edi edx ecx ebx
	mov	bl, al
	pop	eax
	xor	ax,ax
	popf
	retf
endp	xms_move_xms

;------------------------------------------------------------------------------
; locks an XMS block
; In:	AH=0ch
;	DX=XMS handle to be locked
; Out:	AX=1 if block is locked
;	  DX:BX=32-bit linear address of block
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid
;	  BL=0ach -> lock count overflow
;	  BL=0adh -> lock fails

proc	xms_lock_xms
	call	xms_handle_valid
	jc	@@xms_lock_not_valid
	
	mov	bx,dx
	inc	[cs:bx+xms_handle.locks]	; increase lock counter
	jnc	@@locked_successful		; go on if no overflow
	xor	ax,ax
	mov	bl,XMS_LOCK_COUNT_OVERFLOW	; overflow, return with error
	
@@xms_lock_not_valid:	
	popf
	retf
	
@@locked_successful:
	push	eax				; save EAX
	movzx	eax,[cs:bx+xms_handle.xbase]	; get block base address
	shl	eax,10				; calculate linear address
	mov	bx,ax				; store LSW
	shr	eax,16
	mov	dx,ax				; store MSW
	pop	eax				; restore EAX
	mov	ax,1
	popf
	retf
endp	xms_lock_xms

;------------------------------------------------------------------------------
; unlocks an XMS block
; In:	AH=0dh
;	DX=XMS handle to unlock
; Out:	AX=1 if block is unlocked
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid
;	  BL=0aah -> block is not locked

proc	xms_unlock_xms
	call	xms_handle_valid
	jc	@@xms_unlock_not_valid
	
	push	bx
	mov	bx,dx
	cmp	[cs:bx+xms_handle.locks],0	; check if block is locked
	jnz	@@is_locked			; go on if true
	pop	bx
	xor	ax,ax
	mov	bl,XMS_BLOCK_NOT_LOCKED
	popf
	retf
@@is_locked:
	dec	[cs:bx+xms_handle.locks]	; decrease lock counter
	pop	bx
	mov	ax,1
	xor	bl,bl
@@xms_unlock_not_valid:	
	popf
	retf

endp	xms_unlock_xms

;------------------------------------------------------------------------------
; returns XMS handle information
; In:	AH=0eh
;	DX=XMS block handle
; Out:	AX=1 if successful
;	  BH=block's lock count
;	  BL=number of free XMS handles
;	  DX=block's length in kbytes
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a2h -> handle is invalid

proc	xms_get_handle_info
	call 	xms_handle_valid
	jc	@@not_valid

	push 	ds cx		; Save ds and cx
	mov	ax, cs		; and setup ds to this segment.
	mov	ds, ax

	push	dx		; Save handle for later.

	xor	ax, ax		; ax is number of free handles.

	;; Setup for loop.
	mov	bx, offset driver_end
	mov	cx, [ xms_num_handles ]
	
@@look_again:
	cmp	[ bx + xms_handle.used ], 0	; In use?
	jne	@@add_some			; Yes, go to next.
	inc	ax				; No, one more free handle.

@@add_some:
	add	bx, size xms_handle
	loop	@@look_again

	;;  Now ax contains number of free handles.
	
	pop 	bx 				; Get handle saved earlier.
	mov	dx, [bx+xms_handle.xsize]	; Store block size.
	mov	bh, [bx+xms_handle.locks]	; Store lock count.
	
	cmp	ax, 100h	; Make sure that we don't overflow bl.
	jb	@@less_than_256
	mov	al, 0ffh
@@less_than_256:	
	mov	bl, al				; Store number of free handles.

	pop	cx ds		; Restore.
	mov	ax, 1		; Success.

@@not_valid:	
	popf
	retf

endp	xms_get_handle_info

;------------------------------------------------------------------------------
; reallocates an XMS block. only supports shrinking.
; In:	AH=0fh
;	BX=new size for the XMS block in kbytes
;	DX=unlocked XMS handle
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=081h -> VDISK is detected
;	  BL=0a0h -> all XMS is allocated
;	  BL=0a1h -> all handles are in use
;	  BL=0a2h -> invalid handle
;	  BL=0abh -> block is locked

proc	xms_realloc_xms
	call	xms_handle_valid
	jc	@@xms_realloc_not_valid
	
	push	bx dx si ds
	mov	ax,cs
	mov	ds,ax

	xchg	bx,dx
	cmp	dx,[bx+xms_handle.xsize]
	jbe	@@shrink_it

@@no_xms_handles_left:
	pop	ds si dx bx
	xor	ax,ax
	mov	bl,XMS_NO_HANDLE_LEFT		; simulate a "no handle" error
	popf
	retf

@@shrink_it:
	mov	si,bx
	call	xms_find_free_handle		; get blank handle
	jc	@@no_xms_handles_left		; return if there's an error
	mov	ax,[si+xms_handle.xsize]	; get old size
	mov	[si+xms_handle.xsize],dx
	sub	ax,dx				; calculate what's left over
	jz	@@dont_need_handle		; skip if we don't need it
	add	dx,[si+xms_handle.xbase]	; calculate new base address
	mov	[bx+xms_handle.xbase],dx	; store it
	mov	[bx+xms_handle.xsize],ax	; store size
	mov	[bx+xms_handle.locks],0		; block is not locked...
	mov	[bx+xms_handle.used],0		; ...and not used
@@dont_need_handle:
	pop	ds si dx bx
	mov	ax,1
	xor	bl,bl
	
@@xms_realloc_not_valid:	
	popf
	retf
endp	xms_realloc_xms

;------------------------------------------------------------------------------
; requests an UMB block
; In:	AH=10h
;	DX=size of requested memory block in paragraphs
; Out:	AX=1 if successful
;	  BX=segment number of UMB
;	  DX=actual size of the allocated block in paragraphs
;	AX=0 if not successful
;	  DX=size of largest available UMB in paragraphs
;	  BL=080h -> function not implemented
;	  BL=0b0h -> only a smaller UMB are available
;	  BL=0b1h -> no UMBs are available

proc	xms_request_umb
	xor	ax,ax			; function fails...
	mov	bl, XMS_NOT_IMPLEMENTED	; ...because there are no UMBs we...
	xor	dx,dx			; ...are managing
	popf
	retf
endp	xms_request_umb

;------------------------------------------------------------------------------
; releases an UMB block
; In:	AH=11h
;	DX=segment of UMB
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=0b2h -> UMB segment number is invalid

proc	xms_release_umb
	xor	ax,ax
	mov	bl,XMS_NOT_IMPLEMENTED
	popf
	retf
endp	xms_release_umb

;------------------------------------------------------------------------------
; reallocates an UMB
; In:	AH=12h
;	BX=new size for UMB in paragraphs
;	DX=segment of UMB to reallocate
; Out:	AX=1 if successful
;	AX=0 if not successful
;	  BL=080h -> function not implemented
;	  BL=0b0h -> no UMB large enough to satisfy request
;	    DX=size of largest UMB in paragraphs
;	  BL=0b2h -> UMB segment is invalid

proc	xms_realloc_umb
	xor	ax,ax
	mov	bl,XMS_NOT_IMPLEMENTED
	popf
	retf
endp	xms_realloc_umb

trace_get_version		db	'get_version',13,10,0
trace_request_hma		db	'request_hma',13,10,0
trace_release_hma		db	'release_hma',13,10,0
trace_global_enable_a20		db	'global_enable_a20',13,10,0
trace_global_disable_a20	db	'global_disable_a20',13,10,0
trace_local_enable_a20		db	'local_enable_a20',13,10,0
trace_local_disable_a20		db	'local_disable_a20',13,10,0
trace_query_a20			db	'query_a20',13,10,0
trace_query_free_xms		db	'query_free_xms',13,10,0
trace_alloc_xms			db	'alloc_xms',13,10,0
trace_free_xms			db	'free_xms',13,10,0
trace_move_xms			db	'move_xms',13,10,0
trace_lock_xms			db	'lock_xms',13,10,0
trace_unlock_xms		db	'unlock_xms',13,10,0
trace_get_handle_info		db	'get_handle_info',13,10,0
trace_realloc_xms		db	'realloc_xms',13,10,0
trace_request_umb		db	'request_umb',13,10,0
trace_release_umb		db	'release_umb',13,10,0
trace_realloc_umb		db	'realloc_umb',13,10,0


;------------------------------------------------------------------------------
; XMS dispatcher
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; XMS dispatcher
; In:	AH - function number
; Out:	AX=0 -> function not supported
;	else see appr. routine

xms_table	dw	xms_get_version, xms_request_hma
		dw	xms_release_hma, xms_global_enable_a20
		dw	xms_global_disable_a20, xms_local_enable_a20
		dw	xms_local_disable_a20, xms_query_a20
		dw	xms_query_free_xms, xms_alloc_xms
		dw	xms_free_xms, xms_move_xms
		dw	xms_lock_xms, xms_unlock_xms
		dw	xms_get_handle_info, xms_realloc_xms
		dw	xms_request_umb, xms_release_umb
		dw	xms_realloc_umb
	
trace_table	dw	trace_get_version, trace_request_hma
		dw	trace_release_hma, trace_global_enable_a20
		dw	trace_global_disable_a20, trace_local_enable_a20
		dw	trace_local_disable_a20, trace_query_a20
		dw	trace_query_free_xms, trace_alloc_xms
		dw	trace_free_xms, trace_move_xms
		dw	trace_lock_xms, trace_unlock_xms
		dw	trace_get_handle_info, trace_realloc_xms
		dw	trace_request_umb, trace_release_umb
		dw	trace_realloc_umb

proc	xms_dispatcher
	jmp	short @@dispatcher_entry
	nop					;
	nop					; guarantee hookability
	nop					;
@@dispatcher_entry:
	pushf					; save flags
	cld

	cmp	ah,12h				; is it a supported function?
	ja	@@not_supported
	call	check_vdisk			; is VDISK installed?
	jz	@@vdisk_installed
hook_patch:
	jmp	short @@hook_ints
@@hook_return:
	push	bx
	movzx	bx, ah
	push	cx
	mov	cl, bl		;  Put function number in cl for trace shift check.
	shl	bx, 1
	
	;; Check if a trace is wanted.
	push	eax
	mov	eax, 1
	shl	eax, cl
	test	[ cs:trace ], eax
	jz	@@no_trace	;  No trace wanted.
	mov	ax, [ cs:trace_table + bx ]
	call	print_string
	
@@no_trace:	
	pop	eax
	pop	cx
	mov	ax, [ cs:xms_table + bx ]
	pop	bx
	jmp	ax
@@not_supported:
	xor	ax,ax				; everything else fails
	mov	bl,XMS_NOT_IMPLEMENTED
	popf					; and Yury strikes again...
	retf
@@vdisk_installed:
	xor	ax,ax
	mov	bl,XMS_VDISK_DETECTED
	popf					; flags should be forbidden
	retf

@@hook_ints:
	or	ah,ah				; non-version call?
	jz	@@hook_return			; no, don't patch

	push	ax bx cx dx ds es		; save registers
	mov	ax,cs
	mov	ds,ax
	xor	ax,ax				; get INT15h vector
	mov	es,ax
	les	bx,[es:15h*4]
	mov	[word old_int15+2],es
	mov	[word old_int15],bx

	mov	ax,2515h			; install own INT15h
	mov	dx,offset int15_handler
	int	21h

	mov	cx,[xms_num_handles]		; get number of handles
	mov	bx,offset driver_end		; get start of handle table
@@clear_table:
	mov	[bx+xms_handle.xbase],0		; blank handle
	mov	[bx+xms_handle.xsize],0		; blk doesn't occupy any space
	mov	[bx+xms_handle.used],0		; handle not used
	mov	[bx+xms_handle.locks], 0	; No locks on unused handle.
	add	bx,size xms_handle
	loop	@@clear_table

	mov	bx,offset driver_end
	mov	[bx+xms_handle.xbase],XMS_START	; init first block and give
	mov	ax,[xms_size]			; it all available memory
	mov	[bx+xms_handle.xsize],ax

	pop	es ds dx cx bx ax		; restore registers

	mov	[word cs:hook_patch],9090h	; insert two NOPs
	jmp	@@hook_return			; and finish it
endp	xms_dispatcher

	
;------------------------------------------------------------------------------
; Prints null-terminated string pointed to by ax.
; In:	AX - pointer to string to print.
; Out:	AX - destroyed.

proc	print_string
	push	bp di bx
	push	ds
	mov	di, ax
	mov	ax, cs
	mov	ds, ax
	mov	ah, 0eh
	mov	bx, 0070h
@@loop:	
	mov	al, [di]
	or	al, al
	jz	@@epilogue
	int	10h
	inc	di
	jmp	@@loop
@@epilogue:	
	pop	ds
	pop	bx di bp
	ret
endp	print_string

;------------------------------------------------------------------------------
; mark for the driver end. above has to be the resident part, below the
; transient.

label	driver_end	near

;------------------------------------------------------------------------------
; 16-bit transient code and data. only used once.
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; checks if CPU is a 386
; In:	nothing
; Out:	CY=0 - processor is a 386 or higher
;	CY=1 - processor lower than 386

proc	check_cpu
	pushf
	xor	ax,ax
	push	ax
	popf
	pushf
	pop	ax
	and	ah,0fh
	cmp	ah,0fh
	je	@@not386
	mov	ah,7
	push	ax
	popf
	pushf
	pop	ax
	and	ah,7
	je	@@not386
	popf
	clc
	ret
@@not386:
	popf
	stc
	ret
endp	check_cpu

;------------------------------------------------------------------------------
; checks if A20 can be enabled and disabled
; Out:	CF=0 - A20 switching works
;	CF=1 - A20 failure

proc	check_a20
	call	enable_a20
	call	test_a20			; TEST_A20 should return ZF=0
	jz	@@a20failed
	call	disable_a20
	call	test_a20			; TEST_A20 should return ZF=1
	jnz	@@a20failed
	clc
	ret
@@a20failed:
	stc
	ret
endp	check_a20


;------------------------------------------------------------------------------
; Prints null-terminated string pointed to by fs:ax.
; In:	FS:AX - pointer to string to print.
; Out:	AX - destroyed.
	
proc	print_far_string
	push	bp di bx
	push	ds
	mov	di, ax
	mov	ax, fs
	mov	ds, ax
	mov	ah, 0eh
	mov	bx, 0070h
@@loop:	
	mov	al, [di]
	or	al, al
	jz	@@epilogue
	int	10h
	inc	di
	jmp	@@loop
@@epilogue:	
	pop	ds
	pop	bx di bp
	ret
endp	print_far_string

;------------------------------------------------------------------------------
; initializes the driver. called only once!
; may modify DI
; In:	ES:DI - pointer to init structure
;	DS - initialized with code segment

init_message		db	13,10,80 dup (''),'Free-DOS XMS-Driver',13,10
			db	'Copyright (c) 1995, Till Gerken',13,10
			db	'Copyright 2001, Martin Strmberg',13,10,13,10
			db	'Driver Version: ',DRIVER_VERSION,9
			db	'Interface Version: ',INTERFACE_VERSION,13,10
			db	'Information: ',INFO_STR,13,10,'$'

old_dos			db	'XMS needs at least DOS version 3.00.$'
xms_twice		db	'XMS is already installed.$'
vdisk_detected		db	'VDISK has been detected.$'
no_386			db	'At least a 80386 is required.$'
a20_error		db	'Unable to switch A20 address line.$'
xms_sizeerr		db	'Unable to determine size of extended memory.$'
xms_toosmall		db	'Extended memory is too small or not available.$'

error_msg		db	' Driver won''t be installed.',7,13,10,'$'

init_finished		db	80 dup (''),13,10,'$'

command_line_pre	db	13,10,'Command line is "$'
command_line_post	db	'".',13,10,'$'

command_line		db	XMS_COMMAND_LINE_LENGTH_MAX dup (0), 5 dup (0)
	
proc	initialize
	pushf
	push	ax bx cx dx si

	cld

	mov	ah,9				; first, welcome the user!
	mov	dx,offset init_message
	int	21h

	mov	ax,3000h			; get DOS version number
	int	21h
	xchg	ah,al				; convert to BSD
	cmp	ax,300h				; we need at least 3.00
	mov	dx,offset old_dos
	jb	@@error_exit

	mov	ax,4300h			; check if XMS is already
	int	2fh				; installed
	cmp	al,80h
	mov	dx,offset xms_twice
	je	@@error_exit

	call	check_cpu			; do we have at least a 386?
	mov	dx,offset no_386
	jc	@@error_exit

	call	check_a20			; check A20
	mov	dx,offset a20_error
	jc	@@error_exit

	call	check_vdisk			; is VDISK installed?
	mov	dx,offset vdisk_detected
	jz	@@error_exit

	clc
	mov	ah,88h				; extended memory size
	int	15h
	mov	dx,offset xms_sizeerr
	jc	@@error_exit
	mov	dx,offset xms_toosmall
	sub	ax,64				; save HIMEM area
	jc	@@error_exit			; if there aren't 64k,
						; there's nothing to do

	mov	[xms_size],ax			; save size

	push	eax				; save EAX

	mov	ax,cs				; setup descriptors
	mov	[code_seg],ax			; eliminate relocation entry
	movzx	eax,ax
	shl	eax,4
	or	[dword code16dsc+2],eax
	add	[dword gdt32+2],eax

	pop	eax				; restore EAX

	push	es
	xor	ax,ax				; get INT2Fh vector
	mov	es,ax
	les	bx,[es:2fh*4]
	mov	[word old_int2f+2],es
	mov	[word old_int2f],bx

	mov	ax,252fh			; install own INT2Fh
	mov	dx,offset int2f_handler
	int	21h
	pop	es

	mov	ah, 9
	mov	dx, offset command_line_pre
	int	21h
	push	fs
	lfs	ax, [es:di+init_strc.cmd_line]
	call	print_far_string
	pop	fs
	mov	ah, 9
	mov	dx, offset command_line_post
	int	21h

	push	fs
	lfs	ax, [es:di+init_strc.cmd_line]
	call	copy_command_line
	pop	fs

	call	interpret_command_line
		
	mov	[word es:di+2+init_strc.end_addr],cs	; set end address
	xor	dx,dx
	mov	ax,size xms_handle
	mul	[xms_num_handles]
	add	ax,offset driver_end
	mov	[word es:di+init_strc.end_addr],ax
	mov	[es:di+request_hdr.status],STATUS_OK	; we're alright
	jmp	short @@exit

@@error_exit:
	mov	[word es:di+2+init_strc.end_addr],cs	; set end address
	mov	[word es:di+init_strc.end_addr],0	; now, we don't use
							; any space
	mov	[es:di+request_hdr.status],STATUS_BAD	; waaaah!
	mov	ah,9					; print msg
	int	21h
	mov	dx,offset error_msg
	int	21h

@@exit:
	mov	ah,9
	mov	dx,offset init_finished
	int	21h
	pop	si dx cx bx ax
	popf
	ret
endp	initialize

;------------------------------------------------------------------------------
; Saves a string pointed to by fs:ax into command_line. 
; command_line is truncated to a length of 255 bytes and upper cased.
; In:	FS:AX - pointer to string.
; Out:	AL - destroyed.

proc	copy_command_line
	push	di si cx

	mov	cx, XMS_COMMAND_LINE_LENGTH_MAX
	mov	si, ax
	mov	di, offset command_line
@@loop:
	mov	al, [ fs:si ]
	cmp	al, 'a'
	jb	@@do_move
	cmp	al, 'z'
	ja	@@do_move
	;; Must be a lower case letter
	add	al, 'A'-'a'	; which now is uppercase.
@@do_move:	
	mov	[di], al
	dec	cx
	jcxz	@@too_long
	inc	di
	inc	si
	or	al, al
	jnz	@@loop		; Stop if we did copy nul, else continue.

@@the_end:
	pop	cx si di
	ret

@@too_long:
	mov	[byte di], 0		; Terminate command line.
	jmp	@@the_end
		
endp	copy_command_line

	
;------------------------------------------------------------------------------
; Analyses the contents of command_line and sets variables accordingly.
; In:	Nothing.
; Out:	AX - destroyed.
	
proc	interpret_command_line
	push	es
	push	di si cx bx

	mov	ax, ds
	mov	es, ax
	;; First we must step over FDXMS.SYS
	mov	bx, offset command_line		; Remember where search started.
	mov	cx, XMS_COMMAND_LINE_LENGTH_MAX
	add	cx, bx				; cx is guard.
	
@@look_again_for_fdxms_sys:	
	mov	si, bx
	mov	di, offset fdxms_sys_str
	call	strcmp
	jnc	@@found_fdxms_sys
	inc	bx
	cmp	bx, cx
	jae	@@done
	jmp	@@look_again_for_fdxms_sys
			
@@found_fdxms_sys:
@@next_arg:
	push	bx
	mov	bx, 0d0ah	; BH=
, BL=^J
	mov	ax, 0920h	; AH=tab, AL=space
	call	eat_characters
	pop	bx
	
	cmp	[byte si], 0
	je	@@done		; End of string?
	
	cmp	si, cx
	jae	@@done		; Is this necessary?
	
	mov	bx, si		; Remember current position
	
	;; TRACE argument.
	mov	di, offset trace_str
	call	strcmp
	jc	@@try_bios

	push	eax
	cmp	[ byte si ], '='
	je	@@trace_equal_sign
	mov	eax, 0ffffffffh
	jmp	@@trace_no_value

@@trace_equal_sign:	
	inc	si
	call	read_integer
	jc	@@trace_expects_integer
	
@@trace_no_value:
	mov	[ trace ], eax
	pop	eax
	jmp	@@next_arg

@@trace_expects_integer:
	pop	eax
	mov	ax, offset trace_no_int_str
	call	print_string
	jmp	@@next_arg

	;; BIOS argument.
@@try_bios:
	mov	si, bx
	mov	di, offset bios_str
	call	strcmp
	jc	@@try_debug

	mov	[ bios ], 1
	jmp	@@next_arg

	;; DEBUG argument.
@@try_debug:	
	mov	si, bx
	mov	di, offset debug_str
	call	strcmp
	jc	@@try_numhandles

	mov	[ debug ], 1
	jmp	@@next_arg	

	;; NUMHANDLES argument.
@@try_numhandles:
	mov	si, bx
	mov	di, offset numhandles_str
	call	strcmp
	jc	@@bad_arg

	push	eax
	cmp	[ byte si ], '='
	jne	@@numhandles_no_value_given

	inc	si
	call	read_integer
	jc	@@numhandles_no_value_given

	cmp	eax, 1
	jae	@@numhandles_at_least_one

	mov	ax, offset numhandles_zero_str
	call	print_string
	mov	eax, 1

@@numhandles_at_least_one:
	cmp	eax, 1024
	jbe	@@numhandles_value_ok
	
	mov	ax, offset numhandles_too_big_str
	call	print_string
	mov	eax, 1024

@@numhandles_value_ok:
	mov	[ xms_num_handles ], ax
	pop	eax
	jmp	@@next_arg	

@@numhandles_no_value_given:	
	mov	ax, offset numhandles_no_value_str
	call	print_string
	pop	eax
	jmp	@@next_arg

	;; Bad argument.
@@bad_arg:
	mov	ax, offset bad_arg_pre
	call	print_string
	mov	si, bx
	mov	di, offset bad_arg_arg
	
@@bad_arg_loop:	
	mov	al, [si]
	mov	[di], al
	or	al, al
	jz	@@bad_arg_end
	cmp	al, ' '
	je	@@bad_arg_end
	cmp	al, 9		; tab
	je	@@bad_arg_end
	cmp	al, 0ah		; lf
	je	@@bad_arg_end
	cmp	al, 0dh		; cr
	je	@@bad_arg_end
	inc	si
	inc	di
	jmp	@@bad_arg_loop
	
@@bad_arg_end:
	mov	[byte di], 0
	mov	ax, offset bad_arg_arg
	call	print_string
	mov	ax, offset bad_arg_post
	call	print_string	

	mov	al, [si]
	or	al, al
	jz	@@done
	
	inc	si
	jmp	@@next_arg

	;; The end.
@@done:	
	pop	bx cx si di
	pop	es	
	ret	
endp	interpret_command_line
	
fdxms_sys_str		db	'FDXMS.SYS',0
trace_str		db	'TRACE',0
trace_no_int_str	db	'TRACE expects an integer (e.g. TRACE=0xff)',13,10,0
numhandles_str		db	'NUMHANDLES',0
numhandles_zero_str	db	'Ridiculous small argument to NUMHANDLES increased to 1',13,10,0
numhandles_too_big_str	db	'Ridiculous big argument to NUMHANDLES clamped down to 0x400',13,10,0
numhandles_no_value_str	db	'NUMHANDLES expects an argument (e.g. NUMHANDLES=32)',13,10,0
bios_str		db	'BIOS',0
debug_str		db	'DEBUG',0
bad_arg_pre		db	'Ignoring invalid option: ',0
bad_arg_post		db	13,10,0
bad_arg_arg		db	256 dup (?)
dot			db	'.',0
yohoo			db	'Yohoo',13,10,0

;------------------------------------------------------------------------------
; Reads an integer from a string pointed ot by SI.
; In:	SI - pointers to string.
; Out:	CY=0 - successful conversion.
;	  SI - updated.
;	  EAX - integer read.
;	CY=1 - First character wasn't a digit.
;	  EAX - destroyed.
		
proc	read_integer
	push	edx
	xor	eax, eax
	mov	al, [si]
	cmp	al, 0
	je	@@failure
	cmp	al, '0'
	jb	@@failure
	cmp	al, '9'
	ja	@@failure
	sub	al, '0'
	movzx	edx, al
	inc	si
	mov	al, [si]
	cmp	al, 'X'
	je	@@perhaps_hex_number
	
@@decimal_loop:	
	cmp	al, '0'
	jb	@@done
	cmp	al, '9'
	ja	@@done
	lea	edx, [edx+edx*4-'0'/2]	; Lucky zero and '0' are even numbers!
	shl	edx, 1
	add	edx, eax
	inc	si
	mov	al, [si]
	cmp	al, 0
	je	@@done
	jmp	@@decimal_loop
	
@@done:
	mov	eax, edx
	pop	edx
	clc
	ret

@@perhaps_hex_number:
	cmp	dx, 0
	jne	@@done

	inc	si
	mov	al, [si]
	cmp	al, 0
	je	@@looked_like_hex_but_was_not
@@hex_number:	
	cmp	al, '0'
	jb	@@looked_like_hex_but_was_not
	cmp	al, 'F'
	ja	@@looked_like_hex_but_was_not
	cmp	al, '9'
	jbe	@@digit
	cmp	al, 'A'
	jb	@@looked_like_hex_but_was_not
	add	al, '0'-'A'+10	; Sets al to the ASCII code for the characters
				; after '9'.
@@digit:
	sub	al, '0'
	movzx	edx, al
	inc	si
	mov	al, [si]
@@hex_loop:	
	cmp	al, '0'
	jb	@@done
	cmp	al, 'F'
	ja	@@done
	cmp	al, '9'
	jbe	@@another_digit
	cmp	al, 'A'
	jb	@@done
	add	al, '0'-'A'+10	; Sets al to the ASCII code for the characters
				; after '9'.
@@another_digit:
	sub	al, '0'
	shl	edx, 4
	add	edx, eax
	inc	si
	mov	al, [si]
	cmp	al, 0
	je	@@done
	jmp	@@hex_loop
	
@@looked_like_hex_but_was_not:
	dec	si
	jmp	@@done

@@failure:
	pop	edx
	stc
	ret
	
endp	read_integer
;------------------------------------------------------------------------------
; Increases SI until a character that don't match contents of AH, AL, BH or BL is found.
; In:	SI - pointer to string.
;	AL - character to step over
;	AH - character to step over, 0x0 to ignore and to ignore BX contents.
;	BL - character to step over, 0x0 to ignore and to ignore BH contents.
;	BH - character to step over, 0x0 to ignore.
; Out:	SI - updated
	
proc eat_characters
@@loop:	
	cmp	al, [si]
	jne	@@try_ah
	inc	si
	jmp	@@loop
@@try_ah:
	or	ah, ah
	jz	@@done
	cmp	ah, [si]
	jne	@@try_bl
	inc	si
	jmp	@@loop
@@try_bl:
	or	bl, bl
	jz	@@done
	cmp	bl, [si]
	jne	@@try_bh
	inc	si
	jmp	@@loop
@@try_bh:
	or	bh, bh
	jz	@@done
	cmp	bh, [si]
	jne	@@done
	inc	si
	jmp	@@loop

@@done:	
	ret
endp	eat_characters
	
;------------------------------------------------------------------------------
; Compares two strings up to DI points at nul.
; In:	DI, SI - pointers to strings to compare.
; Out:	CY=0 - strings equal
;	  DI - points at null
;	  SI - points at character where DI is null.
;	  AX - destroyed
;	CY=1 - strings not equal
;	  DI - points at character which diffs
;	  SI - points at character which diffs
;	  AX - destroyed
		
proc	strcmp
@@loop:	
	mov	al, [di]
	cmp	al, 0		; Is it nul?
	je	@@done		; Yes, we have a match!
	
	cmp	al, [si]
	jne	@@failure	; Not equal so no match.
	inc	si
	inc	di
	jmp	@@loop

@@failure:	
	stc
	ret
	
@@done:
	clc
	ret		
endp	strcmp
	
ends	code16

;******************************************************************************

end
