/*
// Program:  Free FDISK
// Module:  DISKIO.CPP
// Module Description:  Disk Input/Output Code Module
//                      All functions that access the hard disk are here.
// Written By:  Brian E. Reifsnyder
// Version:  0.98 (Final Release)
// Copyright:  1998-2000 under the terms of the GNU GPL
*/


/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <dir.h>
#include <conio.h>
#include <bios.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bootcode.h"
#include "fdisk.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

/* External Prototype Declarations */
/* ******************************* */
extern int Create_Logical_Drive(int numeric_type, long size_in_MB);
extern int Create_Primary_Partition(int numeric_type,long size_in_MB);
extern int Determine_Drive_Letters();

extern long double far Translate_CHS_To_LBA(unsigned long cylinder
 ,unsigned long head,unsigned long sector,unsigned long total_heads
 ,unsigned long total_sectors);

extern void Clear_Screen(int type);
extern void Clear_Extended_Partition_Table(int drive);
extern void Convert_Long_To_Integer(long number);
extern void Determine_Free_Space();
extern void Pause();
extern void Print_Centered(int y,char *text,int style);
extern void Position_Cursor(int row,int column);
extern void Set_Active_Partition(int partition_number);

/* Module Prototype Declarations */
/* ***************************** */
int Get_Hard_Drive_Parameters(int physical_drive);
int Read_Physical_Sector(int drive, long cylinder, long head, long sector);
int Read_Physical_Sector_CHS(int drive,long cylinder, long head, long sector);
int Read_Physical_Sector_LBA(int drive,long cylinder, long head, long sector);
int Write_Partition_Tables();
int Write_Physical_Sector(int drive, long cylinder, long head, long sector);
int Write_Physical_Sector_CHS(int drive, long cylinder, long head, long sector);
int Write_Physical_Sector_LBA(int drive, long cylinder, long head, long sector);

long double Translate_CHS_To_LBA(unsigned long cylinder,unsigned long head
 ,unsigned long sector,unsigned long total_heads,unsigned long total_sectors);

unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4);

void Automatically_Partition_Hard_Drive();
void Clear_Boot_Sector(int drive,long cylinder,long head,long sector);
void Clear_Flag(int flag_number);
void Clear_Partition_Table();
void Clear_Partition_Table_Area_Of_Sector_Buffer();
void Clear_Sector_Buffer();
void Check_For_INT13_Extensions();
void Create_Alternate_MBR();
void Create_BootEasy_MBR();
void Create_MBR();
void Create_MBR_If_Not_Present();
void Get_Partition_Information();
void Remove_MBR();
void Save_MBR();

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Automatically partition the selected hard drive */
void Automatically_Partition_Hard_Drive()
{
  int index=0;
  unsigned long maximum_partition_size_in_MB;

  Determine_Drive_Letters();

  /* First, make sure no primary or extended partitions exist. */
  /* Compaq diagnostic partitions are ok, though.              */
  do
    {
    if( (brief_partition_table[(flags.drive_number-128)] [index]>0) && (brief_partition_table[(flags.drive_number-128)][index]!=18) )
      {
      printf("\nThe hard drive has already been partitioned...Operation Terminated.\n");
      exit(7);
      }

    index++;
    }while(index<4);

  /* Create a primary partition...if the size or type is incorrect,     */
  /* int Create_Primary_Partition(...) will adjust it automatically.    */
  Determine_Free_Space();
  Set_Active_Partition(Create_Primary_Partition(6,2048));

  /* Create an extended partition, if space allows. */
  Determine_Free_Space();
  if( ( ( (part_table[(flags.drive_number-128)]
   .pri_part_largest_free_space+1) * (part_table[(flags.drive_number-128)]
   .total_head+1) * part_table[(flags.drive_number-128)].total_sect)/2048)>5)
    {
    Create_Primary_Partition(5,9000);

    /* Fill the extended partition with logical drives. */
    Determine_Free_Space();
    do
      {
      Create_Logical_Drive(6,2048);

      Determine_Free_Space();
      maximum_partition_size_in_MB
       =(((part_table[(flags.drive_number-128)]
       .ext_part_largest_free_space+1)*(part_table[(flags.drive_number-128)]
       .total_head+1)*(part_table[(flags.drive_number-128)]
       .total_sect))/2048);
      }while( ( (100*maximum_partition_size_in_MB)
       /part_table[(flags.drive_number-128)].ext_part_size_in_MB ) >1);
    }
}

/* Check for interrupt 0x13 extensions */
void Check_For_INT13_Extensions()
{
  int carry;
  int drive_number=0x80;

  unsigned int ah_register;
  unsigned int bx_register;
  unsigned int cx_register;

  if(debug.lba==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"void Check_For_INT13_Extensions() debugging screen",BOLD);
    printf("\n\n    drive     int 0x13 ext?     access w/packet\n\n");
    }

  do
    {
    carry=99;

    asm{
      mov ah,0x41
      mov bx,0x55aa
      mov dl,BYTE PTR drive_number
      int 0x13

      mov BYTE PTR ah_register,ah
      mov WORD PTR bx_register,bx
      mov WORD PTR cx_register,cx

      jnc carry_flag_not_set    /* Jump if the carry flag is clear  */
      }                         /* If the carry flag is clear, then */
				/* the extensions exist.            */
    carry=1;
    part_table[(drive_number-128)].ext_int_13=FALSE;

    carry_flag_not_set:
    if( (carry==99) && (bx_register==0xaa55) )
      {
      flags.use_extended_int_13=TRUE;
      part_table[(drive_number-128)].ext_int_13=TRUE;

      if((cx_register&0x0001)==1) part_table[(drive_number-128)].device_access_using_packet_structure=TRUE;
      else part_table[(drive_number-128)].device_access_using_packet_structure=FALSE;

      part_table[(drive_number-128)].ext_int_13_version=ah_register;

      if(debug.lba==TRUE)
	{
	printf("     0x%2x          yes",drive_number);

	if((cx_register&0x0001)==1) printf("                 yes");
	else printf("                  no");

	printf("\n");
	}
      }
    else if(debug.lba==TRUE) printf("     0x%2x           no\n",drive_number);

    drive_number++;
    }while(drive_number<0x88);

  if(debug.lba==TRUE)
    {
    printf("\n\n\n");
    Pause();
    }
}

/* Clear the Boot Sector of a partition */
void Clear_Boot_Sector(int drive,long cylinder,long head,long sector)
{
  unsigned char stored_sector_buffer[512];
  long index;

  /* Save sector_buffer[512] into stored_sector_buffer[512] */
  index=0;
  do
    {
    stored_sector_buffer[index]=sector_buffer[index];
    index++;
    }while(index<512);

  /* Write all 0xf6 values to sector_buffer[index] */
  index=0;
  do
    {
    sector_buffer[index]=0xf6;
    index++;
    }while(index<512);

  Write_Physical_Sector(drive,cylinder,head,sector);

  /* Restore sector_buffer[512] to its original contents */
  index=0;
  do
    {
    sector_buffer[index]=stored_sector_buffer[index];
    index++;
    }while(index<512);
}

/* Clear the Flag */
void Clear_Flag(int flag_number)
{
  Read_Physical_Sector((flags.drive_number),0,0,(flags.flag_sector));
  sector_buffer[(446+(flag_number-1))]=0;
  Write_Physical_Sector((flags.drive_number),0,0,(flags.flag_sector));
  printf("\nFlag has been cleared.\n");
  exit(0);
}

/* Clear Partition Table */
void Clear_Partition_Table()
{
  Clear_Sector_Buffer();
  Write_Physical_Sector(flags.drive_number, 0, 0, 1);
}

/* Clear The Partition Table Area Of sector_buffer only. */
void Clear_Partition_Table_Area_Of_Sector_Buffer()
{
  long index=0x1be;

  do
    {
    sector_buffer[index]=0;
    index++;
    }while(index<=0x1fd);
}

/* Clear Sector Buffer */
void Clear_Sector_Buffer()
{
  int index=0;

  do
    {
    sector_buffer[index]=0;
    index ++;
    } while(index<512);
}

/* Combine Cylinder and Sector Values */
unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector)
{
  long value = 0;

  asm{
    mov ax,WORD PTR cylinder
    mov bx,WORD PTR sector

    mov dl,ah
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1
    shl dl,1

    mov dh,al

    add dx,bx

    mov WORD PTR value,dx
    }

  return(value);
}

/* Create Alternate Master Boot Code */
void Create_Alternate_MBR()
{
  char home_path[255];
  long index=0;

  FILE *file_pointer;

  Read_Physical_Sector(flags.drive_number,0,0,1);

  /* Clear old MBR, if any */
  do
    {
    sector_buffer[index]=0x00;
    index++;
    }while(index<0x1be);
  index=0;

  strcpy(home_path,path);
  strcat(home_path,"boot.mbr");
  /* Search the directory Free FDISK resides in before searching the PATH */
  /* in the environment for the boot.mbr file.                            */
  file_pointer=fopen(home_path,"rb");

  if(!file_pointer) file_pointer=fopen(searchpath("boot.mbr"),"rb");

  if(!file_pointer)
    {
    printf("\nThe \"boot.mbr\" file has not been found...Operation Terminated.\n");
    exit(8);
    }

  do
    {
    sector_buffer[index]=fgetc(file_pointer);
    index++;
    }while(index<0x1be);

  fclose(file_pointer);

  sector_buffer[0x1fe]=0x55;
  sector_buffer[0x1ff]=0xaa;

  Write_Physical_Sector(flags.drive_number,0,0,1);
}

/* Create Booteasy MBR */
void Create_BootEasy_MBR()
{
  long number_of_bytes=SIZE_OF_MBR;

  long loop=0;

  Read_Physical_Sector(flags.drive_number,0,0,1);

  do
    {
    sector_buffer[loop]=boot_code[loop];

    loop++;
    }while(loop<=number_of_bytes);

  sector_buffer[0x1fe]=0x55;
  sector_buffer[0x1ff]=0xaa;

  Write_Physical_Sector(flags.drive_number,0,0,1);
}

/* Create Master Boot Code */
void Create_MBR()
{

  if(flags.use_ambr==TRUE)
    {
    Create_Alternate_MBR();
    }
  else
    {
    Create_BootEasy_MBR();
    }
}

/* Create Master Boot Code if it is not present */
void Create_MBR_If_Not_Present()
{
  Read_Physical_Sector(0x80,0,0,1);

  if( (sector_buffer[0x1fe]!=0x055) && (sector_buffer[0x1ff]!=0xaa) )
    {
    Create_MBR();
    }
}

/* Convert Hexidecimal Number to a Decimal Number */
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2, unsigned long hex3, unsigned long hex4)
{
  return((hex1) + (hex2*256) + (hex3*65536) + (hex4*16777216));
}

/* Extract Cylinder */
unsigned long Extract_Cylinder(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );
  unsigned long extracted_cylinder = ( ( (cylinder_and_sector*4) & 768) + (cylinder_and_sector /256) );

  return(extracted_cylinder);
}

/* Extract the Cylinder from an LBA Value */
unsigned long Extract_Cylinder_From_LBA_Value(long double lba_value
 ,long double head,long double sector,long double total_heads
 ,long double total_sectors)
{
  return( ( ( ( (lba_value-(sector-1))/total_sectors)-head)/(total_heads+1) ) );
}

/* Extract Sector */
unsigned long Extract_Sector(unsigned long hex1, unsigned long hex2)
{
  unsigned long cylinder_and_sector = ( (256*hex2) + hex1 );
  unsigned long extracted_sector = cylinder_and_sector % 64;

  return(extracted_sector);
}

/* Get the parameters of the hard disk */
int Get_Hard_Drive_Parameters(int physical_drive)
{
  int error_code=0;

  unsigned long total_cylinders=0;
  unsigned long total_heads=0;
  unsigned long total_sectors=0;

  if(part_table[(physical_drive-128)].ext_int_13==FALSE)
    {
    /* Get the hard drive parameters with normal int 0x13 calls. */
    asm{
      mov ah, 0x08
      mov dl, BYTE PTR physical_drive
      int 0x13

      mov bl,cl
      and bl,00111111B

      mov BYTE PTR error_code, ah
      mov BYTE PTR total_sectors, bl

      mov bl,cl
      mov cl,ch
      shr bl,1
      shr bl,1
      shr bl,1
      shr bl,1
      shr bl,1
      shr bl,1

      mov ch,bl

      mov WORD PTR total_cylinders, cx
      mov BYTE PTR total_heads, dh
      }

    part_table[(physical_drive-0x80)].total_cyl=total_cylinders;
    part_table[(physical_drive-0x80)].total_head=total_heads;
    part_table[(physical_drive-0x80)].total_sect=total_sectors;
    part_table[(physical_drive-0x80)].num_of_log_drives=0;
    }
  else
    {
    /* Get the hard drive parameters with extended int 0x13 calls. */


/* ********************************************************************* */
/* ********************************************************************* */
/* NOTE:  EVENTUALLY MODIFY THIS TO GET THE INFORMATION FLAGS !!!!!!!!!! */

    unsigned int result_buffer_segment=FP_SEG(result_buffer);
    unsigned int result_buffer_offset=FP_OFF(result_buffer);

    asm {
      mov ah,0x48
      mov dl,BYTE PTR physical_drive
      mov ds,result_buffer_segment
      mov si,result_buffer_offset
      int 0x13

      mov BYTE PTR error_code,ah
      }

    part_table[(physical_drive-0x80)].total_cyl=Decimal_Number(result_buffer[4],result_buffer[5],result_buffer[6],result_buffer[7]);
    part_table[(physical_drive-0x80)].total_head=Decimal_Number(result_buffer[8],result_buffer[9],result_buffer[10],result_buffer[11])-1;
    part_table[(physical_drive-0x80)].total_sect=Decimal_Number(result_buffer[12],result_buffer[13],result_buffer[14],result_buffer[15]);
    part_table[(physical_drive-0x80)].num_of_log_drives=0;
    }

  return(error_code);
}

/* Get the volume labels and file system types from the boot sectors */
void Get_Partition_Information()
{
  int index;
  int sub_index;

  long sector_buffer_index;

  /* First get the information from the primary partitions. */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      /* Check for and get the volume label on a FAT12/FAT16 partition. */
      if( ( (part_table[index].pri_part_num_type[sub_index]==1)
       || (part_table[index].pri_part_num_type[sub_index]==4)
       || (part_table[index].pri_part_num_type[sub_index]==6) )
       || ( ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) )
       && (part_table[index].pri_part_num_type[sub_index]==14) ) )
	{
	Read_Physical_Sector((index+128)
	 ,part_table[index].pri_part_start_cyl[sub_index]
	 ,part_table[index].pri_part_start_head[sub_index]
	 ,part_table[index].pri_part_start_sect[sub_index]);

	if( (sector_buffer[53]>=32) && (sector_buffer[53]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=43;
	  do
	    {
	    part_table[index].pri_part_vol_label[sub_index]
	     [(sector_buffer_index-43)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=53);
	  }
	}
      else
	{
	strcpy(part_table[index].pri_part_vol_label[sub_index],"           ");
	}

      /* Check for and get the volume label on a FAT32 partition. */
      if( ( (flags.version==W95B) || (flags.version==W98) )
       && ( (part_table[index].pri_part_num_type[sub_index]==11)
       || (part_table[index].pri_part_num_type[sub_index]==12) ) )
	{
	Read_Physical_Sector((index+128)
	 ,part_table[index].pri_part_start_cyl[sub_index]
	 ,part_table[index].pri_part_start_head[sub_index]
	 ,part_table[index].pri_part_start_sect[sub_index]);

	if( (sector_buffer[81]>=32) && (sector_buffer[81]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=71;
	  do
	    {
	    part_table[index].pri_part_vol_label[sub_index]
	     [(sector_buffer_index-71)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=81);
	  }
	}

      sub_index++;
      }while(sub_index<4);

    index++;
    }while(index<8);

  /* Get the information from the extended partitions. */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      if( ( (part_table[index].log_drive_num_type[sub_index]==1)
       || (part_table[index].log_drive_num_type[sub_index]==4)
       || (part_table[index].log_drive_num_type[sub_index]==6) )
       || ( ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) )
       && (part_table[index].log_drive_num_type[sub_index]==14) ) )
	{
	Read_Physical_Sector((index+128)
	 ,part_table[index].log_drive_start_cyl[sub_index]
	 ,part_table[index].log_drive_start_head[sub_index]
	 ,part_table[index].log_drive_start_sect[sub_index]);

	if( (sector_buffer[53]>=32) && (sector_buffer[53]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=43;
	  do
	    {
	    part_table[index].log_drive_vol_label[sub_index]
	     [(sector_buffer_index-43)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=53);

	  }
	}
      else
	{
	strcpy(part_table[index].log_drive_vol_label[sub_index],"           ");
	}

      /* Check for and get the volume label on a FAT32 partition. */
      if( ( (flags.version==W95B) || (flags.version==W98) )
       && ( (part_table[index].log_drive_num_type[sub_index]==11)
       || (part_table[index].log_drive_num_type[sub_index]==12) ) )
	{
	Read_Physical_Sector((index+128)
	 ,part_table[index].log_drive_start_cyl[sub_index]
	 ,part_table[index].log_drive_start_head[sub_index]
	 ,part_table[index].log_drive_start_sect[sub_index]);

	if( (sector_buffer[81]>=32) && (sector_buffer[81]<=122) )
	  {
	  /* Get Volume Label */
	  sector_buffer_index=71;
	  do
	    {
	    part_table[index].log_drive_vol_label[sub_index]
	     [(sector_buffer_index-71)]=sector_buffer[sector_buffer_index];
	    sector_buffer_index++;
	    }while(sector_buffer_index<=81);
	  }
	}

      sub_index++;
      }while(sub_index<23);

    index++;
    }while(index<8);
}

/* Load External Partition Type Lookup Table */
void Load_External_Lookup_Table()
{
  int index=0;
  int end_of_file_marker_encountered=FALSE;
  int offset=0;
  int sub_index=0;

  long line_counter=1;

  FILE *file;

  char character_number[5];

  char home_path[255];
  char line_buffer[256];

  /* Clear the buffers */
  do
    {
    sub_index=0;
    do
      {
      partition_lookup_table_buffer_short[index] [sub_index]=0;
      sub_index++;
      }while(sub_index<9);

    sub_index=0;
    do
      {
      partition_lookup_table_buffer_long[index] [sub_index]=0;
      sub_index++;
      }while(sub_index<16);

    index++;
    }while(index<256);
  index=0;

  strcpy(home_path,path);
  strcat(home_path,"part.ini");
  /* Search the directory Free FDISK resides in before searching the PATH */
  /* in the environment for the part.cfg file.                            */
  file=fopen(home_path,"rt");

  if(!file) file=fopen(searchpath("part.ini"),"rt");

  flags.partition_type_lookup_table=INTERNAL;
  if(file)
    {
    while(fgets(line_buffer,255,file) !=NULL)
      {
      if( (0!=strncmp(line_buffer,";",1)) && (0!=strncmp(line_buffer,"999",3)) && (end_of_file_marker_encountered==FALSE) )
	{
	/* Determine what partition type this line is referring to. */
	character_number[3]=0;

	if(line_buffer[0]=='0')
	  {
	  character_number[0]=line_buffer[1];
	  character_number[1]=line_buffer[2];
	  character_number[2]=0;
	  }
	else
	  {
	  character_number[0]=line_buffer[0];
	  character_number[1]=line_buffer[1];
	  character_number[2]=line_buffer[2];
	  }

	index=atoi(character_number);

	if( (index<0) || (index>255) )
	  {
	  printf("\nPartition type out of range in line %d of \"part.ini\"...Operation Terminated.\n",line_counter);
	  exit(9);
	  }

	/* Load the short description buffer (8) */
	offset=4;
	do
	  {
	  /* */
	  partition_lookup_table_buffer_short[index] [(offset-4)]=line_buffer[offset];
	  offset++;
	  } while(offset<=11);
	/* Load the long description buffer (15) */
	offset=13;
	do
	  {
	  partition_lookup_table_buffer_long[index] [(offset-13)]=line_buffer[offset];
	  offset++;
	  }while(offset<=27);

	index++;
	}
      line_counter++;
      if(0==strncmp(line_buffer,"999",3)) end_of_file_marker_encountered=TRUE;
      }

    fclose(file);
    flags.partition_type_lookup_table=EXTERNAL;
    }
}

/* Read and process the fdisk.ini file */
void Process_Fdiskini_File()
{
  char char_number[2];
  char command_buffer[20];
  char home_path[255];
  char line_buffer[256];
  char setting_buffer[20];

  int index=0;
  int command_ok=FALSE;
  int done_looking=FALSE;
  int end_of_file_marker_encountered=FALSE;
  int number;
  int object_found=FALSE;
  int sub_buffer_index=0;

  long line_counter=1;
  long setting;

  FILE *file;

  /* Set values to UNCHANGED */
  debug.all=UNCHANGED;
  debug.create_partition=UNCHANGED;
  debug.determine_free_space=UNCHANGED;
  debug.emulate_disk=UNCHANGED;
  debug.input_routine=UNCHANGED;
  debug.lba=UNCHANGED;
  debug.path=UNCHANGED;
  debug.read_sector=UNCHANGED;
  debug.write=UNCHANGED;

  flags.allow_abort=UNCHANGED;
  flags.del_non_dos_log_drives=UNCHANGED;
  flags.extended_options_flag=UNCHANGED;
  flags.flag_sector=UNCHANGED;
  flags.monochrome=UNCHANGED;
  flags.label=UNCHANGED;
  flags.reboot=UNCHANGED;
  flags.screen_color=UNCHANGED;
  flags.set_any_pri_part_active=UNCHANGED;
  flags.use_ambr=UNCHANGED;
  flags.version=UNCHANGED;

  flags.use_freedos_label=FALSE;

  strcpy(home_path,path);
  strcat(home_path,"fdisk.ini");

  /* Search the directory Free FDISK resides in before searching the PATH */
  /* in the environment for the fdisk.ini file.                           */
  file=fopen(home_path,"rt");

  if(!file) file=fopen(searchpath("fdisk.ini"),"rt");

  if(file)
    {
    while(fgets(line_buffer,255,file) !=NULL)
      {
      if( (0!=strncmp(line_buffer,";",1)) && (0!=strncmp(line_buffer,"999",3)) && (end_of_file_marker_encountered==FALSE) )
	{
	/* Clear the command_buffer and setting_buffer */
	index=0;

	do
	  {
	  command_buffer[index]=0x00;
	  setting_buffer[index]=0x00;

	  index++;
	  }while(index<20);

	/* Extract the command and setting from the line_buffer */

	/* Find the command */
	index=0;
	sub_buffer_index=0;
	done_looking=FALSE;
	object_found=FALSE;
	do
	  {
	  if( (line_buffer[index]!='=') && ( (line_buffer[index]>=0x30) && (line_buffer[index]<=0x7a) ) )
	    {
	    object_found=TRUE;
	    command_buffer[sub_buffer_index]=line_buffer[index];
	    sub_buffer_index++;
	    }

	  if( (object_found==TRUE) && ( (line_buffer[index]=='=') || (line_buffer[index]==' ') ) )
	    {
	    //command_buffer[sub_buffer_index]=0x0a;
	    done_looking=TRUE;
	    }

	  if( (index==254) || (line_buffer[index]==0x0a) )
	    {
	    printf("Error encountered on line %n of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }

	  index++;
	  }while(done_looking==FALSE);

	/* Find the setting */
	sub_buffer_index=0;
	object_found=FALSE;
	done_looking=FALSE;

	do
	  {
	  if( (line_buffer[index]!='=') && ( (line_buffer[index]>=0x30) && (line_buffer[index]<=0x7a) ) )
	    {
	    object_found=TRUE;
	    setting_buffer[sub_buffer_index]=line_buffer[index];
	    sub_buffer_index++;
	    }

	  if( (object_found==TRUE) && (line_buffer[index]==0x0a) )
	    {
	    done_looking=TRUE;
	    //setting_buffer[sub_buffer_index]=0x0a;
	    }

	  if(index==254)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }

	  index++;
	  }while(done_looking==FALSE);

	/* Adjust for the possibility of TRUE or FALSE in the fdisk.ini file. */
	if(0==stricmp(setting_buffer,"TRUE")) strcpy(setting_buffer,"ON");
	if(0==stricmp(setting_buffer,"FALSE")) strcpy(setting_buffer,"OFF");

	/* Process the command found in the line buffer */

	command_ok=FALSE;

	/* Check for the ALLOW_ABORT statement */
	if(0==stricmp(command_buffer,"ALLOW_ABORT"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.allow_abort=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.allow_abort=FALSE;
	  if(flags.allow_abort==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the AMBR statement */
	if(0==stricmp(command_buffer,"AMBR"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.use_ambr=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.use_ambr=FALSE;
	  if(flags.use_ambr==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the COLORS statement */
	if(0==stricmp(command_buffer,"COLORS"))
	  {
	  number=atoi(setting_buffer);

	  if( (number>=0) && (number<=127) )
	    {
	    flags.screen_color=number;
	    }

	  if(flags.screen_color==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_ALL statement */
	if(0==stricmp(command_buffer,"D_ALL"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.all=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.all=FALSE;
	  if(debug.all==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_CR_PART statement */
	if(0==stricmp(command_buffer,"D_CR_PART"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.create_partition=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.create_partition=FALSE;
	  if(debug.create_partition==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_DET_FR_SPC statement */
	if(0==stricmp(command_buffer,"D_DET_FR_SPC"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.determine_free_space=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.determine_free_space=FALSE;
	  if(debug.determine_free_space==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_INPUT statement */
	if(0==stricmp(command_buffer,"D_INPUT"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.input_routine=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.input_routine=FALSE;
	  if(debug.input_routine==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_LBA statement */
	if(0==stricmp(command_buffer,"D_LBA"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.lba=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.lba=FALSE;
	  if(debug.lba==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_PATH statement */
	if(0==stricmp(command_buffer,"D_PATH"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.path=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.path=FALSE;
	  if(debug.path==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the D_READ_S statement */
	if(0==stricmp(command_buffer,"D_READ_S"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.read_sector=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.read_sector=FALSE;
	  if(debug.read_sector==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the DEL_ND_LOG statement */
	if(0==stricmp(command_buffer,"DEL_ND_LOG"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.del_non_dos_log_drives=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.del_non_dos_log_drives=FALSE;
	  if(flags.del_non_dos_log_drives==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }


	/* Check for the EMULATE_DISK statement */
	if(0==stricmp(command_buffer,"EMULATE_DISK"))
	  {
	  if(0==stricmp(setting_buffer,"OFF")) debug.emulate_disk=0;
	  if(0==stricmp(setting_buffer,"1")) debug.emulate_disk=1;
	  if(0==stricmp(setting_buffer,"2")) debug.emulate_disk=2;
	  if(0==stricmp(setting_buffer,"3")) debug.emulate_disk=3;
	  if(0==stricmp(setting_buffer,"4")) debug.emulate_disk=4;
	  if(0==stricmp(setting_buffer,"5")) debug.emulate_disk=5;
	  if(0==stricmp(setting_buffer,"6")) debug.emulate_disk=6;
	  if(0==stricmp(setting_buffer,"7")) debug.emulate_disk=7;
	  if(0==stricmp(setting_buffer,"8")) debug.emulate_disk=8;
	  if(debug.emulate_disk==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the FLAG_SECTOR statement */
	if(0==stricmp(command_buffer,"FLAG_SECTOR"))
	  {
	  number=atoi(setting_buffer);
	  if( (number>=2) && (number<=64) ) flags.flag_sector=number;
	  if(flags.flag_sector==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the LABEL statement */
	if(0==stricmp(command_buffer,"LABEL"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.label=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.label=FALSE;
	  if(flags.label==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the MONO statement */
	if(0==stricmp(command_buffer,"MONO"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.monochrome=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.monochrome=FALSE;

	  if(flags.monochrome==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the REBOOT statement */
	if(0==stricmp(command_buffer,"REBOOT"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.reboot=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.reboot=FALSE;

	  if(flags.reboot==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the SET_ANY_ACT statement */
	if(0==stricmp(command_buffer,"SET_ANY_ACT"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.set_any_pri_part_active=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.set_any_pri_part_active=FALSE;

	  if(flags.set_any_pri_part_active==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }


	/* Check for the VERSION statement */
	if(0==stricmp(command_buffer,"VERSION"))
	  {
	  if(0==stricmp(setting_buffer,"4")) flags.version=FOUR;
	  if(0==stricmp(setting_buffer,"5")) flags.version=FIVE;
	  if(0==stricmp(setting_buffer,"6")) flags.version=SIX;
	  if(0==stricmp(setting_buffer,"W95")) flags.version=W95;
	  if(0==stricmp(setting_buffer,"W95B")) flags.version=W95B;
	  if(0==stricmp(setting_buffer,"W98")) flags.version=W98;
	  if(0==stricmp(setting_buffer,"FD"))
	    {
	    flags.version=FREEDOS_VERSION;
	    flags.use_freedos_label=TRUE;
	    }
	  if(flags.version==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the XO statement */
	if(0==stricmp(command_buffer,"XO"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) flags.extended_options_flag=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) flags.extended_options_flag=FALSE;
	  if(flags.extended_options_flag==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	/* Check for the WRITE statement */
	if(0==stricmp(command_buffer,"WRITE"))
	  {
	  if(0==stricmp(setting_buffer,"ON")) debug.write=TRUE;
	  if(0==stricmp(setting_buffer,"OFF")) debug.write=FALSE;
	  if(debug.write==UNCHANGED)
	    {
	    printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	    exit(3);
	    }
	  command_ok=TRUE;
	  }

	if(command_ok==FALSE)
	  {
	  printf("Error encountered on line %d of the \"fdisk.ini\" file...Program Terminated.\n",line_counter);
	  exit(3);
	  }

	}

      if(0==strncmp(line_buffer,"999",3)) end_of_file_marker_encountered=TRUE;

      line_counter++;
      }

    fclose(file);
    }

  /* Set options to defaults, if not already set */
  if(debug.all==UNCHANGED) debug.all=FALSE;
  if(debug.create_partition==UNCHANGED) debug.create_partition=FALSE;
  if(debug.determine_free_space==UNCHANGED) debug.determine_free_space=FALSE;
  if(debug.emulate_disk==UNCHANGED) debug.emulate_disk=0;
  if(debug.lba==UNCHANGED) debug.lba=FALSE;
  if(debug.input_routine==UNCHANGED) debug.input_routine=FALSE;
  if(debug.path==UNCHANGED) debug.path=FALSE;
  if(debug.read_sector==UNCHANGED) debug.read_sector=FALSE;
  if(debug.write==UNCHANGED) debug.write=TRUE;

  if(flags.allow_abort==UNCHANGED) flags.allow_abort=FALSE;
  if(flags.del_non_dos_log_drives==UNCHANGED) flags.del_non_dos_log_drives=FALSE;
  if(flags.extended_options_flag==UNCHANGED) flags.extended_options_flag=FALSE;
  if(flags.flag_sector==UNCHANGED) flags.flag_sector=2;
  if(flags.label==UNCHANGED) flags.label=FALSE;
  if(flags.monochrome==UNCHANGED) flags.monochrome=FALSE;
  if(flags.reboot==UNCHANGED) flags.reboot=FALSE;
  if(flags.screen_color==UNCHANGED) flags.screen_color=0x07; /* light grey on black */
  if(flags.set_any_pri_part_active==UNCHANGED) flags.set_any_pri_part_active=FALSE;
  if(flags.use_ambr==UNCHANGED) flags.use_ambr=FALSE;
  #pragma warn -ccc
  #pragma warn -rch
  if(flags.version==UNCHANGED)
    {
    if(DEFAULT_VERSION!=FD)
      {
      flags.version=DEFAULT_VERSION;
      }
    else
      {
      flags.version=FREEDOS_VERSION;
      flags.use_freedos_label=TRUE;
      }
    }
  #pragma warn +ccc
  #pragma warn +rch

  /* If debug.all==TRUE then set all debugging options to true */
  if(debug.all==TRUE)
    {
    debug.create_partition=TRUE;
    debug.determine_free_space=TRUE;
    debug.input_routine=TRUE;
    debug.lba=TRUE;
    debug.path=TRUE;
    debug.read_sector=TRUE;
    debug.write=FALSE;
    }

  /* If an emulated disk is specified, do not write anything to the disk. */
  if(debug.emulate_disk!=0) debug.write=FALSE;
}

/* Load the Partition Tables and get information on all drives */
int Read_Partition_Tables()
{
  int drive=0;
  int error_code=0;
  int index;
  int sub_index;

  long entry_offset;

  int physical_drive=0x80;

  do
    {
    /* Get the hard drive parameters and ensure that the drive exists. */
    error_code=Get_Hard_Drive_Parameters(physical_drive);

    /* If there was an error accessing the drive, skip that drive. */
    /* If this drive is emulated, then load the emulation values instead. */
    if( (error_code==0) && (debug.emulate_disk!=(drive+1) ) )
      {
      /* Pre-compute the total size of the hard drive */
      /* */
      part_table[drive].total_hard_disk_size_in_log_sect
       =(part_table[drive].total_cyl+1)
       *(part_table[drive].total_head+1)
       *part_table[drive].total_sect;
      part_table[drive].total_hard_disk_size_in_MB
       =part_table[drive].total_hard_disk_size_in_log_sectors/2048;
      }
    else
      {
      if(debug.emulate_disk==(drive+1) )
	{
	/* If this is an emulated drive, set it up. */
	part_table[drive].total_cyl=EMULATED_CYLINDERS;
	part_table[drive].total_head=EMULATED_HEADS;
	part_table[drive].total_sect=EMULATED_SECTORS;

	part_table[drive].total_hard_disk_size_in_log_sectors
	 =(part_table[drive].total_cyl+1)
	 *(part_table[drive].total_head+1)
	 *part_table[drive].total_sect;

	part_table[drive].total_hard_disk_size_in_MB
	 =part_table[drive].total_hard_disk_size_in_log_sectors/2048;

	flags.maximum_drive_number=drive+128;
	}
      else
	{
	if(drive==0)
	  {
	  cprintf("\n    No fixed disks present.\n");
	  exit(6);
	  }
	part_table[drive].total_cyl=0;
	part_table[drive].total_head=0;
	part_table[drive].total_sect=0;
	}
      }

    /* Clear the partition_table_structure structure. */
    part_table[drive].pri_part_largest_free_space=0;

    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_start_head=0;
    part_table[drive].pp_largest_free_space_start_sect=0;

    part_table[drive].pp_largest_free_space_end_cyl=0;

    index=0;
    do
      {
      part_table[drive].active_status[index]=0;
      part_table[drive].pri_part_num_type[index]=0;

      part_table[drive].pri_part_start_cyl[index]=0;
      part_table[drive].pri_part_start_head[index]=0;
      part_table[drive].pri_part_start_sect[index]=0;

      part_table[drive].pri_part_end_cyl[index]=0;
      part_table[drive].pri_part_end_head[index]=0;
      part_table[drive].pri_part_end_sect[index]=0;

      part_table[drive].pri_part_rel_sect[index]=0;
      /* */
      part_table[drive].pri_part_num_sect[index]=0;

      part_table[drive].pri_part_size_in_MB[index]=0;

      part_table[drive].pri_part_physical_order[index]=index;
      part_table[drive].pri_part_created[index]=FALSE;

      index++;
      }while(index<4);

    Clear_Extended_Partition_Table(drive);

    /* Read the Primary Partition Table. */
    if(error_code==0)
      {
      error_code=Read_Physical_Sector(physical_drive,0,0,1);

      if(error_code!=0) return(error_code);

      flags.maximum_drive_number=drive+128;

      index=0;
      do
	{
	entry_offset=0x1be+(index*16);

	part_table[drive].active_status[index]=sector_buffer[(entry_offset+0x00)];

	part_table[drive].pri_part_num_type[index]=sector_buffer[(entry_offset+0x04)];

	if(part_table[drive].ext_int_13==FALSE)
	  {
	  /* If int 0x13 extensions are not used get the CHS values. */
	  part_table[drive].pri_part_start_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);
	  part_table[drive].pri_part_start_head[index]=sector_buffer[(entry_offset+0x01)];
	  part_table[drive].pri_part_start_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);

	  part_table[drive].pri_part_end_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	  part_table[drive].pri_part_end_head[index]=sector_buffer[(entry_offset+0x05)];
	  part_table[drive].pri_part_end_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	  }

	part_table[drive].pri_part_rel_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x08)],sector_buffer[(entry_offset+0x09)],sector_buffer[(entry_offset+0x0a)],sector_buffer[(entry_offset+0x0b)]);
	part_table[drive].pri_part_num_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x0c)],sector_buffer[(entry_offset+0x0d)],sector_buffer[(entry_offset+0x0e)],sector_buffer[(entry_offset+0x0f)]);

	if(part_table[drive].ext_int_13==TRUE)
	  {
	  /* If int 0x13 extensions are used compute the virtual CHS values. */

	  /* The head number is 0 unless the cyl is 0...then the head is 1. */
	  if(part_table[drive].pri_part_rel_sect[index]==part_table[drive].total_sect)
	    {
	    part_table[drive].pri_part_start_head[index]=1;
	    }
	  else part_table[drive].pri_part_start_head[index]=0;
	  part_table[drive].pri_part_start_sect[index]=1;
	  part_table[drive].pri_part_start_cyl[index]
	   =Extract_Cylinder_From_LBA_Value(
	   part_table[drive].pri_part_rel_sect[index]
	   ,part_table[drive].pri_part_start_head[index]
	   ,part_table[drive].pri_part_start_sect[index]
	   ,part_table[drive].total_head
	   ,part_table[drive].total_sect);

	  part_table[drive].pri_part_end_head[index]=part_table[drive].total_head;
	  part_table[drive].pri_part_end_sect[index]=part_table[drive].total_sect;
	  part_table[drive].pri_part_end_cyl[index]
	   =Extract_Cylinder_From_LBA_Value(
	   /* */
	   (part_table[drive].pri_part_rel_sect[index]
	   +part_table[drive].pri_part_num_sect[index])
	   ,part_table[drive].pri_part_end_head[index]
	   ,part_table[drive].pri_part_end_sect[index]
	   ,part_table[drive].total_head
	   ,part_table[drive].total_sect);
	  }

	part_table[drive].pri_part_size_in_MB[index]=(part_table[drive].pri_part_num_sect[index]/2048);

	/* Record the necessary information to easilly and quickly find the */
	/* extended partition when it is time to read it.                   */
	if( (part_table[drive].pri_part_num_type[index]==0x05)
	 || ( (part_table[drive].pri_part_num_type[index]==0x0f)
	 && ( (flags.version==W95) || (flags.version==W95B)
	 || (flags.version==W98) ) ) )
	  {
	  part_table[drive].ext_part_exists=TRUE;
	  part_table[drive].num_of_ext_part=index;
	  part_table[drive].ext_part_num_sect=part_table[drive].pri_part_num_sect[index];
	  part_table[drive].ext_part_size_in_MB=part_table[drive].pri_part_size_in_MB[index];
	  }

	index++;
	}while(index<4);

      /* Read the Extended Partition Table, if applicable. */
      if(part_table[drive].ext_part_exists==TRUE)
	{
	error_code=Read_Physical_Sector(physical_drive,part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part],part_table[drive].pri_part_start_head[part_table[drive].num_of_ext_part],part_table[drive].pri_part_start_sect[part_table[drive].num_of_ext_part]);
	if(error_code!=0) return(error_code);

	/* Ensure that the sector has a valid partition table before        */
	/* any information is loaded into part_table[drive].           */
	if( (sector_buffer[0x1fe]==0x55) && (sector_buffer[0x1ff]==0xaa) )
	  {
	  index=0;
	  do
	    {
	    entry_offset=0x1be;

	    if(sector_buffer[(entry_offset+0x04)]>0)
	     part_table[drive].num_of_log_drives++;

	    part_table[drive].log_drive_num_type[index]
	     =sector_buffer[(entry_offset+0x04)];

	    if(part_table[drive].ext_int_13==FALSE)
	      {
	      /* If int 0x13 extensions are not used get the CHS values. */

	      part_table[drive].log_drive_start_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);
	      part_table[drive].log_drive_start_head[index]=sector_buffer[(entry_offset+0x01)];
	      part_table[drive].log_drive_start_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);

	      part_table[drive].log_drive_end_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	      part_table[drive].log_drive_end_head[index]=sector_buffer[(entry_offset+0x05)];
	      part_table[drive].log_drive_end_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
	      }

	    part_table[drive].log_drive_rel_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x08)],sector_buffer[(entry_offset+0x09)],sector_buffer[(entry_offset+0x0a)],sector_buffer[(entry_offset+0x0b)]);
	    part_table[drive].log_drive_num_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x0c)],sector_buffer[(entry_offset+0x0d)],sector_buffer[(entry_offset+0x0e)],sector_buffer[(entry_offset+0x0f)]);

	    if(part_table[drive].ext_int_13==TRUE)
	      {
	      /* If int 0x13 extensions are used compute the virtual CHS values. */

	      /* The head number is 0 unless the cyl is 0...then the head is 1. */
	      if(part_table[drive].log_drive_rel_sect[index]==part_table[drive].total_sect)
		{
		part_table[drive].log_drive_start_head[index]=1;
		}
	      else part_table[drive].log_drive_start_head[index]=0;
	      part_table[drive].log_drive_start_sect[index]=1;

	      if(index==0)
		part_table[drive].log_drive_start_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect[part_table
		 [drive].num_of_ext_part])
		 ,part_table[drive].log_drive_start_head[index]
		 ,part_table[drive].log_drive_start_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);
	      else
		part_table[drive].log_drive_start_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect[part_table[drive]
		 .num_of_ext_part]
		 +part_table[drive].next_ext_rel_sect[index-1])
		 ,part_table[drive].log_drive_start_head[index]
		 ,part_table[drive].log_drive_start_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);


	      part_table[drive].log_drive_end_head[index]=part_table[drive].total_head;
	      part_table[drive].log_drive_end_sect[index]=part_table[drive].total_sect;

	      if(index==0)
		part_table[drive].log_drive_end_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect
		 [part_table[drive].num_of_ext_part]
		 +part_table[drive].log_drive_num_sect[index])
		 ,part_table[drive].log_drive_end_head[index]
		 ,part_table[drive].log_drive_end_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);
	      else
		part_table[drive].log_drive_end_cyl[index]
		 =Extract_Cylinder_From_LBA_Value(
		 (part_table[drive].log_drive_rel_sect[index]
		 +part_table[drive].pri_part_rel_sect
		 [part_table[drive].num_of_ext_part]
		 +part_table[drive].log_drive_num_sect[index]
		 +part_table[drive].next_ext_rel_sect[index-1])
		 ,part_table[drive].log_drive_end_head[index]
		 ,part_table[drive].log_drive_end_sect[index]
		 ,part_table[drive].total_head
		 ,part_table[drive].total_sect);
	      }

	    part_table[drive].log_drive_size_in_MB[index]
	     =part_table[drive].log_drive_num_sect[index]/2048;

	    entry_offset=entry_offset+16;
	    if(sector_buffer[(entry_offset+0x04)]==0x05)
	      {
	      part_table[drive].next_ext_exists[index]=TRUE;

	      part_table[drive].next_ext_num_type[index]=sector_buffer[(entry_offset+0x04)];

	      if(part_table[drive].ext_int_13==FALSE)
		{
		/* If int 0x13 extensions are not used get the CHS values. */

		part_table[drive].next_ext_start_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);
		part_table[drive].next_ext_start_head[index]=sector_buffer[(entry_offset+0x01)];
		part_table[drive].next_ext_start_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x02)],sector_buffer[(entry_offset+0x03)]);

		part_table[drive].next_ext_end_cyl[index]=Extract_Cylinder(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
		part_table[drive].next_ext_end_head[index]=sector_buffer[(entry_offset+0x05)];
		part_table[drive].next_ext_end_sect[index]=Extract_Sector(sector_buffer[(entry_offset+0x06)],sector_buffer[(entry_offset+0x07)]);
		}

	      part_table[drive].next_ext_rel_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x08)],sector_buffer[(entry_offset+0x09)],sector_buffer[(entry_offset+0x0a)],sector_buffer[(entry_offset+0x0b)]);
	      part_table[drive].next_ext_num_sect[index]=Decimal_Number(sector_buffer[(entry_offset+0x0c)],sector_buffer[(entry_offset+0x0d)],sector_buffer[(entry_offset+0x0e)],sector_buffer[(entry_offset+0x0f)]);

	    if(part_table[drive].ext_int_13==TRUE)
	      {
	      /* If int 0x13 extensions are used compute the virtual CHS values. */

	      part_table[drive].next_ext_start_head[index]=0;
	      part_table[drive].next_ext_start_sect[index]=1;
	      /* */
	      part_table[drive].next_ext_start_cyl[index]
	       =Extract_Cylinder_From_LBA_Value(
	       (part_table[drive].next_ext_rel_sect[index]
	       +part_table[drive].pri_part_rel_sect
	       [part_table[drive].num_of_ext_part])
	       ,part_table[drive].next_ext_start_head[index]
	       ,part_table[drive].next_ext_start_sect[index]
	       ,part_table[drive].total_head
	       ,part_table[drive].total_sect);

	      part_table[drive].next_ext_end_head[index]=part_table[drive].total_head;
	      part_table[drive].next_ext_end_sect[index]=part_table[drive].total_sect;
	      part_table[drive].next_ext_end_cyl[index]
	       =Extract_Cylinder_From_LBA_Value(
	       (part_table[drive].next_ext_rel_sect[index]
	       +part_table[drive].pri_part_rel_sect
	       [part_table[drive].num_of_ext_part]
	       +part_table[drive].next_ext_num_sect[index])
	       ,part_table[drive].next_ext_end_head[index]
	       ,part_table[drive].next_ext_end_sect[index]
	       ,part_table[drive].total_head
	       ,part_table[drive].total_sect);
	      }

	      error_code=Read_Physical_Sector(physical_drive,part_table[drive].next_ext_start_cyl[index],part_table[drive].next_ext_start_head[index],part_table[drive].next_ext_start_sect[index]);

	      if(error_code!=0) return(error_code);
	      }
	    else index=24;

	    index++;
	    }while(index<24);
	  }
	}
     }
  drive++;
  physical_drive=drive+0x80;
  }while(drive<8);

  Determine_Drive_Letters();
  Get_Partition_Information();
  return(0);
}

/* Read_Physical_Sector */
int Read_Physical_Sector(int drive, long cylinder, long head, long sector)
{
  int current_line;
  int error_code;
  int index;
  int offset;

  if(flags.use_extended_int_13==FALSE)
    {
    error_code=Read_Physical_Sector_CHS(drive,cylinder,head,sector);
    }
  else
    {
    error_code=Read_Physical_Sector_LBA(drive,cylinder,head,sector);
    }

  if(debug.read_sector==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(4,"Read_Physical_Sector() function debugging screen",BOLD);

    Position_Cursor(4,10);
    printf("Information passed to this function:");

    Position_Cursor(50,11);
    printf("Drive:     0x%X",drive);
    Position_Cursor(50,12);
    printf("Cylinder:  %d",cylinder);
    Position_Cursor(50,13);
    printf("Head:      %d",head);
    Position_Cursor(50,14);
    printf("Sector:    %d",sector);
    Position_Cursor(4,16);
    printf("Contents of partition table area in sector_buffer[]:");

    current_line=0;
    offset=0x1be;
    do
      {
      index=0;

      Position_Cursor(4,(current_line+18));
      printf("%d:  ",(current_line+1));
      do
	{
	printf("%02X ",sector_buffer[(index+offset)]);
	index++;
	}while(index<16);

      current_line++;
      offset=offset+16;
      }while(offset<(0x1be+64));

    Position_Cursor(4,23);
    printf("Press any key to continue.");

    asm{
      mov ah,7
      int 0x21
      }
    }

  return(error_code);
}

/* Read a physical sector using CHS values */
int Read_Physical_Sector_CHS(int drive, long cylinder, long head, long sector)
{
  int error_code;

  error_code=biosdisk(2, drive, head, cylinder, sector, 1, sector_buffer);

  return(error_code);
}

/* Read a physical sector using LBA values */
int Read_Physical_Sector_LBA(int drive, long cylinder, long head, long sector)
{
  unsigned int error_code=0;

  /* Get the segment and offset of disk_address_packet. */
  unsigned int disk_address_packet_address_segment=FP_SEG(disk_address_packet);
  unsigned int disk_address_packet_address_offset=FP_OFF(disk_address_packet);

  /* Translate CHS values to LBA values. */
  unsigned long LBA_address=Translate_CHS_To_LBA(cylinder,head,sector
  ,part_table[(drive-128)].total_head
  ,part_table[(drive-128)].total_sect);

  /* Transfer LBA_address to disk_address_packet */
  unsigned long LBA_address_high=(LBA_address&0xffff0000)>>16;
  unsigned long LBA_address_low=LBA_address&0x0000ffff;

  Convert_Long_To_Integer(LBA_address_low);
  disk_address_packet[8]=integer1;
  disk_address_packet[9]=integer2;

  Convert_Long_To_Integer(LBA_address_high);
  disk_address_packet[10]=integer1;
  disk_address_packet[11]=integer2;

  /* Load the registers and call the interrupt. */
  asm {
    mov ah,0x42
    mov dl,BYTE PTR drive
    mov ds,disk_address_packet_address_segment
    mov si,disk_address_packet_address_offset
    int 0x13

    mov BYTE PTR error_code,ah
    }

  return(error_code);
}

/* Remove MBR */
void Remove_MBR()
{
  int index=0;

  Read_Physical_Sector((flags.drive_number),0,0,1);

  do
    {
    sector_buffer[index]=0x00;
    index++;
    }while(index<0x1be);

  Write_Physical_Sector((flags.drive_number),0,0,1);
}

/* Save MBR */
void Save_MBR()
{
  long index=0;

  FILE *file_pointer;

  Read_Physical_Sector(flags.drive_number,0,0,1);

  file_pointer = fopen("boot.mbr","wb");

  if(!file_pointer)
    {
    printf("\nError opening or creating \"BOOT.MBR\" for writing...Operation Terminated.\n");
    exit(8);
    }

  do
    {
    fputc(sector_buffer[index],file_pointer);
    index++;
    }while(index<0x1be);

  do{
    fputc(0,file_pointer);
    index++;
    }while(index<512);

  fclose(file_pointer);
}

/* Set the flag */
void Set_Flag(int flag_number)
{
  Read_Physical_Sector((flags.drive_number),0,0,(flags.flag_sector));
  sector_buffer[(446+(flag_number-1))]=1;
  Write_Physical_Sector((flags.drive_number),0,0,(flags.flag_sector));
  printf("\nFlag has been set.\n");
  exit(0);
}

/* Test the flag */
void Test_Flag(int flag_number)
{
  Read_Physical_Sector((flags.drive_number),0,0,(flags.flag_sector));
  if(sector_buffer[(446+flag_number-1)]==1)
    {
    printf("\nFlag has been set.\n");
    exit(20);
    }
  else
    {
    printf("\nFlag has not been set.\n");
    exit(21);
    }
}

/* Translate a CHS value to an LBA value. */
long double Translate_CHS_To_LBA(unsigned long cylinder,unsigned long head
 ,unsigned long sector,unsigned long total_heads,unsigned long total_sectors)
{
  return( ( (cylinder*(total_heads+1)+head)*total_sectors+sector-1) );
}

/* Write partition tables */
int Write_Partition_Tables()
{
  int error_code;
  int index;

  int drive_index=0;

  long entry_offset;

  unsigned long high;
  unsigned long low;
  unsigned long temp;

  long extended_cylinder;
  long extended_head;
  long extended_sector;

  long next_extended_cylinder;
  long next_extended_head;
  long next_extended_sector;

  do
    {
    if(part_table[drive_index].part_values_changed==TRUE)
      {
      index=0;

      Clear_Sector_Buffer();

      if(debug.write==TRUE) error_code=Read_Physical_Sector((drive_index+128),0,0,1);
      else error_code=0;
      if(error_code!=0) return(error_code);

      Clear_Partition_Table_Area_Of_Sector_Buffer();

      do
	{
	/* If this partition was just created, clear its boot sector. */
	if(part_table[drive_index].pri_part_created[index]==TRUE)
	  {
	  Clear_Boot_Sector((drive_index+128),part_table[drive_index].pri_part_start_cyl[index],part_table[drive_index].pri_part_start_head[index],part_table[drive_index].pri_part_start_sect[index]);
	  }

	if( (part_table[drive_index]
	 .pri_part_num_type[index]==0x05)
	 || (part_table[drive_index]
	 .pri_part_num_type[index]==0x0f) )

	  {
	  extended_cylinder=part_table[drive_index].pri_part_start_cyl[index];
	  extended_head=part_table[drive_index].pri_part_start_head[index];
	  extended_sector=part_table[drive_index].pri_part_start_sect[index];
	  }

	entry_offset=0x1be+(index*16);

	sector_buffer[(entry_offset+0x00)]=part_table[drive_index].active_status[index];

	sector_buffer[(entry_offset+0x01)]=part_table[drive_index].pri_part_start_head[index];

	Convert_Long_To_Integer(Combine_Cylinder_and_Sector
	 (part_table[drive_index]
	 .pri_part_start_cyl[index]
	 ,part_table[drive_index]
	 .pri_part_start_sect[index]));

	sector_buffer[(entry_offset+0x02)]=integer1;
	sector_buffer[(entry_offset+0x03)]=integer2;

	sector_buffer[(entry_offset+0x04)]=part_table[drive_index].pri_part_num_type[index];

	sector_buffer[(entry_offset+0x05)]
	 =part_table[drive_index].pri_part_end_head[index];

	Convert_Long_To_Integer(Combine_Cylinder_and_Sector
	 (part_table[drive_index]
	 .pri_part_end_cyl[index]
	 ,part_table[drive_index]
	 .pri_part_end_sect[index]));

	sector_buffer[(entry_offset+0x06)]=integer1;
	sector_buffer[(entry_offset+0x07)]=integer2;

	high = part_table[drive_index].pri_part_rel_sect[index] >> 16;
	temp=high << 16;

	if(part_table[drive_index].pri_part_rel_sect[index] > 0xffff)
	  {
	  low = part_table[drive_index].pri_part_rel_sect[index] - temp;
	  }
	else
	  {
	  low = part_table[drive_index].pri_part_rel_sect[index];
	  }

	Convert_Long_To_Integer(low);

	sector_buffer[entry_offset+0x08]=integer1;
	sector_buffer[entry_offset+0x09]=integer2;

	Convert_Long_To_Integer(high);

	sector_buffer[entry_offset+0x0a]=integer1;
	sector_buffer[entry_offset+0x0b]=integer2;

	high = part_table[drive_index].pri_part_num_sect[index] >> 16;
	temp=high << 16;

	if(part_table[drive_index].pri_part_num_sect[index] > 0xffff)
	  {
	  low = part_table[drive_index].pri_part_num_sect[index] - temp;
	  }
	else
	  {
	  low = part_table[drive_index].pri_part_num_sect[index];
	  }

	Convert_Long_To_Integer(low);

	sector_buffer[entry_offset+0x0c]=integer1;
	sector_buffer[entry_offset+0x0d]=integer2;

	Convert_Long_To_Integer(high);

	sector_buffer[entry_offset+0x0e]=integer1;
	sector_buffer[entry_offset+0x0f]=integer2;

	index++;
	}while(index<4);

      /* Add the partition table marker values */
      sector_buffer[0x1fe]=0x55;
      sector_buffer[0x1ff]=0xaa;

      error_code=Write_Physical_Sector((drive_index+128),0,0,1);
      if(error_code>0) return(error_code);

      /* Write the Extended Partition Table, if applicable. */

      if(part_table[drive_index].ext_part_exists==TRUE)
	{
	index=0;
	do
	  {
	  /* If this logical drive was just created, clear its boot sector. */
	  if(part_table[drive_index].log_drive_created[index]==TRUE)
	    {
	    Clear_Boot_Sector((drive_index+128),part_table[drive_index].log_drive_start_cyl[index],part_table[drive_index].log_drive_start_head[index],part_table[drive_index].log_drive_start_sect[index]);
	    }

	  Clear_Sector_Buffer();

	  entry_offset=0x1be;

	  /* Add the partition table marker values */
	  sector_buffer[0x1fe]=0x55;
	  sector_buffer[0x1ff]=0xaa;

	  sector_buffer[(entry_offset+0x01)]=part_table[drive_index].log_drive_start_head[index];

	  Convert_Long_To_Integer(Combine_Cylinder_and_Sector(part_table[drive_index].log_drive_start_cyl[index],part_table[drive_index].log_drive_start_sect[index]));
	  sector_buffer[(entry_offset+0x02)]=integer1;
	  sector_buffer[(entry_offset+0x03)]=integer2;

	  sector_buffer[(entry_offset+0x04)]=part_table[drive_index].log_drive_num_type[index];

	  sector_buffer[(entry_offset+0x05)]=part_table[drive_index].log_drive_end_head[index];

	  Convert_Long_To_Integer(Combine_Cylinder_and_Sector(part_table[drive_index].log_drive_end_cyl[index],part_table[drive_index].log_drive_end_sect[index]));
	  sector_buffer[(entry_offset+0x06)]=integer1;
	  sector_buffer[(entry_offset+0x07)]=integer2;

	  high = part_table[drive_index].log_drive_rel_sect[index] >> 16;
	  temp=high << 16;

	  if(part_table[drive_index].log_drive_rel_sect[index] > 0xffff)
	    {
	    low = part_table[drive_index].log_drive_rel_sect[index] - temp;
	    }
	  else
	    {
	    low = part_table[drive_index].log_drive_rel_sect[index];
	    }

	  Convert_Long_To_Integer(low);

	  sector_buffer[entry_offset+0x08]=integer1;
	  sector_buffer[entry_offset+0x09]=integer2;

	  Convert_Long_To_Integer(high);

	  sector_buffer[entry_offset+0x0a]=integer1;
	  sector_buffer[entry_offset+0x0b]=integer2;

	  high = part_table[drive_index].log_drive_num_sect[index] >> 16;
	  temp=high << 16;

	  if(part_table[drive_index].log_drive_num_sect[index] > 0xffff)
	    {
	    low = part_table[drive_index].log_drive_num_sect[index] - temp;
	    }
	  else
	    {
	    low = part_table[drive_index].log_drive_num_sect[index];
	    }

	  Convert_Long_To_Integer(low);

	  sector_buffer[entry_offset+0x0c]=integer1;
	  sector_buffer[entry_offset+0x0d]=integer2;

	  Convert_Long_To_Integer(high);

	  sector_buffer[entry_offset+0x0e]=integer1;
	  sector_buffer[entry_offset+0x0f]=integer2;

	  if(part_table[drive_index].next_ext_exists[index]==TRUE)
	    {
	    next_extended_cylinder=part_table[drive_index]
	     .next_ext_start_cyl[index];

	    next_extended_head=part_table[drive_index]
	     .next_ext_start_head[index];

	    next_extended_sector=part_table[drive_index]
	     .next_ext_start_sect[index];

	    entry_offset=entry_offset+16;

	    sector_buffer[(entry_offset+0x01)]
	     =part_table[drive_index].next_ext_start_head[index];

	    Convert_Long_To_Integer(Combine_Cylinder_and_Sector(
	     part_table[drive_index].next_ext_start_cyl
	     [index],part_table[drive_index]
	     .next_ext_start_sect[index]));

	    sector_buffer[(entry_offset+0x02)]=integer1;
	    sector_buffer[(entry_offset+0x03)]=integer2;

	    sector_buffer[(entry_offset+0x04)]=part_table[drive_index]
	     .next_ext_num_type[index];

	    sector_buffer[(entry_offset+0x05)]=part_table[drive_index]
	     .next_ext_end_head[index];

	    Convert_Long_To_Integer(Combine_Cylinder_and_Sector(
	     part_table[drive_index].next_ext_end_cyl
	     [index],part_table[drive_index].next_ext_end_sect
	     [index]));

	    sector_buffer[(entry_offset+0x06)]=integer1;
	    sector_buffer[(entry_offset+0x07)]=integer2;

	    high = part_table[drive_index].next_ext_rel_sect[index] >> 16;
	    temp=high << 16;

	    if(part_table[drive_index].next_ext_rel_sect[index] > 0xffff)
	      {
	      low = part_table[drive_index].next_ext_rel_sect[index] - temp;
	      }
	    else
	      {
	      low = part_table[drive_index].next_ext_rel_sect[index];
	      }

	    Convert_Long_To_Integer(low);

	    sector_buffer[entry_offset+0x08]=integer1;
	    sector_buffer[entry_offset+0x09]=integer2;

	    Convert_Long_To_Integer(high);

	    sector_buffer[entry_offset+0x0a]=integer1;
	    sector_buffer[entry_offset+0x0b]=integer2;

	    high = part_table[drive_index].next_ext_num_sect[index] >> 16;
	    temp=high << 16;

	    if(part_table[drive_index].next_ext_num_sect[index] > 0xffff)
	      {
	      low = part_table[drive_index].next_ext_num_sect[index] - temp;
	      }
	    else
	      {
	      low = part_table[drive_index].next_ext_num_sect[index];
	      }

	    Convert_Long_To_Integer(low);

	    sector_buffer[entry_offset+0x0c]=integer1;
	    sector_buffer[entry_offset+0x0d]=integer2;

	    Convert_Long_To_Integer(high);

	    sector_buffer[entry_offset+0x0e]=integer1;
	    sector_buffer[entry_offset+0x0f]=integer2;

	    error_code
	     =Write_Physical_Sector((drive_index+128)
	     ,extended_cylinder,extended_head,extended_sector);
	    if(error_code!=0) return(error_code);

	    extended_cylinder=next_extended_cylinder;
	    extended_head=next_extended_head;
	    extended_sector=next_extended_sector;
	    }
	  else
	    {
	    error_code=Write_Physical_Sector((drive_index+128),extended_cylinder,extended_head,extended_sector);
	    if(error_code!=0) return(error_code);

	    index=24;
	    }

	  index++;
	  }while(index<24);
	}
      }
    drive_index++;
    }while(drive_index<7);

  return(0);
}

/* Write a physical sector */
int Write_Physical_Sector(int drive, long cylinder, long head, long sector)
{
  int error_code;

  if(part_table[(drive-128)].ext_int_13==FALSE)
    {
    error_code=Write_Physical_Sector_CHS(drive,cylinder,head,sector);
    }
  else
    {
    error_code=Write_Physical_Sector_LBA(drive,cylinder,head,sector);
    }
  return(error_code);
}

/* Write a physical sector using CHS format. */
int Write_Physical_Sector_CHS(int drive, long cylinder, long head, long sector)
{
  int current_line=0;
  int error_code;

  long index=0;
  long offset=0x1be;

  if(debug.write==TRUE)
    {
    error_code=biosdisk(3, drive, head, cylinder, sector, 1, sector_buffer);
    }
  else
    {
    Clear_Screen(NULL);
    Print_Centered(4,"Write_Physical_Sector_CHS() function debugging screen",BOLD);
    Position_Cursor(4,6);
    printf("Note:  WRITE=OFF is set or an emulated disk is in existence...no");
    Position_Cursor(4,7);
    printf("       changes will be made.  Please check the \"fdisk.ini\" file");
    Position_Cursor(4,8);
    printf("       for details.");

    Position_Cursor(4,10);
    printf("Information passed to this function:");

    Position_Cursor(50,11);
    printf("Drive:     0x%X",drive);
    Position_Cursor(50,12);
    printf("Cylinder:  %d",cylinder);
    Position_Cursor(50,13);
    printf("Head:      %d",head);
    Position_Cursor(50,14);
    printf("Sector:    %d",sector);
    Position_Cursor(4,16);
    printf("Contents of partition table area in sector_buffer[]:");

    do
      {
      index=0;

      Position_Cursor(4,(current_line+18));
      printf("%d:  ",(current_line+1));
      do
	{
	printf("%02X ",sector_buffer[(index+offset)]);
	index++;
	}while(index<16);

      current_line++;
      offset=offset+16;
      }while(offset<(0x1be+64));

    Position_Cursor(4,23);
    printf("Press any key to continue.");

    asm{
      mov ah,7
      int 0x21
      }

    error_code=0;
    }

  return(error_code);
}

/* Write a physical sector using LBA format. */
int Write_Physical_Sector_LBA(int drive, long cylinder, long head, long sector)
{
  int current_line=0;
  int index=0;

  long offset=0x1be;

  unsigned int error_code=0;

  /* Get the segment and offset of disk_address_packet. */
  unsigned int disk_address_packet_address_seg=FP_SEG(disk_address_packet);
  unsigned int disk_address_packet_address_off=FP_OFF(disk_address_packet);

  /* Translate CHS values to LBA values. */
  unsigned long LBA_address=Translate_CHS_To_LBA(cylinder,head,sector
   ,part_table[(drive-128)].total_head
   ,part_table[(drive-128)].total_sect);

  /* Transfer LBA_address to disk_address_packet */
  unsigned long LBA_address_high=(LBA_address&0xffff0000)>>16;
  unsigned long LBA_address_low=LBA_address&0x0000ffff;

  Convert_Long_To_Integer(LBA_address_low);
  disk_address_packet[8]=integer1;
  disk_address_packet[9]=integer2;

  Convert_Long_To_Integer(LBA_address_high);
  disk_address_packet[10]=integer1;
  disk_address_packet[11]=integer2;

  if(debug.write==TRUE)
    {
    /* Load the registers and call the interrupt. */
    asm {
      mov ah,0x43
      mov al,0x00
      mov dl,BYTE PTR drive
      nop
      mov ds,disk_address_packet_address_seg
      mov si,disk_address_packet_address_off
      int 0x13

      mov BYTE PTR error_code,ah
      }
    }
  else
    {
    Clear_Screen(NULL);
    Print_Centered(4,"Write_Physical_Sector_LBA() function debugging screen",BOLD);
    Position_Cursor(4,6);
    printf("Note:  WRITE=OFF is set or an emulated disk is in existence...no");
    Position_Cursor(4,7);
    printf("       changes will be made.  Please check the \"fdisk.ini\" file");
    Position_Cursor(4,8);
    printf("       for details.");

    Position_Cursor(4,10);
    printf("Information passed to this function:");

    Position_Cursor(50,11);
    printf("Drive:     0x%X",drive);
    Position_Cursor(50,12);
    printf("Cylinder:  %d",cylinder);
    Position_Cursor(50,13);
    printf("Head:      %d",head);
    Position_Cursor(50,14);
    printf("Sector:    %d",sector);
    Position_Cursor(4,16);
    printf("Contents of partition table area in sector_buffer[]:");

    do
      {
      index=0;

      Position_Cursor(4,(current_line+18));
      printf("%d:  ",(current_line+1));
      do
	{
	printf("%02X ",sector_buffer[(index+offset)]);
	index++;
	}while(index<16);

      current_line++;
      offset=offset+16;
      }while(offset<(0x1be+64));

    Position_Cursor(4,23);
    printf("Press any key to continue.");

    asm{
      mov ah,7
      int 0x21
      }

    error_code=0;
    }

  return(error_code);
}
