/*
// Program:  Free FDISK
// Written By:  Brian E. Reifsnyder
// Module:  FDISK.CPP
// Module Description:  Main Free FDISK Code Module
// Version:  0.99
// Copyright:  1998-2000 under the terms of the GNU GPL
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define MAIN

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <conio.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cmd.h"
#include "main.h"
#include "fdisk.h"
#include "fdiskio.h"
#include "pdiskio.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  GLOBAL VARIABLES
/////////////////////////////////////////////////////////////////////////////
*/

extern char **environ;

/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Ask user if they want to use large disk support (FAT 32) */
void Ask_User_About_FAT32_Support()
{
  Clear_Screen(NULL);

  Print_Centered(5,"Free FDISK is capable of using large disk support to allow you to    ",0);
  Print_Centered(6,"create partitions that are greater than 2,048 MB by using FAT32      ",0);
  Print_Centered(7,"partitions.  If you enable large disk support, any partitions or     ",0);
  Print_Centered(8,"logical drives greater than 512 MB will be created using FAT32.      ",0);
  Print_Centered(10,"IMPORTANT:  If you enable large disk support many operating systems,",0);
  Print_Centered(11,"including FreeDOS, will be unable to access the partitions and      ",0);
  Print_Centered(12,"logical drives created that are over 512 MB in size.                ",0);

  Print_Centered(17,"Do you want to use large disk support (Y/N)....?    ",0);

  flags.fat32=Input(1,62,17,YN,0,0,NONE,1,0,NULL,NULL);
}

/* Calculate the end of a Partition */
void Calculate_Partition_Ending_Cylinder(long start_cylinder,unsigned long size)
{
  /* unsigned long size has to be in sectors @ 512 bytes/sector */
  /* Returns computed_ending_cylinder and computed_partition_size. */
  unsigned long delta_cylinders;

  unsigned long cylinder_size=(part_table[(flags.drive_number-0x80)]
   .total_head+1)*(part_table[(flags.drive_number-0x80)].total_sect);

  unsigned long number_of_cylinders_required=(size/cylinder_size);

  computed_partition_size=(cylinder_size*number_of_cylinders_required);
  computed_ending_cylinder=number_of_cylinders_required+start_cylinder-1;

  /* Keep the ending cylinder from going past the end of the hard disk. */
  if(computed_ending_cylinder>=part_table[(flags.drive_number-0x80)].
   total_cyl)
    {
    delta_cylinders=(computed_ending_cylinder
     -part_table[(flags.drive_number-0x80)].total_cyl)+1;
    computed_ending_cylinder=part_table[(flags.drive_number-0x80)]
     .total_cyl-1;
    computed_partition_size=computed_partition_size
     -(cylinder_size*delta_cylinders);
    }
}

/* Change Current Fixed Disk Drive */
void Change_Current_Fixed_Disk_Drive()
{
  int new_drive_number;
  int old_drive_number=flags.drive_number;

  Clear_Screen(NULL);
  Print_Centered(0,"Change Current Fixed Disk Drive",BOLD);

  Display_All_Drives();

  Position_Cursor(4,21);
  printf("Enter Fixed Disk Drive Number (1-%d)......................."
   ,(flags.maximum_drive_number-127));

  new_drive_number=Input(1,62,21,NUM,1,(flags.maximum_drive_number-127)
   ,ESCR,(flags.drive_number-127),0,NULL,NULL);

  if( (new_drive_number<=0)
   || (new_drive_number>(flags.maximum_drive_number-127)) )
    {
    flags.drive_number=old_drive_number;
    }
  else
    {
    flags.drive_number=new_drive_number+127;
    }
}

/* Clear the Active Partition */
void Clear_Active_Partition()
{
  int index=0;

  do
    {
    part_table[(flags.drive_number-128)].active_status[index]=0x00;
    index++;
    }while(index<4);

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Clear the Extended Partition Table Buffers */
void Clear_Extended_Partition_Table(int drive)
{
  int index;

  part_table[drive].ext_part_exists=FALSE;
  part_table[drive].ext_part_size_in_MB=0;
  part_table[drive].ext_part_num_sect=0;
  part_table[drive].ext_part_largest_free_space=0;

  part_table[drive].log_drive_free_space_start_cyl=0;
  part_table[drive].log_drive_free_space_end_cyl=0;

  part_table[drive].log_drive_largest_free_space_location=0;
  part_table[drive].num_of_ext_part=UNUSED;
  part_table[drive].num_of_log_drives=0;
  part_table[drive].num_of_non_dos_log_drives=0;

  index=0;
  do
    {
    part_table[drive].log_drive_num_type[index]=0;

    part_table[drive].log_drive_start_cyl[index]=0;
    part_table[drive].log_drive_start_head[index]=0;
    part_table[drive].log_drive_start_sect[index]=0;

    part_table[drive].log_drive_end_cyl[index]=0;
    part_table[drive].log_drive_end_head[index]=0;
    part_table[drive].log_drive_end_sect[index]=0;

    part_table[drive].log_drive_rel_sect[index]=0;
    part_table[drive].log_drive_num_sect[index]=0;

    part_table[drive].log_drive_size_in_MB[index]=0;
    part_table[drive].log_drive_created[index]=FALSE;

    part_table[drive].next_ext_exists[index]=FALSE;

    part_table[drive].next_ext_num_type[index]=0;

    part_table[drive].next_ext_start_cyl[index]=0;
    part_table[drive].next_ext_start_head[index]=0;
    part_table[drive].next_ext_start_sect[index]=0;

    part_table[drive].next_ext_end_cyl[index]=0;
    part_table[drive].next_ext_end_head[index]=0;
    part_table[drive].next_ext_end_sect[index]=0;

    part_table[drive].next_ext_rel_sect[index]=0;
    part_table[drive].next_ext_num_sect[index]=0;

    index++;
    }while(index<24);
}

/* Clear Screen */
void Clear_Screen(int type)       /* Clear screen code as suggested by     */
{                                 /* Ralf Quint                            */
  asm{
    mov ah,0x06   /* scroll up */
    mov al,0x00   /* 0 rows, clear whole window */
    mov bh,BYTE PTR flags.screen_color   /* set color */
    mov cx,0x0000 /* coordinates of upper left corner of screen */
    mov dh,25     /* maximum row */
    mov dl,79     /* maximum column */
    int 0x10
    }

  if(type!=NOEXTRAS)
    {
    Display_Information();
    Display_Label();
    }
}

/* Convert Long number to 2 integers */
void Convert_Long_To_Integer(long number)
{
  integer1=0;
  integer2=0;

  asm{
    mov ax,WORD PTR number

    mov BYTE PTR integer1, al
    mov BYTE PTR integer2, ah
    }
}

/* Create DOS Partition Interface */
int Create_DOS_Partition_Interface(int type)
{
  int numeric_type;
  int partition_created=FALSE;
  int partition_slot_just_used;

  long maximum_possible_percentage;

  unsigned long input=0;
  unsigned long maximum_partition_size_in_MB;

  Determine_Free_Space();

  maximum_partition_size_in_MB
   =(((part_table[(flags.drive_number-128)]
   .pri_part_largest_free_space+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect))/2048);

  /* Adjust maximum_partition_size_in_MB depending upon version */
  if( (type!=EXTENDED) && (flags.version==FOUR)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && (flags.version==FIVE)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && (flags.version==SIX)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && (flags.version==W95)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==FALSE) && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if(type==PRIMARY)
    {
    Clear_Screen(NULL);

    Print_Centered(4,"Create Primary DOS Partition",BOLD);

    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    Position_Cursor(4,8);
    printf("Do you wish to use the maximum available size for a Primary DOS Partition");

    if((flags.drive_number-128)==0)
      {
      Position_Cursor(4,9);
      printf("and make the partition active (Y/N).....................? ");
      }
    else
      {
      Position_Cursor(4,9);
      printf("(Y/N)...................................................? ");
      }

    flags.esc=FALSE;
    input=Input(1,62,9,YN,0,0,ESCR,1,0,NULL,NULL);
    if(flags.esc==TRUE) return(1);

    if(input==1)
      {
      input=maximum_partition_size_in_MB;

      if( (flags.fprmt==TRUE) && (type==PRIMARY) && (input>=128) && (input<=2048) )
	{
	Position_Cursor(4,22);
	printf("This drive is a FAT32 by default, switch to FAT16 (Y/N)?    ");
	flags.fat32=!Input(1,61,22,YN,0,0,NONE,1,0,NULL,NULL);
	}

      /* Use the maximum available free space to create a DOS Partition */

      /* Adjust numeric type depending upon partition size and the FDISK */
      /* version emulated.                                               */
      numeric_type=Partition_Type_To_Create(input);

      partition_slot_just_used=Create_Primary_Partition(numeric_type,input);
      if((flags.drive_number-128)==0) Set_Active_Partition(partition_slot_just_used);
      partition_created=TRUE;
      }
    }

  if(partition_created==FALSE)
    {
    Clear_Screen(NULL);

    if(type==PRIMARY) Print_Centered(4,"Create Primary DOS Partition",BOLD);
    else Print_Centered(4,"Create Extended DOS Partition",BOLD);

    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    Display_Primary_Partition_Information_SS();

    Position_Cursor(4,15);
    printf("Maximum space available for partition is ");
    cprintf("%4d",maximum_partition_size_in_MB);
    printf(" Mbytes ");

    maximum_possible_percentage=(100*maximum_partition_size_in_MB)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB;
    cprintf("(%3d%%)",maximum_possible_percentage);

    Position_Cursor(4,18);
    printf("Enter partition size in Mbytes or percent of disk space (%) to");
    Position_Cursor(4,19);

    if(type==PRIMARY) printf("create a Primary DOS Partition.................................: ");
    else printf("create an Extended DOS Partition...............................: ");

    flags.esc=FALSE;

    if( (flags.version==4) || (flags.version==5) || (flags.version==6) )
     input=Input(4,69,19,NUMP,1,maximum_partition_size_in_MB,ESCR
     ,maximum_partition_size_in_MB,maximum_possible_percentage,NULL,NULL);
    else input=Input(5,69,19,NUMP,1,maximum_partition_size_in_MB,ESCR
     ,maximum_partition_size_in_MB,maximum_possible_percentage,NULL,NULL);

    if(flags.esc==TRUE) return(1);

    if( (flags.fprmt==TRUE) && (type==PRIMARY) && (input>=128) && (input<=2048) )
      {
      Position_Cursor(4,22);
      printf("This drive is a FAT32 by default, switch to FAT16 (Y/N)?    ");
      flags.fat32=!Input(1,61,22,YN,0,0,NONE,1,0,NULL,NULL);
      }

    if(type==PRIMARY) numeric_type=Partition_Type_To_Create(input);
    else numeric_type=5;

    Create_Primary_Partition(numeric_type,input);
    }

  if(flags.fprmt==TRUE) flags.fat32=FALSE;

  Clear_Screen(NULL);

  if(type==PRIMARY) Print_Centered(4,"Create Primary DOS Partition",BOLD);
  else Print_Centered(4,"Create Extended DOS Partition",BOLD);

  Position_Cursor(4,6);
  printf("Current fixed disk drive: ");
  cprintf("%d",(flags.drive_number-127));

  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,21);
  if(type==PRIMARY) cprintf("Primary DOS Partition created");
  else cprintf("Extended DOS Partition created");

  Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);

  if(type==EXTENDED) Create_Logical_Drive_Interface();

  return(0);
}

/* Create Logical Drive */
/* Returns a 0 if successful and a 1 if unsuccessful */
int Create_Logical_Drive(int numeric_type, long size_in_MB)
{
  int index;
  int offset;

  unsigned long maximum_size_in_logical_sectors
   =((part_table[(flags.drive_number-128)]
   .ext_part_largest_free_space+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect));
  unsigned long maximum_size_in_MB=maximum_size_in_logical_sectors/2048;
  unsigned long size_in_logical_sectors;

  /* Adjust the size of the partition to fit boundaries, if necessary. */
  if(size_in_MB>maximum_size_in_MB)
    {
    size_in_MB=maximum_size_in_MB;
    size_in_logical_sectors=maximum_size_in_MB*2048;

    numeric_type=Partition_Type_To_Create(size_in_MB);
    }
  else
    {
    size_in_logical_sectors=size_in_MB*2048;
    }

  /* Make space in the part_table structure, if necessary. */
  if( (part_table[(flags.drive_number-128)].log_drive_largest_free_space_location<=part_table[(flags.drive_number-128)].num_of_log_drives) && (part_table[(flags.drive_number-128)].log_drive_largest_free_space_location>0) )
    {
    index=part_table[(flags.drive_number-128)].num_of_log_drives+1;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=part_table[(flags.drive_number-128)].log_drive_num_type[(index-1)];
      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],part_table[(flags.drive_number-128)].log_drive_vol_label[(index-1)]);

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=part_table[(flags.drive_number-128)].log_drive_start_head[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=part_table[(flags.drive_number-128)].log_drive_start_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=part_table[(flags.drive_number-128)].log_drive_end_head[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=part_table[(flags.drive_number-128)].log_drive_end_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=part_table[(flags.drive_number-128)].log_drive_rel_sect[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=part_table[(flags.drive_number-128)].log_drive_num_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_exists[index]=part_table[(flags.drive_number-128)].next_ext_exists[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=part_table[(flags.drive_number-128)].next_ext_num_type[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=part_table[(flags.drive_number-128)].next_ext_start_cyl[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=part_table[(flags.drive_number-128)].next_ext_start_head[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=part_table[(flags.drive_number-128)].next_ext_start_sect[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=part_table[(flags.drive_number-128)].next_ext_end_cyl[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=part_table[(flags.drive_number-128)].next_ext_end_head[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=part_table[(flags.drive_number-128)].next_ext_end_sect[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=part_table[(flags.drive_number-128)].next_ext_rel_sect[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=part_table[(flags.drive_number-128)].next_ext_num_sect[(index-1)];

      index--;
      }while(index>=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location);
    }

  /* Add the logical drive entry. */
  part_table[(flags.drive_number-128)].log_drive_num_type[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=numeric_type;
  strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location],"");

  part_table[(flags.drive_number-128)].log_drive_start_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl;
  part_table[(flags.drive_number-128)].log_drive_start_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;
  part_table[(flags.drive_number-128)].log_drive_start_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;

  Calculate_Partition_Ending_Cylinder(part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl,size_in_logical_sectors);

  part_table[(flags.drive_number-128)].log_drive_end_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_ending_cylinder;
  part_table[(flags.drive_number-128)].log_drive_end_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_head;
  part_table[(flags.drive_number-128)].log_drive_end_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;

  if( (part_table[(flags.drive_number-128)]
   .log_drive_end_cyl[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]>1023)
   && (part_table[(flags.drive_number-128)].ext_int_13==TRUE) )
   part_table[(flags.drive_number-128)]
   .log_drive_num_type[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]
   =LBA_Partition_Type_To_Create(numeric_type);

  part_table[(flags.drive_number-128)].log_drive_rel_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;
  part_table[(flags.drive_number-128)].log_drive_num_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_partition_size-part_table[(flags.drive_number-128)].total_sect;
  part_table[(flags.drive_number-128)].log_drive_size_in_MB[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_partition_size/2048;

  part_table[(flags.drive_number-128)].num_of_log_drives++;
  part_table[(flags.drive_number-128)].log_drive_created[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=TRUE;

  /* Add the linkage entry. */

  /* Add the linkage entry if there is a logical drive after this one. */
  if(part_table[(flags.drive_number-128)].log_drive_num_type[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)]>0)
    {
    part_table[(flags.drive_number-128)].next_ext_exists[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=TRUE;

    part_table[(flags.drive_number-128)].next_ext_num_type[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=5;

    part_table[(flags.drive_number-128)].next_ext_start_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)];
    part_table[(flags.drive_number-128)].next_ext_start_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=0;
    part_table[(flags.drive_number-128)].next_ext_start_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;

    part_table[(flags.drive_number-128)].next_ext_end_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)];
    part_table[(flags.drive_number-128)].next_ext_end_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_head;
    part_table[(flags.drive_number-128)].next_ext_end_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_rel_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=(part_table[(flags.drive_number-128)].log_drive_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)]-part_table[(flags.drive_number-128)].pri_part_start_cyl[part_table[(flags.drive_number-128)].num_of_ext_part])*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;
    part_table[(flags.drive_number-128)].next_ext_num_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_num_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)]+part_table[(flags.drive_number-128)].total_sect;
    }

  /* Add the linkage entry if there is a logical drive before this one. */
  if(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location>0)
    {
    part_table[(flags.drive_number-128)].next_ext_exists[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=TRUE;

    part_table[(flags.drive_number-128)].next_ext_num_type[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=5;

    part_table[(flags.drive_number-128)].next_ext_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl;
    part_table[(flags.drive_number-128)].next_ext_start_head[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=0;
    part_table[(flags.drive_number-128)].next_ext_start_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=1;

    part_table[(flags.drive_number-128)].next_ext_end_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=computed_ending_cylinder;
    part_table[(flags.drive_number-128)].next_ext_end_head[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=part_table[(flags.drive_number-128)].total_head;
    part_table[(flags.drive_number-128)].next_ext_end_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_rel_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=( (part_table[(flags.drive_number-128)].next_ext_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]) - (part_table[(flags.drive_number-128)].pri_part_start_cyl[part_table[(flags.drive_number-128)].num_of_ext_part]) )*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_num_sect
     [(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=computed_partition_size;
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  if(debug.create_partition==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(1,"int Create_Logical_Drive(int numeric_type,long size_in_MB)",BOLD);
    Position_Cursor(4,3);
    printf("int numeric_type=%d",numeric_type);
    Position_Cursor(4,4);
    printf("long size_in_MB=%d",size_in_MB);
    Position_Cursor(4,5);
    printf("Number of partition that was created:  %d",part_table[(flags.drive_number-128)].log_drive_largest_free_space_location);
    Position_Cursor(4,7);
    printf("Brief logical drive table:");

    index=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1;
    offset=9;

    Position_Cursor(4,8);
    printf(" #  NT     SC    SH    SS      EC   EH   ES      Rel. Sect.    Size in MB ");

    do
      {
      if( (index>=0) && (index<24) )
	{
	Position_Cursor(4,offset);
	printf("%2d",index);
	Position_Cursor(7,offset);
	printf("%3d",part_table[(flags.drive_number-128)].log_drive_num_type[index]);
	Position_Cursor(13,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_cyl[index]);
	Position_Cursor(19,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_head[index]);
	Position_Cursor(25,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_sect[index]);

	Position_Cursor(33,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_cyl[index]);
	Position_Cursor(38,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_head[index]);
	Position_Cursor(43,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_sect[index]);

	Position_Cursor(58,offset);
	printf("%d",part_table[(flags.drive_number-128)].log_drive_rel_sect[index]);

	Position_Cursor(72,offset);
	printf("%5d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]);
	}
      else
	{
	Position_Cursor(4,offset);
	printf("N/A");
	}

      offset++;
      index++;

      }while(offset<=11);

    Position_Cursor(4,15);
    printf("Next extended location table:");

    Position_Cursor(4,16);
    printf(" #         SC    SH    SS      EC   EH   ES      Rel. Sect.    Size in MB ");

    index=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1;
    offset=17;

    do
      {
      if( (index>=0) && (index<24) && (part_table[(flags.drive_number-128)].next_ext_exists[index]==TRUE) )
	{
	Position_Cursor(4,offset);
	printf("%2d",index);

	Position_Cursor(13,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_cyl[index]);
	Position_Cursor(19,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_head[index]);
	Position_Cursor(25,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_sect[index]);

	Position_Cursor(33,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_cyl[index]);
	Position_Cursor(38,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_head[index]);
	Position_Cursor(43,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_sect[index]);

	/*
	Temporarily removed because the size of the relative sector field
	exceeds that handled by the printf statement.  As a result,
	incorrect information is returned.

	Position_Cursor(58,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_rel_sect[index]);
	*/

	Position_Cursor(72,offset);
	printf("%4d",((part_table[(flags.drive_number-128)].next_ext_num_sect[index])/2048));
	}
      else
	{
	Position_Cursor(4,offset);
	printf("N/A");
	}

      offset++;
      index++;

      }while(offset<=19);

    Pause();
    }

  return(0);
}

/* Create Logical Drive Interface */
/* Returns a 0 if successful and a 1 if unsuccessful */
int Create_Logical_Drive_Interface()
{
  long input=0;

  int drive_created=FALSE;
  int maximum_possible_percentage;
  int numeric_type;

  unsigned long maximum_partition_size_in_MB;

  Determine_Free_Space();

  maximum_partition_size_in_MB
   =(((part_table[(flags.drive_number-128)]
   .ext_part_largest_free_space+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect))/2048);

  if(part_table[(flags.drive_number-128)]
   .ext_part_largest_free_space>=2)
    {
    do
      {
      if(flags.fprmt==TRUE) flags.fat32=TRUE;

      /* Adjust maximum_partition_size_in_MB depending upon version */
      if( (flags.version==FOUR) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( (flags.version==FIVE) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( (flags.version==SIX) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( (flags.version==W95) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( ( (flags.version==W95B) || (flags.version==W98) )
       && (flags.fat32==FALSE) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;

      Clear_Screen(NULL);

      if(drive_created==TRUE)
	{
	Position_Cursor(4,22);
	cprintf("Logical DOS Drive created, drive letters changed or added");
	}

      Print_Centered(1,"Create Logical DOS Drive in the Extended DOS Partition",BOLD);

      Display_Extended_Partition_Information_SS();

      if(1==Determine_Drive_Letters())
	{
	Position_Cursor(4,22);
	printf("                                                           ");
	Position_Cursor(4,22);
	cprintf("Maximum number of Logical DOS Drives installed.");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	if(flags.fprmt==TRUE) flags.fat32=FALSE;
	return(1);
	}

      Position_Cursor(4,17);
      printf("Total Extended DOS Partition size is ");
      cprintf("%4d",part_table[(flags.drive_number-128)].ext_part_size_in_MB);
      printf(" Mbytes (1 Mbyte = 1048576 bytes)");

      Position_Cursor(4,18);
      printf("Maximum space available for partition is ");

      if( (flags.version==4) || (flags.version==5) || (flags.version==6) )
	cprintf("%4d",maximum_partition_size_in_MB);
      else cprintf("%5d",maximum_partition_size_in_MB);

      printf(" Mbytes ");
      maximum_possible_percentage=(100*maximum_partition_size_in_MB)/part_table[(flags.drive_number-128)].ext_part_size_in_MB;
      cprintf("(%3d%%)",maximum_possible_percentage);

      Position_Cursor(4,20);
      printf("Enter logical drive size in Mbytes or percent of disk space (%)...");

      flags.esc=FALSE;

      if( (flags.version==4) || (flags.version==5) || (flags.version==6) )
       input=Input(4,70,20,NUMP,1,maximum_partition_size_in_MB,ESCR
       ,maximum_partition_size_in_MB,maximum_possible_percentage,NULL,NULL);
      else input=Input(5,70,20,NUMP,1,maximum_partition_size_in_MB,ESCR
       ,maximum_partition_size_in_MB,maximum_possible_percentage,NULL,NULL);

      if(flags.esc==TRUE)
	{
	if(flags.fprmt==TRUE) flags.fat32=FALSE;
	return(1);
	}

      if( (flags.fprmt==TRUE) && (input>=128) && (input<=2048) )
	{
	Position_Cursor(4,21);
	printf("This drive is a FAT32 by default, switch to FAT16 (Y/N)?    ");
	flags.fat32=!Input(1,61,21,YN,0,0,NONE,1,0,NULL,NULL);
	}

      numeric_type=Partition_Type_To_Create(input);

      Create_Logical_Drive(numeric_type,input);
      drive_created=TRUE;

      Determine_Free_Space();
      maximum_partition_size_in_MB
       =(((part_table[(flags.drive_number-128)]
       .ext_part_largest_free_space+1)
       *(part_table[(flags.drive_number-128)].total_head+1)
       *(part_table[(flags.drive_number-128)].total_sect))/2048);
      }while(part_table[(flags.drive_number-128)].ext_part_largest_free_space>=2);
    }

  Clear_Screen(NULL);
  Print_Centered(1,"Create Logical DOS Drive in the Extended DOS Partition",BOLD);
  Display_Extended_Partition_Information_SS();
  Position_Cursor(4,22);
  cprintf("All available space in the Extended DOS Partition");
  Position_Cursor(4,23);
  cprintf("is assigned to logical drives.");
  Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);

  if(flags.fprmt==TRUE) flags.fat32=FALSE;

  return(0);
}

/* Create a Primary Partition */
/* Returns partition number if successful and a 99 if unsuccessful */
int Create_Primary_Partition(int numeric_type,long size_in_MB)
{
  int index=0;
  int empty_partition_number=99;

  unsigned long maximum_size_in_logical_sectors
   =((part_table[(flags.drive_number-128)].pri_part_largest_free_space)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect))
   -part_table[(flags.drive_number-128)].total_sect;

  unsigned long maximum_size_in_MB=maximum_size_in_logical_sectors/2048;
  unsigned long size_in_logical_sectors;

  /* Ensure that an empty primary partition exists. */
  do
    {
    if( (empty_partition_number==99) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0) )
      {
      empty_partition_number=index;
      }
    index++;
    }while(index<4);

  /* If all primary partitions are full, report failure. */
  if(empty_partition_number==99) return(99);

  /* Adjust the size of the partition to fit boundaries, if necessary. */
  if(size_in_MB>maximum_size_in_MB)
    {
    size_in_MB=maximum_size_in_MB;
    size_in_logical_sectors=maximum_size_in_MB*2048;

    if( (numeric_type!=5) && (numeric_type!=0x0f) )
     numeric_type=Partition_Type_To_Create(size_in_MB);
    }
  else
    {
    size_in_logical_sectors=size_in_MB*2048;
    }

  /* Make sure the starting cylinder of an extended partition is at least  */
  /* 1.  If the cylinder number is 0, increment it to 1.                   */
  if( (numeric_type==5) || (numeric_type==0x0f) )
    {
    if(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl==0)
      part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl=1;
      size_in_logical_sectors=size_in_logical_sectors
       -( (part_table[(flags.drive_number-128)].total_head+1)
       *(part_table[(flags.drive_number-128)].total_sect)
       -part_table[(flags.drive_number-128)].total_sect );
    }

  Calculate_Partition_Ending_Cylinder(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl,size_in_logical_sectors);

  part_table[(flags.drive_number-128)].active_status[empty_partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_type[empty_partition_number]=numeric_type;

  part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]=part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl;
  /* If the starting cylinder is 0, then the starting head is 1...otherwise */
  /* the starting head is 1.                                                */
  if(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl==0) part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]=1;
  else part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_sect[empty_partition_number]=1;

  part_table[(flags.drive_number-128)].pri_part_end_cyl[empty_partition_number]=computed_ending_cylinder;
  part_table[(flags.drive_number-128)].pri_part_end_head[empty_partition_number]=part_table[(flags.drive_number-128)].total_head;
  part_table[(flags.drive_number-128)].pri_part_end_sect[empty_partition_number]=part_table[(flags.drive_number-128)].total_sect;

  if( (part_table[(flags.drive_number-128)]
   .pri_part_end_cyl[empty_partition_number]>1023)
   && (part_table[(flags.drive_number-128)].ext_int_13==TRUE) )
    {
    numeric_type=LBA_Partition_Type_To_Create(numeric_type);
    part_table[(flags.drive_number-128)]
     .pri_part_num_type[empty_partition_number]=numeric_type;
    }

  if(part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]>0)
    {
    part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]=(part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number])*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;
    }
  else part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]=part_table[(flags.drive_number-128)].total_sect;

  if(part_table[(flags.drive_number-128)]
   .pp_largest_free_space_start_cyl==0)
   computed_partition_size=computed_partition_size-part_table
   [(flags.drive_number-128)].total_sect;

  part_table[(flags.drive_number-128)]
   .pri_part_num_sect[empty_partition_number]
   =computed_partition_size;

  part_table[(flags.drive_number-128)].pri_part_size_in_MB[empty_partition_number]=computed_partition_size/2048;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  part_table[(flags.drive_number-128)].pri_part_created[empty_partition_number]=TRUE;

  if( (numeric_type==5) || (numeric_type==0x0f) )
    {
    part_table[(flags.drive_number-128)].ext_part_exists=TRUE;
    part_table[(flags.drive_number-128)].num_of_ext_part=empty_partition_number;
    part_table[(flags.drive_number-128)].ext_part_num_sect=computed_partition_size;
    part_table[(flags.drive_number-128)].ext_part_size_in_MB=size_in_MB;
    }

  if(debug.create_partition==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(1,"int Create_Primary_Partition(int numeric_type,long size_in_MB)",BOLD);
    Position_Cursor(4,3);
    printf("int numeric_type=%d",numeric_type);
    Position_Cursor(4,4);
    printf("long size_in_MB=%d",size_in_MB);
    Position_Cursor(4,5);
    printf("empty_partition_number=%d",empty_partition_number);

    Position_Cursor(4,8);
    printf("New Partition Information:");
    Position_Cursor(4,10);
    printf("Starting Cylinder:  %d",part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]);
    Position_Cursor(4,11);
    printf("Starting Head:      %d",part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]);
    Position_Cursor(4,12);
    printf("Starting Sector:    %d",part_table[(flags.drive_number-128)].pri_part_start_sect[empty_partition_number]);

    Position_Cursor(40,10);
    printf("Ending Cylinder:    %d",part_table[(flags.drive_number-128)].pri_part_end_cyl[empty_partition_number]);
    Position_Cursor(40,11);
    printf("Ending Head:        %d",part_table[(flags.drive_number-128)].pri_part_end_head[empty_partition_number]);
    Position_Cursor(40,12);
    printf("Ending Sector:      %d",part_table[(flags.drive_number-128)].pri_part_end_sect[empty_partition_number]);

    Position_Cursor(4,14);
    printf("Relative Sectors:   %d",part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]);

    Position_Cursor(40,14);
    printf("Size of partition in MB:    %d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[empty_partition_number]);

    Pause();
    }

  return(empty_partition_number);
}

/* Delete Extended DOS Partition Interface */
void Delete_Extended_DOS_Partition_Interface()
{
  int input=0;

  Clear_Screen(NULL);

  Print_Centered(4,"Delete Extended DOS Partition",BOLD);

  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,18);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in the deleted Extended DOS Partition will be lost.");
  Position_Cursor(4,19);
  printf("Do you wish to continue (Y/N).................? ");

  flags.esc=FALSE;
  input=Input(1,52,19,YN,0,0,ESCR,0,0,NULL,NULL);

  if( (flags.esc==FALSE) && (input==TRUE) )
    {
    Delete_Primary_Partition(part_table[(flags.drive_number-128)].num_of_ext_part);
    Clear_Extended_Partition_Table(flags.drive_number-128);

    Clear_Screen(NULL);
    Print_Centered(4,"Delete Extended DOS Partition",BOLD);
    Display_Primary_Partition_Information_SS();

    Position_Cursor(4,21);
    cprintf("Extended DOS Partition deleted");

    Position_Cursor(4,24);
    printf("                                    ");

    Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    }
}

/* Delete Logical DOS Drive */
void Delete_Logical_Drive(int logical_drive_number)
{
  int index;

  /* Zero out the partition table entry for the logical drive. */
  part_table[(flags.drive_number-128)]
   .log_drive_num_type[logical_drive_number]=0;

  strcpy(part_table[(flags.drive_number-128)]
   .log_drive_vol_label[logical_drive_number],"           ");

  part_table[(flags.drive_number-128)]
   .log_drive_start_cyl[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_start_head[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_start_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_end_cyl[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_end_head[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_end_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_rel_sect[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_num_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_size_in_MB[logical_drive_number]=0;

  /* If the logical drive to be deleted is not the first logical drive.     */
  /* Assume that there are extended partition tables after this one.  If    */
  /* there are not any more extended partition tables, nothing will be      */
  /* harmed by the shift.                                                   */
  if(logical_drive_number>0)
    {
    /* Move the extended partition information from this table to the       */
    /* previous table.                                                      */
    part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_start_head[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_head[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_start_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_head[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_head[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_num_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_num_sect[logical_drive_number];

    /* Shift all the following extended partition tables left by 1.         */
    index=logical_drive_number;

    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=part_table[(flags.drive_number-128)].log_drive_num_type[(index+1)];

      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],part_table[(flags.drive_number-128)].log_drive_vol_label[(index+1)]);

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=part_table[(flags.drive_number-128)].log_drive_start_head[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=part_table[(flags.drive_number-128)].log_drive_start_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=part_table[(flags.drive_number-128)].log_drive_end_head[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=part_table[(flags.drive_number-128)].log_drive_end_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=part_table[(flags.drive_number-128)].log_drive_rel_sect[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=part_table[(flags.drive_number-128)].log_drive_num_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=part_table[(flags.drive_number-128)].next_ext_num_type[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=part_table[(flags.drive_number-128)].next_ext_start_cyl[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=part_table[(flags.drive_number-128)].next_ext_start_head[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=part_table[(flags.drive_number-128)].next_ext_start_sect[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=part_table[(flags.drive_number-128)].next_ext_end_cyl[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=part_table[(flags.drive_number-128)].next_ext_end_head[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=part_table[(flags.drive_number-128)].next_ext_end_sect[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=part_table[(flags.drive_number-128)].next_ext_rel_sect[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=part_table[(flags.drive_number-128)].next_ext_num_sect[(index+1)];

      if(part_table[(flags.drive_number-128)].log_drive_num_type[index]>0)
	{
	part_table[(flags.drive_number-128)].next_ext_exists[(index-1)]=TRUE;
	}
      else
	{
	part_table[(flags.drive_number-128)].next_ext_exists[(index-1)]=FALSE;
	}

      index++;
      }while(index<22);
    }
  part_table[(flags.drive_number-128)].num_of_log_drives--;

  /* If there aren't any more logical drives, clear the extended        */
  /* partition table to prevent lockups by any other partition utils.   */
  if(part_table[(flags.drive_number-128)].num_of_log_drives==0)
    {
    index=0;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=0;

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=0;
      part_table[(flags.drive_number-128)].log_drive_created[index]=FALSE;

      part_table[(flags.drive_number-128)].next_ext_exists[index]=FALSE;

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=0;

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=0;

      index++;
      }while(index<24);
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Delete Logical Drive Interface */
int Delete_Logical_Drive_Interface()
{
  char char_number[1];

  int drive_to_delete=0;
  int index=0;
  int input=0;
  int input_ok;

  Clear_Screen(NULL);

  Print_Centered(1,"Delete Logical DOS Drive(s) in the Extended DOS Partition",BOLD);

  Display_Extended_Partition_Information_SS();

  Position_Cursor(4,19);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in a deleted Logical DOS Drive will be lost.");
  Position_Cursor(4,20);
  printf ("What drive do you want to delete...............................? ");

  Determine_Drive_Letters();

  //char drive_lettering_buffer[8] [27];   this line is for reference
  /* Place code to find the min and max drive letter here. */


  input_ok=FALSE;

  do
    {
    flags.esc=FALSE;

    if( (flags.del_non_dos_log_drives==TRUE) && (part_table[(flags.drive_number-128)].num_of_non_dos_log_drives>0) )
     {
     if(part_table[(flags.drive_number-128)].num_of_non_dos_log_drives>9) part_table[(flags.drive_number-128)].num_of_non_dos_log_drives=9;
     itoa(part_table[(flags.drive_number)].num_of_non_dos_log_drives,char_number,10);
     input=Input(1,69,20,CHAR,67,90,ESCR,0,0,"1",char_number);
     }
    else input=Input(1,69,20,CHAR,67,90,ESCR,0,0,NULL,NULL);
    /* Note:  min_range and max_range will need adjusted!!!!! */
    /* Changes will have to be made because the first logical drive letter */
    /* on the selected drive may not be D:, the drive letters on the       */
    /* drive may not be sequential.                                        */

    if(flags.esc==TRUE) return(1);


    if(flags.esc==FALSE)
      {
      /* Ensure that the entered character is legitimate. */
      index=4;
      do
	{
	if( (drive_lettering_buffer[(flags.drive_number-128)] [index]>0)
	 && (drive_lettering_buffer[(flags.drive_number-128)] [index]==input) )
	  {
	  input=index-4;
	  input_ok=TRUE;
	  index=30; /* break out of the loop */
	  }

	index++;
	}while(index<=26);
      }

    }while(input_ok==FALSE);

  drive_to_delete=input;

  Position_Cursor(4,22);
  printf("Are you sure (Y/N)..............................? ");
  flags.esc=FALSE;
  input=Input(1,54,22,YN,0,0,ESCR,0,0,NULL,NULL);

  if( (input==TRUE) && (flags.esc==FALSE) )
    {
    Delete_Logical_Drive(drive_to_delete);

    Clear_Screen(NULL);
    Print_Centered(1,"Delete Logical DOS Drive(s) in the Extended DOS Partition",BOLD);
    Display_Extended_Partition_Information_SS();
    input=Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    }

  return(0);
}

/* Delete Non-DOS Partition User Interface */
void Delete_N_DOS_Partition_Interface()
{
  int input=0;

  Clear_Screen(NULL);
  Print_Centered(4,"Delete Non-DOS Partition",BOLD);

  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,18);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in the deleted Non-DOS Partition will be lost.");
  Position_Cursor(4,19);
  printf("What Non-DOS Partition do you want to delete..? ");

  flags.esc=FALSE;
  input=Input(1,52,19,NUM,1,4,ESCR,-1,0,NULL,NULL); /* 4 needs changed to the max num of partitions */

  if(flags.esc==FALSE)
    {
    Delete_Primary_Partition(input-1);

    Clear_Screen(NULL);
    Print_Centered(4,"Delete Non-DOS Partition",BOLD);
    Display_Primary_Partition_Information_SS();
    Position_Cursor(4,21);
    cprintf("Non-DOS Partition deleted");
    Position_Cursor(4,24);
    printf("                                    ");

    Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    }
}

/* Delete Primary Partition */
void Delete_Primary_Partition(int partition_number)
{
  int index;

  /* If partition_number is an extended partition, first delete the */
  /* extended partition.                                            */
  if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==5)
    || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0f) )
    {
    part_table[(flags.drive_number-128)].ext_part_exists=FALSE;
    part_table[(flags.drive_number-128)].num_of_ext_part=0;

    part_table[(flags.drive_number-128)].ext_part_size_in_MB=0;
    part_table[(flags.drive_number-128)].ext_part_num_sect=0;
    part_table[(flags.drive_number-128)].ext_part_largest_free_space=0;

    part_table[(flags.drive_number-128)].log_drive_largest_free_space_location=0;
    part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl=0;
    part_table[(flags.drive_number-128)].log_drive_free_space_end_cyl=0;

    part_table[(flags.drive_number-128)].num_of_log_drives=0;

    index=0;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=0;

      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],"            ");;

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=0;

      part_table[(flags.drive_number-128)].next_ext_exists[index]=0;

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=0;

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=0;

      index++;
      }while(index<23);
    }

  part_table[(flags.drive_number-128)].active_status[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_head[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_end_cyl[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_end_head[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_end_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_rel_sect[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]=0;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  /* Delete volume label from buffer */
  strcpy(part_table[(flags.drive_number-128)].pri_part_vol_label[partition_number],"           ");
}

/* Delete Primary DOS Partition Interface */
void Delete_Primary_DOS_Partition_Interface()
{
  int input=0;
  int partition_to_delete;

  Clear_Screen(NULL);

  Print_Centered(4,"Delete Primary DOS Partition",BOLD);
  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,19);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in the deleted Primary DOS Partition will be lost.");
  Position_Cursor(4,20);
  printf("What primary partition do you want to delete..? ");

  flags.esc=FALSE;
  input=Input(1,52,20,NUM,1,4,ESCR,-1,0,NULL,NULL); /* 4 needs changed to the max num of partitions */

  if(flags.esc==FALSE)
    {
    partition_to_delete=input-1;

    Position_Cursor(4,22);
    printf("Are you sure (Y/N)..............................? ");
    flags.esc=FALSE;
    input=Input(1,54,22,YN,0,0,ESCR,0,0,NULL,NULL);

    if( (input==TRUE) && (flags.esc==FALSE) )
      {
      Delete_Primary_Partition(partition_to_delete);

      Clear_Screen(NULL);

      Print_Centered(4,"Delete Primary DOS Partition",BOLD);
      Display_Primary_Partition_Information_SS();
      Position_Cursor(4,21);
      cprintf("Primary DOS Partition deleted");

      Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
      }
    }
}

/* Determine if the video display will support boldfacing text */
void Determine_Color_Video_Support()
{
  /* Changed to code suggested by Ralf Quint. */

  unsigned char videomode;
  unsigned char maxcolumn;

  asm{

    mov ah,0x0f
    int 0x10
    mov videomode,al
    mov maxcolumn,ah
    }

  if(videomode==7)                /* monochrome mode */
    {
    flags.monochrome=TRUE;
    textcolor(7);
    }
  else                            /* assume color mode */
    {
    flags.monochrome=FALSE;
    textcolor(15);
    }
}

/* Determine the locations of free space in the partition table */
void Determine_Free_Space()
{
  int first_used_partition=UNUSED;
  int last_used_partition=UNUSED;
  int index;
  int sub_index;
  int swap;

  int drive=flags.drive_number-0x80;

  unsigned long free_space_after_last_used_partition=0;
  unsigned long free_space_before_first_used_partition=0;
  unsigned long free_space_between_partitions_0_and_1=0;
  unsigned long free_space_between_partitions_1_and_2=0;
  unsigned long free_space_between_partitions_2_and_3=0;

  unsigned long cylinder_size=(part_table[drive].total_head+1)*(part_table[drive].total_sect);

  /* Reset the physical order to default */
  index=0;
  do
    {
    part_table[drive].pri_part_physical_order[index]=index;

    index++;
    }while(index<4);

  /* Determine the location and size of the largest free space in the */
  /* primary partition.                                               */

  /* 1.  Sort the primary partitions based upon starting cylinder and their*/
  /*     contents...or lack of use.                                        */

  /* Place all empty partitions at the end of the table. */
  index=0;
  do
    {

    sub_index=0;
    do
      {
      if(part_table[drive].pri_part_num_type[part_table[drive].pri_part_physical_order[sub_index]]==0)
	{
	swap=part_table[drive].pri_part_physical_order[sub_index];
	part_table[drive].pri_part_physical_order[sub_index]=part_table[drive].pri_part_physical_order[(sub_index+1)];
	part_table[drive].pri_part_physical_order[(sub_index+1)]=swap;
	}
      sub_index++;
      }while(sub_index<3);

    index++;
    }while(index<4);

  if(debug.determine_free_space==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 1",BOLD);

    printf("\n\nCylinder Size (total heads * total sectors)=%d\n",cylinder_size);

    printf("\nContents after initial sorting of unused partitions to end:\n\n");

    index=0;
    do
      {
      printf("Partition %1d:  %1d    ",index,part_table[drive].pri_part_physical_order[index]);
      printf("SC:  %4d    ",part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("EC:  %4d    ",part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("Size in MB:  %4d\n",part_table[drive].pri_part_size_in_MB[part_table[drive].pri_part_physical_order[index]]);

      index++;
      }while(index<4);
    Position_Cursor(4,20);
    Pause();
    }

  /* Order the partitions based upon starting cylinder */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      if( (part_table[drive].pri_part_num_type
       [part_table[drive].pri_part_physical_order
       [sub_index]]!=0) && (part_table[drive]
       .pri_part_num_type[part_table[drive]
       .pri_part_physical_order[(sub_index+1)]]!=0)
       && (part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[sub_index]]
       >part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[(sub_index+1)]]) )
	{
	swap=part_table[drive].pri_part_physical_order[sub_index];
	part_table[drive].pri_part_physical_order[sub_index]=part_table[drive].pri_part_physical_order[(sub_index+1)];
	part_table[drive].pri_part_physical_order[(sub_index+1)]=swap;
	}
      sub_index++;
      }while(sub_index<3);
    index++;
    }while(index<4);

  if(debug.determine_free_space==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 2",BOLD);

    printf("\n\nCylinder Size (total heads * total sectors)=%d\n",cylinder_size);
    printf("\nContents after sorting partitions by starting cylinder:\n\n");

    index=0;
    do
      {
      printf("Partition %d:  %1d    ",index,part_table[drive].pri_part_physical_order[index]);
      printf("SC:  %4d    ",part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("EC:  %4d    ",part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("Size in MB:  %4d\n",part_table[drive].pri_part_size_in_MB[part_table[drive].pri_part_physical_order[index]]);

      index++;
      }while(index<4);
    }

  /* 2.  Is there any free space before the first partition? */

  /* Find the first used partition and the last used partition. */
  index=0;
  do
    {
    if( (first_used_partition==UNUSED)
     && (part_table[drive].pri_part_num_type[part_table[drive]
     .pri_part_physical_order[index]]>0) )
      {
      first_used_partition=index;
      }

    if(part_table[drive].pri_part_num_type
     [part_table[drive].pri_part_physical_order[index]]>0)
      {
      last_used_partition=index;
      }

    index++;
    }while(index<4);

  if(first_used_partition!=UNUSED)
    {
    if(part_table[drive].pri_part_start_cyl
     [part_table[drive].pri_part_physical_order[first_used_partition]]>0)
      {
      free_space_before_first_used_partition
       =(part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[first_used_partition]]);
      }
    else free_space_before_first_used_partition=0;
    }

  /* 3.  Is there any free space after the last used partition? */
  if(first_used_partition!=UNUSED)
    {
    if(part_table[drive].pri_part_end_cyl
     [part_table[drive].pri_part_physical_order[last_used_partition]]<=part_table[drive].total_cyl)
      {
      free_space_after_last_used_partition
       =(part_table[drive].total_cyl-part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[last_used_partition]]);
      }
    }

  /* 4.  Is there any free space between partitions?                    */
  /*                                                                    */
  if( (first_used_partition!=UNUSED) && (last_used_partition>=1) )
    {
    if( (part_table[drive].pri_part_end_cyl
     [part_table[drive].pri_part_physical_order[0]]+1)
     <(part_table[drive].pri_part_start_cyl[part_table[drive]
     .pri_part_physical_order[1]]) )
      {
      free_space_between_partitions_0_and_1
       =(part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[1]]
       -part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[0]]);
      }
    }

  if( (first_used_partition!=UNUSED) && (last_used_partition>=2) )
    {
    if( (part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]+1)<(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]) )
      {
      free_space_between_partitions_1_and_2=(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]-part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]);
      }
    }

  if( (first_used_partition!=UNUSED) && (last_used_partition==3) )
    {
    if( (part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]+1)<(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]) )
      {
      free_space_between_partitions_2_and_3=(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]-part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]);
      }
    }

  /* Locate the largest free space */
  if(first_used_partition!=UNUSED)
    {
    /* */
    part_table[drive].pp_largest_free_space_start_head=0;
    part_table[drive].pp_largest_free_space_start_sect=1;

    /* Default the largest free space to before the first used partition */
    part_table[drive].pri_part_largest_free_space=free_space_before_first_used_partition;
    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_end_cyl
     =part_table[drive].pri_part_start_cyl
     [part_table[drive].pri_part_physical_order[first_used_partition]]-1;

    /* If the largest free space is not before the first used partition  */
    /* make the correct adjustments.                                     */
    if(free_space_after_last_used_partition>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_after_last_used_partition;
      part_table[drive].pp_largest_free_space_start_cyl
       =part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[last_used_partition]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].total_cyl;
      }

    if(free_space_between_partitions_0_and_1>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_0_and_1;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[0]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[1]]-1;
      }

    if(free_space_between_partitions_1_and_2>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_1_and_2;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]-1;
      }

    if(free_space_between_partitions_2_and_3>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_2_and_3;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]-1;
      }
    }
  else
    {
    part_table[drive].pri_part_largest_free_space=part_table[drive].total_cyl;
    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].total_cyl-1;
    }

  if(part_table[drive].pri_part_largest_free_space<=0) part_table[drive].pri_part_largest_free_space=0;

  if(debug.determine_free_space==TRUE)
    {
    printf("\n\nFree space (in cylinders) in primary partition table:\n");
    printf("\nFree space before first used partition:  %d",free_space_before_first_used_partition);
/* */
    printf("\nFree space after last used partition:  %d",free_space_after_last_used_partition);
    printf("\nFree space between partitions 0 and 1:  %d",free_space_between_partitions_0_and_1);
    printf("\nFree space between partitions 1 and 2:  %d",free_space_between_partitions_1_and_2);
/* */
    printf("\nFree space between partitions 2 and 3:  %d",free_space_between_partitions_2_and_3);
    printf("\n\nLargest free space in primary partition table:  %d",part_table[drive].pri_part_largest_free_space);
    printf("\nStarting cylinder of largest free space:  %d",part_table[drive].pp_largest_free_space_start_cyl);
    printf("\nEnding cylinder of largest free space:  %d",part_table[drive].pp_largest_free_space_end_cyl);
    Pause();
    }

  /* Determine the location and size of the largest free space in the */
  /* extended partition, if it exists.                                */
  if(part_table[drive].ext_part_exists==TRUE)
    {
    part_table[drive].ext_part_largest_free_space=0;
    part_table[drive].log_drive_free_space_start_cyl=0;
    part_table[drive].log_drive_free_space_end_cyl=0;
    part_table[drive].log_drive_largest_free_space_location=0;

    if(debug.determine_free_space==TRUE)
      {
      Clear_Screen(NULL);
      Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 3",BOLD);
      printf("\n\n");
      }

    if(part_table[drive].num_of_log_drives>0)
      {
      /* If there are logical drives in the extended partition first find  */
      /* the largest free space between the logical drives...if it exists. */
      last_used_partition=UNUSED;
      index=0;

      /* Check to see if the first possible entry in the extended partition */
      /* is unused.  If it is unused and there is a logical drive after it  */
      /* then skip checking for free space between entry 0 and 1.           */
      if(part_table[drive].log_drive_num_type[0]==0) index=1;

      do
	{
	if(part_table[drive].log_drive_num_type[index]>0)
	  {
	  last_used_partition=index;
	  }

	if( (part_table[drive].log_drive_start_cyl[(index+1)]-part_table[drive].log_drive_end_cyl[index]) >1 )
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[(index+1)]-1)-(part_table[drive].log_drive_end_cyl[index]+1);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].log_drive_end_cyl[index]+1;
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[(index+1)]-1;
	  part_table[drive].log_drive_largest_free_space_location=index+1;
	  }

	if(debug.determine_free_space==TRUE)
	  {
	  if(index==12)
	    {
	    printf("\n");
	    Pause();
	    }

	  printf("\nLogical Drive #: %2d    ",index);
	  printf("SC: %4d    ",part_table[drive].log_drive_start_cyl[index]);
	  printf("EC: %4d    ",part_table[drive].log_drive_end_cyl[index]);
	  }

	index++;
	}while(index<22);

      if(debug.determine_free_space==TRUE)
	{
	printf("\nLogical Drive #: %2d    ",index);
	printf("SC: %4d    ",part_table[drive].log_drive_start_cyl[22]);
	printf("EC: %4d    \n",part_table[drive].log_drive_end_cyl[22]);
	Pause();

	Clear_Screen(NULL);
	Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 4",BOLD);
	printf("\n\nNote:  All values are in cylinders.\n\n");
	printf("Results of free space calculations after computing spaces between\n  logical drives:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}

      /* Determine if there is any free space before the first logical      */
      /* drive in the extended partition.                                   */
      if(part_table[drive].log_drive_num_type[0]!=0)
	{
	if( (part_table[drive].log_drive_start_cyl[0]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part])>part_table[drive].ext_part_largest_free_space)
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[0]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[0]-1;
	  part_table[drive].log_drive_largest_free_space_location=0;
	  }
	}
      else
	{
	if( (part_table[drive].log_drive_start_cyl[1]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part])>part_table[drive].ext_part_largest_free_space)
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[1]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[1]-1;
	  part_table[drive].log_drive_largest_free_space_location=0;
	  }
	}

      if(debug.determine_free_space==TRUE)
	{
	Clear_Screen(NULL);
	Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 5",BOLD);
	printf("\n\nResults of free space calculations after computing space before\n  the first logical drive:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	}

      /* Determine if there is any free space after the last logical drive  */
      /* in the extended partition.                                         */
      if( (last_used_partition<23) && (part_table[drive].log_drive_end_cyl[last_used_partition]<part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]) )
	{
	if( (part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].log_drive_end_cyl[last_used_partition])>(part_table[drive].ext_part_largest_free_space) )
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].log_drive_end_cyl[last_used_partition]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].log_drive_end_cyl[last_used_partition]+1;
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-1;
	  part_table[drive].log_drive_largest_free_space_location=last_used_partition+1;
	  }
	}

      if(debug.determine_free_space==TRUE)
	{
	printf("\n\nResults of free space calculations after computing space after\n  the last logical drive:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
      }
    else
      {
      /* If the extended partition is empty. */
      part_table[drive].ext_part_largest_free_space=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]-1;

      part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
      part_table[drive].log_drive_free_space_end_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part];

      if(debug.determine_free_space==TRUE)
	{
	printf("\n\nThere are not any Logical DOS Drives in the Extended DOS Partition\n\n");
	printf("Location of largest free space:     0\n");
	printf("Size of largest free space:  %d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
      }
    }
}

/* Display information for all hard drives */
void Display_All_Drives()
{
  int current_column_offset_of_general_drive_information;
  int current_column_offset=4;
  int current_line=3;
  int current_line_of_general_drive_information;
  int drive=1;
  int drive_letter_index=0;
  int index;

  long space_used_on_drive_in_MB;

  unsigned long usage;

  Determine_Drive_Letters();

  Position_Cursor(2,2);
  printf("Disk   Drv   Mbytes   Free   Usage");

  do
    {
    if(current_line>18)
      {
      current_line=3;
      current_column_offset=45;

      Position_Cursor(43,2);
      printf("Disk   Drv   Mbytes   Free   Usage");
      }

    /* Print physical drive information */
    current_column_offset_of_general_drive_information=current_column_offset;
    current_line_of_general_drive_information=current_line;
    space_used_on_drive_in_MB=0;

    /* Print drive number */
    Position_Cursor(current_column_offset_of_general_drive_information,current_line);
    cprintf("%d",drive);

    /* Print size of drive */
    Position_Cursor((current_column_offset_of_general_drive_information+12),current_line);
    printf("%4d",part_table[(drive-1)].total_hard_disk_size_in_MB);

    /* Get space_used_on_drive_in_MB */
    index=0;
    do
      {
      if( (part_table[(drive-1)].pri_part_num_type[index]!=5)
       && (part_table[(drive-1)].pri_part_num_type[index]!=15)
       && (part_table[(drive-1)].pri_part_num_type[index]!=0) )
       space_used_on_drive_in_MB
       =space_used_on_drive_in_MB
       +part_table[(drive-1)].pri_part_size_in_MB[index];

      index++;
      }while(index<=3);

    index=0;
    do
      {
      if(part_table[(drive-1)].log_drive_num_type[index]>0)
       space_used_on_drive_in_MB
       =space_used_on_drive_in_MB
       +part_table[(drive-1)].log_drive_size_in_MB[index];

      index++;
      }while(index<=22);

    /* Print logical drives on disk, if applicable */

    drive_letter_index=0;
    do
      {
      if(drive_lettering_buffer[(drive-1)] [drive_letter_index]>0)
	{
	current_line++;

	if(current_line>18)
	  {
	  current_line=3;
	  current_column_offset=45;

	  Position_Cursor(43,2);
	  printf("Disk   Drv   Mbytes   Free   Usage");
	  }

	/* Print drive letter of logical drive */
	Position_Cursor((current_column_offset+6),current_line);
	printf("%c",drive_lettering_buffer[(drive-1)] [drive_letter_index]);
	Position_Cursor((current_column_offset+7),current_line);
	printf(":");

	/* Print size of logical drive */
	Position_Cursor((current_column_offset+12),current_line);

	if(drive_letter_index<4)
	  {
	  printf("%4d",part_table[(drive-1)].pri_part_size_in_MB[drive_letter_index]);
	  }
	else
	  {
	  printf("%4d",part_table[(drive-1)].log_drive_size_in_MB[(drive_letter_index-4)]);
	  }
	}

      drive_letter_index++;
      }while(drive_letter_index<27);

    /* Print amount of free space on drive */
    if(part_table[(drive-1)].total_hard_disk_size_in_MB>space_used_on_drive_in_MB)
      {
      Position_Cursor((current_column_offset_of_general_drive_information+20),current_line_of_general_drive_information);
      printf("%4d",(part_table[(drive-1)].total_hard_disk_size_in_MB-space_used_on_drive_in_MB));
      }

    /* Print drive usage percentage */
    if(space_used_on_drive_in_MB==0) usage=0;
    else usage=1+((100*space_used_on_drive_in_MB)/(part_table[(drive-1)].total_hard_disk_size_in_MB));
    if(usage>100) usage=100;

    Position_Cursor((current_column_offset_of_general_drive_information+28),current_line_of_general_drive_information);
    printf("%3d%%",usage);

    current_line++;
    drive++;
    }while(drive<=(flags.maximum_drive_number-127));

  Position_Cursor(4,20);
  printf("(1 Mbyte = 1048576 bytes)");
}

void Display_CL_Partition_Table()
{
  int index=0;

  unsigned long usage=0;

  Determine_Drive_Letters();

  printf("\n\nCurrent fixed disk drive: %1d",(flags.drive_number-127));
  if(flags.extended_options_flag==TRUE)
    {
    printf("                  (TC: %4d",part_table[(flags.drive_number-128)].total_cyl);
    printf(" TH: %3d",part_table[(flags.drive_number-128)].total_head);
    printf(" TS: %3d)",part_table[(flags.drive_number-128)].total_sect);
    }

  printf("\n\nPartition   Status   Mbytes   Description     Usage  ");
  if(flags.extended_options_flag==TRUE) printf("Start Cyl  End Cyl");
  printf("\n");

  index=0;
  do
    {
    if(part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
      {
      /* Drive Letter of Partition */
      if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) || ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) && ( (flags.version==W95B) || (flags.version==W98) ) ) )
	{
	printf(" %1c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	}
      else printf("   ");

      /* Partition Number */
      printf(" %1d",(index+1));

      if(flags.extended_options_flag==TRUE)
	{
	/* Partition Type */
	printf(" %3d",(part_table[(flags.drive_number-128)].pri_part_num_type[index]));
	}
      else printf("    ");

      /* Status */
      if(part_table[(flags.drive_number-128)].active_status[index]>0)
	{
	printf("      A");
	}
      else printf("       ");

      /* Mbytes */
      printf("    %6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]);

      /* Description */
      printf("   %15s",partition_lookup_table_buffer_long[part_table[(flags.drive_number-128)].pri_part_num_type[(index)]]);

      /* Usage */
      usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
      if(usage>100) usage=100;

      printf("   %3d%%",usage);

      if(flags.extended_options_flag==TRUE)
	{
	/* Starting Cylinder */
	printf("    %4d",part_table[(flags.drive_number-128)].pri_part_start_cyl[index]);

	/* Ending Cylinder */
	printf("      %4d",part_table[(flags.drive_number-128)].pri_part_end_cyl[index]);
	}
      printf("\n");
      }

    index++;
    }while(index<4);

  /* Check to see if there are any drives to display */
  if( (brief_partition_table[(flags.drive_number-128)] [4]>0) || (brief_partition_table[(flags.drive_number-128)] [5]>0) )
    {
    printf("\nContents of Extended DOS Partition:\n");
    printf("Drv Volume Label  Mbytes  System  Usage\n");

    /* Display information for each Logical DOS Drive */
    index=4;
    do
      {
      if( (brief_partition_table[(flags.drive_number-128)] [index]==1) || (brief_partition_table[flags.drive_number-128] [index]==4) || (brief_partition_table[flags.drive_number-128] [index]==6) )
	{
	/* Display drive letter */
	printf(" %1c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);

	/* Display volume label */
	printf(" %11s",part_table[(flags.drive_number-128)].log_drive_vol_label[index-4]);

	/* Display size in MB */
	printf("    %4d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-4)]);

	/* Display file system type */
	printf("  %-8s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].log_drive_num_type[(index-4)]]);

	/* Display usage in % */
	usage=((part_table[(flags.drive_number-128)].log_drive_num_sect[index-4]*100)/part_table[(flags.drive_number-128)].ext_part_num_sect);
	if(usage>100) usage=100;

	printf("  %3d%%",usage);

	printf("\n");
	}

      index++;
      }while(index<27);
    }
}

/* Display Help Screens */
void Display_Help_Screen()
{
  char name[20];

  if(flags.use_freedos_label==TRUE)
    {
    strcpy(name,ALTNAME);
    strcat(name," FDISK");
    }
  else strcpy(name,PRINAME);

  char version[40];
  strcpy(version,"Version ");
  strcat(version,VERSION);

  if(flags.do_not_pause_help_information==FALSE) Clear_Screen(NOEXTRAS);

  printf("\n\n%-20s                   %40s\n", name, version);
  printf("Written By:  Brian E. Reifsnyder\n\n");
  printf("Syntax:\n\n");
  printf("  %s                   Runs %s in interactive mode.\n",filename,name);
  printf("  %s /? [/NOPAUSE]     Displays this help information.\n\n",filename);

  printf("%s compatibility switches:\n\n",name);
  printf("  %s /ACTOK\n",filename);
  printf("  %s /CMBR <drive#>\n",filename);
  printf("  %s /EXT:<size> <drive#> [{/LOG:<size>} | {/LOGO:<size>}]\n",filename);
  printf("  %s /FPRMT\n",filename);
  printf("  %s /MBR\n",filename);
  printf("  %s /PRI:<size> <drive#>\n",filename);
  printf("  %s /PRIO:<size> <drive#>\n",filename);
  printf("  %s /Q\n",filename);
  printf("  %s /STATUS\n",filename);
  printf("  %s /X\n",filename);
  printf("\n\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    }

  printf("%s extended switches:\n\n",name);
  printf("For partition table modification:\n");

  printf("  %s /AUTO [<drive#>]\n",filename);
  printf("  %s /ACTIVATE:<partition#> [<drive#>]\n",filename);
  printf("  %s /CLEAR [<drive#>]\n",filename);
  printf("  %s /DELETE {/PRI | /EXT | /LOG:<partition#>\n",filename);
  printf("           | /NUM:<partition#>} [<drive#>]\n");
  printf("  %s /DEACTIVATE [<drive#>]\n",filename);
  printf("  %s /EXT:<size>[,100] \n",filename);
  printf("           [ {/LOG:<size>[,100] [/SPEC:<type#>]} \n");
  printf("           | {/LOGO:<size>[,100] [/SPEC:<type#>]} ] [<drive#>]\n");
  printf("  %s /LOG:<size>[,100] [/SPEC:<type#>] [<drive#>]\n",filename);
  printf("  %s /LOGO:<size>[,100] [/SPEC:<type#>] [<drive#>] \n",filename);
  printf("  %s /MODIFY:<partition#>,<newtype#> [<drive#>]\n",filename);
  printf("  %s /PRI:<size>[,100] [/SPEC:<type#>] [<drive#>] \n",filename);
  printf("  %s /PRIO:<size>[,100] [/SPEC:<type#>] [<drive#>] \n",filename);

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n\n\n\n\n");
    Pause();
    Clear_Screen(NOEXTRAS);
    printf("%s extended switches(cont.):\n",name);
    }

  printf("\nFor MBR modification:\n");
  printf("  %s /AMBR [<drive#>]\n",filename);
  printf("  %s /MBR [<drive#>]\n",filename);
  printf("  %s /RMBR [<drive#>]\n",filename);
  printf("  %s /SMBR [<drive#>]\n",filename);

  printf("\nFor handling flags on a hard disk:\n");
  printf("  %s /CLEARFLAG[{:<flag#>} | /ALL} ] [<drive#>]\n",filename);
  printf("  %s /SETFLAG:<flag#>[,<flag_value>] [<drive#>]\n",filename);
  printf("  %s /TESTFLAG:<flag#>[,<flag_value>] [<drive#>]\n",filename);

  printf("\nFor obtaining information about the hard disk(s):\n");
  printf("  %s /DUMP\n",filename);
  printf("  %s /INFO [/TECH] [<drive#>]\n",filename);

  printf("\nFor rebooting the computer:\n");
  printf("  %s /REBOOT\n",filename);
  printf("\n\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n");
    Pause();
    Clear_Screen(NOEXTRAS);
    }

  printf("\nDescription of %s compatibility switches:\n\n",name);
  printf("  /ACTOK   Normally skips integrity checking...kept for compatibility.\n");
  printf("  /CMBR    Writes the MBR to <drive#>.\n");
  printf("  /EXT     Creates an Extended DOS Partition of <size>MB on <drive#>.  /LOG\n");
  printf("             creates a logical drive of <size>MB.  /LOGO will force the\n");
  printf("             creation of a FAT16 logical drive of <size>MB.\n");
  printf("  /FPRMT   Prompts for FAT32/FAT16 in interactive mode.\n");
  printf("  /MBR     Writes the MBR to the first hard disk.\n");
  printf("  /PRI     Creates a partition of <size>MB on <drive#>.\n");
  printf("  /PRIO    Creates a FAT16 partition of <size>MB on <drive#>.\n");
  printf("  /Q       Keeps the system from rebooting after you exit %s.\n",name);
  printf("             (Note:  %s will not reboot after you exit unless\n",name);
  printf("             rebooting is enabled in the \"fdisk.ini\" file.)\n");
  printf("  /STATUS  Displays the current partition layout.\n",filename);
  printf("  /X       Do not use LBA partitions.\n",filename);

  printf("\n\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n\n\n\n");
    Pause();
    Clear_Screen(NOEXTRAS);
    }

  printf("\nDescription of %s extended switches:\n",name);
  printf("\nFor partition table modification:\n");

  printf("  /AUTO    Automatically partitions the hard disk with FAT16 partitions.\n");
  printf("  /ACTIVATE\n");
  printf("           Sets <partition#> active.\n");
  printf("  /CLEAR   Erases the partition tables.  USE WITH CAUTION!\n");
  printf("  /DELETE  Deletes a partition.\n");
  printf("  /DEACTIVATE\n");
  printf("           Removes the active status from a partition on <drive#>.\n");
  printf("  /EXT     Same as /EXT above, except \",100\" will indicate that <size>\n");
  printf("             is a percentage and \"/SPEC\" will specify the partition\n");
  printf("             <type#>.  /LOGO will force %s to not use FAT32\n",name);
  printf("             partitions.\n");
  printf("  /LOG     Same as /LOG above, except for the addition of extra\n");
  printf("             functionallity.  See /EXT, above, for more information.\n");
  printf("  /LOGO    Same as /LOGO above, except for the addition of extra\n");
  printf("             functionallity.\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n\n\n\n");
    Pause();
    Clear_Screen(NOEXTRAS);

    printf("For partition table modification(cont.):\n");
    }

  printf("  /MODIFY  Changes the partition type of <partition#> to <newtype#>.\n");
  printf("             Note:  The starting <partition#> for logical partitions\n");
  printf("                    is \"5\".\n");
  printf("  /PRI     Same as /PRI above, except for the addition of extra\n");
  printf("             functionallity.  See /EXT, above, for more information.\n");
  printf("  /PRIO    Same as /PRIO above, except for the addition of extra\n");
  printf("             functionallity.  See /EXT, above, for more information.\n");
  printf("\nFor MBR modification:\n");
  printf("  /AMBR    Writes the MBR stored in the \"boot.mbr\" file.\n");
  printf("  /MBR     Writes the MBR to <drive#>.\n");
  printf("  /RMBR    Removes the MBR from <drive#>.\n",filename);
  printf("  /SMBR    Saves the MBR, on <drive#>, into a \"boot.mbr\" file.\n",filename);

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n\n\n\n\n\n\n\n");
    Pause();
    Clear_Screen(NOEXTRAS);
    }

  printf("\nFor handling flags on a hard disk:\n");
  printf("  /CLEARFLAG\n");
  printf("           Resets the <flag#> or all the flags on <drive#> to 0.\n");
  printf("  /SETFLAG\n");
  printf("           Sets <flag#> to 1 or <flag_value>, if specified.\n");
  printf("  /TESTFLAG:\n");
  printf("           Tests <flag#> for 1 or <flag_value>, if specified.\n");
  printf("\nFor obtaining information about the hard disk(s):\n");
  printf("  /DUMP    Dumps all partition information from all hard disks.  This\n");
  printf("             function is mainly used for debugging by redirecting the\n");
  printf("             output to a file.\n");
  printf("  /INFO    Displays partition information from <drive#>\n");
  printf("\nFor rebooting the computer:\n");
  printf("  /REBOOT  Forces a reboot.\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n\n\n\n\n\n");
    Pause();
    Clear_Screen(NOEXTRAS);
    }
  else printf("\n\n");

  printf("\n\n");
  printf("This program is Copyright %s, by Brian E. Reifsnyder, under\n",COPYLEFT);
  printf("the terms of the GNU General Public License.\n");
  printf("\nThis program comes as-is and without warranty of any kind.  The author of\n");
  printf("this software assumes no responsibility pertaining to the use or mis-use of\n");
  printf("this software.  By using this software, the operator is understood to be\n");
  printf("agreeing to the terms of the above.\n");

  if(flags.do_not_pause_help_information==FALSE) printf("\n\n\n\n\n\n\n\n");
}

/* Display Information */
void Display_Information()
{
  if(flags.extended_options_flag==TRUE)
    {
    Position_Cursor(0,0);
    if(flags.version==FOUR) cprintf("4");
    if(flags.version==FIVE) cprintf("5");
    if(flags.version==SIX) cprintf("6");
    if(flags.version==W95) cprintf("W95");
    if(flags.version==W95B) cprintf("W95B");
    if(flags.version==W98) cprintf("W98");

    Position_Cursor(5,0);
    if(flags.partition_type_lookup_table==INTERNAL)
     cprintf("INT");
    else
     cprintf("EXT");

    Position_Cursor(9,0);
    if(flags.use_extended_int_13==TRUE) cprintf("LBA");

    Position_Cursor(13,0);
    if(flags.fat32==TRUE) cprintf("FAT32");

    Position_Cursor(72,0);
    if(flags.use_ambr==TRUE) cprintf("AMBR");

    Position_Cursor(77,0);
    if(flags.partitions_have_changed==TRUE) cprintf("C");

    Position_Cursor(79,0);
    if(flags.extended_options_flag==TRUE) cprintf("X");
    }

  if(debug.emulate_disk>0)
    {
    Position_Cursor(66,0);
    cprintf("E%1d",debug.emulate_disk);
    }

  if(debug.write==FALSE)
    {
    Position_Cursor(69,0);
    cprintf("RO");
    }
}

/* Display Label */
void Display_Label()
{
  if(flags.label==TRUE)
    {
    int index=0;

    char label[20];

    strcpy(label,PRINAME);

    do
      {
      Position_Cursor(79,((index*2)+3));
      printf("%c",label[index]);
      index++;
      }while(index<10);
    }
}

/* Display Extended Partition Information Sub Screen */
void Display_Extended_Partition_Information_SS()
{
  int column_index=0;
  int index;
  int print_index=4;

  unsigned long usage;

  Determine_Drive_Letters();

  /* Check to see if there are any drives to display */
  if( (brief_partition_table[(flags.drive_number-128)] [4]>0) || (brief_partition_table[(flags.drive_number-128)] [5]>0) )
    {
    Position_Cursor(0,3);
    printf("Drv Volume Label  Mbytes  System  Usage");

    /* Display information for each Logical DOS Drive */
    index=4;
    print_index=4;
    do
      {
      if( (brief_partition_table[(flags.drive_number-128)] [index]==1)
       || (brief_partition_table[(flags.drive_number-128)] [index]==4)
       || (brief_partition_table[(flags.drive_number-128)] [index]==6)

       || ( ( (brief_partition_table[(flags.drive_number-128)] [index]==0x0b)
       || (brief_partition_table[(flags.drive_number-128)] [index]==0x0c) )
       && ( (flags.version==W95B) || (flags.version==W98) ) )

       || ( (brief_partition_table[(flags.drive_number-128)] [index]==0x0e)
       && ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) ) ) )
	{
	if(print_index>15)
	  {
	  column_index=41;
	  print_index=4;

	  Position_Cursor(41,3);
	  printf("Drv Volume Label  Mbytes  System  Usage");
	  }

	/* Display drive letter */
	Position_Cursor((column_index+0),print_index);
	cprintf("%c",drive_lettering_buffer[(flags.drive_number-128)] [index]);

	Position_Cursor((column_index+1),print_index);
	cprintf(":");

	/* Display volume label */
	Position_Cursor((column_index+4),print_index);
	printf("%11s",part_table[(flags.drive_number-128)].log_drive_vol_label[index-4]);

	/* Display size in MB */
	Position_Cursor((column_index+19),print_index);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-4)]);

	/* Display file system type */
	Position_Cursor((column_index+25),print_index);
	printf("%s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].log_drive_num_type[(index-4)]]);

	/* Display usage in % */
	usage=((part_table[(flags.drive_number-128)].log_drive_num_sect[index-4]*100)/part_table[(flags.drive_number-128)].ext_part_num_sect);
	if(usage>100) usage=100;

	Position_Cursor((column_index+35),print_index);
	printf("%3d%%",usage);

	print_index++;
	}

      else if(brief_partition_table[(flags.drive_number-128)] [index]>0)
	     {
	     if(print_index>15)
	       {
	       column_index=41;
	       print_index=4;

	       Position_Cursor(41,3);
	       printf("Drv Volume Label  Mbytes  System  Usage");
	       }

	     if(flags.del_non_dos_log_drives==TRUE)
	       {
	       /* Display drive number */
	       Position_Cursor((column_index+0),print_index);
	       cprintf("%c",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	       }

	     /* Display size in MB */
	     Position_Cursor((column_index+19),print_index);
	     printf("%4d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-4)]);

	     /* Display file system type */
	     Position_Cursor((column_index+25),print_index);
	     printf("%s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].log_drive_num_type[(index-4)]]);

	     /* Display usage in % */
	     usage=((part_table[(flags.drive_number-128)].log_drive_num_sect[index-4]*100)/part_table[(flags.drive_number-128)].ext_part_num_sect);
	     if(usage>100) usage=100;

	     Position_Cursor((column_index+35),print_index);
	     printf("%3d%%",usage);

	     print_index++;
	     }
      index++;
      }while(index<27);
    }
  else
    {
    Position_Cursor(4,10);
    cprintf("No logical drives defined");
    }

  Position_Cursor(4,17);
  printf("Total Extended DOS Partition size is ");
  cprintf("%4d",(part_table[flags.drive_number-128].ext_part_size_in_MB) );
  printf(" Mbytes (1 Mbyte = 1048576 bytes)");
}

/* Display Or Modify Logical Drive Information in the extended partition */
void Display_Or_Modify_Logical_Drive_Information()
{
  char char_number[1];

  int continue_loop;
  int index;
  int input;
  int input_ok;

  Beginning:

  Clear_Screen(NOEXTRAS);

  if(flags.extended_options_flag==FALSE)
   Print_Centered(1,"Display Logical DOS Drive Information",BOLD);
  else Print_Centered(1,"Display/Modify Logical DOS Drive Information",BOLD);

  Display_Extended_Partition_Information_SS();

  if(flags.extended_options_flag==FALSE)
   Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
  else
    {
    Position_Cursor(4,18);
    printf("Enter the character of the logical drive you want to modify.....?");

    Determine_Drive_Letters();

    continue_loop=TRUE;
    do
      {
      flags.esc=FALSE;

      if( (flags.del_non_dos_log_drives==TRUE) && (part_table[(flags.drive_number-128)].num_of_non_dos_log_drives>0) )
	{
	if(part_table[(flags.drive_number-128)].num_of_non_dos_log_drives>9) part_table[(flags.drive_number-128)].num_of_non_dos_log_drives=9;
	itoa(part_table[(flags.drive_number-128)].num_of_non_dos_log_drives,char_number,10);
	input=Input(1,69,18,CHAR,68,90,ESCC,0,0,"1",char_number);
	}
      else input=Input(1,69,18,CHAR,68,90,ESCC,0,0,NULL,NULL);

      if(flags.esc==FALSE)
	{
	/* Ensure that the entered character is legitimate. */
	index=4;
	do
	  {
	  if( (drive_lettering_buffer[(flags.drive_number-128)] [index]>0) && (drive_lettering_buffer[(flags.drive_number-128)] [index]==input) )
	    {
	    input=index-4;
	    input_ok=TRUE;
	    index=30; /* break out of the loop */
	    }

	    index++;
	  }while(index<=26);
	}

      if(input_ok==TRUE) continue_loop=FALSE;
      if(flags.esc==TRUE) continue_loop=FALSE;

      }while(continue_loop==TRUE);

    if( (input_ok==TRUE) && (flags.esc==FALSE) )
      {
      Modify_Extended_Partition_Information(input);
      goto Beginning;
      }
    }
}

/* Display/Modify Partition Information */
void Display_Partition_Information()
{
  int input;

  Beginning:

  Clear_Screen(NULL);
  if(flags.extended_options_flag==FALSE)
   Print_Centered(4,"Display Partition Information",BOLD);
  else Print_Centered(4,"Display/Modify Partition Information",BOLD);

  Display_Primary_Partition_Information_SS();

  if(part_table[(flags.drive_number-128)].num_of_log_drives>0)
    {
    Position_Cursor(4,17);
    printf("The Extended DOS Partition contains Logical DOS Drives.");

    Position_Cursor(4,18);
    printf("Do you want to display the logical drive information (Y/N)......?");

    if(flags.extended_options_flag==TRUE)
      {
      Position_Cursor(4,19);
      printf("  (Optional:  Type the number of the partition to modify.)");

      input=Input(1,69,18,YN,0,0,ESCR,1,0,"1","4");

      if( ((input-48)>=1) && ((input-48)<=4) )
	{
	Modify_Primary_Partition_Information((input-48));
	goto Beginning;
	}
      }
    else input=Input(1,69,18,YN,0,0,ESCR,1,0,NULL,NULL);

    if(input==TRUE)
      {
      Display_Or_Modify_Logical_Drive_Information();
      if(flags.extended_options_flag==TRUE) goto Beginning;
      }
    }
  else
    {
    if(flags.extended_options_flag==FALSE)
     Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    else
     {
     Position_Cursor(4,18);
     printf("Enter the number of the partition you want to modify (1-4)......?");

     flags.esc=FALSE;
     input=Input(1,69,18,NUM,1,4,ESCR,1,0,NULL,NULL);

     if(flags.esc==FALSE)
       {
       Modify_Primary_Partition_Information(input);
       goto Beginning;
       }
     }
    }
}

/* Display Primary Partition information Sub-screen */
void Display_Primary_Partition_Information_SS()
{
  int cursor_offset=0;
  int index=0;

  unsigned long usage=0;

  Determine_Drive_Letters();

  Position_Cursor(4,6);
  printf("Current fixed disk drive: ");
  cprintf("%d",(flags.drive_number-127));

  if( (part_table[(flags.drive_number-128)].pri_part_num_type[0]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[1]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[2]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[3]>0) )
    {
    if(flags.extended_options_flag==FALSE)
      {
      Position_Cursor(4,8);
      printf("Partition  Status   Type    Volume Label  Mbytes   System   Usage");

      index=0;
      do
	{
	if(part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
	  {
	  /* Drive Letter of Partition */
	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  /* Partition Number */
	  Position_Cursor(8,(cursor_offset+9));
	  cprintf("%d",(index+1));

	  /* Status */
	  if(part_table[(flags.drive_number-128)].active_status[index]>0)
	    {
	    Position_Cursor(18,(cursor_offset+9));
	    printf("A");
	    }

	  /* Type */
	  if(  (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6)  )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  if(part_table[(flags.drive_number-128)].pri_part_num_type[index]==5)
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("EXT DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=1) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=4) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=5) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=6)  )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("Non-DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0f) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("EXT DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  /* Volume Label */
	  Position_Cursor(33,(cursor_offset+9));
	  printf("%11s",part_table[(flags.drive_number-128)].pri_part_vol_label[index]);

	  /* Mbytes */
	  Position_Cursor(45,(cursor_offset+9));
	  printf("%6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]);

	  /* System */
	  Position_Cursor(54,(cursor_offset+9));
	  printf("%s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].pri_part_num_type[(index)]]);

	  /* Usage */
	  usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
	  if(usage>100) usage=100;

	  Position_Cursor(65,(cursor_offset+9));
	  printf("%3d%%",usage);

	  cursor_offset++;
	  }

	index++;
	}while(index<4);
      }
    else
      {
      Position_Cursor(4,8);
      printf("Partition   Status   Mbytes    Description    Usage  Start Cyl  End Cyl");

      index=0;
      do
	{
	if(part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
	  {
	  /* Drive Letter of Partition */
	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  /* Partition Number */
	  Position_Cursor(8,(cursor_offset+9));
	  cprintf("%d",(index+1));

	  /* Partition Type */
	  Position_Cursor(10,(cursor_offset+9));
	  printf("%3d",(part_table[(flags.drive_number-128)].pri_part_num_type[index]));

	  /* Status */
	  if(part_table[(flags.drive_number-128)].active_status[index]>0)
	    {
	    Position_Cursor(19,(cursor_offset+9));
	    printf("A");
	    }

	  /* Mbytes */
	  Position_Cursor(24,(cursor_offset+9));
	  printf("%6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]);

	  /* Description */
	  Position_Cursor(33,(cursor_offset+9));
	  printf("%15s",partition_lookup_table_buffer_long[part_table[(flags.drive_number-128)].pri_part_num_type[index]]);

	  /* Usage */
	  usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
	  if(usage>100) usage=100;

	  Position_Cursor(51,(cursor_offset+9));
	  printf("%3d%%",usage);

	  /* Starting Cylinder */
	  Position_Cursor(59,(cursor_offset+9));
	  /* */
	  printf("%4d",part_table[(flags.drive_number-128)].pri_part_start_cyl[index]);

	  /* Ending Cylinder */
	  Position_Cursor(69,(cursor_offset+9));
	  printf("%4d",part_table[(flags.drive_number-128)].pri_part_end_cyl[index]);

	  cursor_offset++;
	  }

	index++;
	}while(index<4);
      }
    }
  else
    {
    Position_Cursor(4,21);
    cprintf("No partitions defined");
    }

  Position_Cursor(4,14);
  printf("Total disk space is ");
  cprintf("%4d",part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
  printf(" Mbytes (1 Mbyte = 1048576 bytes)");
}

/* Dump the partition tables from all drives to screen */
void Dump_Partition_Information()
{
  int index=0;
  flags.extended_options_flag=TRUE;

  do
    {
    flags.drive_number=index+128;
    Display_CL_Partition_Table();
    index++;
    }while(index<=7);
}

/* Get Environment Settings */
int Get_Environment_Settings(char *environment[])
{
  char command_buffer[255];
  char setting_buffer[255];

  int character_index=0;
  int done_looking;
  int line_index=0;
  int number;
  int sub_buffer_index;

  if(environment[0] [0]==NULL) return(1);


  while( (environment[line_index] [0]!=NULL) && (line_index<64) )
    {
    /* Clear the command_buffer and setting_buffer */
    character_index=0;

    do
      {
      command_buffer[character_index]=0x00;
      setting_buffer[character_index]=0x00;

      character_index++;
      }while(character_index<255);

    /* Extract the command and setting from the line_buffer */

    /* Find the command */
    character_index=0;
    sub_buffer_index=0;

    done_looking=FALSE;
    do
      {
      if(environment[line_index] [character_index]!='=')
	{
	command_buffer[sub_buffer_index]
	 =environment[line_index][character_index];
	}

      if(environment[line_index] [character_index]=='=')
	 {
	 done_looking=TRUE;
	 }

      sub_buffer_index++;
      character_index++;
      if(character_index>=255) done_looking=TRUE;
      }while(done_looking==FALSE);

    /* Find the setting */
    sub_buffer_index=0;
    done_looking=FALSE;

    do
      {
      if( (environment[line_index] [character_index]==NULL)
       || (environment[line_index] [character_index]==0   )
       || (environment[line_index] [character_index]==32) )done_looking=TRUE;

      if( (environment[line_index] [character_index]!='=')
       && (environment[line_index] [character_index]!=NULL) )
	{
	setting_buffer[sub_buffer_index]
	 =environment[line_index][character_index];

	setting_buffer[sub_buffer_index]
	 =toupper(setting_buffer[sub_buffer_index]);

	sub_buffer_index++;
	}

      character_index++;
      if(character_index>=255) done_looking=TRUE;
      }while(done_looking==FALSE);

    /* Adjust for the possibility of TRUE or FALSE in the environment. */
    if(0==strcmp(setting_buffer,"TRUE")) strcpy(setting_buffer,"ON");
    if(0==strcmp(setting_buffer,"FALSE")) strcpy(setting_buffer,"OFF");

    /* Process the command found in the line buffer */

    /* Check for the ALLOW_ABORT statement */
    if(0==strcmp(command_buffer,"FFD_ALLOW_ABORT"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.allow_abort=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.allow_abort=FALSE;
      }

    /* Check for the AMBR statement */
    if(0==strcmp(command_buffer,"FFD_AMBR"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.use_ambr=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.use_ambr=FALSE;
      }

    /* Check for the COLORS statement */
    if(0==strcmp(command_buffer,"FFD_COLORS"))
      {
      number=atoi(setting_buffer);

      if( (number>=0) && (number<=127) )
	{
	flags.screen_color=number;
	}
      }

    /* Check for the DEL_ND_LOG statement */
    if(0==strcmp(command_buffer,"FFD_DEL_ND_LOG"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.del_non_dos_log_drives=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.del_non_dos_log_drives=FALSE;
      }

    /* Check for the EIPTLT statement */
    if(0==strcmp(command_buffer,"FFD_EIPTLT"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.eiptlt=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.eiptlt=FALSE;
      }

    /* Check for the FLAG_SECTOR statement */
    if(0==strcmp(command_buffer,"FFD_FLAG_SECTOR"))
      {
      number=atoi(setting_buffer);
      if(number==0) flags.flag_sector=0;
      if( (number>=2) && (number<=64) ) flags.flag_sector=number;
      if(number==256) flags.flag_sector=part_table[0].total_sect;
      }

    /* Check for the LABEL statement */
    if(0==strcmp(command_buffer,"FFD_LABEL"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.label=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.label=FALSE;
      }

    /* Check for the MONO statement */
    if(0==strcmp(command_buffer,"FFD_MONO"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.monochrome=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.monochrome=FALSE;
      }

    /* Check for the REBOOT statement */
    if(0==strcmp(command_buffer,"FFD_REBOOT"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.reboot=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.reboot=FALSE;
      }

    /* Check for the SET_ANY_ACT statement */
    if(0==strcmp(command_buffer,"FFD_SET_ANY_ACT"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.set_any_pri_part_active=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.set_any_pri_part_active=FALSE;
      }


    /* Check for the VERSION statement */
    if(0==strcmp(command_buffer,"FFD_VERSION"))
      {
      if(0==strcmp(setting_buffer,"4")) flags.version=FOUR;
      if(0==strcmp(setting_buffer,"5")) flags.version=FIVE;
      if(0==strcmp(setting_buffer,"6")) flags.version=SIX;
      if(0==strcmp(setting_buffer,"W95")) flags.version=W95;
      if(0==strcmp(setting_buffer,"W95B")) flags.version=W95B;
      if(0==strcmp(setting_buffer,"W98")) flags.version=W98;
      if(0==strcmp(setting_buffer,"FD"))
	{
	flags.version=FREEDOS_VERSION;
	flags.use_freedos_label=TRUE;
	}
      }

    /* Check for the XO statement */
    if(0==strcmp(command_buffer,"FFD_XO"))
      {
      if(0==strcmp(setting_buffer,"ON")) flags.extended_options_flag=TRUE;
      if(0==strcmp(setting_buffer,"OFF")) flags.extended_options_flag=FALSE;
      }

    /* Check for the LANG statement */
    if(0==strcmp(command_buffer,"LANG"))
      {
      strncpy(flags.language,setting_buffer,2);
      }

    line_index++;
    }

  return(0);
}

/* Initialize flags, variables, load fdisk.ini, load part.ini, etc. */
void Initialization(char *environment[])
{
  int value1;
  int value2;

  long index;

  unsigned long sector_buffer_segment;
  unsigned long sector_buffer_offset;

  flags.display_name_description_copyright=TRUE;
  flags.do_not_pause_help_information=FALSE;
  flags.monochrome=FALSE;
  flags.partitions_have_changed=FALSE;
  flags.fprmt=FALSE;

  flags.drive_number=128;

  /* Clear the user_defined_chs_settings structure */
  index=0;
  do
    {
    user_defined_chs_settings[index].defined=FALSE;
    user_defined_chs_settings[index].total_cylinders=0;
    user_defined_chs_settings[index].total_heads=0;
    user_defined_chs_settings[index].total_sectors=0;

    index++;
    }while(index<8);

  Load_External_Lookup_Table();

  /* If the part.ini file is not found, load the internal lookup table. */
  if(flags.partition_type_lookup_table==INTERNAL)
    {
    index=1;
    do
      {
      if( (index!=5) && (index!=15) )
	{
	strcpy(partition_lookup_table_buffer_short[index],"Unknown ");
	strcpy(partition_lookup_table_buffer_long[index],"Unknown        ");
	}
      index++;
      }while(index<=255);

    strcpy(partition_lookup_table_buffer_short[1],"FAT12   ");
    strcpy(partition_lookup_table_buffer_short[4],"FAT16   ");
    strcpy(partition_lookup_table_buffer_short[5],"Extended");
    strcpy(partition_lookup_table_buffer_short[6],"FAT16   ");
    strcpy(partition_lookup_table_buffer_short[11],"FAT32   ");
    strcpy(partition_lookup_table_buffer_short[12],"FAT32   ");
    strcpy(partition_lookup_table_buffer_short[14],"FAT16   ");
    strcpy(partition_lookup_table_buffer_short[15],"Extended");

    strcpy(partition_lookup_table_buffer_long[1],"FAT12          ");
    strcpy(partition_lookup_table_buffer_long[4],"FAT16          ");
    strcpy(partition_lookup_table_buffer_long[5],"Extended       ");
    strcpy(partition_lookup_table_buffer_long[6],"FAT16          ");
    strcpy(partition_lookup_table_buffer_long[11],"FAT32          ");
    strcpy(partition_lookup_table_buffer_long[12],"FAT32          ");
    strcpy(partition_lookup_table_buffer_long[14],"FAT16          ");
    strcpy(partition_lookup_table_buffer_long[15],"Extended       ");

    if(flags.eiptlt==TRUE)
      {
      }
    }

  Determine_Color_Video_Support();
  Process_Fdiskini_File();

  Get_Environment_Settings(&*environment);

  if(flags.extended_options_flag==TRUE)
    {
    flags.allow_abort=TRUE;
    flags.del_non_dos_log_drives=TRUE;
    flags.set_any_pri_part_active=TRUE;
    }

  /* Set monochrome mode, if it is desired. */
  if(flags.monochrome==TRUE) textcolor(7);
  else textcolor(15);

  /* Check for interrupt 0x13 extensions (If the proper version is set.) */
  if( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) )
   Check_For_INT13_Extensions();

  /* If the version is W95B or W98 then default to FAT32 support.  */
  if( (flags.version==W95B) || (flags.version==W98) ) flags.fat32=TRUE;

  /* Initialize LBA structures, if necessary. */
  if(flags.use_extended_int_13==TRUE) Initialize_LBA_Structures();

  Read_Partition_Tables();

  if( (flags.flag_sector>part_table[(flags.drive_number-128)].total_sect)
   && (flags.flag_sector!=0) )
    {
    printf("The \"FLAG_SECTOR\" value in the \"fdisk.ini\" file is out of range...\n");
    printf("Operation Terminated.\n");
    exit(3);
    }
}

/* Get input from keyboard */
unsigned long Input(int size_of_field,int x_position,int y_position,int type
 ,int min_range,long max_range,int return_message,int default_value
 ,long maximum_possible_percentage,char optional_char_1[1],char optional_char_2[1])
{
  /*
  size_of_field:                 number of characters for the user to enter,
				 if size of field is 0 then no input box
				 is drawn on the screen.
  x_position, y_position:        screen coordinates to place the input box
  type                           type of input--CHAR  A single character as
						      specified by min_range
						      and max_range.  min_range
						      and max_range are the
						      min. and max. ASCII
						      values possible for this
						      input field. The ASCII
						      values in these 2 fields
						      should be of capital
						      characters only.  If the
						      user enters a lower case
						      character it will be
						      converted to uppercase.
						YN    Either a yes or no.
						NUM   A number as specified
						      by min_range and
						      max_range.
						NUMP  A number or percentage
						      within the limits of
						      the size_of_field.
						ESC   Waits for the ESC key
						      only
  return_message				ESCR  Displays "Press
						      Esc to return to FDISK
						      Options"
						ESCE  Displays "Press
						      Esc to exit FDISK"
						ESCC  Displays "Press Esc to
						      continue"
						NONE  Does not display a
						      return message.
  default_value                  The default value that is displayed for
				 input.  This option only works with the NUM
				 type or the YN type.
				 Set this to -1 if it is not used.
  maximum_possible_percentage                   If type is NUMP, this is the
						maximum percentage possible.
  optional_char_1[1] and
  optional_char_2[1]             2 optional character fields for use with
				 the NUM type when size_of_field==1
				 Also is used as two option number fields
				 (converted to char value) when type==CHAR
				 and a single digit numeric input is possible.
				 In this case these two variables define a
				 range.  When type==YN this functions the same
				 as with NUM type above.
  */

  char input;
  char line_buffer[18];

  unsigned long multiplier;

  int char_max_range;
  int default_value_preentered=FALSE;
  int index;
  int invalid_input=FALSE;
  int line_buffer_index=0;
  int proper_input_given=FALSE;
  int percent_entered=FALSE;
  int percent_just_entered=FALSE;

  unsigned long data_max_range=max_range;
  unsigned long data;

  /* Clear line buffer */
  index=0;
  do
    {
    line_buffer[index]=0;
    index++;
    }while(index<10);

  /* Place appropriate text on the screen prior to obtaining input */
  if(type!=ESC)
    {
    Position_Cursor(x_position,y_position);

    cprintf("[");

    index=0;
    do
      {
      cprintf(" ");
      index++;
      }while(index<size_of_field);

    cprintf("]");
    }

  /* Display the return message */
  if( (return_message==ESCR) || (return_message==ESCE) || (return_message==ESCC) )
    {
    Position_Cursor(4,24);
    printf("                                                 ");
    Position_Cursor(4,24);
    printf("Press ");
    cprintf("Esc");
    printf(" to ");
    }

  if(return_message==ESCR) printf("return to FDISK options");

  if(return_message==ESCE) printf("exit FDISK");

  if(return_message==ESCC) printf("continue");

  /* Set the default value for NUM type, if applicable */
  if( (default_value>=0) && (type==NUM) && (size_of_field==1) )
    {
    Position_Cursor(x_position+1,y_position);
    printf("%d",default_value);
    line_buffer_index=0;
    line_buffer[0]=default_value+48;
    }

  /* Set the default value for NUMP type, if applicable */
  if( (default_value>=0) && (type==NUMP) && (size_of_field>1) )
    {
    itoa(default_value,line_buffer,10);
    line_buffer_index=strlen(line_buffer);

    /* Display line_buffer */
    index=line_buffer_index;
    do
      {
      Position_Cursor((x_position+size_of_field-line_buffer_index+index)
       ,y_position);
      index--;
      cprintf("%c",line_buffer[index]);
      }while(index>0);

    default_value_preentered=TRUE;
    }

  /* Set the default value for YN type, if applicable */
  if( (default_value>=0) && (type==YN) && (size_of_field==1) )
    {
    Position_Cursor(x_position+1,y_position);

    if(default_value==1)
      {
      printf("Y");
      line_buffer_index=0;
      line_buffer[0]='Y';
      data=TRUE;
      }

    if(default_value==0)
      {
      printf("N");
      line_buffer_index=0;
      line_buffer[0]='N';
      data=FALSE;
      }
    }

  do
    {
    if(type!=ESC) Position_Cursor((size_of_field+x_position),y_position);

    /* Obtain keypress from keyboard */
    asm{
      mov ah,7
      int 0x21
      mov BYTE PTR input,al
      }

    /* Zero the default value if type==NUMP, the enter, esc, or backspace key */
    /* has not been pressed, and the default value is pre-entered. */
    if( (default_value>=0) && (type==NUMP) && (size_of_field>1)
     && (input!=8) && (input!=13) && (input!=27)
     && (default_value_preentered==TRUE) )
      {
      line_buffer_index=0;

      index=0;
      do
	{
	line_buffer[index]=0;
	index++;
	}while(index<10);

      default_value_preentered=FALSE;
      }

    /* Clear error messages from screen */
    if(type!=YN)
      {
      Position_Cursor(4,22);
      printf("                                                              ");
      Position_Cursor(4,24);
      }

    Position_Cursor(4,23);
    printf("                                                    ");
    Position_Cursor(4,24);

    /* Esc key has been hit */
    if(input==27)
      {
      flags.esc=TRUE;
      proper_input_given=TRUE;
      data=0;
      type=99;
      }

    /* Enter key has been hit */
    if(input==13)
      {
      if( ( (type==CHAR) || (type==YN) ) && (line_buffer[0]!=0) && ( (data==TRUE) || (data==FALSE) || (data!=99) ) )
	{
	proper_input_given=TRUE;

	type=99;
	}

      if( (type==NUMYN) && (line_buffer[0]!=0) )
	{
	data=line_buffer[0];
	proper_input_given=TRUE;
	type=99;
	}

      if( (type==CHARNUM) && (line_buffer[0]!=0) )
	{
	proper_input_given=TRUE;
	data=line_buffer[0];

	type=99;
	}

      if( (type==NUMCHAR) && (line_buffer[0]!=0) )
	{
	proper_input_given=TRUE;
	data=line_buffer[0];

	type=99;
	}

      if( (type==NUM) && (line_buffer[0]!=0) )
	{
	proper_input_given=TRUE;

	/* Convert line_buffer to an unsigned integer in data */
	data=0;
	index=strlen(line_buffer)-1;
	multiplier=1;
	do
	  {
	  data=data+((line_buffer[index]-48)*multiplier);
	  index--;
	  multiplier=multiplier*10;
	  }while(index>=0);

	/* Make sure that data is <= max_range */
	if(data>data_max_range)
	  {
	  data=0;
	  proper_input_given=FALSE;

	  Position_Cursor(4,22);
	  cprintf("Requested partition size exceeds the maximum available space");

	  /* Set input=0xff to avoid processing this time around */
	  input='\xff';
	  }
	else type=99;
	}

      if( (type==NUMP) && (line_buffer[0]!=0) )
	{
	proper_input_given=TRUE;

	/* Convert line_buffer to an unsigned integer in data */
	data=0;
	index=strlen(line_buffer)-1;

	if(percent_entered==TRUE) index--;

	multiplier=1;
	do
	  {
	  data=data+((line_buffer[index]-48)*multiplier);
	  index--;
	  multiplier=multiplier*10;
	  }while(index>=0);


	if(percent_entered==TRUE) data=(data*data_max_range)/maximum_possible_percentage;

	/* Make sure that data is <= max_range */
	if(data>data_max_range)
	  {
	  data=0;
	  proper_input_given=FALSE;

	  Position_Cursor(4,22);
	  cprintf("Requested partition size exceeds the maximum available space");

	  /* Set input=0xff to avoid processing this time around */
	  input='\xff';
	  }
	else type=99;
	}

      if( (debug.input_routine==TRUE) && (type==99) )
	{
	Clear_Screen(NULL);

	printf("Input entered by user:  %d",data);
	Pause();
	}
      }

    if(debug.input_routine==TRUE)
      {
      Position_Cursor(50,22);
      printf("                  ");

      Position_Cursor(50,22);
      printf("Input:  %d",input);
      }

    /* Process the backspace key if type==CHARNUM. */
    if( (type==CHARNUM) && (input==8) )
      {
      type=NUM;

      input='\xff';
      line_buffer[0]='0';
      line_buffer_index=1;

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    /* Process a legitimate entry if type==CHARNUM. */
    if( (type==CHARNUM) && ( ((input-48)>=1) && ((input-48)<=max_range) ) )
      {
      type=NUM;

      line_buffer[0]='0';
      line_buffer_index=1;
      }

    if( (type==CHARNUM)
     && ( (input==optional_char_1[0])
     || ( (input-32)==optional_char_1[0])
     || (input==optional_char_2[0])
     || ( (input-32)==optional_char_2[0]) ) )
      {
      if(input>=97) input=input-32;

      line_buffer_index=1;
      line_buffer[0]=input;
      input='\xff';
      type=CHARNUM;

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    /* Process a legitimate entry if type==NUMYN. */
    if( (type==NUMYN) && ( (input=='Y') || (input=='N') || (input=='y')
     || (input=='n') ) )
      {
      type=YN;

      line_buffer[0]=' ';
      line_buffer_index=1;
      }

    /* Process a legitimate entry if type==NUMCHAR. */
    if( (type==NUMCHAR) && (optional_char_1[0]!=NULL)
     && (optional_char_2[0]!=NULL) )
      {
      char_max_range=atoi(optional_char_2);

      if( (input>='1') && (input<=(char_max_range+48)) )
	{
	line_buffer_index=1;
	line_buffer[0]=input;
	type=NUMCHAR;

	Position_Cursor((x_position+1),y_position);
	cprintf("%c",line_buffer[0]);
	}

      if( (input<'1') || (input>(char_max_range+48)) )
	{
	line_buffer_index=0;
	line_buffer[0]=0;
	type=CHAR;
	}
      }

    /* Process optional character fields. */
    if( (type==NUM) && ( (optional_char_1[0]!=NULL)
     || (optional_char_2[0]!=NULL) ) )
      {
      if( (input==optional_char_1[0]) || ( (input-32)==optional_char_1[0]) )
	{
	if(input>=97) input=input-32;

	line_buffer_index=1;
	line_buffer[0]=input;
	input='\xff';
	type=CHARNUM;

	Position_Cursor((x_position+1),y_position);
	cprintf("%c",line_buffer[0]);
	}

      if( (input==optional_char_2[0]) || ( (input-32)==optional_char_2[0]) )
	{
	if(input>=97) input=input-32;

	line_buffer_index=1;
	line_buffer[0]=input;
	input='\xff';
	type=CHARNUM;

	Position_Cursor((x_position+1),y_position);
	cprintf("%c",line_buffer[0]);
	}
      }

    if( (type==CHAR) && (optional_char_1[0]!=NULL)
     && (optional_char_2[0]!=NULL) )
      {
      char_max_range=atoi(optional_char_2);

      if( (input>='1') && (input<=(char_max_range+48)) )
	{
	line_buffer_index=1;
	line_buffer[0]=input;
	type=NUMCHAR;

	Position_Cursor((x_position+1),y_position);
	cprintf("%c",line_buffer[0]);
	}
      }

    if( ( (type==YN) || (type==NUMYN) ) && (optional_char_1[0]!=NULL)
     && (optional_char_2[0]!=NULL) )
      {
      char_max_range=atoi(optional_char_2);

      if( (input>='1') && (input<=(char_max_range+48)) )
	{
	line_buffer_index=1;
	line_buffer[0]=input;
	type=NUMYN;

	Position_Cursor((x_position+1),y_position);
	cprintf("%c",line_buffer[0]);
	}
      }

    if(type==CHAR)
      {
      /* Convert to upper case, if necessary. */
      if(input>=97) input=input-32;

      if( (input>=min_range) && (input<=max_range) )
	{
	line_buffer[0]=input;
	data=input;
	}
      else
	{
	proper_input_given=FALSE;
	line_buffer[0]=' ';
	data=99;

	Position_Cursor(4,23);
	cprintf("Invalid entry, please enter %c-",min_range);
	cprintf("%c.",max_range);
	}

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    /* Process the backspace key if type==NUMCHAR. */
    if( (type==NUMCHAR) && (input==8) )
      {
      type=CHAR;

      line_buffer[0]=' ';
      line_buffer_index=1;

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    if(type==YN)
      {
      switch (input) {
	case 'Y':
	  line_buffer[0]='Y';
	  data=TRUE;
	  break;
	case 'y':
	  line_buffer[0]='Y';
	  data=TRUE;
	  break;
	case 'N':
	  line_buffer[0]='N';
	  data=FALSE;
	  break;
	case 'n':
	  line_buffer[0]='N';
	  data=FALSE;
	  break;
	default:
	  proper_input_given=FALSE;
	  line_buffer[0]=' ';
	  data=99;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry, please enter Y-N.");

	}

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    /* Process the backspace key if type==NUMYN. */
    if( (type==NUMYN) && (input==8) )
      {
      type=YN;
      line_buffer[0]=' ';
      line_buffer_index=1;

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    if( (type==NUM) && (input!='\xff') )
      {
      /* If the backspace key has not been hit. */
      if(input!=8)
	{
	invalid_input=FALSE;

	if(size_of_field>1)
	  {
	  min_range=0;
	  max_range=9;
	  }

	if( (input>='0') && (input<='9') )input=input-48;
	else
	  {
	  if(input<10) input=11;
	  }

	if( ( (size_of_field>1) && (input>max_range) ) || (input>9) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry, please enter %d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( (size_of_field==1) && ( (input<min_range) || ( (input>max_range) && (input<10) ) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("%d is not a choice, please enter ",input);
	  cprintf("%d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field>1) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry.");
	  invalid_input=TRUE;
	  }

	if( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field==1) )
	  {
	  line_buffer_index=0;
	  }

	if(invalid_input==FALSE)
	  {
	  if( (line_buffer_index==1) && (line_buffer[0]=='0') )
	    {
	    line_buffer[0]=0;
	    line_buffer_index=0;
	    }

	  line_buffer[line_buffer_index]=(input+48);
	  line_buffer_index++;
	  }
	}
      else
	{
	/* If the backspace key has been hit */
	line_buffer_index--;
	if(line_buffer_index<0) line_buffer_index=0;
	line_buffer[line_buffer_index]=0;

	if(line_buffer_index==0)
	  {
	  line_buffer[0]='0';
	  line_buffer_index=1;
	  }
	}

      /* Clear text box before displaying line_buffer */
      index=0;
      do
	{
	Position_Cursor((x_position+1+index),y_position);
	printf(" ");

	index++;
	}while(index<size_of_field);

      /* Display line_buffer */
      index=line_buffer_index;
      do
	{
	Position_Cursor((x_position+size_of_field-line_buffer_index+index),y_position);
	index--;
	cprintf("%c",line_buffer[index]);
	}while(index>0);
      }

    if( (type==NUMP) && (input!='\xff') )
      {
      /* If the backspace key has not been hit. */
      if(input!=8)
	{
	invalid_input=FALSE;

	if(size_of_field>1)
	  {
	  min_range=0;
	  max_range=9;
	  }

	if( (input=='%') && (percent_entered==FALSE) )
	  {
	  percent_entered=TRUE;
	  percent_just_entered=TRUE;
	  }

	if( (input>='0') && (input<='9') )input=input-48;
	else
	  {
	  if(input<10) input=11;
	  }

	if( (percent_entered==FALSE) && (percent_just_entered==FALSE) && ( ( (size_of_field>1) && (input>max_range) ) || (input>9) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry, please enter %d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( (percent_entered==FALSE) && (size_of_field==1) && ( (input<min_range) || ( (input>max_range) && (input<10) ) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("%d is not a choice, please enter ",input);
	  cprintf("%d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( ( (percent_entered==TRUE) && (percent_just_entered==FALSE) ) || ( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field>1) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry.");
	  invalid_input=TRUE;
	  }

	if( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field==1) )
	  {
	  line_buffer_index=0;
	  }

	if(invalid_input==FALSE)
	  {
	  if( (line_buffer_index==1) && (line_buffer[0]=='0') )
	    {
	    line_buffer[0]=0;
	    line_buffer_index=0;
	    }

	  if(percent_just_entered==TRUE)
	    {
	    percent_just_entered=FALSE;
	    line_buffer[line_buffer_index]='%';
	    line_buffer_index++;
	    }
	  else
	    {
	    line_buffer[line_buffer_index]=(input+48);
	    line_buffer_index++;
	    }
	  }
	}
      else
	{
	/* If the backspace key has been hit */
	line_buffer_index--;
	if(line_buffer_index<0) line_buffer_index=0;
	line_buffer[line_buffer_index]=0;

	if(line_buffer_index==0)
	  {
	  line_buffer[0]='0';
	  line_buffer_index=1;
	  }

	if(percent_entered==TRUE) percent_entered=FALSE;
	}

      /* Clear text box before displaying line_buffer */
      index=0;
      do
	{
	Position_Cursor((x_position+1+index),y_position);
	printf(" ");

	index++;
	}while(index<size_of_field);

      /* Display line_buffer */
      index=line_buffer_index;
      do
	{
	Position_Cursor((x_position+size_of_field-line_buffer_index+index),y_position);
	index--;
	cprintf("%c",line_buffer[index]);
	}while(index>0);
      }

    if(debug.input_routine==TRUE)
      {
      Position_Cursor(60,23);
      printf("                ");

      Position_Cursor(60,24);
      printf("                ");

      Position_Cursor(50,23);
      printf("Line Buffer:  %10s",line_buffer);

      Position_Cursor(50,24);
      printf("Line Buffer Index:  %d",line_buffer_index);

      Position_Cursor(75,24);
      if(percent_entered==TRUE)
	{
	printf("P");
	}
      else
	{
	printf("  ");
	}
      }

    /* Place brackets back on screen as a precautionary measure. */
    if(type!=ESC)
      {
      Position_Cursor(x_position,y_position);
      cprintf("[");

      Position_Cursor((x_position+size_of_field+1),y_position);
      cprintf("]");
      }

    }while(proper_input_given==FALSE);

  return(data);
}

/* Interactive User Interface Control Routine */
void Interactive_User_Interface()
{
  int counter=0;
  int index=0;
  int menu=MM;

  Create_MBR_If_Not_Present();

  do
    {
    menu=Standard_Menu(menu);

/* Definitions for the menus */
/* MM   0x00                  Main Menu                     */

/*   CP   0x10                Create PDP or LDD             */

/*     CPDP 0x11              Create Primary DOS Partition  */
/*     CEDP 0x12              Create Extended DOS Partition */
/*     CLDD 0x13              Create Logical DOS Drive      */

/*   SAP  0x20                Set Active Partition          */

/*   DP   0x30                Delete partition or LDD       */

/*     DPDP 0x31              Delete Primary DOS Partition  */
/*     DEDP 0x32              Delete Extended DOS Partition */
/*     DLDD 0x33              Delete Logical DOS Drive      */
/*     DNDP 0x34              Delete Non-DOS Partition      */

/*   DPI  0x40                Display Partition Information */

/*   CD   0x50                Change Drive                  */

/*   MBR  0x60                MBR Functions                 */

/*     BMBR 0x61              Write booteasy MBR to drive   */
/*     AMBR 0x62              Write alternate MBR to drive  */
/*     SMBR 0x63              Save MBR to file              */
/*     RMBR 0x64              Remove MBR from disk          */

/* EXIT 0x0f                  Code to Exit from Program     */

    if( (menu==CPDP) || (menu==CEDP) )
      {
      /* Ensure that space is available in the primary partition table */
      /* to create a partition.                                        */

      /* First make sure that an empty slot is available.  */
      index=0;
      counter=0;
      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0) counter++;
	index++;
	}while(index<4);

      /* Next, make sure that there is a space available of at least   */
      /* two cylinders.                                                */
      Determine_Free_Space();
      if(part_table[(flags.drive_number-128)]
       .pri_part_largest_free_space<2) counter=4;

      if(counter>3)
	{
	Clear_Screen(NULL);

	if(menu==CPDP)
	 Print_Centered(4,"Create Primary DOS Partition",BOLD);
	else Print_Centered(4,"Create Extended DOS Partition",BOLD);
	Position_Cursor(4,6);
	printf("Current fixed disk drive: ");
	cprintf("%d",(flags.drive_number-127));

	Display_Primary_Partition_Information_SS();

	Position_Cursor(4,22);
	cprintf("No space to create a DOS partition.");

	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	}
      }

    if(menu==CPDP) Create_DOS_Partition_Interface(PRIMARY);
    if(menu==CEDP)
      {
      if(part_table[(flags.drive_number-128)]
       .ext_part_exists==TRUE)
	{
	Clear_Screen(NULL);

	Print_Centered(4,"Create Extended DOS Partition",BOLD);
	Position_Cursor(4,6);
	printf("Current fixed disk drive: ");
	cprintf("%d",(flags.drive_number-127));

	Display_Primary_Partition_Information_SS();

	Position_Cursor(4,22);
	cprintf("Extended DOS Partition already exists.");

	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	}
      else Create_DOS_Partition_Interface(EXTENDED);
      }

    if(menu==CLDD)
      {
      if(part_table[(flags.drive_number-128)].ext_part_exists==FALSE)
	{
	Position_Cursor(4,22);
	cprintf("Cannot create Logical DOS Drive without");
	Position_Cursor(4,23);
	cprintf("an Extended DOS Partition on the current drive.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	}
      else Create_Logical_Drive_Interface();
      }

    if(menu==SAP) Set_Active_Partition_Interface();

    if(menu==DPDP)
      {
      /* Ensure that primary partitions are available to delete. */
      counter=0;
      index=0;

      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0)
	  {
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==1) counter++;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==4) counter++;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==6) counter++;

	  if( (flags.version==W95) || (flags.version==W95B)
	   || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0e) counter++;
	    }

	  if( (flags.version==W95B) || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0b) counter++;
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0c) counter++;
	    }
	  }

	index++;
	}while(index<4);

      if(counter==0)
	{
	Position_Cursor(4,22);
	cprintf("No Primary DOS Partition to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	}
/* */
      else Delete_Primary_DOS_Partition_Interface();
      }

    if(menu==DEDP)
      {
      if(part_table[(flags.drive_number-128)].ext_part_exists==FALSE)
	{
	Position_Cursor(4,22);
	cprintf("No Extended DOS Partition to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	}
      else Delete_Extended_DOS_Partition_Interface();
      }

    if(menu==DLDD)
      {
      if( (part_table[(flags.drive_number-128)].num_of_log_drives==0) || (part_table[(flags.drive_number-128)].ext_part_exists==FALSE) )
	{
	Position_Cursor(4,22);
	cprintf("No Logical DOS Drive(s) to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	}
      else Delete_Logical_Drive_Interface();
      }

    if(menu==DNDP)
      {
      /* First Ensure that Non-DOS partitions are available to delete. */
      index=0;
      counter=0;

      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0)
	  {
	  counter++;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==1) counter--;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==4) counter--;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==5) counter--;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==6) counter--;

	  if( (flags.version==W95) || (flags.version==W95B)
	   || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0e) counter--;
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0f) counter--;
	    }

	  if( (flags.version==W95B) || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0b) counter--;
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0c) counter--;
	    }
	  }
	index++;
	}while(index<4);

      if(counter==0)
	{
	Position_Cursor(4,22);
	cprintf("No Non-DOS Partition to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	}
      else Delete_N_DOS_Partition_Interface();
      }
    if(menu==DPI) Display_Partition_Information();

    if(menu==CD) Change_Current_Fixed_Disk_Drive();

    if(menu==BMBR)
      {
      Create_BootEasy_MBR();
      Position_Cursor(4,22);
      cprintf("BootEasy MBR has been created.");
      Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
      }

    if(menu==AMBR)
      {
      char home_path[255];
      FILE *file_pointer;

      strcpy(home_path,path);
      strcat(home_path,"boot.mbr");
      /* Search the directory Free FDISK resides in before searching the */
      /* PATH in the environment for the boot.mbr file.                  */
      file_pointer=fopen(home_path,"rb");

      if(!file_pointer) file_pointer=fopen(searchpath("boot.mbr"),"rb");

      if(!file_pointer)
	{
	Position_Cursor(4,22);
	cprintf("\Unable to find the \"boot.mbr\" file...MBR has not been created.\n");
	}
      else
	{
	Create_Alternate_MBR();
	Position_Cursor(4,22);
	cprintf("MBR has been written using \"boot.mbr\"");
	}
      Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
      }

    if(menu==SMBR)
      {
      Save_MBR();
      Position_Cursor(4,22);
      cprintf("MBR has been saved to \"boot.mbr\"");
      Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
      }

    if(menu==RMBR)
      {
      Remove_MBR();
      Position_Cursor(4,22);
      cprintf("MBR has been removed from the hard disk.");
      Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
      }

    if(menu!=EXIT)
      {
      if( (menu>0x0f) || (menu==MM) )menu=MM;
      else menu=menu&0xf0;
      }

    }while(menu!=EXIT);

  if(flags.partitions_have_changed==TRUE)
    {
    Write_Partition_Tables();

    Clear_Screen(NOEXTRAS);

    if(flags.reboot==FALSE)
      {
      Position_Cursor(4,11);
      printf("You ");
      cprintf("MUST");
      printf(" restart your system for your changes to take effect.");
      Position_Cursor(4,12);
      printf("Any drives you have created or changed must be formatted");
      Position_Cursor(4,13);
      cprintf("AFTER");
      printf(" you restart.");

      Input(0,0,0,ESC,0,0,ESCE,0,0,NULL,NULL);
      Clear_Screen(NOEXTRAS);
      }
    else
      {
      Position_Cursor(4,13);
      cprintf("System will now restart");
      Position_Cursor(4,15);
      printf("Press any key when ready . . .");

      /* Wait for a keypress. */
      asm{
	mov ah,7
	int 0x21
	}

      Reboot_PC();
      }
    }
  else Clear_Screen(NOEXTRAS);
}

/* Convert the standard_partition_type to an LBA partition type */
int LBA_Partition_Type_To_Create(int standard_partition_type)
{
  int lba_partition_type;

  /* Extended int 0x13 FAT 32 */
  if(standard_partition_type==0x0b) lba_partition_type=0x0c;

  /* Extended int 0x13 FAT 16 */
  if( (standard_partition_type==1) || (standard_partition_type==4)
   || (standard_partition_type==6) ) lba_partition_type=0x0e;

  /* Extended int 0x13 Extended Partition */
  if(standard_partition_type==0x05) lba_partition_type=0x0f;

  return(lba_partition_type);
}

/* List the Partition Types */
void List_Partition_Types()
{
  int index=0;
  int row=4;
  int column=0;
  do
    {
    if( (index==0) || (index==64) || (index==128) || (index==192) )
      {
      Clear_Screen(NULL);
      Print_Centered(1,"List Partition Types",BOLD);
      row=4;
      column=0;
      }

    if( (row==20) && (column==0) )
      {
      row=4;
      column=20;
      }

    if( (row==20) && (column==20) )
      {
      row=4;
      column=40;
      }

    if( (row==20) && (column==40) )
      {
      row=4;
      column=60;
      }

    Position_Cursor(column,row);
    cprintf("%3d ",index);
    printf("%s",partition_lookup_table_buffer_long[index]);

    if( (index==63) || (index==127) || (index==191) || (index==255) )
      {
      Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
      }

    row++;
    index++;
    }while(index<=255);
}

/* Modify Extended Partition Information */
void Modify_Extended_Partition_Information(int logical_drive_number)
{
  int finished=FALSE;
  int input;

  unsigned long new_partition_size_in_sectors;
  unsigned long usage;

  do
    {
    Clear_Screen(NULL);
    Print_Centered(4,"Display/Modify Logical Drive Information",BOLD);

    Determine_Drive_Letters();

    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    Position_Cursor(4,8);
    printf("Partition            Mbytes    Description    Usage  Start Cyl  End Cyl");

    /* Drive Letter of Partition */
    if( (part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]==1)
     || (part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]==4)
     || (part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]==6) )
      {
      Position_Cursor(5,9);
      cprintf("%c:",drive_lettering_buffer[(flags.drive_number-128)]
       [(logical_drive_number+4)]);
      }

    if( ( (part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]==0x0b)
     || (part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]==0x0c) )
     && ( (flags.version==W95B) || (flags.version==W98) ) )
      {
      Position_Cursor(5,9);
      cprintf("%c:",drive_lettering_buffer[(flags.drive_number-128)]
       [(logical_drive_number+4)]);
      }

    if( (part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]==0x0e)
     && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
      {
      Position_Cursor(5,9);
      cprintf("%c:",drive_lettering_buffer[(flags.drive_number-128)]
       [(logical_drive_number+4)]);
      }

    /* Partition Number */
    Position_Cursor(8,9);
    printf("%d",(logical_drive_number+1));

    /* Partition Type */
    Position_Cursor(10,9);
    printf("%3d",(part_table[(flags.drive_number-128)]
     .log_drive_num_type[logical_drive_number]));

    /* Mbytes */
    Position_Cursor(24,9);
    printf("%6d",part_table[(flags.drive_number-128)]
     .log_drive_size_in_MB[logical_drive_number]);

    /* Description */
    Position_Cursor(33,9);
    printf("%15s",partition_lookup_table_buffer_long
     [part_table[(flags.drive_number-128)].log_drive_num_type
     [logical_drive_number]]);

    /* Usage */
    usage=((part_table[(flags.drive_number-128)]
     .log_drive_size_in_MB[logical_drive_number]*100)
     /part_table[(flags.drive_number-128)].ext_part_size_in_MB);
    if(usage>100) usage=100;

    Position_Cursor(51,9);
    printf("%3d%%",usage);

    /* Starting Cylinder */
    Position_Cursor(59,9);
    printf("%4d",part_table[(flags.drive_number-128)]
     .log_drive_start_cyl[logical_drive_number]);

    /* Ending Cylinder */
    Position_Cursor(69,9);
    printf("%4d",part_table[(flags.drive_number-128)]
     .log_drive_end_cyl[logical_drive_number]);

    Position_Cursor(4,12);
    printf("Choose one of the following:");

    Position_Cursor(4,14);
    cprintf("1.");
    printf("  Change partition type");
    Position_Cursor(4,15);
    cprintf("2.");
    printf("  List partition types");
    Position_Cursor(44,14);
    cprintf("3.");
    printf("  Hide/Unhide partition");
    Position_Cursor(44,15);
/*
    cprintf("4.");
    printf("  Reserved for future use.");
*/
    Position_Cursor(4,17);
    printf("Enter choice: ");

    flags.esc=FALSE;
    input=Input(1,19,17,NUM,1,3,ESCC,-1,0,NULL,NULL);
    if(flags.esc==TRUE)
      {
      input=99;
      finished=TRUE;
      }

    if(input==1)
      {
      /* Change partition type */
      Position_Cursor(4,19);
      printf("Enter new partition type (1-255)...................................");

      flags.esc=FALSE;
      input=Input(3,71,19,NUM,1,255,ESCC,-1,0,NULL,NULL);
      if(flags.esc==FALSE)
	{
	part_table[(flags.drive_number-128)]
	 .log_drive_num_type[logical_drive_number]=input;

	part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	flags.partitions_have_changed=TRUE;
	input=99;
	}
      else input=99;
      }

    if(input==2)
      {
      List_Partition_Types();
      }

    if(input==3)
      {
      /* Hide/Unhide partition */

      if(part_table[(flags.drive_number-128)]
       .log_drive_num_type[logical_drive_number]>140)
	{
	part_table[(flags.drive_number-128)]
	 .log_drive_num_type[logical_drive_number]
	 =part_table[(flags.drive_number-128)]
	 .log_drive_num_type[logical_drive_number]-140;

	part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	flags.partitions_have_changed=TRUE;
	input=99;
	}
      if( (input==3)
       && (part_table[(flags.drive_number-128)]
       .log_drive_num_type[logical_drive_number]<16) )
	{
	part_table[(flags.drive_number-128)]
	 .log_drive_num_type[logical_drive_number]
	 =part_table[(flags.drive_number-128)]
	 .log_drive_num_type[logical_drive_number]+140;

	part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	flags.partitions_have_changed=TRUE;
	input=99;
	}
      }

    if(input==4)
      {
      /* Reserved */
      }

  }while(finished==FALSE);
}

/* Modify Primary Partition Information */
void Modify_Primary_Partition_Information(int partition_number)
{
  int finished=FALSE;
  int input;

  unsigned long new_partition_size_in_sectors;
  unsigned long usage;

  partition_number--;   /* Adjust partition number to start with 0. */

  do
    {
      Clear_Screen(NULL);
      Print_Centered(4,"Display/Modify Partition Information",BOLD);

      Determine_Drive_Letters();

      Position_Cursor(4,6);
      printf("Current fixed disk drive: ");
      cprintf("%d",(flags.drive_number-127));

      Position_Cursor(4,8);
      printf("Partition   Status   Mbytes    Description    Usage  Start Cyl  End Cyl");

      /* Drive Letter of Partition */
      if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==6) )
	{
	Position_Cursor(5,9);
	printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [partition_number]);
	}

      if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0b) || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	Position_Cursor(5,9);
	printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [partition_number]);
	}

      if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	{
	Position_Cursor(5,9);
	printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [partition_number]);
	}

      /* Partition Number */
      Position_Cursor(8,9);
      cprintf("%d",(partition_number+1));

      /* Partition Type */
      Position_Cursor(10,9);
      printf("%3d",(part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]));

      /* Status */
      if(part_table[(flags.drive_number-128)].active_status[partition_number]>0)
	{
	Position_Cursor(19,9);
	printf("A");
	}

      /* Mbytes */
      Position_Cursor(24,9);
      printf("%6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]);

      /* Description */
      Position_Cursor(33,9);
      printf("%15s",partition_lookup_table_buffer_long[part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]]);

      /* Usage */
      usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
      if(usage>100) usage=100;

      Position_Cursor(51,9);
      printf("%3d%%",usage);

      /* Starting Cylinder */
      Position_Cursor(59,9);
      printf("%4d",part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number]);

      /* Ending Cylinder */
      Position_Cursor(69,9);
      printf("%4d",part_table[(flags.drive_number-128)].pri_part_end_cyl[partition_number]);

      Position_Cursor(4,12);
      printf("Choose one of the following:");

      Position_Cursor(4,14);
      cprintf("1.");
      printf("  Change partition type");
      Position_Cursor(4,15);
      cprintf("2.");
      printf("  List partition types");
      Position_Cursor(44,14);
      cprintf("3.");
      printf("  Hide/Unhide partition");
      Position_Cursor(44,15);
      cprintf("4.");
      printf("  Remove active status");

      Position_Cursor(4,17);
      printf("Enter choice: ");

      flags.esc=FALSE;
      input=Input(1,19,17,NUM,1,4,ESCC,-1,0,NULL,NULL);
      if(flags.esc==TRUE)
	{
	input=99;
	finished=TRUE;
	}

      if(input==1)
	{
	/* Change partition type */
	Position_Cursor(4,19);
	printf("Enter new partition type (1-255)...................................");

	flags.esc=FALSE;
	input=Input(3,71,19,NUM,1,255,ESCC,-1,0,NULL,NULL);
	if(flags.esc==FALSE)
	  {
	  Modify_Partition_Type(partition_number,input);
	  input=99;
	  }
	else input=99;
	}

      if(input==2)
	{
	List_Partition_Types();
	}

      if(input==3)
	{
	/* Hide/Unhide partition */

	if(part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]>140)
	  {
	  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]-140;

	  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	  flags.partitions_have_changed=TRUE;
	  input=99;
	  }
	if( (input==3) && (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]<16) )
	  {
	  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]+140;

	  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	  flags.partitions_have_changed=TRUE;
	  input=99;
	  }
	}

      if(input==4)
	{
	/* Remove active status */
	Clear_Active_Partition();
	}

    }while(finished==FALSE);

}

/* Modify Partition Type */
/* This will eventually be modified to include logical drives */
void Modify_Partition_Type(int partition_number,int type_number)
{
  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=type_number;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Compute the partition type to create. */
int Partition_Type_To_Create(unsigned long size_in_mb)
{
  int numeric_type;

  /* FAT 12 */
  if(size_in_mb<=16) numeric_type=1;

  /* Small FAT 16 */
  if( (size_in_mb>16) && (size_in_mb<=32) ) numeric_type=4;

  /* Large FAT 16 */
  if(size_in_mb>32) numeric_type=6;

  /* FAT 32 */
  if( (size_in_mb>128) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==TRUE) && (flags.fprmt==TRUE) ) numeric_type=0x0b;

  if( (size_in_mb>512) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==TRUE) ) numeric_type=0x0b;

  return(numeric_type);
}

/* Pause Routine */
void Pause()
{
  printf("\nPress any key to continue.\n");

  asm{
    mov ah,7
    int 0x21
    }
}

/* Position cursor on the screen */
void Position_Cursor(int column,int row)
{
  asm{
    /* Get video page number */
    mov ah,0x0f
    int 0x10

    /* Position Cursor */
    mov ah,0x02
    mov dh,BYTE PTR row
    mov dl,BYTE PTR column
    int 0x10
    }
}

/* Print Centered Text */
void Print_Centered(int y,char *text,int style)
{
  int x=40-strlen(text)/2;

  Position_Cursor(x,y);

  if(style==BOLD) cprintf(text);
  else printf(text);
}

/* Reboot the PC */
void Reboot_PC()
{
  /* Note:  Reboot is a cold start. */
  void ((far * fp) ())=(void(far*) () ) ((0xffffL<<16) | 0x0000L);
  *(int far *) ((0x0040L << 16) | 0x0072)=0;
  fp();
}

/* Re-Initialize LBA related functions. */
void Re_Initialization()
{
  /* Check for interrupt 0x13 extensions (If the proper version is set.) */
  if( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) )
   Check_For_INT13_Extensions();

  /* If the version is W95B or W98 then default to FAT32 support.  */
  if( (flags.version==W95B) || (flags.version==W98) ) flags.fat32=TRUE;

  /* Initialize LBA structures, if necessary. */
  if(flags.use_extended_int_13==TRUE) Initialize_LBA_Structures();

  Read_Partition_Tables();
}


/* Set Active Partition */
void Set_Active_Partition(int partition_number)
{
  int index=0;

  do
    {
    if(index==partition_number) part_table[(flags.drive_number-128)].active_status[index]=0x80;
    else part_table[(flags.drive_number-128)].active_status[index]=0x00;

    index++;
    }while(index<4);

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Set Active Partition Interface */
int Set_Active_Partition_Interface()
{
  int correct_input=FALSE;
  int index=0;
  int input;

  int available_partition_counter=0;
  int first_available_partition_active=FALSE;
  int only_active_partition_active=FALSE;

  int partition_settable[4];

  /* Check to see if other partitions that can be set active exist.*/
  /* Also check to see what partitions are available to set active.*/
  do
    {
    partition_settable[index]=FALSE;

    if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1)
     || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4)
     || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6)

     || ( ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b)
     || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) )
     && ( (flags.version==W95B) || (flags.version==W98) ) )

     || ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e)
     && ( (flags.version==W95) || (flags.version==W95B)
     || (flags.version==W98) ) )
     && (flags.set_any_pri_part_active==FALSE) )
      {
      available_partition_counter++;
      if( (available_partition_counter==1)
       && (part_table[(flags.drive_number-128)]
       .active_status[index]==0x80) )first_available_partition_active=TRUE;
      partition_settable[index]=TRUE;
      }

    if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
     && (flags.set_any_pri_part_active==TRUE) )
      {
      available_partition_counter++;
      if( (available_partition_counter==1)
       && (part_table[(flags.drive_number-128)]
       .active_status[index]==0x80) )first_available_partition_active=TRUE;
      partition_settable[index]=TRUE;
      }

    index++;
    }while(index<=3);

  if( (available_partition_counter==1) && (first_available_partition_active==TRUE) ) only_active_partition_active=TRUE;

  Clear_Screen(NULL);
  Print_Centered(4,"Set Active Partition",BOLD);

  Display_Primary_Partition_Information_SS();

  if(available_partition_counter==0)
    {
    Position_Cursor(4,22);
    cprintf("No partitions to make active.");

    Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    }

  if( (only_active_partition_active==FALSE) && (available_partition_counter>0) )
    {
    Position_Cursor(4,16);
    printf("Enter the number of the partition you want to make active...........: ");

    do
      {
      flags.esc=FALSE;
      input=Input(1,70,16,NUM,1,4,ESCR,-1,0,NULL,NULL);
      if(flags.esc==TRUE) return(1);

      /* Ensure that input is valid. */
      if(partition_settable[(input-1)]==TRUE) correct_input=TRUE;
      else
	{
	Position_Cursor(4,23);
	cprintf("%d is not a choice. Please enter a valid choice.",input);
	}

      }while(correct_input==FALSE);

    Set_Active_Partition(input-1);

    Clear_Screen(NULL);
    Print_Centered(4,"Set Active Partition",BOLD);

    /* */
    Display_Primary_Partition_Information_SS();

    Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    }

  if(only_active_partition_active==TRUE)
    {
    Position_Cursor(4,22);
    cprintf("The only startable partition on Drive %d is already set active.",(flags.drive_number-127));

    Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
    }

  return(0);
}

/* Standard Menu Routine */
/* Displays the menus laid out in a standard format and returns the */
/* selection chosen by the user.                                    */
int Standard_Menu(int menu)
{
  int counter;
  int index;

  int display_menu=TRUE;
  int result;
  int maximum_number_of_options=0;

  int input;

  char copyleft[60]="";
  char program_name[60]="";
  char program_description[60]="";
  char version[30]="";

  char title[60]="";

  char option_1[60]="";
  char option_2[60]="";
  char option_3[60]="";
  char option_4[60]="";
  char option_5[60]="Change current fixed disk drive";

  char optional_char_1[1]={NULL};
  char optional_char_2[1]={NULL};

  do
    {
    /* Load Menu Text */

    if(flags.use_freedos_label==FALSE)
      {
      strcpy(program_name,PRINAME);
      strcat(program_name,"    Version ");
      strcat(program_name,VERSION);
      }
    else strcpy(program_name,ALTNAME);

    strcpy(program_description,"Fixed Disk Setup Program");
    strcpy(copyleft,"GNU GPL Copyright Brian E. Reifsnyder ");
    strcat(copyleft,COPYLEFT);

    if(menu==MM)
      {
      maximum_number_of_options=4;
      strcpy(title,"FDISK Options");
      strcpy(option_1,"Create DOS partition or Logical DOS Drive");
      strcpy(option_2,"Set Active partition");
      strcpy(option_3,"Delete partition or Logical DOS Drive");

      if(flags.extended_options_flag==FALSE) strcpy(option_4,"Display partition information");
      else strcpy(option_4,"Display/Modify partition information");
      }

    if(menu==CP)
      {
      maximum_number_of_options=3;
      strcpy(title,"Create DOS Partition or Logical DOS Drive");
      strcpy(option_1,"Create Primary DOS Partition");
      strcpy(option_2,"Create Extended DOS Partition");
      strcpy(option_3,"Create Logical DOS Drive(s) in the Extended DOS Partition");
      strcpy(option_4,"");
      }

    if(menu==DP)
      {
      maximum_number_of_options=4;
      strcpy(title,"Delete DOS Partition or Logical DOS Drive");
      strcpy(option_1,"Delete Primary DOS Partition");
      strcpy(option_2,"Delete Extended DOS Partition");
      strcpy(option_3,"Delete Logical DOS Drive(s) in the Extended DOS Partition");
      strcpy(option_4,"Delete Non-DOS Partition");
      if(flags.version==FOUR) maximum_number_of_options=3;
      }

    if(menu==MBR)
      {
      maximum_number_of_options=4;
      strcpy(title,"MBR Maintenance");
      strcpy(option_1,"Create BootEasy MBR");
      strcpy(option_2,"Create MBR using the saved file");
      strcpy(option_3,"Save the MBR to a file");
      strcpy(option_4,"Remove the MBR from the disk");
      }

    /* Display Program Name and Copyright Information */
    Clear_Screen(NULL);

    if( (flags.extended_options_flag==TRUE) && (menu==MM) )
     flags.display_name_description_copyright=TRUE;

    if(flags.display_name_description_copyright==TRUE)
      {
      Print_Centered(0,program_name,STANDARD);
      Print_Centered(1,program_description,STANDARD);
      Print_Centered(2,copyleft,STANDARD);

      if(flags.use_freedos_label==TRUE)
	{
	strcpy(version,"Version:  ");
	strcat(version,VERSION);
	Position_Cursor(49,24);
	printf("%30s",version);
	}
      }

    flags.display_name_description_copyright=FALSE;

    /* Display Menu Title(s) */
    Print_Centered(4,title,BOLD);

    /* Display Current Drive Number */
    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    if(menu==DP)
      {
      /* Ensure that primary partitions are available to delete. */
      counter=0;
      index=0;

      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0)
	 counter++;
	index++;
	}while(index<4);

      if(counter==0)
	{
	Position_Cursor(4,22);
	cprintf("No partitions to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0,NULL,NULL);
	menu=MM;
	return(menu);
	}
      }

    /* Display Menu */
    Position_Cursor(4,8);
    printf("Choose one of the following:");

    Position_Cursor(4,10);
    cprintf("1.  ");
    printf("%s",option_1);

    if(maximum_number_of_options>1)
      {
      Position_Cursor(4,11);
      cprintf("2.  ");
      printf("%s",option_2);
      }

    if(maximum_number_of_options>2)
      {
      Position_Cursor(4,12);
      cprintf("3.  ");
      printf("%s",option_3);
      }

    if(maximum_number_of_options>3)
      {
      Position_Cursor(4,13);
      cprintf("4.  ");
      printf("%s",option_4);
      }

    if( (menu==MM) && (flags.more_than_one_drive==TRUE) )
      {
      maximum_number_of_options=5;
      Position_Cursor(4,14);
      cprintf("5.  ");
      printf("%s",option_5);
      }

    if( (menu==MM) && (flags.extended_options_flag==TRUE) )
      {
      Position_Cursor(50,15);
      cprintf("M.  ");
      printf("MBR maintenance");

      optional_char_1[0]='M';
      }
    else optional_char_1[0]=NULL;

    if( (menu==MM) && (flags.allow_abort==TRUE) )
      {
      Position_Cursor(50,16);
      cprintf("A.  ");
      printf("Abort changes and exit");

      optional_char_2[0]='A';
      }
    else optional_char_2[0]=NULL;

    /* Display Special Messages */

    /* If there is not an active partition */
    if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[0]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[1]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[2]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[3]>0) ) && (flags.drive_number==0x80) && (menu==MM) && (part_table[flags.drive_number-128].active_status[0]==0) && (part_table[flags.drive_number-128].active_status[1]==0) && (part_table[flags.drive_number-128].active_status[2]==0) && (part_table[flags.drive_number-128].active_status[3]==0) )
      {
      Position_Cursor(4,21);
      cprintf("WARNING! ");
      printf("No partitions are set active - disk 1 is not startable unless");
      Position_Cursor(4,22);
      printf("a partition is set active");
      }

    /* Get input from user */
    Position_Cursor(4,17);
    printf("Enter choice: ");

    if(menu==MM) input=Input(1,19,17,NUM,1,maximum_number_of_options,ESCE,1,0,optional_char_1,optional_char_2);
    else input=Input(1,19,17,NUM,1,maximum_number_of_options,ESCR,-1,0,NULL,NULL);

    /* Process the input */
    if(input=='A')
      {
      /* Abort any changes and exit the program immediately. */
      flags.screen_color=7;  /* Set screen colors back to default. */
      Clear_Screen(NOEXTRAS);
      exit(0);
      }

    if(input=='M') input=6;

    if(input!=0)
      {
      if(menu==MM) menu=input<<4;
      else menu=menu|input;
      }
    else
      {
      if(menu==MM)
	{
	menu=EXIT;
	}
      else
	{
	if(menu>0x0f) menu=MM;
	else menu=menu&0xf0;
	}
      }

    if( (menu==MM) || (menu==CP) || (menu==DP) || (menu==MBR) ) display_menu=TRUE;
    else display_menu=FALSE;

    }while(display_menu==TRUE);

  return(menu);
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[], char *env[])
{
  int command_ok;
  int index;
  int location;

  /* Place the filename of this program into filename */
  index=strlen(argv[0]);
  location=0;
  do
    {
    if(argv[0] [index]=='\\')
      {
      location=index+1;
      index=-1;
      }
    index--;
    }while(index>=0);

  index=location;
  do
    {
    filename[index-location]=argv[0] [index];
    index++;
    }while(index<=(strlen(argv[0])) );

  index=0;
  do
    {
    if(filename[index]=='.') filename[index]=0;
    index++;
    }while(index<12);

  /* Place the path of this program into path. */
  if(location>0)
    {
    index=0;
    do
      {
      path[index]=argv[0] [index];

      index++;
      }while(index<location);
    path[index]=0;
    }
  else path[0]=0;

  Initialization(&*env);

  if(debug.path==TRUE)
    {
    printf("\nThe PATH to \"%s\" is:  ",filename);
    printf("\"%s\"\n\n",path);
    Pause();
    }

  /* New Parsing Routine */
  /* The command line format is:                                            */
  /* /aaaaaaaaaa:999999,999 9 /aaaaaaaaaa:999999,999 /aaaaaaaaaa:999999,999 /aaaaaaaaaa:999999,999   */
  /* Where:   "a" is an ascii character and "9" is a number                 */
  /* Note:  The second "9" in the above command line format is the drive    */
  /*        number.  This drive number can now be anywhere on the line.     */

  /* If "FDISK" is typed without any options */
  number_of_command_line_options=Get_Options(&*argv,argc);
  if(number_of_command_line_options==0)
    {
    /* Ask the user if FAT32 is desired. */
    if( (flags.version==W95B) || (flags.version==W98) )
     Ask_User_About_FAT32_Support();

    Interactive_User_Interface();
    exit(0);
    }
  else
    {
    do
      {
      if(debug.command_line_arguments==TRUE)
	{
	int command_line_index=0;

	printf("\n");
	do
	  {
	  printf("/%s:",arg[command_line_index].choice);
	  printf("%d,",arg[command_line_index].value);
	  printf("%d ",arg[command_line_index].extra_value);
	  command_line_index++;
	  }while(command_line_index<number_of_command_line_options);

	Pause();
	}

      command_ok=FALSE;

      switch(arg[0].choice[0])
	{
	case 'A':
	  {
	  if(0==strcmp(arg[0].choice,"ACTIVATE"))
	    {
	    if((arg[0].value<1) || (arg[0].value>4))
	      {
	      printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	      exit(9);
	      }

	    Set_Active_Partition((arg[0].value-1));
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }

	  if(0==strcmp(arg[0].choice,"ACTOK"))
	    {
	    if( (flags.version==W95B) || (flags.version==W98) )
	     Ask_User_About_FAT32_Support();

	    Interactive_User_Interface();
	    exit(0);
	    }

	  if(0==strcmp(arg[0].choice,"AMBR"))
	    {
	    Create_Alternate_MBR();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }

	  if(0==strcmp(arg[0].choice,"AUTO"))
	    {
	    Automatically_Partition_Hard_Drive();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }
	  }
	  break;

	case 'C':
	  {
	  if(0==strcmp(arg[0].choice,"CLEAR"))
	    {
	    Clear_Partition_Table();
	    command_ok=TRUE;

	    Re_Initialization();
	    Shift_Command_Line_Options(1);
	    }

	  if(0==strcmp(arg[0].choice,"CLEARFLAG"))
	    {
	    Command_Line_Clear_Flag();
	    command_ok=TRUE;
	    }

	  if(0==strcmp(arg[0].choice,"CMBR"))
	    {
	    Create_MBR();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }
	  }
	  break;

	case 'D':
	  {
	  if(0==strcmp(arg[0].choice,"DEACTIVATE"))
	    {
	    Clear_Active_Partition();
	    Write_Partition_Tables();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }

	  if(0==strcmp(arg[0].choice,"DELETE"))
	    {
	    Command_Line_Delete();
	    command_ok=TRUE;
	    }

	  if(0==strcmp(arg[0].choice,"DUMP"))
	    {
	    Dump_Partition_Information();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }
	  }
	  break;

	case 'E':
	  {
	  if(0==strcmp(arg[0].choice,"EXT"))
	    {
	    Command_Line_Create_Extended_Partition();
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'F':
	  {
	  if(0==strcmp(arg[0].choice,"FPRMT"))
	    {
	    if( (flags.version==W95B) || (flags.version==W98) ) flags.fprmt=TRUE;
	    Interactive_User_Interface();
	    exit(0);
	    }
	  }
	  break;

	case 'I':
	  {
	  if(0==strcmp(arg[0].choice,"INFO"))
	    {
	    Command_Line_Info();
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'L':
	  {
	  if(0==strcmp(arg[0].choice,"LOG"))
	    {
	    Command_Line_Create_Logical_DOS_Drive();
	    command_ok=TRUE;
	    }

	  if(0==strcmp(arg[0].choice,"LOGO"))
	    {
	    flags.fat32=FALSE;
	    Command_Line_Create_Logical_DOS_Drive();
	    flags.fat32=TRUE;
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'M':
	  {
	  if(0==strcmp(arg[0].choice,"MBR"))
	    {
	    Create_MBR();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }

	  if(0==strcmp(arg[0].choice,"MODIFY"))
	    {
	    Command_Line_Modify();
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'P':
	  {
	  if(0==strcmp(arg[0].choice,"PRI"))
	    {
	    Command_Line_Create_Primary_Partition();
	    command_ok=TRUE;
	    }

	  if(0==strcmp(arg[0].choice,"PRIO"))
	    {
	    flags.fat32=FALSE;
	    Command_Line_Create_Primary_Partition();
	    flags.fat32=TRUE;
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'Q':
	  {
	  if(0==strcmp(arg[0].choice,"Q"))
	    {
	    flags.reboot=FALSE;

	    if( (flags.version==W95B) || (flags.version==W98) )
	     Ask_User_About_FAT32_Support();

	    Interactive_User_Interface();
	    exit(0);
	    }
	  }
	  break;

	case 'R':
	  {
	  if(0==strcmp(arg[0].choice,"REBOOT")) Reboot_PC();

	  if(0==strcmp(arg[0].choice,"RMBR"))
	    {
	    Remove_MBR();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }
	  }
	  break;

	case 'S':
	  {
	  if(0==strcmp(arg[0].choice,"SETFLAG"))
	    {
	    Command_Line_Set_Flag();
	    command_ok=TRUE;
	    }

	  if(0==strcmp(arg[0].choice,"SMBR"))
	    {
	    Save_MBR();
	    command_ok=TRUE;

	    Shift_Command_Line_Options(1);
	    }

	  if(0==strcmp(arg[0].choice,"STATUS"))
	    {
	    Command_Line_Status();
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'T':
	  {
	  if(0==strcmp(arg[0].choice,"TESTFLAG"))
	    {
	    Command_Line_Test_Flag();
	    command_ok=TRUE;
	    }
	  }
	  break;

	case 'X':
	  {
	  if(0==strcmp(arg[0].choice,"X"))
	    {
	    Command_Line_X();
	    exit(0);
	    }
	  }
	  break;

	case '?':
	  {
	  if(0==strcmp(arg[1].choice,"NOPAUSE"))
	    {
	    flags.do_not_pause_help_information=TRUE;
	    Shift_Command_Line_Options(1);
	    }
	  Display_Help_Screen();
	  command_ok=TRUE;

	  Shift_Command_Line_Options(1);
	  }
	  break;

	default:
	  {
	  printf("\nSyntax Error...Operation Terminated.\n");
	  exit(1);
	  }
	}

      if(command_ok==FALSE)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      }while(number_of_command_line_options>0);

    Write_Partition_Tables();
    exit(0);
    }
}
